// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.


//+-----------------------------------------------------------------------------
//

//
//  $TAG ENGR

//      $Module:    win_mil_graphics_geometry
//      $Keywords:
//
//  $Description:
//      Geometry sink (IGeometrySink) and generator (IGeometryGenerator)
//      interface declarations
//
//  $ENDTAG
//
//------------------------------------------------------------------------------

struct CCoverageInterval;


//+-----------------------------------------------------------------------------
//
//  Interface:
//      Geometry Sink
//
//  Synopsis:
//      Common interface for all classes that consume geometry data generated by
//      geometry generators
//
//      Tessellators, and other geometry generating code, output results using
//      this interface, and hence be independent of the brush/primitive type.
//
//------------------------------------------------------------------------------
interface IGeometrySink
{
    //
    // Aliased geometry output
    //

    virtual HRESULT AddVertex(
        __in_ecount(1) const MilPoint2F &ptPosition,
            // In: Vertex coordinates
        __out_ecount(1) WORD *pidxOut
            // Out: Index of vertex
        ) PURE;

    virtual HRESULT AddIndexedVertices(
        UINT cVertices,
            // In: number of vertices
        __in_bcount(cVertices*uVertexStride) const void *pVertexBuffer,
            // In: vertex buffer containing the vertices
        UINT uVertexStride,
            // In: size of each vertex
        MilVertexFormat mvfFormat,
            // In: format of each vertex
        UINT cIndices,
            // In: Number of indices
        __in_ecount(cIndices) const UINT *puIndexBuffer
            // In: index buffer                                                             
        ) PURE;

    virtual void SetTransformMapping(
        __in_ecount(1) const MILMatrix3x2 &mat2DTransform
        ) PURE;

    virtual HRESULT AddTriangle(
        DWORD idx1,
            // In: Index of triangle's first vertex
        DWORD idx2,
            // In: Index of triangle's second vertex
        DWORD idx3
            // In: Index of triangle's third vertex
        ) PURE;

    //
    // Trapezoidal AA geometry output
    //

    virtual HRESULT AddComplexScan(
        INT nPixelY,
            // In: y coordinate in pixel space
        __in_ecount(1) const CCoverageInterval *pIntervalSpanStart
            // In: coverage segments
        ) PURE;
    
    virtual HRESULT AddTrapezoid(
        float rYMin,
            // In: y coordinate of top of trapezoid
        float rXLeftYMin,
            // In: x coordinate for top left
        float rXRightYMin,
            // In: x coordinate for top right
        float rYMax,
            // In: y coordinate of bottom of trapezoid
        float rXLeftYMax,
            // In: x coordinate for bottom left
        float rXRightYMax,
            // In: x coordinate for bottom right
        float rXDeltaLeft,
            // In: trapezoid expand radius
        float rXDeltaRight
            // In: trapezoid expand radius
        ) PURE;
    
    virtual HRESULT AddParallelogram(
        __in_ecount(4) const MilPoint2F *rgPosition
        ) PURE;
    
    //
    // Query sink status
    //

    // Some geometry generators don't actually know if they have output
    // any triangles, so they need to get this information from the geometry sink.

    virtual BOOL IsEmpty() PURE;

};

class CHwPipelineBuilder;

//+-----------------------------------------------------------------------------
//
//  Interface:
//      Geometry Generator
//
//  Synopsis:
//      Common interface for all classes that generate geometry data
//
//  Responsibilities:
//      - Specifying incoming per vertex data to pipeline
//      - Generate tessellation with coordinates and sometimes alpha falloff
//        [texture coordinates for trapezoidal AA] (basic vertex information) for
//        a given fill path
//
//  Not responsible for:
//      - Generating per vertex color source specific data
//
//  Inputs required:
//      - Path
//      - Per-primitve anti-alias setting
//      - Vertex builder to send basic vertex info
//      - Translation from path space to device space
//
//------------------------------------------------------------------------------
interface IGeometryGenerator
{
    //+-------------------------------------------------------------------------
    //
    //  Member:
    //      GetPerVertexDataType
    //
    //  Synopsis:
    //      Return vertex fields that are generated when this is used
    //
    //--------------------------------------------------------------------------

    virtual void GetPerVertexDataType(
        __out_ecount(1) MilVertexFormat &mvfFullyGenerated
        ) const PURE;

    //+-------------------------------------------------------------------------
    //
    //  Member:
    //      SendGeometry
    //
    //  Synopsis:
    //      Geometry is generated and passed to given sink
    //
    //--------------------------------------------------------------------------

    virtual HRESULT SendGeometry(
        __inout_ecount(1) IGeometrySink *pGeomSink
        ) PURE;

    //+-------------------------------------------------------------------------
    //
    //  Member:
    //      SendGeometryModifiers
    //
    //  Synopsis:
    //      Generator has a chance to modify the pipeline colors in order to
    //      apply anti-aliasing, blend colors, etc.
    //
    //--------------------------------------------------------------------------
    virtual HRESULT SendGeometryModifiers(
        __inout_ecount(1) CHwPipelineBuilder *pPipelineBuilder
        ) PURE;

    //+-------------------------------------------------------------------------
    //
    //  Member:
    //      SendLighting
    //
    //  Synopsis:
    //      Geometry generator supplies lighting information to the pipeline.
    //
    //--------------------------------------------------------------------------
    virtual HRESULT SendLighting(
        __inout_ecount(1) CHwPipelineBuilder *pPipelineBuilder
        ) PURE;
};



