// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Package tools shows examples of various tools that Gemini model can use to generate text.
package tools

// [START googlegenaisdk_tools_vais_with_txt]
import (
	"context"
	"fmt"
	"io"

	"google.golang.org/genai"
)

// generateWithGoogleVAIS shows how to generate text using VAIS Search.
func generateWithGoogleVAIS(w io.Writer, datastore string) error {
	//datastore = "gs://your-datastore/your-prefix"
	ctx := context.Background()

	client, err := genai.NewClient(ctx, &genai.ClientConfig{
		HTTPOptions: genai.HTTPOptions{APIVersion: "v1"},
	})
	if err != nil {
		return fmt.Errorf("failed to create genai client: %w", err)
	}

	modelName := "gemini-2.5-flash"
	contents := genai.Text("How do I make an appointment to renew my driver's license?")
	config := &genai.GenerateContentConfig{
		Tools: []*genai.Tool{
			{
				Retrieval: &genai.Retrieval{
					VertexAISearch: &genai.VertexAISearch{
						Datastore: datastore,
					},
				},
			},
		},
	}

	resp, err := client.Models.GenerateContent(ctx, modelName, contents, config)
	if err != nil {
		return fmt.Errorf("failed to generate content: %w", err)
	}

	respText := resp.Text()

	fmt.Fprintln(w, respText)

	// Example response:
	// 'The process for making an appointment to renew your driver's license varies depending on your location. To provide you with the most accurate instructions...'

	return nil
}

// [END googlegenaisdk_tools_vais_with_txt]
