import nconf from 'nconf';
import {
  generateUser,
  requester,
  translate as t,
} from '../../../../../helpers/api-integration/v3';
import {
  bcryptCompare,
  sha1MakeSalt,
  sha1Encrypt as sha1EncryptPassword,
} from '../../../../../../website/server/libs/password';

describe('POST /user/auth/local/login', () => {
  let api;
  let user;
  const endpoint = '/user/auth/local/login';
  const password = 'password';
  beforeEach(async () => {
    api = requester();
    user = await generateUser();
  });

  it('success with username', async () => {
    const response = await api.post(endpoint, {
      username: user.auth.local.username,
      password,
    });
    expect(response.apiToken).to.eql(user.apiToken);
  });

  it('success with email', async () => {
    const response = await api.post(endpoint, {
      username: user.auth.local.email,
      password,
    });
    expect(response.apiToken).to.eql(user.apiToken);
  });

  it('user is blocked', async () => {
    await user.updateOne({ 'auth.blocked': 1 });
    await expect(api.post(endpoint, {
      username: user.auth.local.username,
      password,
    })).to.eventually.be.rejected.and.eql({
      code: 401,
      error: 'NotAuthorized',
      message: t('accountSuspended', { communityManagerEmail: nconf.get('EMAILS_COMMUNITY_MANAGER_EMAIL'), userId: user._id }),
    });
  });

  it('wrong password', async () => {
    await expect(api.post(endpoint, {
      username: user.auth.local.username,
      password: 'wrong-password',
    })).to.eventually.be.rejected.and.eql({
      code: 401,
      error: 'NotAuthorized',
      message: t('invalidLoginCredentialsLong'),
    });
  });

  it('missing username', async () => {
    await expect(api.post(endpoint, {
      password: 'wrong-password',
    })).to.eventually.be.rejected.and.eql({
      code: 400,
      error: 'BadRequest',
      message: t('invalidReqParams'),
    });
  });

  it('missing password', async () => {
    await expect(api.post(endpoint, {
      username: user.auth.local.username,
    })).to.eventually.be.rejected.and.eql({
      code: 400,
      error: 'BadRequest',
      message: t('invalidReqParams'),
    });
  });

  it('converts user with SHA1 encrypted password to bcrypt encryption', async () => {
    const textPassword = 'mySecretPassword';
    const salt = sha1MakeSalt();
    const sha1HashedPassword = sha1EncryptPassword(textPassword, salt);

    await user.updateOne({
      'auth.local.hashed_password': sha1HashedPassword,
      'auth.local.passwordHashMethod': 'sha1',
      'auth.local.salt': salt,
    });

    await user.sync();
    expect(user.auth.local.passwordHashMethod).to.equal('sha1');
    expect(user.auth.local.salt).to.equal(salt);
    expect(user.auth.local.hashed_password).to.equal(sha1HashedPassword);

    // login
    await api.post(endpoint, {
      username: user.auth.local.email,
      password: textPassword,
    });

    await user.sync();
    expect(user.auth.local.passwordHashMethod).to.equal('bcrypt');
    expect(user.auth.local.salt).to.be.undefined;
    expect(user.auth.local.hashed_password).not.to.equal(sha1HashedPassword);

    const isValidPassword = await bcryptCompare(textPassword, user.auth.local.hashed_password);
    expect(isValidPassword).to.equal(true);
  });

  it('sets auth.timestamps.updated', async () => {
    const oldUpdated = new Date(user.auth.timestamps.updated);
    // login
    await api.post(endpoint, {
      username: user.auth.local.email,
      password,
    });

    await user.sync();
    expect(user.auth.timestamps.updated).to.be.greaterThan(oldUpdated);
  });

  it('user uses social authentication and has no password', async () => {
    await user.unset({
      'auth.local.hashed_password': 1,
    });

    await user.sync();
    expect(user.auth.local.hashed_password).to.be.undefined;

    await expect(api.post(endpoint, {
      username: user.auth.local.username,
      password: 'any-password',
    })).to.eventually.be.rejected.and.eql({
      code: 401,
      error: 'NotAuthorized',
      message: t('invalidLoginCredentialsLong'),
    });
  });
});
