/* eslint-disable camelcase */
import { each, camelCase } from 'lodash';
import {
  expectValidTranslationString,
} from '../helpers/content.helper';

import { CLASSES } from '../../website/common/script/content/constants';

import gearData from '../../website/common/script/content/gear';
import * as backerGear from '../../website/common/script/content/gear/sets/special/special-backer';
import * as contributorGear from '../../website/common/script/content/gear/sets/special/special-contributor';

const allGear = gearData.tree;

describe('Gear', () => {
  each(allGear, (piece, gearType) => {
    describe(gearType, () => {
      each(piece, (items, klass) => {
        context(`${klass} ${gearType}s`, () => {
          it('have a value of at least 0 for each stat', () => {
            each(items, gear => {
              expect(gear.con, gear.key).to.be.at.least(0);
              expect(gear.int, gear.key).to.be.at.least(0);
              expect(gear.per, gear.key).to.be.at.least(0);
              expect(gear.str, gear.key).to.be.at.least(0);
            });
          });

          it('have a purchase value of at least 0', () => {
            each(items, gear => {
              expect(gear.value, gear.key).to.be.at.least(0);
            });
          });

          it('has a canBuy function', () => {
            each(items, gear => {
              expect(gear.canBuy, gear.key).to.be.a('function');
            });
          });

          it('have valid translation strings for text and notes', () => {
            each(items, gear => {
              expectValidTranslationString(gear.text, gear.key);
              expectValidTranslationString(gear.notes, gear.key);
            });
          });

          it('has a set attribue', () => {
            each(items, gear => {
              expect(gear.set, gear.key).to.exist;
            });
          });

          it('has a valid value for klass or specialClass', () => {
            const validClassValues = CLASSES + ['base', 'mystery', 'armoire'];
            each(items, gear => {
              const field = gear.klass === 'special' ? gear.specialClass : gear.klass;
              if (gear.klass === 'special' && field === undefined) {
                // some special gear doesn't have a klass
                return;
              }
              expect(field, gear.key).to.exist;
              expect(validClassValues, gear.key).to.include(field);
            });
          });
        });
      });
    });
  });

  it('only assigns mage weapons twoHanded', () => {
    each([allGear.armor.special, allGear.head.special, allGear.shield.special], gearType => {
      each(gearType, gear => {
        if (gear.specialClass === 'wizard') {
          expect(gear.twoHanded, gear.key).to.not.eql(true);
        }
      });
    });
  });

  describe('backer gear', () => {
    let user;

    beforeEach(() => {
      user = {
        backer: {},
        items: { gear: { owned: {} } },
      };
    });

    const cases = {
      armor_special_0: 45,
      armor_special_2: 300,
      head_special_0: 45,
      head_special_2: 300,
      weapon_special_0: 70,
      weapon_special_2: 300,
      weapon_special_3: 300,
    };

    each(cases, (tierRequirement, key) => {
      context(key, () => {
        const camelCaseKey = camelCase(key);

        it(`canOwn returns true if user has a backer tier of ${tierRequirement} or higher`, () => {
          user.backer.tier = tierRequirement;
          expect(backerGear[camelCaseKey].canOwn(user)).to.eql(true);

          user.backer.tier = tierRequirement + 1;
          expect(backerGear[camelCaseKey].canOwn(user)).to.eql(true);
        });

        it('canOwn returns true if user already owns the item', () => {
          user.items.gear.owned[key] = true;
          expect(backerGear[camelCaseKey].canOwn(user)).to.eql(true);
        });

        it('canOwn returns true if user has previously owned the item', () => {
          user.items.gear.owned[key] = false;
          expect(backerGear[camelCaseKey].canOwn(user)).to.eql(true);
        });

        it('canOwn returns false if user does not have tier requirement and did not previously own the item', () => {
          expect(backerGear[camelCaseKey].canOwn(user)).to.eql(false);
        });
      });
    });
  });

  describe('contributor gear', () => {
    let user;

    beforeEach(() => {
      user = {
        contributor: {},
        items: { gear: { owned: {} } },
      };
    });

    const cases = {
      armor_special_1: 2,
      head_special_1: 3,
      shield_special_1: 5,
      weapon_special_1: 4,
    };

    each(cases, (tierRequirement, key) => {
      context(key, () => {
        const camelCaseKey = camelCase(key);

        it(`canOwn returns true if user has a contributor tier of ${tierRequirement} or higher`, () => {
          user.contributor.level = tierRequirement;
          expect(contributorGear[camelCaseKey].canOwn(user)).to.eql(true);

          user.contributor.level = tierRequirement + 1;
          expect(contributorGear[camelCaseKey].canOwn(user)).to.eql(true);
        });

        it('canOwn returns true if user already owns the item', () => {
          user.items.gear.owned[key] = true;
          expect(contributorGear[camelCaseKey].canOwn(user)).to.eql(true);
        });

        it('canOwn returns true if user has previously owned the item', () => {
          user.items.gear.owned[key] = false;
          expect(contributorGear[camelCaseKey].canOwn(user)).to.eql(true);
        });

        it('canOwn returns false if user does not have tier requirement and did not previously own the item', () => {
          expect(contributorGear[camelCaseKey].canOwn(user)).to.eql(false);
        });
      });
    });

    context('hammer of bug smashing', () => {
      it('canOwn returns true if user has a critical flag on their contributor object', () => {
        user.contributor.critical = true;
        expect(contributorGear.weaponSpecialCritical.canOwn(user)).to.eql(true);
      });

      it('canOwn returns true if user already owns the item', () => {
        user.items.gear.owned.weapon_special_critical = true;
        expect(contributorGear.weaponSpecialCritical.canOwn(user)).to.eql(true);
      });

      it('canOwn returns true if user has previously owned the item', () => {
        user.items.gear.owned.weapon_special_critical = false;
        expect(contributorGear.weaponSpecialCritical.canOwn(user)).to.eql(true);
      });

      it('canOwn returns false if user does not have tier requirement and did not previously own the item', () => {
        expect(contributorGear.weaponSpecialCritical.canOwn(user)).to.eql(false);
      });
    });
  });
});
