/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* eslint-disable no-console */

import fs from 'fs';
import * as playwright from '../..';
import type { BrowserType } from '../client/browserType';
import type { LaunchServerOptions } from '../client/types';
import { createPlaywright, DispatcherConnection, Root, PlaywrightDispatcher } from '../server';
import { IpcTransport, PipeTransport } from '../protocol/transport';
import { PlaywrightServer } from '../remote/playwrightServer';
import { gracefullyCloseAll } from '../utils/processLauncher';

export function printApiJson() {
  // Note: this file is generated by build-playwright-driver.sh
  console.log(JSON.stringify(require('../../api.json')));
}

export function runDriver() {
  const dispatcherConnection = new DispatcherConnection();
  new Root(dispatcherConnection, async (rootScope, { sdkLanguage }) => {
    const playwright = createPlaywright(sdkLanguage);
    return new PlaywrightDispatcher(rootScope, playwright);
  });
  const transport = process.send ? new IpcTransport(process) : new PipeTransport(process.stdout, process.stdin);
  transport.onmessage = message => dispatcherConnection.dispatch(JSON.parse(message));
  dispatcherConnection.onmessage = message => transport.send(JSON.stringify(message));
  transport.onclose = async () => {
    // Drop any messages during shutdown on the floor.
    dispatcherConnection.onmessage = () => {};
    // Force exit after 30 seconds.
    setTimeout(() => process.exit(0), 30000);
    // Meanwhile, try to gracefully close all browsers.
    await gracefullyCloseAll();
    process.exit(0);
  };
}

export async function runServer(port: number | undefined, path: string = '/', maxClients: number = Infinity, enableSocksProxy: boolean = true) {
  const server = await PlaywrightServer.startDefault({ path, maxClients, enableSocksProxy });
  const wsEndpoint = await server.listen(port);
  process.on('exit', () => server.close().catch(console.error));
  console.log('Listening on ' + wsEndpoint);  // eslint-disable-line no-console
}

export async function launchBrowserServer(browserName: string, configFile?: string) {
  let options: LaunchServerOptions = {};
  if (configFile)
    options = JSON.parse(fs.readFileSync(configFile).toString());
  const browserType = (playwright as any)[browserName] as BrowserType;
  const server = await browserType.launchServer(options);
  console.log(server.wsEndpoint());
}
