// Copyright (c) 2011 The LevelDB Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file. See the AUTHORS file for names of contributors.

#ifndef STORAGE_LEVELDB_INCLUDE_OPTIONS_H_
#define STORAGE_LEVELDB_INCLUDE_OPTIONS_H_

#include <stddef.h>
#include <stdint.h>
#include <string>
#include <memory>

namespace leveldb {

class Cache;
class Comparator;
class Env;
class ExpiryModule;
class FilterPolicy;
class Logger;
class Snapshot;
namespace log
{
    class Writer;
}  // namespace log

// DB contents are stored in a set of blocks, each of which holds a
// sequence of key,value pairs.  Each block may be compressed before
// being stored in a file.  The following enum describes which
// compression method (if any) is used to compress a block.
enum CompressionType {
  // NOTE: do not change the values of existing entries, as these are
  // part of the persistent format on disk.
  kNoCompression     = 0x0,
  kSnappyCompression = 0x1,
  kLZ4Compression    = 0x2,
  kNoCompressionAutomated = 0x3
};

//  Originally located in db/dbformat.h.  Now available publically.
// Value types encoded as the last component of internal keys.
// DO NOT CHANGE THESE ENUM VALUES: they are embedded in the on-disk
// data structures.
enum ValueType {
  kTypeDeletion = 0x0,
  kTypeValue = 0x1,
  kTypeValueWriteTime = 0x2,
  kTypeValueExplicitExpiry = 0x3
};

//  Originally located in db/dbformat.h
typedef uint64_t SequenceNumber;
typedef uint64_t ExpiryTime;

};  // namespace leveldb

//
// must follow ValueType declaration
#include "leveldb/expiry.h"

namespace leveldb {

// Options to control the behavior of a database (passed to DB::Open)
struct Options {
  // -------------------
  // Parameters that affect behavior

  // Comparator used to define the order of keys in the table.
  // Default: a comparator that uses lexicographic byte-wise ordering
  //
  // REQUIRES: The client must ensure that the comparator supplied
  // here has the same name and orders keys *exactly* the same as the
  // comparator provided to previous open calls on the same DB.
  const Comparator* comparator;

  // If true, the database will be created if it is missing.
  // Default: false
  bool create_if_missing;

  // If true, an error is raised if the database already exists.
  // Default: false
  bool error_if_exists;

  // If true, the implementation will do aggressive checking of the
  // data it is processing and will stop early if it detects any
  // errors.  This may have unforeseen ramifications: for example, a
  // corruption of one DB entry may cause a large number of entries to
  // become unreadable or for the entire DB to become unopenable.
  // Default: false
  bool paranoid_checks;

  // Riak specific: this variable replaces paranoid_checks at one
  // one place in the code.  This variable alone controls whether or not
  // compaction read operations check CRC values.  Riak needs
  // the compaction CRC check, but not other paranoid_checks ... so
  // this independent control.
  // Default: true
  bool verify_compactions;

  // Use the specified object to interact with the environment,
  // e.g. to read/write files, schedule background work, etc.
  // Default: Env::Default()
  Env* env;

  // Any internal progress/error information generated by the db will
  // be written to info_log if it is non-NULL, or to a file stored
  // in the same directory as the DB contents if info_log is NULL.
  // Default: NULL
  Logger* info_log;

  // -------------------
  // Parameters that affect performance

  // Amount of data to build up in memory (backed by an unsorted log
  // on disk) before converting to a sorted on-disk file.
  //
  // Larger values increase performance, especially during bulk loads.
  // Up to two write buffers may be held in memory at the same time,
  // so you may wish to adjust this parameter to control memory usage.
  // Also, a larger write buffer will result in a longer recovery time
  // the next time the database is opened.
  //
  // Default: 4MB
  size_t write_buffer_size;

  // Number of open files that can be used by the DB.  You may need to
  // increase this if your database has a large working set (budget
  // one open file per 2MB of working set).
  // RIAK: NO LONGER USED
  // Default: 1000
  int max_open_files;

  // Control over blocks (user data is stored in a set of blocks, and
  // a block is the unit of reading from disk).

  // If non-NULL, use the specified cache for blocks.
  // If NULL, leveldb will automatically create and use an 8MB internal cache.
  // Default: NULL
  Cache* block_cache;

  // Approximate size of user data packed per block.  Note that the
  // block size specified here corresponds to uncompressed data.  The
  // actual size of the unit read from disk may be smaller if
  // compression is enabled.  This parameter can be changed dynamically.
  //
  // Default: 4K
  size_t block_size;

  // Riak specific:  non-zero value activates code to automatically
  // increase block_size as needed to ensure maximum number of files
  // are available in the file cache.  The value indicates how many
  // incremental increases to use between the original block_size
  // and largest, reasonable block_size.
  //
  // Default: 16
  int block_size_steps;

  // Number of keys between restart points for delta encoding of keys.
  // This parameter can be changed dynamically.  Most clients should
  // leave this parameter alone.
  //
  // Default: 16
  int block_restart_interval;

  // Compress blocks using the specified compression algorithm.  This
  // parameter can be changed dynamically.
  //
  // Default: kSnappyCompression, which gives lightweight but fast
  // compression.
  //
  // Typical speeds of kSnappyCompression on an Intel(R) Core(TM)2 2.4GHz:
  //    ~200-500MB/s compression
  //    ~400-800MB/s decompression
  // Note that these speeds are significantly faster than most
  // persistent storage speeds, and therefore it is typically never
  // worth switching to kNoCompression.  Even if the input data is
  // incompressible, the kSnappyCompression implementation will
  // efficiently detect that and will switch to uncompressed mode.
  CompressionType compression;

  // If non-NULL, use the specified filter policy to reduce disk reads.
  // Many applications will benefit from passing the result of
  // NewBloomFilterPolicy() here.
  //
  // Default: NULL
  const FilterPolicy* filter_policy;

  // Riak specific flag used to indicate when database is open
  // as part of a Repair operation.  Default is false
  bool is_repair;

  // Riak specific flag to mark Riak internal database versus
  //  user database.  (User database gets larger cache resources.)
  bool is_internal_db;

  // Riak replacement for max_open_files and block_cache.  This is
  //  TOTAL memory to be used by leveldb across ALL DATABASES.
  //  Most recent value seen upon database open, wins.  Zero for default.
  uint64_t total_leveldb_mem;

  // Riak specific option specifying block cache space that cannot
  //  be released for page cache use.  The space may still be
  //  released for file cache.
  uint64_t block_cache_threshold;

  // Riak option to override most memory modeling and create
  //  smaller memory footprint for developers.  Helps when
  //  running large number of databases and multiple VMs. Do
  //  NOT use this option if making performance measurements.
  // Default: false
  bool limited_developer_mem;

  // The size of each MMAped file, choose 0 for the default (20M)
  uint64_t mmap_size;

  // Riak option to adjust aggressive delete behavior.
  //  - zero disables aggressive delete
  //  - positive value indicates how many deletes must exist
  //     in a file for it to be compacted due to deletes
  uint64_t delete_threshold;

  // Riak specific flag used to indicate when fadvise() management
  // should default to WILLNEED instead of DONTNEED.  Default is false
  bool fadvise_willneed;

  // *****
  // Riak specific options for establishing two tiers of disk arrays.
  // All three tier options must be valid for the option to activate.
  // When active, leveldb directories are constructed using either
  // the fast or slow prefix followed by the database name given
  // in the DB::Open call.  (a synonym for "prefix" is "mount")
  // *****

  // Riak specific option setting the level number at which the
  // "tiered_slow_prefix" should be used.  Default is zero which
  // disables the option.  Valid values are 1 to 6.  3 or 4 recommended.
  unsigned tiered_slow_level;

  // Riak specific option with the path prefix used for "fast" disk
  // array.  levels 0 to tiered_slow_level-1 use this path prefix
  std::string tiered_fast_prefix;

  // Riak specific option with the path prefix used for "slow" disk
  // array.  levels tiered_slow_level through 6 use this path prefix
  std::string tiered_slow_prefix;

  // Riak specific option that writes a list of open table files
  // to disk on close then automatically opens same files again
  // upon restart.
  bool cache_object_warming;

  // Riak specific object that defines expiry policy for data
  // written to leveldb.
  ExpiryPtr_t expiry_module;

  // Create an Options object with default values for all fields.
  Options();

  void Dump(Logger * log) const;

private:

};

// Options that control read operations
struct ReadOptions {
  // If true, all data read from underlying storage will be
  // verified against corresponding checksums.
  // Default: false
  bool verify_checksums;

  // Should the data read for this iteration be cached in memory?
  // Callers may wish to set this field to false for bulk scans.
  // Default: true
  bool fill_cache;

  // If "snapshot" is non-NULL, read as of the supplied snapshot
  // (which must belong to the DB that is being read and which must
  // not have been released).  If "snapshot" is NULL, use an impliicit
  // snapshot of the state at the beginning of this read operation.
  // Default: NULL
  const Snapshot* snapshot;

  // Riak specific flag, currently used within Erlang adaptor
  //  to enable automatic delete and new of fresh snapshot
  //  and database iterator objects for long running iterations
  //  (only supports iterator NEXT operations).
  // Default: false
  bool iterator_refresh;

  ReadOptions()
  : verify_checksums(true),
      fill_cache(true),
      snapshot(NULL),
      iterator_refresh(false),
      is_compaction(false),
      env(NULL),
      info_log(NULL)
  {
  }


  // accessors to the private data
  bool IsCompaction() const {return(is_compaction);};

  Logger * GetInfoLog() const {return(info_log);};

  const std::string & GetDBName() const {return(dbname);};

  Env * GetEnv() const {return(env);};

  // The items below are internal options, not for external manipulation.
  //  They are populated by VersionSet::MakeInputIterator only during compaction operations
private:
  friend class VersionSet;

  // true when used on background compaction
  bool is_compaction;

  // Database name for potential creation of bad blocks file
  std::string dbname;

  // Needed for file operations if creating bad blocks file
  Env * env;

  // Open log file for error notifications
  // Only valid when is_compation==true
  Logger* info_log;

};

// Options that control write operations
struct WriteOptions {
  // If true, the write will be flushed from the operating system
  // buffer cache (by calling WritableFile::Sync()) before the write
  // is considered complete.  If this flag is true, writes will be
  // slower.
  //
  // If this flag is false, and the machine crashes, some recent
  // writes may be lost.  Note that if it is just the process that
  // crashes (i.e., the machine does not reboot), no writes will be
  // lost even if sync==false.
  //
  // In other words, a DB write with sync==false has similar
  // crash semantics as the "write()" system call.  A DB write
  // with sync==true has similar crash semantics to a "write()"
  // system call followed by "fsync()".
  //
  // Default: false
  bool sync;

  WriteOptions()
      : sync(false) {
  }
};


// Riak specific object that can return key metadata
//  during get or iterate operation
struct KeyMetaData
{
    ValueType m_Type;          // see above
    SequenceNumber m_Sequence; // output only, leveldb internal
    ExpiryTime m_Expiry;       // microseconds since Epoch, UTC

    KeyMetaData()
    : m_Type(kTypeValue), m_Sequence(0), m_Expiry(0)
    {};
};  // struct KeyMetaData

}  // namespace leveldb

#endif  // STORAGE_LEVELDB_INCLUDE_OPTIONS_H_
