// Copyright (c) 2012 The LevelDB Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file. See the AUTHORS file for names of contributors.

#include <memory>
#include "leveldb/filter_policy.h"

#include "db/dbformat.h"
#include "leveldb/slice.h"
#include "util/hash.h"
#include "util/murmurhash.h"

namespace leveldb {

static unsigned Bytes2Prime(unsigned Bytes);
static unsigned Bits2PrimeNBytes(unsigned Bits, unsigned & BytesOut);


namespace {
static uint32_t BloomHash0(const Slice& key) {
  return Hash(key.data(), key.size(), 0xbc9f1d34);
}

static uint32_t BloomHash1(const Slice& key) {
    return((uint32_t)MurmurHash(key.data(), key.size(), 0x5bd1e995));
}

class BloomFilterPolicy2 : public FilterPolicy {
 private:
  size_t bits_per_key_;
  size_t k_;

 public:
  explicit BloomFilterPolicy2(int bits_per_key)
      : bits_per_key_(bits_per_key) {
    // We intentionally round down to reduce probing cost a little bit
    k_ = static_cast<size_t>(bits_per_key * 0.69);  // 0.69 =~ ln(2)
    if (k_ < 1) k_ = 1;
    if (k_ > 30) k_ = 30;
  }

  virtual const char* Name() const {
    return "leveldb.BuiltinBloomFilter2";
  }

  virtual void CreateFilter(const Slice* keys, int n, std::string* dst) const
  {
    unsigned bytes;

    // Compute bloom filter size (in both bits and bytes)
    size_t bits = n * bits_per_key_;

    // For small n, we can see a very high false positive rate.  Fix it
    // by enforcing a minimum bloom filter length.
    if (bits < 61) bits = 61;

    const unsigned prime=Bits2PrimeNBytes(bits, bytes);

    const size_t init_size = dst->size();
    dst->resize(init_size + bytes, 0);
    dst->push_back(static_cast<char>(k_));  // Remember # of probes in filter
    char* array = &(*dst)[init_size];
    for (size_t i = 0; i < (size_t)n; i++) {
      // Use double-hashing to generate a sequence of hash values.
      // See analysis in [Kirsch,Mitzenmacher 2006].
      uint32_t h = BloomHash0(keys[i]);
      uint32_t h2= BloomHash1(keys[i]);
      const uint32_t delta = (h >> 17) | (h << 15);  // Rotate right 17 bits
      for (size_t j = 0; j < k_; j++) {
          const uint32_t bitpos = (h + ((j+1)*h2)) % prime;
        array[bitpos/8] |= (1 << (bitpos % 8));
        h += delta;
      }
    }
  }

  virtual bool KeyMayMatch(const Slice& key, const Slice& bloom_filter) const {
    const size_t len = bloom_filter.size();
    if (len < 2) return false;

    const char* array = bloom_filter.data();
    const unsigned prime=Bytes2Prime(len-1);

    // Use the encoded k so that we can read filters generated by
    // bloom filters created using different parameters.
    const size_t k = array[len-1];
    if (k > 30) {
      // Reserved for potentially new encodings for short bloom filters.
      // Consider it a match.
      return true;
    }

    uint32_t h = BloomHash0(key);
    uint32_t h2= BloomHash1(key);
    const uint32_t delta = (h >> 17) | (h << 15);  // Rotate right 17 bits
    for (size_t j = 0; j < k; j++) {
      const uint32_t bitpos = (h + ((j+1)*h2)) % prime;
      if ((array[bitpos/8] & (1 << (bitpos % 8))) == 0) return false;
      h += delta;
    }
    return true;
  }
};
}

const FilterPolicy* NewBloomFilterPolicy2(int bits_per_key) {
  return new BloomFilterPolicy2(bits_per_key);
}


// container to hold one bloom filter and auto destruct
struct BloomInventoryItem2
{
    std::auto_ptr<const FilterPolicy> m_Item;

    BloomInventoryItem2()
    {
        m_Item.reset(new InternalFilterPolicy2(NewBloomFilterPolicy2(16)));
        FilterInventory::AddFilterToInventory(m_Item.get());
    };
};  // struct BloomInventoryItem2

// bloom filter for reading, created on start-up
static BloomInventoryItem2 lBloom2Item;



// sparse table of primes where index to array is count
//  of bytes to contain the prime number of bits
// The prime table only helps where key count is roughly 6,250
//  or less.  It adds accuracy to smaller populations.
static unsigned ByteSizePrimes[]=
{
 0, 7, 13, 23, 31, 37, 47, 53, 61, 71,
 79, 83, 89, 103, 109, 113, 127, 131, 139, 151,
 157, 167, 173, 181, 191, 199, 199, 211, 223, 229,
 239, 241, 251, 263, 271, 277, 283, 293, 293, 311,
 317, 317, 331, 337, 349, 359, 367, 373, 383, 389,
 397, 401, 409, 421, 431, 439, 443, 449, 463, 467,
 479, 487, 491, 503, 509, 509, 523, 523, 541, 547,
 557, 563, 571, 577, 587, 599, 607, 613, 619, 631,
 631, 647, 653, 661, 661, 677, 683, 691, 701, 709,
 719, 727, 733, 743, 751, 757, 761, 773, 773, 787,
 797, 797, 811, 823, 829, 839, 839, 853, 863, 863,
 877, 887, 887, 887, 911, 919, 919, 929, 941, 947,
 953, 967, 971, 983, 991, 997, 997, 1013, 1021, 1031,
 1039, 1039, 1051, 1063, 1069, 1069, 1087, 1093, 1103, 1109,
 1117, 1123, 1129, 1129, 1151, 1153, 1163, 1171, 1181, 1187,
 1193, 1201, 1213, 1223, 1231, 1237, 1237, 1249, 1259, 1259,
 1279, 1283, 1291, 1303, 1307, 1319, 1327, 1327, 1327, 1327,
 1327, 1367, 1373, 1381, 1381, 1399, 1399, 1409, 1423, 1429,
 1439, 1447, 1453, 1459, 1471, 1471, 1487, 1493, 1499, 1511,
 1511, 1523, 1531, 1543, 1549, 1559, 1567, 1571, 1583, 1583,
 1597, 1607, 1613, 1621, 1627, 1637, 1637, 1637, 1663, 1669,
 1669, 1669, 1693, 1699, 1709, 1709, 1723, 1733, 1741, 1747,
 1759, 1759, 1759, 1783, 1789, 1789, 1801, 1811, 1823, 1831,
 1831, 1847, 1847, 1861, 1871, 1879, 1879, 1889, 1901, 1907,
 1913, 1913, 1933, 1933, 1951, 1951, 1951, 1973, 1979, 1987,
 1999, 2003, 2011, 2017, 2029, 2039, 2039, 2053, 2063, 2069,
 2069, 2087, 2089, 2099, 2111, 2113, 2113, 2131, 2143, 2143,
 2153, 2161, 2161, 2179, 2179, 2179, 2207, 2213, 2221, 2221,
 2239, 2243, 2251, 2251, 2269, 2273, 2287, 2293, 2297, 2311,
 2311, 2311, 2333, 2341, 2351, 2357, 2357, 2371, 2383, 2389,
 2399, 2399, 2411, 2423, 2423, 2437, 2447, 2447, 2459, 2467,
 2477, 2477, 2477, 2503, 2503, 2503, 2521, 2531, 2543, 2551,
 2557, 2557, 2557, 2579, 2591, 2593, 2593, 2609, 2621, 2621,
 2633, 2647, 2647, 2663, 2671, 2677, 2687, 2693, 2699, 2711,
 2719, 2719, 2731, 2741, 2749, 2753, 2767, 2767, 2777, 2791,
 2797, 2803, 2803, 2819, 2819, 2837, 2843, 2851, 2861, 2861,
 2879, 2887, 2887, 2903, 2909, 2917, 2927, 2927, 2939, 2939,
 2957, 2963, 2971, 2971, 2971, 2999, 3001, 3011, 3023, 3023,
 3037, 3041, 3049, 3061, 3067, 3079, 3083, 3089, 3089, 3109,
 3119, 3121, 3121, 3137, 3137, 3137, 3167, 3169, 3181, 3191,
 3191, 3203, 3209, 3221, 3229, 3229, 3229, 3253, 3259, 3271,
 3271, 3271, 3271, 3301, 3307, 3319, 3323, 3331, 3343, 3347,
 3359, 3361, 3373, 3373, 3391, 3391, 3407, 3413, 3413, 3413,
 3433, 3433, 3449, 3463, 3469, 3469, 3469, 3491, 3499, 3511,
 3517, 3527, 3533, 3541, 3547, 3559, 3559, 3571, 3583, 3583,
 3593, 3607, 3613, 3623, 3631, 3637, 3643, 3643, 3659, 3671,
 3677, 3677, 3691, 3701, 3709, 3719, 3727, 3733, 3739, 3739,
 3739, 3767, 3769, 3779, 3779, 3797, 3803, 3803, 3823, 3823,
 3833, 3847, 3853, 3863, 3863, 3877, 3881, 3889, 3889, 3911,
 3919, 3923, 3931, 3943, 3947, 3947, 3967, 3967, 3967, 3989,
 3989, 4007, 4013, 4021, 4027, 4027, 4027, 4051, 4057, 4057,
 4079, 4079, 4093, 4099, 4111, 4111, 4127, 4133, 4139, 4139,
 4159, 4159, 4159, 4177, 4177, 4177, 4201, 4211, 4219, 4231,
 4231, 4243, 4253, 4261, 4271, 4273, 4283, 4289, 4297, 4297,
 4297, 4327, 4327, 4339, 4349, 4357, 4363, 4373, 4373, 4391,
 4397, 4397, 4409, 4423, 4423, 4423, 4447, 4451, 4463, 4463,
 4463, 4483, 4493, 4493, 4507, 4519, 4523, 4523, 4523, 4549,
 4549, 4567, 4567, 4583, 4591, 4597, 4603, 4603, 4621, 4621,
 4639, 4643, 4651, 4663, 4663, 4679, 4679, 4691, 4703, 4703,
 4703, 4723, 4733, 4733, 4751, 4759, 4759, 4759, 4783, 4789,
 4799, 4801, 4813, 4817, 4831, 4831, 4831, 4831, 4861, 4871,
 4877, 4877, 4889, 4903, 4909, 4919, 4919, 4933, 4943, 4951,
 4957, 4967, 4973, 4973, 4987, 4999, 5003, 5011, 5023, 5023,
 5039, 5039, 5051, 5059, 5059, 5077, 5087, 5087, 5101, 5107,
 5119, 5119, 5119, 5119, 5147, 5153, 5167, 5171, 5179, 5189,
 5197, 5197, 5209, 5209, 5231, 5237, 5237, 5237, 5261, 5261,
 5279, 5281, 5281, 5303, 5309, 5309, 5323, 5333, 5333, 5351,
 5351, 5351, 5351, 5381, 5387, 5399, 5407, 5413, 5419, 5431,
 5437, 5443, 5449, 5449, 5471, 5479, 5483, 5483, 5503, 5507,
 5519, 5527, 5531, 5531, 5531, 5557, 5563, 5573, 5581, 5591,
 5591, 5591, 5591, 5623, 5623, 5639, 5647, 5653, 5659, 5669,
 5669, 5683, 5693, 5701, 5711, 5717, 5717, 5717, 5743, 5749,
 5749, 5749, 5749, 5783, 5791, 5791, 5807, 5813, 5821, 5827,
 5839, 5843, 5851, 5861, 5869, 5879, 5881, 5881, 5903, 5903,
 5903, 5927, 5927, 5939, 5939, 5953, 5953, 5953, 5981, 5987,
 5987, 6007, 6011, 6011, 6029, 6037, 6047, 6053, 6053, 6067,
 6079, 6079, 6091, 6101, 6101, 6113, 6121, 6133, 6143, 6151,
 6151, 6163, 6173, 6173, 6173, 6199, 6203, 6211, 6221, 6229,
 6229, 6247, 6247, 6263, 6271, 6277, 6287, 6287, 6301, 6311,
 6317, 6323, 6329, 6343, 6343, 6359, 6367, 6373, 6379, 6389,
 6397, 6397, 6397, 6421, 6427, 6427, 6427, 6451, 6451, 6469,
 6473, 6481, 6491, 6491, 6491, 6491, 6521, 6529, 6529, 6551,
 6553, 6563, 6571, 6581, 6581, 6599, 6607, 6607, 6619, 6619,
 6637, 6637, 6653, 6661, 6661, 6679, 6679, 6691, 6703, 6709,
 6719, 6719, 6733, 6737, 6737, 6737, 6763, 6763, 6781, 6791,
 6793, 6803, 6803, 6823, 6829, 6833, 6841, 6841, 6863, 6871,
 6871, 6883, 6883, 6899, 6911, 6917, 6917, 6917, 6917, 6949,
 6959, 6967, 6971, 6983, 6991, 6997, 7001, 7013, 7019, 7027,
 7039, 7043, 7043, 7057, 7069, 7079, 7079, 7079, 7103, 7109,
 7109, 7127, 7129, 7129, 7151, 7159, 7159, 7159, 7177, 7187,
 7193, 7207, 7213, 7219, 7229, 7237, 7247, 7253, 7253, 7253,
 7253, 7283, 7283, 7297, 7309, 7309, 7321, 7333, 7333, 7351,
 7351, 7351, 7369, 7369, 7369, 7393, 7393, 7411, 7417, 7417,
 7433, 7433, 7451, 7459, 7459, 7477, 7487, 7489, 7499, 7507,
 7517, 7523, 7529, 7541, 7549, 7559, 7561, 7573, 7583, 7591,
 7591, 7607, 7607, 7621, 7621, 7639, 7643, 7649, 7649, 7669,
 7673, 7687, 7691, 7703, 7703, 7717, 7727, 7727, 7741, 7741,
 7759, 7759, 7759, 7759, 7789, 7793, 7793, 7793, 7823, 7829,
 7829, 7841, 7853, 7853, 7867, 7879, 7883, 7883, 7901, 7907,
 7919, 7927, 7933, 7937, 7951, 7951, 7963, 7963, 7963, 7963,
 7993, 7993, 8011, 8017, 8017, 8039, 8039, 8053, 8059, 8069,
 8069, 8087, 8093, 8101, 8111, 8117, 8123, 8123, 8123, 8147,
 8147, 8167, 8171, 8179, 8191, 8191, 8191, 8209, 8221, 8231,
 8237, 8243, 8243, 8263, 8269, 8273, 8287, 8293, 8297, 8311,
 8317, 8317, 8329, 8329, 8329, 8353, 8363, 8369, 8377, 8389,
 8389, 8389, 8389, 8423, 8431, 8431, 8447, 8447, 8461, 8467,
 8467, 8467, 8467, 8501, 8501, 8513, 8527, 8527, 8543, 8543,
 8543, 8563, 8573, 8581, 8581, 8599, 8599, 8609, 8623, 8629,
 8629, 8647, 8647, 8663, 8669, 8677, 8681, 8693, 8699, 8707,
 8719, 8719, 8731, 8741, 8747, 8753, 8761, 8761, 8783, 8783,
 8783, 8807, 8807, 8821, 8831, 8839, 8839, 8849, 8863, 8867,
 8867, 8887, 8893, 8893, 8893, 8893, 8923, 8933, 8941, 8951,
 8951, 8963, 8971, 8971, 8971, 8999, 9007, 9013, 9013, 9029,
 9029, 9043, 9049, 9059, 9067, 9067, 9067, 9091, 9103, 9109,
 9109, 9127, 9133, 9137, 9151, 9157, 9161, 9173, 9181, 9187,
 9199, 9203, 9209, 9221, 9227, 9239, 9241, 9241, 9257, 9257,
 9277, 9283, 9293, 9293, 9311, 9319, 9323, 9323, 9343, 9349,
 9349, 9349, 9371, 9377, 9391, 9397, 9403, 9413, 9421, 9431,
 9439, 9439, 9439, 9463, 9467, 9479, 9479, 9491, 9497, 9511,
 9511, 9521, 9533, 9539, 9551, 9551, 9551, 9551, 9551, 9587,
 9587, 9601, 9613, 9623, 9631, 9631, 9643, 9649, 9661, 9661,
 9679, 9679, 9689, 9697, 9697, 9719, 9721, 9733, 9743, 9749,
 9749, 9767, 9769, 9781, 9791, 9791, 9803, 9811, 9817, 9829,
 9839, 9839, 9851, 9859, 9871, 9871, 9887, 9887, 9901, 9907,
 9907, 9923, 9931, 9941, 9949, 9949, 9967, 9973, 9973, 9973,
 9973, 10007, 10009, 10009, 10009, 10039, 10039, 10039, 10061, 10069,
 10079, 10079, 10093, 10103, 10111, 10111, 10111, 10133, 10141, 10151,
 10159, 10163, 10169, 10181, 10181, 10193, 10193, 10211, 10223, 10223,
 10223, 10247, 10253, 10259, 10271, 10273, 10273, 10289, 10303, 10303,
 10313, 10321, 10333, 10343, 10343, 10357, 10357, 10369, 10369, 10391,
 10399, 10399, 10399, 10399, 10429, 10433, 10433, 10453, 10463, 10463,
 10477, 10487, 10487, 10501, 10501, 10513, 10513, 10531, 10531, 10531,
 10559, 10567, 10567, 10567, 10589, 10597, 10607, 10613, 10613, 10631,
 10639, 10639, 10651, 10663, 10667, 10667, 10687, 10691, 10691, 10711,
 10711, 10723, 10733, 10739, 10739, 10753, 10753, 10771, 10781, 10789,
 10799, 10799, 10799, 10799, 10831, 10837, 10847, 10853, 10861, 10867,
 10867, 10883, 10891, 10903, 10909, 10909, 10909, 10909, 10939, 10949,
 10957, 10957, 10973, 10979, 10987, 10993, 11003, 11003, 11003, 11027,
 11027, 11047, 11047, 11059, 11071, 11071, 11087, 11093, 11093, 11093,
 11119, 11119, 11131, 11131, 11149, 11159, 11161, 11173, 11177, 11177,
 11197, 11197, 11213, 11213, 11213, 11239, 11243, 11251, 11261, 11261,
 11279, 11287, 11287, 11299, 11311, 11317, 11321, 11329, 11329, 11351,
 11353, 11353, 11369, 11383, 11383, 11399, 11399, 11411, 11423, 11423,
 11437, 11447, 11447, 11447, 11471, 11471, 11483, 11491, 11503, 11503,
 11519, 11527, 11527, 11527, 11551, 11551, 11551, 11551, 11579, 11587,
 11597, 11597, 11597, 11621, 11621, 11633, 11633, 11633, 11657, 11657,
 11677, 11681, 11689, 11701, 11701, 11719, 11719, 11731, 11743, 11743,
 11743, 11743, 11743, 11783, 11789, 11789, 11807, 11813, 11821, 11831,
 11839, 11839, 11839, 11863, 11867, 11867, 11887, 11887, 11903, 11909,
 11909, 11927, 11933, 11941, 11941, 11959, 11959, 11971, 11981, 11987,
 11987, 12007, 12011, 12011, 12011, 12037, 12043, 12049, 12049, 12071,
 12073, 12073, 12073, 12101, 12109, 12119, 12119, 12119, 12143, 12149,
 12157, 12163, 12163, 12163, 12163, 12197, 12203, 12211, 12211, 12227,
 12239, 12241, 12253, 12263, 12269, 12277, 12281, 12289, 12301, 12301,
 12301, 12323, 12329, 12343, 12347, 12347, 12347, 12373, 12379, 12391,
 12391, 12401, 12413, 12421, 12421, 12437, 12437, 12451, 12457, 12457,
 12479, 12487, 12491, 12503, 12511, 12517, 12527, 12527, 12541, 12547,
 12553, 12553, 12569, 12583, 12589, 12589, 12601, 12613, 12619, 12619,
 12637, 12647, 12653, 12659, 12671, 12671, 12671, 12689, 12703, 12703,
 12713, 12721, 12721, 12743, 12743, 12757, 12763, 12763, 12781, 12791,
 12799, 12799, 12809, 12823, 12829, 12829, 12841, 12853, 12853, 12853,
 12853, 12853, 12893, 12899, 12911, 12919, 12923, 12923, 12941, 12941,
 12959, 12967, 12973, 12983, 12983, 12983, 13007, 13009, 13009, 13009,
 13037, 13043, 13049, 13063, 13063, 13063, 13063, 13093, 13103, 13109,
 13109, 13127, 13127, 13127, 13151, 13159, 13163, 13171, 13183, 13187,
 13187, 13187, 13187, 13219, 13229, 13229, 13241, 13249, 13259, 13267,
 13267, 13267, 13291, 13297, 13309, 13313, 13327, 13331, 13339, 13339,
 13339, 13367, 13367, 13381, 13381, 13399, 13399, 13411, 13421, 13421,
 13421, 13441, 13451, 13463, 13469, 13477, 13487, 13487, 13499, 13499,
 13513, 13523, 13523, 13537, 13537, 13553, 13567, 13567, 13577, 13591,
 13597, 13597, 13613, 13619, 13627, 13633, 13633, 13649, 13649, 13669,
 13679, 13687, 13693, 13697, 13711, 13711, 13723, 13729, 13729, 13751,
 13759, 13763, 13763, 13781, 13789, 13799, 13807, 13807, 13807, 13831,
 13831, 13841, 13841, 13859, 13859, 13879, 13883, 13883, 13903, 13907,
 13913, 13921, 13933, 13933, 13933, 13933, 13967, 13967, 13967, 13967,
 13999, 13999, 14011, 14011, 14029, 14033, 14033, 14051, 14057, 14071,
 14071, 14087, 14087, 14087, 14107, 14107, 14107, 14107, 14143, 14149,
 14159, 14159, 14173, 14177, 14177, 14197, 14207, 14207, 14221, 14221,
 14221, 14243, 14251, 14251, 14251, 14251, 14281, 14293, 14303, 14303,
 14303, 14327, 14327, 14341, 14347, 14347, 14347, 14369, 14369, 14389,
 14389, 14407, 14411, 14423, 14431, 14437, 14447, 14449, 14461, 14461,
 14479, 14479, 14489, 14503, 14503, 14519, 14519, 14533, 14543, 14551,
 14557, 14563, 14563, 14563, 14591, 14593, 14593, 14593, 14621, 14629,
 14639, 14639, 14653, 14657, 14669, 14669, 14683, 14683, 14699, 14699,
 14717, 14723, 14731, 14741, 14747, 14759, 14767, 14771, 14783, 14783,
 14797, 14797, 14813, 14821, 14831, 14831, 14843, 14851, 14851, 14869,
 14879, 14887, 14891, 14897, 14897, 14897, 14923, 14929, 14939, 14951,
 14957, 14957, 14969, 14983, 14983, 14983, 14983, 15013, 15017, 15031,
 15031, 15031, 15053, 15061, 15061, 15077, 15083, 15091, 15101, 15107,
 15107, 15121, 15131, 15139, 15149, 15149, 15161, 15173, 15173, 15187,
 15199, 15199, 15199, 15217, 15227, 15233, 15241, 15241, 15263, 15271,
 15277, 15287, 15289, 15299, 15307, 15319, 15319, 15331, 15331, 15349,
 15359, 15361, 15373, 15383, 15391, 15391, 15401, 15413, 15413, 15427,
 15439, 15443, 15451, 15461, 15467, 15473, 15473, 15493, 15497, 15511,
 15511, 15527, 15527, 15541, 15551, 15559, 15559, 15569, 15583, 15583,
 15583, 15607, 15607, 15619, 15629, 15629, 15647, 15649, 15661, 15671,
 15679, 15683, 15683, 15683, 15683, 15683, 15727, 15733, 15739, 15749,
 15749, 15767, 15773, 15773, 15791, 15797, 15803, 15809, 15823, 15823,
 15823, 15823, 15823, 15859, 15859, 15877, 15887, 15889, 15901, 15907,
 15919, 15923, 15923, 15937, 15937, 15959, 15959, 15973, 15973, 15991,
 15991, 16007, 16007, 16007, 16007, 16033, 16033, 16033, 16063, 16069,
 16073, 16087, 16091, 16103, 16111, 16111, 16127, 16127, 16141, 16141,
 16141, 16141, 16141, 16183, 16189, 16193, 16193, 16193, 16223, 16231,
 16231, 16231, 16253, 16253, 16267, 16273, 16273, 16273, 16301, 16301,
 16319, 16319, 16333, 16339, 16349, 16349, 16363, 16369, 16381, 16381,
 16381, 16381, 16411, 16421, 16427, 16433, 16447, 16453, 16453, 16453,
 16477, 16487, 16493, 16493, 16493, 16519, 16519, 16529, 16529, 16547,
 16553, 16567, 16573, 16573, 16573, 16573, 16607, 16607, 16619, 16631,
 16633, 16633, 16651, 16661, 16661, 16673, 16673, 16693, 16703, 16703,
 16703, 16703, 16729, 16741, 16747, 16759, 16763, 16763, 16763, 16787,
 16787, 16787, 16811, 16823, 16831, 16831, 16843, 16843, 16843, 16871,
 16879, 16883, 16889, 16903, 16903, 16903, 16927, 16931, 16943, 16943,
 16943, 16963, 16963, 16981, 16987, 16993, 16993, 17011, 17021, 17029,
 17033, 17047, 17053, 17053, 17053, 17077, 17077, 17093, 17099, 17107,
 17117, 17123, 17123, 17137, 17137, 17159, 17167, 17167, 17183, 17191,
 17191, 17207, 17209, 17209, 17231, 17239, 17239, 17239, 17257, 17257,
 17257, 17257, 17293, 17299, 17299, 17317, 17327, 17333, 17341, 17351,
 17359, 17359, 17359, 17383, 17389, 17393, 17401, 17401, 17419, 17431,
 17431, 17443, 17449, 17449, 17471, 17477, 17483, 17491, 17497, 17509,
 17519, 17519, 17519, 17539, 17551, 17551, 17551, 17573, 17581, 17581,
 17599, 17599, 17609, 17623, 17627, 17627, 17627, 17627, 17659, 17669,
 17669, 17683, 17683, 17683, 17707, 17713, 17713, 17729, 17737, 17749,
 17749, 17761, 17761, 17783, 17791, 17791, 17807, 17807, 17807, 17827,
 17839, 17839, 17851, 17863, 17863, 17863, 17881, 17891, 17903, 17911,
 17911, 17923, 17929, 17939, 17939, 17959, 17959, 17971, 17981, 17989,
 17989, 17989, 18013, 18013, 18013, 18013, 18047, 18049, 18061, 18061,
 18077, 18077, 18089, 18097, 18097, 18119, 18127, 18133, 18143, 18149,
 18149, 18149, 18169, 18181, 18191, 18199, 18199, 18211, 18223, 18229,
 18233, 18233, 18253, 18257, 18269, 18269, 18287, 18289, 18301, 18311,
 18313, 18313, 18329, 18341, 18341, 18353, 18367, 18371, 18379, 18379,
 18397, 18401, 18413, 18413, 18427, 18439, 18443, 18451, 18461, 18461,
 18461, 18481, 18493, 18503, 18503, 18517, 18523, 18523, 18541, 18541,
 18553, 18553, 18553, 18583, 18587, 18593, 18593, 18593, 18617, 18617,
 18637, 18637, 18637, 18661, 18671, 18679, 18679, 18691, 18701, 18701,
 18719, 18719, 18731, 18743, 18749, 18757, 18757, 18773, 18773, 18787,
 18797, 18803, 18803, 18803, 18803, 18839, 18839, 18839, 18859, 18869,
 18869, 18869, 18869, 18899, 18911, 18919, 18919, 18919, 18919, 18947,
 18959, 18959, 18973, 18979, 18979, 18979, 19001, 19013, 19013, 19031,
 19037, 19037, 19051, 19051, 19069, 19079, 19087, 19087, 19087, 19087,
 19087, 19121, 19121, 19141, 19141, 19157, 19163, 19163, 19183, 19183,
 19183, 19207, 19213, 19219, 19231, 19237, 19237, 19249, 19259, 19267,
 19273, 19273, 19289, 19301, 19309, 19319, 19319, 19333, 19333, 19333,
 19333, 19333, 19373, 19381, 19391, 19391, 19403, 19403, 19423, 19429,
 19433, 19447, 19447, 19463, 19471, 19477, 19483, 19489, 19501, 19507,
 19507, 19507, 19531, 19543, 19543, 19559, 19559, 19571, 19583, 19583,
 19597, 19603, 19609, 19609, 19609, 19609, 19609, 19609, 19661, 19661,
 19661, 19687, 19687, 19699, 19709, 19717, 19727, 19727, 19739, 19751,
 19759, 19763, 19763, 19777, 19777, 19793, 19801, 19813, 19819, 19819,
 19819, 19843, 19853, 19861, 19867, 19867, 19867, 19891, 19891, 19891,
 19919, 19927, 19927, 19937, 19949, 19949, 19963, 19973, 19979, 19991,
 19997, 19997, 20011, 20023, 20029, 20029, 20047, 20051, 20063, 20071,
 20071, 20071, 20089, 20101, 20107, 20117, 20123, 20129, 20143, 20149,
 20149, 20161, 20173, 20183, 20183, 20183, 20201, 20201, 20219, 20231,
 20233, 20233, 20249, 20261, 20269, 20269, 20287, 20287, 20297, 20297,
 20297, 20327, 20333, 20341, 20347, 20359, 20359, 20369, 20369, 20389,
 20399, 20407, 20411, 20411, 20431, 20431, 20443, 20443, 20443, 20443,
 20479, 20483, 20483, 20483, 20509, 20509, 20521, 20533, 20543, 20551,
 20551, 20563, 20563, 20563, 20563, 20599, 20599, 20611, 20611, 20627,
 20639, 20641, 20641, 20663, 20663, 20663, 20681, 20693, 20693, 20707,
 20719, 20719, 20731, 20743, 20749, 20759, 20759, 20773, 20773, 20789,
 20789, 20807, 20809, 20809, 20809, 20809, 20809, 20849, 20857, 20857,
 20879, 20887, 20887, 20903, 20903, 20903, 20921, 20929, 20939, 20947,
 20959, 20963, 20963, 20983, 20983, 20983, 21001, 21013, 21023, 21031,
 21031, 21031, 21031, 21061, 21067, 21067, 21067, 21089, 21101, 21107,
 21107, 21121, 21121, 21143, 21149, 21157, 21163, 21169, 21179, 21191,
 21193, 21193, 21211, 21221, 21227, 21227, 21247, 21247, 21247, 21269,
 21277, 21283, 21283, 21283, 21283, 21319, 21323, 21323, 21341, 21347,
 21347, 21347, 21347, 21383, 21391, 21397, 21407, 21407, 21419, 21419,
 21433, 21433, 21433, 21433, 21467, 21467, 21487, 21493, 21503, 21503,
 21517, 21523, 21529, 21529, 21529, 21559, 21563, 21569, 21577, 21589,
 21599, 21601, 21613, 21617, 21617, 21617, 21647, 21649, 21661, 21661,
 21673, 21683, 21683, 21701, 21701, 21713, 21727, 21727, 21739, 21751,
 21757, 21767, 21773, 21773, 21787, 21799, 21803, 21803, 21821, 21821,
 21839, 21841, 21851, 21863, 21871, 21871, 21881, 21893, 21893, 21911,
 21911, 21911, 21929, 21943, 21943, 21943, 21961, 21961, 21977, 21991,
 21997, 22003, 22013, 22013, 22031, 22039, 22039, 22051, 22063, 22067,
 22079, 22079, 22093, 22093, 22111, 22111, 22123, 22133, 22133, 22147,
 22159, 22159, 22171, 22171, 22189, 22193, 22193, 22193, 22193, 22229,
 22229, 22247, 22247, 22259, 22271, 22279, 22283, 22291, 22303, 22307,
 22307, 22307, 22307, 22343, 22349, 22349, 22367, 22369, 22381, 22391,
 22397, 22397, 22409, 22409, 22409, 22433, 22447, 22453, 22453, 22469,
 22469, 22483, 22483, 22501, 22511, 22511, 22511, 22531, 22543, 22549,
 22549, 22567, 22573, 22573, 22573, 22573, 22573, 22613, 22621, 22621,
 22639, 22643, 22651, 22651, 22669, 22679, 22679, 22691, 22699, 22709,
 22717, 22727, 22727, 22741, 22751, 22751, 22751, 22769, 22783, 22787,
 22787, 22807, 22811, 22817, 22817, 22817, 22817, 22853, 22861, 22871,
 22877, 22877, 22877, 22901, 22907, 22907, 22921, 22921, 22943, 22943,
 22943, 22963, 22973, 22973, 22973, 22993, 23003, 23011, 23021, 23029,
 23039, 23041, 23053, 23063, 23071, 23071, 23087, 23087, 23099, 23099,
 23117, 23117, 23131, 23143, 23143, 23159, 23167, 23173, 23173, 23189,
 23197, 23203, 23209, 23209, 23227, 23227, 23227, 23251, 23251, 23269,
 23279, 23279, 23293, 23297, 23311, 23311, 23327, 23333, 23339, 23339,
 23357, 23357, 23371, 23371, 23371, 23399, 23399, 23399, 23417, 23431,
 23431, 23447, 23447, 23459, 23459, 23473, 23473, 23473, 23497, 23509,
 23509, 23509, 23531, 23539, 23549, 23557, 23567, 23567, 23581, 23581,
 23599, 23603, 23609, 23623, 23629, 23633, 23633, 23633, 23663, 23671,
 23677, 23687, 23689, 23689, 23689, 23719, 23719, 23719, 23743, 23747,
 23753, 23767, 23773, 23773, 23789, 23789, 23801, 23813, 23819, 23831,
 23833, 23833, 23833, 23857, 23869, 23879, 23887, 23893, 23899, 23911,
 23917, 23917, 23929, 23929, 23929, 23957, 23957, 23971, 23981, 23981,
 23993, 24007, 24007, 24023, 24029, 24029, 24043, 24049, 24061, 24071,
 24077, 24083, 24091, 24103, 24109, 24113, 24121, 24133, 24137, 24151,
 24151, 24151, 24169, 24181, 24181, 24197, 24203, 24203, 24223, 24229,
 24239, 24247, 24251, 24251, 24251, 24251, 24281, 24281, 24281, 24281,
 24317, 24317, 24329, 24337, 24337, 24359, 24359, 24373, 24379, 24391,
 24391, 24407, 24413, 24421, 24421, 24439, 24443, 24443, 24443, 24469,
 24473, 24481, 24481, 24499, 24509, 24517, 24527, 24533, 24533, 24551,
 24551, 24551, 24571, 24571, 24571, 24593, 24593, 24611, 24623, 24631,
 24631, 24631, 24631, 24659, 24671, 24677, 24683, 24691, 24697, 24709,
 24709, 24709, 24733, 24733, 24749, 24749, 24767, 24767, 24781, 24781,
 24799, 24799, 24809, 24821, 24821, 24821, 24847, 24851, 24859, 24859,
 24877, 24877, 24889, 24889, 24907, 24919, 24923, 24923, 24943, 24943,
 24953, 24967, 24971, 24979, 24989, 24989, 24989, 25013, 25013, 25031,
 25037, 25037, 25037, 25057, 25057, 25073, 25087, 25087, 25097, 25111,
 25117, 25127, 25127, 25127, 25147, 25153, 25163, 25171, 25183, 25189,
 25189, 25189, 25189, 25219, 25229, 25237, 25247, 25253, 25261, 25261,
 25261, 25261, 25261, 25303, 25309, 25309, 25321, 25321, 25343, 25349,
 25357, 25367, 25373, 25373, 25391, 25391, 25391, 25411, 25423, 25423,
 25439, 25447, 25453, 25463, 25471, 25471, 25471, 25471, 25471, 25471,
 25471, 25523, 25523, 25541, 25541, 25541, 25561, 25561, 25583, 25589,
 25589, 25603, 25609, 25621, 25621, 25639, 25643, 25643, 25657, 25667,
 25679, 25679, 25693, 25703, 25703, 25717, 25717, 25733, 25741, 25747,
 25759, 25763, 25771, 25771, 25771, 25799, 25801, 25801, 25819, 25819,
 25819, 25847, 25849, 25849, 25867, 25873, 25873, 25889, 25903, 25903,
 25919, 25919, 25933, 25943, 25951, 25951, 25951, 25969, 25981, 25981,
 25999, 26003, 26003, 26021, 26029, 26029, 26041, 26053, 26053, 26053,
 26053, 26083, 26083, 26099, 26111, 26119, 26119, 26119, 26141, 26141,
 26153, 26161, 26171, 26183, 26189, 26189, 26203, 26209, 26209, 26227,
 26237, 26237, 26251, 26263, 26267, 26267, 26267, 26293, 26297, 26309,
 26317, 26321, 26321, 26339, 26347, 26357, 26357, 26371, 26371, 26387,
 26399, 26407, 26407, 26423, 26431, 26437, 26437, 26449, 26459, 26459,
 26479, 26479, 26489, 26501, 26501, 26513, 26513, 26513, 26539, 26539,
 26557, 26561, 26573, 26573, 26591, 26597, 26597, 26597, 26597, 26627,
 26633, 26647, 26647, 26647, 26669, 26669, 26687, 26693, 26701, 26711,
 26717, 26723, 26731, 26737, 26737, 26759, 26759, 26759, 26783, 26783,
 26783, 26801, 26813, 26821, 26821, 26839, 26839, 26849, 26863, 26863,
 26879, 26881, 26893, 26903, 26903, 26903, 26927, 26927, 26927, 26951,
 26959, 26959, 26959, 26981, 26987, 26993, 26993, 27011, 27017, 27031,
 27031, 27043, 27043, 27061, 27067, 27077, 27077, 27091, 27103, 27109,
 27109, 27127, 27127, 27143, 27143, 27143, 27143, 27143, 27179, 27191,
 27197, 27197, 27211, 27211, 27211, 27239, 27241, 27253, 27259, 27271,
 27277, 27283, 27283, 27299, 27299, 27299, 27299, 27329, 27337, 27337,
 27337, 27367, 27367, 27367, 27367, 27397, 27407, 27409, 27409, 27431,
 27437, 27437, 27449, 27457, 27457, 27479, 27487, 27487, 27487, 27509,
 27509, 27527, 27529, 27541, 27551, 27551, 27551, 27551, 27583, 27583,
 27583, 27583, 27611, 27617, 27631, 27631, 27647, 27653, 27653, 27653,
 27673, 27673, 27691, 27701, 27701, 27701, 27701, 27733, 27743, 27751,
 27751, 27767, 27773, 27779, 27791, 27799, 27803, 27809, 27823, 27827,
 27827, 27847, 27851, 27851, 27851, 27851, 27883, 27893, 27901, 27901,
 27919, 27919, 27919, 27943, 27947, 27953, 27967, 27967, 27983, 27983,
 27997, 28001, 28001, 28019, 28031, 28031, 28031, 28051, 28057, 28069,
 28069, 28087, 28087, 28099, 28111, 28111, 28123, 28123, 28123, 28151,
 28151, 28163, 28163, 28183, 28183, 28183, 28201, 28211, 28219, 28229,
 28229, 28229, 28229, 28229, 28229, 28279, 28283, 28289, 28297, 28309,
 28319, 28319, 28319, 28319, 28351, 28351, 28351, 28351, 28351, 28387,
 28393, 28403, 28411, 28411, 28429, 28439, 28447, 28447, 28463, 28463,
 28477, 28477, 28493, 28499, 28499, 28517, 28517, 28517, 28541, 28549,
 28559, 28559, 28573, 28579, 28591, 28597, 28607, 28607, 28621, 28631,
 28631, 28643, 28649, 28663, 28669, 28669, 28687, 28687, 28703, 28711,
 28711, 28723, 28729, 28729, 28751, 28759, 28759, 28771, 28771, 28789,
 28793, 28807, 28813, 28817, 28817, 28837, 28843, 28843, 28859, 28871,
 28879, 28879, 28879, 28901, 28909, 28909, 28927, 28933, 28933, 28949,
 28949, 28961, 28961, 28979, 28979, 28979, 28979, 29009, 29023, 29027,
 29033, 29033, 29033, 29063, 29063, 29077, 29077, 29077, 29101, 29101,
 29101, 29123, 29131, 29137, 29147, 29153, 29167, 29173, 29179, 29191,
 29191, 29207, 29209, 29221, 29231, 29231, 29243, 29251, 29251, 29269,
 29269, 29287, 29287, 29303, 29311, 29311, 29327, 29333, 29339, 29347,
 29347, 29363, 29363, 29383, 29389, 29399, 29401, 29411, 29423, 29429,
 29437, 29443, 29453, 29453, 29453, 29473, 29483, 29483, 29501, 29501,
 29501, 29527, 29531, 29537, 29537, 29537, 29567, 29573, 29581, 29587,
 29599, 29599, 29611, 29611, 29629, 29633, 29641, 29641, 29663, 29671,
 29671, 29683, 29683, 29683, 29683, 29717, 29723, 29723, 29741, 29741,
 29759, 29761, 29761, 29761, 29789, 29789, 29803, 29803, 29819, 29819,
 29837, 29837, 29851, 29863, 29867, 29879, 29881, 29881, 29881, 29881,
 29917, 29927, 29927, 29927, 29947, 29959, 29959, 29959, 29983, 29989,
 29989, 29989, 30013, 30013, 30029, 30029, 30047, 30047, 30059, 30071,
 30071, 30071, 30091, 30103, 30109, 30119, 30119, 30133, 30139, 30139,
 30139, 30161, 30169, 30181, 30187, 30197, 30203, 30211, 30223, 30223,
 30223, 30241, 30253, 30259, 30271, 30271, 30271, 30293, 30293, 30307,
 30319, 30323, 30323, 30341, 30347, 30347, 30367, 30367, 30367, 30391,
 30391, 30403, 30403, 30403, 30431, 30431, 30431, 30449, 30449, 30469,
 30469, 30469, 30493, 30497, 30509, 30517, 30517, 30529, 30539, 30539,
 30559, 30559, 30559, 30577, 30577, 30593, 30593, 30593, 30593, 30631,
 30637, 30643, 30649, 30661, 30671, 30677, 30677, 30689, 30703, 30707,
 30713, 30727, 30727, 30727, 30727, 30757, 30763, 30773, 30781, 30781,
 30781, 30803, 30809, 30817, 30829, 30839, 30841, 30853, 30859, 30871,
 30871, 30881, 30893, 30893, 30911, 30911, 30911, 30931, 30941, 30949,
 30949, 30949, 30971, 30983, 30983, 30983, 30983, 31013, 31019, 31019,
 31039, 31039, 31051, 31063, 31069, 31079, 31081, 31091, 31091, 31091,
 31091, 31123, 31123, 31139, 31151, 31159, 31159, 31159, 31183, 31189,
 31193, 31193, 31193, 31223, 31231, 31237, 31247, 31253, 31259, 31271,
 31277, 31277, 31277, 31277, 31307, 31319, 31327, 31333, 31337, 31337,
 31357, 31357, 31357, 31379, 31391, 31397, 31397, 31397, 31397, 31397,
 31397, 31397, 31397, 31397, 31469, 31477, 31481, 31489, 31489, 31511,
 31517, 31517, 31531, 31543, 31547, 31547, 31567, 31573, 31583, 31583,
 31583, 31607, 31607, 31607, 31627, 31627, 31643, 31649, 31663, 31667,
 31667, 31687, 31687, 31699, 31699, 31699, 31727, 31729, 31741, 31751,
 31751, 31751, 31771, 31771, 31771, 31799, 31799, 31799, 31817, 31817,
 31817, 31847, 31849, 31859, 31859, 31873, 31883, 31891, 31891, 31907,
 31907, 31907, 31907, 31907, 31907, 31957, 31963, 31973, 31981, 31991,
 31991, 32003, 32009, 32009, 32029, 32029, 32029, 32051, 32063, 32069,
 32077, 32083, 32089, 32099, 32099, 32119, 32119, 32119, 32143, 32143,
 32159, 32159, 32173, 32183, 32191, 32191, 32203, 32213, 32213, 32213,
 32237, 32237, 32251, 32261, 32261, 32261, 32261, 32261, 32303, 32309,
 32309, 32327, 32327, 32341, 32341, 32359, 32363, 32371, 32381, 32381,
 32381, 32401, 32413, 32423, 32429, 32429, 32443, 32443, 32443, 32467,
 32479, 32479, 32491, 32503, 32507, 32507, 32507, 32533, 32537, 32537,
 32537, 32563, 32573, 32579, 32587, 32587, 32603, 32611, 32621, 32621,
 32633, 32647, 32653, 32653, 32653, 32653, 32687, 32693, 32693, 32707,
 32719, 32719, 32719, 32719, 32749, 32749, 32749, 32771, 32783, 32789,
 32797, 32803, 32803, 32803, 32831, 32839, 32843, 32843, 32843, 32869,
 32869, 32887, 32887, 32887, 32911, 32917, 32917, 32933, 32941, 32941,
 32957, 32957, 32971, 32983, 32987, 32999, 32999, 33013, 33023, 33029,
 33037, 33037, 33053, 33053, 33071, 33073, 33083, 33091, 33091, 33107,
 33119, 33119, 33119, 33119, 33151, 33151, 33161, 33161, 33181, 33191,
 33199, 33203, 33211, 33223, 33223, 33223, 33247, 33247, 33247, 33247,
 33247, 33287, 33289, 33301, 33311, 33317, 33317, 33331, 33343, 33349,
 33359, 33359, 33359, 33377, 33391, 33391, 33403, 33413, 33413, 33427,
 33427, 33427, 33427, 33461, 33469, 33479, 33487, 33493, 33503, 33503,
 33503, 33521, 33533, 33533, 33547, 33547, 33563, 33569, 33581, 33589,
 33599, 33601, 33613, 33623, 33629, 33637, 33647, 33647, 33647, 33647,
 33679, 33679, 33679, 33703, 33703, 33713, 33721, 33721, 33739, 33751,
 33757, 33767, 33773, 33773, 33791, 33797, 33797, 33811, 33811, 33829,
 33829, 33829, 33851, 33863, 33871, 33871, 33871, 33893, 33893, 33911,
 33911, 33923, 33931, 33941, 33941, 33941, 33967, 33967, 33967, 33967,
 33997, 33997, 33997, 34019, 34031, 34039, 34039, 34039, 34061, 34061,
 34061, 34061, 34061, 34061, 34061, 34061, 34127, 34129, 34141, 34147,
 34159, 34159, 34171, 34183, 34183, 34183, 34183, 34213, 34217, 34231,
 34231, 34231, 34253, 34261, 34267, 34273, 34283, 34283, 34303, 34303,
 34319, 34327, 34327, 34337, 34351, 34351, 34367, 34369, 34381, 34381,
 34381, 34403, 34403, 34421, 34429, 34439, 34439, 34439, 34457, 34471,
 34471, 34487, 34487, 34501, 34511, 34519, 34519, 34519, 34543, 34549,
 34549, 34549, 34549, 34583, 34591, 34591, 34607, 34613, 34613, 34631,
 34631, 34631, 34651, 34651, 34667, 34679, 34687, 34693, 34703, 34703,
 34703, 34721, 34729, 34739, 34747, 34759, 34763, 34763, 34781, 34781,
 34781, 34807, 34807, 34819, 34819, 34819, 34847, 34849, 34849, 34871,
 34877, 34883, 34883, 34897, 34897, 34919, 34919, 34919, 34939, 34949,
 34949, 34963, 34963, 34981, 34981, 34981, 34981, 34981, 35023, 35027,
 35027, 35027, 35053, 35059, 35069, 35069, 35083, 35089, 35099, 35111,
 35117, 35117, 35129, 35141, 35149, 35159, 35159, 35171, 35171, 35171,
 35171, 35201, 35201, 35221, 35227, 35227, 35227, 35251, 35257, 35267,
 35279, 35281, 35291, 35291, 35311, 35317, 35327, 35327, 35339, 35339,
 35353, 35363, 35363, 35381, 35381, 35393, 35407, 35407, 35423, 35423,
 35437, 35447, 35449, 35461, 35461, 35461, 35461, 35491, 35491, 35509,
 35509, 35527, 35533, 35543, 35543, 35543, 35543, 35573, 35573, 35591,
 35597, 35603, 35603, 35617, 35617, 35617, 35617, 35617, 35617, 35671,
 35677, 35677, 35677, 35677, 35677, 35677, 35677, 35731, 35731, 35747,
 35759, 35759, 35771, 35771, 35771, 35797, 35803, 35809, 35809, 35831,
 35839, 35839, 35851, 35863, 35869, 35879, 35879, 35879, 35899, 35911,
 35911, 35923, 35933, 35933, 35951, 35951, 35963, 35969, 35983, 35983,
 35999, 36007, 36013, 36017, 36017, 36037, 36037, 36037, 36061, 36067,
 36073, 36083, 36083, 36097, 36109, 36109, 36109, 36131, 36137, 36151,
 36151, 36161, 36161, 36161, 36191, 36191, 36191, 36209, 36217, 36229,
 36229, 36241, 36251, 36263, 36269, 36277, 36277, 36293, 36299, 36307,
 36319, 36319, 36319, 36343, 36343, 36353, 36353, 36373, 36383, 36389,
 36389, 36389, 36389, 36389, 36389, 36433, 36433, 36451, 36457, 36469,
 36479, 36479, 36493, 36497, 36497, 36497, 36527, 36529, 36541, 36551,
 36559, 36563, 36571, 36583, 36587, 36599, 36607, 36607, 36607, 36629,
 36637, 36643, 36653, 36653, 36671, 36677, 36683, 36691, 36697, 36709,
 36713, 36721, 36721, 36739, 36749, 36749, 36767, 36767, 36781, 36791,
 36793, 36793, 36809, 36821, 36821, 36833, 36847, 36847, 36857, 36871,
 36877, 36887, 36887, 36901, 36901, 36919, 36923, 36931, 36943, 36947,
 36947, 36947, 36973, 36979, 36979, 36997, 37003, 37013, 37021, 37021,
 37039, 37039, 37049, 37061, 37061, 37061, 37087, 37087, 37097, 37097,
 37117, 37123, 37123, 37139, 37139, 37159, 37159, 37171, 37181, 37189,
 37199, 37201, 37201, 37223, 37223, 37223, 37243, 37253, 37253, 37253,
 37277, 37277, 37277, 37277, 37309, 37313, 37321, 37321, 37339, 37339,
 37357, 37363, 37369, 37379, 37379, 37397, 37397, 37409, 37423, 37423,
 37423, 37447, 37447, 37463, 37463, 37463, 37483, 37493, 37501, 37511,
 37517, 37517, 37529, 37537, 37549, 37549, 37567, 37573, 37579, 37591,
 37591, 37607, 37607, 37619, 37619, 37633, 37643, 37649, 37663, 37663,
 37663, 37663, 37693, 37699, 37699, 37717, 37717, 37717, 37717, 37747,
 37747, 37747, 37747, 37783, 37783, 37799, 37799, 37813, 37813, 37831,
 37831, 37847, 37853, 37861, 37871, 37879, 37879, 37889, 37897, 37907,
 37907, 37907, 37907, 37907, 37951, 37957, 37967, 37967, 37967, 37991,
 37997, 37997, 38011, 38011, 38011, 38039, 38047, 38053, 38053, 38069,
 38069, 38083, 38083, 38083, 38083, 38119, 38119, 38119, 38119, 38149,
 38153, 38167, 38167, 38183, 38189, 38197, 38201, 38201, 38219, 38231,
 38239, 38239, 38239, 38261, 38261, 38273, 38287, 38287, 38303, 38303,
 38317, 38327, 38333, 38333, 38351, 38351, 38351, 38371, 38377, 38377,
 38393, 38393, 38393, 38393, 38431, 38431, 38447, 38453, 38461, 38461,
 38461, 38461, 38461, 38501, 38501, 38501, 38501, 38501, 38543, 38543,
 38557, 38567, 38569, 38569, 38569, 38593, 38603, 38611, 38611, 38629,
 38639, 38639, 38653, 38653, 38671, 38677, 38677, 38693, 38699, 38711,
 38713, 38723, 38729, 38737, 38749, 38749, 38767, 38767, 38783, 38791,
 38791, 38803, 38803, 38821, 38821, 38839, 38839, 38851, 38861, 38867,
 38873, 38873, 38891, 38903, 38903, 38917, 38923, 38933, 38933, 38933,
 38959, 38959, 38971, 38977, 38977, 38993, 38993, 38993, 39023, 39023,
 39023, 39047, 39047, 39047, 39047, 39079, 39079, 39089, 39103, 39107,
 39119, 39119, 39133, 39139, 39139, 39157, 39163, 39163, 39181, 39191,
 39199, 39199, 39209, 39217, 39229, 39239, 39241, 39251, 39251, 39251,
 39251, 39251, 39293, 39301, 39301, 39317, 39323, 39323, 39343, 39343,
 39359, 39367, 39373, 39383, 39383, 39397, 39397, 39409, 39419, 39419,
 39439, 39443, 39451, 39461, 39461, 39461, 39461, 39461, 39503, 39511,
 39511, 39521, 39521, 39541, 39551, 39551, 39563, 39569, 39581, 39581,
 39581, 39607, 39607, 39623, 39631, 39631, 39631, 39631, 39659, 39671,
 39679, 39679, 39679, 39703, 39709, 39719, 39727, 39733, 39733, 39749,
 39749, 39761, 39769, 39779, 39791, 39799, 39799, 39799, 39821, 39829,
 39839, 39847, 39847, 39863, 39869, 39877, 39887, 39887, 39901, 39901,
 39901, 39901, 39929, 39937, 39937, 39953, 39953, 39971, 39983, 39989,
 39989, 39989, 40013, 40013, 40031, 40039, 40039, 40039, 40063, 40063,
 40063, 40087, 40093, 40099, 40111, 40111, 40127, 40129, 40129, 40151,
 40153, 40163, 40169, 40177, 40189, 40193, 40193, 40213, 40213, 40231,
 40237, 40241, 40253, 40253, 40253, 40277, 40283, 40289, 40289, 40289,
 40289, 40289, 40289, 40343, 40351, 40357, 40361, 40361, 40361, 40387,
 40387, 40387, 40387, 40423, 40429, 40433, 40433, 40433, 40459, 40471,
 40471, 40487, 40493, 40499, 40507, 40519, 40519, 40531, 40543, 40543,
 40559, 40559, 40559, 40583, 40591, 40597, 40597, 40609, 40609, 40627,
 40639, 40639, 40639, 40639, 40639, 40639, 40639, 40693, 40699, 40709,
 40709, 40709, 40709, 40739, 40751, 40759, 40763, 40771, 40771, 40787,
 40787, 40801, 40813, 40823, 40829, 40829, 40847, 40853, 40853, 40867,
 40879, 40883, 40883, 40903, 40903, 40903, 40927, 40933, 40939, 40949,
 40949, 40961, 40973, 40973, 40973, 40993, 40993, 41011, 41023, 41023,
 41039, 41047, 41051, 41057, 41057, 41077, 41081, 41081, 41081, 41081,
 41117, 41117, 41131, 41143, 41149, 41149, 41161, 41161, 41183, 41189,
 41189, 41203, 41213, 41221, 41231, 41233, 41243, 41243, 41263, 41269,
 41269, 41281, 41281, 41299, 41299, 41299, 41299, 41333, 41341, 41351,
 41357, 41357, 41357, 41381, 41389, 41399, 41399, 41413, 41413, 41413,
 41413, 41443, 41453, 41453, 41467, 41479, 41479, 41491, 41491, 41507,
 41519, 41521, 41521, 41543, 41549, 41549, 41549, 41549, 41579, 41579,
 41597, 41603, 41611, 41621, 41627, 41627, 41647, 41651, 41659, 41669,
 41669, 41687, 41687, 41687, 41687, 41719, 41719, 41729, 41737, 41737,
 41759, 41761, 41771, 41777, 41777, 41777, 41801, 41813, 41813, 41813,
 41813, 41843, 41851, 41863, 41863, 41879, 41887, 41893, 41903, 41911,
 41911, 41927, 41927, 41941, 41947, 41959, 41959, 41969, 41983, 41983,
 41999, 41999, 42013, 42023, 42023, 42023, 42043, 42043, 42061, 42071,
 42073, 42083, 42089, 42101, 42101, 42101, 42101, 42131, 42139, 42139,
 42157, 42157, 42169, 42181, 42187, 42197, 42197, 42209, 42223, 42227,
 42239, 42239, 42239, 42257, 42257, 42257, 42283, 42293, 42299, 42307,
 42307, 42323, 42331, 42337, 42349, 42359, 42359, 42373, 42379, 42391,
 42397, 42407, 42409, 42409, 42409, 42437, 42443, 42451, 42463, 42467,
 42473, 42487, 42491, 42499, 42509, 42509, 42509, 42533, 42533, 42533,
 42557, 42557, 42571, 42577, 42589, 42589, 42589, 42611, 42611, 42611,
 42611, 42643, 42649, 42649, 42667, 42677, 42683, 42689, 42703, 42709,
 42719, 42727, 42727, 42743, 42751, 42751, 42767, 42773, 42773, 42787,
 42797, 42797, 42797, 42821, 42829, 42839, 42841, 42853, 42863, 42863,
 42863, 42863, 42863, 42901, 42901, 42901, 42923, 42929, 42943, 42943,
 42953, 42967, 42967, 42979, 42989, 42989, 43003, 43013, 43019, 43019,
 43037, 43037, 43051, 43063, 43067, 43067, 43067, 43093, 43103, 43103,
 43117, 43117, 43133, 43133, 43151, 43159, 43159, 43159, 43177, 43189,
 43189, 43207, 43207, 43223, 43223, 43237, 43237, 43237, 43261, 43271,
 43271, 43283, 43291, 43291, 43291, 43319, 43321, 43331, 43331, 43331,
 43331, 43331, 43331, 43331, 43391, 43399, 43403, 43411, 43411, 43427,
 43427, 43441, 43451, 43457, 43457, 43457, 43487, 43487, 43499, 43499,
 43517, 43517, 43517, 43543, 43543, 43543, 43543, 43573, 43579, 43591,
 43597, 43607, 43613, 43613, 43627, 43633, 43633, 43651, 43661, 43669,
 43669, 43669, 43691, 43691, 43711, 43717, 43721, 43721, 43721, 43721,
 43759, 43759, 43759, 43783, 43789, 43793, 43801, 43801, 43801, 43801,
 43801, 43801, 43853, 43853, 43867, 43867, 43867, 43891, 43891, 43891,
 43913, 43913, 43933, 43943, 43951, 43951, 43963, 43973, 43973, 43991,
 43997, 43997, 43997, 44021, 44029, 44029, 44041, 44053, 44059, 44071,
 44071, 44087, 44089, 44101, 44111, 44119, 44123, 44131, 44131, 44131,
 44159, 44159, 44171, 44179, 44189, 44189, 44207, 44207, 44221, 44221,
 44221, 44221, 44249, 44263, 44269, 44279, 44281, 44293, 44293, 44293,
 44293, 44293, 44293, 44293, 44351, 44357, 44357, 44371, 44383, 44389,
 44389, 44389, 44389, 44417, 44417, 44417, 44417, 44453, 44453, 44453,
 44453, 44483, 44491, 44501, 44507, 44519, 44519, 44533, 44543, 44549,
 44549, 44563, 44563, 44579, 44587, 44587, 44587, 44587, 44623, 44623,
 44633, 44647, 44651, 44657, 44657, 44657, 44687, 44687, 44701, 44711,
 44711, 44711, 44729, 44741, 44741, 44753, 44753, 44773, 44777, 44789,
 44797, 44797, 44809, 44819, 44819, 44839, 44843, 44851, 44851, 44867,
 44879, 44887, 44893, 44893, 44909, 44917, 44927, 44927, 44939, 44939,
 44959, 44963, 44971, 44983, 44987, 44987, 45007, 45013, 45013, 45013,
 45013, 45013, 45053, 45061, 45061, 45077, 45083, 45083, 45083, 45083,
 45119, 45127, 45131, 45139, 45139, 45139, 45161, 45161, 45181, 45191,
 45197, 45197, 45197, 45197, 45197, 45233, 45247, 45247, 45263, 45263,
 45263, 45281, 45293, 45293, 45307, 45319, 45319, 45329, 45343, 45343,
 45343, 45361, 45361, 45377, 45389, 45389, 45403, 45413, 45413, 45427,
 45439, 45439, 45439, 45439, 45439, 45439, 45481, 45491, 45503, 45503,
 45503, 45523, 45533, 45541, 45541, 45557, 45557, 45569, 45569, 45589,
 45599, 45599, 45613, 45613, 45631, 45631, 45641, 45641, 45659, 45667,
 45677, 45677, 45691, 45697, 45707, 45707, 45707, 45707, 45737, 45751,
 45757, 45767, 45767, 45779, 45779, 45779, 45779, 45779, 45823, 45827,
 45833, 45841, 45853, 45863, 45869, 45869, 45887, 45893, 45893, 45893,
 45893, 45893, 45893, 45943, 45949, 45959, 45959, 45971, 45979, 45989,
 45989, 45989, 45989, 46021, 46027, 46027, 46027, 46051, 46061, 46061,
 46073, 46073, 46093, 46103, 46103, 46103, 46103, 46133, 46141, 46147,
 46153, 46153, 46171, 46183, 46187, 46199, 46199, 46199, 46219, 46229,
 46237, 46237, 46237, 46261, 46271, 46279, 46279, 46279, 46301, 46309,
 46309, 46327, 46327, 46337, 46351, 46351, 46351, 46351, 46381, 46381,
 46399, 46399, 46411, 46411, 46411, 46439, 46447, 46451, 46457, 46471,
 46477, 46477, 46489, 46499, 46511, 46511, 46523, 46523, 46523, 46549,
 46559, 46567, 46573, 46573, 46591, 46591, 46601, 46601, 46619, 46619,
 46639, 46643, 46649, 46663, 46663, 46679, 46687, 46691, 46703, 46703,
 46703, 46727, 46727, 46727, 46751, 46757, 46757, 46771, 46771, 46771,
 46771, 46807, 46811, 46819, 46831, 46831, 46831, 46853, 46861, 46867,
 46877, 46877, 46889, 46901, 46901, 46919, 46919, 46933, 46933, 46933,
 46957, 46957, 46957, 46957, 46957, 46997, 46997, 46997, 47017, 47017,
 47017, 47041, 47051, 47059, 47059, 47059, 47087, 47093, 47093, 47111,
 47119, 47123, 47129, 47143, 47149, 47149, 47161, 47161, 47161, 47189,
 47189, 47207, 47207, 47221, 47221, 47237, 47237, 47251, 47251, 47269,
 47279, 47287, 47293, 47303, 47309, 47317, 47317, 47317, 47339, 47351,
 47353, 47363, 47363, 47381, 47389, 47389, 47407, 47407, 47419, 47431,
 47431, 47441, 47441, 47459, 47459, 47459, 47459, 47491, 47501, 47507,
 47513, 47527, 47533, 47543, 47543, 47543, 47563, 47569, 47581, 47591,
 47599, 47599, 47609, 47623, 47629, 47639, 47639, 47653, 47659, 47659,
 47659, 47681, 47681, 47701, 47711, 47717, 47717, 47717, 47743, 47743,
 47743, 47743, 47743, 47779, 47791, 47797, 47807, 47809, 47819, 47819,
 47837, 47843, 47843, 47857, 47869, 47869, 47881, 47881, 47903, 47911,
 47917, 47917, 47933, 47939, 47951, 47951, 47963, 47969, 47981, 47981,
 47981, 47981, 47981, 48023, 48029, 48029, 48029, 48049, 48049, 48049,
 48079, 48079, 48091, 48091, 48109, 48119, 48121, 48131, 48131, 48131,
 48157, 48163, 48163, 48179, 48187, 48197, 48197, 48197, 48221, 48221,
 48239, 48247, 48247, 48259, 48271, 48271, 48281, 48281, 48299, 48311,
 48313, 48313, 48313, 48341, 48341, 48353, 48353, 48371, 48383, 48383,
 48397, 48407, 48413, 48413, 48413, 48437, 48437, 48449, 48463, 48463,
 48479, 48487, 48491, 48497, 48497, 48497, 48527, 48533, 48541, 48541,
 48541, 48563, 48571, 48571, 48589, 48593, 48593, 48611, 48623, 48623,
 48623, 48647, 48649, 48661, 48661, 48679, 48679, 48679, 48679, 48679,
 48679, 48679, 48733, 48733, 48751, 48757, 48767, 48767, 48781, 48787,
 48799, 48799, 48809, 48823, 48823, 48823, 48847, 48847, 48859, 48871,
 48871, 48883, 48889, 48889, 48907, 48907, 48907, 48907, 48907, 48947,
 48953, 48953, 48973, 48973, 48991, 48991, 49003, 49009, 49019, 49031,
 49037, 49043, 49043, 49057, 49069, 49069, 49081, 49081, 49103, 49109,
 49117, 49123, 49123, 49139, 49139, 49157, 49157, 49171, 49177, 49177,
 49199, 49207, 49211, 49223, 49223, 49223, 49223, 49253, 49261, 49261,
 49279, 49279, 49279, 49297, 49307, 49307, 49307, 49333, 49339, 49339,
 49339, 49367, 49369, 49369, 49391, 49393, 49393, 49411, 49417, 49429,
 49433, 49433, 49451, 49463, 49463, 49477, 49481, 49481, 49499, 49499,
 49499, 49523, 49531, 49537, 49549, 49559, 49559, 49559, 49559, 49559,
 49597, 49603, 49613, 49613, 49627, 49639, 49639, 49639, 49663, 49669,
 49669, 49681, 49681, 49697, 49711, 49711, 49727, 49727, 49741, 49747,
 49757, 49757, 49757, 49783, 49789, 49789, 49807, 49811, 49823, 49831,
 49831, 49843, 49853, 49853, 49871, 49877, 49877, 49891, 49891, 49891,
 49919, 49927, 49927, 49943, 49943, 49957, 49957, 49957, 49957, 49991,
 49999, 49999, 49999, 50023, 50023, 50033, 50047, 50053, 50053, 50069,
 50077, 50087, 50093, 50101, 50111, 50119, 50123, 50131, 50131, 50147,
 50159, 50159, 50159, 50177, 50177, 50177, 50207, 50207, 50221, 50231,
 50231, 50231, 50231, 50263, 50263, 50273, 50287, 50291, 50291, 50311,
 50311, 50321, 50333, 50341, 50341, 50359, 50363, 50363, 50383, 50387,
 50387, 50387, 50411, 50423, 50423, 50423, 50441, 50441, 50461, 50461,
 50461, 50461, 50461, 50503, 50503, 50513, 50527, 50527, 50543, 50551,
 50551, 50551, 50551, 50581, 50591, 50599, 50599, 50599, 50599, 50627,
 50627, 50647, 50651, 50651, 50671, 50671, 50683, 50683, 50683, 50707,
 50707, 50723, 50723, 50741, 50741, 50753, 50767, 50773, 50777, 50789,
 50789, 50789, 50789, 50821, 50821, 50839, 50839, 50849, 50857, 50867,
 50873, 50873, 50893, 50893, 50909, 50909, 50923, 50929, 50929, 50951,
 50957, 50957, 50971, 50971, 50989, 50993, 51001, 51001, 51001, 51031,
 51031, 51047, 51047, 51061, 51071, 51071, 51071, 51071, 51071, 51109,
 51109, 51109, 51133, 51137, 51151, 51157, 51157, 51169, 51169, 51169,
 51199, 51203, 51203, 51217, 51229, 51239, 51241, 51241, 51263, 51263,
 51263, 51287, 51287, 51287, 51307, 51307, 51307, 51329, 51343, 51349,
 51349, 51361, 51361, 51383, 51383, 51383, 51407, 51413, 51421, 51431,
 51439, 51439, 51449, 51461, 51461, 51479, 51487, 51487, 51503, 51511,
 51517, 51521, 51521, 51539, 51551, 51551, 51563, 51563, 51581, 51581,
 51599, 51607, 51613, 51613, 51631, 51637, 51647, 51647, 51659, 51659,
 51679, 51683, 51691, 51691, 51691, 51719, 51721, 51721, 51721, 51749,
 51749, 51767, 51769, 51769, 51787, 51797, 51803, 51803, 51817, 51829,
 51839, 51839, 51853, 51859, 51871, 51871, 51871, 51893, 51899, 51907,
 51913, 51913, 51929, 51941, 51949, 51949, 51949, 51973, 51977, 51991,
 51991, 51991, 52009, 52021, 52027, 52027, 52027, 52051, 52057, 52069,
 52069, 52081, 52081, 52103, 52103, 52103, 52127, 52127, 52127, 52147,
 52153, 52163, 52163, 52183, 52189, 52189, 52201, 52201, 52223, 52223,
 52237, 52237, 52253, 52259, 52267, 52267, 52267, 52291, 52301, 52301,
 52313, 52321, 52321, 52321, 52321, 52321, 52363, 52369, 52379, 52391,
 52391, 52391, 52391, 52391, 52391, 52433, 52433, 52453, 52457, 52457,
 52457, 52457, 52489, 52501, 52511, 52517, 52517, 52529, 52543, 52543,
 52553, 52567, 52571, 52583, 52583, 52583, 52583, 52609, 52609, 52631,
 52639, 52639, 52639, 52639, 52667, 52673, 52673, 52691, 52697, 52711,
 52711, 52727, 52733, 52733, 52747, 52757, 52757, 52769, 52783, 52783,
 52783, 52807, 52813, 52817, 52817, 52837, 52837, 52837, 52861, 52861,
 52879, 52883, 52889, 52903, 52903, 52919, 52919, 52919, 52937, 52951,
 52957, 52967, 52973, 52981, 52981, 52999, 53003, 53003, 53017, 53017,
 53017, 53047, 53051, 53051, 53069, 53077, 53087, 53093, 53101, 53101,
 53117, 53117, 53129, 53129, 53149, 53149, 53161, 53173, 53173, 53189,
 53197, 53201, 53201, 53201, 53231, 53239, 53239, 53239, 53239, 53269,
 53279, 53281, 53281, 53299, 53309, 53309, 53327, 53327, 53327, 53327,
 53359, 53359, 53359, 53381, 53381, 53381, 53407, 53411, 53419, 53419,
 53437, 53441, 53453, 53453, 53453, 53479, 53479, 53479, 53503, 53507,
 53507, 53527, 53527, 53527, 53551, 53551, 53551, 53569, 53569, 53591,
 53597, 53597, 53611, 53623, 53629, 53639, 53639, 53653, 53657, 53657,
 53657, 53681, 53693, 53699, 53699, 53719, 53719, 53731, 53731, 53731,
 53759, 53759, 53773, 53783, 53791, 53791, 53791, 53813, 53819, 53831,
 53831, 53831, 53849, 53861, 53861, 53861, 53887, 53891, 53899, 53899,
 53917, 53927, 53927, 53939, 53951, 53959, 53959, 53959, 53959, 53987,
 53993, 54001, 54013, 54013, 54013, 54037, 54037, 54049, 54059, 54059,
 54059, 54083, 54091, 54101, 54101, 54101, 54121, 54133, 54139, 54151,
 54151, 54167, 54167, 54181, 54181, 54193, 54193, 54193, 54217, 54217,
 54217, 54217, 54251, 54251, 54269, 54277, 54287, 54293, 54293, 54311,
 54319, 54323, 54331, 54331, 54347, 54347, 54367, 54371, 54377, 54377,
 54377, 54403, 54413, 54421, 54421, 54437, 54443, 54449, 54449, 54469,
 54469, 54469, 54493, 54503, 54503, 54517, 54521, 54521, 54541, 54547,
 54559, 54563, 54563, 54583, 54583, 54583, 54601, 54601, 54623, 54631,
 54631, 54647, 54647, 54647, 54667, 54679, 54679, 54679, 54679, 54709,
 54713, 54727, 54727, 54727, 54751, 54751, 54767, 54773, 54779, 54787,
 54799, 54799, 54799, 54799, 54829, 54833, 54833, 54851, 54851, 54869,
 54877, 54881, 54881, 54881, 54907, 54919, 54919, 54919, 54941, 54949,
 54959, 54959, 54973, 54983, 54983, 54983, 55001, 55009, 55021, 55021,
 55021, 55021, 55051, 55061, 55061, 55079, 55079, 55079, 55103, 55109,
 55117, 55127, 55127, 55127, 55147, 55147, 55163, 55171, 55171, 55171,
 55171, 55207, 55213, 55219, 55229, 55229, 55243, 55249, 55259, 55259,
 55259, 55259, 55291, 55291, 55291, 55313, 55313, 55333, 55343, 55351,
 55351, 55351, 55373, 55381, 55381, 55399, 55399, 55411, 55411, 55411,
 55439, 55441, 55441, 55457, 55469, 55469, 55487, 55487, 55501, 55511,
 55511, 55511, 55529, 55541, 55547, 55547, 55547, 55547, 55579, 55589,
 55589, 55603, 55609, 55621, 55631, 55639, 55639, 55639, 55663, 55667,
 55673, 55681, 55691, 55697, 55711, 55717, 55721, 55733, 55733, 55733,
 55733, 55763, 55763, 55763, 55787, 55799, 55807, 55813, 55823, 55829,
 55837, 55843, 55849, 55849, 55871, 55871, 55871, 55889, 55903, 55903,
 55903, 55927, 55933, 55933, 55949, 55949, 55967, 55967, 55967, 55987,
 55997, 56003, 56009, 56009, 56009, 56039, 56041, 56053, 56053, 56053,
 56053, 56087, 56093, 56101, 56101, 56113, 56123, 56131, 56131, 56149,
 56149, 56167, 56171, 56179, 56179, 56197, 56207, 56209, 56209, 56209,
 56239, 56239, 56249, 56263, 56269, 56269, 56269, 56269, 56299, 56311,
 56311, 56311, 56333, 56333, 56333, 56359, 56359, 56369, 56383, 56383,
 56393, 56401, 56401, 56417, 56431, 56437, 56443, 56453, 56453, 56467,
 56479, 56479, 56489, 56503, 56509, 56519, 56527, 56533, 56543, 56543,
 56543, 56543, 56569, 56569, 56591, 56599, 56599, 56611, 56611, 56629,
 56633, 56633, 56633, 56663, 56671, 56671, 56687, 56687, 56701, 56711,
 56713, 56713, 56731, 56737, 56747, 56747, 56767, 56773, 56783, 56783,
 56783, 56807, 56813, 56821, 56827, 56827, 56843, 56843, 56857, 56857,
 56873, 56873, 56893, 56897, 56911, 56911, 56923, 56929, 56941, 56951,
 56957, 56963, 56963, 56983, 56989, 56999, 56999, 56999, 56999, 56999,
 57037, 57047, 57047, 57059, 57059, 57077, 57077, 57089, 57097, 57107,
 57119, 57119, 57131, 57143, 57149, 57149, 57163, 57173, 57179, 57191,
 57193, 57203, 57203, 57223, 57223, 57223, 57241, 57251, 57259, 57271,
 57271, 57287, 57287, 57301, 57301, 57301, 57301, 57331, 57331, 57349,
 57349, 57367, 57373, 57383, 57389, 57397, 57397, 57413, 57413, 57427,
 57427, 57427, 57427, 57457, 57467, 57467, 57487, 57493, 57503, 57503,
 57503, 57527, 57529, 57529, 57529, 57559, 57559, 57571, 57571, 57587,
 57593, 57601, 57601, 57601, 57601, 57637, 57641, 57653, 57653, 57667,
 57679, 57679, 57689, 57697, 57709, 57719, 57727, 57731, 57737, 57751,
 57751, 57751, 57773, 57781, 57791, 57793, 57803, 57809, 57809, 57829,
 57839, 57847, 57853, 57859, 57859, 57859, 57881, 57881, 57901, 57901,
 57917, 57923, 57923, 57943, 57947, 57947, 57947, 57973, 57977, 57991,
 57991, 57991, 58013, 58013, 58031, 58031, 58043, 58049, 58061, 58067,
 58073, 58073, 58073, 58099, 58111, 58111, 58111, 58129, 58129, 58151,
 58153, 58153, 58171, 58171, 58189, 58199, 58207, 58211, 58217, 58231,
 58237, 58243, 58243, 58243, 58271, 58271, 58271, 58271, 58271, 58309,
 58313, 58321, 58321, 58337, 58337, 58337, 58367, 58369, 58379, 58391,
 58393, 58403, 58411, 58417, 58427, 58439, 58441, 58453, 58453, 58453,
 58477, 58481, 58481, 58481, 58511, 58511, 58511, 58511, 58543, 58549,
 58549, 58567, 58573, 58579, 58579, 58579, 58603, 58613, 58613, 58631,
 58631, 58631, 58631, 58661, 58661, 58679, 58687, 58693, 58699, 58711,
 58711, 58727, 58733, 58741, 58741, 58757, 58763, 58771, 58771, 58789,
 58789, 58789, 58789, 58789, 58831, 58831, 58831, 58831, 58831, 58831,
 58831, 58831, 58889, 58901, 58909, 58913, 58921, 58921, 58943, 58943,
 58943, 58967, 58967, 58979, 58991, 58997, 58997, 59011, 59023, 59029,
 59029, 59029, 59053, 59063, 59069, 59077, 59083, 59093, 59093, 59107,
 59119, 59123, 59123, 59141, 59149, 59159, 59167, 59167, 59183, 59183,
 59197, 59207, 59209, 59221, 59221, 59239, 59243, 59243, 59263, 59263,
 59273, 59281, 59281, 59281, 59281, 59281, 59281, 59333, 59341, 59351,
 59359, 59359, 59369, 59377, 59387, 59399, 59407, 59407, 59419, 59419,
 59419, 59447, 59453, 59453, 59471, 59473, 59473, 59473, 59497, 59509,
 59513, 59513, 59513, 59539, 59539, 59557, 59567, 59567, 59581, 59581,
 59581, 59581, 59611, 59621, 59629, 59629, 59629, 59651, 59663, 59671,
 59671, 59671, 59693, 59699, 59707, 59707, 59723, 59729, 59743, 59747,
 59753, 59753, 59771, 59779, 59791, 59797, 59797, 59809, 59809, 59809,
 59833, 59833, 59833, 59863, 59863, 59879, 59887, 59887, 59887, 59887,
 59887, 59921, 59929, 59929, 59951, 59957, 59957, 59971, 59981, 59981,
 59999, 59999, 60013, 60017, 60029, 60037, 60041, 60041, 60041, 60041,
 60077, 60083, 60091, 60103, 60107, 60107, 60127, 60133, 60139, 60149,
 60149, 60167, 60169, 60169, 60169, 60169, 60169, 60209, 60223, 60223,
 60223, 60223, 60251, 60259, 60271, 60271, 60271, 60293, 60293, 60293,
 60317, 60317, 60331, 60343, 60343, 60353, 60353, 60373, 60383, 60383,
 60397, 60397, 60413, 60413, 60427, 60427, 60443, 60449, 60457, 60457,
 60457, 60457, 60493, 60497, 60509, 60509, 60527, 60527, 60539, 60539,
 60539, 60539, 60539, 60539, 60589, 60589, 60607, 60611, 60623, 60631,
 60637, 60647, 60649, 60661, 60661, 60679, 60679, 60689, 60703, 60703,
 60719, 60727, 60733, 60737, 60737, 60757, 60763, 60773, 60779, 60779,
 60793, 60793, 60811, 60821, 60821, 60821, 60821, 60821, 60859, 60869,
 60869, 60887, 60889, 60901, 60901, 60919, 60923, 60923, 60943, 60943,
 60953, 60961, 60961, 60961, 60961, 60961, 61007, 61007, 61007, 61031,
 61031, 61043, 61051, 61057, 61057, 61057, 61057, 61091, 61099, 61099,
 61099, 61121, 61129, 61141, 61151, 61153, 61153, 61169, 61169, 61169,
 61169, 61169, 61211, 61223, 61231, 61231, 61231, 61253, 61261, 61261,
 61261, 61283, 61291, 61297, 61297, 61297, 61297, 61333, 61343, 61343,
 61357, 61363, 61363, 61381, 61381, 61381, 61403, 61409, 61417, 61417,
 61417, 61441, 61441, 61463, 61471, 61471, 61487, 61493, 61493, 61511,
 61519, 61519, 61519, 61543, 61547, 61559, 61561, 61561, 61583, 61583,
 61583, 61603, 61613, 61613, 61631, 61637, 61643, 61651, 61657, 61667,
 61673, 61687, 61687, 61703, 61703, 61717, 61723, 61729, 61729, 61751,
 61757, 61757, 61757, 61781, 61781, 61781, 61781, 61813, 61819, 61819,
 61837, 61843, 61843, 61861, 61871, 61879, 61879, 61879, 61879, 61909,
 61909, 61927, 61933, 61933, 61949, 61949, 61967, 61967, 61981, 61991,
 61991, 62003, 62011, 62017, 62017, 62039, 62047, 62053, 62057, 62071,
 62071, 62081, 62081, 62099, 62099, 62119, 62119, 62131, 62143, 62143,
 62143, 62143, 62171, 62171, 62191, 62191, 62207, 62213, 62219, 62219,
 62233, 62233, 62233, 62233, 62233, 62273, 62273, 62273, 62303, 62311,
 62311, 62327, 62327, 62327, 62351, 62351, 62351, 62351, 62383, 62383,
 62383, 62401, 62401, 62423, 62423, 62423, 62423, 62423, 62459, 62467,
 62477, 62483, 62483, 62501, 62507, 62507, 62507, 62533, 62539, 62549,
 62549, 62563, 62563, 62581, 62591, 62597, 62603, 62603, 62617, 62627,
 62639, 62639, 62653, 62659, 62659, 62659, 62687, 62687, 62701, 62701,
 62701, 62723, 62731, 62743, 62743, 62753, 62761, 62773, 62773, 62791,
 62791, 62801, 62801, 62819, 62827, 62827, 62827, 62851, 62861, 62869,
 62873, 62873, 62873, 62903, 62903, 62903, 62927, 62929, 62939, 62939,
 62939, 62939, 62971, 62983, 62989, 62989, 62989, 62989, 62989, 63031,
 63031, 63031, 63031, 63059, 63067, 63079, 63079, 63079, 63103, 63103,
 63113, 63127, 63131, 63131, 63149, 63149, 63149, 63149, 63179, 63179,
 63199, 63199, 63211, 63211, 63211, 63211, 63247, 63247, 63247, 63247,
 63277, 63281, 63281, 63299, 63311, 63317, 63317, 63331, 63337, 63347,
 63353, 63367, 63367, 63377, 63391, 63397, 63397, 63409, 63421, 63421,
 63439, 63443, 63443, 63463, 63467, 63473, 63487, 63493, 63499, 63499,
 63499, 63527, 63533, 63541, 63541, 63559, 63559, 63559, 63577, 63589,
 63599, 63607, 63611, 63617, 63629, 63629, 63647, 63649, 63659, 63671,
 63671, 63671, 63691, 63703, 63709, 63719, 63727, 63727, 63743, 63743,
 63743, 63761, 63773, 63781, 63781, 63799, 63803, 63809, 63823, 63823,
 63839, 63841, 63853, 63863, 63863, 63863, 63863, 63863, 63901, 63907,
 63913, 63913, 63929, 63929, 63949, 63949, 63949, 63949, 63977, 63977,
 63997, 64007, 64013, 64019, 64019, 64037, 64037, 64037, 64063, 64067,
 64067, 64081, 64091, 64091, 64109, 64109, 64123, 64123, 64123, 64151,
 64157, 64157, 64171, 64171, 64189, 64189, 64189, 64189, 64223, 64231,
 64237, 64237, 64237, 64237, 64271, 64279, 64283, 64283, 64303, 64303,
 64319, 64327, 64333, 64333, 64333, 64333, 64333, 64373, 64381, 64381,
 64399, 64403, 64403, 64403, 64403, 64439, 64439, 64453, 64453, 64453,
 64453, 64483, 64489, 64499, 64499, 64513, 64513, 64513, 64513, 64513,
 64553, 64567, 64567, 64579, 64591, 64591, 64601, 64613, 64621, 64627,
 64633, 64633, 64633, 64663, 64667, 64679, 64679, 64693, 64693, 64709,
 64717, 64717, 64717, 64717, 64747, 64747, 64763, 64763, 64783, 64783,
 64793, 64793, 64811, 64817, 64817, 64817, 64817, 64853, 64853, 64871,
 64879, 64879, 64891, 64901, 64901, 64919, 64927, 64927, 64937, 64951,
 64951, 64951, 64969, 64969, 64969, 64997, 65003, 65011, 65011, 65029,
 65033, 65033, 65053, 65063, 65071, 65071, 65071, 65089, 65101, 65111,
 65119, 65123, 65129, 65141, 65147, 65147, 65167, 65173, 65183, 65183,
 65183, 65203, 65213, 65213, 65213, 65239, 65239, 65239, 65257, 65269,
 65269, 65287, 65293, 65293, 65309, 65309, 65327, 65327, 65327, 65327,
 65357, 65357, 65371, 65381, 65381, 65393, 65407, 65413, 65423, 65423,
 65437, 65447, 65449, 65449, 65449, 65479, 65479, 65479, 65497, 65497,
 65519, 65521, 65521, 65543, 65551, 65557, 65563, 65563, 65581, 65587,
 65599, 65599, 65609, 65617, 65629, 65633, 65647, 65651, 65657, 65657,
 65677, 65687, 65687, 65701, 65707, 65719, 65719, 65731, 65731, 65731,
 65731, 65761, 65761, 65777, 65789, 65789, 65789, 65809, 65809, 65831,
 65839, 65843, 65851, 65851, 65867, 65867, 65881, 65881, 65899, 65899,
 65899, 65927, 65929, 65929, 65951, 65957, 65963, 65963, 65983, 65983,
 65993, 65993, 65993, 65993, 66029, 66037, 66047, 66047, 66047, 66071,
 66071, 66083, 66089, 66103, 66109, 66109, 66109, 66109, 66137, 66137,
 66137, 66161, 66173, 66179, 66191, 66191, 66191, 66191, 66221, 66221,
 66239, 66239, 66239, 66239, 66271, 66271, 66271, 66293, 66301, 66301,
 66301, 66301, 66301, 66343, 66347, 66359, 66361, 66373, 66383, 66383,
 66383, 66403, 66413, 66413, 66431, 66431, 66431, 66449, 66463, 66467,
 66467, 66467, 66491, 66499, 66509, 66509, 66523, 66533, 66541, 66541,
 66553, 66553, 66571, 66571, 66587, 66593, 66601, 66601, 66617, 66629,
 66629, 66643, 66653, 66653, 66653, 66653, 66683, 66683, 66701, 66701,
 66713, 66721, 66733, 66739, 66751, 66751, 66763, 66763, 66763, 66791,
 66797, 66797, 66809, 66821, 66821, 66821, 66841, 66853, 66863, 66863,
 66877, 66883, 66889, 66889, 66889, 66919, 66923, 66931, 66943, 66949,
 66959, 66959, 66973, 66977, 66977, 66977, 67003, 67003, 67021, 67021,
 67033, 67043, 67049, 67061, 67061, 67079, 67079, 67079, 67103, 67103,
 67103, 67121, 67129, 67141, 67141, 67157, 67157, 67169, 67181, 67189,
 67189, 67189, 67213, 67219, 67231, 67231, 67247, 67247, 67261, 67271,
 67273, 67273, 67289, 67289, 67307, 67307, 67307, 67307, 67343, 67349,
 67349, 67349, 67369, 67369, 67391, 67399, 67399, 67411, 67421, 67429,
 67433, 67447, 67453, 67453, 67453, 67477, 67481, 67493, 67499, 67511,
 67511, 67523, 67531, 67537, 67547, 67559, 67567, 67567, 67579, 67589,
 67589, 67607, 67607, 67619, 67631, 67631, 67631, 67651, 67651, 67651,
 67679, 67679, 67679, 67699, 67709, 67709, 67723, 67733, 67741, 67751,
 67759, 67763, 67763, 67783, 67789, 67789, 67807, 67807, 67819, 67829,
 67829, 67843, 67853, 67853, 67867, 67867, 67883, 67891, 67901, 67901,
 67901, 67927, 67933, 67943, 67943, 67957, 67967, 67967, 67979, 67987,
 67993, 67993, 67993, 68023, 68023, 68023, 68041, 68053, 68059, 68071,
 68071, 68087, 68087, 68099, 68111, 68113, 68113, 68113, 68141, 68147,
 68147, 68161, 68171, 68171, 68171, 68171, 68207, 68213, 68219, 68227,
 68239, 68239, 68239, 68261, 68261, 68279, 68281, 68281, 68281, 68311,
 68311, 68311, 68329, 68329, 68351, 68351, 68351, 68371, 68371, 68389,
 68399, 68399, 68399, 68399, 68399, 68437, 68447, 68449, 68449, 68449,
 68477, 68483, 68491, 68501, 68507, 68507, 68521, 68531, 68543, 68543,
 68543, 68567, 68567, 68581, 68581, 68597, 68597, 68611, 68611, 68611,
 68639, 68639, 68639, 68659, 68669, 68669, 68687, 68687, 68699, 68711,
 68713, 68713, 68729, 68743, 68749, 68749, 68767, 68771, 68777, 68791,
 68791, 68791, 68813, 68821, 68821, 68821, 68821, 68821, 68863, 68863,
 68879, 68881, 68891, 68903, 68909, 68917, 68927, 68927, 68927, 68947,
 68947, 68963, 68963, 68963, 68963, 68993, 69001, 69011, 69019, 69031,
 69031, 69031, 69031, 69061, 69067, 69073, 69073, 69073, 69073, 69109,
 69119, 69127, 69127, 69143, 69151, 69151, 69163, 69163, 69163, 69191,
 69197, 69203, 69203, 69221, 69221, 69239, 69247, 69247, 69263, 69263,
 69263, 69263, 69263, 69263, 69263, 69317, 69317, 69317, 69341, 69341,
 69341, 69341, 69371, 69383, 69389, 69389, 69403, 69403, 69403, 69431,
 69439, 69439, 69439, 69463, 69467, 69473, 69481, 69493, 69499, 69499,
 69499, 69499, 69499, 69539, 69539, 69557, 69557, 69557, 69557, 69557,
 69593, 69593, 69593, 69623, 69623, 69623, 69623, 69653, 69661, 69661,
 69677, 69677, 69691, 69697, 69709, 69709, 69709, 69709, 69739, 69739,
 69739, 69767, 69767, 69779, 69779, 69779, 69779, 69809, 69821, 69829,
 69833, 69847, 69847, 69859, 69859, 69877, 69877, 69877, 69899, 69911,
 69911, 69911, 69931, 69941, 69941, 69959, 69959, 69959, 69959, 69991,
 69997, 70003, 70009, 70019, 70019, 70039, 70039, 70051, 70061, 70067,
 70079, 70079, 70079, 70099, 70111, 70117, 70123, 70123, 70141, 70141,
 70157, 70163, 70163, 70183, 70183, 70199, 70207, 70207, 70223, 70229,
 70237, 70241, 70249, 70249, 70271, 70271, 70271, 70289, 70297, 70309,
 70313, 70327, 70327, 70327, 70351, 70351, 70351, 70373, 70381, 70381,
 70393, 70393, 70393, 70423, 70429, 70439, 70439, 70451, 70459, 70459,
 70459, 70487, 70489, 70501, 70507, 70507, 70507, 70529, 70537, 70549,
 70549, 70549, 70573, 70583, 70589, 70589, 70607, 70607, 70621, 70627,
 70639, 70639, 70639, 70663, 70667, 70667, 70687, 70687, 70687, 70709,
 70717, 70717, 70729, 70729, 70729, 70753, 70753, 70769, 70783, 70783,
 70793, 70793, 70793, 70823, 70823, 70823, 70843, 70853, 70853, 70867,
 70879, 70879, 70891, 70901, 70901, 70919, 70921, 70921, 70937, 70951,
 70957, 70957, 70969, 70981, 70991, 70999, 70999, 71011, 71023, 71023,
 71039, 71039, 71039, 71059, 71069, 71069, 71081, 71089, 71089, 71089,
 71119, 71119, 71129, 71143, 71147, 71153, 71167, 71171, 71171, 71191,
 71191, 71191, 71209, 71209, 71209, 71237, 71237, 71249, 71263, 71263,
 71263, 71287, 71293, 71293, 71293, 71317, 71327, 71333, 71341, 71347,
 71359, 71363, 71363, 71363, 71389, 71399, 71399, 71413, 71419, 71429,
 71437, 71443, 71453, 71453, 71471, 71479, 71483, 71483, 71503, 71503,
 71503, 71527, 71527, 71537, 71551, 71551, 71563, 71569, 71569, 71569,
 71597, 71597, 71597, 71597, 71597, 71633, 71647, 71647, 71663, 71671,
 71671, 71671, 71693, 71699, 71711, 71719, 71719, 71719, 71741, 71741,
 71741, 71761, 71761, 71777, 71789, 71789, 71807, 71809, 71821, 71821,
 71837, 71843, 71849, 71861, 71867, 71879, 71887, 71887, 71899, 71909,
 71917, 71917, 71933, 71941, 71947, 71947, 71963, 71971, 71983, 71987,
 71999, 71999, 71999, 72019, 72031, 72031, 72047, 72053, 72053, 72053,
 72077, 72077, 72091, 72103, 72109, 72109, 72109, 72109, 72139, 72139,
 72139, 72167, 72173, 72173, 72173, 72173, 72173, 72211, 72223, 72229,
 72229, 72229, 72253, 72253, 72271, 72277, 72287, 72287, 72287, 72307,
 72313, 72313, 72313, 72341, 72341, 72353, 72367, 72367, 72383, 72383,
 72383, 72383, 72383, 72421, 72431, 72431, 72431, 72431, 72461, 72469,
 72469, 72481, 72493, 72503, 72503, 72503, 72503, 72533, 72533, 72551,
 72559, 72559, 72559, 72577, 72577, 72577, 72577, 72613, 72623, 72623,
 72623, 72647, 72649, 72661, 72671, 72679, 72679, 72689, 72701, 72707,
 72719, 72727, 72733, 72739, 72739, 72739, 72767, 72767, 72767, 72767,
 72797, 72797, 72797, 72823, 72823, 72823, 72823, 72823, 72859, 72871,
 72871, 72883, 72893, 72901, 72911, 72911, 72923, 72931, 72937, 72949,
 72959, 72959, 72973, 72977, 72977, 72997, 72997, 73013, 73019, 73019,
 73039, 73043, 73043, 73063, 73063, 73079, 73079, 73091, 73091, 73091,
 73091, 73127, 73133, 73141, 73141, 73141, 73141, 73141, 73181, 73189,
 73189, 73189, 73189, 73189, 73189, 73237, 73243, 73243, 73259, 73259,
 73277, 73277, 73291, 73303, 73309, 73309, 73327, 73331, 73331, 73351,
 73351, 73363, 73369, 73379, 73387, 73387, 73387, 73387, 73421, 73421,
 73433, 73433, 73453, 73459, 73471, 73477, 73483, 73483, 73483, 73483,
 73517, 73523, 73529, 73529, 73547, 73553, 73561, 73571, 73583, 73589,
 73597, 73607, 73613, 73613, 73613, 73637, 73643, 73651, 73651, 73651,
 73679, 73681, 73693, 73699, 73709, 73709, 73727, 73727, 73727, 73751,
 73757, 73757, 73771, 73783, 73783, 73783, 73783, 73783, 73823, 73823,
 73823, 73847, 73849, 73859, 73867, 73877, 73883, 73883, 73897, 73907,
 73907, 73907, 73907, 73943, 73951, 73951, 73961, 73973, 73973, 73973,
 73999, 73999, 73999, 74021, 74027, 74027, 74047, 74051, 74051, 74071,
 74077, 74077, 74093, 74101, 74101, 74101, 74101, 74131, 74143, 74149,
 74159, 74167, 74167, 74177, 74189, 74197, 74203, 74209, 74219, 74231,
 74231, 74231, 74231, 74257, 74257, 74279, 74287, 74293, 74297, 74311,
 74317, 74323, 74323, 74323, 74323, 74357, 74363, 74363, 74383, 74383,
 74383, 74383, 74413, 74419, 74419, 74419, 74441, 74453, 74453, 74471,
 74471, 74471, 74489, 74489, 74509, 74509, 74527, 74531, 74531, 74551,
 74551, 74567, 74573, 74573, 74587, 74597, 74597, 74611, 74623, 74623,
 74623, 74623, 74653, 74653, 74653, 74653, 74687, 74687, 74699, 74707,
 74719, 74719, 74731, 74731, 74747, 74759, 74761, 74771, 74779, 74779,
 74797, 74797, 74797, 74821, 74831, 74831, 74843, 74843, 74861, 74869,
 74873, 74887, 74891, 74903, 74903, 74903, 74923, 74933, 74941, 74941,
 74959, 74959, 74959, 74959, 74959, 74959, 74959, 75013, 75017, 75029,
 75037, 75041, 75041, 75041, 75041, 75079, 75083, 75083, 75083, 75109,
 75109, 75109, 75133, 75133, 75149, 75149, 75167, 75169, 75181, 75181,
 75193, 75193, 75211, 75223, 75227, 75239, 75239, 75253, 75253, 75269,
 75277, 75277, 75289, 75289, 75307, 75307, 75323, 75329, 75337, 75347,
 75353, 75367, 75367, 75377, 75391, 75391, 75407, 75407, 75407, 75431,
 75437, 75437, 75437, 75437, 75437, 75479, 75479, 75479, 75503, 75511,
 75511, 75527, 75533, 75541, 75541, 75557, 75557, 75571, 75583, 75583,
 75583, 75583, 75611, 75619, 75629, 75629, 75641, 75653, 75659, 75659,
 75679, 75683, 75689, 75703, 75709, 75709, 75721, 75731, 75743, 75743,
 75743, 75767, 75773, 75781, 75787, 75797, 75797, 75797, 75821, 75821,
 75833, 75833, 75853, 75853, 75869, 75869, 75883, 75883, 75883, 75883,
 75913, 75913, 75931, 75941, 75941, 75941, 75967, 75967, 75983, 75991,
 75997, 76003, 76003, 76003, 76031, 76039, 76039, 76039, 76039, 76039,
 76079, 76081, 76091, 76103, 76103, 76103, 76123, 76129, 76129, 76147,
 76159, 76163, 76163, 76163, 76163, 76163, 76207, 76213, 76213, 76231,
 76231, 76243, 76253, 76261, 76261, 76261, 76283, 76289, 76303, 76303,
 76303, 76303, 76333, 76343, 76343, 76343, 76367, 76369, 76379, 76387,
 76387, 76403, 76403, 76423, 76423, 76423, 76441, 76441, 76463, 76471,
 76471, 76487, 76493, 76493, 76511, 76519, 76519, 76519, 76543, 76543,
 76543, 76561, 76561, 76579, 76579, 76597, 76607, 76607, 76607, 76631,
 76631, 76631, 76651, 76651, 76667, 76679, 76679, 76679, 76697, 76697,
 76717, 76717, 76733, 76733, 76733, 76757, 76757, 76771, 76781, 76781,
 76781, 76801, 76801, 76819, 76831, 76837, 76847, 76847, 76847, 76871,
 76873, 76883, 76883, 76883, 76907, 76919, 76919, 76919, 76943, 76949,
 76949, 76963, 76963, 76963, 76991, 76991, 77003, 77003, 77023, 77029,
 77029, 77047, 77047, 77047, 77069, 77069, 77081, 77093, 77101, 77101,
 77101, 77101, 77101, 77141, 77141, 77153, 77167, 77171, 77171, 77191,
 77191, 77201, 77213, 77213, 77213, 77239, 77243, 77249, 77263, 77269,
 77279, 77279, 77291, 77291, 77291, 77317, 77323, 77323, 77339, 77351,
 77359, 77359, 77369, 77383, 77383, 77383, 77383, 77383, 77419, 77431,
 77431, 77447, 77447, 77447, 77471, 77479, 77479, 77491, 77491, 77509,
 77513, 77527, 77527, 77543, 77551, 77557, 77563, 77573, 77573, 77591,
 77591, 77591, 77611, 77621, 77621, 77621, 77647, 77647, 77659, 77659,
 77659, 77687, 77689, 77699, 77711, 77719, 77723, 77731, 77743, 77747,
 77747, 77761, 77773, 77783, 77783, 77797, 77801, 77813, 77813, 77813,
 77839, 77839, 77849, 77863, 77867, 77867, 77867, 77893, 77899, 77899,
 77899, 77899, 77933, 77933, 77951, 77951, 77951, 77969, 77983, 77983,
 77999, 78007, 78007, 78017, 78031, 78031, 78041, 78049, 78059, 78059,
 78079, 78079, 78079, 78101, 78101, 78101, 78121, 78121, 78139, 78139,
 78157, 78167, 78173, 78179, 78191, 78193, 78203, 78203, 78203, 78229,
 78233, 78241, 78241, 78259, 78259, 78277, 78283, 78283, 78301, 78311,
 78317, 78317, 78317, 78341, 78347, 78347, 78367, 78367, 78367, 78367,
 78367, 78401, 78401, 78401, 78427, 78439, 78439, 78439, 78439, 78467,
 78479, 78487, 78487, 78497, 78511, 78517, 78517, 78517, 78541, 78541,
 78553, 78553, 78571, 78583, 78583, 78593, 78607, 78607, 78623, 78623,
 78623, 78643, 78653, 78653, 78653, 78653, 78653, 78691, 78697, 78707,
 78713, 78721, 78721, 78737, 78737, 78737, 78737, 78737, 78781, 78791,
 78797, 78803, 78809, 78823, 78823, 78839, 78839, 78853, 78857, 78857,
 78877, 78887, 78893, 78901, 78901, 78919, 78919, 78929, 78941, 78941,
 78941, 78941, 78941, 78979, 78989, 78989, 78989, 78989, 78989, 79031,
 79039, 79043, 79043, 79063, 79063, 79063, 79087, 79087, 79103, 79111,
 79111, 79111, 79133, 79139, 79151, 79159, 79159, 79159, 79181, 79187,
 79193, 79201, 79201, 79201, 79231, 79231, 79241, 79241, 79259, 79259,
 79279, 79283, 79283, 79301, 79309, 79319, 79319, 79333, 79337, 79349,
 79357, 79367, 79367, 79379, 79379, 79399, 79399, 79411, 79423, 79427,
 79433, 79433, 79451, 79451, 79451, 79451, 79481, 79493, 79493, 79493,
 79493, 79493, 79531, 79537, 79549, 79559, 79561, 79561, 79579, 79589,
 79589, 79601, 79613, 79621, 79631, 79633, 79633, 79633, 79657, 79669,
 79669, 79687, 79693, 79699, 79699, 79699, 79699, 79699, 79699, 79699,
 79757, 79757, 79769, 79777, 79777, 79777, 79801, 79813, 79823, 79829,
 79829, 79847, 79847, 79861, 79867, 79873, 79873, 79889, 79903, 79907,
 79907, 79907, 79907, 79943, 79943, 79943, 79967, 79973, 79979, 79987,
 79999, 79999, 79999, 80021, 80021, 80039, 80039, 80051, 80051, 80071,
 80077, 80077, 80077, 80077, 80111, 80111, 80111, 80111, 80141, 80149,
 80153, 80167, 80173, 80177, 80191, 80191, 80207, 80209, 80221, 80231,
 80239, 80239, 80251, 80263, 80263, 80279, 80287, 80287, 80287, 80309,
 80317, 80317, 80329, 80341, 80347, 80347, 80363, 80369, 80369, 80387,
 80387, 80407, 80407, 80407, 80429, 80429, 80447, 80449, 80449, 80471,
 80473, 80473, 80491, 80491, 80491, 80513, 80527, 80527, 80537, 80537,
 80557, 80567, 80567, 80567, 80567, 80599, 80603, 80611, 80621, 80629,
 80629, 80629, 80651, 80657, 80671, 80677, 80687, 80687, 80701, 80701,
 80713, 80713, 80713, 80737, 80749, 80749, 80761, 80761, 80783, 80789,
 80789, 80803, 80809, 80819, 80831, 80833, 80833, 80849, 80863, 80863,
 80863, 80863, 80863, 80897, 80911, 80917, 80923, 80933, 80933, 80933,
 80953, 80963, 80963, 80963, 80989, 80989, 81001, 81013, 81023, 81031,
 81031, 81047, 81049, 81049, 81071, 81077, 81083, 81083, 81101, 81101,
 81119, 81119, 81131, 81131, 81131, 81157, 81163, 81173, 81181, 81181,
 81199, 81203, 81203, 81223, 81223, 81239, 81239, 81239, 81239, 81239,
 81239, 81283, 81293, 81299, 81307, 81307, 81307, 81331, 81343, 81349,
 81359, 81359, 81373, 81373, 81373, 81373, 81401, 81409, 81421, 81421,
 81439, 81439, 81439, 81463, 81463, 81463, 81463, 81463, 81463, 81509,
 81517, 81527, 81533, 81533, 81551, 81559, 81563, 81569, 81569, 81569,
 81569, 81569, 81611, 81619, 81629, 81637, 81647, 81649, 81649, 81671,
 81677, 81677, 81689, 81703, 81707, 81707, 81727, 81727, 81737, 81749,
 81749, 81761, 81773, 81773, 81773, 81799, 81799, 81799, 81817, 81817,
 81839, 81847, 81853, 81853, 81869, 81869, 81883, 81883, 81901, 81901,
 81919, 81919, 81931, 81943, 81943, 81953, 81967, 81973, 81973, 81973,
 81973, 82007, 82013, 82021, 82031, 82039, 82039, 82051, 82051, 82067,
 82073, 82073, 82073, 82073, 82073, 82073, 82073, 82129, 82141, 82141,
 82153, 82163, 82171, 82183, 82189, 82193, 82207, 82207, 82223, 82231,
 82237, 82241, 82241, 82261, 82267, 82279, 82279, 82279, 82301, 82307,
 82307, 82307, 82307, 82339, 82351, 82351, 82361, 82373, 82373, 82387,
 82393, 82393, 82393, 82421, 82421, 82421, 82421, 82421, 82463, 82471,
 82471, 82487, 82493, 82499, 82507, 82507, 82507, 82531, 82531, 82549,
 82559, 82567, 82571, 82571, 82591, 82591, 82601, 82613, 82619, 82619,
 82633, 82633, 82651, 82657, 82657, 82657, 82657, 82657, 82699, 82699,
 82699, 82727, 82729, 82729, 82729, 82759, 82763, 82763, 82781, 82787,
 82799, 82799, 82813, 82813, 82813, 82837, 82847, 82847, 82847, 82847,
 82847, 82883, 82891, 82903, 82903, 82913, 82913, 82913, 82939, 82939,
 82939, 82963, 82963, 82981, 82981, 82997, 83003, 83009, 83023, 83023,
 83023, 83047, 83047, 83063, 83071, 83077, 83077, 83093, 83101, 83101,
 83117, 83117, 83117, 83137, 83137, 83137, 83137, 83137, 83177, 83177,
 83177, 83207, 83207, 83221, 83231, 83233, 83243, 83243, 83257, 83269,
 83273, 83273, 83273, 83299, 83311, 83311, 83311, 83311, 83341, 83341,
 83357, 83357, 83357, 83383, 83389, 83399, 83407, 83407, 83423, 83431,
 83437, 83443, 83449, 83459, 83471, 83477, 83477, 83477, 83497, 83497,
 83497, 83497, 83497, 83537, 83537, 83557, 83563, 83563, 83579, 83591,
 83597, 83597, 83609, 83621, 83621, 83639, 83641, 83653, 83663, 83663,
 83663, 83663, 83689, 83701, 83701, 83719, 83719, 83719, 83737, 83737,
 83737, 83761, 83773, 83777, 83791, 83791, 83791, 83813, 83813, 83813,
 83833, 83843, 83843, 83857, 83869, 83873, 83873, 83891, 83903, 83911,
 83911, 83921, 83933, 83939, 83939, 83939, 83939, 83969, 83983, 83987,
 83987, 83987, 84011, 84017, 84017, 84017, 84047, 84053, 84061, 84067,
 84067, 84067, 84089, 84089, 84089, 84089, 84127, 84131, 84143, 84143,
 84143, 84163, 84163, 84181, 84191, 84199, 84199, 84211, 84223, 84229,
 84239, 84247, 84247, 84263, 84263, 84263, 84263, 84263, 84299, 84307,
 84319, 84319, 84319, 84319, 84349, 84349, 84349, 84349, 84377, 84391,
 84391, 84407, 84407, 84421, 84431, 84437, 84443, 84449, 84463, 84467,
 84467, 84481, 84481, 84503, 84509, 84509, 84523, 84533, 84533, 84551,
 84559, 84559, 84559, 84559, 84589, 84589, 84589, 84589, 84589, 84631,
 84631, 84631, 84653, 84659, 84659, 84673, 84673, 84691, 84701, 84701,
 84719, 84719, 84731, 84737, 84751, 84751, 84761, 84761, 84761, 84787,
 84793, 84793, 84811, 84811, 84827, 84827, 84827, 84827, 84859, 84871,
 84871, 84871, 84871, 84871, 84871, 84919, 84919, 84919, 84919, 84947,
 84947, 84967, 84967, 84979, 84991, 84991, 84991, 85009, 85021, 85027,
 85037, 85037, 85049, 85061, 85061, 85061, 85087, 85093, 85103, 85109,
 85109, 85121, 85133, 85133, 85147, 85159, 85159, 85159, 85159, 85159,
 85199, 85201, 85213, 85223, 85229, 85237, 85247, 85247, 85259, 85259,
 85259, 85259, 85259, 85303, 85303, 85313, 85313, 85333, 85333, 85333,
 85333, 85363, 85369, 85381, 85381, 85381, 85381, 85411, 85411, 85429,
 85439, 85447, 85453, 85453, 85469, 85469, 85487, 85487, 85487, 85487,
 85517, 85523, 85531, 85531, 85549, 85549, 85549, 85571, 85577, 85577,
 85597, 85607, 85607, 85621, 85627, 85639, 85643, 85643, 85661, 85669,
 85669, 85669, 85691, 85703, 85711, 85717, 85717, 85733, 85733, 85751,
 85751, 85751, 85751, 85781, 85781, 85793, 85793, 85793, 85819, 85831,
 85837, 85847, 85853, 85853, 85853, 85853, 85853, 85889, 85903, 85909,
 85909, 85909, 85933, 85933, 85933, 85933, 85933, 85933, 85933, 85991,
 85999, 85999, 86011, 86017, 86029, 86029, 86029, 86029, 86029, 86069,
 86077, 86083, 86083, 86083, 86111, 86117, 86117, 86131, 86143, 86143,
 86143, 86161, 86171, 86183, 86183, 86197, 86201, 86209, 86209, 86209,
 86239, 86243, 86249, 86263, 86269, 86269, 86287, 86293, 86297, 86311,
 86311, 86323, 86323, 86341, 86351, 86357, 86357, 86371, 86381, 86389,
 86399, 86399, 86413, 86423, 86423, 86423, 86441, 86453, 86461, 86467,
 86477, 86477, 86491, 86501, 86509, 86509, 86509, 86533, 86539, 86539,
 86539, 86561, 86573, 86579, 86587, 86599, 86599, 86599, 86599, 86629,
 86629, 86629, 86629, 86629, 86629, 86677, 86677, 86693, 86693, 86711,
 86719, 86719, 86729, 86743, 86743, 86753, 86767, 86771, 86783, 86783,
 86783, 86783, 86813, 86813, 86813, 86837, 86843, 86851, 86861, 86869,
 86869, 86869, 86869, 86869, 86869, 86869, 86927, 86929, 86939, 86951,
 86959, 86959, 86969, 86981, 86981, 86993, 86993, 87013, 87013, 87013,
 87037, 87041, 87049, 87049, 87071, 87071, 87083, 87083, 87103, 87107,
 87119, 87121, 87133, 87133, 87151, 87151, 87151, 87151, 87181, 87187,
 87187, 87187, 87211, 87223, 87223, 87223, 87223, 87253, 87257, 87257,
 87277, 87281, 87293, 87299, 87299, 87317, 87323, 87323, 87337, 87337,
 87359, 87359, 87359, 87383, 87383, 87383, 87407, 87407, 87421, 87427,
 87433, 87443, 87443, 87443, 87443, 87473, 87481, 87491, 87491, 87511,
 87517, 87523, 87523, 87541, 87547, 87559, 87559, 87559, 87583, 87589,
 87589, 87589, 87613, 87623, 87631, 87631, 87643, 87649, 87649, 87671,
 87679, 87683, 87691, 87701, 87701, 87719, 87721, 87721, 87743, 87751,
 87751, 87767, 87767, 87767, 87767, 87797, 87803, 87811, 87811, 87811,
 87833, 87833, 87853, 87853, 87869, 87877, 87887, 87887, 87887, 87911,
 87917, 87917, 87931, 87943, 87943, 87959, 87961, 87973, 87977, 87991,
 87991, 88007, 88007, 88019, 88019, 88037, 88037, 88037, 88037, 88069,
 88079, 88079, 88093, 88093, 88093, 88117, 88117, 88129, 88129, 88129,
 88129, 88129, 88169, 88177, 88177, 88177, 88177, 88211, 88223, 88223,
 88237, 88241, 88241, 88261, 88261, 88261, 88261, 88289, 88301, 88301,
 88301, 88327, 88327, 88339, 88339, 88339, 88339, 88339, 88379, 88379,
 88397, 88397, 88411, 88423, 88427, 88427, 88427, 88427, 88463, 88471,
 88471, 88471, 88493, 88499, 88499, 88513, 88523, 88523, 88523, 88547,
 88547, 88547, 88547, 88547, 88591, 88591, 88607, 88609, 88609, 88609,
 88609, 88643, 88651, 88663, 88667, 88667, 88681, 88681, 88681, 88681,
 88681, 88721, 88729, 88741, 88747, 88747, 88747, 88771, 88771, 88789,
 88799, 88807, 88813, 88819, 88819, 88819, 88843, 88853, 88861, 88867,
 88873, 88883, 88883, 88903, 88903, 88919, 88919, 88919, 88937, 88951,
 88951, 88951, 88969, 88969, 88969, 88997, 89003, 89009, 89021, 89021,
 89021, 89041, 89051, 89057, 89071, 89071, 89087, 89087, 89101, 89107,
 89119, 89123, 89123, 89137, 89137, 89153, 89153, 89153, 89153, 89189,
 89189, 89203, 89213, 89213, 89231, 89237, 89237, 89237, 89261, 89269,
 89273, 89273, 89293, 89303, 89303, 89317, 89317, 89329, 89329, 89329,
 89329, 89363, 89371, 89381, 89387, 89399, 89399, 89413, 89417, 89431,
 89431, 89443, 89449, 89459, 89459, 89477, 89477, 89491, 89501, 89501,
 89519, 89527, 89533, 89533, 89533, 89533, 89567, 89567, 89567, 89591,
 89599, 89603, 89611, 89611, 89627, 89633, 89633, 89653, 89659, 89671,
 89671, 89681, 89689, 89689, 89689, 89689, 89689, 89689, 89689, 89689,
 89759, 89767, 89767, 89783, 89783, 89797, 89797, 89809, 89821, 89821,
 89839, 89839, 89849, 89849, 89867, 89867, 89867, 89891, 89899, 89909,
 89917, 89923, 89923, 89939, 89939, 89959, 89963, 89963, 89983, 89989,
 89989, 90007, 90011, 90023, 90031, 90031, 90031, 90053, 90059, 90071,
 90073, 90073, 90089, 90089, 90107, 90107, 90127, 90127, 90127, 90149,
 90149, 90163, 90173, 90173, 90191, 90199, 90203, 90203, 90217, 90227,
 90239, 90247, 90247, 90263, 90271, 90271, 90281, 90289, 90289, 90289,
 90313, 90313, 90313, 90313, 90313, 90359, 90359, 90373, 90379, 90379,
 90397, 90407, 90407, 90407, 90407, 90439, 90439, 90439, 90439, 90469,
 90473, 90481, 90481, 90499, 90511, 90511, 90527, 90533, 90533, 90547,
 90547, 90547, 90547, 90583, 90583, 90599, 90599, 90599, 90619, 90631,
 90631, 90647, 90647, 90659, 90659, 90679, 90679, 90679, 90703, 90709,
 90709, 90709, 90731, 90731, 90749, 90749, 90749, 90749, 90749, 90787,
 90793, 90803, 90803, 90823, 90823, 90833, 90847, 90847, 90863, 90863,
 90863, 90887, 90887, 90901, 90911, 90917, 90917, 90931, 90931, 90947,
 90947, 90947, 90971, 90977, 90989, 90997, 90997, 91009, 91019, 91019,
 91033, 91033, 91033, 91033, 91033, 91079, 91081, 91081, 91099, 91099,
 91099, 91127, 91129, 91141, 91151, 91159, 91163, 91163, 91183, 91183,
 91199, 91199, 91199, 91199, 91229, 91237, 91243, 91253, 91253, 91253,
 91253, 91283, 91291, 91303, 91309, 91309, 91309, 91331, 91331, 91331,
 91331, 91367, 91373, 91381, 91387, 91397, 91397, 91411, 91423, 91423,
 91433, 91433, 91453, 91463, 91463, 91463, 91463, 91493, 91499, 91499,
 91513, 91513, 91529, 91541, 91541, 91541, 91541, 91573, 91583, 91591,
 91591, 91591, 91591, 91621, 91631, 91639, 91639, 91639, 91639, 91639,
 91673, 91673, 91691, 91703, 91711, 91711, 91711, 91733, 91733, 91733,
 91757, 91757, 91771, 91781, 91781, 91781, 91807, 91813, 91823, 91823,
 91837, 91841, 91841, 91841, 91867, 91873, 91873, 91873, 91873, 91909,
 91909, 91921, 91921, 91943, 91951, 91957, 91967, 91969, 91969, 91969,
 91997, 92003, 92009, 92009, 92009, 92033, 92041, 92051, 92051, 92051,
 92077, 92083, 92083, 92083, 92111, 92119, 92119, 92119, 92143, 92143,
 92153, 92153, 92173, 92179, 92189, 92189, 92203, 92203, 92221, 92227,
 92237, 92243, 92251, 92251, 92269, 92269, 92269, 92269, 92297, 92311,
 92317, 92317, 92333, 92333, 92347, 92357, 92363, 92369, 92383, 92387,
 92399, 92401, 92413, 92419, 92431, 92431, 92431, 92431, 92461, 92467,
 92479, 92479, 92489, 92503, 92507, 92507, 92507, 92507, 92507, 92551,
 92557, 92567, 92569, 92581, 92581, 92593, 92593, 92593, 92623, 92627,
 92639, 92647, 92647, 92657, 92671, 92671, 92683, 92693, 92699, 92707,
 92717, 92723, 92723, 92737, 92737, 92753, 92767, 92767, 92779, 92791,
 92791, 92801, 92809, 92821, 92831, 92831, 92831, 92849, 92863, 92867,
 92867, 92867, 92893, 92899, 92899, 92899, 92927, 92927, 92941, 92951,
 92959, 92959, 92959, 92959, 92987, 92993, 93001, 93001, 93001, 93001,
 93001, 93047, 93053, 93059, 93059, 93077, 93083, 93089, 93103, 93103,
 93113, 93113, 93133, 93139, 93151, 93151, 93151, 93169, 93179, 93187,
 93199, 93199, 93199, 93199, 93229, 93239, 93241, 93253, 93263, 93263,
 93263, 93287, 93287, 93287, 93307, 93319, 93323, 93329, 93337, 93337,
 93337, 93337, 93371, 93383, 93383, 93383, 93407, 93407, 93419, 93427,
 93427, 93427, 93427, 93463, 93463, 93479, 93487, 93493, 93503, 93503,
 93503, 93523, 93529, 93529, 93529, 93559, 93563, 93563, 93581, 93581,
 93581, 93607, 93607, 93607, 93629, 93637, 93637, 93637, 93637, 93637,
 93637, 93683, 93683, 93703, 93703, 93719, 93719, 93719, 93739, 93739,
 93739, 93763, 93763, 93763, 93787, 93787, 93787, 93811, 93811, 93827,
 93827, 93827, 93851, 93851, 93871, 93871, 93887, 93893, 93901, 93911,
 93913, 93923, 93923, 93941, 93949, 93949, 93967, 93971, 93983, 93983,
 93997, 94007, 94009, 94009, 94009, 94033, 94033, 94049, 94063, 94063,
 94079, 94079, 94079, 94099, 94111, 94117, 94121, 94121, 94121, 94151,
 94153, 94153, 94169, 94169, 94169, 94169, 94207, 94207, 94219, 94229,
 94229, 94229, 94253, 94261, 94261, 94273, 94273, 94291, 94291, 94309,
 94309, 94327, 94331, 94343, 94351, 94351, 94351, 94351, 94379, 94379,
 94399, 94399, 94399, 94421, 94427, 94439, 94447, 94447, 94463, 94463,
 94477, 94483, 94483, 94483, 94483, 94513, 94513, 94531, 94543, 94547,
 94559, 94561, 94573, 94583, 94583, 94597, 94603, 94613, 94621, 94621,
 94621, 94621, 94651, 94651, 94651, 94651, 94687, 94693, 94693, 94709,
 94709, 94727, 94727, 94727, 94747, 94747, 94747, 94771, 94781, 94789,
 94793, 94793, 94811, 94823, 94823, 94837, 94847, 94849, 94849, 94849,
 94873, 94873, 94889, 94903, 94907, 94907, 94907, 94933, 94933, 94951,
 94951, 94961, 94961, 94961, 94961, 94999, 95003, 95009, 95021, 95027,
 95027, 95027, 95027, 95063, 95071, 95071, 95087, 95093, 95101, 95111,
 95111, 95111, 95131, 95143, 95143, 95153, 95153, 95153, 95177, 95191,
 95191, 95203, 95213, 95219, 95231, 95239, 95239, 95239, 95261, 95267,
 95279, 95287, 95287, 95287, 95311, 95317, 95327, 95327, 95339, 95339,
 95339, 95339, 95369, 95383, 95383, 95393, 95401, 95413, 95419, 95429,
 95429, 95443, 95443, 95461, 95471, 95479, 95483, 95483, 95483, 95507,
 95507, 95527, 95531, 95539, 95549, 95549, 95561, 95569, 95581, 95581,
 95597, 95603, 95603, 95621, 95629, 95633, 95633, 95651, 95651, 95651,
 95651, 95651, 95651, 95701, 95707, 95717, 95723, 95731, 95737, 95747,
 95747, 95747, 95773, 95783, 95791, 95791, 95803, 95813, 95819, 95819,
 95819, 95819, 95819, 95857, 95869, 95873, 95881, 95891, 95891, 95911,
 95917, 95923, 95929, 95929, 95947, 95959, 95959, 95971, 95971, 95989,
 95989, 96001, 96013, 96017, 96017, 96017, 96043, 96053, 96059, 96059,
 96079, 96079, 96079, 96097, 96097, 96097, 96097, 96097, 96137, 96149,
 96157, 96167, 96167, 96181, 96181, 96199, 96199, 96211, 96223, 96223,
 96233, 96233, 96233, 96263, 96269, 96269, 96281, 96293, 96293, 96293,
 96293, 96323, 96331, 96337, 96337, 96353, 96353, 96353, 96377, 96377,
 96377, 96401, 96401, 96419, 96431, 96431, 96443, 96451, 96461, 96469,
 96479, 96487, 96493, 96497, 96497, 96517, 96527, 96527, 96527, 96527,
 96557, 96557, 96557, 96581, 96589, 96589, 96601, 96601, 96601, 96601,
 96601, 96643, 96643, 96661, 96671, 96671, 96671, 96671, 96703, 96703,
 96703, 96703, 96731, 96739, 96749, 96757, 96763, 96769, 96779, 96787,
 96799, 96799, 96799, 96823, 96827, 96827, 96847, 96851, 96857, 96857,
 96857, 96857, 96893, 96893, 96911, 96911, 96911, 96931, 96931, 96931,
 96959, 96959, 96973, 96979, 96989, 96997, 97007, 97007, 97021, 97021,
 97039, 97039, 97039, 97039, 97039, 97073, 97081, 97081, 97103, 97103,
 97117, 97127, 97127, 97127, 97151, 97159, 97159, 97171, 97177, 97187,
 97187, 97187, 97213, 97213, 97231, 97231, 97241, 97241, 97259, 97259,
 97259, 97283, 97283, 97303, 97303, 97303, 97327, 97327, 97327, 97327,
 97327, 97367, 97373, 97381, 97387, 97397, 97397, 97397, 97423, 97429,
 97429, 97441, 97453, 97463, 97463, 97463, 97463, 97463, 97501, 97511,
 97511, 97523, 97523, 97523, 97549, 97553, 97561, 97571, 97583, 97583,
 97583, 97607, 97613, 97613, 97613, 97613, 97613, 97651, 97651, 97651,
 97673, 97687, 97687, 97687, 97711, 97711, 97711, 97729, 97729, 97729,
 97729, 97729, 97771, 97777, 97789, 97789, 97789, 97813, 97813, 97829,
 97829, 97847, 97849, 97861, 97871, 97879, 97883, 97883, 97883, 97883,
 97919, 97927, 97931, 97943, 97943, 97943, 97967, 97973, 97973, 97987,
 97987, 97987, 98011, 98017, 98017, 98017, 98047, 98047, 98057, 98057,
 98057, 98081, 98081, 98101, 98101, 98101, 98123, 98129, 98143, 98143,
 98143, 98143, 98143, 98179, 98179, 98179, 98207, 98213, 98221, 98227,
 98227, 98227, 98251, 98257, 98269, 98269, 98269, 98269, 98299, 98299,
 98317, 98327, 98327, 98327, 98347, 98347, 98347, 98369, 98377, 98389,
 98389, 98407, 98411, 98419, 98429, 98429, 98443, 98453, 98459, 98467,
 98479, 98479, 98491, 98491, 98507, 98519, 98519, 98533, 98543, 98543,
 98543, 98563, 98573, 98573, 98573, 98597, 98597, 98597, 98621, 98627,
 98639, 98641, 98641, 98663, 98669, 98669, 98669, 98689, 98689, 98711,
 98717, 98717, 98731, 98737, 98737, 98737, 98737, 98773, 98779, 98779,
 98779, 98807, 98809, 98809, 98809, 98837, 98837, 98849, 98849, 98869,
 98873, 98887, 98893, 98899, 98911, 98911, 98927, 98929, 98939, 98947,
 98953, 98963, 98963, 98981, 98981, 98999, 98999, 99013, 99023, 99023,
 99023, 99041, 99053, 99053, 99053, 99079, 99083, 99089, 99103, 99109,
 99119, 99119, 99133, 99139, 99149, 99149, 99149, 99173, 99181, 99191,
 99191, 99191, 99191, 99223, 99223, 99233, 99241, 99251, 99259, 99259,
 99277, 99277, 99289, 99289, 99289, 99317, 99317, 99317, 99317, 99349,
 99349, 99367, 99371, 99377, 99391, 99397, 99401, 99409, 99409, 99431,
 99439, 99439, 99439, 99439, 99469, 99469, 99487, 99487, 99497, 99497,
 99497, 99527, 99529, 99529, 99551, 99559, 99563, 99571, 99581, 99581,
 99581, 99607, 99611, 99623, 99623, 99623, 99643, 99643, 99661, 99667,
 99679, 99679, 99689, 99689, 99709, 99719, 99721, 99733, 99733, 99733,
 99733, 99767, 99767, 99767, 99787, 99793, 99793, 99809, 99823, 99829,
 99839, 99839, 99839, 99859, 99871, 99877, 99881, 99881, 99901, 99907,
 99907, 99923, 99929, 99929, 99929, 99929, 99961, 99971
};


static const unsigned ByteSizePrimesCount=sizeof(ByteSizePrimes)/sizeof(ByteSizePrimes[0]);


static unsigned
Bits2PrimeNBytes(
    unsigned Bits,
    unsigned & BytesOut)
{
    unsigned prime, bytes, temp;

    prime=0;
    BytesOut=0;

    if (0 < Bits)
    {
        bytes=(Bits+7)/8;
        do
        {
            if (bytes<ByteSizePrimesCount)
            {
                temp=ByteSizePrimes[bytes];
                if (Bits<=temp)
                    prime=temp;
                else
                    ++bytes;
            }   // if
            else
                prime=bytes*8;

        } while(0==prime);

        BytesOut=bytes;
    }   // if

    return(prime);

}   // Bits2PrimeNBytes


static unsigned
Bytes2Prime(
    unsigned Bytes)
{
    unsigned prime;

    prime=0;
    if (Bytes<ByteSizePrimesCount)
    {
        prime=ByteSizePrimes[Bytes];
    }   // if
    else
    {
        prime=Bytes*8;
    }   // else

    return(prime);

}   // Bytes2Prime


}  // namespace leveldb
