<?php

/**
 * SPDX-License-Identifier: MIT
 * Copyright (c) 2017-2018 Tobias Reich
 * Copyright (c) 2018-2025 LycheeOrg.
 */

namespace App\Actions\Albums;

use App\Contracts\Exceptions\InternalLycheeException;
use App\DTO\AlbumSortingCriterion;
use App\DTO\TopAlbumDTO;
use App\Enum\ColumnSortingType;
use App\Enum\OrderSortingType;
use App\Exceptions\ConfigurationKeyMissingException;
use App\Exceptions\Internal\InvalidOrderDirectionException;
use App\Factories\AlbumFactory;
use App\Models\Album;
use App\Models\Builders\AlbumBuilder;
use App\Models\Configs;
use App\Models\Extensions\SortingDecorator;
use App\Models\TagAlbum;
use App\Policies\AlbumPolicy;
use App\Policies\AlbumQueryPolicy;
use App\SmartAlbums\BaseSmartAlbum;
use Illuminate\Support\Collection as BaseCollection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

class Top
{
	private AlbumSortingCriterion $sorting;

	/**
	 * @throws InvalidOrderDirectionException
	 * @throws ConfigurationKeyMissingException
	 */
	public function __construct(
		private AlbumFactory $album_factory,
		private AlbumQueryPolicy $album_query_policy,
	) {
		$this->sorting = AlbumSortingCriterion::createDefault();
	}

	/**
	 * Returns the top-level albums (but not tag albums) visible
	 * to the current user.
	 *
	 * If the user is authenticated, then the result differentiates between
	 * albums which are owned by the user and "shared" albums which the
	 * user does not own, but is allowed to see.
	 * The term "shared album" might be a little misleading here.
	 * Albums which are owned by the user himself may also be shared (with
	 * other users.)
	 * Actually, in this context "shared albums" means "foreign albums".
	 *
	 * Note, the result may include password-protected albums that are not
	 * accessible (but are visible).
	 *
	 * @return TopAlbumDTO
	 *
	 * @throws InternalLycheeException
	 */
	public function get(): TopAlbumDTO
	{
		// Do not eagerly load the relation `photos` for each smart album.
		// On the albums overview, we only need a thumbnail for each album.
		/** @var BaseCollection<int,BaseSmartAlbum> $smart_albums */
		$smart_albums = $this->album_factory
			->getAllBuiltInSmartAlbums(false)
			->filter(fn ($smart_album) => Gate::check(AlbumPolicy::CAN_SEE, $smart_album));

		$tag_album_query = $this->album_query_policy
			->applyVisibilityFilter(TagAlbum::query()->with(['access_permissions', 'owner']));

		/** @var BaseCollection<int,TagAlbum> $tag_albums */
		$tag_albums = (new SortingDecorator($tag_album_query))
			->orderBy($this->sorting->column, $this->sorting->order)
			->get();

		$pinned_album_query = $this->album_query_policy
			->applyVisibilityFilter(Album::query()->with(['access_permissions', 'owner'])
			->joinSub(DB::table('base_albums')->select(['id', 'is_pinned'])->where('is_pinned', '=', true), 'pinned', 'pinned.id', '=', 'albums.id'));

		/** @var BaseCollection<int,Album> $pinned_albums */
		$pinned_albums = (new SortingDecorator($pinned_album_query))
			->orderBy(
				Configs::getValueAsEnum('sorting_pinned_albums_col', ColumnSortingType::class),
				Configs::getValueAsEnum('sorting_pinned_albums_order', OrderSortingType::class)
			)
			->get();

		/** @var AlbumBuilder $query */
		$query = $this->album_query_policy
			->applyVisibilityFilter(Album::query()->with(['access_permissions', 'owner'])->whereIsRoot()
			->when(
				Configs::getValueAsBool('deduplicate_pinned_albums'),
				fn ($q) => $q
					->joinSub(DB::table('base_albums')->select(['id', 'is_pinned'])->where('is_pinned', '=', false), 'not_pinned', 'not_pinned.id', '=', 'albums.id')
			));

		$user_id = Auth::id();
		if ($user_id !== null) {
			// For authenticated users we group albums by ownership.
			$albums = (new SortingDecorator($query))
				->orderBy(ColumnSortingType::OWNER_ID, OrderSortingType::ASC)
				->orderBy($this->sorting->column, $this->sorting->order)
				->get();

			list($a, $b) = $albums->partition(fn ($album) => $album->owner_id === $user_id);

			return new TopAlbumDTO(
				smart_albums: $smart_albums,
				tag_albums: $tag_albums,
				pinned_albums: $pinned_albums,
				albums: $a->values(),
				shared_albums: $b->values());
		} else {
			// For anonymous users we don't want to implicitly expose
			// ownership via sorting.
			/** @var BaseCollection<int,Album> */
			$albums = (new SortingDecorator($query))
				->orderBy($this->sorting->column, $this->sorting->order)
				->get();

			return new TopAlbumDTO(
				smart_albums: $smart_albums,
				tag_albums: $tag_albums,
				pinned_albums: $pinned_albums,
				albums: $albums);
		}
	}
}