<?php

/**
 * SPDX-License-Identifier: MIT
 * Copyright (c) 2017-2018 Tobias Reich
 * Copyright (c) 2018-2025 LycheeOrg.
 */

namespace App\Enum;

use App\Enum\Traits\DecorateBackedEnum;

/**
 * Enum OmnipayProviderType.
 *
 * Available providers
 */
enum OmnipayProviderType: string
{
	use DecorateBackedEnum;

	case DUMMY = 'Dummy';
	case MOLLIE = 'Mollie';
	case PAYPAL_EXPRESS = 'PayPal_Express';
	case PAYPAL_EXPRESSINCONTEXT = 'PayPal_ExpressInContext';
	case PAYPAL_PRO = 'PayPal_Pro';
	case PAYPAL_REST = 'PayPal_Rest';
	case STRIPE = 'Stripe';

	/**
	 * Determine if the provider is allowed in the current environment.
	 *
	 * @return bool
	 */
	public function isAllowed(): bool
	{
		if ($this === self::DUMMY) {
			return config('app.env', 'production') !== 'production';
		}

		return true;
	}

	/**
	 * Get the required configuration keys for the given provider.
	 *
	 * @return string[]
	 */
	public function requiredKeys(): array
	{
		return match ($this) {
			OmnipayProviderType::DUMMY => ['apiKey'],
			OmnipayProviderType::MOLLIE => ['apiKey', 'profileId'],
			OmnipayProviderType::STRIPE => ['apiKey', 'publishableKey'],
			OmnipayProviderType::PAYPAL_REST => ['clientId', 'secret'],
			OmnipayProviderType::PAYPAL_EXPRESS,
			OmnipayProviderType::PAYPAL_PRO,
			OmnipayProviderType::PAYPAL_EXPRESSINCONTEXT => ['username', 'password', 'signature'],
		};
	}
}
