//
//  Logo.swift
//  ApiCore
//
//  Created by Ondrej Rafaj on 19/05/2018.
//

import Foundation
import ErrorsCore
import ImageCore


/// System logo creator
public struct Logo {
    
    /// Logo error
    public enum Error: FrontendError {
        case invalidSize
        case imageCorrupted
        case imageTooSmall(Size)
        case imageNotSquare(Size)
        
        /// Error HTTP status
        public var status: HTTPStatus {
            return .preconditionFailed
        }
        
        /// Error code
        public var identifier: String {
            switch self {
            case .invalidSize:
                return "logo.invalid_size"
            case .imageCorrupted:
                return "logo.image_corrupted"
            case .imageTooSmall:
                return "logo.image_too_small"
            case .imageNotSquare:
                return "logo.image_not_square"
            }
        }
        
        /// Reason for failure
        public var reason: String {
            switch self {
            case .invalidSize:
                return "Invalid image size; Available sizes are 16, 64, 128, 192, 256 & 512 px"
            case .imageCorrupted:
                return "Image seems to be corrupted"
            case .imageTooSmall(let size):
                return "Image needs to be at least 512x512 px large; uploaded image is \(size.toString())"
            case .imageNotSquare(let size):
                return "Image needs to be square; uploaded image is \(size.toString())"
            }
        }
        
    }
    
    /// Logo data
    public static var data: Data {
        guard let data = Data(base64Encoded: base64, options: .ignoreUnknownCharacters) else {
            fatalError("Invalid icon data!")
        }
        return data
    }
    
    /// Logo as a BASE64 encoded string
    public static var base64: String = """
        iVBORw0KGgoAAAANSUhEUgAAAgAAAAIACAYAAAD0eNT6AAAACXBIWXMAAAsTAAAL
        EwEAmpwYAAALcmlUWHRYTUw6Y29tLmFkb2JlLnhtcAAAAAAAPD94cGFja2V0IGJl
        Z2luPSLvu78iIGlkPSJXNU0wTXBDZWhpSHpyZVN6TlRjemtjOWQiPz4gPHg6eG1w
        bWV0YSB4bWxuczp4PSJhZG9iZTpuczptZXRhLyIgeDp4bXB0az0iQWRvYmUgWE1Q
        IENvcmUgNS42LWMxNDAgNzkuMTYwNDUxLCAyMDE3LzA1LzA2LTAxOjA4OjIxICAg
        ICAgICAiPiA8cmRmOlJERiB4bWxuczpyZGY9Imh0dHA6Ly93d3cudzMub3JnLzE5
        OTkvMDIvMjItcmRmLXN5bnRheC1ucyMiPiA8cmRmOkRlc2NyaXB0aW9uIHJkZjph
        Ym91dD0iIiB4bWxuczp4bXA9Imh0dHA6Ly9ucy5hZG9iZS5jb20veGFwLzEuMC8i
        IHhtbG5zOnBob3Rvc2hvcD0iaHR0cDovL25zLmFkb2JlLmNvbS9waG90b3Nob3Av
        MS4wLyIgeG1sbnM6ZGM9Imh0dHA6Ly9wdXJsLm9yZy9kYy9lbGVtZW50cy8xLjEv
        IiB4bWxuczp4bXBNTT0iaHR0cDovL25zLmFkb2JlLmNvbS94YXAvMS4wL21tLyIg
        eG1sbnM6c3RFdnQ9Imh0dHA6Ly9ucy5hZG9iZS5jb20veGFwLzEuMC9zVHlwZS9S
        ZXNvdXJjZUV2ZW50IyIgeG1sbnM6c3RSZWY9Imh0dHA6Ly9ucy5hZG9iZS5jb20v
        eGFwLzEuMC9zVHlwZS9SZXNvdXJjZVJlZiMiIHhtbG5zOnRpZmY9Imh0dHA6Ly9u
        cy5hZG9iZS5jb20vdGlmZi8xLjAvIiB4bWxuczpleGlmPSJodHRwOi8vbnMuYWRv
        YmUuY29tL2V4aWYvMS4wLyIgeG1wOkNyZWF0b3JUb29sPSJBZG9iZSBQaG90b3No
        b3AgQ0MgMjAxNyAoTWFjaW50b3NoKSIgeG1wOkNyZWF0ZURhdGU9IjIwMTctMDct
        MTBUMjE6NDM6MzgrMDE6MDAiIHhtcDpNZXRhZGF0YURhdGU9IjIwMTgtMDUtMTlU
        MjE6NTE6NTArMDI6MDAiIHhtcDpNb2RpZnlEYXRlPSIyMDE4LTA1LTE5VDIxOjUx
        OjUwKzAyOjAwIiBwaG90b3Nob3A6Q29sb3JNb2RlPSIzIiBwaG90b3Nob3A6SUND
        UHJvZmlsZT0ic1JHQiBJRUM2MTk2Ni0yLjEiIGRjOmZvcm1hdD0iaW1hZ2UvcG5n
        IiB4bXBNTTpJbnN0YW5jZUlEPSJ4bXAuaWlkOjM1NjZmNzVkLTY5NTctNDI2Ny05
        NmE1LWYzZDZiOWFkYTg4MiIgeG1wTU06RG9jdW1lbnRJRD0iYWRvYmU6ZG9jaWQ6
        cGhvdG9zaG9wOjYwNjQ2NDlkLTczMzgtMmM0Yy1hY2I1LWU3N2IzNDFmMGZhOCIg
        eG1wTU06T3JpZ2luYWxEb2N1bWVudElEPSJ4bXAuZGlkOmVmMTI2MzMyLWNmZDct
        NDg2Mi05MDE2LTg3YjZiZDc0OTUxOSIgdGlmZjpPcmllbnRhdGlvbj0iMSIgdGlm
        ZjpYUmVzb2x1dGlvbj0iNzIwMDAwLzEwMDAwIiB0aWZmOllSZXNvbHV0aW9uPSI3
        MjAwMDAvMTAwMDAiIHRpZmY6UmVzb2x1dGlvblVuaXQ9IjIiIGV4aWY6Q29sb3JT
        cGFjZT0iMSIgZXhpZjpQaXhlbFhEaW1lbnNpb249IjIwNDgiIGV4aWY6UGl4ZWxZ
        RGltZW5zaW9uPSIyMDQ4Ij4gPHhtcE1NOkhpc3Rvcnk+IDxyZGY6U2VxPiA8cmRm
        OmxpIHN0RXZ0OmFjdGlvbj0iY3JlYXRlZCIgc3RFdnQ6aW5zdGFuY2VJRD0ieG1w
        LmlpZDplZjEyNjMzMi1jZmQ3LTQ4NjItOTAxNi04N2I2YmQ3NDk1MTkiIHN0RXZ0
        OndoZW49IjIwMTctMDctMTBUMjE6NDM6MzgrMDE6MDAiIHN0RXZ0OnNvZnR3YXJl
        QWdlbnQ9IkFkb2JlIFBob3Rvc2hvcCBDQyAyMDE3IChNYWNpbnRvc2gpIi8+IDxy
        ZGY6bGkgc3RFdnQ6YWN0aW9uPSJzYXZlZCIgc3RFdnQ6aW5zdGFuY2VJRD0ieG1w
        LmlpZDplZjgxZjU4Zi02NzY0LTRkMDQtYWY3My05NDkyZTVlMDg1MjIiIHN0RXZ0
        OndoZW49IjIwMTctMDctMTFUMDA6MTQ6NTgrMDE6MDAiIHN0RXZ0OnNvZnR3YXJl
        QWdlbnQ9IkFkb2JlIFBob3Rvc2hvcCBDQyAyMDE3IChNYWNpbnRvc2gpIiBzdEV2
        dDpjaGFuZ2VkPSIvIi8+IDxyZGY6bGkgc3RFdnQ6YWN0aW9uPSJzYXZlZCIgc3RF
        dnQ6aW5zdGFuY2VJRD0ieG1wLmlpZDo4OGJiMjg0My1iMDM0LTQyZTQtOWFiMS0y
        NTUzMjcwYjBhYjEiIHN0RXZ0OndoZW49IjIwMTctMDctMTFUMDA6MTU6MTcrMDE6
        MDAiIHN0RXZ0OnNvZnR3YXJlQWdlbnQ9IkFkb2JlIFBob3Rvc2hvcCBDQyAyMDE3
        IChNYWNpbnRvc2gpIiBzdEV2dDpjaGFuZ2VkPSIvIi8+IDxyZGY6bGkgc3RFdnQ6
        YWN0aW9uPSJjb252ZXJ0ZWQiIHN0RXZ0OnBhcmFtZXRlcnM9ImZyb20gYXBwbGlj
        YXRpb24vdm5kLmFkb2JlLnBob3Rvc2hvcCB0byBpbWFnZS9wbmciLz4gPHJkZjps
        aSBzdEV2dDphY3Rpb249ImRlcml2ZWQiIHN0RXZ0OnBhcmFtZXRlcnM9ImNvbnZl
        cnRlZCBmcm9tIGFwcGxpY2F0aW9uL3ZuZC5hZG9iZS5waG90b3Nob3AgdG8gaW1h
        Z2UvcG5nIi8+IDxyZGY6bGkgc3RFdnQ6YWN0aW9uPSJzYXZlZCIgc3RFdnQ6aW5z
        dGFuY2VJRD0ieG1wLmlpZDo2YzA5MzZjZC00YTNiLTQ0MzctYWU3ZC04ODE1YmI4
        OGNjMDgiIHN0RXZ0OndoZW49IjIwMTctMDctMTFUMDA6MTU6MTcrMDE6MDAiIHN0
        RXZ0OnNvZnR3YXJlQWdlbnQ9IkFkb2JlIFBob3Rvc2hvcCBDQyAyMDE3IChNYWNp
        bnRvc2gpIiBzdEV2dDpjaGFuZ2VkPSIvIi8+IDxyZGY6bGkgc3RFdnQ6YWN0aW9u
        PSJzYXZlZCIgc3RFdnQ6aW5zdGFuY2VJRD0ieG1wLmlpZDozNTY2Zjc1ZC02OTU3
        LTQyNjctOTZhNS1mM2Q2YjlhZGE4ODIiIHN0RXZ0OndoZW49IjIwMTgtMDUtMTlU
        MjE6NTE6NTArMDI6MDAiIHN0RXZ0OnNvZnR3YXJlQWdlbnQ9IkFkb2JlIFBob3Rv
        c2hvcCBDQyAoTWFjaW50b3NoKSIgc3RFdnQ6Y2hhbmdlZD0iLyIvPiA8L3JkZjpT
        ZXE+IDwveG1wTU06SGlzdG9yeT4gPHhtcE1NOkRlcml2ZWRGcm9tIHN0UmVmOmlu
        c3RhbmNlSUQ9InhtcC5paWQ6ODhiYjI4NDMtYjAzNC00MmU0LTlhYjEtMjU1MzI3
        MGIwYWIxIiBzdFJlZjpkb2N1bWVudElEPSJ4bXAuZGlkOmVmMTI2MzMyLWNmZDct
        NDg2Mi05MDE2LTg3YjZiZDc0OTUxOSIgc3RSZWY6b3JpZ2luYWxEb2N1bWVudElE
        PSJ4bXAuZGlkOmVmMTI2MzMyLWNmZDctNDg2Mi05MDE2LTg3YjZiZDc0OTUxOSIv
        PiA8L3JkZjpEZXNjcmlwdGlvbj4gPC9yZGY6UkRGPiA8L3g6eG1wbWV0YT4gPD94
        cGFja2V0IGVuZD0iciI/PjFf2eEAAKKfSURBVHic7X35k2RXld6XS+370tXV1a2l
        1VJrsRa0IRgYSYGQI4AZQxCDB8YmBocJ+2+Y/8Jhm7EhjG2YYGZsIMLMTAyCQKyD
        hDZAC1pGG+q9q6tr36sy0z80r7h1+973Xmae79yX1feL6MisrOp77rvLWb5z7s3S
        Jz/5ybMAhuBBo9EAAJRKpb33JnyfFxGNRgOlUmnvfZFQpL65+hJynn3rzvc7adma
        soD9z8Ye97Tn05CtJcuWx2xf8lnM+XF9JiHHlsFYdyF0B2v/sHSixn635K1UAczk
        /GNyd+Th63NRnsWc8CI5Ui7lEmoD+/rE6o+tDDWMfyIzWQNaTkeWHA0jxpxPRUV6
        1fxJwWWY7fcSMlzOpyS0nFqNdWXKkkQyRuy9b6ydoTKAFZcw10JOFrn9d0VD2iYs
        Un9N45/8bI6x1AZvBaZS05JnvgJXKz+NuTM3n/mPDbYCziObBdeYMuXaY2m/l5Sh
        YWh87bOcJsae15h3EyF1Zytgj4/LvgNYKScf2APmGkAtZdgOzM1vG7AiLwqXwdE2
        Ai6lptUHl/dbhLWmsWZc6RZpFMEhZjuULsaKHdHaTqo07H3AoOVtZ59FmbPB0ptp
        60oarGfw2cGy/Ye+CKxToEFptgt7jIvQL81oPwusKCRNnuu9Rg7cZagYMn1ROBM+
        p9LVLwm42EmN+WOuVUk20DX2LOfCbFNjL3siXJF27dQOK/fPHifXfqiav3S9B8LQ
        knmRlhcrYn+B4hh9QL6gqB35CbTGxxX5aOUP7X4kKNLaaBe24mQ75exx1GZRXLqt
        HVlpBob1DCHWs6RMHxMuBaYD5kuFmXLKyR+6KHOzoaIqJVdxmIv6L0p0W0S4IlEt
        aEdtLtkuI8WmqkM4PqZMLSpeq55Fg7lM05ES0HKWtNa6xl7WkiG9jpmpnWbaKwMH
        I+pIi/yL4sDY/SiCU5LmHbLhmifNmg2fAmSPg6/Qk2382TJseWx62ZSjUTDKTF8k
        7ZmFwXaRsARcTILEWLnmm7WXs9ay5PNkpbCkwNB15hry9b+clTspgpHKgmvDh4po
        XTD74lP42nmytDHTgM/4a6Uk0hRrKCPJaL+V32n1QQIaBZRm+1lRuoQMphw2O2KC
        5QCaDgzLsWDr47SxagdZOt7GvhQAk3aShiZl2y7yeNwa48zOZzUL0zsNlXvXZoe0
        ZfrSDSw5NjRSKZKyXGtQa/xsuZ2KUIFEu3PkCs4Y885su1n2slzkAj8fXDkZ36CG
        ei6fwS9Kf0LWRbgUqmtOmfI15JjyQqehQkXiTOo/keOq5WgVvjUoaWh8xp65/l0M
        hlS77LWVl4aXov6l23S1xxi3VtboVccAm20gJDqN+g8dbdsI0Z88jogGLa6ZcglV
        kKq5BzTSST4Fx6bi037fCnzrvRNZAFcRNlsWAyxHRoupbnUNeR2Aoi7CtHSF/Xch
        4DL6oftivtfMc7ngqoPQVHyS0WIzshJoRUydSsOmwdxPjDXjalfjebTAksVmezT2
        LLM2InktQsRv46oiwKIia3MWIeJPwIgaWoG5cViFM60gFPMQwjCGnndNeZ0O85gf
        w3nXjvCZ609jP2WlShiyGGClgCVYMC8DYAsIAddDha7czoJWVXIeJBvUd9RPa9Gb
        fTFx0AurXMcL2dBed/a8smlUrWJGxjhqHfm022fI0l7brOfRcGI0x6ZZpDoARTGq
        JtgLu1W46JhQEaBLeWlT/y4ZPupfow8hHDGNvLjZrvmqHZlpUeUsB8MeMyaLYjNz
        UrD3uMZRNmaas8iGs5n2pZ1Jyb2Q6gCEhm8ThoisfLBTE0VzSkIXH9oI1ZfQbAcT
        WqcnbHl5P28VWoyRTf1LI40ul6aEzT2v4Syx8tqmHNacsIsKGQGPq912sM8BCE35
        J8haYFqRVR64FEcRDK65UEJU2Zs/azsi2g6ZyyHVpPvt9yHkMcZas8Kcbdg0WUFm
        Goat51y1SkwngwkGwyPNau5zAIpguAC/Ai2Kg2JCiwbNA1c+NnQaIoRcLeo/7Rm1
        lIuGExDC0WDRpy45Wqko5hEwc5wYDqhtmBmpC588STmhHeV2YDuRBzIFkBVlFAUu
        Q1s0FMEhcb0P1QcteSFYBw1ZZs5XQyZj/dgOO9tBtJU1w3Caz8FyamyHTHo+2AwM
        wL9oie2AM2QUygHwbfjQhiwNRaP+Ad3LZlyyQ4+DFs2qlVrJareTqXEXmGuIbSh9
        8kyZjLaT9ll1GCw9p6mrNJwLJmPFQCEcgDRjH9qguGT7KKuQMBeeS7lp909bnmvj
        MSlXTUPiyh+zqfHkVSNKTl619rmWHLMojwEWw2brjYPAaIVgAovUjg9Vaus5YW8U
        18IO5Qj4vGxbYWn2LU2p+PqhGRGHdtoSFKEPktDK+ZswnQBm5GTKY8BHlzNl+X6W
        gksHSMhyOS6slEIijwUNvacBSYfCHO/gDEDagxWNXmd6880gS2lq9tEVuYXM92tG
        Kxp0n2ZknMDlaLDTDGymwXRitBgbafjYSEa06arGlwQzNXJQwBh/u62gDkDI3GYr
        KOLJhKwUhUb/NIvDbCTKQzuPmPUZQ65WNGMrHo3qdTY0nBg22Gkn1x5myHGxMdLQ
        mm+NgkJmbUdQB8BcAC5aPSR8UW0o6r9Vw8Ok7mxoOQHa6RbzvamEtZwOLZrcrvZP
        3mtQ2My9b6enWHAZZgl5Zptaz2JDSpZPj0q3zQTzGbJkSrRj9j+YA2BulpAUtg+u
        9EOoFEAzdCyzKMxuX5Mm9vWJnc9l5VnzwF5v7IgpeU3bm5LyNFIoLrmM9nxzIyXP
        3neSjlmeqF9CFps9K0rtUbPwBRms1Iu5VukOQF4lEiKqzgPGRmgWJsWeRbcz+ueK
        QDWj4axn1Yrq2JG4T0FqRBxaVdghHEU2i8F+nrTiaKn2tZwxzWN/DDB0na+4l+Hg
        2Y493QFIGyzbGykKXAMfqo956cRQzIRWjUFWP6ThU4jMZw5V9Opb76ziI+3Kf5dM
        qXbNz5jrUCu1xp4LaeaH1W6WLOk2NZxUV/sqxwCzFFuRIn4ToftVBPbBjvRdv/f9
        jtGXPJ9JwFYioesb2PJZa007bWYbAca4aa4FjXnXYn/yft5u+wxmjpVaZacwE9hs
        LZ0ByONNatOBabCjhSIYX/voUtoi1OifNsuQZgw1q/61IlbA7XhIy9RMt9mV62yZ
        NtPAjMrZCLnGOwka6UcmM6LRtm//0RwAn9JMowJDw86PFAFauTkfXArc9XtN2RpU
        ZfJey0P3yWDI1Xgm1xxpFTH65EvI0DKWTLpZY52l6VLJkxGdWPintY7y7HOVUwCu
        iKMoE2dPRhGcEtfG0XRMmnGE2H3SXiNZldyh6FJpGZqRP1uuZhqKJcelIzVTPtJI
        06cSsplOLDsQZPfdJcsHqgOQRp0WwfgDV19iU4R+hWZJXMWZIY9qhZTNSAW5jKIm
        w6A1nvbpFbYsFtJSUYz22UyJhgHqNOdF08lnyzF1StZYqd4DUIToutMQwiFxLR7N
        6Eoz5+7rA1MhuI5PahkwDWeDvX60UiUJ7CO4DJkaaRkNh8w+UshILZivLHRSXYG5
        38w1mucZKA6AT8lo5c/ywOxjUfplT2Qo4w9k5/5ZCCU3LTpmGS9TGRch1cFom2EA
        NNaIqRtc7yXhS5dIwRURajJOkmAfwZVk5NIc1aI4GNQiQNOY2Z0LjSJR/9rUbBpC
        9KEIc6BlhDWLCm2ZGvvQ1a6WsyEFrYAgLVCShGZ9E5Pp0ciZS8LWKUxmxJaRFxQH
        oCgFfp0CdtSZB6Yy0p4/bXlpSpdFWabJYudLJaOaLFnmq/25RPta6SGtnLBWQRjz
        SCRrfWnpRQ0mmKVb2w0eRR0Am8I2Py9CdAu4qb0iwDVuQHHO3rNhMjKa+T3fuEvL
        8hkUabl2xMEsKLPlMtcuk/q3HRd7zlhGx5YjPV6uV2lo1bAw+89ggn16hc04NDsP
        Ig6AS7lp0DfNoEg0u4msxc2ODrVl+uQVZX2w4DNejKhJe61rzV3oNcKGpPHRQmLk
        mNQ/Awdhzdp6pBU9KuIApCmaohjc0JE1kM8pCkG9h2RomMbQJUtTni1HSxmY88p0
        rrLGU9qgsWhsM4rVSpVo1YKwKvHZTqxLjrQMxr706XQm09dO2205AC463UejhYTd
        v1ARZxGcEB9CjId2pKqZZtDIK6bJBPjjqrl+O72uyOegJT9Ltc1mXl0pJmloOazM
        ugjmkVGX8S9EDYCJ0JvVVsChFUiIyNMH10YI0QcN2a6x1nheXwUwU55mvlGjfbYM
        myVh52dNmXY/2m1bk7lg1Cok7xkphQSs2g6XLpWeD5++bnft5nYAQinSVmGesTY/
        YxZg5YG5wEM5AhpFTnnBVvCu870h0ixMaDuX2qkTtkFgI42ZkYz+JdprVqaE85K0
        oxGoSTLTmkEMy7HI7QDYOZOQVHoaXN61K0Xh+lsmQjMQCUL1wbVW2GkiW7n4fi+J
        kE6VlkxNp6bIQUYeuMaJoYOYJwrMdtkOO6tdhq3SCC5s4y+NplIAadF0UcHKw7SD
        EHnhrM/Y0CjsMZGHsZKMAkwZrsi1E/ZKXoROH0nApJu15DHgc8YY9DOLPWUbOVuW
        NDp1DwBNOgB5cldFqa7X3Nw+FCHXbisIs2hEuz9aOWo7zcJMubgcTJcyY9OaoZyq
        Tks52OtAIwWlUSinPQ/SzjOj/6w9r5kCYztGuRwA10T5OhQ6utQuhEqDq1gjxOkI
        15EXLSfJl35hIkSdh/a4mjIPihxbXqfm/kPudWkwKO0ERUmLtgIf7S+1tjSc1Eaj
        0fwpgE6i/ovAAmhXg5tyXEbf9Xda/dGSo+10uCK9EPMs3W4CxplpXwTLNP4uGQww
        013stc3UVSYbx2jflsVC2rppV65WyjCRk+oA+IxIkbw2F6WnrYiLNkYuxyf06QcN
        mJtHaw1oR3rm2mLOo+b42etVgzJnngHXCDxYa465lkPsSZ/8VtvVcB4ZTp7PsSv7
        /sjulPnaCdCm2F2v2v1wIVRaxFSEmusmxBFLpjyfc6sljyUnba9Iy3Up7U6Inn1G
        gE2dM9eWVr+l2U4fYyVdDKmNqxgA7cXWLlwTEDr6tqkubfnJq01FaubDNav+0zY+
        +zhUyEiPFaVpzlfaZ1IyOjUXbzpJLOo/aVdrbFj7hZ1W8DmPGvIkYfd3nwOQRGuu
        ByyaI2Av3BDRZtGQRf1rp0M058Inj0WRakPbiGnkUDUYhkQOAyFYRma7WkwWA8x9
        6XL2GcV+5s/S1L+vv94aAF/HQiPrgbSpX59M7fEqYh8OgoJM4HKKtSInNqvhk6MB
        NoXKNGgsJ8OlVzQcMhbYz6HBxtl7Xeo5mIF1njWaWgRY1GjapxC1FZiPoguJ0I6b
        dtRvpltctK+EjLSfGfA5G1p90KRpGdS/xhwxWQymXrHHnjlWGm379n8ngM225HFQ
        y+Yv7VxxEQY0j7EPQTezc06tIkQaJMR4ZBkRSQ89edXMk2pENZpgRswJ7MJCZrTJ
        at9sm6HX7LFhnyTp5NoC1hwkbbP2g33UMq1tJwNQJOWTtw/afc66Z0CbVgtR8Fc0
        54cBm1XQXl/2Z1JwOW3MiFNrPWoEL5osg3Qu2GybNVauYIBtRKXHieXoaTAwzQRj
        5TRjURQFnwyUrz+ahW55FwhbAdkLX5uqZhqnZmVryMtag1Jy7PeaESejPTsaYcCe
        F6ZBY99XYMtjtc+SwdZHrj0oHT2zYKYrWA5RsygXNX9i9inNwIc4nRCCFnbJz/s5
        E1rUfwijHwIae9HXPkORmsqOvV9cdwu0C1Npu34nCebca94fwYb2yaZ2YLMgLFbE
        3md5US7KAGZRVEVgKfIMcshImOXNuz5nKNtm+8EE01M3kYfVYkCrzidEbY60I6Ph
        kDENg4bDl7SXdgy53bbT5Ei0zwhmfM6K5Dq1+91Mu01/F4AG8j5ACGObtVBCpCBY
        SDtXrx35azFVIdik5Lk0z66z0wlZ6Yx22jZluFhCFljOgN13aRkahX++YE3yWTrJ
        WfUZe1YdUasOfWEcAFdOO7Sx9SEU9Z8mW7M47VqC1jyHYlIYkZTvpI4EXMwTcz8y
        nsHVLjv3rOlQahj9dmS4+izNiLFrRaTY38I4AAlCHGPLAptqb6UPzf5eQpZGBO6D
        ZuSv9YwsStAH195iRZtaOWeW8dda6yxGxsUq2TIlwHa+GKlGO73HmAOfPpFaVzYj
        2ur4BHcAsiL7kM6AaxOxPXXXz0kftJQSW2nkgcYacI23Vm4c0KszAPYzalrRJoNS
        9aUXJKE592wZ0tByiuy9yE6LSMKlR1i0f7vPEdwBsPNF9oCFijhNaOYZbZmJXC1D
        4ZKv+fy+3LHWs2tGfYDumtKsBJdW2q7iMubYdTolrzXXjL2Z1netYKAdMAIon15s
        t93gDgDgjrSLYPgBv4PClmXKtN9rsBAuion57L62mYpMw0s32zffm2PLjGLNvRVi
        /iTbZDkYWSxkJ8HctxrG33yVaNP1uaRdMMeGvR9YqRbJvVwIB6CI0FCaeeDqg3Sf
        XGxDCDANYjPypZF2NIoxly6GgblmWA6UVkCgJYdpkDX3DLuKvajBYB5opMBMtCtD
        1QHIyt2FoNrtvvgisxD0u5Yx9j1bETahRmQZKr3AzAtqp9AYazWLFZKWpWH8WUyW
        xnyH1osSCJVGbaeNxCFipBZUHQBXMV0R6DdXv9jV0nn6E8LoSuaX8soLMd8aVKCt
        MJlVxy6ZGlEzgynLakuack5TsFIIdcxTqs00il6i/bT3EtDa69JgF2QHSQEUkdJh
        5LVa6YO2QbQZj6J49JqnLVgyTMOoGZlp5zal5ZlOuGZk3sn0fxF1ah5onm5K5LHA
        ZLBZY6TmABQh0k+DL4o5SPRwlnzGmdssma38jiWTDXaUoDFvbLpZy6BpFEWyn0Er
        Pai5F6XXsIt9k5LBYkWSNjRS4ioOgD3orkUV+rx/K7+TRKiUg+uUw0FQXCHl2s5U
        p9K/ZrvsqF8DvrQPo1gr6zMpOexoU0MXMGqN0p6hE9autJPtS6+oOAC+hylCkVko
        +aGfN+lDqOcOkebQMsTMC0xsWa5X6fZ9nzMMJ0snpDFcknLMOzvYLAOr9kLzvgjW
        GLHm2E6XaqYT24Grzg0Aqm21mgNFov5dkxaKeQhZ6Gf2wTZYTGhuGhPaeUYbGtQ/
        W4lqH0dlodOqwH1gzDm7sNPl3LGdDA19w3TCGDLM9tRqAFwecShDcK3DFTlqbZIQ
        46+pAExnQ+tiIY1ctgaTwaplsNc5i5LXiJhdP0vKYe1TLfaNPVbswl4t6j8B3QHI
        mnjtCDx0lbt2JOrbECGcrxDQoP5DRJXa60hjDKXXpssRYxZMsowDO3I2x4ldJc/e
        j6wgz6b+We1LI6uvNAfAVlAhKVi7DyEuSvH1hQ1beYSo8C/CvLPbdUWXWpta4/w6
        eww18sAuue3AlUoMHWC0A619ytJD7LHXSOloy6A4ALYSZG/EPLBzQqGo6MT5CKEo
        NFMgGpfe+BBCpq/IRgJZ0Yd0xGy+16Cb2caT5ZSZukR6rEIFUNr6WKotFjPSzOft
        ypIuqs3TT1EHwNzQRfKEXQogdP/YG81F/Ws7PRqMg29utdgVjcjfppY1mIUQEW1o
        pqhZuPrbSc/A0tc+3dNp1L8dxHTSkUub5fa1Lc4AuB4mpFdp03Qax3Nc0FDgpiyN
        xZsX2kaE5aG72g3tSErCtUYl2QXXexaYRsfFXkjD1FPMqJaxX3z9Zu1LLXSac5dn
        /EUdANu4hijwS5ukIlD/WrLS3mv0wXzPlK2VD7fnT5L2zVqziXwWNPZFWopEWnaI
        4ENyfzNTWC4dyTzyx5hbdpDJisrt9pjpNRNp67JtB8BWhM0IZyLN8BykXFoeMBVK
        mkxX9bK0jOQ1kaf5jFJr2zdOWqyRBt3PehZf+kWjUpuRmtHaq+Y4MVgeE5JzYTta
        jMCKWfiqUXzeTJF7Sw6AjwrVoJey+mT3I6Qhtg2T1qbO+owl86DRcb4on+lsaJ5O
        OQgUrc8AsGhzhuFM2g3BnLIiXIZe8LXJ3IcssFNT9uc+iNwEGCKn7uqD+T5ktGv2
        J0REqj0XPspPI2o1f2YrApZSYzMlPmjsE9cxVC3qUwomQ8KK0JltHwT4xoTh6DHW
        aCidLMoA+Og2HyOgjVAbyKXA7bSI9hlVrfEPPc8h5lya0rR/PmiFhexokA17jWmk
        Fxjtp6VFJdqWhk+nSc+DPS6dyCgkMpp1JFtKAeTxLLSp/yIhBBWvKc/e8JpHxlyL
        m6HQzPcaCkIzlcJ2mjScMuYZfBMug8mi/lkIyTJJwJdGlXoOTb3FarvVFFLTDgC7
        AKNZ2AsjlKF1Db5m1b1rg2hV3id90Bp7V0W+BjTkaNDjbOrf5RBqrkUp+PaVdPtM
        aETmLDDTInmZbGlZ7LabXau5HYC8ed7QDoELzD7Z+UHNSC6R4Xs+zQhMA5pOHpvu
        1U4RadSHuJgT1+8k5JjGmQmmQ61Zq9RJ1L/ZPsCr9GemRDTRTv9zOQBZuZi0v2Wh
        Gcqj0yivLLgMIdNg2bLtjcOSmTZvWt46Q0EkCi1RQswz0+Z7tgMVoviW1a6GY8Hu
        vzT1r61jTFmMsdJI6Wk4SO3IaDoFoBnx+eQD+ycvBO3vi3A00iJpJx5CUH5Mg2LL
        YFO+zXwuJYO1hplRjgkXoyHVrgtae4yZ++9ksMfB3vdMx1gapp7SpP5bhdMBsA2K
        XV0YkuYvQoohTTFo0+Gu/mhBk07WkGkyGmwj0wyD1Q60ChdNWVJwGQKmk5GWwuhU
        MB1LBmy9xmQXO82xYLTvdABspcHM5zULV/StJddEiDSIZvRoQrPAEPDXmzDhUv5M
        2tH8uVMpyKR99lxJU9kmNBg0s232fpV0Ku01pDHPyav0ET8tm2EHywxItp07BaB5
        1MuFInjj9kLSPgXBOgfbCjQicS1oGUsTWkV5GuyCNDSdTY32tYwPy5HRqFFx/a7d
        cTPZaxOswl7mOJltS/Z/nwPgivhtwaEi79DUv4v219zYIYxUaHlastlHW130dQhm
        o1NlMOlaLYakk6jsBHadlV20KgWzfWm47FinUf/JK8PB2OcAFL2avgjOgHbUnwYt
        minkvLMVp/1sDAVhHxVlQtuxYMwPm+myjb/0nGusqTRIjpfL8Eg9iy/9Ipm+sJ17
        CaSlRtjrSLrNqv1L3yCFiP4ThDRANhVZhL6E6odmbosdoWnWGGgbZSZbx1yDdmqN
        ZTjZc6+5tlzjJGUoOjndqzUHGmk25jOUswxcKOrf7JMNTYXKKj5Kk+nzMJn98CkQ
        9vO72tSeXy1KkD1+mnPFMsxaFeZsaLCVLAdJIz1i9p3N8Ei3a9fDMaJ+lmNht1l2
        PUQRzr9q02a2bCDMtb62LK0x8FWnH6QI2SWTZciYCs4ly/c7lkw2GMpbKwXDytea
        7QPc+dXUQSwnhllTYM+zVLtM58KUlfzbqwEwN0dImttGCCoqa/GzPFa7bU0ay3wf
        QsFr0+QA3zCznSibjpceU9+6kDZAeVgvKbDnhBkV2u1rMkqSbXcq46Zd/8Veo4BR
        BKgRrdhw5a5CUsK2TB/lKdUfe8xDGEWXwdeM/EM4d0xFFJK5kkZW8ZfUc7rWP0N5
        azhjDD2qHQho0POS0NIhTNuklWa357Ucwugk8J3RDGnwNR0hm2nQZmG05j5NDjuK
        cSk1FkIUTXX6c0lTqT4Z2pAOEtjwGbdOeg4Nps31c7ttm3uAof/tNs15LYeubE8Q
        Ihp0gX0EyYRGeiEvmGtAu6YhrXaDtc5c64Xp3GilpphUM9sRNNtl084sPWozFxrO
        XidE/aFkmftcmgGTbtdu0zW3VzEAGrAXtIYiyIKvApmpmHwnHDTpd20H8CDk0Hxy
        GMohTV7ez5tpV4MF03b2mdXUWvtV0jBrM3Kdyu6kza009c/ca2mopv6WBN/Daht/
        lxev1Q9NpsFGKCcrpFxmesVumxFBmYqCFZGHWoPMo5haxofRfxablKXnWpXlWkMa
        TjF77WoWdkojS+81/XXAkkh7YLYyCmmMQjoevnyWllxNGRo1FUzq32VgNIy/tCwf
        22f/ngFGhM6O+k1Hj71uJZGWrtCi5qXBNs5J2xopKpdNVXUATIrR5QHZf8tGMihF
        yLtrbXTtfLwmTAWk6XAk8phH19gUITMVZ+c4mXS2T64UsvRW0dp1ta+1VpnPIb3H
        7fFhyvExecw94XsOVQfAt/FDUf92vpPt1SdyTFls58NHyWkihEyN2gbm2PoUEgNM
        he2SI+kwpR1hZUZt0u2zGTkNPaCxZpmOiyaS9c9YT6aNydpnQVIAaR3SmogQkXBI
        6t+WqxUha+dimYWNvgiZWTWttTYYlLPLIIR4tnZg91EjSGBT/50UmftkaDirGmlo
        NvWfBTUHwKb/QxUCmrJDOQHMCCVLtmbVv9b4alRkm+vGnj9m1Kah6NjRmhbbxZCj
        UeCp7YyznLwEklS2b6+1+ww+x47pyPjWTyvj5eq/b2zSnkPNAWDRHXmRFY2wZNrU
        fwKN6D9tA2lA69lMWexqY41n0jpipmV8THnS7bnalJYjXUyaxlh1EsVtO0bSOs2l
        L6Xgck6ZrKFv/bQq1zce5nPlGTO6A+CjvbVgKlUN6shEFh2tmXNlIs0bZSkepkPl
        chaZykgTIR1v6Ta1nWgppK0fKVkuw8Myokn7LBZJWp/YDhgbaePSzphl/d88bVMc
        AFd0phV1u2BPcggF7hsTNjQKvGwnh830pBV+Sbfv8qg10hlMsFMlGtBwAjQcC0nj
        lkVrS0CDSUyj4SWof5fjwnoett4F2hsTigNgHwUpQsTEzO2kyXRFqxpyzVf7PQNa
        z2dvWE12I5TSkwR7LYZY59JtshksV7sSFLSdSmCkXV3pECbLJw2tuhRtR7hVubQU
        QKho34UQ/ZD2XPPKMtMdGnJ9TA8LpmJjp3TM9rUK8tJ+bgY+x0V6ftKYPul1wIgI
        7bY0apVcxodFBWsxcNLtM3LyLOrf1U/p4lrWuhe9Ctge4BBOgMso2DSuZtTo+lka
        pjE03xfFAWNCg8rWGEuXQmIZakkwItm8cjsxjcFglOyUlVb/O7lt6ZQhOxgB3OmL
        diHqAGhVVmYhLQrWXsBahXi2DE0lrBGJu+QyoeXAahhm7Yi2E51PNmPGMkC2rtOI
        nk1IyWLbDFedkrQTAMivHxcrJQnxFECIiN981TS4PmhUx5rtJzKZC8XlXJiGXysS
        Zx/XMd+zcpvaYKZmXFFJp1H/TLicI8l1xabjTTlslsrVvlRhpEZwJB0IudqTDrjE
        HAB7k2rR/0WKOkI5H+Yi14ooWfk0n2wNGeb4aRhLUzYDGuwMa/+5WB8mNOhb81Ua
        bJaH0W4e49yuTFd6VKLdBOzUgt2mtI0RSQGYNFTICDxkyqFIfZBs01Ys9vMxDYxL
        HtuYMSMF394I7bi2CiZTohHZao27rR8ZjlInVrPbqQvmHGs4XubPnZBeAAQcAHuA
        tTeVqy/a0Kq6N8HeOEnbiay0+g52vpE9vvZaksxr+iJ+Jk1eq9VQr9fRaDRQr9f3
        3ic/u2SbkXy5XN57td+bf2vKlnoGZiTLmmefLNfPrLGShitqZoGxR7QCE1bQYDt2
        LL3XsgOguUBc0KC80sDegM30g9GmS8lrVnjbshlgMxl2hOP7XV40Gg3UajXs7u7u
        /UsMeqlUQrVaRaVSQXd3N7q7u1GtVtHV1YWuri5UKpW93ydG3VQuiXOQtF+r1bCz
        s4Pt7e299xsbG3u/q9fre45BtVrd+1epVFoeR5fDKQm7fS3DxnLWmWwVq01Xuy45
        ktS/9HzbbDcDGkFlqVTK7wC4vGf7vYZxyIpGtRC65kAj8geuHm87V86KZtnj60rb
        MDac2XbeZ0oMfWKAd3d3AQDd3d3o7e3F0NAQBgcH0d/fj4GBgb1/fX196OnpQV9f
        H7q7u/eMf7Va3Yvgzaje5QAk/2q1Gmq1Gra3t7Gzs7PnAGxubmJjYwPr6+tYXV3F
        2travn8rKyvY2dkBAFSr1b1+5HUM2HOvydRJp3pca5YBdlCh0XfbRkgzLxo2j8lW
        7QUNrXTGbCTt9wcRoSJ/zQjclGm/l1ZqvnaYEY3PkZWU04yDXK/XsbW1ha2tLezu
        7qJSqWBgYABjY2MYHR3F6OgoxsbGMDIyguHhYQwMDKC/vx89PT17xjVpJ3EeTGNu
        R/i+fiSKrVQqoaurC93d3RgYGNhzGhIjnqQCEidla2trzyFYWVnBwsICFhYWsLS0
        hIWFBSwvL2NlZQW1Wg3VahU9PT3o7u7e6zcrUrPhCh4k25Rs1wXNVCM7J68tR7JN
        5jplOzC2rLZSAGk/M+CToWkUtXKIaXI1oRGJa0QGSfuJTFM+Q06as1Gr1fYiaQDo
        6+vD6OgoJicncejQIUxOTmJ8fByjo6MYHBxET08PyuUy6vU6dnd3sbOzg1qttheJ
        h2KiEmehUqlgeHgYExMTe2mAWq2Gra0trKysYHFxEfPz85ibm8Ps7Czm5uawuLiI
        9fV1lEol9Pb2oqenB9Wq6LUk+8Cacx8dzCxwk2zPF9AwImYG1W/KAPSYUZYsTScP
        yCgC9HlrIY0REDb/HkLZakb/vmhMwzhrR35sZWFu5CRKrtVqGBgYwOTkJKanpzE9
        PY3Dhw9jYmICw8PD6O3tBQBsb29je3sbm5ubWF9fp/SxXZjMws7ODjY3N/f9vlwu
        o7+/H2NjYzh58iRKpRI2NzextLSEubk5XLx4EefPn8eFCxdw6dIlzM/Po1Kp7LEb
        5XKZkl5iII0JkKT+JdozEbqWSgKadkFLF2ukSDIZgCLR+iGNfiIf0D/rH8IQm69m
        P1gIoXiYtQvAFTp+fX0dGxsbqFarmJiYwIkTJ3D06FEcPXoUU1NTGBkZQbVa3TOe
        GxsbWFtbE+1TSJipjQTlchkDAwMYHx/HnXfeiZ2dHSwtLeHixYs4e/Yszpw5g3Pn
        zmFubg67u7vo7+9HX18fKpVKS/Ol4fDZNSSs9lmQ1q0+Z4ilx3zOFosp0TLOGm2W
        /uiP/mgZwFBaA9pGL60PWv1wydTIU9qyNaN/Uz6gp3g0c8BMJEZ/fX0dvb29OHLk
        CK677jpcd911OHbsGCYmJtDT04Pt7W1sbGxgZ2enY6MuKST1Bknh4tbWFubm5nDm
        zBmcOnUK7733Hs6fP4/NzU309fVhYGCgKWeA7bizc9rsfHMntu2SwXIogKuZ707S
        Ty7nzpCz0rIDoG2UbPlMGWnecAhDrCmPrRBcP3fShjKRjFVSANfb24uZmRnceOON
        uPHGG3HddddhdHQUALC+vo7Nzc1r3uBnIakJGBgYAAAsLCzg9OnTeOedd/Duu+/i
        7Nmz2Nzc3DsFkbZetfRFp7Wt6RSxHHuXkXZ93m77GuxrwHWa7gCEiLx9fQipONle
        fl65TBmajIO2E8Uowtre3sby8jIA4MiRIzhx4gROnDiBG264AaOjo9jd3cXa2hq2
        t7fF5F6L6O7uxuDgICqVChYXF/Hee+/hzTffxFtvvYVz586h0WhgeHgYPT091Fy5
        DW1nWbp9QD7/7zP8vr+RkKVRO9SpAUoiw34G42e3A6BFAfvgig5DpCGKkPJg9yGE
        MdaUK7WJS6US6vU61tbWsL6+jtHRUZw4cQK33norTpw4gYmJCdTr9X3n4CPk0Gg0
        0N3djaGhIVQqFczNzeGtt97Ca6+9hrfeegsLCwvo7+/H4OBgxxZpsRlWn17vpLSF
        LUe6PeaJCHabPjkpOjfbAfD8RwrSPMaQUbfGOIRgOnyOlsbGZVB2DLquVCphd3cX
        S0tLaDQauO6663D77bfjtttuw7Fjx1CpVLCysrKv0C2Ci0ajgd7eXgwPD2N3dxdn
        zpzBq6++it/85jc4deoUAGB0dBTVapVmJDqlXZccNjQi5k5kYUPYsgxZfgegCNS/
        pqwQz8ukyPLIBvRZHjZdZ75vR1apVMLW1hYWFxfR19eHkydP4q677sLJkycxOjqK
        jY0NrK6u7l2JG6GPRqOBcrm8Vw+wsLCAN954Ay+++CJef/11bGxsYHR01JkeaEUW
        qxiMYdjyRLNahXNFd5ZCBELaet6D/Q5AyDy7De2+aNPToZ0sH9PB3KzsIhqJTVwq
        lbC+vo7l5WWMj4/jjjvuwN13340TJ06gq6sLS0tL2Nraika/YGg0Gujp6cHIyAh2
        dnbw1ltv4cUXX8TLL7+M+fl5DA8Po7+/v63116nReUGMjUj7DB2toaNseWzklOF2
        AEIW3YWgwk3ZB1FWnj50+nGgdlEqlfbusp+amsI999yD973vfbjuuuuws7OD5eVl
        1Gq1aPgLjkajgUqlgpGREXR1deHUqVN44YUX8Ktf/QoXL17cu0q52TXDWMcsfRuC
        WdRiFhhgB38aqWPXZznkrpQ+8YlPLJdKpSHt/IpmwUVe+dqyQ+X9Ab36jqJHBeVy
        GWtra1heXsbhw4dx//33433vex+OHDmCjY0NrKyseNdqRHGRzNnQ0BAGBgZw9uxZ
        PP/883j++edx4cIFjIyMYGBgAPV63ft/k/cmOsm4+Z6jnWfIw1wyCxgZbWoxk+Zn
        TDQh4woD0Gg0vPcAsBCaAnf1I5TzEYp50Driok3950G5XMbGxgYWFxdx6NAhPPDA
        A7j//vsxPT2N1dVVrK6u7ms7ojORrInBwUEMDQ3h/PnzePbZZ/HMM8/g0qVLGBsb
        Q19fn9cRYPaJDS0Hg2nsNY75mTKZbAx7PbXQ9/0MwLVI/YeSd9DTHAXzdPdQLpex
        vb2N+fl5jIyM4IEHHsBDDz2EI0eOYGVlBWtra9HoH1A0Go09R+Ds2bN46qmn8Oyz
        z2JpaQkTExPo7u6+yhFgGrmDcMa809vXMs5stCjnigOA39UAaB9PcL3XgDYNbstN
        ZB+EaNzHZmjUFDTDoiTn+Ofn51GtVnHffffhQx/6EG644YYY8V9DSNbJ0NAQBgcH
        8dvf/hY//elP8fzzz2N7exuTk5N737zIStNp0NnS1L9LljQ007LsvDygl0pvQ9Z+
        B0ADRaD+QzoAIZ0OZvuh7jDIY/gBYHl5GZubm7jjjjvw8MMP47bbbsPW1haWlpb2
        /V3EtYFk3YyOjqK3txevvvoqfvjDH+Lll19Gb28vRkdH0WjIfsFPqFQfo+1OZi86
        cR58dUhttr/C+wJuD2zjH4IK1zbCtqwQTAdLpjmf2pdzZMkql8vY3NzEwsICZmZm
        8Oijj+L+++/fu0kunuG/dpHM++LiIkqlEk6ePImbb74Zzz77LJ588kmcOXMG4+Pj
        6O3tddYHtCqzU6l5m3nrdOMvaQM0mAuWnqqcPHnyL0qlUg9TSIIQEbApO0HIWodQ
        BocpN1TxpO+Zks/n5ubQaDTwyCOP4E/+5E9w8uRJLC0tYWVlBaVSKRr/iL01sLa2
        hlqthttvvx133303yuUy3n77baysrOx96VCr0DCandy+C520N0Mw2rYj0yK2KydP
        nvwLAD0SnUqDL++vIc9E8pnmAkv6onUe3vTU2YZfczzzjF+5XMb6+jrm5+dx8uRJ
        fOYzn8GHP/xh7O7uYn5+XmXtRXQekhqR5eVl9PX14d5778VNN92E+fl5vPvuu6hU
        Ki3fKMheb2y9orFf2JG41jMw5BBt57ZaDUCoiDt0H0KkGbQKDF0Lk1Utnbz3IZE9
        NzeH/v5+PPbYY/jwhz+MSqUSDX9EU0jWyuTkJHZ3d/HjH/8Y3//+97G2toZDhw41
        tc47Md+s3Ta7fVb64gAULa6oMAAa0Wia7IOMLFqc6ZFq0oJpz5Gc6b98+TLuuOMO
        fO5zn8P999+PlZUVLC8vR7o/oikka2V1dRWNRgN33303br31VszPz+Ptt99umg1g
        7UEm7ACCDeYYMY1+UXRgi9BhALRySLanl3wGcL+cwteHRK7ZD2k5ZtshKvE1ZPie
        K3n+y5cvo6urC48//jgeeeQRAMD8/Py+v4mIaAXJupuYmAAAPPnkk/jud7+L7e1t
        HDp0aN/f+P4vk4FjQTuXzWxX8zhe0ds02y6VSnwGQJN6dUW/2gYg1LOyZfscDrY8
        3+Ytl8vY2dnB7Owsjh8/jn/zb/4NHnzwwb0iv3K5HI1/RNtI1uDa2hrq9Truuece
        3HLLLTh37hzee+899PT0oKurC41GQ03XMWtvzH2nUUPEBJOldAWbTDBklEolXhFg
        qKr7EPS0LTuErINWBZyGcrmMpaUlrK2t4dFHH8VnP/tZTExMYHZ2FrVaDeVyOXQX
        Iw4YSqUSdnd3sbKygqNHj+KBBx5ArVbDq6++ilqthr6+vn1/32kRYQJpptTWU3bw
        0K6+zCr8Y1D/7Hoq82eWPfldu/xTACGMf/I+JFgT56v07/Qq4Dwyk2e9dOkS+vv7
        8dnPfhaPPfbY3p3+MeqPYCJZf6urq6hUKnjggQdw+PBhvPHGG7h8+TIGBwcB6BgI
        6fZYTj6zJslXhMw0mgz2UzuFa4zVNv0ioFCGOETEaspkydY0cCEcKnMzmOOZUP5z
        c3O45ZZb8JnPfAYzMzOYnZ1FvV6PUX+EGpKjpltbW3j/+9+Po0eP4hvf+AZef/11
        TE1NoaurS+zyIIBjnF1RuUZtlCR8uonxHLbxZwdcbDY3gTgDEPK4lTbbYMvWkqtZ
        hapZvOlDpVLB2toaFhcX8fDDD+PP/uzPMDg4iNnZ2VjhHxEEyb5YXl7GoUOH8P73
        vx+bm5t49dVXUSqVWr4z4CAgrQDa1l3tyDDTCJ2aj7epfi3j/zsZMimAkAtd+7iK
        KUvb8LicDtbpAm1HzierXC5jYWEBtVoNn/70p/Hxj38cGxsbWFpailF/RHAkBYKV
        SgUPPvgghoaG8PLLL2NjYwP9/f0t70/NuhtpXaZVPKiRwmDC5Sixda41RjIpAOnC
        kVZkA2EcEReVJglXxS9zrDU8UZdMH6MyOzuL0dFRfO5zn8Ptt9+OS5cuYXd3Nxr/
        iMKgXC5jbW0NW1tbeOyxx3D48GF87Wtfw8WLF3H48OGW0gEaOlWD0mbqKF+6kAGt
        lC6j8DKt/ZYZABcVzUbag2nnxm3ZGkdaNI8ZhUSpdOVa1osXL+KGG27AF7/4Rdxw
        ww24ePFi/AKfiEIiWbOrq6u48cYbcdddd+Htt9/G6dOnMTAwcM04rL4iOak9axt9
        ybZdslg2jjlOLqfCw8q0ngKwjV5o+oVtFFyROBshZCayQtZx7O7uYnZ2Fvfeey++
        8IUvYHBwEJcuXYr5/ohCI1mbKysrmJiYwIMPPojZ2Vn88z//M/r6+lCpVDLb0NSh
        0vspLfcv1XbSPou90GayWbJyBot6XwbULkIWFwLuCFxyA/merwi1BlpyyuUytra2
        MD8/j0ceeQSf+9znUK/XsbCwcM1EUBGdj1LpylHB/v5+PPTQQ9jY2MArr7yC7u7u
        vUuD0v6v670kJIynS18xdZVGxG+2LS2DaTtccKVJHOgMB8AVCYdwBtJy1VLtJ9Bm
        U7TOnyawN0BytGppaQmf+MQn8MlPfhJra2t7t/pFRHQSku+nAIAHH3wQ5XIZv/71
        r1Eul1NPCGjk+5PXdmT5dBUj+jflaOh/5pFFu33mOOVw9ForAgxR6GfL1YxS0xaD
        VD9CGv9EPpP+she9mZdKIqaNjQ185jOfwSOPPILLly9ja2srGv+IjkVSHFir1fDJ
        T34SAwMD+Nu//VvU63UMDQ3tKw5kMZx2u6yCaZb+crXFGKuDVF/VjJOX2wGwva+i
        GGA2Qp1wYHqIadBaoIkTUC6XsbKygp2dHfzZn/0ZHnrooVjpH3FgkKS1Ll68iI9+
        9KPo6+vD17/+ddTrdYyMjOw5AZ14qselmxn5ePM904FhReY++6FxaiFr3pvSsInS
        DuUthSwC0zomYy5wjQVizikbtqzkTv+dnR18/vOfx0MPPYTZ2Vns7u7GYr+IA4Ok
        sPXChQv40Ic+hC9+8YvY3d3du76avfcy8sAtt2e+l5ZhttWJx/tMuHL/moFWGlId
        gLScrZYTYEaLoQwVE6zjMj6EKPBzFcBUKhUsLS2hXq/jz//8z3Hvvffi4sWLqNVq
        0fhHHDiUSiXUajWcP38eDz74IP7Df/gPqNVqWFhYQKVSEduX9r6TDCjsqNIV/Us/
        B7PaP2mXCVeAx0qx2u/zyEl1AGwPLASF4euTpkwtaHqF5itjTNPaTCL/RqOBL3zh
        C7jnnntw4cKFeMY/4kCjVLpyV8C5c+dw77334j/+x/+IRqOxxwS0C5f+YBUps/ep
        bfCZNQtm+0wdzK76N52LvLKcq872HIsE7ahcO2Jm5Ol8hTTm7zWOpAC/N/67u7v4
        /Oc/j7vuugsXLlwIXusREaGBRFGfP38e99577146YGlpKdc9AVlt24aAHW0mcqXb
        17Q7zDFi1i2ktZlXltfttI8RaIJVjJEmz36v9czs5/PdLaDRDzt1U6lUsLq6upfz
        TyL/aPwjriUkuvXcuXO4//778cUvfhE7OztYXl4udOGrz7FgFf4BMnrY5VCw07zM
        4khJlIH0fK39eybS5IRwQtjta7Es9vG7EIsxORK1vr6Oz372s3s5/2j8I65FmEzA
        gw8+iD//8z/H+vo61tbWWnICbPqXEZUnYFXKJ21LOxfmeGjXkDHGqNnfpaEM5Dtn
        HyI/rSWfeQTEBVPWQalpSCs4KpVK2NzcxPLyMj796U/jAx/4QLzXP+KaR1ITkJwO
        +NznPofl5WVsbm42vS+y6rUk+mq+2u8l2i9qlNwMNIsLfbayGexjADSPKBQJLu+W
        KUvD+NsyNVkG+/Pd3V0sLCzg4x//OB599FHMzs5G4x8Rgd+fDkjuCfjUpz6Fy5cv
        Y2dnx7s/fMERK4BhUf+h6qs6tdAPuJrpaRflNC9C69y9OTHalwwBV1PkTNhFIdpF
        jQzY42gXUV66dAkPP/wwPvaxj+Hy5cvxnH9EhIHESZ6bm8Mf//Ef46Mf/Wiqk+wr
        UmYXmUk7GHZ7LN2vdXqBkdZ1samSRerltJwL20C5ag+02QdtysaWexBk+T4rlUq4
        ePEi7r33Xnz605/G8vIytra2ovGPiLCQpMmWlpbwp3/6p3jwwQdx4cKFvd9n7WFW
        rplZS2B/ztZTUvbFV1BoG2eJsbPbk6L+E+yrNglF/WsWaLigHYlr5v216wASlMtl
        XLp0CTfeeCM++9nPYnt7u+UCp4iIawFJoez29ja+8IUv4Oabb8bFixdRqVRS66Ik
        9QlThmm4NOh4U66kLB8rwwy8WONU1oqA06AdobJlumAuEK18vOYtV7bxX1hYwOjo
        KD7/+c+ju7sbS0tL0fhHRGQg2TtdXV349//+32NsbAxzc3Mol8tO6leaNXXdJZDI
        a1eOqZfMwutOYgR9qXJW+sVnn6Xk7WlktlFyLSa7+p4BrTPvrvbtVy2DrJFO8T1T
        EsU0Gg187nOfw9TUFC5fvhyNf0RETlQqFczNzWF6ehr/7t/9OwDA6uqqqi5jnMU3
        X83PW32GPP9PMhfv+kwyoLPbMR0mVuCoppXTaP5Q1L80XMVwWrl4Uy4LWf0vlUrY
        2dnBysoKPvnJT+L222/H7OxsNP4REU2iXC7j/PnzuOuuu/CZz3wGy8vL2NnZ2fc3
        0tR/Wi1Yu21Ltwn4n1/aWJpBKjNdzT7K6QJdM/seQitCNX/WyvXn6Y902xq5/zz3
        M8zNzeEP//AP8fDDD+PSpUsHxrmLiAiBixcv4rHHHsNHPvIRzM7OAuAYCB/1L9W2
        +Z5Fl+f5rBW46hbYrLUWa6wSmtnGN1TOhy3XdzxD87w/I6eWx7lIiv5uueUW/PEf
        /zGWl5fjcb+IiDZQKpWwvb2NpaUl/Mmf/Mkeo9budwbYYOpmLT2ooWckj9/Z8NH/
        7KC1rGUUWUUMWXLNn1ky0woNtY82hmA5yuUylpeXMTQ0hM985jOoVCpYW1uLxj8i
        ok2Uy2Wsrq6iXC7j3/7bf4vh4WGRbw/0BWXM6Jytm8zaMqn2EmjcJaA9ToBxCoAJ
        m9IIUXCobRQPwoayZfoqYLe3t7G5uYlPfepTmJmZiUV/ERGCKJfLmJubw7Fjx/Cn
        f/qn2NjYaPs+jYQpZEa1LpkakM7/2/pWI/9vf86Q12g09GsANA0WoEcN+Sr/WfKa
        +bxV5B3Dubk5PPzww3jggQdi0V9EBAHlchkXL17EBz7wAXz0ox/FpUuXWmrHNmKs
        vPZBKvbWYo9d6Vt2qoGqqe1Fpn0lrQZcMplFImkOBrt4x0a5XMbly5dx880342Mf
        +xiWl5dRr9dF+xAREXEFtVoNS0tL+OQnP4lbb70Vly5datrZ9p1QAvjGRhouR0ay
        XcbNfmnyNKh/832pVOI4AKZR1KSYbNlaA6rl7Wo6VFm0V6lUwvr6Orq6uvCv/tW/
        Qnd3d8z7R0QQUSqVsLq6iu7ubvzrf/2v0dPT0/Se00q/sg2ndEo5TYeHYHMl4Uvd
        UlMAWrn3tAI8NkwHRysfr+FQ5Wm30WhgaWkJjz/+OG6++ea928oiIiJ4SE7bnDx5
        Ep/4xCewuLiYWw/YEWACRtFcns8l2maCeY+LnXqRPL2V1k+71qDjNbareEKLadCu
        7jdfWTLyjGNSlPQv/sW/wCOPPILLly/T+hQREXE15ubm8NGPfhT33HNPat1Nlr5o
        NYLO0hOMdKQrXy4lR/PsfQJWcWHaCThaDYD5ENo0MMObzSMvxNHC0Eio//7+fnz8
        4x8HgPgNfxERiiiVrnxzYKPRwKc+9SkMDAx4rwrOQqt6xnW5j9TJJ1dtQtK+dKCX
        V5YUNJ2MPEFj2w5AiMI7U27ez9uVoV3VatYwsJwqX//TZCXU/2OPPYYbb7wR8/Pz
        kfqPiFBGUoB74sQJfOxjH8PS0pLTmEnXQtk6kcGEZuk6KZmuuibf7yQRIl3sQ9ua
        2yVE8+FcxpFhLNPkScPlVDHGtNn2E6Vz22234Q//8A8xPz8v3qeIiIj8mJubw0c+
        8hHceeed+04FaOgprSt+pel+E3aglXzGgFZBejMsiUjopkn953k46VyKRv49gXYt
        g0uOT+7m5ia6urrwL//lv0S5XMbm5mak/iMiAiFJBVQqFfzRH/0Ruru7sbGxsfd7
        RuTPzsW79F+oeivJ9tip4lbvD2jLAWjW22hXFqBzftWGdI7LhxD1E4D/umYTyfeU
        f+hDH8LJkycj9R8RUQAkBbm33347Hn30UVy+fFm0ojwNjBSABptrIi0NIAXmM7gc
        sgS5WN1WBWt7ZK4CkBDVmkzYjgYb9gZOqwlYWlrCzMwMHn74YaysrBS2SDEi4lpD
        o9HA8vIyHnvsMVx33XVYXFwU1yEadL8Jph5M6zuz8E+yHVtnt+qINe0AuCiHEDfv
        acjRop5chrhIBrZer2NzcxOPPPIIxsbGWq44joiIkEepVMLKygomJyfx+OOPY2Nj
        g3YjJ0s3uYIQhixtvcpiR0zWtp1asdwOQAgK3lVboFWpqUWjacpzjVeWzHK5jPn5
        edxxxx148MEHI/UfEVFAJBcEffCDH8Tdd9/d0jXBPmhXrLOjcCbT6gocpeW4ov9W
        7UdHpABMeRp1Br6fGfI0n6uVTba9vY1qtYqHH34YpdKVb/6LiIgoHra3t1GpVPD4
        44+jWq1S9ir7PnwWXJfhsOTZhZHSckwHhp4CYB+NyAN2oYbNbrAXo9k+0/Nt98RE
        Ev3fd999uO2227CwsBCj/4iIgsK8ofOhhx5q63puW0ex9KItR+M6XA1oseLtjlfm
        6pDwMlpBGkXD8qi0YH+HAFOOjWbmcGNjAyMjI/jQhz6Era2t+E1/EREFR71ex9bW
        Fj7ykY9gdHQU6+vruf9vWjU5KwC0daGUPsw6wi3xPC7GWOvuBdMha0em1wGwJ0Oz
        Ol1LpisvlLxnIoR32uwzlUolLC4u4oEHHsANN9yApaWlWPgXEVFwlEolLCws4Pjx
        4/iDP/iD3KydqWs12El2u7aMVs/J52k7gUZQZ8qx37eCq1aGzyjav2MhhOOh/Vwh
        Tk00612vr6/j0KFDeOihh7C2tsbqXkREBAGrq6v48Ic/jMOHD+c6teM75qdZc8VI
        7bLP4DMZ8jzttSvT6xpq0+LA/gnTqgq1iylY8to9rtEKWt0A5XIZy8vLeOCBB3Dk
        yBGsrKzE6D8iokNQKpWwvLyMo0eP4oMf/CCWlpa8f2vrPe0UL8vZ0CjgZsnzBYqM
        OoyrHACz4EMD2pX3iYxk8Wk6OhpetU9mM1hbW8Phw4dx//33Y3V1ldCriIgINpaX
        l/GBD3wAR44cyb2P2dG/RnG1K5hjOhcaly6x2Iw9ByCEcQL8g8fOObkWC+tcqOuV
        AQnHLYke7rvvPkxPT8dLfyIiOhClUgmrq6uYmZnBQw89hOXlZec+9l0uwwSLMrdZ
        VmbhYgKN1C7zjph9DIBmoZ85cKHy464+SUPzxkST0Wj1WdbX1zE1NYV77703Rv8R
        ER2OlZUVvP/978f09HRqLQ+DXvaBHWiZxXhMoyzNXvtsImtuGo3G7x2AayHKM6l/
        EwelyNBGK5X/y8vLuPvuuzEzMxOj/4iIDkapdOWK4KNHj+L+++/3nuSxDWWr+irt
        /9kGTRJpLCtDFitYNQM413tplEollDUpalu4C2zqP3mvdcSFXVxo5ofalbe1tYWx
        sTG8733va+r8cERERHGxvr6OBx54AOPj49jc3Nz3O0m9HypY0GStzfeabAnLkSkn
        P4Q4mqYpS+uMpilXI6fmeq5W5CXn/u+8805cf/31sfI/IuIAIGH1jh8/jnvvvRcL
        CwtqulCClcxqVyN1bAZZjGjcV4vGpP4TlEMYxrSfWTJtg8yO+rWqaV0boBV5u7u7
        6O/vx913342dnR31YtCIiAgOGo0GdnZ2cN9996G/vx87Ozsqcll61nXLn7Qj4Esn
        MAyyq3ZLi10oaxt+bYfD1QfNnLxGJaopr9W2FhcXcfLkSdx0003eiuGIiIjOQ7K/
        b7nlFtxxxx1YWFgQbT9L97DToM30JS/MYJFZhZ8ln92+2je7aFbDA3766SAZNqnF
        njAWd955J7q6ulCr1UTajYiIKAZqtRq6urpw7733ApA1xL4La6SRVsfFkJPGskrJ
        cY0Ts77AZk3oDoCm92fKMeknO6fCkGenGczfScsx25VYlCsrKzh27BhuueUWLC4u
        HignKSIi4vcswO23344bb7wRy8vLbbfp021sllcjteqSJa3Lbeqf4WC43pty1L/b
        lV09yTbCWW1qphck5JVKJWxsbOD222/H2NgY5TvEIyIiwmNrawvj4+O46667sL6+
        3rbB8dUdsYrWtG5tDVkQL1nD4LpR0G6f7gAwqf+0xcBeKOwrGm2waKjk6N/tt9+O
        jY2NGP1HRBxQlEolrK+v484778TExAS2trZE29c6GscCK9pPk+e7VbBduJwl11Fx
        mgPgqp6URlblO9vhsKkuljyWB5xcFHLixAkcPXoUKysrYm1HREQUD8vLy7jhhhtw
        8uTJlr/iO1R9VSdG/z47aNL/0vCdDHPJEncAfDkUDbCP32VNWKdUuJrtlUol3Hrr
        rahUKh3tvUdERGSj0WigUqngzjvvbNkAaVP/jLbTZEnCd+usbZAlHQ9fCtwlQ8wB
        cD0Q0xnwVYRqUv8smKkFFrsAXLkhbHp6GjfddFO8+Cci4hpAqVTC0tISbrnlFhw9
        ejT1+wHSYOtAyRNJ9s8MvcSs4WqlOJJhG/PMkSgDoHG9btK2OZgaN+65jL4WfcOi
        itbW1nDixAlMTk6K5wMjIiKKic3NTRw6dAi33nprU9/34WJWOzVoYNoqW19rMBl2
        esF8TQOlBkCbSk4eXiP6N1+ZcsznYTxXvV5HT08Pbr75Zuzu7kb6PyLiGkKtVsNt
        t92G3t7e3Pd++PSfdF2S/b4TdVMRLg7KM25tOwBMitonT3twTTpe+5gfqyhlbW0N
        MzMz8d7/iIhrDEnx7/Hjx3Hdddc19bXfLOrfbIupZ9lpaV+BpKbtaEZW2w6A+XAa
        hRour5BN/dvyWNS/79iGJJKx2tjYwPHjxzEyMqJ2N3hEREQxkBz/vfnmmzO/+bMT
        I3Af2EXiWe0z6xlauba4JQfAZQxDRJBMirwIYDka9Xodvb29OH78OGq1Woz+IyKu
        MZRKJezu7uLEiRPo7+9HvV5P/VsNPZ+WO5eWowHXXTHskxLNBuItOQD2RGmf/+zU
        IyHa8lyLI7kMZHp6uq0q4IiIiM7GysoKbrjhBszMzGSyAACHmg+RVnX9LNW2L0XN
        dmZabb9pB8BHUzMnMY1WYTgfWkf90uRLw2xzY2MD1113HUZHR+PVvxER1yi2t7cx
        Pj6OG2+80RkIMFK7aW0xAyGGTrdZEe00cStV/zbaqgHQipSTCdT0rNhIy+9LPZPL
        cWo0GqhWq7j++utFZERERHQ2jh8/jmq1mhrJslhezRvxpGl4l341ZUkXrGfl91t5
        rtwOQJp3o3UXPjsi14jKmVW0ebC1tYWJiQkcO3YsF+0XERFxcLG2tobrr78eU1NT
        ++4CYR310zzax3As8jpJ0vVpacFiOzKaZgBcCyMUVS4J0xN1yZTsh2bNhO2Jrq+v
        Y2ZmBhMTE9jc3KT2IyIiotjY2NjA1NQUjh07tnccUJr6zzrhxDCSrvonCbgKIjWo
        f/O9ZEGm0wFIK5jQLNgIecKAhVA1E8lrrVbDzMwMenp6OjZ9EhERIYNGo4Genh5c
        d911qNVqFOrfpq7NgmQJHZRmkNM+bxWuSF/aocli2aWexekA5PmyB80ollkcwih0
        8ckyXxltZ6FWq6G/vx9Hjx6NxX8REREArqQFr7vuOgwODu45AQmko3NG264v3LFz
        8e0ijVFgF23n6VOrSE0BmN4g8/pHu81QET/T0UhemYslq75ga2sLk5OTmJqawsbG
        hng/IiIiOg9JWnBqako8LahhLNPkspyMRIbrvRRMpoQlw+sAmBSNpkHWrClo5eak
        VmDKYI5n1kLZ2NjA9PR0vP0vIiJiDzs7OxgdHcXMzAwlMPAFKNLQYlk1gmFbFkvG
        VQ6AKwpnTpzpqbmoHBbMZ2J6byaYt0HlLQ6Znp6+6shPRETEtYtG48rR4KNHj3aU
        XnBF4IwAy9cm2z6xj0cCv3MAfEI0I3GXPKncTQgKypbF6oPPYbNRr9fR39+Pw4cP
        x+g/IiJiH3Z2dnDkyBEMDAzk/nZAH1yBlWaeXLpuwS7c1nCSsvS5VNvl5APpysxm
        O8O+aMKEq3KTIdeOzCUWUKv/d2trCyMjI/H4X0RExFXY2NjA5OQkxsbG9t0H0ApC
        HRWXDqxcjDQz7582Tiz7uC8FIF05mQamh+MCswrVlpN1hKMd2OOWN62QFAAODw9H
        BiAiImIftre3MTo6ikOHDrUVIGhEyUzG2udMhLCJbFsFAGU7QmVX4jOiYp8cE/YE
        MlMBrKsnTTQ7bru7uzh06BB6e3tTv/krIiLi2kOSIpyensbu7m5LbWhR/y6ZUvBF
        +exnYttdH/YYALtSvZMrG/PIYDkcGmPY7GJMinwmJycp/YmIiOh8NBoNTE1NtVwk
        nJZulYTGqS0TjNo0X3uajkCj0UBZ81wmoEOluGRrphkSaE1iFpILgMbGxuIFQBER
        EU5sb29jcnKy6ULAUPpVUqaPbmdG+8l7O1jVGM+k/bL5AeMIhS1Ua6FoPI8J9r3Q
        7bS1s7OD4eHh6ABERER4sbW1hbGxsVxfE57FrDKO4qXJbbdt7cJFF5h30vicC+89
        AAxoG2MNaBQXmqczWmk7KfAZHByMBYARERFOJIHC+Ph45kkA+2IzZoW8KYPxXSqu
        wmpTlpQ9yeMcsevFbNlljagf0KH+Xe2HoKYATuV/q23u7u5idHQUPT09sQAwIiLC
        iXq9jt7eXoyNjeUqBNSOkBky05yJdlmHvClhjZMTiXxbRupFQO1C68HYRRp5wTja
        aKYz2mlzdHQU5XLT3/4cERFxjaDRaKBSqWB8fNz7e9/nkvrO9bN0asFsz74Hx9WP
        VpB1rE8rRe17plKplP5lQO3ANcAsmO1r1xjYn0u0Ld1ed3c3RkdHY/QfERGRilqt
        hvHxcXR3d1+lL1wBjrQBS5Mh6WikXe3LeDb7lB0LLpvkczTETwFkUSksaBb7AZ1V
        y1Cr1dDb24uhoaGWz/dGRERcG0jqAPr6+q46CcAKetIgeVOsKwK3P/f9fatyXD9L
        McXN/F/f35aliylcXhsLoah+JqS93d3dXfT19cUCwIiIiEzs7OxgaGgIAwMDewGD
        qYfYVequz6V0oesyOLtGjUH9Z/2NhJxmZJsQSwFo5zfMynhbPkMWWw6r3Z2dHQwO
        DqK/vz8yABEREanY3d3FwMAAhoaG9gIG+4p46Qp/+2dTBivFwL7X35RnB3WahelZ
        RwvbcgCyPEPmkQbNC4XsOoZOKppMNnS8AjgiIiILtVptjzE0GQCW4bKP2tkGS7qu
        Kq1N9qU/eT6XlJnHYaMwACzYUb8mNGQynq3RaGBgYKDl6z0jIiKuHTQaV64NHxwc
        RK1Wyx1JSvdBEq7gzcUwSDMbofUtPQWgRaOY8tJkaX+RggTY6ZJ6vY6BgQFUKpXg
        CzIiIqLYSByAoaGhq/QSg/o3gzqm0WQyDDZcdQZMOe3Y4JYcAI2ikDxgffGEdqVr
        IoMR/ZfLZQwMDIi2GxERcbBhBw2sqNzU4Yzrg121adJpBdcYsRgTl+1t5/hiSw4A
        24PywTWhDGh7iMzK2mq1ir6+vpj/j4iIyAUma+hqj53aZQRXWY4Li9HIuk3Qfp+F
        lhkATeqfLSPLm5OGTXmxUgCJA9Db2xvp/4iIiFyo1+vo6+uj1Q25dB6rwNB81QJb
        nqT9bcoBCGVE7CpRafjuLGA7AWxZ9XodXV1d6O3tberrPSMiIq5dJJeHdXd3i+sN
        jRy83W6IAnWmEyDJGjflANh5G/bAmp6i1hE8NhVlgl1ZmzgAjI0cERFxMFGv19Hd
        3Y2uri5KXZWp05l5cmZxtU9e8qohWyLF0PIpAI0zk+zvEbAHUOOIoU39MwsXG40r
        X+4huZEjIiIONhIHoFqtitYOufSdRqqVhaxoX1K3s2oKMh0A26uxP5eEixZnLxBt
        doHlFbqeI2EAurq6YhFgRERELpjMYSt6w2UzNFKeSbsMhtrluGh//4xtPyTkZzoA
        oe737xQKpRlZ5iujbdvjTTZyvAQoIiIiL9pNHfqOp7HBtB1pUT77+VxXCUvJ9ToA
        NjVufyYJHw2veSxEcsH4+q2RXrB/rlQqKJfL0QGIiIjIhURvSAQO2k5AAq1TYxpg
        1hSkMgCsvEOa8WUaZrtdVjWqq13GJRdpPyefVavV6ABERETkRqNx5QKxVh0AV20V
        G2yDn8UuMAJIVwpFOo285wC4qjSZxteGz9Fod2LT/r9WHodRjJLnisl6vY5KpRKv
        AY6IiMiNxAFoNXCwo35t5lMapi00X1k1ca4j76xCwD0HIARVoyHLNYisNIMvKmfn
        /dPuSND8Eo+IiIiDgVKp1JIDoFEvFjKYYbHiLhkacKYAtM7C24bpoBgq7eLCNEcj
        VA4uIiKic1Eul5vSxy49xE57+j5jytOUZTLxrECualeQa13P6GqfUWvgcjJYE6t1
        UZKv6t/1d5EBiIiIaAam3sjSYb6/ZdoOLX1mPxPzXhofeyvNUNtyyrYgjehVy7Oy
        79xny2ZuAG3vNyIiIiINdjpVO9gIEf2HZgWkUXV9qHUjXicjLdJneYh5I3+7L50+
        1hEREXowA6c0g+4r5mbAdjY0ZDDrxVyy2XCx72WXB6eR/2chhLHTuLAo7WcfXJc4
        RURERGSh0Wik3gJoG0nzM2afAN419L7qewZsVjpUqrZqPqA2Nc9qOwHzeXzPIPFs
        WZ53s23Fa4AjIiKaQb1e9+ohOy+uFb0yI3GXE6NhpzTsb9rJjLL2JQ1MSlrLA03z
        RCX6YF/92Gq7pVIJtVoN9Xo9MgARERG5UCqVUK/Xc10DrK1ztWQw9aX9LGwnKi0o
        LktGm810RBpmvqbT2QVf+82OX7lcxu7ubnQAIiIicsN0AMrlKyfFfZS/ZgDJluW6
        M4YlJ0+NhbRM189lgHdZg2Z6wc7fSB+fcMkLVbjRzP+t1+vRAYiIiMiNhDnc3d3N
        PHrHNmKdelOrjbRTBRp1Br7+ZH4bYKvQqhC1aXI2VcRe8JKOU6lUwvb29r6NHBER
        EZGGcrmMnZ0dbG9v7zEAgO7ROJcuZ8mx37MuMEqeSZvRsOWaEHUAXLkNLWjlUBJZ
        rMUoedlFuVzG9vb2VRs5IiIiwgfTAahUKnufa50S02KONRwaX8CoySCn2RJRq2Bf
        vCNtKNPy4qyjIeZ7zRyUBMrlMmq1GnZ2diIDEBERkQtJ4JAwh+0WI+eFqctZ+spV
        6KdB/9vPpCEzz5yJh4WaZ+I1FqbtxLAcDfs5JI4SJp78zs7OPk8+IiIiwofEAdjZ
        2cmVApAAK2jMkmWCodu1nAyfjKzAVcQB0Hg4QLcKP5HnqgyVXigsai1xADY3N6MD
        EBERkQvVahUbGxvY2tracwA0TnDZnzGQ9hzSbLWrnovJVPsq/VUYAC3vzfWQkoPq
        isBZi5NNp5VKpT0HIKYAIiIi8qBUKmFjY2MvBcCm47XvotGGOYaMfviC1DyyWnYA
        0ugTdj4+z+ettG8PnIZDw5ZTq9Wwvr4eiwAjIiJyoVwuY319Hbu7uwB0jqppGWhm
        LZdpNzRz/i5bnFdeS1bBHESNyk1bHut4iK8IhfE8iRxW2wnq9TrW1tZEZURERBxs
        rK6uol6vH5jAgX2VsEuGRn1aM5+7kHt2bUpck1JxsQuMPDygV1vA8nrt5yiVSlhb
        W0OtVotpgIiIiFSUSiXs7u5iZWWFGim7omSWLPNVA3YgyUhRmz+3c7NgbgfAZVg0
        jsb5jhUy8vBs6p+5GH0pmXK5jLW1tXgZUERERCaSuqHV1VWUy2URXeXSTexr24sA
        RnrDDr7brU9rit+xvY1WBDYrD+CeC3XJYrSrOV6mjEqlgrW1NWxubh4YOi8iIoKD
        SqWCra0trK6uolqtirSZZaSYTIMmW63NNohcFtesQG2PTePyHXY1vvnKlOOqW+ju
        7sbq6irW19fFNnRERMTBRLVaxerqKlZWVtDV1SXath04stKu2gGq/Z5B/ZtwVf23
        ipZqAOxOSMKV42BQRb42NZwBSdiL0O5/pVLB+vo6VldXxTd0RETEwUJ3dzdWVlZE
        9IWtj7TO+pvQSnu6jDKrwFuy7aZrAOzOSML1gBpGkwlX/YIksrzocrmMzc1NLC8v
        RwcgIiIiFdVqFUtLS9jY2Gj78rAQBp91QsyFtJNpnYKmGACtKBzgfvGERk0BwD1z
        asMnJ7nWc3FxMdYAREREpKJSqWBhYQHb29siustV7CfRrov51M712+/ZsuzCewnZ
        qRaBPai2kbdvTGJUoGouFmZ9QRb1b2NxcRG1Wo3Sl4iIiM5HqVRCrVbD5cuXm/6/
        aZE3gwnwXXvr+1karR67axZZVf/twusAaHo2THkhbmPScDKa8QKr1SoWFxf33e0d
        ERERYSJJF87Pz7dcMGwaRq1CPFegpRG4suUk7ZssirQ8pzWw8+8hqBXpdtnP0M5l
        DGx0d3djcXExFgJGRER40dXVheXlZczPz6Onp6ep/2tffuMy/lLUvzblb8p2vdcA
        K5C9ygHQekjfdwdIydQ6taAJe/HnHauuri4sLS1hYWGh6Y0dERFxbaCnpwfz8/NY
        XFxEd3d3S22YQRCrCNA8Zqf9RUJMx8PFYrAvTNrnAGhXaWpUxyfv2TQU23Hybao8
        SC4Dmp+fjwxARESEE93d3Zibm8Pq6mruEwDaEbldI6bF7iavTIfDVcfAdja8CWHm
        g2oZY43FodV+O4shud97bm7uQDAhERER8iiVSpidnW3q2nBXNM6KWjVqCnxy2bLt
        9rWcmj0HQKOIop0othn4on/J4yeuNhnPZddhtDovXV1dmJ2djVcCR0REXIXkK4Av
        XLiQyhL6mE4mA2rXVmnR/5qFhSHkAL9zADSq47Wr/l3FeFLHT3wMA8tpkmi3p6cH
        c3NzWFpaajm/FxERcTCRFArPzs6it7fX+3cuoysVtab9/xDUvy1bQ4amU9NoNJr7
        LoBWkLZgWGAbZbb36WIZ2pWXbPC5ubnUDR4REXHtoa+vD5cuXWrrBEC7SDt1pnnC
        SjONnFZgyK6NA4CyVuEf82a/RAaLZdDOOSWQvBQp+U6AixcvxkLAiIiIfeju7sa5
        c+ewtra2rwAwK7pnGUjXKS7tujTNUwZ2ikML4gxAlhFmV1EyvERXUYtGMQhDxvnz
        55sq8omIiDjYKJVK2NnZwdmzZ7036mlcqOZiPO2jcMzUsW2ItU41SNV5NYO9UwAM
        Y2kLSd6zikM0Bk7zix9Yxrmvrw8XLlzA4uJiZAEiIiIAXCkQXlhYwNmzZ9HX15f7
        /zEDLS29ntU+m7GWLFD3yUkDLQXgy19LyvMNGrvAkAGJ435Z6O3txdzcHC5evNjU
        Ro+IiDi4GBgYwPnz5zE7O+vVC65jauz6qhCV/r7PpGXZNpHNbPh+Ry8C1KJRWNCq
        OGXf+ARcOeqzurqKs2fPxpMAERERAK7k/0+dOpV6AZDGpTQueQfp6J1t9LVqDEyI
        nwLIM3DM43HS5/xtaOS+zLbZVFC1WsWZM2fiFwNFRESgXC5ja2sL7733Xu7b/6Th
        ov41j/iFMsKhUhvm521bANsz1Dg24SvGY9QYANxFYht+yWdwtdXf349z587h8uXL
        8XsBIiKucfT29mJ2dhZnzpzBwMDAvt9pGGObFtcyxhoF3Xna15LtkyMaAmo8DPvm
        PVOW65Uli5VPS9pPkNz3ffr06as2fERExLWFwcFBvPfee94LgJgV8b6cOBN2caGm
        zbJPGbDlAtlfXSziALBpeJccoLOvZgzV93K5jN3dXZw6dUpFfkRERLHx7rvvYmdn
        x3sEkAXzbhjNWitNhLgIzwdXX+j3AHQSzEUYqgJVqt20/E9/fz/ee+89LCwsxOOA
        ERHXKLq7u3H58mW88847+9hADYNsRuG2rpXUu5rp6SKg2RR5Ww6AZsGGxplJrYI/
        Lfieob+/HxcuXMCZM2cwNDSk3KuIiIgiYGhoCO+99x7Onj17VTpQWsemHbezfyd9
        VDwE9W6+15RvIk/6pi0HIMT1jGmftyuLuRC1aK48C61cLmNjYwPvvPMOqtXqgXB2
        IiIi8qPRaKCrqwtvvvkm1tfX950IYp+mcv3M1EHsa+htWcD+GwwThCo2TJvPlh2A
        0EcZJNvWPHLCRN7nSNIA77zzDhYWFuKdABER1xh6enowPz+PN998E/39/QB0gxRA
        9/6WIuXdGUgL/EQZAI1BtSsYNRaMBlg0UCvMRX9/P86ePYv33nsPQ0NDkQWIiLhG
        0Gg0MDw8jHfeeQenTp2ipgF9OWm2kxEinetzOjS/TtjuTxaacgA0juHZC4RJobC+
        PMgE+xmA1k5hVCoVbG5u4s0330SlUul45yoiIiI/KpUKXnvtNWxsbKBSqTiPjbUC
        W79lHUOTgqvtkBX/7O+OcQV9rchpygFg5m3Y3lIac6FZXCgJ12ZrJg0wODiIt956
        C3Nzc/FSoIiIawR9fX2YnZ3F66+/jqGhIdTr9b3ftWuwsnQdW8eGovxNeZoRf7vP
        mekAsAvjEtjVmtIFhq5CEK1nk2YZfP1utv/JrYBvv/02hoeHYxogIuKAo9FoYGRk
        BG+88Ybz9j+GPNerdPvJe1PHa1L/ttxOqStLdQDMB2M8lC9iZRaisCpCXZPPoP5N
        Z6Ld50j+/2uvvYZarRbTABERBxzJRWCvvPKKaGDi0tfs9KotS1tmAq3baYH045St
        4Jr4NhgNj9A0ykxIOi9JMdBbb72F06dPxzsBIiIOOJKz/6+//jpGRkboOjFEepXp
        BPhy7xoRvx1QSshNdQA0zk66qH+WjATM0wvsQhdpmik5DvTqq6+iv78/pgEiIg4o
        Go0GBgYG8NJLL4l/GZir2I9JiWe1x0xVm59r32cgDacDEOIYQyKPNXFpn7FkSYPB
        MjQaV+4E+M1vfoOFhYVYDBgRcUDR09ODy5cv48UXX9w7+98uXDpb69IdVxTMvsTI
        TjWwn5Nti/ccgLQH0fbepKFxwoB5K6K5yBkyBgcHcerUKRVaMCIiQh+NRgNjY2N4
        9dVX8e6774oX/WrrDEahuI0sxlWj8t/FXkuO9T4GwFfAJgXbiLGoITOlkIDtrWnk
        gaTzTfY4vfzyy9jd3UWlUhFpPyIiohioVCrY2dnBCy+8AEDOYJmGWKPWKs0IM2Tb
        rKvv7hhmUTlLBgCUO53mt2F7hqxFqeXxMufHbHN0dBSvv/463nrrrcgCREQcICTR
        /xtvvIHf/OY3GB8f76hUqB1paxXeaRcYJu0nsjV0sMopANuIad0BrUH9a1Wcsr3q
        arWK9fV1/PrXv0ZXV1c8EhgRcUBQKpXQ1dWF559/Hmtra2JfAW7T4xqV/ux0qykn
        ebWdDWatmuu5mM9aNj0qDbDkmcaefWOhvdgPQrTcaDQwOjqKl156KX4/QETEAUGj
        ceXin3feeQe//OUvMTY2tu/mv1bb1K6Az8rHS8M+wsh0OtKifTYToHYPgAb1b8pi
        o1NTCqY8myFJjgS+8MILYlXCERERYTEwMIBnnnkGc3Nz6OvrE21bM98P6H+7np1y
        0GAeNAPyMluQxqAlclgDpzEZtgMjPW5pY2N61iMjI/j1r3+Nc+fOYXBwMLIAEREd
        ikajgaGhIZw+fRrPPfccRkdHRaJ/u+iv075DJQt5K/8l4XtWtv6lMgCu3Ib2ALYL
        0xCHiNKl2sj7DAMDA7h48SKef/75eDNgRESHY3h4GE8//TTOnz/f9r3/WQGEZNs+
        I6wRsAL769W0on5NJPJoDoBmAVsCZqW8RtUsYyG47qlOk9FoXLke+Pnnn8f58+cj
        CxAR0YFIov+zZ8/i6aefxsjIiGjbTD0VUt+46ho0T3xpXikPKNcAMNrTOBbCqvgE
        9htl6clvxXtuNK5cF3rhwgU899xzkQWIiOhQjIyM4Kmnnmo7nefTrdJGip1SsOXY
        n9mFhppgH1v3geIAsCbSjvbZRl/L+2OfKmj2ORIW4JlnnsHZs2fjiYCIiA5Csn9P
        nz6Nf/qnf8Lo6KhIuzYdLk39+04VSNuRPM/AoP59bK/m1cn2e3EHgEn9a9NEjEXg
        ewbG87TT/8HBQczOzuIXv/gFBgcHhXsWERHBxNDQEH7yk5/gwoULbafxmGlQn37S
        1PVmYaP5GdN+2fKz/kZafuJ0iDgAPlqbCYaMEEdapJ0AqWeo1+sYGxvDL37xC/z2
        t7/F6OhoZAEiIgqORqOB8fFxvP322/jZz37W1Ln/rP2tcZ06G2kpWK0UREjYzy/K
        ALAXh/1PGq48jJQczeISqXHq6+vD0tISfvrTn6KnpwflslrJSERERAsol8vo6enB
        D37wAywuLjZV+Z/FTmoZaIZuzGvctb4zJtSxP9vGtaXRXYuEMYBsioRh8ENBMm1R
        r9cxMTGB5557Dq+++iomJibaPkccERHBQb1ex+TkJF566SU89dRTOHToUEv7NTFQ
        bH3IDuhMuJwbl1PAfl5GHYVPlt22y5Fr2gFI8wZZxl+LknEdmWsXmp60uZmk+t/V
        1YWdnR388Ic/RK1WQ3d3t0i7ERERsujp6UG9Xsf3vvc97O7u5t6rthHWpMC17r43
        da9WsaFLRvLeZz+l4HM02k4BpF2QwIz8E5lSMnzeke/nVuEqvJCErz3JcZqcnMQr
        r7yCZ555BpOTk5EFiIgoGOr1Og4dOoR/+qd/wq9//eumon/XXfcakakGDd6MrBA1
        CHl/1y58zHBLDIA2ZW5SUlJwnb/XoLs6rW3gSl6xr68PP/zhD3H58uV4OVBERIHQ
        aFy59Gdubg7f/e530dfXl7texxWZMva2bYi1bIdt9MzUBvP22OTVlKVdXJhnfHM7
        AK4JZFMmbI9Uq6JVeqy0ayIajSvfEXDmzBn86Ec/wsjIyIGulI2I6DSMjo7iiSee
        wKlTpzA2NpZLt7loYa3LfTRuWHXpMY2I33ehmzbDYNdxuOSLFQFKgpkrd+W7NOoW
        GDK0FnCj0UC9Xsf4+Dh++tOf4rXXXosFgRERBUC9XsfU1BReeeUVPPnkk5icnGxK
        L5hf7MPQha6z9drBg9b1uja00wuAv6bCJ78pB0CTtmFNmoZXxqa5NKtWTfT29mJ7
        exv/+I//iFqtht7e3pgKiIgIhEajgb6+PtRqNfy///f/sL29jb6+vqai/6zPJPro
        +oytN9Jq1LSofy3YQW0zqZxMB0DTa9OkaVjQyKMB/IpVV/+TY0avvvoqfvzjH2Ni
        YoLWh4iIiHQkBbrf//738eKLL2JqaqppVk4r55+817Ijyavr+RhFhkC4C4TsgLCZ
        wvPC3OyiMUk29S8NtqNk5nSYjlHaM5RKJYyMjOD73/8+3n333ZgKiIgIgKTq/623
        3sI//MM/YGxsLLfu0ShKc1H/GpExM32cJlM76rfhOsmRJ6XTdBEgA8wF6TPKLGq+
        0xgLE3kWcaNx5dsC19bW8Hd/93cAEFMBERGKaDQa6O3tRalUwre+9S2srq7mPplz
        kPepqeu12AZXvZT9OybaTeWkOgCaBRvsfDm7YJElQ8uzNOc6S14Sfbz44ov44Q9/
        GFMBERHKmJqawhNPPIEXXngBhw8fzmThXAyrVs4fCFOAF7LwLumDhux2Uh1eB6DT
        qRof7cRiGVibyazSLZL3XiqVMDo6in/8x3/Em2++2fK1oxEREfmRVP2/9tpr+M53
        vpOL+jd1LSMy9ulXuzBNGq7gSzMCD0H9+4y8fTw/77PnKgJkgE2XM70ye5GFOmYi
        gVYXcJIK2NnZwbe+9S1sb2/HC4IiIohoNBoYHBzE9vY2/vqv/xpbW1u59pypoxgG
        y2f02frQZixDXLYTQp4rqLWdobxzvM8B0C7UYIM1Oa4BN1+lId2ur71m5SSnAt58
        80383d/9HUZGRlCpVCS6GBERYaFSqWBsbAzf/va38dprr+Wq+g+h0zVuVvX1gZmS
        TdPzIdIpNrPTCsohKJPkPVMGY9Fr5dFMWcy6CCkcOnQIP/rRj/DMM8/kykdGREQ0
        h3q9jiNHjuDnP/85nnjiCUxNTTn/LjQdrZmu9MnRKCZn2zFgf6CZ9qztsOnlEBSG
        Vg6KAS16iy1DaswajQa6u7vR09OD//t//y/OnDkTvzAoIkIQtVoNhw4dwqlTp/BX
        f/VX6OvrQ09Pj9cgaBgnW5Z9/EwTna6L88hNO2XQzjyXkwakvba0tlg5KLZx1qK2
        tDawLa9V1Ot1jI6OYnl5Gd/4xjdQr9cxODgYnYCIiDZRr9cxPDyMer2Or371q1ha
        WsLY2Ni+vZWVE2YjqzAtlHyptkMUGLrgov7bfday2bAk7KITrTy5Jv3EaFOzPkJK
        VqNx5bsCDh8+jNdffx3f/va3MTw8jK6urlgUGBHRIhJ2bXR0FH/zN3+D3/zmN5ie
        nkatVtv3d6GLj31XkzMCSu3gKKsfTBkaMlVuApRcoGkLjJX7ccliyGD23/yZVR/R
        aDQwNTWFH/3oR/jRj36Ew4cPi8qIiLjWMD09je9973v43ve+h+npae/faVL/Lhls
        /Wh+YVECjap/1ykvrdMN5s+s8S0zqX9z4qTguuqQXYiRoJ1qyyxIPoerOpQ5TqaM
        rq4uDA0N4Zvf/CZeeeUVTE9Px1RARESTqNfrmJmZwYsvvohvfOMbGB0dRXd3t6rh
        TYNpkNnF1rZMLbZXm1UOAXEGwJWn0AKLlpeW4crZMaB5qiNBkv8vlUr42te+htnZ
        WUxOTl5FW0ZERLhRq9UwNTWFCxcu4Mtf/jIAYGhoyOtImwyipr5lybODlhC1BGly
        mWlsF8PNHIPCfBlQs2AuDnMiGNWtGhdzsHNxaZ/V63VMTExgYWEBX/3qV7Gzs3NV
        4VJERMTVqNfrGBsbw/b2Nr70pS9hfn7+KgfapZ+YUaqrjksjKvbpSY2AyXWznubz
        Jj/bgbT0s4s5APYC0fTapGXZ1+9KsxkaBZGsEwu+sbY/T6KYt99+G1/72tfQ3d2N
        gYGB6ARERHiQsGe9vb34yle+gn/+53/GkSNHrtozrhwxU9/6ivykYQctPpaUbYhd
        +llLpvlznmCrXYg5ABrUv8/zlDbMyXvmhRLS7bs2i7TXmDY+vjmYnp7Gc889h//z
        f/4PRkZG4jcHRkQ40Ghc+Ya/0dFRfP3rX8fTTz+NmZmZpvLhjD6Z8mz50vAFLdpp
        ZPvZtOS75pIdSLftAGhQM8z283peUjJYnqzv2KWknGadvGQzHT58GE8++ST+/u//
        HpOTk6hWq9EJiIj4HRqNBqrVKg4dOoRvf/vbeOKJJzA9PY1yuZwaFZufa4MpM40K
        Z8AX8WucMkigXW+QoCUHwDZozGp8Nl3uyvXYsiVksL11ZrvtyEiU28TEBL7zne/g
        Bz/4AQ4fPoxKpRKdgIhrHo1GA5VKBUeOHMETTzyBb37zm5icnHRW/GtE4ZrtJ/AF
        YexTXr7AMjT17+oTCy05AGyKRDv3w5xwLTrL57Ey0gzNtpnQm8PDw/jbv/1b/Pzn
        P3dGOBER1xIajQbK5TJmZmbwk5/8BF//+tcxMjKCvr6+vby/ywnQKsKz+8qC7w4D
        zTRy0g/7bzVgOjzasltOATCpf42rfbWoNMYpAh8Y9RGueW6lzXq9joGBAfT39+Pr
        X/86nnvuORw5ckT96FJERBGQKPyZmRk8/fTT+MpXvoL+/v6rrtDWqPI3+6QlKxSS
        IMk2ttp5/rT+2L9nomkHwI7OpQdPayJYVfI2NAp12N65lJx6vY6hoSF0dXXhf/yP
        /4Ff/vKX0QmIuOaQGP9jx47hueeew1/+5V+iu7sbIyMjub7eV9oouAItX1TOkMuW
        40OoiD+R5arv0O5TywyARmGG6WBo5USkEMKgSbMNDOeiXq9jZGQE1WoVX/nKV/Dr
        X/8aMzMz0QmIuCaQ6LKjR4/i+eefx3/9r/8VlUoFo6OjqZdlaR3zM/tpvrJg63Z2
        4Z2dAtCum0qeLfSJh2QMmnYAmIraNPrJz+bvpGUdFIPjWkwS1D8LybcHlstlfOUr
        X8Evf/lLzMzMxJqAiAONJOd/9OhRPPfcc/jP//k/o1wuY3x83Pvtfoy0XlpbmvvP
        d/FOKDqehay6jZDBYls1AAxoRPudbmS0vFeWN95oNPZuPCuVSvjyl7+8VxMQTwdE
        HEQ0Gleq/Y8ePYpf/OIX+C//5b+gUqlgfHzcG/mb1DwrfZj2MxP2s2npfTsC10JW
        5K8F03aUSiVUbrrppr8A0OP6Q+18BGNytKorNb1WNkWm2XZ/fz+2t7fx3HPPYXx8
        HHfccQc2NjZQr9c7Lu0TEeFCo3HlS7KOHDmCn/zkJ/jv//2/o6urC2NjY07jr3UG
        3jzC7ZMvDU1ZLmie7U+Q6GzNgnCXfAe2vQ6Adgc1ZTJvsWK1L1GJ72qTvdnT7lko
        lUqo1+vo7+9HrVbDs88+i4GBAdx9993Y3t7Gzs5OdAIiOhr1eh29vb04fPgwvvvd
        7+KrX/0q+vr6vDn/EIFEqPP+oWVrI0t3s+xG4uSZ+vd3r24HwGWQOzlyZnp9WhuJ
        1bYZAWQdSWmlbZccG43GlXsCSqUSnn/+eZRKJbzvfe9DvV7H9vZ2dAIiOhLJ3f7j
        4+P41re+hW984xsYHh6+6pv97NonFmw5IfZViAg8kRsaoQqdXYHX7/qRjwHQov47
        rW1tsO5b0GABstZQo9FAT08Purq68Mtf/hKrq6u477770NXVhfX1dZTLHfvFlRHX
        IGq1GsbHx9HX14evfe1r+M53voOJiQnnF2JpnQE397u2MdJycnyyNWXZrCcQ1vin
        jP3VDoD9AExoeaLM59DyaF0UumS70kij+3zyGo0r1wb39vbi5Zdfxvnz53HPPfdg
        ZGQEa2trB8qZizi4qNfrOHz4MHZ3d/GXf/mX+PGPf4zp6Wn09vaiXq8Hqa9yQXM/
        2QZIo85AU6YtQ1uuDRfb4+jHdjWtEWmvxeeJaFS4MsD2atOORTIg+SwuzzPPc5gF
        U88//zwWFhbwxS9+ETMzM7hw4YJKtXBERCtI1vXRo0dx/vx5fOlLX8Jbb72Fo0eP
        olwu70X+doSopasSmHtTA9oOTxH0gznHIesPsoL5fQwAe0HaE8POyxdhIbQKDeei
        yCiXyxgeHsbZs2fxq1/9CseOHcOJEyfiCYGIQqJer6O7uxtHjx7Fyy+/jP/0n/4T
        zp8/j6NHj3r/jyYt7CgA2/c7SbhY5BBV75qyXTJDG/4c8n+fAtDyCDXkdLLhLLph
        zgN747U6341GA0NDQ1haWsKzzz6Lvr4+3HXXXajX69ja2opOQEQhUKvVMDw8jImJ
        CXz/+9/Hl7/8ZWxtbWF6etpZ7KcN7Yp/Vw5cC6EKHNPkaZ90SKn6t3HFAWg0Gj3m
        f2R0TKvKFeAep0jAYhg06iE6yXA2Go29wqlnn30WKysruOuuuzAwMBDrAiKCo16v
        Y2pqCuVyGX/1V3+Fb37zmxgYGPBe8KNJv2fpW608PKt9V9shCxwT+aFg2ycgcx6u
        rgFgDF6nK2lXNaf0GGlSgZLQKBpNvkmwu7sbP/jBD3DmzBl8/vOfx7Fjx3Dx4kXU
        arWOX2MRnYVG48rNfjMzMzh9+jT+5//8n3jllVcwPT2N7u7ufcbflZLUOv5s9znt
        53bhMoSu2gMJuBypEDS8Vn1WXvjsk7cG4Pjx439RKpWuOgbI6pw0tI+XdKKhOQgF
        N4nCHRoawunTp/HLX/4SY2NjuO2222JKIEIVyfn+Q4cO4amnnsKXvvQlnD17FkeP
        HvVeZR2CFtfMhWueHjMRstYgkRda7/iOHubAduXEiRN7KQAW2AYo9JELKbCi804c
        kzSKb2hoCGtra3jmmWewurqKO+64A8PDw1hbW9v7m4gIaSR65vDhwwCAv/mbv8Ff
        //Vfo1Qq4dChQ149xzSMRTlSGAKuYsZQzx6ytqONUyXblePHj/8FgB5mrkYLoSsv
        80LTW9bKwTHTRi46q6+vD11dXXjppZf2jlodP348XiEcQUFype+RI0fw5ptv4r/9
        t/+Gp556CpOTkxgcHLzqch/AfQ2r9LpMy4NrOf85C87E5PheNVEUO9NmymO79Nhj
        jy0DGGq3I75JYFbjsw2+3TZDnmb/pdvWcGLS2k7kX7p0Cd3d3fjEJz6Bxx9/HI1G
        A5cvX973NxERrSDJpx46dAilUglPPPEEvvOd72BrawtTU1OZa5+VA/fJsg0jGyGo
        f5fBC1H4pyEjzaYKPPOK9yrgZqFd4GLLZtDn0mgjV9O0HKAzr1huRaEMDg6iVqvh
        hRdewKlTp3D06FHccMMNkQ2IaAv1eh19fX04cuQIfvvb3+KrX/0qvve972FgYAAT
        ExPOqD8kDvI6twsM7Yp3rWAn+dnVD2mktS10Ik2OAbCh5ekCnReZd6oMV62FRvSf
        Z36Tv5mdncXAwAA+9rGP4SMf+QgqlQrm5uaC0IQRnYlkrRw6dAi7u7v4wQ9+gL//
        +7/H6uoqDh8+nLkeQ9LDWsxoqKN2vgLLgxb9Z8kX0r9yDIDZGXa1aQLmkTYmmGkR
        gLsh7OOQDLTT7tDQEHZ2dvDCCy/g3XffxeTkJG666SY0Go14UiAiE/V6HUNDQ5ia
        msKrr76K//W//hd+8IMfoK+vDxMTE7nWplB0loksyl9CrksnphliFrROM/igeaIi
        qx8JXPPSJGQZAO0ckFRb7CpatmHW8siZ+T7J9pKxuHTpErq6uvDoo4/i8ccfx8TE
        BC5duoTt7e347YIR+5Bc5Ts1NYW5uTk88cQTePLJJ7G9vY2pqalcUb9dkBYiUmTp
        lwQhns1ldLVZiNBjYEPIprTPAGjlspl5F62FxKqStSMOSbg2GmO+GW0ODV3xa196
        6SW8+uqrqFarOHHiBAYHB7G+vh7TAhF7a2Bqago9PT342c9+hv/9v/83nn32WYyM
        jGBsbKzpqD/P55Jg6y6fztU6ZWDKY+uhNNkHEPKnADqpql3bg+uksTHb1GQXGHLK
        5TLm5+exubmJe+65B48//jjuuOMObG5uYnFxEcCB3eARHiTrbGxsDL29vXjllVfw
        xBNP4Fe/+hX6+vr2DH+e9agd9WtGvZonu9L6AehG3q70Clt+WnqBJHMl9euA88Ds
        GKPgTPqcp92WxqRq5AAZyJv7aweMjWVvpHq9jrGxMdTrdbz88st4/fXX8dBDD+HR
        Rx/FTTfdhJWVFaysrOz7PxEHE8naGB4extDQEN5++208+eSTePrpp7G9vY3Dhw/v
        ++revO1psEm+6JepBzSDAFOm67RUCEdAE1nrhzEHbTEAzMXho54l5LAXknZeKq0Q
        SKptSWiNv2tcyuUytra2cPnyZYyOjuIP/uAP8KEPfQjHjh3D8vIyVldXoxNwAJHM
        /9DQEEZGRnDq1Cn85Cc/wc9//nMsLCxgcnISPT09mYZfo2aoGRwkxiGvPGmdlOW8
        harlUHDyVtpOATCL2ljtm+1q0dvMthkbAuBetOGi2KTbTt67UC6Xsb6+joWFBRw+
        fBgf/OAH8YEPfAAzMzOREThAMA3/8PAwzp49i6eeego/+9nPcPHiRYyPj6O/v7+l
        M/1aOkQzvapNe2f1J4EWA8EIOPPKDZBuac0B6LRctktGp1bkaxjnRA7b05bufytt
        lMtlrK6uYmlpCUeOHMFDDz2E97///Th27BjW1tawvLysQu9GyCKZs+HhYQwODuL0
        6dN4+umn8fTTT+PcuXMYHR31XuFrt+H7HRAugCjqMdx25WrS/T4Hx3ekUnNcNJw+
        NOsAaOWazJ9Z6NTNw3QANPvOkmev0bxjUyqVsLq6iuXlZUxPT+OBBx7AAw88gBtv
        vBE7OztYXFyMXzvcAWg0rnxr5NjYGLq6uvDOO+/g2WefxbPPPovz589jZGQEg4OD
        ba89jSDC9TnQ+UW5aTJNaDErefvDkh2QacnnAGgZZ8ZA+KJOabhoM2l57IWiSTUm
        KFrdQqlUwtraGpaWljAxMYF77rkH9913H06ePInu7m4sLCzEC4UKiEajgZ6eHoyN
        jWFnZwdvvPEGnn/+efzqV7/CpUuXMDo6ioGBgY5wlDXlhKb+s2Rppj58fWAiEPWf
        IN0BYA+OVq7fd36U4WR0anqE4bhkKRSG8Zdqt1QqYXNzEwsLC+jv78ftt9+O++67
        D7fffjvGx8exvr6OlZUV1Ov16AwEQqPRQLlcxtDQEAYGBnD58mW8+uqreOGFF/DK
        K69gbW0N4+Pj6O3tFTf8Iah/dkFukah/bdkhGA+7H4D6FxzlZwDY1D/AiwZtOQyE
        9pTbbZsFrQ3Gar9UKu2lABqNBm688UbcdddduPPOO3HDDTegUqlgeXkZGxsb0RFQ
        QDK/fX19GBkZwe7uLt577z289NJLePHFF/Huu+8CwF4KoJP2ii0nhFEsAvWvIc91
        kkMriNM6wpkDfgdAe6FLD0TIfJmETNeC7EQHgC1Da7OWSiXU63UsLy9jfX0d4+Pj
        uPXWW3HHHXfg1ltvxaFDh1Cr1bC8vIzt7e3oDAgjofiHh4dRqVQwOzuLN954Y+9e
        h8uXL2NgYADDw8OiNTEa6UNbphZceovdjzTjWwTnSpvZ0V5fFrIdAGaFuS1LGqyF
        pRHVahp95jFClzxJsMbel5dLPt/a2sLS0hJKpRKOHj2KkydP4tZbb8VNN92E8fFx
        7O7uYmVlBdvb2+L9u5bQ3d29Z/QXFhbwzjvv4LXXXsPrr7+OM2fOoNFoYGRkBD09
        PR2ZfgvhZJhyXA4Ae68yTwI105eQlL+NQjIAWlWuTAPNgMaiZbAKdvssdDJ96IsO
        0o4Gra2tYXV1FT09Pbj++utx4sQJ3HzzzbjhhhswPj4OAFhbW8PGxkZhFE4R0Whc
        yen39fVhcHAQAHD58mX89re/xZtvvok333wTp0+fxsbGxl7eP0QNERusqNTn2Prk
        aiDEuGo6Pc30y4SGbYHtAGgZNoYMW3FrVrMzi9nszyTBZC4Y0Ny0zbZdKpVQq9Ww
        traGtbU19PX1YWZmBsePH8fx48dx/fXX733hzPb2NtbW1rCzs3PNOwSlUgnd3d3o
        7+9Hd3c3tra2MDs7i/feew/vvPMO3n33XZw9exYbGxvo7+/H4ODg3jc5alG1oY77
        aSCLzZROZ9qfa8F+Tm35PgTuwxUHoNFoiHwdcF5obaZOMpzstjUdJI3USAKWI9lO
        m4kzsL6+jrW1NXR1deHQoUM4duwYrrvuOlx//fWYnp7G2NgYqtUqdnZ2sLGxge3t
        7ZZupOsklMtldHd3o6+vD93d3djZ2cHCwgLOnz+PU6dO4dSpUzh9+jRmZ2exu7uL
        /v5+DAwMoFKp7M1JCIpaA5psQ6ho33UiS9uRC1Hsl9YvTVj9ujoFoDEonWiYNdpP
        ZEguUI0co4/CaldeVuTQCdFJUjy4ubmJtbU11Go1DA4OYmpqCjMzMzh69CiOHDmC
        yclJjI6Ooq+vDwCwvb2Nra0t7OzsoFarifRFE6VSCeVyGV1dXejp6UF3dzdKpRI2
        NjawsLCAS5cu4dy5czh79izOnTuHixcvYnV1FZVKBYODg+jp6VGvltam+0MYI02G
        0dcXEyHy8KHz/6HGwMGg/v7bADWNJ5P+Z7bNhN1njZyfFFxjLtF/lwGQluECY/30
        9fXtGffd3V1cuHBh78jawMAAxsbGcOjQIUxPT2NqagqTk5MYGxvD8PAwent7UalU
        UKvVsLOzg52dHezu7qJWq6HRyPeVtQyUSle+D75SqaBaraKrqwtdXV17fd3c3MTy
        8jJOnz6Nubk5zM7O4sKFC5idncX8/DzW1tb2jc3AwIBTDjvyT9o2dYgtmyUrec80
        Aubz2HLYbJ39nHa/tO2O63M2ssZYqy8uW1D6yEc+sscAsCv+2dGnRkW+ptGRLP7R
        QKetHVuG+RkTtkKo1WrY2trC5uYmdnd3Ua1WMTAwgNHRUYyPj+/9Gxsbw8jIyF4B
        XE9PD7q6ulCtXvHj6/U66vU6arXa3vvkX+IouIypbSCSf+Vyed+/SqWy9x644sjs
        7u7usRsrKytYXFzE/Pz8vn+Li4tYW1vbe7be3l709PSgUqk0NWZMaKfIAL3v83C9
        T3DQUw4hZLr0SSjnw9UHY33/ngGQ7hhTebNpW58cdvv2eEnS0GwDKq3INCMGDSWc
        xpBUKhX09/ejv79/73e1Wg2XL1/GuXPnsLu7CwB7xXIDAwMYHBzc+zc0NITBwUH0
        9/ejr68Pvb296O7u3vtXrVavMuBm1Gn+SxyIWq2G3d1dbG9v7/3b3NzExsbG3qmH
        1dVVrKys7L2ura1hfX197+hjtVrdo//HxsZys1GuNcWGKzKWQBYLZ86BNHzRvqln
        NCL/UAhR9OeMsgOPRZrcqvc3bYK5aV0Gh0H9m68MMJQOS3mmKTKG4nRtGqaSZEdg
        zfapWq2iWq3uOQUA9ozz0tIS5ubmsLu7uxfhVyqVPSo+MfyVSmWPJUh+ZzsBLuOf
        pBeSOoQk9bC1tbUX9SdXIJfL5T36v1qtYmhoaI8haHWcNBkZtpPhS2OFyv2HMoIa
        0AoMm+1HSCcI2L/W7PVAcwC0FhprYYf22loFK5JxLWjGRm/VaLYiJ4GG8pdwMBLD
        3dXVtVdPkMA05ElKIbm50KwVsFMBZj9N+t+VAkhkmlGkBIqitFntuvaOluFP4Fp/
        EmsyL8PBQlGMrakTfSxAyDSIb47EHQCmArcHWYvWZkGDNmeASVsCOvPMdB6TNrWi
        vASm4WbI0cpZa+w79j5JM8Ia9QX2q8sZZaQ5fO9ZcD1XCGOb9ZxFCSTtfshrCjI0
        Nq75ypYjLUsrbyrdvkshMZ2MBCEMQavtsNpuRV5RFFqrSNYWax2brIq5jtnGyWZz
        zH653kvIc6UZNFMOppy0VJJWH5L3RdgjWeMh5gCYC5+tmFiDyzKerM1ng0WF+aIK
        adj5URbstcpqG5BT+PbcsiMdX9uSUaNrX2goTTZlzKLbs2DrLy3myUV/axpf2/HQ
        ZAF8MkI6yuY4uJwyEzQGgEUPsiNPxsKxKTFpR0NDedp9ZzpgWqkMSWNmwi6uY0BD
        wTFTPRp5Wx+jpBWB2/tcI/rXeEZbFuAeaw2YdiFELYArwg4d+TfDgrRVA+Ba4Aww
        I2gXfcWAq31GlM4Ae2EzI03ptrLallbyLmeIZUhsR5gli+XM29AuTnPNlWY0qg0t
        3ZkH7CAuS3ZRxqJZuW0xAGaRllS0oDWAWrQ8ExpjpOUYAfLV5Un7TGPDjjxc1D8L
        rrw1myJnsz02LcxyZtOoVk1dxoZLJ4Sm/vN+zpKtXWvQDLLGou0UAFPh2e1LyLLb
        01osNi0jTf8z22EpS7Ycdru2DC2mQcOgsI0yO52UQMuBco2XdoGmFtIcXq2gxBV1
        a8I1t0UIJJtdcy07AFo5puR90du1YSu3EI5HM/ApMIZnn8YWdQr1r9G+LUcjUrZl
        SrZtK24WQkXEWjnorOp+puPhMrqa+swOnkJG3/a6DolW5TdVA5DmuTMjUXZkqJFT
        lW5Tw3nJ+qwdsPN09txKrs8QxUYMuIwxswg2kcmG7eQzI8W0tln735ZpzxlT72g7
        V/ba0XjWrD4VBeb4tLpvm2YANDx4JiQ3atr/66QoJ8Qm6jS63OV4aZ9UaPdZ8rTF
        nhfmyQLf3tZab67UXiedZklr35e2ZNfX+D7XKvSzZbhOHBQBZj+b6VsuB0A7v8QY
        XKb375LD9v4l4VJaknJcm4gJjTHScmBcsluFrw6FySqZ+4FdKKaVE/btcZejwWJT
        QkWirn2gSf9rP7fveTWYWFOm62ffWmhmjDIdAJfRYVLyrAXFXDzM/KndHtvwMObA
        jvokHRnNsbHfs+RpKnmN8TPHi2kwbMXIdjRCRYGaxseUqWn40vabptNhytNK89hI
        S9O1Kz/VAXDlm5hgRyBsMA0CmzLVSu1Iz4crJyhJlSc/m/uAlQbwGWPpdI8GI2PL
        0djbLkdTWpb9WRo9zpCpgTxy2X3TTjfYcK3d0I6fCalTJ02lABjKTtqb06abTRnS
        sjQjTY3cJUtJSrfrKzxyzYfE3PgcGK0oS2p9+faeRiSuYZB8OovJWpoyzFf7PUNu
        CKbBBWbdiIk0hzVE+sFG1pprpX9OB8D0bAEOJcnKJfmOqUghrS2GMdKMzNljxopk
        mRvT3ANSEUiak8qumGcVULmYEW3KNkRUqoXQ6YYQ/dB0egB/TRdTN2bBdkqkZTsd
        ALtAKLTn0w6k+67plbM2m5bx1Ghf0jl1edi2ApB0Um3ZGnUq7KgxkeX6XArayth2
        ZDQr/bX2alo0qR1MuZ45BPVvI4QT5mJjJOcglQHQXHDSYFM27AXK3PjsYzTMsXFR
        5IwqeZ/sduFSptr7jBH928/CNBQh6Ngs5k+qfVdbmmlMsw8hTuuYstlsmKvtNJmh
        A2GGTd5zAGyjH+J4g2R7nahUTSUgXfCitZhdEVKnRrUJpGj/NJo8+RtpaOXHEzmM
        uc96BrYjq7WeXbI0i7BthJCpZfSzZIY29jYkU5AmvEWA0h68xoAyi0VcUYg0TCUq
        /RwahTQ2VcoEMz3CkJO1eVkFpNpgpxg0n8sukmUGRb6aKC3dWTTDF4rhCVnxn/XM
        VAaA+cB2PQEz8tfOx0lGhwywWB0Xo8BcQxqK0BfBakSYbBmstkMaDpazpp2m8UHT
        +ISu9DcRIt/vYx5Cppo0nMCyLYDtwTPb1oj+JehA3/9xGSAJpB1ja6dNsy1tj1ky
        5ZLAjvikkJUyYjjEGgwJu5bEhTR5Uv3wyWDqFk32zJatIde33n0GTwMuu1Ekh09D
        l7b9dcA+aHlR7AljRM9pE82g/s330ovKdC40GB5JpPW9VZmu/xeKRmSMW4hI0TaO
        bOfGfNWm/tk5cK2Az4bPuQ5Z42DKDWF8XdBI1ZqgOQDMibU3KAtsKlB6orOUv4Q8
        l8euNQdA8S8oypPLDaXs2oVtiLUUVQiWIYFWZOxj1Biw2brQY6vhaJmw57QIDomm
        0Tch7gDkoXqk5LAnq5PobCDd+DChyb5IU//s1IW2YtNqW6sI01wD7KBC29FIcwjZ
        Oi10zt83zxqw15Ht0IYs/NNizhNQGACm0dHKW3Vq22xoRPts5eA6fsdMW7A3tUtx
        MZgl+6gfm43R2utZkC4u9M2XhqPuYm/Y9S55fpfn95JIe+YQ0X8otO0A2JtUO2fG
        ANMLLMoibxa2gpJ6Dl80zjSUpgwtCpt1MsJsn5lusz/TNMihFbI0QxeieDJNN0v2
        I89RV+0o24WiBGCaR4NdEGUA2JS8Zr6oKAukWbCiTRYFaypC9phr0vHskymJDG1j
        3InHbPMgFP2vLSv0M4bUq669YjtlocB2zHyotvOf7Tw8YxNpLCKN/B/TKXLltKTb
        NttljlWIqFLyeVzzYb9vtV3A3VeTmmdS/+ZnDDDGrVn5nagDWpWvUUPlMrZMmT7Y
        +8NmAEPNTWj5IgwAk9JJPDOttIJ0vo8N5jE/1waWUpKuHKRkbtmXsnBFy0zqXGo+
        fHPLMs5a0Zsme+GTr9WPEMylyZj6ioSZMGWzWZ20Z3E9uwbSdE1ohxBowwFwGX02
        9a8BDQZDoj3XwmKMk21gpPOhSbsmJGTYCk86yvf9rLEn2BXxiYwEGlF/IkeT+mdG
        X742NQyuT66GDvVF1RqGL6uA1CVXu+LfZh5C10K07ACwaGcT7EXL6ruPOpeARq7I
        FzmzwCzIczEY7cJuQ4tONBUZy5nJ+pwB+3SBhvw0p40hKwTlHYJetvdYiCjXXEtZ
        8jUdMhfDGZoFaMkBMBcWk9bqxCiAmQ7RArtCnt227zOGXI10gguSqQUT2mybpgJM
        Y88k2wf8TqIWNFKnaXAZYXYBqQtF0cVFo/4TNO0AuDz1Ij1QGjQngW3gWBSw3T7b
        uZNeP2kUqDRsg6JNJzKgnae2U0zSsjWdGR87F8IIadH95vusYEojlWQ7HkVwDEIw
        MXmR2wHQzs0xHAsNj1TzWFunpEd8UTmr4NJlmFlzbb7vRMYkhKy0ok/Wnk/es/PP
        LkN4kBgvH3xFhpqwjX/antdykEyWXJsF8vXJfJ/bAbA7z/bmGAqbSdeyjXFRKaQ8
        YEX7dvvmz77oRAraEVYChtPEcrh9cu1/TFm2TLZecRWfaiAEE+tLH2mmHooQ4dvw
        sbRF0Nv22sx0AFx5rZA5naK2m7TN2gAa3nXSdy3HiMkuALwjna5xYjk2LKfVbltL
        Ybpy0xo0OfvURAgDDPhvF2SmUnwBSYjiNte8ajq0aSha0ObqT1v3AEgrcKmJcy1Q
        U8kwNgerLqIZWqtdOeYrUJwCmjSYBYuSY27LMH+nmSPXaJudvrB/1jAY5nrWci5M
        2ZpRn0uWtFw7cnSxHKEiXXM9FcHQ2o5YkfSoaQtzpQBsBctSsoksCbjyLazcvLSC
        8fVNsu95FUU7suxNyXQsTMdOQgmmFRJqRJKu91LQSI/Y8lxymHJZrIlPjjbtbcoO
        jVCRdpo+CVV0maW7i4RMBoDtzWkcP2IX5JltS1FvrqiINS6uKEmryJPRBiOKdK3L
        Im7oZqFttFzUPwO+fchkGPLIl0aogjttdsMH27HMo+uZc+LqT1pfmPCx6ub+S95T
        vg44L1gRuf1eY+Jd8tuBzSwwomaGs6GpACUdL8DPSmkVpLKj/wTsokwbLkdZSyYL
        vv3Dls9gY/PITJBF/4dCHvlsJihEDURWP1x9MnWn0wHQ8PBYlLCLcmY4GNJt2u3Z
        ETkjamZRzGb7WgZGQo7vNAGLnUpgO3ssMA2Hr13NaFF7rWmzKD6dqc022H3RRBEY
        iLxytVMRvnky16i9T69yALQXdSKziG3Z7bJoZ1fNgqSydjlELEqUDTvqb3ed+vpt
        e9PMSIdZva0F31wwdYkGY5K1Psz3DEcuLfpOfq/tXGkxVS4UoajOFyjYKAL9n7z6
        9Ms+B4BNn2tTVxJwReZazIgU7CI5jTwsg7EA5CuOpQoHWwWb+k9rT+NEifR+zzK2
        rAJT32cHgb3JI9vsA8B3VLMMahGif9s5K7JdcwWZgOEAmAuZMdCdSgH6cvxSOWcX
        rcc2AswoVrLtNEoreS8Nu5iISTdqsG1p88Jil5jK0FWsyjYMvmdk6ssE9liaMkMW
        cLKDoLRncwVlGnDpHpd+CHUKwZRtj6FPB5TN/8w20tJsQtrPkmDSO5qMAiMSs8Fc
        /JLGxRdJaqUXzGdhUccaTreLDmXD5aAx4XM4tfatb4w15JvMg5aR87Xt+1wruMwq
        rtPsjwut2PGyBqXDcDA0vFEm5WUbs1D0Xqtgjn+awyVlLNPSIVIKNssDLwKl2S7S
        oiJpOQlCF93Zv2PAV7SluU7SdB9T56a1H2qfuPZyEeoR0pBHV5YB+ShUe8EWeRJc
        0FJmGs5d0r6k8vetR43IJ2mfQeWyo2Wf48SYf9cYsZ9Hg6nMuzc1mQ5bHtvBMmVq
        MjzmWrUdIM0AqRn7paVj88g3xy6vrqzm+aNmoTkYIRaFZJu+4gyJthlwRXyAvBPp
        UnjMdVpE5dIszDnwRZASSHMqWGvArmVgzVdWpM+odckLdm2Fy+HSLJL1sXHaY+2i
        +tNYSd/vtdCObNpFQKxNorH5tCKM5OdOfh5fXqwVuBR8yM3fDmzloGk0JI1UWhv2
        c7H3uwZbZr9P+zt2PzTl+qAt17VvigBbz4VmnaXYN1EHQFtpMyKbUIZHEpq0lDle
        7Y6ZK+JjgKngXWPPVhhaDp69P5qhGtuRr1X5bs6dtvFxMTe+v2HKdv3TREjD6tJl
        RXPI7D60O14UBoCZc5SkzH15LybsHJcUXAtWcpGa49/JjAXgzi9Ktq2pvE2Z5vyw
        1rFtGDQUoRb1r2188qTQmP1yOTz2WGsHcyGjflcO3dWnIrIxrfZJxAHQpG0k27cn
        mR11MsdGQ3lpVf6Hiv6l2mc6GHnkm6+S0Hgul6JlR/3aMn39cDlVGuOcIFRNjM0q
        mfKLUFxXJORhiszfZ0GMAWDTcxoIoagl2mAofW0vlxW5upS7GS1LQztXqK2o2c69
        i+7XYhkAXfrfZm7MfrBgt+/bC1q6PKuGSHPuteS1gmbXSN7naNsBYEdq5mZkUmEa
        NJs0zEhZ2vjb0R5DMfraY7ALrnlgshgsGXnlSrdvHzNiIG/1tbTMkIGLTzZzjO35
        TBDCENqOVwj4nJCQ6QgbWadTWkW13QaYx41sOUwqTKptlzPBZkekx0areNBF90nP
        sWYhJFuJhWDZfAyKVNu+NJxWrYnp1IRU9q7InC0DCFfwWFS2uCgG3wRzPzTNALjy
        Uwzq2XxgtvGXhC/CZEa1rLFh9t+X75SCr12WDE2FxlTYdruSRbdZ0DT+JrQMoL1n
        NQ1v2loNyYLY6ysUY1ZEw2/CtS8lQLsHoFVo54G0i+eK2qa9AdmFV+arBhjK1q5b
        YBpll2Hs9PQCi9Z0wbXm7CBDG5oy7dMbAD+ISOuLKV/b+LrG3ZfeCslUmGuTtVda
        cgCKMmmtwFx0kpGGy7s3fyclQ7I9s13XWDAWv7bRBzjOkqYhTssTS+4LWxabYfLJ
        12LmNOfQN74hUjra69eUqykzL8wx0UjF5IWZnmH1o2kHgFks5zOgyWcSMphFYFp5
        S/NVArb3y1x0DErZ1RYjKnfRqJqV/rYR0XDENQr+TLAjco1ixjTZgL7jkbZOipKL
        D9EPe1yKMhYJNJyP3A4AS9mY7bJo7QQMys9kEzoZnUz9+5w6aSfA57gwDXGytpiO
        K6Cn/FyOPTMn61tzGs4T87nywl47tgOk1R9TbhHGxYTWiZO8sG0ic2/mdgCKNkh5
        kHakQ9IosME2MPZ77aivE6GZBtNOudlsHBPsNeHSAZpGj53aALLnyEzxac6tKd/1
        PiSK2KcEtoPG7F+mA6AR3TBlsSN0s21G39nw0f8S0DaSAGfTmArTLqZiPp9WblZT
        nmbhny03lAPK3ge+wuk0Y6+ZtrL3ivaJGfO9by5Cro3k1Z6rwqUAJMF+SNPwM48R
        Mg2BVk0BS4Y59hrRl4vmlYDrmBI7mtM0jIk8zTSAJqvlSgvZkHx2baYhkWmiCGyb
        b01pO7aMv5dCM/UojDm9ygEo4kJqFrZhzqMA8rarCSaj4FJQ7BoMJjT6rmEsNfOy
        LjlaDhpTr/iK7dKMsuRz2zl3FtrpM9v5Cm03QhV6too8jj9jzrwMANuL1ZoYScrR
        Fwl2IvWfQJL6t8ckGS/WcULXGpWM+l0/s6n/g2z82YWMLmgZYx+0qH+NEzytwOxP
        aOofKN45/9BzdZUDwPacGA/tMwASz+CKkhlerr1IWSmFRBZjfk2D73KWpGVp5OKL
        sElZYBUZZbEnLPicQk3Z2nDpp5BGLU0/agc2eZzcEGkac3xCMxR7DoC9cTpJ4eUp
        gmm3bXbEpEUVuhgMSYSsdmaD7bgyns+n8Jjz5Ku8Z8+fS6FqyLWdXg15aUGIxj7J
        MqSh9mpoHZEGX3Ad0glwMgDMiICZq9Wg25g5Ls2jllL9d1HwLOpfq6BQm57WookT
        aEWrzL0SGllGmAkX8+XbE5pHeovoBBTNIbDny/wsBPYcACYd4TqDKgHTyEjTKq7F
        zIieXAZUGkzq36b7mScukvbNV6lnyqKMO7lA0keJs2SZYNaB2HLtPcrWaSy9lhd5
        ZGqnPUIb/dA1H2nQTkvlQbnTqX+XAW33GcwoNvnHKmDSWgha1D/LmUnb2JLz7WtT
        4llCKQDNXKy9Dth6xef4a6cdWLVBNuw1pF3t7htPe38WoeAytC2z134Rcv429qUA
        GNS/y4AWHRqbyrU4WDJY0KDaXIZEWrGb861FjWsrJ81n04Rd/c6GT44W7exyHENR
        /z4nNhQjYvYltPEHrtYrIdkiH8oAz/iwombXQErmtE3Y3n27bfuocubCYNOAgE4E
        ouVISq8re841WR8tZaO5Dmy5mgq1CNR/0o80aM9HaKNWVOrflZZKPi8CyoCsYtVY
        eMzKV9fmlmxbkyqUrlVwtS9d+eyLrpgbJ61NCXku5aShpLTTDJ3qxDaDUGkcn4ws
        2RLpMVc7LmZXi+LOSkOEXivmmGmzi81i3zFAKbCof9fASrbtMszShs2msjXYFwmY
        C9h+Bmk5ZtuSDEyWTC0wxs01N0BxlGK70KxjMGW49muIsXSlOjQdO98e0WIx7T7Z
        fSgSte4y+kXpm43c3wWQB77oTapNzQFlGVBXsVI78EXmEm3bMBd2J1fF++SzjLLZ
        Pus5XYaJ9Uy2UQw1dy65kuvSLrIL4YT45GmnPex+hKbZXesudJ9MFJ36T/oh4gCw
        FRsbGoqa5bjk8czbAcOpyytTklZ2jb+mE2lWizPlsfeLzyhqKra8EWk77fs+C2n0
        tYuTs/Z+SIObpxgxJOy5KlLfgN/3S8QBYBdgsKn/BFqV+IzFwDJsPu+f4cjYSofx
        HPaGZDmvLqqW6ShrpJZCI238pCN/MxCw13yIsdWi/n2Mhwa7lAeae6qV/iTvi9K/
        LLTlAGgYN3vzMQeTpTiZ7WqCMQcaz+BSXhryWKdgXPI06GkX9V8U+l8q8rfnLhT1
        b7NgCUI7dqHlA8VIQSTIWhtF6acJc++2zQCwoxoNuivrM4n2GWPEVkxsp0t7s9iK
        VGtzalD/GsZY0xj5oijNFEpo4+tyRjT2ok+GNgPi6kfRKfXQdSJpMOcv+deSA8Ck
        TZv5nCGrU9pPZNhz0e7m9LXFcMZcOWSGcjELMNlUudkmkwb0talRA5C8sqlol1LV
        Sp+5xjcU5W2vXQZc68Y3BiHHoUgGFfBT/yaKwgK4nBPRUwCtIo0yZU24PRhFmaRm
        4Nu07bbpiwbabTsrN8ZiMAB+saRPHnP9MubfBdecabFCTFm2DNezsdkVW6YdpYWC
        7ciG6Esn6eQ0XVMkp8Vey4VwANImWmIR5FEq0pOkSWd3iow8So35XNoUJlsWKyry
        pWY0qtC16n1smfY/LfhYDpuh0t7vrjEIVfxoyw7ZD9/vNPZGq3CtpwTVZhtiIM3D
        lJCpXYyllV6Qjl5NqtyEVu5RGr5516KRGbI0nBjfOmCsb1ekqbVf7fZdz6YdvYXO
        cbuYCG3q3V7j9loMMTa+veBK9xURrnXVVA2AhqJm5nl8HpBU2673jPYBfp7climt
        8PNQ8+3KSPtZWpamkQL8Bpoh10dRM2WGAntd2rLypFRCG5RQxrboNLr9cxEj/wRp
        9R25HAANwwnwaR5G9MKefA1jZht7jc2WbHJWhOzL50rDXrMaMljGWMPRtOEaP419
        ZcsMQf272I4Q0XaWU36tU//NsHtFcgRsB9PFsBWCAXC1LW2opdsEdDaKzY4k76Vg
        tqmpBLUVXSKT2R77mbTYBtP4mmyD1tpg0+A+h5fN5PnkA+EMR55oOwT9n7zXmhMf
        bKfU9bnZtyIxFUD2urrKAdCgN+z8UvKZRpqB0S47dcGMUEIYfMZ6YjuRvrbZRsoV
        mWmlTVwRBEOuRuTrGseQxsXWgRryY5qhedg6y5X3LxpMJ9526O21dpUDoLE5NPNr
        koo6pJFhKgnf4mgFIcYoFE2oRVFrRULalcwu55a1zotQ1Z7lhGjINT8LzT4kCE3z
        u+Dac+w12irsvjRjw50pAEbU7+oIg5Jl0pW+TcuMyCTazjLKkvPtymnav5Ocd5uq
        ZkPaqfTJ8BXusMCsLciCdgTsWpMaCF3Y5jK0RTFkWkxwXqTVaBQN7axhbwpA+mHT
        cjudBptaYUUsUm37oh+WMcvjkUrK0XAqk/eahXHsPeLb6xrPGMLZsJ3FkDooxHP7
        ZIcYB9eeLZpNMJnXIrIULtjMTu4aAPbgs+nFUA4G2+BIten6THIu0ihF1nOYBWpM
        hKyC1nAC2JGpL/IOQcGHMH6udavlZKXpwiIYMiZj2wpc4xWauWkVvrn3ngJgU/9M
        ypTZrh0phcrbSbTBivpdeVzptWTTca730mAyJT5Ipn/SoKV4bedfyyDZc2Y+bwjq
        X5vmTnvW0EZMswC8Gfj2RFH6ZyPNWfGtcfP/ZJ4CaBf25LKoZgZFY7ejUbwoKSOv
        4m0XvvnVjF4ZsjRZJJ+zxpDFlpFHPjsaN9NzrjxuyOfWRFoOOxT1r+XgNgvX2jQd
        laL114RrXeeZ3zJ7MzA9bl/hEsPJ0GIvWA4SC1l0GUOGBkzD4SoIkoRmQZbJaGiv
        bbMPbLicX+b4ZrFt7DWUJttGaENWhBoMF7ICvqL019y37c4l5cuA7MllFcqF8uLZ
        7bU7qbYzxPS6mUe47D6znT27Tdc4slEUJSMBnwJlG2JX6o6NtH0VguqWLCJuFq79
        yUoLtgOfHrF1WtEgOYa0bwN0RRpS0KRmmTA3qaRRM6lPLS+W0W4IBeZyOlisjDYd
        GlKZaaSDEmivG1/0H9qQ+BwPDb3m+6xIBtXsk2u/F6mvNswUVzv9pDgAdj5eQrnZ
        G7zTUeTFlQculkdDlnY+Tpq2zVI6bIRgMTQjcBf1r3lKRNvY2jJ8kWwouGqziqj7
        itgnEywmtyzZGHA17Sal3FhVo752tDaupBFNo8g7kfpP5Jivrr9hyJVca0VXLtLw
        ndZggaVzWumD9rMnMn1pv1BMSAhHKC/sgEKTKW0VvhoSkRoAaWPKNM7SHn2oCmHG
        URO7YpW9AVljZc+JxukLFpvh6r9J37GZE/NVexw14GIbtE6/JG27xlk78vYxICH6
        kLwvasRv9i0ECycBKVslmgJgRJkuSos1UZKbJ80DDkHDSrft85wlZQBXK3HWaRLX
        e+m2zc8084waVdea82XCVuYhDK+LDdOOvM1/oWl/ExpsZDN9sD9nOv9SYK6nqlRD
        pkGQ7KwZKUkyC652pfrv2oTMiFx6fJqRy2gvjY5jP6NE+2lKWNP501wTvtoGJrLa
        194PoSJeDbYvL/KsvVDzksguUt/S4EsrSdpZEQZAI/fmovZaRZpyluh/Wh6QFcFq
        eNedRpNlgRWt+QyB5thpGP889DtLXuh16KL+O53ZawdFj6ITMNk+Nhh2VsQBsKk3
        6ar/pE3WRpOm/u33LmXBgLTST2uLScVrKXqt6nBtOjh51Xg+c22bPzOe2ZVjDkH/
        +yIz85UFe4yLSGEXJVCwx6cIfXLB1nf2+JmpHen5bdsBcA2qVBQtbQg0KENfIQwD
        7AWe1neGo2ErMsmUT57PWNCi4m3DpCFTO+dvwl77IcY4BNICltD0fxEckATmONnr
        JETKNA9cqWMmmnYAsiJcaUguKlfFJ7vvplxpMOkslwfKBJMu187HpzlkGgpSS7Fp
        GdzQDpwLIaNcdqql2b6Y/SgaG+FiiIrQLxN5axIYa64lBoC58G3lKeWpuagyhtF0
        5QW1aMGit2nCpyg6vYbBdlhddB5Dpmb0a743n1eyH+b+952kMF/Z8NGw2k6ArVdC
        R/xmX3y/DwEflW7+rmhIS+ew1nkuB8A1YNID6KOypeQwTigAVytCtgFLXlnUP3Oe
        XW2zGRjNiMQ2/iGOpGm1z6YqXSlA+1+IUw5Zn2mgKBFsUQvqis5K+OA7/cRc57kc
        ANcRHw10QjToikSkqX97IWgYMsaCs08ruBSINPWvnSe1jZQGmDn/PG2y5GYZGM0o
        zna+NZ0PhqPPQJEodlvXdMLY5UkDSOMqB8AXAYaYVEk60VTIGtS//b4daNKcISrH
        E9lMsNexyQBp5fk1HA2f4WUzgvY6sedPu9gwkal9ssPlbBTBmLno9SI4Kll9KMLY
        AcVJ5VzlAPjyW6zJdbUnublZuSmbVtKi/lnerDZ9bSt4abg2FkOehgwXtBwNG+x1
        n+VcaOihNOPBdJLZulACndBHGyGD2Dzw9Utjj3tTAOZC14rOfD+30yazAt+l/NuB
        axxshSPJirg+0/DimbUYpgztaE0L7KghbT+mFThK90HTyfE5v6wAwtWmKwfsWtch
        EVp+FnzzWIQ0gGtfZVH/7D7vOQA+QYyo0KdQpGAOrMakS8jJiiwYFGte2VIwDZcW
        Y8JqOwGbkg7FLtjyNKhKn7OruTZ90EzDpX2macR8bAwjYMvTF99n5tosMvVv2qSi
        BCd7DoBr40lPNNOb1lyUDAfGVwHKBjPCSHsGlkNjp0tYsBUOyzi6qu21o2LtfLsW
        y2Aiq8hQW59oOeVpyFproQvCs9ZnUaj/VoIGrT7vSwG4Bkxy4ZsPLr2xbaZCo1Je
        ilbyOS/Jc2goA0bUbI8Pi+2xZTK969DKBOAZI5dTw5TnaztUVGTqJc0+hDit4kLW
        c4eI/PPAZbdYNqwdNLOvtPrsPQWgVSnPgCb1r9U+M+/INGoaStVnuBgyEpjOLMux
        cc2/9LPZz2RHoFrFoPZn2qyDb1y1HO88fWHDnn82u5UHzaRETNtVFKMPuPtTlP7t
        cwBYkZMWncVYqL7F1u7z+NpiLpSsjSMpx5wLtoNhy2bK0aL+E5la4+gDa/2l/Y7t
        LLpefbRsaEUdmnEKnYrIoveb/Z0mXP0oUloF+J0D4IrQWE4AKzpLIJ3D9jlF7T5H
        mqKRjoDS5pZFk4dWXK2iKMpDI/9u7kmX0ZU2/q7cviv6ZyLN0GtEus0+ryaT6WO4
        irInEpipUdfvXO+LhCI5L+VGozHEatz2rtmRkpQMX8QlvRnysgDtItkwrueRhI+u
        k5wTRttmez7DkIDt3Gg6a640BptBcck2wWQ7fDoidDEbEJ76T15dDFcRkMdBs/Vz
        6L679lToPlkYqpZKpXMAhtiUm2R7bJrOLLxj5pXS2m9XlkuRauShWEyPnZtmMRc+
        aClEl9IoYhTWDNJoZN9zacxvSHo7ZFonC0Vfb5prphUwbYYwVv4/Cw/oNftrx+AA
        AAAASUVORK5CYII=
        """
    
    /// Install default logo
    public static func install(on req: Request) throws -> Future<Void> {
        return try create(from: data, mime: .png, on: req)
    }
    
}


extension Logo {
    
    /// Create logo from data
    public static func create(from data: Data, mime: MediaType? = nil, path: String = "server/image/", minSize: UInt = 512, on req: Request) throws -> Future<Void> {
        guard let mime = mime ?? data.imageFileMediaType(), let gdMime = mime.gdMime() else {
            throw ImageError.invalidImageFormat
        }
        let image = try Image(data: data, as: gdMime)
        
        let size = image.size
        
        // Check image is square
        if size.width != size.height {
            throw Error.imageNotSquare(size)
        }
        
        // Image too small
        if size.width < minSize {
            throw Error.imageTooSmall(size)
        }
        
        // Resize all images
        guard let favicon = image.resizedTo(width: IconSize.favicon.rawValue), let faviconData = try? favicon.export(as: .png),
            let at1x = image.resizedTo(width: IconSize.at1x.rawValue), let at1xData = try? at1x.export(as: .png),
            let at2x = image.resizedTo(width: IconSize.at2x.rawValue), let at2xData = try? at2x.export(as: .png),
            let at3x = image.resizedTo(width: IconSize.at3x.rawValue), let at3xData = try? at3x.export(as: .png),
            let reg = image.resizedTo(width: IconSize.regular.rawValue), let regData = try? reg.export(as: .png),
            let large = image.resizedTo(width: IconSize.large.rawValue), let largeData = try? large.export(as: .png) else {
                throw Error.imageCorrupted
        }
        
        // Save all images
        let fm = try req.makeFileCore()
        return try fm.save(file: faviconData, to: "\(path)\(IconSize.favicon.rawValue)", mime: mime, on: req).flatMap({ _ in
            return try fm.save(file: at1xData, to: "\(path)\(IconSize.at1x.rawValue)", mime: mime, on: req).flatMap({ _ in
                return try fm.save(file: at2xData, to: "\(path)\(IconSize.at2x.rawValue)", mime: mime, on: req).flatMap({ _ in
                    return try fm.save(file: at3xData, to: "\(path)\(IconSize.at3x.rawValue)", mime: mime, on: req).flatMap({ _ in
                        return try fm.save(file: regData, to: "\(path)\(IconSize.regular.rawValue)", mime: mime, on: req).flatMap({ _ in
                            return try fm.save(file: largeData, to: "\(path)\(IconSize.large.rawValue)", mime: mime, on: req)
                        })
                    })
                })
            })
        })
    }
    
}

