package knf.kuma.animeinfo;

import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.TextView;

import com.afollestad.materialdialogs.MaterialDialog;
import com.michaelflisar.dragselectrecyclerview.DragSelectTouchListener;
import com.squareup.picasso.Callback;

import java.io.File;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import androidx.annotation.NonNull;
import androidx.appcompat.widget.PopupMenu;
import androidx.cardview.widget.CardView;
import androidx.fragment.app.Fragment;
import androidx.lifecycle.LifecycleOwner;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;
import androidx.lifecycle.Observer;
import androidx.recyclerview.widget.RecyclerView;
import butterknife.BindView;
import butterknife.ButterKnife;
import knf.kuma.R;
import knf.kuma.animeinfo.fragments.ChaptersFragment;
import knf.kuma.commons.CastUtil;
import knf.kuma.commons.EAHelper;
import knf.kuma.commons.Network;
import knf.kuma.commons.PicassoSingle;
import knf.kuma.commons.PrefsUtil;
import knf.kuma.commons.SelfServer;
import knf.kuma.database.CacheDB;
import knf.kuma.database.dao.ChaptersDAO;
import knf.kuma.database.dao.DownloadsDAO;
import knf.kuma.database.dao.RecordsDAO;
import knf.kuma.database.dao.SeeingDAO;
import knf.kuma.download.DownloadManager;
import knf.kuma.download.FileAccessHelper;
import knf.kuma.pojos.AnimeObject;
import knf.kuma.pojos.DownloadObject;
import knf.kuma.pojos.RecordObject;
import knf.kuma.pojos.SeeingObject;
import knf.kuma.queue.QueueManager;
import knf.kuma.videoservers.ServersFactory;
import me.zhanghai.android.materialprogressbar.MaterialProgressBar;
import xdroid.toaster.Toaster;

public class AnimeChaptersAdapter extends RecyclerView.Adapter<AnimeChaptersAdapter.ChapterImgHolder> {

    private Context context;
    private Fragment fragment;
    private ChaptersDAO chaptersDAO = CacheDB.INSTANCE.chaptersDAO();
    private RecordsDAO recordsDAO = CacheDB.INSTANCE.recordsDAO();
    private SeeingDAO seeingDAO = CacheDB.INSTANCE.seeingDAO();
    private DownloadsDAO downloadsDAO = CacheDB.INSTANCE.downloadsDAO();
    private List<AnimeObject.WebInfo.AnimeChapter> chapters;
    private DragSelectTouchListener touchListener;
    private boolean isNetworkAvailable = Network.isConnected();
    private HashSet<Integer> selected = new HashSet<>();
    private SeeingObject seeingObject;

    public AnimeChaptersAdapter(Fragment fragment, List<AnimeObject.WebInfo.AnimeChapter> chapters, DragSelectTouchListener touchListener) {
        this.context = fragment.getContext();
        this.fragment = fragment;
        this.chapters = chapters;
        this.touchListener = touchListener;
        chaptersDAO.init();
        if (chapters.size() > 0)
            seeingObject = seeingDAO.getByAid(chapters.get(0).aid);
    }

    @NonNull
    @Override
    public ChapterImgHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        return new ChapterImgHolder(LayoutInflater.from(parent.getContext()).inflate(R.layout.item_chapter_preview, parent, false));
    }

    @Override
    public void onBindViewHolder(@NonNull final ChapterImgHolder holder, int position) {
        final AnimeObject.WebInfo.AnimeChapter chapter = chapters.get(position);
        AtomicReference<DownloadObject> downloadObject = new AtomicReference<>(downloadsDAO.getByEid(chapter.eid));
        final File d_file = FileAccessHelper.INSTANCE.getFile(chapter.getFileName());
        if (selected.contains(position)) {
            holder.cardView.setCardBackgroundColor(context.getResources().getColor(EAHelper.getThemeColorLight(context)));
        } else {
            holder.cardView.setCardBackgroundColor(context.getResources().getColor(R.color.cardview_background));
        }
        holder.setQueue(QueueManager.isInQueue(chapter.eid), isPlayAvailable(d_file, downloadObject.get()));
        chapter.isDownloaded = canPlay(d_file);
        holder.setDownloadObserver(downloadsDAO.getLiveByEid(chapter.eid), fragment, object -> {
            holder.setDownloadState(object);
            String casting = CastUtil.get().getCasting().getValue();
            boolean isCasting = casting != null && casting.equals(chapter.eid);
            if (!isCasting)
                holder.setQueue(QueueManager.isInQueue(chapter.eid), isPlayAvailable(d_file, object));
            else
                holder.setDownloaded(isPlayAvailable(d_file, object), true);
            downloadObject.set(object);
        });
        if (!Network.isConnected() || chapter.img == null)
            holder.imageView.setVisibility(View.GONE);
        if (chapter.img != null)
            PicassoSingle.get(context).load(chapter.img).into(holder.imageView, new Callback() {
                @Override
                public void onSuccess() {
                    holder.imageView.setVisibility(View.VISIBLE);
                }

                @Override
                public void onError() {

                }
            });
        holder.setCastingObserver(fragment, s -> {
            if (!chapter.eid.equals(s))
                holder.setQueue(QueueManager.isInQueue(chapter.eid), isPlayAvailable(d_file, downloadObject.get()));
            else
                holder.setDownloaded(isPlayAvailable(d_file, downloadObject.get()), chapter.eid.equals(s));
        });
        holder.chapter.setTextColor(context.getResources().getColor(chaptersDAO.chapterIsSeen(chapter.eid) ? EAHelper.getThemeColor(context) : R.color.textPrimary));
        holder.separator.setVisibility(position == 0 ? View.GONE : View.VISIBLE);
        holder.chapter.setText(chapter.number);
        holder.actions.setOnClickListener(view -> {
            PopupMenu menu = new PopupMenu(context, view);
            if (CastUtil.get().getCasting().getValue().equals(chapter.eid)) {
                menu.inflate(R.menu.chapter_casting_menu);
                if (canPlay(d_file))
                    menu.getMenu().findItem(R.id.download).setVisible(false);
            } else if (isPlayAvailable(d_file, downloadObject.get())) {
                menu.inflate(R.menu.chapter_downloaded_menu);
                if (!CastUtil.get().connected())
                    menu.getMenu().findItem(R.id.cast).setVisible(false);
            } else if (isNetworkAvailable)
                menu.inflate(R.menu.chapter_menu);
            else
                menu.inflate(R.menu.chapter_menu_offline);
            if (QueueManager.isInQueue(chapter.eid) && menu.getMenu().findItem(R.id.queue) != null)
                menu.getMenu().findItem(R.id.queue).setVisible(false);
            if (!PrefsUtil.INSTANCE.showImport())
                menu.getMenu().findItem(R.id.import_file).setVisible(false);
            menu.setOnMenuItemClickListener(item -> {
                switch (item.getItemId()) {
                    case R.id.play:
                        if (canPlay(d_file)) {
                            chaptersDAO.addChapter(chapter);
                            recordsDAO.add(RecordObject.fromChapter(chapter));
                            updateSeeing(chapter.number);
                            holder.setSeen(context, true);
                            ServersFactory.startPlay(context, chapter.getEpTitle(), chapter.getFileName());
                        } else {
                            Toaster.toast("Aun no se está descargando");
                        }
                        break;
                    case R.id.cast:
                        if (canPlay(d_file)) {
                            CastUtil.get().play(fragment.getActivity(), chapter.eid, SelfServer.start(chapter.getFileName(), true), chapter.name, chapter.number, chapter.img == null ? chapter.aid : chapter.img, chapter.img == null);
                            chaptersDAO.addChapter(chapter);
                            recordsDAO.add(RecordObject.fromChapter(chapter));
                            updateSeeing(chapter.number);
                            holder.setSeen(context, true);
                        }
                        break;
                    case R.id.casting:
                        CastUtil.get().openControls();
                        break;
                    case R.id.delete:
                        new MaterialDialog.Builder(context)
                                .content("¿Eliminar el " + chapter.number.toLowerCase() + "?")
                                .positiveText("CONFIRMAR")
                                .negativeText("CANCELAR")
                                .onPositive((dialog, which) -> {
                                    if (downloadObject.get() != null)
                                        downloadObject.get().state = -8;
                                    chapter.isDownloaded = false;
                                    holder.setDownloaded(false, false);
                                    FileAccessHelper.INSTANCE.delete(chapter.getFileName(), () -> new Handler(Looper.getMainLooper()).post(this::notifyDataSetChanged));
                                    //CacheDB.INSTANCE.downloadsDAO().deleteByEid(chapter.eid);
                                    DownloadManager.cancel(chapter.eid);
                                    QueueManager.remove(chapter.eid);
                                }).build().show();
                        break;
                    case R.id.download:
                        ServersFactory.start(context, fragment.getChildFragmentManager(), chapter.link, chapter, false, new ServersFactory.ServersInterface() {
                            @Override
                            public void onFinish(boolean started, boolean success) {
                                if (started) {
                                    holder.setQueue(CacheDB.INSTANCE.queueDAO().isInQueue(chapter.eid), true);
                                    chapter.isDownloaded = true;
                                }
                            }

                            @Override
                            public void onCast(String url) {

                            }
                        });
                        break;
                    case R.id.streaming:
                        ServersFactory.start(context, fragment.getChildFragmentManager(), chapter.link, chapter, true, new ServersFactory.ServersInterface() {
                            @Override
                            public void onFinish(boolean started, boolean success) {
                                if (!started && success) {
                                    chaptersDAO.addChapter(chapter);
                                    recordsDAO.add(RecordObject.fromChapter(chapter));
                                    updateSeeing(chapter.number);
                                    holder.setSeen(context, true);
                                }
                            }

                            @Override
                            public void onCast(String url) {
                                CastUtil.get().play(fragment.getActivity(), chapter.eid, url, chapter.name, chapter.number, chapter.img == null ? chapter.aid : chapter.img, chapter.img == null);
                                chaptersDAO.addChapter(chapter);
                                recordsDAO.add(RecordObject.fromChapter(chapter));
                                updateSeeing(chapter.number);
                                holder.setSeen(context, true);
                            }
                        });
                        break;
                    case R.id.queue:
                        if (isPlayAvailable(d_file, downloadObject.get())) {
                            QueueManager.add(Uri.fromFile(d_file), true, chapter);
                            holder.setQueue(true, true);
                        } else
                            ServersFactory.start(context, fragment.getChildFragmentManager(), chapter.link, chapter, true, true, new ServersFactory.ServersInterface() {
                                @Override
                                public void onFinish(boolean started, boolean success) {
                                    if (success) {
                                        holder.setQueue(true, false);
                                    }
                                }

                                @Override
                                public void onCast(String url) {
                                }
                            });
                        break;
                    case R.id.share:
                        fragment.getActivity().startActivity(Intent.createChooser(new Intent(Intent.ACTION_SEND)
                                .setType("text/plain")
                                .putExtra(Intent.EXTRA_TEXT, chapter.getEpTitle() + "\n" + chapter.link), "Compartir"));
                        break;
                    case R.id.import_file:
                        if (fragment != null)
                            ((ChaptersFragment) fragment).onMove(chapter.getFileName());
                        break;
                }
                return true;
            });
            menu.show();
        });
        holder.cardView.setOnClickListener(view -> {
            if (chaptersDAO.chapterIsSeen(chapter.eid)) {
                chaptersDAO.deleteChapter(chapter);
                holder.chapter.setTextColor(context.getResources().getColor(R.color.textPrimary));
            } else {
                chaptersDAO.addChapter(chapter);
                holder.chapter.setTextColor(context.getResources().getColor(EAHelper.getThemeColor(context)));
            }
            updateSeeing(chapter.number);
        });
        holder.cardView.setOnLongClickListener(v -> {
            touchListener.startDragSelection(holder.getAdapterPosition());
            return true;
        });
    }

    private void updateSeeing(String chapter) {
        if (seeingObject != null) {
            seeingObject.chapter = chapter;
            seeingDAO.update(seeingObject);
        }
    }

    private boolean isPlayAvailable(File file, DownloadObject downloadObject) {
        return (file != null && file.exists()) || (downloadObject != null && downloadObject.isDownloading());
    }

    private boolean canPlay(File file) {
        return file != null && file.exists();
    }

    @Override
    public int getItemViewType(int position) {
        return chapters.get(position).chapterType.value;
    }

    @Override
    public int getItemCount() {
        return chapters.size();
    }

    public void select(int pos, boolean sel) {
        if (sel) {
            selected.add(pos);
        } else {
            selected.remove(pos);
        }
        notifyItemChanged(pos);
    }

    public void selectRange(int start, int end, boolean sel) {
        for (int i = start; i <= end; i++) {
            if (sel)
                selected.add(i);
            else
                selected.remove(i);
        }
        notifyItemRangeChanged(start, end - start + 1);
    }

    public void deselectAll() {
        selected.clear();
        notifyDataSetChanged();
    }

    public int getCountSelected() {
        return selected.size();
    }

    public HashSet<Integer> getSelection() {
        return selected;
    }

    @Override
    public void onViewRecycled(@NonNull ChapterImgHolder holder) {
        holder.unsetCastingObserver();
        holder.unsetDownloadObserver();
        super.onViewRecycled(holder);
    }

    class ChapterImgHolder extends RecyclerView.ViewHolder {
        @BindView(R.id.card)
        CardView cardView;
        @BindView(R.id.separator)
        View separator;
        @BindView(R.id.img)
        ImageView imageView;
        @BindView(R.id.chapter)
        TextView chapter;
        @BindView(R.id.in_down)
        ImageView in_down;
        @BindView(R.id.actions)
        ImageButton actions;
        @BindView(R.id.progress)
        MaterialProgressBar progressBar;

        private LiveData<DownloadObject> downloadLiveData = new MutableLiveData<>();

        private Observer<DownloadObject> downloadObserver;
        private Observer<String> castingObserver;

        ChapterImgHolder(View itemView) {
            super(itemView);
            ButterKnife.bind(this, itemView);
        }

        void setDownloadObserver(LiveData<DownloadObject> downloadLiveData, LifecycleOwner owner, Observer<DownloadObject> observer) {
            this.downloadLiveData = downloadLiveData;
            this.downloadObserver = observer;
            this.downloadLiveData.observe(owner, downloadObserver);
        }

        void unsetDownloadObserver() {
            if (downloadObserver != null) {
                downloadLiveData.removeObserver(downloadObserver);
                downloadObserver = null;
            }
        }

        void setCastingObserver(LifecycleOwner owner, Observer<String> observer) {
            this.castingObserver = observer;
            CastUtil.get().getCasting().observe(owner, castingObserver);
        }

        void unsetCastingObserver() {
            if (castingObserver != null) {
                CastUtil.get().getCasting().removeObserver(castingObserver);
                castingObserver = null;
            }
        }

        void setDownloaded(final boolean downloaded, final boolean isCasting) {
            in_down.post(() -> {
                if (downloaded)
                    in_down.setImageResource(R.drawable.ic_chap_down);
                if (isCasting)
                    in_down.setImageResource(R.drawable.ic_casting);
                in_down.setVisibility(downloaded || isCasting ? View.VISIBLE : View.GONE);
            });
        }

        void setQueue(final boolean isInQueue, final boolean isDownloaded) {
            in_down.post(() -> {
                if (!isInQueue)
                    setDownloaded(isDownloaded, false);
                else {
                    in_down.setImageResource(isDownloaded ? R.drawable.ic_queue_file : R.drawable.ic_queue_normal);
                    in_down.setVisibility(View.VISIBLE);
                }
            });
        }

        void setSeen(final Context context, final boolean seen) {
            chapter.post(() -> chapter.setTextColor(context.getResources().getColor(seen ? EAHelper.getThemeColor(context) : R.color.textPrimary)));
        }

        void setDownloadState(DownloadObject object) {
            progressBar.post(() -> {
                if (object != null && PrefsUtil.INSTANCE.showProgress())
                    switch (object.state) {
                        case DownloadObject.PENDING:
                            progressBar.setVisibility(View.VISIBLE);
                            progressBar.setIndeterminate(true);
                            break;
                        case DownloadObject.PAUSED:
                        case DownloadObject.DOWNLOADING:
                            progressBar.setVisibility(View.VISIBLE);
                            progressBar.setIndeterminate(false);
                            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N)
                                progressBar.setProgress(object.progress, true);
                            else
                                progressBar.setProgress(object.progress);
                            break;
                        default:
                            progressBar.setVisibility(View.GONE);
                            break;
                    }
                else progressBar.setVisibility(View.GONE);
            });
        }
    }

}
