// Copyright (c) Facebook, Inc. and its affiliates.
// This source code is licensed under the MIT license found in the
// LICENSE file in the root directory of this source tree.
#include "SceneGraph.h"

#include <Corrade/Utility/Assert.h>
#include <Corrade/Utility/Debug.h>
#include <Corrade/Utility/DebugStl.h>

namespace esp {
namespace scene {

SceneGraph::SceneGraph()
    : rootNode_{world_},
      defaultRenderCameraNode_{rootNode_},
      defaultRenderCamera_{defaultRenderCameraNode_} {
  // For now, just create one drawable group with empty string uuid
  createDrawableGroup(std::string{});
}

// set transformation, projection matrix, viewport to the default camera
void SceneGraph::setDefaultRenderCamera(sensor::VisualSensor& sensor) {
  ASSERT(sensor.isVisualSensor());

  sensor.setTransformationMatrix(defaultRenderCamera_)
      .setProjectionMatrix(defaultRenderCamera_)
      .setViewport(defaultRenderCamera_);
}

bool SceneGraph::isRootNode(SceneNode& node) {
  auto parent = node.parent();
  // if the parent is null, it means the node is the world_ node.
  CORRADE_ASSERT(parent != nullptr,
                 "SceneGraph::isRootNode: the node is illegal.", false);
  return (parent->parent() == nullptr ? true : false);
}

gfx::DrawableGroup* SceneGraph::getDrawableGroup(const std::string& id) {
  auto it = drawableGroups_.find(id);
  return it == drawableGroups_.end() ? nullptr : &it->second;
}

const gfx::DrawableGroup* SceneGraph::getDrawableGroup(
    const std::string& id) const {
  auto it = drawableGroups_.find(id);
  return it == drawableGroups_.end() ? nullptr : &it->second;
}

bool SceneGraph::deleteDrawableGroup(const std::string& id) {
  return drawableGroups_.erase(id);
}

}  // namespace scene
}  // namespace esp
