// https://gist.github.com/nbasham/3b2de0566d5f716894fc
//
//  Survey+Util.swift
//  previously Color+HexAndCSSColorNames.swift
//
//  Created by Norman Basham on 12/8/15.
//  Copyright ©2018 Black Labs. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#if os(iOS)
    import SwiftUI
    import UIKit

    // swiftlint:disable identifier_name
    public extension UIColor {
        /**
         Creates an immuatble UIColor instance specified by a hex string, CSS color name, or nil.

         - parameter hexString: A case insensitive String? representing a hex or CSS value e.g.

         - **"abc"**
         - **"abc7"**
         - **"#abc7"**
         - **"00FFFF"**
         - **"#00FFFF"**
         - **"00FFFF77"**
         - **"Orange", "Azure", "Tomato"** Modern browsers support 140 color names (<http://www.w3schools.com/cssref/css_colornames.asp>)
         - **"Clear"** [UIColor clearColor]
         - **"Transparent"** [UIColor clearColor]
         - **nil** [UIColor clearColor]
         - **empty string** [UIColor clearColor]
         */
        convenience init(hex: String?) {
            let normalizedHexString: String = UIColor.normalize(hex)
            var c: CUnsignedLongLong = 0
            Scanner(string: normalizedHexString).scanHexInt64(&c)
            self.init(
                red: UIColorMasks.redValue(CUnsignedInt(c)),
                green: UIColorMasks.greenValue(CUnsignedInt(c)),
                blue: UIColorMasks.blueValue(CUnsignedInt(c)),
                alpha: UIColorMasks.alphaValue(CUnsignedInt(c))
            )
        }

        /**
         Returns a hex equivalent of this UIColor.

         - Parameter includeAlpha:   Optional parameter to include the alpha hex.

         color.hexDescription() -> "ff0000"

         color.hexDescription(true) -> "ff0000aa"

         - Returns: A new string with `String` with the color's hexidecimal value.
         */
        func hexDescription(_ includeAlpha: Bool = false) -> String {
            guard cgColor.numberOfComponents == 4 else {
                return "Color not RGB."
            }
            let a = cgColor.components!.map { Int($0 * CGFloat(255)) }
            let color = String(format: "%02x%02x%02x", a[0], a[1], a[2])
            if includeAlpha {
                let alpha = String(format: "%02x", a[3])
                return "\(color)\(alpha)"
            }
            return color
        }

        fileprivate enum UIColorMasks: CUnsignedInt {
            case redMask = 0xFF000000
            case greenMask = 0x00FF0000
            case blueMask = 0x0000FF00
            case alphaMask = 0x000000FF

            static func redValue(_ value: CUnsignedInt) -> CGFloat {
                CGFloat((value & redMask.rawValue) >> 24) / 255.0
            }

            static func greenValue(_ value: CUnsignedInt) -> CGFloat {
                CGFloat((value & greenMask.rawValue) >> 16) / 255.0
            }

            static func blueValue(_ value: CUnsignedInt) -> CGFloat {
                CGFloat((value & blueMask.rawValue) >> 8) / 255.0
            }

            static func alphaValue(_ value: CUnsignedInt) -> CGFloat {
                CGFloat(value & alphaMask.rawValue) / 255.0
            }
        }

        fileprivate static func normalize(_ hex: String?) -> String {
            guard var hexString = hex else {
                return "00000000"
            }
            if let cssColor = cssToHexDictionary[hexString.uppercased()] {
                return cssColor.count == 8 ? cssColor : cssColor + "ff"
            }
            if hexString.hasPrefix("#") {
                hexString = String(hexString.dropFirst())
            }
            if hexString.count == 3 || hexString.count == 4 {
                hexString = hexString.map { "\($0)\($0)" }.joined()
            }
            let hasAlpha = hexString.count > 7
            if !hasAlpha {
                hexString += "ff"
            }
            return hexString
        }

        /**
         All modern browsers support the following 140 color names (see http://www.w3schools.com/cssref/css_colornames.asp)
         */
        fileprivate static func hexFromCssName(_ cssName: String) -> String {
            let key = cssName.uppercased()
            if let hex = cssToHexDictionary[key] {
                return hex
            }
            return cssName
        }

        fileprivate static let cssToHexDictionary: [String: String] = [
            "CLEAR": "00000000",
            "TRANSPARENT": "00000000",
            "": "00000000",
            "ALICEBLUE": "F0F8FF",
            "ANTIQUEWHITE": "FAEBD7",
            "AQUA": "00FFFF",
            "AQUAMARINE": "7FFFD4",
            "AZURE": "F0FFFF",
            "BEIGE": "F5F5DC",
            "BISQUE": "FFE4C4",
            "BLACK": "000000",
            "BLANCHEDALMOND": "FFEBCD",
            "BLUE": "0000FF",
            "BLUEVIOLET": "8A2BE2",
            "BROWN": "A52A2A",
            "BURLYWOOD": "DEB887",
            "CADETBLUE": "5F9EA0",
            "CHARTREUSE": "7FFF00",
            "CHOCOLATE": "D2691E",
            "CORAL": "FF7F50",
            "CORNFLOWERBLUE": "6495ED",
            "CORNSILK": "FFF8DC",
            "CRIMSON": "DC143C",
            "CYAN": "00FFFF",
            "DARKBLUE": "00008B",
            "DARKCYAN": "008B8B",
            "DARKGOLDENROD": "B8860B",
            "DARKGRAY": "A9A9A9",
            "DARKGREY": "A9A9A9",
            "DARKGREEN": "006400",
            "DARKKHAKI": "BDB76B",
            "DARKMAGENTA": "8B008B",
            "DARKOLIVEGREEN": "556B2F",
            "DARKORANGE": "FF8C00",
            "DARKORCHID": "9932CC",
            "DARKRED": "8B0000",
            "DARKSALMON": "E9967A",
            "DARKSEAGREEN": "8FBC8F",
            "DARKSLATEBLUE": "483D8B",
            "DARKSLATEGRAY": "2F4F4F",
            "DARKSLATEGREY": "2F4F4F",
            "DARKTURQUOISE": "00CED1",
            "DARKVIOLET": "9400D3",
            "DEEPPINK": "FF1493",
            "DEEPSKYBLUE": "00BFFF",
            "DIMGRAY": "696969",
            "DIMGREY": "696969",
            "DODGERBLUE": "1E90FF",
            "FIREBRICK": "B22222",
            "FLORALWHITE": "FFFAF0",
            "FORESTGREEN": "228B22",
            "FUCHSIA": "FF00FF",
            "GAINSBORO": "DCDCDC",
            "GHOSTWHITE": "F8F8FF",
            "GOLD": "FFD700",
            "GOLDENROD": "DAA520",
            "GRAY": "808080",
            "GREY": "808080",
            "GREEN": "008000",
            "GREENYELLOW": "ADFF2F",
            "HONEYDEW": "F0FFF0",
            "HOTPINK": "FF69B4",
            "INDIANRED": "CD5C5C",
            "INDIGO": "4B0082",
            "IVORY": "FFFFF0",
            "KHAKI": "F0E68C",
            "LAVENDER": "E6E6FA",
            "LAVENDERBLUSH": "FFF0F5",
            "LAWNGREEN": "7CFC00",
            "LEMONCHIFFON": "FFFACD",
            "LIGHTBLUE": "ADD8E6",
            "LIGHTCORAL": "F08080",
            "LIGHTCYAN": "E0FFFF",
            "LIGHTGOLDENRODYELLOW": "FAFAD2",
            "LIGHTGRAY": "D3D3D3",
            "LIGHTGREY": "D3D3D3",
            "LIGHTGREEN": "90EE90",
            "LIGHTPINK": "FFB6C1",
            "LIGHTSALMON": "FFA07A",
            "LIGHTSEAGREEN": "20B2AA",
            "LIGHTSKYBLUE": "87CEFA",
            "LIGHTSLATEGRAY": "778899",
            "LIGHTSLATEGREY": "778899",
            "LIGHTSTEELBLUE": "B0C4DE",
            "LIGHTYELLOW": "FFFFE0",
            "LIME": "00FF00",
            "LIMEGREEN": "32CD32",
            "LINEN": "FAF0E6",
            "MAGENTA": "FF00FF",
            "MAROON": "800000",
            "MEDIUMAQUAMARINE": "66CDAA",
            "MEDIUMBLUE": "0000CD",
            "MEDIUMORCHID": "BA55D3",
            "MEDIUMPURPLE": "9370DB",
            "MEDIUMSEAGREEN": "3CB371",
            "MEDIUMSLATEBLUE": "7B68EE",
            "MEDIUMSPRINGGREEN": "00FA9A",
            "MEDIUMTURQUOISE": "48D1CC",
            "MEDIUMVIOLETRED": "C71585",
            "MIDNIGHTBLUE": "191970",
            "MINTCREAM": "F5FFFA",
            "MISTYROSE": "FFE4E1",
            "MOCCASIN": "FFE4B5",
            "NAVAJOWHITE": "FFDEAD",
            "NAVY": "000080",
            "OLDLACE": "FDF5E6",
            "OLIVE": "808000",
            "OLIVEDRAB": "6B8E23",
            "ORANGE": "FFA500",
            "ORANGERED": "FF4500",
            "ORCHID": "DA70D6",
            "PALEGOLDENROD": "EEE8AA",
            "PALEGREEN": "98FB98",
            "PALETURQUOISE": "AFEEEE",
            "PALEVIOLETRED": "DB7093",
            "PAPAYAWHIP": "FFEFD5",
            "PEACHPUFF": "FFDAB9",
            "PERU": "CD853F",
            "PINK": "FFC0CB",
            "PLUM": "DDA0DD",
            "POWDERBLUE": "B0E0E6",
            "PURPLE": "800080",
            "RED": "FF0000",
            "ROSYBROWN": "BC8F8F",
            "ROYALBLUE": "4169E1",
            "SADDLEBROWN": "8B4513",
            "SALMON": "FA8072",
            "SANDYBROWN": "F4A460",
            "SEAGREEN": "2E8B57",
            "SEASHELL": "FFF5EE",
            "SIENNA": "A0522D",
            "SILVER": "C0C0C0",
            "SKYBLUE": "87CEEB",
            "SLATEBLUE": "6A5ACD",
            "SLATEGRAY": "708090",
            "SLATEGREY": "708090",
            "SNOW": "FFFAFA",
            "SPRINGGREEN": "00FF7F",
            "STEELBLUE": "4682B4",
            "TAN": "D2B48C",
            "TEAL": "008080",
            "THISTLE": "D8BFD8",
            "TOMATO": "FF6347",
            "TURQUOISE": "40E0D0",
            "VIOLET": "EE82EE",
            "WHEAT": "F5DEB3",
            "WHITE": "FFFFFF",
            "WHITESMOKE": "F5F5F5",
            "YELLOW": "FFFF00",
            "YELLOWGREEN": "9ACD32",
        ]
    }

    extension Color {
        @available(iOS 15.0, *)
        func getContrastingTextColor() -> Color {
            var r, g, b, a: CGFloat
            (r, g, b, a) = (0, 0, 0, 0)
            UIColor(self).getRed(&r, green: &g, blue: &b, alpha: &a)
            let luminance = 0.2126 * r + 0.7152 * g + 0.0722 * b
            return luminance < 0.6 ? .white : .black
        }
    }

    // swiftlint:enable identifier_name
#endif
