package systemcontracts

import (
	"encoding/hex"
	"fmt"
	"math/big"

	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/state"
	"github.com/ethereum/go-ethereum/log"
	"github.com/ethereum/go-ethereum/params"
)

type UpgradeConfig struct {
	BeforeUpgrade upgradeHook
	AfterUpgrade  upgradeHook
	ContractAddr  common.Address
	CommitUrl     string
	Code          string
}

type Upgrade struct {
	UpgradeName string
	Configs     []*UpgradeConfig
}

type upgradeHook func(blockNumber *big.Int, contractAddr common.Address, statedb *state.StateDB) error

const (
	mainNet    = "Mainnet"
	chapelNet  = "Chapel"
	rialtoNet  = "Rialto"
	defaultNet = "Default"
)

var (
	GenesisHash common.Hash
	//upgrade config
	ramanujanUpgrade = make(map[string]*Upgrade)

	nielsUpgrade = make(map[string]*Upgrade)

	mirrorUpgrade = make(map[string]*Upgrade)

	brunoUpgrade = make(map[string]*Upgrade)
)

func init() {
	ramanujanUpgrade[rialtoNet] = &Upgrade{
		UpgradeName: "ramanujan",
		Configs: []*UpgradeConfig{
			{
				ContractAddr: common.HexToAddress(ValidatorContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/f4bc161dac5937b8cbd4fe3089c7514c415430f9",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(SlashContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/f4bc161dac5937b8cbd4fe3089c7514c415430f9",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(SystemRewardContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/f4bc161dac5937b8cbd4fe3089c7514c415430f9",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(LightClientContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/f4bc161dac5937b8cbd4fe3089c7514c415430f9",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(TokenHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/f4bc161dac5937b8cbd4fe3089c7514c415430f9",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(RelayerIncentivizeContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/f4bc161dac5937b8cbd4fe3089c7514c415430f9",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(RelayerHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/f4bc161dac5937b8cbd4fe3089c7514c415430f9",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(GovHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/f4bc161dac5937b8cbd4fe3089c7514c415430f9",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(TokenManagerContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/f4bc161dac5937b8cbd4fe3089c7514c415430f9",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(CrossChainContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/f4bc161dac5937b8cbd4fe3089c7514c415430f9",
				Code:         "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",
			},
		},
	}

	ramanujanUpgrade[chapelNet] = &Upgrade{
		UpgradeName: "ramanujan",
		Configs: []*UpgradeConfig{
			{
				ContractAddr: common.HexToAddress(ValidatorContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(SlashContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(SystemRewardContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "60806040526004361061014f5760003560e01c806396713da9116100b6578063c81b16621161006f578063c81b1662146103dc578063dc927faf146103f1578063f9a2bbc714610406578063fb5478b31461041b578063fc3e590814610430578063fd6a68791461044557610193565b806396713da91461033a5780639a99b4f01461034f5780639dc0926214610388578063a1a11bf51461039d578063a78abc16146103b2578063ab51bb96146103c757610193565b806351e806721161010857806351e806721461028a5780636d70f7ae1461029f5780636e47b482146102e657806370fd5bad146102fb57806375d47a0a146103105780637942fd051461032557610193565b80630bee7a67146101985780633a0b0eff146101c65780633dffc387146101ed57806343756e5c14610218578063493279b1146102495780634bf6c8821461027557610193565b366101935734156101915760408051348152905133917f6c98249d85d88c3753a04a22230f595e4dc8d3dc86c34af35deeeedc861b89db919081900360200190a25b005b600080fd5b3480156101a457600080fd5b506101ad61045a565b6040805163ffffffff9092168252519081900360200190f35b3480156101d257600080fd5b506101db61045f565b60408051918252519081900360200190f35b3480156101f957600080fd5b50610202610465565b6040805160ff9092168252519081900360200190f35b34801561022457600080fd5b5061022d61046a565b604080516001600160a01b039092168252519081900360200190f35b34801561025557600080fd5b5061025e610470565b6040805161ffff9092168252519081900360200190f35b34801561028157600080fd5b50610202610475565b34801561029657600080fd5b5061022d61047a565b3480156102ab57600080fd5b506102d2600480360360208110156102c257600080fd5b50356001600160a01b0316610480565b604080519115158252519081900360200190f35b3480156102f257600080fd5b5061022d61049e565b34801561030757600080fd5b506102026104a4565b34801561031c57600080fd5b5061022d6104a9565b34801561033157600080fd5b506102026104af565b34801561034657600080fd5b506102026104b4565b34801561035b57600080fd5b506101db6004803603604081101561037257600080fd5b506001600160a01b0381351690602001356104b9565b34801561039457600080fd5b5061022d610664565b3480156103a957600080fd5b5061022d61066a565b3480156103be57600080fd5b506102d2610670565b3480156103d357600080fd5b506101ad610679565b3480156103e857600080fd5b5061022d61067e565b3480156103fd57600080fd5b5061022d610684565b34801561041257600080fd5b5061022d61068a565b34801561042757600080fd5b506101db610690565b34801561043c57600080fd5b5061020261069c565b34801561045157600080fd5b5061022d6106a1565b606481565b60015481565b600181565b61100181565b606081565b600881565b61200081565b6001600160a01b031660009081526002602052604090205460ff1690565b61100581565b600281565b61100881565b600b81565b600981565b6000805460ff1661053657600260208190527fe57bda0a954a7c7381b17b2c763e646ba2c60f67292d287ba583603e2c1c41668054600160ff19918216811790925561100560009081527fe25235fc0de9d7165652bef0846fefda506174abb9a190f03d0f7bcc6146dbce80548316841790559282558254161790555b3360009081526002602052604090205460ff166105845760405162461bcd60e51b815260040180806020018281038252602b8152602001806106a8602b913960400191505060405180910390fd5b60004783106105935747610595565b825b9050670de0b6b3a76400008111156105b25750670de0b6b3a76400005b8015610633576040516001600160a01b0385169082156108fc029083906000818181858888f193505050501580156105ee573d6000803e3d6000fd5b506040805182815290516001600160a01b038616917ff8b71c64315fc33b2ead2adfa487955065152a8ac33d9d5193aafd7f45dc15a0919081900360200190a261065d565b6040517fe589651933c2457488cc0d8e0941518abf748e799435e4e396d9c4d0b2db2d4d90600090a15b9392505050565b61100781565b61100681565b60005460ff1681565b600081565b61100281565b61100381565b61100081565b670de0b6b3a764000081565b600381565b6110048156fe6f6e6c79206f70657261746f7220697320616c6c6f77656420746f2063616c6c20746865206d6574686f64a2646970667358221220d0a7618a32d393c0757994fad066c9aaf7342a982498e0748019ec232f5dc86764736f6c63430006040033",
			},
			{
				ContractAddr: common.HexToAddress(LightClientContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(TokenHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(RelayerIncentivizeContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "6080604052600436106102765760003560e01c80637e146cc51161014f578063af400681116100c1578063e75d72c71161007a578063e75d72c714610795578063e89a3020146107c8578063f9a2bbc7146107f2578063fc3e590814610807578063fd6a68791461081c578063fdd31fcd146108315761027d565b8063af400681146106ed578063bd4cc83014610717578063c81b166214610741578063dc927faf14610756578063dcae76ab1461076b578063e1c7392a146107805761027d565b8063a3c3c0ad11610113578063a3c3c0ad146105b3578063a78abc16146105c8578063a7c6a59d146105dd578063ab51bb96146105f2578063ac43175114610607578063ace9fcc2146106d85761027d565b80637e146cc51461052c578063930e1b091461054157806396713da9146105745780639dc0926214610589578063a1a11bf51461059e5761027d565b806343756e5c116101e85780636e47b482116101ac5780636e47b482146104645780636f93d2e61461047957806370fd5bad146104d857806374f2272d146104ed57806375d47a0a146105025780637942fd05146105175761027d565b806343756e5c146103e4578063493279b1146103f95780634bf6c8821461042557806351e806721461043a578063541333071461044f5761027d565b806312950c461161023a57806312950c46146103165780631b20087c1461032b5780631c643312146103405780633a975612146102825780633dffc3871461038657806340bb43c0146103b15761027d565b8063081e9d131461028257806308f2ec06146102a9578063093f2fc4146102be5780630bee7a67146102d357806310e06a76146103015761027d565b3661027d57005b600080fd5b34801561028e57600080fd5b50610297610864565b60408051918252519081900360200190f35b3480156102b557600080fd5b50610297610869565b3480156102ca57600080fd5b5061029761086e565b3480156102df57600080fd5b506102e8610873565b6040805163ffffffff9092168252519081900360200190f35b34801561030d57600080fd5b50610297610878565b34801561032257600080fd5b5061029761087e565b34801561033757600080fd5b50610297610884565b34801561034c57600080fd5b5061036a6004803603602081101561036357600080fd5b503561088a565b604080516001600160a01b039092168252519081900360200190f35b34801561039257600080fd5b5061039b610864565b6040805160ff9092168252519081900360200190f35b3480156103bd57600080fd5b50610297600480360360208110156103d457600080fd5b50356001600160a01b03166108b1565b3480156103f057600080fd5b5061036a6108c3565b34801561040557600080fd5b5061040e6108c9565b6040805161ffff9092168252519081900360200190f35b34801561043157600080fd5b5061039b6108ce565b34801561044657600080fd5b5061036a6108d3565b34801561045b57600080fd5b50610297610873565b34801561047057600080fd5b5061036a6108d9565b34801561048557600080fd5b506104c46004803603608081101561049c57600080fd5b506001600160a01b0381358116916020810135909116906040810135906060013515156108df565b604080519115158252519081900360200190f35b3480156104e457600080fd5b5061039b610cad565b3480156104f957600080fd5b50610297610cb2565b34801561050e57600080fd5b5061036a610cb8565b34801561052357600080fd5b5061039b610cbe565b34801561053857600080fd5b50610297610cc3565b34801561054d57600080fd5b506102976004803603602081101561056457600080fd5b50356001600160a01b0316610cc8565b34801561058057600080fd5b5061039b610cda565b34801561059557600080fd5b5061036a610cdf565b3480156105aa57600080fd5b5061036a610ce5565b3480156105bf57600080fd5b50610297610ceb565b3480156105d457600080fd5b506104c4610cf1565b3480156105e957600080fd5b50610297610cfa565b3480156105fe57600080fd5b506102e8610d00565b34801561061357600080fd5b506106d66004803603604081101561062a57600080fd5b81019060208101813564010000000081111561064557600080fd5b82018360208201111561065757600080fd5b8035906020019184600183028401116401000000008311171561067957600080fd5b91939092909160208101903564010000000081111561069757600080fd5b8201836020820111156106a957600080fd5b803590602001918460018302840111640100000000831117156106cb57600080fd5b509092509050610d05565b005b3480156106e457600080fd5b50610297611354565b3480156106f957600080fd5b506102976004803603602081101561071057600080fd5b503561135a565b34801561072357600080fd5b506102976004803603602081101561073a57600080fd5b50356113b5565b34801561074d57600080fd5b5061036a6113ce565b34801561076257600080fd5b5061036a6113d4565b34801561077757600080fd5b506102976113da565b34801561078c57600080fd5b506106d66113e0565b3480156107a157600080fd5b506106d6600480360360208110156107b857600080fd5b50356001600160a01b03166114a9565b3480156107d457600080fd5b5061036a600480360360208110156107eb57600080fd5b5035611602565b3480156107fe57600080fd5b5061036a61160f565b34801561081357600080fd5b5061039b611615565b34801561082857600080fd5b5061036a61161a565b34801561083d57600080fd5b506102976004803603602081101561085457600080fd5b50356001600160a01b0316611620565b600181565b602881565b605081565b606481565b600b5481565b60015481565b600c5481565b6006818154811061089757fe5b6000918252602090912001546001600160a01b0316905081565b60076020526000908152604090205481565b61100181565b606081565b600881565b61200081565b61100581565b6000805460ff16610937576040805162461bcd60e51b815260206004820152601960248201527f74686520636f6e7472616374206e6f7420696e69742079657400000000000000604482015290519081900360640190fd5b33612000146109775760405162461bcd60e51b815260040180806020018281038252602f815260200180612164602f913960400191505060405180910390fd5b600082156109fc57604080516309a99b4f60e41b8152611005600482015260248101869052905161100291639a99b4f09160448083019260209291908290030181600087803b1580156109c957600080fd5b505af11580156109dd573d6000803e3d6000fd5b505050506040513d60208110156109f357600080fd5b50519050610a75565b604080516309a99b4f60e41b8152611005600482015260248101869052905161100491639a99b4f09160448083019260209291908290030181600087803b158015610a4657600080fd5b505af1158015610a5a573d6000803e3d6000fd5b505050506040513d6020811015610a7057600080fd5b505190505b600c805460010190556000610a8982611632565b600954909150610a9f908263ffffffff61166116565b600955600a54610ac7908290610abb908563ffffffff61166116565b9063ffffffff6116c216565b600a556001600160a01b038716600090815260056020526040902054610b3357600680546001810182556000919091527ff652222313e28459528d920b65115c16c04f3efc82aaedc97be59f3f377c0d3f0180546001600160a01b0319166001600160a01b0389161790555b6001600160a01b038088166000908152600560209081526040808320805460010190559289168252600790522054610bb157600880546001810182556000919091527ff3f7a9fe364faab93b216da50a3214154f22a0a2b415b23a84c8169e8b636ee30180546001600160a01b0319166001600160a01b0388161790555b6001600160a01b038616600090815260076020526040902080546001019055600c54606411610ca057600b54600954600a5460408051938452602084019290925282820152517f2649b1b772a1a74bd332a67695e285317dd722941166595741c60a00fa65bb759181900360600190a16000610c2b611704565b90506000610c376119e8565b6001600160a01b0389166000908152600d6020526040902054909150610c75908290610c69908563ffffffff61166116565b9063ffffffff61166116565b6001600160a01b0389166000908152600d6020526040812091909155600b80546001019055600c5550505b5060019695505050505050565b600281565b60035481565b61100881565b600b81565b600581565b60056020526000908152604090205481565b600981565b61100781565b61100681565b600a5481565b60005460ff1681565b60045481565b600081565b3361100714610d455760405162461bcd60e51b815260040180806020018281038252602e8152602001806120a9602e913960400191505060405180910390fd5b60005460ff16610d865760405162461bcd60e51b81526004018080602001828103825260218152602001806120d76021913960400191505060405180910390fd5b610dfa84848080601f0160208091040260200160405190810160405280939291908181526020018383808284376000920191909152505060408051808201909152601f81527f68656164657252656c61796572526577617264526174654d6f6c6563756c650060208201529150611c669050565b15610ec75760208114610e3e5760405162461bcd60e51b81526004018080602001828103825260328152602001806121936032913960400191505060405180910390fd5b604080516020601f8401819004810282018101909252828152600091610e7c91858580838501838280828437600092019190915250611d4d92505050565b9050600254811115610ebf5760405162461bcd60e51b8152600401808060200182810382526060815260200180611f826060913960600191505060405180910390fd5b6001556112c2565b610f2084848080601f016020809104026020016040519081016040528093929190818152602001838380828437600092019190915250506040805160608101909152602280825290925090506120876020830139611c66565b15610ff85760208114610f645760405162461bcd60e51b815260040180806020018281038252602e815260200180612038602e913960400191505060405180910390fd5b604080516020601f8401819004810282018101909252828152600091610fa291858580838501838280828437600092019190915250611d4d92505050565b90508015801590610fb557506001548110155b610ff05760405162461bcd60e51b815260040180806020018281038252606c8152602001806120f8606c913960800191505060405180910390fd5b6002556112c2565b61106c84848080601f0160208091040260200160405190810160405280939291908181526020018383808284376000920191909152505060408051808201909152601a81527f63616c6c6572436f6d70656e736174696f6e4d6f6c6563756c6500000000000060208201529150611c669050565b1561113957602081146110b05760405162461bcd60e51b815260040180806020018281038252602e815260200180612038602e913960400191505060405180910390fd5b604080516020601f84018190048102820181019092528281526000916110ee91858580838501838280828437600092019190915250611d4d92505050565b90506004548111156111315760405162461bcd60e51b8152600401808060200182810382526056815260200180611fe26056913960600191505060405180910390fd5b6003556112c2565b6111ad84848080601f0160208091040260200160405190810160405280939291908181526020018383808284376000920191909152505060408051808201909152601d81527f63616c6c6572436f6d70656e736174696f6e44656e6f6d696e61746f7200000060208201529150611c669050565b1561128557602081146111f15760405162461bcd60e51b815260040180806020018281038252602e815260200180612038602e913960400191505060405180910390fd5b604080516020601f840181900481028201810190925282815260009161122f91858580838501838280828437600092019190915250611d4d92505050565b9050801580159061124257506003548110155b61127d5760405162461bcd60e51b8152600401808060200182810382526061815260200180611f216061913960800191505060405180910390fd5b6004556112c2565b6040805162461bcd60e51b815260206004820152600d60248201526c756e6b6e6f776e20706172616d60981b604482015290519081900360640190fd5b7f6cdb0ac70ab7f2e2d035cca5be60d89906f2dede7648ddbd7402189c1eeed17a848484846040518080602001806020018381038352878782818152602001925080828437600083820152601f01601f191690910184810383528581526020019050858580828437600083820152604051601f909101601f19169092018290039850909650505050505050a150505050565b60025481565b60006028821161136b5750806113b0565b81602810801561137c575060508211155b15611389575060286113b0565b60508211801561139a5750606e8211155b156113aa575060788190036113b0565b50600481045b919050565b6000602882116113c65750806113b0565b5060286113b0565b61100281565b61100381565b60095481565b60005460ff1615611438576040805162461bcd60e51b815260206004820152601960248201527f74686520636f6e747261637420616c726561647920696e697400000000000000604482015290519081900360640190fd5b60005460ff1615611486576040805162461bcd60e51b8152602060048201526013602482015272185b1c9958591e481a5b9a5d1a585b1a5e9959606a1b604482015290519081900360640190fd5b60018080556005600255600381905560506004556000805460ff19169091179055565b6001600160a01b0381166000908152600d602052604090205480611508576040805162461bcd60e51b81526020600482015260116024820152701b9bc81c995b185e595c881c995dd85c99607a1b604482015290519081900360640190fd5b6001600160a01b0382166000818152600d60205260408082208290555184929184156108fc02918591818181858888f193505050506115b85760405161100290819084156108fc029085906000818181858888f19350505050158015611572573d6000803e3d6000fd5b506040805161100281526020810185905281517f24502838a334c8f2bb2ee1f8262a4fa7183e4489a717e96cc824e325f8b39e11929181900390910190a15050506115ff565b604080516001600160a01b03851681526020810184905281517f24502838a334c8f2bb2ee1f8262a4fa7183e4489a717e96cc824e325f8b39e11929181900390910190a150505b50565b6008818154811061089757fe5b61100081565b600381565b61100481565b600d6020526000908152604090205481565b600061165b60025461164f60015485611d5290919063ffffffff16565b9063ffffffff611dab16565b92915050565b6000828201838110156116bb576040805162461bcd60e51b815260206004820152601b60248201527f536166654d6174683a206164646974696f6e206f766572666c6f770000000000604482015290519081900360640190fd5b9392505050565b60006116bb83836040518060400160405280601e81526020017f536166654d6174683a207375627472616374696f6e206f766572666c6f770000815250611ded565b600954600680546040805160208084028201810190925282815260009493859360609383018282801561176057602002820191906000526020600020905b81546001600160a01b03168152600190910190602001808311611742575b5050505050905060608151604051908082528060200260200182016040528015611794578160200160208202803683370190505b50905060005b82518110156118215760008382815181106117b157fe5b6020026020010151905060006117eb60056000846001600160a01b03166001600160a01b03168152602001908152602001600020546113b5565b9050808484815181106117fa57fe5b6020908102919091010152611815868263ffffffff61166116565b9550505060010161179a565b50600061183f60045461164f60035488611d5290919063ffffffff16565b9050611851858263ffffffff6116c216565b94508460015b845181101561192857600061188c8761164f8a88868151811061187657fe5b6020026020010151611d5290919063ffffffff16565b90506118d981600d60008986815181106118a257fe5b60200260200101516001600160a01b03166001600160a01b031681526020019081526020016000205461166190919063ffffffff16565b600d60008885815181106118e957fe5b6020908102919091018101516001600160a01b031682528101919091526040016000205561191d838263ffffffff6116c216565b925050600101611857565b5061193e81600d6000876000815181106118a257fe5b600d60008660008151811061194f57fe5b60200260200101516001600160a01b03166001600160a01b0316815260200190815260200160002081905550600060098190555060008090505b84518110156119d057600560008683815181106119a257fe5b6020908102919091018101516001600160a01b03168252810191909152604001600090812055600101611989565b506119dd60066000611ee9565b509450505050505b90565b600a546008805460408051602080840282018101909252828152600094938593606093830182828015611a4457602002820191906000526020600020905b81546001600160a01b03168152600190910190602001808311611a26575b5050505050905060608151604051908082528060200260200182016040528015611a78578160200160208202803683370190505b50905060005b8251811015611af7576000838281518110611a9557fe5b602002602001015190506000611acf60076000846001600160a01b03166001600160a01b031681526020019081526020016000205461135a565b905080848481518110611ade57fe5b6020908102919091010152949094019350600101611a7e565b506000611b1560045461164f60035488611d5290919063ffffffff16565b9050611b27858263ffffffff6116c216565b94508460015b8451811015611bb1576000611b4c8761164f8a88868151811061187657fe5b9050611b6281600d60008986815181106118a257fe5b600d6000888581518110611b7257fe5b6020908102919091018101516001600160a01b0316825281019190915260400160002055611ba6838263ffffffff6116c216565b925050600101611b2d565b50611bc781600d6000876000815181106118a257fe5b600d600086600081518110611bd857fe5b60200260200101516001600160a01b03166001600160a01b03168152602001908152602001600020819055506000600a8190555060008090505b8451811015611c595760076000868381518110611c2b57fe5b6020908102919091018101516001600160a01b03168252810191909152604001600090812055600101611c12565b506119dd60086000611ee9565b6000816040516020018082805190602001908083835b60208310611c9b5780518252601f199092019160209182019101611c7c565b6001836020036101000a03801982511681845116808217855250505050505090500191505060405160208183030381529060405280519060200120836040516020018082805190602001908083835b60208310611d095780518252601f199092019160209182019101611cea565b6001836020036101000a0380198251168184511680821785525050505050509050019150506040516020818303038152906040528051906020012014905092915050565b015190565b600082611d615750600061165b565b82820282848281611d6e57fe5b04146116bb5760405162461bcd60e51b81526004018080602001828103825260218152602001806120666021913960400191505060405180910390fd5b60006116bb83836040518060400160405280601a81526020017f536166654d6174683a206469766973696f6e206279207a65726f000000000000815250611e84565b60008184841115611e7c5760405162461bcd60e51b81526004018080602001828103825283818151815260200191508051906020019080838360005b83811015611e41578181015183820152602001611e29565b50505050905090810190601f168015611e6e5780820380516001836020036101000a031916815260200191505b509250505060405180910390fd5b505050900390565b60008183611ed35760405162461bcd60e51b8152602060048201818152835160248401528351909283926044909101919085019080838360008315611e41578181015183820152602001611e29565b506000838581611edf57fe5b0495945050505050565b50805460008255906000526020600020908101906115ff91906119e591905b80821115611f1c5760008155600101611f08565b509056fe746865206e657743616c6c6572436f6d70656e736174696f6e44656e6f6d696e61746f72206d757374206e6f74206265207a65726f20616e64206e6f206c657373207468616e2063616c6c6572436f6d70656e736174696f6e4d6f6c6563756c656e65772068656164657252656c61796572526577617264526174654d6f6c6563756c652073686f756c646e27742062652067726561746572207468616e2068656164657252656c617965725265776172645261746544656e6f6d696e61746f726e65772063616c6c6572436f6d70656e736174696f6e4d6f6c6563756c652073686f756c646e27742062652067726561746572207468616e2063616c6c6572436f6d70656e736174696f6e44656e6f6d696e61746f726c656e677468206f6620726577617264466f7256616c696461746f725365744368616e6765206d69736d61746368536166654d6174683a206d756c7469706c69636174696f6e206f766572666c6f7768656164657252656c617965725265776172645261746544656e6f6d696e61746f72746865206d6573736167652073656e646572206d75737420626520676f7665726e616e636520636f6e7472616374636f6e747261637420686173206e6f74206265656e20696e697469616c697a6564746865206e65772068656164657252656c617965725265776172645261746544656e6f6d696e61746f72206d757374206e6f74206265207a65726f20616e64206e6f206c657373207468616e2068656164657252656c61796572526577617264526174654d6f6c6563756c65746865206d6573736167652073656e646572206d7573742062652063726f737320636861696e20636f6e74726163746c656e677468206f662068656164657252656c61796572526577617264526174654d6f6c6563756c65206d69736d61746368a264697066735822122046aa8f3564133ae19cf0f29fe8a7ebdf2a5f74f0268c42431d82384fe137b2ed64736f6c63430006040033",
			},
			{
				ContractAddr: common.HexToAddress(RelayerHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(GovHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(TokenManagerContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(CrossChainContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
		},
	}

	nielsUpgrade[chapelNet] = &Upgrade{
		UpgradeName: "niels",
		Configs: []*UpgradeConfig{
			{
				ContractAddr: common.HexToAddress(ValidatorContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(SlashContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(SystemRewardContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(LightClientContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(TokenHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(RelayerIncentivizeContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "6080604052600436106102765760003560e01c80637e146cc51161014f578063af400681116100c1578063e75d72c71161007a578063e75d72c714610795578063e89a3020146107c8578063f9a2bbc7146107f2578063fc3e590814610807578063fd6a68791461081c578063fdd31fcd146108315761027d565b8063af400681146106ed578063bd4cc83014610717578063c81b166214610741578063dc927faf14610756578063dcae76ab1461076b578063e1c7392a146107805761027d565b8063a3c3c0ad11610113578063a3c3c0ad146105b3578063a78abc16146105c8578063a7c6a59d146105dd578063ab51bb96146105f2578063ac43175114610607578063ace9fcc2146106d85761027d565b80637e146cc51461052c578063930e1b091461054157806396713da9146105745780639dc0926214610589578063a1a11bf51461059e5761027d565b806343756e5c116101e85780636e47b482116101ac5780636e47b482146104645780636f93d2e61461047957806370fd5bad146104d857806374f2272d146104ed57806375d47a0a146105025780637942fd05146105175761027d565b806343756e5c146103e4578063493279b1146103f95780634bf6c8821461042557806351e806721461043a578063541333071461044f5761027d565b806312950c461161023a57806312950c46146103165780631b20087c1461032b5780631c643312146103405780633a975612146102825780633dffc3871461038657806340bb43c0146103b15761027d565b8063081e9d131461028257806308f2ec06146102a9578063093f2fc4146102be5780630bee7a67146102d357806310e06a76146103015761027d565b3661027d57005b600080fd5b34801561028e57600080fd5b50610297610864565b60408051918252519081900360200190f35b3480156102b557600080fd5b50610297610869565b3480156102ca57600080fd5b5061029761086e565b3480156102df57600080fd5b506102e8610873565b6040805163ffffffff9092168252519081900360200190f35b34801561030d57600080fd5b50610297610878565b34801561032257600080fd5b5061029761087e565b34801561033757600080fd5b50610297610884565b34801561034c57600080fd5b5061036a6004803603602081101561036357600080fd5b503561088a565b604080516001600160a01b039092168252519081900360200190f35b34801561039257600080fd5b5061039b610864565b6040805160ff9092168252519081900360200190f35b3480156103bd57600080fd5b50610297600480360360208110156103d457600080fd5b50356001600160a01b03166108b1565b3480156103f057600080fd5b5061036a6108c3565b34801561040557600080fd5b5061040e6108c9565b6040805161ffff9092168252519081900360200190f35b34801561043157600080fd5b5061039b6108ce565b34801561044657600080fd5b5061036a6108d3565b34801561045b57600080fd5b50610297610873565b34801561047057600080fd5b5061036a6108d9565b34801561048557600080fd5b506104c46004803603608081101561049c57600080fd5b506001600160a01b0381358116916020810135909116906040810135906060013515156108df565b604080519115158252519081900360200190f35b3480156104e457600080fd5b5061039b610cad565b3480156104f957600080fd5b50610297610cb2565b34801561050e57600080fd5b5061036a610cb8565b34801561052357600080fd5b5061039b610cbe565b34801561053857600080fd5b50610297610cc3565b34801561054d57600080fd5b506102976004803603602081101561056457600080fd5b50356001600160a01b0316610cc8565b34801561058057600080fd5b5061039b610cda565b34801561059557600080fd5b5061036a610cdf565b3480156105aa57600080fd5b5061036a610ce5565b3480156105bf57600080fd5b50610297610ceb565b3480156105d457600080fd5b506104c4610cf1565b3480156105e957600080fd5b50610297610cfa565b3480156105fe57600080fd5b506102e8610d00565b34801561061357600080fd5b506106d66004803603604081101561062a57600080fd5b81019060208101813564010000000081111561064557600080fd5b82018360208201111561065757600080fd5b8035906020019184600183028401116401000000008311171561067957600080fd5b91939092909160208101903564010000000081111561069757600080fd5b8201836020820111156106a957600080fd5b803590602001918460018302840111640100000000831117156106cb57600080fd5b509092509050610d05565b005b3480156106e457600080fd5b50610297611354565b3480156106f957600080fd5b506102976004803603602081101561071057600080fd5b503561135a565b34801561072357600080fd5b506102976004803603602081101561073a57600080fd5b50356113b5565b34801561074d57600080fd5b5061036a6113ce565b34801561076257600080fd5b5061036a6113d4565b34801561077757600080fd5b506102976113da565b34801561078c57600080fd5b506106d66113e0565b3480156107a157600080fd5b506106d6600480360360208110156107b857600080fd5b50356001600160a01b03166114a9565b3480156107d457600080fd5b5061036a600480360360208110156107eb57600080fd5b5035611602565b3480156107fe57600080fd5b5061036a61160f565b34801561081357600080fd5b5061039b611615565b34801561082857600080fd5b5061036a61161a565b34801561083d57600080fd5b506102976004803603602081101561085457600080fd5b50356001600160a01b0316611620565b600181565b602881565b605081565b606481565b600b5481565b60015481565b600c5481565b6006818154811061089757fe5b6000918252602090912001546001600160a01b0316905081565b60076020526000908152604090205481565b61100181565b606181565b600881565b61200081565b61100581565b6000805460ff16610937576040805162461bcd60e51b815260206004820152601960248201527f74686520636f6e7472616374206e6f7420696e69742079657400000000000000604482015290519081900360640190fd5b33612000146109775760405162461bcd60e51b815260040180806020018281038252602f815260200180612164602f913960400191505060405180910390fd5b600082156109fc57604080516309a99b4f60e41b8152611005600482015260248101869052905161100291639a99b4f09160448083019260209291908290030181600087803b1580156109c957600080fd5b505af11580156109dd573d6000803e3d6000fd5b505050506040513d60208110156109f357600080fd5b50519050610a75565b604080516309a99b4f60e41b8152611005600482015260248101869052905161100491639a99b4f09160448083019260209291908290030181600087803b158015610a4657600080fd5b505af1158015610a5a573d6000803e3d6000fd5b505050506040513d6020811015610a7057600080fd5b505190505b600c805460010190556000610a8982611632565b600954909150610a9f908263ffffffff61166116565b600955600a54610ac7908290610abb908563ffffffff61166116565b9063ffffffff6116c216565b600a556001600160a01b038716600090815260056020526040902054610b3357600680546001810182556000919091527ff652222313e28459528d920b65115c16c04f3efc82aaedc97be59f3f377c0d3f0180546001600160a01b0319166001600160a01b0389161790555b6001600160a01b038088166000908152600560209081526040808320805460010190559289168252600790522054610bb157600880546001810182556000919091527ff3f7a9fe364faab93b216da50a3214154f22a0a2b415b23a84c8169e8b636ee30180546001600160a01b0319166001600160a01b0388161790555b6001600160a01b038616600090815260076020526040902080546001019055600c54606411610ca057600b54600954600a5460408051938452602084019290925282820152517f2649b1b772a1a74bd332a67695e285317dd722941166595741c60a00fa65bb759181900360600190a16000610c2b611704565b90506000610c376119e8565b6001600160a01b0389166000908152600d6020526040902054909150610c75908290610c69908563ffffffff61166116565b9063ffffffff61166116565b6001600160a01b0389166000908152600d6020526040812091909155600b80546001019055600c5550505b5060019695505050505050565b600281565b60035481565b61100881565b600b81565b600581565b60056020526000908152604090205481565b600981565b61100781565b61100681565b600a5481565b60005460ff1681565b60045481565b600081565b3361100714610d455760405162461bcd60e51b815260040180806020018281038252602e8152602001806120a9602e913960400191505060405180910390fd5b60005460ff16610d865760405162461bcd60e51b81526004018080602001828103825260218152602001806120d76021913960400191505060405180910390fd5b610dfa84848080601f0160208091040260200160405190810160405280939291908181526020018383808284376000920191909152505060408051808201909152601f81527f68656164657252656c61796572526577617264526174654d6f6c6563756c650060208201529150611c669050565b15610ec75760208114610e3e5760405162461bcd60e51b81526004018080602001828103825260328152602001806121936032913960400191505060405180910390fd5b604080516020601f8401819004810282018101909252828152600091610e7c91858580838501838280828437600092019190915250611d4d92505050565b9050600254811115610ebf5760405162461bcd60e51b8152600401808060200182810382526060815260200180611f826060913960600191505060405180910390fd5b6001556112c2565b610f2084848080601f016020809104026020016040519081016040528093929190818152602001838380828437600092019190915250506040805160608101909152602280825290925090506120876020830139611c66565b15610ff85760208114610f645760405162461bcd60e51b815260040180806020018281038252602e815260200180612038602e913960400191505060405180910390fd5b604080516020601f8401819004810282018101909252828152600091610fa291858580838501838280828437600092019190915250611d4d92505050565b90508015801590610fb557506001548110155b610ff05760405162461bcd60e51b815260040180806020018281038252606c8152602001806120f8606c913960800191505060405180910390fd5b6002556112c2565b61106c84848080601f0160208091040260200160405190810160405280939291908181526020018383808284376000920191909152505060408051808201909152601a81527f63616c6c6572436f6d70656e736174696f6e4d6f6c6563756c6500000000000060208201529150611c669050565b1561113957602081146110b05760405162461bcd60e51b815260040180806020018281038252602e815260200180612038602e913960400191505060405180910390fd5b604080516020601f84018190048102820181019092528281526000916110ee91858580838501838280828437600092019190915250611d4d92505050565b90506004548111156111315760405162461bcd60e51b8152600401808060200182810382526056815260200180611fe26056913960600191505060405180910390fd5b6003556112c2565b6111ad84848080601f0160208091040260200160405190810160405280939291908181526020018383808284376000920191909152505060408051808201909152601d81527f63616c6c6572436f6d70656e736174696f6e44656e6f6d696e61746f7200000060208201529150611c669050565b1561128557602081146111f15760405162461bcd60e51b815260040180806020018281038252602e815260200180612038602e913960400191505060405180910390fd5b604080516020601f840181900481028201810190925282815260009161122f91858580838501838280828437600092019190915250611d4d92505050565b9050801580159061124257506003548110155b61127d5760405162461bcd60e51b8152600401808060200182810382526061815260200180611f216061913960800191505060405180910390fd5b6004556112c2565b6040805162461bcd60e51b815260206004820152600d60248201526c756e6b6e6f776e20706172616d60981b604482015290519081900360640190fd5b7f6cdb0ac70ab7f2e2d035cca5be60d89906f2dede7648ddbd7402189c1eeed17a848484846040518080602001806020018381038352878782818152602001925080828437600083820152601f01601f191690910184810383528581526020019050858580828437600083820152604051601f909101601f19169092018290039850909650505050505050a150505050565b60025481565b60006028821161136b5750806113b0565b81602810801561137c575060508211155b15611389575060286113b0565b60508211801561139a5750606e8211155b156113aa575060788190036113b0565b50600481045b919050565b6000602882116113c65750806113b0565b5060286113b0565b61100281565b61100381565b60095481565b60005460ff1615611438576040805162461bcd60e51b815260206004820152601960248201527f74686520636f6e747261637420616c726561647920696e697400000000000000604482015290519081900360640190fd5b60005460ff1615611486576040805162461bcd60e51b8152602060048201526013602482015272185b1c9958591e481a5b9a5d1a585b1a5e9959606a1b604482015290519081900360640190fd5b60018080556005600255600381905560506004556000805460ff19169091179055565b6001600160a01b0381166000908152600d602052604090205480611508576040805162461bcd60e51b81526020600482015260116024820152701b9bc81c995b185e595c881c995dd85c99607a1b604482015290519081900360640190fd5b6001600160a01b0382166000818152600d60205260408082208290555184929184156108fc02918591818181858888f193505050506115b85760405161100290819084156108fc029085906000818181858888f19350505050158015611572573d6000803e3d6000fd5b506040805161100281526020810185905281517f24502838a334c8f2bb2ee1f8262a4fa7183e4489a717e96cc824e325f8b39e11929181900390910190a15050506115ff565b604080516001600160a01b03851681526020810184905281517f24502838a334c8f2bb2ee1f8262a4fa7183e4489a717e96cc824e325f8b39e11929181900390910190a150505b50565b6008818154811061089757fe5b61100081565b600381565b61100481565b600d6020526000908152604090205481565b600061165b60025461164f60015485611d5290919063ffffffff16565b9063ffffffff611dab16565b92915050565b6000828201838110156116bb576040805162461bcd60e51b815260206004820152601b60248201527f536166654d6174683a206164646974696f6e206f766572666c6f770000000000604482015290519081900360640190fd5b9392505050565b60006116bb83836040518060400160405280601e81526020017f536166654d6174683a207375627472616374696f6e206f766572666c6f770000815250611ded565b600954600680546040805160208084028201810190925282815260009493859360609383018282801561176057602002820191906000526020600020905b81546001600160a01b03168152600190910190602001808311611742575b5050505050905060608151604051908082528060200260200182016040528015611794578160200160208202803683370190505b50905060005b82518110156118215760008382815181106117b157fe5b6020026020010151905060006117eb60056000846001600160a01b03166001600160a01b03168152602001908152602001600020546113b5565b9050808484815181106117fa57fe5b6020908102919091010152611815868263ffffffff61166116565b9550505060010161179a565b50600061183f60045461164f60035488611d5290919063ffffffff16565b9050611851858263ffffffff6116c216565b94508460015b845181101561192857600061188c8761164f8a88868151811061187657fe5b6020026020010151611d5290919063ffffffff16565b90506118d981600d60008986815181106118a257fe5b60200260200101516001600160a01b03166001600160a01b031681526020019081526020016000205461166190919063ffffffff16565b600d60008885815181106118e957fe5b6020908102919091018101516001600160a01b031682528101919091526040016000205561191d838263ffffffff6116c216565b925050600101611857565b5061193e81600d6000876000815181106118a257fe5b600d60008660008151811061194f57fe5b60200260200101516001600160a01b03166001600160a01b0316815260200190815260200160002081905550600060098190555060008090505b84518110156119d057600560008683815181106119a257fe5b6020908102919091018101516001600160a01b03168252810191909152604001600090812055600101611989565b506119dd60066000611ee9565b509450505050505b90565b600a546008805460408051602080840282018101909252828152600094938593606093830182828015611a4457602002820191906000526020600020905b81546001600160a01b03168152600190910190602001808311611a26575b5050505050905060608151604051908082528060200260200182016040528015611a78578160200160208202803683370190505b50905060005b8251811015611af7576000838281518110611a9557fe5b602002602001015190506000611acf60076000846001600160a01b03166001600160a01b031681526020019081526020016000205461135a565b905080848481518110611ade57fe5b6020908102919091010152949094019350600101611a7e565b506000611b1560045461164f60035488611d5290919063ffffffff16565b9050611b27858263ffffffff6116c216565b94508460015b8451811015611bb1576000611b4c8761164f8a88868151811061187657fe5b9050611b6281600d60008986815181106118a257fe5b600d6000888581518110611b7257fe5b6020908102919091018101516001600160a01b0316825281019190915260400160002055611ba6838263ffffffff6116c216565b925050600101611b2d565b50611bc781600d6000876000815181106118a257fe5b600d600086600081518110611bd857fe5b60200260200101516001600160a01b03166001600160a01b03168152602001908152602001600020819055506000600a8190555060008090505b8451811015611c595760076000868381518110611c2b57fe5b6020908102919091018101516001600160a01b03168252810191909152604001600090812055600101611c12565b506119dd60086000611ee9565b6000816040516020018082805190602001908083835b60208310611c9b5780518252601f199092019160209182019101611c7c565b6001836020036101000a03801982511681845116808217855250505050505090500191505060405160208183030381529060405280519060200120836040516020018082805190602001908083835b60208310611d095780518252601f199092019160209182019101611cea565b6001836020036101000a0380198251168184511680821785525050505050509050019150506040516020818303038152906040528051906020012014905092915050565b015190565b600082611d615750600061165b565b82820282848281611d6e57fe5b04146116bb5760405162461bcd60e51b81526004018080602001828103825260218152602001806120666021913960400191505060405180910390fd5b60006116bb83836040518060400160405280601a81526020017f536166654d6174683a206469766973696f6e206279207a65726f000000000000815250611e84565b60008184841115611e7c5760405162461bcd60e51b81526004018080602001828103825283818151815260200191508051906020019080838360005b83811015611e41578181015183820152602001611e29565b50505050905090810190601f168015611e6e5780820380516001836020036101000a031916815260200191505b509250505060405180910390fd5b505050900390565b60008183611ed35760405162461bcd60e51b8152602060048201818152835160248401528351909283926044909101919085019080838360008315611e41578181015183820152602001611e29565b506000838581611edf57fe5b0495945050505050565b50805460008255906000526020600020908101906115ff91906119e591905b80821115611f1c5760008155600101611f08565b509056fe746865206e657743616c6c6572436f6d70656e736174696f6e44656e6f6d696e61746f72206d757374206e6f74206265207a65726f20616e64206e6f206c657373207468616e2063616c6c6572436f6d70656e736174696f6e4d6f6c6563756c656e65772068656164657252656c61796572526577617264526174654d6f6c6563756c652073686f756c646e27742062652067726561746572207468616e2068656164657252656c617965725265776172645261746544656e6f6d696e61746f726e65772063616c6c6572436f6d70656e736174696f6e4d6f6c6563756c652073686f756c646e27742062652067726561746572207468616e2063616c6c6572436f6d70656e736174696f6e44656e6f6d696e61746f726c656e677468206f6620726577617264466f7256616c696461746f725365744368616e6765206d69736d61746368536166654d6174683a206d756c7469706c69636174696f6e206f766572666c6f7768656164657252656c617965725265776172645261746544656e6f6d696e61746f72746865206d6573736167652073656e646572206d75737420626520676f7665726e616e636520636f6e7472616374636f6e747261637420686173206e6f74206265656e20696e697469616c697a6564746865206e65772068656164657252656c617965725265776172645261746544656e6f6d696e61746f72206d757374206e6f74206265207a65726f20616e64206e6f206c657373207468616e2068656164657252656c61796572526577617264526174654d6f6c6563756c65746865206d6573736167652073656e646572206d7573742062652063726f737320636861696e20636f6e74726163746c656e677468206f662068656164657252656c61796572526577617264526174654d6f6c6563756c65206d69736d61746368a2646970667358221220b0f586134d92421afeb7764dfa9711bf04e0920aa1729c40e9183ccccafbb49d64736f6c63430006040033",
			},
			{
				ContractAddr: common.HexToAddress(RelayerHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(GovHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(TokenManagerContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(CrossChainContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/823b953232a344ba3c32d6690e70a245477e5760",
				Code:         "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",
			},
		},
	}

	mirrorUpgrade[mainNet] = &Upgrade{
		UpgradeName: "mirror",
		Configs: []*UpgradeConfig{
			{
				ContractAddr: common.HexToAddress(TokenManagerContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/af4f3993303213052222f55c721e661862d19638",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(TokenHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/af4f3993303213052222f55c721e661862d19638",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(RelayerIncentivizeContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/af4f3993303213052222f55c721e661862d19638",
				Code:         "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",
			},
		},
	}

	mirrorUpgrade[chapelNet] = &Upgrade{
		UpgradeName: "mirror",
		Configs: []*UpgradeConfig{
			{
				ContractAddr: common.HexToAddress(TokenManagerContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/af4f3993303213052222f55c721e661862d19638",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(TokenHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/af4f3993303213052222f55c721e661862d19638",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(RelayerIncentivizeContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/af4f3993303213052222f55c721e661862d19638",
				Code:         "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",
			},
		},
	}

	mirrorUpgrade[rialtoNet] = &Upgrade{
		UpgradeName: "mirror",
		Configs: []*UpgradeConfig{
			{
				ContractAddr: common.HexToAddress(TokenManagerContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/af4f3993303213052222f55c721e661862d19638",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(TokenHubContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/af4f3993303213052222f55c721e661862d19638",
				Code:         "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",
			},
			{
				ContractAddr: common.HexToAddress(RelayerIncentivizeContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/af4f3993303213052222f55c721e661862d19638",
				Code:         "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",
			},
		},
	}

	brunoUpgrade[mainNet] = &Upgrade{
		UpgradeName: "bruno",
		Configs: []*UpgradeConfig{
			{
				ContractAddr: common.HexToAddress(ValidatorContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/ce622fef469d84ee418fa6181f3ac962412a5f4f",
				Code:         "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",
			},
		},
	}

	brunoUpgrade[chapelNet] = &Upgrade{
		UpgradeName: "bruno",
		Configs: []*UpgradeConfig{
			{
				ContractAddr: common.HexToAddress(ValidatorContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/ce622fef469d84ee418fa6181f3ac962412a5f4f",
				Code:         "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",
			},
		},
	}

	brunoUpgrade[rialtoNet] = &Upgrade{
		UpgradeName: "bruno",
		Configs: []*UpgradeConfig{
			{
				ContractAddr: common.HexToAddress(ValidatorContract),
				CommitUrl:    "https://github.com/binance-chain/bsc-genesis-contract/commit/ce622fef469d84ee418fa6181f3ac962412a5f4f",
				Code:         "6080604052600436106102e45760003560e01c80638624988211610190578063c8509d81116100dc578063eb57e20211610095578063f9a2bbc71161006f578063f9a2bbc714610a54578063fc3e590814610a69578063fccc281314610a7e578063fd6a687914610a93576102e4565b8063eb57e202146109e6578063eda5868c14610a19578063f340fa0114610a2e576102e4565b8063c8509d81146106d8578063d86222d51461097d578063daacdb6614610992578063dc927faf146109a7578063e086c7b1146109bc578063e1c7392a146109d1576102e4565b8063aaf5eb6811610149578063ad3c9da611610123578063ad3c9da6146108d0578063b7ab4db514610903578063bf9f4995146104c0578063c81b166214610968576102e4565b8063aaf5eb68146107db578063ab51bb96146107f0578063ac43175114610805576102e4565b8063862498821461075d57806396713da9146107725780639dc0926214610787578063a1a11bf51461079c578063a5422d5c146107b1578063a78abc16146107c6576102e4565b806351e806721161024f57806370fd5bad116102085780637942fd05116101e25780637942fd05146106ae57806381650b62146106c3578063831d65d1146106d8578063853230aa14610699576102e4565b806370fd5bad1461066f57806375d47a0a1461068457806378dfed4a14610699576102e4565b806351e8067214610572578063565c56b3146105875780635667515a146105ba5780635d77156c146105cf5780636969a25c146105e45780636e47b4821461065a576102e4565b80633de0f0d8116102a15780633de0f0d8146104ab5780633dffc387146104c057806343756e5c146104eb578063493279b11461051c5780634bf6c882146105485780635192c82c1461055d576102e4565b80630bee7a67146102e95780631182b87514610317578063152ad3b8146104115780631ff180691461043a578063219f22d51461046157806335409f7f14610476575b600080fd5b3480156102f557600080fd5b506102fe610aa8565b6040805163ffffffff9092168252519081900360200190f35b34801561032357600080fd5b5061039c6004803603604081101561033a57600080fd5b60ff8235169190810190604081016020820135600160201b81111561035e57600080fd5b82018360208201111561037057600080fd5b803590602001918460018302840111600160201b8311171561039157600080fd5b509092509050610aad565b6040805160208082528351818301528351919283929083019185019080838360005b838110156103d65781810151838201526020016103be565b50505050905090810190601f1680156104035780820380516001836020036101000a031916815260200191505b509250505060405180910390f35b34801561041d57600080fd5b50610426610c9a565b604080519115158252519081900360200190f35b34801561044657600080fd5b5061044f610ca3565b60408051918252519081900360200190f35b34801561046d57600080fd5b506102fe610ca9565b34801561048257600080fd5b506104a96004803603602081101561049957600080fd5b50356001600160a01b0316610cae565b005b3480156104b757600080fd5b5061044f610fc2565b3480156104cc57600080fd5b506104d5610fc8565b6040805160ff9092168252519081900360200190f35b3480156104f757600080fd5b50610500610fcd565b604080516001600160a01b039092168252519081900360200190f35b34801561052857600080fd5b50610531610fd3565b6040805161ffff9092168252519081900360200190f35b34801561055457600080fd5b506104d5610fd9565b34801561056957600080fd5b5061044f610fde565b34801561057e57600080fd5b50610500610fe4565b34801561059357600080fd5b5061044f600480360360208110156105aa57600080fd5b50356001600160a01b0316610fea565b3480156105c657600080fd5b506104d561103c565b3480156105db57600080fd5b506102fe611041565b3480156105f057600080fd5b5061060e6004803603602081101561060757600080fd5b5035611046565b604080516001600160a01b039788168152958716602087015293909516848401526001600160401b0390911660608401521515608083015260a082019290925290519081900360c00190f35b34801561066657600080fd5b506105006110aa565b34801561067b57600080fd5b506104d56110b0565b34801561069057600080fd5b506105006110b5565b3480156106a557600080fd5b5061044f6110bb565b3480156106ba57600080fd5b506104d56110c1565b3480156106cf57600080fd5b506102fe6110c6565b3480156106e457600080fd5b506104a9600480360360408110156106fb57600080fd5b60ff8235169190810190604081016020820135600160201b81111561071f57600080fd5b82018360208201111561073157600080fd5b803590602001918460018302840111600160201b8311171561075257600080fd5b5090925090506110cb565b34801561076957600080fd5b5061044f61117e565b34801561077e57600080fd5b506104d5611184565b34801561079357600080fd5b50610500611189565b3480156107a857600080fd5b5061050061118f565b3480156107bd57600080fd5b5061039c611195565b3480156107d257600080fd5b506104266111b4565b3480156107e757600080fd5b5061044f6111bd565b3480156107fc57600080fd5b506102fe61103c565b34801561081157600080fd5b506104a96004803603604081101561082857600080fd5b810190602081018135600160201b81111561084257600080fd5b82018360208201111561085457600080fd5b803590602001918460018302840111600160201b8311171561087557600080fd5b919390929091602081019035600160201b81111561089257600080fd5b8201836020820111156108a457600080fd5b803590602001918460018302840111600160201b831117156108c557600080fd5b5090925090506111c6565b3480156108dc57600080fd5b5061044f600480360360208110156108f357600080fd5b50356001600160a01b03166115bd565b34801561090f57600080fd5b506109186115cf565b60408051602080825283518183015283519192839290830191858101910280838360005b8381101561095457818101518382015260200161093c565b505050509050019250505060405180910390f35b34801561097457600080fd5b506105006116f5565b34801561098957600080fd5b5061044f6116fb565b34801561099e57600080fd5b5061044f611707565b3480156109b357600080fd5b5061050061170d565b3480156109c857600080fd5b5061044f611713565b3480156109dd57600080fd5b506104a9611718565b3480156109f257600080fd5b506104a960048036036020811015610a0957600080fd5b50356001600160a01b031661191b565b348015610a2557600080fd5b506102fe611aea565b6104a960048036036020811015610a4457600080fd5b50356001600160a01b0316611aef565b348015610a6057600080fd5b50610500611df4565b348015610a7557600080fd5b506104d5611dfa565b348015610a8a57600080fd5b50610500611dff565b348015610a9f57600080fd5b50610500611e05565b606481565b60005460609060ff16610b03576040805162461bcd60e51b81526020600482015260196024820152781d1a194818dbdb9d1c9858dd081b9bdd081a5b9a5d081e595d603a1b604482015290519081900360640190fd5b3361200014610b435760405162461bcd60e51b815260040180806020018281038252602f815260200180614952602f913960400191505060405180910390fd5b610b4b614159565b6000610b8c85858080601f016020809104026020016040519081016040528093929190818152602001838380828437600092019190915250611e0b92505050565b9150915080610ba857610b9f6064611f64565b92505050610c93565b815160009060ff16610bc857610bc18360200151611fc5565b9050610c5f565b825160ff1660011415610c5b57826020015151600114610c35577f70e72399380dcfb0338abc03dc8d47f9f470ada8e769c9a78d644ea97385ecb26040518080602001828103825260258152602001806142876025913960400191505060405180910390a1506067610c56565b610bc18360200151600081518110610c4957fe5b6020026020010151612dcb565b610c5f565b5060655b63ffffffff8116610c845750506040805160008152602081019091529150610c939050565b610c8d81611f64565b93505050505b9392505050565b60075460ff1681565b60035481565b606881565b3361100114610cee5760405162461bcd60e51b81526004018080602001828103825260298152602001806149ae6029913960400191505060405180910390fd5b6001600160a01b03811660009081526004602052604090205480610d125750610fbf565b600181039050600060018281548110610d2757fe5b60009182526020909120600360049092020101546001549091506000190180610d7657600060018481548110610d5957fe5b906000526020600020906004020160030181905550505050610fbf565b6040805183815290516001600160a01b038616917f3b6f9ef90462b512a1293ecec018670bf7b7f1876fb727590a8a6d7643130a70919081900360200190a26001600160a01b038416600090815260046020526040812055600154600019018314610ef857600180546000198101908110610ded57fe5b906000526020600020906004020160018481548110610e0857fe5b6000918252602082208354600492830290910180546001600160a01b03199081166001600160a01b0393841617825560018087015481840180548416918616919091179055600280880180549185018054909416919095161780835584546001600160401b03600160a01b91829004160267ffffffffffffffff60a01b1990911617808355935460ff600160e01b918290041615150260ff60e01b199094169390931790556003948501549401939093558254868401939192919087908110610ecd57fe5b600091825260208083206004909202909101546001600160a01b031683528201929092526040019020555b6001805480610f0357fe5b60008281526020812060046000199093019283020180546001600160a01b0319908116825560018201805490911690556002810180546001600160e81b03191690556003018190559155818381610f5657fe5b0490508015610fba5760015460005b81811015610fb7578260018281548110610f7b57fe5b9060005260206000209060040201600301540160018281548110610f9b57fe5b6000918252602090912060036004909202010155600101610f65565b50505b505050505b50565b61271081565b600181565b61100181565b6102ca81565b600881565b60065481565b61200081565b6001600160a01b03811660009081526004602052604081205480611012576000915050611037565b60018082038154811061102157fe5b9060005260206000209060040201600301549150505b919050565b600081565b606781565b6001818154811061105357fe5b600091825260209091206004909102018054600182015460028301546003909301546001600160a01b0392831694509082169291821691600160a01b81046001600160401b031691600160e01b90910460ff169086565b61100581565b600281565b61100881565b6103e881565b600b81565b606681565b336120001461110b5760405162461bcd60e51b815260040180806020018281038252602f815260200180614952602f913960400191505060405180910390fd5b7f41ce201247b6ceb957dcdb217d0b8acb50b9ea0e12af9af4f5e7f38902101605838383604051808460ff1660ff168152602001806020018281038252848482818152602001925080828437600083820152604051601f909101601f1916909201829003965090945050505050a1505050565b60025481565b600981565b61100781565b61100681565b6040518061062001604052806105ef81526020016143636105ef913981565b60005460ff1681565b6402540be40081565b60005460ff16611219576040805162461bcd60e51b81526020600482015260196024820152781d1a194818dbdb9d1c9858dd081b9bdd081a5b9a5d081e595d603a1b604482015290519081900360640190fd5b33611007146112595760405162461bcd60e51b815260040180806020018281038252602e8152602001806142cd602e913960400191505060405180910390fd5b6112c384848080601f01602080910402602001604051908101604052809392919081815260200183838082843760009201919091525050604080518082019091526013815272065787069726554696d655365636f6e6447617606c1b60208201529150612f429050565b1561139e57602081146113075760405162461bcd60e51b815260040180806020018281038252602681526020018061431c6026913960400191505060405180910390fd5b604080516020601f84018190048102820181019092528281526000916113459185858083850183828082843760009201919091525061302a92505050565b90506064811015801561135b5750620186a08111155b6113965760405162461bcd60e51b81526004018080602001828103825260278152602001806142606027913960400191505060405180910390fd5b60025561152b565b6113fe84848080601f016020809104026020016040519081016040528093929190818152602001838380828437600092019190915250506040805180820190915260098152686275726e526174696f60b81b60208201529150612f429050565b156114ee5760208114611458576040805162461bcd60e51b815260206004820152601c60248201527f6c656e677468206f66206275726e526174696f206d69736d6174636800000000604482015290519081900360640190fd5b604080516020601f84018190048102820181019092528281526000916114969185858083850183828082843760009201919091525061302a92505050565b90506127108111156114d95760405162461bcd60e51b815260040180806020018281038252602b81526020018061420a602b913960400191505060405180910390fd5b6006556007805460ff1916600117905561152b565b6040805162461bcd60e51b815260206004820152600d60248201526c756e6b6e6f776e20706172616d60981b604482015290519081900360640190fd5b7f6cdb0ac70ab7f2e2d035cca5be60d89906f2dede7648ddbd7402189c1eeed17a848484846040518080602001806020018381038352878782818152602001925080828437600083820152601f01601f191690910184810383528581526020019050858580828437600083820152604051601f909101601f19169092018290039850909650505050505050a150505050565b60046020526000908152604090205481565b6001546060906000805b8281101561162057600181815481106115ee57fe5b9060005260206000209060040201600201601c9054906101000a900460ff16611618576001909101905b6001016115d9565b5060608160405190808252806020026020018201604052801561164d578160200160208202803683370190505b50600092509050815b838110156116ed576001818154811061166b57fe5b9060005260206000209060040201600201601c9054906101000a900460ff166116e5576001818154811061169b57fe5b600091825260209091206004909102015482516001600160a01b03909116908390859081106116c657fe5b6001600160a01b03909216602092830291909101909101526001909201915b600101611656565b509250505090565b61100281565b67016345785d8a000081565b60055481565b61100381565b602981565b60005460ff1615611770576040805162461bcd60e51b815260206004820152601960248201527f74686520636f6e747261637420616c726561647920696e697400000000000000604482015290519081900360640190fd5b611778614159565b600061179e6040518061062001604052806105ef81526020016143636105ef9139611e0b565b91509150806117de5760405162461bcd60e51b81526004018080602001828103825260218152602001806143426021913960400191505060405180910390fd5b60005b8260200151518110156119035760018360200151828151811061180057fe5b60209081029190910181015182546001818101855560009485528385208351600493840290910180546001600160a01b039283166001600160a01b03199182161782558587015182850180549185169183169190911790556040860151600283018054606089015160808a01511515600160e01b0260ff60e01b196001600160401b03909216600160a01b0267ffffffffffffffff60a01b199590981692909516919091179290921694909417161790915560a0909301516003909301929092559186015180519185019391859081106118d657fe5b602090810291909101810151516001600160a01b03168252810191909152604001600020556001016117e1565b50506103e8600255506000805460ff19166001179055565b336110011461195b5760405162461bcd60e51b81526004018080602001828103825260298152602001806149ae6029913960400191505060405180910390fd5b6001600160a01b0381166000908152600460205260409020548061197f5750610fbf565b60018103905060006001828154811061199457fe5b90600052602060002090600402016003015490506000600183815481106119b757fe5b906000526020600020906004020160030181905550600060018080549050039050836001600160a01b03167f8cd4e147d8af98a9e3b6724021b8bf6aed2e5dac71c38f2dce8161b82585b25d836040518082815260200191505060405180910390a280611a2657505050610fbf565b6000818381611a3157fe5b0490508015610fba5760005b84811015611a8f578160018281548110611a5357fe5b9060005260206000209060040201600301540160018281548110611a7357fe5b6000918252602090912060036004909202010155600101611a3d565b50600180549085015b81811015610fb7578260018281548110611aae57fe5b9060005260206000209060040201600301540160018281548110611ace57fe5b6000918252602090912060036004909202010155600101611a98565b606581565b334114611b2d5760405162461bcd60e51b815260040180806020018281038252602d815260200180614981602d913960400191505060405180910390fd5b60005460ff16611b80576040805162461bcd60e51b81526020600482015260196024820152781d1a194818dbdb9d1c9858dd081b9bdd081a5b9a5d081e595d603a1b604482015290519081900360640190fd5b60003411611bcd576040805162461bcd60e51b81526020600482015260156024820152746465706f7369742076616c7565206973207a65726f60581b604482015290519081900360640190fd5b6001600160a01b0381166000908152600460205260409020546007543491906103e89060ff1615611bfd57506006545b600083118015611c0d5750600081115b15611cb6576000611c36612710611c2a868563ffffffff61302f16565b9063ffffffff61308816565b90508015611cb45760405161dead9082156108fc029083906000818181858888f19350505050158015611c6d573d6000803e3d6000fd5b506040805182815290517f627059660ea01c4733a328effb2294d2f86905bf806da763a89cee254de8bee59181900360200190a1611cb1848263ffffffff6130ca16565b93505b505b8115611dae576000600180840381548110611ccd57fe5b9060005260206000209060040201905080600201601c9054906101000a900460ff1615611d38576040805185815290516001600160a01b038716917ff177e5d6c5764d79c32883ed824111d9b13f5668cf6ab1cc12dd36791dd955b4919081900360200190a2611da8565b600354611d4b908563ffffffff61310c16565b6003908155810154611d63908563ffffffff61310c16565b60038201556040805185815290516001600160a01b038716917f93a090ecc682c002995fad3c85b30c5651d7fd29b0be5da9d784a3302aedc055919081900360200190a25b50611dee565b6040805184815290516001600160a01b038616917ff177e5d6c5764d79c32883ed824111d9b13f5668cf6ab1cc12dd36791dd955b4919081900360200190a25b50505050565b61100081565b600381565b61dead81565b61100481565b611e13614159565b6000611e1d614159565b611e25614171565b611e36611e3186613166565b61318b565b90506000805b611e45836131d5565b15611f565780611e6a57611e60611e5b846131f6565b613244565b60ff168452611f4e565b8060011415611f49576060611e86611e81856131f6565b6132fb565b90508051604051908082528060200260200182016040528015611ec357816020015b611eb0614191565b815260200190600190039081611ea85790505b50602086015260005b8151811015611f3e57611edd614191565b6000611efb848481518110611eee57fe5b60200260200101516133cc565b9150915080611f1857876000995099505050505050505050611f5f565b8188602001518481518110611f2957fe5b60209081029190910101525050600101611ecc565b506001925050611f4e565b611f56565b600101611e3c565b50919350909150505b915091565b604080516001808252818301909252606091829190816020015b6060815260200190600190039081611f7e579050509050611fa48363ffffffff166134a9565b81600081518110611fb157fe5b6020026020010181905250610c93816134bc565b6000806060611fd384613546565b9150915081612080577f70e72399380dcfb0338abc03dc8d47f9f470ada8e769c9a78d644ea97385ecb2816040518080602001828103825283818151815260200191508051906020019080838360005b8381101561203b578181015183820152602001612023565b50505050905090810190601f1680156120685780820380516001836020036101000a031916815260200191505b509250505060405180910390a1606692505050611037565b600080805b6001548110156120fd5767016345785d8a0000600182815481106120a557fe5b906000526020600020906004020160030154106120c7576001909201916120f5565b6000600182815481106120d657fe5b90600052602060002090600402016003015411156120f5576001909101905b600101612085565b5060608260405190808252806020026020018201604052801561212a578160200160208202803683370190505b509050606083604051908082528060200260200182016040528015612159578160200160208202803683370190505b509050606084604051908082528060200260200182016040528015612188578160200160208202803683370190505b5090506060856040519080825280602002602001820160405280156121b7578160200160208202803683370190505b50905060006060866040519080825280602002602001820160405280156121e8578160200160208202803683370190505b509050606087604051908082528060200260200182016040528015612217578160200160208202803683370190505b509050600098506000975060608d905060006110046001600160a01b031663149d14d96040518163ffffffff1660e01b815260040160206040518083038186803b15801561226457600080fd5b505afa158015612278573d6000803e3d6000fd5b505050506040513d602081101561228e57600080fd5b5051905067016345785d8a0000811115612303577f70e72399380dcfb0338abc03dc8d47f9f470ada8e769c9a78d644ea97385ecb26040518080602001828103825260218152602001806142fb6021913960400191505060405180910390a160689d5050505050505050505050505050611037565b60005b6001548110156125765767016345785d8a00006001828154811061232657fe5b906000526020600020906004020160030154106124ac576001818154811061234a57fe5b906000526020600020906004020160020160009054906101000a90046001600160a01b03168a8d8151811061237b57fe5b60200260200101906001600160a01b031690816001600160a01b03168152505060006402540be400600183815481106123b057fe5b906000526020600020906004020160030154816123c957fe5b06600183815481106123d757fe5b9060005260206000209060040201600301540390506123ff83826130ca90919063ffffffff16565b8a8e8151811061240b57fe5b6020026020010181815250506001828154811061242457fe5b906000526020600020906004020160020160009054906101000a90046001600160a01b0316888e8151811061245557fe5b60200260200101906001600160a01b031690816001600160a01b03168152505081898e8151811061248257fe5b602090810291909101015261249d878263ffffffff61310c16565b6001909d019c965061256e9050565b6000600182815481106124bb57fe5b906000526020600020906004020160030154111561256e57600181815481106124e057fe5b906000526020600020906004020160010160009054906101000a90046001600160a01b0316858c8151811061251157fe5b60200260200101906001600160a01b031690816001600160a01b0316815250506001818154811061253e57fe5b906000526020600020906004020160030154848c8151811061255c57fe5b60209081029190910101526001909a01995b600101612306565b50600085156129b4576110046001600160a01b0316636e056520878c8c8b60025442016040518663ffffffff1660e01b815260040180806020018060200180602001856001600160401b03166001600160401b03168152602001848103845288818151815260200191508051906020019060200280838360005b838110156126085781810151838201526020016125f0565b50505050905001848103835287818151815260200191508051906020019060200280838360005b8381101561264757818101518382015260200161262f565b50505050905001848103825286818151815260200191508051906020019060200280838360005b8381101561268657818101518382015260200161266e565b505050509050019750505050505050506020604051808303818588803b1580156126af57600080fd5b505af1935050505080156126d557506040513d60208110156126d057600080fd5b505160015b612910576040516000815260443d10156126f15750600061278c565b60046000803e60005160e01c6308c379a0811461271257600091505061278c565b60043d036004833e81513d60248201116001600160401b038211171561273d5760009250505061278c565b80830180516001600160401b0381111561275e57600094505050505061278c565b8060208301013d860181111561277c5760009550505050505061278c565b601f01601f191660405250925050505b80612797575061283b565b60019150867fa7cdeed7d0db45e3219a6e5d60838824c16f1d39991fcfe3f963029c844bf280826040518080602001828103825283818151815260200191508051906020019080838360005b838110156127fb5781810151838201526020016127e3565b50505050905090810190601f1680156128285780820380516001836020036101000a031916815260200191505b509250505060405180910390a25061290b565b3d808015612865576040519150601f19603f3d011682016040523d82523d6000602084013e61286a565b606091505b5060019150867fbfa884552dd8921b6ce90bfe906952ae5b3b29be0cc1a951d4f62697635a3a45826040518080602001828103825283818151815260200191508051906020019080838360005b838110156128cf5781810151838201526020016128b7565b50505050905090810190601f1680156128fc5780820380516001836020036101000a031916815260200191505b509250505060405180910390a2505b6129b4565b801561294e576040805188815290517fa217d08e65f80c73121cd9db834d81652d544bfbf452f6d04922b16c90a37b709181900360200190a16129b2565b604080516020808252601b908201527f6261746368207472616e736665722072657475726e2066616c7365000000000081830152905188917fa7cdeed7d0db45e3219a6e5d60838824c16f1d39991fcfe3f963029c844bf280919081900360600190a25b505b8015612b6a5760005b8851811015612b685760008982815181106129d457fe5b602002602001015190506000600182815481106129ed57fe5b60009182526020909120600160049092020181015481546001600160a01b03909116916108fc9185908110612a1e57fe5b9060005260206000209060040201600301549081150290604051600060405180830381858888f1935050505090508015612ada5760018281548110612a5f57fe5b60009182526020909120600160049092020181015481546001600160a01b03909116917f6c61d60f69a7beb3e1c80db7f39f37b208537cbb19da3174511b477812b2fc7d9185908110612aae57fe5b9060005260206000209060040201600301546040518082815260200191505060405180910390a2612b5e565b60018281548110612ae757fe5b60009182526020909120600160049092020181015481546001600160a01b03909116917f25d0ce7d2f0cec669a8c17efe49d195c13455bb8872b65fa610ac7f53fe4ca7d9185908110612b3657fe5b9060005260206000209060040201600301546040518082815260200191505060405180910390a25b50506001016129bd565b505b845115612cb45760005b8551811015612cb2576000868281518110612b8b57fe5b60200260200101516001600160a01b03166108fc878481518110612bab57fe5b60200260200101519081150290604051600060405180830381858888f1935050505090508015612c4157868281518110612be157fe5b60200260200101516001600160a01b03167f6c61d60f69a7beb3e1c80db7f39f37b208537cbb19da3174511b477812b2fc7d878481518110612c1f57fe5b60200260200101516040518082815260200191505060405180910390a2612ca9565b868281518110612c4d57fe5b60200260200101516001600160a01b03167f25d0ce7d2f0cec669a8c17efe49d195c13455bb8872b65fa610ac7f53fe4ca7d878481518110612c8b57fe5b60200260200101516040518082815260200191505060405180910390a25b50600101612b74565b505b4715612d1d576040805147815290517f6ecc855f9440a9282c90913bbc91619fd44f5ec0b462af28d127b116f130aa4d9181900360200190a1604051611002904780156108fc02916000818181858888f19350505050158015612d1b573d6000803e3d6000fd5b505b60006003819055600555825115612d3757612d3783613628565b6110016001600160a01b031663fc4333cd6040518163ffffffff1660e01b8152600401600060405180830381600087803b158015612d7457600080fd5b505af1158015612d88573d6000803e3d6000fd5b50506040517fedd8d7296956dd970ab4de3f2fc03be2b0ffc615d20cd4c72c6e44f928630ebf925060009150a15060009f9e505050505050505050505050505050565b80516001600160a01b0316600090815260046020526040812054801580612e1c5750600180820381548110612dfc57fe5b9060005260206000209060040201600201601c9054906101000a900460ff165b15612e625782516040516001600160a01b03909116907fe209c46bebf57cf265d5d9009a00870e256d9150f3ed5281ab9d9eb3cec6e4be90600090a26000915050611037565b600154600554600019820111801590612eb85784516040516001600160a01b03909116907fe209c46bebf57cf265d5d9009a00870e256d9150f3ed5281ab9d9eb3cec6e4be90600090a260009350505050611037565b600580546001908101909155805481906000198601908110612ed657fe5b6000918252602082206002600490920201018054921515600160e01b0260ff60e01b199093169290921790915585516040516001600160a01b03909116917ff226e7d8f547ff903d9d419cf5f54e0d7d07efa9584135a53a057c5f1f27f49a91a2506000949350505050565b6000816040516020018082805190602001908083835b60208310612f775780518252601f199092019160209182019101612f58565b6001836020036101000a03801982511681845116808217855250505050505090500191505060405160208183030381529060405280519060200120836040516020018082805190602001908083835b60208310612fe55780518252601f199092019160209182019101612fc6565b6001836020036101000a038019825116818451168082178552505050505050905001915050604051602081830303815290604052805190602001201490505b92915050565b015190565b60008261303e57506000613024565b8282028284828161304b57fe5b0414610c935760405162461bcd60e51b81526004018080602001828103825260218152602001806142ac6021913960400191505060405180910390fd5b6000610c9383836040518060400160405280601a81526020017f536166654d6174683a206469766973696f6e206279207a65726f000000000000815250613ae9565b6000610c9383836040518060400160405280601e81526020017f536166654d6174683a207375627472616374696f6e206f766572666c6f770000815250613b8b565b600082820183811015610c93576040805162461bcd60e51b815260206004820152601b60248201527f536166654d6174683a206164646974696f6e206f766572666c6f770000000000604482015290519081900360640190fd5b61316e6141c6565b506040805180820190915281518152602082810190820152919050565b613193614171565b61319c82613be5565b6131a557600080fd5b60006131b48360200151613c1f565b60208085015160408051808201909152868152920190820152915050919050565b60006131df6141c6565b505080518051602091820151919092015191011190565b6131fe6141c6565b613207826131d5565b61321057600080fd5b6020820151600061322082613c82565b80830160209586015260408051808201909152908152938401919091525090919050565b80516000901580159061325957508151602110155b61326257600080fd5b60006132718360200151613c1f565b905080836000015110156132cc576040805162461bcd60e51b815260206004820152601a60248201527f6c656e677468206973206c657373207468616e206f6666736574000000000000604482015290519081900360640190fd5b8251602080850151830180519284900392918310156132f257826020036101000a820491505b50949350505050565b606061330682613be5565b61330f57600080fd5b600061331a83613db5565b905060608160405190808252806020026020018201604052801561335857816020015b6133456141c6565b81526020019060019003908161333d5790505b509050600061336a8560200151613c1f565b60208601510190506000805b848110156133c15761338783613c82565b91506040518060400160405280838152602001848152508482815181106133aa57fe5b602090810291909101015291810191600101613376565b509195945050505050565b6133d4614191565b60006133de614191565b6133e6614171565b6133ef8561318b565b90506000805b6133fe836131d5565b15611f56578061342957613419613414846131f6565b613e11565b6001600160a01b031684526134a1565b80600114156134515761343e613414846131f6565b6001600160a01b031660208501526134a1565b806002141561347957613466613414846131f6565b6001600160a01b031660408501526134a1565b8060031415611f495761348e611e5b846131f6565b6001600160401b03166060850152600191505b6001016133f5565b60606130246134b783613e2b565b613f11565b60608151600014156134dd5750604080516000815260208101909152611037565b6060826000815181106134ec57fe5b602002602001015190506000600190505b835181101561352d576135238285838151811061351657fe5b6020026020010151613f63565b91506001016134fd565b50610c93613540825160c060ff16613fe0565b82613f63565b600060606029835111156135785760006040518060600160405280602981526020016141e16029913991509150611f5f565b60005b835181101561360e5760005b818110156136055784818151811061359b57fe5b6020026020010151600001516001600160a01b03168583815181106135bc57fe5b6020026020010151600001516001600160a01b031614156135fd5760006040518060600160405280602b8152602001614235602b9139935093505050611f5f565b600101613587565b5060010161357b565b505060408051602081019091526000815260019150915091565b600154815160005b82811015613745576001613642614191565b6001838154811061364f57fe5b600091825260208083206040805160c08101825260049490940290910180546001600160a01b0390811685526001820154811693850193909352600281015492831691840191909152600160a01b82046001600160401b03166060840152600160e01b90910460ff16151560808301526003015460a082015291505b84811015613719578681815181106136df57fe5b6020026020010151600001516001600160a01b031682600001516001600160a01b031614156137115760009250613719565b6001016136cb565b50811561373b5780516001600160a01b03166000908152600460205260408120555b5050600101613630565b50808211156137ba57805b828110156137b857600180548061376357fe5b60008281526020812060046000199093019283020180546001600160a01b03199081168255600182810180549092169091556002820180546001600160e81b0319169055600390910191909155915501613750565b505b60008183106137c957816137cb565b825b905060005b818110156139c55761387d8582815181106137e757fe5b6020026020010151600183815481106137fc57fe5b60009182526020918290206040805160c08101825260049390930290910180546001600160a01b0390811684526001820154811694840194909452600281015493841691830191909152600160a01b83046001600160401b03166060830152600160e01b90920460ff161515608082015260039091015460a08201526140d8565b61399857806001016004600087848151811061389557fe5b6020026020010151600001516001600160a01b03166001600160a01b03168152602001908152602001600020819055508481815181106138d157fe5b6020026020010151600182815481106138e657fe5b6000918252602091829020835160049092020180546001600160a01b039283166001600160a01b0319918216178255928401516001820180549184169185169190911790556040840151600282018054606087015160808801511515600160e01b0260ff60e01b196001600160401b03909216600160a01b0267ffffffffffffffff60a01b1995909716929097169190911792909216939093171692909217905560a0909101516003909101556139bd565b6000600182815481106139a757fe5b9060005260206000209060040201600301819055505b6001016137d0565b5082821115611dee57825b82811015610fba5760018582815181106139e657fe5b60209081029190910181015182546001818101855560009485528385208351600493840290910180546001600160a01b039283166001600160a01b03199182161782559585015181840180549184169188169190911790556040850151600282018054606088015160808901511515600160e01b0260ff60e01b196001600160401b03909216600160a01b0267ffffffffffffffff60a01b199590971692909a169190911792909216939093171695909517905560a09092015160039093019290925587519084019290889085908110613abc57fe5b602090810291909101810151516001600160a01b03168252810191909152604001600020556001016139d0565b60008183613b755760405162461bcd60e51b81526004018080602001828103825283818151815260200191508051906020019080838360005b83811015613b3a578181015183820152602001613b22565b50505050905090810190601f168015613b675780820380516001836020036101000a031916815260200191505b509250505060405180910390fd5b506000838581613b8157fe5b0495945050505050565b60008184841115613bdd5760405162461bcd60e51b8152602060048201818152835160248401528351909283926044909101919085019080838360008315613b3a578181015183820152602001613b22565b505050900390565b8051600090613bf657506000611037565b6020820151805160001a9060c0821015613c1557600092505050611037565b5060019392505050565b8051600090811a6080811015613c39576000915050611037565b60b8811080613c54575060c08110801590613c54575060f881105b15613c63576001915050611037565b60c0811015613c775760b519019050611037565b60f519019050611037565b80516000908190811a6080811015613c9d5760019150613dae565b60b8811015613cb257607e1981019150613dae565b60c0811015613d2c57600060b78203600186019550806020036101000a865104915060018101820193505080831015613d26576040805162461bcd60e51b81526020600482015260116024820152706164646974