/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// cl_serverlist.c  -- interactuates with the master server

#include "client.h"
#include "../qcommon/trie.h"

//=========================================================

typedef struct serverlist_s
{
	char address[48];
	unsigned int pingTimeStamp;
	unsigned int lastValidPing;
	struct serverlist_s *pnext;
} serverlist_t;

typedef struct masteradrcache_s
{
	netadr_t adr;
	struct masteradrcache_s *next;
} masteradrcache_t;

serverlist_t *masterList, *favoritesList;

// cache of resolved master server addresses/names
static trie_t *serverlist_masters_trie = NULL;
static masteradrcache_t *serverlist_masters_head = NULL;

static qboolean filter_allow_full = qfalse;
static qboolean filter_allow_empty = qfalse;

//=========================================================

/*
* CL_FreeServerlist
*/
static void CL_FreeServerlist( serverlist_t **serversList )
{
	serverlist_t *ptr;

	while( *serversList )
	{
		ptr = *serversList;
		*serversList = ptr->pnext;
		Mem_ZoneFree( ptr );
	}
}

/*
* CL_ServerIsInList
*/
static serverlist_t *CL_ServerFindInList( serverlist_t *serversList, char *adr )
{

	serverlist_t *server;

	server = serversList;
	while( server )
	{
		if( !Q_stricmp( server->address, adr ) )
			return server;
		server = server->pnext;
	}

	return NULL;
}

/*
* CL_AddServerToList
*/
static qboolean CL_AddServerToList( serverlist_t **serversList, char *adr, unsigned int days )
{
	serverlist_t *newserv;
	netadr_t nadr;

	if( !adr || !strlen( adr ) )
		return qfalse;

	if( !NET_StringToAddress( adr, &nadr ) )
		return qfalse;

	if( CL_ServerFindInList( *serversList, adr ) )
		return qfalse;

	newserv = (serverlist_t *)Mem_ZoneMalloc( sizeof( serverlist_t ) );
	Q_strncpyz( newserv->address, adr, sizeof( newserv->address ) );
	newserv->pingTimeStamp = 0;
	if( days == 0 )
		newserv->lastValidPing = Com_DaysSince1900();
	else
		newserv->lastValidPing = days;
	newserv->pnext = *serversList;
	*serversList = newserv;

	return qtrue;
}

#define SERVERSFILE "serverscache.txt"
/*
* CL_WriteServerCache
*/
void CL_WriteServerCache( void )
{
	serverlist_t *server;
	int filehandle;
	char str[256];
	netadr_t adr;

	if( FS_FOpenFile( SERVERSFILE, &filehandle, FS_WRITE ) == -1 )
	{
		Com_Printf( "CL_WriteServerList: Couldn't create the cache file\n" );
		return;
	}

	Q_snprintfz( str, sizeof( str ), "// servers cache file generated by %s. Do not modify\n", APPLICATION );
	FS_Print( filehandle, str );

	FS_Print( filehandle, "master\n" );
	server = masterList;
	while( server )
	{
		if( server->lastValidPing + 7 > Com_DaysSince1900() )
		{
			if( NET_StringToAddress( server->address, &adr ) )
			{
				Q_snprintfz( str, sizeof( str ), "%s %i\n", server->address, (int)server->lastValidPing );
				FS_Print( filehandle, str );
			}
		}

		server = server->pnext;
	}

	FS_Print( filehandle, "favorites\n" );
	server = favoritesList;
	while( server )
	{
		if( server->lastValidPing + 7 > Com_DaysSince1900() )
		{
			if( NET_StringToAddress( server->address, &adr ) )
			{
				Q_snprintfz( str, sizeof( str ), "%s %i\n", server->address, (int)server->lastValidPing );
				FS_Print( filehandle, str );
			}
		}

		server = server->pnext;
	}

	FS_FCloseFile( filehandle );
}

/*
* CL_ReadServerCache
*/
void CL_ReadServerCache( void )
{
	int filelen, filehandle;
	qbyte *buf = NULL;
	char *ptr, *token;
	netadr_t adr;
	char adrString[64];
	qboolean favorite = qfalse;

	filelen = FS_FOpenFile( SERVERSFILE, &filehandle, FS_READ );
	if( !filehandle || filelen < 1 )
	{
		FS_FCloseFile( filehandle );
	}
	else
	{
		buf = Mem_TempMalloc( filelen + 1 );
		filelen = FS_Read( buf, filelen, filehandle );
		FS_FCloseFile( filehandle );
	}

	if( !buf )
		return;

	ptr = ( char * )buf;
	while( ptr )
	{
		token = COM_ParseExt( &ptr, qtrue );
		if( !token[0] )
			break;

		if( !Q_stricmp( token, "master" ) )
		{
			favorite = qfalse;
			continue;
		}

		if( !Q_stricmp( token, "favorites" ) )
		{
			favorite = qtrue;
			continue;
		}

		if( NET_StringToAddress( token, &adr ) )
		{
			Q_strncpyz( adrString, token, sizeof( adrString ) );
			token = COM_ParseExt( &ptr, qfalse );
			if( !token[0] )
				continue;

			if( favorite )
				CL_AddServerToList( &favoritesList, adrString, (unsigned int)atoi( token ) );
			else
				CL_AddServerToList( &masterList, adrString, (unsigned int)atoi( token ) );
		}
	}

	Mem_TempFree( buf );
}

/*
* CL_ParseGetInfoResponse
*
* Handle a server responding to a detailed info broadcast
*/
void CL_ParseGetInfoResponse( const socket_t *socket, const netadr_t *address, msg_t *msg )
{
	char *s = MSG_ReadString( msg );
	Com_DPrintf( "%s\n", s );
}

/*
* CL_ParseGetStatusResponse
*
* Handle a server responding to a detailed info broadcast
*/
void CL_ParseGetStatusResponse( const socket_t *socket, const netadr_t *address, msg_t *msg )
{
	char *s = MSG_ReadString( msg );
	Com_DPrintf( "%s\n", s );
}


/*
* CL_QueryGetInfoMessage
*/
static void CL_QueryGetInfoMessage( const char *cmdname )
{
	netadr_t adr;
	char *requeststring;
	char *server;

	//get what master
	server = Cmd_Argv( 1 );
	if( !server || !( *server ) )
	{
		Com_Printf( "%s: no address provided %s...\n", Cmd_Argv( 0 ), server ? server : "" );
		return;
	}

	requeststring = va( cmdname );

	// send a broadcast packet
	Com_DPrintf( "quering %s...\n", server );

	if( NET_StringToAddress( server, &adr ) )
	{
		socket_t *socket;

		if( NET_GetAddressPort( &adr ) == 0 )
			NET_SetAddressPort( &adr, PORT_SERVER );

		socket = ( adr.type == NA_IP6 ? &cls.socket_udp6 : &cls.socket_udp );
		Netchan_OutOfBandPrint( socket, &adr, requeststring );
	}
	else
	{
		Com_Printf( "Bad address: %s\n", server );
	}
}


/*
* CL_QueryGetInfoMessage_f - getinfo 83.97.146.17:27911
*/
void CL_QueryGetInfoMessage_f( void )
{
	CL_QueryGetInfoMessage( "getinfo" );
}


/*
* CL_QueryGetStatusMessage_f - getstatus 83.97.146.17:27911
*/
void CL_QueryGetStatusMessage_f( void )
{
	CL_QueryGetInfoMessage( "getstatus" );
}

/*
* CL_PingServer_f
*/
void CL_PingServer_f( void )
{
	char *address_string;
	char requestString[64];
	netadr_t adr;
	serverlist_t *pingserver;
	socket_t *socket;

	if( Cmd_Argc() < 2 )
		Com_Printf( "Usage: pingserver [ip:port]\n" );

	address_string = Cmd_Argv( 1 );

	if( !NET_StringToAddress( address_string, &adr ) )
		return;

	pingserver = CL_ServerFindInList( masterList, address_string );
	if( !pingserver )
		pingserver = CL_ServerFindInList( favoritesList, address_string );
	if( !pingserver )
		return;

	// never request a second ping while awaiting for a ping reply
	if( pingserver->pingTimeStamp + SERVER_PINGING_TIMEOUT > cls.realtime )
		return;

	pingserver->pingTimeStamp = cls.realtime;

	Q_snprintfz( requestString, sizeof( requestString ), "info %i %s %s", APP_PROTOCOL_VERSION,
		filter_allow_full ? "full" : "",
		filter_allow_empty ? "empty" : "" );

	socket = ( adr.type == NA_IP6 ? &cls.socket_udp6 : &cls.socket_udp );
	Netchan_OutOfBandPrint( socket, &adr, requestString );
}

/*
* CL_ParseStatusMessage
* Handle a reply from a ping
*/
void CL_ParseStatusMessage( const socket_t *socket, const netadr_t *address, msg_t *msg )
{
	char *s = MSG_ReadString( msg );
	serverlist_t *pingserver;
	char adrString[64];

	Com_DPrintf( "%s\n", s );

	Q_strncpyz( adrString, NET_AddressToString( address ), sizeof( adrString ) );

	// ping response
	pingserver = CL_ServerFindInList( masterList, adrString );
	if( !pingserver )
		pingserver = CL_ServerFindInList( favoritesList, adrString );

	if( pingserver && pingserver->pingTimeStamp ) // valid ping
	{
		unsigned int ping = cls.realtime - pingserver->pingTimeStamp;
		CL_UIModule_AddToServerList( adrString, va( "\\\\ping\\\\%i%s", ping, s ) );
		pingserver->pingTimeStamp = 0;
		pingserver->lastValidPing = Com_DaysSince1900();
		return;
	}

	// add the server info, but ignore the ping, cause it's not valid
	CL_UIModule_AddToServerList( adrString, s );
}

/*
* CL_ParseGetServersResponseMessage
* Handle a reply from getservers message to master server
*/
static void CL_ParseGetServersResponseMessage( msg_t *msg, qboolean extended )
{
	const char *header;
	char adrString[64];
	qbyte addr[16];
	unsigned short port;
	netadr_t adr;

	MSG_BeginReading( msg );
	MSG_ReadLong( msg ); // skip the -1

	//jump over the command name
	header = ( extended ? "getserversExtResponse" : "getserversResponse" );
	if( !MSG_SkipData( msg, strlen( header ) ) )
	{
		Com_Printf( "Invalid master packet ( missing %s )\n", header );
		return;
	}

	while( msg->readcount + 7 <= msg->cursize )
	{
		char prefix = MSG_ReadChar( msg );

		switch( prefix )
		{
		case '\\':
			MSG_ReadData( msg, addr, 4 );
			port = ShortSwap( MSG_ReadShort( msg ) ); // both endians need this swapped.
			Q_snprintfz( adrString, sizeof( adrString ), "%u.%u.%u.%u:%u", addr[0], addr[1], addr[2], addr[3], port );
			break;

		case '/':
			if( extended )
			{
				MSG_ReadData( msg, addr, 16 );
				port = ShortSwap( MSG_ReadShort( msg ) ); // both endians need this swapped.
				Q_snprintfz( adrString, sizeof( adrString ), "[%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x]:%hu",
								addr[ 0], addr[ 1], addr[ 2], addr[ 3], addr[ 4], addr[ 5], addr[ 6], addr[ 7],
								addr[ 8], addr[ 9], addr[10], addr[11], addr[12], addr[13], addr[14], addr[15],
								port );
			}
			else
			{
				Com_Printf( "Invalid master packet ( IPv6 prefix in a non-extended response )\n" );
				return;
			}

			break;

		default:
			Com_Printf( "Invalid master packet ( missing separator )\n" );
			return;
		}

		if( port == 0 )  // last server seen
			return;

		Com_DPrintf( "%s\n", adrString );
		if( !NET_StringToAddress( adrString, &adr ) )
		{
			Com_Printf( "Bad address: %s\n", adrString );
			continue;
		}

		CL_AddServerToList( &masterList, adrString, 0 );
	}
}

/*
* CL_ParseGetServersResponse
* Handle a reply from getservers message to master server
*/
void CL_ParseGetServersResponse( const socket_t *socket, const netadr_t *address, msg_t *msg, qboolean extended )
{
	serverlist_t *server;
	netadr_t adr;

	CL_ReadServerCache();

	// add the new server addresses to the local addresses list
	CL_ParseGetServersResponseMessage( msg, extended );

	CL_WriteServerCache();

	// dump the whole list to the ui
	server = masterList;
	while( server )
	{
		if( NET_StringToAddress( server->address, &adr ) )
			CL_UIModule_AddToServerList( server->address, "\\\\EOT" );

		server = server->pnext;
	}
}

/*
//jal: I will remove this function soon
//jal: the browser bug was the response string being parsed by
//jal: using MSG_ParseStringLine instead of using MSG_ParseString.
//jal: MSG_ParseStringLine cut off the string when it found
//jal: a line ending, a zero or a -1.
void CL_ParseGetServersResponse( const socket_t *socket, const netadr_t *address, msg_t *msg )
{
	CL_FreeServerlist( &masterList );
	CL_ParseGetServersResponseMessage( msg );
	//	CL_LoadServerList(); //jal: tmp
	//	CL_WriteServerList();
#if 1
	//send the servers to the ui
	{
		serverlist_t *server;
		netadr_t adr;

		server = masterList;
		while( server )
		{
			if( NET_StringToAddress( server->address, &adr ) )
			{
				CL_UIModule_AddToServerList( server->address, "\\\\EOT" );
			}
			server = server->pnext;
		}
	}
#else
	{
		serverlist_t *server;
		char requestString[32];
		netadr_t adr;

		Q_snprintfz( requestString, sizeof( requestString ), "info %i %s %s", APP_PROTOCOL_VERSION,
			filter_allow_full ? "full" : "",
			filter_allow_empty ? "empty" : "" );

		server = masterList;
		while( server )
		{
			if( NET_StringToAddress( server->address, &adr ) )
				Netchan_OutOfBandPrint( &cls.socket_udp, &adr, requestString );
			server = server->pnext;
		}
	}
#endif
	CL_FreeServerlist( &masterList );
}
*/

/*
* CL_ResolveMasterAddress
*/
static void CL_ResolveMasterAddress( const char *master, netadr_t *adr )
{
	trie_error_t err;
	masteradrcache_t *cache;

	// check memory cache
	err = Trie_Find( serverlist_masters_trie, master, TRIE_EXACT_MATCH, (void **)&cache );
	if( err == TRIE_OK )
	{
		if( adr )
			*adr = cache->adr;
		return;
	}

	// send a unicast packet
	cache = Mem_ZoneMalloc( sizeof( *cache ) );
	NET_StringToAddress( master, &cache->adr );

	if( adr )
		*adr = cache->adr;
	cache->next = serverlist_masters_head;
	serverlist_masters_head = cache;

	Trie_Insert( serverlist_masters_trie, master, (void *)cache );
}

/*
* CL_MasterAddressCache_Init
*/
static void CL_MasterAddressCache_Init( void )
{
	char *master;
	const char *mlist;

	serverlist_masters_head = NULL;
	Trie_Create( TRIE_CASE_INSENSITIVE, &serverlist_masters_trie );

	// synchronous DNS queries cause interruption of sounds, background music, etc
	// so precache results of resolution queries for all master servers
	mlist = Cvar_String( "masterservers" );
	while( mlist )
	{
		master = COM_Parse( &mlist );
		if( !*master )
			break;
		CL_ResolveMasterAddress( master, NULL );
	}
}

/*
* CL_MasterAddressCache_Shutdown
*/
static void CL_MasterAddressCache_Shutdown( void )
{
	masteradrcache_t *next;

	// free allocated memory
	Trie_Destroy( serverlist_masters_trie );
	while( serverlist_masters_head )
	{
		next = serverlist_masters_head->next;
		Mem_ZoneFree( serverlist_masters_head );
		serverlist_masters_head = next;
	}
}

/*
* CL_GetServers_f
*/
void CL_GetServers_f( void )
{
	netadr_t adr;
	char *requeststring;
	int i;
	char *modname, *master;

	filter_allow_full = qfalse;
	filter_allow_empty = qfalse;
	for( i = 0; i < Cmd_Argc(); i++ )
	{
		if( !Q_stricmp( "full", Cmd_Argv( i ) ) )
			filter_allow_full = qtrue;

		if( !Q_stricmp( "empty", Cmd_Argv( i ) ) )
			filter_allow_empty = qtrue;
	}

	if( !Q_stricmp( Cmd_Argv( 1 ), "local" ) )
	{
		// send a broadcast packet
		Com_Printf( "pinging broadcast...\n" );

		// erm... modname isn't sent in local queries?

		requeststring = va( "info %i %s %s", APP_PROTOCOL_VERSION,
			filter_allow_full ? "full" : "",
			filter_allow_empty ? "empty" : "" );

		for( i = 0; i < NUM_BROADCAST_PORTS; i++ )
		{
			NET_BroadcastAddress( &adr, PORT_SERVER + i );
			Netchan_OutOfBandPrint( &cls.socket_udp, &adr, requeststring );
		}
		return;
	}

	//get what master
	master = Cmd_Argv( 2 );
	if( !master || !( *master ) )
		return;

	modname = Cmd_Argv( 3 );
	// never allow anyone to use DEFAULT_BASEGAME as mod name
	if( !modname || !modname[0] || !Q_stricmp( modname, DEFAULT_BASEGAME ) )
		modname = APPLICATION;

	assert( modname[0] );

	// fetch from cache or query DNS server
	CL_ResolveMasterAddress( master, &adr );

	if( adr.type == NA_IP || adr.type == NA_IP6 )
	{
		const char *cmdname;
		socket_t *socket;

		if ( adr.type == NA_IP )
		{
			cmdname = "getservers";
			socket = &cls.socket_udp;
		}
		else
		{
			cmdname = "getserversExt";
			socket = &cls.socket_udp6;
		}

		// create the message
		requeststring = va( "%s %c%s %i %s %s", cmdname, toupper( modname[0] ), modname+1, APP_PROTOCOL_VERSION,
			filter_allow_full ? "full" : "",
			filter_allow_empty ? "empty" : "" );

		if( NET_GetAddressPort( &adr ) == 0 )
			NET_SetAddressPort( &adr, PORT_MASTER );

		Netchan_OutOfBandPrint( socket, &adr, requeststring );

		Com_DPrintf( "quering %s...%s: %s\n", master, NET_AddressToString( &adr), requeststring );
	}
	else
	{
		Com_Printf( "Bad address: %s\n", master );
	}
}

/*
* CL_InitServerList
*/
void CL_InitServerList( void )
{
	CL_FreeServerlist( &masterList );
	CL_FreeServerlist( &favoritesList );

	CL_ReadServerCache();

	CL_MasterAddressCache_Init();
}

/*
* CL_ShutDownServerList
*/
void CL_ShutDownServerList( void )
{
	CL_WriteServerCache();

	CL_FreeServerlist( &masterList );
	CL_FreeServerlist( &favoritesList );

	CL_MasterAddressCache_Shutdown();
}
