package rearth.oritech.block.entity.generators;

import dev.architectury.fluid.FluidStack;
import dev.architectury.platform.Platform;
import org.jetbrains.annotations.Nullable;
import rearth.oritech.Oritech;
import rearth.oritech.api.fluid.FluidApi;
import rearth.oritech.api.fluid.FluidApi.SingleSlotStorage;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.block.base.entity.FluidMultiblockGeneratorBlockEntity;
import rearth.oritech.block.base.entity.MultiblockGeneratorBlockEntity;
import rearth.oritech.client.init.ModScreens;
import rearth.oritech.client.init.ParticleContent;
import rearth.oritech.init.BlockEntitiesContent;
import rearth.oritech.init.FluidContent;
import rearth.oritech.init.recipes.OritechRecipe;
import rearth.oritech.init.recipes.OritechRecipeType;
import rearth.oritech.init.recipes.RecipeContent;
import rearth.oritech.util.Geometry;
import rearth.oritech.util.InventorySlotAssignment;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.HolderLookup;
import net.minecraft.core.Vec3i;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.Tuple;
import net.minecraft.world.inventory.MenuType;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.material.Fluid;
import net.minecraft.world.level.material.Fluids;
import net.minecraft.world.phys.Vec3;

// progress is abused to sync active speed.
public class SteamEngineEntity extends MultiblockGeneratorBlockEntity implements FluidApi.BlockProvider {
    
    private static final int MAX_SPEED = 10;
    private static final int MAX_CHAIN_SIZE = 20;
    private static final float WATER_RATIO = 0.9f;
    
    // use lower amounts of steam on neo, because on fabric a bucket is 81000 droplets, and on neo its only 1000
    // due to this, the pipe capacities would be reached super fast.
    // So on neo, we reduce the amount of steam/water used/produced by X, and then increase the energy amount by X again
    public static float STEAM_AMOUNT_MULTIPLIER = Platform.isNeoForge() ? 0.05f : 4f;
    
    
    // how chaining works:
    // (non-chained non-empty) generator checks neighbors in both sides in N dist
    // generator marks neighbors as chained, with timestamp
    // slaved generator shows only chain notice in popup, moves all inserted steam to master (get api returns master entries)
    // master processes at X rate, shows chained count in UI
    
    // progress is used to store/sync animation speed
    
    private static Fluid USED_STEAM_FLUID;
    
    public long masterHeartbeat; // set from master, used by slave
    public SteamEngineEntity master;
    
    private final Set<SteamEngineEntity> slaves = new HashSet<>();
    
    // client only
    @SyncField({SyncType.GUI_TICK, SyncType.GUI_OPEN})
    public SteamEngineSyncPacket clientStats;
    
    public SteamEngineEntity(BlockPos pos, BlockState state) {
        super(BlockEntitiesContent.STEAM_ENGINE_ENTITY, pos, state, Oritech.CONFIG.generators.steamEngineData.steamToRfRatio());
        clientStats = new SteamEngineSyncPacket(pos, 1f, 1f, 0, 0, 0);
    }
    
    @Override
    public void serverTick(Level world, BlockPos pos, BlockState state, NetworkedBlockEntity blockEntity) {
        
        if (world.isClientSide || !isActive(state)) return;
        
        var slaved = inSlaveMode();
        var hasInput = !boilerStorage.getInStack().isEmpty();
        if (world.getGameTime() % 80 == 0 && !slaved && hasInput)
            setupMaster();
        
        if (!slaved && hasInput) tickMaster();
        
        if (slaved) tickSlave();
        
        
        outputEnergy();
    }
    
    // this is only called when steam is available
    private void tickMaster() {
        
        var steamTank = boilerStorage.getInputContainer();
        var waterTank = boilerStorage.getOutputContainer();
        
        // optional config stops (energy full / water full)
        if (energyStorage.getAmount() >= energyStorage.getCapacity() && Oritech.CONFIG.generators.steamEngineData.stopOnEnergyFull())
            return;
        if (waterTank.getStack().getAmount() >= waterTank.getCapacity() && Oritech.CONFIG.generators.steamEngineData.stopOnWaterFull())
            return;
        
        // if not recipe is currently set, or it does not match the steam tank, search for a recipe
        if (currentRecipe == OritechRecipe.DUMMY || !currentRecipe.getFluidInput().matchesFluid(steamTank.getStack())) {
            var candidate = FluidMultiblockGeneratorBlockEntity.getRecipe(steamTank, level, getOwnRecipeType());
            candidate.ifPresent(recipe -> currentRecipe = recipe.value());
            if (candidate.isEmpty()) return;
            currentRecipe = candidate.get().value();
        }
        
        var speed = getSteamProcessingSpeed(steamTank);
        var workerCount = slaves.size() + 1;
        
        var consumedCount = currentRecipe.getFluidInput().amount() * speed * workerCount * STEAM_AMOUNT_MULTIPLIER;
        var producedCount = consumedCount * WATER_RATIO;
        
        consumedCount = Math.max(consumedCount, 1);
        producedCount = Math.max(producedCount, 1);
        
        // update tanks
        steamTank.extract(steamTank.getStack().copyWithAmount((long) consumedCount), false);
        waterTank.insert(FluidStack.create(Fluids.WATER, (long) producedCount), false);
        
        // produce energy
        var energyEfficiency = getSteamEnergyEfficiency(speed);
        var energyProduced = consumedCount * energyEfficiency * energyPerTick / STEAM_AMOUNT_MULTIPLIER;
        energyStorage.insertIgnoringLimit((long) energyProduced, false);
        
        spawnParticles();
        lastWorkedAt = level.getGameTime();
        
        // used for animation speed
        progress = (int) (speed * 100f);
        
        // order/data: speed, efficiency, rf produced, steam consumed, slave count
        clientStats = new SteamEngineSyncPacket(worldPosition, speed, energyEfficiency, (long) energyProduced, (long) (consumedCount / STEAM_AMOUNT_MULTIPLIER), slaves.size());
        this.setChanged();
        
    }
    
    private void tickSlave() {
        // check if master is actually working
        var masterStats = master.clientStats;
        if (masterStats == null) return;
        var wasWorking = master.isActivelyWorking();
        var speed = masterStats.speed();
        
        if (wasWorking) {
            spawnParticles();
            this.lastWorkedAt = level.getGameTime();
            this.setChanged();
        }
        
        // used for animation speed
        progress = (int) (speed * 100f);
    }
    
    private void setupMaster() {
        slaves.clear();
        
        for (int direction = -1; direction <= 1; direction++) {
            if (direction == 0) continue;
            for (int i = 1; i <= MAX_CHAIN_SIZE; i++) {
                var checkPos = new BlockPos(Geometry.offsetToWorldPosition(getFacing(), new Vec3i(i * direction, 0, 0), worldPosition));
                
                var coreCandidate = level.getBlockEntity(checkPos, BlockEntitiesContent.MACHINE_CORE_ENTITY);
                if (coreCandidate.isPresent() && coreCandidate.get().getCachedController() != null)
                    checkPos = coreCandidate.get().getControllerPos();
                
                var candidate = level.getBlockEntity(checkPos, BlockEntitiesContent.STEAM_ENGINE_ENTITY);
                if (candidate.isEmpty() || !candidate.get().isActive(candidate.get().getBlockState())) {
                    break;
                } else if (!candidate.get().boilerStorage.getInStack().isEmpty()) {
                    break;
                } else {
                    var slave = candidate.get();
                    slaves.add(slave);
                    slave.masterHeartbeat = level.getGameTime();
                    slave.master = this;
                }
            }
        }
    }
    
    public boolean inSlaveMode() {
        var heartbeatAge = level.getGameTime() - masterHeartbeat;
        return heartbeatAge <= 100 && master != null && !master.isRemoved();
    }
    
    @Override
    public boolean boilerAcceptsInput(Fluid fluid) {
        return fluid.equals(getUsedSteamFluid());
    }
    
    private void spawnParticles() {
        if (level.random.nextFloat() > 0.5) return;
        // emit particles
        var facing = getFacing();
        var offsetLocal = Geometry.rotatePosition(new Vec3(0, 0, -0.5), facing);
        var emitPosition = Vec3.atCenterOf(worldPosition).add(offsetLocal);
        
        ParticleContent.STEAM_ENGINE_WORKING.spawn(level, emitPosition, 1);
    }
    
    private float getSteamEnergyEfficiency(float x) {
        // basically a curve that goes through 0:0.5, 7:1 and 10:0.2
        return (float) (0.5f - 0.1966667f * x + 0.09166667f * Math.pow(x, 2) - 0.0075f * Math.pow(x, 3)) + 0.4f;
    }
    
    @Override
    protected void loadAdditional(CompoundTag nbt, HolderLookup.Provider registryLookup) {
        super.loadAdditional(nbt, registryLookup);
    }
    
    private float getSteamProcessingSpeed(FluidApi.SingleSlotStorage usedTank) {
        var fillPercentage = usedTank.getStack().getAmount() / (float) usedTank.getCapacity();
        return fillPercentage * MAX_SPEED;
    }
    
    @Override
    protected float getAnimationSpeed() {
        if (progress == 0) return 1;
        return (float) progress / 100f;
    }
    
    @Override
    public BarConfiguration getFluidConfiguration() {
        return new BarConfiguration(149, 10, 18, 64);
    }
    
    @Override
    protected OritechRecipeType getOwnRecipeType() {
        return RecipeContent.STEAM_ENGINE;
    }
    
    @Override
    public InventorySlotAssignment getSlotAssignments() {
        return new InventorySlotAssignment(0, 0, 0, 0);
    }
    
    @Override
    public List<GuiSlot> getGuiSlots() {
        return List.of();
    }
    
    @Override
    public MenuType<?> getScreenHandlerType() {
        return ModScreens.STEAM_ENGINE_SCREEN;
    }
    
    @Override
    public int getInventorySize() {
        return 0;
    }
    
    @Override
    public long getDefaultCapacity() {
        return Oritech.CONFIG.generators.steamEngineData.energyCapacity();
    }
    
    @Override
    public long getDefaultExtractionRate() {
        return Oritech.CONFIG.generators.steamEngineData.maxEnergyExtraction();
    }
    
    @Override
    protected Set<Tuple<BlockPos, Direction>> getOutputTargets(BlockPos pos, Level world) {
        
        var res = new HashSet<Tuple<BlockPos, Direction>>();
        
        var facing = getFacingForAddon();
        var posA = new Vec3i(0, 0, 1); // front
        var posB = new Vec3i(-1, 0, 0); // right
        var posC = new Vec3i(1, 0, 0);  // left
        var posD = new Vec3i(-1, 0, -1); // back left
        var posE = new Vec3i(1, 0, -1); // back right
        var posF = new Vec3i(0, 0, -2);  // back
        var worldPosA = (BlockPos) Geometry.offsetToWorldPosition(facing, posA, pos);
        var worldPosB = (BlockPos) Geometry.offsetToWorldPosition(facing, posB, pos);
        var worldPosC = (BlockPos) Geometry.offsetToWorldPosition(facing, posC, pos);
        var worldPosD = (BlockPos) Geometry.offsetToWorldPosition(facing, posD, pos);
        var worldPosE = (BlockPos) Geometry.offsetToWorldPosition(facing, posE, pos);
        var worldPosF = (BlockPos) Geometry.offsetToWorldPosition(facing, posF, pos);
        
        res.add(new Tuple<>(worldPosA, Geometry.fromVector(Geometry.getForward(facing))));
        res.add(new Tuple<>(worldPosB, Geometry.fromVector(Geometry.getLeft(facing))));
        res.add(new Tuple<>(worldPosC, Geometry.fromVector(Geometry.getRight(facing))));
        res.add(new Tuple<>(worldPosD, Geometry.fromVector(Geometry.getLeft(facing))));
        res.add(new Tuple<>(worldPosE, Geometry.fromVector(Geometry.getRight(facing))));
        res.add(new Tuple<>(worldPosF, Geometry.fromVector(Geometry.getBackward(facing))));
        
        return res;
        
    }
    
    @Override
    public List<Vec3i> getAddonSlots() {
        return List.of();
    }
    
    @Override
    public List<Vec3i> getCorePositions() {
        return List.of(
          new Vec3i(0, 1, 0),
          new Vec3i(0, 0, -1),
          new Vec3i(0, 1, -1)
        );
    }
    
    @Override
    public boolean showProgress() {
        return false;
    }
    
    @Override
    public FluidApi.FluidStorage getFluidStorage(@Nullable Direction direction) {
        if (inSlaveMode()) return master.boilerStorage;
        return boilerStorage;
    }
    
    public record SteamEngineSyncPacket(BlockPos position, float speed, float efficiency, long energyProduced,
                                        long steamConsumed, int slaves) {
    }
    
    public static Fluid getUsedSteamFluid() {
        if (USED_STEAM_FLUID == null) {
            USED_STEAM_FLUID = BuiltInRegistries.FLUID.get(ResourceLocation.parse(Oritech.CONFIG.generators.steamId()));
        }
        
        return USED_STEAM_FLUID;
    }
}
