// Copyright Project Contour Authors
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Package certgen contains the code that handles the `certgen` subcommand
// for the main `contour` binary.
package certgen

import (
	"context"
	"fmt"
	"path"

	core_v1 "k8s.io/api/core/v1"
	k8serrors "k8s.io/apimachinery/pkg/api/errors"
	meta_v1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/util/validation"
	"k8s.io/client-go/kubernetes"

	"github.com/projectcontour/contour/internal/dag"
	"github.com/projectcontour/contour/pkg/certs"
)

const (
	// CACertificateKey is the dictionary key for the CA certificate bundle.
	CACertificateKey = "cacert.pem"
	// ContourCertificateKey is the dictionary key for the Contour certificate.
	ContourCertificateKey = "contourcert.pem"
	// ContourPrivateKeyKey is the dictionary key for the Contour private key.
	ContourPrivateKeyKey = "contourkey.pem"
	// EnvoyCertificateKey is the dictionary key for the Envoy certificate.
	EnvoyCertificateKey = "envoycert.pem"
	// EnvoyPrivateKeyKey is the dictionary key for the Envoy private key.
	EnvoyPrivateKeyKey = "envoykey.pem"
)

// OverwritePolicy specifies whether an output should be overwritten.
type OverwritePolicy int

const (
	// NoOverwrite specifies outputs must not be overwritten.
	NoOverwrite OverwritePolicy = 0
	// Overwrite specifies outputs may be overwritten.
	Overwrite OverwritePolicy = 1
)

func newSecret(secretType core_v1.SecretType, name, namespace string, data map[string][]byte) *core_v1.Secret {
	return &core_v1.Secret{
		Type: secretType,
		TypeMeta: meta_v1.TypeMeta{
			Kind:       "Secret",
			APIVersion: "v1",
		},
		ObjectMeta: meta_v1.ObjectMeta{
			Name:      name,
			Namespace: namespace,
			Labels: map[string]string{
				"app": "contour",
			},
		},
		Data: data,
	}
}

// WritePEM writes a certificate out to its filename in outputDir.
func writePEM(outputDir, filename string, data []byte, force OverwritePolicy) error {
	filepath := path.Join(outputDir, filename)
	f, err := createFile(filepath, force == Overwrite)
	if err != nil {
		return err
	}
	_, err = f.Write(data)
	return checkFile(filepath, err)
}

// WriteCertsPEM writes out all the certs in certdata to
// individual PEM files in outputDir
func WriteCertsPEM(outputDir string, certdata *certs.Certificates, force OverwritePolicy) error {
	err := writePEM(outputDir, "cacert.pem", certdata.CACertificate, force)
	if err != nil {
		return err
	}

	err = writePEM(outputDir, "contourcert.pem", certdata.ContourCertificate, force)
	if err != nil {
		return err
	}

	err = writePEM(outputDir, "contourkey.pem", certdata.ContourPrivateKey, force)
	if err != nil {
		return err
	}

	err = writePEM(outputDir, "envoycert.pem", certdata.EnvoyCertificate, force)
	if err != nil {
		return err
	}

	return writePEM(outputDir, "envoykey.pem", certdata.EnvoyPrivateKey, force)
}

// WriteSecretsYAML writes all the keypairs out to Kubernetes Secrets in YAML form
// in outputDir.
func WriteSecretsYAML(outputDir string, secrets []*core_v1.Secret, force OverwritePolicy) error {
	for _, s := range secrets {
		filename := path.Join(outputDir, s.Name+".yaml")
		f, err := createFile(filename, force == Overwrite)
		if err != nil {
			return err
		}
		if err := checkFile(filename, writeSecret(f, s)); err != nil {
			return err
		}
	}

	return nil
}

// WriteSecretsKube writes all the keypairs out to Kubernetes Secrets in the
// compact format which is compatible with Secrets generated by cert-manager.
func WriteSecretsKube(client *kubernetes.Clientset, secrets []*core_v1.Secret, force OverwritePolicy) error {
	for _, s := range secrets {
		if _, err := client.CoreV1().Secrets(s.Namespace).Create(context.TODO(), s, meta_v1.CreateOptions{}); err != nil {
			if !k8serrors.IsAlreadyExists(err) {
				return err
			}

			if force == NoOverwrite {
				fmt.Printf("secret/%s already exists\n", s.Name)
				continue
			}

			if _, err := client.CoreV1().Secrets(s.Namespace).Update(context.TODO(), s, meta_v1.UpdateOptions{}); err != nil {
				return err
			}
		}

		fmt.Printf("secret/%s updated\n", s.Name)
	}

	return nil
}

// AsSecrets transforms the given Certificates struct into a slice of
// Secrets in in compact Secret format, which is compatible with
// both cert-manager and Contour.
func AsSecrets(namespace, nameSuffix string, certdata *certs.Certificates) ([]*core_v1.Secret, []error) {
	// Only check the "contourcert" name because suffixes are the same
	// for all, and "contourcert" is the longest.
	if errs := validateSecretNamespaceAndName(namespace, "contourcert"+nameSuffix); len(errs) > 0 {
		return nil, errs
	}

	return []*core_v1.Secret{
		newSecret(
			core_v1.SecretTypeTLS,
			"contourcert"+nameSuffix,
			namespace,
			map[string][]byte{
				dag.CACertificateKey:     certdata.CACertificate,
				core_v1.TLSCertKey:       certdata.ContourCertificate,
				core_v1.TLSPrivateKeyKey: certdata.ContourPrivateKey,
			}),
		newSecret(
			core_v1.SecretTypeTLS,
			"envoycert"+nameSuffix,
			namespace,
			map[string][]byte{
				dag.CACertificateKey:     certdata.CACertificate,
				core_v1.TLSCertKey:       certdata.EnvoyCertificate,
				core_v1.TLSPrivateKeyKey: certdata.EnvoyPrivateKey,
			}),
	}, nil
}

// AsLegacySecrets transforms the given Certificates struct into a slice of
// Secrets that is compatible with certgen from contour 1.4 and earlier.
// The difference is that the CA cert is in a separate secret, rather
// than duplicated inline in each TLS secrets.
func AsLegacySecrets(namespace, nameSuffix string, certdata *certs.Certificates) ([]*core_v1.Secret, []error) {
	// Only check the "contourcert" name because suffixes are the same
	// for all, and "contourcert" is the longest.
	if errs := validateSecretNamespaceAndName(namespace, "contourcert"+nameSuffix); len(errs) > 0 {
		return nil, errs
	}

	return []*core_v1.Secret{
		newSecret(
			core_v1.SecretTypeTLS,
			"contourcert"+nameSuffix,
			namespace,
			map[string][]byte{
				core_v1.TLSCertKey:       certdata.ContourCertificate,
				core_v1.TLSPrivateKeyKey: certdata.ContourPrivateKey,
			}),
		newSecret(
			core_v1.SecretTypeTLS,
			"envoycert"+nameSuffix,
			namespace,
			map[string][]byte{
				core_v1.TLSCertKey:       certdata.EnvoyCertificate,
				core_v1.TLSPrivateKeyKey: certdata.EnvoyPrivateKey,
			}),
		newSecret(
			core_v1.SecretTypeOpaque,
			"cacert"+nameSuffix,
			namespace,
			map[string][]byte{
				"cacert.pem": certdata.CACertificate,
			}),
	}, nil
}

func validateSecretNamespaceAndName(namespace, name string) []error {
	var errs []error

	for _, errstring := range validation.IsDNS1123Label(namespace) {
		errs = append(errs, fmt.Errorf("invalid namespace name %q: %s", namespace, errstring))
	}

	for _, errstring := range validation.IsDNS1123Subdomain(name) {
		errs = append(errs, fmt.Errorf("invalid secret name %q: %s", name, errstring))
	}

	if len(errs) > 0 {
		return errs
	}

	return nil
}
