package main

import (
	"context"
	"flag"
	"fmt"
	"io"
	"maps"
	"os"
	"os/signal"
	"slices"
	"text/template"

	"go.sia.tech/core/consensus"
	"go.sia.tech/core/types"
	"go.sia.tech/coreutils"
	"go.sia.tech/coreutils/chain"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

func writeTemplate(w io.Writer, overwriteIDs map[types.BlockID][]types.FileContractID) error {
	tmpl, err := template.New("").Parse(`
// Code generated by calcswaps; DO NOT EDIT.
package chain

import (
	"go.sia.tech/core/types"
)

var defaultExpiringFileContractOrder = {{.}}
`)
	if err != nil {
		return fmt.Errorf("failed to parse template: %w", err)
	}

	return tmpl.Execute(w, fmt.Sprintf("%#v", overwriteIDs))
}

func main() {
	var (
		network  string
		dbPath   string
		logLevel zap.AtomicLevel
	)

	flag.TextVar(&logLevel, "log.level", zap.NewAtomicLevelAt(zap.InfoLevel), "log level (debug, info, warn, error, dpanic, panic, fatal)")
	flag.StringVar(&dbPath, "db", "consensus.db", "path to the chain database")
	flag.StringVar(&network, "network", "mainnet", "the network to use (mainnet, zen, anagami)")
	flag.Parse()

	ctx, cancel := signal.NotifyContext(context.Background(), os.Interrupt)
	defer cancel()

	cfg := zap.NewProductionEncoderConfig()
	cfg.EncodeTime = zapcore.RFC3339TimeEncoder
	cfg.EncodeDuration = zapcore.StringDurationEncoder
	cfg.EncodeLevel = zapcore.CapitalColorLevelEncoder

	cfg.StacktraceKey = ""
	cfg.CallerKey = ""
	encoder := zapcore.NewConsoleEncoder(cfg)

	log := zap.New(zapcore.NewCore(encoder, zapcore.Lock(os.Stdout), logLevel))
	defer log.Sync()

	if _, err := os.Stat(dbPath); err != nil {
		log.Panic("failed to stat db", zap.String("dbPath", dbPath))
	}

	var n *consensus.Network
	var genesis types.Block
	switch network {
	case "mainnet":
		n, genesis = chain.Mainnet()
	case "zen":
		n, genesis = chain.TestnetZen()
	default:
		log.Panic("unknown network", zap.String("network", network))
	}
	maxCheckHeight := n.HardforkV2.RequireHeight

	tipDB, err := coreutils.OpenBoltChainDB(dbPath)
	if err != nil {
		log.Panic("failed to open db", zap.Error(err))
	}
	defer tipDB.Close()

	tipStore, tipState, err := chain.NewDBStore(tipDB, n, genesis, nil)
	if err != nil {
		log.Panic("failed to create tip store", zap.Error(err))
	}

	log.Info("starting expiring file contract order calculation",
		zap.String("network", network), zap.Stringer("index", tipState.Index))

	cleanStore, cs, err := chain.NewDBStore(chain.NewMemDB(), n, genesis, nil)
	if err != nil {
		log.Panic("failed to create clean db", zap.Error(err))
	}
	cm := chain.NewManager(cleanStore, cs, chain.WithLog(log.Named("chain")), chain.WithPruneTarget(144))

	tip := min(tipState.Index.Height, maxCheckHeight)
	overwriteIDs := make(map[types.BlockID][]types.FileContractID)
	for height := range tip - 144 {
		select {
		case <-ctx.Done():
			return
		default:
		}

		index, ok := tipStore.BestIndex(height)
		if !ok {
			log.Panic("failed to get best index", zap.Uint64("height", height))
		}
		log := log.With(zap.Stringer("index", index))
		b, bs, ok := tipStore.Block(index.ID)
		if !ok {
			log.Panic("failed to get block")
		} else if bs == nil {
			log.Panic("block state is nil")
		}

		cs, ok := tipStore.State(index.ID)
		if !ok {
			log.Panic("failed to get state for block")
		}

		order := make([]types.FileContractID, 0, len(bs.ExpiringFileContracts))
		for _, fc := range bs.ExpiringFileContracts {
			order = append(order, fc.ID)
		}

		cleanOrder := cleanStore.ExpiringFileContractIDs(height)
		seen := make(map[types.FileContractID]bool)
		if !slices.Equal(order, cleanOrder) {
			// ensure that all expiring file contracts in the clean db are also
			// in the tip db
			for _, id := range cleanOrder {
				seen[id] = true
			}
			for _, id := range order {
				if !seen[id] {
					log.Panic("expiring file contract not found in clean db")
				}
				delete(seen, id)
			}
			if len(seen) > 0 {
				log.Panic("clean db has extra expiring file contracts", zap.Stringers("extra", slices.Collect(maps.Keys(seen))))
			}
			log.Warn("expiring file contracts do not match", zap.Stringer("blockID", index.ID),
				zap.Stringers("expected", order),
				zap.Stringers("actual", cleanOrder))

			cleanStore.OverwriteExpiringFileContractIDs(height, order)
			if err := cleanStore.Flush(); err != nil {
				log.Panic("failed to flush clean db", zap.Error(err))
			}
			overwriteIDs[index.ID] = order
		}

		if err := cm.AddBlocks([]types.Block{b}); err != nil {
			log.Panic("failed to add block to clean db", zap.Error(err))
		}

		cleanState := cm.TipState()
		log = log.With(zap.Stringer("cleanIndex", cleanState.Index))
		if cleanState.Index != cs.Index {
			log.Panic("clean state index does not match tip state index")
		} else if cs.MerkleLeafHash(types.VoidAddress) != cleanState.MerkleLeafHash(types.VoidAddress) {
			log.Panic("clean state merkle leaf hash does not match tip state merkle leaf hash",
				zap.Stringer("expected", cs.MerkleLeafHash(types.VoidAddress)),
				zap.Stringer("actual", cleanState.MerkleLeafHash(types.VoidAddress)))
		}
	}

	f, err := os.Create("./chain/expiringcontracts.go")
	if err != nil {
		log.Panic("failed to create expiringcontracts.go", zap.Error(err))
	}
	defer f.Close()

	if err := writeTemplate(f, overwriteIDs); err != nil {
		log.Panic("failed to write template", zap.Error(err))
	} else if err := f.Sync(); err != nil {
		log.Panic("failed to sync file", zap.Error(err))
	} else if err := f.Close(); err != nil {
		log.Panic("failed to close file", zap.Error(err))
	}
}
