/*
 * Hypertex modifications to DVI previewer for X.
 * This portion of xhdvi is completely in the public domain. The
 * author renounces any copyright claims. It may be freely used for
 * commercial or non-commercial purposes. The author makes no claims
 * or guarantees - use this at your own risk.
 * 
 * Arthur Smith, U. of Washington, 1994
 * Copyright (c) 1994.  All rights reserved.
 *
 * 5/1994       code written from scratch.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "xhdvi.h"

/* Implementation of HyperTeX through \specials */
/* One option: allow HTML tags and recognize them */
/* applicationDoSpecial strips leading blanks, so first char should
   be '<' if it's a tag */

char *StrAllocCopy();
char *refscan();

#define HT_A_NAME 1
#define HT_A_HREF 2

#define ht_shrinkfactor mane.shrinkfactor /* Only main win has refs */

typedef struct {
	int type; /* Type of anchor: URL, etc from WWW anchor list */
	char *name; /* Name string for anchor (Null by default) */
	char *href; /* A reference from this anchor */
	int llx, lly, urx, ury; /* Box on page where anchor located */
} HT_Anchor;

/* Structure to remember where we have been: */
typedef struct {
	char *refname; /* File name (or href?) */
	int pageno; /* Which page anchor was on */
	int which; /* Number of anchor on this page */
	int type; /* And the other properties of this anchor */
	char *name;
	char *href;
} Anchors;

int waiting_for_anchor = -1; /* If waiting for anchor to be properly parsed? */
int cur_anchor_on_page; /* Keep track of current page as it's read in */

#define HT_AnchorSTEP 20
HT_Anchor **HT_anchorlist = NULL;
int *nHT_anchors, *maxHT_anchors;

Anchors *HT_visited = NULL;
int nHT_visited = 0, maxHT_visited = 0;

#define HT_NSTACK 32
int HTAnest[HT_NSTACK]; /* Maximum number of nested anchors */
int HTAnestlevel; /* Current nesting level */

int HTreflevel; /* 0 if not currently inside an href anchor */

int *ht_parsedpages = NULL; /* List of all pages, = 1 if already parsed, zero if not */
int ht_total_pages;
/* size = total_pages, current page = current_page defined in xhdvi.h */

void ht_parseanchor ARGS((char *, HT_Anchor *));
char *StrAllocCopy ARGS((char **, char *));
char *refscan      ARGS(( char*, char **, char **));
void ht_loc_on_page ARGS((Anchors *));
void freeHTAnchors ARGS((HT_Anchor *, int));
void ht_img	ARGS((int, char *, int));
void ht_base	ARGS((int, char *, int));

int checkHyperTeX(cp, pageno)
char *cp;
int pageno;
{
	int htfound = 0;

	if (strncasecmp(cp, "html:", 5) == 0) {
		cp += 5;
		while (isspace(*cp)) cp++;
		htfound = 1;
	}
	if (*cp == '<') { /* Possibly Missing the header part */
		htfound = 1;
		ht_handletag(cp, pageno);
	} else if (strncasecmp(cp, "hyp", 3) == 0) {
		/* Dave Oliver's HyperTeX */
		htfound = 1;
		cp += 4;
		hy_handletag(cp, pageno);
	}

	return htfound;
}

#define BEGIN 0
#define END 1

void ht_handletag(cp, pageno)
char *cp;
int pageno;
{
	int beginend=BEGIN;

	if (*cp != '<') return;
	++cp;
	while (isspace(*cp)) cp++;
	if (*cp == '/') {
		beginend = END;
		cp++;
	}
	switch(*cp) {
	    case 'A':
	    case 'a': /* Anchors */
		ht_anchor(beginend, cp+1, pageno);
		break;
	    case 'b': /* Base name? */
		ht_base(beginend, cp, pageno);
		break;
	    case 'i': /* Included images? */
		ht_img(beginend, cp, pageno);
		break;
	    default: /* Tag not implemented yet */
		break;
	}
}

/* Basically just want to parse the line... */
/* Should use WWW library stuff ? */

void ht_anchor(beginend, cp, pageno)
int beginend, pageno;
char *cp;
{
	int i, *nap, *maxp;
	int oldllx, oldlly, oldurx, oldury;
	HT_Anchor *HTAp, *HTAp2, **HTApp;

	HTApp = HT_anchorlist + pageno;
	nap = nHT_anchors + pageno;
	maxp = maxHT_anchors + pageno;
	if (*HTApp == NULL) {
		*maxp = HT_AnchorSTEP;
		*HTApp = (HT_Anchor *) xmalloc((*maxp)*sizeof(HT_Anchor), "Anchor");
	} else if (*nap == *maxp) {
		*maxp += HT_AnchorSTEP;
		*HTApp = (HT_Anchor *) realloc(*HTApp,
					(*maxp)*sizeof(HT_Anchor));
	}
	if (ht_parsedpages[pageno] != 1) {  /* Only do if page not done yet */
	  if (beginend == END) {
	    HTAnestlevel--;
	    if (HTAnestlevel < 0) {
		HTAnestlevel = 0; /* Extra </a>'s? */
	    } else {
		HTAp = *HTApp + HTAnest[HTAnestlevel];
		if (HTAp->llx > DVI_H) {
			HTAp->llx = DVI_H;
		}
		if (HTAp->urx < DVI_H) {
			HTAp->urx = DVI_H;
		}
		if (HTAp->lly > DVI_V) {
			HTAp->lly = DVI_V;
		}
		if (HTAp->ury < DVI_V) {
			HTAp->ury = DVI_V;
		}
		oldllx = HTAp->llx;
		oldlly = HTAp->lly;
		oldurx = HTAp->urx;
		oldury = HTAp->ury;
		if (debug & DBG_HYPER) {
		    fprintf(stderr, "Added anchor %d, level %d:\n",
			    HTAnest[HTAnestlevel], HTAnestlevel);
		    if (HTAp->type&HT_A_HREF) {
			fprintf(stderr, "href = %s\n", HTAp->href);
		    }
		    if (HTAp->type&HT_A_NAME) {
			fprintf(stderr, "name = %s\n", HTAp->name);
		    }
		    fprintf(stderr, "box %d %d %d %d\n",
			HTAp->llx, HTAp->lly, HTAp->urx, HTAp->ury);
		}
		if (waiting_for_anchor == HTAnest[HTAnestlevel]) {
			ht_to_anchor(current_page, waiting_for_anchor);
			waiting_for_anchor = -1; /* Reset it! */
		}
/* End of debug section */
		if (HTAnestlevel > 0) {
			HTAp = *HTApp + HTAnest[HTAnestlevel-1];
			/* Check llx, lly, urx, ury info */
			if (oldllx < HTAp->llx) {
				HTAp->llx = oldllx;
			}
			if (oldlly < HTAp->lly) {
				HTAp->lly = oldlly;
			}
			if (oldurx > HTAp->urx) {
				HTAp->urx = oldurx;
			}
			if (oldury > HTAp->ury) {
				HTAp->ury = oldury;
			}
		}
	    }
	  } else {
		HTAp = *HTApp + *nap;
	/* Set type, and the name, href */
		ht_parseanchor(cp, HTAp);
		if (HTAp->type != 0) {
		    cur_anchor_on_page++; /* Increment the count of anchors here */
		    if (ht_parsedpages[pageno] == 2) {
    			/* Go to this anchor in list we already have: */
			HTAp = *HTApp + cur_anchor_on_page;
			HTAp->urx = HTAp->llx = DVI_H; /* Current horiz pos.*/
			HTAp->ury = HTAp->lly = DVI_V; /* Current vert. pos. */
			if (HTAnestlevel >= HT_NSTACK) {
				/* Error - too many nested anchors! */
			} else {
				HTAnest[HTAnestlevel++] = cur_anchor_on_page;
			}
		    } else if (ht_parsedpages[pageno] != 1) {  /* Only do if page not done yet */
			HTAp->urx = HTAp->llx = DVI_H; /* Current horiz pos.*/
			HTAp->ury = HTAp->lly = DVI_V; /* Current vert. pos. */
			if (HTAnestlevel >= HT_NSTACK) {
				/* Error - too many nested anchors! */
			} else {
				HTAnest[HTAnestlevel++] = *nap;
			}
			(*nap)++;
		    }
		}
	  }
	} else { /* if page has been properly parsed before */
	  if (beginend != END) {
		HTAp = *HTApp + *nap;
	/* Set type, and the name, href */
		ht_parseanchor(cp, HTAp);
	  }
	}
	if (beginend == END) {
		if (HTreflevel > 0) HTreflevel--;
	} else {
		if (HTAp->type&HT_A_HREF) HTreflevel++;
	}
}

void ht_initpage() /* Starting a new page */
{
	if (ht_parsedpages == NULL) ht_reinit();
	HTAnestlevel = 0; /* Start with zero nesting level for a page */
	HTreflevel = 0;
	HTnext_extern = 0; /* Go to links in current window */
	cur_anchor_on_page = -1;
	paint_anchor(NULL);
}

/* A character or something was written: record position for current anchor */
void ht_recordbits(x, y, w, h)
int x, y, w, h;
{
	HT_Anchor *HTAp;
	int dvix, dviy, dvix2, dviy2;
	int ch = 0;

	dvix = x*(ht_shrinkfactor << 16);
	dviy = y*(ht_shrinkfactor << 16);
	dvix2 = (x+w)*(ht_shrinkfactor << 16);
	dviy2 = (y+h)*(ht_shrinkfactor << 16);
	HTAp = HT_anchorlist[current_page] + HTAnest[HTAnestlevel-1];
	if (HTAp->llx > dvix) {
		HTAp->llx = dvix;
		ch++;
	}
	if (HTAp->lly > dviy) {
		HTAp->lly = dviy;
		ch++;
	}
	if (HTAp->urx < dvix2) {
		HTAp->urx = dvix2;
		ch++;
	}
	if (HTAp->ury > dviy2) {
		HTAp->ury = dviy2;
		ch++;
	}
	if (debug & DBG_HYPER) {
	    if (ch > 0) {
		fprintf(stderr, "New box for anchor %d, level %d: %d %d %d %d\n",
			HTAnest[HTAnestlevel-1], HTAnestlevel,
			HTAp->llx, HTAp->lly, HTAp->urx, HTAp->ury);
	    }
	}
}

void ht_donepage(i, pflag) /* This page has been completed */
int i, pflag;
{
	HT_Anchor *HTAp;

	/* Finish off boxes for nested anchors not done on this page */
	while (HTAnestlevel > 0) {
		HTAnestlevel--;
		HTAp = HT_anchorlist[i] + HTAnest[HTAnestlevel];
		if (HTAp->llx > DVI_H) {
			HTAp->llx = DVI_H;
		}
		if (HTAp->urx < DVI_H) {
			HTAp->urx = DVI_H;
		}
		if (HTAp->lly > DVI_V) {
			HTAp->lly = DVI_V;
		}
		if (HTAp->ury < DVI_V) {
			HTAp->ury = DVI_V;
		}
	}
	if (pflag == 1) { /* Really parsed this page */
		ht_drawboxes(); /* Draw boxes around the anchor positions */
		ht_parsedpages[i] = 1;
	} else {
		ht_parsedpages[i] = 2; /* Means ht_parsed, not done properly */
	}
}

/* If the dvi file has changed, assume all links have changed also,
   and reset everything! */

void ht_reinit()
{
	int i;

	if (ht_parsedpages == NULL) { /* First call to this routine */
		ht_parsedpages = (int *) xmalloc(total_pages*sizeof(int), "HTpages");
		HT_anchorlist = (HT_Anchor **) xmalloc(total_pages*
				sizeof(HT_Anchor *), "Anchors");
		nHT_anchors = (int *) xmalloc(total_pages* sizeof(int), "Anchors");
		maxHT_anchors = (int *) xmalloc(total_pages* sizeof(int), "Anchors");
		for (i=0; i < total_pages; i++) maxHT_anchors[i] = 0;
	} else if (ht_total_pages != total_pages) {
		ht_parsedpages = (int *) realloc(ht_parsedpages,
					total_pages*sizeof(int));
	/* Following operates if new has fewer pages than old: */
		for (i=total_pages; i < ht_total_pages; i++) {
		    	if (maxHT_anchors[i] > 0)
			    freeHTAnchors(HT_anchorlist[i], nHT_anchors[i]);
		}
		HT_anchorlist = (HT_Anchor **) realloc(HT_anchorlist,
				total_pages*sizeof(HT_Anchor *));
		nHT_anchors = (int *) realloc(nHT_anchors,
					total_pages* sizeof(int));
		maxHT_anchors = (int *) realloc(maxHT_anchors,
					total_pages* sizeof(int));
	/* Following operates if new has more pages than old: */
		for (i= ht_total_pages; i < total_pages; i++)
				maxHT_anchors[i] = 0;
	}
	ht_total_pages = total_pages;
	for (i=0; i < total_pages; i++) {
	    ht_parsedpages[i] = 0;
	    if (maxHT_anchors[i] > 0) { /* Get rid of the old anchor lists: */
		   freeHTAnchors(HT_anchorlist[i], nHT_anchors[i]);
		   free(HT_anchorlist[i]);
	    }
	    HT_anchorlist[i] = NULL;
	    nHT_anchors[i] = 0;
	    maxHT_anchors[i] = 0;
	}
}

/* Following parses the stuff after the '<' in the html tag */
/* Only understands name and href in anchor */
/*     html: <A HREF="..." NAME="..."> */
/*     html: <A NAME="..." HREF="...> */

void ht_parseanchor(cp, anchor)
char *cp;
HT_Anchor *anchor;
{
	char *ref, *str;

	anchor->type = 0;
	anchor->href = NULL;
	anchor->name = NULL;
	while (isspace(*cp)) cp++;
	while ((*cp) && (*cp != '>')) {
		cp = refscan(cp, &ref, &str);
		if (cp == NULL) break;
		if (strcasecmp(ref, "href") == 0) {
			anchor->type |= HT_A_HREF;
			StrAllocCopy(&(anchor->href), str);
		} else if (strcasecmp(ref, "name") == 0) {
			anchor->type |= HT_A_NAME;
			StrAllocCopy(&(anchor->name), str);
		}
	}
}

char *StrAllocCopy(dest, src)
char **dest, *src;
{
	if (*dest) free(*dest);
	if (! src)
		*dest = NULL;
	else {
		*dest = (char *) xmalloc(strlen(src) + 1, "StrAllocCopy");
		strcpy(*dest, src);
	}
	return *dest;
}

/* Parses cp containing 'ref="string"more', returning pointer to "more" */
char *refscan(name, ref, str)
char *name, **ref, **str;
{
	char *cp;

	*str = name;
	for (cp=name; *cp; cp++) {
		if (*cp == '=') {
			*cp = 0;
			*ref = name;
			*str = cp+1;
			break;
		}
	}
	cp = *str;
	if (cp != name) {
	    while (isspace(*cp)) cp++;
	    if (*cp == '"') { /* Yes, this really is a string being set */
		*str = cp+1;
		while ((cp = strchr(cp+1, '"')) != NULL) {
			if (cp[-1] != '\\') break; /* Check if quote escaped */
		}
		if (cp != NULL) {
			*cp = 0;
			cp++;
		}
	    } else {
		cp = NULL;
	    }
	} else {
		cp = NULL;
	}
	return cp;
}

/* What happens when mouse moves on screen: */
void ht_displayanchor(x, y)
int x, y; /* x,y coordinates of mouse position */
{
	char astr[256];
	char namestr[256];
	char hrefstr[256];
	HT_Anchor *HTAp;
	long dvix, dviy;
	int i;

/* Don't display until we've finished with the page: */
	if (ht_parsedpages == NULL) return;
	if (ht_parsedpages[current_page] != 1) return;
	/* First need to convert mouse coords to dvi coords: */
	dvix = (x + mane.base_x)*(ht_shrinkfactor << 16);
	dviy = (y + mane.base_y)*(ht_shrinkfactor << 16);
	/* Find anchor that fits current position: */
	HTAp = HT_anchorlist[current_page] + nHT_anchors[current_page] - 1;
	for (i=nHT_anchors[current_page] - 1; i >= 0; i--, HTAp--) {
		if (HTAp->llx > dvix) continue;
		if (HTAp->lly > dviy) continue;
		if (HTAp->urx < dvix) continue;
		if (HTAp->ury < dviy) continue;
		if (debug & DBG_HYPER) {
		    fprintf(stderr, "In anchor #%d\n", i);
		}
		if (HTAp->type & HT_A_NAME) {
			sprintf(namestr, "name = %s ", HTAp->name);
		} else {
			sprintf(namestr, "");
		}
		if (HTAp->type & HT_A_HREF) {
			sprintf(hrefstr, "href = %s ", HTAp->href);
		} else {
			sprintf(hrefstr, "");
		}
		sprintf(astr, "anchor #%d: %s%s", i, namestr, hrefstr);
		paint_anchor(astr);
		break;
	}
	if (i == -1) paint_anchor(NULL);
}

/* What happens when mouse is clicked: */
int ht_handleref(x, y)
int x, y; /* current mouse location when ref clicked */
{
	HT_Anchor *HTAp;
	long dvix, dviy;
	int i, afound;

/* Check that we've finished the page first! */
	if (ht_parsedpages == NULL) return;
	if (ht_parsedpages[current_page] != 1) return;
	/* First need to convert mouse coords to dvi coords: */
	dvix = (x + mane.base_x)*(ht_shrinkfactor << 16);
	dviy = (y + mane.base_y)*(ht_shrinkfactor << 16);
	/* Find anchor that fits current position: */
	HTAp = HT_anchorlist[current_page] + nHT_anchors[current_page] - 1;
	afound = -1;
	for (i=nHT_anchors[current_page]-1; i >= 0; i--, HTAp--) {
		if ((HTAp->type&HT_A_HREF) == 0) continue; /* Only ref on hrefs */
		if (HTAp->llx > dvix) continue;
		if (HTAp->lly > dviy) continue;
		if (HTAp->urx < dvix) continue;
		if (HTAp->ury < dviy) continue;
		afound = i; /* Get the last of them in case of nesting */
		break;
	}
	if (afound == -1) return 0; /* There was no href at this location */
/* Update the list of where we used to be: */
	if (HT_visited == NULL) {
		maxHT_visited = HT_AnchorSTEP;
		HT_visited = (Anchors *) xmalloc(maxHT_visited*sizeof(Anchors), "Visited");
		for (i=0; i < maxHT_visited; i++) {
			HT_visited[i].refname = NULL;
			HT_visited[i].name = NULL;
			HT_visited[i].href = NULL;
		}
	} else if (nHT_visited >= maxHT_visited - 1) {
		maxHT_visited += HT_AnchorSTEP;
		HT_visited = (Anchors *) realloc(HT_visited,
			maxHT_visited*sizeof(Anchors));
		for (i=nHT_visited; i < maxHT_visited; i++) {
			HT_visited[i].refname = NULL;
			HT_visited[i].name = NULL;
			HT_visited[i].href = NULL;
		}
	}
	StrAllocCopy(&(HT_visited[nHT_visited].refname), dvi_name);
	HT_visited[nHT_visited].pageno = current_page;
	HT_visited[nHT_visited].type = HTAp->type;
	if (HTAp->type & HT_A_NAME) {
		StrAllocCopy(&(HT_visited[nHT_visited].name), HTAp->name);
	}
	if (HTAp->type & HT_A_HREF) {
		StrAllocCopy(&(HT_visited[nHT_visited].href), HTAp->href);
	}
	HT_visited[nHT_visited].which = HTAp - HT_anchorlist[current_page];
	nHT_visited++;
/* Then just do it: */
	ht_dohref(HTAp->href);
	return 1;
}

int ht_dohref(href)
char *href;
{
	/* Need to put the HT_visited stuff here too... */
	if (ht_is_url(href)) ht_do_url(href);
	else ht_do_loc(href);
	/* Need to handle properly when ref doesn't exist! */
}

/* Draw boxes around the anchor positions */
void ht_drawboxes()
{
	HT_Anchor *HTAp;
	int i;
	int x, y, w, h;
	
	if (!underline_link) return;
	HTAp = HT_anchorlist[current_page];
	for (i=0; i < nHT_anchors[current_page]; i++, HTAp++) {
		if ((HTAp->type&HT_A_HREF) == 0) continue; /* Only box hrefs */
		x = pixel_conv(HTAp->llx)-1;
		y = pixel_conv(HTAp->lly)-1;
		w = pixel_conv(HTAp->urx) - x+2;
		h = pixel_conv(HTAp->ury) - y+2;
/* The last arg of put_rule is whether or not to
   use the "highlight" graphics context. */
/*		put_rule(x, y, w, 1, True);
		put_rule(x+w, y, 1, h, True); */
		put_rule(x+1, y+h, w, 1, True);
/*		put_rule(x, y+1, 1, h, True); */
	}
}

/* It's a local reference - find the anchor and go to it */
void ht_do_loc(href)
char *href;
{
	int ipage, ia, reffound;
	HT_Anchor *HTAp, **HTApp;
	char astr[256];
	char *cp;
	
	if (href == NULL) return; /* shouldn't happen? */
	cp = href;
	while (*cp == '#') cp++;
	if (HTnext_extern == 1) {
		invokedviviewer(dvi_name, cp);
		return;
	}
	HTApp = HT_anchorlist;
	reffound = 0;
/* Should hash based on "name" value? - to speed this up! */
	for (ipage = 0; ipage < total_pages; ipage++, HTApp++) {
	    	if (ht_parsedpages[ipage] == 0) continue;
		HTAp = *HTApp;
		for (ia=0; ia < nHT_anchors[ipage]; ia++, HTAp++) {
			if ((HTAp->type&HT_A_NAME) == 0) continue;
			if (!strcmp(HTAp->name, cp)) {
				reffound = 1;
				break;
			}
		}
		if (reffound) break;
	}
	if (reffound == 0) { /* Need to parse remaining pages */
		if (debug & DBG_HYPER) {
		    fprintf(stderr, "Searching for remaining anchors\n");
		}
		ht_parsepages();
		/* And try again: */
		HTApp = HT_anchorlist;
		for (ipage = 0; ipage < total_pages; ipage++, HTApp++) {
			if (ht_parsedpages[ipage] < 2) continue;
			HTAp = *HTApp;
			for (ia=0; ia < nHT_anchors[ipage]; ia++, HTAp++) {
				if ((HTAp->type&HT_A_NAME) == 0) continue;
				if (!strcmp(HTAp->name, cp)) {
					reffound = 1;
					break;
				}
			}
			if (reffound) break;
		}
	}
	if (reffound) {
		ht_to_anchor(ipage, ia); /* move to anchor */
	} else {
		if ((nHT_visited == 0) ||
		   (!strcmp(HT_visited[nHT_visited-1].refname, dvi_name))) {
		/* Really was from same file - just print error message */
			sprintf(astr, "Error: reference \"%s\" not found\n", cp);
			paint_anchor(astr);
		} else {
			/* Go to page 1 and print error message */
		sprintf(astr, "Error: reference \"%s\" in file %s not found\n",
				cp, dvi_name);
			ht_to_page(0); /* Go to first page! */
			paint_anchor(astr);
		}
	}
}

void ht_to_page(pageno)
int pageno;
{
	/* taken from keystroke subroutine: */
	current_page = pageno;
	hush_spec_now = hush_spec;
	ht_can_it();
}

void ht_to_anchor(pageno, n)
int pageno, n;
{
	int x, y;
	HT_Anchor *HTAp = HT_anchorlist[pageno] + n;

	if ((n < 0) || (n >= nHT_anchors[pageno])) return; /* Error message for this?*/
	if (pageno != current_page) {
		if (ht_parsedpages[pageno] != 1) waiting_for_anchor = n;
	/* taken from keystroke subroutine: */
		current_page = pageno;
		hush_spec_now = hush_spec;
		ht_can_it();
	}
	x = (HTAp->llx + HTAp->urx)/(2*ht_shrinkfactor << 16);
	y = (HTAp->lly + HTAp->ury)/(2*ht_shrinkfactor << 16);
	if (debug & DBG_HYPER) {
		fprintf(stderr, "moving to pixel %d %d\n", x, y);
	}
	if (ht_parsedpages[pageno] > 0) centerpage(x, y);
}

/* Following goes back to previous anchor point */
void ht_goback()
{
	int i;

	if (nHT_visited <= 0) return; /* There's nowhere to go! */

	if (debug & DBG_HYPER) {
	    fprintf(stderr, "Currently %d anchors in sequence:\n", nHT_visited);
	    for (i=0; i < nHT_visited; i++) {
		fprintf(stderr, "%d file %s, href=%s\n", i,
			HT_visited[i].refname, HT_visited[i].href);
	    }
	}
	nHT_visited--;
	if (strcmp(HT_visited[nHT_visited].refname, dvi_name) != 0) {
		/* Need to read in old file again! */
		StrAllocCopy(&dvi_name, HT_visited[nHT_visited].refname);
		if (open_dvi_file() == 0) {
			perror(dvi_name);
			exit(1);
		}
		ht_reinit();
	}
	ht_loc_on_page(HT_visited + nHT_visited);
	/* taken from keystroke subroutine: */
	hush_spec_now = hush_spec;
	ht_can_it();
}

/* Is this a url we recognize? */
int ht_is_url(href)
char *href;
{
	char *cp;
	int ret = 0;

	cp = href;
/* Be flexible on local files: */
	if (strncasecmp(href, "file:", 5) == 0) {
		ret = 1;
/* regular URL's (from lynx): */
	} else if(!strncmp(cp,"news:",5)) {
		ret = 1;
	} else if(!strncmp(cp,"news\\:",6)) {
		ret = 1;
	} else if(!strncmp(cp,"mailto:",7)) {
        	ret = 1;
	} else if(!strncmp(cp,"mailto\\:",8)) {
		ret = 1;

        /* if it doesn't contain ":/" then it can't be a url 
         * except for news:
         */
	} else if(!strstr(cp+3,":/")) {  
		ret = 0;
	} else if(!strncmp(cp,"http",4)) {
		ret = 1;
	} else if(!strncmp(cp,"gopher",6)) {
		ret = 1;
	} else if(!strncmp(cp,"ftp",3)) {
		ret = 1;
	} else if(!strncmp(cp,"wais",4)) {
		ret = 1;
	} else if(!strncmp(cp,"telnet",6)) {
		ret = 1;
	} else if(!strncmp(cp,"tn3270",6)) {
		ret = 1;
	} else if(!strncmp(cp,"rlogin",6)) {
		ret = 1;
	} else if(!strncmp(cp,"afs",3)) {
		ret = 1;
	} else if(!strncmp(cp,"prospero",8)) {
		ret = 1;
	} else {
		ret = 0;
	}
	return ret;
}

/* Can handle href's of form file:?.dvi#name */
void ht_do_url(href)
char *href;
{
	char *command;
	static char *filename = NULL;
	char *aname, *cp;

/* Probably should resolve href as a full URL, using URLbase, before
   doing anything else? */
	StrAllocCopy(&filename, href);
	/* First fix up the URL value in "filename": */
	if ((cp = strchr(filename, ';')) != NULL) {
		*cp = '\0'; /* Disallow extra commands on line */
	} else { /* Get rid of trailing > and spaces */
		cp = filename + strlen(filename) - 1;
		if (*cp == '>') cp --;
		while (isspace(*cp)) cp--;
		cp++;
		*cp = '\0'; 
	}
	if (strncasecmp(filename, "file:", 5) == 0) {
		filename += 5; /* We're opening a new file! */
		cp = aname = strchr(filename, '#');
		if (aname != NULL) {
			*aname = '\0'; /* Terminate filename string */
			aname++;
			while (*aname == '#') aname++;
		} else {
			cp = filename + strlen(filename);
		}
		add_dir(filename, &fullpathname);
		if (strcmp(cp - 4,".dvi") != 0) { /* Try another mime viewer */
			if (invokeviewer(fullpathname) == 1) return;
		} else {
			if (HTnext_extern == 1) {
				invokedviviewer(fullpathname, aname);
				return;
			}
			StrAllocCopy(&dvi_name, fullpathname);
			if (open_dvi_file() == 0) { /* Couldn't open dvi file */
				ht_goback(); /* Go back to where we were! */
			} else {
				ht_reinit();
				ht_do_loc(aname);
				return;
			}
		}
	}
	/* Spawn a web browser with this URL */
	if (browser == NULL) {
		paint_anchor("Error: WWWBROWSER not set for URL access\n");
		fprintf(stderr, "You need to set the environment variable WWWBROWSER\n");
		fprintf(stderr, "or specify -browser on command line to access\n");
		fprintf(stderr, "networked URL's\n");
	} else {
		if (URLbase == NULL) {
			command = (char *) xmalloc(strlen(filename)
			 + strlen(browser) + 10, "Command");
			sprintf(command, "%s '%s' &", browser, filename);
		} else {
			command = (char *) xmalloc(strlen(filename)
			 + strlen(URLbase) + strlen(browser) + 20, "Command");
			sprintf(command, "%s -base '%s' '%s' &",
				 browser, URLbase, filename);
		}
		if (debug & DBG_HYPER) {
			fprintf(stderr, "Executing: %s\n", command);
		}
		system(command);
		free(command);
	}
}

/* Find the anchor pointed to by ap on the given page */

void ht_loc_on_page(ap)
Anchors *ap;
{
	int i;
	HT_Anchor *HTAp;

	if (ht_parsedpages[ap->pageno] == 0) {
		ht_parse_page(ap->pageno); /* Parse the needed page! */
	}
	ht_to_anchor(ap->pageno, ap->which); /* move to anchor i */
}

void freeHTAnchors(HTAp, nHT)
HT_Anchor *HTAp;
int nHT;
{
	int i;

	for (i=0; i < nHT; i++) {
		if (HTAp[i].type&HT_A_NAME)
			free(HTAp[i].name);
		if (HTAp[i].type&HT_A_HREF)
			free(HTAp[i].href);
	}
}

/* Add the string cp to the current search string */
#define LINE 1024
char anchor_search_string[LINE];
int anchor_search_len = 0;

void add_search(cp, n)
char *cp;
int n;
{
	while (n>0) {
	    switch(*cp) {
		case '\v':
		case '\f':
		case '\r':
		case '\n': /* Finish string and search on it */
			anchor_search_string[anchor_search_len] == '\0';
			if (anchor_search_len > 0) {
				ht_dohref(anchor_search_string);
			/* Need to handle properly when ref doesn't exist! */
			}
			return;
			break;
		case '\b':
		case '\177':	/* Del */
			if (anchor_search_len > 0) anchor_search_len--; 
			break;
		case '':
			anchor_search_len = 0;
			break;
		default:
			if (*cp > 10) {
				anchor_search_string[anchor_search_len++] = *cp;
			}
			break;
		}
		cp++;
		n--;
	}
	anchor_search_string[anchor_search_len] = '\0';
	if (debug & DBG_HYPER) {
		fprintf(stderr, "search string: %s\n", anchor_search_string);
	}
	paint_search(anchor_search_string);
}

void ht_base(beginend, cp, pageno)
int beginend, pageno;
char *cp;
{
	char *ref, *str;
	if (beginend == END) return;

	if (!strncasecmp(cp, "base", 4)) {
		cp += 4;
		cp = refscan(cp, &ref, &str);
		if (cp == NULL) return;
		while (isspace(*ref)) ref++;
		while (isspace(*str)) str++;
		if (strcasecmp(ref, "href") == 0) {
			cp = str + strlen(str) - 1; /* Fix end of anchor */
			while (isspace(*cp)) cp--;
			if (*cp == '>') cp --;
			while (isspace(*cp)) cp--;
			cp++;
			*cp = '\0'; 
			StrAllocCopy(&URLbase, str); /* Change base */
			if (debug & DBG_HYPER) {
				fprintf(stderr, "Changing base name to: %s\n", URLbase);
			}
		}
	}
}

void ht_img(beginend, cp, pageno)
int beginend, pageno;
char *cp;
{
	char *ref, *str;

	if (beginend == END) return;
	if (pageno != current_page) return; /* Only do when on page */
	if (ht_parsedpages[pageno] == 1) return; /* And first time through */
	if (!strncasecmp(cp, "img", 3)) {
		cp += 3;
		cp = refscan(cp, &ref, &str);
		if (cp == NULL) return;
		while (isspace(*ref)) ref++;
		while (isspace(*str)) str++;
		if (strcasecmp(ref, "src") == 0) {
			cp = str + strlen(str) - 1; /* Fix end of anchor */
			while (isspace(*cp)) cp--;
			if (*cp == '>') cp --;
			while (isspace(*cp)) cp--;
			cp++;
			*cp = '\0'; 
			add_dir(str, &fullpathname);
			if (invokeviewer(fullpathname) != 1) {
	fprintf(stderr, "Don't know how to deal with <img src=%s>\n", fullpathname);
			}
		}
	}
}

/* Dave Oliver's hypertex format: */
/* Only understand my version of anchors so far */
/* ie.: his TYPE=text for hrefs, frag for names */
hy_handletag(cp, pageno)
char *cp;
int pageno;
{
	int beginend=BEGIN;

	while (isspace(*cp)) cp++;
	if (!strncasecmp(cp, "END", 3)) {
		beginend = END;
		cp += 3;
	}
/* Leave the parsing to ht_anchor! */
	ht_anchor(beginend, cp, pageno);
}

add_dir(file, fullpath)
char *file, **fullpath;
{
	if (*fullpath != NULL) {
		free(*fullpath);
		*fullpath = NULL;
	}
	if ((current_dir == NULL) || (file[0] == '/')) {
		*fullpath = xmalloc(strlen(file)+1, "full path name");
		Sprintf(*fullpath, "%s", file);
	} else {
		*fullpath = xmalloc(strlen(file) + strlen(current_dir) + 1,
					"full path name");
		Sprintf(*fullpath, "%s%s", current_dir, file);
	}
}
