#include "pdfmark.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <float.h>

int curpage;

static enum tokentype nextobj(char **infile);
static string readstr(char **infile);
static string readlit(char **infile);
static void skipnextobj(char **infile, enum tokentype nexttok);
static enum marktype linktype(char **infile);
static struct annmark getpdfann(char **infile);
static struct lnkmark getpdflnk(char **infile);
static struct outmark getpdfout(char **infile);
static struct artmark getpdfart(char **infile);
static struct destmark getpdfdest(char **infile);
static struct infomark getpdfinfo(char **infile);
static struct viewmark getpdfview(char **infile);

static enum tokentype
nextobj(char **infile)
{int ch;
 while ( ( (ch=(*(*infile)++)) != '\0' ) && isspace(ch) ) ;
 if (ch == '/') return lit;
 if (ch == '(') return str;
 if (ch == ')') return estr;
 if (ch == '[') return arr;
 if (ch == ']') return earr;
 if (ch == '{') return proc;
 if (ch == '}') return eproc;
 if (ch == '<') {
   if ( (ch=(*(*infile)++)) != '<' ) {
     if (ch != '\0') (*infile)--;
     return hex;
   } else {
     return dict;
   }
 }
 if (ch == '>') {
   if ( (ch=(*(*infile)++)) != '>') {
     if (ch != '\0') (*infile)--;
     return ehex;
   } else {
     return edict;
   }
 }
 if (ch != '\0') (*infile)--;
 return other;
}

static string 
readstr (char **infile)
{
  int ch1, ch2;
  int i=0;
  char buffer[65536]="\0";
  string result;

  while (!(**infile == '\0')) {
    int ch;
    while ( ((ch=(*(*infile)++)) != '\0') && (ch != '\\') && ch != ')' )
      buffer[i++] = ch;
    if (ch == ')' || (**infile == '\0') || ( (ch=(*(*infile)++)) == '\0' ) ) break;
    if (isdigit(ch)) {
      if ( ((ch1=(*(*infile)++)) == '\0')||((ch2=(*(*infile)++)) == '\0') ) break;
      buffer[i++] = (ch<<6 | ch1<<3 | ch2);
    } else if (ch != '\n') {
      buffer[i++] = ch;
    }
  }
  buffer[i++] = '\0';

  if (!(result = malloc((size_t)i))) return NULL;
  strcpy(result, buffer);

  return result;
}

static string
readlit(char **infile)
{
 char name[256], *result;
 if (sscanf(*infile,"%255[^][%(){}<> \t\n\r]",name)!=1) {
   return NULL;
 }
 *infile += strlen(name);
 if (!(result=malloc(strlen(name)+1))) return NULL;
 return strcpy(result,name);
}

static void
skipnextobj(char **infile,enum tokentype nexttok)
{
 switch (nexttok) {
 case lit:
 case other:
   free(readlit(infile)); break;
 case str:
   free(readstr(infile)); break;
 case hex:
   while (*(*infile)++ != '>');
   break;
 case arr:
   while ( nexttok = nextobj(infile), (nexttok!=earr) ) 
     skipnextobj(infile,nexttok); break;
 case proc:
 case dict:
   while ( nexttok = nextobj(infile), (nexttok!=eproc) && (nexttok!=edict) ) 
     skipnextobj(infile,nexttok); break;
 default:  break;
 }
}

static enum marktype
linktype(char **infile)
{
 char *name;
 enum tokentype next = nextobj(infile);

 if (next != lit) {
   skipnextobj(infile,next);
   skipnextobj(infile,arr);
   return OTHER;
 }

 name = readlit(infile);

 if (strcmp(name,"ANN")==0) return ANN;
 if (strcmp(name,"LNK")==0) return LNK;
 if (strcmp(name,"OUT")==0) return OUT;
 if (strcmp(name,"ARTICLE")==0) return ARTICLE;
 if (strcmp(name,"DEST")==0) return DEST;
 if (strcmp(name,"PS")==0) return OTHER;
 if (strcmp(name,"PAGES")==0) return OTHER;
 if (strcmp(name,"PAGE")==0) return OTHER;
 if (strcmp(name,"DOCINFO")==0) return DOCINFO;
 if (strcmp(name,"DOCVIEW")==0) return DOCVIEW;

 skipnextobj(infile,arr);
 return OTHER;
}

static struct annmark 
getpdfann(char **infile)
{ struct annmark result
    ={OTHER, NULL, NULL, NULL, 0, {0, 0, 0, 0}, NULL, NULL, 0};
  char *name;
  enum tokentype next;

  result.SrcPage = curpage;

  while ( (next = nextobj(infile)) == lit ) {

    name=readlit(infile);

    if (strcmp(name,"Title")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Title = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"SrcPage")==0) {
      char *tmp = *infile;
      result.SrcPage = strtol(tmp, infile, 10);
      if (tmp == *infile) {
	result.SrcPage = curpage;
	skipnextobj(infile,nextobj(infile));
      }
    } else if (strcmp(name,"Rect")==0) {
      if ( (next=nextobj(infile)) == arr ) {
	int i;
	for (i=0; i<4; i++) {
	  char *tmp = *infile;
	  result.Rect[i] = strtod(tmp, infile);
	}
	skipnextobj(infile,arr);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"ModDate")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.ModDate = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Contents")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Contents = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Open")==0) {
      if ( (next=nextobj(infile)) == other ) {
	char *val = readlit(infile);
	result.Open = (strcmp(val,"false") == 0);
	free(val);
      } else {
	skipnextobj(infile,next);
      }
    } else skipnextobj(infile,nextobj(infile));

  }

  if (next == earr) {
    result.type = ANN;
  } else {
    skipnextobj(infile,next);
    skipnextobj(infile,arr);
  }

  return result;
}

static struct lnkmark
getpdflnk(char **infile)
{ struct lnkmark result={OTHER, NULL, NULL, NULL, 0, {0, 0, 0, 0}, NULL,
			 GoTo, NULL, {0, Fit, {0, 0, 0, 0}, NULL, Open,
					{NULL, NULL} } };
  char *name;
  enum tokentype next;
  
  result.SrcPage = curpage;

  while ( (next = nextobj(infile)) == lit ) {

    name=readlit(infile);

    if (strcmp(name,"Title")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Title = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"SrcPage")==0) {
      char *tmp = *infile;
      result.SrcPage = strtol(tmp, infile, 10);
      if (tmp == *infile) {
	result.SrcPage = curpage;
	skipnextobj(infile,nextobj(infile));
      }
    } else if (strcmp(name,"Rect")==0) {
      if ( (next=nextobj(infile)) == arr ) {
	int i;
	for (i=0; i<4; i++) {
	  char *tmp = *infile;
	  result.Rect[i] = strtod(tmp, infile);
	}
	skipnextobj(infile,arr);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"ModDate")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.ModDate = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Action")==0) {
      if ( (next=nextobj(infile)) == lit ) {
	char *val=readlit(infile);
	if (strcmp(val,"GoTo")==0)
	  result.Action = GoTo;
	if (strcmp(val,"GoToR")==0)
	  result.Action = GoToR;
	if (strcmp(val,"Launch")==0)
	  result.Action = Launch;
	if (strcmp(val,"Article")==0)
	  result.Action = Article;
	free(val);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"Dest")==0) {
      if ( (next=nextobj(infile)) == other ) {
	result.Dest = readlit(infile);
      } else if ( next == lit ) {
	result.Dest = readlit(infile);
      } else if ( next == str ) {
	result.Dest = readstr(infile);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"Page")==0) {
      char *tmp = *infile;
      result.View.Page = strtol(tmp, infile, 10);
      if (tmp == *infile) {
	if ( (next = nextobj(infile)) == lit ) {
	  char *val = readlit(infile);
	  if (strcmp(val,"Prev")==0) {
	    result.View.Page = curpage-1;
	  } else if (strcmp(val,"Next")==0) {
	    result.View.Page = curpage+1;
	  }
	} else skipnextobj(infile,next);
      }
    } else if (strcmp(name,"View")==0)  {
      if ( (next=nextobj(infile)) == arr ) {
	int first = 1;

	if ( (next=nextobj(infile)) == other ) {
	  char *tmp = *infile;
	  result.View.Page = strtol(tmp, infile, 10);
	  if (tmp == *infile) {
	    skipnextobj(infile,other);
	  }
	  first = 0;
	} 
      reloop:
	if (next == lit) {
	  char *val = readlit(infile);
	  if (first) {
	    if (strcmp(val,"Next")==0) {
	      result.View.Page = curpage+1;
	      first = 0;
	      next=nextobj(infile);
	      free(val);
	      goto reloop;
	    } else if (strcmp(val,"Prev")==0) {
	      result.View.Page = curpage-1;
	      first = 0;
	      next=nextobj(infile);
	      free(val);
	      goto reloop;
	    }
	    if (val) free(val);
	  }
	  if (strcmp(val,"Fit")==0) {
	    result.View.View = Fit;
	  } else if (strcmp(val,"FitB")==0) {
	    result.View.View = FitB;
	  } else if (strcmp(val,"FitH")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitH;
	    }
	  } else if (strcmp(val,"FitBH")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitBH;
	    }
	  } else if (strcmp(val,"FitR")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<4; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitR;
	    }
	  } else if (strcmp(val,"FitV")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitV;
	    }
	  } else if (strcmp(val,"FitBV")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitBV;
	    }
	  } else if (strcmp(val,"XYZ")==0) {
	    int i;
	    for (i=0; i<3; i++) {
	      char *tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	      if (tmp == *infile) {
		char *val=readlit(infile);
		if (strcmp(val,"null")==i)
		  result.View.params[i] = -DBL_MIN;
		else break;
	      }
	    }
	    if (i==3) {
	      result.View.View = XYZ;
	    }
	  } 
	}
	skipnextobj(infile,arr);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"File")==0) {
      if ((next=nextobj(infile))==str) {
	if(result.View.FileName) free(result.View.FileName);
	result.View.FileName=readstr(infile);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"Op")==0) {
      if ((next=nextobj(infile))==lit) {
	char *val=readlit(infile);
	if (strcmp(val,"Print")==0)
	  result.View.Op = Print;
	free(val);
      } else skipnextobj(infile,next);
    } else if (((next=nextobj(infile))==str) &&
	       !result.View.FileName && strstr(name,"File")) {
      result.View.FileName = readstr(infile);
    } else 
      skipnextobj(infile,next);
  }
    
  if (next == earr) {
    result.type = LNK;
  } else {
    skipnextobj(infile,next);
    skipnextobj(infile,arr);
  }

  return result;
}

static struct outmark
getpdfout(char **infile)
{ struct outmark result={OTHER, NULL, NULL, NULL,
			 GoTo, NULL, {0, Fit, {0, 0, 0, 0}, NULL, Open,
					{NULL, NULL} } };
  char *name;
  enum tokentype next;
  
  while ( (next = nextobj(infile)) == lit ) {

    name=readlit(infile);

    if (strcmp(name,"Title")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Title = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Action")==0) {
      if ( (next=nextobj(infile)) == lit ) {
	char *val=readlit(infile);
	if (strcmp(val,"GoTo")==0)
	  result.Action = GoTo;
	if (strcmp(val,"GoToR")==0)
	  result.Action = GoToR;
	if (strcmp(val,"Launch")==0)
	  result.Action = Launch;
	if (strcmp(val,"Article")==0)
	  result.Action = Article;
	free(val);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"Dest")==0) {
      if ( (next=nextobj(infile)) == other ) {
	result.Dest = readlit(infile);
      } else if ( next == lit ) {
	result.Dest = readlit(infile);
      } else if ( next == str ) {
	result.Dest = readstr(infile);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"Page")==0) {
      char *tmp = *infile;
      result.View.Page = strtol(tmp, infile, 10);
      if (tmp == *infile) {
	skipnextobj(infile,nextobj(infile));
      }
    } else if (strcmp(name,"View")==0)  {
      if ( (next=nextobj(infile)) == arr ) {
	int first = 1;

	if ( (next=nextobj(infile)) == other ) {
	  char *tmp = *infile;
	  result.View.Page = strtol(tmp, infile, 10);
	  if (tmp == *infile) {
	    skipnextobj(infile,other);
	  }
	  first = 0;
	} 
	if (next == lit) {
	  char *val = readlit(infile);
	  if (strcmp(val,"Fit")==0) {
	    result.View.View = Fit;
	  } else if (strcmp(val,"FitB")==0) {
	    result.View.View = FitB;
	  } else if (strcmp(val,"FitH")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitH;
	    }
	  } else if (strcmp(val,"FitBH")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitBH;
	    }
	  } else if (strcmp(val,"FitR")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<4; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitR;
	    }
	  } else if (strcmp(val,"FitV")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitV;
	    }
	  } else if (strcmp(val,"FitBV")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitBV;
	    }
	  } else if (strcmp(val,"XYZ")==0) {
	    int i;
	    for (i=0; i<3; i++) {
	      char *tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	      if (tmp == *infile) {
		char *val=readlit(infile);
		if (strcmp(val,"null")==i)
		  result.View.params[i] = -DBL_MIN;
		else break;
	      }
	    }
	    if (i==3) {
	      result.View.View = XYZ;
	    }
	  } 
	}
	skipnextobj(infile,arr);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"File")==0) {
      if ((next=nextobj(infile))==str) {
	if(result.View.FileName) free(result.View.FileName);
	result.View.FileName=readstr(infile);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"Op")==0) {
      if ((next=nextobj(infile))==lit) {
	char *val=readlit(infile);
	if (strcmp(val,"Print")==0)
	  result.View.Op = Print;
	free(val);
      } else skipnextobj(infile,next);
    } else if (((next=nextobj(infile))==str) &&
	       !result.View.FileName && strstr(name,"File")) {
      result.View.FileName = readstr(infile);
    } else skipnextobj(infile,next);
  }

  if (next == earr) {
    result.type = OUT;
  } else {
    skipnextobj(infile,next);
    skipnextobj(infile,arr);
  }

  return result;
}

static struct artmark 
getpdfart(char **infile)
{ struct artmark result
    ={OTHER, NULL, NULL, NULL, 0, {0, 0, 0, 0}, NULL, NULL, NULL};
  char *name;
  enum tokentype next;
  
  result.SrcPage = curpage;

  while ( (next = nextobj(infile)) == lit ) {

    name=readlit(infile);

    if (strcmp(name,"Title")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Title = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Page")==0) {
      char *tmp = *infile;
      result.SrcPage = strtol(tmp, infile, 10);
      if (tmp == *infile) {
	result.SrcPage = curpage;
	skipnextobj(infile,nextobj(infile));
      }
    } else if (strcmp(name,"Rect")==0) {
      if ( (next=nextobj(infile)) == arr ) {
	int i;
	for (i=0; i<4; i++) {
	  char *tmp = *infile;
	  result.Rect[i] = strtod(tmp, infile);
	}
	skipnextobj(infile,arr);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Subject")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Subject = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Author")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Author = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Keywords")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Keywords = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else skipnextobj(infile,nextobj(infile));

  }

  if (next == earr) {
    result.type = ARTICLE;
  } else {
    skipnextobj(infile,next);
    skipnextobj(infile,arr);
  }

  return result;
}

static struct destmark 
getpdfdest(char **infile)
{ struct destmark result
    ={OTHER, NULL, NULL, NULL, {0, Fit, {0, 0, 0, 0}, NULL, Open,
					{NULL, NULL} } };
  char *name;
  enum tokentype next;
  
  while ( (next = nextobj(infile)) == lit ) {

    name=readlit(infile);

    if (strcmp(name,"Dest")==0) {
      if ( (next=nextobj(infile)) == lit ) {
	result.Dest = readlit(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Page")==0) {
      char *tmp = *infile;
      result.View.Page = strtol(tmp, infile, 10);
      if (tmp == *infile) {
	result.View.Page = 0;
	skipnextobj(infile,nextobj(infile));
      } 
    } else if (strcmp(name,"View")==0)  {
      if ( (next=nextobj(infile)) == arr ) {
	int first = 1;

	if ( (next=nextobj(infile)) == other ) {
	  char *tmp = *infile;
	  result.View.Page = strtol(tmp, infile, 10);
	  if (tmp == *infile) {
	    skipnextobj(infile,other);
	  }
	  first = 0;
	} 
	if (next == lit) {
	  char *val = readlit(infile);
	  if (strcmp(val,"Fit")==0) {
	    result.View.View = Fit;
	  } else if (strcmp(val,"FitB")==0) {
	    result.View.View = FitB;
	  } else if (strcmp(val,"FitH")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitH;
	    }
	  } else if (strcmp(val,"FitBH")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitBH;
	    }
	  } else if (strcmp(val,"FitR")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<4; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitR;
	    }
	  } else if (strcmp(val,"FitV")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitV;
	    }
	  } else if (strcmp(val,"FitBV")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitBV;
	    }
	  } else if (strcmp(val,"XYZ")==0) {
	    int i;
	    for (i=0; i<3; i++) {
	      char *tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	      if (tmp == *infile) {
		char *val=readlit(infile);
		if (strcmp(val,"null")==i)
		  result.View.params[i] = -DBL_MIN;
		else break;
	      }
	    }
	    if (i==3) {
	      result.View.View = XYZ;
	    }
	  } 
	}
	skipnextobj(infile,arr);
      } else skipnextobj(infile,next);
    } else skipnextobj(infile,nextobj(infile));
  }

  if (next == earr) {
    result.type = DEST;
  } else {
    skipnextobj(infile,next);
    skipnextobj(infile,arr);
  }

  return result;
}


static struct infomark 
getpdfinfo(char **infile)
{ struct infomark result
    ={OTHER, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL};
  char *name;
  enum tokentype next;
  
  while ( (next = nextobj(infile)) == lit ) {

    name=readlit(infile);

    if (strcmp(name,"Title")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Title = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"ModDate")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.ModDate = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Author")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Author = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"CreationDate")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.CreationDate = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Creator")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Creator = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Producer")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Producer = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Subject")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Subject = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else if (strcmp(name,"Keywords")==0) {
      if ( (next=nextobj(infile)) == str ) {
	result.Keywords = readstr(infile);
      } else {
	skipnextobj(infile,next);
      }
    } else skipnextobj(infile,nextobj(infile));
  }

  if (next == earr) {
    result.type = DOCINFO;
  } else {
    skipnextobj(infile,next);
    skipnextobj(infile,arr);
  }

  return result;
}

static struct viewmark
getpdfview(char **infile)
{ struct viewmark result={OTHER, NULL, NULL, 
			 GoTo, NULL, {0, Fit, {0, 0, 0, 0}, NULL, Open,
					{NULL, NULL} }, {NULL, NULL}, UseNone };
  char *name;
  enum tokentype next;
  
  while ( (next = nextobj(infile)) == lit ) {

    name=readlit(infile);

    if (strcmp(name,"Action")==0) {
      if ( (next=nextobj(infile)) == lit ) {
	char *val=readlit(infile);
	if (strcmp(val,"GoTo")==0)
	  result.Action = GoTo;
	if (strcmp(val,"GoToR")==0)
	  result.Action = GoToR;
	if (strcmp(val,"Launch")==0)
	  result.Action = Launch;
	if (strcmp(val,"Article")==0)
	  result.Action = Article;
	free(val);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"PageMode")==0) {
      if ( (next=nextobj(infile)) == lit ) {
	char *val=readlit(infile);
	if (strcmp(val,"UseNone")==0)
	  result.PageMode = UseNone;
	if (strcmp(val,"UseOutlines")==0)
	  result.PageMode = UseOutlines;
	if (strcmp(val,"UseThumbs")==0)
	  result.PageMode = UseThumbs;
	if (strcmp(val,"FullScreen")==0)
	  result.PageMode = FullScreen;
	free(val);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"Dest")==0) {
      if ( (next=nextobj(infile)) == other ) {
	result.Dest = readlit(infile);
      } else if ( next == lit ) {
	result.Dest = readlit(infile);
      } else if ( next == str ) {
	result.Dest = readstr(infile);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"Page")==0) {
      char *tmp = *infile;
      result.View.Page = strtol(tmp, infile, 10);
      if (tmp == *infile) {
	skipnextobj(infile,nextobj(infile));
      }
    } else if (strcmp(name,"View")==0)  {
      if ( (next=nextobj(infile)) == arr ) {
	int first = 1;

	if ( (next=nextobj(infile)) == other ) {
	  char *tmp = *infile;
	  result.View.Page = strtol(tmp, infile, 10);
	  if (tmp == *infile) {
	    skipnextobj(infile,other);
	  }
	  first = 0;
	} 
	if (next == lit) {
	  char *val = readlit(infile);
	  if (strcmp(val,"Fit")==0) {
	    result.View.View = Fit;
	  } else if (strcmp(val,"FitB")==0) {
	    result.View.View = FitB;
	  } else if (strcmp(val,"FitH")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitH;
	    }
	  } else if (strcmp(val,"FitBH")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitBH;
	    }
	  } else if (strcmp(val,"FitR")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<4; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitR;
	    }
	  } else if (strcmp(val,"FitV")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitV;
	    }
	  } else if (strcmp(val,"FitBV")==0) {
	    int i;
	    char *tmp;
	    for (i=0; i<1; i++) {
	      tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	    }
	    if (tmp != *infile) {
	      result.View.View = FitBV;
	    }
	  } else if (strcmp(val,"XYZ")==0) {
	    int i;
	    for (i=0; i<3; i++) {
	      char *tmp = *infile;
	      result.View.params[i] = strtod(tmp, infile);
	      if (tmp == *infile) {
		char *val=readlit(infile);
		if (strcmp(val,"null")==i)
		  result.View.params[i] = -DBL_MIN;
		else break;
	      }
	    }
	    if (i==3) {
	      result.View.View = XYZ;
	    }
	  } 
	}
	skipnextobj(infile,arr);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"File")==0) {
      if ((next=nextobj(infile))==str) {
	if(result.View.FileName) free(result.View.FileName);
	result.View.FileName=readstr(infile);
      } else skipnextobj(infile,next);
    } else if (strcmp(name,"Op")==0) {
      if ((next=nextobj(infile))==lit) {
	char *val=readlit(infile);
	if (strcmp(val,"Print")==0)
	  result.View.Op = Print;
	free(val);
      } else skipnextobj(infile,next);
    } else if (((next=nextobj(infile))==str) &&
	       !result.View.FileName && strstr(name,"File")) {
      result.View.FileName = readstr(infile);
    } else skipnextobj(infile,next);
  }

  if (next == earr) {
    result.type = DOCVIEW;
  } else {
    skipnextobj(infile,next);
    skipnextobj(infile,arr);
  }

  return result;
}

union pdfmark *
getpdfmark(char **infile)
{
  static union pdfmark result;

  result.flag.type = OTHER;
  result.flag.next = NULL;
  result.flag.prev = NULL;
  
  switch(linktype(infile)) {
  case ANN:
    result.pdfann = getpdfann(infile); break;
  case LNK:
    result.pdflnk = getpdflnk(infile); break;
  case OUT:
    result.pdfout = getpdfout(infile); break;
  case ARTICLE:
    result.pdfart = getpdfart(infile); break;
  case DEST:
    result.pdfdest = getpdfdest(infile); break;
  case DOCINFO:
    result.pdfinfo = getpdfinfo(infile); break;
  case DOCVIEW:
    result.pdfview = getpdfview(infile); break;
  default:
    break;
  }

  return (result.flag.type == OTHER)? NULL : &result;
}
