namespace FunStripe

open FunStripe.Json
open StripeModel
open System

module StripeRequest =

    module ThreeDSecure =

        type CreateOptions = {
            ///Amount of the charge that you will create when authentication completes.
            [<Config.Form>]Amount: int
            ///The ID of a card token, or the ID of a card belonging to the given customer.
            [<Config.Form>]Card: string option
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string
            ///The customer associated with this 3D secure authentication.
            [<Config.Form>]Customer: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The URL that the cardholder's browser will be returned to when authentication completes.
            [<Config.Form>]ReturnUrl: string
        }
        with
            static member New(amount: int, currency: string, returnUrl: string, ?card: string, ?customer: string, ?expand: string list) =
                {
                    Amount = amount
                    Card = card
                    Currency = currency
                    Customer = customer
                    Expand = expand
                    ReturnUrl = returnUrl
                }

        ///<p>Initiate 3D Secure authentication.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/3d_secure"
            |> RestApi.postAsync<_, ThreeDSecure> settings (Map.empty) options

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]ThreeDSecure: string
        }
        with
            static member New(threeDSecure: string, ?expand: string list) =
                {
                    Expand = expand
                    ThreeDSecure = threeDSecure
                }

        ///<p>Retrieves a 3D Secure object.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/3d_secure/{options.ThreeDSecure}"
            |> RestApi.getAsync<ThreeDSecure> settings qs

    module Account =

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(?expand: string list) =
                {
                    Expand = expand
                }

        ///<p>Retrieves the details of an account.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/account"
            |> RestApi.getAsync<Account> settings qs

    module AccountLinks =

        type Create'Collect =
        | CurrentlyDue
        | EventuallyDue

        type Create'Type =
        | AccountOnboarding
        | AccountUpdate
        | CustomAccountUpdate
        | CustomAccountVerification

        type CreateOptions = {
            ///The identifier of the account to create an account link for.
            [<Config.Form>]Account: string
            ///Which information the platform needs to collect from the user. One of `currently_due` or `eventually_due`. Default is `currently_due`.
            [<Config.Form>]Collect: Create'Collect option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The URL the user will be redirected to if the account link is expired, has been previously-visited, or is otherwise invalid. The URL you specify should attempt to generate a new account link with the same parameters used to create the original account link, then redirect the user to the new account link's URL so they can continue with Connect Onboarding. If a new account link cannot be generated or the redirect fails you should display a useful error to the user.
            [<Config.Form>]RefreshUrl: string option
            ///The URL that the user will be redirected to upon leaving or completing the linked flow.
            [<Config.Form>]ReturnUrl: string option
            ///The type of account link the user is requesting. Possible values are `account_onboarding` or `account_update`.
            [<Config.Form>]Type: Create'Type
        }
        with
            static member New(account: string, type': Create'Type, ?collect: Create'Collect, ?expand: string list, ?refreshUrl: string, ?returnUrl: string) =
                {
                    Account = account
                    Collect = collect
                    Expand = expand
                    RefreshUrl = refreshUrl
                    ReturnUrl = returnUrl
                    Type = type'
                }

        ///<p>Creates an AccountLink object that includes a single-use Stripe URL that the platform can redirect their user to in order to take them through the Connect Onboarding flow.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/account_links"
            |> RestApi.postAsync<_, AccountLink> settings (Map.empty) options

    module Accounts =

        type ListOptions = {
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of accounts connected to your platform via <a href="/docs/connect">Connect</a>. If you’re not a platform, the list is empty.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/accounts"
            |> RestApi.getAsync<Account list> settings qs

        type Create'BusinessProfileSupportAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'BusinessProfile = {
            ///[The merchant category code for the account](https://stripe.com/docs/connect/setting-mcc). MCCs are used to classify businesses based on the goods or services they provide.
            [<Config.Form>]Mcc: string option
            ///The customer-facing business name.
            [<Config.Form>]Name: string option
            ///Internal-only description of the product sold by, or service provided by, the business. Used by Stripe for risk and underwriting purposes.
            [<Config.Form>]ProductDescription: string option
            ///A publicly available mailing address for sending support issues to.
            [<Config.Form>]SupportAddress: Create'BusinessProfileSupportAddress option
            ///A publicly available email address for sending support issues to.
            [<Config.Form>]SupportEmail: string option
            ///A publicly available phone number to call with support issues.
            [<Config.Form>]SupportPhone: string option
            ///A publicly available website for handling support issues.
            [<Config.Form>]SupportUrl: Choice<string,string> option
            ///The business's publicly available website.
            [<Config.Form>]Url: string option
        }
        with
            static member New(?mcc: string, ?name: string, ?productDescription: string, ?supportAddress: Create'BusinessProfileSupportAddress, ?supportEmail: string, ?supportPhone: string, ?supportUrl: Choice<string,string>, ?url: string) =
                {
                    Mcc = mcc
                    Name = name
                    ProductDescription = productDescription
                    SupportAddress = supportAddress
                    SupportEmail = supportEmail
                    SupportPhone = supportPhone
                    SupportUrl = supportUrl
                    Url = url
                }

        type Create'CapabilitiesAcssDebitPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesAfterpayClearpayPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesAuBecsDebitPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesBacsDebitPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesBancontactPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesBoletoPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesCardIssuing = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesCardPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesCartesBancairesPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesEpsPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesFpxPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesGiropayPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesGrabpayPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesIdealPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesJcbPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesLegacyPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesOxxoPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesP24Payments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesSepaDebitPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesSofortPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesTaxReportingUs1099K = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesTaxReportingUs1099Misc = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'CapabilitiesTransfers = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Create'Capabilities = {
            ///The acss_debit_payments capability.
            [<Config.Form>]AcssDebitPayments: Create'CapabilitiesAcssDebitPayments option
            ///The afterpay_clearpay_payments capability.
            [<Config.Form>]AfterpayClearpayPayments: Create'CapabilitiesAfterpayClearpayPayments option
            ///The au_becs_debit_payments capability.
            [<Config.Form>]AuBecsDebitPayments: Create'CapabilitiesAuBecsDebitPayments option
            ///The bacs_debit_payments capability.
            [<Config.Form>]BacsDebitPayments: Create'CapabilitiesBacsDebitPayments option
            ///The bancontact_payments capability.
            [<Config.Form>]BancontactPayments: Create'CapabilitiesBancontactPayments option
            ///The boleto_payments capability.
            [<Config.Form>]BoletoPayments: Create'CapabilitiesBoletoPayments option
            ///The card_issuing capability.
            [<Config.Form>]CardIssuing: Create'CapabilitiesCardIssuing option
            ///The card_payments capability.
            [<Config.Form>]CardPayments: Create'CapabilitiesCardPayments option
            ///The cartes_bancaires_payments capability.
            [<Config.Form>]CartesBancairesPayments: Create'CapabilitiesCartesBancairesPayments option
            ///The eps_payments capability.
            [<Config.Form>]EpsPayments: Create'CapabilitiesEpsPayments option
            ///The fpx_payments capability.
            [<Config.Form>]FpxPayments: Create'CapabilitiesFpxPayments option
            ///The giropay_payments capability.
            [<Config.Form>]GiropayPayments: Create'CapabilitiesGiropayPayments option
            ///The grabpay_payments capability.
            [<Config.Form>]GrabpayPayments: Create'CapabilitiesGrabpayPayments option
            ///The ideal_payments capability.
            [<Config.Form>]IdealPayments: Create'CapabilitiesIdealPayments option
            ///The jcb_payments capability.
            [<Config.Form>]JcbPayments: Create'CapabilitiesJcbPayments option
            ///The legacy_payments capability.
            [<Config.Form>]LegacyPayments: Create'CapabilitiesLegacyPayments option
            ///The oxxo_payments capability.
            [<Config.Form>]OxxoPayments: Create'CapabilitiesOxxoPayments option
            ///The p24_payments capability.
            [<Config.Form>]P24Payments: Create'CapabilitiesP24Payments option
            ///The sepa_debit_payments capability.
            [<Config.Form>]SepaDebitPayments: Create'CapabilitiesSepaDebitPayments option
            ///The sofort_payments capability.
            [<Config.Form>]SofortPayments: Create'CapabilitiesSofortPayments option
            ///The tax_reporting_us_1099_k capability.
            [<Config.Form>]TaxReportingUs1099K: Create'CapabilitiesTaxReportingUs1099K option
            ///The tax_reporting_us_1099_misc capability.
            [<Config.Form>]TaxReportingUs1099Misc: Create'CapabilitiesTaxReportingUs1099Misc option
            ///The transfers capability.
            [<Config.Form>]Transfers: Create'CapabilitiesTransfers option
        }
        with
            static member New(?acssDebitPayments: Create'CapabilitiesAcssDebitPayments, ?taxReportingUs1099K: Create'CapabilitiesTaxReportingUs1099K, ?sofortPayments: Create'CapabilitiesSofortPayments, ?sepaDebitPayments: Create'CapabilitiesSepaDebitPayments, ?p24Payments: Create'CapabilitiesP24Payments, ?oxxoPayments: Create'CapabilitiesOxxoPayments, ?legacyPayments: Create'CapabilitiesLegacyPayments, ?jcbPayments: Create'CapabilitiesJcbPayments, ?idealPayments: Create'CapabilitiesIdealPayments, ?grabpayPayments: Create'CapabilitiesGrabpayPayments, ?taxReportingUs1099Misc: Create'CapabilitiesTaxReportingUs1099Misc, ?giropayPayments: Create'CapabilitiesGiropayPayments, ?epsPayments: Create'CapabilitiesEpsPayments, ?cartesBancairesPayments: Create'CapabilitiesCartesBancairesPayments, ?cardPayments: Create'CapabilitiesCardPayments, ?cardIssuing: Create'CapabilitiesCardIssuing, ?boletoPayments: Create'CapabilitiesBoletoPayments, ?bancontactPayments: Create'CapabilitiesBancontactPayments, ?bacsDebitPayments: Create'CapabilitiesBacsDebitPayments, ?auBecsDebitPayments: Create'CapabilitiesAuBecsDebitPayments, ?afterpayClearpayPayments: Create'CapabilitiesAfterpayClearpayPayments, ?fpxPayments: Create'CapabilitiesFpxPayments, ?transfers: Create'CapabilitiesTransfers) =
                {
                    AcssDebitPayments = acssDebitPayments
                    AfterpayClearpayPayments = afterpayClearpayPayments
                    AuBecsDebitPayments = auBecsDebitPayments
                    BacsDebitPayments = bacsDebitPayments
                    BancontactPayments = bancontactPayments
                    BoletoPayments = boletoPayments
                    CardIssuing = cardIssuing
                    CardPayments = cardPayments
                    CartesBancairesPayments = cartesBancairesPayments
                    EpsPayments = epsPayments
                    FpxPayments = fpxPayments
                    GiropayPayments = giropayPayments
                    GrabpayPayments = grabpayPayments
                    IdealPayments = idealPayments
                    JcbPayments = jcbPayments
                    LegacyPayments = legacyPayments
                    OxxoPayments = oxxoPayments
                    P24Payments = p24Payments
                    SepaDebitPayments = sepaDebitPayments
                    SofortPayments = sofortPayments
                    TaxReportingUs1099K = taxReportingUs1099K
                    TaxReportingUs1099Misc = taxReportingUs1099Misc
                    Transfers = transfers
                }

        type Create'CompanyAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'CompanyAddressKana = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Create'CompanyAddressKanji = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Create'CompanyVerificationDocument = {
            ///The back of a document returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `additional_verification`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Back: string option
            ///The front of a document returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `additional_verification`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Create'CompanyVerification = {
            ///A document verifying the business.
            [<Config.Form>]Document: Create'CompanyVerificationDocument option
        }
        with
            static member New(?document: Create'CompanyVerificationDocument) =
                {
                    Document = document
                }

        type Create'CompanyStructure =
        | FreeZoneEstablishment
        | FreeZoneLlc
        | GovernmentInstrumentality
        | GovernmentalUnit
        | IncorporatedNonProfit
        | LimitedLiabilityPartnership
        | Llc
        | MultiMemberLlc
        | PrivateCompany
        | PrivateCorporation
        | PrivatePartnership
        | PublicCompany
        | PublicCorporation
        | PublicPartnership
        | SingleMemberLlc
        | SoleEstablishment
        | SoleProprietorship
        | TaxExemptGovernmentInstrumentality
        | UnincorporatedAssociation
        | UnincorporatedNonProfit

        type Create'Company = {
            ///The company's primary address.
            [<Config.Form>]Address: Create'CompanyAddress option
            ///The Kana variation of the company's primary address (Japan only).
            [<Config.Form>]AddressKana: Create'CompanyAddressKana option
            ///The Kanji variation of the company's primary address (Japan only).
            [<Config.Form>]AddressKanji: Create'CompanyAddressKanji option
            ///Whether the company's directors have been provided. Set this Boolean to `true` after creating all the company's directors with [the Persons API](https://stripe.com/docs/api/persons) for accounts with a `relationship.director` requirement. This value is not automatically set to `true` after creating directors, so it needs to be updated to indicate all directors have been provided.
            [<Config.Form>]DirectorsProvided: bool option
            ///Whether the company's executives have been provided. Set this Boolean to `true` after creating all the company's executives with [the Persons API](https://stripe.com/docs/api/persons) for accounts with a `relationship.executive` requirement.
            [<Config.Form>]ExecutivesProvided: bool option
            ///The company's legal name.
            [<Config.Form>]Name: string option
            ///The Kana variation of the company's legal name (Japan only).
            [<Config.Form>]NameKana: string option
            ///The Kanji variation of the company's legal name (Japan only).
            [<Config.Form>]NameKanji: string option
            ///Whether the company's owners have been provided. Set this Boolean to `true` after creating all the company's owners with [the Persons API](https://stripe.com/docs/api/persons) for accounts with a `relationship.owner` requirement.
            [<Config.Form>]OwnersProvided: bool option
            ///The company's phone number (used for verification).
            [<Config.Form>]Phone: string option
            ///The identification number given to a company when it is registered or incorporated, if distinct from the identification number used for filing taxes. (Examples are the CIN for companies and LLP IN for partnerships in India, and the Company Registration Number in Hong Kong).
            [<Config.Form>]RegistrationNumber: string option
            ///The category identifying the legal structure of the company or legal entity. See [Business structure](https://stripe.com/docs/connect/identity-verification#business-structure) for more details.
            [<Config.Form>]Structure: Create'CompanyStructure option
            ///The business ID number of the company, as appropriate for the company’s country. (Examples are an Employer ID Number in the U.S., a Business Number in Canada, or a Company Number in the UK.)
            [<Config.Form>]TaxId: string option
            ///The jurisdiction in which the `tax_id` is registered (Germany-based companies only).
            [<Config.Form>]TaxIdRegistrar: string option
            ///The VAT number of the company.
            [<Config.Form>]VatId: string option
            ///Information on the verification state of the company.
            [<Config.Form>]Verification: Create'CompanyVerification option
        }
        with
            static member New(?address: Create'CompanyAddress, ?addressKana: Create'CompanyAddressKana, ?addressKanji: Create'CompanyAddressKanji, ?directorsProvided: bool, ?executivesProvided: bool, ?name: string, ?nameKana: string, ?nameKanji: string, ?ownersProvided: bool, ?phone: string, ?registrationNumber: string, ?structure: Create'CompanyStructure, ?taxId: string, ?taxIdRegistrar: string, ?vatId: string, ?verification: Create'CompanyVerification) =
                {
                    Address = address
                    AddressKana = addressKana
                    AddressKanji = addressKanji
                    DirectorsProvided = directorsProvided
                    ExecutivesProvided = executivesProvided
                    Name = name
                    NameKana = nameKana
                    NameKanji = nameKanji
                    OwnersProvided = ownersProvided
                    Phone = phone
                    RegistrationNumber = registrationNumber
                    Structure = structure
                    TaxId = taxId
                    TaxIdRegistrar = taxIdRegistrar
                    VatId = vatId
                    Verification = verification
                }

        type Create'DocumentsBankAccountOwnershipVerification = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Create'DocumentsCompanyLicense = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Create'DocumentsCompanyMemorandumOfAssociation = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Create'DocumentsCompanyMinisterialDecree = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Create'DocumentsCompanyRegistrationVerification = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Create'DocumentsCompanyTaxIdVerification = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Create'Documents = {
            ///One or more documents that support the [Bank account ownership verification](https://support.stripe.com/questions/bank-account-ownership-verification) requirement. Must be a document associated with the account’s primary active bank account that displays the last 4 digits of the account number, either a statement or a voided check.
            [<Config.Form>]BankAccountOwnershipVerification: Create'DocumentsBankAccountOwnershipVerification option
            ///One or more documents that demonstrate proof of a company's license to operate.
            [<Config.Form>]CompanyLicense: Create'DocumentsCompanyLicense option
            ///One or more documents showing the company's Memorandum of Association.
            [<Config.Form>]CompanyMemorandumOfAssociation: Create'DocumentsCompanyMemorandumOfAssociation option
            ///(Certain countries only) One or more documents showing the ministerial decree legalizing the company's establishment.
            [<Config.Form>]CompanyMinisterialDecree: Create'DocumentsCompanyMinisterialDecree option
            ///One or more documents that demonstrate proof of a company's registration with the appropriate local authorities.
            [<Config.Form>]CompanyRegistrationVerification: Create'DocumentsCompanyRegistrationVerification option
            ///One or more documents that demonstrate proof of a company's tax ID.
            [<Config.Form>]CompanyTaxIdVerification: Create'DocumentsCompanyTaxIdVerification option
        }
        with
            static member New(?bankAccountOwnershipVerification: Create'DocumentsBankAccountOwnershipVerification, ?companyLicense: Create'DocumentsCompanyLicense, ?companyMemorandumOfAssociation: Create'DocumentsCompanyMemorandumOfAssociation, ?companyMinisterialDecree: Create'DocumentsCompanyMinisterialDecree, ?companyRegistrationVerification: Create'DocumentsCompanyRegistrationVerification, ?companyTaxIdVerification: Create'DocumentsCompanyTaxIdVerification) =
                {
                    BankAccountOwnershipVerification = bankAccountOwnershipVerification
                    CompanyLicense = companyLicense
                    CompanyMemorandumOfAssociation = companyMemorandumOfAssociation
                    CompanyMinisterialDecree = companyMinisterialDecree
                    CompanyRegistrationVerification = companyRegistrationVerification
                    CompanyTaxIdVerification = companyTaxIdVerification
                }

        type Create'IndividualAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'IndividualAddressKana = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Create'IndividualAddressKanji = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Create'IndividualDobDateOfBirthSpecs = {
            ///The day of birth, between 1 and 31.
            [<Config.Form>]Day: int option
            ///The month of birth, between 1 and 12.
            [<Config.Form>]Month: int option
            ///The four-digit year of birth.
            [<Config.Form>]Year: int option
        }
        with
            static member New(?day: int, ?month: int, ?year: int) =
                {
                    Day = day
                    Month = month
                    Year = year
                }

        type Create'IndividualVerificationAdditionalDocument = {
            ///The back of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Back: string option
            ///The front of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Create'IndividualVerificationDocument = {
            ///The back of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Back: string option
            ///The front of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Create'IndividualVerification = {
            ///A document showing address, either a passport, local ID card, or utility bill from a well-known utility company.
            [<Config.Form>]AdditionalDocument: Create'IndividualVerificationAdditionalDocument option
            ///An identifying document, either a passport or local ID card.
            [<Config.Form>]Document: Create'IndividualVerificationDocument option
        }
        with
            static member New(?additionalDocument: Create'IndividualVerificationAdditionalDocument, ?document: Create'IndividualVerificationDocument) =
                {
                    AdditionalDocument = additionalDocument
                    Document = document
                }

        type Create'IndividualPoliticalExposure =
        | Existing
        | None'

        type Create'Individual = {
            ///The individual's primary address.
            [<Config.Form>]Address: Create'IndividualAddress option
            ///The Kana variation of the the individual's primary address (Japan only).
            [<Config.Form>]AddressKana: Create'IndividualAddressKana option
            ///The Kanji variation of the the individual's primary address (Japan only).
            [<Config.Form>]AddressKanji: Create'IndividualAddressKanji option
            ///The individual's date of birth.
            [<Config.Form>]Dob: Choice<Create'IndividualDobDateOfBirthSpecs,string> option
            ///The individual's email address.
            [<Config.Form>]Email: string option
            ///The individual's first name.
            [<Config.Form>]FirstName: string option
            ///The Kana variation of the the individual's first name (Japan only).
            [<Config.Form>]FirstNameKana: string option
            ///The Kanji variation of the individual's first name (Japan only).
            [<Config.Form>]FirstNameKanji: string option
            ///The individual's gender (International regulations require either "male" or "female").
            [<Config.Form>]Gender: string option
            ///The government-issued ID number of the individual, as appropriate for the representative’s country. (Examples are a Social Security Number in the U.S., or a Social Insurance Number in Canada). Instead of the number itself, you can also provide a [PII token created with Stripe.js](https://stripe.com/docs/stripe.js#collecting-pii-data).
            [<Config.Form>]IdNumber: string option
            ///The individual's last name.
            [<Config.Form>]LastName: string option
            ///The Kana varation of the individual's last name (Japan only).
            [<Config.Form>]LastNameKana: string option
            ///The Kanji varation of the individual's last name (Japan only).
            [<Config.Form>]LastNameKanji: string option
            ///The individual's maiden name.
            [<Config.Form>]MaidenName: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The individual's phone number.
            [<Config.Form>]Phone: string option
            ///Indicates if the person or any of their representatives, family members, or other closely related persons, declares that they hold or have held an important public job or function, in any jurisdiction.
            [<Config.Form>]PoliticalExposure: Create'IndividualPoliticalExposure option
            ///The last four digits of the individual's Social Security Number (U.S. only).
            [<Config.Form>]SsnLast4: string option
            ///The individual's verification document information.
            [<Config.Form>]Verification: Create'IndividualVerification option
        }
        with
            static member New(?address: Create'IndividualAddress, ?politicalExposure: Create'IndividualPoliticalExposure, ?phone: string, ?metadata: Map<string, string>, ?maidenName: string, ?lastNameKanji: string, ?lastNameKana: string, ?lastName: string, ?ssnLast4: string, ?idNumber: string, ?firstNameKanji: string, ?firstNameKana: string, ?firstName: string, ?email: string, ?dob: Choice<Create'IndividualDobDateOfBirthSpecs,string>, ?addressKanji: Create'IndividualAddressKanji, ?addressKana: Create'IndividualAddressKana, ?gender: string, ?verification: Create'IndividualVerification) =
                {
                    Address = address
                    AddressKana = addressKana
                    AddressKanji = addressKanji
                    Dob = dob
                    Email = email
                    FirstName = firstName
                    FirstNameKana = firstNameKana
                    FirstNameKanji = firstNameKanji
                    Gender = gender
                    IdNumber = idNumber
                    LastName = lastName
                    LastNameKana = lastNameKana
                    LastNameKanji = lastNameKanji
                    MaidenName = maidenName
                    Metadata = metadata
                    Phone = phone
                    PoliticalExposure = politicalExposure
                    SsnLast4 = ssnLast4
                    Verification = verification
                }

        type Create'SettingsBranding = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) An icon for the account. Must be square and at least 128px x 128px.
            [<Config.Form>]Icon: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) A logo for the account that will be used in Checkout instead of the icon and without the account's name next to it if provided. Must be at least 128px x 128px.
            [<Config.Form>]Logo: string option
            ///A CSS hex color value representing the primary branding color for this account.
            [<Config.Form>]PrimaryColor: string option
            ///A CSS hex color value representing the secondary branding color for this account.
            [<Config.Form>]SecondaryColor: string option
        }
        with
            static member New(?icon: string, ?logo: string, ?primaryColor: string, ?secondaryColor: string) =
                {
                    Icon = icon
                    Logo = logo
                    PrimaryColor = primaryColor
                    SecondaryColor = secondaryColor
                }

        type Create'SettingsCardIssuingTosAcceptance = {
            ///The Unix timestamp marking when the account representative accepted the service agreement.
            [<Config.Form>]Date: DateTime option
            ///The IP address from which the account representative accepted the service agreement.
            [<Config.Form>]Ip: string option
            ///The user agent of the browser from which the account representative accepted the service agreement.
            [<Config.Form>]UserAgent: string option
        }
        with
            static member New(?date: DateTime, ?ip: string, ?userAgent: string) =
                {
                    Date = date
                    Ip = ip
                    UserAgent = userAgent
                }

        type Create'SettingsCardIssuing = {
            ///Details on the account's acceptance of the [Stripe Issuing Terms and Disclosures](https://stripe.com/docs/issuing/connect/tos_acceptance).
            [<Config.Form>]TosAcceptance: Create'SettingsCardIssuingTosAcceptance option
        }
        with
            static member New(?tosAcceptance: Create'SettingsCardIssuingTosAcceptance) =
                {
                    TosAcceptance = tosAcceptance
                }

        type Create'SettingsCardPaymentsDeclineOn = {
            ///Whether Stripe automatically declines charges with an incorrect ZIP or postal code. This setting only applies when a ZIP or postal code is provided and they fail bank verification.
            [<Config.Form>]AvsFailure: bool option
            ///Whether Stripe automatically declines charges with an incorrect CVC. This setting only applies when a CVC is provided and it fails bank verification.
            [<Config.Form>]CvcFailure: bool option
        }
        with
            static member New(?avsFailure: bool, ?cvcFailure: bool) =
                {
                    AvsFailure = avsFailure
                    CvcFailure = cvcFailure
                }

        type Create'SettingsCardPayments = {
            ///Automatically declines certain charge types regardless of whether the card issuer accepted or declined the charge.
            [<Config.Form>]DeclineOn: Create'SettingsCardPaymentsDeclineOn option
            ///The default text that appears on credit card statements when a charge is made. This field prefixes any dynamic `statement_descriptor` specified on the charge. `statement_descriptor_prefix` is useful for maximizing descriptor space for the dynamic portion.
            [<Config.Form>]StatementDescriptorPrefix: string option
        }
        with
            static member New(?declineOn: Create'SettingsCardPaymentsDeclineOn, ?statementDescriptorPrefix: string) =
                {
                    DeclineOn = declineOn
                    StatementDescriptorPrefix = statementDescriptorPrefix
                }

        type Create'SettingsPayments = {
            ///The default text that appears on credit card statements when a charge is made. This field prefixes any dynamic `statement_descriptor` specified on the charge.
            [<Config.Form>]StatementDescriptor: string option
            ///The Kana variation of the default text that appears on credit card statements when a charge is made (Japan only).
            [<Config.Form>]StatementDescriptorKana: string option
            ///The Kanji variation of the default text that appears on credit card statements when a charge is made (Japan only).
            [<Config.Form>]StatementDescriptorKanji: string option
        }
        with
            static member New(?statementDescriptor: string, ?statementDescriptorKana: string, ?statementDescriptorKanji: string) =
                {
                    StatementDescriptor = statementDescriptor
                    StatementDescriptorKana = statementDescriptorKana
                    StatementDescriptorKanji = statementDescriptorKanji
                }

        type Create'SettingsPayoutsScheduleDelayDays =
        | Minimum

        type Create'SettingsPayoutsScheduleInterval =
        | Daily
        | Manual
        | Monthly
        | Weekly

        type Create'SettingsPayoutsScheduleWeeklyAnchor =
        | Friday
        | Monday
        | Saturday
        | Sunday
        | Thursday
        | Tuesday
        | Wednesday

        type Create'SettingsPayoutsSchedule = {
            ///The number of days charge funds are held before being paid out. May also be set to `minimum`, representing the lowest available value for the account country. Default is `minimum`. The `delay_days` parameter does not apply when the `interval` is `manual`.
            [<Config.Form>]DelayDays: Choice<Create'SettingsPayoutsScheduleDelayDays,int> option
            ///How frequently available funds are paid out. One of: `daily`, `manual`, `weekly`, or `monthly`. Default is `daily`.
            [<Config.Form>]Interval: Create'SettingsPayoutsScheduleInterval option
            ///The day of the month when available funds are paid out, specified as a number between 1--31. Payouts nominally scheduled between the 29th and 31st of the month are instead sent on the last day of a shorter month. Required and applicable only if `interval` is `monthly`.
            [<Config.Form>]MonthlyAnchor: int option
            ///The day of the week when available funds are paid out, specified as `monday`, `tuesday`, etc. (required and applicable only if `interval` is `weekly`.)
            [<Config.Form>]WeeklyAnchor: Create'SettingsPayoutsScheduleWeeklyAnchor option
        }
        with
            static member New(?delayDays: Choice<Create'SettingsPayoutsScheduleDelayDays,int>, ?interval: Create'SettingsPayoutsScheduleInterval, ?monthlyAnchor: int, ?weeklyAnchor: Create'SettingsPayoutsScheduleWeeklyAnchor) =
                {
                    DelayDays = delayDays
                    Interval = interval
                    MonthlyAnchor = monthlyAnchor
                    WeeklyAnchor = weeklyAnchor
                }

        type Create'SettingsPayouts = {
            ///A Boolean indicating whether Stripe should try to reclaim negative balances from an attached bank account. For details, see [Understanding Connect Account Balances](https://stripe.com/docs/connect/account-balances).
            [<Config.Form>]DebitNegativeBalances: bool option
            ///Details on when funds from charges are available, and when they are paid out to an external account. For details, see our [Setting Bank and Debit Card Payouts](https://stripe.com/docs/connect/bank-transfers#payout-information) documentation.
            [<Config.Form>]Schedule: Create'SettingsPayoutsSchedule option
            ///The text that appears on the bank account statement for payouts. If not set, this defaults to the platform's bank descriptor as set in the Dashboard.
            [<Config.Form>]StatementDescriptor: string option
        }
        with
            static member New(?debitNegativeBalances: bool, ?schedule: Create'SettingsPayoutsSchedule, ?statementDescriptor: string) =
                {
                    DebitNegativeBalances = debitNegativeBalances
                    Schedule = schedule
                    StatementDescriptor = statementDescriptor
                }

        type Create'Settings = {
            ///Settings used to apply the account's branding to email receipts, invoices, Checkout, and other products.
            [<Config.Form>]Branding: Create'SettingsBranding option
            ///Settings specific to the account's use of the Card Issuing product.
            [<Config.Form>]CardIssuing: Create'SettingsCardIssuing option
            ///Settings specific to card charging on the account.
            [<Config.Form>]CardPayments: Create'SettingsCardPayments option
            ///Settings that apply across payment methods for charging on the account.
            [<Config.Form>]Payments: Create'SettingsPayments option
            ///Settings specific to the account's payouts.
            [<Config.Form>]Payouts: Create'SettingsPayouts option
        }
        with
            static member New(?branding: Create'SettingsBranding, ?cardIssuing: Create'SettingsCardIssuing, ?cardPayments: Create'SettingsCardPayments, ?payments: Create'SettingsPayments, ?payouts: Create'SettingsPayouts) =
                {
                    Branding = branding
                    CardIssuing = cardIssuing
                    CardPayments = cardPayments
                    Payments = payments
                    Payouts = payouts
                }

        type Create'TosAcceptance = {
            ///The Unix timestamp marking when the account representative accepted their service agreement.
            [<Config.Form>]Date: DateTime option
            ///The IP address from which the account representative accepted their service agreement.
            [<Config.Form>]Ip: string option
            ///The user's service agreement type.
            [<Config.Form>]ServiceAgreement: string option
            ///The user agent of the browser from which the account representative accepted their service agreement.
            [<Config.Form>]UserAgent: string option
        }
        with
            static member New(?date: DateTime, ?ip: string, ?serviceAgreement: string, ?userAgent: string) =
                {
                    Date = date
                    Ip = ip
                    ServiceAgreement = serviceAgreement
                    UserAgent = userAgent
                }

        type Create'BusinessType =
        | Company
        | GovernmentEntity
        | Individual
        | NonProfit

        type Create'Type =
        | Custom
        | Express
        | Standard

        type CreateOptions = {
            ///An [account token](https://stripe.com/docs/api#create_account_token), used to securely provide details to the account.
            [<Config.Form>]AccountToken: string option
            ///Business information about the account.
            [<Config.Form>]BusinessProfile: Create'BusinessProfile option
            ///The business type.
            [<Config.Form>]BusinessType: Create'BusinessType option
            ///Each key of the dictionary represents a capability, and each capability maps to its settings (e.g. whether it has been requested or not). Each capability will be inactive until you have provided its specific requirements and Stripe has verified them. An account may have some of its requested capabilities be active and some be inactive.
            [<Config.Form>]Capabilities: Create'Capabilities option
            ///Information about the company or business. This field is available for any `business_type`.
            [<Config.Form>]Company: Create'Company option
            ///The country in which the account holder resides, or in which the business is legally established. This should be an ISO 3166-1 alpha-2 country code. For example, if you are in the United States and the business for which you're creating an account is legally represented in Canada, you would use `CA` as the country for the account being created.
            [<Config.Form>]Country: string option
            ///Three-letter ISO currency code representing the default currency for the account. This must be a currency that [Stripe supports in the account's country](https://stripe.com/docs/payouts).
            [<Config.Form>]DefaultCurrency: string option
            ///Documents that may be submitted to satisfy various informational requests.
            [<Config.Form>]Documents: Create'Documents option
            ///The email address of the account holder. This is only to make the account easier to identify to you. Stripe will never directly email Custom accounts.
            [<Config.Form>]Email: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///A card or bank account to attach to the account for receiving [payouts](https://stripe.com/docs/connect/bank-debit-card-payouts) (you won’t be able to use it for top-ups). You can provide either a token, like the ones returned by [Stripe.js](https://stripe.com/docs/stripe.js), or a dictionary, as documented in the `external_account` parameter for [bank account](https://stripe.com/docs/api#account_create_bank_account) creation. <br><br>By default, providing an external account sets it as the new default external account for its currency, and deletes the old default if one exists. To add additional external accounts without replacing the existing default for the currency, use the bank account or card creation API.
            [<Config.Form>]ExternalAccount: string option
            ///Information about the person represented by the account. This field is null unless `business_type` is set to `individual`.
            [<Config.Form>]Individual: Create'Individual option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Options for customizing how the account functions within Stripe.
            [<Config.Form>]Settings: Create'Settings option
            ///Details on the account's acceptance of the [Stripe Services Agreement](https://stripe.com/docs/connect/updating-accounts#tos-acceptance).
            [<Config.Form>]TosAcceptance: Create'TosAcceptance option
            ///The type of Stripe account to create. May be one of `custom`, `express` or `standard`.
            [<Config.Form>]Type: Create'Type option
        }
        with
            static member New(?accountToken: string, ?businessProfile: Create'BusinessProfile, ?businessType: Create'BusinessType, ?capabilities: Create'Capabilities, ?company: Create'Company, ?country: string, ?defaultCurrency: string, ?documents: Create'Documents, ?email: string, ?expand: string list, ?externalAccount: string, ?individual: Create'Individual, ?metadata: Map<string, string>, ?settings: Create'Settings, ?tosAcceptance: Create'TosAcceptance, ?type': Create'Type) =
                {
                    AccountToken = accountToken
                    BusinessProfile = businessProfile
                    BusinessType = businessType
                    Capabilities = capabilities
                    Company = company
                    Country = country
                    DefaultCurrency = defaultCurrency
                    Documents = documents
                    Email = email
                    Expand = expand
                    ExternalAccount = externalAccount
                    Individual = individual
                    Metadata = metadata
                    Settings = settings
                    TosAcceptance = tosAcceptance
                    Type = type'
                }

        ///<p>With <a href="/docs/connect">Connect</a>, you can create Stripe accounts for your users.
        ///To do this, you’ll first need to <a href="https://dashboard.stripe.com/account/applications/settings">register your platform</a>.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/accounts"
            |> RestApi.postAsync<_, Account> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Account: string
        }
        with
            static member New(account: string) =
                {
                    Account = account
                }

        ///<p>With <a href="/docs/connect">Connect</a>, you can delete accounts you manage.
        ///Accounts created using test-mode keys can be deleted at any time. Custom or Express accounts created using live-mode keys can only be deleted once all balances are zero.
        ///If you want to delete your own account, use the <a href="https://dashboard.stripe.com/account">account information tab in your account settings</a> instead.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/accounts/{options.Account}"
            |> RestApi.deleteAsync<DeletedAccount> settings (Map.empty)

        type RetrieveOptions = {
            [<Config.Path>]Account: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(account: string, ?expand: string list) =
                {
                    Account = account
                    Expand = expand
                }

        ///<p>Retrieves the details of an account.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/accounts/{options.Account}"
            |> RestApi.getAsync<Account> settings qs

        type Update'BusinessProfileSupportAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'BusinessProfile = {
            ///[The merchant category code for the account](https://stripe.com/docs/connect/setting-mcc). MCCs are used to classify businesses based on the goods or services they provide.
            [<Config.Form>]Mcc: string option
            ///The customer-facing business name.
            [<Config.Form>]Name: string option
            ///Internal-only description of the product sold by, or service provided by, the business. Used by Stripe for risk and underwriting purposes.
            [<Config.Form>]ProductDescription: string option
            ///A publicly available mailing address for sending support issues to.
            [<Config.Form>]SupportAddress: Update'BusinessProfileSupportAddress option
            ///A publicly available email address for sending support issues to.
            [<Config.Form>]SupportEmail: string option
            ///A publicly available phone number to call with support issues.
            [<Config.Form>]SupportPhone: string option
            ///A publicly available website for handling support issues.
            [<Config.Form>]SupportUrl: Choice<string,string> option
            ///The business's publicly available website.
            [<Config.Form>]Url: string option
        }
        with
            static member New(?mcc: string, ?name: string, ?productDescription: string, ?supportAddress: Update'BusinessProfileSupportAddress, ?supportEmail: string, ?supportPhone: string, ?supportUrl: Choice<string,string>, ?url: string) =
                {
                    Mcc = mcc
                    Name = name
                    ProductDescription = productDescription
                    SupportAddress = supportAddress
                    SupportEmail = supportEmail
                    SupportPhone = supportPhone
                    SupportUrl = supportUrl
                    Url = url
                }

        type Update'CapabilitiesAcssDebitPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesAfterpayClearpayPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesAuBecsDebitPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesBacsDebitPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesBancontactPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesBoletoPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesCardIssuing = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesCardPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesCartesBancairesPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesEpsPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesFpxPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesGiropayPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesGrabpayPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesIdealPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesJcbPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesLegacyPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesOxxoPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesP24Payments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesSepaDebitPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesSofortPayments = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesTaxReportingUs1099K = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesTaxReportingUs1099Misc = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'CapabilitiesTransfers = {
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(?requested: bool) =
                {
                    Requested = requested
                }

        type Update'Capabilities = {
            ///The acss_debit_payments capability.
            [<Config.Form>]AcssDebitPayments: Update'CapabilitiesAcssDebitPayments option
            ///The afterpay_clearpay_payments capability.
            [<Config.Form>]AfterpayClearpayPayments: Update'CapabilitiesAfterpayClearpayPayments option
            ///The au_becs_debit_payments capability.
            [<Config.Form>]AuBecsDebitPayments: Update'CapabilitiesAuBecsDebitPayments option
            ///The bacs_debit_payments capability.
            [<Config.Form>]BacsDebitPayments: Update'CapabilitiesBacsDebitPayments option
            ///The bancontact_payments capability.
            [<Config.Form>]BancontactPayments: Update'CapabilitiesBancontactPayments option
            ///The boleto_payments capability.
            [<Config.Form>]BoletoPayments: Update'CapabilitiesBoletoPayments option
            ///The card_issuing capability.
            [<Config.Form>]CardIssuing: Update'CapabilitiesCardIssuing option
            ///The card_payments capability.
            [<Config.Form>]CardPayments: Update'CapabilitiesCardPayments option
            ///The cartes_bancaires_payments capability.
            [<Config.Form>]CartesBancairesPayments: Update'CapabilitiesCartesBancairesPayments option
            ///The eps_payments capability.
            [<Config.Form>]EpsPayments: Update'CapabilitiesEpsPayments option
            ///The fpx_payments capability.
            [<Config.Form>]FpxPayments: Update'CapabilitiesFpxPayments option
            ///The giropay_payments capability.
            [<Config.Form>]GiropayPayments: Update'CapabilitiesGiropayPayments option
            ///The grabpay_payments capability.
            [<Config.Form>]GrabpayPayments: Update'CapabilitiesGrabpayPayments option
            ///The ideal_payments capability.
            [<Config.Form>]IdealPayments: Update'CapabilitiesIdealPayments option
            ///The jcb_payments capability.
            [<Config.Form>]JcbPayments: Update'CapabilitiesJcbPayments option
            ///The legacy_payments capability.
            [<Config.Form>]LegacyPayments: Update'CapabilitiesLegacyPayments option
            ///The oxxo_payments capability.
            [<Config.Form>]OxxoPayments: Update'CapabilitiesOxxoPayments option
            ///The p24_payments capability.
            [<Config.Form>]P24Payments: Update'CapabilitiesP24Payments option
            ///The sepa_debit_payments capability.
            [<Config.Form>]SepaDebitPayments: Update'CapabilitiesSepaDebitPayments option
            ///The sofort_payments capability.
            [<Config.Form>]SofortPayments: Update'CapabilitiesSofortPayments option
            ///The tax_reporting_us_1099_k capability.
            [<Config.Form>]TaxReportingUs1099K: Update'CapabilitiesTaxReportingUs1099K option
            ///The tax_reporting_us_1099_misc capability.
            [<Config.Form>]TaxReportingUs1099Misc: Update'CapabilitiesTaxReportingUs1099Misc option
            ///The transfers capability.
            [<Config.Form>]Transfers: Update'CapabilitiesTransfers option
        }
        with
            static member New(?acssDebitPayments: Update'CapabilitiesAcssDebitPayments, ?taxReportingUs1099K: Update'CapabilitiesTaxReportingUs1099K, ?sofortPayments: Update'CapabilitiesSofortPayments, ?sepaDebitPayments: Update'CapabilitiesSepaDebitPayments, ?p24Payments: Update'CapabilitiesP24Payments, ?oxxoPayments: Update'CapabilitiesOxxoPayments, ?legacyPayments: Update'CapabilitiesLegacyPayments, ?jcbPayments: Update'CapabilitiesJcbPayments, ?idealPayments: Update'CapabilitiesIdealPayments, ?grabpayPayments: Update'CapabilitiesGrabpayPayments, ?taxReportingUs1099Misc: Update'CapabilitiesTaxReportingUs1099Misc, ?giropayPayments: Update'CapabilitiesGiropayPayments, ?epsPayments: Update'CapabilitiesEpsPayments, ?cartesBancairesPayments: Update'CapabilitiesCartesBancairesPayments, ?cardPayments: Update'CapabilitiesCardPayments, ?cardIssuing: Update'CapabilitiesCardIssuing, ?boletoPayments: Update'CapabilitiesBoletoPayments, ?bancontactPayments: Update'CapabilitiesBancontactPayments, ?bacsDebitPayments: Update'CapabilitiesBacsDebitPayments, ?auBecsDebitPayments: Update'CapabilitiesAuBecsDebitPayments, ?afterpayClearpayPayments: Update'CapabilitiesAfterpayClearpayPayments, ?fpxPayments: Update'CapabilitiesFpxPayments, ?transfers: Update'CapabilitiesTransfers) =
                {
                    AcssDebitPayments = acssDebitPayments
                    AfterpayClearpayPayments = afterpayClearpayPayments
                    AuBecsDebitPayments = auBecsDebitPayments
                    BacsDebitPayments = bacsDebitPayments
                    BancontactPayments = bancontactPayments
                    BoletoPayments = boletoPayments
                    CardIssuing = cardIssuing
                    CardPayments = cardPayments
                    CartesBancairesPayments = cartesBancairesPayments
                    EpsPayments = epsPayments
                    FpxPayments = fpxPayments
                    GiropayPayments = giropayPayments
                    GrabpayPayments = grabpayPayments
                    IdealPayments = idealPayments
                    JcbPayments = jcbPayments
                    LegacyPayments = legacyPayments
                    OxxoPayments = oxxoPayments
                    P24Payments = p24Payments
                    SepaDebitPayments = sepaDebitPayments
                    SofortPayments = sofortPayments
                    TaxReportingUs1099K = taxReportingUs1099K
                    TaxReportingUs1099Misc = taxReportingUs1099Misc
                    Transfers = transfers
                }

        type Update'CompanyAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'CompanyAddressKana = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Update'CompanyAddressKanji = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Update'CompanyVerificationDocument = {
            ///The back of a document returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `additional_verification`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Back: string option
            ///The front of a document returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `additional_verification`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Update'CompanyVerification = {
            ///A document verifying the business.
            [<Config.Form>]Document: Update'CompanyVerificationDocument option
        }
        with
            static member New(?document: Update'CompanyVerificationDocument) =
                {
                    Document = document
                }

        type Update'CompanyStructure =
        | FreeZoneEstablishment
        | FreeZoneLlc
        | GovernmentInstrumentality
        | GovernmentalUnit
        | IncorporatedNonProfit
        | LimitedLiabilityPartnership
        | Llc
        | MultiMemberLlc
        | PrivateCompany
        | PrivateCorporation
        | PrivatePartnership
        | PublicCompany
        | PublicCorporation
        | PublicPartnership
        | SingleMemberLlc
        | SoleEstablishment
        | SoleProprietorship
        | TaxExemptGovernmentInstrumentality
        | UnincorporatedAssociation
        | UnincorporatedNonProfit

        type Update'Company = {
            ///The company's primary address.
            [<Config.Form>]Address: Update'CompanyAddress option
            ///The Kana variation of the company's primary address (Japan only).
            [<Config.Form>]AddressKana: Update'CompanyAddressKana option
            ///The Kanji variation of the company's primary address (Japan only).
            [<Config.Form>]AddressKanji: Update'CompanyAddressKanji option
            ///Whether the company's directors have been provided. Set this Boolean to `true` after creating all the company's directors with [the Persons API](https://stripe.com/docs/api/persons) for accounts with a `relationship.director` requirement. This value is not automatically set to `true` after creating directors, so it needs to be updated to indicate all directors have been provided.
            [<Config.Form>]DirectorsProvided: bool option
            ///Whether the company's executives have been provided. Set this Boolean to `true` after creating all the company's executives with [the Persons API](https://stripe.com/docs/api/persons) for accounts with a `relationship.executive` requirement.
            [<Config.Form>]ExecutivesProvided: bool option
            ///The company's legal name.
            [<Config.Form>]Name: string option
            ///The Kana variation of the company's legal name (Japan only).
            [<Config.Form>]NameKana: string option
            ///The Kanji variation of the company's legal name (Japan only).
            [<Config.Form>]NameKanji: string option
            ///Whether the company's owners have been provided. Set this Boolean to `true` after creating all the company's owners with [the Persons API](https://stripe.com/docs/api/persons) for accounts with a `relationship.owner` requirement.
            [<Config.Form>]OwnersProvided: bool option
            ///The company's phone number (used for verification).
            [<Config.Form>]Phone: string option
            ///The identification number given to a company when it is registered or incorporated, if distinct from the identification number used for filing taxes. (Examples are the CIN for companies and LLP IN for partnerships in India, and the Company Registration Number in Hong Kong).
            [<Config.Form>]RegistrationNumber: string option
            ///The category identifying the legal structure of the company or legal entity. See [Business structure](https://stripe.com/docs/connect/identity-verification#business-structure) for more details.
            [<Config.Form>]Structure: Update'CompanyStructure option
            ///The business ID number of the company, as appropriate for the company’s country. (Examples are an Employer ID Number in the U.S., a Business Number in Canada, or a Company Number in the UK.)
            [<Config.Form>]TaxId: string option
            ///The jurisdiction in which the `tax_id` is registered (Germany-based companies only).
            [<Config.Form>]TaxIdRegistrar: string option
            ///The VAT number of the company.
            [<Config.Form>]VatId: string option
            ///Information on the verification state of the company.
            [<Config.Form>]Verification: Update'CompanyVerification option
        }
        with
            static member New(?address: Update'CompanyAddress, ?addressKana: Update'CompanyAddressKana, ?addressKanji: Update'CompanyAddressKanji, ?directorsProvided: bool, ?executivesProvided: bool, ?name: string, ?nameKana: string, ?nameKanji: string, ?ownersProvided: bool, ?phone: string, ?registrationNumber: string, ?structure: Update'CompanyStructure, ?taxId: string, ?taxIdRegistrar: string, ?vatId: string, ?verification: Update'CompanyVerification) =
                {
                    Address = address
                    AddressKana = addressKana
                    AddressKanji = addressKanji
                    DirectorsProvided = directorsProvided
                    ExecutivesProvided = executivesProvided
                    Name = name
                    NameKana = nameKana
                    NameKanji = nameKanji
                    OwnersProvided = ownersProvided
                    Phone = phone
                    RegistrationNumber = registrationNumber
                    Structure = structure
                    TaxId = taxId
                    TaxIdRegistrar = taxIdRegistrar
                    VatId = vatId
                    Verification = verification
                }

        type Update'DocumentsBankAccountOwnershipVerification = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Update'DocumentsCompanyLicense = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Update'DocumentsCompanyMemorandumOfAssociation = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Update'DocumentsCompanyMinisterialDecree = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Update'DocumentsCompanyRegistrationVerification = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Update'DocumentsCompanyTaxIdVerification = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Update'Documents = {
            ///One or more documents that support the [Bank account ownership verification](https://support.stripe.com/questions/bank-account-ownership-verification) requirement. Must be a document associated with the account’s primary active bank account that displays the last 4 digits of the account number, either a statement or a voided check.
            [<Config.Form>]BankAccountOwnershipVerification: Update'DocumentsBankAccountOwnershipVerification option
            ///One or more documents that demonstrate proof of a company's license to operate.
            [<Config.Form>]CompanyLicense: Update'DocumentsCompanyLicense option
            ///One or more documents showing the company's Memorandum of Association.
            [<Config.Form>]CompanyMemorandumOfAssociation: Update'DocumentsCompanyMemorandumOfAssociation option
            ///(Certain countries only) One or more documents showing the ministerial decree legalizing the company's establishment.
            [<Config.Form>]CompanyMinisterialDecree: Update'DocumentsCompanyMinisterialDecree option
            ///One or more documents that demonstrate proof of a company's registration with the appropriate local authorities.
            [<Config.Form>]CompanyRegistrationVerification: Update'DocumentsCompanyRegistrationVerification option
            ///One or more documents that demonstrate proof of a company's tax ID.
            [<Config.Form>]CompanyTaxIdVerification: Update'DocumentsCompanyTaxIdVerification option
        }
        with
            static member New(?bankAccountOwnershipVerification: Update'DocumentsBankAccountOwnershipVerification, ?companyLicense: Update'DocumentsCompanyLicense, ?companyMemorandumOfAssociation: Update'DocumentsCompanyMemorandumOfAssociation, ?companyMinisterialDecree: Update'DocumentsCompanyMinisterialDecree, ?companyRegistrationVerification: Update'DocumentsCompanyRegistrationVerification, ?companyTaxIdVerification: Update'DocumentsCompanyTaxIdVerification) =
                {
                    BankAccountOwnershipVerification = bankAccountOwnershipVerification
                    CompanyLicense = companyLicense
                    CompanyMemorandumOfAssociation = companyMemorandumOfAssociation
                    CompanyMinisterialDecree = companyMinisterialDecree
                    CompanyRegistrationVerification = companyRegistrationVerification
                    CompanyTaxIdVerification = companyTaxIdVerification
                }

        type Update'IndividualAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'IndividualAddressKana = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Update'IndividualAddressKanji = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Update'IndividualDobDateOfBirthSpecs = {
            ///The day of birth, between 1 and 31.
            [<Config.Form>]Day: int option
            ///The month of birth, between 1 and 12.
            [<Config.Form>]Month: int option
            ///The four-digit year of birth.
            [<Config.Form>]Year: int option
        }
        with
            static member New(?day: int, ?month: int, ?year: int) =
                {
                    Day = day
                    Month = month
                    Year = year
                }

        type Update'IndividualVerificationAdditionalDocument = {
            ///The back of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Back: string option
            ///The front of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Update'IndividualVerificationDocument = {
            ///The back of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Back: string option
            ///The front of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Update'IndividualVerification = {
            ///A document showing address, either a passport, local ID card, or utility bill from a well-known utility company.
            [<Config.Form>]AdditionalDocument: Update'IndividualVerificationAdditionalDocument option
            ///An identifying document, either a passport or local ID card.
            [<Config.Form>]Document: Update'IndividualVerificationDocument option
        }
        with
            static member New(?additionalDocument: Update'IndividualVerificationAdditionalDocument, ?document: Update'IndividualVerificationDocument) =
                {
                    AdditionalDocument = additionalDocument
                    Document = document
                }

        type Update'IndividualPoliticalExposure =
        | Existing
        | None'

        type Update'Individual = {
            ///The individual's primary address.
            [<Config.Form>]Address: Update'IndividualAddress option
            ///The Kana variation of the the individual's primary address (Japan only).
            [<Config.Form>]AddressKana: Update'IndividualAddressKana option
            ///The Kanji variation of the the individual's primary address (Japan only).
            [<Config.Form>]AddressKanji: Update'IndividualAddressKanji option
            ///The individual's date of birth.
            [<Config.Form>]Dob: Choice<Update'IndividualDobDateOfBirthSpecs,string> option
            ///The individual's email address.
            [<Config.Form>]Email: string option
            ///The individual's first name.
            [<Config.Form>]FirstName: string option
            ///The Kana variation of the the individual's first name (Japan only).
            [<Config.Form>]FirstNameKana: string option
            ///The Kanji variation of the individual's first name (Japan only).
            [<Config.Form>]FirstNameKanji: string option
            ///The individual's gender (International regulations require either "male" or "female").
            [<Config.Form>]Gender: string option
            ///The government-issued ID number of the individual, as appropriate for the representative’s country. (Examples are a Social Security Number in the U.S., or a Social Insurance Number in Canada). Instead of the number itself, you can also provide a [PII token created with Stripe.js](https://stripe.com/docs/stripe.js#collecting-pii-data).
            [<Config.Form>]IdNumber: string option
            ///The individual's last name.
            [<Config.Form>]LastName: string option
            ///The Kana varation of the individual's last name (Japan only).
            [<Config.Form>]LastNameKana: string option
            ///The Kanji varation of the individual's last name (Japan only).
            [<Config.Form>]LastNameKanji: string option
            ///The individual's maiden name.
            [<Config.Form>]MaidenName: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The individual's phone number.
            [<Config.Form>]Phone: string option
            ///Indicates if the person or any of their representatives, family members, or other closely related persons, declares that they hold or have held an important public job or function, in any jurisdiction.
            [<Config.Form>]PoliticalExposure: Update'IndividualPoliticalExposure option
            ///The last four digits of the individual's Social Security Number (U.S. only).
            [<Config.Form>]SsnLast4: string option
            ///The individual's verification document information.
            [<Config.Form>]Verification: Update'IndividualVerification option
        }
        with
            static member New(?address: Update'IndividualAddress, ?politicalExposure: Update'IndividualPoliticalExposure, ?phone: string, ?metadata: Map<string, string>, ?maidenName: string, ?lastNameKanji: string, ?lastNameKana: string, ?lastName: string, ?ssnLast4: string, ?idNumber: string, ?firstNameKanji: string, ?firstNameKana: string, ?firstName: string, ?email: string, ?dob: Choice<Update'IndividualDobDateOfBirthSpecs,string>, ?addressKanji: Update'IndividualAddressKanji, ?addressKana: Update'IndividualAddressKana, ?gender: string, ?verification: Update'IndividualVerification) =
                {
                    Address = address
                    AddressKana = addressKana
                    AddressKanji = addressKanji
                    Dob = dob
                    Email = email
                    FirstName = firstName
                    FirstNameKana = firstNameKana
                    FirstNameKanji = firstNameKanji
                    Gender = gender
                    IdNumber = idNumber
                    LastName = lastName
                    LastNameKana = lastNameKana
                    LastNameKanji = lastNameKanji
                    MaidenName = maidenName
                    Metadata = metadata
                    Phone = phone
                    PoliticalExposure = politicalExposure
                    SsnLast4 = ssnLast4
                    Verification = verification
                }

        type Update'SettingsBranding = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) An icon for the account. Must be square and at least 128px x 128px.
            [<Config.Form>]Icon: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) A logo for the account that will be used in Checkout instead of the icon and without the account's name next to it if provided. Must be at least 128px x 128px.
            [<Config.Form>]Logo: string option
            ///A CSS hex color value representing the primary branding color for this account.
            [<Config.Form>]PrimaryColor: string option
            ///A CSS hex color value representing the secondary branding color for this account.
            [<Config.Form>]SecondaryColor: string option
        }
        with
            static member New(?icon: string, ?logo: string, ?primaryColor: string, ?secondaryColor: string) =
                {
                    Icon = icon
                    Logo = logo
                    PrimaryColor = primaryColor
                    SecondaryColor = secondaryColor
                }

        type Update'SettingsCardIssuingTosAcceptance = {
            ///The Unix timestamp marking when the account representative accepted the service agreement.
            [<Config.Form>]Date: DateTime option
            ///The IP address from which the account representative accepted the service agreement.
            [<Config.Form>]Ip: string option
            ///The user agent of the browser from which the account representative accepted the service agreement.
            [<Config.Form>]UserAgent: string option
        }
        with
            static member New(?date: DateTime, ?ip: string, ?userAgent: string) =
                {
                    Date = date
                    Ip = ip
                    UserAgent = userAgent
                }

        type Update'SettingsCardIssuing = {
            ///Details on the account's acceptance of the [Stripe Issuing Terms and Disclosures](https://stripe.com/docs/issuing/connect/tos_acceptance).
            [<Config.Form>]TosAcceptance: Update'SettingsCardIssuingTosAcceptance option
        }
        with
            static member New(?tosAcceptance: Update'SettingsCardIssuingTosAcceptance) =
                {
                    TosAcceptance = tosAcceptance
                }

        type Update'SettingsCardPaymentsDeclineOn = {
            ///Whether Stripe automatically declines charges with an incorrect ZIP or postal code. This setting only applies when a ZIP or postal code is provided and they fail bank verification.
            [<Config.Form>]AvsFailure: bool option
            ///Whether Stripe automatically declines charges with an incorrect CVC. This setting only applies when a CVC is provided and it fails bank verification.
            [<Config.Form>]CvcFailure: bool option
        }
        with
            static member New(?avsFailure: bool, ?cvcFailure: bool) =
                {
                    AvsFailure = avsFailure
                    CvcFailure = cvcFailure
                }

        type Update'SettingsCardPayments = {
            ///Automatically declines certain charge types regardless of whether the card issuer accepted or declined the charge.
            [<Config.Form>]DeclineOn: Update'SettingsCardPaymentsDeclineOn option
            ///The default text that appears on credit card statements when a charge is made. This field prefixes any dynamic `statement_descriptor` specified on the charge. `statement_descriptor_prefix` is useful for maximizing descriptor space for the dynamic portion.
            [<Config.Form>]StatementDescriptorPrefix: string option
        }
        with
            static member New(?declineOn: Update'SettingsCardPaymentsDeclineOn, ?statementDescriptorPrefix: string) =
                {
                    DeclineOn = declineOn
                    StatementDescriptorPrefix = statementDescriptorPrefix
                }

        type Update'SettingsPayments = {
            ///The default text that appears on credit card statements when a charge is made. This field prefixes any dynamic `statement_descriptor` specified on the charge.
            [<Config.Form>]StatementDescriptor: string option
            ///The Kana variation of the default text that appears on credit card statements when a charge is made (Japan only).
            [<Config.Form>]StatementDescriptorKana: string option
            ///The Kanji variation of the default text that appears on credit card statements when a charge is made (Japan only).
            [<Config.Form>]StatementDescriptorKanji: string option
        }
        with
            static member New(?statementDescriptor: string, ?statementDescriptorKana: string, ?statementDescriptorKanji: string) =
                {
                    StatementDescriptor = statementDescriptor
                    StatementDescriptorKana = statementDescriptorKana
                    StatementDescriptorKanji = statementDescriptorKanji
                }

        type Update'SettingsPayoutsScheduleDelayDays =
        | Minimum

        type Update'SettingsPayoutsScheduleInterval =
        | Daily
        | Manual
        | Monthly
        | Weekly

        type Update'SettingsPayoutsScheduleWeeklyAnchor =
        | Friday
        | Monday
        | Saturday
        | Sunday
        | Thursday
        | Tuesday
        | Wednesday

        type Update'SettingsPayoutsSchedule = {
            ///The number of days charge funds are held before being paid out. May also be set to `minimum`, representing the lowest available value for the account country. Default is `minimum`. The `delay_days` parameter does not apply when the `interval` is `manual`.
            [<Config.Form>]DelayDays: Choice<Update'SettingsPayoutsScheduleDelayDays,int> option
            ///How frequently available funds are paid out. One of: `daily`, `manual`, `weekly`, or `monthly`. Default is `daily`.
            [<Config.Form>]Interval: Update'SettingsPayoutsScheduleInterval option
            ///The day of the month when available funds are paid out, specified as a number between 1--31. Payouts nominally scheduled between the 29th and 31st of the month are instead sent on the last day of a shorter month. Required and applicable only if `interval` is `monthly`.
            [<Config.Form>]MonthlyAnchor: int option
            ///The day of the week when available funds are paid out, specified as `monday`, `tuesday`, etc. (required and applicable only if `interval` is `weekly`.)
            [<Config.Form>]WeeklyAnchor: Update'SettingsPayoutsScheduleWeeklyAnchor option
        }
        with
            static member New(?delayDays: Choice<Update'SettingsPayoutsScheduleDelayDays,int>, ?interval: Update'SettingsPayoutsScheduleInterval, ?monthlyAnchor: int, ?weeklyAnchor: Update'SettingsPayoutsScheduleWeeklyAnchor) =
                {
                    DelayDays = delayDays
                    Interval = interval
                    MonthlyAnchor = monthlyAnchor
                    WeeklyAnchor = weeklyAnchor
                }

        type Update'SettingsPayouts = {
            ///A Boolean indicating whether Stripe should try to reclaim negative balances from an attached bank account. For details, see [Understanding Connect Account Balances](https://stripe.com/docs/connect/account-balances).
            [<Config.Form>]DebitNegativeBalances: bool option
            ///Details on when funds from charges are available, and when they are paid out to an external account. For details, see our [Setting Bank and Debit Card Payouts](https://stripe.com/docs/connect/bank-transfers#payout-information) documentation.
            [<Config.Form>]Schedule: Update'SettingsPayoutsSchedule option
            ///The text that appears on the bank account statement for payouts. If not set, this defaults to the platform's bank descriptor as set in the Dashboard.
            [<Config.Form>]StatementDescriptor: string option
        }
        with
            static member New(?debitNegativeBalances: bool, ?schedule: Update'SettingsPayoutsSchedule, ?statementDescriptor: string) =
                {
                    DebitNegativeBalances = debitNegativeBalances
                    Schedule = schedule
                    StatementDescriptor = statementDescriptor
                }

        type Update'Settings = {
            ///Settings used to apply the account's branding to email receipts, invoices, Checkout, and other products.
            [<Config.Form>]Branding: Update'SettingsBranding option
            ///Settings specific to the account's use of the Card Issuing product.
            [<Config.Form>]CardIssuing: Update'SettingsCardIssuing option
            ///Settings specific to card charging on the account.
            [<Config.Form>]CardPayments: Update'SettingsCardPayments option
            ///Settings that apply across payment methods for charging on the account.
            [<Config.Form>]Payments: Update'SettingsPayments option
            ///Settings specific to the account's payouts.
            [<Config.Form>]Payouts: Update'SettingsPayouts option
        }
        with
            static member New(?branding: Update'SettingsBranding, ?cardIssuing: Update'SettingsCardIssuing, ?cardPayments: Update'SettingsCardPayments, ?payments: Update'SettingsPayments, ?payouts: Update'SettingsPayouts) =
                {
                    Branding = branding
                    CardIssuing = cardIssuing
                    CardPayments = cardPayments
                    Payments = payments
                    Payouts = payouts
                }

        type Update'TosAcceptance = {
            ///The Unix timestamp marking when the account representative accepted their service agreement.
            [<Config.Form>]Date: DateTime option
            ///The IP address from which the account representative accepted their service agreement.
            [<Config.Form>]Ip: string option
            ///The user's service agreement type.
            [<Config.Form>]ServiceAgreement: string option
            ///The user agent of the browser from which the account representative accepted their service agreement.
            [<Config.Form>]UserAgent: string option
        }
        with
            static member New(?date: DateTime, ?ip: string, ?serviceAgreement: string, ?userAgent: string) =
                {
                    Date = date
                    Ip = ip
                    ServiceAgreement = serviceAgreement
                    UserAgent = userAgent
                }

        type Update'BusinessType =
        | Company
        | GovernmentEntity
        | Individual
        | NonProfit

        type UpdateOptions = {
            [<Config.Path>]Account: string
            ///An [account token](https://stripe.com/docs/api#create_account_token), used to securely provide details to the account.
            [<Config.Form>]AccountToken: string option
            ///Business information about the account.
            [<Config.Form>]BusinessProfile: Update'BusinessProfile option
            ///The business type.
            [<Config.Form>]BusinessType: Update'BusinessType option
            ///Each key of the dictionary represents a capability, and each capability maps to its settings (e.g. whether it has been requested or not). Each capability will be inactive until you have provided its specific requirements and Stripe has verified them. An account may have some of its requested capabilities be active and some be inactive.
            [<Config.Form>]Capabilities: Update'Capabilities option
            ///Information about the company or business. This field is available for any `business_type`.
            [<Config.Form>]Company: Update'Company option
            ///Three-letter ISO currency code representing the default currency for the account. This must be a currency that [Stripe supports in the account's country](https://stripe.com/docs/payouts).
            [<Config.Form>]DefaultCurrency: string option
            ///Documents that may be submitted to satisfy various informational requests.
            [<Config.Form>]Documents: Update'Documents option
            ///The email address of the account holder. This is only to make the account easier to identify to you. Stripe will never directly email Custom accounts.
            [<Config.Form>]Email: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///A card or bank account to attach to the account for receiving [payouts](https://stripe.com/docs/connect/bank-debit-card-payouts) (you won’t be able to use it for top-ups). You can provide either a token, like the ones returned by [Stripe.js](https://stripe.com/docs/stripe.js), or a dictionary, as documented in the `external_account` parameter for [bank account](https://stripe.com/docs/api#account_create_bank_account) creation. <br><br>By default, providing an external account sets it as the new default external account for its currency, and deletes the old default if one exists. To add additional external accounts without replacing the existing default for the currency, use the bank account or card creation API.
            [<Config.Form>]ExternalAccount: string option
            ///Information about the person represented by the account. This field is null unless `business_type` is set to `individual`.
            [<Config.Form>]Individual: Update'Individual option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Options for customizing how the account functions within Stripe.
            [<Config.Form>]Settings: Update'Settings option
            ///Details on the account's acceptance of the [Stripe Services Agreement](https://stripe.com/docs/connect/updating-accounts#tos-acceptance).
            [<Config.Form>]TosAcceptance: Update'TosAcceptance option
        }
        with
            static member New(account: string, ?accountToken: string, ?businessProfile: Update'BusinessProfile, ?businessType: Update'BusinessType, ?capabilities: Update'Capabilities, ?company: Update'Company, ?defaultCurrency: string, ?documents: Update'Documents, ?email: string, ?expand: string list, ?externalAccount: string, ?individual: Update'Individual, ?metadata: Map<string, string>, ?settings: Update'Settings, ?tosAcceptance: Update'TosAcceptance) =
                {
                    Account = account
                    AccountToken = accountToken
                    BusinessProfile = businessProfile
                    BusinessType = businessType
                    Capabilities = capabilities
                    Company = company
                    DefaultCurrency = defaultCurrency
                    Documents = documents
                    Email = email
                    Expand = expand
                    ExternalAccount = externalAccount
                    Individual = individual
                    Metadata = metadata
                    Settings = settings
                    TosAcceptance = tosAcceptance
                }

        ///<p>Updates a <a href="/docs/connect/accounts">connected account</a> by setting the values of the parameters passed. Any parameters not provided are left unchanged. Most parameters can be changed only for Custom accounts. (These are marked <strong>Custom Only</strong> below.) Parameters marked <strong>Custom and Express</strong> are not supported for Standard accounts.
        ///To update your own account, use the <a href="https://dashboard.stripe.com/account">Dashboard</a>. Refer to our <a href="/docs/connect/updating-accounts">Connect</a> documentation to learn more about updating accounts.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/accounts/{options.Account}"
            |> RestApi.postAsync<_, Account> settings (Map.empty) options

    module AccountsCapabilities =

        type CapabilitiesOptions = {
            [<Config.Path>]Account: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(account: string, ?expand: string list) =
                {
                    Account = account
                    Expand = expand
                }

        ///<p>Returns a list of capabilities associated with the account. The capabilities are returned sorted by creation date, with the most recent capability appearing first.</p>
        let Capabilities settings (options: CapabilitiesOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/accounts/{options.Account}/capabilities"
            |> RestApi.getAsync<Capability list> settings qs

        type RetrieveOptions = {
            [<Config.Path>]Account: string
            [<Config.Path>]Capability: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(account: string, capability: string, ?expand: string list) =
                {
                    Account = account
                    Capability = capability
                    Expand = expand
                }

        ///<p>Retrieves information about the specified Account Capability.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/accounts/{options.Account}/capabilities/{options.Capability}"
            |> RestApi.getAsync<Capability> settings qs

        type UpdateOptions = {
            [<Config.Path>]Account: string
            [<Config.Path>]Capability: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Passing true requests the capability for the account, if it is not already requested. A requested capability may not immediately become active. Any requirements to activate the capability are returned in the `requirements` arrays.
            [<Config.Form>]Requested: bool option
        }
        with
            static member New(account: string, capability: string, ?expand: string list, ?requested: bool) =
                {
                    Account = account
                    Capability = capability
                    Expand = expand
                    Requested = requested
                }

        ///<p>Updates an existing Account Capability.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/accounts/{options.Account}/capabilities/{options.Capability}"
            |> RestApi.postAsync<_, Capability> settings (Map.empty) options

    module AccountsExternalAccounts =

        type ListOptions = {
            [<Config.Path>]Account: string
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(account: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Account = account
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>List external accounts for an account.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/accounts/{options.Account}/external_accounts"
            |> RestApi.getAsync<ExternalAccount list> settings qs

        type CreateOptions = {
            [<Config.Path>]Account: string
            ///When set to true, or if this is the first external account added in this currency, this account becomes the default external account for its currency.
            [<Config.Form>]DefaultForCurrency: bool option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Please refer to full [documentation](https://stripe.com/docs/api) instead.
            [<Config.Form>]ExternalAccount: string
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(account: string, externalAccount: string, ?defaultForCurrency: bool, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Account = account
                    DefaultForCurrency = defaultForCurrency
                    Expand = expand
                    ExternalAccount = externalAccount
                    Metadata = metadata
                }

        ///<p>Create an external account for a given account.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/accounts/{options.Account}/external_accounts"
            |> RestApi.postAsync<_, ExternalAccount> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Account: string
            [<Config.Path>]Id: string
        }
        with
            static member New(account: string, id: string) =
                {
                    Account = account
                    Id = id
                }

        ///<p>Delete a specified external account for a given account.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/accounts/{options.Account}/external_accounts/{options.Id}"
            |> RestApi.deleteAsync<DeletedExternalAccount> settings (Map.empty)

        type RetrieveOptions = {
            [<Config.Path>]Account: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
        }
        with
            static member New(account: string, id: string, ?expand: string list) =
                {
                    Account = account
                    Expand = expand
                    Id = id
                }

        ///<p>Retrieve a specified external account for a given account.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/accounts/{options.Account}/external_accounts/{options.Id}"
            |> RestApi.getAsync<ExternalAccount> settings qs

        type Update'AccountHolderType =
        | Company
        | Individual

        type UpdateOptions = {
            [<Config.Path>]Account: string
            [<Config.Path>]Id: string
            ///The name of the person or business that owns the bank account.
            [<Config.Form>]AccountHolderName: string option
            ///The type of entity that holds the account. This can be either `individual` or `company`.
            [<Config.Form>]AccountHolderType: Update'AccountHolderType option
            ///City/District/Suburb/Town/Village.
            [<Config.Form>]AddressCity: string option
            ///Billing address country, if provided when creating card.
            [<Config.Form>]AddressCountry: string option
            ///Address line 1 (Street address/PO Box/Company name).
            [<Config.Form>]AddressLine1: string option
            ///Address line 2 (Apartment/Suite/Unit/Building).
            [<Config.Form>]AddressLine2: string option
            ///State/County/Province/Region.
            [<Config.Form>]AddressState: string option
            ///ZIP or postal code.
            [<Config.Form>]AddressZip: string option
            ///When set to true, this becomes the default external account for its currency.
            [<Config.Form>]DefaultForCurrency: bool option
            ///Two digit number representing the card’s expiration month.
            [<Config.Form>]ExpMonth: string option
            ///Four digit number representing the card’s expiration year.
            [<Config.Form>]ExpYear: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Cardholder name.
            [<Config.Form>]Name: string option
        }
        with
            static member New(account: string, id: string, ?accountHolderName: string, ?accountHolderType: Update'AccountHolderType, ?addressCity: string, ?addressCountry: string, ?addressLine1: string, ?addressLine2: string, ?addressState: string, ?addressZip: string, ?defaultForCurrency: bool, ?expMonth: string, ?expYear: string, ?expand: string list, ?metadata: Map<string, string>, ?name: string) =
                {
                    Account = account
                    Id = id
                    AccountHolderName = accountHolderName
                    AccountHolderType = accountHolderType
                    AddressCity = addressCity
                    AddressCountry = addressCountry
                    AddressLine1 = addressLine1
                    AddressLine2 = addressLine2
                    AddressState = addressState
                    AddressZip = addressZip
                    DefaultForCurrency = defaultForCurrency
                    ExpMonth = expMonth
                    ExpYear = expYear
                    Expand = expand
                    Metadata = metadata
                    Name = name
                }

        ///<p>Updates the metadata, account holder name, account holder type of a bank account belonging to a <a href="/docs/connect/custom-accounts">Custom account</a>, and optionally sets it as the default for its currency. Other bank account details are not editable by design.
        ///You can re-enable a disabled bank account by performing an update call without providing any arguments or changes.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/accounts/{options.Account}/external_accounts/{options.Id}"
            |> RestApi.postAsync<_, ExternalAccount> settings (Map.empty) options

    module AccountsLoginLinks =

        type CreateOptions = {
            [<Config.Path>]Account: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Where to redirect the user after they log out of their dashboard.
            [<Config.Form>]RedirectUrl: string option
        }
        with
            static member New(account: string, ?expand: string list, ?redirectUrl: string) =
                {
                    Account = account
                    Expand = expand
                    RedirectUrl = redirectUrl
                }

        ///<p>Creates a single-use login link for an Express account to access their Stripe dashboard.
        ///<strong>You may only create login links for <a href="/docs/connect/express-accounts">Express accounts</a> connected to your platform</strong>.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/accounts/{options.Account}/login_links"
            |> RestApi.postAsync<_, LoginLink> settings (Map.empty) options

    module AccountsPersons =

        type ListOptions = {
            [<Config.Path>]Account: string
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///Filters on the list of people returned based on the person's relationship to the account's company.
            [<Config.Query>]Relationship: Map<string, string> option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(account: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?relationship: Map<string, string>, ?startingAfter: string) =
                {
                    Account = account
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    Relationship = relationship
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of people associated with the account’s legal entity. The people are returned sorted by creation date, with the most recent people appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("relationship", options.Relationship |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/accounts/{options.Account}/persons"
            |> RestApi.getAsync<Person list> settings qs

        type Create'Address = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'AddressKana = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Create'AddressKanji = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Create'DobDateOfBirthSpecs = {
            ///The day of birth, between 1 and 31.
            [<Config.Form>]Day: int option
            ///The month of birth, between 1 and 12.
            [<Config.Form>]Month: int option
            ///The four-digit year of birth.
            [<Config.Form>]Year: int option
        }
        with
            static member New(?day: int, ?month: int, ?year: int) =
                {
                    Day = day
                    Month = month
                    Year = year
                }

        type Create'DocumentsCompanyAuthorization = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Create'DocumentsPassport = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Create'DocumentsVisa = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Create'Documents = {
            ///One or more documents that demonstrate proof that this person is authorized to represent the company.
            [<Config.Form>]CompanyAuthorization: Create'DocumentsCompanyAuthorization option
            ///One or more documents showing the person's passport page with photo and personal data.
            [<Config.Form>]Passport: Create'DocumentsPassport option
            ///One or more documents showing the person's visa required for living in the country where they are residing.
            [<Config.Form>]Visa: Create'DocumentsVisa option
        }
        with
            static member New(?companyAuthorization: Create'DocumentsCompanyAuthorization, ?passport: Create'DocumentsPassport, ?visa: Create'DocumentsVisa) =
                {
                    CompanyAuthorization = companyAuthorization
                    Passport = passport
                    Visa = visa
                }

        type Create'Relationship = {
            ///Whether the person is a director of the account's legal entity. Currently only required for accounts in the EU. Directors are typically members of the governing board of the company, or responsible for ensuring the company meets its regulatory obligations.
            [<Config.Form>]Director: bool option
            ///Whether the person has significant responsibility to control, manage, or direct the organization.
            [<Config.Form>]Executive: bool option
            ///Whether the person is an owner of the account’s legal entity.
            [<Config.Form>]Owner: bool option
            ///The percent owned by the person of the account's legal entity.
            [<Config.Form>]PercentOwnership: Choice<decimal,string> option
            ///Whether the person is authorized as the primary representative of the account. This is the person nominated by the business to provide information about themselves, and general information about the account. There can only be one representative at any given time. At the time the account is created, this person should be set to the person responsible for opening the account.
            [<Config.Form>]Representative: bool option
            ///The person's title (e.g., CEO, Support Engineer).
            [<Config.Form>]Title: string option
        }
        with
            static member New(?director: bool, ?executive: bool, ?owner: bool, ?percentOwnership: Choice<decimal,string>, ?representative: bool, ?title: string) =
                {
                    Director = director
                    Executive = executive
                    Owner = owner
                    PercentOwnership = percentOwnership
                    Representative = representative
                    Title = title
                }

        type Create'VerificationAdditionalDocument = {
            ///The back of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Back: string option
            ///The front of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Create'VerificationDocument = {
            ///The back of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Back: string option
            ///The front of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Create'Verification = {
            ///A document showing address, either a passport, local ID card, or utility bill from a well-known utility company.
            [<Config.Form>]AdditionalDocument: Create'VerificationAdditionalDocument option
            ///An identifying document, either a passport or local ID card.
            [<Config.Form>]Document: Create'VerificationDocument option
        }
        with
            static member New(?additionalDocument: Create'VerificationAdditionalDocument, ?document: Create'VerificationDocument) =
                {
                    AdditionalDocument = additionalDocument
                    Document = document
                }

        type CreateOptions = {
            [<Config.Path>]Account: string
            ///The person's address.
            [<Config.Form>]Address: Create'Address option
            ///The Kana variation of the person's address (Japan only).
            [<Config.Form>]AddressKana: Create'AddressKana option
            ///The Kanji variation of the person's address (Japan only).
            [<Config.Form>]AddressKanji: Create'AddressKanji option
            ///The person's date of birth.
            [<Config.Form>]Dob: Choice<Create'DobDateOfBirthSpecs,string> option
            ///Documents that may be submitted to satisfy various informational requests.
            [<Config.Form>]Documents: Create'Documents option
            ///The person's email address.
            [<Config.Form>]Email: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The person's first name.
            [<Config.Form>]FirstName: string option
            ///The Kana variation of the person's first name (Japan only).
            [<Config.Form>]FirstNameKana: string option
            ///The Kanji variation of the person's first name (Japan only).
            [<Config.Form>]FirstNameKanji: string option
            ///The person's gender (International regulations require either "male" or "female").
            [<Config.Form>]Gender: string option
            ///The person's ID number, as appropriate for their country. For example, a social security number in the U.S., social insurance number in Canada, etc. Instead of the number itself, you can also provide a [PII token provided by Stripe.js](https://stripe.com/docs/stripe.js#collecting-pii-data).
            [<Config.Form>]IdNumber: string option
            ///The person's last name.
            [<Config.Form>]LastName: string option
            ///The Kana variation of the person's last name (Japan only).
            [<Config.Form>]LastNameKana: string option
            ///The Kanji variation of the person's last name (Japan only).
            [<Config.Form>]LastNameKanji: string option
            ///The person's maiden name.
            [<Config.Form>]MaidenName: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The country where the person is a national. Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)), or "XX" if unavailable.
            [<Config.Form>]Nationality: string option
            ///A [person token](https://stripe.com/docs/connect/account-tokens), used to securely provide details to the person.
            [<Config.Form>]PersonToken: string option
            ///The person's phone number.
            [<Config.Form>]Phone: string option
            ///Indicates if the person or any of their representatives, family members, or other closely related persons, declares that they hold or have held an important public job or function, in any jurisdiction.
            [<Config.Form>]PoliticalExposure: string option
            ///The relationship that this person has with the account's legal entity.
            [<Config.Form>]Relationship: Create'Relationship option
            ///The last four digits of the person's Social Security number (U.S. only).
            [<Config.Form>]SsnLast4: string option
            ///The person's verification status.
            [<Config.Form>]Verification: Create'Verification option
        }
        with
            static member New(account: string, ?relationship: Create'Relationship, ?politicalExposure: string, ?phone: string, ?personToken: string, ?nationality: string, ?metadata: Map<string, string>, ?maidenName: string, ?lastNameKanji: string, ?lastNameKana: string, ?lastName: string, ?ssnLast4: string, ?idNumber: string, ?firstNameKanji: string, ?firstNameKana: string, ?firstName: string, ?expand: string list, ?email: string, ?documents: Create'Documents, ?dob: Choice<Create'DobDateOfBirthSpecs,string>, ?addressKanji: Create'AddressKanji, ?addressKana: Create'AddressKana, ?address: Create'Address, ?gender: string, ?verification: Create'Verification) =
                {
                    Account = account
                    Address = address
                    AddressKana = addressKana
                    AddressKanji = addressKanji
                    Dob = dob
                    Documents = documents
                    Email = email
                    Expand = expand
                    FirstName = firstName
                    FirstNameKana = firstNameKana
                    FirstNameKanji = firstNameKanji
                    Gender = gender
                    IdNumber = idNumber
                    LastName = lastName
                    LastNameKana = lastNameKana
                    LastNameKanji = lastNameKanji
                    MaidenName = maidenName
                    Metadata = metadata
                    Nationality = nationality
                    PersonToken = personToken
                    Phone = phone
                    PoliticalExposure = politicalExposure
                    Relationship = relationship
                    SsnLast4 = ssnLast4
                    Verification = verification
                }

        ///<p>Creates a new person.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/accounts/{options.Account}/persons"
            |> RestApi.postAsync<_, Person> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Account: string
            [<Config.Path>]Person: string
        }
        with
            static member New(account: string, person: string) =
                {
                    Account = account
                    Person = person
                }

        ///<p>Deletes an existing person’s relationship to the account’s legal entity. Any person with a relationship for an account can be deleted through the API, except if the person is the <code>account_opener</code>. If your integration is using the <code>executive</code> parameter, you cannot delete the only verified <code>executive</code> on file.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/accounts/{options.Account}/persons/{options.Person}"
            |> RestApi.deleteAsync<DeletedPerson> settings (Map.empty)

        type RetrieveOptions = {
            [<Config.Path>]Account: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Person: string
        }
        with
            static member New(account: string, person: string, ?expand: string list) =
                {
                    Account = account
                    Expand = expand
                    Person = person
                }

        ///<p>Retrieves an existing person.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/accounts/{options.Account}/persons/{options.Person}"
            |> RestApi.getAsync<Person> settings qs

        type Update'Address = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'AddressKana = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Update'AddressKanji = {
            ///City or ward.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Block or building number.
            [<Config.Form>]Line1: string option
            ///Building details.
            [<Config.Form>]Line2: string option
            ///Postal code.
            [<Config.Form>]PostalCode: string option
            ///Prefecture.
            [<Config.Form>]State: string option
            ///Town or cho-me.
            [<Config.Form>]Town: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string, ?town: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                    Town = town
                }

        type Update'DobDateOfBirthSpecs = {
            ///The day of birth, between 1 and 31.
            [<Config.Form>]Day: int option
            ///The month of birth, between 1 and 12.
            [<Config.Form>]Month: int option
            ///The four-digit year of birth.
            [<Config.Form>]Year: int option
        }
        with
            static member New(?day: int, ?month: int, ?year: int) =
                {
                    Day = day
                    Month = month
                    Year = year
                }

        type Update'DocumentsCompanyAuthorization = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Update'DocumentsPassport = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Update'DocumentsVisa = {
            ///One or more document ids returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `account_requirement`.
            [<Config.Form>]Files: string list option
        }
        with
            static member New(?files: string list) =
                {
                    Files = files
                }

        type Update'Documents = {
            ///One or more documents that demonstrate proof that this person is authorized to represent the company.
            [<Config.Form>]CompanyAuthorization: Update'DocumentsCompanyAuthorization option
            ///One or more documents showing the person's passport page with photo and personal data.
            [<Config.Form>]Passport: Update'DocumentsPassport option
            ///One or more documents showing the person's visa required for living in the country where they are residing.
            [<Config.Form>]Visa: Update'DocumentsVisa option
        }
        with
            static member New(?companyAuthorization: Update'DocumentsCompanyAuthorization, ?passport: Update'DocumentsPassport, ?visa: Update'DocumentsVisa) =
                {
                    CompanyAuthorization = companyAuthorization
                    Passport = passport
                    Visa = visa
                }

        type Update'Relationship = {
            ///Whether the person is a director of the account's legal entity. Currently only required for accounts in the EU. Directors are typically members of the governing board of the company, or responsible for ensuring the company meets its regulatory obligations.
            [<Config.Form>]Director: bool option
            ///Whether the person has significant responsibility to control, manage, or direct the organization.
            [<Config.Form>]Executive: bool option
            ///Whether the person is an owner of the account’s legal entity.
            [<Config.Form>]Owner: bool option
            ///The percent owned by the person of the account's legal entity.
            [<Config.Form>]PercentOwnership: Choice<decimal,string> option
            ///Whether the person is authorized as the primary representative of the account. This is the person nominated by the business to provide information about themselves, and general information about the account. There can only be one representative at any given time. At the time the account is created, this person should be set to the person responsible for opening the account.
            [<Config.Form>]Representative: bool option
            ///The person's title (e.g., CEO, Support Engineer).
            [<Config.Form>]Title: string option
        }
        with
            static member New(?director: bool, ?executive: bool, ?owner: bool, ?percentOwnership: Choice<decimal,string>, ?representative: bool, ?title: string) =
                {
                    Director = director
                    Executive = executive
                    Owner = owner
                    PercentOwnership = percentOwnership
                    Representative = representative
                    Title = title
                }

        type Update'VerificationAdditionalDocument = {
            ///The back of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Back: string option
            ///The front of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Update'VerificationDocument = {
            ///The back of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Back: string option
            ///The front of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`. The uploaded file needs to be a color image (smaller than 8,000px by 8,000px), in JPG, PNG, or PDF format, and less than 10 MB in size.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Update'Verification = {
            ///A document showing address, either a passport, local ID card, or utility bill from a well-known utility company.
            [<Config.Form>]AdditionalDocument: Update'VerificationAdditionalDocument option
            ///An identifying document, either a passport or local ID card.
            [<Config.Form>]Document: Update'VerificationDocument option
        }
        with
            static member New(?additionalDocument: Update'VerificationAdditionalDocument, ?document: Update'VerificationDocument) =
                {
                    AdditionalDocument = additionalDocument
                    Document = document
                }

        type UpdateOptions = {
            [<Config.Path>]Account: string
            [<Config.Path>]Person: string
            ///The person's address.
            [<Config.Form>]Address: Update'Address option
            ///The Kana variation of the person's address (Japan only).
            [<Config.Form>]AddressKana: Update'AddressKana option
            ///The Kanji variation of the person's address (Japan only).
            [<Config.Form>]AddressKanji: Update'AddressKanji option
            ///The person's date of birth.
            [<Config.Form>]Dob: Choice<Update'DobDateOfBirthSpecs,string> option
            ///Documents that may be submitted to satisfy various informational requests.
            [<Config.Form>]Documents: Update'Documents option
            ///The person's email address.
            [<Config.Form>]Email: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The person's first name.
            [<Config.Form>]FirstName: string option
            ///The Kana variation of the person's first name (Japan only).
            [<Config.Form>]FirstNameKana: string option
            ///The Kanji variation of the person's first name (Japan only).
            [<Config.Form>]FirstNameKanji: string option
            ///The person's gender (International regulations require either "male" or "female").
            [<Config.Form>]Gender: string option
            ///The person's ID number, as appropriate for their country. For example, a social security number in the U.S., social insurance number in Canada, etc. Instead of the number itself, you can also provide a [PII token provided by Stripe.js](https://stripe.com/docs/stripe.js#collecting-pii-data).
            [<Config.Form>]IdNumber: string option
            ///The person's last name.
            [<Config.Form>]LastName: string option
            ///The Kana variation of the person's last name (Japan only).
            [<Config.Form>]LastNameKana: string option
            ///The Kanji variation of the person's last name (Japan only).
            [<Config.Form>]LastNameKanji: string option
            ///The person's maiden name.
            [<Config.Form>]MaidenName: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The country where the person is a national. Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)), or "XX" if unavailable.
            [<Config.Form>]Nationality: string option
            ///A [person token](https://stripe.com/docs/connect/account-tokens), used to securely provide details to the person.
            [<Config.Form>]PersonToken: string option
            ///The person's phone number.
            [<Config.Form>]Phone: string option
            ///Indicates if the person or any of their representatives, family members, or other closely related persons, declares that they hold or have held an important public job or function, in any jurisdiction.
            [<Config.Form>]PoliticalExposure: string option
            ///The relationship that this person has with the account's legal entity.
            [<Config.Form>]Relationship: Update'Relationship option
            ///The last four digits of the person's Social Security number (U.S. only).
            [<Config.Form>]SsnLast4: string option
            ///The person's verification status.
            [<Config.Form>]Verification: Update'Verification option
        }
        with
            static member New(account: string, person: string, ?relationship: Update'Relationship, ?politicalExposure: string, ?phone: string, ?personToken: string, ?nationality: string, ?metadata: Map<string, string>, ?maidenName: string, ?lastNameKanji: string, ?lastNameKana: string, ?lastName: string, ?idNumber: string, ?gender: string, ?firstNameKanji: string, ?firstNameKana: string, ?firstName: string, ?expand: string list, ?email: string, ?documents: Update'Documents, ?dob: Choice<Update'DobDateOfBirthSpecs,string>, ?addressKanji: Update'AddressKanji, ?addressKana: Update'AddressKana, ?address: Update'Address, ?ssnLast4: string, ?verification: Update'Verification) =
                {
                    Account = account
                    Person = person
                    Address = address
                    AddressKana = addressKana
                    AddressKanji = addressKanji
                    Dob = dob
                    Documents = documents
                    Email = email
                    Expand = expand
                    FirstName = firstName
                    FirstNameKana = firstNameKana
                    FirstNameKanji = firstNameKanji
                    Gender = gender
                    IdNumber = idNumber
                    LastName = lastName
                    LastNameKana = lastNameKana
                    LastNameKanji = lastNameKanji
                    MaidenName = maidenName
                    Metadata = metadata
                    Nationality = nationality
                    PersonToken = personToken
                    Phone = phone
                    PoliticalExposure = politicalExposure
                    Relationship = relationship
                    SsnLast4 = ssnLast4
                    Verification = verification
                }

        ///<p>Updates an existing person.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/accounts/{options.Account}/persons/{options.Person}"
            |> RestApi.postAsync<_, Person> settings (Map.empty) options

    module AccountsReject =

        type RejectOptions = {
            [<Config.Path>]Account: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The reason for rejecting the account. Can be `fraud`, `terms_of_service`, or `other`.
            [<Config.Form>]Reason: string
        }
        with
            static member New(account: string, reason: string, ?expand: string list) =
                {
                    Account = account
                    Expand = expand
                    Reason = reason
                }

        ///<p>With <a href="/docs/connect">Connect</a>, you may flag accounts as suspicious.
        ///Test-mode Custom and Express accounts can be rejected at any time. Accounts created using live-mode keys may only be rejected once all balances are zero.</p>
        let Reject settings (options: RejectOptions) =
            $"/v1/accounts/{options.Account}/reject"
            |> RestApi.postAsync<_, Account> settings (Map.empty) options

    module ApplePayDomains =

        type ListOptions = {
            [<Config.Query>]DomainName: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?domainName: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    DomainName = domainName
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>List apple pay domains.</p>
        let List settings (options: ListOptions) =
            let qs = [("domain_name", options.DomainName |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/apple_pay/domains"
            |> RestApi.getAsync<ApplePayDomain list> settings qs

        type CreateOptions = {
            [<Config.Form>]DomainName: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(domainName: string, ?expand: string list) =
                {
                    DomainName = domainName
                    Expand = expand
                }

        ///<p>Create an apple pay domain.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/apple_pay/domains"
            |> RestApi.postAsync<_, ApplePayDomain> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Domain: string
        }
        with
            static member New(domain: string) =
                {
                    Domain = domain
                }

        ///<p>Delete an apple pay domain.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/apple_pay/domains/{options.Domain}"
            |> RestApi.deleteAsync<DeletedApplePayDomain> settings (Map.empty)

        type RetrieveOptions = {
            [<Config.Path>]Domain: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(domain: string, ?expand: string list) =
                {
                    Domain = domain
                    Expand = expand
                }

        ///<p>Retrieve an apple pay domain.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/apple_pay/domains/{options.Domain}"
            |> RestApi.getAsync<ApplePayDomain> settings qs

    module ApplicationFees =

        type ListOptions = {
            ///Only return application fees for the charge specified by this charge ID.
            [<Config.Query>]Charge: string option
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?charge: string, ?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Charge = charge
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of application fees you’ve previously collected. The application fees are returned in sorted order, with the most recent fees appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("charge", options.Charge |> box); ("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/application_fees"
            |> RestApi.getAsync<ApplicationFee list> settings qs

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
        }
        with
            static member New(id: string, ?expand: string list) =
                {
                    Expand = expand
                    Id = id
                }

        ///<p>Retrieves the details of an application fee that your account has collected. The same information is returned when refunding the application fee.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/application_fees/{options.Id}"
            |> RestApi.getAsync<ApplicationFee> settings qs

    module ApplicationFeesRefunds =

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Fee: string
            [<Config.Path>]Id: string
        }
        with
            static member New(fee: string, id: string, ?expand: string list) =
                {
                    Expand = expand
                    Fee = fee
                    Id = id
                }

        ///<p>By default, you can see the 10 most recent refunds stored directly on the application fee object, but you can also retrieve details about a specific refund stored on the application fee.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/application_fees/{options.Fee}/refunds/{options.Id}"
            |> RestApi.getAsync<FeeRefund> settings qs

        type UpdateOptions = {
            [<Config.Path>]Fee: string
            [<Config.Path>]Id: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(fee: string, id: string, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Fee = fee
                    Id = id
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Updates the specified application fee refund by setting the values of the parameters passed. Any parameters not provided will be left unchanged.
        ///This request only accepts metadata as an argument.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/application_fees/{options.Fee}/refunds/{options.Id}"
            |> RestApi.postAsync<_, FeeRefund> settings (Map.empty) options

        type ListOptions = {
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(id: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    EndingBefore = endingBefore
                    Expand = expand
                    Id = id
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>You can see a list of the refunds belonging to a specific application fee. Note that the 10 most recent refunds are always available by default on the application fee object. If you need more than those 10, you can use this API method and the <code>limit</code> and <code>starting_after</code> parameters to page through additional refunds.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/application_fees/{options.Id}/refunds"
            |> RestApi.getAsync<FeeRefund list> settings qs

        type CreateOptions = {
            [<Config.Path>]Id: string
            ///A positive integer, in _%s_, representing how much of this fee to refund. Can refund only up to the remaining unrefunded amount of the fee.
            [<Config.Form>]Amount: int option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(id: string, ?amount: int, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Id = id
                    Amount = amount
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Refunds an application fee that has previously been collected but not yet refunded.
        ///Funds will be refunded to the Stripe account from which the fee was originally collected.
        ///You can optionally refund only part of an application fee.
        ///You can do so multiple times, until the entire fee has been refunded.
        ///Once entirely refunded, an application fee can’t be refunded again.
        ///This method will raise an error when called on an already-refunded application fee,
        ///or when trying to refund more money than is left on an application fee.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/application_fees/{options.Id}/refunds"
            |> RestApi.postAsync<_, FeeRefund> settings (Map.empty) options

    module Balance =

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(?expand: string list) =
                {
                    Expand = expand
                }

        ///<p>Retrieves the current account balance, based on the authentication that was used to make the request.
        /// For a sample request, see <a href="/docs/connect/account-balances#accounting-for-negative-balances">Accounting for negative balances</a>.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/balance"
            |> RestApi.getAsync<Balance> settings qs

    module BalanceTransactions =

        type ListOptions = {
            [<Config.Query>]AvailableOn: int option
            [<Config.Query>]Created: int option
            ///Only return transactions in a certain currency. Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Query>]Currency: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///For automatic Stripe payouts only, only returns transactions that were paid out on the specified payout ID.
            [<Config.Query>]Payout: string option
            ///Only returns the original transaction.
            [<Config.Query>]Source: string option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only returns transactions of the given type. One of: `adjustment`, `advance`, `advance_funding`, `anticipation_repayment`, `application_fee`, `application_fee_refund`, `charge`, `connect_collection_transfer`, `contribution`, `issuing_authorization_hold`, `issuing_authorization_release`, `issuing_dispute`, `issuing_transaction`, `payment`, `payment_failure_refund`, `payment_refund`, `payout`, `payout_cancel`, `payout_failure`, `refund`, `refund_failure`, `reserve_transaction`, `reserved_funds`, `stripe_fee`, `stripe_fx_fee`, `tax_fee`, `topup`, `topup_reversal`, `transfer`, `transfer_cancel`, `transfer_failure`, or `transfer_refund`.
            [<Config.Query>]Type: string option
        }
        with
            static member New(?availableOn: int, ?created: int, ?currency: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?payout: string, ?source: string, ?startingAfter: string, ?type': string) =
                {
                    AvailableOn = availableOn
                    Created = created
                    Currency = currency
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    Payout = payout
                    Source = source
                    StartingAfter = startingAfter
                    Type = type'
                }

        ///<p>Returns a list of transactions that have contributed to the Stripe account balance (e.g., charges, transfers, and so forth). The transactions are returned in sorted order, with the most recent transactions appearing first.
        ///Note that this endpoint was previously called “Balance history” and used the path <code>/v1/balance/history</code>.</p>
        let List settings (options: ListOptions) =
            let qs = [("available_on", options.AvailableOn |> box); ("created", options.Created |> box); ("currency", options.Currency |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("payout", options.Payout |> box); ("source", options.Source |> box); ("starting_after", options.StartingAfter |> box); ("type", options.Type |> box)] |> Map.ofList
            $"/v1/balance_transactions"
            |> RestApi.getAsync<BalanceTransaction list> settings qs

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
        }
        with
            static member New(id: string, ?expand: string list) =
                {
                    Expand = expand
                    Id = id
                }

        ///<p>Retrieves the balance transaction with the given ID.
        ///Note that this endpoint previously used the path <code>/v1/balance/history/:id</code>.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/balance_transactions/{options.Id}"
            |> RestApi.getAsync<BalanceTransaction> settings qs

    module BillingPortalConfigurations =

        type ListOptions = {
            ///Only return configurations that are active or inactive (e.g., pass `true` to only list active configurations).
            [<Config.Query>]Active: bool option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///Only return the default or non-default configurations (e.g., pass `true` to only list the default configuration).
            [<Config.Query>]IsDefault: bool option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?active: bool, ?endingBefore: string, ?expand: string list, ?isDefault: bool, ?limit: int, ?startingAfter: string) =
                {
                    Active = active
                    EndingBefore = endingBefore
                    Expand = expand
                    IsDefault = isDefault
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of configurations that describe the functionality of the customer portal.</p>
        let List settings (options: ListOptions) =
            let qs = [("active", options.Active |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("is_default", options.IsDefault |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/billing_portal/configurations"
            |> RestApi.getAsync<BillingPortalConfiguration list> settings qs

        type Create'BusinessProfile = {
            ///The messaging shown to customers in the portal.
            [<Config.Form>]Headline: string option
            ///A link to the business’s publicly available privacy policy.
            [<Config.Form>]PrivacyPolicyUrl: string option
            ///A link to the business’s publicly available terms of service.
            [<Config.Form>]TermsOfServiceUrl: string option
        }
        with
            static member New(?headline: string, ?privacyPolicyUrl: string, ?termsOfServiceUrl: string) =
                {
                    Headline = headline
                    PrivacyPolicyUrl = privacyPolicyUrl
                    TermsOfServiceUrl = termsOfServiceUrl
                }

        type Create'FeaturesCustomerUpdateAllowedUpdates =
        | Address
        | Email
        | Phone
        | Shipping
        | TaxId

        type Create'FeaturesCustomerUpdate = {
            ///The types of customer updates that are supported. When empty, customers are not updateable.
            [<Config.Form>]AllowedUpdates: Choice<Create'FeaturesCustomerUpdateAllowedUpdates list,string> option
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?allowedUpdates: Choice<Create'FeaturesCustomerUpdateAllowedUpdates list,string>, ?enabled: bool) =
                {
                    AllowedUpdates = allowedUpdates
                    Enabled = enabled
                }

        type Create'FeaturesInvoiceHistory = {
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?enabled: bool) =
                {
                    Enabled = enabled
                }

        type Create'FeaturesPaymentMethodUpdate = {
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?enabled: bool) =
                {
                    Enabled = enabled
                }

        type Create'FeaturesSubscriptionCancelMode =
        | AtPeriodEnd
        | Immediately

        type Create'FeaturesSubscriptionCancelProrationBehavior =
        | AlwaysInvoice
        | CreateProrations
        | None'

        type Create'FeaturesSubscriptionCancel = {
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
            ///Whether to cancel subscriptions immediately or at the end of the billing period.
            [<Config.Form>]Mode: Create'FeaturesSubscriptionCancelMode option
            ///Whether to create prorations when canceling subscriptions. Possible values are `none` and `create_prorations`, which is only compatible with `mode=immediately`. No prorations are generated when canceling a subscription at the end of its natural billing period.
            [<Config.Form>]ProrationBehavior: Create'FeaturesSubscriptionCancelProrationBehavior option
        }
        with
            static member New(?enabled: bool, ?mode: Create'FeaturesSubscriptionCancelMode, ?prorationBehavior: Create'FeaturesSubscriptionCancelProrationBehavior) =
                {
                    Enabled = enabled
                    Mode = mode
                    ProrationBehavior = prorationBehavior
                }

        type Create'FeaturesSubscriptionPause = {
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?enabled: bool) =
                {
                    Enabled = enabled
                }

        type Create'FeaturesSubscriptionUpdateDefaultAllowedUpdates =
        | Price
        | PromotionCode
        | Quantity

        type Create'FeaturesSubscriptionUpdateProducts = {
            ///The list of prices IDs that a subscription can be updated to.
            [<Config.Form>]Prices: string list option
            ///The product id.
            [<Config.Form>]Product: string option
        }
        with
            static member New(?prices: string list, ?product: string) =
                {
                    Prices = prices
                    Product = product
                }

        type Create'FeaturesSubscriptionUpdateProrationBehavior =
        | AlwaysInvoice
        | CreateProrations
        | None'

        type Create'FeaturesSubscriptionUpdate = {
            ///The types of subscription updates that are supported. When empty, subscriptions are not updateable.
            [<Config.Form>]DefaultAllowedUpdates: Choice<Create'FeaturesSubscriptionUpdateDefaultAllowedUpdates list,string> option
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
            ///The list of products that support subscription updates.
            [<Config.Form>]Products: Choice<Create'FeaturesSubscriptionUpdateProducts list,string> option
            ///Determines how to handle prorations resulting from subscription updates. Valid values are `none`, `create_prorations`, and `always_invoice`.
            [<Config.Form>]ProrationBehavior: Create'FeaturesSubscriptionUpdateProrationBehavior option
        }
        with
            static member New(?defaultAllowedUpdates: Choice<Create'FeaturesSubscriptionUpdateDefaultAllowedUpdates list,string>, ?enabled: bool, ?products: Choice<Create'FeaturesSubscriptionUpdateProducts list,string>, ?prorationBehavior: Create'FeaturesSubscriptionUpdateProrationBehavior) =
                {
                    DefaultAllowedUpdates = defaultAllowedUpdates
                    Enabled = enabled
                    Products = products
                    ProrationBehavior = prorationBehavior
                }

        type Create'Features = {
            ///Information about updating the customer details in the portal.
            [<Config.Form>]CustomerUpdate: Create'FeaturesCustomerUpdate option
            ///Information about showing the billing history in the portal.
            [<Config.Form>]InvoiceHistory: Create'FeaturesInvoiceHistory option
            ///Information about updating payment methods in the portal.
            [<Config.Form>]PaymentMethodUpdate: Create'FeaturesPaymentMethodUpdate option
            ///Information about canceling subscriptions in the portal.
            [<Config.Form>]SubscriptionCancel: Create'FeaturesSubscriptionCancel option
            ///Information about pausing subscriptions in the portal.
            [<Config.Form>]SubscriptionPause: Create'FeaturesSubscriptionPause option
            ///Information about updating subscriptions in the portal.
            [<Config.Form>]SubscriptionUpdate: Create'FeaturesSubscriptionUpdate option
        }
        with
            static member New(?customerUpdate: Create'FeaturesCustomerUpdate, ?invoiceHistory: Create'FeaturesInvoiceHistory, ?paymentMethodUpdate: Create'FeaturesPaymentMethodUpdate, ?subscriptionCancel: Create'FeaturesSubscriptionCancel, ?subscriptionPause: Create'FeaturesSubscriptionPause, ?subscriptionUpdate: Create'FeaturesSubscriptionUpdate) =
                {
                    CustomerUpdate = customerUpdate
                    InvoiceHistory = invoiceHistory
                    PaymentMethodUpdate = paymentMethodUpdate
                    SubscriptionCancel = subscriptionCancel
                    SubscriptionPause = subscriptionPause
                    SubscriptionUpdate = subscriptionUpdate
                }

        type CreateOptions = {
            ///The business information shown to customers in the portal.
            [<Config.Form>]BusinessProfile: Create'BusinessProfile
            ///The default URL to redirect customers to when they click on the portal's link to return to your website. This can be [overriden](https://stripe.com/docs/api/customer_portal/sessions/create#create_portal_session-return_url) when creating the session.
            [<Config.Form>]DefaultReturnUrl: Choice<string,string> option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Information about the features available in the portal.
            [<Config.Form>]Features: Create'Features
        }
        with
            static member New(businessProfile: Create'BusinessProfile, features: Create'Features, ?defaultReturnUrl: Choice<string,string>, ?expand: string list) =
                {
                    BusinessProfile = businessProfile
                    DefaultReturnUrl = defaultReturnUrl
                    Expand = expand
                    Features = features
                }

        ///<p>Creates a configuration that describes the functionality and behavior of a PortalSession</p>
        let Create settings (options: CreateOptions) =
            $"/v1/billing_portal/configurations"
            |> RestApi.postAsync<_, BillingPortalConfiguration> settings (Map.empty) options

        type RetrieveOptions = {
            [<Config.Path>]Configuration: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(configuration: string, ?expand: string list) =
                {
                    Configuration = configuration
                    Expand = expand
                }

        ///<p>Retrieves a configuration that describes the functionality of the customer portal.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/billing_portal/configurations/{options.Configuration}"
            |> RestApi.getAsync<BillingPortalConfiguration> settings qs

        type Update'BusinessProfile = {
            ///The messaging shown to customers in the portal.
            [<Config.Form>]Headline: string option
            ///A link to the business’s publicly available privacy policy.
            [<Config.Form>]PrivacyPolicyUrl: string option
            ///A link to the business’s publicly available terms of service.
            [<Config.Form>]TermsOfServiceUrl: string option
        }
        with
            static member New(?headline: string, ?privacyPolicyUrl: string, ?termsOfServiceUrl: string) =
                {
                    Headline = headline
                    PrivacyPolicyUrl = privacyPolicyUrl
                    TermsOfServiceUrl = termsOfServiceUrl
                }

        type Update'FeaturesCustomerUpdateAllowedUpdates =
        | Address
        | Email
        | Phone
        | Shipping
        | TaxId

        type Update'FeaturesCustomerUpdate = {
            ///The types of customer updates that are supported. When empty, customers are not updateable.
            [<Config.Form>]AllowedUpdates: Choice<Update'FeaturesCustomerUpdateAllowedUpdates list,string> option
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?allowedUpdates: Choice<Update'FeaturesCustomerUpdateAllowedUpdates list,string>, ?enabled: bool) =
                {
                    AllowedUpdates = allowedUpdates
                    Enabled = enabled
                }

        type Update'FeaturesInvoiceHistory = {
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?enabled: bool) =
                {
                    Enabled = enabled
                }

        type Update'FeaturesPaymentMethodUpdate = {
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?enabled: bool) =
                {
                    Enabled = enabled
                }

        type Update'FeaturesSubscriptionCancelMode =
        | AtPeriodEnd
        | Immediately

        type Update'FeaturesSubscriptionCancelProrationBehavior =
        | AlwaysInvoice
        | CreateProrations
        | None'

        type Update'FeaturesSubscriptionCancel = {
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
            ///Whether to cancel subscriptions immediately or at the end of the billing period.
            [<Config.Form>]Mode: Update'FeaturesSubscriptionCancelMode option
            ///Whether to create prorations when canceling subscriptions. Possible values are `none` and `create_prorations`, which is only compatible with `mode=immediately`. No prorations are generated when canceling a subscription at the end of its natural billing period.
            [<Config.Form>]ProrationBehavior: Update'FeaturesSubscriptionCancelProrationBehavior option
        }
        with
            static member New(?enabled: bool, ?mode: Update'FeaturesSubscriptionCancelMode, ?prorationBehavior: Update'FeaturesSubscriptionCancelProrationBehavior) =
                {
                    Enabled = enabled
                    Mode = mode
                    ProrationBehavior = prorationBehavior
                }

        type Update'FeaturesSubscriptionPause = {
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?enabled: bool) =
                {
                    Enabled = enabled
                }

        type Update'FeaturesSubscriptionUpdateDefaultAllowedUpdates =
        | Price
        | PromotionCode
        | Quantity

        type Update'FeaturesSubscriptionUpdateProducts = {
            ///The list of prices IDs that a subscription can be updated to.
            [<Config.Form>]Prices: string list option
            ///The product id.
            [<Config.Form>]Product: string option
        }
        with
            static member New(?prices: string list, ?product: string) =
                {
                    Prices = prices
                    Product = product
                }

        type Update'FeaturesSubscriptionUpdateProrationBehavior =
        | AlwaysInvoice
        | CreateProrations
        | None'

        type Update'FeaturesSubscriptionUpdate = {
            ///The types of subscription updates that are supported. When empty, subscriptions are not updateable.
            [<Config.Form>]DefaultAllowedUpdates: Choice<Update'FeaturesSubscriptionUpdateDefaultAllowedUpdates list,string> option
            ///Whether the feature is enabled.
            [<Config.Form>]Enabled: bool option
            ///The list of products that support subscription updates.
            [<Config.Form>]Products: Choice<Update'FeaturesSubscriptionUpdateProducts list,string> option
            ///Determines how to handle prorations resulting from subscription updates. Valid values are `none`, `create_prorations`, and `always_invoice`.
            [<Config.Form>]ProrationBehavior: Update'FeaturesSubscriptionUpdateProrationBehavior option
        }
        with
            static member New(?defaultAllowedUpdates: Choice<Update'FeaturesSubscriptionUpdateDefaultAllowedUpdates list,string>, ?enabled: bool, ?products: Choice<Update'FeaturesSubscriptionUpdateProducts list,string>, ?prorationBehavior: Update'FeaturesSubscriptionUpdateProrationBehavior) =
                {
                    DefaultAllowedUpdates = defaultAllowedUpdates
                    Enabled = enabled
                    Products = products
                    ProrationBehavior = prorationBehavior
                }

        type Update'Features = {
            ///Information about updating the customer details in the portal.
            [<Config.Form>]CustomerUpdate: Update'FeaturesCustomerUpdate option
            ///Information about showing the billing history in the portal.
            [<Config.Form>]InvoiceHistory: Update'FeaturesInvoiceHistory option
            ///Information about updating payment methods in the portal.
            [<Config.Form>]PaymentMethodUpdate: Update'FeaturesPaymentMethodUpdate option
            ///Information about canceling subscriptions in the portal.
            [<Config.Form>]SubscriptionCancel: Update'FeaturesSubscriptionCancel option
            ///Information about pausing subscriptions in the portal.
            [<Config.Form>]SubscriptionPause: Update'FeaturesSubscriptionPause option
            ///Information about updating subscriptions in the portal.
            [<Config.Form>]SubscriptionUpdate: Update'FeaturesSubscriptionUpdate option
        }
        with
            static member New(?customerUpdate: Update'FeaturesCustomerUpdate, ?invoiceHistory: Update'FeaturesInvoiceHistory, ?paymentMethodUpdate: Update'FeaturesPaymentMethodUpdate, ?subscriptionCancel: Update'FeaturesSubscriptionCancel, ?subscriptionPause: Update'FeaturesSubscriptionPause, ?subscriptionUpdate: Update'FeaturesSubscriptionUpdate) =
                {
                    CustomerUpdate = customerUpdate
                    InvoiceHistory = invoiceHistory
                    PaymentMethodUpdate = paymentMethodUpdate
                    SubscriptionCancel = subscriptionCancel
                    SubscriptionPause = subscriptionPause
                    SubscriptionUpdate = subscriptionUpdate
                }

        type UpdateOptions = {
            [<Config.Path>]Configuration: string
            ///Whether the configuration is active and can be used to create portal sessions.
            [<Config.Form>]Active: bool option
            ///The business information shown to customers in the portal.
            [<Config.Form>]BusinessProfile: Update'BusinessProfile option
            ///The default URL to redirect customers to when they click on the portal's link to return to your website. This can be [overriden](https://stripe.com/docs/api/customer_portal/sessions/create#create_portal_session-return_url) when creating the session.
            [<Config.Form>]DefaultReturnUrl: Choice<string,string> option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Information about the features available in the portal.
            [<Config.Form>]Features: Update'Features option
        }
        with
            static member New(configuration: string, ?active: bool, ?businessProfile: Update'BusinessProfile, ?defaultReturnUrl: Choice<string,string>, ?expand: string list, ?features: Update'Features) =
                {
                    Configuration = configuration
                    Active = active
                    BusinessProfile = businessProfile
                    DefaultReturnUrl = defaultReturnUrl
                    Expand = expand
                    Features = features
                }

        ///<p>Updates a configuration that describes the functionality of the customer portal.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/billing_portal/configurations/{options.Configuration}"
            |> RestApi.postAsync<_, BillingPortalConfiguration> settings (Map.empty) options

    module BillingPortalSessions =

        type CreateOptions = {
            ///The ID of an existing [configuration](https://stripe.com/docs/api/customer_portal/configuration) to use for this session, describing its functionality and features. If not specified, the session uses the default configuration.
            [<Config.Form>]Configuration: string option
            ///The ID of an existing customer.
            [<Config.Form>]Customer: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The `on_behalf_of` account to use for this session. When specified, only subscriptions and invoices with this `on_behalf_of` account appear in the portal. For more information, see the [docs](https://stripe.com/docs/connect/charges-transfers#on-behalf-of). Use the [Accounts API](https://stripe.com/docs/api/accounts/object#account_object-settings-branding) to modify the `on_behalf_of` account's branding settings, which the portal displays.
            [<Config.Form>]OnBehalfOf: string option
            ///The default URL to redirect customers to when they click on the portal's link to return to your website.
            [<Config.Form>]ReturnUrl: string option
        }
        with
            static member New(customer: string, ?configuration: string, ?expand: string list, ?onBehalfOf: string, ?returnUrl: string) =
                {
                    Configuration = configuration
                    Customer = customer
                    Expand = expand
                    OnBehalfOf = onBehalfOf
                    ReturnUrl = returnUrl
                }

        ///<p>Creates a session of the customer portal.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/billing_portal/sessions"
            |> RestApi.postAsync<_, BillingPortalSession> settings (Map.empty) options

    module BitcoinReceivers =

        type ListOptions = {
            ///Filter for active receivers.
            [<Config.Query>]Active: bool option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///Filter for filled receivers.
            [<Config.Query>]Filled: bool option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Filter for receivers with uncaptured funds.
            [<Config.Query>]UncapturedFunds: bool option
        }
        with
            static member New(?active: bool, ?endingBefore: string, ?expand: string list, ?filled: bool, ?limit: int, ?startingAfter: string, ?uncapturedFunds: bool) =
                {
                    Active = active
                    EndingBefore = endingBefore
                    Expand = expand
                    Filled = filled
                    Limit = limit
                    StartingAfter = startingAfter
                    UncapturedFunds = uncapturedFunds
                }

        ///<p>Returns a list of your receivers. Receivers are returned sorted by creation date, with the most recently created receivers appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("active", options.Active |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("filled", options.Filled |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("uncaptured_funds", options.UncapturedFunds |> box)] |> Map.ofList
            $"/v1/bitcoin/receivers"
            |> RestApi.getAsync<BitcoinReceiver list> settings qs

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
        }
        with
            static member New(id: string, ?expand: string list) =
                {
                    Expand = expand
                    Id = id
                }

        ///<p>Retrieves the Bitcoin receiver with the given ID.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/bitcoin/receivers/{options.Id}"
            |> RestApi.getAsync<BitcoinReceiver> settings qs

    module BitcoinReceiversTransactions =

        type ListOptions = {
            ///Only return transactions for the customer specified by this customer ID.
            [<Config.Query>]Customer: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            [<Config.Path>]Receiver: string
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(receiver: string, ?customer: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Customer = customer
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    Receiver = receiver
                    StartingAfter = startingAfter
                }

        ///<p>List bitcoin transacitons for a given receiver.</p>
        let List settings (options: ListOptions) =
            let qs = [("customer", options.Customer |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/bitcoin/receivers/{options.Receiver}/transactions"
            |> RestApi.getAsync<BitcoinTransaction list> settings qs

    module Charges =

        type ListOptions = {
            [<Config.Query>]Created: int option
            ///Only return charges for the customer specified by this customer ID.
            [<Config.Query>]Customer: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///Only return charges that were created by the PaymentIntent specified by this PaymentIntent ID.
            [<Config.Query>]PaymentIntent: string option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return charges for this transfer group.
            [<Config.Query>]TransferGroup: string option
        }
        with
            static member New(?created: int, ?customer: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?paymentIntent: string, ?startingAfter: string, ?transferGroup: string) =
                {
                    Created = created
                    Customer = customer
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    PaymentIntent = paymentIntent
                    StartingAfter = startingAfter
                    TransferGroup = transferGroup
                }

        ///<p>Returns a list of charges you’ve previously created. The charges are returned in sorted order, with the most recent charges appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("customer", options.Customer |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("payment_intent", options.PaymentIntent |> box); ("starting_after", options.StartingAfter |> box); ("transfer_group", options.TransferGroup |> box)] |> Map.ofList
            $"/v1/charges"
            |> RestApi.getAsync<Charge list> settings qs

        type Create'Destination = {
            ///ID of an existing, connected Stripe account.
            [<Config.Form>]Account: string option
            ///The amount to transfer to the destination account without creating an `Application Fee` object. Cannot be combined with the `application_fee` parameter. Must be less than or equal to the charge amount.
            [<Config.Form>]Amount: int option
        }
        with
            static member New(?account: string, ?amount: int) =
                {
                    Account = account
                    Amount = amount
                }

        type Create'ShippingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'Shipping = {
            ///Shipping address.
            [<Config.Form>]Address: Create'ShippingAddress option
            ///The delivery service that shipped a physical product, such as Fedex, UPS, USPS, etc.
            [<Config.Form>]Carrier: string option
            ///Recipient name.
            [<Config.Form>]Name: string option
            ///Recipient phone (including extension).
            [<Config.Form>]Phone: string option
            ///The tracking number for a physical product, obtained from the delivery service. If multiple tracking numbers were generated for this purchase, please separate them with commas.
            [<Config.Form>]TrackingNumber: string option
        }
        with
            static member New(?address: Create'ShippingAddress, ?carrier: string, ?name: string, ?phone: string, ?trackingNumber: string) =
                {
                    Address = address
                    Carrier = carrier
                    Name = name
                    Phone = phone
                    TrackingNumber = trackingNumber
                }

        type Create'TransferData = {
            ///The amount transferred to the destination account, if specified. By default, the entire charge amount is transferred to the destination account.
            [<Config.Form>]Amount: int option
            ///ID of an existing, connected Stripe account.
            [<Config.Form>]Destination: string option
        }
        with
            static member New(?amount: int, ?destination: string) =
                {
                    Amount = amount
                    Destination = destination
                }

        type CreateOptions = {
            ///Amount intended to be collected by this payment. A positive integer representing how much to charge in the [smallest currency unit](https://stripe.com/docs/currencies#zero-decimal) (e.g., 100 cents to charge $1.00 or 100 to charge ¥100, a zero-decimal currency). The minimum amount is $0.50 US or [equivalent in charge currency](https://stripe.com/docs/currencies#minimum-and-maximum-charge-amounts). The amount value supports up to eight digits (e.g., a value of 99999999 for a USD charge of $999,999.99).
            [<Config.Form>]Amount: int option
            [<Config.Form>]ApplicationFee: int option
            ///A fee in %s that will be applied to the charge and transferred to the application owner's Stripe account. The request must be made with an OAuth key or the `Stripe-Account` header in order to take an application fee. For more information, see the application fees [documentation](https://stripe.com/docs/connect/direct-charges#collecting-fees).
            [<Config.Form>]ApplicationFeeAmount: int option
            ///Whether to immediately capture the charge. Defaults to `true`. When `false`, the charge issues an authorization (or pre-authorization), and will need to be [captured](https://stripe.com/docs/api#capture_charge) later. Uncaptured charges expire in _seven days_. For more information, see the [authorizing charges and settling later](https://stripe.com/docs/charges/placing-a-hold) documentation.
            [<Config.Form>]Capture: bool option
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string option
            ///The ID of an existing customer that will be charged in this request.
            [<Config.Form>]Customer: string option
            ///An arbitrary string which you can attach to a `Charge` object. It is displayed when in the web interface alongside the charge. Note that if you use Stripe to send automatic email receipts to your customers, your receipt emails will include the `description` of the charge(s) that they are describing.
            [<Config.Form>]Description: string option
            [<Config.Form>]Destination: Create'Destination option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The Stripe account ID for which these funds are intended. Automatically set if you use the `destination` parameter. For details, see [Creating Separate Charges and Transfers](https://stripe.com/docs/connect/charges-transfers#on-behalf-of).
            [<Config.Form>]OnBehalfOf: string option
            ///The email address to which this charge's [receipt](https://stripe.com/docs/dashboard/receipts) will be sent. The receipt will not be sent until the charge is paid, and no receipts will be sent for test mode charges. If this charge is for a [Customer](https://stripe.com/docs/api/customers/object), the email address specified here will override the customer's email address. If `receipt_email` is specified for a charge in live mode, a receipt will be sent regardless of your [email settings](https://dashboard.stripe.com/account/emails).
            [<Config.Form>]ReceiptEmail: string option
            ///Shipping information for the charge. Helps prevent fraud on charges for physical goods.
            [<Config.Form>]Shipping: Create'Shipping option
            ///A payment source to be charged. This can be the ID of a [card](https://stripe.com/docs/api#cards) (i.e., credit or debit card), a [bank account](https://stripe.com/docs/api#bank_accounts), a [source](https://stripe.com/docs/api#sources), a [token](https://stripe.com/docs/api#tokens), or a [connected account](https://stripe.com/docs/connect/account-debits#charging-a-connected-account). For certain sources---namely, [cards](https://stripe.com/docs/api#cards), [bank accounts](https://stripe.com/docs/api#bank_accounts), and attached [sources](https://stripe.com/docs/api#sources)---you must also pass the ID of the associated customer.
            [<Config.Form>]Source: string option
            ///For card charges, use `statement_descriptor_suffix` instead. Otherwise, you can use this value as the complete description of a charge on your customers’ statements. Must contain at least one letter, maximum 22 characters.
            [<Config.Form>]StatementDescriptor: string option
            ///Provides information about the charge that customers see on their statements. Concatenated with the prefix (shortened descriptor) or statement descriptor that’s set on the account to form the complete statement descriptor. Maximum 22 characters for the concatenated descriptor.
            [<Config.Form>]StatementDescriptorSuffix: string option
            ///An optional dictionary including the account to automatically transfer to as part of a destination charge. [See the Connect documentation](https://stripe.com/docs/connect/destination-charges) for details.
            [<Config.Form>]TransferData: Create'TransferData option
            ///A string that identifies this transaction as part of a group. For details, see [Grouping transactions](https://stripe.com/docs/connect/charges-transfers#transfer-options).
            [<Config.Form>]TransferGroup: string option
        }
        with
            static member New(?amount: int, ?statementDescriptorSuffix: string, ?statementDescriptor: string, ?source: string, ?shipping: Create'Shipping, ?receiptEmail: string, ?onBehalfOf: string, ?metadata: Map<string, string>, ?expand: string list, ?destination: Create'Destination, ?description: string, ?customer: string, ?currency: string, ?capture: bool, ?applicationFeeAmount: int, ?applicationFee: int, ?transferData: Create'TransferData, ?transferGroup: string) =
                {
                    Amount = amount
                    ApplicationFee = applicationFee
                    ApplicationFeeAmount = applicationFeeAmount
                    Capture = capture
                    Currency = currency
                    Customer = customer
                    Description = description
                    Destination = destination
                    Expand = expand
                    Metadata = metadata
                    OnBehalfOf = onBehalfOf
                    ReceiptEmail = receiptEmail
                    Shipping = shipping
                    Source = source
                    StatementDescriptor = statementDescriptor
                    StatementDescriptorSuffix = statementDescriptorSuffix
                    TransferData = transferData
                    TransferGroup = transferGroup
                }

        ///<p>To charge a credit card or other payment source, you create a <code>Charge</code> object. If your API key is in test mode, the supplied payment source (e.g., card) won’t actually be charged, although everything else will occur as if in live mode. (Stripe assumes that the charge would have completed successfully).</p>
        let Create settings (options: CreateOptions) =
            $"/v1/charges"
            |> RestApi.postAsync<_, Charge> settings (Map.empty) options

        type RetrieveOptions = {
            [<Config.Path>]Charge: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(charge: string, ?expand: string list) =
                {
                    Charge = charge
                    Expand = expand
                }

        ///<p>Retrieves the details of a charge that has previously been created. Supply the unique charge ID that was returned from your previous request, and Stripe will return the corresponding charge information. The same information is returned when creating or refunding the charge.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/charges/{options.Charge}"
            |> RestApi.getAsync<Charge> settings qs

        type Update'FraudDetailsUserReport =
        | Fraudulent
        | Safe

        type Update'FraudDetails = {
            ///Either `safe` or `fraudulent`.
            [<Config.Form>]UserReport: Update'FraudDetailsUserReport option
        }
        with
            static member New(?userReport: Update'FraudDetailsUserReport) =
                {
                    UserReport = userReport
                }

        type Update'ShippingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'Shipping = {
            ///Shipping address.
            [<Config.Form>]Address: Update'ShippingAddress option
            ///The delivery service that shipped a physical product, such as Fedex, UPS, USPS, etc.
            [<Config.Form>]Carrier: string option
            ///Recipient name.
            [<Config.Form>]Name: string option
            ///Recipient phone (including extension).
            [<Config.Form>]Phone: string option
            ///The tracking number for a physical product, obtained from the delivery service. If multiple tracking numbers were generated for this purchase, please separate them with commas.
            [<Config.Form>]TrackingNumber: string option
        }
        with
            static member New(?address: Update'ShippingAddress, ?carrier: string, ?name: string, ?phone: string, ?trackingNumber: string) =
                {
                    Address = address
                    Carrier = carrier
                    Name = name
                    Phone = phone
                    TrackingNumber = trackingNumber
                }

        type UpdateOptions = {
            [<Config.Path>]Charge: string
            ///The ID of an existing customer that will be associated with this request. This field may only be updated if there is no existing associated customer with this charge.
            [<Config.Form>]Customer: string option
            ///An arbitrary string which you can attach to a charge object. It is displayed when in the web interface alongside the charge. Note that if you use Stripe to send automatic email receipts to your customers, your receipt emails will include the `description` of the charge(s) that they are describing.
            [<Config.Form>]Description: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///A set of key-value pairs you can attach to a charge giving information about its riskiness. If you believe a charge is fraudulent, include a `user_report` key with a value of `fraudulent`. If you believe a charge is safe, include a `user_report` key with a value of `safe`. Stripe will use the information you send to improve our fraud detection algorithms.
            [<Config.Form>]FraudDetails: Update'FraudDetails option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///This is the email address that the receipt for this charge will be sent to. If this field is updated, then a new email receipt will be sent to the updated address.
            [<Config.Form>]ReceiptEmail: string option
            ///Shipping information for the charge. Helps prevent fraud on charges for physical goods.
            [<Config.Form>]Shipping: Update'Shipping option
            ///A string that identifies this transaction as part of a group. `transfer_group` may only be provided if it has not been set. See the [Connect documentation](https://stripe.com/docs/connect/charges-transfers#transfer-options) for details.
            [<Config.Form>]TransferGroup: string option
        }
        with
            static member New(charge: string, ?customer: string, ?description: string, ?expand: string list, ?fraudDetails: Update'FraudDetails, ?metadata: Map<string, string>, ?receiptEmail: string, ?shipping: Update'Shipping, ?transferGroup: string) =
                {
                    Charge = charge
                    Customer = customer
                    Description = description
                    Expand = expand
                    FraudDetails = fraudDetails
                    Metadata = metadata
                    ReceiptEmail = receiptEmail
                    Shipping = shipping
                    TransferGroup = transferGroup
                }

        ///<p>Updates the specified charge by setting the values of the parameters passed. Any parameters not provided will be left unchanged.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/charges/{options.Charge}"
            |> RestApi.postAsync<_, Charge> settings (Map.empty) options

    module ChargesCapture =

        type Capture'TransferData = {
            ///The amount transferred to the destination account, if specified. By default, the entire charge amount is transferred to the destination account.
            [<Config.Form>]Amount: int option
        }
        with
            static member New(?amount: int) =
                {
                    Amount = amount
                }

        type CaptureOptions = {
            [<Config.Path>]Charge: string
            ///The amount to capture, which must be less than or equal to the original amount. Any additional amount will be automatically refunded.
            [<Config.Form>]Amount: int option
            ///An application fee to add on to this charge.
            [<Config.Form>]ApplicationFee: int option
            ///An application fee amount to add on to this charge, which must be less than or equal to the original amount.
            [<Config.Form>]ApplicationFeeAmount: int option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The email address to send this charge's receipt to. This will override the previously-specified email address for this charge, if one was set. Receipts will not be sent in test mode.
            [<Config.Form>]ReceiptEmail: string option
            ///For card charges, use `statement_descriptor_suffix` instead. Otherwise, you can use this value as the complete description of a charge on your customers’ statements. Must contain at least one letter, maximum 22 characters.
            [<Config.Form>]StatementDescriptor: string option
            ///Provides information about the charge that customers see on their statements. Concatenated with the prefix (shortened descriptor) or statement descriptor that’s set on the account to form the complete statement descriptor. Maximum 22 characters for the concatenated descriptor.
            [<Config.Form>]StatementDescriptorSuffix: string option
            ///An optional dictionary including the account to automatically transfer to as part of a destination charge. [See the Connect documentation](https://stripe.com/docs/connect/destination-charges) for details.
            [<Config.Form>]TransferData: Capture'TransferData option
            ///A string that identifies this transaction as part of a group. `transfer_group` may only be provided if it has not been set. See the [Connect documentation](https://stripe.com/docs/connect/charges-transfers#transfer-options) for details.
            [<Config.Form>]TransferGroup: string option
        }
        with
            static member New(charge: string, ?amount: int, ?applicationFee: int, ?applicationFeeAmount: int, ?expand: string list, ?receiptEmail: string, ?statementDescriptor: string, ?statementDescriptorSuffix: string, ?transferData: Capture'TransferData, ?transferGroup: string) =
                {
                    Charge = charge
                    Amount = amount
                    ApplicationFee = applicationFee
                    ApplicationFeeAmount = applicationFeeAmount
                    Expand = expand
                    ReceiptEmail = receiptEmail
                    StatementDescriptor = statementDescriptor
                    StatementDescriptorSuffix = statementDescriptorSuffix
                    TransferData = transferData
                    TransferGroup = transferGroup
                }

        ///<p>Capture the payment of an existing, uncaptured, charge. This is the second half of the two-step payment flow, where first you <a href="#create_charge">created a charge</a> with the capture option set to false.
        ///Uncaptured payments expire exactly seven days after they are created. If they are not captured by that point in time, they will be marked as refunded and will no longer be capturable.</p>
        let Capture settings (options: CaptureOptions) =
            $"/v1/charges/{options.Charge}/capture"
            |> RestApi.postAsync<_, Charge> settings (Map.empty) options

    module ChargesRefunds =

        type ListOptions = {
            [<Config.Path>]Charge: string
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(charge: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Charge = charge
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>You can see a list of the refunds belonging to a specific charge. Note that the 10 most recent refunds are always available by default on the charge object. If you need more than those 10, you can use this API method and the <code>limit</code> and <code>starting_after</code> parameters to page through additional refunds.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/charges/{options.Charge}/refunds"
            |> RestApi.getAsync<Refund list> settings qs

        type RetrieveOptions = {
            [<Config.Path>]Charge: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Refund: string
        }
        with
            static member New(charge: string, refund: string, ?expand: string list) =
                {
                    Charge = charge
                    Expand = expand
                    Refund = refund
                }

        ///<p>Retrieves the details of an existing refund.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/charges/{options.Charge}/refunds/{options.Refund}"
            |> RestApi.getAsync<Refund> settings qs

    module CheckoutSessions =

        type ListOptions = {
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///Only return the Checkout Session for the PaymentIntent specified.
            [<Config.Query>]PaymentIntent: string option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return the Checkout Session for the subscription specified.
            [<Config.Query>]Subscription: string option
        }
        with
            static member New(?endingBefore: string, ?expand: string list, ?limit: int, ?paymentIntent: string, ?startingAfter: string, ?subscription: string) =
                {
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    PaymentIntent = paymentIntent
                    StartingAfter = startingAfter
                    Subscription = subscription
                }

        ///<p>Returns a list of Checkout Sessions.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("payment_intent", options.PaymentIntent |> box); ("starting_after", options.StartingAfter |> box); ("subscription", options.Subscription |> box)] |> Map.ofList
            $"/v1/checkout/sessions"
            |> RestApi.getAsync<CheckoutSession list> settings qs

        type Create'AutomaticTax = {
            ///Set to true to enable automatic taxes.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?enabled: bool) =
                {
                    Enabled = enabled
                }

        type Create'CustomerUpdateAddress =
        | Auto
        | Never

        type Create'CustomerUpdateName =
        | Auto
        | Never

        type Create'CustomerUpdateShipping =
        | Auto
        | Never

        type Create'CustomerUpdate = {
            ///Describes whether Checkout saves the billing address onto `customer.address`.
            ///To always collect a full billing address, use `billing_address_collection`. Defaults to `never`.
            [<Config.Form>]Address: Create'CustomerUpdateAddress option
            ///Describes whether Checkout saves the name onto `customer.name`. Defaults to `never`.
            [<Config.Form>]Name: Create'CustomerUpdateName option
            ///Describes whether Checkout saves shipping information onto `customer.shipping`.
            ///To collect shipping information, use `shipping_address_collection`. Defaults to `never`.
            [<Config.Form>]Shipping: Create'CustomerUpdateShipping option
        }
        with
            static member New(?address: Create'CustomerUpdateAddress, ?name: Create'CustomerUpdateName, ?shipping: Create'CustomerUpdateShipping) =
                {
                    Address = address
                    Name = name
                    Shipping = shipping
                }

        type Create'Discounts = {
            ///The ID of the coupon to apply to this Session.
            [<Config.Form>]Coupon: string option
            ///The ID of a promotion code to apply to this Session.
            [<Config.Form>]PromotionCode: string option
        }
        with
            static member New(?coupon: string, ?promotionCode: string) =
                {
                    Coupon = coupon
                    PromotionCode = promotionCode
                }

        type Create'LineItemsAdjustableQuantity = {
            ///Set to true if the quantity can be adjusted to any non-negative integer. By default customers will be able to remove the line item by setting the quantity to 0.
            [<Config.Form>]Enabled: bool option
            ///The maximum quantity the customer can purchase for the Checkout Session. By default this value is 99.
            [<Config.Form>]Maximum: int option
            ///The minimum quantity the customer must purchase for the Checkout Session. By default this value is 0.
            [<Config.Form>]Minimum: int option
        }
        with
            static member New(?enabled: bool, ?maximum: int, ?minimum: int) =
                {
                    Enabled = enabled
                    Maximum = maximum
                    Minimum = minimum
                }

        type Create'LineItemsPriceDataProductData = {
            ///The product's description, meant to be displayable to the customer. Use this field to optionally store a long form explanation of the product being sold for your own rendering purposes.
            [<Config.Form>]Description: string option
            ///A list of up to 8 URLs of images for this product, meant to be displayable to the customer.
            [<Config.Form>]Images: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The product's name, meant to be displayable to the customer. Whenever this product is sold via a subscription, name will show up on associated invoice line item descriptions.
            [<Config.Form>]Name: string option
            ///A [tax code](https://stripe.com/docs/tax/tax-codes) ID.
            [<Config.Form>]TaxCode: string option
        }
        with
            static member New(?description: string, ?images: string list, ?metadata: Map<string, string>, ?name: string, ?taxCode: string) =
                {
                    Description = description
                    Images = images
                    Metadata = metadata
                    Name = name
                    TaxCode = taxCode
                }

        type Create'LineItemsPriceDataRecurringInterval =
        | Day
        | Month
        | Week
        | Year

        type Create'LineItemsPriceDataRecurring = {
            ///Specifies billing frequency. Either `day`, `week`, `month` or `year`.
            [<Config.Form>]Interval: Create'LineItemsPriceDataRecurringInterval option
            ///The number of intervals between subscription billings. For example, `interval=month` and `interval_count=3` bills every 3 months. Maximum of one year interval allowed (1 year, 12 months, or 52 weeks).
            [<Config.Form>]IntervalCount: int option
        }
        with
            static member New(?interval: Create'LineItemsPriceDataRecurringInterval, ?intervalCount: int) =
                {
                    Interval = interval
                    IntervalCount = intervalCount
                }

        type Create'LineItemsPriceDataTaxBehavior =
        | Exclusive
        | Inclusive
        | Unspecified

        type Create'LineItemsPriceData = {
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string option
            ///The ID of the product that this price will belong to. One of `product` or `product_data` is required.
            [<Config.Form>]Product: string option
            ///Data used to generate a new product object inline. One of `product` or `product_data` is required.
            [<Config.Form>]ProductData: Create'LineItemsPriceDataProductData option
            ///The recurring components of a price such as `interval` and `usage_type`.
            [<Config.Form>]Recurring: Create'LineItemsPriceDataRecurring option
            ///Specifies whether the price is considered inclusive of taxes or exclusive of taxes. One of `inclusive`, `exclusive`, or `unspecified`. Once specified as either `inclusive` or `exclusive`, it cannot be changed.
            [<Config.Form>]TaxBehavior: Create'LineItemsPriceDataTaxBehavior option
            ///A non-negative integer in %s representing how much to charge. One of `unit_amount` or `unit_amount_decimal` is required.
            [<Config.Form>]UnitAmount: int option
            ///Same as `unit_amount`, but accepts a decimal value in %s with at most 12 decimal places. Only one of `unit_amount` and `unit_amount_decimal` can be set.
            [<Config.Form>]UnitAmountDecimal: string option
        }
        with
            static member New(?currency: string, ?product: string, ?productData: Create'LineItemsPriceDataProductData, ?recurring: Create'LineItemsPriceDataRecurring, ?taxBehavior: Create'LineItemsPriceDataTaxBehavior, ?unitAmount: int, ?unitAmountDecimal: string) =
                {
                    Currency = currency
                    Product = product
                    ProductData = productData
                    Recurring = recurring
                    TaxBehavior = taxBehavior
                    UnitAmount = unitAmount
                    UnitAmountDecimal = unitAmountDecimal
                }

        type Create'LineItems = {
            ///When set, provides configuration for this item’s quantity to be adjusted by the customer during Checkout.
            [<Config.Form>]AdjustableQuantity: Create'LineItemsAdjustableQuantity option
            ///The amount to be collected per unit of the line item. If specified, must also pass `currency` and `name`.
            [<Config.Form>]Amount: int option
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies). Required if `amount` is passed.
            [<Config.Form>]Currency: string option
            ///The description for the line item, to be displayed on the Checkout page.
            ///If using `price` or `price_data`, will default to the name of the associated product.
            [<Config.Form>]Description: string option
            ///The [tax rates](https://stripe.com/docs/api/tax_rates) that will be applied to this line item depending on the customer's billing/shipping address. We currently support the following countries: US, GB, AU, and all countries in the EU.
            [<Config.Form>]DynamicTaxRates: string list option
            ///A list of image URLs representing this line item. Each image can be up to 5 MB in size. If passing `price` or `price_data`, specify images on the associated product instead.
            [<Config.Form>]Images: string list option
            ///The name for the item to be displayed on the Checkout page. Required if `amount` is passed.
            [<Config.Form>]Name: string option
            ///The ID of the [Price](https://stripe.com/docs/api/prices) or [Plan](https://stripe.com/docs/api/plans) object. One of `price`, `price_data` or `amount` is required.
            [<Config.Form>]Price: string option
            ///Data used to generate a new [Price](https://stripe.com/docs/api/prices) object inline. One of `price`, `price_data` or `amount` is required.
            [<Config.Form>]PriceData: Create'LineItemsPriceData option
            ///The quantity of the line item being purchased. Quantity should not be defined when `recurring.usage_type=metered`.
            [<Config.Form>]Quantity: int option
            ///The [tax rates](https://stripe.com/docs/api/tax_rates) which apply to this line item.
            [<Config.Form>]TaxRates: string list option
        }
        with
            static member New(?adjustableQuantity: Create'LineItemsAdjustableQuantity, ?amount: int, ?currency: string, ?description: string, ?dynamicTaxRates: string list, ?images: string list, ?name: string, ?price: string, ?priceData: Create'LineItemsPriceData, ?quantity: int, ?taxRates: string list) =
                {
                    AdjustableQuantity = adjustableQuantity
                    Amount = amount
                    Currency = currency
                    Description = description
                    DynamicTaxRates = dynamicTaxRates
                    Images = images
                    Name = name
                    Price = price
                    PriceData = priceData
                    Quantity = quantity
                    TaxRates = taxRates
                }

        type Create'PaymentIntentDataShippingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'PaymentIntentDataShipping = {
            ///Shipping address.
            [<Config.Form>]Address: Create'PaymentIntentDataShippingAddress option
            ///The delivery service that shipped a physical product, such as Fedex, UPS, USPS, etc.
            [<Config.Form>]Carrier: string option
            ///Recipient name.
            [<Config.Form>]Name: string option
            ///Recipient phone (including extension).
            [<Config.Form>]Phone: string option
            ///The tracking number for a physical product, obtained from the delivery service. If multiple tracking numbers were generated for this purchase, please separate them with commas.
            [<Config.Form>]TrackingNumber: string option
        }
        with
            static member New(?address: Create'PaymentIntentDataShippingAddress, ?carrier: string, ?name: string, ?phone: string, ?trackingNumber: string) =
                {
                    Address = address
                    Carrier = carrier
                    Name = name
                    Phone = phone
                    TrackingNumber = trackingNumber
                }

        type Create'PaymentIntentDataTransferData = {
            ///The amount that will be transferred automatically when a charge succeeds.
            [<Config.Form>]Amount: int option
            ///If specified, successful charges will be attributed to the destination
            ///account for tax reporting, and the funds from charges will be transferred
            ///to the destination account. The ID of the resulting transfer will be
            ///returned on the successful charge's `transfer` field.
            [<Config.Form>]Destination: string option
        }
        with
            static member New(?amount: int, ?destination: string) =
                {
                    Amount = amount
                    Destination = destination
                }

        type Create'PaymentIntentDataCaptureMethod =
        | Automatic
        | Manual

        type Create'PaymentIntentDataSetupFutureUsage =
        | OffSession
        | OnSession

        type Create'PaymentIntentData = {
            ///The amount of the application fee (if any) that will be requested to be applied to the payment and
            ///transferred to the application owner's Stripe account. The amount of the application fee collected
            ///will be capped at the total payment amount. To use an application fee, the request must be made on
            ///behalf of another account, using the `Stripe-Account` header or an OAuth key. For more information,
            ///see the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts).
            [<Config.Form>]ApplicationFeeAmount: int option
            ///Controls when the funds will be captured from the customer's account.
            [<Config.Form>]CaptureMethod: Create'PaymentIntentDataCaptureMethod option
            ///An arbitrary string attached to the object. Often useful for displaying to users.
            [<Config.Form>]Description: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The Stripe account ID for which these funds are intended. For details,
            ///see the PaymentIntents [use case for connected
            ///accounts](/docs/payments/connected-accounts).
            [<Config.Form>]OnBehalfOf: string option
            ///Email address that the receipt for the resulting payment will be sent to. If `receipt_email` is specified for a payment in live mode, a receipt will be sent regardless of your [email settings](https://dashboard.stripe.com/account/emails).
            [<Config.Form>]ReceiptEmail: string option
            ///Indicates that you intend to [make future payments](https://stripe.com/docs/payments/payment-intents#future-usage) with the payment
            ///method collected by this Checkout Session.
            ///When setting this to `on_session`, Checkout will show a notice to the
            ///customer that their payment details will be saved.
            ///When setting this to `off_session`, Checkout will show a notice to the
            ///customer that their payment details will be saved and used for future
            ///payments.
            ///For both values, Checkout will attach the payment method to either the
            ///provided Customer for the session, or a new Customer created by Checkout
            ///if one has not been provided.
            ///When processing card payments, Checkout also uses `setup_future_usage`
            ///to dynamically optimize your payment flow and comply with regional
            ///legislation and network rules, such as SCA.
            [<Config.Form>]SetupFutureUsage: Create'PaymentIntentDataSetupFutureUsage option
            ///Shipping information for this payment.
            [<Config.Form>]Shipping: Create'PaymentIntentDataShipping option
            ///Extra information about the payment. This will appear on your
            ///customer's statement when this payment succeeds in creating a charge.
            [<Config.Form>]StatementDescriptor: string option
            ///Provides information about the charge that customers see on their statements. Concatenated with the
            ///prefix (shortened descriptor) or statement descriptor that’s set on the account to form the complete
            ///statement descriptor. Maximum 22 characters for the concatenated descriptor.
            [<Config.Form>]StatementDescriptorSuffix: string option
            ///The parameters used to automatically create a Transfer when the payment succeeds.
            ///For more information, see the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts).
            [<Config.Form>]TransferData: Create'PaymentIntentDataTransferData option
            ///A string that identifies the resulting payment as part of a group. See the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts) for details.
            [<Config.Form>]TransferGroup: string option
        }
        with
            static member New(?applicationFeeAmount: int, ?captureMethod: Create'PaymentIntentDataCaptureMethod, ?description: string, ?metadata: Map<string, string>, ?onBehalfOf: string, ?receiptEmail: string, ?setupFutureUsage: Create'PaymentIntentDataSetupFutureUsage, ?shipping: Create'PaymentIntentDataShipping, ?statementDescriptor: string, ?statementDescriptorSuffix: string, ?transferData: Create'PaymentIntentDataTransferData, ?transferGroup: string) =
                {
                    ApplicationFeeAmount = applicationFeeAmount
                    CaptureMethod = captureMethod
                    Description = description
                    Metadata = metadata
                    OnBehalfOf = onBehalfOf
                    ReceiptEmail = receiptEmail
                    SetupFutureUsage = setupFutureUsage
                    Shipping = shipping
                    StatementDescriptor = statementDescriptor
                    StatementDescriptorSuffix = statementDescriptorSuffix
                    TransferData = transferData
                    TransferGroup = transferGroup
                }

        type Create'PaymentMethodOptionsAcssDebitMandateOptionsPaymentSchedule =
        | Combined
        | Interval
        | Sporadic

        type Create'PaymentMethodOptionsAcssDebitMandateOptionsTransactionType =
        | Business
        | Personal

        type Create'PaymentMethodOptionsAcssDebitMandateOptions = {
            ///A URL for custom mandate text to render during confirmation step.
            ///The URL will be rendered with additional GET parameters `payment_intent` and `payment_intent_client_secret` when confirming a Payment Intent,
            ///or `setup_intent` and `setup_intent_client_secret` when confirming a Setup Intent.
            [<Config.Form>]CustomMandateUrl: Choice<string,string> option
            ///Description of the mandate interval. Only required if 'payment_schedule' parameter is 'interval' or 'combined'.
            [<Config.Form>]IntervalDescription: string option
            ///Payment schedule for the mandate.
            [<Config.Form>]PaymentSchedule: Create'PaymentMethodOptionsAcssDebitMandateOptionsPaymentSchedule option
            ///Transaction type of the mandate.
            [<Config.Form>]TransactionType: Create'PaymentMethodOptionsAcssDebitMandateOptionsTransactionType option
        }
        with
            static member New(?customMandateUrl: Choice<string,string>, ?intervalDescription: string, ?paymentSchedule: Create'PaymentMethodOptionsAcssDebitMandateOptionsPaymentSchedule, ?transactionType: Create'PaymentMethodOptionsAcssDebitMandateOptionsTransactionType) =
                {
                    CustomMandateUrl = customMandateUrl
                    IntervalDescription = intervalDescription
                    PaymentSchedule = paymentSchedule
                    TransactionType = transactionType
                }

        type Create'PaymentMethodOptionsAcssDebitCurrency =
        | Cad
        | Usd

        type Create'PaymentMethodOptionsAcssDebitVerificationMethod =
        | Automatic
        | Instant
        | Microdeposits

        type Create'PaymentMethodOptionsAcssDebit = {
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies). This is only accepted for Checkout Sessions in `setup` mode.
            [<Config.Form>]Currency: Create'PaymentMethodOptionsAcssDebitCurrency option
            ///Additional fields for Mandate creation
            [<Config.Form>]MandateOptions: Create'PaymentMethodOptionsAcssDebitMandateOptions option
            ///Verification method for the intent
            [<Config.Form>]VerificationMethod: Create'PaymentMethodOptionsAcssDebitVerificationMethod option
        }
        with
            static member New(?currency: Create'PaymentMethodOptionsAcssDebitCurrency, ?mandateOptions: Create'PaymentMethodOptionsAcssDebitMandateOptions, ?verificationMethod: Create'PaymentMethodOptionsAcssDebitVerificationMethod) =
                {
                    Currency = currency
                    MandateOptions = mandateOptions
                    VerificationMethod = verificationMethod
                }

        type Create'PaymentMethodOptionsBoleto = {
            ///The number of calendar days before a Boleto voucher expires. For example, if you create a Boleto voucher on Monday and you set expires_after_days to 2, the Boleto invoice will expire on Wednesday at 23:59 America/Sao_Paulo time.
            [<Config.Form>]ExpiresAfterDays: int option
        }
        with
            static member New(?expiresAfterDays: int) =
                {
                    ExpiresAfterDays = expiresAfterDays
                }

        type Create'PaymentMethodOptionsOxxo = {
            ///The number of calendar days before an OXXO voucher expires. For example, if you create an OXXO voucher on Monday and you set expires_after_days to 2, the OXXO invoice will expire on Wednesday at 23:59 America/Mexico_City time.
            [<Config.Form>]ExpiresAfterDays: int option
        }
        with
            static member New(?expiresAfterDays: int) =
                {
                    ExpiresAfterDays = expiresAfterDays
                }

        type Create'PaymentMethodOptionsWechatPayClient =
        | Android
        | Ios
        | Web

        type Create'PaymentMethodOptionsWechatPay = {
            ///The app ID registered with WeChat Pay. Only required when client is ios or android.
            [<Config.Form>]AppId: string option
            ///The client type that the end customer will pay from
            [<Config.Form>]Client: Create'PaymentMethodOptionsWechatPayClient option
        }
        with
            static member New(?appId: string, ?client: Create'PaymentMethodOptionsWechatPayClient) =
                {
                    AppId = appId
                    Client = client
                }

        type Create'PaymentMethodOptions = {
            ///contains details about the ACSS Debit payment method options.
            [<Config.Form>]AcssDebit: Create'PaymentMethodOptionsAcssDebit option
            ///contains details about the Boleto payment method options.
            [<Config.Form>]Boleto: Create'PaymentMethodOptionsBoleto option
            ///contains details about the OXXO payment method options.
            [<Config.Form>]Oxxo: Create'PaymentMethodOptionsOxxo option
            ///contains details about the Wechat Pay payment method options.
            [<Config.Form>]WechatPay: Create'PaymentMethodOptionsWechatPay option
        }
        with
            static member New(?acssDebit: Create'PaymentMethodOptionsAcssDebit, ?boleto: Create'PaymentMethodOptionsBoleto, ?oxxo: Create'PaymentMethodOptionsOxxo, ?wechatPay: Create'PaymentMethodOptionsWechatPay) =
                {
                    AcssDebit = acssDebit
                    Boleto = boleto
                    Oxxo = oxxo
                    WechatPay = wechatPay
                }

        type Create'PaymentMethodTypes =
        | AcssDebit
        | AfterpayClearpay
        | Alipay
        | BacsDebit
        | Bancontact
        | Boleto
        | Card
        | Eps
        | Fpx
        | Giropay
        | Grabpay
        | Ideal
        | Oxxo
        | P24
        | SepaDebit
        | Sofort
        | WechatPay

        type Create'SetupIntentData = {
            ///An arbitrary string attached to the object. Often useful for displaying to users.
            [<Config.Form>]Description: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The Stripe account for which the setup is intended.
            [<Config.Form>]OnBehalfOf: string option
        }
        with
            static member New(?description: string, ?metadata: Map<string, string>, ?onBehalfOf: string) =
                {
                    Description = description
                    Metadata = metadata
                    OnBehalfOf = onBehalfOf
                }

        type Create'ShippingAddressCollectionAllowedCountries =
        | [<JsonUnionCase("AC")>] AC
        | [<JsonUnionCase("AD")>] AD
        | [<JsonUnionCase("AE")>] AE
        | [<JsonUnionCase("AF")>] AF
        | [<JsonUnionCase("AG")>] AG
        | [<JsonUnionCase("AI")>] AI
        | [<JsonUnionCase("AL")>] AL
        | [<JsonUnionCase("AM")>] AM
        | [<JsonUnionCase("AO")>] AO
        | [<JsonUnionCase("AQ")>] AQ
        | [<JsonUnionCase("AR")>] AR
        | [<JsonUnionCase("AT")>] AT
        | [<JsonUnionCase("AU")>] AU
        | [<JsonUnionCase("AW")>] AW
        | [<JsonUnionCase("AX")>] AX
        | [<JsonUnionCase("AZ")>] AZ
        | [<JsonUnionCase("BA")>] BA
        | [<JsonUnionCase("BB")>] BB
        | [<JsonUnionCase("BD")>] BD
        | [<JsonUnionCase("BE")>] BE
        | [<JsonUnionCase("BF")>] BF
        | [<JsonUnionCase("BG")>] BG
        | [<JsonUnionCase("BH")>] BH
        | [<JsonUnionCase("BI")>] BI
        | [<JsonUnionCase("BJ")>] BJ
        | [<JsonUnionCase("BL")>] BL
        | [<JsonUnionCase("BM")>] BM
        | [<JsonUnionCase("BN")>] BN
        | [<JsonUnionCase("BO")>] BO
        | [<JsonUnionCase("BQ")>] BQ
        | [<JsonUnionCase("BR")>] BR
        | [<JsonUnionCase("BS")>] BS
        | [<JsonUnionCase("BT")>] BT
        | [<JsonUnionCase("BV")>] BV
        | [<JsonUnionCase("BW")>] BW
        | [<JsonUnionCase("BY")>] BY
        | [<JsonUnionCase("BZ")>] BZ
        | [<JsonUnionCase("CA")>] CA
        | [<JsonUnionCase("CD")>] CD
        | [<JsonUnionCase("CF")>] CF
        | [<JsonUnionCase("CG")>] CG
        | [<JsonUnionCase("CH")>] CH
        | [<JsonUnionCase("CI")>] CI
        | [<JsonUnionCase("CK")>] CK
        | [<JsonUnionCase("CL")>] CL
        | [<JsonUnionCase("CM")>] CM
        | [<JsonUnionCase("CN")>] CN
        | [<JsonUnionCase("CO")>] CO
        | [<JsonUnionCase("CR")>] CR
        | [<JsonUnionCase("CV")>] CV
        | [<JsonUnionCase("CW")>] CW
        | [<JsonUnionCase("CY")>] CY
        | [<JsonUnionCase("CZ")>] CZ
        | [<JsonUnionCase("DE")>] DE
        | [<JsonUnionCase("DJ")>] DJ
        | [<JsonUnionCase("DK")>] DK
        | [<JsonUnionCase("DM")>] DM
        | [<JsonUnionCase("DO")>] DO
        | [<JsonUnionCase("DZ")>] DZ
        | [<JsonUnionCase("EC")>] EC
        | [<JsonUnionCase("EE")>] EE
        | [<JsonUnionCase("EG")>] EG
        | [<JsonUnionCase("EH")>] EH
        | [<JsonUnionCase("ER")>] ER
        | [<JsonUnionCase("ES")>] ES
        | [<JsonUnionCase("ET")>] ET
        | [<JsonUnionCase("FI")>] FI
        | [<JsonUnionCase("FJ")>] FJ
        | [<JsonUnionCase("FK")>] FK
        | [<JsonUnionCase("FO")>] FO
        | [<JsonUnionCase("FR")>] FR
        | [<JsonUnionCase("GA")>] GA
        | [<JsonUnionCase("GB")>] GB
        | [<JsonUnionCase("GD")>] GD
        | [<JsonUnionCase("GE")>] GE
        | [<JsonUnionCase("GF")>] GF
        | [<JsonUnionCase("GG")>] GG
        | [<JsonUnionCase("GH")>] GH
        | [<JsonUnionCase("GI")>] GI
        | [<JsonUnionCase("GL")>] GL
        | [<JsonUnionCase("GM")>] GM
        | [<JsonUnionCase("GN")>] GN
        | [<JsonUnionCase("GP")>] GP
        | [<JsonUnionCase("GQ")>] GQ
        | [<JsonUnionCase("GR")>] GR
        | [<JsonUnionCase("GS")>] GS
        | [<JsonUnionCase("GT")>] GT
        | [<JsonUnionCase("GU")>] GU
        | [<JsonUnionCase("GW")>] GW
        | [<JsonUnionCase("GY")>] GY
        | [<JsonUnionCase("HK")>] HK
        | [<JsonUnionCase("HN")>] HN
        | [<JsonUnionCase("HR")>] HR
        | [<JsonUnionCase("HT")>] HT
        | [<JsonUnionCase("HU")>] HU
        | [<JsonUnionCase("ID")>] ID
        | [<JsonUnionCase("IE")>] IE
        | [<JsonUnionCase("IL")>] IL
        | [<JsonUnionCase("IM")>] IM
        | [<JsonUnionCase("IN")>] IN
        | [<JsonUnionCase("IO")>] IO
        | [<JsonUnionCase("IQ")>] IQ
        | [<JsonUnionCase("IS")>] IS
        | [<JsonUnionCase("IT")>] IT
        | [<JsonUnionCase("JE")>] JE
        | [<JsonUnionCase("JM")>] JM
        | [<JsonUnionCase("JO")>] JO
        | [<JsonUnionCase("JP")>] JP
        | [<JsonUnionCase("KE")>] KE
        | [<JsonUnionCase("KG")>] KG
        | [<JsonUnionCase("KH")>] KH
        | [<JsonUnionCase("KI")>] KI
        | [<JsonUnionCase("KM")>] KM
        | [<JsonUnionCase("KN")>] KN
        | [<JsonUnionCase("KR")>] KR
        | [<JsonUnionCase("KW")>] KW
        | [<JsonUnionCase("KY")>] KY
        | [<JsonUnionCase("KZ")>] KZ
        | [<JsonUnionCase("LA")>] LA
        | [<JsonUnionCase("LB")>] LB
        | [<JsonUnionCase("LC")>] LC
        | [<JsonUnionCase("LI")>] LI
        | [<JsonUnionCase("LK")>] LK
        | [<JsonUnionCase("LR")>] LR
        | [<JsonUnionCase("LS")>] LS
        | [<JsonUnionCase("LT")>] LT
        | [<JsonUnionCase("LU")>] LU
        | [<JsonUnionCase("LV")>] LV
        | [<JsonUnionCase("LY")>] LY
        | [<JsonUnionCase("MA")>] MA
        | [<JsonUnionCase("MC")>] MC
        | [<JsonUnionCase("MD")>] MD
        | [<JsonUnionCase("ME")>] ME
        | [<JsonUnionCase("MF")>] MF
        | [<JsonUnionCase("MG")>] MG
        | [<JsonUnionCase("MK")>] MK
        | [<JsonUnionCase("ML")>] ML
        | [<JsonUnionCase("MM")>] MM
        | [<JsonUnionCase("MN")>] MN
        | [<JsonUnionCase("MO")>] MO
        | [<JsonUnionCase("MQ")>] MQ
        | [<JsonUnionCase("MR")>] MR
        | [<JsonUnionCase("MS")>] MS
        | [<JsonUnionCase("MT")>] MT
        | [<JsonUnionCase("MU")>] MU
        | [<JsonUnionCase("MV")>] MV
        | [<JsonUnionCase("MW")>] MW
        | [<JsonUnionCase("MX")>] MX
        | [<JsonUnionCase("MY")>] MY
        | [<JsonUnionCase("MZ")>] MZ
        | [<JsonUnionCase("NA")>] NA
        | [<JsonUnionCase("NC")>] NC
        | [<JsonUnionCase("NE")>] NE
        | [<JsonUnionCase("NG")>] NG
        | [<JsonUnionCase("NI")>] NI
        | [<JsonUnionCase("NL")>] NL
        | [<JsonUnionCase("NO")>] NO
        | [<JsonUnionCase("NP")>] NP
        | [<JsonUnionCase("NR")>] NR
        | [<JsonUnionCase("NU")>] NU
        | [<JsonUnionCase("NZ")>] NZ
        | [<JsonUnionCase("OM")>] OM
        | [<JsonUnionCase("PA")>] PA
        | [<JsonUnionCase("PE")>] PE
        | [<JsonUnionCase("PF")>] PF
        | [<JsonUnionCase("PG")>] PG
        | [<JsonUnionCase("PH")>] PH
        | [<JsonUnionCase("PK")>] PK
        | [<JsonUnionCase("PL")>] PL
        | [<JsonUnionCase("PM")>] PM
        | [<JsonUnionCase("PN")>] PN
        | [<JsonUnionCase("PR")>] PR
        | [<JsonUnionCase("PS")>] PS
        | [<JsonUnionCase("PT")>] PT
        | [<JsonUnionCase("PY")>] PY
        | [<JsonUnionCase("QA")>] QA
        | [<JsonUnionCase("RE")>] RE
        | [<JsonUnionCase("RO")>] RO
        | [<JsonUnionCase("RS")>] RS
        | [<JsonUnionCase("RU")>] RU
        | [<JsonUnionCase("RW")>] RW
        | [<JsonUnionCase("SA")>] SA
        | [<JsonUnionCase("SB")>] SB
        | [<JsonUnionCase("SC")>] SC
        | [<JsonUnionCase("SE")>] SE
        | [<JsonUnionCase("SG")>] SG
        | [<JsonUnionCase("SH")>] SH
        | [<JsonUnionCase("SI")>] SI
        | [<JsonUnionCase("SJ")>] SJ
        | [<JsonUnionCase("SK")>] SK
        | [<JsonUnionCase("SL")>] SL
        | [<JsonUnionCase("SM")>] SM
        | [<JsonUnionCase("SN")>] SN
        | [<JsonUnionCase("SO")>] SO
        | [<JsonUnionCase("SR")>] SR
        | [<JsonUnionCase("SS")>] SS
        | [<JsonUnionCase("ST")>] ST
        | [<JsonUnionCase("SV")>] SV
        | [<JsonUnionCase("SX")>] SX
        | [<JsonUnionCase("SZ")>] SZ
        | [<JsonUnionCase("TA")>] TA
        | [<JsonUnionCase("TC")>] TC
        | [<JsonUnionCase("TD")>] TD
        | [<JsonUnionCase("TF")>] TF
        | [<JsonUnionCase("TG")>] TG
        | [<JsonUnionCase("TH")>] TH
        | [<JsonUnionCase("TJ")>] TJ
        | [<JsonUnionCase("TK")>] TK
        | [<JsonUnionCase("TL")>] TL
        | [<JsonUnionCase("TM")>] TM
        | [<JsonUnionCase("TN")>] TN
        | [<JsonUnionCase("TO")>] TO
        | [<JsonUnionCase("TR")>] TR
        | [<JsonUnionCase("TT")>] TT
        | [<JsonUnionCase("TV")>] TV
        | [<JsonUnionCase("TW")>] TW
        | [<JsonUnionCase("TZ")>] TZ
        | [<JsonUnionCase("UA")>] UA
        | [<JsonUnionCase("UG")>] UG
        | [<JsonUnionCase("US")>] US
        | [<JsonUnionCase("UY")>] UY
        | [<JsonUnionCase("UZ")>] UZ
        | [<JsonUnionCase("VA")>] VA
        | [<JsonUnionCase("VC")>] VC
        | [<JsonUnionCase("VE")>] VE
        | [<JsonUnionCase("VG")>] VG
        | [<JsonUnionCase("VN")>] VN
        | [<JsonUnionCase("VU")>] VU
        | [<JsonUnionCase("WF")>] WF
        | [<JsonUnionCase("WS")>] WS
        | [<JsonUnionCase("XK")>] XK
        | [<JsonUnionCase("YE")>] YE
        | [<JsonUnionCase("YT")>] YT
        | [<JsonUnionCase("ZA")>] ZA
        | [<JsonUnionCase("ZM")>] ZM
        | [<JsonUnionCase("ZW")>] ZW
        | [<JsonUnionCase("ZZ")>] ZZ

        type Create'ShippingAddressCollection = {
            ///An array of two-letter ISO country codes representing which countries Checkout should provide as options for
            ///shipping locations. Unsupported country codes: `AS, CX, CC, CU, HM, IR, KP, MH, FM, NF, MP, PW, SD, SY, UM, VI`.
            [<Config.Form>]AllowedCountries: Create'ShippingAddressCollectionAllowedCountries list option
        }
        with
            static member New(?allowedCountries: Create'ShippingAddressCollectionAllowedCountries list) =
                {
                    AllowedCountries = allowedCountries
                }

        type Create'SubscriptionDataItems = {
            ///Plan ID for this item.
            [<Config.Form>]Plan: string option
            ///The quantity of the subscription item being purchased. Quantity should not be defined when `recurring.usage_type=metered`.
            [<Config.Form>]Quantity: int option
            ///The tax rates which apply to this item. When set, the `default_tax_rates`
            ///on `subscription_data` do not apply to this item.
            [<Config.Form>]TaxRates: string list option
        }
        with
            static member New(?plan: string, ?quantity: int, ?taxRates: string list) =
                {
                    Plan = plan
                    Quantity = quantity
                    TaxRates = taxRates
                }

        type Create'SubscriptionDataTransferData = {
            ///A non-negative decimal between 0 and 100, with at most two decimal places. This represents the percentage of the subscription invoice subtotal that will be transferred to the destination account. By default, the entire amount is transferred to the destination.
            [<Config.Form>]AmountPercent: decimal option
            ///ID of an existing, connected Stripe account.
            [<Config.Form>]Destination: string option
        }
        with
            static member New(?amountPercent: decimal, ?destination: string) =
                {
                    AmountPercent = amountPercent
                    Destination = destination
                }

        type Create'SubscriptionData = {
            ///A non-negative decimal between 0 and 100, with at most two decimal places. This represents the percentage of the subscription invoice subtotal that will be transferred to the application owner's Stripe account. To use an application fee percent, the request must be made on behalf of another account, using the `Stripe-Account` header or an OAuth key. For more information, see the application fees [documentation](https://stripe.com/docs/connect/subscriptions#collecting-fees-on-subscriptions).
            [<Config.Form>]ApplicationFeePercent: decimal option
            ///The ID of the coupon to apply to this subscription. A coupon applied to a subscription will only affect invoices created for that particular subscription.
            [<Config.Form>]Coupon: string option
            ///The tax rates that will apply to any subscription item that does not have
            ///`tax_rates` set. Invoices created will have their `default_tax_rates` populated
            ///from the subscription.
            [<Config.Form>]DefaultTaxRates: string list option
            ///A list of items, each with an attached plan, that the customer is subscribing to. Prefer using `line_items`.
            [<Config.Form>]Items: Create'SubscriptionDataItems list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///If specified, the funds from the subscription's invoices will be transferred to the destination and the ID of the resulting transfers will be found on the resulting charges.
            [<Config.Form>]TransferData: Create'SubscriptionDataTransferData option
            ///Unix timestamp representing the end of the trial period the customer
            ///will get before being charged for the first time. Has to be at least
            ///48 hours in the future.
            [<Config.Form>]TrialEnd: DateTime option
            ///Indicates if a plan’s `trial_period_days` should be applied to the subscription. Setting `trial_end` on `subscription_data` is preferred. Defaults to `false`.
            [<Config.Form>]TrialFromPlan: bool option
            ///Integer representing the number of trial period days before the
            ///customer is charged for the first time. Has to be at least 1.
            [<Config.Form>]TrialPeriodDays: int option
        }
        with
            static member New(?applicationFeePercent: decimal, ?coupon: string, ?defaultTaxRates: string list, ?items: Create'SubscriptionDataItems list, ?metadata: Map<string, string>, ?transferData: Create'SubscriptionDataTransferData, ?trialEnd: DateTime, ?trialFromPlan: bool, ?trialPeriodDays: int) =
                {
                    ApplicationFeePercent = applicationFeePercent
                    Coupon = coupon
                    DefaultTaxRates = defaultTaxRates
                    Items = items
                    Metadata = metadata
                    TransferData = transferData
                    TrialEnd = trialEnd
                    TrialFromPlan = trialFromPlan
                    TrialPeriodDays = trialPeriodDays
                }

        type Create'TaxIdCollection = {
            ///Set to true to enable Tax ID collection.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?enabled: bool) =
                {
                    Enabled = enabled
                }

        type Create'BillingAddressCollection =
        | Auto
        | Required

        type Create'Locale =
        | Auto
        | Bg
        | Cs
        | Da
        | De
        | El
        | En
        | [<JsonUnionCase("en-GB")>] EnGB
        | Es
        | [<JsonUnionCase("es-419")>] Es419
        | Et
        | Fi
        | Fr
        | [<JsonUnionCase("fr-CA")>] FrCA
        | Hu
        | Id
        | It
        | Ja
        | Lt
        | Lv
        | Ms
        | Mt
        | Nb
        | Nl
        | Pl
        | Pt
        | [<JsonUnionCase("pt-BR")>] PtBR
        | Ro
        | Ru
        | Sk
        | Sl
        | Sv
        | Th
        | Tr
        | Zh
        | [<JsonUnionCase("zh-HK")>] ZhHK
        | [<JsonUnionCase("zh-TW")>] ZhTW

        type Create'Mode =
        | Payment
        | Setup
        | Subscription

        type Create'SubmitType =
        | Auto
        | Book
        | Donate
        | Pay

        type CreateOptions = {
            ///Enables user redeemable promotion codes.
            [<Config.Form>]AllowPromotionCodes: bool option
            [<Config.Form>]AutomaticTax: Create'AutomaticTax option
            ///Specify whether Checkout should collect the customer's billing address.
            [<Config.Form>]BillingAddressCollection: Create'BillingAddressCollection option
            ///The URL the customer will be directed to if they decide to cancel payment and return to your website.
            [<Config.Form>]CancelUrl: string
            ///A unique string to reference the Checkout Session. This can be a
            ///customer ID, a cart ID, or similar, and can be used to reconcile the
            ///session with your internal systems.
            [<Config.Form>]ClientReferenceId: string option
            ///ID of an existing Customer, if one exists. In `payment` mode, the customer’s most recent card
            ///payment method will be used to prefill the email, name, card details, and billing address
            ///on the Checkout page. In `subscription` mode, the customer’s [default payment method](https://stripe.com/docs/api/customers/update#update_customer-invoice_settings-default_payment_method)
            ///will be used if it’s a card, and otherwise the most recent card will be used. A valid billing address is required for Checkout to prefill the customer's card details.
            ///If the customer changes their email on the Checkout page, the Customer object will be updated with the new email.
            ///If blank for Checkout Sessions in `payment` or `subscription` mode, Checkout will create a new Customer object based on information provided during the payment flow.
            ///You can set [`payment_intent_data.setup_future_usage`](https://stripe.com/docs/api/checkout/sessions/create#create_checkout_session-payment_intent_data-setup_future_usage) to have Checkout automatically attach the payment method to the Customer you pass in for future reuse.
            [<Config.Form>]Customer: string option
            ///If provided, this value will be used when the Customer object is created.
            ///If not provided, customers will be asked to enter their email address.
            ///Use this parameter to prefill customer data if you already have an email
            ///on file. To access information about the customer once a session is
            ///complete, use the `customer` field.
            [<Config.Form>]CustomerEmail: string option
            ///Controls what fields on Customer can be updated by the Checkout Session. Can only be provided when `customer` is provided.
            [<Config.Form>]CustomerUpdate: Create'CustomerUpdate option
            ///The coupon or promotion code to apply to this Session. Currently, only up to one may be specified.
            [<Config.Form>]Discounts: Create'Discounts list option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///A list of items the customer is purchasing. Use this parameter to pass one-time or recurring [Prices](https://stripe.com/docs/api/prices).
            ///For `payment` mode, there is a maximum of 100 line items, however it is recommended to consolidate line items if there are more than a few dozen.
            ///For `subscription` mode, there is a maximum of 20 line items with recurring Prices and 20 line items with one-time Prices. Line items with one-time Prices in will be on the initial invoice only.
            [<Config.Form>]LineItems: Create'LineItems list option
            ///The IETF language tag of the locale Checkout is displayed in. If blank or `auto`, the browser's locale is used.
            [<Config.Form>]Locale: Create'Locale option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The mode of the Checkout Session. Required when using prices or `setup` mode. Pass `subscription` if the Checkout Session includes at least one recurring item.
            [<Config.Form>]Mode: Create'Mode option
            ///A subset of parameters to be passed to PaymentIntent creation for Checkout Sessions in `payment` mode.
            [<Config.Form>]PaymentIntentData: Create'PaymentIntentData option
            ///Payment-method-specific configuration.
            [<Config.Form>]PaymentMethodOptions: Create'PaymentMethodOptions option
            ///A list of the types of payment methods (e.g., `card`) this Checkout Session can accept.
            ///Read more about the supported payment methods and their requirements in our [payment
            ///method details guide](/docs/payments/checkout/payment-methods).
            ///If multiple payment methods are passed, Checkout will dynamically reorder them to
            ///prioritize the most relevant payment methods based on the customer's location and
            ///other characteristics.
            [<Config.Form>]PaymentMethodTypes: Create'PaymentMethodTypes list option
            ///A subset of parameters to be passed to SetupIntent creation for Checkout Sessions in `setup` mode.
            [<Config.Form>]SetupIntentData: Create'SetupIntentData option
            ///When set, provides configuration for Checkout to collect a shipping address from a customer.
            [<Config.Form>]ShippingAddressCollection: Create'ShippingAddressCollection option
            ///The shipping rate to apply to this Session. Currently, only up to one may be specified.
            [<Config.Form>]ShippingRates: string list option
            ///Describes the type of transaction being performed by Checkout in order to customize
            ///relevant text on the page, such as the submit button. `submit_type` can only be
            ///specified on Checkout Sessions in `payment` mode, but not Checkout Sessions
            ///in `subscription` or `setup` mode.
            [<Config.Form>]SubmitType: Create'SubmitType option
            ///A subset of parameters to be passed to subscription creation for Checkout Sessions in `subscription` mode.
            [<Config.Form>]SubscriptionData: Create'SubscriptionData option
            ///The URL to which Stripe should send customers when payment or setup
            ///is complete.
            ///If you’d like access to the Checkout Session for the successful
            ///payment, read more about it in the guide on [fulfilling orders](https://stripe.com/docs/payments/checkout/fulfill-orders).
            [<Config.Form>]SuccessUrl: string
            ///Controls tax ID collection settings for the session.
            [<Config.Form>]TaxIdCollection: Create'TaxIdCollection option
        }
        with
            static member New(cancelUrl: string, successUrl: string, ?allowPromotionCodes: bool, ?subscriptionData: Create'SubscriptionData, ?submitType: Create'SubmitType, ?shippingRates: string list, ?shippingAddressCollection: Create'ShippingAddressCollection, ?setupIntentData: Create'SetupIntentData, ?paymentMethodTypes: Create'PaymentMethodTypes list, ?paymentMethodOptions: Create'PaymentMethodOptions, ?paymentIntentData: Create'PaymentIntentData, ?mode: Create'Mode, ?locale: Create'Locale, ?lineItems: Create'LineItems list, ?expand: string list, ?discounts: Create'Discounts list, ?customerUpdate: Create'CustomerUpdate, ?customerEmail: string, ?customer: string, ?clientReferenceId: string, ?billingAddressCollection: Create'BillingAddressCollection, ?automaticTax: Create'AutomaticTax, ?metadata: Map<string, string>, ?taxIdCollection: Create'TaxIdCollection) =
                {
                    AllowPromotionCodes = allowPromotionCodes
                    AutomaticTax = automaticTax
                    BillingAddressCollection = billingAddressCollection
                    CancelUrl = cancelUrl
                    ClientReferenceId = clientReferenceId
                    Customer = customer
                    CustomerEmail = customerEmail
                    CustomerUpdate = customerUpdate
                    Discounts = discounts
                    Expand = expand
                    LineItems = lineItems
                    Locale = locale
                    Metadata = metadata
                    Mode = mode
                    PaymentIntentData = paymentIntentData
                    PaymentMethodOptions = paymentMethodOptions
                    PaymentMethodTypes = paymentMethodTypes
                    SetupIntentData = setupIntentData
                    ShippingAddressCollection = shippingAddressCollection
                    ShippingRates = shippingRates
                    SubmitType = submitType
                    SubscriptionData = subscriptionData
                    SuccessUrl = successUrl
                    TaxIdCollection = taxIdCollection
                }

        ///<p>Creates a Session object.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/checkout/sessions"
            |> RestApi.postAsync<_, CheckoutSession> settings (Map.empty) options

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Session: string
        }
        with
            static member New(session: string, ?expand: string list) =
                {
                    Expand = expand
                    Session = session
                }

        ///<p>Retrieves a Session object.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/checkout/sessions/{options.Session}"
            |> RestApi.getAsync<CheckoutSession> settings qs

    module CheckoutSessionsLineItems =

        type ListOptions = {
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            [<Config.Path>]Session: string
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(session: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    Session = session
                    StartingAfter = startingAfter
                }

        ///<p>When retrieving a Checkout Session, there is an includable <strong>line_items</strong> property containing the first handful of those items. There is also a URL where you can retrieve the full (paginated) list of line items.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/checkout/sessions/{options.Session}/line_items"
            |> RestApi.getAsync<Item list> settings qs

    module CountrySpecs =

        type ListOptions = {
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Lists all Country Spec objects available in the API.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/country_specs"
            |> RestApi.getAsync<CountrySpec list> settings qs

        type RetrieveOptions = {
            [<Config.Path>]Country: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(country: string, ?expand: string list) =
                {
                    Country = country
                    Expand = expand
                }

        ///<p>Returns a Country Spec for a given Country code.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/country_specs/{options.Country}"
            |> RestApi.getAsync<CountrySpec> settings qs

    module Coupons =

        type ListOptions = {
            ///A filter on the list, based on the object `created` field. The value can be a string with an integer Unix timestamp, or it can be a dictionary with a number of different query options.
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of your coupons.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/coupons"
            |> RestApi.getAsync<Coupon list> settings qs

        type Create'AppliesTo = {
            ///An array of Product IDs that this Coupon will apply to.
            [<Config.Form>]Products: string list option
        }
        with
            static member New(?products: string list) =
                {
                    Products = products
                }

        type Create'Duration =
        | Forever
        | Once
        | Repeating

        type CreateOptions = {
            ///A positive integer representing the amount to subtract from an invoice total (required if `percent_off` is not passed).
            [<Config.Form>]AmountOff: int option
            ///A hash containing directions for what this Coupon will apply discounts to.
            [<Config.Form>]AppliesTo: Create'AppliesTo option
            ///Three-letter [ISO code for the currency](https://stripe.com/docs/currencies) of the `amount_off` parameter (required if `amount_off` is passed).
            [<Config.Form>]Currency: string option
            ///Specifies how long the discount will be in effect if used on a subscription. Can be `forever`, `once`, or `repeating`. Defaults to `once`.
            [<Config.Form>]Duration: Create'Duration option
            ///Required only if `duration` is `repeating`, in which case it must be a positive integer that specifies the number of months the discount will be in effect.
            [<Config.Form>]DurationInMonths: int option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Unique string of your choice that will be used to identify this coupon when applying it to a customer. If you don't want to specify a particular code, you can leave the ID blank and we'll generate a random code for you.
            [<Config.Form>]Id: string option
            ///A positive integer specifying the number of times the coupon can be redeemed before it's no longer valid. For example, you might have a 50% off coupon that the first 20 readers of your blog can use.
            [<Config.Form>]MaxRedemptions: int option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Name of the coupon displayed to customers on, for instance invoices, or receipts. By default the `id` is shown if `name` is not set.
            [<Config.Form>]Name: string option
            ///A positive float larger than 0, and smaller or equal to 100, that represents the discount the coupon will apply (required if `amount_off` is not passed).
            [<Config.Form>]PercentOff: decimal option
            ///Unix timestamp specifying the last time at which the coupon can be redeemed. After the redeem_by date, the coupon can no longer be applied to new customers.
            [<Config.Form>]RedeemBy: DateTime option
        }
        with
            static member New(?amountOff: int, ?appliesTo: Create'AppliesTo, ?currency: string, ?duration: Create'Duration, ?durationInMonths: int, ?expand: string list, ?id: string, ?maxRedemptions: int, ?metadata: Map<string, string>, ?name: string, ?percentOff: decimal, ?redeemBy: DateTime) =
                {
                    AmountOff = amountOff
                    AppliesTo = appliesTo
                    Currency = currency
                    Duration = duration
                    DurationInMonths = durationInMonths
                    Expand = expand
                    Id = id
                    MaxRedemptions = maxRedemptions
                    Metadata = metadata
                    Name = name
                    PercentOff = percentOff
                    RedeemBy = redeemBy
                }

        ///<p>You can create coupons easily via the <a href="https://dashboard.stripe.com/coupons">coupon management</a> page of the Stripe dashboard. Coupon creation is also accessible via the API if you need to create coupons on the fly.
        ///A coupon has either a <code>percent_off</code> or an <code>amount_off</code> and <code>currency</code>. If you set an <code>amount_off</code>, that amount will be subtracted from any invoice’s subtotal. For example, an invoice with a subtotal of <currency>100</currency> will have a final total of <currency>0</currency> if a coupon with an <code>amount_off</code> of <amount>200</amount> is applied to it and an invoice with a subtotal of <currency>300</currency> will have a final total of <currency>100</currency> if a coupon with an <code>amount_off</code> of <amount>200</amount> is applied to it.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/coupons"
            |> RestApi.postAsync<_, Coupon> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Coupon: string
        }
        with
            static member New(coupon: string) =
                {
                    Coupon = coupon
                }

        ///<p>You can delete coupons via the <a href="https://dashboard.stripe.com/coupons">coupon management</a> page of the Stripe dashboard. However, deleting a coupon does not affect any customers who have already applied the coupon; it means that new customers can’t redeem the coupon. You can also delete coupons via the API.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/coupons/{options.Coupon}"
            |> RestApi.deleteAsync<DeletedCoupon> settings (Map.empty)

        type RetrieveOptions = {
            [<Config.Path>]Coupon: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(coupon: string, ?expand: string list) =
                {
                    Coupon = coupon
                    Expand = expand
                }

        ///<p>Retrieves the coupon with the given ID.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/coupons/{options.Coupon}"
            |> RestApi.getAsync<Coupon> settings qs

        type UpdateOptions = {
            [<Config.Path>]Coupon: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Name of the coupon displayed to customers on, for instance invoices, or receipts. By default the `id` is shown if `name` is not set.
            [<Config.Form>]Name: string option
        }
        with
            static member New(coupon: string, ?expand: string list, ?metadata: Map<string, string>, ?name: string) =
                {
                    Coupon = coupon
                    Expand = expand
                    Metadata = metadata
                    Name = name
                }

        ///<p>Updates the metadata of a coupon. Other coupon details (currency, duration, amount_off) are, by design, not editable.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/coupons/{options.Coupon}"
            |> RestApi.postAsync<_, Coupon> settings (Map.empty) options

    module CreditNotes =

        type ListOptions = {
            ///Only return credit notes for the customer specified by this customer ID.
            [<Config.Query>]Customer: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///Only return credit notes for the invoice specified by this invoice ID.
            [<Config.Query>]Invoice: string option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?customer: string, ?endingBefore: string, ?expand: string list, ?invoice: string, ?limit: int, ?startingAfter: string) =
                {
                    Customer = customer
                    EndingBefore = endingBefore
                    Expand = expand
                    Invoice = invoice
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of credit notes.</p>
        let List settings (options: ListOptions) =
            let qs = [("customer", options.Customer |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("invoice", options.Invoice |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/credit_notes"
            |> RestApi.getAsync<CreditNote list> settings qs

        type Create'LinesType =
        | CustomLineItem
        | InvoiceLineItem

        type Create'Lines = {
            ///The line item amount to credit. Only valid when `type` is `invoice_line_item` and the referenced invoice line item does not have a quantity, only an amount.
            [<Config.Form>]Amount: int option
            ///The description of the credit note line item. Only valid when the `type` is `custom_line_item`.
            [<Config.Form>]Description: string option
            ///The invoice line item to credit. Only valid when the `type` is `invoice_line_item`.
            [<Config.Form>]InvoiceLineItem: string option
            ///The line item quantity to credit.
            [<Config.Form>]Quantity: int option
            ///The tax rates which apply to the credit note line item. Only valid when the `type` is `custom_line_item`.
            [<Config.Form>]TaxRates: Choice<string list,string> option
            ///Type of the credit note line item, one of `invoice_line_item` or `custom_line_item`
            [<Config.Form>]Type: Create'LinesType option
            ///The integer unit amount in %s of the credit note line item. This `unit_amount` will be multiplied by the quantity to get the full amount to credit for this line item. Only valid when `type` is `custom_line_item`.
            [<Config.Form>]UnitAmount: int option
            ///Same as `unit_amount`, but accepts a decimal value in %s with at most 12 decimal places. Only one of `unit_amount` and `unit_amount_decimal` can be set.
            [<Config.Form>]UnitAmountDecimal: string option
        }
        with
            static member New(?amount: int, ?description: string, ?invoiceLineItem: string, ?quantity: int, ?taxRates: Choice<string list,string>, ?type': Create'LinesType, ?unitAmount: int, ?unitAmountDecimal: string) =
                {
                    Amount = amount
                    Description = description
                    InvoiceLineItem = invoiceLineItem
                    Quantity = quantity
                    TaxRates = taxRates
                    Type = type'
                    UnitAmount = unitAmount
                    UnitAmountDecimal = unitAmountDecimal
                }

        type Create'Reason =
        | Duplicate
        | Fraudulent
        | OrderChange
        | ProductUnsatisfactory

        type CreateOptions = {
            ///The integer amount in %s representing the total amount of the credit note.
            [<Config.Form>]Amount: int option
            ///The integer amount in %s representing the amount to credit the customer's balance, which will be automatically applied to their next invoice.
            [<Config.Form>]CreditAmount: int option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///ID of the invoice.
            [<Config.Form>]Invoice: string
            ///Line items that make up the credit note.
            [<Config.Form>]Lines: Create'Lines list option
            ///The credit note's memo appears on the credit note PDF.
            [<Config.Form>]Memo: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The integer amount in %s representing the amount that is credited outside of Stripe.
            [<Config.Form>]OutOfBandAmount: int option
            ///Reason for issuing this credit note, one of `duplicate`, `fraudulent`, `order_change`, or `product_unsatisfactory`
            [<Config.Form>]Reason: Create'Reason option
            ///ID of an existing refund to link this credit note to.
            [<Config.Form>]Refund: string option
            ///The integer amount in %s representing the amount to refund. If set, a refund will be created for the charge associated with the invoice.
            [<Config.Form>]RefundAmount: int option
        }
        with
            static member New(invoice: string, ?amount: int, ?creditAmount: int, ?expand: string list, ?lines: Create'Lines list, ?memo: string, ?metadata: Map<string, string>, ?outOfBandAmount: int, ?reason: Create'Reason, ?refund: string, ?refundAmount: int) =
                {
                    Amount = amount
                    CreditAmount = creditAmount
                    Expand = expand
                    Invoice = invoice
                    Lines = lines
                    Memo = memo
                    Metadata = metadata
                    OutOfBandAmount = outOfBandAmount
                    Reason = reason
                    Refund = refund
                    RefundAmount = refundAmount
                }

        ///<p>Issue a credit note to adjust the amount of a finalized invoice. For a <code>status=open</code> invoice, a credit note reduces
        ///its <code>amount_due</code>. For a <code>status=paid</code> invoice, a credit note does not affect its <code>amount_due</code>. Instead, it can result
        ///in any combination of the following:
        ///<ul>
        ///<li>Refund: create a new refund (using <code>refund_amount</code>) or link an existing refund (using <code>refund</code>).</li>
        ///<li>Customer balance credit: credit the customer’s balance (using <code>credit_amount</code>) which will be automatically applied to their next invoice when it’s finalized.</li>
        ///<li>Outside of Stripe credit: record the amount that is or will be credited outside of Stripe (using <code>out_of_band_amount</code>).</li>
        ///</ul>
        ///For post-payment credit notes the sum of the refund, credit and outside of Stripe amounts must equal the credit note total.
        ///You may issue multiple credit notes for an invoice. Each credit note will increment the invoice’s <code>pre_payment_credit_notes_amount</code>
        ///or <code>post_payment_credit_notes_amount</code> depending on its <code>status</code> at the time of credit note creation.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/credit_notes"
            |> RestApi.postAsync<_, CreditNote> settings (Map.empty) options

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
        }
        with
            static member New(id: string, ?expand: string list) =
                {
                    Expand = expand
                    Id = id
                }

        ///<p>Retrieves the credit note object with the given identifier.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/credit_notes/{options.Id}"
            |> RestApi.getAsync<CreditNote> settings qs

        type UpdateOptions = {
            [<Config.Path>]Id: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Credit note memo.
            [<Config.Form>]Memo: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(id: string, ?expand: string list, ?memo: string, ?metadata: Map<string, string>) =
                {
                    Id = id
                    Expand = expand
                    Memo = memo
                    Metadata = metadata
                }

        ///<p>Updates an existing credit note.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/credit_notes/{options.Id}"
            |> RestApi.postAsync<_, CreditNote> settings (Map.empty) options

    module CreditNotesPreview =

        type PreviewOptions = {
            ///The integer amount in %s representing the total amount of the credit note.
            [<Config.Query>]Amount: int option
            ///The integer amount in %s representing the amount to credit the customer's balance, which will be automatically applied to their next invoice.
            [<Config.Query>]CreditAmount: int option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///ID of the invoice.
            [<Config.Query>]Invoice: string
            ///Line items that make up the credit note.
            [<Config.Query>]Lines: string list option
            ///The credit note's memo appears on the credit note PDF.
            [<Config.Query>]Memo: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Query>]Metadata: Map<string, string> option
            ///The integer amount in %s representing the amount that is credited outside of Stripe.
            [<Config.Query>]OutOfBandAmount: int option
            ///Reason for issuing this credit note, one of `duplicate`, `fraudulent`, `order_change`, or `product_unsatisfactory`
            [<Config.Query>]Reason: string option
            ///ID of an existing refund to link this credit note to.
            [<Config.Query>]Refund: string option
            ///The integer amount in %s representing the amount to refund. If set, a refund will be created for the charge associated with the invoice.
            [<Config.Query>]RefundAmount: int option
        }
        with
            static member New(invoice: string, ?amount: int, ?creditAmount: int, ?expand: string list, ?lines: string list, ?memo: string, ?metadata: Map<string, string>, ?outOfBandAmount: int, ?reason: string, ?refund: string, ?refundAmount: int) =
                {
                    Amount = amount
                    CreditAmount = creditAmount
                    Expand = expand
                    Invoice = invoice
                    Lines = lines
                    Memo = memo
                    Metadata = metadata
                    OutOfBandAmount = outOfBandAmount
                    Reason = reason
                    Refund = refund
                    RefundAmount = refundAmount
                }

        ///<p>Get a preview of a credit note without creating it.</p>
        let Preview settings (options: PreviewOptions) =
            let qs = [("amount", options.Amount |> box); ("credit_amount", options.CreditAmount |> box); ("expand", options.Expand |> box); ("invoice", options.Invoice |> box); ("lines", options.Lines |> box); ("memo", options.Memo |> box); ("metadata", options.Metadata |> box); ("out_of_band_amount", options.OutOfBandAmount |> box); ("reason", options.Reason |> box); ("refund", options.Refund |> box); ("refund_amount", options.RefundAmount |> box)] |> Map.ofList
            $"/v1/credit_notes/preview"
            |> RestApi.getAsync<CreditNote> settings qs

    module CreditNotesPreviewLines =

        type PreviewLinesOptions = {
            ///The integer amount in %s representing the total amount of the credit note.
            [<Config.Query>]Amount: int option
            ///The integer amount in %s representing the amount to credit the customer's balance, which will be automatically applied to their next invoice.
            [<Config.Query>]CreditAmount: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///ID of the invoice.
            [<Config.Query>]Invoice: string
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///Line items that make up the credit note.
            [<Config.Query>]Lines: string list option
            ///The credit note's memo appears on the credit note PDF.
            [<Config.Query>]Memo: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Query>]Metadata: Map<string, string> option
            ///The integer amount in %s representing the amount that is credited outside of Stripe.
            [<Config.Query>]OutOfBandAmount: int option
            ///Reason for issuing this credit note, one of `duplicate`, `fraudulent`, `order_change`, or `product_unsatisfactory`
            [<Config.Query>]Reason: string option
            ///ID of an existing refund to link this credit note to.
            [<Config.Query>]Refund: string option
            ///The integer amount in %s representing the amount to refund. If set, a refund will be created for the charge associated with the invoice.
            [<Config.Query>]RefundAmount: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(invoice: string, ?amount: int, ?creditAmount: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?lines: string list, ?memo: string, ?metadata: Map<string, string>, ?outOfBandAmount: int, ?reason: string, ?refund: string, ?refundAmount: int, ?startingAfter: string) =
                {
                    Amount = amount
                    CreditAmount = creditAmount
                    EndingBefore = endingBefore
                    Expand = expand
                    Invoice = invoice
                    Limit = limit
                    Lines = lines
                    Memo = memo
                    Metadata = metadata
                    OutOfBandAmount = outOfBandAmount
                    Reason = reason
                    Refund = refund
                    RefundAmount = refundAmount
                    StartingAfter = startingAfter
                }

        ///<p>When retrieving a credit note preview, you’ll get a <strong>lines</strong> property containing the first handful of those items. This URL you can retrieve the full (paginated) list of line items.</p>
        let PreviewLines settings (options: PreviewLinesOptions) =
            let qs = [("amount", options.Amount |> box); ("credit_amount", options.CreditAmount |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("invoice", options.Invoice |> box); ("limit", options.Limit |> box); ("lines", options.Lines |> box); ("memo", options.Memo |> box); ("metadata", options.Metadata |> box); ("out_of_band_amount", options.OutOfBandAmount |> box); ("reason", options.Reason |> box); ("refund", options.Refund |> box); ("refund_amount", options.RefundAmount |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/credit_notes/preview/lines"
            |> RestApi.getAsync<CreditNoteLineItem list> settings qs

    module CreditNotesLines =

        type ListOptions = {
            [<Config.Path>]CreditNote: string
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(creditNote: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    CreditNote = creditNote
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>When retrieving a credit note, you’ll get a <strong>lines</strong> property containing the the first handful of those items. There is also a URL where you can retrieve the full (paginated) list of line items.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/credit_notes/{options.CreditNote}/lines"
            |> RestApi.getAsync<CreditNoteLineItem list> settings qs

    module CreditNotesVoid =

        type VoidCreditNoteOptions = {
            [<Config.Path>]Id: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(id: string, ?expand: string list) =
                {
                    Id = id
                    Expand = expand
                }

        ///<p>Marks a credit note as void. Learn more about <a href="/docs/billing/invoices/credit-notes#voiding">voiding credit notes</a>.</p>
        let VoidCreditNote settings (options: VoidCreditNoteOptions) =
            $"/v1/credit_notes/{options.Id}/void"
            |> RestApi.postAsync<_, CreditNote> settings (Map.empty) options

    module Customers =

        type ListOptions = {
            [<Config.Query>]Created: int option
            ///A case-sensitive filter on the list based on the customer's `email` field. The value must be a string.
            [<Config.Query>]Email: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?created: int, ?email: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Created = created
                    Email = email
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of your customers. The customers are returned sorted by creation date, with the most recent customers appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("email", options.Email |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/customers"
            |> RestApi.getAsync<Customer list> settings qs

        type Create'AddressOptionalFieldsAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'InvoiceSettingsCustomFields = {
            ///The name of the custom field. This may be up to 30 characters.
            [<Config.Form>]Name: string option
            ///The value of the custom field. This may be up to 30 characters.
            [<Config.Form>]Value: string option
        }
        with
            static member New(?name: string, ?value: string) =
                {
                    Name = name
                    Value = value
                }

        type Create'InvoiceSettings = {
            ///Default custom fields to be displayed on invoices for this customer. When updating, pass an empty string to remove previously-defined fields.
            [<Config.Form>]CustomFields: Choice<Create'InvoiceSettingsCustomFields list,string> option
            ///ID of a payment method that's attached to the customer, to be used as the customer's default payment method for subscriptions and invoices.
            [<Config.Form>]DefaultPaymentMethod: string option
            ///Default footer to be displayed on invoices for this customer.
            [<Config.Form>]Footer: string option
        }
        with
            static member New(?customFields: Choice<Create'InvoiceSettingsCustomFields list,string>, ?defaultPaymentMethod: string, ?footer: string) =
                {
                    CustomFields = customFields
                    DefaultPaymentMethod = defaultPaymentMethod
                    Footer = footer
                }

        type Create'ShippingCustomerShippingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'ShippingCustomerShipping = {
            ///Customer shipping address.
            [<Config.Form>]Address: Create'ShippingCustomerShippingAddress option
            ///Customer name.
            [<Config.Form>]Name: string option
            ///Customer phone (including extension).
            [<Config.Form>]Phone: string option
        }
        with
            static member New(?address: Create'ShippingCustomerShippingAddress, ?name: string, ?phone: string) =
                {
                    Address = address
                    Name = name
                    Phone = phone
                }

        type Create'Tax = {
            ///A recent IP address of the customer used for tax reporting and tax location inference. Stripe recommends updating the IP address when a new PaymentMethod is attached or the address field on the customer is updated. We recommend against updating this field more frequently since it could result in unexpected tax location/reporting outcomes.
            [<Config.Form>]IpAddress: Choice<string,string> option
        }
        with
            static member New(?ipAddress: Choice<string,string>) =
                {
                    IpAddress = ipAddress
                }

        type Create'TaxIdDataType =
        | AeTrn
        | AuAbn
        | BrCnpj
        | BrCpf
        | CaBn
        | CaGstHst
        | CaPstBc
        | CaPstMb
        | CaPstSk
        | CaQst
        | ChVat
        | ClTin
        | EsCif
        | EuVat
        | GbVat
        | HkBr
        | IdNpwp
        | IlVat
        | InGst
        | JpCn
        | JpRn
        | KrBrn
        | LiUid
        | MxRfc
        | MyFrp
        | MyItn
        | MySst
        | NoVat
        | NzGst
        | RuInn
        | RuKpp
        | SaVat
        | SgGst
        | SgUen
        | ThVat
        | TwVat
        | UsEin
        | ZaVat

        type Create'TaxIdData = {
            ///Type of the tax ID, one of `ae_trn`, `au_abn`, `br_cnpj`, `br_cpf`, `ca_bn`, `ca_gst_hst`, `ca_pst_bc`, `ca_pst_mb`, `ca_pst_sk`, `ca_qst`, `ch_vat`, `cl_tin`, `es_cif`, `eu_vat`, `gb_vat`, `hk_br`, `id_npwp`, `il_vat`, `in_gst`, `jp_cn`, `jp_rn`, `kr_brn`, `li_uid`, `mx_rfc`, `my_frp`, `my_itn`, `my_sst`, `no_vat`, `nz_gst`, `ru_inn`, `ru_kpp`, `sa_vat`, `sg_gst`, `sg_uen`, `th_vat`, `tw_vat`, `us_ein`, or `za_vat`
            [<Config.Form>]Type: Create'TaxIdDataType option
            ///Value of the tax ID.
            [<Config.Form>]Value: string option
        }
        with
            static member New(?type': Create'TaxIdDataType, ?value: string) =
                {
                    Type = type'
                    Value = value
                }

        type Create'TaxExempt =
        | Exempt
        | None'
        | Reverse

        type CreateOptions = {
            ///The customer's address.
            [<Config.Form>]Address: Choice<Create'AddressOptionalFieldsAddress,string> option
            ///An integer amount in %s that represents the customer's current balance, which affect the customer's future invoices. A negative amount represents a credit that decreases the amount due on an invoice; a positive amount increases the amount due on an invoice.
            [<Config.Form>]Balance: int option
            [<Config.Form>]Coupon: string option
            ///An arbitrary string that you can attach to a customer object. It is displayed alongside the customer in the dashboard.
            [<Config.Form>]Description: string option
            ///Customer's email address. It's displayed alongside the customer in your dashboard and can be useful for searching and tracking. This may be up to *512 characters*.
            [<Config.Form>]Email: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The prefix for the customer used to generate unique invoice numbers. Must be 3–12 uppercase letters or numbers.
            [<Config.Form>]InvoicePrefix: string option
            ///Default invoice settings for this customer.
            [<Config.Form>]InvoiceSettings: Create'InvoiceSettings option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The customer's full name or business name.
            [<Config.Form>]Name: string option
            ///The sequence to be used on the customer's next invoice. Defaults to 1.
            [<Config.Form>]NextInvoiceSequence: int option
            [<Config.Form>]PaymentMethod: string option
            ///The customer's phone number.
            [<Config.Form>]Phone: string option
            ///Customer's preferred languages, ordered by preference.
            [<Config.Form>]PreferredLocales: string list option
            ///The API ID of a promotion code to apply to the customer. The customer will have a discount applied on all recurring payments. Charges you create through the API will not have the discount.
            [<Config.Form>]PromotionCode: string option
            ///The customer's shipping information. Appears on invoices emailed to this customer.
            [<Config.Form>]Shipping: Choice<Create'ShippingCustomerShipping,string> option
            [<Config.Form>]Source: string option
            ///Tax details about the customer.
            [<Config.Form>]Tax: Create'Tax option
            ///The customer's tax exemption. One of `none`, `exempt`, or `reverse`.
            [<Config.Form>]TaxExempt: Create'TaxExempt option
            ///The customer's tax IDs.
            [<Config.Form>]TaxIdData: Create'TaxIdData list option
        }
        with
            static member New(?address: Choice<Create'AddressOptionalFieldsAddress,string>, ?tax: Create'Tax, ?source: string, ?shipping: Choice<Create'ShippingCustomerShipping,string>, ?promotionCode: string, ?preferredLocales: string list, ?phone: string, ?paymentMethod: string, ?nextInvoiceSequence: int, ?name: string, ?metadata: Map<string, string>, ?invoiceSettings: Create'InvoiceSettings, ?invoicePrefix: string, ?expand: string list, ?email: string, ?description: string, ?coupon: string, ?balance: int, ?taxExempt: Create'TaxExempt, ?taxIdData: Create'TaxIdData list) =
                {
                    Address = address
                    Balance = balance
                    Coupon = coupon
                    Description = description
                    Email = email
                    Expand = expand
                    InvoicePrefix = invoicePrefix
                    InvoiceSettings = invoiceSettings
                    Metadata = metadata
                    Name = name
                    NextInvoiceSequence = nextInvoiceSequence
                    PaymentMethod = paymentMethod
                    Phone = phone
                    PreferredLocales = preferredLocales
                    PromotionCode = promotionCode
                    Shipping = shipping
                    Source = source
                    Tax = tax
                    TaxExempt = taxExempt
                    TaxIdData = taxIdData
                }

        ///<p>Creates a new customer object.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/customers"
            |> RestApi.postAsync<_, Customer> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Customer: string
        }
        with
            static member New(customer: string) =
                {
                    Customer = customer
                }

        ///<p>Permanently deletes a customer. It cannot be undone. Also immediately cancels any active subscriptions on the customer.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/customers/{options.Customer}"
            |> RestApi.deleteAsync<DeletedCustomer> settings (Map.empty)

        type RetrieveOptions = {
            [<Config.Path>]Customer: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(customer: string, ?expand: string list) =
                {
                    Customer = customer
                    Expand = expand
                }

        ///<p>Retrieves the details of an existing customer. You need only supply the unique customer identifier that was returned upon customer creation.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/customers/{options.Customer}"
            |> RestApi.getAsync<Customer> settings qs

        type Update'AddressOptionalFieldsAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'InvoiceSettingsCustomFields = {
            ///The name of the custom field. This may be up to 30 characters.
            [<Config.Form>]Name: string option
            ///The value of the custom field. This may be up to 30 characters.
            [<Config.Form>]Value: string option
        }
        with
            static member New(?name: string, ?value: string) =
                {
                    Name = name
                    Value = value
                }

        type Update'InvoiceSettings = {
            ///Default custom fields to be displayed on invoices for this customer. When updating, pass an empty string to remove previously-defined fields.
            [<Config.Form>]CustomFields: Choice<Update'InvoiceSettingsCustomFields list,string> option
            ///ID of a payment method that's attached to the customer, to be used as the customer's default payment method for subscriptions and invoices.
            [<Config.Form>]DefaultPaymentMethod: string option
            ///Default footer to be displayed on invoices for this customer.
            [<Config.Form>]Footer: string option
        }
        with
            static member New(?customFields: Choice<Update'InvoiceSettingsCustomFields list,string>, ?defaultPaymentMethod: string, ?footer: string) =
                {
                    CustomFields = customFields
                    DefaultPaymentMethod = defaultPaymentMethod
                    Footer = footer
                }

        type Update'ShippingCustomerShippingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'ShippingCustomerShipping = {
            ///Customer shipping address.
            [<Config.Form>]Address: Update'ShippingCustomerShippingAddress option
            ///Customer name.
            [<Config.Form>]Name: string option
            ///Customer phone (including extension).
            [<Config.Form>]Phone: string option
        }
        with
            static member New(?address: Update'ShippingCustomerShippingAddress, ?name: string, ?phone: string) =
                {
                    Address = address
                    Name = name
                    Phone = phone
                }

        type Update'Tax = {
            ///A recent IP address of the customer used for tax reporting and tax location inference. Stripe recommends updating the IP address when a new PaymentMethod is attached or the address field on the customer is updated. We recommend against updating this field more frequently since it could result in unexpected tax location/reporting outcomes.
            [<Config.Form>]IpAddress: Choice<string,string> option
        }
        with
            static member New(?ipAddress: Choice<string,string>) =
                {
                    IpAddress = ipAddress
                }

        type Update'TrialEnd =
        | Now

        type Update'TaxExempt =
        | Exempt
        | None'
        | Reverse

        type UpdateOptions = {
            [<Config.Path>]Customer: string
            ///The customer's address.
            [<Config.Form>]Address: Choice<Update'AddressOptionalFieldsAddress,string> option
            ///An integer amount in %s that represents the customer's current balance, which affect the customer's future invoices. A negative amount represents a credit that decreases the amount due on an invoice; a positive amount increases the amount due on an invoice.
            [<Config.Form>]Balance: int option
            [<Config.Form>]Coupon: string option
            ///If you are using payment methods created via the PaymentMethods API, see the [invoice_settings.default_payment_method](https://stripe.com/docs/api/customers/update#update_customer-invoice_settings-default_payment_method) parameter.
            ///Provide the ID of a payment source already attached to this customer to make it this customer's default payment source.
            ///If you want to add a new payment source and make it the default, see the [source](https://stripe.com/docs/api/customers/update#update_customer-source) property.
            [<Config.Form>]DefaultSource: string option
            ///An arbitrary string that you can attach to a customer object. It is displayed alongside the customer in the dashboard.
            [<Config.Form>]Description: string option
            ///Customer's email address. It's displayed alongside the customer in your dashboard and can be useful for searching and tracking. This may be up to *512 characters*.
            [<Config.Form>]Email: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The prefix for the customer used to generate unique invoice numbers. Must be 3–12 uppercase letters or numbers.
            [<Config.Form>]InvoicePrefix: string option
            ///Default invoice settings for this customer.
            [<Config.Form>]InvoiceSettings: Update'InvoiceSettings option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The customer's full name or business name.
            [<Config.Form>]Name: string option
            ///The sequence to be used on the customer's next invoice. Defaults to 1.
            [<Config.Form>]NextInvoiceSequence: int option
            ///The customer's phone number.
            [<Config.Form>]Phone: string option
            ///Customer's preferred languages, ordered by preference.
            [<Config.Form>]PreferredLocales: string list option
            ///The API ID of a promotion code to apply to the customer. The customer will have a discount applied on all recurring payments. Charges you create through the API will not have the discount.
            [<Config.Form>]PromotionCode: string option
            ///The customer's shipping information. Appears on invoices emailed to this customer.
            [<Config.Form>]Shipping: Choice<Update'ShippingCustomerShipping,string> option
            [<Config.Form>]Source: string option
            ///Tax details about the customer.
            [<Config.Form>]Tax: Update'Tax option
            ///The customer's tax exemption. One of `none`, `exempt`, or `reverse`.
            [<Config.Form>]TaxExempt: Update'TaxExempt option
            ///Unix timestamp representing the end of the trial period the customer will get before being charged for the first time. This will always overwrite any trials that might apply via a subscribed plan. If set, trial_end will override the default trial period of the plan the customer is being subscribed to. The special value `now` can be provided to end the customer's trial immediately. Can be at most two years from `billing_cycle_anchor`.
            [<Config.Form>]TrialEnd: Choice<Update'TrialEnd,DateTime> option
        }
        with
            static member New(customer: string, ?tax: Update'Tax, ?source: string, ?shipping: Choice<Update'ShippingCustomerShipping,string>, ?promotionCode: string, ?preferredLocales: string list, ?phone: string, ?nextInvoiceSequence: int, ?name: string, ?taxExempt: Update'TaxExempt, ?metadata: Map<string, string>, ?invoicePrefix: string, ?expand: string list, ?email: string, ?description: string, ?defaultSource: string, ?coupon: string, ?balance: int, ?address: Choice<Update'AddressOptionalFieldsAddress,string>, ?invoiceSettings: Update'InvoiceSettings, ?trialEnd: Choice<Update'TrialEnd,DateTime>) =
                {
                    Customer = customer
                    Address = address
                    Balance = balance
                    Coupon = coupon
                    DefaultSource = defaultSource
                    Description = description
                    Email = email
                    Expand = expand
                    InvoicePrefix = invoicePrefix
                    InvoiceSettings = invoiceSettings
                    Metadata = metadata
                    Name = name
                    NextInvoiceSequence = nextInvoiceSequence
                    Phone = phone
                    PreferredLocales = preferredLocales
                    PromotionCode = promotionCode
                    Shipping = shipping
                    Source = source
                    Tax = tax
                    TaxExempt = taxExempt
                    TrialEnd = trialEnd
                }

        ///<p>Updates the specified customer by setting the values of the parameters passed. Any parameters not provided will be left unchanged. For example, if you pass the <strong>source</strong> parameter, that becomes the customer’s active source (e.g., a card) to be used for all charges in the future. When you update a customer to a new valid card source by passing the <strong>source</strong> parameter: for each of the customer’s current subscriptions, if the subscription bills automatically and is in the <code>past_due</code> state, then the latest open invoice for the subscription with automatic collection enabled will be retried. This retry will not count as an automatic retry, and will not affect the next regularly scheduled payment for the invoice. Changing the <strong>default_source</strong> for a customer will not trigger this behavior.
        ///This request accepts mostly the same arguments as the customer creation call.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/customers/{options.Customer}"
            |> RestApi.postAsync<_, Customer> settings (Map.empty) options

    module CustomersBalanceTransactions =

        type ListOptions = {
            [<Config.Path>]Customer: string
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(customer: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Customer = customer
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of transactions that updated the customer’s <a href="/docs/billing/customer/balance">balances</a>.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/customers/{options.Customer}/balance_transactions"
            |> RestApi.getAsync<CustomerBalanceTransaction list> settings qs

        type CreateOptions = {
            [<Config.Path>]Customer: string
            ///The integer amount in **%s** to apply to the customer's credit balance.
            [<Config.Form>]Amount: int
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies). If the customer's [`currency`](https://stripe.com/docs/api/customers/object#customer_object-currency) is set, this value must match it. If the customer's `currency` is not set, it will be updated to this value.
            [<Config.Form>]Currency: string
            ///An arbitrary string attached to the object. Often useful for displaying to users.
            [<Config.Form>]Description: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(customer: string, amount: int, currency: string, ?description: string, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Customer = customer
                    Amount = amount
                    Currency = currency
                    Description = description
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Creates an immutable transaction that updates the customer’s credit <a href="/docs/billing/customer/balance">balance</a>.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/customers/{options.Customer}/balance_transactions"
            |> RestApi.postAsync<_, CustomerBalanceTransaction> settings (Map.empty) options

        type RetrieveOptions = {
            [<Config.Path>]Customer: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Transaction: string
        }
        with
            static member New(customer: string, transaction: string, ?expand: string list) =
                {
                    Customer = customer
                    Expand = expand
                    Transaction = transaction
                }

        ///<p>Retrieves a specific customer balance transaction that updated the customer’s <a href="/docs/billing/customer/balance">balances</a>.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/customers/{options.Customer}/balance_transactions/{options.Transaction}"
            |> RestApi.getAsync<CustomerBalanceTransaction> settings qs

        type UpdateOptions = {
            [<Config.Path>]Customer: string
            [<Config.Path>]Transaction: string
            ///An arbitrary string attached to the object. Often useful for displaying to users.
            [<Config.Form>]Description: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(customer: string, transaction: string, ?description: string, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Customer = customer
                    Transaction = transaction
                    Description = description
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Most credit balance transaction fields are immutable, but you may update its <code>description</code> and <code>metadata</code>.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/customers/{options.Customer}/balance_transactions/{options.Transaction}"
            |> RestApi.postAsync<_, CustomerBalanceTransaction> settings (Map.empty) options

    module CustomersDiscount =

        type DeleteDiscountOptions = {
            [<Config.Path>]Customer: string
        }
        with
            static member New(customer: string) =
                {
                    Customer = customer
                }

        ///<p>Removes the currently applied discount on a customer.</p>
        let DeleteDiscount settings (options: DeleteDiscountOptions) =
            $"/v1/customers/{options.Customer}/discount"
            |> RestApi.deleteAsync<DeletedDiscount> settings (Map.empty)

    module CustomersSources =

        type ListOptions = {
            [<Config.Path>]Customer: string
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///Filter sources according to a particular object type.
            [<Config.Query>]Object: string option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(customer: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?object: string, ?startingAfter: string) =
                {
                    Customer = customer
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    Object = object
                    StartingAfter = startingAfter
                }

        ///<p>List sources for a specified customer.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("object", options.Object |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/customers/{options.Customer}/sources"
            |> RestApi.getAsync<PaymentSource list> settings qs

        type CreateOptions = {
            [<Config.Path>]Customer: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Please refer to full [documentation](https://stripe.com/docs/api) instead.
            [<Config.Form>]Source: string
        }
        with
            static member New(customer: string, source: string, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Customer = customer
                    Expand = expand
                    Metadata = metadata
                    Source = source
                }

        ///<p>When you create a new credit card, you must specify a customer or recipient on which to create it.
        ///If the card’s owner has no default card, then the new card will become the default.
        ///However, if the owner already has a default, then it will not change.
        ///To change the default, you should <a href="/docs/api#update_customer">update the customer</a> to have a new <code>default_source</code>.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/customers/{options.Customer}/sources"
            |> RestApi.postAsync<_, PaymentSource> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Customer: string
            [<Config.Path>]Id: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(customer: string, id: string, ?expand: string list) =
                {
                    Customer = customer
                    Id = id
                    Expand = expand
                }

        ///<p>Delete a specified source for a given customer.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/customers/{options.Customer}/sources/{options.Id}"
            |> RestApi.deleteAsync<PaymentSource> settings (Map.empty)

        type RetrieveOptions = {
            [<Config.Path>]Customer: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
        }
        with
            static member New(customer: string, id: string, ?expand: string list) =
                {
                    Customer = customer
                    Expand = expand
                    Id = id
                }

        ///<p>Retrieve a specified source for a given customer.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/customers/{options.Customer}/sources/{options.Id}"
            |> RestApi.getAsync<PaymentSource> settings qs

        type Update'OwnerAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'Owner = {
            ///Owner's address.
            [<Config.Form>]Address: Update'OwnerAddress option
            ///Owner's email address.
            [<Config.Form>]Email: string option
            ///Owner's full name.
            [<Config.Form>]Name: string option
            ///Owner's phone number.
            [<Config.Form>]Phone: string option
        }
        with
            static member New(?address: Update'OwnerAddress, ?email: string, ?name: string, ?phone: string) =
                {
                    Address = address
                    Email = email
                    Name = name
                    Phone = phone
                }

        type Update'AccountHolderType =
        | Company
        | Individual

        type UpdateOptions = {
            [<Config.Path>]Customer: string
            [<Config.Path>]Id: string
            ///The name of the person or business that owns the bank account.
            [<Config.Form>]AccountHolderName: string option
            ///The type of entity that holds the account. This can be either `individual` or `company`.
            [<Config.Form>]AccountHolderType: Update'AccountHolderType option
            ///City/District/Suburb/Town/Village.
            [<Config.Form>]AddressCity: string option
            ///Billing address country, if provided when creating card.
            [<Config.Form>]AddressCountry: string option
            ///Address line 1 (Street address/PO Box/Company name).
            [<Config.Form>]AddressLine1: string option
            ///Address line 2 (Apartment/Suite/Unit/Building).
            [<Config.Form>]AddressLine2: string option
            ///State/County/Province/Region.
            [<Config.Form>]AddressState: string option
            ///ZIP or postal code.
            [<Config.Form>]AddressZip: string option
            ///Two digit number representing the card’s expiration month.
            [<Config.Form>]ExpMonth: string option
            ///Four digit number representing the card’s expiration year.
            [<Config.Form>]ExpYear: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Cardholder name.
            [<Config.Form>]Name: string option
            [<Config.Form>]Owner: Update'Owner option
        }
        with
            static member New(customer: string, id: string, ?accountHolderName: string, ?accountHolderType: Update'AccountHolderType, ?addressCity: string, ?addressCountry: string, ?addressLine1: string, ?addressLine2: string, ?addressState: string, ?addressZip: string, ?expMonth: string, ?expYear: string, ?expand: string list, ?metadata: Map<string, string>, ?name: string, ?owner: Update'Owner) =
                {
                    Customer = customer
                    Id = id
                    AccountHolderName = accountHolderName
                    AccountHolderType = accountHolderType
                    AddressCity = addressCity
                    AddressCountry = addressCountry
                    AddressLine1 = addressLine1
                    AddressLine2 = addressLine2
                    AddressState = addressState
                    AddressZip = addressZip
                    ExpMonth = expMonth
                    ExpYear = expYear
                    Expand = expand
                    Metadata = metadata
                    Name = name
                    Owner = owner
                }

        ///<p>Update a specified source for a given customer.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/customers/{options.Customer}/sources/{options.Id}"
            |> RestApi.postAsync<_, Card> settings (Map.empty) options

    module CustomersSourcesVerify =

        type VerifyOptions = {
            [<Config.Path>]Customer: string
            [<Config.Path>]Id: string
            ///Two positive integers, in *cents*, equal to the values of the microdeposits sent to the bank account.
            [<Config.Form>]Amounts: int list option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(customer: string, id: string, ?amounts: int list, ?expand: string list) =
                {
                    Customer = customer
                    Id = id
                    Amounts = amounts
                    Expand = expand
                }

        ///<p>Verify a specified bank account for a given customer.</p>
        let Verify settings (options: VerifyOptions) =
            $"/v1/customers/{options.Customer}/sources/{options.Id}/verify"
            |> RestApi.postAsync<_, BankAccount> settings (Map.empty) options

    module CustomersTaxIds =

        type ListOptions = {
            [<Config.Path>]Customer: string
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(customer: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Customer = customer
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of tax IDs for a customer.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/customers/{options.Customer}/tax_ids"
            |> RestApi.getAsync<TaxId list> settings qs

        type Create'Type =
        | AeTrn
        | AuAbn
        | BrCnpj
        | BrCpf
        | CaBn
        | CaGstHst
        | CaPstBc
        | CaPstMb
        | CaPstSk
        | CaQst
        | ChVat
        | ClTin
        | EsCif
        | EuVat
        | GbVat
        | HkBr
        | IdNpwp
        | IlVat
        | InGst
        | JpCn
        | JpRn
        | KrBrn
        | LiUid
        | MxRfc
        | MyFrp
        | MyItn
        | MySst
        | NoVat
        | NzGst
        | RuInn
        | RuKpp
        | SaVat
        | SgGst
        | SgUen
        | ThVat
        | TwVat
        | UsEin
        | ZaVat

        type CreateOptions = {
            [<Config.Path>]Customer: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Type of the tax ID, one of `ae_trn`, `au_abn`, `br_cnpj`, `br_cpf`, `ca_bn`, `ca_gst_hst`, `ca_pst_bc`, `ca_pst_mb`, `ca_pst_sk`, `ca_qst`, `ch_vat`, `cl_tin`, `es_cif`, `eu_vat`, `gb_vat`, `hk_br`, `id_npwp`, `il_vat`, `in_gst`, `jp_cn`, `jp_rn`, `kr_brn`, `li_uid`, `mx_rfc`, `my_frp`, `my_itn`, `my_sst`, `no_vat`, `nz_gst`, `ru_inn`, `ru_kpp`, `sa_vat`, `sg_gst`, `sg_uen`, `th_vat`, `tw_vat`, `us_ein`, or `za_vat`
            [<Config.Form>]Type: Create'Type
            ///Value of the tax ID.
            [<Config.Form>]Value: string
        }
        with
            static member New(customer: string, type': Create'Type, value: string, ?expand: string list) =
                {
                    Customer = customer
                    Expand = expand
                    Type = type'
                    Value = value
                }

        ///<p>Creates a new <code>TaxID</code> object for a customer.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/customers/{options.Customer}/tax_ids"
            |> RestApi.postAsync<_, TaxId> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Customer: string
            [<Config.Path>]Id: string
        }
        with
            static member New(customer: string, id: string) =
                {
                    Customer = customer
                    Id = id
                }

        ///<p>Deletes an existing <code>TaxID</code> object.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/customers/{options.Customer}/tax_ids/{options.Id}"
            |> RestApi.deleteAsync<DeletedTaxId> settings (Map.empty)

        type RetrieveOptions = {
            [<Config.Path>]Customer: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
        }
        with
            static member New(customer: string, id: string, ?expand: string list) =
                {
                    Customer = customer
                    Expand = expand
                    Id = id
                }

        ///<p>Retrieves the <code>TaxID</code> object with the given identifier.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/customers/{options.Customer}/tax_ids/{options.Id}"
            |> RestApi.getAsync<TaxId> settings qs

    module Disputes =

        type ListOptions = {
            ///Only return disputes associated to the charge specified by this charge ID.
            [<Config.Query>]Charge: string option
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///Only return disputes associated to the PaymentIntent specified by this PaymentIntent ID.
            [<Config.Query>]PaymentIntent: string option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?charge: string, ?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?paymentIntent: string, ?startingAfter: string) =
                {
                    Charge = charge
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    PaymentIntent = paymentIntent
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of your disputes.</p>
        let List settings (options: ListOptions) =
            let qs = [("charge", options.Charge |> box); ("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("payment_intent", options.PaymentIntent |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/disputes"
            |> RestApi.getAsync<Dispute list> settings qs

        type RetrieveOptions = {
            [<Config.Path>]Dispute: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(dispute: string, ?expand: string list) =
                {
                    Dispute = dispute
                    Expand = expand
                }

        ///<p>Retrieves the dispute with the given ID.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/disputes/{options.Dispute}"
            |> RestApi.getAsync<Dispute> settings qs

        type Update'Evidence = {
            ///Any server or activity logs showing proof that the customer accessed or downloaded the purchased digital product. This information should include IP addresses, corresponding timestamps, and any detailed recorded activity. Has a maximum character count of 20,000.
            [<Config.Form>]AccessActivityLog: string option
            ///The billing address provided by the customer.
            [<Config.Form>]BillingAddress: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Your subscription cancellation policy, as shown to the customer.
            [<Config.Form>]CancellationPolicy: string option
            ///An explanation of how and when the customer was shown your refund policy prior to purchase. Has a maximum character count of 20,000.
            [<Config.Form>]CancellationPolicyDisclosure: string option
            ///A justification for why the customer's subscription was not canceled. Has a maximum character count of 20,000.
            [<Config.Form>]CancellationRebuttal: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Any communication with the customer that you feel is relevant to your case. Examples include emails proving that the customer received the product or service, or demonstrating their use of or satisfaction with the product or service.
            [<Config.Form>]CustomerCommunication: string option
            ///The email address of the customer.
            [<Config.Form>]CustomerEmailAddress: string option
            ///The name of the customer.
            [<Config.Form>]CustomerName: string option
            ///The IP address that the customer used when making the purchase.
            [<Config.Form>]CustomerPurchaseIp: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) A relevant document or contract showing the customer's signature.
            [<Config.Form>]CustomerSignature: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Documentation for the prior charge that can uniquely identify the charge, such as a receipt, shipping label, work order, etc. This document should be paired with a similar document from the disputed payment that proves the two payments are separate.
            [<Config.Form>]DuplicateChargeDocumentation: string option
            ///An explanation of the difference between the disputed charge versus the prior charge that appears to be a duplicate. Has a maximum character count of 20,000.
            [<Config.Form>]DuplicateChargeExplanation: string option
            ///The Stripe ID for the prior charge which appears to be a duplicate of the disputed charge.
            [<Config.Form>]DuplicateChargeId: string option
            ///A description of the product or service that was sold. Has a maximum character count of 20,000.
            [<Config.Form>]ProductDescription: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Any receipt or message sent to the customer notifying them of the charge.
            [<Config.Form>]Receipt: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Your refund policy, as shown to the customer.
            [<Config.Form>]RefundPolicy: string option
            ///Documentation demonstrating that the customer was shown your refund policy prior to purchase. Has a maximum character count of 20,000.
            [<Config.Form>]RefundPolicyDisclosure: string option
            ///A justification for why the customer is not entitled to a refund. Has a maximum character count of 20,000.
            [<Config.Form>]RefundRefusalExplanation: string option
            ///The date on which the customer received or began receiving the purchased service, in a clear human-readable format.
            [<Config.Form>]ServiceDate: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Documentation showing proof that a service was provided to the customer. This could include a copy of a signed contract, work order, or other form of written agreement.
            [<Config.Form>]ServiceDocumentation: string option
            ///The address to which a physical product was shipped. You should try to include as complete address information as possible.
            [<Config.Form>]ShippingAddress: string option
            ///The delivery service that shipped a physical product, such as Fedex, UPS, USPS, etc. If multiple carriers were used for this purchase, please separate them with commas.
            [<Config.Form>]ShippingCarrier: string option
            ///The date on which a physical product began its route to the shipping address, in a clear human-readable format.
            [<Config.Form>]ShippingDate: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Documentation showing proof that a product was shipped to the customer at the same address the customer provided to you. This could include a copy of the shipment receipt, shipping label, etc. It should show the customer's full shipping address, if possible.
            [<Config.Form>]ShippingDocumentation: string option
            ///The tracking number for a physical product, obtained from the delivery service. If multiple tracking numbers were generated for this purchase, please separate them with commas.
            [<Config.Form>]ShippingTrackingNumber: string option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Any additional evidence or statements.
            [<Config.Form>]UncategorizedFile: string option
            ///Any additional evidence or statements. Has a maximum character count of 20,000.
            [<Config.Form>]UncategorizedText: string option
        }
        with
            static member New(?accessActivityLog: string, ?shippingTrackingNumber: string, ?shippingDocumentation: string, ?shippingDate: string, ?shippingCarrier: string, ?shippingAddress: string, ?serviceDocumentation: string, ?serviceDate: string, ?refundRefusalExplanation: string, ?refundPolicyDisclosure: string, ?refundPolicy: string, ?receipt: string, ?uncategorizedFile: string, ?productDescription: string, ?duplicateChargeExplanation: string, ?duplicateChargeDocumentation: string, ?customerSignature: string, ?customerPurchaseIp: string, ?customerName: string, ?customerEmailAddress: string, ?customerCommunication: string, ?cancellationRebuttal: string, ?cancellationPolicyDisclosure: string, ?cancellationPolicy: string, ?billingAddress: string, ?duplicateChargeId: string, ?uncategorizedText: string) =
                {
                    AccessActivityLog = accessActivityLog
                    BillingAddress = billingAddress
                    CancellationPolicy = cancellationPolicy
                    CancellationPolicyDisclosure = cancellationPolicyDisclosure
                    CancellationRebuttal = cancellationRebuttal
                    CustomerCommunication = customerCommunication
                    CustomerEmailAddress = customerEmailAddress
                    CustomerName = customerName
                    CustomerPurchaseIp = customerPurchaseIp
                    CustomerSignature = customerSignature
                    DuplicateChargeDocumentation = duplicateChargeDocumentation
                    DuplicateChargeExplanation = duplicateChargeExplanation
                    DuplicateChargeId = duplicateChargeId
                    ProductDescription = productDescription
                    Receipt = receipt
                    RefundPolicy = refundPolicy
                    RefundPolicyDisclosure = refundPolicyDisclosure
                    RefundRefusalExplanation = refundRefusalExplanation
                    ServiceDate = serviceDate
                    ServiceDocumentation = serviceDocumentation
                    ShippingAddress = shippingAddress
                    ShippingCarrier = shippingCarrier
                    ShippingDate = shippingDate
                    ShippingDocumentation = shippingDocumentation
                    ShippingTrackingNumber = shippingTrackingNumber
                    UncategorizedFile = uncategorizedFile
                    UncategorizedText = uncategorizedText
                }

        type UpdateOptions = {
            [<Config.Path>]Dispute: string
            ///Evidence to upload, to respond to a dispute. Updating any field in the hash will submit all fields in the hash for review. The combined character count of all fields is limited to 150,000.
            [<Config.Form>]Evidence: Update'Evidence option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Whether to immediately submit evidence to the bank. If `false`, evidence is staged on the dispute. Staged evidence is visible in the API and Dashboard, and can be submitted to the bank by making another request with this attribute set to `true` (the default).
            [<Config.Form>]Submit: bool option
        }
        with
            static member New(dispute: string, ?evidence: Update'Evidence, ?expand: string list, ?metadata: Map<string, string>, ?submit: bool) =
                {
                    Dispute = dispute
                    Evidence = evidence
                    Expand = expand
                    Metadata = metadata
                    Submit = submit
                }

        ///<p>When you get a dispute, contacting your customer is always the best first step. If that doesn’t work, you can submit evidence to help us resolve the dispute in your favor. You can do this in your <a href="https://dashboard.stripe.com/disputes">dashboard</a>, but if you prefer, you can use the API to submit evidence programmatically.
        ///Depending on your dispute type, different evidence fields will give you a better chance of winning your dispute. To figure out which evidence fields to provide, see our <a href="/docs/disputes/categories">guide to dispute types</a>.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/disputes/{options.Dispute}"
            |> RestApi.postAsync<_, Dispute> settings (Map.empty) options

    module DisputesClose =

        type CloseOptions = {
            [<Config.Path>]Dispute: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(dispute: string, ?expand: string list) =
                {
                    Dispute = dispute
                    Expand = expand
                }

        ///<p>Closing the dispute for a charge indicates that you do not have any evidence to submit and are essentially dismissing the dispute, acknowledging it as lost.
        ///The status of the dispute will change from <code>needs_response</code> to <code>lost</code>. <em>Closing a dispute is irreversible</em>.</p>
        let Close settings (options: CloseOptions) =
            $"/v1/disputes/{options.Dispute}/close"
            |> RestApi.postAsync<_, Dispute> settings (Map.empty) options

    module EphemeralKeys =

        type CreateOptions = {
            ///The ID of the Customer you'd like to modify using the resulting ephemeral key.
            [<Config.Form>]Customer: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The ID of the Issuing Card you'd like to access using the resulting ephemeral key.
            [<Config.Form>]IssuingCard: string option
        }
        with
            static member New(?customer: string, ?expand: string list, ?issuingCard: string) =
                {
                    Customer = customer
                    Expand = expand
                    IssuingCard = issuingCard
                }

        ///<p>Creates a short-lived API key for a given resource.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/ephemeral_keys"
            |> RestApi.postAsync<_, EphemeralKey> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Key: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(key: string, ?expand: string list) =
                {
                    Key = key
                    Expand = expand
                }

        ///<p>Invalidates a short-lived API key for a given resource.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/ephemeral_keys/{options.Key}"
            |> RestApi.deleteAsync<EphemeralKey> settings (Map.empty)

    module Events =

        type ListOptions = {
            [<Config.Query>]Created: int option
            ///Filter events by whether all webhooks were successfully delivered. If false, events which are still pending or have failed all delivery attempts to a webhook endpoint will be returned.
            [<Config.Query>]DeliverySuccess: bool option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///A string containing a specific event name, or group of events using * as a wildcard. The list will be filtered to include only events with a matching event property.
            [<Config.Query>]Type: string option
            ///An array of up to 20 strings containing specific event names. The list will be filtered to include only events with a matching event property. You may pass either `type` or `types`, but not both.
            [<Config.Query>]Types: string list option
        }
        with
            static member New(?created: int, ?deliverySuccess: bool, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string, ?type': string, ?types: string list) =
                {
                    Created = created
                    DeliverySuccess = deliverySuccess
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                    Type = type'
                    Types = types
                }

        ///<p>List events, going back up to 30 days. Each event data is rendered according to Stripe API version at its creation time, specified in <a href="/docs/api/events/object">event object</a> <code>api_version</code> attribute (not according to your current Stripe API version or <code>Stripe-Version</code> header).</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("delivery_success", options.DeliverySuccess |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("type", options.Type |> box); ("types", options.Types |> box)] |> Map.ofList
            $"/v1/events"
            |> RestApi.getAsync<Event list> settings qs

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
        }
        with
            static member New(id: string, ?expand: string list) =
                {
                    Expand = expand
                    Id = id
                }

        ///<p>Retrieves the details of an event. Supply the unique identifier of the event, which you might have received in a webhook.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/events/{options.Id}"
            |> RestApi.getAsync<Event> settings qs

    module ExchangeRates =

        type ListOptions = {
            ///A cursor for use in pagination. `ending_before` is the currency that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with the exchange rate for currency X your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and total number of supported payout currencies, and the default is the max.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is the currency that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with the exchange rate for currency X, your subsequent call can include `starting_after=X` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of objects that contain the rates at which foreign currencies are converted to one another. Only shows the currencies for which Stripe supports.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/exchange_rates"
            |> RestApi.getAsync<ExchangeRate list> settings qs

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]RateId: string
        }
        with
            static member New(rateId: string, ?expand: string list) =
                {
                    Expand = expand
                    RateId = rateId
                }

        ///<p>Retrieves the exchange rates from the given currency to every supported currency.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/exchange_rates/{options.RateId}"
            |> RestApi.getAsync<ExchangeRate> settings qs

    module FileLinks =

        type ListOptions = {
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///Filter links by their expiration status. By default, all links are returned.
            [<Config.Query>]Expired: bool option
            ///Only return links for the given file.
            [<Config.Query>]File: string option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?created: int, ?endingBefore: string, ?expand: string list, ?expired: bool, ?file: string, ?limit: int, ?startingAfter: string) =
                {
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Expired = expired
                    File = file
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of file links.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("expired", options.Expired |> box); ("file", options.File |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/file_links"
            |> RestApi.getAsync<FileLink list> settings qs

        type CreateOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///A future timestamp after which the link will no longer be usable.
            [<Config.Form>]ExpiresAt: DateTime option
            ///The ID of the file. The file's `purpose` must be one of the following: `business_icon`, `business_logo`, `customer_signature`, `dispute_evidence`, `finance_report_run`, `identity_document_downloadable`, `pci_document`, `selfie`, `sigma_scheduled_query`, or `tax_document_user_upload`.
            [<Config.Form>]File: string
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(file: string, ?expand: string list, ?expiresAt: DateTime, ?metadata: Map<string, string>) =
                {
                    Expand = expand
                    ExpiresAt = expiresAt
                    File = file
                    Metadata = metadata
                }

        ///<p>Creates a new file link object.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/file_links"
            |> RestApi.postAsync<_, FileLink> settings (Map.empty) options

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Link: string
        }
        with
            static member New(link: string, ?expand: string list) =
                {
                    Expand = expand
                    Link = link
                }

        ///<p>Retrieves the file link with the given ID.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/file_links/{options.Link}"
            |> RestApi.getAsync<FileLink> settings qs

        type Update'ExpiresAt =
        | Now

        type UpdateOptions = {
            [<Config.Path>]Link: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///A future timestamp after which the link will no longer be usable, or `now` to expire the link immediately.
            [<Config.Form>]ExpiresAt: Choice<Update'ExpiresAt,DateTime,string> option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(link: string, ?expand: string list, ?expiresAt: Choice<Update'ExpiresAt,DateTime,string>, ?metadata: Map<string, string>) =
                {
                    Link = link
                    Expand = expand
                    ExpiresAt = expiresAt
                    Metadata = metadata
                }

        ///<p>Updates an existing file link object. Expired links can no longer be updated.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/file_links/{options.Link}"
            |> RestApi.postAsync<_, FileLink> settings (Map.empty) options

    module Files =

        type ListOptions = {
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///The file purpose to filter queries by. If none is provided, files will not be filtered by purpose.
            [<Config.Query>]Purpose: string option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?purpose: string, ?startingAfter: string) =
                {
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    Purpose = purpose
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of the files that your account has access to. The files are returned sorted by creation date, with the most recently created files appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("purpose", options.Purpose |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/files"
            |> RestApi.getAsync<File list> settings qs

        type Create'FileLinkData = {
            ///Set this to `true` to create a file link for the newly created file. Creating a link is only possible when the file's `purpose` is one of the following: `business_icon`, `business_logo`, `customer_signature`, `dispute_evidence`, `pci_document`, or `tax_document_user_upload`.
            [<Config.Form>]Create: bool option
            ///A future timestamp after which the link will no longer be usable.
            [<Config.Form>]ExpiresAt: DateTime option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(?create: bool, ?expiresAt: DateTime, ?metadata: Map<string, string>) =
                {
                    Create = create
                    ExpiresAt = expiresAt
                    Metadata = metadata
                }

        type Create'Purpose =
        | AccountRequirement
        | AdditionalVerification
        | BusinessIcon
        | BusinessLogo
        | CustomerSignature
        | DisputeEvidence
        | IdentityDocument
        | PciDocument
        | TaxDocumentUserUpload

        type CreateOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///A file to upload. The file should follow the specifications of RFC 2388 (which defines file transfers for the `multipart/form-data` protocol).
            [<Config.Form>]File: string
            ///Optional parameters to automatically create a [file link](https://stripe.com/docs/api#file_links) for the newly created file.
            [<Config.Form>]FileLinkData: Create'FileLinkData option
            ///The [purpose](https://stripe.com/docs/file-upload#uploading-a-file) of the uploaded file.
            [<Config.Form>]Purpose: Create'Purpose
        }
        with
            static member New(file: string, purpose: Create'Purpose, ?expand: string list, ?fileLinkData: Create'FileLinkData) =
                {
                    Expand = expand
                    File = file
                    FileLinkData = fileLinkData
                    Purpose = purpose
                }

        ///<p>To upload a file to Stripe, you’ll need to send a request of type <code>multipart/form-data</code>. The request should contain the file you would like to upload, as well as the parameters for creating a file.
        ///All of Stripe’s officially supported Client libraries should have support for sending <code>multipart/form-data</code>.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/files"
            |> RestApi.postAsync<_, File> settings (Map.empty) options

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]File: string
        }
        with
            static member New(file: string, ?expand: string list) =
                {
                    Expand = expand
                    File = file
                }

        ///<p>Retrieves the details of an existing file object. Supply the unique file ID from a file, and Stripe will return the corresponding file object. To access file contents, see the <a href="/docs/file-upload#download-file-contents">File Upload Guide</a>.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/files/{options.File}"
            |> RestApi.getAsync<File> settings qs

    module IdentityVerificationReports =

        type ListOptions = {
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return VerificationReports of this type
            [<Config.Query>]Type: string option
            ///Only return VerificationReports created by this VerificationSession ID. It is allowed to provide a VerificationIntent ID.
            [<Config.Query>]VerificationSession: string option
        }
        with
            static member New(?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string, ?type': string, ?verificationSession: string) =
                {
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                    Type = type'
                    VerificationSession = verificationSession
                }

        ///<p>List all verification reports.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("type", options.Type |> box); ("verification_session", options.VerificationSession |> box)] |> Map.ofList
            $"/v1/identity/verification_reports"
            |> RestApi.getAsync<IdentityVerificationReport list> settings qs

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Report: string
        }
        with
            static member New(report: string, ?expand: string list) =
                {
                    Expand = expand
                    Report = report
                }

        ///<p>Retrieves an existing VerificationReport</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/identity/verification_reports/{options.Report}"
            |> RestApi.getAsync<IdentityVerificationReport> settings qs

    module IdentityVerificationSessions =

        type ListOptions = {
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return VerificationSessions with this status. [Learn more about the lifecycle of sessions](https://stripe.com/docs/identity/how-sessions-work).
            [<Config.Query>]Status: string option
        }
        with
            static member New(?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string, ?status: string) =
                {
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                    Status = status
                }

        ///<p>Returns a list of VerificationSessions</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("status", options.Status |> box)] |> Map.ofList
            $"/v1/identity/verification_sessions"
            |> RestApi.getAsync<IdentityVerificationSession list> settings qs

        type Create'OptionsDocumentDocumentOptionsAllowedTypes =
        | DrivingLicense
        | IdCard
        | Passport

        type Create'OptionsDocumentDocumentOptions = {
            ///Array of strings of allowed identity document types. If the provided identity document isn’t one of the allowed types, the verification check will fail with a document_type_not_allowed error code.
            [<Config.Form>]AllowedTypes: Create'OptionsDocumentDocumentOptionsAllowedTypes list option
            ///Collect an ID number and perform an [ID number check](https://stripe.com/docs/identity/verification-checks?type=id-number) with the document’s extracted name and date of birth.
            [<Config.Form>]RequireIdNumber: bool option
            ///Disable image uploads, identity document images have to be captured using the device’s camera.
            [<Config.Form>]RequireLiveCapture: bool option
            ///Capture a face image and perform a [selfie check](https://stripe.com/docs/identity/verification-checks?type=selfie) comparing a photo ID and a picture of your user’s face. [Learn more](https://stripe.com/docs/identity/selfie).
            [<Config.Form>]RequireMatchingSelfie: bool option
        }
        with
            static member New(?allowedTypes: Create'OptionsDocumentDocumentOptionsAllowedTypes list, ?requireIdNumber: bool, ?requireLiveCapture: bool, ?requireMatchingSelfie: bool) =
                {
                    AllowedTypes = allowedTypes
                    RequireIdNumber = requireIdNumber
                    RequireLiveCapture = requireLiveCapture
                    RequireMatchingSelfie = requireMatchingSelfie
                }

        type Create'Options = {
            ///Options that apply to the [document check](https://stripe.com/docs/identity/verification-checks?type=document).
            [<Config.Form>]Document: Choice<Create'OptionsDocumentDocumentOptions,string> option
        }
        with
            static member New(?document: Choice<Create'OptionsDocumentDocumentOptions,string>) =
                {
                    Document = document
                }

        type Create'Type =
        | Document
        | IdNumber

        type CreateOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///A set of options for the session’s verification checks.
            [<Config.Form>]Options: Create'Options option
            ///The URL that the user will be redirected to upon completing the verification flow.
            [<Config.Form>]ReturnUrl: string option
            ///The type of [verification check](https://stripe.com/docs/identity/verification-checks) to be performed.
            [<Config.Form>]Type: Create'Type
        }
        with
            static member New(type': Create'Type, ?expand: string list, ?metadata: Map<string, string>, ?options: Create'Options, ?returnUrl: string) =
                {
                    Expand = expand
                    Metadata = metadata
                    Options = options
                    ReturnUrl = returnUrl
                    Type = type'
                }

        ///<p>Creates a VerificationSession object.
        ///After the VerificationSession is created, display a verification modal using the session <code>client_secret</code> or send your users to the session’s <code>url</code>.
        ///If your API key is in test mode, verification checks won’t actually process, though everything else will occur as if in live mode.
        ///Related guide: <a href="/docs/identity/verify-identity-documents">Verify your users’ identity documents</a>.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/identity/verification_sessions"
            |> RestApi.postAsync<_, IdentityVerificationSession> settings (Map.empty) options

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Session: string
        }
        with
            static member New(session: string, ?expand: string list) =
                {
                    Expand = expand
                    Session = session
                }

        ///<p>Retrieves the details of a VerificationSession that was previously created.
        ///When the session status is <code>requires_input</code>, you can use this method to retrieve a valid
        ///<code>client_secret</code> or <code>url</code> to allow re-submission.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/identity/verification_sessions/{options.Session}"
            |> RestApi.getAsync<IdentityVerificationSession> settings qs

        type Update'OptionsDocumentDocumentOptionsAllowedTypes =
        | DrivingLicense
        | IdCard
        | Passport

        type Update'OptionsDocumentDocumentOptions = {
            ///Array of strings of allowed identity document types. If the provided identity document isn’t one of the allowed types, the verification check will fail with a document_type_not_allowed error code.
            [<Config.Form>]AllowedTypes: Update'OptionsDocumentDocumentOptionsAllowedTypes list option
            ///Collect an ID number and perform an [ID number check](https://stripe.com/docs/identity/verification-checks?type=id-number) with the document’s extracted name and date of birth.
            [<Config.Form>]RequireIdNumber: bool option
            ///Disable image uploads, identity document images have to be captured using the device’s camera.
            [<Config.Form>]RequireLiveCapture: bool option
            ///Capture a face image and perform a [selfie check](https://stripe.com/docs/identity/verification-checks?type=selfie) comparing a photo ID and a picture of your user’s face. [Learn more](https://stripe.com/docs/identity/selfie).
            [<Config.Form>]RequireMatchingSelfie: bool option
        }
        with
            static member New(?allowedTypes: Update'OptionsDocumentDocumentOptionsAllowedTypes list, ?requireIdNumber: bool, ?requireLiveCapture: bool, ?requireMatchingSelfie: bool) =
                {
                    AllowedTypes = allowedTypes
                    RequireIdNumber = requireIdNumber
                    RequireLiveCapture = requireLiveCapture
                    RequireMatchingSelfie = requireMatchingSelfie
                }

        type Update'Options = {
            ///Options that apply to the [document check](https://stripe.com/docs/identity/verification-checks?type=document).
            [<Config.Form>]Document: Choice<Update'OptionsDocumentDocumentOptions,string> option
        }
        with
            static member New(?document: Choice<Update'OptionsDocumentDocumentOptions,string>) =
                {
                    Document = document
                }

        type Update'Type =
        | Document
        | IdNumber

        type UpdateOptions = {
            [<Config.Path>]Session: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///A set of options for the session’s verification checks.
            [<Config.Form>]Options: Update'Options option
            ///The type of [verification check](https://stripe.com/docs/identity/verification-checks) to be performed.
            [<Config.Form>]Type: Update'Type option
        }
        with
            static member New(session: string, ?expand: string list, ?metadata: Map<string, string>, ?options: Update'Options, ?type': Update'Type) =
                {
                    Session = session
                    Expand = expand
                    Metadata = metadata
                    Options = options
                    Type = type'
                }

        ///<p>Updates a VerificationSession object.
        ///When the session status is <code>requires_input</code>, you can use this method to update the
        ///verification check and options.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/identity/verification_sessions/{options.Session}"
            |> RestApi.postAsync<_, IdentityVerificationSession> settings (Map.empty) options

    module IdentityVerificationSessionsCancel =

        type CancelOptions = {
            [<Config.Path>]Session: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(session: string, ?expand: string list) =
                {
                    Session = session
                    Expand = expand
                }

        ///<p>A VerificationSession object can be canceled when it is in <code>requires_input</code> <a href="/docs/identity/how-sessions-work">status</a>.
        ///Once canceled, future submission attempts are disabled. This cannot be undone. <a href="/docs/identity/verification-sessions#cancel">Learn more</a>.</p>
        let Cancel settings (options: CancelOptions) =
            $"/v1/identity/verification_sessions/{options.Session}/cancel"
            |> RestApi.postAsync<_, IdentityVerificationSession> settings (Map.empty) options

    module IdentityVerificationSessionsRedact =

        type RedactOptions = {
            [<Config.Path>]Session: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(session: string, ?expand: string list) =
                {
                    Session = session
                    Expand = expand
                }

        ///<p>Redact a VerificationSession to remove all collected information from Stripe. This will redact
        ///the VerificationSession and all objects related to it, including VerificationReports, Events,
        ///request logs, etc.
        ///A VerificationSession object can be redacted when it is in <code>requires_input</code> or <code>verified</code>
        ///<a href="/docs/identity/how-sessions-work">status</a>. Redacting a VerificationSession in <code>requires_action</code>
        ///state will automatically cancel it.
        ///The redaction process may take up to four days. When the redaction process is in progress, the
        ///VerificationSession’s <code>redaction.status</code> field will be set to <code>processing</code>; when the process is
        ///finished, it will change to <code>redacted</code> and an <code>identity.verification_session.redacted</code> event
        ///will be emitted.
        ///Redaction is irreversible. Redacted objects are still accessible in the Stripe API, but all the
        ///fields that contain personal data will be replaced by the string <code>[redacted]</code> or a similar
        ///placeholder. The <code>metadata</code> field will also be erased. Redacted objects cannot be updated or
        ///used for any purpose.
        ///<a href="/docs/identity/verification-sessions#redact">Learn more</a>.</p>
        let Redact settings (options: RedactOptions) =
            $"/v1/identity/verification_sessions/{options.Session}/redact"
            |> RestApi.postAsync<_, IdentityVerificationSession> settings (Map.empty) options

    module Invoiceitems =

        type ListOptions = {
            [<Config.Query>]Created: int option
            ///The identifier of the customer whose invoice items to return. If none is provided, all invoice items will be returned.
            [<Config.Query>]Customer: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///Only return invoice items belonging to this invoice. If none is provided, all invoice items will be returned. If specifying an invoice, no customer identifier is needed.
            [<Config.Query>]Invoice: string option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///Set to `true` to only show pending invoice items, which are not yet attached to any invoices. Set to `false` to only show invoice items already attached to invoices. If unspecified, no filter is applied.
            [<Config.Query>]Pending: bool option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?created: int, ?customer: string, ?endingBefore: string, ?expand: string list, ?invoice: string, ?limit: int, ?pending: bool, ?startingAfter: string) =
                {
                    Created = created
                    Customer = customer
                    EndingBefore = endingBefore
                    Expand = expand
                    Invoice = invoice
                    Limit = limit
                    Pending = pending
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of your invoice items. Invoice items are returned sorted by creation date, with the most recently created invoice items appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("customer", options.Customer |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("invoice", options.Invoice |> box); ("limit", options.Limit |> box); ("pending", options.Pending |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/invoiceitems"
            |> RestApi.getAsync<Invoiceitem list> settings qs

        type Create'Discounts = {
            ///ID of the coupon to create a new discount for.
            [<Config.Form>]Coupon: string option
            ///ID of an existing discount on the object (or one of its ancestors) to reuse.
            [<Config.Form>]Discount: string option
        }
        with
            static member New(?coupon: string, ?discount: string) =
                {
                    Coupon = coupon
                    Discount = discount
                }

        type Create'Period = {
            ///The end of the period, which must be greater than or equal to the start.
            [<Config.Form>]End: DateTime option
            ///The start of the period.
            [<Config.Form>]Start: DateTime option
        }
        with
            static member New(?end': DateTime, ?start: DateTime) =
                {
                    End = end'
                    Start = start
                }

        type Create'PriceDataTaxBehavior =
        | Exclusive
        | Inclusive
        | Unspecified

        type Create'PriceData = {
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string option
            ///The ID of the product that this price will belong to.
            [<Config.Form>]Product: string option
            ///Specifies whether the price is considered inclusive of taxes or exclusive of taxes. One of `inclusive`, `exclusive`, or `unspecified`. Once specified as either `inclusive` or `exclusive`, it cannot be changed.
            [<Config.Form>]TaxBehavior: Create'PriceDataTaxBehavior option
            ///A positive integer in %s (or 0 for a free price) representing how much to charge.
            [<Config.Form>]UnitAmount: int option
            ///Same as `unit_amount`, but accepts a decimal value in %s with at most 12 decimal places. Only one of `unit_amount` and `unit_amount_decimal` can be set.
            [<Config.Form>]UnitAmountDecimal: string option
        }
        with
            static member New(?currency: string, ?product: string, ?taxBehavior: Create'PriceDataTaxBehavior, ?unitAmount: int, ?unitAmountDecimal: string) =
                {
                    Currency = currency
                    Product = product
                    TaxBehavior = taxBehavior
                    UnitAmount = unitAmount
                    UnitAmountDecimal = unitAmountDecimal
                }

        type CreateOptions = {
            ///The integer amount in %s of the charge to be applied to the upcoming invoice. Passing in a negative `amount` will reduce the `amount_due` on the invoice.
            [<Config.Form>]Amount: int option
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string option
            ///The ID of the customer who will be billed when this invoice item is billed.
            [<Config.Form>]Customer: string
            ///An arbitrary string which you can attach to the invoice item. The description is displayed in the invoice for easy tracking.
            [<Config.Form>]Description: string option
            ///Controls whether discounts apply to this invoice item. Defaults to false for prorations or negative invoice items, and true for all other invoice items.
            [<Config.Form>]Discountable: bool option
            ///The coupons to redeem into discounts for the invoice item or invoice line item.
            [<Config.Form>]Discounts: Choice<Create'Discounts list,string> option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///The ID of an existing invoice to add this invoice item to. When left blank, the invoice item will be added to the next upcoming scheduled invoice. This is useful when adding invoice items in response to an invoice.created webhook. You can only add invoice items to draft invoices and there is a maximum of 250 items per invoice.
            [<Config.Form>]Invoice: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The period associated with this invoice item.
            [<Config.Form>]Period: Create'Period option
            ///The ID of the price object.
            [<Config.Form>]Price: string option
            ///Data used to generate a new [Price](https://stripe.com/docs/api/prices) object inline.
            [<Config.Form>]PriceData: Create'PriceData option
            ///Non-negative integer. The quantity of units for the invoice item.
            [<Config.Form>]Quantity: int option
            ///The ID of a subscription to add this invoice item to. When left blank, the invoice item will be be added to the next upcoming scheduled invoice. When set, scheduled invoices for subscriptions other than the specified subscription will ignore the invoice item. Use this when you want to express that an invoice item has been accrued within the context of a particular subscription.
            [<Config.Form>]Subscription: string option
            ///The tax rates which apply to the invoice item. When set, the `default_tax_rates` on the invoice do not apply to this invoice item.
            [<Config.Form>]TaxRates: string list option
            ///The integer unit amount in %s of the charge to be applied to the upcoming invoice. This `unit_amount` will be multiplied by the quantity to get the full amount. Passing in a negative `unit_amount` will reduce the `amount_due` on the invoice.
            [<Config.Form>]UnitAmount: int option
            ///Same as `unit_amount`, but accepts a decimal value in %s with at most 12 decimal places. Only one of `unit_amount` and `unit_amount_decimal` can be set.
            [<Config.Form>]UnitAmountDecimal: string option
        }
        with
            static member New(customer: string, ?amount: int, ?taxRates: string list, ?subscription: string, ?quantity: int, ?priceData: Create'PriceData, ?price: string, ?period: Create'Period, ?metadata: Map<string, string>, ?invoice: string, ?expand: string list, ?discounts: Choice<Create'Discounts list,string>, ?discountable: bool, ?description: string, ?currency: string, ?unitAmount: int, ?unitAmountDecimal: string) =
                {
                    Amount = amount
                    Currency = currency
                    Customer = customer
                    Description = description
                    Discountable = discountable
                    Discounts = discounts
                    Expand = expand
                    Invoice = invoice
                    Metadata = metadata
                    Period = period
                    Price = price
                    PriceData = priceData
                    Quantity = quantity
                    Subscription = subscription
                    TaxRates = taxRates
                    UnitAmount = unitAmount
                    UnitAmountDecimal = unitAmountDecimal
                }

        ///<p>Creates an item to be added to a draft invoice (up to 250 items per invoice). If no invoice is specified, the item will be on the next invoice created for the customer specified.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/invoiceitems"
            |> RestApi.postAsync<_, Invoiceitem> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Invoiceitem: string
        }
        with
            static member New(invoiceitem: string) =
                {
                    Invoiceitem = invoiceitem
                }

        ///<p>Deletes an invoice item, removing it from an invoice. Deleting invoice items is only possible when they’re not attached to invoices, or if it’s attached to a draft invoice.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/invoiceitems/{options.Invoiceitem}"
            |> RestApi.deleteAsync<DeletedInvoiceitem> settings (Map.empty)

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Invoiceitem: string
        }
        with
            static member New(invoiceitem: string, ?expand: string list) =
                {
                    Expand = expand
                    Invoiceitem = invoiceitem
                }

        ///<p>Retrieves the invoice item with the given ID.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/invoiceitems/{options.Invoiceitem}"
            |> RestApi.getAsync<Invoiceitem> settings qs

        type Update'Discounts = {
            ///ID of the coupon to create a new discount for.
            [<Config.Form>]Coupon: string option
            ///ID of an existing discount on the object (or one of its ancestors) to reuse.
            [<Config.Form>]Discount: string option
        }
        with
            static member New(?coupon: string, ?discount: string) =
                {
                    Coupon = coupon
                    Discount = discount
                }

        type Update'Period = {
            ///The end of the period, which must be greater than or equal to the start.
            [<Config.Form>]End: DateTime option
            ///The start of the period.
            [<Config.Form>]Start: DateTime option
        }
        with
            static member New(?end': DateTime, ?start: DateTime) =
                {
                    End = end'
                    Start = start
                }

        type Update'PriceDataTaxBehavior =
        | Exclusive
        | Inclusive
        | Unspecified

        type Update'PriceData = {
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string option
            ///The ID of the product that this price will belong to.
            [<Config.Form>]Product: string option
            ///Specifies whether the price is considered inclusive of taxes or exclusive of taxes. One of `inclusive`, `exclusive`, or `unspecified`. Once specified as either `inclusive` or `exclusive`, it cannot be changed.
            [<Config.Form>]TaxBehavior: Update'PriceDataTaxBehavior option
            ///A positive integer in %s (or 0 for a free price) representing how much to charge.
            [<Config.Form>]UnitAmount: int option
            ///Same as `unit_amount`, but accepts a decimal value in %s with at most 12 decimal places. Only one of `unit_amount` and `unit_amount_decimal` can be set.
            [<Config.Form>]UnitAmountDecimal: string option
        }
        with
            static member New(?currency: string, ?product: string, ?taxBehavior: Update'PriceDataTaxBehavior, ?unitAmount: int, ?unitAmountDecimal: string) =
                {
                    Currency = currency
                    Product = product
                    TaxBehavior = taxBehavior
                    UnitAmount = unitAmount
                    UnitAmountDecimal = unitAmountDecimal
                }

        type UpdateOptions = {
            [<Config.Path>]Invoiceitem: string
            ///The integer amount in %s of the charge to be applied to the upcoming invoice. If you want to apply a credit to the customer's account, pass a negative amount.
            [<Config.Form>]Amount: int option
            ///An arbitrary string which you can attach to the invoice item. The description is displayed in the invoice for easy tracking.
            [<Config.Form>]Description: string option
            ///Controls whether discounts apply to this invoice item. Defaults to false for prorations or negative invoice items, and true for all other invoice items. Cannot be set to true for prorations.
            [<Config.Form>]Discountable: bool option
            ///The coupons & existing discounts which apply to the invoice item or invoice line item. Item discounts are applied before invoice discounts. Pass an empty string to remove previously-defined discounts.
            [<Config.Form>]Discounts: Choice<Update'Discounts list,string> option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The period associated with this invoice item.
            [<Config.Form>]Period: Update'Period option
            ///The ID of the price object.
            [<Config.Form>]Price: string option
            ///Data used to generate a new [Price](https://stripe.com/docs/api/prices) object inline.
            [<Config.Form>]PriceData: Update'PriceData option
            ///Non-negative integer. The quantity of units for the invoice item.
            [<Config.Form>]Quantity: int option
            ///The tax rates which apply to the invoice item. When set, the `default_tax_rates` on the invoice do not apply to this invoice item. Pass an empty string to remove previously-defined tax rates.
            [<Config.Form>]TaxRates: Choice<string list,string> option
            ///The integer unit amount in %s of the charge to be applied to the upcoming invoice. This unit_amount will be multiplied by the quantity to get the full amount. If you want to apply a credit to the customer's account, pass a negative unit_amount.
            [<Config.Form>]UnitAmount: int option
            ///Same as `unit_amount`, but accepts a decimal value in %s with at most 12 decimal places. Only one of `unit_amount` and `unit_amount_decimal` can be set.
            [<Config.Form>]UnitAmountDecimal: string option
        }
        with
            static member New(invoiceitem: string, ?amount: int, ?description: string, ?discountable: bool, ?discounts: Choice<Update'Discounts list,string>, ?expand: string list, ?metadata: Map<string, string>, ?period: Update'Period, ?price: string, ?priceData: Update'PriceData, ?quantity: int, ?taxRates: Choice<string list,string>, ?unitAmount: int, ?unitAmountDecimal: string) =
                {
                    Invoiceitem = invoiceitem
                    Amount = amount
                    Description = description
                    Discountable = discountable
                    Discounts = discounts
                    Expand = expand
                    Metadata = metadata
                    Period = period
                    Price = price
                    PriceData = priceData
                    Quantity = quantity
                    TaxRates = taxRates
                    UnitAmount = unitAmount
                    UnitAmountDecimal = unitAmountDecimal
                }

        ///<p>Updates the amount or description of an invoice item on an upcoming invoice. Updating an invoice item is only possible before the invoice it’s attached to is closed.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/invoiceitems/{options.Invoiceitem}"
            |> RestApi.postAsync<_, Invoiceitem> settings (Map.empty) options

    module Invoices =

        type ListOptions = {
            ///The collection method of the invoice to retrieve. Either `charge_automatically` or `send_invoice`.
            [<Config.Query>]CollectionMethod: string option
            [<Config.Query>]Created: int option
            ///Only return invoices for the customer specified by this customer ID.
            [<Config.Query>]Customer: string option
            [<Config.Query>]DueDate: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///The status of the invoice, one of `draft`, `open`, `paid`, `uncollectible`, or `void`. [Learn more](https://stripe.com/docs/billing/invoices/workflow#workflow-overview)
            [<Config.Query>]Status: string option
            ///Only return invoices for the subscription specified by this subscription ID.
            [<Config.Query>]Subscription: string option
        }
        with
            static member New(?collectionMethod: string, ?created: int, ?customer: string, ?dueDate: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string, ?status: string, ?subscription: string) =
                {
                    CollectionMethod = collectionMethod
                    Created = created
                    Customer = customer
                    DueDate = dueDate
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                    Status = status
                    Subscription = subscription
                }

        ///<p>You can list all invoices, or list the invoices for a specific customer. The invoices are returned sorted by creation date, with the most recently created invoices appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("collection_method", options.CollectionMethod |> box); ("created", options.Created |> box); ("customer", options.Customer |> box); ("due_date", options.DueDate |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("status", options.Status |> box); ("subscription", options.Subscription |> box)] |> Map.ofList
            $"/v1/invoices"
            |> RestApi.getAsync<Invoice list> settings qs

        type Create'AutomaticTax = {
            ///Controls whether Stripe will automatically compute tax on this invoice.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?enabled: bool) =
                {
                    Enabled = enabled
                }

        type Create'CustomFields = {
            ///The name of the custom field. This may be up to 30 characters.
            [<Config.Form>]Name: string option
            ///The value of the custom field. This may be up to 30 characters.
            [<Config.Form>]Value: string option
        }
        with
            static member New(?name: string, ?value: string) =
                {
                    Name = name
                    Value = value
                }

        type Create'Discounts = {
            ///ID of the coupon to create a new discount for.
            [<Config.Form>]Coupon: string option
            ///ID of an existing discount on the object (or one of its ancestors) to reuse.
            [<Config.Form>]Discount: string option
        }
        with
            static member New(?coupon: string, ?discount: string) =
                {
                    Coupon = coupon
                    Discount = discount
                }

        type Create'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptionsPreferredLanguage =
        | De
        | En
        | Fr
        | Nl

        type Create'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptions = {
            ///Preferred language of the Bancontact authorization page that the customer is redirected to.
            [<Config.Form>]PreferredLanguage: Create'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptionsPreferredLanguage option
        }
        with
            static member New(?preferredLanguage: Create'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptionsPreferredLanguage) =
                {
                    PreferredLanguage = preferredLanguage
                }

        type Create'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptionsRequestThreeDSecure =
        | Any
        | Automatic

        type Create'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptions = {
            ///We strongly recommend that you rely on our SCA Engine to automatically prompt your customers for authentication based on risk level and [other requirements](https://stripe.com/docs/strong-customer-authentication). However, if you wish to request 3D Secure based on logic from your own fraud engine, provide this option. Read our guide on [manually requesting 3D Secure](https://stripe.com/docs/payments/3d-secure#manual-three-ds) for more information on how this configuration interacts with Radar and our SCA Engine.
            [<Config.Form>]RequestThreeDSecure: Create'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptionsRequestThreeDSecure option
        }
        with
            static member New(?requestThreeDSecure: Create'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptionsRequestThreeDSecure) =
                {
                    RequestThreeDSecure = requestThreeDSecure
                }

        type Create'PaymentSettingsPaymentMethodOptions = {
            ///If paying by `bancontact`, this sub-hash contains details about the Bancontact payment method options to pass to the invoice’s PaymentIntent.
            [<Config.Form>]Bancontact: Choice<Create'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptions,string> option
            ///If paying by `card`, this sub-hash contains details about the Card payment method options to pass to the invoice’s PaymentIntent.
            [<Config.Form>]Card: Choice<Create'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptions,string> option
        }
        with
            static member New(?bancontact: Choice<Create'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptions,string>, ?card: Choice<Create'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptions,string>) =
                {
                    Bancontact = bancontact
                    Card = card
                }

        type Create'PaymentSettingsPaymentMethodTypes =
        | AchCreditTransfer
        | AchDebit
        | AuBecsDebit
        | BacsDebit
        | Bancontact
        | Boleto
        | Card
        | Fpx
        | Giropay
        | Ideal
        | SepaCreditTransfer
        | SepaDebit
        | Sofort
        | WechatPay

        type Create'PaymentSettings = {
            ///Payment-method-specific configuration to provide to the invoice’s PaymentIntent.
            [<Config.Form>]PaymentMethodOptions: Create'PaymentSettingsPaymentMethodOptions option
            ///The list of payment method types (e.g. card) to provide to the invoice’s PaymentIntent. If not set, Stripe attempts to automatically determine the types to use by looking at the invoice’s default payment method, the subscription’s default payment method, the customer’s default payment method, and your [invoice template settings](https://dashboard.stripe.com/settings/billing/invoice).
            [<Config.Form>]PaymentMethodTypes: Choice<Create'PaymentSettingsPaymentMethodTypes list,string> option
        }
        with
            static member New(?paymentMethodOptions: Create'PaymentSettingsPaymentMethodOptions, ?paymentMethodTypes: Choice<Create'PaymentSettingsPaymentMethodTypes list,string>) =
                {
                    PaymentMethodOptions = paymentMethodOptions
                    PaymentMethodTypes = paymentMethodTypes
                }

        type Create'TransferData = {
            ///The amount that will be transferred automatically when the invoice is paid. If no amount is set, the full amount is transferred.
            [<Config.Form>]Amount: int option
            ///ID of an existing, connected Stripe account.
            [<Config.Form>]Destination: string option
        }
        with
            static member New(?amount: int, ?destination: string) =
                {
                    Amount = amount
                    Destination = destination
                }

        type Create'CollectionMethod =
        | ChargeAutomatically
        | SendInvoice

        type CreateOptions = {
            ///The account tax IDs associated with the invoice. Only editable when the invoice is a draft.
            [<Config.Form>]AccountTaxIds: Choice<string list,string> option
            ///A fee in %s that will be applied to the invoice and transferred to the application owner's Stripe account. The request must be made with an OAuth key or the Stripe-Account header in order to take an application fee. For more information, see the application fees [documentation](https://stripe.com/docs/billing/invoices/connect#collecting-fees).
            [<Config.Form>]ApplicationFeeAmount: int option
            ///Controls whether Stripe will perform [automatic collection](https://stripe.com/docs/billing/invoices/workflow/#auto_advance) of the invoice. When `false`, the invoice's state will not automatically advance without an explicit action.
            [<Config.Form>]AutoAdvance: bool option
            ///Settings for automatic tax lookup for this invoice.
            [<Config.Form>]AutomaticTax: Create'AutomaticTax option
            ///Either `charge_automatically`, or `send_invoice`. When charging automatically, Stripe will attempt to pay this invoice using the default source attached to the customer. When sending an invoice, Stripe will email this invoice to the customer with payment instructions. Defaults to `charge_automatically`.
            [<Config.Form>]CollectionMethod: Create'CollectionMethod option
            ///A list of up to 4 custom fields to be displayed on the invoice.
            [<Config.Form>]CustomFields: Choice<Create'CustomFields list,string> option
            ///The ID of the customer who will be billed.
            [<Config.Form>]Customer: string
            ///The number of days from when the invoice is created until it is due. Valid only for invoices where `collection_method=send_invoice`.
            [<Config.Form>]DaysUntilDue: int option
            ///ID of the default payment method for the invoice. It must belong to the customer associated with the invoice. If not set, defaults to the subscription's default payment method, if any, or to the default payment method in the customer's invoice settings.
            [<Config.Form>]DefaultPaymentMethod: string option
            ///ID of the default payment source for the invoice. It must belong to the customer associated with the invoice and be in a chargeable state. If not set, defaults to the subscription's default source, if any, or to the customer's default source.
            [<Config.Form>]DefaultSource: string option
            ///The tax rates that will apply to any line item that does not have `tax_rates` set.
            [<Config.Form>]DefaultTaxRates: string list option
            ///An arbitrary string attached to the object. Often useful for displaying to users. Referenced as 'memo' in the Dashboard.
            [<Config.Form>]Description: string option
            ///The coupons to redeem into discounts for the invoice. If not specified, inherits the discount from the invoice's customer. Pass an empty string to avoid inheriting any discounts.
            [<Config.Form>]Discounts: Choice<Create'Discounts list,string> option
            ///The date on which payment for this invoice is due. Valid only for invoices where `collection_method=send_invoice`.
            [<Config.Form>]DueDate: DateTime option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Footer to be displayed on the invoice.
            [<Config.Form>]Footer: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The account (if any) for which the funds of the invoice payment are intended. If set, the invoice will be presented with the branding and support information of the specified account. See the [Invoices with Connect](https://stripe.com/docs/billing/invoices/connect) documentation for details.
            [<Config.Form>]OnBehalfOf: string option
            ///Configuration settings for the PaymentIntent that is generated when the invoice is finalized.
            [<Config.Form>]PaymentSettings: Create'PaymentSettings option
            ///Extra information about a charge for the customer's credit card statement. It must contain at least one letter. If not specified and this invoice is part of a subscription, the default `statement_descriptor` will be set to the first subscription item's product's `statement_descriptor`.
            [<Config.Form>]StatementDescriptor: string option
            ///The ID of the subscription to invoice, if any. If not set, the created invoice will include all pending invoice items for the customer. If set, the created invoice will only include pending invoice items for that subscription and pending invoice items not associated with any subscription. The subscription's billing cycle and regular subscription events won't be affected.
            [<Config.Form>]Subscription: string option
            ///If specified, the funds from the invoice will be transferred to the destination and the ID of the resulting transfer will be found on the invoice's charge.
            [<Config.Form>]TransferData: Create'TransferData option
        }
        with
            static member New(customer: string, ?accountTaxIds: Choice<string list,string>, ?statementDescriptor: string, ?paymentSettings: Create'PaymentSettings, ?onBehalfOf: string, ?metadata: Map<string, string>, ?footer: string, ?expand: string list, ?dueDate: DateTime, ?discounts: Choice<Create'Discounts list,string>, ?description: string, ?defaultTaxRates: string list, ?defaultSource: string, ?defaultPaymentMethod: string, ?daysUntilDue: int, ?customFields: Choice<Create'CustomFields list,string>, ?collectionMethod: Create'CollectionMethod, ?automaticTax: Create'AutomaticTax, ?autoAdvance: bool, ?applicationFeeAmount: int, ?subscription: string, ?transferData: Create'TransferData) =
                {
                    AccountTaxIds = accountTaxIds
                    ApplicationFeeAmount = applicationFeeAmount
                    AutoAdvance = autoAdvance
                    AutomaticTax = automaticTax
                    CollectionMethod = collectionMethod
                    CustomFields = customFields
                    Customer = customer
                    DaysUntilDue = daysUntilDue
                    DefaultPaymentMethod = defaultPaymentMethod
                    DefaultSource = defaultSource
                    DefaultTaxRates = defaultTaxRates
                    Description = description
                    Discounts = discounts
                    DueDate = dueDate
                    Expand = expand
                    Footer = footer
                    Metadata = metadata
                    OnBehalfOf = onBehalfOf
                    PaymentSettings = paymentSettings
                    StatementDescriptor = statementDescriptor
                    Subscription = subscription
                    TransferData = transferData
                }

        ///<p>This endpoint creates a draft invoice for a given customer. The draft invoice created pulls in all pending invoice items on that customer, including prorations. The invoice remains a draft until you <a href="#finalize_invoice">finalize</a> the invoice, which allows you to <a href="#pay_invoice">pay</a> or <a href="#send_invoice">send</a> the invoice to your customers.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/invoices"
            |> RestApi.postAsync<_, Invoice> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Invoice: string
        }
        with
            static member New(invoice: string) =
                {
                    Invoice = invoice
                }

        ///<p>Permanently deletes a one-off invoice draft. This cannot be undone. Attempts to delete invoices that are no longer in a draft state will fail; once an invoice has been finalized or if an invoice is for a subscription, it must be <a href="#void_invoice">voided</a>.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/invoices/{options.Invoice}"
            |> RestApi.deleteAsync<DeletedInvoice> settings (Map.empty)

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Invoice: string
        }
        with
            static member New(invoice: string, ?expand: string list) =
                {
                    Expand = expand
                    Invoice = invoice
                }

        ///<p>Retrieves the invoice with the given ID.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/invoices/{options.Invoice}"
            |> RestApi.getAsync<Invoice> settings qs

        type Update'AutomaticTax = {
            ///Controls whether Stripe will automatically compute tax on this invoice.
            [<Config.Form>]Enabled: bool option
        }
        with
            static member New(?enabled: bool) =
                {
                    Enabled = enabled
                }

        type Update'CustomFields = {
            ///The name of the custom field. This may be up to 30 characters.
            [<Config.Form>]Name: string option
            ///The value of the custom field. This may be up to 30 characters.
            [<Config.Form>]Value: string option
        }
        with
            static member New(?name: string, ?value: string) =
                {
                    Name = name
                    Value = value
                }

        type Update'Discounts = {
            ///ID of the coupon to create a new discount for.
            [<Config.Form>]Coupon: string option
            ///ID of an existing discount on the object (or one of its ancestors) to reuse.
            [<Config.Form>]Discount: string option
        }
        with
            static member New(?coupon: string, ?discount: string) =
                {
                    Coupon = coupon
                    Discount = discount
                }

        type Update'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptionsPreferredLanguage =
        | De
        | En
        | Fr
        | Nl

        type Update'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptions = {
            ///Preferred language of the Bancontact authorization page that the customer is redirected to.
            [<Config.Form>]PreferredLanguage: Update'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptionsPreferredLanguage option
        }
        with
            static member New(?preferredLanguage: Update'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptionsPreferredLanguage) =
                {
                    PreferredLanguage = preferredLanguage
                }

        type Update'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptionsRequestThreeDSecure =
        | Any
        | Automatic

        type Update'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptions = {
            ///We strongly recommend that you rely on our SCA Engine to automatically prompt your customers for authentication based on risk level and [other requirements](https://stripe.com/docs/strong-customer-authentication). However, if you wish to request 3D Secure based on logic from your own fraud engine, provide this option. Read our guide on [manually requesting 3D Secure](https://stripe.com/docs/payments/3d-secure#manual-three-ds) for more information on how this configuration interacts with Radar and our SCA Engine.
            [<Config.Form>]RequestThreeDSecure: Update'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptionsRequestThreeDSecure option
        }
        with
            static member New(?requestThreeDSecure: Update'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptionsRequestThreeDSecure) =
                {
                    RequestThreeDSecure = requestThreeDSecure
                }

        type Update'PaymentSettingsPaymentMethodOptions = {
            ///If paying by `bancontact`, this sub-hash contains details about the Bancontact payment method options to pass to the invoice’s PaymentIntent.
            [<Config.Form>]Bancontact: Choice<Update'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptions,string> option
            ///If paying by `card`, this sub-hash contains details about the Card payment method options to pass to the invoice’s PaymentIntent.
            [<Config.Form>]Card: Choice<Update'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptions,string> option
        }
        with
            static member New(?bancontact: Choice<Update'PaymentSettingsPaymentMethodOptionsBancontactInvoicePaymentMethodOptions,string>, ?card: Choice<Update'PaymentSettingsPaymentMethodOptionsCardInvoicePaymentMethodOptions,string>) =
                {
                    Bancontact = bancontact
                    Card = card
                }

        type Update'PaymentSettingsPaymentMethodTypes =
        | AchCreditTransfer
        | AchDebit
        | AuBecsDebit
        | BacsDebit
        | Bancontact
        | Boleto
        | Card
        | Fpx
        | Giropay
        | Ideal
        | SepaCreditTransfer
        | SepaDebit
        | Sofort
        | WechatPay

        type Update'PaymentSettings = {
            ///Payment-method-specific configuration to provide to the invoice’s PaymentIntent.
            [<Config.Form>]PaymentMethodOptions: Update'PaymentSettingsPaymentMethodOptions option
            ///The list of payment method types (e.g. card) to provide to the invoice’s PaymentIntent. If not set, Stripe attempts to automatically determine the types to use by looking at the invoice’s default payment method, the subscription’s default payment method, the customer’s default payment method, and your [invoice template settings](https://dashboard.stripe.com/settings/billing/invoice).
            [<Config.Form>]PaymentMethodTypes: Choice<Update'PaymentSettingsPaymentMethodTypes list,string> option
        }
        with
            static member New(?paymentMethodOptions: Update'PaymentSettingsPaymentMethodOptions, ?paymentMethodTypes: Choice<Update'PaymentSettingsPaymentMethodTypes list,string>) =
                {
                    PaymentMethodOptions = paymentMethodOptions
                    PaymentMethodTypes = paymentMethodTypes
                }

        type Update'TransferDataTransferDataSpecs = {
            ///The amount that will be transferred automatically when the invoice is paid. If no amount is set, the full amount is transferred.
            [<Config.Form>]Amount: int option
            ///ID of an existing, connected Stripe account.
            [<Config.Form>]Destination: string option
        }
        with
            static member New(?amount: int, ?destination: string) =
                {
                    Amount = amount
                    Destination = destination
                }

        type Update'CollectionMethod =
        | ChargeAutomatically
        | SendInvoice

        type UpdateOptions = {
            [<Config.Path>]Invoice: string
            ///The account tax IDs associated with the invoice. Only editable when the invoice is a draft.
            [<Config.Form>]AccountTaxIds: Choice<string list,string> option
            ///A fee in %s that will be applied to the invoice and transferred to the application owner's Stripe account. The request must be made with an OAuth key or the Stripe-Account header in order to take an application fee. For more information, see the application fees [documentation](https://stripe.com/docs/billing/invoices/connect#collecting-fees).
            [<Config.Form>]ApplicationFeeAmount: int option
            ///Controls whether Stripe will perform [automatic collection](https://stripe.com/docs/billing/invoices/workflow/#auto_advance) of the invoice.
            [<Config.Form>]AutoAdvance: bool option
            ///Settings for automatic tax lookup for this invoice.
            [<Config.Form>]AutomaticTax: Update'AutomaticTax option
            ///Either `charge_automatically` or `send_invoice`. This field can be updated only on `draft` invoices.
            [<Config.Form>]CollectionMethod: Update'CollectionMethod option
            ///A list of up to 4 custom fields to be displayed on the invoice. If a value for `custom_fields` is specified, the list specified will replace the existing custom field list on this invoice. Pass an empty string to remove previously-defined fields.
            [<Config.Form>]CustomFields: Choice<Update'CustomFields list,string> option
            ///The number of days from which the invoice is created until it is due. Only valid for invoices where `collection_method=send_invoice`. This field can only be updated on `draft` invoices.
            [<Config.Form>]DaysUntilDue: int option
            ///ID of the default payment method for the invoice. It must belong to the customer associated with the invoice. If not set, defaults to the subscription's default payment method, if any, or to the default payment method in the customer's invoice settings.
            [<Config.Form>]DefaultPaymentMethod: string option
            ///ID of the default payment source for the invoice. It must belong to the customer associated with the invoice and be in a chargeable state. If not set, defaults to the subscription's default source, if any, or to the customer's default source.
            [<Config.Form>]DefaultSource: string option
            ///The tax rates that will apply to any line item that does not have `tax_rates` set. Pass an empty string to remove previously-defined tax rates.
            [<Config.Form>]DefaultTaxRates: Choice<string list,string> option
            ///An arbitrary string attached to the object. Often useful for displaying to users. Referenced as 'memo' in the Dashboard.
            [<Config.Form>]Description: string option
            ///The discounts that will apply to the invoice. Pass an empty string to remove previously-defined discounts.
            [<Config.Form>]Discounts: Choice<Update'Discounts list,string> option
            ///The date on which payment for this invoice is due. Only valid for invoices where `collection_method=send_invoice`. This field can only be updated on `draft` invoices.
            [<Config.Form>]DueDate: DateTime option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Footer to be displayed on the invoice.
            [<Config.Form>]Footer: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The account (if any) for which the funds of the invoice payment are intended. If set, the invoice will be presented with the branding and support information of the specified account. See the [Invoices with Connect](https://stripe.com/docs/billing/invoices/connect) documentation for details.
            [<Config.Form>]OnBehalfOf: Choice<string,string> option
            ///Configuration settings for the PaymentIntent that is generated when the invoice is finalized.
            [<Config.Form>]PaymentSettings: Update'PaymentSettings option
            ///Extra information about a charge for the customer's credit card statement. It must contain at least one letter. If not specified and this invoice is part of a subscription, the default `statement_descriptor` will be set to the first subscription item's product's `statement_descriptor`.
            [<Config.Form>]StatementDescriptor: string option
            ///If specified, the funds from the invoice will be transferred to the destination and the ID of the resulting transfer will be found on the invoice's charge. This will be unset if you POST an empty value.
            [<Config.Form>]TransferData: Choice<Update'TransferDataTransferDataSpecs,string> option
        }
        with
            static member New(invoice: string, ?paymentSettings: Update'PaymentSettings, ?onBehalfOf: Choice<string,string>, ?metadata: Map<string, string>, ?footer: string, ?expand: string list, ?dueDate: DateTime, ?discounts: Choice<Update'Discounts list,string>, ?description: string, ?statementDescriptor: string, ?defaultTaxRates: Choice<string list,string>, ?defaultPaymentMethod: string, ?daysUntilDue: int, ?customFields: Choice<Update'CustomFields list,string>, ?collectionMethod: Update'CollectionMethod, ?automaticTax: Update'AutomaticTax, ?autoAdvance: bool, ?applicationFeeAmount: int, ?accountTaxIds: Choice<string list,string>, ?defaultSource: string, ?transferData: Choice<Update'TransferDataTransferDataSpecs,string>) =
                {
                    Invoice = invoice
                    AccountTaxIds = accountTaxIds
                    ApplicationFeeAmount = applicationFeeAmount
                    AutoAdvance = autoAdvance
                    AutomaticTax = automaticTax
                    CollectionMethod = collectionMethod
                    CustomFields = customFields
                    DaysUntilDue = daysUntilDue
                    DefaultPaymentMethod = defaultPaymentMethod
                    DefaultSource = defaultSource
                    DefaultTaxRates = defaultTaxRates
                    Description = description
                    Discounts = discounts
                    DueDate = dueDate
                    Expand = expand
                    Footer = footer
                    Metadata = metadata
                    OnBehalfOf = onBehalfOf
                    PaymentSettings = paymentSettings
                    StatementDescriptor = statementDescriptor
                    TransferData = transferData
                }

        ///<p>Draft invoices are fully editable. Once an invoice is <a href="/docs/billing/invoices/workflow#finalized">finalized</a>,
        ///monetary values, as well as <code>collection_method</code>, become uneditable.
        ///If you would like to stop the Stripe Billing engine from automatically finalizing, reattempting payments on,
        ///sending reminders for, or <a href="/docs/billing/invoices/reconciliation">automatically reconciling</a> invoices, pass
        ///<code>auto_advance=false</code>.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/invoices/{options.Invoice}"
            |> RestApi.postAsync<_, Invoice> settings (Map.empty) options

    module InvoicesUpcoming =

        type UpcomingOptions = {
            ///Settings for automatic tax lookup for this invoice preview.
            [<Config.Query>]AutomaticTax: Map<string, string> option
            ///The code of the coupon to apply. If `subscription` or `subscription_items` is provided, the invoice returned will preview updating or creating a subscription with that coupon. Otherwise, it will preview applying that coupon to the customer for the next upcoming invoice from among the customer's subscriptions. The invoice can be previewed without a coupon by passing this value as an empty string.
            [<Config.Query>]Coupon: string option
            ///The identifier of the customer whose upcoming invoice you'd like to retrieve.
            [<Config.Query>]Customer: string option
            ///Details about the customer you want to invoice
            [<Config.Query>]CustomerDetails: Map<string, string> option
            ///The coupons to redeem into discounts for the invoice preview. If not specified, inherits the discount from the customer or subscription. Pass an empty string to avoid inheriting any discounts. To preview the upcoming invoice for a subscription that hasn't been created, use `coupon` instead.
            [<Config.Query>]Discounts: string list option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///List of invoice items to add or update in the upcoming invoice preview.
            [<Config.Query>]InvoiceItems: string list option
            ///The identifier of the unstarted schedule whose upcoming invoice you'd like to retrieve. Cannot be used with subscription or subscription fields.
            [<Config.Query>]Schedule: string option
            ///The identifier of the subscription for which you'd like to retrieve the upcoming invoice. If not provided, but a `subscription_items` is provided, you will preview creating a subscription with those items. If neither `subscription` nor `subscription_items` is provided, you will retrieve the next upcoming invoice from among the customer's subscriptions.
            [<Config.Query>]Subscription: string option
            ///For new subscriptions, a future timestamp to anchor the subscription's [billing cycle](https://stripe.com/docs/subscriptions/billing-cycle). This is used to determine the date of the first full invoice, and, for plans with `month` or `year` intervals, the day of the month for subsequent invoices. For existing subscriptions, the value can only be set to `now` or `unchanged`.
            [<Config.Query>]SubscriptionBillingCycleAnchor: string option
            ///Timestamp indicating when the subscription should be scheduled to cancel. Will prorate if within the current period and prorations have been enabled using `proration_behavior`.
            [<Config.Query>]SubscriptionCancelAt: int option
            ///Boolean indicating whether this subscription should cancel at the end of the current period.
            [<Config.Query>]SubscriptionCancelAtPeriodEnd: bool option
            ///This simulates the subscription being canceled or expired immediately.
            [<Config.Query>]SubscriptionCancelNow: bool option
            ///If provided, the invoice returned will preview updating or creating a subscription with these default tax rates. The default tax rates will apply to any line item that does not have `tax_rates` set.
            [<Config.Query>]SubscriptionDefaultTaxRates: string list option
            ///A list of up to 20 subscription items, each with an attached price.
            [<Config.Query>]SubscriptionItems: string list option
            ///Determines how to handle [prorations](https://stripe.com/docs/subscriptions/billing-cycle#prorations) when the billing cycle changes (e.g., when switching plans, resetting `billing_cycle_anchor=now`, or starting a trial), or if an item's `quantity` changes. Valid values are `create_prorations`, `none`, or `always_invoice`.
            ///Passing `create_prorations` will cause proration invoice items to be created when applicable. These proration items will only be invoiced immediately under [certain conditions](https://stripe.com/docs/subscriptions/upgrading-downgrading#immediate-payment). In order to always invoice immediately for prorations, pass `always_invoice`.
            ///Prorations can be disabled by passing `none`.
            [<Config.Query>]SubscriptionProrationBehavior: string option
            ///If previewing an update to a subscription, and doing proration, `subscription_proration_date` forces the proration to be calculated as though the update was done at the specified time. The time given must be within the current subscription period, and cannot be before the subscription was on its current plan. If set, `subscription`, and one of `subscription_items`, or `subscription_trial_end` are required. Also, `subscription_proration_behavior` cannot be set to 'none'.
            [<Config.Query>]SubscriptionProrationDate: int option
            ///Date a subscription is intended to start (can be future or past)
            [<Config.Query>]SubscriptionStartDate: int option
            ///If provided, the invoice returned will preview updating or creating a subscription with that trial end. If set, one of `subscription_items` or `subscription` is required.
            [<Config.Query>]SubscriptionTrialEnd: string option
            ///Indicates if a plan's `trial_period_days` should be applied to the subscription. Setting `subscription_trial_end` per subscription is preferred, and this defaults to `false`. Setting this flag to `true` together with `subscription_trial_end` is not allowed.
            [<Config.Query>]SubscriptionTrialFromPlan: bool option
        }
        with
            static member New(?automaticTax: Map<string, string>, ?subscriptionStartDate: int, ?subscriptionProrationDate: int, ?subscriptionProrationBehavior: string, ?subscriptionItems: string list, ?subscriptionDefaultTaxRates: string list, ?subscriptionCancelNow: bool, ?subscriptionCancelAtPeriodEnd: bool, ?subscriptionCancelAt: int, ?subscriptionBillingCycleAnchor: string, ?subscription: string, ?schedule: string, ?invoiceItems: string list, ?expand: string list, ?discounts: string list, ?customerDetails: Map<string, string>, ?customer: string, ?coupon: string, ?subscriptionTrialEnd: string, ?subscriptionTrialFromPlan: bool) =
                {
                    AutomaticTax = automaticTax
                    Coupon = coupon
                    Customer = customer
                    CustomerDetails = customerDetails
                    Discounts = discounts
                    Expand = expand
                    InvoiceItems = invoiceItems
                    Schedule = schedule
                    Subscription = subscription
                    SubscriptionBillingCycleAnchor = subscriptionBillingCycleAnchor
                    SubscriptionCancelAt = subscriptionCancelAt
                    SubscriptionCancelAtPeriodEnd = subscriptionCancelAtPeriodEnd
                    SubscriptionCancelNow = subscriptionCancelNow
                    SubscriptionDefaultTaxRates = subscriptionDefaultTaxRates
                    SubscriptionItems = subscriptionItems
                    SubscriptionProrationBehavior = subscriptionProrationBehavior
                    SubscriptionProrationDate = subscriptionProrationDate
                    SubscriptionStartDate = subscriptionStartDate
                    SubscriptionTrialEnd = subscriptionTrialEnd
                    SubscriptionTrialFromPlan = subscriptionTrialFromPlan
                }

        ///<p>At any time, you can preview the upcoming invoice for a customer. This will show you all the charges that are pending, including subscription renewal charges, invoice item charges, etc. It will also show you any discounts that are applicable to the invoice.
        ///Note that when you are viewing an upcoming invoice, you are simply viewing a preview – the invoice has not yet been created. As such, the upcoming invoice will not show up in invoice listing calls, and you cannot use the API to pay or edit the invoice. If you want to change the amount that your customer will be billed, you can add, remove, or update pending invoice items, or update the customer’s discount.
        ///You can preview the effects of updating a subscription, including a preview of what proration will take place. To ensure that the actual proration is calculated exactly the same as the previewed proration, you should pass a <code>proration_date</code> parameter when doing the actual subscription update. The value passed in should be the same as the <code>subscription_proration_date</code> returned on the upcoming invoice resource. The recommended way to get only the prorations being previewed is to consider only proration line items where <code>period[start]</code> is equal to the <code>subscription_proration_date</code> on the upcoming invoice resource.</p>
        let Upcoming settings (options: UpcomingOptions) =
            let qs = [("automatic_tax", options.AutomaticTax |> box); ("coupon", options.Coupon |> box); ("customer", options.Customer |> box); ("customer_details", options.CustomerDetails |> box); ("discounts", options.Discounts |> box); ("expand", options.Expand |> box); ("invoice_items", options.InvoiceItems |> box); ("schedule", options.Schedule |> box); ("subscription", options.Subscription |> box); ("subscription_billing_cycle_anchor", options.SubscriptionBillingCycleAnchor |> box); ("subscription_cancel_at", options.SubscriptionCancelAt |> box); ("subscription_cancel_at_period_end", options.SubscriptionCancelAtPeriodEnd |> box); ("subscription_cancel_now", options.SubscriptionCancelNow |> box); ("subscription_default_tax_rates", options.SubscriptionDefaultTaxRates |> box); ("subscription_items", options.SubscriptionItems |> box); ("subscription_proration_behavior", options.SubscriptionProrationBehavior |> box); ("subscription_proration_date", options.SubscriptionProrationDate |> box); ("subscription_start_date", options.SubscriptionStartDate |> box); ("subscription_trial_end", options.SubscriptionTrialEnd |> box); ("subscription_trial_from_plan", options.SubscriptionTrialFromPlan |> box)] |> Map.ofList
            $"/v1/invoices/upcoming"
            |> RestApi.getAsync<Invoice> settings qs

    module InvoicesUpcomingLines =

        type UpcomingLinesOptions = {
            ///Settings for automatic tax lookup for this invoice preview.
            [<Config.Query>]AutomaticTax: Map<string, string> option
            ///The code of the coupon to apply. If `subscription` or `subscription_items` is provided, the invoice returned will preview updating or creating a subscription with that coupon. Otherwise, it will preview applying that coupon to the customer for the next upcoming invoice from among the customer's subscriptions. The invoice can be previewed without a coupon by passing this value as an empty string.
            [<Config.Query>]Coupon: string option
            ///The identifier of the customer whose upcoming invoice you'd like to retrieve.
            [<Config.Query>]Customer: string option
            ///Details about the customer you want to invoice
            [<Config.Query>]CustomerDetails: Map<string, string> option
            ///The coupons to redeem into discounts for the invoice preview. If not specified, inherits the discount from the customer or subscription. Pass an empty string to avoid inheriting any discounts. To preview the upcoming invoice for a subscription that hasn't been created, use `coupon` instead.
            [<Config.Query>]Discounts: string list option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///List of invoice items to add or update in the upcoming invoice preview.
            [<Config.Query>]InvoiceItems: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///The identifier of the unstarted schedule whose upcoming invoice you'd like to retrieve. Cannot be used with subscription or subscription fields.
            [<Config.Query>]Schedule: string option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///The identifier of the subscription for which you'd like to retrieve the upcoming invoice. If not provided, but a `subscription_items` is provided, you will preview creating a subscription with those items. If neither `subscription` nor `subscription_items` is provided, you will retrieve the next upcoming invoice from among the customer's subscriptions.
            [<Config.Query>]Subscription: string option
            ///For new subscriptions, a future timestamp to anchor the subscription's [billing cycle](https://stripe.com/docs/subscriptions/billing-cycle). This is used to determine the date of the first full invoice, and, for plans with `month` or `year` intervals, the day of the month for subsequent invoices. For existing subscriptions, the value can only be set to `now` or `unchanged`.
            [<Config.Query>]SubscriptionBillingCycleAnchor: string option
            ///Timestamp indicating when the subscription should be scheduled to cancel. Will prorate if within the current period and prorations have been enabled using `proration_behavior`.
            [<Config.Query>]SubscriptionCancelAt: int option
            ///Boolean indicating whether this subscription should cancel at the end of the current period.
            [<Config.Query>]SubscriptionCancelAtPeriodEnd: bool option
            ///This simulates the subscription being canceled or expired immediately.
            [<Config.Query>]SubscriptionCancelNow: bool option
            ///If provided, the invoice returned will preview updating or creating a subscription with these default tax rates. The default tax rates will apply to any line item that does not have `tax_rates` set.
            [<Config.Query>]SubscriptionDefaultTaxRates: string list option
            ///A list of up to 20 subscription items, each with an attached price.
            [<Config.Query>]SubscriptionItems: string list option
            ///Determines how to handle [prorations](https://stripe.com/docs/subscriptions/billing-cycle#prorations) when the billing cycle changes (e.g., when switching plans, resetting `billing_cycle_anchor=now`, or starting a trial), or if an item's `quantity` changes. Valid values are `create_prorations`, `none`, or `always_invoice`.
            ///Passing `create_prorations` will cause proration invoice items to be created when applicable. These proration items will only be invoiced immediately under [certain conditions](https://stripe.com/docs/subscriptions/upgrading-downgrading#immediate-payment). In order to always invoice immediately for prorations, pass `always_invoice`.
            ///Prorations can be disabled by passing `none`.
            [<Config.Query>]SubscriptionProrationBehavior: string option
            ///If previewing an update to a subscription, and doing proration, `subscription_proration_date` forces the proration to be calculated as though the update was done at the specified time. The time given must be within the current subscription period, and cannot be before the subscription was on its current plan. If set, `subscription`, and one of `subscription_items`, or `subscription_trial_end` are required. Also, `subscription_proration_behavior` cannot be set to 'none'.
            [<Config.Query>]SubscriptionProrationDate: int option
            ///Date a subscription is intended to start (can be future or past)
            [<Config.Query>]SubscriptionStartDate: int option
            ///If provided, the invoice returned will preview updating or creating a subscription with that trial end. If set, one of `subscription_items` or `subscription` is required.
            [<Config.Query>]SubscriptionTrialEnd: string option
            ///Indicates if a plan's `trial_period_days` should be applied to the subscription. Setting `subscription_trial_end` per subscription is preferred, and this defaults to `false`. Setting this flag to `true` together with `subscription_trial_end` is not allowed.
            [<Config.Query>]SubscriptionTrialFromPlan: bool option
        }
        with
            static member New(?automaticTax: Map<string, string>, ?subscriptionStartDate: int, ?subscriptionProrationDate: int, ?subscriptionProrationBehavior: string, ?subscriptionItems: string list, ?subscriptionDefaultTaxRates: string list, ?subscriptionCancelNow: bool, ?subscriptionCancelAtPeriodEnd: bool, ?subscriptionCancelAt: int, ?subscriptionBillingCycleAnchor: string, ?subscriptionTrialEnd: string, ?subscription: string, ?schedule: string, ?limit: int, ?invoiceItems: string list, ?expand: string list, ?endingBefore: string, ?discounts: string list, ?customerDetails: Map<string, string>, ?customer: string, ?coupon: string, ?startingAfter: string, ?subscriptionTrialFromPlan: bool) =
                {
                    AutomaticTax = automaticTax
                    Coupon = coupon
                    Customer = customer
                    CustomerDetails = customerDetails
                    Discounts = discounts
                    EndingBefore = endingBefore
                    Expand = expand
                    InvoiceItems = invoiceItems
                    Limit = limit
                    Schedule = schedule
                    StartingAfter = startingAfter
                    Subscription = subscription
                    SubscriptionBillingCycleAnchor = subscriptionBillingCycleAnchor
                    SubscriptionCancelAt = subscriptionCancelAt
                    SubscriptionCancelAtPeriodEnd = subscriptionCancelAtPeriodEnd
                    SubscriptionCancelNow = subscriptionCancelNow
                    SubscriptionDefaultTaxRates = subscriptionDefaultTaxRates
                    SubscriptionItems = subscriptionItems
                    SubscriptionProrationBehavior = subscriptionProrationBehavior
                    SubscriptionProrationDate = subscriptionProrationDate
                    SubscriptionStartDate = subscriptionStartDate
                    SubscriptionTrialEnd = subscriptionTrialEnd
                    SubscriptionTrialFromPlan = subscriptionTrialFromPlan
                }

        ///<p>When retrieving an upcoming invoice, you’ll get a <strong>lines</strong> property containing the total count of line items and the first handful of those items. There is also a URL where you can retrieve the full (paginated) list of line items.</p>
        let UpcomingLines settings (options: UpcomingLinesOptions) =
            let qs = [("automatic_tax", options.AutomaticTax |> box); ("coupon", options.Coupon |> box); ("customer", options.Customer |> box); ("customer_details", options.CustomerDetails |> box); ("discounts", options.Discounts |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("invoice_items", options.InvoiceItems |> box); ("limit", options.Limit |> box); ("schedule", options.Schedule |> box); ("starting_after", options.StartingAfter |> box); ("subscription", options.Subscription |> box); ("subscription_billing_cycle_anchor", options.SubscriptionBillingCycleAnchor |> box); ("subscription_cancel_at", options.SubscriptionCancelAt |> box); ("subscription_cancel_at_period_end", options.SubscriptionCancelAtPeriodEnd |> box); ("subscription_cancel_now", options.SubscriptionCancelNow |> box); ("subscription_default_tax_rates", options.SubscriptionDefaultTaxRates |> box); ("subscription_items", options.SubscriptionItems |> box); ("subscription_proration_behavior", options.SubscriptionProrationBehavior |> box); ("subscription_proration_date", options.SubscriptionProrationDate |> box); ("subscription_start_date", options.SubscriptionStartDate |> box); ("subscription_trial_end", options.SubscriptionTrialEnd |> box); ("subscription_trial_from_plan", options.SubscriptionTrialFromPlan |> box)] |> Map.ofList
            $"/v1/invoices/upcoming/lines"
            |> RestApi.getAsync<LineItem list> settings qs

    module InvoicesFinalize =

        type FinalizeInvoiceOptions = {
            [<Config.Path>]Invoice: string
            ///Controls whether Stripe will perform [automatic collection](https://stripe.com/docs/billing/invoices/overview#auto-advance) of the invoice. When `false`, the invoice's state will not automatically advance without an explicit action.
            [<Config.Form>]AutoAdvance: bool option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(invoice: string, ?autoAdvance: bool, ?expand: string list) =
                {
                    Invoice = invoice
                    AutoAdvance = autoAdvance
                    Expand = expand
                }

        ///<p>Stripe automatically finalizes drafts before sending and attempting payment on invoices. However, if you’d like to finalize a draft invoice manually, you can do so using this method.</p>
        let FinalizeInvoice settings (options: FinalizeInvoiceOptions) =
            $"/v1/invoices/{options.Invoice}/finalize"
            |> RestApi.postAsync<_, Invoice> settings (Map.empty) options

    module InvoicesLines =

        type ListOptions = {
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Invoice: string
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(invoice: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    EndingBefore = endingBefore
                    Expand = expand
                    Invoice = invoice
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>When retrieving an invoice, you’ll get a <strong>lines</strong> property containing the total count of line items and the first handful of those items. There is also a URL where you can retrieve the full (paginated) list of line items.</p>
        let List settings (options: ListOptions) =
            let qs = [("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/invoices/{options.Invoice}/lines"
            |> RestApi.getAsync<LineItem list> settings qs

    module InvoicesMarkUncollectible =

        type MarkUncollectibleOptions = {
            [<Config.Path>]Invoice: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(invoice: string, ?expand: string list) =
                {
                    Invoice = invoice
                    Expand = expand
                }

        ///<p>Marking an invoice as uncollectible is useful for keeping track of bad debts that can be written off for accounting purposes.</p>
        let MarkUncollectible settings (options: MarkUncollectibleOptions) =
            $"/v1/invoices/{options.Invoice}/mark_uncollectible"
            |> RestApi.postAsync<_, Invoice> settings (Map.empty) options

    module InvoicesPay =

        type PayOptions = {
            [<Config.Path>]Invoice: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///In cases where the source used to pay the invoice has insufficient funds, passing `forgive=true` controls whether a charge should be attempted for the full amount available on the source, up to the amount to fully pay the invoice. This effectively forgives the difference between the amount available on the source and the amount due. 
            ///Passing `forgive=false` will fail the charge if the source hasn't been pre-funded with the right amount. An example for this case is with ACH Credit Transfers and wires: if the amount wired is less than the amount due by a small amount, you might want to forgive the difference. Defaults to `false`.
            [<Config.Form>]Forgive: bool option
            ///Indicates if a customer is on or off-session while an invoice payment is attempted. Defaults to `true` (off-session).
            [<Config.Form>]OffSession: bool option
            ///Boolean representing whether an invoice is paid outside of Stripe. This will result in no charge being made. Defaults to `false`.
            [<Config.Form>]PaidOutOfBand: bool option
            ///A PaymentMethod to be charged. The PaymentMethod must be the ID of a PaymentMethod belonging to the customer associated with the invoice being paid.
            [<Config.Form>]PaymentMethod: string option
            ///A payment source to be charged. The source must be the ID of a source belonging to the customer associated with the invoice being paid.
            [<Config.Form>]Source: string option
        }
        with
            static member New(invoice: string, ?expand: string list, ?forgive: bool, ?offSession: bool, ?paidOutOfBand: bool, ?paymentMethod: string, ?source: string) =
                {
                    Invoice = invoice
                    Expand = expand
                    Forgive = forgive
                    OffSession = offSession
                    PaidOutOfBand = paidOutOfBand
                    PaymentMethod = paymentMethod
                    Source = source
                }

        ///<p>Stripe automatically creates and then attempts to collect payment on invoices for customers on subscriptions according to your <a href="https://dashboard.stripe.com/account/billing/automatic">subscriptions settings</a>. However, if you’d like to attempt payment on an invoice out of the normal collection schedule or for some other reason, you can do so.</p>
        let Pay settings (options: PayOptions) =
            $"/v1/invoices/{options.Invoice}/pay"
            |> RestApi.postAsync<_, Invoice> settings (Map.empty) options

    module InvoicesSend =

        type SendInvoiceOptions = {
            [<Config.Path>]Invoice: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(invoice: string, ?expand: string list) =
                {
                    Invoice = invoice
                    Expand = expand
                }

        ///<p>Stripe will automatically send invoices to customers according to your <a href="https://dashboard.stripe.com/account/billing/automatic">subscriptions settings</a>. However, if you’d like to manually send an invoice to your customer out of the normal schedule, you can do so. When sending invoices that have already been paid, there will be no reference to the payment in the email.
        ///Requests made in test-mode result in no emails being sent, despite sending an <code>invoice.sent</code> event.</p>
        let SendInvoice settings (options: SendInvoiceOptions) =
            $"/v1/invoices/{options.Invoice}/send"
            |> RestApi.postAsync<_, Invoice> settings (Map.empty) options

    module InvoicesVoid =

        type VoidInvoiceOptions = {
            [<Config.Path>]Invoice: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(invoice: string, ?expand: string list) =
                {
                    Invoice = invoice
                    Expand = expand
                }

        ///<p>Mark a finalized invoice as void. This cannot be undone. Voiding an invoice is similar to <a href="#delete_invoice">deletion</a>, however it only applies to finalized invoices and maintains a papertrail where the invoice can still be found.</p>
        let VoidInvoice settings (options: VoidInvoiceOptions) =
            $"/v1/invoices/{options.Invoice}/void"
            |> RestApi.postAsync<_, Invoice> settings (Map.empty) options

    module IssuerFraudRecords =

        type ListOptions = {
            ///Only return issuer fraud records for the charge specified by this charge ID.
            [<Config.Query>]Charge: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?charge: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Charge = charge
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of issuer fraud records.</p>
        let List settings (options: ListOptions) =
            let qs = [("charge", options.Charge |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/issuer_fraud_records"
            |> RestApi.getAsync<IssuerFraudRecord list> settings qs

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]IssuerFraudRecord: string
        }
        with
            static member New(issuerFraudRecord: string, ?expand: string list) =
                {
                    Expand = expand
                    IssuerFraudRecord = issuerFraudRecord
                }

        ///<p>Retrieves the details of an issuer fraud record that has previously been created. 
        ///Please refer to the <a href="#issuer_fraud_record_object">issuer fraud record</a> object reference for more details.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/issuer_fraud_records/{options.IssuerFraudRecord}"
            |> RestApi.getAsync<IssuerFraudRecord> settings qs

    module IssuingAuthorizations =

        type ListOptions = {
            ///Only return authorizations that belong to the given card.
            [<Config.Query>]Card: string option
            ///Only return authorizations that belong to the given cardholder.
            [<Config.Query>]Cardholder: string option
            ///Only return authorizations that were created during the given date interval.
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return authorizations with the given status. One of `pending`, `closed`, or `reversed`.
            [<Config.Query>]Status: string option
        }
        with
            static member New(?card: string, ?cardholder: string, ?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string, ?status: string) =
                {
                    Card = card
                    Cardholder = cardholder
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                    Status = status
                }

        ///<p>Returns a list of Issuing <code>Authorization</code> objects. The objects are sorted in descending order by creation date, with the most recently created object appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("card", options.Card |> box); ("cardholder", options.Cardholder |> box); ("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("status", options.Status |> box)] |> Map.ofList
            $"/v1/issuing/authorizations"
            |> RestApi.getAsync<IssuingAuthorization list> settings qs

        type RetrieveOptions = {
            [<Config.Path>]Authorization: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(authorization: string, ?expand: string list) =
                {
                    Authorization = authorization
                    Expand = expand
                }

        ///<p>Retrieves an Issuing <code>Authorization</code> object.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/issuing/authorizations/{options.Authorization}"
            |> RestApi.getAsync<IssuingAuthorization> settings qs

        type UpdateOptions = {
            [<Config.Path>]Authorization: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(authorization: string, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Authorization = authorization
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Updates the specified Issuing <code>Authorization</code> object by setting the values of the parameters passed. Any parameters not provided will be left unchanged.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/issuing/authorizations/{options.Authorization}"
            |> RestApi.postAsync<_, IssuingAuthorization> settings (Map.empty) options

    module IssuingAuthorizationsApprove =

        type ApproveOptions = {
            [<Config.Path>]Authorization: string
            ///If the authorization's `pending_request.is_amount_controllable` property is `true`, you may provide this value to control how much to hold for the authorization. Must be positive (use [`decline`](https://stripe.com/docs/api/issuing/authorizations/decline) to decline an authorization request).
            [<Config.Form>]Amount: int option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(authorization: string, ?amount: int, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Authorization = authorization
                    Amount = amount
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Approves a pending Issuing <code>Authorization</code> object. This request should be made within the timeout window of the <a href="/docs/issuing/controls/real-time-authorizations">real-time authorization</a> flow.</p>
        let Approve settings (options: ApproveOptions) =
            $"/v1/issuing/authorizations/{options.Authorization}/approve"
            |> RestApi.postAsync<_, IssuingAuthorization> settings (Map.empty) options

    module IssuingAuthorizationsDecline =

        type DeclineOptions = {
            [<Config.Path>]Authorization: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(authorization: string, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Authorization = authorization
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Declines a pending Issuing <code>Authorization</code> object. This request should be made within the timeout window of the <a href="/docs/issuing/controls/real-time-authorizations">real time authorization</a> flow.</p>
        let Decline settings (options: DeclineOptions) =
            $"/v1/issuing/authorizations/{options.Authorization}/decline"
            |> RestApi.postAsync<_, IssuingAuthorization> settings (Map.empty) options

    module IssuingCardholders =

        type ListOptions = {
            ///Only return cardholders that were created during the given date interval.
            [<Config.Query>]Created: int option
            ///Only return cardholders that have the given email address.
            [<Config.Query>]Email: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///Only return cardholders that have the given phone number.
            [<Config.Query>]PhoneNumber: string option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return cardholders that have the given status. One of `active`, `inactive`, or `blocked`.
            [<Config.Query>]Status: string option
            ///Only return cardholders that have the given type. One of `individual` or `company`.
            [<Config.Query>]Type: string option
        }
        with
            static member New(?created: int, ?email: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?phoneNumber: string, ?startingAfter: string, ?status: string, ?type': string) =
                {
                    Created = created
                    Email = email
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    PhoneNumber = phoneNumber
                    StartingAfter = startingAfter
                    Status = status
                    Type = type'
                }

        ///<p>Returns a list of Issuing <code>Cardholder</code> objects. The objects are sorted in descending order by creation date, with the most recently created object appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("email", options.Email |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("phone_number", options.PhoneNumber |> box); ("starting_after", options.StartingAfter |> box); ("status", options.Status |> box); ("type", options.Type |> box)] |> Map.ofList
            $"/v1/issuing/cardholders"
            |> RestApi.getAsync<IssuingCardholder list> settings qs

        type Create'BillingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'Billing = {
            ///The cardholder’s billing address.
            [<Config.Form>]Address: Create'BillingAddress option
        }
        with
            static member New(?address: Create'BillingAddress) =
                {
                    Address = address
                }

        type Create'Company = {
            ///The entity's business ID number.
            [<Config.Form>]TaxId: string option
        }
        with
            static member New(?taxId: string) =
                {
                    TaxId = taxId
                }

        type Create'IndividualDob = {
            ///The day of birth, between 1 and 31.
            [<Config.Form>]Day: int option
            ///The month of birth, between 1 and 12.
            [<Config.Form>]Month: int option
            ///The four-digit year of birth.
            [<Config.Form>]Year: int option
        }
        with
            static member New(?day: int, ?month: int, ?year: int) =
                {
                    Day = day
                    Month = month
                    Year = year
                }

        type Create'IndividualVerificationDocument = {
            ///The back of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`.
            [<Config.Form>]Back: string option
            ///The front of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Create'IndividualVerification = {
            ///An identifying document, either a passport or local ID card.
            [<Config.Form>]Document: Create'IndividualVerificationDocument option
        }
        with
            static member New(?document: Create'IndividualVerificationDocument) =
                {
                    Document = document
                }

        type Create'Individual = {
            ///The date of birth of this cardholder.
            [<Config.Form>]Dob: Create'IndividualDob option
            ///The first name of this cardholder.
            [<Config.Form>]FirstName: string option
            ///The last name of this cardholder.
            [<Config.Form>]LastName: string option
            ///Government-issued ID document for this cardholder.
            [<Config.Form>]Verification: Create'IndividualVerification option
        }
        with
            static member New(?dob: Create'IndividualDob, ?firstName: string, ?lastName: string, ?verification: Create'IndividualVerification) =
                {
                    Dob = dob
                    FirstName = firstName
                    LastName = lastName
                    Verification = verification
                }

        type Create'SpendingControlsAllowedCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Create'SpendingControlsBlockedCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Create'SpendingControlsSpendingLimitsCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Create'SpendingControlsSpendingLimitsInterval =
        | AllTime
        | Daily
        | Monthly
        | PerAuthorization
        | Weekly
        | Yearly

        type Create'SpendingControlsSpendingLimits = {
            ///Maximum amount allowed to spend per interval.
            [<Config.Form>]Amount: int option
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) this limit applies to. Omitting this field will apply the limit to all categories.
            [<Config.Form>]Categories: Create'SpendingControlsSpendingLimitsCategories list option
            ///Interval (or event) to which the amount applies.
            [<Config.Form>]Interval: Create'SpendingControlsSpendingLimitsInterval option
        }
        with
            static member New(?amount: int, ?categories: Create'SpendingControlsSpendingLimitsCategories list, ?interval: Create'SpendingControlsSpendingLimitsInterval) =
                {
                    Amount = amount
                    Categories = categories
                    Interval = interval
                }

        type Create'SpendingControls = {
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to allow. All other categories will be blocked. Cannot be set with `blocked_categories`.
            [<Config.Form>]AllowedCategories: Create'SpendingControlsAllowedCategories list option
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to decline. All other categories will be allowed. Cannot be set with `allowed_categories`.
            [<Config.Form>]BlockedCategories: Create'SpendingControlsBlockedCategories list option
            ///Limit spending with amount-based rules that apply across this cardholder's cards.
            [<Config.Form>]SpendingLimits: Create'SpendingControlsSpendingLimits list option
            ///Currency of amounts within `spending_limits`. Defaults to your merchant country's currency.
            [<Config.Form>]SpendingLimitsCurrency: string option
        }
        with
            static member New(?allowedCategories: Create'SpendingControlsAllowedCategories list, ?blockedCategories: Create'SpendingControlsBlockedCategories list, ?spendingLimits: Create'SpendingControlsSpendingLimits list, ?spendingLimitsCurrency: string) =
                {
                    AllowedCategories = allowedCategories
                    BlockedCategories = blockedCategories
                    SpendingLimits = spendingLimits
                    SpendingLimitsCurrency = spendingLimitsCurrency
                }

        type Create'Status =
        | Active
        | Inactive

        type Create'Type =
        | Company
        | Individual

        type CreateOptions = {
            ///The cardholder's billing address.
            [<Config.Form>]Billing: Create'Billing
            ///Additional information about a `company` cardholder.
            [<Config.Form>]Company: Create'Company option
            ///The cardholder's email address.
            [<Config.Form>]Email: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Additional information about an `individual` cardholder.
            [<Config.Form>]Individual: Create'Individual option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The cardholder's name. This will be printed on cards issued to them.
            [<Config.Form>]Name: string
            ///The cardholder's phone number. This will be transformed to [E.164](https://en.wikipedia.org/wiki/E.164) if it is not provided in that format already.
            [<Config.Form>]PhoneNumber: string option
            ///Rules that control spending across this cardholder's cards. Refer to our [documentation](https://stripe.com/docs/issuing/controls/spending-controls) for more details.
            [<Config.Form>]SpendingControls: Create'SpendingControls option
            ///Specifies whether to permit authorizations on this cardholder's cards. Defaults to `active`.
            [<Config.Form>]Status: Create'Status option
            ///One of `individual` or `company`.
            [<Config.Form>]Type: Create'Type
        }
        with
            static member New(billing: Create'Billing, name: string, type': Create'Type, ?company: Create'Company, ?email: string, ?expand: string list, ?individual: Create'Individual, ?metadata: Map<string, string>, ?phoneNumber: string, ?spendingControls: Create'SpendingControls, ?status: Create'Status) =
                {
                    Billing = billing
                    Company = company
                    Email = email
                    Expand = expand
                    Individual = individual
                    Metadata = metadata
                    Name = name
                    PhoneNumber = phoneNumber
                    SpendingControls = spendingControls
                    Status = status
                    Type = type'
                }

        ///<p>Creates a new Issuing <code>Cardholder</code> object that can be issued cards.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/issuing/cardholders"
            |> RestApi.postAsync<_, IssuingCardholder> settings (Map.empty) options

        type RetrieveOptions = {
            [<Config.Path>]Cardholder: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(cardholder: string, ?expand: string list) =
                {
                    Cardholder = cardholder
                    Expand = expand
                }

        ///<p>Retrieves an Issuing <code>Cardholder</code> object.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/issuing/cardholders/{options.Cardholder}"
            |> RestApi.getAsync<IssuingCardholder> settings qs

        type Update'BillingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'Billing = {
            ///The cardholder’s billing address.
            [<Config.Form>]Address: Update'BillingAddress option
        }
        with
            static member New(?address: Update'BillingAddress) =
                {
                    Address = address
                }

        type Update'Company = {
            ///The entity's business ID number.
            [<Config.Form>]TaxId: string option
        }
        with
            static member New(?taxId: string) =
                {
                    TaxId = taxId
                }

        type Update'IndividualDob = {
            ///The day of birth, between 1 and 31.
            [<Config.Form>]Day: int option
            ///The month of birth, between 1 and 12.
            [<Config.Form>]Month: int option
            ///The four-digit year of birth.
            [<Config.Form>]Year: int option
        }
        with
            static member New(?day: int, ?month: int, ?year: int) =
                {
                    Day = day
                    Month = month
                    Year = year
                }

        type Update'IndividualVerificationDocument = {
            ///The back of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`.
            [<Config.Form>]Back: string option
            ///The front of an ID returned by a [file upload](https://stripe.com/docs/api#create_file) with a `purpose` value of `identity_document`.
            [<Config.Form>]Front: string option
        }
        with
            static member New(?back: string, ?front: string) =
                {
                    Back = back
                    Front = front
                }

        type Update'IndividualVerification = {
            ///An identifying document, either a passport or local ID card.
            [<Config.Form>]Document: Update'IndividualVerificationDocument option
        }
        with
            static member New(?document: Update'IndividualVerificationDocument) =
                {
                    Document = document
                }

        type Update'Individual = {
            ///The date of birth of this cardholder.
            [<Config.Form>]Dob: Update'IndividualDob option
            ///The first name of this cardholder.
            [<Config.Form>]FirstName: string option
            ///The last name of this cardholder.
            [<Config.Form>]LastName: string option
            ///Government-issued ID document for this cardholder.
            [<Config.Form>]Verification: Update'IndividualVerification option
        }
        with
            static member New(?dob: Update'IndividualDob, ?firstName: string, ?lastName: string, ?verification: Update'IndividualVerification) =
                {
                    Dob = dob
                    FirstName = firstName
                    LastName = lastName
                    Verification = verification
                }

        type Update'SpendingControlsAllowedCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Update'SpendingControlsBlockedCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Update'SpendingControlsSpendingLimitsCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Update'SpendingControlsSpendingLimitsInterval =
        | AllTime
        | Daily
        | Monthly
        | PerAuthorization
        | Weekly
        | Yearly

        type Update'SpendingControlsSpendingLimits = {
            ///Maximum amount allowed to spend per interval.
            [<Config.Form>]Amount: int option
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) this limit applies to. Omitting this field will apply the limit to all categories.
            [<Config.Form>]Categories: Update'SpendingControlsSpendingLimitsCategories list option
            ///Interval (or event) to which the amount applies.
            [<Config.Form>]Interval: Update'SpendingControlsSpendingLimitsInterval option
        }
        with
            static member New(?amount: int, ?categories: Update'SpendingControlsSpendingLimitsCategories list, ?interval: Update'SpendingControlsSpendingLimitsInterval) =
                {
                    Amount = amount
                    Categories = categories
                    Interval = interval
                }

        type Update'SpendingControls = {
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to allow. All other categories will be blocked. Cannot be set with `blocked_categories`.
            [<Config.Form>]AllowedCategories: Update'SpendingControlsAllowedCategories list option
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to decline. All other categories will be allowed. Cannot be set with `allowed_categories`.
            [<Config.Form>]BlockedCategories: Update'SpendingControlsBlockedCategories list option
            ///Limit spending with amount-based rules that apply across this cardholder's cards.
            [<Config.Form>]SpendingLimits: Update'SpendingControlsSpendingLimits list option
            ///Currency of amounts within `spending_limits`. Defaults to your merchant country's currency.
            [<Config.Form>]SpendingLimitsCurrency: string option
        }
        with
            static member New(?allowedCategories: Update'SpendingControlsAllowedCategories list, ?blockedCategories: Update'SpendingControlsBlockedCategories list, ?spendingLimits: Update'SpendingControlsSpendingLimits list, ?spendingLimitsCurrency: string) =
                {
                    AllowedCategories = allowedCategories
                    BlockedCategories = blockedCategories
                    SpendingLimits = spendingLimits
                    SpendingLimitsCurrency = spendingLimitsCurrency
                }

        type Update'Status =
        | Active
        | Inactive

        type UpdateOptions = {
            [<Config.Path>]Cardholder: string
            ///The cardholder's billing address.
            [<Config.Form>]Billing: Update'Billing option
            ///Additional information about a `company` cardholder.
            [<Config.Form>]Company: Update'Company option
            ///The cardholder's email address.
            [<Config.Form>]Email: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Additional information about an `individual` cardholder.
            [<Config.Form>]Individual: Update'Individual option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The cardholder's phone number.
            [<Config.Form>]PhoneNumber: string option
            ///Rules that control spending across this cardholder's cards. Refer to our [documentation](https://stripe.com/docs/issuing/controls/spending-controls) for more details.
            [<Config.Form>]SpendingControls: Update'SpendingControls option
            ///Specifies whether to permit authorizations on this cardholder's cards.
            [<Config.Form>]Status: Update'Status option
        }
        with
            static member New(cardholder: string, ?billing: Update'Billing, ?company: Update'Company, ?email: string, ?expand: string list, ?individual: Update'Individual, ?metadata: Map<string, string>, ?phoneNumber: string, ?spendingControls: Update'SpendingControls, ?status: Update'Status) =
                {
                    Cardholder = cardholder
                    Billing = billing
                    Company = company
                    Email = email
                    Expand = expand
                    Individual = individual
                    Metadata = metadata
                    PhoneNumber = phoneNumber
                    SpendingControls = spendingControls
                    Status = status
                }

        ///<p>Updates the specified Issuing <code>Cardholder</code> object by setting the values of the parameters passed. Any parameters not provided will be left unchanged.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/issuing/cardholders/{options.Cardholder}"
            |> RestApi.postAsync<_, IssuingCardholder> settings (Map.empty) options

    module IssuingCards =

        type ListOptions = {
            ///Only return cards belonging to the Cardholder with the provided ID.
            [<Config.Query>]Cardholder: string option
            ///Only return cards that were issued during the given date interval.
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Only return cards that have the given expiration month.
            [<Config.Query>]ExpMonth: int option
            ///Only return cards that have the given expiration year.
            [<Config.Query>]ExpYear: int option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///Only return cards that have the given last four digits.
            [<Config.Query>]Last4: string option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return cards that have the given status. One of `active`, `inactive`, or `canceled`.
            [<Config.Query>]Status: string option
            ///Only return cards that have the given type. One of `virtual` or `physical`.
            [<Config.Query>]Type: string option
        }
        with
            static member New(?cardholder: string, ?created: int, ?endingBefore: string, ?expMonth: int, ?expYear: int, ?expand: string list, ?last4: string, ?limit: int, ?startingAfter: string, ?status: string, ?type': string) =
                {
                    Cardholder = cardholder
                    Created = created
                    EndingBefore = endingBefore
                    ExpMonth = expMonth
                    ExpYear = expYear
                    Expand = expand
                    Last4 = last4
                    Limit = limit
                    StartingAfter = startingAfter
                    Status = status
                    Type = type'
                }

        ///<p>Returns a list of Issuing <code>Card</code> objects. The objects are sorted in descending order by creation date, with the most recently created object appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("cardholder", options.Cardholder |> box); ("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("exp_month", options.ExpMonth |> box); ("exp_year", options.ExpYear |> box); ("expand", options.Expand |> box); ("last4", options.Last4 |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("status", options.Status |> box); ("type", options.Type |> box)] |> Map.ofList
            $"/v1/issuing/cards"
            |> RestApi.getAsync<IssuingCard list> settings qs

        type Create'ShippingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'ShippingService =
        | Express
        | Priority
        | Standard

        type Create'ShippingType =
        | Bulk
        | Individual

        type Create'Shipping = {
            ///The address that the card is shipped to.
            [<Config.Form>]Address: Create'ShippingAddress option
            ///The name printed on the shipping label when shipping the card.
            [<Config.Form>]Name: string option
            ///Shipment service.
            [<Config.Form>]Service: Create'ShippingService option
            ///Packaging options.
            [<Config.Form>]Type: Create'ShippingType option
        }
        with
            static member New(?address: Create'ShippingAddress, ?name: string, ?service: Create'ShippingService, ?type': Create'ShippingType) =
                {
                    Address = address
                    Name = name
                    Service = service
                    Type = type'
                }

        type Create'SpendingControlsAllowedCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Create'SpendingControlsBlockedCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Create'SpendingControlsSpendingLimitsCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Create'SpendingControlsSpendingLimitsInterval =
        | AllTime
        | Daily
        | Monthly
        | PerAuthorization
        | Weekly
        | Yearly

        type Create'SpendingControlsSpendingLimits = {
            ///Maximum amount allowed to spend per interval.
            [<Config.Form>]Amount: int option
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) this limit applies to. Omitting this field will apply the limit to all categories.
            [<Config.Form>]Categories: Create'SpendingControlsSpendingLimitsCategories list option
            ///Interval (or event) to which the amount applies.
            [<Config.Form>]Interval: Create'SpendingControlsSpendingLimitsInterval option
        }
        with
            static member New(?amount: int, ?categories: Create'SpendingControlsSpendingLimitsCategories list, ?interval: Create'SpendingControlsSpendingLimitsInterval) =
                {
                    Amount = amount
                    Categories = categories
                    Interval = interval
                }

        type Create'SpendingControls = {
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to allow. All other categories will be blocked. Cannot be set with `blocked_categories`.
            [<Config.Form>]AllowedCategories: Create'SpendingControlsAllowedCategories list option
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to decline. All other categories will be allowed. Cannot be set with `allowed_categories`.
            [<Config.Form>]BlockedCategories: Create'SpendingControlsBlockedCategories list option
            ///Limit spending with amount-based rules that apply across any cards this card replaced (i.e., its `replacement_for` card and _that_ card's `replacement_for` card, up the chain).
            [<Config.Form>]SpendingLimits: Create'SpendingControlsSpendingLimits list option
        }
        with
            static member New(?allowedCategories: Create'SpendingControlsAllowedCategories list, ?blockedCategories: Create'SpendingControlsBlockedCategories list, ?spendingLimits: Create'SpendingControlsSpendingLimits list) =
                {
                    AllowedCategories = allowedCategories
                    BlockedCategories = blockedCategories
                    SpendingLimits = spendingLimits
                }

        type Create'ReplacementReason =
        | Damaged
        | Expired
        | Lost
        | Stolen

        type Create'Status =
        | Active
        | Inactive

        type Create'Type =
        | Physical
        | Virtual

        type CreateOptions = {
            ///The [Cardholder](https://stripe.com/docs/api#issuing_cardholder_object) object with which the card will be associated.
            [<Config.Form>]Cardholder: string option
            ///The currency for the card.
            [<Config.Form>]Currency: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The card this is meant to be a replacement for (if any).
            [<Config.Form>]ReplacementFor: string option
            ///If `replacement_for` is specified, this should indicate why that card is being replaced.
            [<Config.Form>]ReplacementReason: Create'ReplacementReason option
            ///The address where the card will be shipped.
            [<Config.Form>]Shipping: Create'Shipping option
            ///Rules that control spending for this card. Refer to our [documentation](https://stripe.com/docs/issuing/controls/spending-controls) for more details.
            [<Config.Form>]SpendingControls: Create'SpendingControls option
            ///Whether authorizations can be approved on this card. Defaults to `inactive`.
            [<Config.Form>]Status: Create'Status option
            ///The type of card to issue. Possible values are `physical` or `virtual`.
            [<Config.Form>]Type: Create'Type
        }
        with
            static member New(currency: string, type': Create'Type, ?cardholder: string, ?expand: string list, ?metadata: Map<string, string>, ?replacementFor: string, ?replacementReason: Create'ReplacementReason, ?shipping: Create'Shipping, ?spendingControls: Create'SpendingControls, ?status: Create'Status) =
                {
                    Cardholder = cardholder
                    Currency = currency
                    Expand = expand
                    Metadata = metadata
                    ReplacementFor = replacementFor
                    ReplacementReason = replacementReason
                    Shipping = shipping
                    SpendingControls = spendingControls
                    Status = status
                    Type = type'
                }

        ///<p>Creates an Issuing <code>Card</code> object.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/issuing/cards"
            |> RestApi.postAsync<_, IssuingCard> settings (Map.empty) options

        type RetrieveOptions = {
            [<Config.Path>]Card: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(card: string, ?expand: string list) =
                {
                    Card = card
                    Expand = expand
                }

        ///<p>Retrieves an Issuing <code>Card</code> object.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/issuing/cards/{options.Card}"
            |> RestApi.getAsync<IssuingCard> settings qs

        type Update'SpendingControlsAllowedCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Update'SpendingControlsBlockedCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Update'SpendingControlsSpendingLimitsCategories =
        | AcRefrigerationRepair
        | AccountingBookkeepingServices
        | AdvertisingServices
        | AgriculturalCooperative
        | AirlinesAirCarriers
        | AirportsFlyingFields
        | AmbulanceServices
        | AmusementParksCarnivals
        | AntiqueReproductions
        | AntiqueShops
        | Aquariums
        | ArchitecturalSurveyingServices
        | ArtDealersAndGalleries
        | ArtistsSupplyAndCraftShops
        | AutoAndHomeSupplyStores
        | AutoBodyRepairShops
        | AutoPaintShops
        | AutoServiceShops
        | AutomatedCashDisburse
        | AutomatedFuelDispensers
        | AutomobileAssociations
        | AutomotivePartsAndAccessoriesStores
        | AutomotiveTireStores
        | BailAndBondPayments
        | Bakeries
        | BandsOrchestras
        | BarberAndBeautyShops
        | BettingCasinoGambling
        | BicycleShops
        | BilliardPoolEstablishments
        | BoatDealers
        | BoatRentalsAndLeases
        | BookStores
        | BooksPeriodicalsAndNewspapers
        | BowlingAlleys
        | BusLines
        | BusinessSecretarialSchools
        | BuyingShoppingServices
        | CableSatelliteAndOtherPayTelevisionAndRadio
        | CameraAndPhotographicSupplyStores
        | CandyNutAndConfectioneryStores
        | CarAndTruckDealersNewUsed
        | CarAndTruckDealersUsedOnly
        | CarRentalAgencies
        | CarWashes
        | CarpentryServices
        | CarpetUpholsteryCleaning
        | Caterers
        | CharitableAndSocialServiceOrganizationsFundraising
        | ChemicalsAndAlliedProducts
        | ChildCareServices
        | ChildrensAndInfantsWearStores
        | ChiropodistsPodiatrists
        | Chiropractors
        | CigarStoresAndStands
        | CivicSocialFraternalAssociations
        | CleaningAndMaintenance
        | ClothingRental
        | CollegesUniversities
        | CommercialEquipment
        | CommercialFootwear
        | CommercialPhotographyArtAndGraphics
        | CommuterTransportAndFerries
        | ComputerNetworkServices
        | ComputerProgramming
        | ComputerRepair
        | ComputerSoftwareStores
        | ComputersPeripheralsAndSoftware
        | ConcreteWorkServices
        | ConstructionMaterials
        | ConsultingPublicRelations
        | CorrespondenceSchools
        | CosmeticStores
        | CounselingServices
        | CountryClubs
        | CourierServices
        | CourtCosts
        | CreditReportingAgencies
        | CruiseLines
        | DairyProductsStores
        | DanceHallStudiosSchools
        | DatingEscortServices
        | DentistsOrthodontists
        | DepartmentStores
        | DetectiveAgencies
        | DigitalGoodsApplications
        | DigitalGoodsGames
        | DigitalGoodsLargeVolume
        | DigitalGoodsMedia
        | DirectMarketingCatalogMerchant
        | DirectMarketingCombinationCatalogAndRetailMerchant
        | DirectMarketingInboundTelemarketing
        | DirectMarketingInsuranceServices
        | DirectMarketingOther
        | DirectMarketingOutboundTelemarketing
        | DirectMarketingSubscription
        | DirectMarketingTravel
        | DiscountStores
        | Doctors
        | DoorToDoorSales
        | DraperyWindowCoveringAndUpholsteryStores
        | DrinkingPlaces
        | DrugStoresAndPharmacies
        | DrugsDrugProprietariesAndDruggistSundries
        | DryCleaners
        | DurableGoods
        | DutyFreeStores
        | EatingPlacesRestaurants
        | EducationalServices
        | ElectricRazorStores
        | ElectricalPartsAndEquipment
        | ElectricalServices
        | ElectronicsRepairShops
        | ElectronicsStores
        | ElementarySecondarySchools
        | EmploymentTempAgencies
        | EquipmentRental
        | ExterminatingServices
        | FamilyClothingStores
        | FastFoodRestaurants
        | FinancialInstitutions
        | FinesGovernmentAdministrativeEntities
        | FireplaceFireplaceScreensAndAccessoriesStores
        | FloorCoveringStores
        | Florists
        | FloristsSuppliesNurseryStockAndFlowers
        | FreezerAndLockerMeatProvisioners
        | FuelDealersNonAutomotive
        | FuneralServicesCrematories
        | FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances
        | FurnitureRepairRefinishing
        | FurriersAndFurShops
        | GeneralServices
        | GiftCardNoveltyAndSouvenirShops
        | GlassPaintAndWallpaperStores
        | GlasswareCrystalStores
        | GolfCoursesPublic
        | GovernmentServices
        | GroceryStoresSupermarkets
        | HardwareEquipmentAndSupplies
        | HardwareStores
        | HealthAndBeautySpas
        | HearingAidsSalesAndSupplies
        | HeatingPlumbingAC
        | HobbyToyAndGameShops
        | HomeSupplyWarehouseStores
        | Hospitals
        | HotelsMotelsAndResorts
        | HouseholdApplianceStores
        | IndustrialSupplies
        | InformationRetrievalServices
        | InsuranceDefault
        | InsuranceUnderwritingPremiums
        | IntraCompanyPurchases
        | JewelryStoresWatchesClocksAndSilverwareStores
        | LandscapingServices
        | Laundries
        | LaundryCleaningServices
        | LegalServicesAttorneys
        | LuggageAndLeatherGoodsStores
        | LumberBuildingMaterialsStores
        | ManualCashDisburse
        | MarinasServiceAndSupplies
        | MasonryStoneworkAndPlaster
        | MassageParlors
        | MedicalAndDentalLabs
        | MedicalDentalOphthalmicAndHospitalEquipmentAndSupplies
        | MedicalServices
        | MembershipOrganizations
        | MensAndBoysClothingAndAccessoriesStores
        | MensWomensClothingStores
        | MetalServiceCenters
        | Miscellaneous
        | MiscellaneousApparelAndAccessoryShops
        | MiscellaneousAutoDealers
        | MiscellaneousBusinessServices
        | MiscellaneousFoodStores
        | MiscellaneousGeneralMerchandise
        | MiscellaneousGeneralServices
        | MiscellaneousHomeFurnishingSpecialtyStores
        | MiscellaneousPublishingAndPrinting
        | MiscellaneousRecreationServices
        | MiscellaneousRepairShops
        | MiscellaneousSpecialtyRetail
        | MobileHomeDealers
        | MotionPictureTheaters
        | MotorFreightCarriersAndTrucking
        | MotorHomesDealers
        | MotorVehicleSuppliesAndNewParts
        | MotorcycleShopsAndDealers
        | MotorcycleShopsDealers
        | MusicStoresMusicalInstrumentsPianosAndSheetMusic
        | NewsDealersAndNewsstands
        | NonFiMoneyOrders
        | NonFiStoredValueCardPurchaseLoad
        | NondurableGoods
        | NurseriesLawnAndGardenSupplyStores
        | NursingPersonalCare
        | OfficeAndCommercialFurniture
        | OpticiansEyeglasses
        | OptometristsOphthalmologist
        | OrthopedicGoodsProstheticDevices
        | Osteopaths
        | PackageStoresBeerWineAndLiquor
        | PaintsVarnishesAndSupplies
        | ParkingLotsGarages
        | PassengerRailways
        | PawnShops
        | PetShopsPetFoodAndSupplies
        | PetroleumAndPetroleumProducts
        | PhotoDeveloping
        | PhotographicPhotocopyMicrofilmEquipmentAndSupplies
        | PhotographicStudios
        | PictureVideoProduction
        | PieceGoodsNotionsAndOtherDryGoods
        | PlumbingHeatingEquipmentAndSupplies
        | PoliticalOrganizations
        | PostalServicesGovernmentOnly
        | PreciousStonesAndMetalsWatchesAndJewelry
        | ProfessionalServices
        | PublicWarehousingAndStorage
        | QuickCopyReproAndBlueprint
        | Railroads
        | RealEstateAgentsAndManagersRentals
        | RecordStores
        | RecreationalVehicleRentals
        | ReligiousGoodsStores
        | ReligiousOrganizations
        | RoofingSidingSheetMetal
        | SecretarialSupportServices
        | SecurityBrokersDealers
        | ServiceStations
        | SewingNeedleworkFabricAndPieceGoodsStores
        | ShoeRepairHatCleaning
        | ShoeStores
        | SmallApplianceRepair
        | SnowmobileDealers
        | SpecialTradeServices
        | SpecialtyCleaning
        | SportingGoodsStores
        | SportingRecreationCamps
        | SportsAndRidingApparelStores
        | SportsClubsFields
        | StampAndCoinStores
        | StationaryOfficeSuppliesPrintingAndWritingPaper
        | StationeryStoresOfficeAndSchoolSupplyStores
        | SwimmingPoolsSales
        | TUiTravelGermany
        | TailorsAlterations
        | TaxPaymentsGovernmentAgencies
        | TaxPreparationServices
        | TaxicabsLimousines
        | TelecommunicationEquipmentAndTelephoneSales
        | TelecommunicationServices
        | TelegraphServices
        | TentAndAwningShops
        | TestingLaboratories
        | TheatricalTicketAgencies
        | Timeshares
        | TireRetreadingAndRepair
        | TollsBridgeFees
        | TouristAttractionsAndExhibits
        | TowingServices
        | TrailerParksCampgrounds
        | TransportationServices
        | TravelAgenciesTourOperators
        | TruckStopIteration
        | TruckUtilityTrailerRentals
        | TypesettingPlateMakingAndRelatedServices
        | TypewriterStores
        | USFederalGovernmentAgenciesOrDepartments
        | UniformsCommercialClothing
        | UsedMerchandiseAndSecondhandStores
        | Utilities
        | VarietyStores
        | VeterinaryServices
        | VideoAmusementGameSupplies
        | VideoGameArcades
        | VideoTapeRentalStores
        | VocationalTradeSchools
        | WatchJewelryRepair
        | WeldingRepair
        | WholesaleClubs
        | WigAndToupeeStores
        | WiresMoneyOrders
        | WomensAccessoryAndSpecialtyShops
        | WomensReadyToWearStores
        | WreckingAndSalvageYards

        type Update'SpendingControlsSpendingLimitsInterval =
        | AllTime
        | Daily
        | Monthly
        | PerAuthorization
        | Weekly
        | Yearly

        type Update'SpendingControlsSpendingLimits = {
            ///Maximum amount allowed to spend per interval.
            [<Config.Form>]Amount: int option
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) this limit applies to. Omitting this field will apply the limit to all categories.
            [<Config.Form>]Categories: Update'SpendingControlsSpendingLimitsCategories list option
            ///Interval (or event) to which the amount applies.
            [<Config.Form>]Interval: Update'SpendingControlsSpendingLimitsInterval option
        }
        with
            static member New(?amount: int, ?categories: Update'SpendingControlsSpendingLimitsCategories list, ?interval: Update'SpendingControlsSpendingLimitsInterval) =
                {
                    Amount = amount
                    Categories = categories
                    Interval = interval
                }

        type Update'SpendingControls = {
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to allow. All other categories will be blocked. Cannot be set with `blocked_categories`.
            [<Config.Form>]AllowedCategories: Update'SpendingControlsAllowedCategories list option
            ///Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to decline. All other categories will be allowed. Cannot be set with `allowed_categories`.
            [<Config.Form>]BlockedCategories: Update'SpendingControlsBlockedCategories list option
            ///Limit spending with amount-based rules that apply across any cards this card replaced (i.e., its `replacement_for` card and _that_ card's `replacement_for` card, up the chain).
            [<Config.Form>]SpendingLimits: Update'SpendingControlsSpendingLimits list option
        }
        with
            static member New(?allowedCategories: Update'SpendingControlsAllowedCategories list, ?blockedCategories: Update'SpendingControlsBlockedCategories list, ?spendingLimits: Update'SpendingControlsSpendingLimits list) =
                {
                    AllowedCategories = allowedCategories
                    BlockedCategories = blockedCategories
                    SpendingLimits = spendingLimits
                }

        type Update'CancellationReason =
        | Lost
        | Stolen

        type Update'Status =
        | Active
        | Canceled
        | Inactive

        type UpdateOptions = {
            [<Config.Path>]Card: string
            ///Reason why the `status` of this card is `canceled`.
            [<Config.Form>]CancellationReason: Update'CancellationReason option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Rules that control spending for this card. Refer to our [documentation](https://stripe.com/docs/issuing/controls/spending-controls) for more details.
            [<Config.Form>]SpendingControls: Update'SpendingControls option
            ///Dictates whether authorizations can be approved on this card. If this card is being canceled because it was lost or stolen, this information should be provided as `cancellation_reason`.
            [<Config.Form>]Status: Update'Status option
        }
        with
            static member New(card: string, ?cancellationReason: Update'CancellationReason, ?expand: string list, ?metadata: Map<string, string>, ?spendingControls: Update'SpendingControls, ?status: Update'Status) =
                {
                    Card = card
                    CancellationReason = cancellationReason
                    Expand = expand
                    Metadata = metadata
                    SpendingControls = spendingControls
                    Status = status
                }

        ///<p>Updates the specified Issuing <code>Card</code> object by setting the values of the parameters passed. Any parameters not provided will be left unchanged.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/issuing/cards/{options.Card}"
            |> RestApi.postAsync<_, IssuingCard> settings (Map.empty) options

    module IssuingDisputes =

        type ListOptions = {
            ///Select Issuing disputes that were created during the given date interval.
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Select Issuing disputes with the given status.
            [<Config.Query>]Status: string option
            ///Select the Issuing dispute for the given transaction.
            [<Config.Query>]Transaction: string option
        }
        with
            static member New(?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string, ?status: string, ?transaction: string) =
                {
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                    Status = status
                    Transaction = transaction
                }

        ///<p>Returns a list of Issuing <code>Dispute</code> objects. The objects are sorted in descending order by creation date, with the most recently created object appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("status", options.Status |> box); ("transaction", options.Transaction |> box)] |> Map.ofList
            $"/v1/issuing/disputes"
            |> RestApi.getAsync<IssuingDispute list> settings qs

        type Create'EvidenceCanceledCanceledProductType =
        | Merchandise
        | Service

        type Create'EvidenceCanceledCanceledReturnStatus =
        | MerchantRejected
        | Successful

        type Create'EvidenceCanceledCanceled = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Date when order was canceled.
            [<Config.Form>]CanceledAt: Choice<DateTime,string> option
            ///Whether the cardholder was provided with a cancellation policy.
            [<Config.Form>]CancellationPolicyProvided: Choice<bool,string> option
            ///Reason for canceling the order.
            [<Config.Form>]CancellationReason: string option
            ///Date when the cardholder expected to receive the product.
            [<Config.Form>]ExpectedAt: Choice<DateTime,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Description of the merchandise or service that was purchased.
            [<Config.Form>]ProductDescription: string option
            ///Whether the product was a merchandise or service.
            [<Config.Form>]ProductType: Create'EvidenceCanceledCanceledProductType option
            ///Result of cardholder's attempt to return the product.
            [<Config.Form>]ReturnStatus: Create'EvidenceCanceledCanceledReturnStatus option
            ///Date when the product was returned or attempted to be returned.
            [<Config.Form>]ReturnedAt: Choice<DateTime,string> option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?canceledAt: Choice<DateTime,string>, ?cancellationPolicyProvided: Choice<bool,string>, ?cancellationReason: string, ?expectedAt: Choice<DateTime,string>, ?explanation: string, ?productDescription: string, ?productType: Create'EvidenceCanceledCanceledProductType, ?returnStatus: Create'EvidenceCanceledCanceledReturnStatus, ?returnedAt: Choice<DateTime,string>) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    CanceledAt = canceledAt
                    CancellationPolicyProvided = cancellationPolicyProvided
                    CancellationReason = cancellationReason
                    ExpectedAt = expectedAt
                    Explanation = explanation
                    ProductDescription = productDescription
                    ProductType = productType
                    ReturnStatus = returnStatus
                    ReturnedAt = returnedAt
                }

        type Create'EvidenceDuplicateDuplicate = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Copy of the card statement showing that the product had already been paid for.
            [<Config.Form>]CardStatement: Choice<string,string> option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Copy of the receipt showing that the product had been paid for in cash.
            [<Config.Form>]CashReceipt: Choice<string,string> option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Image of the front and back of the check that was used to pay for the product.
            [<Config.Form>]CheckImage: Choice<string,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Transaction (e.g., ipi_...) that the disputed transaction is a duplicate of. Of the two or more transactions that are copies of each other, this is original undisputed one.
            [<Config.Form>]OriginalTransaction: string option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?cardStatement: Choice<string,string>, ?cashReceipt: Choice<string,string>, ?checkImage: Choice<string,string>, ?explanation: string, ?originalTransaction: string) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    CardStatement = cardStatement
                    CashReceipt = cashReceipt
                    CheckImage = checkImage
                    Explanation = explanation
                    OriginalTransaction = originalTransaction
                }

        type Create'EvidenceFraudulentFraudulent = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?explanation: string) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    Explanation = explanation
                }

        type Create'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribedReturnStatus =
        | MerchantRejected
        | Successful

        type Create'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribed = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Date when the product was received.
            [<Config.Form>]ReceivedAt: Choice<DateTime,string> option
            ///Description of the cardholder's attempt to return the product.
            [<Config.Form>]ReturnDescription: string option
            ///Result of cardholder's attempt to return the product.
            [<Config.Form>]ReturnStatus: Create'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribedReturnStatus option
            ///Date when the product was returned or attempted to be returned.
            [<Config.Form>]ReturnedAt: Choice<DateTime,string> option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?explanation: string, ?receivedAt: Choice<DateTime,string>, ?returnDescription: string, ?returnStatus: Create'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribedReturnStatus, ?returnedAt: Choice<DateTime,string>) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    Explanation = explanation
                    ReceivedAt = receivedAt
                    ReturnDescription = returnDescription
                    ReturnStatus = returnStatus
                    ReturnedAt = returnedAt
                }

        type Create'EvidenceNotReceivedNotReceivedProductType =
        | Merchandise
        | Service

        type Create'EvidenceNotReceivedNotReceived = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Date when the cardholder expected to receive the product.
            [<Config.Form>]ExpectedAt: Choice<DateTime,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Description of the merchandise or service that was purchased.
            [<Config.Form>]ProductDescription: string option
            ///Whether the product was a merchandise or service.
            [<Config.Form>]ProductType: Create'EvidenceNotReceivedNotReceivedProductType option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?expectedAt: Choice<DateTime,string>, ?explanation: string, ?productDescription: string, ?productType: Create'EvidenceNotReceivedNotReceivedProductType) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    ExpectedAt = expectedAt
                    Explanation = explanation
                    ProductDescription = productDescription
                    ProductType = productType
                }

        type Create'EvidenceOtherOtherProductType =
        | Merchandise
        | Service

        type Create'EvidenceOtherOther = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Description of the merchandise or service that was purchased.
            [<Config.Form>]ProductDescription: string option
            ///Whether the product was a merchandise or service.
            [<Config.Form>]ProductType: Create'EvidenceOtherOtherProductType option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?explanation: string, ?productDescription: string, ?productType: Create'EvidenceOtherOtherProductType) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    Explanation = explanation
                    ProductDescription = productDescription
                    ProductType = productType
                }

        type Create'EvidenceServiceNotAsDescribedServiceNotAsDescribed = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Date when order was canceled.
            [<Config.Form>]CanceledAt: Choice<DateTime,string> option
            ///Reason for canceling the order.
            [<Config.Form>]CancellationReason: string option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Date when the product was received.
            [<Config.Form>]ReceivedAt: Choice<DateTime,string> option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?canceledAt: Choice<DateTime,string>, ?cancellationReason: string, ?explanation: string, ?receivedAt: Choice<DateTime,string>) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    CanceledAt = canceledAt
                    CancellationReason = cancellationReason
                    Explanation = explanation
                    ReceivedAt = receivedAt
                }

        type Create'EvidenceReason =
        | Canceled
        | Duplicate
        | Fraudulent
        | MerchandiseNotAsDescribed
        | NotReceived
        | Other
        | ServiceNotAsDescribed

        type Create'Evidence = {
            ///Evidence provided when `reason` is 'canceled'.
            [<Config.Form>]Canceled: Choice<Create'EvidenceCanceledCanceled,string> option
            ///Evidence provided when `reason` is 'duplicate'.
            [<Config.Form>]Duplicate: Choice<Create'EvidenceDuplicateDuplicate,string> option
            ///Evidence provided when `reason` is 'fraudulent'.
            [<Config.Form>]Fraudulent: Choice<Create'EvidenceFraudulentFraudulent,string> option
            ///Evidence provided when `reason` is 'merchandise_not_as_described'.
            [<Config.Form>]MerchandiseNotAsDescribed: Choice<Create'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribed,string> option
            ///Evidence provided when `reason` is 'not_received'.
            [<Config.Form>]NotReceived: Choice<Create'EvidenceNotReceivedNotReceived,string> option
            ///Evidence provided when `reason` is 'other'.
            [<Config.Form>]Other: Choice<Create'EvidenceOtherOther,string> option
            ///The reason for filing the dispute. The evidence should be submitted in the field of the same name.
            [<Config.Form>]Reason: Create'EvidenceReason option
            ///Evidence provided when `reason` is 'service_not_as_described'.
            [<Config.Form>]ServiceNotAsDescribed: Choice<Create'EvidenceServiceNotAsDescribedServiceNotAsDescribed,string> option
        }
        with
            static member New(?canceled: Choice<Create'EvidenceCanceledCanceled,string>, ?duplicate: Choice<Create'EvidenceDuplicateDuplicate,string>, ?fraudulent: Choice<Create'EvidenceFraudulentFraudulent,string>, ?merchandiseNotAsDescribed: Choice<Create'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribed,string>, ?notReceived: Choice<Create'EvidenceNotReceivedNotReceived,string>, ?other: Choice<Create'EvidenceOtherOther,string>, ?reason: Create'EvidenceReason, ?serviceNotAsDescribed: Choice<Create'EvidenceServiceNotAsDescribedServiceNotAsDescribed,string>) =
                {
                    Canceled = canceled
                    Duplicate = duplicate
                    Fraudulent = fraudulent
                    MerchandiseNotAsDescribed = merchandiseNotAsDescribed
                    NotReceived = notReceived
                    Other = other
                    Reason = reason
                    ServiceNotAsDescribed = serviceNotAsDescribed
                }

        type CreateOptions = {
            ///Evidence provided for the dispute.
            [<Config.Form>]Evidence: Create'Evidence option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The ID of the issuing transaction to create a dispute for.
            [<Config.Form>]Transaction: string
        }
        with
            static member New(transaction: string, ?evidence: Create'Evidence, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Evidence = evidence
                    Expand = expand
                    Metadata = metadata
                    Transaction = transaction
                }

        ///<p>Creates an Issuing <code>Dispute</code> object. Individual pieces of evidence within the <code>evidence</code> object are optional at this point. Stripe only validates that required evidence is present during submission. Refer to <a href="/docs/issuing/purchases/disputes#dispute-reasons-and-evidence">Dispute reasons and evidence</a> for more details about evidence requirements.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/issuing/disputes"
            |> RestApi.postAsync<_, IssuingDispute> settings (Map.empty) options

        type RetrieveOptions = {
            [<Config.Path>]Dispute: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
        }
        with
            static member New(dispute: string, ?expand: string list) =
                {
                    Dispute = dispute
                    Expand = expand
                }

        ///<p>Retrieves an Issuing <code>Dispute</code> object.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/issuing/disputes/{options.Dispute}"
            |> RestApi.getAsync<IssuingDispute> settings qs

        type Update'EvidenceCanceledCanceledProductType =
        | Merchandise
        | Service

        type Update'EvidenceCanceledCanceledReturnStatus =
        | MerchantRejected
        | Successful

        type Update'EvidenceCanceledCanceled = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Date when order was canceled.
            [<Config.Form>]CanceledAt: Choice<DateTime,string> option
            ///Whether the cardholder was provided with a cancellation policy.
            [<Config.Form>]CancellationPolicyProvided: Choice<bool,string> option
            ///Reason for canceling the order.
            [<Config.Form>]CancellationReason: string option
            ///Date when the cardholder expected to receive the product.
            [<Config.Form>]ExpectedAt: Choice<DateTime,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Description of the merchandise or service that was purchased.
            [<Config.Form>]ProductDescription: string option
            ///Whether the product was a merchandise or service.
            [<Config.Form>]ProductType: Update'EvidenceCanceledCanceledProductType option
            ///Result of cardholder's attempt to return the product.
            [<Config.Form>]ReturnStatus: Update'EvidenceCanceledCanceledReturnStatus option
            ///Date when the product was returned or attempted to be returned.
            [<Config.Form>]ReturnedAt: Choice<DateTime,string> option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?canceledAt: Choice<DateTime,string>, ?cancellationPolicyProvided: Choice<bool,string>, ?cancellationReason: string, ?expectedAt: Choice<DateTime,string>, ?explanation: string, ?productDescription: string, ?productType: Update'EvidenceCanceledCanceledProductType, ?returnStatus: Update'EvidenceCanceledCanceledReturnStatus, ?returnedAt: Choice<DateTime,string>) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    CanceledAt = canceledAt
                    CancellationPolicyProvided = cancellationPolicyProvided
                    CancellationReason = cancellationReason
                    ExpectedAt = expectedAt
                    Explanation = explanation
                    ProductDescription = productDescription
                    ProductType = productType
                    ReturnStatus = returnStatus
                    ReturnedAt = returnedAt
                }

        type Update'EvidenceDuplicateDuplicate = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Copy of the card statement showing that the product had already been paid for.
            [<Config.Form>]CardStatement: Choice<string,string> option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Copy of the receipt showing that the product had been paid for in cash.
            [<Config.Form>]CashReceipt: Choice<string,string> option
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Image of the front and back of the check that was used to pay for the product.
            [<Config.Form>]CheckImage: Choice<string,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Transaction (e.g., ipi_...) that the disputed transaction is a duplicate of. Of the two or more transactions that are copies of each other, this is original undisputed one.
            [<Config.Form>]OriginalTransaction: string option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?cardStatement: Choice<string,string>, ?cashReceipt: Choice<string,string>, ?checkImage: Choice<string,string>, ?explanation: string, ?originalTransaction: string) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    CardStatement = cardStatement
                    CashReceipt = cashReceipt
                    CheckImage = checkImage
                    Explanation = explanation
                    OriginalTransaction = originalTransaction
                }

        type Update'EvidenceFraudulentFraudulent = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?explanation: string) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    Explanation = explanation
                }

        type Update'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribedReturnStatus =
        | MerchantRejected
        | Successful

        type Update'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribed = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Date when the product was received.
            [<Config.Form>]ReceivedAt: Choice<DateTime,string> option
            ///Description of the cardholder's attempt to return the product.
            [<Config.Form>]ReturnDescription: string option
            ///Result of cardholder's attempt to return the product.
            [<Config.Form>]ReturnStatus: Update'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribedReturnStatus option
            ///Date when the product was returned or attempted to be returned.
            [<Config.Form>]ReturnedAt: Choice<DateTime,string> option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?explanation: string, ?receivedAt: Choice<DateTime,string>, ?returnDescription: string, ?returnStatus: Update'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribedReturnStatus, ?returnedAt: Choice<DateTime,string>) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    Explanation = explanation
                    ReceivedAt = receivedAt
                    ReturnDescription = returnDescription
                    ReturnStatus = returnStatus
                    ReturnedAt = returnedAt
                }

        type Update'EvidenceNotReceivedNotReceivedProductType =
        | Merchandise
        | Service

        type Update'EvidenceNotReceivedNotReceived = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Date when the cardholder expected to receive the product.
            [<Config.Form>]ExpectedAt: Choice<DateTime,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Description of the merchandise or service that was purchased.
            [<Config.Form>]ProductDescription: string option
            ///Whether the product was a merchandise or service.
            [<Config.Form>]ProductType: Update'EvidenceNotReceivedNotReceivedProductType option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?expectedAt: Choice<DateTime,string>, ?explanation: string, ?productDescription: string, ?productType: Update'EvidenceNotReceivedNotReceivedProductType) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    ExpectedAt = expectedAt
                    Explanation = explanation
                    ProductDescription = productDescription
                    ProductType = productType
                }

        type Update'EvidenceOtherOtherProductType =
        | Merchandise
        | Service

        type Update'EvidenceOtherOther = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Description of the merchandise or service that was purchased.
            [<Config.Form>]ProductDescription: string option
            ///Whether the product was a merchandise or service.
            [<Config.Form>]ProductType: Update'EvidenceOtherOtherProductType option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?explanation: string, ?productDescription: string, ?productType: Update'EvidenceOtherOtherProductType) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    Explanation = explanation
                    ProductDescription = productDescription
                    ProductType = productType
                }

        type Update'EvidenceServiceNotAsDescribedServiceNotAsDescribed = {
            ///(ID of a [file upload](https://stripe.com/docs/guides/file-upload)) Additional documentation supporting the dispute.
            [<Config.Form>]AdditionalDocumentation: Choice<string,string> option
            ///Date when order was canceled.
            [<Config.Form>]CanceledAt: Choice<DateTime,string> option
            ///Reason for canceling the order.
            [<Config.Form>]CancellationReason: string option
            ///Explanation of why the cardholder is disputing this transaction.
            [<Config.Form>]Explanation: string option
            ///Date when the product was received.
            [<Config.Form>]ReceivedAt: Choice<DateTime,string> option
        }
        with
            static member New(?additionalDocumentation: Choice<string,string>, ?canceledAt: Choice<DateTime,string>, ?cancellationReason: string, ?explanation: string, ?receivedAt: Choice<DateTime,string>) =
                {
                    AdditionalDocumentation = additionalDocumentation
                    CanceledAt = canceledAt
                    CancellationReason = cancellationReason
                    Explanation = explanation
                    ReceivedAt = receivedAt
                }

        type Update'EvidenceReason =
        | Canceled
        | Duplicate
        | Fraudulent
        | MerchandiseNotAsDescribed
        | NotReceived
        | Other
        | ServiceNotAsDescribed

        type Update'Evidence = {
            ///Evidence provided when `reason` is 'canceled'.
            [<Config.Form>]Canceled: Choice<Update'EvidenceCanceledCanceled,string> option
            ///Evidence provided when `reason` is 'duplicate'.
            [<Config.Form>]Duplicate: Choice<Update'EvidenceDuplicateDuplicate,string> option
            ///Evidence provided when `reason` is 'fraudulent'.
            [<Config.Form>]Fraudulent: Choice<Update'EvidenceFraudulentFraudulent,string> option
            ///Evidence provided when `reason` is 'merchandise_not_as_described'.
            [<Config.Form>]MerchandiseNotAsDescribed: Choice<Update'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribed,string> option
            ///Evidence provided when `reason` is 'not_received'.
            [<Config.Form>]NotReceived: Choice<Update'EvidenceNotReceivedNotReceived,string> option
            ///Evidence provided when `reason` is 'other'.
            [<Config.Form>]Other: Choice<Update'EvidenceOtherOther,string> option
            ///The reason for filing the dispute. The evidence should be submitted in the field of the same name.
            [<Config.Form>]Reason: Update'EvidenceReason option
            ///Evidence provided when `reason` is 'service_not_as_described'.
            [<Config.Form>]ServiceNotAsDescribed: Choice<Update'EvidenceServiceNotAsDescribedServiceNotAsDescribed,string> option
        }
        with
            static member New(?canceled: Choice<Update'EvidenceCanceledCanceled,string>, ?duplicate: Choice<Update'EvidenceDuplicateDuplicate,string>, ?fraudulent: Choice<Update'EvidenceFraudulentFraudulent,string>, ?merchandiseNotAsDescribed: Choice<Update'EvidenceMerchandiseNotAsDescribedMerchandiseNotAsDescribed,string>, ?notReceived: Choice<Update'EvidenceNotReceivedNotReceived,string>, ?other: Choice<Update'EvidenceOtherOther,string>, ?reason: Update'EvidenceReason, ?serviceNotAsDescribed: Choice<Update'EvidenceServiceNotAsDescribedServiceNotAsDescribed,string>) =
                {
                    Canceled = canceled
                    Duplicate = duplicate
                    Fraudulent = fraudulent
                    MerchandiseNotAsDescribed = merchandiseNotAsDescribed
                    NotReceived = notReceived
                    Other = other
                    Reason = reason
                    ServiceNotAsDescribed = serviceNotAsDescribed
                }

        type UpdateOptions = {
            [<Config.Path>]Dispute: string
            ///Evidence provided for the dispute.
            [<Config.Form>]Evidence: Update'Evidence option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(dispute: string, ?evidence: Update'Evidence, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Dispute = dispute
                    Evidence = evidence
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Updates the specified Issuing <code>Dispute</code> object by setting the values of the parameters passed. Any parameters not provided will be left unchanged. Properties on the <code>evidence</code> object can be unset by passing in an empty string.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/issuing/disputes/{options.Dispute}"
            |> RestApi.postAsync<_, IssuingDispute> settings (Map.empty) options

    module IssuingDisputesSubmit =

        type SubmitOptions = {
            [<Config.Path>]Dispute: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(dispute: string, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Dispute = dispute
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Submits an Issuing <code>Dispute</code> to the card network. Stripe validates that all evidence fields required for the dispute’s reason are present. For more details, see <a href="/docs/issuing/purchases/disputes#dispute-reasons-and-evidence">Dispute reasons and evidence</a>.</p>
        let Submit settings (options: SubmitOptions) =
            $"/v1/issuing/disputes/{options.Dispute}/submit"
            |> RestApi.postAsync<_, IssuingDispute> settings (Map.empty) options

    module IssuingTransactions =

        type ListOptions = {
            ///Only return transactions that belong to the given card.
            [<Config.Query>]Card: string option
            ///Only return transactions that belong to the given cardholder.
            [<Config.Query>]Cardholder: string option
            ///Only return transactions that were created during the given date interval.
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return transactions that have the given type. One of `capture` or `refund`.
            [<Config.Query>]Type: string option
        }
        with
            static member New(?card: string, ?cardholder: string, ?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string, ?type': string) =
                {
                    Card = card
                    Cardholder = cardholder
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                    Type = type'
                }

        ///<p>Returns a list of Issuing <code>Transaction</code> objects. The objects are sorted in descending order by creation date, with the most recently created object appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("card", options.Card |> box); ("cardholder", options.Cardholder |> box); ("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("type", options.Type |> box)] |> Map.ofList
            $"/v1/issuing/transactions"
            |> RestApi.getAsync<IssuingTransaction list> settings qs

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Transaction: string
        }
        with
            static member New(transaction: string, ?expand: string list) =
                {
                    Expand = expand
                    Transaction = transaction
                }

        ///<p>Retrieves an Issuing <code>Transaction</code> object.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/issuing/transactions/{options.Transaction}"
            |> RestApi.getAsync<IssuingTransaction> settings qs

        type UpdateOptions = {
            [<Config.Path>]Transaction: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(transaction: string, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Transaction = transaction
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Updates the specified Issuing <code>Transaction</code> object by setting the values of the parameters passed. Any parameters not provided will be left unchanged.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/issuing/transactions/{options.Transaction}"
            |> RestApi.postAsync<_, IssuingTransaction> settings (Map.empty) options

    module Mandates =

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Mandate: string
        }
        with
            static member New(mandate: string, ?expand: string list) =
                {
                    Expand = expand
                    Mandate = mandate
                }

        ///<p>Retrieves a Mandate object.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/mandates/{options.Mandate}"
            |> RestApi.getAsync<Mandate> settings qs

    module OrderReturns =

        type ListOptions = {
            ///Date this return was created.
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///The order to retrieve returns for.
            [<Config.Query>]Order: string option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?order: string, ?startingAfter: string) =
                {
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    Order = order
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of your order returns. The returns are returned sorted by creation date, with the most recently created return appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("order", options.Order |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/order_returns"
            |> RestApi.getAsync<OrderReturn list> settings qs

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
        }
        with
            static member New(id: string, ?expand: string list) =
                {
                    Expand = expand
                    Id = id
                }

        ///<p>Retrieves the details of an existing order return. Supply the unique order ID from either an order return creation request or the order return list, and Stripe will return the corresponding order information.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/order_returns/{options.Id}"
            |> RestApi.getAsync<OrderReturn> settings qs

    module Orders =

        type ListOptions = {
            ///Date this order was created.
            [<Config.Query>]Created: int option
            ///Only return orders for the given customer.
            [<Config.Query>]Customer: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///Only return orders with the given IDs.
            [<Config.Query>]Ids: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return orders that have the given status. One of `created`, `paid`, `fulfilled`, or `refunded`.
            [<Config.Query>]Status: string option
            ///Filter orders based on when they were paid, fulfilled, canceled, or returned.
            [<Config.Query>]StatusTransitions: Map<string, string> option
            ///Only return orders with the given upstream order IDs.
            [<Config.Query>]UpstreamIds: string list option
        }
        with
            static member New(?created: int, ?customer: string, ?endingBefore: string, ?expand: string list, ?ids: string list, ?limit: int, ?startingAfter: string, ?status: string, ?statusTransitions: Map<string, string>, ?upstreamIds: string list) =
                {
                    Created = created
                    Customer = customer
                    EndingBefore = endingBefore
                    Expand = expand
                    Ids = ids
                    Limit = limit
                    StartingAfter = startingAfter
                    Status = status
                    StatusTransitions = statusTransitions
                    UpstreamIds = upstreamIds
                }

        ///<p>Returns a list of your orders. The orders are returned sorted by creation date, with the most recently created orders appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("customer", options.Customer |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("ids", options.Ids |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("status", options.Status |> box); ("status_transitions", options.StatusTransitions |> box); ("upstream_ids", options.UpstreamIds |> box)] |> Map.ofList
            $"/v1/orders"
            |> RestApi.getAsync<Order list> settings qs

        type Create'ItemsType =
        | Discount
        | Shipping
        | Sku
        | Tax

        type Create'Items = {
            [<Config.Form>]Amount: int option
            [<Config.Form>]Currency: string option
            [<Config.Form>]Description: string option
            ///The ID of the SKU being ordered.
            [<Config.Form>]Parent: string option
            ///The quantity of this order item. When type is `sku`, this is the number of instances of the SKU to be ordered.
            [<Config.Form>]Quantity: int option
            [<Config.Form>]Type: Create'ItemsType option
        }
        with
            static member New(?amount: int, ?currency: string, ?description: string, ?parent: string, ?quantity: int, ?type': Create'ItemsType) =
                {
                    Amount = amount
                    Currency = currency
                    Description = description
                    Parent = parent
                    Quantity = quantity
                    Type = type'
                }

        type Create'ShippingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'Shipping = {
            ///Customer shipping address.
            [<Config.Form>]Address: Create'ShippingAddress option
            ///Customer name.
            [<Config.Form>]Name: string option
            ///Customer phone (including extension).
            [<Config.Form>]Phone: string option
        }
        with
            static member New(?address: Create'ShippingAddress, ?name: string, ?phone: string) =
                {
                    Address = address
                    Name = name
                    Phone = phone
                }

        type CreateOptions = {
            ///A coupon code that represents a discount to be applied to this order. Must be one-time duration and in same currency as the order. An order can have multiple coupons.
            [<Config.Form>]Coupon: string option
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string
            ///The ID of an existing customer to use for this order. If provided, the customer email and shipping address will be used to create the order. Subsequently, the customer will also be charged to pay the order. If `email` or `shipping` are also provided, they will override the values retrieved from the customer object.
            [<Config.Form>]Customer: string option
            ///The email address of the customer placing the order.
            [<Config.Form>]Email: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///List of items constituting the order. An order can have up to 25 items.
            [<Config.Form>]Items: Create'Items list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Shipping address for the order. Required if any of the SKUs are for products that have `shippable` set to true.
            [<Config.Form>]Shipping: Create'Shipping option
        }
        with
            static member New(currency: string, ?coupon: string, ?customer: string, ?email: string, ?expand: string list, ?items: Create'Items list, ?metadata: Map<string, string>, ?shipping: Create'Shipping) =
                {
                    Coupon = coupon
                    Currency = currency
                    Customer = customer
                    Email = email
                    Expand = expand
                    Items = items
                    Metadata = metadata
                    Shipping = shipping
                }

        ///<p>Creates a new order object.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/orders"
            |> RestApi.postAsync<_, Order> settings (Map.empty) options

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Id: string
        }
        with
            static member New(id: string, ?expand: string list) =
                {
                    Expand = expand
                    Id = id
                }

        ///<p>Retrieves the details of an existing order. Supply the unique order ID from either an order creation request or the order list, and Stripe will return the corresponding order information.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/orders/{options.Id}"
            |> RestApi.getAsync<Order> settings qs

        type Update'Shipping = {
            ///The name of the carrier like `USPS`, `UPS`, or `FedEx`.
            [<Config.Form>]Carrier: string option
            ///The tracking number provided by the carrier.
            [<Config.Form>]TrackingNumber: string option
        }
        with
            static member New(?carrier: string, ?trackingNumber: string) =
                {
                    Carrier = carrier
                    TrackingNumber = trackingNumber
                }

        type Update'Status =
        | Canceled
        | Created
        | Fulfilled
        | Paid
        | Returned

        type UpdateOptions = {
            [<Config.Path>]Id: string
            ///A coupon code that represents a discount to be applied to this order. Must be one-time duration and in same currency as the order. An order can have multiple coupons.
            [<Config.Form>]Coupon: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The shipping method to select for fulfilling this order. If specified, must be one of the `id`s of a shipping method in the `shipping_methods` array. If specified, will overwrite the existing selected shipping method, updating `items` as necessary.
            [<Config.Form>]SelectedShippingMethod: string option
            ///Tracking information once the order has been fulfilled.
            [<Config.Form>]Shipping: Update'Shipping option
            ///Current order status. One of `created`, `paid`, `canceled`, `fulfilled`, or `returned`. More detail in the [Orders Guide](https://stripe.com/docs/orders/guide#understanding-order-statuses).
            [<Config.Form>]Status: Update'Status option
        }
        with
            static member New(id: string, ?coupon: string, ?expand: string list, ?metadata: Map<string, string>, ?selectedShippingMethod: string, ?shipping: Update'Shipping, ?status: Update'Status) =
                {
                    Id = id
                    Coupon = coupon
                    Expand = expand
                    Metadata = metadata
                    SelectedShippingMethod = selectedShippingMethod
                    Shipping = shipping
                    Status = status
                }

        ///<p>Updates the specific order by setting the values of the parameters passed. Any parameters not provided will be left unchanged.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/orders/{options.Id}"
            |> RestApi.postAsync<_, Order> settings (Map.empty) options

    module OrdersPay =

        type PayOptions = {
            [<Config.Path>]Id: string
            ///A fee in %s that will be applied to the order and transferred to the application owner's Stripe account. The request must be made with an OAuth key or the `Stripe-Account` header in order to take an application fee. For more information, see the application fees [documentation](https://stripe.com/docs/connect/direct-charges#collecting-fees).
            [<Config.Form>]ApplicationFee: int option
            ///The ID of an existing customer that will be charged for this order. If no customer was attached to the order at creation, either `source` or `customer` is required. Otherwise, the specified customer will be charged instead of the one attached to the order.
            [<Config.Form>]Customer: string option
            ///The email address of the customer placing the order. Required if not previously specified for the order.
            [<Config.Form>]Email: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///A [Token](https://stripe.com/docs/api#tokens)'s or a [Source](https://stripe.com/docs/api#sources)'s ID, as returned by [Elements](https://stripe.com/docs/elements). If no customer was attached to the order at creation, either `source` or `customer` is required. Otherwise, the specified source will be charged intead of the customer attached to the order.
            [<Config.Form>]Source: string option
        }
        with
            static member New(id: string, ?applicationFee: int, ?customer: string, ?email: string, ?expand: string list, ?metadata: Map<string, string>, ?source: string) =
                {
                    Id = id
                    ApplicationFee = applicationFee
                    Customer = customer
                    Email = email
                    Expand = expand
                    Metadata = metadata
                    Source = source
                }

        ///<p>Pay an order by providing a <code>source</code> to create a payment.</p>
        let Pay settings (options: PayOptions) =
            $"/v1/orders/{options.Id}/pay"
            |> RestApi.postAsync<_, Order> settings (Map.empty) options

    module OrdersReturns =

        type ReturnOrder'ItemsType =
        | Discount
        | Shipping
        | Sku
        | Tax

        type ReturnOrder'Items = {
            ///The amount (price) for this order item to return.
            [<Config.Form>]Amount: int option
            ///If returning a `tax` item, use description to disambiguate which one to return.
            [<Config.Form>]Description: string option
            ///The ID of the SKU, tax, or shipping item being returned.
            [<Config.Form>]Parent: string option
            ///When type is `sku`, this is the number of instances of the SKU to be returned.
            [<Config.Form>]Quantity: int option
            ///The type of this order item. Must be `sku`, `tax`, or `shipping`.
            [<Config.Form>]Type: ReturnOrder'ItemsType option
        }
        with
            static member New(?amount: int, ?description: string, ?parent: string, ?quantity: int, ?type': ReturnOrder'ItemsType) =
                {
                    Amount = amount
                    Description = description
                    Parent = parent
                    Quantity = quantity
                    Type = type'
                }

        type ReturnOrderOptions = {
            [<Config.Path>]Id: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///List of items to return.
            [<Config.Form>]Items: Choice<ReturnOrder'Items list,string> option
        }
        with
            static member New(id: string, ?expand: string list, ?items: Choice<ReturnOrder'Items list,string>) =
                {
                    Id = id
                    Expand = expand
                    Items = items
                }

        ///<p>Return all or part of an order. The order must have a status of <code>paid</code> or <code>fulfilled</code> before it can be returned. Once all items have been returned, the order will become <code>canceled</code> or <code>returned</code> depending on which status the order started in.</p>
        let ReturnOrder settings (options: ReturnOrderOptions) =
            $"/v1/orders/{options.Id}/returns"
            |> RestApi.postAsync<_, OrderReturn> settings (Map.empty) options

    module PaymentIntents =

        type ListOptions = {
            ///A filter on the list, based on the object `created` field. The value can be a string with an integer Unix timestamp, or it can be a dictionary with a number of different query options.
            [<Config.Query>]Created: int option
            ///Only return PaymentIntents for the customer specified by this customer ID.
            [<Config.Query>]Customer: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?created: int, ?customer: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Created = created
                    Customer = customer
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of PaymentIntents.</p>
        let List settings (options: ListOptions) =
            let qs = [("created", options.Created |> box); ("customer", options.Customer |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/payment_intents"
            |> RestApi.getAsync<PaymentIntent list> settings qs

        type Create'MandateDataCustomerAcceptanceOnline = {
            ///The IP address from which the Mandate was accepted by the customer.
            [<Config.Form>]IpAddress: string option
            ///The user agent of the browser from which the Mandate was accepted by the customer.
            [<Config.Form>]UserAgent: string option
        }
        with
            static member New(?ipAddress: string, ?userAgent: string) =
                {
                    IpAddress = ipAddress
                    UserAgent = userAgent
                }

        type Create'MandateDataCustomerAcceptanceType =
        | Offline
        | Online

        type Create'MandateDataCustomerAcceptance = {
            ///The time at which the customer accepted the Mandate.
            [<Config.Form>]AcceptedAt: DateTime option
            ///If this is a Mandate accepted offline, this hash contains details about the offline acceptance.
            [<Config.Form>]Offline: string option
            ///If this is a Mandate accepted online, this hash contains details about the online acceptance.
            [<Config.Form>]Online: Create'MandateDataCustomerAcceptanceOnline option
            ///The type of customer acceptance information included with the Mandate. One of `online` or `offline`.
            [<Config.Form>]Type: Create'MandateDataCustomerAcceptanceType option
        }
        with
            static member New(?acceptedAt: DateTime, ?offline: string, ?online: Create'MandateDataCustomerAcceptanceOnline, ?type': Create'MandateDataCustomerAcceptanceType) =
                {
                    AcceptedAt = acceptedAt
                    Offline = offline
                    Online = online
                    Type = type'
                }

        type Create'MandateData = {
            ///This hash contains details about the customer acceptance of the Mandate.
            [<Config.Form>]CustomerAcceptance: Create'MandateDataCustomerAcceptance option
        }
        with
            static member New(?customerAcceptance: Create'MandateDataCustomerAcceptance) =
                {
                    CustomerAcceptance = customerAcceptance
                }

        type Create'OffSession =
        | OneOff
        | Recurring

        type Create'PaymentMethodDataAcssDebit = {
            ///Customer's bank account number.
            [<Config.Form>]AccountNumber: string option
            ///Institution number of the customer's bank.
            [<Config.Form>]InstitutionNumber: string option
            ///Transit number of the customer's bank.
            [<Config.Form>]TransitNumber: string option
        }
        with
            static member New(?accountNumber: string, ?institutionNumber: string, ?transitNumber: string) =
                {
                    AccountNumber = accountNumber
                    InstitutionNumber = institutionNumber
                    TransitNumber = transitNumber
                }

        type Create'PaymentMethodDataAuBecsDebit = {
            ///The account number for the bank account.
            [<Config.Form>]AccountNumber: string option
            ///Bank-State-Branch number of the bank account.
            [<Config.Form>]BsbNumber: string option
        }
        with
            static member New(?accountNumber: string, ?bsbNumber: string) =
                {
                    AccountNumber = accountNumber
                    BsbNumber = bsbNumber
                }

        type Create'PaymentMethodDataBacsDebit = {
            ///Account number of the bank account that the funds will be debited from.
            [<Config.Form>]AccountNumber: string option
            ///Sort code of the bank account. (e.g., `10-20-30`)
            [<Config.Form>]SortCode: string option
        }
        with
            static member New(?accountNumber: string, ?sortCode: string) =
                {
                    AccountNumber = accountNumber
                    SortCode = sortCode
                }

        type Create'PaymentMethodDataBillingDetailsAddressBillingDetailsAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'PaymentMethodDataBillingDetails = {
            ///Billing address.
            [<Config.Form>]Address: Choice<Create'PaymentMethodDataBillingDetailsAddressBillingDetailsAddress,string> option
            ///Email address.
            [<Config.Form>]Email: string option
            ///Full name.
            [<Config.Form>]Name: string option
            ///Billing phone number (including extension).
            [<Config.Form>]Phone: string option
        }
        with
            static member New(?address: Choice<Create'PaymentMethodDataBillingDetailsAddressBillingDetailsAddress,string>, ?email: string, ?name: string, ?phone: string) =
                {
                    Address = address
                    Email = email
                    Name = name
                    Phone = phone
                }

        type Create'PaymentMethodDataBoleto = {
            ///Uniquely identifies this customer tax_id (CNPJ or CPF)
            [<Config.Form>]TaxId: string option
        }
        with
            static member New(?taxId: string) =
                {
                    TaxId = taxId
                }

        type Create'PaymentMethodDataEpsBank =
        | ArzteUndApothekerBank
        | AustrianAnadiBankAg
        | BankAustria
        | BankhausCarlSpangler
        | BankhausSchelhammerUndSchatteraAg
        | BawagPskAg
        | BksBankAg
        | BrullKallmusBankAg
        | BtvVierLanderBank
        | CapitalBankGraweGruppeAg
        | Dolomitenbank
        | EasybankAg
        | ErsteBankUndSparkassen
        | HypoAlpeadriabankInternationalAg
        | HypoBankBurgenlandAktiengesellschaft
        | HypoNoeLbFurNiederosterreichUWien
        | HypoOberosterreichSalzburgSteiermark
        | HypoTirolBankAg
        | HypoVorarlbergBankAg
        | MarchfelderBank
        | OberbankAg
        | RaiffeisenBankengruppeOsterreich
        | SchoellerbankAg
        | SpardaBankWien
        | VolksbankGruppe
        | VolkskreditbankAg
        | VrBankBraunau

        type Create'PaymentMethodDataEps = {
            ///The customer's bank.
            [<Config.Form>]Bank: Create'PaymentMethodDataEpsBank option
        }
        with
            static member New(?bank: Create'PaymentMethodDataEpsBank) =
                {
                    Bank = bank
                }

        type Create'PaymentMethodDataFpxAccountHolderType =
        | Company
        | Individual

        type Create'PaymentMethodDataFpxBank =
        | AffinBank
        | AllianceBank
        | Ambank
        | BankIslam
        | BankMuamalat
        | BankRakyat
        | Bsn
        | Cimb
        | DeutscheBank
        | HongLeongBank
        | Hsbc
        | Kfh
        | Maybank2e
        | Maybank2u
        | Ocbc
        | PbEnterprise
        | PublicBank
        | Rhb
        | StandardChartered
        | Uob

        type Create'PaymentMethodDataFpx = {
            ///Account holder type for FPX transaction
            [<Config.Form>]AccountHolderType: Create'PaymentMethodDataFpxAccountHolderType option
            ///The customer's bank.
            [<Config.Form>]Bank: Create'PaymentMethodDataFpxBank option
        }
        with
            static member New(?accountHolderType: Create'PaymentMethodDataFpxAccountHolderType, ?bank: Create'PaymentMethodDataFpxBank) =
                {
                    AccountHolderType = accountHolderType
                    Bank = bank
                }

        type Create'PaymentMethodDataIdealBank =
        | AbnAmro
        | AsnBank
        | Bunq
        | Handelsbanken
        | Ing
        | Knab
        | Moneyou
        | Rabobank
        | Regiobank
        | Revolut
        | SnsBank
        | TriodosBank
        | VanLanschot

        type Create'PaymentMethodDataIdeal = {
            ///The customer's bank.
            [<Config.Form>]Bank: Create'PaymentMethodDataIdealBank option
        }
        with
            static member New(?bank: Create'PaymentMethodDataIdealBank) =
                {
                    Bank = bank
                }

        type Create'PaymentMethodDataP24Bank =
        | AliorBank
        | BankMillennium
        | BankNowyBfgSa
        | BankPekaoSa
        | BankiSpbdzielcze
        | Blik
        | BnpParibas
        | Boz
        | CitiHandlowy
        | CreditAgricole
        | Envelobank
        | EtransferPocztowy24
        | GetinBank
        | Ideabank
        | Ing
        | Inteligo
        | MbankMtransfer
        | NestPrzelew
        | NoblePay
        | PbacZIpko
        | PlusBank
        | SantanderPrzelew24
        | TmobileUsbugiBankowe
        | ToyotaBank
        | VolkswagenBank

        type Create'PaymentMethodDataP24 = {
            ///The customer's bank.
            [<Config.Form>]Bank: Create'PaymentMethodDataP24Bank option
        }
        with
            static member New(?bank: Create'PaymentMethodDataP24Bank) =
                {
                    Bank = bank
                }

        type Create'PaymentMethodDataSepaDebit = {
            ///IBAN of the bank account.
            [<Config.Form>]Iban: string option
        }
        with
            static member New(?iban: string) =
                {
                    Iban = iban
                }

        type Create'PaymentMethodDataSofortCountry =
        | [<JsonUnionCase("AT")>] AT
        | [<JsonUnionCase("BE")>] BE
        | [<JsonUnionCase("DE")>] DE
        | [<JsonUnionCase("ES")>] ES
        | [<JsonUnionCase("IT")>] IT
        | [<JsonUnionCase("NL")>] NL

        type Create'PaymentMethodDataSofort = {
            ///Two-letter ISO code representing the country the bank account is located in.
            [<Config.Form>]Country: Create'PaymentMethodDataSofortCountry option
        }
        with
            static member New(?country: Create'PaymentMethodDataSofortCountry) =
                {
                    Country = country
                }

        type Create'PaymentMethodDataType =
        | AcssDebit
        | AfterpayClearpay
        | Alipay
        | AuBecsDebit
        | BacsDebit
        | Bancontact
        | Boleto
        | Eps
        | Fpx
        | Giropay
        | Grabpay
        | Ideal
        | Oxxo
        | P24
        | SepaDebit
        | Sofort
        | WechatPay

        type Create'PaymentMethodData = {
            ///If this is an `acss_debit` PaymentMethod, this hash contains details about the ACSS Debit payment method.
            [<Config.Form>]AcssDebit: Create'PaymentMethodDataAcssDebit option
            ///If this is an `AfterpayClearpay` PaymentMethod, this hash contains details about the AfterpayClearpay payment method.
            [<Config.Form>]AfterpayClearpay: string option
            ///If this is an `Alipay` PaymentMethod, this hash contains details about the Alipay payment method.
            [<Config.Form>]Alipay: string option
            ///If this is an `au_becs_debit` PaymentMethod, this hash contains details about the bank account.
            [<Config.Form>]AuBecsDebit: Create'PaymentMethodDataAuBecsDebit option
            ///If this is a `bacs_debit` PaymentMethod, this hash contains details about the Bacs Direct Debit bank account.
            [<Config.Form>]BacsDebit: Create'PaymentMethodDataBacsDebit option
            ///If this is a `bancontact` PaymentMethod, this hash contains details about the Bancontact payment method.
            [<Config.Form>]Bancontact: string option
            ///Billing information associated with the PaymentMethod that may be used or required by particular types of payment methods.
            [<Config.Form>]BillingDetails: Create'PaymentMethodDataBillingDetails option
            ///If this is a `boleto` PaymentMethod, this hash contains details about the Boleto payment method.
            [<Config.Form>]Boleto: Create'PaymentMethodDataBoleto option
            ///If this is an `eps` PaymentMethod, this hash contains details about the EPS payment method.
            [<Config.Form>]Eps: Create'PaymentMethodDataEps option
            ///If this is an `fpx` PaymentMethod, this hash contains details about the FPX payment method.
            [<Config.Form>]Fpx: Create'PaymentMethodDataFpx option
            ///If this is a `giropay` PaymentMethod, this hash contains details about the Giropay payment method.
            [<Config.Form>]Giropay: string option
            ///If this is a `grabpay` PaymentMethod, this hash contains details about the GrabPay payment method.
            [<Config.Form>]Grabpay: string option
            ///If this is an `ideal` PaymentMethod, this hash contains details about the iDEAL payment method.
            [<Config.Form>]Ideal: Create'PaymentMethodDataIdeal option
            ///If this is an `interac_present` PaymentMethod, this hash contains details about the Interac Present payment method.
            [<Config.Form>]InteracPresent: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///If this is an `oxxo` PaymentMethod, this hash contains details about the OXXO payment method.
            [<Config.Form>]Oxxo: string option
            ///If this is a `p24` PaymentMethod, this hash contains details about the P24 payment method.
            [<Config.Form>]P24: Create'PaymentMethodDataP24 option
            ///If this is a `sepa_debit` PaymentMethod, this hash contains details about the SEPA debit bank account.
            [<Config.Form>]SepaDebit: Create'PaymentMethodDataSepaDebit option
            ///If this is a `sofort` PaymentMethod, this hash contains details about the SOFORT payment method.
            [<Config.Form>]Sofort: Create'PaymentMethodDataSofort option
            ///The type of the PaymentMethod. An additional hash is included on the PaymentMethod with a name matching this value. It contains additional information specific to the PaymentMethod type.
            [<Config.Form>]Type: Create'PaymentMethodDataType option
            ///If this is an `wechat_pay` PaymentMethod, this hash contains details about the wechat_pay payment method.
            [<Config.Form>]WechatPay: string option
        }
        with
            static member New(?acssDebit: Create'PaymentMethodDataAcssDebit, ?sofort: Create'PaymentMethodDataSofort, ?sepaDebit: Create'PaymentMethodDataSepaDebit, ?p24: Create'PaymentMethodDataP24, ?oxxo: string, ?metadata: Map<string, string>, ?interacPresent: string, ?ideal: Create'PaymentMethodDataIdeal, ?grabpay: string, ?type': Create'PaymentMethodDataType, ?giropay: string, ?eps: Create'PaymentMethodDataEps, ?boleto: Create'PaymentMethodDataBoleto, ?billingDetails: Create'PaymentMethodDataBillingDetails, ?bancontact: string, ?bacsDebit: Create'PaymentMethodDataBacsDebit, ?auBecsDebit: Create'PaymentMethodDataAuBecsDebit, ?alipay: string, ?afterpayClearpay: string, ?fpx: Create'PaymentMethodDataFpx, ?wechatPay: string) =
                {
                    AcssDebit = acssDebit
                    AfterpayClearpay = afterpayClearpay
                    Alipay = alipay
                    AuBecsDebit = auBecsDebit
                    BacsDebit = bacsDebit
                    Bancontact = bancontact
                    BillingDetails = billingDetails
                    Boleto = boleto
                    Eps = eps
                    Fpx = fpx
                    Giropay = giropay
                    Grabpay = grabpay
                    Ideal = ideal
                    InteracPresent = interacPresent
                    Metadata = metadata
                    Oxxo = oxxo
                    P24 = p24
                    SepaDebit = sepaDebit
                    Sofort = sofort
                    Type = type'
                    WechatPay = wechatPay
                }

        type Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsPaymentSchedule =
        | Combined
        | Interval
        | Sporadic

        type Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsTransactionType =
        | Business
        | Personal

        type Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptions = {
            ///A URL for custom mandate text to render during confirmation step.
            ///The URL will be rendered with additional GET parameters `payment_intent` and `payment_intent_client_secret` when confirming a Payment Intent,
            ///or `setup_intent` and `setup_intent_client_secret` when confirming a Setup Intent.
            [<Config.Form>]CustomMandateUrl: Choice<string,string> option
            ///Description of the mandate interval. Only required if 'payment_schedule' parameter is 'interval' or 'combined'.
            [<Config.Form>]IntervalDescription: string option
            ///Payment schedule for the mandate.
            [<Config.Form>]PaymentSchedule: Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsPaymentSchedule option
            ///Transaction type of the mandate.
            [<Config.Form>]TransactionType: Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsTransactionType option
        }
        with
            static member New(?customMandateUrl: Choice<string,string>, ?intervalDescription: string, ?paymentSchedule: Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsPaymentSchedule, ?transactionType: Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsTransactionType) =
                {
                    CustomMandateUrl = customMandateUrl
                    IntervalDescription = intervalDescription
                    PaymentSchedule = paymentSchedule
                    TransactionType = transactionType
                }

        type Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsVerificationMethod =
        | Automatic
        | Instant
        | Microdeposits

        type Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptions = {
            ///Additional fields for Mandate creation
            [<Config.Form>]MandateOptions: Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptions option
            ///Verification method for the intent
            [<Config.Form>]VerificationMethod: Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsVerificationMethod option
        }
        with
            static member New(?mandateOptions: Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptions, ?verificationMethod: Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsVerificationMethod) =
                {
                    MandateOptions = mandateOptions
                    VerificationMethod = verificationMethod
                }

        type Create'PaymentMethodOptionsAfterpayClearpayPaymentMethodOptions = {
            ///Order identifier shown to the customer in Afterpay’s online portal. We recommend using a value that helps you answer any questions a customer might have about
            ///the payment. The identifier is limited to 128 characters and may contain only letters, digits, underscores, backslashes and dashes.
            [<Config.Form>]Reference: string option
        }
        with
            static member New(?reference: string) =
                {
                    Reference = reference
                }

        type Create'PaymentMethodOptionsBancontactPaymentMethodOptionsPreferredLanguage =
        | De
        | En
        | Fr
        | Nl

        type Create'PaymentMethodOptionsBancontactPaymentMethodOptions = {
            ///Preferred language of the Bancontact authorization page that the customer is redirected to.
            [<Config.Form>]PreferredLanguage: Create'PaymentMethodOptionsBancontactPaymentMethodOptionsPreferredLanguage option
        }
        with
            static member New(?preferredLanguage: Create'PaymentMethodOptionsBancontactPaymentMethodOptionsPreferredLanguage) =
                {
                    PreferredLanguage = preferredLanguage
                }

        type Create'PaymentMethodOptionsBoletoPaymentMethodOptions = {
            ///The number of calendar days before a Boleto voucher expires. For example, if you create a Boleto voucher on Monday and you set expires_after_days to 2, the Boleto invoice will expire on Wednesday at 23:59 America/Sao_Paulo time.
            [<Config.Form>]ExpiresAfterDays: int option
        }
        with
            static member New(?expiresAfterDays: int) =
                {
                    ExpiresAfterDays = expiresAfterDays
                }

        type Create'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanInterval =
        | Month

        type Create'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanType =
        | FixedCount

        type Create'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlan = {
            ///For `fixed_count` installment plans, this is the number of installment payments your customer will make to their credit card.
            [<Config.Form>]Count: int option
            ///For `fixed_count` installment plans, this is the interval between installment payments your customer will make to their credit card.
            ///One of `month`.
            [<Config.Form>]Interval: Create'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanInterval option
            ///Type of installment plan, one of `fixed_count`.
            [<Config.Form>]Type: Create'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanType option
        }
        with
            static member New(?count: int, ?interval: Create'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanInterval, ?type': Create'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanType) =
                {
                    Count = count
                    Interval = interval
                    Type = type'
                }

        type Create'PaymentMethodOptionsCardPaymentIntentInstallments = {
            ///Setting to true enables installments for this PaymentIntent.
            ///This will cause the response to contain a list of available installment plans.
            ///Setting to false will prevent any selected plan from applying to a charge.
            [<Config.Form>]Enabled: bool option
            ///The selected installment plan to use for this payment attempt.
            ///This parameter can only be provided during confirmation.
            [<Config.Form>]Plan: Choice<Create'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlan,string> option
        }
        with
            static member New(?enabled: bool, ?plan: Choice<Create'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlan,string>) =
                {
                    Enabled = enabled
                    Plan = plan
                }

        type Create'PaymentMethodOptionsCardPaymentIntentNetwork =
        | Amex
        | CartesBancaires
        | Diners
        | Discover
        | Interac
        | Jcb
        | Mastercard
        | Unionpay
        | Unknown
        | Visa

        type Create'PaymentMethodOptionsCardPaymentIntentRequestThreeDSecure =
        | Any
        | Automatic

        type Create'PaymentMethodOptionsCardPaymentIntent = {
            ///A single-use `cvc_update` Token that represents a card CVC value. When provided, the CVC value will be verified during the card payment attempt. This parameter can only be provided during confirmation.
            [<Config.Form>]CvcToken: string option
            ///Installment configuration for payments attempted on this PaymentIntent (Mexico Only).
            ///For more information, see the [installments integration guide](https://stripe.com/docs/payments/installments).
            [<Config.Form>]Installments: Create'PaymentMethodOptionsCardPaymentIntentInstallments option
            ///When specified, this parameter indicates that a transaction will be marked
            ///as MOTO (Mail Order Telephone Order) and thus out of scope for SCA. This
            ///parameter can only be provided during confirmation.
            [<Config.Form>]Moto: bool option
            ///Selected network to process this PaymentIntent on. Depends on the available networks of the card attached to the PaymentIntent. Can be only set confirm-time.
            [<Config.Form>]Network: Create'PaymentMethodOptionsCardPaymentIntentNetwork option
            ///We strongly recommend that you rely on our SCA Engine to automatically prompt your customers for authentication based on risk level and [other requirements](https://stripe.com/docs/strong-customer-authentication). However, if you wish to request 3D Secure based on logic from your own fraud engine, provide this option. Permitted values include: `automatic` or `any`. If not provided, defaults to `automatic`. Read our guide on [manually requesting 3D Secure](https://stripe.com/docs/payments/3d-secure#manual-three-ds) for more information on how this configuration interacts with Radar and our SCA Engine.
            [<Config.Form>]RequestThreeDSecure: Create'PaymentMethodOptionsCardPaymentIntentRequestThreeDSecure option
        }
        with
            static member New(?cvcToken: string, ?installments: Create'PaymentMethodOptionsCardPaymentIntentInstallments, ?moto: bool, ?network: Create'PaymentMethodOptionsCardPaymentIntentNetwork, ?requestThreeDSecure: Create'PaymentMethodOptionsCardPaymentIntentRequestThreeDSecure) =
                {
                    CvcToken = cvcToken
                    Installments = installments
                    Moto = moto
                    Network = network
                    RequestThreeDSecure = requestThreeDSecure
                }

        type Create'PaymentMethodOptionsOxxoPaymentMethodOptions = {
            ///The number of calendar days before an OXXO voucher expires. For example, if you create an OXXO voucher on Monday and you set expires_after_days to 2, the OXXO invoice will expire on Wednesday at 23:59 America/Mexico_City time.
            [<Config.Form>]ExpiresAfterDays: int option
        }
        with
            static member New(?expiresAfterDays: int) =
                {
                    ExpiresAfterDays = expiresAfterDays
                }

        type Create'PaymentMethodOptionsP24PaymentMethodOptions = {
            ///Confirm that the payer has accepted the P24 terms and conditions.
            [<Config.Form>]TosShownAndAccepted: bool option
        }
        with
            static member New(?tosShownAndAccepted: bool) =
                {
                    TosShownAndAccepted = tosShownAndAccepted
                }

        type Create'PaymentMethodOptionsSepaDebitPaymentIntentPaymentMethodOptions = {
            ///Additional fields for Mandate creation
            [<Config.Form>]MandateOptions: string option
        }
        with
            static member New(?mandateOptions: string) =
                {
                    MandateOptions = mandateOptions
                }

        type Create'PaymentMethodOptionsSofortPaymentMethodOptionsPreferredLanguage =
        | De
        | En
        | Es
        | Fr
        | It
        | Nl
        | Pl

        type Create'PaymentMethodOptionsSofortPaymentMethodOptions = {
            ///Language shown to the payer on redirect.
            [<Config.Form>]PreferredLanguage: Create'PaymentMethodOptionsSofortPaymentMethodOptionsPreferredLanguage option
        }
        with
            static member New(?preferredLanguage: Create'PaymentMethodOptionsSofortPaymentMethodOptionsPreferredLanguage) =
                {
                    PreferredLanguage = preferredLanguage
                }

        type Create'PaymentMethodOptionsWechatPayPaymentMethodOptionsClient =
        | Android
        | Ios
        | Web

        type Create'PaymentMethodOptionsWechatPayPaymentMethodOptions = {
            ///The app ID registered with WeChat Pay. Only required when client is ios or android.
            [<Config.Form>]AppId: string option
            ///The client type that the end customer will pay from
            [<Config.Form>]Client: Create'PaymentMethodOptionsWechatPayPaymentMethodOptionsClient option
        }
        with
            static member New(?appId: string, ?client: Create'PaymentMethodOptionsWechatPayPaymentMethodOptionsClient) =
                {
                    AppId = appId
                    Client = client
                }

        type Create'PaymentMethodOptions = {
            ///If this is a `acss_debit` PaymentMethod, this sub-hash contains details about the ACSS Debit payment method options.
            [<Config.Form>]AcssDebit: Choice<Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptions,string> option
            ///If this is a `afterpay_clearpay` PaymentMethod, this sub-hash contains details about the Afterpay Clearpay payment method options.
            [<Config.Form>]AfterpayClearpay: Choice<Create'PaymentMethodOptionsAfterpayClearpayPaymentMethodOptions,string> option
            ///If this is a `alipay` PaymentMethod, this sub-hash contains details about the Alipay payment method options.
            [<Config.Form>]Alipay: Choice<string,string> option
            ///If this is a `bancontact` PaymentMethod, this sub-hash contains details about the Bancontact payment method options.
            [<Config.Form>]Bancontact: Choice<Create'PaymentMethodOptionsBancontactPaymentMethodOptions,string> option
            ///If this is a `boleto` PaymentMethod, this sub-hash contains details about the Boleto payment method options.
            [<Config.Form>]Boleto: Choice<Create'PaymentMethodOptionsBoletoPaymentMethodOptions,string> option
            ///Configuration for any card payments attempted on this PaymentIntent.
            [<Config.Form>]Card: Choice<Create'PaymentMethodOptionsCardPaymentIntent,string> option
            ///If this is a `card_present` PaymentMethod, this sub-hash contains details about the Card Present payment method options.
            [<Config.Form>]CardPresent: Choice<string,string> option
            ///If this is a `oxxo` PaymentMethod, this sub-hash contains details about the OXXO payment method options.
            [<Config.Form>]Oxxo: Choice<Create'PaymentMethodOptionsOxxoPaymentMethodOptions,string> option
            ///If this is a `p24` PaymentMethod, this sub-hash contains details about the Przelewy24 payment method options.
            [<Config.Form>]P24: Choice<Create'PaymentMethodOptionsP24PaymentMethodOptions,string> option
            ///If this is a `sepa_debit` PaymentIntent, this sub-hash contains details about the SEPA Debit payment method options.
            [<Config.Form>]SepaDebit: Choice<Create'PaymentMethodOptionsSepaDebitPaymentIntentPaymentMethodOptions,string> option
            ///If this is a `sofort` PaymentMethod, this sub-hash contains details about the SOFORT payment method options.
            [<Config.Form>]Sofort: Choice<Create'PaymentMethodOptionsSofortPaymentMethodOptions,string> option
            ///If this is a `wechat_pay` PaymentMethod, this sub-hash contains details about the WeChat Pay payment method options.
            [<Config.Form>]WechatPay: Choice<Create'PaymentMethodOptionsWechatPayPaymentMethodOptions,string> option
        }
        with
            static member New(?acssDebit: Choice<Create'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptions,string>, ?afterpayClearpay: Choice<Create'PaymentMethodOptionsAfterpayClearpayPaymentMethodOptions,string>, ?alipay: Choice<string,string>, ?bancontact: Choice<Create'PaymentMethodOptionsBancontactPaymentMethodOptions,string>, ?boleto: Choice<Create'PaymentMethodOptionsBoletoPaymentMethodOptions,string>, ?card: Choice<Create'PaymentMethodOptionsCardPaymentIntent,string>, ?cardPresent: Choice<string,string>, ?oxxo: Choice<Create'PaymentMethodOptionsOxxoPaymentMethodOptions,string>, ?p24: Choice<Create'PaymentMethodOptionsP24PaymentMethodOptions,string>, ?sepaDebit: Choice<Create'PaymentMethodOptionsSepaDebitPaymentIntentPaymentMethodOptions,string>, ?sofort: Choice<Create'PaymentMethodOptionsSofortPaymentMethodOptions,string>, ?wechatPay: Choice<Create'PaymentMethodOptionsWechatPayPaymentMethodOptions,string>) =
                {
                    AcssDebit = acssDebit
                    AfterpayClearpay = afterpayClearpay
                    Alipay = alipay
                    Bancontact = bancontact
                    Boleto = boleto
                    Card = card
                    CardPresent = cardPresent
                    Oxxo = oxxo
                    P24 = p24
                    SepaDebit = sepaDebit
                    Sofort = sofort
                    WechatPay = wechatPay
                }

        type Create'ShippingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'Shipping = {
            ///Shipping address.
            [<Config.Form>]Address: Create'ShippingAddress option
            ///The delivery service that shipped a physical product, such as Fedex, UPS, USPS, etc.
            [<Config.Form>]Carrier: string option
            ///Recipient name.
            [<Config.Form>]Name: string option
            ///Recipient phone (including extension).
            [<Config.Form>]Phone: string option
            ///The tracking number for a physical product, obtained from the delivery service. If multiple tracking numbers were generated for this purchase, please separate them with commas.
            [<Config.Form>]TrackingNumber: string option
        }
        with
            static member New(?address: Create'ShippingAddress, ?carrier: string, ?name: string, ?phone: string, ?trackingNumber: string) =
                {
                    Address = address
                    Carrier = carrier
                    Name = name
                    Phone = phone
                    TrackingNumber = trackingNumber
                }

        type Create'TransferData = {
            ///The amount that will be transferred automatically when a charge succeeds.
            ///The amount is capped at the total transaction amount and if no amount is set,
            ///the full amount is transferred.
            ///If you intend to collect a fee and you need a more robust reporting experience, using
            ///[application_fee_amount](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-application_fee_amount)
            ///might be a better fit for your integration.
            [<Config.Form>]Amount: int option
            ///If specified, successful charges will be attributed to the destination
            ///account for tax reporting, and the funds from charges will be transferred
            ///to the destination account. The ID of the resulting transfer will be
            ///returned on the successful charge's `transfer` field.
            [<Config.Form>]Destination: string option
        }
        with
            static member New(?amount: int, ?destination: string) =
                {
                    Amount = amount
                    Destination = destination
                }

        type Create'CaptureMethod =
        | Automatic
        | Manual

        type Create'ConfirmationMethod =
        | Automatic
        | Manual

        type Create'SetupFutureUsage =
        | OffSession
        | OnSession

        type CreateOptions = {
            ///Amount intended to be collected by this PaymentIntent. A positive integer representing how much to charge in the [smallest currency unit](https://stripe.com/docs/currencies#zero-decimal) (e.g., 100 cents to charge $1.00 or 100 to charge ¥100, a zero-decimal currency). The minimum amount is $0.50 US or [equivalent in charge currency](https://stripe.com/docs/currencies#minimum-and-maximum-charge-amounts). The amount value supports up to eight digits (e.g., a value of 99999999 for a USD charge of $999,999.99).
            [<Config.Form>]Amount: int
            ///The amount of the application fee (if any) that will be requested to be applied to the payment and transferred to the application owner's Stripe account. The amount of the application fee collected will be capped at the total payment amount. For more information, see the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts).
            [<Config.Form>]ApplicationFeeAmount: int option
            ///Controls when the funds will be captured from the customer's account.
            [<Config.Form>]CaptureMethod: Create'CaptureMethod option
            ///Set to `true` to attempt to [confirm](https://stripe.com/docs/api/payment_intents/confirm) this PaymentIntent immediately. This parameter defaults to `false`. When creating and confirming a PaymentIntent at the same time, parameters available in the [confirm](https://stripe.com/docs/api/payment_intents/confirm) API may also be provided.
            [<Config.Form>]Confirm: bool option
            [<Config.Form>]ConfirmationMethod: Create'ConfirmationMethod option
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string
            ///ID of the Customer this PaymentIntent belongs to, if one exists.
            ///Payment methods attached to other Customers cannot be used with this PaymentIntent.
            ///If present in combination with [setup_future_usage](https://stripe.com/docs/api#payment_intent_object-setup_future_usage), this PaymentIntent's payment method will be attached to the Customer after the PaymentIntent has been confirmed and any required actions from the user are complete.
            [<Config.Form>]Customer: string option
            ///An arbitrary string attached to the object. Often useful for displaying to users.
            [<Config.Form>]Description: string option
            ///Set to `true` to fail the payment attempt if the PaymentIntent transitions into `requires_action`. This parameter is intended for simpler integrations that do not handle customer actions, like [saving cards without authentication](https://stripe.com/docs/payments/save-card-without-authentication). This parameter can only be used with [`confirm=true`](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-confirm).
            [<Config.Form>]ErrorOnRequiresAction: bool option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///ID of the mandate to be used for this payment. This parameter can only be used with [`confirm=true`](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-confirm).
            [<Config.Form>]Mandate: string option
            ///This hash contains details about the Mandate to create. This parameter can only be used with [`confirm=true`](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-confirm).
            [<Config.Form>]MandateData: Create'MandateData option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///Set to `true` to indicate that the customer is not in your checkout flow during this payment attempt, and therefore is unable to authenticate. This parameter is intended for scenarios where you collect card details and [charge them later](https://stripe.com/docs/payments/cards/charging-saved-cards). This parameter can only be used with [`confirm=true`](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-confirm).
            [<Config.Form>]OffSession: Choice<bool,Create'OffSession> option
            ///The Stripe account ID for which these funds are intended. For details, see the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts).
            [<Config.Form>]OnBehalfOf: string option
            ///ID of the payment method (a PaymentMethod, Card, or [compatible Source](https://stripe.com/docs/payments/payment-methods#compatibility) object) to attach to this PaymentIntent.
            ///If neither the `payment_method` parameter nor the `source` parameter are provided with `confirm=true`, `source` will be automatically populated with `customer.default_source` to improve the migration experience for users of the Charges API. We recommend that you explicitly provide the `payment_method` going forward.
            [<Config.Form>]PaymentMethod: string option
            ///If provided, this hash will be used to create a PaymentMethod. The new PaymentMethod will appear
            ///in the [payment_method](https://stripe.com/docs/api/payment_intents/object#payment_intent_object-payment_method)
            ///property on the PaymentIntent.
            [<Config.Form>]PaymentMethodData: Create'PaymentMethodData option
            ///Payment-method-specific configuration for this PaymentIntent.
            [<Config.Form>]PaymentMethodOptions: Create'PaymentMethodOptions option
            ///The list of payment method types (e.g. card) that this PaymentIntent is allowed to use. If this is not provided, defaults to ["card"].
            [<Config.Form>]PaymentMethodTypes: string list option
            ///Email address that the receipt for the resulting payment will be sent to. If `receipt_email` is specified for a payment in live mode, a receipt will be sent regardless of your [email settings](https://dashboard.stripe.com/account/emails).
            [<Config.Form>]ReceiptEmail: string option
            ///The URL to redirect your customer back to after they authenticate or cancel their payment on the payment method's app or site. If you'd prefer to redirect to a mobile application, you can alternatively supply an application URI scheme. This parameter can only be used with [`confirm=true`](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-confirm).
            [<Config.Form>]ReturnUrl: string option
            ///Indicates that you intend to make future payments with this PaymentIntent's payment method.
            ///Providing this parameter will [attach the payment method](https://stripe.com/docs/payments/save-during-payment) to the PaymentIntent's Customer, if present, after the PaymentIntent is confirmed and any required actions from the user are complete. If no Customer was provided, the payment method can still be [attached](https://stripe.com/docs/api/payment_methods/attach) to a Customer after the transaction completes.
            ///When processing card payments, Stripe also uses `setup_future_usage` to dynamically optimize your payment flow and comply with regional legislation and network rules, such as [SCA](https://stripe.com/docs/strong-customer-authentication).
            [<Config.Form>]SetupFutureUsage: Create'SetupFutureUsage option
            ///Shipping information for this PaymentIntent.
            [<Config.Form>]Shipping: Create'Shipping option
            ///For non-card charges, you can use this value as the complete description that appears on your customers’ statements. Must contain at least one letter, maximum 22 characters.
            [<Config.Form>]StatementDescriptor: string option
            ///Provides information about a card payment that customers see on their statements. Concatenated with the prefix (shortened descriptor) or statement descriptor that’s set on the account to form the complete statement descriptor. Maximum 22 characters for the concatenated descriptor.
            [<Config.Form>]StatementDescriptorSuffix: string option
            ///The parameters used to automatically create a Transfer when the payment succeeds.
            ///For more information, see the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts).
            [<Config.Form>]TransferData: Create'TransferData option
            ///A string that identifies the resulting payment as part of a group. See the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts) for details.
            [<Config.Form>]TransferGroup: string option
            ///Set to `true` only when using manual confirmation and the iOS or Android SDKs to handle additional authentication steps.
            [<Config.Form>]UseStripeSdk: bool option
        }
        with
            static member New(amount: int, currency: string, ?transferData: Create'TransferData, ?statementDescriptorSuffix: string, ?statementDescriptor: string, ?shipping: Create'Shipping, ?setupFutureUsage: Create'SetupFutureUsage, ?returnUrl: string, ?receiptEmail: string, ?paymentMethodTypes: string list, ?paymentMethodOptions: Create'PaymentMethodOptions, ?paymentMethodData: Create'PaymentMethodData, ?paymentMethod: string, ?onBehalfOf: string, ?offSession: Choice<bool,Create'OffSession>, ?metadata: Map<string, string>, ?mandateData: Create'MandateData, ?mandate: string, ?expand: string list, ?errorOnRequiresAction: bool, ?description: string, ?customer: string, ?confirmationMethod: Create'ConfirmationMethod, ?confirm: bool, ?captureMethod: Create'CaptureMethod, ?applicationFeeAmount: int, ?transferGroup: string, ?useStripeSdk: bool) =
                {
                    Amount = amount
                    ApplicationFeeAmount = applicationFeeAmount
                    CaptureMethod = captureMethod
                    Confirm = confirm
                    ConfirmationMethod = confirmationMethod
                    Currency = currency
                    Customer = customer
                    Description = description
                    ErrorOnRequiresAction = errorOnRequiresAction
                    Expand = expand
                    Mandate = mandate
                    MandateData = mandateData
                    Metadata = metadata
                    OffSession = offSession
                    OnBehalfOf = onBehalfOf
                    PaymentMethod = paymentMethod
                    PaymentMethodData = paymentMethodData
                    PaymentMethodOptions = paymentMethodOptions
                    PaymentMethodTypes = paymentMethodTypes
                    ReceiptEmail = receiptEmail
                    ReturnUrl = returnUrl
                    SetupFutureUsage = setupFutureUsage
                    Shipping = shipping
                    StatementDescriptor = statementDescriptor
                    StatementDescriptorSuffix = statementDescriptorSuffix
                    TransferData = transferData
                    TransferGroup = transferGroup
                    UseStripeSdk = useStripeSdk
                }

        ///<p>Creates a PaymentIntent object.
        ///After the PaymentIntent is created, attach a payment method and <a href="/docs/api/payment_intents/confirm">confirm</a>
        ///to continue the payment. You can read more about the different payment flows
        ///available via the Payment Intents API <a href="/docs/payments/payment-intents">here</a>.
        ///When <code>confirm=true</code> is used during creation, it is equivalent to creating
        ///and confirming the PaymentIntent in the same call. You may use any parameters
        ///available in the <a href="/docs/api/payment_intents/confirm">confirm API</a> when <code>confirm=true</code>
        ///is supplied.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/payment_intents"
            |> RestApi.postAsync<_, PaymentIntent> settings (Map.empty) options

        type RetrieveOptions = {
            ///The client secret of the PaymentIntent. Required if a publishable key is used to retrieve the source.
            [<Config.Query>]ClientSecret: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Intent: string
        }
        with
            static member New(intent: string, ?expand: string list, ?clientSecret: string) =
                {
                    ClientSecret = clientSecret
                    Expand = expand
                    Intent = intent
                }

        ///<p>Retrieves the details of a PaymentIntent that has previously been created. 
        ///Client-side retrieval using a publishable key is allowed when the <code>client_secret</code> is provided in the query string. 
        ///When retrieved with a publishable key, only a subset of properties will be returned. Please refer to the <a href="#payment_intent_object">payment intent</a> object reference for more details.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("client_secret", options.ClientSecret |> box); ("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/payment_intents/{options.Intent}"
            |> RestApi.getAsync<PaymentIntent> settings qs

        type Update'PaymentMethodDataAcssDebit = {
            ///Customer's bank account number.
            [<Config.Form>]AccountNumber: string option
            ///Institution number of the customer's bank.
            [<Config.Form>]InstitutionNumber: string option
            ///Transit number of the customer's bank.
            [<Config.Form>]TransitNumber: string option
        }
        with
            static member New(?accountNumber: string, ?institutionNumber: string, ?transitNumber: string) =
                {
                    AccountNumber = accountNumber
                    InstitutionNumber = institutionNumber
                    TransitNumber = transitNumber
                }

        type Update'PaymentMethodDataAuBecsDebit = {
            ///The account number for the bank account.
            [<Config.Form>]AccountNumber: string option
            ///Bank-State-Branch number of the bank account.
            [<Config.Form>]BsbNumber: string option
        }
        with
            static member New(?accountNumber: string, ?bsbNumber: string) =
                {
                    AccountNumber = accountNumber
                    BsbNumber = bsbNumber
                }

        type Update'PaymentMethodDataBacsDebit = {
            ///Account number of the bank account that the funds will be debited from.
            [<Config.Form>]AccountNumber: string option
            ///Sort code of the bank account. (e.g., `10-20-30`)
            [<Config.Form>]SortCode: string option
        }
        with
            static member New(?accountNumber: string, ?sortCode: string) =
                {
                    AccountNumber = accountNumber
                    SortCode = sortCode
                }

        type Update'PaymentMethodDataBillingDetailsAddressBillingDetailsAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'PaymentMethodDataBillingDetails = {
            ///Billing address.
            [<Config.Form>]Address: Choice<Update'PaymentMethodDataBillingDetailsAddressBillingDetailsAddress,string> option
            ///Email address.
            [<Config.Form>]Email: string option
            ///Full name.
            [<Config.Form>]Name: string option
            ///Billing phone number (including extension).
            [<Config.Form>]Phone: string option
        }
        with
            static member New(?address: Choice<Update'PaymentMethodDataBillingDetailsAddressBillingDetailsAddress,string>, ?email: string, ?name: string, ?phone: string) =
                {
                    Address = address
                    Email = email
                    Name = name
                    Phone = phone
                }

        type Update'PaymentMethodDataBoleto = {
            ///Uniquely identifies this customer tax_id (CNPJ or CPF)
            [<Config.Form>]TaxId: string option
        }
        with
            static member New(?taxId: string) =
                {
                    TaxId = taxId
                }

        type Update'PaymentMethodDataEpsBank =
        | ArzteUndApothekerBank
        | AustrianAnadiBankAg
        | BankAustria
        | BankhausCarlSpangler
        | BankhausSchelhammerUndSchatteraAg
        | BawagPskAg
        | BksBankAg
        | BrullKallmusBankAg
        | BtvVierLanderBank
        | CapitalBankGraweGruppeAg
        | Dolomitenbank
        | EasybankAg
        | ErsteBankUndSparkassen
        | HypoAlpeadriabankInternationalAg
        | HypoBankBurgenlandAktiengesellschaft
        | HypoNoeLbFurNiederosterreichUWien
        | HypoOberosterreichSalzburgSteiermark
        | HypoTirolBankAg
        | HypoVorarlbergBankAg
        | MarchfelderBank
        | OberbankAg
        | RaiffeisenBankengruppeOsterreich
        | SchoellerbankAg
        | SpardaBankWien
        | VolksbankGruppe
        | VolkskreditbankAg
        | VrBankBraunau

        type Update'PaymentMethodDataEps = {
            ///The customer's bank.
            [<Config.Form>]Bank: Update'PaymentMethodDataEpsBank option
        }
        with
            static member New(?bank: Update'PaymentMethodDataEpsBank) =
                {
                    Bank = bank
                }

        type Update'PaymentMethodDataFpxAccountHolderType =
        | Company
        | Individual

        type Update'PaymentMethodDataFpxBank =
        | AffinBank
        | AllianceBank
        | Ambank
        | BankIslam
        | BankMuamalat
        | BankRakyat
        | Bsn
        | Cimb
        | DeutscheBank
        | HongLeongBank
        | Hsbc
        | Kfh
        | Maybank2e
        | Maybank2u
        | Ocbc
        | PbEnterprise
        | PublicBank
        | Rhb
        | StandardChartered
        | Uob

        type Update'PaymentMethodDataFpx = {
            ///Account holder type for FPX transaction
            [<Config.Form>]AccountHolderType: Update'PaymentMethodDataFpxAccountHolderType option
            ///The customer's bank.
            [<Config.Form>]Bank: Update'PaymentMethodDataFpxBank option
        }
        with
            static member New(?accountHolderType: Update'PaymentMethodDataFpxAccountHolderType, ?bank: Update'PaymentMethodDataFpxBank) =
                {
                    AccountHolderType = accountHolderType
                    Bank = bank
                }

        type Update'PaymentMethodDataIdealBank =
        | AbnAmro
        | AsnBank
        | Bunq
        | Handelsbanken
        | Ing
        | Knab
        | Moneyou
        | Rabobank
        | Regiobank
        | Revolut
        | SnsBank
        | TriodosBank
        | VanLanschot

        type Update'PaymentMethodDataIdeal = {
            ///The customer's bank.
            [<Config.Form>]Bank: Update'PaymentMethodDataIdealBank option
        }
        with
            static member New(?bank: Update'PaymentMethodDataIdealBank) =
                {
                    Bank = bank
                }

        type Update'PaymentMethodDataP24Bank =
        | AliorBank
        | BankMillennium
        | BankNowyBfgSa
        | BankPekaoSa
        | BankiSpbdzielcze
        | Blik
        | BnpParibas
        | Boz
        | CitiHandlowy
        | CreditAgricole
        | Envelobank
        | EtransferPocztowy24
        | GetinBank
        | Ideabank
        | Ing
        | Inteligo
        | MbankMtransfer
        | NestPrzelew
        | NoblePay
        | PbacZIpko
        | PlusBank
        | SantanderPrzelew24
        | TmobileUsbugiBankowe
        | ToyotaBank
        | VolkswagenBank

        type Update'PaymentMethodDataP24 = {
            ///The customer's bank.
            [<Config.Form>]Bank: Update'PaymentMethodDataP24Bank option
        }
        with
            static member New(?bank: Update'PaymentMethodDataP24Bank) =
                {
                    Bank = bank
                }

        type Update'PaymentMethodDataSepaDebit = {
            ///IBAN of the bank account.
            [<Config.Form>]Iban: string option
        }
        with
            static member New(?iban: string) =
                {
                    Iban = iban
                }

        type Update'PaymentMethodDataSofortCountry =
        | [<JsonUnionCase("AT")>] AT
        | [<JsonUnionCase("BE")>] BE
        | [<JsonUnionCase("DE")>] DE
        | [<JsonUnionCase("ES")>] ES
        | [<JsonUnionCase("IT")>] IT
        | [<JsonUnionCase("NL")>] NL

        type Update'PaymentMethodDataSofort = {
            ///Two-letter ISO code representing the country the bank account is located in.
            [<Config.Form>]Country: Update'PaymentMethodDataSofortCountry option
        }
        with
            static member New(?country: Update'PaymentMethodDataSofortCountry) =
                {
                    Country = country
                }

        type Update'PaymentMethodDataType =
        | AcssDebit
        | AfterpayClearpay
        | Alipay
        | AuBecsDebit
        | BacsDebit
        | Bancontact
        | Boleto
        | Eps
        | Fpx
        | Giropay
        | Grabpay
        | Ideal
        | Oxxo
        | P24
        | SepaDebit
        | Sofort
        | WechatPay

        type Update'PaymentMethodData = {
            ///If this is an `acss_debit` PaymentMethod, this hash contains details about the ACSS Debit payment method.
            [<Config.Form>]AcssDebit: Update'PaymentMethodDataAcssDebit option
            ///If this is an `AfterpayClearpay` PaymentMethod, this hash contains details about the AfterpayClearpay payment method.
            [<Config.Form>]AfterpayClearpay: string option
            ///If this is an `Alipay` PaymentMethod, this hash contains details about the Alipay payment method.
            [<Config.Form>]Alipay: string option
            ///If this is an `au_becs_debit` PaymentMethod, this hash contains details about the bank account.
            [<Config.Form>]AuBecsDebit: Update'PaymentMethodDataAuBecsDebit option
            ///If this is a `bacs_debit` PaymentMethod, this hash contains details about the Bacs Direct Debit bank account.
            [<Config.Form>]BacsDebit: Update'PaymentMethodDataBacsDebit option
            ///If this is a `bancontact` PaymentMethod, this hash contains details about the Bancontact payment method.
            [<Config.Form>]Bancontact: string option
            ///Billing information associated with the PaymentMethod that may be used or required by particular types of payment methods.
            [<Config.Form>]BillingDetails: Update'PaymentMethodDataBillingDetails option
            ///If this is a `boleto` PaymentMethod, this hash contains details about the Boleto payment method.
            [<Config.Form>]Boleto: Update'PaymentMethodDataBoleto option
            ///If this is an `eps` PaymentMethod, this hash contains details about the EPS payment method.
            [<Config.Form>]Eps: Update'PaymentMethodDataEps option
            ///If this is an `fpx` PaymentMethod, this hash contains details about the FPX payment method.
            [<Config.Form>]Fpx: Update'PaymentMethodDataFpx option
            ///If this is a `giropay` PaymentMethod, this hash contains details about the Giropay payment method.
            [<Config.Form>]Giropay: string option
            ///If this is a `grabpay` PaymentMethod, this hash contains details about the GrabPay payment method.
            [<Config.Form>]Grabpay: string option
            ///If this is an `ideal` PaymentMethod, this hash contains details about the iDEAL payment method.
            [<Config.Form>]Ideal: Update'PaymentMethodDataIdeal option
            ///If this is an `interac_present` PaymentMethod, this hash contains details about the Interac Present payment method.
            [<Config.Form>]InteracPresent: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///If this is an `oxxo` PaymentMethod, this hash contains details about the OXXO payment method.
            [<Config.Form>]Oxxo: string option
            ///If this is a `p24` PaymentMethod, this hash contains details about the P24 payment method.
            [<Config.Form>]P24: Update'PaymentMethodDataP24 option
            ///If this is a `sepa_debit` PaymentMethod, this hash contains details about the SEPA debit bank account.
            [<Config.Form>]SepaDebit: Update'PaymentMethodDataSepaDebit option
            ///If this is a `sofort` PaymentMethod, this hash contains details about the SOFORT payment method.
            [<Config.Form>]Sofort: Update'PaymentMethodDataSofort option
            ///The type of the PaymentMethod. An additional hash is included on the PaymentMethod with a name matching this value. It contains additional information specific to the PaymentMethod type.
            [<Config.Form>]Type: Update'PaymentMethodDataType option
            ///If this is an `wechat_pay` PaymentMethod, this hash contains details about the wechat_pay payment method.
            [<Config.Form>]WechatPay: string option
        }
        with
            static member New(?acssDebit: Update'PaymentMethodDataAcssDebit, ?sofort: Update'PaymentMethodDataSofort, ?sepaDebit: Update'PaymentMethodDataSepaDebit, ?p24: Update'PaymentMethodDataP24, ?oxxo: string, ?metadata: Map<string, string>, ?interacPresent: string, ?ideal: Update'PaymentMethodDataIdeal, ?grabpay: string, ?type': Update'PaymentMethodDataType, ?giropay: string, ?eps: Update'PaymentMethodDataEps, ?boleto: Update'PaymentMethodDataBoleto, ?billingDetails: Update'PaymentMethodDataBillingDetails, ?bancontact: string, ?bacsDebit: Update'PaymentMethodDataBacsDebit, ?auBecsDebit: Update'PaymentMethodDataAuBecsDebit, ?alipay: string, ?afterpayClearpay: string, ?fpx: Update'PaymentMethodDataFpx, ?wechatPay: string) =
                {
                    AcssDebit = acssDebit
                    AfterpayClearpay = afterpayClearpay
                    Alipay = alipay
                    AuBecsDebit = auBecsDebit
                    BacsDebit = bacsDebit
                    Bancontact = bancontact
                    BillingDetails = billingDetails
                    Boleto = boleto
                    Eps = eps
                    Fpx = fpx
                    Giropay = giropay
                    Grabpay = grabpay
                    Ideal = ideal
                    InteracPresent = interacPresent
                    Metadata = metadata
                    Oxxo = oxxo
                    P24 = p24
                    SepaDebit = sepaDebit
                    Sofort = sofort
                    Type = type'
                    WechatPay = wechatPay
                }

        type Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsPaymentSchedule =
        | Combined
        | Interval
        | Sporadic

        type Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsTransactionType =
        | Business
        | Personal

        type Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptions = {
            ///A URL for custom mandate text to render during confirmation step.
            ///The URL will be rendered with additional GET parameters `payment_intent` and `payment_intent_client_secret` when confirming a Payment Intent,
            ///or `setup_intent` and `setup_intent_client_secret` when confirming a Setup Intent.
            [<Config.Form>]CustomMandateUrl: Choice<string,string> option
            ///Description of the mandate interval. Only required if 'payment_schedule' parameter is 'interval' or 'combined'.
            [<Config.Form>]IntervalDescription: string option
            ///Payment schedule for the mandate.
            [<Config.Form>]PaymentSchedule: Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsPaymentSchedule option
            ///Transaction type of the mandate.
            [<Config.Form>]TransactionType: Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsTransactionType option
        }
        with
            static member New(?customMandateUrl: Choice<string,string>, ?intervalDescription: string, ?paymentSchedule: Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsPaymentSchedule, ?transactionType: Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsTransactionType) =
                {
                    CustomMandateUrl = customMandateUrl
                    IntervalDescription = intervalDescription
                    PaymentSchedule = paymentSchedule
                    TransactionType = transactionType
                }

        type Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsVerificationMethod =
        | Automatic
        | Instant
        | Microdeposits

        type Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptions = {
            ///Additional fields for Mandate creation
            [<Config.Form>]MandateOptions: Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptions option
            ///Verification method for the intent
            [<Config.Form>]VerificationMethod: Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsVerificationMethod option
        }
        with
            static member New(?mandateOptions: Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptions, ?verificationMethod: Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsVerificationMethod) =
                {
                    MandateOptions = mandateOptions
                    VerificationMethod = verificationMethod
                }

        type Update'PaymentMethodOptionsAfterpayClearpayPaymentMethodOptions = {
            ///Order identifier shown to the customer in Afterpay’s online portal. We recommend using a value that helps you answer any questions a customer might have about
            ///the payment. The identifier is limited to 128 characters and may contain only letters, digits, underscores, backslashes and dashes.
            [<Config.Form>]Reference: string option
        }
        with
            static member New(?reference: string) =
                {
                    Reference = reference
                }

        type Update'PaymentMethodOptionsBancontactPaymentMethodOptionsPreferredLanguage =
        | De
        | En
        | Fr
        | Nl

        type Update'PaymentMethodOptionsBancontactPaymentMethodOptions = {
            ///Preferred language of the Bancontact authorization page that the customer is redirected to.
            [<Config.Form>]PreferredLanguage: Update'PaymentMethodOptionsBancontactPaymentMethodOptionsPreferredLanguage option
        }
        with
            static member New(?preferredLanguage: Update'PaymentMethodOptionsBancontactPaymentMethodOptionsPreferredLanguage) =
                {
                    PreferredLanguage = preferredLanguage
                }

        type Update'PaymentMethodOptionsBoletoPaymentMethodOptions = {
            ///The number of calendar days before a Boleto voucher expires. For example, if you create a Boleto voucher on Monday and you set expires_after_days to 2, the Boleto invoice will expire on Wednesday at 23:59 America/Sao_Paulo time.
            [<Config.Form>]ExpiresAfterDays: int option
        }
        with
            static member New(?expiresAfterDays: int) =
                {
                    ExpiresAfterDays = expiresAfterDays
                }

        type Update'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanInterval =
        | Month

        type Update'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanType =
        | FixedCount

        type Update'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlan = {
            ///For `fixed_count` installment plans, this is the number of installment payments your customer will make to their credit card.
            [<Config.Form>]Count: int option
            ///For `fixed_count` installment plans, this is the interval between installment payments your customer will make to their credit card.
            ///One of `month`.
            [<Config.Form>]Interval: Update'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanInterval option
            ///Type of installment plan, one of `fixed_count`.
            [<Config.Form>]Type: Update'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanType option
        }
        with
            static member New(?count: int, ?interval: Update'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanInterval, ?type': Update'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanType) =
                {
                    Count = count
                    Interval = interval
                    Type = type'
                }

        type Update'PaymentMethodOptionsCardPaymentIntentInstallments = {
            ///Setting to true enables installments for this PaymentIntent.
            ///This will cause the response to contain a list of available installment plans.
            ///Setting to false will prevent any selected plan from applying to a charge.
            [<Config.Form>]Enabled: bool option
            ///The selected installment plan to use for this payment attempt.
            ///This parameter can only be provided during confirmation.
            [<Config.Form>]Plan: Choice<Update'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlan,string> option
        }
        with
            static member New(?enabled: bool, ?plan: Choice<Update'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlan,string>) =
                {
                    Enabled = enabled
                    Plan = plan
                }

        type Update'PaymentMethodOptionsCardPaymentIntentNetwork =
        | Amex
        | CartesBancaires
        | Diners
        | Discover
        | Interac
        | Jcb
        | Mastercard
        | Unionpay
        | Unknown
        | Visa

        type Update'PaymentMethodOptionsCardPaymentIntentRequestThreeDSecure =
        | Any
        | Automatic

        type Update'PaymentMethodOptionsCardPaymentIntent = {
            ///A single-use `cvc_update` Token that represents a card CVC value. When provided, the CVC value will be verified during the card payment attempt. This parameter can only be provided during confirmation.
            [<Config.Form>]CvcToken: string option
            ///Installment configuration for payments attempted on this PaymentIntent (Mexico Only).
            ///For more information, see the [installments integration guide](https://stripe.com/docs/payments/installments).
            [<Config.Form>]Installments: Update'PaymentMethodOptionsCardPaymentIntentInstallments option
            ///When specified, this parameter indicates that a transaction will be marked
            ///as MOTO (Mail Order Telephone Order) and thus out of scope for SCA. This
            ///parameter can only be provided during confirmation.
            [<Config.Form>]Moto: bool option
            ///Selected network to process this PaymentIntent on. Depends on the available networks of the card attached to the PaymentIntent. Can be only set confirm-time.
            [<Config.Form>]Network: Update'PaymentMethodOptionsCardPaymentIntentNetwork option
            ///We strongly recommend that you rely on our SCA Engine to automatically prompt your customers for authentication based on risk level and [other requirements](https://stripe.com/docs/strong-customer-authentication). However, if you wish to request 3D Secure based on logic from your own fraud engine, provide this option. Permitted values include: `automatic` or `any`. If not provided, defaults to `automatic`. Read our guide on [manually requesting 3D Secure](https://stripe.com/docs/payments/3d-secure#manual-three-ds) for more information on how this configuration interacts with Radar and our SCA Engine.
            [<Config.Form>]RequestThreeDSecure: Update'PaymentMethodOptionsCardPaymentIntentRequestThreeDSecure option
        }
        with
            static member New(?cvcToken: string, ?installments: Update'PaymentMethodOptionsCardPaymentIntentInstallments, ?moto: bool, ?network: Update'PaymentMethodOptionsCardPaymentIntentNetwork, ?requestThreeDSecure: Update'PaymentMethodOptionsCardPaymentIntentRequestThreeDSecure) =
                {
                    CvcToken = cvcToken
                    Installments = installments
                    Moto = moto
                    Network = network
                    RequestThreeDSecure = requestThreeDSecure
                }

        type Update'PaymentMethodOptionsOxxoPaymentMethodOptions = {
            ///The number of calendar days before an OXXO voucher expires. For example, if you create an OXXO voucher on Monday and you set expires_after_days to 2, the OXXO invoice will expire on Wednesday at 23:59 America/Mexico_City time.
            [<Config.Form>]ExpiresAfterDays: int option
        }
        with
            static member New(?expiresAfterDays: int) =
                {
                    ExpiresAfterDays = expiresAfterDays
                }

        type Update'PaymentMethodOptionsP24PaymentMethodOptions = {
            ///Confirm that the payer has accepted the P24 terms and conditions.
            [<Config.Form>]TosShownAndAccepted: bool option
        }
        with
            static member New(?tosShownAndAccepted: bool) =
                {
                    TosShownAndAccepted = tosShownAndAccepted
                }

        type Update'PaymentMethodOptionsSepaDebitPaymentIntentPaymentMethodOptions = {
            ///Additional fields for Mandate creation
            [<Config.Form>]MandateOptions: string option
        }
        with
            static member New(?mandateOptions: string) =
                {
                    MandateOptions = mandateOptions
                }

        type Update'PaymentMethodOptionsSofortPaymentMethodOptionsPreferredLanguage =
        | De
        | En
        | Es
        | Fr
        | It
        | Nl
        | Pl

        type Update'PaymentMethodOptionsSofortPaymentMethodOptions = {
            ///Language shown to the payer on redirect.
            [<Config.Form>]PreferredLanguage: Update'PaymentMethodOptionsSofortPaymentMethodOptionsPreferredLanguage option
        }
        with
            static member New(?preferredLanguage: Update'PaymentMethodOptionsSofortPaymentMethodOptionsPreferredLanguage) =
                {
                    PreferredLanguage = preferredLanguage
                }

        type Update'PaymentMethodOptionsWechatPayPaymentMethodOptionsClient =
        | Android
        | Ios
        | Web

        type Update'PaymentMethodOptionsWechatPayPaymentMethodOptions = {
            ///The app ID registered with WeChat Pay. Only required when client is ios or android.
            [<Config.Form>]AppId: string option
            ///The client type that the end customer will pay from
            [<Config.Form>]Client: Update'PaymentMethodOptionsWechatPayPaymentMethodOptionsClient option
        }
        with
            static member New(?appId: string, ?client: Update'PaymentMethodOptionsWechatPayPaymentMethodOptionsClient) =
                {
                    AppId = appId
                    Client = client
                }

        type Update'PaymentMethodOptions = {
            ///If this is a `acss_debit` PaymentMethod, this sub-hash contains details about the ACSS Debit payment method options.
            [<Config.Form>]AcssDebit: Choice<Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptions,string> option
            ///If this is a `afterpay_clearpay` PaymentMethod, this sub-hash contains details about the Afterpay Clearpay payment method options.
            [<Config.Form>]AfterpayClearpay: Choice<Update'PaymentMethodOptionsAfterpayClearpayPaymentMethodOptions,string> option
            ///If this is a `alipay` PaymentMethod, this sub-hash contains details about the Alipay payment method options.
            [<Config.Form>]Alipay: Choice<string,string> option
            ///If this is a `bancontact` PaymentMethod, this sub-hash contains details about the Bancontact payment method options.
            [<Config.Form>]Bancontact: Choice<Update'PaymentMethodOptionsBancontactPaymentMethodOptions,string> option
            ///If this is a `boleto` PaymentMethod, this sub-hash contains details about the Boleto payment method options.
            [<Config.Form>]Boleto: Choice<Update'PaymentMethodOptionsBoletoPaymentMethodOptions,string> option
            ///Configuration for any card payments attempted on this PaymentIntent.
            [<Config.Form>]Card: Choice<Update'PaymentMethodOptionsCardPaymentIntent,string> option
            ///If this is a `card_present` PaymentMethod, this sub-hash contains details about the Card Present payment method options.
            [<Config.Form>]CardPresent: Choice<string,string> option
            ///If this is a `oxxo` PaymentMethod, this sub-hash contains details about the OXXO payment method options.
            [<Config.Form>]Oxxo: Choice<Update'PaymentMethodOptionsOxxoPaymentMethodOptions,string> option
            ///If this is a `p24` PaymentMethod, this sub-hash contains details about the Przelewy24 payment method options.
            [<Config.Form>]P24: Choice<Update'PaymentMethodOptionsP24PaymentMethodOptions,string> option
            ///If this is a `sepa_debit` PaymentIntent, this sub-hash contains details about the SEPA Debit payment method options.
            [<Config.Form>]SepaDebit: Choice<Update'PaymentMethodOptionsSepaDebitPaymentIntentPaymentMethodOptions,string> option
            ///If this is a `sofort` PaymentMethod, this sub-hash contains details about the SOFORT payment method options.
            [<Config.Form>]Sofort: Choice<Update'PaymentMethodOptionsSofortPaymentMethodOptions,string> option
            ///If this is a `wechat_pay` PaymentMethod, this sub-hash contains details about the WeChat Pay payment method options.
            [<Config.Form>]WechatPay: Choice<Update'PaymentMethodOptionsWechatPayPaymentMethodOptions,string> option
        }
        with
            static member New(?acssDebit: Choice<Update'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptions,string>, ?afterpayClearpay: Choice<Update'PaymentMethodOptionsAfterpayClearpayPaymentMethodOptions,string>, ?alipay: Choice<string,string>, ?bancontact: Choice<Update'PaymentMethodOptionsBancontactPaymentMethodOptions,string>, ?boleto: Choice<Update'PaymentMethodOptionsBoletoPaymentMethodOptions,string>, ?card: Choice<Update'PaymentMethodOptionsCardPaymentIntent,string>, ?cardPresent: Choice<string,string>, ?oxxo: Choice<Update'PaymentMethodOptionsOxxoPaymentMethodOptions,string>, ?p24: Choice<Update'PaymentMethodOptionsP24PaymentMethodOptions,string>, ?sepaDebit: Choice<Update'PaymentMethodOptionsSepaDebitPaymentIntentPaymentMethodOptions,string>, ?sofort: Choice<Update'PaymentMethodOptionsSofortPaymentMethodOptions,string>, ?wechatPay: Choice<Update'PaymentMethodOptionsWechatPayPaymentMethodOptions,string>) =
                {
                    AcssDebit = acssDebit
                    AfterpayClearpay = afterpayClearpay
                    Alipay = alipay
                    Bancontact = bancontact
                    Boleto = boleto
                    Card = card
                    CardPresent = cardPresent
                    Oxxo = oxxo
                    P24 = p24
                    SepaDebit = sepaDebit
                    Sofort = sofort
                    WechatPay = wechatPay
                }

        type Update'ShippingShippingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'ShippingShipping = {
            ///Shipping address.
            [<Config.Form>]Address: Update'ShippingShippingAddress option
            ///The delivery service that shipped a physical product, such as Fedex, UPS, USPS, etc.
            [<Config.Form>]Carrier: string option
            ///Recipient name.
            [<Config.Form>]Name: string option
            ///Recipient phone (including extension).
            [<Config.Form>]Phone: string option
            ///The tracking number for a physical product, obtained from the delivery service. If multiple tracking numbers were generated for this purchase, please separate them with commas.
            [<Config.Form>]TrackingNumber: string option
        }
        with
            static member New(?address: Update'ShippingShippingAddress, ?carrier: string, ?name: string, ?phone: string, ?trackingNumber: string) =
                {
                    Address = address
                    Carrier = carrier
                    Name = name
                    Phone = phone
                    TrackingNumber = trackingNumber
                }

        type Update'TransferData = {
            ///The amount that will be transferred automatically when a charge succeeds.
            [<Config.Form>]Amount: int option
        }
        with
            static member New(?amount: int) =
                {
                    Amount = amount
                }

        type Update'SetupFutureUsage =
        | OffSession
        | OnSession

        type UpdateOptions = {
            [<Config.Path>]Intent: string
            ///Amount intended to be collected by this PaymentIntent. A positive integer representing how much to charge in the [smallest currency unit](https://stripe.com/docs/currencies#zero-decimal) (e.g., 100 cents to charge $1.00 or 100 to charge ¥100, a zero-decimal currency). The minimum amount is $0.50 US or [equivalent in charge currency](https://stripe.com/docs/currencies#minimum-and-maximum-charge-amounts). The amount value supports up to eight digits (e.g., a value of 99999999 for a USD charge of $999,999.99).
            [<Config.Form>]Amount: int option
            ///The amount of the application fee (if any) that will be requested to be applied to the payment and transferred to the application owner's Stripe account. The amount of the application fee collected will be capped at the total payment amount. For more information, see the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts).
            [<Config.Form>]ApplicationFeeAmount: Choice<int,string> option
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string option
            ///ID of the Customer this PaymentIntent belongs to, if one exists.
            ///Payment methods attached to other Customers cannot be used with this PaymentIntent.
            ///If present in combination with [setup_future_usage](https://stripe.com/docs/api#payment_intent_object-setup_future_usage), this PaymentIntent's payment method will be attached to the Customer after the PaymentIntent has been confirmed and any required actions from the user are complete.
            [<Config.Form>]Customer: string option
            ///An arbitrary string attached to the object. Often useful for displaying to users.
            [<Config.Form>]Description: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///ID of the payment method (a PaymentMethod, Card, or [compatible Source](https://stripe.com/docs/payments/payment-methods/transitioning#compatibility) object) to attach to this PaymentIntent.
            [<Config.Form>]PaymentMethod: string option
            ///If provided, this hash will be used to create a PaymentMethod. The new PaymentMethod will appear
            ///in the [payment_method](https://stripe.com/docs/api/payment_intents/object#payment_intent_object-payment_method)
            ///property on the PaymentIntent.
            [<Config.Form>]PaymentMethodData: Update'PaymentMethodData option
            ///Payment-method-specific configuration for this PaymentIntent.
            [<Config.Form>]PaymentMethodOptions: Update'PaymentMethodOptions option
            ///The list of payment method types (e.g. card) that this PaymentIntent is allowed to use.
            [<Config.Form>]PaymentMethodTypes: string list option
            ///Email address that the receipt for the resulting payment will be sent to. If `receipt_email` is specified for a payment in live mode, a receipt will be sent regardless of your [email settings](https://dashboard.stripe.com/account/emails).
            [<Config.Form>]ReceiptEmail: Choice<string,string> option
            ///Indicates that you intend to make future payments with this PaymentIntent's payment method.
            ///Providing this parameter will [attach the payment method](https://stripe.com/docs/payments/save-during-payment) to the PaymentIntent's Customer, if present, after the PaymentIntent is confirmed and any required actions from the user are complete. If no Customer was provided, the payment method can still be [attached](https://stripe.com/docs/api/payment_methods/attach) to a Customer after the transaction completes.
            ///When processing card payments, Stripe also uses `setup_future_usage` to dynamically optimize your payment flow and comply with regional legislation and network rules, such as [SCA](https://stripe.com/docs/strong-customer-authentication).
            ///If `setup_future_usage` is already set and you are performing a request using a publishable key, you may only update the value from `on_session` to `off_session`.
            [<Config.Form>]SetupFutureUsage: Update'SetupFutureUsage option
            ///Shipping information for this PaymentIntent.
            [<Config.Form>]Shipping: Choice<Update'ShippingShipping,string> option
            ///For non-card charges, you can use this value as the complete description that appears on your customers’ statements. Must contain at least one letter, maximum 22 characters.
            [<Config.Form>]StatementDescriptor: string option
            ///Provides information about a card payment that customers see on their statements. Concatenated with the prefix (shortened descriptor) or statement descriptor that’s set on the account to form the complete statement descriptor. Maximum 22 characters for the concatenated descriptor.
            [<Config.Form>]StatementDescriptorSuffix: string option
            ///The parameters used to automatically create a Transfer when the payment succeeds. For more information, see the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts).
            [<Config.Form>]TransferData: Update'TransferData option
            ///A string that identifies the resulting payment as part of a group. `transfer_group` may only be provided if it has not been set. See the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts) for details.
            [<Config.Form>]TransferGroup: string option
        }
        with
            static member New(intent: string, ?statementDescriptorSuffix: string, ?statementDescriptor: string, ?shipping: Choice<Update'ShippingShipping,string>, ?setupFutureUsage: Update'SetupFutureUsage, ?receiptEmail: Choice<string,string>, ?paymentMethodTypes: string list, ?paymentMethodOptions: Update'PaymentMethodOptions, ?transferData: Update'TransferData, ?paymentMethodData: Update'PaymentMethodData, ?metadata: Map<string, string>, ?expand: string list, ?description: string, ?customer: string, ?currency: string, ?applicationFeeAmount: Choice<int,string>, ?amount: int, ?paymentMethod: string, ?transferGroup: string) =
                {
                    Intent = intent
                    Amount = amount
                    ApplicationFeeAmount = applicationFeeAmount
                    Currency = currency
                    Customer = customer
                    Description = description
                    Expand = expand
                    Metadata = metadata
                    PaymentMethod = paymentMethod
                    PaymentMethodData = paymentMethodData
                    PaymentMethodOptions = paymentMethodOptions
                    PaymentMethodTypes = paymentMethodTypes
                    ReceiptEmail = receiptEmail
                    SetupFutureUsage = setupFutureUsage
                    Shipping = shipping
                    StatementDescriptor = statementDescriptor
                    StatementDescriptorSuffix = statementDescriptorSuffix
                    TransferData = transferData
                    TransferGroup = transferGroup
                }

        ///<p>Updates properties on a PaymentIntent object without confirming.
        ///Depending on which properties you update, you may need to confirm the
        ///PaymentIntent again. For example, updating the <code>payment_method</code> will
        ///always require you to confirm the PaymentIntent again. If you prefer to
        ///update and confirm at the same time, we recommend updating properties via
        ///the <a href="/docs/api/payment_intents/confirm">confirm API</a> instead.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/payment_intents/{options.Intent}"
            |> RestApi.postAsync<_, PaymentIntent> settings (Map.empty) options

    module PaymentIntentsCancel =

        type Cancel'CancellationReason =
        | Abandoned
        | Duplicate
        | Fraudulent
        | RequestedByCustomer

        type CancelOptions = {
            [<Config.Path>]Intent: string
            ///Reason for canceling this PaymentIntent. Possible values are `duplicate`, `fraudulent`, `requested_by_customer`, or `abandoned`
            [<Config.Form>]CancellationReason: Cancel'CancellationReason option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(intent: string, ?cancellationReason: Cancel'CancellationReason, ?expand: string list) =
                {
                    Intent = intent
                    CancellationReason = cancellationReason
                    Expand = expand
                }

        ///<p>A PaymentIntent object can be canceled when it is in one of these statuses: <code>requires_payment_method</code>, <code>requires_capture</code>, <code>requires_confirmation</code>, or <code>requires_action</code>. 
        ///Once canceled, no additional charges will be made by the PaymentIntent and any operations on the PaymentIntent will fail with an error. For PaymentIntents with <code>status=’requires_capture’</code>, the remaining <code>amount_capturable</code> will automatically be refunded.</p>
        let Cancel settings (options: CancelOptions) =
            $"/v1/payment_intents/{options.Intent}/cancel"
            |> RestApi.postAsync<_, PaymentIntent> settings (Map.empty) options

    module PaymentIntentsCapture =

        type Capture'TransferData = {
            ///The amount that will be transferred automatically when a charge succeeds.
            [<Config.Form>]Amount: int option
        }
        with
            static member New(?amount: int) =
                {
                    Amount = amount
                }

        type CaptureOptions = {
            [<Config.Path>]Intent: string
            ///The amount to capture from the PaymentIntent, which must be less than or equal to the original amount. Any additional amount will be automatically refunded. Defaults to the full `amount_capturable` if not provided.
            [<Config.Form>]AmountToCapture: int option
            ///The amount of the application fee (if any) that will be requested to be applied to the payment and transferred to the application owner's Stripe account. The amount of the application fee collected will be capped at the total payment amount. For more information, see the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts).
            [<Config.Form>]ApplicationFeeAmount: int option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///For non-card charges, you can use this value as the complete description that appears on your customers’ statements. Must contain at least one letter, maximum 22 characters.
            [<Config.Form>]StatementDescriptor: string option
            ///Provides information about a card payment that customers see on their statements. Concatenated with the prefix (shortened descriptor) or statement descriptor that’s set on the account to form the complete statement descriptor. Maximum 22 characters for the concatenated descriptor.
            [<Config.Form>]StatementDescriptorSuffix: string option
            ///The parameters used to automatically create a Transfer when the payment
            ///is captured. For more information, see the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts).
            [<Config.Form>]TransferData: Capture'TransferData option
        }
        with
            static member New(intent: string, ?amountToCapture: int, ?applicationFeeAmount: int, ?expand: string list, ?statementDescriptor: string, ?statementDescriptorSuffix: string, ?transferData: Capture'TransferData) =
                {
                    Intent = intent
                    AmountToCapture = amountToCapture
                    ApplicationFeeAmount = applicationFeeAmount
                    Expand = expand
                    StatementDescriptor = statementDescriptor
                    StatementDescriptorSuffix = statementDescriptorSuffix
                    TransferData = transferData
                }

        ///<p>Capture the funds of an existing uncaptured PaymentIntent when its status is <code>requires_capture</code>.
        ///Uncaptured PaymentIntents will be canceled exactly seven days after they are created.
        ///Learn more about <a href="/docs/payments/capture-later">separate authorization and capture</a>.</p>
        let Capture settings (options: CaptureOptions) =
            $"/v1/payment_intents/{options.Intent}/capture"
            |> RestApi.postAsync<_, PaymentIntent> settings (Map.empty) options

    module PaymentIntentsConfirm =

        type Confirm'MandateDataSecretKeyCustomerAcceptanceOnline = {
            ///The IP address from which the Mandate was accepted by the customer.
            [<Config.Form>]IpAddress: string option
            ///The user agent of the browser from which the Mandate was accepted by the customer.
            [<Config.Form>]UserAgent: string option
        }
        with
            static member New(?ipAddress: string, ?userAgent: string) =
                {
                    IpAddress = ipAddress
                    UserAgent = userAgent
                }

        type Confirm'MandateDataSecretKeyCustomerAcceptanceType =
        | Offline
        | Online

        type Confirm'MandateDataSecretKeyCustomerAcceptance = {
            ///The time at which the customer accepted the Mandate.
            [<Config.Form>]AcceptedAt: DateTime option
            ///If this is a Mandate accepted offline, this hash contains details about the offline acceptance.
            [<Config.Form>]Offline: string option
            ///If this is a Mandate accepted online, this hash contains details about the online acceptance.
            [<Config.Form>]Online: Confirm'MandateDataSecretKeyCustomerAcceptanceOnline option
            ///The type of customer acceptance information included with the Mandate. One of `online` or `offline`.
            [<Config.Form>]Type: Confirm'MandateDataSecretKeyCustomerAcceptanceType option
        }
        with
            static member New(?acceptedAt: DateTime, ?offline: string, ?online: Confirm'MandateDataSecretKeyCustomerAcceptanceOnline, ?type': Confirm'MandateDataSecretKeyCustomerAcceptanceType) =
                {
                    AcceptedAt = acceptedAt
                    Offline = offline
                    Online = online
                    Type = type'
                }

        type Confirm'MandateDataSecretKey = {
            ///This hash contains details about the customer acceptance of the Mandate.
            [<Config.Form>]CustomerAcceptance: Confirm'MandateDataSecretKeyCustomerAcceptance option
        }
        with
            static member New(?customerAcceptance: Confirm'MandateDataSecretKeyCustomerAcceptance) =
                {
                    CustomerAcceptance = customerAcceptance
                }

        type Confirm'MandateDataClientKeyCustomerAcceptanceOnline = {
            ///The IP address from which the Mandate was accepted by the customer.
            [<Config.Form>]IpAddress: string option
            ///The user agent of the browser from which the Mandate was accepted by the customer.
            [<Config.Form>]UserAgent: string option
        }
        with
            static member New(?ipAddress: string, ?userAgent: string) =
                {
                    IpAddress = ipAddress
                    UserAgent = userAgent
                }

        type Confirm'MandateDataClientKeyCustomerAcceptanceType =
        | Online

        type Confirm'MandateDataClientKeyCustomerAcceptance = {
            ///If this is a Mandate accepted online, this hash contains details about the online acceptance.
            [<Config.Form>]Online: Confirm'MandateDataClientKeyCustomerAcceptanceOnline option
            ///The type of customer acceptance information included with the Mandate.
            [<Config.Form>]Type: Confirm'MandateDataClientKeyCustomerAcceptanceType option
        }
        with
            static member New(?online: Confirm'MandateDataClientKeyCustomerAcceptanceOnline, ?type': Confirm'MandateDataClientKeyCustomerAcceptanceType) =
                {
                    Online = online
                    Type = type'
                }

        type Confirm'MandateDataClientKey = {
            ///This hash contains details about the customer acceptance of the Mandate.
            [<Config.Form>]CustomerAcceptance: Confirm'MandateDataClientKeyCustomerAcceptance option
        }
        with
            static member New(?customerAcceptance: Confirm'MandateDataClientKeyCustomerAcceptance) =
                {
                    CustomerAcceptance = customerAcceptance
                }

        type Confirm'OffSession =
        | OneOff
        | Recurring

        type Confirm'PaymentMethodDataAcssDebit = {
            ///Customer's bank account number.
            [<Config.Form>]AccountNumber: string option
            ///Institution number of the customer's bank.
            [<Config.Form>]InstitutionNumber: string option
            ///Transit number of the customer's bank.
            [<Config.Form>]TransitNumber: string option
        }
        with
            static member New(?accountNumber: string, ?institutionNumber: string, ?transitNumber: string) =
                {
                    AccountNumber = accountNumber
                    InstitutionNumber = institutionNumber
                    TransitNumber = transitNumber
                }

        type Confirm'PaymentMethodDataAuBecsDebit = {
            ///The account number for the bank account.
            [<Config.Form>]AccountNumber: string option
            ///Bank-State-Branch number of the bank account.
            [<Config.Form>]BsbNumber: string option
        }
        with
            static member New(?accountNumber: string, ?bsbNumber: string) =
                {
                    AccountNumber = accountNumber
                    BsbNumber = bsbNumber
                }

        type Confirm'PaymentMethodDataBacsDebit = {
            ///Account number of the bank account that the funds will be debited from.
            [<Config.Form>]AccountNumber: string option
            ///Sort code of the bank account. (e.g., `10-20-30`)
            [<Config.Form>]SortCode: string option
        }
        with
            static member New(?accountNumber: string, ?sortCode: string) =
                {
                    AccountNumber = accountNumber
                    SortCode = sortCode
                }

        type Confirm'PaymentMethodDataBillingDetailsAddressBillingDetailsAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Confirm'PaymentMethodDataBillingDetails = {
            ///Billing address.
            [<Config.Form>]Address: Choice<Confirm'PaymentMethodDataBillingDetailsAddressBillingDetailsAddress,string> option
            ///Email address.
            [<Config.Form>]Email: string option
            ///Full name.
            [<Config.Form>]Name: string option
            ///Billing phone number (including extension).
            [<Config.Form>]Phone: string option
        }
        with
            static member New(?address: Choice<Confirm'PaymentMethodDataBillingDetailsAddressBillingDetailsAddress,string>, ?email: string, ?name: string, ?phone: string) =
                {
                    Address = address
                    Email = email
                    Name = name
                    Phone = phone
                }

        type Confirm'PaymentMethodDataBoleto = {
            ///Uniquely identifies this customer tax_id (CNPJ or CPF)
            [<Config.Form>]TaxId: string option
        }
        with
            static member New(?taxId: string) =
                {
                    TaxId = taxId
                }

        type Confirm'PaymentMethodDataEpsBank =
        | ArzteUndApothekerBank
        | AustrianAnadiBankAg
        | BankAustria
        | BankhausCarlSpangler
        | BankhausSchelhammerUndSchatteraAg
        | BawagPskAg
        | BksBankAg
        | BrullKallmusBankAg
        | BtvVierLanderBank
        | CapitalBankGraweGruppeAg
        | Dolomitenbank
        | EasybankAg
        | ErsteBankUndSparkassen
        | HypoAlpeadriabankInternationalAg
        | HypoBankBurgenlandAktiengesellschaft
        | HypoNoeLbFurNiederosterreichUWien
        | HypoOberosterreichSalzburgSteiermark
        | HypoTirolBankAg
        | HypoVorarlbergBankAg
        | MarchfelderBank
        | OberbankAg
        | RaiffeisenBankengruppeOsterreich
        | SchoellerbankAg
        | SpardaBankWien
        | VolksbankGruppe
        | VolkskreditbankAg
        | VrBankBraunau

        type Confirm'PaymentMethodDataEps = {
            ///The customer's bank.
            [<Config.Form>]Bank: Confirm'PaymentMethodDataEpsBank option
        }
        with
            static member New(?bank: Confirm'PaymentMethodDataEpsBank) =
                {
                    Bank = bank
                }

        type Confirm'PaymentMethodDataFpxAccountHolderType =
        | Company
        | Individual

        type Confirm'PaymentMethodDataFpxBank =
        | AffinBank
        | AllianceBank
        | Ambank
        | BankIslam
        | BankMuamalat
        | BankRakyat
        | Bsn
        | Cimb
        | DeutscheBank
        | HongLeongBank
        | Hsbc
        | Kfh
        | Maybank2e
        | Maybank2u
        | Ocbc
        | PbEnterprise
        | PublicBank
        | Rhb
        | StandardChartered
        | Uob

        type Confirm'PaymentMethodDataFpx = {
            ///Account holder type for FPX transaction
            [<Config.Form>]AccountHolderType: Confirm'PaymentMethodDataFpxAccountHolderType option
            ///The customer's bank.
            [<Config.Form>]Bank: Confirm'PaymentMethodDataFpxBank option
        }
        with
            static member New(?accountHolderType: Confirm'PaymentMethodDataFpxAccountHolderType, ?bank: Confirm'PaymentMethodDataFpxBank) =
                {
                    AccountHolderType = accountHolderType
                    Bank = bank
                }

        type Confirm'PaymentMethodDataIdealBank =
        | AbnAmro
        | AsnBank
        | Bunq
        | Handelsbanken
        | Ing
        | Knab
        | Moneyou
        | Rabobank
        | Regiobank
        | Revolut
        | SnsBank
        | TriodosBank
        | VanLanschot

        type Confirm'PaymentMethodDataIdeal = {
            ///The customer's bank.
            [<Config.Form>]Bank: Confirm'PaymentMethodDataIdealBank option
        }
        with
            static member New(?bank: Confirm'PaymentMethodDataIdealBank) =
                {
                    Bank = bank
                }

        type Confirm'PaymentMethodDataP24Bank =
        | AliorBank
        | BankMillennium
        | BankNowyBfgSa
        | BankPekaoSa
        | BankiSpbdzielcze
        | Blik
        | BnpParibas
        | Boz
        | CitiHandlowy
        | CreditAgricole
        | Envelobank
        | EtransferPocztowy24
        | GetinBank
        | Ideabank
        | Ing
        | Inteligo
        | MbankMtransfer
        | NestPrzelew
        | NoblePay
        | PbacZIpko
        | PlusBank
        | SantanderPrzelew24
        | TmobileUsbugiBankowe
        | ToyotaBank
        | VolkswagenBank

        type Confirm'PaymentMethodDataP24 = {
            ///The customer's bank.
            [<Config.Form>]Bank: Confirm'PaymentMethodDataP24Bank option
        }
        with
            static member New(?bank: Confirm'PaymentMethodDataP24Bank) =
                {
                    Bank = bank
                }

        type Confirm'PaymentMethodDataSepaDebit = {
            ///IBAN of the bank account.
            [<Config.Form>]Iban: string option
        }
        with
            static member New(?iban: string) =
                {
                    Iban = iban
                }

        type Confirm'PaymentMethodDataSofortCountry =
        | [<JsonUnionCase("AT")>] AT
        | [<JsonUnionCase("BE")>] BE
        | [<JsonUnionCase("DE")>] DE
        | [<JsonUnionCase("ES")>] ES
        | [<JsonUnionCase("IT")>] IT
        | [<JsonUnionCase("NL")>] NL

        type Confirm'PaymentMethodDataSofort = {
            ///Two-letter ISO code representing the country the bank account is located in.
            [<Config.Form>]Country: Confirm'PaymentMethodDataSofortCountry option
        }
        with
            static member New(?country: Confirm'PaymentMethodDataSofortCountry) =
                {
                    Country = country
                }

        type Confirm'PaymentMethodDataType =
        | AcssDebit
        | AfterpayClearpay
        | Alipay
        | AuBecsDebit
        | BacsDebit
        | Bancontact
        | Boleto
        | Eps
        | Fpx
        | Giropay
        | Grabpay
        | Ideal
        | Oxxo
        | P24
        | SepaDebit
        | Sofort
        | WechatPay

        type Confirm'PaymentMethodData = {
            ///If this is an `acss_debit` PaymentMethod, this hash contains details about the ACSS Debit payment method.
            [<Config.Form>]AcssDebit: Confirm'PaymentMethodDataAcssDebit option
            ///If this is an `AfterpayClearpay` PaymentMethod, this hash contains details about the AfterpayClearpay payment method.
            [<Config.Form>]AfterpayClearpay: string option
            ///If this is an `Alipay` PaymentMethod, this hash contains details about the Alipay payment method.
            [<Config.Form>]Alipay: string option
            ///If this is an `au_becs_debit` PaymentMethod, this hash contains details about the bank account.
            [<Config.Form>]AuBecsDebit: Confirm'PaymentMethodDataAuBecsDebit option
            ///If this is a `bacs_debit` PaymentMethod, this hash contains details about the Bacs Direct Debit bank account.
            [<Config.Form>]BacsDebit: Confirm'PaymentMethodDataBacsDebit option
            ///If this is a `bancontact` PaymentMethod, this hash contains details about the Bancontact payment method.
            [<Config.Form>]Bancontact: string option
            ///Billing information associated with the PaymentMethod that may be used or required by particular types of payment methods.
            [<Config.Form>]BillingDetails: Confirm'PaymentMethodDataBillingDetails option
            ///If this is a `boleto` PaymentMethod, this hash contains details about the Boleto payment method.
            [<Config.Form>]Boleto: Confirm'PaymentMethodDataBoleto option
            ///If this is an `eps` PaymentMethod, this hash contains details about the EPS payment method.
            [<Config.Form>]Eps: Confirm'PaymentMethodDataEps option
            ///If this is an `fpx` PaymentMethod, this hash contains details about the FPX payment method.
            [<Config.Form>]Fpx: Confirm'PaymentMethodDataFpx option
            ///If this is a `giropay` PaymentMethod, this hash contains details about the Giropay payment method.
            [<Config.Form>]Giropay: string option
            ///If this is a `grabpay` PaymentMethod, this hash contains details about the GrabPay payment method.
            [<Config.Form>]Grabpay: string option
            ///If this is an `ideal` PaymentMethod, this hash contains details about the iDEAL payment method.
            [<Config.Form>]Ideal: Confirm'PaymentMethodDataIdeal option
            ///If this is an `interac_present` PaymentMethod, this hash contains details about the Interac Present payment method.
            [<Config.Form>]InteracPresent: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///If this is an `oxxo` PaymentMethod, this hash contains details about the OXXO payment method.
            [<Config.Form>]Oxxo: string option
            ///If this is a `p24` PaymentMethod, this hash contains details about the P24 payment method.
            [<Config.Form>]P24: Confirm'PaymentMethodDataP24 option
            ///If this is a `sepa_debit` PaymentMethod, this hash contains details about the SEPA debit bank account.
            [<Config.Form>]SepaDebit: Confirm'PaymentMethodDataSepaDebit option
            ///If this is a `sofort` PaymentMethod, this hash contains details about the SOFORT payment method.
            [<Config.Form>]Sofort: Confirm'PaymentMethodDataSofort option
            ///The type of the PaymentMethod. An additional hash is included on the PaymentMethod with a name matching this value. It contains additional information specific to the PaymentMethod type.
            [<Config.Form>]Type: Confirm'PaymentMethodDataType option
            ///If this is an `wechat_pay` PaymentMethod, this hash contains details about the wechat_pay payment method.
            [<Config.Form>]WechatPay: string option
        }
        with
            static member New(?acssDebit: Confirm'PaymentMethodDataAcssDebit, ?sofort: Confirm'PaymentMethodDataSofort, ?sepaDebit: Confirm'PaymentMethodDataSepaDebit, ?p24: Confirm'PaymentMethodDataP24, ?oxxo: string, ?metadata: Map<string, string>, ?interacPresent: string, ?ideal: Confirm'PaymentMethodDataIdeal, ?grabpay: string, ?type': Confirm'PaymentMethodDataType, ?giropay: string, ?eps: Confirm'PaymentMethodDataEps, ?boleto: Confirm'PaymentMethodDataBoleto, ?billingDetails: Confirm'PaymentMethodDataBillingDetails, ?bancontact: string, ?bacsDebit: Confirm'PaymentMethodDataBacsDebit, ?auBecsDebit: Confirm'PaymentMethodDataAuBecsDebit, ?alipay: string, ?afterpayClearpay: string, ?fpx: Confirm'PaymentMethodDataFpx, ?wechatPay: string) =
                {
                    AcssDebit = acssDebit
                    AfterpayClearpay = afterpayClearpay
                    Alipay = alipay
                    AuBecsDebit = auBecsDebit
                    BacsDebit = bacsDebit
                    Bancontact = bancontact
                    BillingDetails = billingDetails
                    Boleto = boleto
                    Eps = eps
                    Fpx = fpx
                    Giropay = giropay
                    Grabpay = grabpay
                    Ideal = ideal
                    InteracPresent = interacPresent
                    Metadata = metadata
                    Oxxo = oxxo
                    P24 = p24
                    SepaDebit = sepaDebit
                    Sofort = sofort
                    Type = type'
                    WechatPay = wechatPay
                }

        type Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsPaymentSchedule =
        | Combined
        | Interval
        | Sporadic

        type Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsTransactionType =
        | Business
        | Personal

        type Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptions = {
            ///A URL for custom mandate text to render during confirmation step.
            ///The URL will be rendered with additional GET parameters `payment_intent` and `payment_intent_client_secret` when confirming a Payment Intent,
            ///or `setup_intent` and `setup_intent_client_secret` when confirming a Setup Intent.
            [<Config.Form>]CustomMandateUrl: Choice<string,string> option
            ///Description of the mandate interval. Only required if 'payment_schedule' parameter is 'interval' or 'combined'.
            [<Config.Form>]IntervalDescription: string option
            ///Payment schedule for the mandate.
            [<Config.Form>]PaymentSchedule: Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsPaymentSchedule option
            ///Transaction type of the mandate.
            [<Config.Form>]TransactionType: Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsTransactionType option
        }
        with
            static member New(?customMandateUrl: Choice<string,string>, ?intervalDescription: string, ?paymentSchedule: Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsPaymentSchedule, ?transactionType: Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptionsTransactionType) =
                {
                    CustomMandateUrl = customMandateUrl
                    IntervalDescription = intervalDescription
                    PaymentSchedule = paymentSchedule
                    TransactionType = transactionType
                }

        type Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsVerificationMethod =
        | Automatic
        | Instant
        | Microdeposits

        type Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptions = {
            ///Additional fields for Mandate creation
            [<Config.Form>]MandateOptions: Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptions option
            ///Verification method for the intent
            [<Config.Form>]VerificationMethod: Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsVerificationMethod option
        }
        with
            static member New(?mandateOptions: Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsMandateOptions, ?verificationMethod: Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptionsVerificationMethod) =
                {
                    MandateOptions = mandateOptions
                    VerificationMethod = verificationMethod
                }

        type Confirm'PaymentMethodOptionsAfterpayClearpayPaymentMethodOptions = {
            ///Order identifier shown to the customer in Afterpay’s online portal. We recommend using a value that helps you answer any questions a customer might have about
            ///the payment. The identifier is limited to 128 characters and may contain only letters, digits, underscores, backslashes and dashes.
            [<Config.Form>]Reference: string option
        }
        with
            static member New(?reference: string) =
                {
                    Reference = reference
                }

        type Confirm'PaymentMethodOptionsBancontactPaymentMethodOptionsPreferredLanguage =
        | De
        | En
        | Fr
        | Nl

        type Confirm'PaymentMethodOptionsBancontactPaymentMethodOptions = {
            ///Preferred language of the Bancontact authorization page that the customer is redirected to.
            [<Config.Form>]PreferredLanguage: Confirm'PaymentMethodOptionsBancontactPaymentMethodOptionsPreferredLanguage option
        }
        with
            static member New(?preferredLanguage: Confirm'PaymentMethodOptionsBancontactPaymentMethodOptionsPreferredLanguage) =
                {
                    PreferredLanguage = preferredLanguage
                }

        type Confirm'PaymentMethodOptionsBoletoPaymentMethodOptions = {
            ///The number of calendar days before a Boleto voucher expires. For example, if you create a Boleto voucher on Monday and you set expires_after_days to 2, the Boleto invoice will expire on Wednesday at 23:59 America/Sao_Paulo time.
            [<Config.Form>]ExpiresAfterDays: int option
        }
        with
            static member New(?expiresAfterDays: int) =
                {
                    ExpiresAfterDays = expiresAfterDays
                }

        type Confirm'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanInterval =
        | Month

        type Confirm'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanType =
        | FixedCount

        type Confirm'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlan = {
            ///For `fixed_count` installment plans, this is the number of installment payments your customer will make to their credit card.
            [<Config.Form>]Count: int option
            ///For `fixed_count` installment plans, this is the interval between installment payments your customer will make to their credit card.
            ///One of `month`.
            [<Config.Form>]Interval: Confirm'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanInterval option
            ///Type of installment plan, one of `fixed_count`.
            [<Config.Form>]Type: Confirm'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanType option
        }
        with
            static member New(?count: int, ?interval: Confirm'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanInterval, ?type': Confirm'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlanType) =
                {
                    Count = count
                    Interval = interval
                    Type = type'
                }

        type Confirm'PaymentMethodOptionsCardPaymentIntentInstallments = {
            ///Setting to true enables installments for this PaymentIntent.
            ///This will cause the response to contain a list of available installment plans.
            ///Setting to false will prevent any selected plan from applying to a charge.
            [<Config.Form>]Enabled: bool option
            ///The selected installment plan to use for this payment attempt.
            ///This parameter can only be provided during confirmation.
            [<Config.Form>]Plan: Choice<Confirm'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlan,string> option
        }
        with
            static member New(?enabled: bool, ?plan: Choice<Confirm'PaymentMethodOptionsCardPaymentIntentInstallmentsPlanInstallmentPlan,string>) =
                {
                    Enabled = enabled
                    Plan = plan
                }

        type Confirm'PaymentMethodOptionsCardPaymentIntentNetwork =
        | Amex
        | CartesBancaires
        | Diners
        | Discover
        | Interac
        | Jcb
        | Mastercard
        | Unionpay
        | Unknown
        | Visa

        type Confirm'PaymentMethodOptionsCardPaymentIntentRequestThreeDSecure =
        | Any
        | Automatic

        type Confirm'PaymentMethodOptionsCardPaymentIntent = {
            ///A single-use `cvc_update` Token that represents a card CVC value. When provided, the CVC value will be verified during the card payment attempt. This parameter can only be provided during confirmation.
            [<Config.Form>]CvcToken: string option
            ///Installment configuration for payments attempted on this PaymentIntent (Mexico Only).
            ///For more information, see the [installments integration guide](https://stripe.com/docs/payments/installments).
            [<Config.Form>]Installments: Confirm'PaymentMethodOptionsCardPaymentIntentInstallments option
            ///When specified, this parameter indicates that a transaction will be marked
            ///as MOTO (Mail Order Telephone Order) and thus out of scope for SCA. This
            ///parameter can only be provided during confirmation.
            [<Config.Form>]Moto: bool option
            ///Selected network to process this PaymentIntent on. Depends on the available networks of the card attached to the PaymentIntent. Can be only set confirm-time.
            [<Config.Form>]Network: Confirm'PaymentMethodOptionsCardPaymentIntentNetwork option
            ///We strongly recommend that you rely on our SCA Engine to automatically prompt your customers for authentication based on risk level and [other requirements](https://stripe.com/docs/strong-customer-authentication). However, if you wish to request 3D Secure based on logic from your own fraud engine, provide this option. Permitted values include: `automatic` or `any`. If not provided, defaults to `automatic`. Read our guide on [manually requesting 3D Secure](https://stripe.com/docs/payments/3d-secure#manual-three-ds) for more information on how this configuration interacts with Radar and our SCA Engine.
            [<Config.Form>]RequestThreeDSecure: Confirm'PaymentMethodOptionsCardPaymentIntentRequestThreeDSecure option
        }
        with
            static member New(?cvcToken: string, ?installments: Confirm'PaymentMethodOptionsCardPaymentIntentInstallments, ?moto: bool, ?network: Confirm'PaymentMethodOptionsCardPaymentIntentNetwork, ?requestThreeDSecure: Confirm'PaymentMethodOptionsCardPaymentIntentRequestThreeDSecure) =
                {
                    CvcToken = cvcToken
                    Installments = installments
                    Moto = moto
                    Network = network
                    RequestThreeDSecure = requestThreeDSecure
                }

        type Confirm'PaymentMethodOptionsOxxoPaymentMethodOptions = {
            ///The number of calendar days before an OXXO voucher expires. For example, if you create an OXXO voucher on Monday and you set expires_after_days to 2, the OXXO invoice will expire on Wednesday at 23:59 America/Mexico_City time.
            [<Config.Form>]ExpiresAfterDays: int option
        }
        with
            static member New(?expiresAfterDays: int) =
                {
                    ExpiresAfterDays = expiresAfterDays
                }

        type Confirm'PaymentMethodOptionsP24PaymentMethodOptions = {
            ///Confirm that the payer has accepted the P24 terms and conditions.
            [<Config.Form>]TosShownAndAccepted: bool option
        }
        with
            static member New(?tosShownAndAccepted: bool) =
                {
                    TosShownAndAccepted = tosShownAndAccepted
                }

        type Confirm'PaymentMethodOptionsSepaDebitPaymentIntentPaymentMethodOptions = {
            ///Additional fields for Mandate creation
            [<Config.Form>]MandateOptions: string option
        }
        with
            static member New(?mandateOptions: string) =
                {
                    MandateOptions = mandateOptions
                }

        type Confirm'PaymentMethodOptionsSofortPaymentMethodOptionsPreferredLanguage =
        | De
        | En
        | Es
        | Fr
        | It
        | Nl
        | Pl

        type Confirm'PaymentMethodOptionsSofortPaymentMethodOptions = {
            ///Language shown to the payer on redirect.
            [<Config.Form>]PreferredLanguage: Confirm'PaymentMethodOptionsSofortPaymentMethodOptionsPreferredLanguage option
        }
        with
            static member New(?preferredLanguage: Confirm'PaymentMethodOptionsSofortPaymentMethodOptionsPreferredLanguage) =
                {
                    PreferredLanguage = preferredLanguage
                }

        type Confirm'PaymentMethodOptionsWechatPayPaymentMethodOptionsClient =
        | Android
        | Ios
        | Web

        type Confirm'PaymentMethodOptionsWechatPayPaymentMethodOptions = {
            ///The app ID registered with WeChat Pay. Only required when client is ios or android.
            [<Config.Form>]AppId: string option
            ///The client type that the end customer will pay from
            [<Config.Form>]Client: Confirm'PaymentMethodOptionsWechatPayPaymentMethodOptionsClient option
        }
        with
            static member New(?appId: string, ?client: Confirm'PaymentMethodOptionsWechatPayPaymentMethodOptionsClient) =
                {
                    AppId = appId
                    Client = client
                }

        type Confirm'PaymentMethodOptions = {
            ///If this is a `acss_debit` PaymentMethod, this sub-hash contains details about the ACSS Debit payment method options.
            [<Config.Form>]AcssDebit: Choice<Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptions,string> option
            ///If this is a `afterpay_clearpay` PaymentMethod, this sub-hash contains details about the Afterpay Clearpay payment method options.
            [<Config.Form>]AfterpayClearpay: Choice<Confirm'PaymentMethodOptionsAfterpayClearpayPaymentMethodOptions,string> option
            ///If this is a `alipay` PaymentMethod, this sub-hash contains details about the Alipay payment method options.
            [<Config.Form>]Alipay: Choice<string,string> option
            ///If this is a `bancontact` PaymentMethod, this sub-hash contains details about the Bancontact payment method options.
            [<Config.Form>]Bancontact: Choice<Confirm'PaymentMethodOptionsBancontactPaymentMethodOptions,string> option
            ///If this is a `boleto` PaymentMethod, this sub-hash contains details about the Boleto payment method options.
            [<Config.Form>]Boleto: Choice<Confirm'PaymentMethodOptionsBoletoPaymentMethodOptions,string> option
            ///Configuration for any card payments attempted on this PaymentIntent.
            [<Config.Form>]Card: Choice<Confirm'PaymentMethodOptionsCardPaymentIntent,string> option
            ///If this is a `card_present` PaymentMethod, this sub-hash contains details about the Card Present payment method options.
            [<Config.Form>]CardPresent: Choice<string,string> option
            ///If this is a `oxxo` PaymentMethod, this sub-hash contains details about the OXXO payment method options.
            [<Config.Form>]Oxxo: Choice<Confirm'PaymentMethodOptionsOxxoPaymentMethodOptions,string> option
            ///If this is a `p24` PaymentMethod, this sub-hash contains details about the Przelewy24 payment method options.
            [<Config.Form>]P24: Choice<Confirm'PaymentMethodOptionsP24PaymentMethodOptions,string> option
            ///If this is a `sepa_debit` PaymentIntent, this sub-hash contains details about the SEPA Debit payment method options.
            [<Config.Form>]SepaDebit: Choice<Confirm'PaymentMethodOptionsSepaDebitPaymentIntentPaymentMethodOptions,string> option
            ///If this is a `sofort` PaymentMethod, this sub-hash contains details about the SOFORT payment method options.
            [<Config.Form>]Sofort: Choice<Confirm'PaymentMethodOptionsSofortPaymentMethodOptions,string> option
            ///If this is a `wechat_pay` PaymentMethod, this sub-hash contains details about the WeChat Pay payment method options.
            [<Config.Form>]WechatPay: Choice<Confirm'PaymentMethodOptionsWechatPayPaymentMethodOptions,string> option
        }
        with
            static member New(?acssDebit: Choice<Confirm'PaymentMethodOptionsAcssDebitPaymentIntentPaymentMethodOptions,string>, ?afterpayClearpay: Choice<Confirm'PaymentMethodOptionsAfterpayClearpayPaymentMethodOptions,string>, ?alipay: Choice<string,string>, ?bancontact: Choice<Confirm'PaymentMethodOptionsBancontactPaymentMethodOptions,string>, ?boleto: Choice<Confirm'PaymentMethodOptionsBoletoPaymentMethodOptions,string>, ?card: Choice<Confirm'PaymentMethodOptionsCardPaymentIntent,string>, ?cardPresent: Choice<string,string>, ?oxxo: Choice<Confirm'PaymentMethodOptionsOxxoPaymentMethodOptions,string>, ?p24: Choice<Confirm'PaymentMethodOptionsP24PaymentMethodOptions,string>, ?sepaDebit: Choice<Confirm'PaymentMethodOptionsSepaDebitPaymentIntentPaymentMethodOptions,string>, ?sofort: Choice<Confirm'PaymentMethodOptionsSofortPaymentMethodOptions,string>, ?wechatPay: Choice<Confirm'PaymentMethodOptionsWechatPayPaymentMethodOptions,string>) =
                {
                    AcssDebit = acssDebit
                    AfterpayClearpay = afterpayClearpay
                    Alipay = alipay
                    Bancontact = bancontact
                    Boleto = boleto
                    Card = card
                    CardPresent = cardPresent
                    Oxxo = oxxo
                    P24 = p24
                    SepaDebit = sepaDebit
                    Sofort = sofort
                    WechatPay = wechatPay
                }

        type Confirm'ShippingShippingAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Confirm'ShippingShipping = {
            ///Shipping address.
            [<Config.Form>]Address: Confirm'ShippingShippingAddress option
            ///The delivery service that shipped a physical product, such as Fedex, UPS, USPS, etc.
            [<Config.Form>]Carrier: string option
            ///Recipient name.
            [<Config.Form>]Name: string option
            ///Recipient phone (including extension).
            [<Config.Form>]Phone: string option
            ///The tracking number for a physical product, obtained from the delivery service. If multiple tracking numbers were generated for this purchase, please separate them with commas.
            [<Config.Form>]TrackingNumber: string option
        }
        with
            static member New(?address: Confirm'ShippingShippingAddress, ?carrier: string, ?name: string, ?phone: string, ?trackingNumber: string) =
                {
                    Address = address
                    Carrier = carrier
                    Name = name
                    Phone = phone
                    TrackingNumber = trackingNumber
                }

        type Confirm'SetupFutureUsage =
        | OffSession
        | OnSession

        type ConfirmOptions = {
            [<Config.Path>]Intent: string
            ///Set to `true` to fail the payment attempt if the PaymentIntent transitions into `requires_action`. This parameter is intended for simpler integrations that do not handle customer actions, like [saving cards without authentication](https://stripe.com/docs/payments/save-card-without-authentication).
            [<Config.Form>]ErrorOnRequiresAction: bool option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///ID of the mandate to be used for this payment.
            [<Config.Form>]Mandate: string option
            ///This hash contains details about the Mandate to create
            [<Config.Form>]MandateData: Choice<Confirm'MandateDataSecretKey,Confirm'MandateDataClientKey> option
            ///Set to `true` to indicate that the customer is not in your checkout flow during this payment attempt, and therefore is unable to authenticate. This parameter is intended for scenarios where you collect card details and [charge them later](https://stripe.com/docs/payments/cards/charging-saved-cards).
            [<Config.Form>]OffSession: Choice<bool,Confirm'OffSession> option
            ///ID of the payment method (a PaymentMethod, Card, or [compatible Source](https://stripe.com/docs/payments/payment-methods/transitioning#compatibility) object) to attach to this PaymentIntent.
            [<Config.Form>]PaymentMethod: string option
            ///If provided, this hash will be used to create a PaymentMethod. The new PaymentMethod will appear
            ///in the [payment_method](https://stripe.com/docs/api/payment_intents/object#payment_intent_object-payment_method)
            ///property on the PaymentIntent.
            [<Config.Form>]PaymentMethodData: Confirm'PaymentMethodData option
            ///Payment-method-specific configuration for this PaymentIntent.
            [<Config.Form>]PaymentMethodOptions: Confirm'PaymentMethodOptions option
            ///Email address that the receipt for the resulting payment will be sent to. If `receipt_email` is specified for a payment in live mode, a receipt will be sent regardless of your [email settings](https://dashboard.stripe.com/account/emails).
            [<Config.Form>]ReceiptEmail: Choice<string,string> option
            ///The URL to redirect your customer back to after they authenticate or cancel their payment on the payment method's app or site.
            ///If you'd prefer to redirect to a mobile application, you can alternatively supply an application URI scheme.
            ///This parameter is only used for cards and other redirect-based payment methods.
            [<Config.Form>]ReturnUrl: string option
            ///Indicates that you intend to make future payments with this PaymentIntent's payment method.
            ///Providing this parameter will [attach the payment method](https://stripe.com/docs/payments/save-during-payment) to the PaymentIntent's Customer, if present, after the PaymentIntent is confirmed and any required actions from the user are complete. If no Customer was provided, the payment method can still be [attached](https://stripe.com/docs/api/payment_methods/attach) to a Customer after the transaction completes.
            ///When processing card payments, Stripe also uses `setup_future_usage` to dynamically optimize your payment flow and comply with regional legislation and network rules, such as [SCA](https://stripe.com/docs/strong-customer-authentication).
            ///If `setup_future_usage` is already set and you are performing a request using a publishable key, you may only update the value from `on_session` to `off_session`.
            [<Config.Form>]SetupFutureUsage: Confirm'SetupFutureUsage option
            ///Shipping information for this PaymentIntent.
            [<Config.Form>]Shipping: Choice<Confirm'ShippingShipping,string> option
            ///Set to `true` only when using manual confirmation and the iOS or Android SDKs to handle additional authentication steps.
            [<Config.Form>]UseStripeSdk: bool option
        }
        with
            static member New(intent: string, ?errorOnRequiresAction: bool, ?expand: string list, ?mandate: string, ?mandateData: Choice<Confirm'MandateDataSecretKey,Confirm'MandateDataClientKey>, ?offSession: Choice<bool,Confirm'OffSession>, ?paymentMethod: string, ?paymentMethodData: Confirm'PaymentMethodData, ?paymentMethodOptions: Confirm'PaymentMethodOptions, ?receiptEmail: Choice<string,string>, ?returnUrl: string, ?setupFutureUsage: Confirm'SetupFutureUsage, ?shipping: Choice<Confirm'ShippingShipping,string>, ?useStripeSdk: bool) =
                {
                    Intent = intent
                    ErrorOnRequiresAction = errorOnRequiresAction
                    Expand = expand
                    Mandate = mandate
                    MandateData = mandateData
                    OffSession = offSession
                    PaymentMethod = paymentMethod
                    PaymentMethodData = paymentMethodData
                    PaymentMethodOptions = paymentMethodOptions
                    ReceiptEmail = receiptEmail
                    ReturnUrl = returnUrl
                    SetupFutureUsage = setupFutureUsage
                    Shipping = shipping
                    UseStripeSdk = useStripeSdk
                }

        ///<p>Confirm that your customer intends to pay with current or provided
        ///payment method. Upon confirmation, the PaymentIntent will attempt to initiate
        ///a payment.
        ///If the selected payment method requires additional authentication steps, the
        ///PaymentIntent will transition to the <code>requires_action</code> status and
        ///suggest additional actions via <code>next_action</code>. If payment fails,
        ///the PaymentIntent will transition to the <code>requires_payment_method</code> status. If
        ///payment succeeds, the PaymentIntent will transition to the <code>succeeded</code>
        ///status (or <code>requires_capture</code>, if <code>capture_method</code> is set to <code>manual</code>).
        ///If the <code>confirmation_method</code> is <code>automatic</code>, payment may be attempted
        ///using our <a href="/docs/stripe-js/reference#stripe-handle-card-payment">client SDKs</a>
        ///and the PaymentIntent’s <a href="#payment_intent_object-client_secret">client_secret</a>.
        ///After <code>next_action</code>s are handled by the client, no additional
        ///confirmation is required to complete the payment.
        ///If the <code>confirmation_method</code> is <code>manual</code>, all payment attempts must be
        ///initiated using a secret key.
        ///If any actions are required for the payment, the PaymentIntent will
        ///return to the <code>requires_confirmation</code> state
        ///after those actions are completed. Your server needs to then
        ///explicitly re-confirm the PaymentIntent to initiate the next payment
        ///attempt. Read the <a href="/docs/payments/payment-intents/web-manual">expanded documentation</a>
        ///to learn more about manual confirmation.</p>
        let Confirm settings (options: ConfirmOptions) =
            $"/v1/payment_intents/{options.Intent}/confirm"
            |> RestApi.postAsync<_, PaymentIntent> settings (Map.empty) options

    module PaymentMethods =

        type ListOptions = {
            ///The ID of the customer whose PaymentMethods will be retrieved.
            [<Config.Query>]Customer: string
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///A required filter on the list, based on the object `type` field.
            [<Config.Query>]Type: string
        }
        with
            static member New(customer: string, type': string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string) =
                {
                    Customer = customer
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                    Type = type'
                }

        ///<p>Returns a list of PaymentMethods for a given Customer</p>
        let List settings (options: ListOptions) =
            let qs = [("customer", options.Customer |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("type", options.Type |> box)] |> Map.ofList
            $"/v1/payment_methods"
            |> RestApi.getAsync<PaymentMethod list> settings qs

        type Create'AcssDebit = {
            ///Customer's bank account number.
            [<Config.Form>]AccountNumber: string option
            ///Institution number of the customer's bank.
            [<Config.Form>]InstitutionNumber: string option
            ///Transit number of the customer's bank.
            [<Config.Form>]TransitNumber: string option
        }
        with
            static member New(?accountNumber: string, ?institutionNumber: string, ?transitNumber: string) =
                {
                    AccountNumber = accountNumber
                    InstitutionNumber = institutionNumber
                    TransitNumber = transitNumber
                }

        type Create'AuBecsDebit = {
            ///The account number for the bank account.
            [<Config.Form>]AccountNumber: string option
            ///Bank-State-Branch number of the bank account.
            [<Config.Form>]BsbNumber: string option
        }
        with
            static member New(?accountNumber: string, ?bsbNumber: string) =
                {
                    AccountNumber = accountNumber
                    BsbNumber = bsbNumber
                }

        type Create'BacsDebit = {
            ///Account number of the bank account that the funds will be debited from.
            [<Config.Form>]AccountNumber: string option
            ///Sort code of the bank account. (e.g., `10-20-30`)
            [<Config.Form>]SortCode: string option
        }
        with
            static member New(?accountNumber: string, ?sortCode: string) =
                {
                    AccountNumber = accountNumber
                    SortCode = sortCode
                }

        type Create'BillingDetailsAddressBillingDetailsAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Create'BillingDetails = {
            ///Billing address.
            [<Config.Form>]Address: Choice<Create'BillingDetailsAddressBillingDetailsAddress,string> option
            ///Email address.
            [<Config.Form>]Email: string option
            ///Full name.
            [<Config.Form>]Name: string option
            ///Billing phone number (including extension).
            [<Config.Form>]Phone: string option
        }
        with
            static member New(?address: Choice<Create'BillingDetailsAddressBillingDetailsAddress,string>, ?email: string, ?name: string, ?phone: string) =
                {
                    Address = address
                    Email = email
                    Name = name
                    Phone = phone
                }

        type Create'Boleto = {
            ///Uniquely identifies this customer tax_id (CNPJ or CPF)
            [<Config.Form>]TaxId: string option
        }
        with
            static member New(?taxId: string) =
                {
                    TaxId = taxId
                }

        type Create'CardCardDetailsParams = {
            ///The card's CVC. It is highly recommended to always include this value.
            [<Config.Form>]Cvc: string option
            ///Two-digit number representing the card's expiration month.
            [<Config.Form>]ExpMonth: int option
            ///Four-digit number representing the card's expiration year.
            [<Config.Form>]ExpYear: int option
            ///The card number, as a string without any separators.
            [<Config.Form>]Number: string option
        }
        with
            static member New(?cvc: string, ?expMonth: int, ?expYear: int, ?number: string) =
                {
                    Cvc = cvc
                    ExpMonth = expMonth
                    ExpYear = expYear
                    Number = number
                }

        type Create'CardTokenParams = {
            [<Config.Form>]Token: string option
        }
        with
            static member New(?token: string) =
                {
                    Token = token
                }

        type Create'EpsBank =
        | ArzteUndApothekerBank
        | AustrianAnadiBankAg
        | BankAustria
        | BankhausCarlSpangler
        | BankhausSchelhammerUndSchatteraAg
        | BawagPskAg
        | BksBankAg
        | BrullKallmusBankAg
        | BtvVierLanderBank
        | CapitalBankGraweGruppeAg
        | Dolomitenbank
        | EasybankAg
        | ErsteBankUndSparkassen
        | HypoAlpeadriabankInternationalAg
        | HypoBankBurgenlandAktiengesellschaft
        | HypoNoeLbFurNiederosterreichUWien
        | HypoOberosterreichSalzburgSteiermark
        | HypoTirolBankAg
        | HypoVorarlbergBankAg
        | MarchfelderBank
        | OberbankAg
        | RaiffeisenBankengruppeOsterreich
        | SchoellerbankAg
        | SpardaBankWien
        | VolksbankGruppe
        | VolkskreditbankAg
        | VrBankBraunau

        type Create'Eps = {
            ///The customer's bank.
            [<Config.Form>]Bank: Create'EpsBank option
        }
        with
            static member New(?bank: Create'EpsBank) =
                {
                    Bank = bank
                }

        type Create'FpxAccountHolderType =
        | Company
        | Individual

        type Create'FpxBank =
        | AffinBank
        | AllianceBank
        | Ambank
        | BankIslam
        | BankMuamalat
        | BankRakyat
        | Bsn
        | Cimb
        | DeutscheBank
        | HongLeongBank
        | Hsbc
        | Kfh
        | Maybank2e
        | Maybank2u
        | Ocbc
        | PbEnterprise
        | PublicBank
        | Rhb
        | StandardChartered
        | Uob

        type Create'Fpx = {
            ///Account holder type for FPX transaction
            [<Config.Form>]AccountHolderType: Create'FpxAccountHolderType option
            ///The customer's bank.
            [<Config.Form>]Bank: Create'FpxBank option
        }
        with
            static member New(?accountHolderType: Create'FpxAccountHolderType, ?bank: Create'FpxBank) =
                {
                    AccountHolderType = accountHolderType
                    Bank = bank
                }

        type Create'IdealBank =
        | AbnAmro
        | AsnBank
        | Bunq
        | Handelsbanken
        | Ing
        | Knab
        | Moneyou
        | Rabobank
        | Regiobank
        | Revolut
        | SnsBank
        | TriodosBank
        | VanLanschot

        type Create'Ideal = {
            ///The customer's bank.
            [<Config.Form>]Bank: Create'IdealBank option
        }
        with
            static member New(?bank: Create'IdealBank) =
                {
                    Bank = bank
                }

        type Create'P24Bank =
        | AliorBank
        | BankMillennium
        | BankNowyBfgSa
        | BankPekaoSa
        | BankiSpbdzielcze
        | Blik
        | BnpParibas
        | Boz
        | CitiHandlowy
        | CreditAgricole
        | Envelobank
        | EtransferPocztowy24
        | GetinBank
        | Ideabank
        | Ing
        | Inteligo
        | MbankMtransfer
        | NestPrzelew
        | NoblePay
        | PbacZIpko
        | PlusBank
        | SantanderPrzelew24
        | TmobileUsbugiBankowe
        | ToyotaBank
        | VolkswagenBank

        type Create'P24 = {
            ///The customer's bank.
            [<Config.Form>]Bank: Create'P24Bank option
        }
        with
            static member New(?bank: Create'P24Bank) =
                {
                    Bank = bank
                }

        type Create'SepaDebit = {
            ///IBAN of the bank account.
            [<Config.Form>]Iban: string option
        }
        with
            static member New(?iban: string) =
                {
                    Iban = iban
                }

        type Create'SofortCountry =
        | [<JsonUnionCase("AT")>] AT
        | [<JsonUnionCase("BE")>] BE
        | [<JsonUnionCase("DE")>] DE
        | [<JsonUnionCase("ES")>] ES
        | [<JsonUnionCase("IT")>] IT
        | [<JsonUnionCase("NL")>] NL

        type Create'Sofort = {
            ///Two-letter ISO code representing the country the bank account is located in.
            [<Config.Form>]Country: Create'SofortCountry option
        }
        with
            static member New(?country: Create'SofortCountry) =
                {
                    Country = country
                }

        type Create'Type =
        | AcssDebit
        | AfterpayClearpay
        | Alipay
        | AuBecsDebit
        | BacsDebit
        | Bancontact
        | Boleto
        | Card
        | Eps
        | Fpx
        | Giropay
        | Grabpay
        | Ideal
        | Oxxo
        | P24
        | SepaDebit
        | Sofort
        | WechatPay

        type CreateOptions = {
            ///If this is an `acss_debit` PaymentMethod, this hash contains details about the ACSS Debit payment method.
            [<Config.Form>]AcssDebit: Create'AcssDebit option
            ///If this is an `AfterpayClearpay` PaymentMethod, this hash contains details about the AfterpayClearpay payment method.
            [<Config.Form>]AfterpayClearpay: string option
            ///If this is an `Alipay` PaymentMethod, this hash contains details about the Alipay payment method.
            [<Config.Form>]Alipay: string option
            ///If this is an `au_becs_debit` PaymentMethod, this hash contains details about the bank account.
            [<Config.Form>]AuBecsDebit: Create'AuBecsDebit option
            ///If this is a `bacs_debit` PaymentMethod, this hash contains details about the Bacs Direct Debit bank account.
            [<Config.Form>]BacsDebit: Create'BacsDebit option
            ///If this is a `bancontact` PaymentMethod, this hash contains details about the Bancontact payment method.
            [<Config.Form>]Bancontact: string option
            ///Billing information associated with the PaymentMethod that may be used or required by particular types of payment methods.
            [<Config.Form>]BillingDetails: Create'BillingDetails option
            ///If this is a `boleto` PaymentMethod, this hash contains details about the Boleto payment method.
            [<Config.Form>]Boleto: Create'Boleto option
            ///If this is a `card` PaymentMethod, this hash contains the user's card details. For backwards compatibility, you can alternatively provide a Stripe token (e.g., for Apple Pay, Amex Express Checkout, or legacy Checkout) into the card hash with format `card: {token: "tok_visa"}`. When providing a card number, you must meet the requirements for [PCI compliance](https://stripe.com/docs/security#validating-pci-compliance). We strongly recommend using Stripe.js instead of interacting with this API directly.
            [<Config.Form>]Card: Choice<Create'CardCardDetailsParams,Create'CardTokenParams> option
            ///The `Customer` to whom the original PaymentMethod is attached.
            [<Config.Form>]Customer: string option
            ///If this is an `eps` PaymentMethod, this hash contains details about the EPS payment method.
            [<Config.Form>]Eps: Create'Eps option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///If this is an `fpx` PaymentMethod, this hash contains details about the FPX payment method.
            [<Config.Form>]Fpx: Create'Fpx option
            ///If this is a `giropay` PaymentMethod, this hash contains details about the Giropay payment method.
            [<Config.Form>]Giropay: string option
            ///If this is a `grabpay` PaymentMethod, this hash contains details about the GrabPay payment method.
            [<Config.Form>]Grabpay: string option
            ///If this is an `ideal` PaymentMethod, this hash contains details about the iDEAL payment method.
            [<Config.Form>]Ideal: Create'Ideal option
            ///If this is an `interac_present` PaymentMethod, this hash contains details about the Interac Present payment method.
            [<Config.Form>]InteracPresent: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///If this is an `oxxo` PaymentMethod, this hash contains details about the OXXO payment method.
            [<Config.Form>]Oxxo: string option
            ///If this is a `p24` PaymentMethod, this hash contains details about the P24 payment method.
            [<Config.Form>]P24: Create'P24 option
            ///The PaymentMethod to share.
            [<Config.Form>]PaymentMethod: string option
            ///If this is a `sepa_debit` PaymentMethod, this hash contains details about the SEPA debit bank account.
            [<Config.Form>]SepaDebit: Create'SepaDebit option
            ///If this is a `sofort` PaymentMethod, this hash contains details about the SOFORT payment method.
            [<Config.Form>]Sofort: Create'Sofort option
            ///The type of the PaymentMethod. An additional hash is included on the PaymentMethod with a name matching this value. It contains additional information specific to the PaymentMethod type.
            [<Config.Form>]Type: Create'Type option
            ///If this is an `wechat_pay` PaymentMethod, this hash contains details about the wechat_pay payment method.
            [<Config.Form>]WechatPay: string option
        }
        with
            static member New(?acssDebit: Create'AcssDebit, ?sofort: Create'Sofort, ?sepaDebit: Create'SepaDebit, ?paymentMethod: string, ?p24: Create'P24, ?oxxo: string, ?metadata: Map<string, string>, ?interacPresent: string, ?ideal: Create'Ideal, ?grabpay: string, ?giropay: string, ?type': Create'Type, ?fpx: Create'Fpx, ?eps: Create'Eps, ?customer: string, ?card: Choice<Create'CardCardDetailsParams,Create'CardTokenParams>, ?boleto: Create'Boleto, ?billingDetails: Create'BillingDetails, ?bancontact: string, ?bacsDebit: Create'BacsDebit, ?auBecsDebit: Create'AuBecsDebit, ?alipay: string, ?afterpayClearpay: string, ?expand: string list, ?wechatPay: string) =
                {
                    AcssDebit = acssDebit
                    AfterpayClearpay = afterpayClearpay
                    Alipay = alipay
                    AuBecsDebit = auBecsDebit
                    BacsDebit = bacsDebit
                    Bancontact = bancontact
                    BillingDetails = billingDetails
                    Boleto = boleto
                    Card = card
                    Customer = customer
                    Eps = eps
                    Expand = expand
                    Fpx = fpx
                    Giropay = giropay
                    Grabpay = grabpay
                    Ideal = ideal
                    InteracPresent = interacPresent
                    Metadata = metadata
                    Oxxo = oxxo
                    P24 = p24
                    PaymentMethod = paymentMethod
                    SepaDebit = sepaDebit
                    Sofort = sofort
                    Type = type'
                    WechatPay = wechatPay
                }

        ///<p>Creates a PaymentMethod object. Read the <a href="/docs/stripe-js/reference#stripe-create-payment-method">Stripe.js reference</a> to learn how to create PaymentMethods via Stripe.js.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/payment_methods"
            |> RestApi.postAsync<_, PaymentMethod> settings (Map.empty) options

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]PaymentMethod: string
        }
        with
            static member New(paymentMethod: string, ?expand: string list) =
                {
                    Expand = expand
                    PaymentMethod = paymentMethod
                }

        ///<p>Retrieves a PaymentMethod object.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/payment_methods/{options.PaymentMethod}"
            |> RestApi.getAsync<PaymentMethod> settings qs

        type Update'BillingDetailsAddressBillingDetailsAddress = {
            ///City, district, suburb, town, or village.
            [<Config.Form>]City: string option
            ///Two-letter country code ([ISO 3166-1 alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)).
            [<Config.Form>]Country: string option
            ///Address line 1 (e.g., street, PO Box, or company name).
            [<Config.Form>]Line1: string option
            ///Address line 2 (e.g., apartment, suite, unit, or building).
            [<Config.Form>]Line2: string option
            ///ZIP or postal code.
            [<Config.Form>]PostalCode: string option
            ///State, county, province, or region.
            [<Config.Form>]State: string option
        }
        with
            static member New(?city: string, ?country: string, ?line1: string, ?line2: string, ?postalCode: string, ?state: string) =
                {
                    City = city
                    Country = country
                    Line1 = line1
                    Line2 = line2
                    PostalCode = postalCode
                    State = state
                }

        type Update'BillingDetails = {
            ///Billing address.
            [<Config.Form>]Address: Choice<Update'BillingDetailsAddressBillingDetailsAddress,string> option
            ///Email address.
            [<Config.Form>]Email: string option
            ///Full name.
            [<Config.Form>]Name: string option
            ///Billing phone number (including extension).
            [<Config.Form>]Phone: string option
        }
        with
            static member New(?address: Choice<Update'BillingDetailsAddressBillingDetailsAddress,string>, ?email: string, ?name: string, ?phone: string) =
                {
                    Address = address
                    Email = email
                    Name = name
                    Phone = phone
                }

        type Update'Card = {
            ///Two-digit number representing the card's expiration month.
            [<Config.Form>]ExpMonth: int option
            ///Four-digit number representing the card's expiration year.
            [<Config.Form>]ExpYear: int option
        }
        with
            static member New(?expMonth: int, ?expYear: int) =
                {
                    ExpMonth = expMonth
                    ExpYear = expYear
                }

        type UpdateOptions = {
            [<Config.Path>]PaymentMethod: string
            ///This is a legacy parameter that will be removed in the future. It is a hash that does not accept any keys.
            [<Config.Form>]AcssDebit: string option
            ///This is a legacy parameter that will be removed in the future. It is a hash that does not accept any keys.
            [<Config.Form>]AuBecsDebit: string option
            ///This is a legacy parameter that will be removed in the future. It is a hash that does not accept any keys.
            [<Config.Form>]BacsDebit: string option
            ///Billing information associated with the PaymentMethod that may be used or required by particular types of payment methods.
            [<Config.Form>]BillingDetails: Update'BillingDetails option
            ///If this is a `card` PaymentMethod, this hash contains the user's card details.
            [<Config.Form>]Card: Update'Card option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///This is a legacy parameter that will be removed in the future. It is a hash that does not accept any keys.
            [<Config.Form>]SepaDebit: string option
        }
        with
            static member New(paymentMethod: string, ?acssDebit: string, ?auBecsDebit: string, ?bacsDebit: string, ?billingDetails: Update'BillingDetails, ?card: Update'Card, ?expand: string list, ?metadata: Map<string, string>, ?sepaDebit: string) =
                {
                    PaymentMethod = paymentMethod
                    AcssDebit = acssDebit
                    AuBecsDebit = auBecsDebit
                    BacsDebit = bacsDebit
                    BillingDetails = billingDetails
                    Card = card
                    Expand = expand
                    Metadata = metadata
                    SepaDebit = sepaDebit
                }

        ///<p>Updates a PaymentMethod object. A PaymentMethod must be attached a customer to be updated.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/payment_methods/{options.PaymentMethod}"
            |> RestApi.postAsync<_, PaymentMethod> settings (Map.empty) options

    module PaymentMethodsAttach =

        type AttachOptions = {
            [<Config.Path>]PaymentMethod: string
            ///The ID of the customer to which to attach the PaymentMethod.
            [<Config.Form>]Customer: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(paymentMethod: string, customer: string, ?expand: string list) =
                {
                    PaymentMethod = paymentMethod
                    Customer = customer
                    Expand = expand
                }

        ///<p>Attaches a PaymentMethod object to a Customer.
        ///To attach a new PaymentMethod to a customer for future payments, we recommend you use a <a href="/docs/api/setup_intents">SetupIntent</a>
        ///or a PaymentIntent with <a href="/docs/api/payment_intents/create#create_payment_intent-setup_future_usage">setup_future_usage</a>.
        ///These approaches will perform any necessary steps to ensure that the PaymentMethod can be used in a future payment. Using the
        ///<code>/v1/payment_methods/:id/attach</code> endpoint does not ensure that future payments can be made with the attached PaymentMethod.
        ///See <a href="/docs/payments/payment-intents#future-usage">Optimizing cards for future payments</a> for more information about setting up future payments.
        ///To use this PaymentMethod as the default for invoice or subscription payments,
        ///set <a href="/docs/api/customers/update#update_customer-invoice_settings-default_payment_method"><code>invoice_settings.default_payment_method</code></a>,
        ///on the Customer to the PaymentMethod’s ID.</p>
        let Attach settings (options: AttachOptions) =
            $"/v1/payment_methods/{options.PaymentMethod}/attach"
            |> RestApi.postAsync<_, PaymentMethod> settings (Map.empty) options

    module PaymentMethodsDetach =

        type DetachOptions = {
            [<Config.Path>]PaymentMethod: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(paymentMethod: string, ?expand: string list) =
                {
                    PaymentMethod = paymentMethod
                    Expand = expand
                }

        ///<p>Detaches a PaymentMethod object from a Customer.</p>
        let Detach settings (options: DetachOptions) =
            $"/v1/payment_methods/{options.PaymentMethod}/detach"
            |> RestApi.postAsync<_, PaymentMethod> settings (Map.empty) options

    module Payouts =

        type ListOptions = {
            [<Config.Query>]ArrivalDate: int option
            [<Config.Query>]Created: int option
            ///The ID of an external account - only return payouts sent to this external account.
            [<Config.Query>]Destination: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return payouts that have the given status: `pending`, `paid`, `failed`, or `canceled`.
            [<Config.Query>]Status: string option
        }
        with
            static member New(?arrivalDate: int, ?created: int, ?destination: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?startingAfter: string, ?status: string) =
                {
                    ArrivalDate = arrivalDate
                    Created = created
                    Destination = destination
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    StartingAfter = startingAfter
                    Status = status
                }

        ///<p>Returns a list of existing payouts sent to third-party bank accounts or that Stripe has sent you. The payouts are returned in sorted order, with the most recently created payouts appearing first.</p>
        let List settings (options: ListOptions) =
            let qs = [("arrival_date", options.ArrivalDate |> box); ("created", options.Created |> box); ("destination", options.Destination |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("starting_after", options.StartingAfter |> box); ("status", options.Status |> box)] |> Map.ofList
            $"/v1/payouts"
            |> RestApi.getAsync<Payout list> settings qs

        type Create'Method =
        | Instant
        | Standard

        type Create'SourceType =
        | BankAccount
        | Card
        | Fpx

        type CreateOptions = {
            ///A positive integer in cents representing how much to payout.
            [<Config.Form>]Amount: int
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string
            ///An arbitrary string attached to the object. Often useful for displaying to users.
            [<Config.Form>]Description: string option
            ///The ID of a bank account or a card to send the payout to. If no destination is supplied, the default external account for the specified currency will be used.
            [<Config.Form>]Destination: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The method used to send this payout, which can be `standard` or `instant`. `instant` is only supported for payouts to debit cards. (See [Instant payouts for marketplaces for more information](https://stripe.com/blog/instant-payouts-for-marketplaces).)
            [<Config.Form>]Method: Create'Method option
            ///The balance type of your Stripe balance to draw this payout from. Balances for different payment sources are kept separately. You can find the amounts with the balances API. One of `bank_account`, `card`, or `fpx`.
            [<Config.Form>]SourceType: Create'SourceType option
            ///A string to be displayed on the recipient's bank or card statement. This may be at most 22 characters. Attempting to use a `statement_descriptor` longer than 22 characters will return an error. Note: Most banks will truncate this information and/or display it inconsistently. Some may not display it at all.
            [<Config.Form>]StatementDescriptor: string option
        }
        with
            static member New(amount: int, currency: string, ?description: string, ?destination: string, ?expand: string list, ?metadata: Map<string, string>, ?method: Create'Method, ?sourceType: Create'SourceType, ?statementDescriptor: string) =
                {
                    Amount = amount
                    Currency = currency
                    Description = description
                    Destination = destination
                    Expand = expand
                    Metadata = metadata
                    Method = method
                    SourceType = sourceType
                    StatementDescriptor = statementDescriptor
                }

        ///<p>To send funds to your own bank account, you create a new payout object. Your <a href="#balance">Stripe balance</a> must be able to cover the payout amount, or you’ll receive an “Insufficient Funds” error.
        ///If your API key is in test mode, money won’t actually be sent, though everything else will occur as if in live mode.
        ///If you are creating a manual payout on a Stripe account that uses multiple payment source types, you’ll need to specify the source type balance that the payout should draw from. The <a href="#balance_object">balance object</a> details available and pending amounts by source type.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/payouts"
            |> RestApi.postAsync<_, Payout> settings (Map.empty) options

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Payout: string
        }
        with
            static member New(payout: string, ?expand: string list) =
                {
                    Expand = expand
                    Payout = payout
                }

        ///<p>Retrieves the details of an existing payout. Supply the unique payout ID from either a payout creation request or the payout list, and Stripe will return the corresponding payout information.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/payouts/{options.Payout}"
            |> RestApi.getAsync<Payout> settings qs

        type UpdateOptions = {
            [<Config.Path>]Payout: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(payout: string, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Payout = payout
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Updates the specified payout by setting the values of the parameters passed. Any parameters not provided will be left unchanged. This request accepts only the metadata as arguments.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/payouts/{options.Payout}"
            |> RestApi.postAsync<_, Payout> settings (Map.empty) options

    module PayoutsCancel =

        type CancelOptions = {
            [<Config.Path>]Payout: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
        }
        with
            static member New(payout: string, ?expand: string list) =
                {
                    Payout = payout
                    Expand = expand
                }

        ///<p>A previously created payout can be canceled if it has not yet been paid out. Funds will be refunded to your available balance. You may not cancel automatic Stripe payouts.</p>
        let Cancel settings (options: CancelOptions) =
            $"/v1/payouts/{options.Payout}/cancel"
            |> RestApi.postAsync<_, Payout> settings (Map.empty) options

    module PayoutsReverse =

        type ReverseOptions = {
            [<Config.Path>]Payout: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
        }
        with
            static member New(payout: string, ?expand: string list, ?metadata: Map<string, string>) =
                {
                    Payout = payout
                    Expand = expand
                    Metadata = metadata
                }

        ///<p>Reverses a payout by debiting the destination bank account. Only payouts for connected accounts to US bank accounts may be reversed at this time. If the payout is in the <code>pending</code> status, <code>/v1/payouts/:id/cancel</code> should be used instead.
        ///By requesting a reversal via <code>/v1/payouts/:id/reverse</code>, you confirm that the authorized signatory of the selected bank account has authorized the debit on the bank account and that no other authorization is required.</p>
        let Reverse settings (options: ReverseOptions) =
            $"/v1/payouts/{options.Payout}/reverse"
            |> RestApi.postAsync<_, Payout> settings (Map.empty) options

    module Plans =

        type ListOptions = {
            ///Only return plans that are active or inactive (e.g., pass `false` to list all inactive plans).
            [<Config.Query>]Active: bool option
            ///A filter on the list, based on the object `created` field. The value can be a string with an integer Unix timestamp, or it can be a dictionary with a number of different query options.
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///Only return plans for the given product.
            [<Config.Query>]Product: string option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
        }
        with
            static member New(?active: bool, ?created: int, ?endingBefore: string, ?expand: string list, ?limit: int, ?product: string, ?startingAfter: string) =
                {
                    Active = active
                    Created = created
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    Product = product
                    StartingAfter = startingAfter
                }

        ///<p>Returns a list of your plans.</p>
        let List settings (options: ListOptions) =
            let qs = [("active", options.Active |> box); ("created", options.Created |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("product", options.Product |> box); ("starting_after", options.StartingAfter |> box)] |> Map.ofList
            $"/v1/plans"
            |> RestApi.getAsync<Plan list> settings qs

        type Create'ProductInlineProductParams = {
            ///Whether the product is currently available for purchase. Defaults to `true`.
            [<Config.Form>]Active: bool option
            ///The identifier for the product. Must be unique. If not provided, an identifier will be randomly generated.
            [<Config.Form>]Id: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The product's name, meant to be displayable to the customer. Whenever this product is sold via a subscription, name will show up on associated invoice line item descriptions.
            [<Config.Form>]Name: string option
            ///An arbitrary string to be displayed on your customer's credit card or bank statement. While most banks display this information consistently, some may display it incorrectly or not at all.
            ///This may be up to 22 characters. The statement description may not include `<`, `>`, `\`, `"`, `'` characters, and will appear on your customer's statement in capital letters. Non-ASCII characters are automatically stripped.
            [<Config.Form>]StatementDescriptor: string option
            ///A [tax code](https://stripe.com/docs/tax/tax-codes) ID.
            [<Config.Form>]TaxCode: string option
            ///A label that represents units of this product in Stripe and on customers’ receipts and invoices. When set, this will be included in associated invoice line item descriptions.
            [<Config.Form>]UnitLabel: string option
        }
        with
            static member New(?active: bool, ?id: string, ?metadata: Map<string, string>, ?name: string, ?statementDescriptor: string, ?taxCode: string, ?unitLabel: string) =
                {
                    Active = active
                    Id = id
                    Metadata = metadata
                    Name = name
                    StatementDescriptor = statementDescriptor
                    TaxCode = taxCode
                    UnitLabel = unitLabel
                }

        type Create'TiersUpTo =
        | Inf

        type Create'Tiers = {
            ///The flat billing amount for an entire tier, regardless of the number of units in the tier.
            [<Config.Form>]FlatAmount: int option
            ///Same as `flat_amount`, but accepts a decimal value representing an integer in the minor units of the currency. Only one of `flat_amount` and `flat_amount_decimal` can be set.
            [<Config.Form>]FlatAmountDecimal: string option
            ///The per unit billing amount for each individual unit for which this tier applies.
            [<Config.Form>]UnitAmount: int option
            ///Same as `unit_amount`, but accepts a decimal value in %s with at most 12 decimal places. Only one of `unit_amount` and `unit_amount_decimal` can be set.
            [<Config.Form>]UnitAmountDecimal: string option
            ///Specifies the upper bound of this tier. The lower bound of a tier is the upper bound of the previous tier adding one. Use `inf` to define a fallback tier.
            [<Config.Form>]UpTo: Choice<Create'TiersUpTo,int> option
        }
        with
            static member New(?flatAmount: int, ?flatAmountDecimal: string, ?unitAmount: int, ?unitAmountDecimal: string, ?upTo: Choice<Create'TiersUpTo,int>) =
                {
                    FlatAmount = flatAmount
                    FlatAmountDecimal = flatAmountDecimal
                    UnitAmount = unitAmount
                    UnitAmountDecimal = unitAmountDecimal
                    UpTo = upTo
                }

        type Create'TransformUsageRound =
        | Down
        | Up

        type Create'TransformUsage = {
            ///Divide usage by this number.
            [<Config.Form>]DivideBy: int option
            ///After division, either round the result `up` or `down`.
            [<Config.Form>]Round: Create'TransformUsageRound option
        }
        with
            static member New(?divideBy: int, ?round: Create'TransformUsageRound) =
                {
                    DivideBy = divideBy
                    Round = round
                }

        type Create'AggregateUsage =
        | LastDuringPeriod
        | LastEver
        | Max
        | Sum

        type Create'BillingScheme =
        | PerUnit
        | Tiered

        type Create'Interval =
        | Day
        | Month
        | Week
        | Year

        type Create'TiersMode =
        | Graduated
        | Volume

        type Create'UsageType =
        | Licensed
        | Metered

        type CreateOptions = {
            ///Whether the plan is currently available for new subscriptions. Defaults to `true`.
            [<Config.Form>]Active: bool option
            ///Specifies a usage aggregation strategy for plans of `usage_type=metered`. Allowed values are `sum` for summing up all usage during a period, `last_during_period` for using the last usage record reported within a period, `last_ever` for using the last usage record ever (across period bounds) or `max` which uses the usage record with the maximum reported usage during a period. Defaults to `sum`.
            [<Config.Form>]AggregateUsage: Create'AggregateUsage option
            ///A positive integer in %s (or 0 for a free plan) representing how much to charge on a recurring basis.
            [<Config.Form>]Amount: int option
            ///Same as `amount`, but accepts a decimal value with at most 12 decimal places. Only one of `amount` and `amount_decimal` can be set.
            [<Config.Form>]AmountDecimal: string option
            ///Describes how to compute the price per period. Either `per_unit` or `tiered`. `per_unit` indicates that the fixed amount (specified in `amount`) will be charged per unit in `quantity` (for plans with `usage_type=licensed`), or per unit of total usage (for plans with `usage_type=metered`). `tiered` indicates that the unit pricing will be computed using a tiering strategy as defined using the `tiers` and `tiers_mode` attributes.
            [<Config.Form>]BillingScheme: Create'BillingScheme option
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///An identifier randomly generated by Stripe. Used to identify this plan when subscribing a customer. You can optionally override this ID, but the ID must be unique across all plans in your Stripe account. You can, however, use the same plan ID in both live and test modes.
            [<Config.Form>]Id: string option
            ///Specifies billing frequency. Either `day`, `week`, `month` or `year`.
            [<Config.Form>]Interval: Create'Interval
            ///The number of intervals between subscription billings. For example, `interval=month` and `interval_count=3` bills every 3 months. Maximum of one year interval allowed (1 year, 12 months, or 52 weeks).
            [<Config.Form>]IntervalCount: int option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///A brief description of the plan, hidden from customers.
            [<Config.Form>]Nickname: string option
            [<Config.Form>]Product: Choice<Create'ProductInlineProductParams,string> option
            ///Each element represents a pricing tier. This parameter requires `billing_scheme` to be set to `tiered`. See also the documentation for `billing_scheme`.
            [<Config.Form>]Tiers: Create'Tiers list option
            ///Defines if the tiering price should be `graduated` or `volume` based. In `volume`-based tiering, the maximum quantity within a period determines the per unit price, in `graduated` tiering pricing can successively change as the quantity grows.
            [<Config.Form>]TiersMode: Create'TiersMode option
            ///Apply a transformation to the reported usage or set quantity before computing the billed price. Cannot be combined with `tiers`.
            [<Config.Form>]TransformUsage: Create'TransformUsage option
            ///Default number of trial days when subscribing a customer to this plan using [`trial_from_plan=true`](https://stripe.com/docs/api#create_subscription-trial_from_plan).
            [<Config.Form>]TrialPeriodDays: int option
            ///Configures how the quantity per period should be determined. Can be either `metered` or `licensed`. `licensed` automatically bills the `quantity` set when adding it to a subscription. `metered` aggregates the total usage based on usage records. Defaults to `licensed`.
            [<Config.Form>]UsageType: Create'UsageType option
        }
        with
            static member New(interval: Create'Interval, currency: string, ?transformUsage: Create'TransformUsage, ?tiersMode: Create'TiersMode, ?tiers: Create'Tiers list, ?product: Choice<Create'ProductInlineProductParams,string>, ?nickname: string, ?metadata: Map<string, string>, ?intervalCount: int, ?active: bool, ?id: string, ?expand: string list, ?billingScheme: Create'BillingScheme, ?amountDecimal: string, ?amount: int, ?aggregateUsage: Create'AggregateUsage, ?trialPeriodDays: int, ?usageType: Create'UsageType) =
                {
                    Active = active
                    AggregateUsage = aggregateUsage
                    Amount = amount
                    AmountDecimal = amountDecimal
                    BillingScheme = billingScheme
                    Currency = currency
                    Expand = expand
                    Id = id
                    Interval = interval
                    IntervalCount = intervalCount
                    Metadata = metadata
                    Nickname = nickname
                    Product = product
                    Tiers = tiers
                    TiersMode = tiersMode
                    TransformUsage = transformUsage
                    TrialPeriodDays = trialPeriodDays
                    UsageType = usageType
                }

        ///<p>You can now model subscriptions more flexibly using the <a href="#prices">Prices API</a>. It replaces the Plans API and is backwards compatible to simplify your migration.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/plans"
            |> RestApi.postAsync<_, Plan> settings (Map.empty) options

        type DeleteOptions = {
            [<Config.Path>]Plan: string
        }
        with
            static member New(plan: string) =
                {
                    Plan = plan
                }

        ///<p>Deleting plans means new subscribers can’t be added. Existing subscribers aren’t affected.</p>
        let Delete settings (options: DeleteOptions) =
            $"/v1/plans/{options.Plan}"
            |> RestApi.deleteAsync<DeletedPlan> settings (Map.empty)

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Plan: string
        }
        with
            static member New(plan: string, ?expand: string list) =
                {
                    Expand = expand
                    Plan = plan
                }

        ///<p>Retrieves the plan with the given ID.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/plans/{options.Plan}"
            |> RestApi.getAsync<Plan> settings qs

        type UpdateOptions = {
            [<Config.Path>]Plan: string
            ///Whether the plan is currently available for new subscriptions.
            [<Config.Form>]Active: bool option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///A brief description of the plan, hidden from customers.
            [<Config.Form>]Nickname: string option
            ///The product the plan belongs to. This cannot be changed once it has been used in a subscription or subscription schedule.
            [<Config.Form>]Product: string option
            ///Default number of trial days when subscribing a customer to this plan using [`trial_from_plan=true`](https://stripe.com/docs/api#create_subscription-trial_from_plan).
            [<Config.Form>]TrialPeriodDays: int option
        }
        with
            static member New(plan: string, ?active: bool, ?expand: string list, ?metadata: Map<string, string>, ?nickname: string, ?product: string, ?trialPeriodDays: int) =
                {
                    Plan = plan
                    Active = active
                    Expand = expand
                    Metadata = metadata
                    Nickname = nickname
                    Product = product
                    TrialPeriodDays = trialPeriodDays
                }

        ///<p>Updates the specified plan by setting the values of the parameters passed. Any parameters not provided are left unchanged. By design, you cannot change a plan’s ID, amount, currency, or billing cycle.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/plans/{options.Plan}"
            |> RestApi.postAsync<_, Plan> settings (Map.empty) options

    module Prices =

        type ListOptions = {
            ///Only return prices that are active or inactive (e.g., pass `false` to list all inactive prices).
            [<Config.Query>]Active: bool option
            ///A filter on the list, based on the object `created` field. The value can be a string with an integer Unix timestamp, or it can be a dictionary with a number of different query options.
            [<Config.Query>]Created: int option
            ///Only return prices for the given currency.
            [<Config.Query>]Currency: string option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
            [<Config.Query>]Limit: int option
            ///Only return the price with these lookup_keys, if any exist.
            [<Config.Query>]LookupKeys: string list option
            ///Only return prices for the given product.
            [<Config.Query>]Product: string option
            ///Only return prices with these recurring fields.
            [<Config.Query>]Recurring: Map<string, string> option
            ///A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
            [<Config.Query>]StartingAfter: string option
            ///Only return prices of type `recurring` or `one_time`.
            [<Config.Query>]Type: string option
        }
        with
            static member New(?active: bool, ?created: int, ?currency: string, ?endingBefore: string, ?expand: string list, ?limit: int, ?lookupKeys: string list, ?product: string, ?recurring: Map<string, string>, ?startingAfter: string, ?type': string) =
                {
                    Active = active
                    Created = created
                    Currency = currency
                    EndingBefore = endingBefore
                    Expand = expand
                    Limit = limit
                    LookupKeys = lookupKeys
                    Product = product
                    Recurring = recurring
                    StartingAfter = startingAfter
                    Type = type'
                }

        ///<p>Returns a list of your prices.</p>
        let List settings (options: ListOptions) =
            let qs = [("active", options.Active |> box); ("created", options.Created |> box); ("currency", options.Currency |> box); ("ending_before", options.EndingBefore |> box); ("expand", options.Expand |> box); ("limit", options.Limit |> box); ("lookup_keys", options.LookupKeys |> box); ("product", options.Product |> box); ("recurring", options.Recurring |> box); ("starting_after", options.StartingAfter |> box); ("type", options.Type |> box)] |> Map.ofList
            $"/v1/prices"
            |> RestApi.getAsync<Price list> settings qs

        type Create'ProductData = {
            ///Whether the product is currently available for purchase. Defaults to `true`.
            [<Config.Form>]Active: bool option
            ///The identifier for the product. Must be unique. If not provided, an identifier will be randomly generated.
            [<Config.Form>]Id: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///The product's name, meant to be displayable to the customer. Whenever this product is sold via a subscription, name will show up on associated invoice line item descriptions.
            [<Config.Form>]Name: string option
            ///An arbitrary string to be displayed on your customer's credit card or bank statement. While most banks display this information consistently, some may display it incorrectly or not at all.
            ///This may be up to 22 characters. The statement description may not include `<`, `>`, `\`, `"`, `'` characters, and will appear on your customer's statement in capital letters. Non-ASCII characters are automatically stripped.
            [<Config.Form>]StatementDescriptor: string option
            ///A [tax code](https://stripe.com/docs/tax/tax-codes) ID.
            [<Config.Form>]TaxCode: string option
            ///A label that represents units of this product in Stripe and on customers’ receipts and invoices. When set, this will be included in associated invoice line item descriptions.
            [<Config.Form>]UnitLabel: string option
        }
        with
            static member New(?active: bool, ?id: string, ?metadata: Map<string, string>, ?name: string, ?statementDescriptor: string, ?taxCode: string, ?unitLabel: string) =
                {
                    Active = active
                    Id = id
                    Metadata = metadata
                    Name = name
                    StatementDescriptor = statementDescriptor
                    TaxCode = taxCode
                    UnitLabel = unitLabel
                }

        type Create'RecurringAggregateUsage =
        | LastDuringPeriod
        | LastEver
        | Max
        | Sum

        type Create'RecurringInterval =
        | Day
        | Month
        | Week
        | Year

        type Create'RecurringUsageType =
        | Licensed
        | Metered

        type Create'Recurring = {
            ///Specifies a usage aggregation strategy for prices of `usage_type=metered`. Allowed values are `sum` for summing up all usage during a period, `last_during_period` for using the last usage record reported within a period, `last_ever` for using the last usage record ever (across period bounds) or `max` which uses the usage record with the maximum reported usage during a period. Defaults to `sum`.
            [<Config.Form>]AggregateUsage: Create'RecurringAggregateUsage option
            ///Specifies billing frequency. Either `day`, `week`, `month` or `year`.
            [<Config.Form>]Interval: Create'RecurringInterval option
            ///The number of intervals between subscription billings. For example, `interval=month` and `interval_count=3` bills every 3 months. Maximum of one year interval allowed (1 year, 12 months, or 52 weeks).
            [<Config.Form>]IntervalCount: int option
            ///Default number of trial days when subscribing a customer to this price using [`trial_from_plan=true`](https://stripe.com/docs/api#create_subscription-trial_from_plan).
            [<Config.Form>]TrialPeriodDays: int option
            ///Configures how the quantity per period should be determined. Can be either `metered` or `licensed`. `licensed` automatically bills the `quantity` set when adding it to a subscription. `metered` aggregates the total usage based on usage records. Defaults to `licensed`.
            [<Config.Form>]UsageType: Create'RecurringUsageType option
        }
        with
            static member New(?aggregateUsage: Create'RecurringAggregateUsage, ?interval: Create'RecurringInterval, ?intervalCount: int, ?trialPeriodDays: int, ?usageType: Create'RecurringUsageType) =
                {
                    AggregateUsage = aggregateUsage
                    Interval = interval
                    IntervalCount = intervalCount
                    TrialPeriodDays = trialPeriodDays
                    UsageType = usageType
                }

        type Create'TiersUpTo =
        | Inf

        type Create'Tiers = {
            ///The flat billing amount for an entire tier, regardless of the number of units in the tier.
            [<Config.Form>]FlatAmount: int option
            ///Same as `flat_amount`, but accepts a decimal value representing an integer in the minor units of the currency. Only one of `flat_amount` and `flat_amount_decimal` can be set.
            [<Config.Form>]FlatAmountDecimal: string option
            ///The per unit billing amount for each individual unit for which this tier applies.
            [<Config.Form>]UnitAmount: int option
            ///Same as `unit_amount`, but accepts a decimal value in %s with at most 12 decimal places. Only one of `unit_amount` and `unit_amount_decimal` can be set.
            [<Config.Form>]UnitAmountDecimal: string option
            ///Specifies the upper bound of this tier. The lower bound of a tier is the upper bound of the previous tier adding one. Use `inf` to define a fallback tier.
            [<Config.Form>]UpTo: Choice<Create'TiersUpTo,int> option
        }
        with
            static member New(?flatAmount: int, ?flatAmountDecimal: string, ?unitAmount: int, ?unitAmountDecimal: string, ?upTo: Choice<Create'TiersUpTo,int>) =
                {
                    FlatAmount = flatAmount
                    FlatAmountDecimal = flatAmountDecimal
                    UnitAmount = unitAmount
                    UnitAmountDecimal = unitAmountDecimal
                    UpTo = upTo
                }

        type Create'TransformQuantityRound =
        | Down
        | Up

        type Create'TransformQuantity = {
            ///Divide usage by this number.
            [<Config.Form>]DivideBy: int option
            ///After division, either round the result `up` or `down`.
            [<Config.Form>]Round: Create'TransformQuantityRound option
        }
        with
            static member New(?divideBy: int, ?round: Create'TransformQuantityRound) =
                {
                    DivideBy = divideBy
                    Round = round
                }

        type Create'BillingScheme =
        | PerUnit
        | Tiered

        type Create'TaxBehavior =
        | Exclusive
        | Inclusive
        | Unspecified

        type Create'TiersMode =
        | Graduated
        | Volume

        type CreateOptions = {
            ///Whether the price can be used for new purchases. Defaults to `true`.
            [<Config.Form>]Active: bool option
            ///Describes how to compute the price per period. Either `per_unit` or `tiered`. `per_unit` indicates that the fixed amount (specified in `unit_amount` or `unit_amount_decimal`) will be charged per unit in `quantity` (for prices with `usage_type=licensed`), or per unit of total usage (for prices with `usage_type=metered`). `tiered` indicates that the unit pricing will be computed using a tiering strategy as defined using the `tiers` and `tiers_mode` attributes.
            [<Config.Form>]BillingScheme: Create'BillingScheme option
            ///Three-letter [ISO currency code](https://www.iso.org/iso-4217-currency-codes.html), in lowercase. Must be a [supported currency](https://stripe.com/docs/currencies).
            [<Config.Form>]Currency: string
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///A lookup key used to retrieve prices dynamically from a static string.
            [<Config.Form>]LookupKey: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///A brief description of the price, hidden from customers.
            [<Config.Form>]Nickname: string option
            ///The ID of the product that this price will belong to.
            [<Config.Form>]Product: string option
            ///These fields can be used to create a new product that this price will belong to.
            [<Config.Form>]ProductData: Create'ProductData option
            ///The recurring components of a price such as `interval` and `usage_type`.
            [<Config.Form>]Recurring: Create'Recurring option
            ///Specifies whether the price is considered inclusive of taxes or exclusive of taxes. One of `inclusive`, `exclusive`, or `unspecified`. Once specified as either `inclusive` or `exclusive`, it cannot be changed.
            [<Config.Form>]TaxBehavior: Create'TaxBehavior option
            ///Each element represents a pricing tier. This parameter requires `billing_scheme` to be set to `tiered`. See also the documentation for `billing_scheme`.
            [<Config.Form>]Tiers: Create'Tiers list option
            ///Defines if the tiering price should be `graduated` or `volume` based. In `volume`-based tiering, the maximum quantity within a period determines the per unit price, in `graduated` tiering pricing can successively change as the quantity grows.
            [<Config.Form>]TiersMode: Create'TiersMode option
            ///If set to true, will atomically remove the lookup key from the existing price, and assign it to this price.
            [<Config.Form>]TransferLookupKey: bool option
            ///Apply a transformation to the reported usage or set quantity before computing the billed price. Cannot be combined with `tiers`.
            [<Config.Form>]TransformQuantity: Create'TransformQuantity option
            ///A positive integer in %s (or 0 for a free price) representing how much to charge.
            [<Config.Form>]UnitAmount: int option
            ///Same as `unit_amount`, but accepts a decimal value in %s with at most 12 decimal places. Only one of `unit_amount` and `unit_amount_decimal` can be set.
            [<Config.Form>]UnitAmountDecimal: string option
        }
        with
            static member New(currency: string, ?active: bool, ?transformQuantity: Create'TransformQuantity, ?transferLookupKey: bool, ?tiersMode: Create'TiersMode, ?tiers: Create'Tiers list, ?taxBehavior: Create'TaxBehavior, ?recurring: Create'Recurring, ?productData: Create'ProductData, ?product: string, ?nickname: string, ?metadata: Map<string, string>, ?lookupKey: string, ?expand: string list, ?billingScheme: Create'BillingScheme, ?unitAmount: int, ?unitAmountDecimal: string) =
                {
                    Active = active
                    BillingScheme = billingScheme
                    Currency = currency
                    Expand = expand
                    LookupKey = lookupKey
                    Metadata = metadata
                    Nickname = nickname
                    Product = product
                    ProductData = productData
                    Recurring = recurring
                    TaxBehavior = taxBehavior
                    Tiers = tiers
                    TiersMode = tiersMode
                    TransferLookupKey = transferLookupKey
                    TransformQuantity = transformQuantity
                    UnitAmount = unitAmount
                    UnitAmountDecimal = unitAmountDecimal
                }

        ///<p>Creates a new price for an existing product. The price can be recurring or one-time.</p>
        let Create settings (options: CreateOptions) =
            $"/v1/prices"
            |> RestApi.postAsync<_, Price> settings (Map.empty) options

        type RetrieveOptions = {
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            [<Config.Path>]Price: string
        }
        with
            static member New(price: string, ?expand: string list) =
                {
                    Expand = expand
                    Price = price
                }

        ///<p>Retrieves the price with the given ID.</p>
        let Retrieve settings (options: RetrieveOptions) =
            let qs = [("expand", options.Expand |> box)] |> Map.ofList
            $"/v1/prices/{options.Price}"
            |> RestApi.getAsync<Price> settings qs

        type Update'RecurringUpdateRecurringParams = {
            ///Default number of trial days when subscribing a customer to this plan using [`trial_from_plan=true`](https://stripe.com/docs/api#create_subscription-trial_from_plan).
            [<Config.Form>]TrialPeriodDays: int option
        }
        with
            static member New(?trialPeriodDays: int) =
                {
                    TrialPeriodDays = trialPeriodDays
                }

        type Update'TaxBehavior =
        | Exclusive
        | Inclusive
        | Unspecified

        type UpdateOptions = {
            [<Config.Path>]Price: string
            ///Whether the price can be used for new purchases. Defaults to `true`.
            [<Config.Form>]Active: bool option
            ///Specifies which fields in the response should be expanded.
            [<Config.Form>]Expand: string list option
            ///A lookup key used to retrieve prices dynamically from a static string.
            [<Config.Form>]LookupKey: string option
            ///Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format. Individual keys can be unset by posting an empty value to them. All keys can be unset by posting an empty value to `metadata`.
            [<Config.Form>]Metadata: Map<string, string> option
            ///A brief description of the price, hidden from customers.
            [<Config.Form>]Nickname: string option
            ///The recurring components of a price such as `interval` and `usage_type`.
            [<Config.Form>]Recurring: Choice<Update'RecurringUpdateRecurringParams,string> option
            ///Specifies whether the price is considered inclusive of taxes or exclusive of taxes. One of `inclusive`, `exclusive`, or `unspecified`. Once specified as either `inclusive` or `exclusive`, it cannot be changed.
            [<Config.Form>]TaxBehavior: Update'TaxBehavior option
            ///If set to true, will atomically remove the lookup key from the existing price, and assign it to this price.
            [<Config.Form>]TransferLookupKey: bool option
        }
        with
            static member New(price: string, ?active: bool, ?expand: string list, ?lookupKey: string, ?metadata: Map<string, string>, ?nickname: string, ?recurring: Choice<Update'RecurringUpdateRecurringParams,string>, ?taxBehavior: Update'TaxBehavior, ?transferLookupKey: bool) =
                {
                    Price = price
                    Active = active
                    Expand = expand
                    LookupKey = lookupKey
                    Metadata = metadata
                    Nickname = nickname
                    Recurring = recurring
                    TaxBehavior = taxBehavior
                    TransferLookupKey = transferLookupKey
                }

        ///<p>Updates the specified price by setting the values of the parameters passed. Any parameters not provided are left unchanged.</p>
        let Update settings (options: UpdateOptions) =
            $"/v1/prices/{options.Price}"
            |> RestApi.postAsync<_, Price> settings (Map.empty) options

    module Products =

        type ListOptions = {
            ///Only return products that are active or inactive (e.g., pass `false` to list all inactive products).
            [<Config.Query>]Active: bool option
            ///Only return products that were created during the given date interval.
            [<Config.Query>]Created: int option
            ///A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
            [<Config.Query>]EndingBefore: string option
            ///Specifies which fields in the response should be expanded.
            [<Config.Query>]Expand: string list option
            ///Only return products with the given IDs.
            [<Config.Query>]Ids: string list option
            ///A limit on the number of objects to be returned. Limit can range between 1 and 100, and the defa