namespace ProveSDK

open ProveSDK

module private SmsDeliveryEndpoints =
    let [<Literal>] SmsDelivery = "/platform"
    
module SmsDelivery =
    let [<Literal>] AppName = "SmsDelivery"

    type RequestNamedData =
        { /// Message to send to user.
          MessageText: string }
    
    type RequestData =
        { /// End user's telephone number.
          PhoneNumber: string
          
          /// Data to be sent to the user.
          NamedData: RequestNamedData }
    
    /// The SMS OTP delivery function within the MFA platform is a function based call to send an SMS message.
    type Request =
        { /// Customer ID as assigned by Prove.
          ClientId: string
          
          /// Sub-identifier that represents a particular customer.
          /// Value as provided in the request.
          ClientAcctId: string option
          
          /// Assigned by Prove for access to the service.
          /// The license key is specific to the service environment.
          /// Note: Provision configurability of this parameter,
          /// preferably outside the code base, assists with installation
          /// of new license keys.
          License: string
          
          /// Application identifier as assigned by Prove.
          App: string
          
          /// Unique identifier representing the session assigned by the customer.
          /// Character set: ASCII
          /// Avoid including sensitive information (user name, account #) in the field.
          ClientContext: string
          
          /// Data to be sent to the user.
          Data: RequestData }
        
    type Response =
        { /// Customer ID as assigned by Prove. Value as provided in the request.
          ClientId: string
          
          /// Unique identifier representing the session as assigned by the customer. Value as provided in the request.
          ClientContext: string
          
          /// Application identifier as assigned by Prove.
          App: string
          
          /// Sub-identifier that represents a particular customer. Value as provided in the request.
          ClientAcctId: string option
          
          /// A globally unique identifier generated by Prove.
          /// Returned in the report message in response to the initial request.
          /// This identifier ties all information about the session together and should be easily accessible
          /// by customer staff submitting support requests.
          /// Character set: a-f, 0-9, -
          EwSID: string
          
          /// Service URL for all poll requests. Due to the multi-site nature of our production environment,
          /// you must design the service implementation to send subsequent messages to the entire replyTo URL
          /// provided in the response from Prove.
          ReplyTo: string
          
          /// Timestamp of message in ISO 8601 extended combined format.
          /// Format: {YYYY}-{MM|-{DD}T{HH}:{MM}:{SS}Z
          TimestampIso8601: string
          
          /// A list of applicable status codes is provided in the API Reference Section, Error and Status Codes.
          /// Hexadecimal Value Format: 0-FFFF.
          StatusCode: string
          
          /// A description of the value returned in the statusCode field.
          StatusText: string }
        
    let execute (env: ProveEnvironment) (request: Request) =
        Request.executeMfa<Request, Response> env SmsDeliveryEndpoints.SmsDelivery request

module Sms =
    open SmsDelivery

    type License =
        { Environment: ProveEnvironment
          ClientId: string
          LicenseKey: string }

    type SmsNotSent =
        { /// A globally unique identifier generated by Prove.
          /// Returned in the report message in response to the initial request.
          /// This identifier ties all information about the session together and should be easily accessible
          /// by customer staff submitting support requests.
          /// Character set: a-f, 0-9, -
          EwSID: string

          /// Timestamp of message in ISO 8601 extended combined format.
          /// Format: {YYYY}-{MM|-{DD}T{HH}:{MM}:{SS}Z
          TimestampIso8601: string
          
          /// A list of applicable status codes is provided in the API Reference Section, Error and Status Codes.
          /// Hexadecimal Value Format: 0-FFFF.
          StatusCode: string
          
          /// A description of the value returned in the statusCode field.
          StatusText: string }

    type SmsSent =
        { /// Service URL for all poll requests. Due to the multi-site nature of our production environment,
          /// you must design the service implementation to send subsequent messages to the entire replyTo URL
          /// provided in the response from Prove.
          ReplyTo: string }

    type SmsError =
        | UnknownError of string
        | MismatchedClient
        | SmsNotSent of SmsNotSent

    let send (license: License) (phoneNumber: string) (message: string) =
        async {
            let clientAcctId = None
            let clientContext = System.Guid.NewGuid().ToString()

            let request =
                { ClientId = license.ClientId
                  ClientAcctId = clientAcctId
                  License = license.LicenseKey
                  App = SmsDelivery.AppName
                  ClientContext = clientContext
                  Data =
                    { PhoneNumber = phoneNumber
                      NamedData = { MessageText = message } } }

            let! response = execute license.Environment request

            return
                match response with
                | Error error ->
                    Error (UnknownError error.Message)

                | Ok data ->
                    if
                        data.ClientId <> license.ClientId
                        || data.ClientAcctId <> clientAcctId
                        || data.ClientContext <> clientContext
                    then
                        Error MismatchedClient
                    elif data.StatusCode <> "0" then
                        Error
                            (SmsNotSent
                                { EwSID = data.EwSID
                                  TimestampIso8601 = data.TimestampIso8601
                                  StatusCode = data.StatusCode
                                  StatusText = data.StatusText })
                    else
                        Ok { ReplyTo = data.ReplyTo }
        }
