{
  config,
  lib,
  pkgs,
  ...
}: let
  # These flags cause Bazel builds to connect to NativeLink's read-only cache.
  #
  # ```nix
  # devShells.default = pkgs.mkShell {
  #   shellHook = ''
  #   # Generate the `lre.bazelrc` config file.
  #   ${config.nativelink.installationScript}
  #   '';
  # };
  # ```
  defaultConfig = [
    "--remote_cache=${config.endpoint}"
    "--remote_header=x-nativelink-api-key=${config.api-key}"
    "--remote_header=x-nativelink-project=nativelink-ci"
    "--nogenerate_json_trace_profile"
    "--remote_upload_local_results=false"
    "--experimental_remote_cache_async"
  ];

  # If the `nativelink.settings.prefix` is set to a nonempty string,
  # prefix the Bazel build commands with that string. This will disable
  # connecting to the nativelink-cloud by default and require adding
  # `--config=<prefix>` to Bazel invocations.
  maybePrefixedConfig =
    if (config.prefix == "")
    then map (x: "build " + x) defaultConfig
    else map (x: "build:" + config.prefix + " " + x) defaultConfig;

  configFile = pkgs.runCommand "nativelink.bazelrc" {} ''
    printf '# These flags are dynamically generated by the nativelink flake module.
    #
    # Add `try-import %%workspace%%/nativelink.bazelrc` to your .bazelrc to
    # include these flags when running Bazel in a nix environment.

    ${lib.concatLines maybePrefixedConfig}' >$out
  '';
in {
  options = {
    installationScript = lib.mkOption {
      type = lib.types.str;
      description = lib.mkDoc ''
        A bash snippet which creates a nativelink.bazelrc file in the repository.
      '';
    };
    endpoint = lib.mkOption {
      type = lib.types.str;
      description = lib.mdDoc ''
        The NativeLink Cloud endpoint.

        Defaults to NativeLink's shared cache.
      '';
      default = "grpcs://cas-tracemachina-shared.build-faster.nativelink.net";
    };
    api-key = lib.mkOption {
      type = lib.types.str;
      description = lib.mdDoc ''
        The API key to connect to the NativeLink Cloud.

        You should only use read-only keys here to prevent cache-poisoning and
        malicious artifact extractions.

        Defaults to NativeLink's shared read-only api key.
      '';
      default = "065f02f53f26a12331d5cfd00a778fb243bfb4e857b8fcd4c99273edfb15deae";
    };
    prefix = lib.mkOption {
      type = lib.types.str;
      description = lib.mdDoc ''
        An optional Bazel config prefix for the flags in `nativelink.bazelrc`.

        If set, builds need to explicitly enable the nativelink config via
        `--config=<prefix>`.

        Defaults to an empty string, enabling the cache by default.
      '';
      default = "";
    };
  };

  config = {
    installationScript = ''
      if ! type -t git >/dev/null; then
        # In pure shells
        echo 1>&2 "WARNING: NativeLink: git command not found; skipping installation."
      elif ! ${pkgs.git}/bin/git rev-parse --git-dir &> /dev/null; then
        echo 1>&2 "WARNING: NativeLink: .git not found; skipping installation."
      else
        GIT_WC=`${pkgs.git}/bin/git rev-parse --show-toplevel`

        # These update procedures compare before they write, to avoid
        # filesystem churn. This improves performance with watch tools like
        # lorri and prevents installation loops by lorri.

        if ! readlink "''${GIT_WC}/nativelink.bazelrc" >/dev/null \
          || [[ $(readlink "''${GIT_WC}/nativelink.bazelrc") != ${configFile} ]]; then
          echo 1>&2 "NativeLink: updating $PWD repository"
          [ -L nativelink.bazelrc ] && unlink nativelink.bazelrc

          if [ -e "''${GIT_WC}/nativelink.bazelrc" ]; then
            echo 1>&2 "NativeLink: WARNING: Refusing to install because of pre-existing nativelink.bazelrc"
            echo 1>&2 "  Remove the nativelink.bazelrc file and add nativelink.bazelrc to .gitignore."
          else
            ln -fs ${configFile} "''${GIT_WC}/nativelink.bazelrc"
          fi
        fi
      fi
    '';
  };
}
