# LENS

## 1. Коротко о главном


* Встраиваемый скриптовый язык
* Платформа .NET
* Функциональная парадигма, статическая типизация
* Функции - объекты первого рода, алгебраические типы
* Взаимодействие со сборками .NET

## 2. Синтаксис и возможности

Блоки выделяются отступами, выражения разделяются переводом строки.

### 2.1. Типы данных

В интерпретатор встроена поддержка следующих типов:

* `unit = void`
* `object = System.Object`
* `bool = System.Boolean`
* `int = System.Int32`
* `long = System.Int64`
* `float = System.Single`
* `double = System.Double`
* `string = System.String`

### 2.2. Объявление констант и переменных

Изменяемые переменные объявляются ключевым словом `var`, константы - `let`.
Значение констант не обязательно должно быть константой во время компиляции.
Константы нельзя использовать слева от знака присваивания или передавать по ссылке.

### 2.3. Операторы

В языке объявлены следующие операторы, перечисленные в порядке приоритета:

1. Возведение в степень (`**`)
2. Умножение (`*`), деление (`/`), получение остатка от деления (`%`)
3. Сложение (`+`), вычитание (`-`)
4. Унарное отрицание (`-`), инверсия (`not`)
5. Сравнение (`==`, `<>`, `<`, `>`, `<=`, `>=`)
6. Логические операции (`&&`, `||`, `^^`)

Оператор сложения также используется для конкатенации строк. Если у объекта есть переопределенный
оператор, он будет использован.

### 2.4. Записи

Запись - то же самое, что структура. Объект, имеющий только поля. Без методов
и модификаторов доступа. Объявляется ключевым словом `record`:

    record Student
        Name : string
        Age : int
        
Все поля структуры являются публичными.

Структуры могут быть рекурсивными, т.е. включать в себя элементы собственного
типа.

### 2.5. Алгебраические типы

Объявляются ключевым словом `type` и перечислением возможных ярлыков типа. К
каждому ярлыку может быть прикреплена метка с помощью ключевого слова `of`:

    type Suit
        Hearts
        Clubs
        Spades
        Diamonds

    type Card
        Ace of Suit
        King of Suit
        Queen of Suit
        Jack of Suit
        ValueCard of Tuple<Suit, int>

Ярлыки должны быть глобально уникальными идентификаторами в контексте скрипта,
поскольку они же являются статическими конструкторами:

    let jack = Jack Hearts
    let two = ValueCard new (Diamonds; 2)

### 2.6. Функции

Функции объявляются в теле программы ключевым словом `fun`:

    fun negate of int x:int -> -x

    fun hypo of int a:int b:int ->
        let sq1 = a * a
        let sq2 = b * b
        Math::Sqrt (sq1 + sq2)
        
После названия функции идет ее тип, после - список параметров с типами.

Каждая функция имеет свое пространство имен. Переменные, объявленные в глобальной
области видимости, _не доступны_ внутри функций.

#### 2.61. Аргументы функции

После слова `fun` идет название функции и тип возвращаемого значения,
а потом ее аргументы с указанием типа. Если у функции не объявлено
ни одного параметра, она будет принимать тип `unit` для вызова.
Литералом `unit` является выражение `()`.

Ключевое слово `unit` является внутренним именованием типа. Его нельзя использовать
для описания типа аргумента функции, в качестве generic-параметра другого типа и в
операторах `default` и `typeof`.

#### 2.6.2. Возвращаемое значение функции

Любая функция должна возвращать значение. Возвращаемым значением является последнее
выражение тела функции. Если последнее выражение - управляющая конструкция или вызов
функции типа `void`, функция возвращает тип `unit`.

Если функция не должна возвращать никакого значения, а ее последнее выражение не является
`void`, следует использоаать литерал `()`.

#### 2.6.3 Вызов функции

Функция вызывается, когда ей передаются все требуемые параметры. Для того, чтобы
вызвать функцию без параметров, ей нужно передать параметр типа `unit` - пара скобок `()`.

    fun sum of int a:int b:int c:int -> a + b + c
    fun getTen of int -> 10
    
    let five = sum 1 1 3
    let ten = getTen ()

При вызове функции можно использовать только литералы и имена переменных. Любые более сложные
выражения должны быть взяты в скобки.

    fun sum of double a:double b:double -> a + b

    let sum = sqrt sin 1        // sqrt(sin, 1) - wtf?
    let sum = sqrt (sin 1)      // компилируется
    let someData = sum (sin 1) (cos 2)
    
#### 2.6.4. Передача аргумента по ссылке

Аргумент в функцию можно передать по ссылке. Для этого как в объявлении, так и при вызове
следует использовать модификатор `ref`:

    fun test of bool str: ref string ->
        if(str.Length > 100)
            str = str.Substring 0 100
            true
        else
            false
            
    var a = "hello world"
    var b = "test"
    println (test ref a) // true
    println (test ref b) // false
    
После `ref` может использоваться:

* Имя переменной, объявленной с помощью `var`
* Имя аргумента текущей функции
* Обращение к полю
* Обращение к индексу массива
 
Не может быть использовано:

* Литерал, выражение или имя константы, объявленной с помощью `let`
* Обращение к свойству
* Обращение к индексу объекта с переопределенным индексатором

#### 2.6.5. Анонимные функции

Анонимные функции могут быть объявлены (практически) в любом месте программы.
Помимо отсутствия имени они отличаются от именованных функций следующими моментами:

1. Анонимная функция замыкает переменные и константы из внешней области видимости.
2. Тип анонимной функции выводится автоматически, поскольку она не может быть рекурсивной.

Анонимная функция может быть описана следующим образом:

    let sum = (a:int b:int) -> a + b
    let getTen = -> sum 5 5
    let addFive = (a:int) ->
        let b = 5
        sum a b // то же самое, что sum 5
        
Как видно из следующего примера, оператор `->` разделяет параметры функции и
ее тело. Даже если параметров нет, `->` все равно необходимо указывать.

#### 2.6.6. Чистые функции и мемоизация

При объявлении именованной функции ее можно пометить модификатором `pure`. Это
означает, что при равных входных параметрах ее результат всегда будет
одинаковым. В таком случае при первом вызове ее результат будет закеширован,
а при повторных вызовах будет использоваться именно этот кеш, а сама функция
не будет повторно вызвана.

Чистота функции не проверяется компилятором. Фактическое наличие побочных
эффектов остается на совести программиста.

#### 2.6.7. Порядок объявления и вызова функций

Порядок не играет роли. Рекурсивные вызовы допустимы без какого-либо явного
указания (например, в F# требуется модификатор `rec`), взаимная рекурсия
также допустима.

#### 2.6.8. Оператор передачи значения

Для передачи значения в функцию может быть использован оператор `<|`.
Однако этот оператор будет полезен, если аргументы не умещаются на одной строке, или
если требуется передать многострочное выражение.

Оператор `<|` требует увеличения отступа относительно выражения, к которому он применяется.

    somefx
        <| value1
        <| (a:int b:int) ->
            let sum = a + b
            sum * sum
        <| (s:string) -> log s
        
#### 2.6.9. Оператор передачи контекста

Для вызова функций по цепочке, особенно со сложными аргументами, удобно использовать
оператор передачи контекста, аналогичный точке. Он позволяет размещать длинное
выражение на нескольких строках.

    someData
        |> Where ((a:MyObj) -> a.Value > 10)
        |> Select ((a:MyObj) -> a.Value ** 2)
        |> Sum ()

### 2.7. Ключевые слова и конструкции

#### 2.7.1. Создание объектов

Новые объекты создаются с помощью ключевого слова `new`:

    let tuple = new Tuple<string, int> "hello" 2

#### 2.7.2. Условие

Условие записывается с помощью блока if / else:
    
    let a = if (1 > 2) 3 else 4

Выражение может также использоваться по правую сторону от знака присваивания,
если указаны обе ветки (`if` и `else`). Если блок `else` не используется, конструкция
`if` всегда возвращает тип `unit`.

#### 2.7.3. Цикл

Цикл записывается с помощью блока `while`:

    var a = 0
    while (a < 10)
       Console.WriteLine "{0} loop iteration" a
       a = a + 1

Цикл `while` всегда возвращает значение последнего выражения в теле цикла.
Если цикл не был выполнен ни одного раза, будет возвращено выражение `default(T)`.

#### 2.7.4. try-catch

Блоки `try-catch` записываются следующим образом:

    try
        doSomethingHorrible()
    catch(WebException ex)
        notify "web exception" ex.Message
    catch(DivideByZeroException ex)
        notify "whoops!"
    catch
        notify "something weird has happened"

Блок `try-catch` всегда возвращает `unit`.

#### 2.7.5. using

Ключевое слово using открывает пространство имен, добавляя объявленные в нем
классы в глобальное:

    using System.Text.RegularExpressions
    let rx = new Regex "[a-z]{2}"

#### 2.7.6. Приведение и проверка типов

Для приведения типов используется оператор `as`. В отличие от C#, он кидает
`InvalidCastException` в случае неудачи, а не возвращает `null`. Может быть
использован на любых типах, в том числе `int` / `string` / `bool` / `object`.

Для проверки того, является ли объект экземпляром некоторого класса, используется
оператор `is`. Он возвращает `bool`.

### 2.8. Создание структур данных

В языке есть поддержка для упрощенного создания заранее инициализированных
коллекций разного типа. Для этого используется специальный синтаксис оператора `new`.

Данный синтаксис используется только в том случае, если количество элементов
заранее известно и оно отлично от нуля. Для объявления пустых структур данных
следует пользоваться их классическими конструкторами. Объявить пустой массив можно
с помощью `System.Array.CreateInstance(...)`. Возможно, следует добавить для этого
случая generic-метод.

Тип коллекции выводится автоматически из типов аргументов. Для этого в коллекции должен
присутствовать хотя бы один элемент, отличный от null.

Ключи `Dictionary` проверяются более строго - они не могут иметь значение `null` и их тип
должен совпадать в точности.

#### 2.8.1. Массивы

    // int[]
    let ints = new [1; 2; 3]

#### 2.8.2. Списки

     // System.Collections.Generic.List<int>
    let ints = new [[1; 2; 3]]

#### 2.8.3 Словари

    // System.Collections.Generic.Dictionary<string, int>
    let dict = new { "hello" => 1; "world" => 2 }

#### 2.8.4

    // System.Tuple<int, string, object>
    let t = new (1, "hello world", new object())

В кортеже должно быть от 1 до 7 элементов. Кортежи неограниченной длины, возможно,
будут поддерживаться в следующей версии.

### 2.9 Делегаты

Анонимные функции всегда являются выражениями типа `Func<>` или `Action<>`.
Для того, чтобы использовать их в функциях, принимающих другие типы делегатов, можно
использовать приведение типов. Для этого типы принимаемых и возвращаемых значений должны
в точности соответствовать:

    let filter = (x:int) -> x % 2 == 0
    let data = Enumerable.Range 1 100 |> ToArray ()
    let even = Array::FindAll data (filter as Predicate<int>)

## 3. Встраиваемость

Технически, интерпретатор будет реализован в виде сборки .NET, которую
программист может подключить к своей программе, чтобы добавить в нее
поддержку скриптового языка.

Сборка содержит основной класс интерпретатора. Схема работы программиста с
интерпретатором следующая:

1. Добавить в проект ссылку на сборки LENS
2. Создать объект интерпретатора
3. Зарегистрировать в интерпретаторе свои типы, функции и свойства
4. Передать интерпретатору текст исполняемой программы

Результатом является объект типа `Func<object>`, позволяющий исполнять скрипт многократно
без необходимости перекомпиляции.
  
Примерный код этого взаимодействия на языке C# представлен ниже:

    public void Run()
    {
        var source = "a = 1 + 2";
        var a = 0;

        var compiler = new LensCompiler();
        compiler.RegisterProperty("a", () => a, newA => a = newA);
        
        try
        {
            var fx = compiler.Compile(source);
            fx();

            Console.WriteLine("Success: {0}", a);
        }
        catch (LensCompilerException ex)
        {
            Console.WriteLine("Error: {0}", ex.FullMessage);
        }
    }

## 4. Дополнительные возможности

* Поддержка переопределенных операторов
* Раскрутка констант во время компиляции
* Сохранение сгенерированной в виде исполняемого файла
* Возможость отключать поиск extension-методов для ускорения компиляции

## 5. Ограничения

### 5.1. Планы на будущее

Уже реализовано, но не включено в парсер:

* Мемоизация функций
* Цикл `foreach`
* Блок `finally` в конструкции `try`
* Побитовые операторы и операторы сдвига
 
Будет реализовано в дальнейших версиях:

* Сокращенное присваивание (+= и т.д.)
* Подписка на события
* Pattern matching
* Объявление generic-функций
* Объявление generic-структур

### 5.2. Сознательные ограничения

Поскольку LENS является встраиваемым языком, в нем не будет вещей, присущих
классическим языкам программирования, как то:

* Создание полноценных классов с методами
* Модификаторы доступа
* Объявление интерфейсов
* Управляющие конструкции, прерывающие поток выполнения: `return`, `break`, `continue`
