# Custom Scripts for MrRSS

MrRSS supports custom scripts that can be used to fetch RSS feeds from sources that don't provide standard RSS/Atom feeds.

## How It Works

1. Place your scripts in the `scripts` folder inside the MrRSS data directory
2. When adding a new feed, select "Custom Script" instead of "RSS URL"
3. Choose your script from the dropdown
4. The script will be executed whenever the feed is refreshed

## Script Requirements

Your script must output valid RSS or Atom XML to stdout. The output should follow this structure:

```xml
<?xml version="1.0" encoding="UTF-8"?>
<rss version="2.0">
  <channel>
    <title>Feed Title</title>
    <link>https://example.com</link>
    <description>Feed description</description>
    <item>
      <title>Article Title</title>
      <link>https://example.com/article1</link>
      <description>Article content or summary</description>
      <pubDate>Mon, 01 Jan 2024 12:00:00 GMT</pubDate>
    </item>
    <!-- More items... -->
  </channel>
</rss>
```

## Supported Script Types

| Extension | Language | Command Used |
|-----------|----------|--------------|
| `.py` | Python | `python3` (or `python` on Windows) |
| `.sh` | Shell | `bash` |
| `.ps1` | PowerShell | `powershell.exe` (Windows) or `pwsh` |
| `.js` | Node.js | `node` |
| `.rb` | Ruby | `ruby` |

## Example Scripts

### Python Example

```python
#!/usr/bin/env python3
import json
from datetime import datetime

# Fetch data from your source
# items = fetch_from_api()

# Generate RSS output
print('<?xml version="1.0" encoding="UTF-8"?>')
print('<rss version="2.0">')
print('  <channel>')
print('    <title>My Custom Feed</title>')
print('    <link>https://example.com</link>')
print('    <description>Custom feed generated by script</description>')

# Example items
items = [
    {"title": "Article 1", "link": "https://example.com/1", "desc": "Description 1"},
    {"title": "Article 2", "link": "https://example.com/2", "desc": "Description 2"},
]

for item in items:
    print(f'    <item>')
    print(f'      <title>{item["title"]}</title>')
    print(f'      <link>{item["link"]}</link>')
    print(f'      <description>{item["desc"]}</description>')
    print(f'    </item>')

print('  </channel>')
print('</rss>')
```

### Shell Example

```bash
#!/bin/bash

cat << 'EOF'
<?xml version="1.0" encoding="UTF-8"?>
<rss version="2.0">
  <channel>
    <title>Shell Script Feed</title>
    <link>https://example.com</link>
    <description>Feed generated by shell script</description>
    <item>
      <title>Sample Article</title>
      <link>https://example.com/sample</link>
      <description>This is a sample article</description>
    </item>
  </channel>
</rss>
EOF
```

### Node.js Example

```javascript
#!/usr/bin/env node

const items = [
  { title: 'Article 1', link: 'https://example.com/1', desc: 'Description 1' },
  { title: 'Article 2', link: 'https://example.com/2', desc: 'Description 2' },
];

console.log('<?xml version="1.0" encoding="UTF-8"?>');
console.log('<rss version="2.0">');
console.log('  <channel>');
console.log('    <title>Node.js Feed</title>');
console.log('    <link>https://example.com</link>');
console.log('    <description>Feed generated by Node.js</description>');

items.forEach(item => {
  console.log('    <item>');
  console.log(`      <title>${item.title}</title>`);
  console.log(`      <link>${item.link}</link>`);
  console.log(`      <description>${item.desc}</description>`);
  console.log('    </item>');
});

console.log('  </channel>');
console.log('</rss>');
```

## Tips

1. **Error Handling**: If your script encounters an error, write the error message to stderr. MrRSS will display this in the feed's error indicator.

2. **Timeout**: Scripts have a 30-second timeout. If your script takes longer, it will be terminated.

3. **Working Directory**: Scripts are executed with the scripts folder as the working directory.

4. **Dependencies**: Make sure any required dependencies (Python packages, Node modules, etc.) are installed on your system.

5. **Testing**: Test your script from the command line first to ensure it produces valid RSS output:

   ```bash
   python3 your_script.py | xmllint --noout -
   ```

## Troubleshooting

- **Script not found**: Make sure the script file is in the scripts folder and has the correct extension
- **Permission denied**: On Unix-like systems, ensure the script is executable (`chmod +x script.py`)
- **Invalid output**: Verify your script outputs valid XML by testing it manually
- **Encoding issues**: Always use UTF-8 encoding in your scripts

## Data Directory Location

The scripts folder is located in the MrRSS data directory:

- **Windows**: `%APPDATA%\MrRSS\scripts\`
- **macOS**: `~/Library/Application Support/MrRSS/scripts/`
- **Linux**: `~/.local/share/MrRSS/scripts/`

You can also open the scripts folder directly from the MrRSS interface when adding a new feed.
