package discovery

import (
	"context"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"
	"time"
)

func TestNewService(t *testing.T) {
	service := NewService()
	if service == nil {
		t.Fatal("NewService returned nil")
	}
	if service.client == nil {
		t.Error("client is nil")
	}
	if service.feedParser == nil {
		t.Error("feedParser is nil")
	}
}

func TestResolveURL(t *testing.T) {
	service := NewService()

	tests := []struct {
		base     string
		href     string
		expected string
	}{
		{"https://example.com/", "/about", "https://example.com/about"},
		{"https://example.com/blog/", "post.html", "https://example.com/blog/post.html"},
		{"https://example.com/", "https://other.com/page", "https://other.com/page"},
		{"https://example.com/", "", ""},
	}

	for _, test := range tests {
		result := service.resolveURL(test.base, test.href)
		if result != test.expected {
			t.Errorf("resolveURL(%q, %q) = %q; want %q", test.base, test.href, result, test.expected)
		}
	}
}

func TestIsValidBlogDomain(t *testing.T) {
	service := NewService()

	validDomains := []string{
		"myblog.com",
		"example.blog",
		"personal-website.net",
	}

	invalidDomains := []string{
		"facebook.com",
		"www.twitter.com",
		"github.com",
		"stackoverflow.com",
	}

	for _, domain := range validDomains {
		if !service.isValidBlogDomain(domain) {
			t.Errorf("Expected %q to be valid blog domain", domain)
		}
	}

	for _, domain := range invalidDomains {
		if service.isValidBlogDomain(domain) {
			t.Errorf("Expected %q to be invalid blog domain", domain)
		}
	}
}

func TestGetFavicon(t *testing.T) {
	service := NewService()

	blogURL := "https://example.com/blog"
	favicon := service.getFavicon(blogURL)

	expected := "https://www.google.com/s2/favicons?domain=example.com"
	if favicon != expected {
		t.Errorf("getFavicon(%q) = %q; want %q", blogURL, favicon, expected)
	}
}

func TestDiscoverFromFeedWithTimeout(t *testing.T) {
	if testing.Short() {
		t.Skip("Skipping network test in short mode")
	}

	service := NewService()
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()

	// Test with a non-existent feed URL (should fail gracefully)
	_, err := service.DiscoverFromFeed(ctx, "https://nonexistent-feed-url-12345.com/feed")
	if err == nil {
		t.Log("Expected error for non-existent feed, but got none (this is acceptable)")
	}
}

func TestProgressCallbackCalled(t *testing.T) {
	if testing.Short() {
		t.Skip("Skipping network test in short mode")
	}

	service := NewService()
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()

	progressCalled := false
	progressCb := func(progress Progress) {
		progressCalled = true
		// Verify progress has expected fields
		if progress.Stage == "" {
			t.Error("Progress stage should not be empty")
		}
	}

	// Test with a non-existent feed URL - progress callback should still be called
	_, _ = service.DiscoverFromFeedWithProgress(ctx, "https://nonexistent-feed-url-12345.com/feed", progressCb)

	if !progressCalled {
		t.Error("Progress callback should have been called at least once")
	}
}

func TestProgressStructFields(t *testing.T) {
	// Test that Progress struct can hold all expected fields
	p := Progress{
		Stage:      "checking_rss",
		Message:    "Checking RSS feed",
		Detail:     "https://example.com",
		Current:    5,
		Total:      10,
		FeedName:   "Test Feed",
		FoundCount: 3,
	}

	if p.Stage != "checking_rss" {
		t.Errorf("Expected stage 'checking_rss', got %q", p.Stage)
	}
	if p.Current != 5 {
		t.Errorf("Expected current 5, got %d", p.Current)
	}
	if p.Total != 10 {
		t.Errorf("Expected total 10, got %d", p.Total)
	}
	if p.FoundCount != 3 {
		t.Errorf("Expected found_count 3, got %d", p.FoundCount)
	}
}

// Helper to create a service with a custom http client (used in network tests)
func newServiceWithClient(client *http.Client) *Service {
	s := NewService()
	if client != nil {
		s.client = client
	}
	return s
}

func TestIsValidFeed_HEAD_OK(t *testing.T) {
	srv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.Method == "HEAD" {
			w.Header().Set("Content-Type", "application/rss+xml")
			w.WriteHeader(200)
			return
		}
		w.WriteHeader(200)
		_, _ = w.Write([]byte("<rss></rss>"))
	}))
	defer srv.Close()

	s := newServiceWithClient(srv.Client())
	ok := s.isValidFeed(context.Background(), srv.URL)
	if !ok {
		t.Fatalf("expected feed to be valid via HEAD")
	}
}

func TestIsValidFeed_HEAD_FallbackToGET_XML(t *testing.T) {
	srv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.Method == "HEAD" {
			w.WriteHeader(404)
			return
		}
		w.WriteHeader(200)
		_, _ = w.Write([]byte("<?xml version=\"1.0\"?><rss><channel><title>t</title></channel></rss>"))
	}))
	defer srv.Close()

	s := newServiceWithClient(srv.Client())
	ok := s.isValidFeed(context.Background(), srv.URL)
	if !ok {
		t.Fatalf("expected feed to be valid via GET content check")
	}
}

func TestFindRSSFeed_LinkInHead(t *testing.T) {
	// Serve homepage with link rel to /feed.xml and feed at /feed.xml
	mux := http.NewServeMux()
	mux.HandleFunc("/", func(w http.ResponseWriter, r *http.Request) {
		html := `<html><head><link rel="alternate" type="application/rss+xml" href="/feed.xml"></head><body></body></html>`
		w.WriteHeader(200)
		_, _ = w.Write([]byte(html))
	})
	mux.HandleFunc("/feed.xml", func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(200)
		_, _ = w.Write([]byte("<?xml version=\"1.0\"?><rss><channel><title>Feed</title><item><title>one</title></item></channel></rss>"))
	})

	srv := httptest.NewServer(mux)
	defer srv.Close()

	s := newServiceWithClient(srv.Client())
	feedURL, err := s.findRSSFeed(context.Background(), srv.URL)
	if err != nil {
		t.Fatalf("findRSSFeed error: %v", err)
	}
	if !strings.HasSuffix(feedURL, "/feed.xml") {
		t.Fatalf("expected feed URL to end with /feed.xml, got %s", feedURL)
	}
}

func TestGetFaviconAndResolveURLAndExtractLinks(t *testing.T) {
	// Serve homepage with friend link page and friends page pointing to external blog
	mux := http.NewServeMux()
	mux.HandleFunc("/", func(w http.ResponseWriter, r *http.Request) {
		html := `<html><body><a href="/friends">Friend Links</a></body></html>`
		w.WriteHeader(200)
		_, _ = w.Write([]byte(html))
	})
	mux.HandleFunc("/friends", func(w http.ResponseWriter, r *http.Request) {
		html := `<html><body><a href="https://external.example.com/blog">Blog</a></body></html>`
		w.WriteHeader(200)
		_, _ = w.Write([]byte(html))
	})

	srv := httptest.NewServer(mux)
	defer srv.Close()

	s := newServiceWithClient(srv.Client())

	// getFavicon
	fav := s.getFavicon(srv.URL)
	if fav == "" || !strings.Contains(fav, "google.com/s2/favicons") {
		t.Fatalf("unexpected favicon: %s", fav)
	}

	// findFriendLinkPage
	friendURL, err := s.findFriendLinkPage(context.Background(), srv.URL)
	if err != nil {
		t.Fatalf("findFriendLinkPage error: %v", err)
	}
	if !strings.HasSuffix(friendURL, "/friends") {
		t.Fatalf("expected friend page to end with /friends, got %s", friendURL)
	}

	// fetch and extract external links via findFriendLinks
	links, err := s.findFriendLinks(context.Background(), srv.URL)
	if err != nil {
		t.Fatalf("findFriendLinks error: %v", err)
	}
	if len(links) != 1 || !strings.Contains(links[0], "external.example.com") {
		t.Fatalf("expected external link, got %v", links)
	}

	// resolveURL
	resolved := s.resolveURL(srv.URL, "/a/b")
	if !strings.HasPrefix(resolved, srv.URL) {
		t.Fatalf("resolveURL failed: %s", resolved)
	}
}
