#!/usr/bin/env node
/**
 * Example script for MrRSS - Node.js RSS Feed Generator
 *
 * This script demonstrates how to create a custom RSS feed script using Node.js.
 * Place this file in the MrRSS scripts directory and select it when adding a new feed.
 *
 * The script outputs valid RSS 2.0 XML to stdout.
 */

function escapeXml(text) {
  return text
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&apos;');
}

function formatDate(date) {
  return date.toUTCString();
}

function generateRss() {
  const now = new Date();
  const currentDate = formatDate(now);

  // Example items - replace with your own data source
  const items = [
    {
      title: 'Welcome to MrRSS Custom Scripts',
      link: 'https://github.com/WCY-dt/MrRSS',
      description: 'This is an example article generated by a custom Node.js script.',
      pubDate: currentDate
    },
    {
      title: 'How to Create Custom Feed Scripts',
      link: 'https://github.com/WCY-dt/MrRSS/blob/main/docs/CUSTOM_SCRIPTS.md',
      description: 'Learn how to create your own custom scripts to fetch content from any source.',
      pubDate: currentDate
    }
  ];

  // RSS header
  console.log('<?xml version="1.0" encoding="UTF-8"?>');
  console.log('<rss version="2.0">');
  console.log('  <channel>');
  console.log('    <title>Example Node.js Feed</title>');
  console.log('    <link>https://github.com/WCY-dt/MrRSS</link>');
  console.log('    <description>An example feed generated by a Node.js script</description>');
  console.log(`    <lastBuildDate>${currentDate}</lastBuildDate>`);

  // Generate items
  for (const item of items) {
    console.log('    <item>');
    console.log(`      <title>${escapeXml(item.title)}</title>`);
    console.log(`      <link>${escapeXml(item.link)}</link>`);
    console.log(`      <description>${escapeXml(item.description)}</description>`);
    console.log(`      <pubDate>${item.pubDate}</pubDate>`);
    console.log('    </item>');
  }

  // RSS footer
  console.log('  </channel>');
  console.log('</rss>');
}

try {
  generateRss();
} catch (error) {
  console.error(`Error generating feed: ${error.message}`);
  process.exit(1);
}
