#!/usr/bin/env python3
"""
Example script for MrRSS - Python RSS Feed Generator

This script demonstrates how to create a custom RSS feed script.
Place this file in the MrRSS scripts directory and select it when adding a new feed.

The script outputs valid RSS 2.0 XML to stdout.
"""

import sys
from datetime import datetime, timezone

def generate_rss():
    """Generate a sample RSS feed."""
    # Example items - replace with your own data source
    items = [
        {
            "title": "Welcome to MrRSS Custom Scripts",
            "link": "https://github.com/WCY-dt/MrRSS",
            "description": "This is an example article generated by a custom Python script.",
            "pubDate": datetime.now(timezone.utc).strftime("%a, %d %b %Y %H:%M:%S GMT")
        },
        {
            "title": "How to Create Custom Feed Scripts",
            "link": "https://github.com/WCY-dt/MrRSS/blob/main/docs/CUSTOM_SCRIPTS.md",
            "description": "Learn how to create your own custom scripts to fetch content from any source.",
            "pubDate": datetime.now(timezone.utc).strftime("%a, %d %b %Y %H:%M:%S GMT")
        }
    ]

    # RSS header
    print('<?xml version="1.0" encoding="UTF-8"?>')
    print('<rss version="2.0">')
    print('  <channel>')
    print('    <title>Example Python Feed</title>')
    print('    <link>https://github.com/WCY-dt/MrRSS</link>')
    print('    <description>An example feed generated by a Python script</description>')
    print(f'    <lastBuildDate>{datetime.now(timezone.utc).strftime("%a, %d %b %Y %H:%M:%S GMT")}</lastBuildDate>')

    # Generate items
    for item in items:
        print('    <item>')
        print(f'      <title>{escape_xml(item["title"])}</title>')
        print(f'      <link>{escape_xml(item["link"])}</link>')
        print(f'      <description>{escape_xml(item["description"])}</description>')
        print(f'      <pubDate>{item["pubDate"]}</pubDate>')
        print('    </item>')

    # RSS footer
    print('  </channel>')
    print('</rss>')

def escape_xml(text):
    """Escape special XML characters."""
    return (text
        .replace('&', '&amp;')
        .replace('<', '&lt;')
        .replace('>', '&gt;')
        .replace('"', '&quot;')
        .replace("'", '&apos;'))

if __name__ == '__main__':
    try:
        generate_rss()
    except Exception as e:
        print(f"Error generating feed: {e}", file=sys.stderr)
        sys.exit(1)
