# Copyright (c) Meta Platforms, Inc. and affiliates.
# Copyright (c) WhatsApp LLC
#
# This source code is licensed under the MIT license found in the
# LICENSE.md file in the root directory of this source tree.

import glob
import hashlib
import os
import re
from dataclasses import dataclass, field
from typing import Any, Optional

import yaml

from jinja2 import Environment, FileSystemLoader, Template
from jinja2.ext import Extension

from .context import Context
from .schema import Root as RootSchema

TOKEN: str = "".join(["<<", "SignedSource", "::", "*O*zOeWoEQle#+L!plEphiEmie@IsG", ">>"])


class SignedSourceC(Extension):
    HEADER: str = "".join(
        [
            "// DO NOT EDIT: this file was generated by 'just codegen'\n",
            "// ",
            "@",
            "g",
            "enerated ",
            TOKEN,
            "\n",
        ]
    )

    def preprocess(self, source: str, name: Optional[str], filename: Optional[str] = None) -> str:
        return self.HEADER + source


class SignedSourceErl(Extension):
    PREFIX: str = "".join(["%" * 3, " %"])
    HEADER: str = "".join(
        [
            PREFIX,
            " DO NOT EDIT: this file was generated by 'just codegen'\n",
            PREFIX,
            " @",
            "g",
            "enerated ",
            TOKEN,
            "\n",
        ]
    )

    def preprocess(self, source: str, name: Optional[str], filename: Optional[str] = None) -> str:
        return self.HEADER + source


@dataclass
class Stage:
    codegen: "CodeGenerator" = field(repr=False)
    templates_path: str
    targets: list["StageTarget"] = field(init=False)
    c_env: Environment = field(init=False)
    erl_env: Environment = field(init=False)

    def __post_init__(self) -> None:
        self.targets = []
        for template_name in glob.iglob("**", root_dir=self.templates_path, recursive=True):
            if os.path.isfile(os.path.join(self.templates_path, template_name)):
                output_name, template_ext = os.path.splitext(template_name)
                if template_ext == ".j2":
                    self.targets.append(
                        StageTarget(
                            stage=self,
                            template_name=template_name,
                            output_name=output_name,
                        )
                    )
        self.c_env = Environment(
            loader=FileSystemLoader(self.templates_path, encoding="utf-8"),
            comment_start_string="/* clang-format ",
            comment_end_string=" */ //-",
            trim_blocks=True,
            lstrip_blocks=True,
        )
        self.c_env.add_extension("jinja2.ext.debug")
        self.c_env.add_extension(SignedSourceC)
        self.erl_env = Environment(
            loader=FileSystemLoader(self.templates_path, encoding="utf-8"),
            # block_start_string="<<{%",
            # block_end_string="%}>>",
            # variable_start_string="<<{{",
            # variable_end_string="}}>>",
            # comment_start_string="<<{#",
            # comment_end_string="#}>>",
            block_start_string="~J{",
            block_end_string="}.",
            variable_start_string="~J<",
            variable_end_string=">.",
            comment_start_string="~J#",
            comment_end_string="#.",
            trim_blocks=True,
            lstrip_blocks=True,
        )
        self.erl_env.add_extension("jinja2.ext.debug")
        self.erl_env.add_extension(SignedSourceErl)

    def clean(self) -> None:
        for target in self.targets:
            target.clean()
        return

    def render(self) -> None:
        for target in self.targets:
            target.render()
        return

    def sign(self) -> None:
        for target in self.targets:
            target.sign()
        return


@dataclass
class StageTarget:
    stage: "Stage" = field(repr=False)
    template_name: str
    output_name: str
    output_path: str = field(init=False)
    output_file: str = field(init=False)
    is_erl: bool = field(init=False)

    def __post_init__(self) -> None:
        _, output_ext = os.path.splitext(self.output_name)
        if output_ext in (".erl", ".hrl", ".xrl", ".yrl"):
            self.is_erl = True
        else:
            self.is_erl = False
        self.output_file = os.path.join(self.codegen.output_path, self.output_name)
        self.output_path = os.path.dirname(self.output_file)
        return

    @property
    def codegen(self) -> "CodeGenerator":
        return self.stage.codegen

    @property
    def env(self) -> Environment:
        if self.is_erl:
            return self.stage.erl_env
        else:
            return self.stage.c_env

    @property
    def label(self) -> str:
        if self.is_erl:
            return "E"
        else:
            return "C"

    def clean(self) -> None:
        if os.path.isfile(self.output_file):
            os.remove(self.output_file)
        return

    def render(self) -> None:
        os.makedirs(self.output_path, exist_ok=True)
        template: Template = self.env.get_template(self.template_name)
        print(f"[{self.label}] {self.output_name}")
        template.stream(ctx=self.codegen.ctx).dump(self.output_file, encoding="utf-8")
        # self.sign()
        return

    def sign(self) -> None:
        token: bytes = TOKEN.encode("utf-8")
        needle: re.Pattern = re.compile(rb"(?:SignedSource<<(?P<md5>[a-f0-9]{32})>>)|" + re.escape(token))
        with open(self.output_file, "rb+") as f:
            haystack: bytes = f.read()
            md5: hashlib.md5 = hashlib.md5()
            index: int = 0
            for match in needle.finditer(haystack):
                if match.group(1) is None:
                    md5.update(haystack[index : match.end()])
                else:
                    md5.update(haystack[index : match.end()])
                    md5.update(token)
                index = match.end()
            if index > 0:
                md5.update(haystack[index:])
                signature: str = f"SignedSource<<{md5.hexdigest()}>>"
                for match in needle.finditer(haystack):
                    f.seek(match.start())
                    f.write(signature.encode("utf-8"))


@dataclass
class CodeGenerator:
    config_file: str
    output_path: str
    templates0_path: str
    templates1_path: str
    ctx: Context = field(init=False)
    stage0: Stage = field(init=False)
    stage1: Stage = field(init=False)

    def __post_init__(self) -> None:
        with open(self.config_file, "r", encoding="utf-8") as f:
            data: Any = yaml.safe_load(f)
            schema: RootSchema = RootSchema(**data)
            self.ctx = Context(_output_path=self.output_path, schema=schema)
        self.stage0 = Stage(codegen=self, templates_path=self.templates0_path)
        self.stage1 = Stage(codegen=self, templates_path=self.templates1_path)

    def render(self) -> None:
        print("Stage1 Clean")
        self.stage1.clean()
        print("Stage0 Render")
        self.stage0.render()
        self._analyze_c_src_nif_atoms()
        print("Stage1 Render")
        self.stage1.render()

    def sign(self) -> None:
        print("Stage0 Sign")
        self.stage0.sign()
        print("Stage1 Sign")
        self.stage1.sign()

    def _analyze_c_src_nif_atoms(self) -> None:
        c_src_nif_root: str = os.path.join(self.output_path, "apps", "erldist_filter", "c_src", "nif")
        for source_name in glob.iglob("**", root_dir=c_src_nif_root, recursive=True):
            source_file: str = os.path.join(c_src_nif_root, source_name)
            if os.path.isfile(source_file):
                _, actual_ext = os.path.splitext(source_name)
                if actual_ext in (".c", ".h", ".c.h"):
                    with open(source_file, "r", encoding="utf-8") as f:
                        contents: str = f.read()
                        for atom_name in re.finditer(r"\bATOM\((.*?)\)", contents):
                            self.ctx.nif.make_atom(atom_name.group(1))
