// Copyright (c) 2010 Satoshi Nakamoto
// Copyright (c) 2009-2014 The Bitcoin Core developers
// Distributed under the MIT software license, see the accompanying
// file COPYING or http://www.opensource.org/licenses/mit-license.php.

/******************************************************************************
 * Copyright © 2014-2019 The SuperNET Developers.                             *
 *                                                                            *
 * See the AUTHORS, DEVELOPER-AGREEMENT and LICENSE files at                  *
 * the top-level directory of this distribution for the individual copyright  *
 * holder information and the developer policies on copyright and licensing.  *
 *                                                                            *
 * Unless otherwise agreed in a custom licensing agreement, no part of the    *
 * SuperNET software, including this file may be copied, modified, propagated *
 * or distributed except according to the terms contained in the LICENSE file *
 *                                                                            *
 * Removal or modification of this copyright notice is prohibited.            *
 *                                                                            *
 ******************************************************************************/

#include "rpc/client.h"
#include "rpc/protocol.h"
#include "util.h"

#include <set>
#include <stdint.h>

#include <univalue.h>

using namespace std;

class CRPCConvertParam
{
public:
    std::string methodName;            //! method whose params want conversion
    int paramIdx;                      //! 0-based idx of param to convert
};

static const CRPCConvertParam vRPCConvertParams[] =
{
    { "stop", 0 },
    { "setmocktime", 0 },
    { "getaddednodeinfo", 0 },
    { "setgenerate", 0 },
    { "setgenerate", 1 },
    { "generate", 0 },
    { "getnetworkhashps", 0 },
    { "getnetworkhashps", 1 },
    { "getnetworksolps", 0 },
    { "getnetworksolps", 1 },
    { "sendtoaddress", 1 },
    { "sendtoaddress", 4 },
    { "settxfee", 0 },
    { "getnotarysendmany", 0 },
    { "getnotarysendmany", 1 },
    { "getreceivedbyaddress", 1 },
    { "getreceivedbyaccount", 1 },
    { "listreceivedbyaddress", 0 },
    { "listreceivedbyaddress", 1 },
    { "listreceivedbyaddress", 2 },
    { "listreceivedbyaccount", 0 },
    { "listreceivedbyaccount", 1 },
    { "listreceivedbyaccount", 2 },
    { "getbalance", 1 },
    { "getbalance", 2 },
    { "getblockhash", 0 },
    { "move", 2 },
    { "move", 3 },
    { "sendfrom", 2 },
    { "sendfrom", 3 },
    { "listtransactions", 1 },
    { "listtransactions", 2 },
    { "listtransactions", 3 },
    { "listaccounts", 0 },
    { "listaccounts", 1 },
    { "walletpassphrase", 1 },
    { "getblocktemplate", 0 },
    { "listsinceblock", 1 },
    { "listsinceblock", 2 },
    { "sendmany", 1 },
    { "sendmany", 2 },
    { "sendmany", 4 },
    { "addmultisigaddress", 0 },
    { "addmultisigaddress", 1 },
    { "createmultisig", 0 },
    { "createmultisig", 1 },
    { "listunspent", 0 },
    { "listunspent", 1 },
    { "listunspent", 2 },
    { "getblock", 1 },
    { "getblockheader", 1 },
    { "getchaintxstats", 0  },
    { "getlastsegidstakes", 0 },
    { "gettransaction", 1 },
    { "getrawtransaction", 1 },
    { "getlastsegidstakes", 0 },
    { "createrawtransaction", 0 },
    { "createrawtransaction", 1 },
    { "createrawtransaction", 2 },
    { "createrawtransaction", 3 },
    { "signrawtransaction", 1 },
    { "signrawtransaction", 2 },
    { "sendrawtransaction", 1 },
    { "fundrawtransaction", 1 },
    { "gettxout", 1 },
    { "gettxout", 2 },
    { "gettxoutproof", 0 },
    { "lockunspent", 0 },
    { "lockunspent", 1 },
    { "importprivkey", 2 },
    { "importprivkey", 3 },
    { "importprivkey", 4 },
    { "importaddress", 2 },
    { "verifychain", 0 },
    { "verifychain", 1 },
    { "keypoolrefill", 0 },
    { "getrawmempool", 0 },
    { "estimatefee", 0 },
    { "estimatepriority", 0 },
    { "prioritisetransaction", 1 },
    { "prioritisetransaction", 2 },
    { "setban", 2 },
    { "setban", 3 },
    { "setstakingsplit", 0 },
    { "getblockhashes", 0 },
    { "getblockhashes", 1 },
    { "getblockhashes", 2 },
    { "getspentinfo", 0},
    { "getaddresstxids", 0},
    { "getaddressbalance", 0},
    { "getaddressdeltas", 0},
    { "getaddressutxos", 0},
    { "getaddressmempool", 0},
    { "zcrawjoinsplit", 1 },
    { "zcrawjoinsplit", 2 },
    { "zcrawjoinsplit", 3 },
    { "zcrawjoinsplit", 4 },
    { "zcbenchmark", 1 },
    { "zcbenchmark", 2 },
    { "getblocksubsidy", 0},
    { "z_listaddresses", 0},
    { "z_listreceivedbyaddress", 1},
    { "z_listunspent", 0 },
    { "z_listunspent", 1 },
    { "z_listunspent", 2 },
    { "z_listunspent", 3 },
    { "z_getbalance", 1},
    { "z_getbalances", 1},
    { "z_gettotalbalance", 0},
    { "z_gettotalbalance", 1},
    { "z_gettotalbalance", 2},
    { "z_mergetoaddress", 0},
    { "z_mergetoaddress", 2},
    { "z_mergetoaddress", 3},
    { "z_mergetoaddress", 4},
    { "z_viewtransaction", 1},
    { "z_sendmany", 1},
    { "z_sendmany", 2},
    { "z_sendmany", 3},
    { "z_shieldcoinbase", 2},
    { "z_shieldcoinbase", 3},
    { "z_getoperationstatus", 0},
    { "z_getoperationresult", 0},
    { "paxprice", 4 },
    { "paxprices", 3 },
    { "paxpending", 0 },
    { "notaries", 2 },
    { "minerids", 1 },
    { "kvsearch", 1 },
    { "kvupdate", 4 },
    { "z_importkey", 2 },
    { "z_importviewingkey", 2 },
    { "z_getpaymentdisclosure", 1},
    { "z_getpaymentdisclosure", 2},
    // crosschain
    { "assetchainproof", 1},
    { "crosschainproof", 1},
    { "getproofroot", 2},
    { "getNotarisationsForBlock", 0},
    { "height_MoM", 1},
    { "calc_MoM", 2},
    { "migrate_completeimporttransaction", 1},

    { "getalldata", 0},
    { "getalldata", 1},
    { "getalldata", 2},
    { "getalldata", 3},
    { "zs_listtransactions", 0},
    { "zs_listtransactions", 1},
    { "zs_listtransactions", 2},
    { "zs_listtransactions", 3},
    { "zs_listtransactions", 4},
    { "zs_listsentbyaddress", 1},
    { "zs_listsentbyaddress", 2},
    { "zs_listsentbyaddress", 3},
    { "zs_listsentbyaddress", 4},
    { "zs_listsentbyaddress", 5},
    { "zs_listreceivedbyaddress", 1},
    { "zs_listreceivedbyaddress", 2},
    { "zs_listreceivedbyaddress", 3},
    { "zs_listreceivedbyaddress", 4},
    { "zs_listreceivedbyaddress", 5},
    { "zs_listspentbyaddress", 1},
    { "zs_listspentbyaddress", 2},
    { "zs_listspentbyaddress", 3},
    { "zs_listspentbyaddress", 4},
    { "zs_listspentbyaddress", 5},

    { "getsaplingwitness", 1},
    { "getsaplingwitness", 2},
    { "getsaplingwitnessatheight", 1},
    { "getsaplingwitnessatheight", 2},
    { "getsaplingwitnessatheight", 3},
    { "getsaplingblocks", 0},
    { "getsaplingblocks", 1},
    { "getsaplingblocks", 2},

    { "z_createbuildinstructions", 0 },
    { "z_createbuildinstructions", 1 },
    { "z_createbuildinstructions", 2 },
    { "z_createbuildinstructions", 3 }
};

class CRPCConvertTable
{
private:
    std::set<std::pair<std::string, int> > members;

public:
    CRPCConvertTable();

    bool convert(const std::string& method, int idx) {
        return (members.count(std::make_pair(method, idx)) > 0);
    }
};

CRPCConvertTable::CRPCConvertTable()
{
    const unsigned int n_elem =
        (sizeof(vRPCConvertParams) / sizeof(vRPCConvertParams[0]));

    for (unsigned int i = 0; i < n_elem; i++) {
        members.insert(std::make_pair(vRPCConvertParams[i].methodName,
                                      vRPCConvertParams[i].paramIdx));
    }
}

static CRPCConvertTable rpcCvtTable;

/** Non-RFC4627 JSON parser, accepts internal values (such as numbers, true, false, null)
 * as well as objects and arrays.
 */
UniValue ParseNonRFCJSONValue(const std::string& strVal)
{
    UniValue jVal;
    if (!jVal.read(std::string("[")+strVal+std::string("]")) ||
        !jVal.isArray() || jVal.size()!=1)
        throw runtime_error(string("Error JSON:")+strVal);
    return jVal[0];
}

/** Convert strings to command-specific RPC representation */
UniValue RPCConvertValues(const std::string &strMethod, const std::vector<std::string> &strParams)
{
    UniValue params(UniValue::VARR);

    for (unsigned int idx = 0; idx < strParams.size(); idx++) {
        const std::string& strVal = strParams[idx];
        if (!rpcCvtTable.convert(strMethod, idx)) {
            // insert string value directly
            params.push_back(strVal);
        } else {
            // parse string as JSON, insert bool/number/object/etc. value
            params.push_back(ParseNonRFCJSONValue(strVal));
        }
    }

    return params;
}
