// Copyright 2012-2013 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

// The following code was generated by "src/etc/unicode.py"

#![allow(missing_doc)]
#![allow(non_uppercase_statics)]

fn bsearch_range_table(c: char, r: &'static [(char,char)]) -> bool {
    use cmp::{Equal, Less, Greater};
    use slice::ImmutableVector;
    use option::None;
    r.bsearch(|&(lo,hi)| {
        if lo <= c && c <= hi { Equal }
        else if hi < c { Less }
        else { Greater }
    }) != None
}

pub mod general_category {
    static Cc_table : &'static [(char,char)] = &[
        ('\x00', '\x1f'), ('\x7f', '\x9f')
    ];

    pub fn Cc(c: char) -> bool {
        super::bsearch_range_table(c, Cc_table)
    }

    static Nd_table : &'static [(char,char)] = &[
        ('\x30', '\x39'), ('\u0660', '\u0669'),
        ('\u06f0', '\u06f9'), ('\u07c0', '\u07c9'),
        ('\u0966', '\u096f'), ('\u09e6', '\u09ef'),
        ('\u0a66', '\u0a6f'), ('\u0ae6', '\u0aef'),
        ('\u0b66', '\u0b6f'), ('\u0be6', '\u0bef'),
        ('\u0c66', '\u0c6f'), ('\u0ce6', '\u0cef'),
        ('\u0d66', '\u0d6f'), ('\u0e50', '\u0e59'),
        ('\u0ed0', '\u0ed9'), ('\u0f20', '\u0f29'),
        ('\u1040', '\u1049'), ('\u1090', '\u1099'),
        ('\u17e0', '\u17e9'), ('\u1810', '\u1819'),
        ('\u1946', '\u194f'), ('\u19d0', '\u19d9'),
        ('\u1a80', '\u1a99'), ('\u1b50', '\u1b59'),
        ('\u1bb0', '\u1bb9'), ('\u1c40', '\u1c49'),
        ('\u1c50', '\u1c59'), ('\ua620', '\ua629'),
        ('\ua8d0', '\ua8d9'), ('\ua900', '\ua909'),
        ('\ua9d0', '\ua9d9'), ('\uaa50', '\uaa59'),
        ('\uabf0', '\uabf9'), ('\uff10', '\uff19'),
        ('\U000104a0', '\U000104a9'), ('\U00011066', '\U0001106f'),
        ('\U000110f0', '\U000110f9'), ('\U00011136', '\U0001113f'),
        ('\U000111d0', '\U000111d9'), ('\U000116c0', '\U000116c9'),
        ('\U0001d7ce', '\U0001d7ff')
    ];

    pub fn Nd(c: char) -> bool {
        super::bsearch_range_table(c, Nd_table)
    }

    static Nl_table : &'static [(char,char)] = &[
        ('\u16ee', '\u16f0'), ('\u2160', '\u2182'),
        ('\u2185', '\u2188'), ('\u3007', '\u3007'),
        ('\u3021', '\u3029'), ('\u3038', '\u303a'),
        ('\ua6e6', '\ua6ef'), ('\U00010140', '\U00010174'),
        ('\U00010341', '\U00010341'), ('\U0001034a', '\U0001034a'),
        ('\U000103d1', '\U000103d5'), ('\U00012400', '\U00012462')
    ];

    pub fn Nl(c: char) -> bool {
        super::bsearch_range_table(c, Nl_table)
    }

    static No_table : &'static [(char,char)] = &[
        ('\xb2', '\xb3'), ('\xb9', '\xb9'),
        ('\xbc', '\xbe'), ('\u09f4', '\u09f9'),
        ('\u0b72', '\u0b77'), ('\u0bf0', '\u0bf2'),
        ('\u0c78', '\u0c7e'), ('\u0d70', '\u0d75'),
        ('\u0f2a', '\u0f33'), ('\u1369', '\u137c'),
        ('\u17f0', '\u17f9'), ('\u19da', '\u19da'),
        ('\u2070', '\u2070'), ('\u2074', '\u2079'),
        ('\u2080', '\u2089'), ('\u2150', '\u215f'),
        ('\u2189', '\u2189'), ('\u2460', '\u249b'),
        ('\u24ea', '\u24ff'), ('\u2776', '\u2793'),
        ('\u2cfd', '\u2cfd'), ('\u3192', '\u3195'),
        ('\u3220', '\u3229'), ('\u3248', '\u324f'),
        ('\u3251', '\u325f'), ('\u3280', '\u3289'),
        ('\u32b1', '\u32bf'), ('\ua830', '\ua835'),
        ('\U00010107', '\U00010133'), ('\U00010175', '\U00010178'),
        ('\U0001018a', '\U0001018a'), ('\U00010320', '\U00010323'),
        ('\U00010858', '\U0001085f'), ('\U00010916', '\U0001091b'),
        ('\U00010a40', '\U00010a47'), ('\U00010a7d', '\U00010a7e'),
        ('\U00010b58', '\U00010b5f'), ('\U00010b78', '\U00010b7f'),
        ('\U00010e60', '\U00010e7e'), ('\U00011052', '\U00011065'),
        ('\U0001d360', '\U0001d371'), ('\U0001f100', '\U0001f10a')
    ];

    pub fn No(c: char) -> bool {
        super::bsearch_range_table(c, No_table)
    }

}
pub mod decompose {
    use option::Option;
    use option::{Some, None};
    use slice::ImmutableVector;

    fn bsearch_table(c: char, r: &'static [(char, &'static [char])]) -> Option<&'static [char]> {
        use cmp::{Equal, Less, Greater};
        match r.bsearch(|&(val, _)| {
            if c == val { Equal }
            else if val < c { Less }
            else { Greater }
        }) {
            Some(idx) => {
                let (_, result) = r[idx];
                Some(result)
            }
            None => None
        }
    }


    fn bsearch_range_value_table(c: char, r: &'static [(char, char, u8)]) -> u8 {
        use cmp::{Equal, Less, Greater};
        match r.bsearch(|&(lo, hi, _)| {
            if lo <= c && c <= hi { Equal }
            else if hi < c { Less }
            else { Greater }
        }) {
            Some(idx) => {
                let (_, _, result) = r[idx];
                result
            }
            None => 0
        }
    }


    // Canonical decompositions
    static canonical_table : &'static [(char, &'static [char])] = &[
        ('\xc0', &['\x41', '\u0300']), ('\xc1', &['\x41', '\u0301']), ('\xc2', &['\x41', '\u0302']),
        ('\xc3', &['\x41', '\u0303']), ('\xc4', &['\x41', '\u0308']), ('\xc5', &['\x41', '\u030a']),
        ('\xc7', &['\x43', '\u0327']), ('\xc8', &['\x45', '\u0300']), ('\xc9', &['\x45', '\u0301']),
        ('\xca', &['\x45', '\u0302']), ('\xcb', &['\x45', '\u0308']), ('\xcc', &['\x49', '\u0300']),
        ('\xcd', &['\x49', '\u0301']), ('\xce', &['\x49', '\u0302']), ('\xcf', &['\x49', '\u0308']),
        ('\xd1', &['\x4e', '\u0303']), ('\xd2', &['\x4f', '\u0300']), ('\xd3', &['\x4f', '\u0301']),
        ('\xd4', &['\x4f', '\u0302']), ('\xd5', &['\x4f', '\u0303']), ('\xd6', &['\x4f', '\u0308']),
        ('\xd9', &['\x55', '\u0300']), ('\xda', &['\x55', '\u0301']), ('\xdb', &['\x55', '\u0302']),
        ('\xdc', &['\x55', '\u0308']), ('\xdd', &['\x59', '\u0301']), ('\xe0', &['\x61', '\u0300']),
        ('\xe1', &['\x61', '\u0301']), ('\xe2', &['\x61', '\u0302']), ('\xe3', &['\x61', '\u0303']),
        ('\xe4', &['\x61', '\u0308']), ('\xe5', &['\x61', '\u030a']), ('\xe7', &['\x63', '\u0327']),
        ('\xe8', &['\x65', '\u0300']), ('\xe9', &['\x65', '\u0301']), ('\xea', &['\x65', '\u0302']),
        ('\xeb', &['\x65', '\u0308']), ('\xec', &['\x69', '\u0300']), ('\xed', &['\x69', '\u0301']),
        ('\xee', &['\x69', '\u0302']), ('\xef', &['\x69', '\u0308']), ('\xf1', &['\x6e', '\u0303']),
        ('\xf2', &['\x6f', '\u0300']), ('\xf3', &['\x6f', '\u0301']), ('\xf4', &['\x6f', '\u0302']),
        ('\xf5', &['\x6f', '\u0303']), ('\xf6', &['\x6f', '\u0308']), ('\xf9', &['\x75', '\u0300']),
        ('\xfa', &['\x75', '\u0301']), ('\xfb', &['\x75', '\u0302']), ('\xfc', &['\x75', '\u0308']),
        ('\xfd', &['\x79', '\u0301']), ('\xff', &['\x79', '\u0308']), ('\u0100', &['\x41',
        '\u0304']), ('\u0101', &['\x61', '\u0304']), ('\u0102', &['\x41', '\u0306']), ('\u0103',
        &['\x61', '\u0306']), ('\u0104', &['\x41', '\u0328']), ('\u0105', &['\x61', '\u0328']),
        ('\u0106', &['\x43', '\u0301']), ('\u0107', &['\x63', '\u0301']), ('\u0108', &['\x43',
        '\u0302']), ('\u0109', &['\x63', '\u0302']), ('\u010a', &['\x43', '\u0307']), ('\u010b',
        &['\x63', '\u0307']), ('\u010c', &['\x43', '\u030c']), ('\u010d', &['\x63', '\u030c']),
        ('\u010e', &['\x44', '\u030c']), ('\u010f', &['\x64', '\u030c']), ('\u0112', &['\x45',
        '\u0304']), ('\u0113', &['\x65', '\u0304']), ('\u0114', &['\x45', '\u0306']), ('\u0115',
        &['\x65', '\u0306']), ('\u0116', &['\x45', '\u0307']), ('\u0117', &['\x65', '\u0307']),
        ('\u0118', &['\x45', '\u0328']), ('\u0119', &['\x65', '\u0328']), ('\u011a', &['\x45',
        '\u030c']), ('\u011b', &['\x65', '\u030c']), ('\u011c', &['\x47', '\u0302']), ('\u011d',
        &['\x67', '\u0302']), ('\u011e', &['\x47', '\u0306']), ('\u011f', &['\x67', '\u0306']),
        ('\u0120', &['\x47', '\u0307']), ('\u0121', &['\x67', '\u0307']), ('\u0122', &['\x47',
        '\u0327']), ('\u0123', &['\x67', '\u0327']), ('\u0124', &['\x48', '\u0302']), ('\u0125',
        &['\x68', '\u0302']), ('\u0128', &['\x49', '\u0303']), ('\u0129', &['\x69', '\u0303']),
        ('\u012a', &['\x49', '\u0304']), ('\u012b', &['\x69', '\u0304']), ('\u012c', &['\x49',
        '\u0306']), ('\u012d', &['\x69', '\u0306']), ('\u012e', &['\x49', '\u0328']), ('\u012f',
        &['\x69', '\u0328']), ('\u0130', &['\x49', '\u0307']), ('\u0134', &['\x4a', '\u0302']),
        ('\u0135', &['\x6a', '\u0302']), ('\u0136', &['\x4b', '\u0327']), ('\u0137', &['\x6b',
        '\u0327']), ('\u0139', &['\x4c', '\u0301']), ('\u013a', &['\x6c', '\u0301']), ('\u013b',
        &['\x4c', '\u0327']), ('\u013c', &['\x6c', '\u0327']), ('\u013d', &['\x4c', '\u030c']),
        ('\u013e', &['\x6c', '\u030c']), ('\u0143', &['\x4e', '\u0301']), ('\u0144', &['\x6e',
        '\u0301']), ('\u0145', &['\x4e', '\u0327']), ('\u0146', &['\x6e', '\u0327']), ('\u0147',
        &['\x4e', '\u030c']), ('\u0148', &['\x6e', '\u030c']), ('\u014c', &['\x4f', '\u0304']),
        ('\u014d', &['\x6f', '\u0304']), ('\u014e', &['\x4f', '\u0306']), ('\u014f', &['\x6f',
        '\u0306']), ('\u0150', &['\x4f', '\u030b']), ('\u0151', &['\x6f', '\u030b']), ('\u0154',
        &['\x52', '\u0301']), ('\u0155', &['\x72', '\u0301']), ('\u0156', &['\x52', '\u0327']),
        ('\u0157', &['\x72', '\u0327']), ('\u0158', &['\x52', '\u030c']), ('\u0159', &['\x72',
        '\u030c']), ('\u015a', &['\x53', '\u0301']), ('\u015b', &['\x73', '\u0301']), ('\u015c',
        &['\x53', '\u0302']), ('\u015d', &['\x73', '\u0302']), ('\u015e', &['\x53', '\u0327']),
        ('\u015f', &['\x73', '\u0327']), ('\u0160', &['\x53', '\u030c']), ('\u0161', &['\x73',
        '\u030c']), ('\u0162', &['\x54', '\u0327']), ('\u0163', &['\x74', '\u0327']), ('\u0164',
        &['\x54', '\u030c']), ('\u0165', &['\x74', '\u030c']), ('\u0168', &['\x55', '\u0303']),
        ('\u0169', &['\x75', '\u0303']), ('\u016a', &['\x55', '\u0304']), ('\u016b', &['\x75',
        '\u0304']), ('\u016c', &['\x55', '\u0306']), ('\u016d', &['\x75', '\u0306']), ('\u016e',
        &['\x55', '\u030a']), ('\u016f', &['\x75', '\u030a']), ('\u0170', &['\x55', '\u030b']),
        ('\u0171', &['\x75', '\u030b']), ('\u0172', &['\x55', '\u0328']), ('\u0173', &['\x75',
        '\u0328']), ('\u0174', &['\x57', '\u0302']), ('\u0175', &['\x77', '\u0302']), ('\u0176',
        &['\x59', '\u0302']), ('\u0177', &['\x79', '\u0302']), ('\u0178', &['\x59', '\u0308']),
        ('\u0179', &['\x5a', '\u0301']), ('\u017a', &['\x7a', '\u0301']), ('\u017b', &['\x5a',
        '\u0307']), ('\u017c', &['\x7a', '\u0307']), ('\u017d', &['\x5a', '\u030c']), ('\u017e',
        &['\x7a', '\u030c']), ('\u01a0', &['\x4f', '\u031b']), ('\u01a1', &['\x6f', '\u031b']),
        ('\u01af', &['\x55', '\u031b']), ('\u01b0', &['\x75', '\u031b']), ('\u01cd', &['\x41',
        '\u030c']), ('\u01ce', &['\x61', '\u030c']), ('\u01cf', &['\x49', '\u030c']), ('\u01d0',
        &['\x69', '\u030c']), ('\u01d1', &['\x4f', '\u030c']), ('\u01d2', &['\x6f', '\u030c']),
        ('\u01d3', &['\x55', '\u030c']), ('\u01d4', &['\x75', '\u030c']), ('\u01d5', &['\xdc',
        '\u0304']), ('\u01d6', &['\xfc', '\u0304']), ('\u01d7', &['\xdc', '\u0301']), ('\u01d8',
        &['\xfc', '\u0301']), ('\u01d9', &['\xdc', '\u030c']), ('\u01da', &['\xfc', '\u030c']),
        ('\u01db', &['\xdc', '\u0300']), ('\u01dc', &['\xfc', '\u0300']), ('\u01de', &['\xc4',
        '\u0304']), ('\u01df', &['\xe4', '\u0304']), ('\u01e0', &['\u0226', '\u0304']), ('\u01e1',
        &['\u0227', '\u0304']), ('\u01e2', &['\xc6', '\u0304']), ('\u01e3', &['\xe6', '\u0304']),
        ('\u01e6', &['\x47', '\u030c']), ('\u01e7', &['\x67', '\u030c']), ('\u01e8', &['\x4b',
        '\u030c']), ('\u01e9', &['\x6b', '\u030c']), ('\u01ea', &['\x4f', '\u0328']), ('\u01eb',
        &['\x6f', '\u0328']), ('\u01ec', &['\u01ea', '\u0304']), ('\u01ed', &['\u01eb', '\u0304']),
        ('\u01ee', &['\u01b7', '\u030c']), ('\u01ef', &['\u0292', '\u030c']), ('\u01f0', &['\x6a',
        '\u030c']), ('\u01f4', &['\x47', '\u0301']), ('\u01f5', &['\x67', '\u0301']), ('\u01f8',
        &['\x4e', '\u0300']), ('\u01f9', &['\x6e', '\u0300']), ('\u01fa', &['\xc5', '\u0301']),
        ('\u01fb', &['\xe5', '\u0301']), ('\u01fc', &['\xc6', '\u0301']), ('\u01fd', &['\xe6',
        '\u0301']), ('\u01fe', &['\xd8', '\u0301']), ('\u01ff', &['\xf8', '\u0301']), ('\u0200',
        &['\x41', '\u030f']), ('\u0201', &['\x61', '\u030f']), ('\u0202', &['\x41', '\u0311']),
        ('\u0203', &['\x61', '\u0311']), ('\u0204', &['\x45', '\u030f']), ('\u0205', &['\x65',
        '\u030f']), ('\u0206', &['\x45', '\u0311']), ('\u0207', &['\x65', '\u0311']), ('\u0208',
        &['\x49', '\u030f']), ('\u0209', &['\x69', '\u030f']), ('\u020a', &['\x49', '\u0311']),
        ('\u020b', &['\x69', '\u0311']), ('\u020c', &['\x4f', '\u030f']), ('\u020d', &['\x6f',
        '\u030f']), ('\u020e', &['\x4f', '\u0311']), ('\u020f', &['\x6f', '\u0311']), ('\u0210',
        &['\x52', '\u030f']), ('\u0211', &['\x72', '\u030f']), ('\u0212', &['\x52', '\u0311']),
        ('\u0213', &['\x72', '\u0311']), ('\u0214', &['\x55', '\u030f']), ('\u0215', &['\x75',
        '\u030f']), ('\u0216', &['\x55', '\u0311']), ('\u0217', &['\x75', '\u0311']), ('\u0218',
        &['\x53', '\u0326']), ('\u0219', &['\x73', '\u0326']), ('\u021a', &['\x54', '\u0326']),
        ('\u021b', &['\x74', '\u0326']), ('\u021e', &['\x48', '\u030c']), ('\u021f', &['\x68',
        '\u030c']), ('\u0226', &['\x41', '\u0307']), ('\u0227', &['\x61', '\u0307']), ('\u0228',
        &['\x45', '\u0327']), ('\u0229', &['\x65', '\u0327']), ('\u022a', &['\xd6', '\u0304']),
        ('\u022b', &['\xf6', '\u0304']), ('\u022c', &['\xd5', '\u0304']), ('\u022d', &['\xf5',
        '\u0304']), ('\u022e', &['\x4f', '\u0307']), ('\u022f', &['\x6f', '\u0307']), ('\u0230',
        &['\u022e', '\u0304']), ('\u0231', &['\u022f', '\u0304']), ('\u0232', &['\x59', '\u0304']),
        ('\u0233', &['\x79', '\u0304']), ('\u0340', &['\u0300']), ('\u0341', &['\u0301']),
        ('\u0343', &['\u0313']), ('\u0344', &['\u0308', '\u0301']), ('\u0374', &['\u02b9']),
        ('\u037e', &['\x3b']), ('\u0385', &['\xa8', '\u0301']), ('\u0386', &['\u0391', '\u0301']),
        ('\u0387', &['\xb7']), ('\u0388', &['\u0395', '\u0301']), ('\u0389', &['\u0397', '\u0301']),
        ('\u038a', &['\u0399', '\u0301']), ('\u038c', &['\u039f', '\u0301']), ('\u038e', &['\u03a5',
        '\u0301']), ('\u038f', &['\u03a9', '\u0301']), ('\u0390', &['\u03ca', '\u0301']), ('\u03aa',
        &['\u0399', '\u0308']), ('\u03ab', &['\u03a5', '\u0308']), ('\u03ac', &['\u03b1',
        '\u0301']), ('\u03ad', &['\u03b5', '\u0301']), ('\u03ae', &['\u03b7', '\u0301']), ('\u03af',
        &['\u03b9', '\u0301']), ('\u03b0', &['\u03cb', '\u0301']), ('\u03ca', &['\u03b9',
        '\u0308']), ('\u03cb', &['\u03c5', '\u0308']), ('\u03cc', &['\u03bf', '\u0301']), ('\u03cd',
        &['\u03c5', '\u0301']), ('\u03ce', &['\u03c9', '\u0301']), ('\u03d3', &['\u03d2',
        '\u0301']), ('\u03d4', &['\u03d2', '\u0308']), ('\u0400', &['\u0415', '\u0300']), ('\u0401',
        &['\u0415', '\u0308']), ('\u0403', &['\u0413', '\u0301']), ('\u0407', &['\u0406',
        '\u0308']), ('\u040c', &['\u041a', '\u0301']), ('\u040d', &['\u0418', '\u0300']), ('\u040e',
        &['\u0423', '\u0306']), ('\u0419', &['\u0418', '\u0306']), ('\u0439', &['\u0438',
        '\u0306']), ('\u0450', &['\u0435', '\u0300']), ('\u0451', &['\u0435', '\u0308']), ('\u0453',
        &['\u0433', '\u0301']), ('\u0457', &['\u0456', '\u0308']), ('\u045c', &['\u043a',
        '\u0301']), ('\u045d', &['\u0438', '\u0300']), ('\u045e', &['\u0443', '\u0306']), ('\u0476',
        &['\u0474', '\u030f']), ('\u0477', &['\u0475', '\u030f']), ('\u04c1', &['\u0416',
        '\u0306']), ('\u04c2', &['\u0436', '\u0306']), ('\u04d0', &['\u0410', '\u0306']), ('\u04d1',
        &['\u0430', '\u0306']), ('\u04d2', &['\u0410', '\u0308']), ('\u04d3', &['\u0430',
        '\u0308']), ('\u04d6', &['\u0415', '\u0306']), ('\u04d7', &['\u0435', '\u0306']), ('\u04da',
        &['\u04d8', '\u0308']), ('\u04db', &['\u04d9', '\u0308']), ('\u04dc', &['\u0416',
        '\u0308']), ('\u04dd', &['\u0436', '\u0308']), ('\u04de', &['\u0417', '\u0308']), ('\u04df',
        &['\u0437', '\u0308']), ('\u04e2', &['\u0418', '\u0304']), ('\u04e3', &['\u0438',
        '\u0304']), ('\u04e4', &['\u0418', '\u0308']), ('\u04e5', &['\u0438', '\u0308']), ('\u04e6',
        &['\u041e', '\u0308']), ('\u04e7', &['\u043e', '\u0308']), ('\u04ea', &['\u04e8',
        '\u0308']), ('\u04eb', &['\u04e9', '\u0308']), ('\u04ec', &['\u042d', '\u0308']), ('\u04ed',
        &['\u044d', '\u0308']), ('\u04ee', &['\u0423', '\u0304']), ('\u04ef', &['\u0443',
        '\u0304']), ('\u04f0', &['\u0423', '\u0308']), ('\u04f1', &['\u0443', '\u0308']), ('\u04f2',
        &['\u0423', '\u030b']), ('\u04f3', &['\u0443', '\u030b']), ('\u04f4', &['\u0427',
        '\u0308']), ('\u04f5', &['\u0447', '\u0308']), ('\u04f8', &['\u042b', '\u0308']), ('\u04f9',
        &['\u044b', '\u0308']), ('\u0622', &['\u0627', '\u0653']), ('\u0623', &['\u0627',
        '\u0654']), ('\u0624', &['\u0648', '\u0654']), ('\u0625', &['\u0627', '\u0655']), ('\u0626',
        &['\u064a', '\u0654']), ('\u06c0', &['\u06d5', '\u0654']), ('\u06c2', &['\u06c1',
        '\u0654']), ('\u06d3', &['\u06d2', '\u0654']), ('\u0929', &['\u0928', '\u093c']), ('\u0931',
        &['\u0930', '\u093c']), ('\u0934', &['\u0933', '\u093c']), ('\u0958', &['\u0915',
        '\u093c']), ('\u0959', &['\u0916', '\u093c']), ('\u095a', &['\u0917', '\u093c']), ('\u095b',
        &['\u091c', '\u093c']), ('\u095c', &['\u0921', '\u093c']), ('\u095d', &['\u0922',
        '\u093c']), ('\u095e', &['\u092b', '\u093c']), ('\u095f', &['\u092f', '\u093c']), ('\u09cb',
        &['\u09c7', '\u09be']), ('\u09cc', &['\u09c7', '\u09d7']), ('\u09dc', &['\u09a1',
        '\u09bc']), ('\u09dd', &['\u09a2', '\u09bc']), ('\u09df', &['\u09af', '\u09bc']), ('\u0a33',
        &['\u0a32', '\u0a3c']), ('\u0a36', &['\u0a38', '\u0a3c']), ('\u0a59', &['\u0a16',
        '\u0a3c']), ('\u0a5a', &['\u0a17', '\u0a3c']), ('\u0a5b', &['\u0a1c', '\u0a3c']), ('\u0a5e',
        &['\u0a2b', '\u0a3c']), ('\u0b48', &['\u0b47', '\u0b56']), ('\u0b4b', &['\u0b47',
        '\u0b3e']), ('\u0b4c', &['\u0b47', '\u0b57']), ('\u0b5c', &['\u0b21', '\u0b3c']), ('\u0b5d',
        &['\u0b22', '\u0b3c']), ('\u0b94', &['\u0b92', '\u0bd7']), ('\u0bca', &['\u0bc6',
        '\u0bbe']), ('\u0bcb', &['\u0bc7', '\u0bbe']), ('\u0bcc', &['\u0bc6', '\u0bd7']), ('\u0c48',
        &['\u0c46', '\u0c56']), ('\u0cc0', &['\u0cbf', '\u0cd5']), ('\u0cc7', &['\u0cc6',
        '\u0cd5']), ('\u0cc8', &['\u0cc6', '\u0cd6']), ('\u0cca', &['\u0cc6', '\u0cc2']), ('\u0ccb',
        &['\u0cca', '\u0cd5']), ('\u0d4a', &['\u0d46', '\u0d3e']), ('\u0d4b', &['\u0d47',
        '\u0d3e']), ('\u0d4c', &['\u0d46', '\u0d57']), ('\u0dda', &['\u0dd9', '\u0dca']), ('\u0ddc',
        &['\u0dd9', '\u0dcf']), ('\u0ddd', &['\u0ddc', '\u0dca']), ('\u0dde', &['\u0dd9',
        '\u0ddf']), ('\u0f43', &['\u0f42', '\u0fb7']), ('\u0f4d', &['\u0f4c', '\u0fb7']), ('\u0f52',
        &['\u0f51', '\u0fb7']), ('\u0f57', &['\u0f56', '\u0fb7']), ('\u0f5c', &['\u0f5b',
        '\u0fb7']), ('\u0f69', &['\u0f40', '\u0fb5']), ('\u0f73', &['\u0f71', '\u0f72']), ('\u0f75',
        &['\u0f71', '\u0f74']), ('\u0f76', &['\u0fb2', '\u0f80']), ('\u0f78', &['\u0fb3',
        '\u0f80']), ('\u0f81', &['\u0f71', '\u0f80']), ('\u0f93', &['\u0f92', '\u0fb7']), ('\u0f9d',
        &['\u0f9c', '\u0fb7']), ('\u0fa2', &['\u0fa1', '\u0fb7']), ('\u0fa7', &['\u0fa6',
        '\u0fb7']), ('\u0fac', &['\u0fab', '\u0fb7']), ('\u0fb9', &['\u0f90', '\u0fb5']), ('\u1026',
        &['\u1025', '\u102e']), ('\u1b06', &['\u1b05', '\u1b35']), ('\u1b08', &['\u1b07',
        '\u1b35']), ('\u1b0a', &['\u1b09', '\u1b35']), ('\u1b0c', &['\u1b0b', '\u1b35']), ('\u1b0e',
        &['\u1b0d', '\u1b35']), ('\u1b12', &['\u1b11', '\u1b35']), ('\u1b3b', &['\u1b3a',
        '\u1b35']), ('\u1b3d', &['\u1b3c', '\u1b35']), ('\u1b40', &['\u1b3e', '\u1b35']), ('\u1b41',
        &['\u1b3f', '\u1b35']), ('\u1b43', &['\u1b42', '\u1b35']), ('\u1e00', &['\x41', '\u0325']),
        ('\u1e01', &['\x61', '\u0325']), ('\u1e02', &['\x42', '\u0307']), ('\u1e03', &['\x62',
        '\u0307']), ('\u1e04', &['\x42', '\u0323']), ('\u1e05', &['\x62', '\u0323']), ('\u1e06',
        &['\x42', '\u0331']), ('\u1e07', &['\x62', '\u0331']), ('\u1e08', &['\xc7', '\u0301']),
        ('\u1e09', &['\xe7', '\u0301']), ('\u1e0a', &['\x44', '\u0307']), ('\u1e0b', &['\x64',
        '\u0307']), ('\u1e0c', &['\x44', '\u0323']), ('\u1e0d', &['\x64', '\u0323']), ('\u1e0e',
        &['\x44', '\u0331']), ('\u1e0f', &['\x64', '\u0331']), ('\u1e10', &['\x44', '\u0327']),
        ('\u1e11', &['\x64', '\u0327']), ('\u1e12', &['\x44', '\u032d']), ('\u1e13', &['\x64',
        '\u032d']), ('\u1e14', &['\u0112', '\u0300']), ('\u1e15', &['\u0113', '\u0300']), ('\u1e16',
        &['\u0112', '\u0301']), ('\u1e17', &['\u0113', '\u0301']), ('\u1e18', &['\x45', '\u032d']),
        ('\u1e19', &['\x65', '\u032d']), ('\u1e1a', &['\x45', '\u0330']), ('\u1e1b', &['\x65',
        '\u0330']), ('\u1e1c', &['\u0228', '\u0306']), ('\u1e1d', &['\u0229', '\u0306']), ('\u1e1e',
        &['\x46', '\u0307']), ('\u1e1f', &['\x66', '\u0307']), ('\u1e20', &['\x47', '\u0304']),
        ('\u1e21', &['\x67', '\u0304']), ('\u1e22', &['\x48', '\u0307']), ('\u1e23', &['\x68',
        '\u0307']), ('\u1e24', &['\x48', '\u0323']), ('\u1e25', &['\x68', '\u0323']), ('\u1e26',
        &['\x48', '\u0308']), ('\u1e27', &['\x68', '\u0308']), ('\u1e28', &['\x48', '\u0327']),
        ('\u1e29', &['\x68', '\u0327']), ('\u1e2a', &['\x48', '\u032e']), ('\u1e2b', &['\x68',
        '\u032e']), ('\u1e2c', &['\x49', '\u0330']), ('\u1e2d', &['\x69', '\u0330']), ('\u1e2e',
        &['\xcf', '\u0301']), ('\u1e2f', &['\xef', '\u0301']), ('\u1e30', &['\x4b', '\u0301']),
        ('\u1e31', &['\x6b', '\u0301']), ('\u1e32', &['\x4b', '\u0323']), ('\u1e33', &['\x6b',
        '\u0323']), ('\u1e34', &['\x4b', '\u0331']), ('\u1e35', &['\x6b', '\u0331']), ('\u1e36',
        &['\x4c', '\u0323']), ('\u1e37', &['\x6c', '\u0323']), ('\u1e38', &['\u1e36', '\u0304']),
        ('\u1e39', &['\u1e37', '\u0304']), ('\u1e3a', &['\x4c', '\u0331']), ('\u1e3b', &['\x6c',
        '\u0331']), ('\u1e3c', &['\x4c', '\u032d']), ('\u1e3d', &['\x6c', '\u032d']), ('\u1e3e',
        &['\x4d', '\u0301']), ('\u1e3f', &['\x6d', '\u0301']), ('\u1e40', &['\x4d', '\u0307']),
        ('\u1e41', &['\x6d', '\u0307']), ('\u1e42', &['\x4d', '\u0323']), ('\u1e43', &['\x6d',
        '\u0323']), ('\u1e44', &['\x4e', '\u0307']), ('\u1e45', &['\x6e', '\u0307']), ('\u1e46',
        &['\x4e', '\u0323']), ('\u1e47', &['\x6e', '\u0323']), ('\u1e48', &['\x4e', '\u0331']),
        ('\u1e49', &['\x6e', '\u0331']), ('\u1e4a', &['\x4e', '\u032d']), ('\u1e4b', &['\x6e',
        '\u032d']), ('\u1e4c', &['\xd5', '\u0301']), ('\u1e4d', &['\xf5', '\u0301']), ('\u1e4e',
        &['\xd5', '\u0308']), ('\u1e4f', &['\xf5', '\u0308']), ('\u1e50', &['\u014c', '\u0300']),
        ('\u1e51', &['\u014d', '\u0300']), ('\u1e52', &['\u014c', '\u0301']), ('\u1e53', &['\u014d',
        '\u0301']), ('\u1e54', &['\x50', '\u0301']), ('\u1e55', &['\x70', '\u0301']), ('\u1e56',
        &['\x50', '\u0307']), ('\u1e57', &['\x70', '\u0307']), ('\u1e58', &['\x52', '\u0307']),
        ('\u1e59', &['\x72', '\u0307']), ('\u1e5a', &['\x52', '\u0323']), ('\u1e5b', &['\x72',
        '\u0323']), ('\u1e5c', &['\u1e5a', '\u0304']), ('\u1e5d', &['\u1e5b', '\u0304']), ('\u1e5e',
        &['\x52', '\u0331']), ('\u1e5f', &['\x72', '\u0331']), ('\u1e60', &['\x53', '\u0307']),
        ('\u1e61', &['\x73', '\u0307']), ('\u1e62', &['\x53', '\u0323']), ('\u1e63', &['\x73',
        '\u0323']), ('\u1e64', &['\u015a', '\u0307']), ('\u1e65', &['\u015b', '\u0307']), ('\u1e66',
        &['\u0160', '\u0307']), ('\u1e67', &['\u0161', '\u0307']), ('\u1e68', &['\u1e62',
        '\u0307']), ('\u1e69', &['\u1e63', '\u0307']), ('\u1e6a', &['\x54', '\u0307']), ('\u1e6b',
        &['\x74', '\u0307']), ('\u1e6c', &['\x54', '\u0323']), ('\u1e6d', &['\x74', '\u0323']),
        ('\u1e6e', &['\x54', '\u0331']), ('\u1e6f', &['\x74', '\u0331']), ('\u1e70', &['\x54',
        '\u032d']), ('\u1e71', &['\x74', '\u032d']), ('\u1e72', &['\x55', '\u0324']), ('\u1e73',
        &['\x75', '\u0324']), ('\u1e74', &['\x55', '\u0330']), ('\u1e75', &['\x75', '\u0330']),
        ('\u1e76', &['\x55', '\u032d']), ('\u1e77', &['\x75', '\u032d']), ('\u1e78', &['\u0168',
        '\u0301']), ('\u1e79', &['\u0169', '\u0301']), ('\u1e7a', &['\u016a', '\u0308']), ('\u1e7b',
        &['\u016b', '\u0308']), ('\u1e7c', &['\x56', '\u0303']), ('\u1e7d', &['\x76', '\u0303']),
        ('\u1e7e', &['\x56', '\u0323']), ('\u1e7f', &['\x76', '\u0323']), ('\u1e80', &['\x57',
        '\u0300']), ('\u1e81', &['\x77', '\u0300']), ('\u1e82', &['\x57', '\u0301']), ('\u1e83',
        &['\x77', '\u0301']), ('\u1e84', &['\x57', '\u0308']), ('\u1e85', &['\x77', '\u0308']),
        ('\u1e86', &['\x57', '\u0307']), ('\u1e87', &['\x77', '\u0307']), ('\u1e88', &['\x57',
        '\u0323']), ('\u1e89', &['\x77', '\u0323']), ('\u1e8a', &['\x58', '\u0307']), ('\u1e8b',
        &['\x78', '\u0307']), ('\u1e8c', &['\x58', '\u0308']), ('\u1e8d', &['\x78', '\u0308']),
        ('\u1e8e', &['\x59', '\u0307']), ('\u1e8f', &['\x79', '\u0307']), ('\u1e90', &['\x5a',
        '\u0302']), ('\u1e91', &['\x7a', '\u0302']), ('\u1e92', &['\x5a', '\u0323']), ('\u1e93',
        &['\x7a', '\u0323']), ('\u1e94', &['\x5a', '\u0331']), ('\u1e95', &['\x7a', '\u0331']),
        ('\u1e96', &['\x68', '\u0331']), ('\u1e97', &['\x74', '\u0308']), ('\u1e98', &['\x77',
        '\u030a']), ('\u1e99', &['\x79', '\u030a']), ('\u1e9b', &['\u017f', '\u0307']), ('\u1ea0',
        &['\x41', '\u0323']), ('\u1ea1', &['\x61', '\u0323']), ('\u1ea2', &['\x41', '\u0309']),
        ('\u1ea3', &['\x61', '\u0309']), ('\u1ea4', &['\xc2', '\u0301']), ('\u1ea5', &['\xe2',
        '\u0301']), ('\u1ea6', &['\xc2', '\u0300']), ('\u1ea7', &['\xe2', '\u0300']), ('\u1ea8',
        &['\xc2', '\u0309']), ('\u1ea9', &['\xe2', '\u0309']), ('\u1eaa', &['\xc2', '\u0303']),
        ('\u1eab', &['\xe2', '\u0303']), ('\u1eac', &['\u1ea0', '\u0302']), ('\u1ead', &['\u1ea1',
        '\u0302']), ('\u1eae', &['\u0102', '\u0301']), ('\u1eaf', &['\u0103', '\u0301']), ('\u1eb0',
        &['\u0102', '\u0300']), ('\u1eb1', &['\u0103', '\u0300']), ('\u1eb2', &['\u0102',
        '\u0309']), ('\u1eb3', &['\u0103', '\u0309']), ('\u1eb4', &['\u0102', '\u0303']), ('\u1eb5',
        &['\u0103', '\u0303']), ('\u1eb6', &['\u1ea0', '\u0306']), ('\u1eb7', &['\u1ea1',
        '\u0306']), ('\u1eb8', &['\x45', '\u0323']), ('\u1eb9', &['\x65', '\u0323']), ('\u1eba',
        &['\x45', '\u0309']), ('\u1ebb', &['\x65', '\u0309']), ('\u1ebc', &['\x45', '\u0303']),
        ('\u1ebd', &['\x65', '\u0303']), ('\u1ebe', &['\xca', '\u0301']), ('\u1ebf', &['\xea',
        '\u0301']), ('\u1ec0', &['\xca', '\u0300']), ('\u1ec1', &['\xea', '\u0300']), ('\u1ec2',
        &['\xca', '\u0309']), ('\u1ec3', &['\xea', '\u0309']), ('\u1ec4', &['\xca', '\u0303']),
        ('\u1ec5', &['\xea', '\u0303']), ('\u1ec6', &['\u1eb8', '\u0302']), ('\u1ec7', &['\u1eb9',
        '\u0302']), ('\u1ec8', &['\x49', '\u0309']), ('\u1ec9', &['\x69', '\u0309']), ('\u1eca',
        &['\x49', '\u0323']), ('\u1ecb', &['\x69', '\u0323']), ('\u1ecc', &['\x4f', '\u0323']),
        ('\u1ecd', &['\x6f', '\u0323']), ('\u1ece', &['\x4f', '\u0309']), ('\u1ecf', &['\x6f',
        '\u0309']), ('\u1ed0', &['\xd4', '\u0301']), ('\u1ed1', &['\xf4', '\u0301']), ('\u1ed2',
        &['\xd4', '\u0300']), ('\u1ed3', &['\xf4', '\u0300']), ('\u1ed4', &['\xd4', '\u0309']),
        ('\u1ed5', &['\xf4', '\u0309']), ('\u1ed6', &['\xd4', '\u0303']), ('\u1ed7', &['\xf4',
        '\u0303']), ('\u1ed8', &['\u1ecc', '\u0302']), ('\u1ed9', &['\u1ecd', '\u0302']), ('\u1eda',
        &['\u01a0', '\u0301']), ('\u1edb', &['\u01a1', '\u0301']), ('\u1edc', &['\u01a0',
        '\u0300']), ('\u1edd', &['\u01a1', '\u0300']), ('\u1ede', &['\u01a0', '\u0309']), ('\u1edf',
        &['\u01a1', '\u0309']), ('\u1ee0', &['\u01a0', '\u0303']), ('\u1ee1', &['\u01a1',
        '\u0303']), ('\u1ee2', &['\u01a0', '\u0323']), ('\u1ee3', &['\u01a1', '\u0323']), ('\u1ee4',
        &['\x55', '\u0323']), ('\u1ee5', &['\x75', '\u0323']), ('\u1ee6', &['\x55', '\u0309']),
        ('\u1ee7', &['\x75', '\u0309']), ('\u1ee8', &['\u01af', '\u0301']), ('\u1ee9', &['\u01b0',
        '\u0301']), ('\u1eea', &['\u01af', '\u0300']), ('\u1eeb', &['\u01b0', '\u0300']), ('\u1eec',
        &['\u01af', '\u0309']), ('\u1eed', &['\u01b0', '\u0309']), ('\u1eee', &['\u01af',
        '\u0303']), ('\u1eef', &['\u01b0', '\u0303']), ('\u1ef0', &['\u01af', '\u0323']), ('\u1ef1',
        &['\u01b0', '\u0323']), ('\u1ef2', &['\x59', '\u0300']), ('\u1ef3', &['\x79', '\u0300']),
        ('\u1ef4', &['\x59', '\u0323']), ('\u1ef5', &['\x79', '\u0323']), ('\u1ef6', &['\x59',
        '\u0309']), ('\u1ef7', &['\x79', '\u0309']), ('\u1ef8', &['\x59', '\u0303']), ('\u1ef9',
        &['\x79', '\u0303']), ('\u1f00', &['\u03b1', '\u0313']), ('\u1f01', &['\u03b1', '\u0314']),
        ('\u1f02', &['\u1f00', '\u0300']), ('\u1f03', &['\u1f01', '\u0300']), ('\u1f04', &['\u1f00',
        '\u0301']), ('\u1f05', &['\u1f01', '\u0301']), ('\u1f06', &['\u1f00', '\u0342']), ('\u1f07',
        &['\u1f01', '\u0342']), ('\u1f08', &['\u0391', '\u0313']), ('\u1f09', &['\u0391',
        '\u0314']), ('\u1f0a', &['\u1f08', '\u0300']), ('\u1f0b', &['\u1f09', '\u0300']), ('\u1f0c',
        &['\u1f08', '\u0301']), ('\u1f0d', &['\u1f09', '\u0301']), ('\u1f0e', &['\u1f08',
        '\u0342']), ('\u1f0f', &['\u1f09', '\u0342']), ('\u1f10', &['\u03b5', '\u0313']), ('\u1f11',
        &['\u03b5', '\u0314']), ('\u1f12', &['\u1f10', '\u0300']), ('\u1f13', &['\u1f11',
        '\u0300']), ('\u1f14', &['\u1f10', '\u0301']), ('\u1f15', &['\u1f11', '\u0301']), ('\u1f18',
        &['\u0395', '\u0313']), ('\u1f19', &['\u0395', '\u0314']), ('\u1f1a', &['\u1f18',
        '\u0300']), ('\u1f1b', &['\u1f19', '\u0300']), ('\u1f1c', &['\u1f18', '\u0301']), ('\u1f1d',
        &['\u1f19', '\u0301']), ('\u1f20', &['\u03b7', '\u0313']), ('\u1f21', &['\u03b7',
        '\u0314']), ('\u1f22', &['\u1f20', '\u0300']), ('\u1f23', &['\u1f21', '\u0300']), ('\u1f24',
        &['\u1f20', '\u0301']), ('\u1f25', &['\u1f21', '\u0301']), ('\u1f26', &['\u1f20',
        '\u0342']), ('\u1f27', &['\u1f21', '\u0342']), ('\u1f28', &['\u0397', '\u0313']), ('\u1f29',
        &['\u0397', '\u0314']), ('\u1f2a', &['\u1f28', '\u0300']), ('\u1f2b', &['\u1f29',
        '\u0300']), ('\u1f2c', &['\u1f28', '\u0301']), ('\u1f2d', &['\u1f29', '\u0301']), ('\u1f2e',
        &['\u1f28', '\u0342']), ('\u1f2f', &['\u1f29', '\u0342']), ('\u1f30', &['\u03b9',
        '\u0313']), ('\u1f31', &['\u03b9', '\u0314']), ('\u1f32', &['\u1f30', '\u0300']), ('\u1f33',
        &['\u1f31', '\u0300']), ('\u1f34', &['\u1f30', '\u0301']), ('\u1f35', &['\u1f31',
        '\u0301']), ('\u1f36', &['\u1f30', '\u0342']), ('\u1f37', &['\u1f31', '\u0342']), ('\u1f38',
        &['\u0399', '\u0313']), ('\u1f39', &['\u0399', '\u0314']), ('\u1f3a', &['\u1f38',
        '\u0300']), ('\u1f3b', &['\u1f39', '\u0300']), ('\u1f3c', &['\u1f38', '\u0301']), ('\u1f3d',
        &['\u1f39', '\u0301']), ('\u1f3e', &['\u1f38', '\u0342']), ('\u1f3f', &['\u1f39',
        '\u0342']), ('\u1f40', &['\u03bf', '\u0313']), ('\u1f41', &['\u03bf', '\u0314']), ('\u1f42',
        &['\u1f40', '\u0300']), ('\u1f43', &['\u1f41', '\u0300']), ('\u1f44', &['\u1f40',
        '\u0301']), ('\u1f45', &['\u1f41', '\u0301']), ('\u1f48', &['\u039f', '\u0313']), ('\u1f49',
        &['\u039f', '\u0314']), ('\u1f4a', &['\u1f48', '\u0300']), ('\u1f4b', &['\u1f49',
        '\u0300']), ('\u1f4c', &['\u1f48', '\u0301']), ('\u1f4d', &['\u1f49', '\u0301']), ('\u1f50',
        &['\u03c5', '\u0313']), ('\u1f51', &['\u03c5', '\u0314']), ('\u1f52', &['\u1f50',
        '\u0300']), ('\u1f53', &['\u1f51', '\u0300']), ('\u1f54', &['\u1f50', '\u0301']), ('\u1f55',
        &['\u1f51', '\u0301']), ('\u1f56', &['\u1f50', '\u0342']), ('\u1f57', &['\u1f51',
        '\u0342']), ('\u1f59', &['\u03a5', '\u0314']), ('\u1f5b', &['\u1f59', '\u0300']), ('\u1f5d',
        &['\u1f59', '\u0301']), ('\u1f5f', &['\u1f59', '\u0342']), ('\u1f60', &['\u03c9',
        '\u0313']), ('\u1f61', &['\u03c9', '\u0314']), ('\u1f62', &['\u1f60', '\u0300']), ('\u1f63',
        &['\u1f61', '\u0300']), ('\u1f64', &['\u1f60', '\u0301']), ('\u1f65', &['\u1f61',
        '\u0301']), ('\u1f66', &['\u1f60', '\u0342']), ('\u1f67', &['\u1f61', '\u0342']), ('\u1f68',
        &['\u03a9', '\u0313']), ('\u1f69', &['\u03a9', '\u0314']), ('\u1f6a', &['\u1f68',
        '\u0300']), ('\u1f6b', &['\u1f69', '\u0300']), ('\u1f6c', &['\u1f68', '\u0301']), ('\u1f6d',
        &['\u1f69', '\u0301']), ('\u1f6e', &['\u1f68', '\u0342']), ('\u1f6f', &['\u1f69',
        '\u0342']), ('\u1f70', &['\u03b1', '\u0300']), ('\u1f71', &['\u03ac']), ('\u1f72',
        &['\u03b5', '\u0300']), ('\u1f73', &['\u03ad']), ('\u1f74', &['\u03b7', '\u0300']),
        ('\u1f75', &['\u03ae']), ('\u1f76', &['\u03b9', '\u0300']), ('\u1f77', &['\u03af']),
        ('\u1f78', &['\u03bf', '\u0300']), ('\u1f79', &['\u03cc']), ('\u1f7a', &['\u03c5',
        '\u0300']), ('\u1f7b', &['\u03cd']), ('\u1f7c', &['\u03c9', '\u0300']), ('\u1f7d',
        &['\u03ce']), ('\u1f80', &['\u1f00', '\u0345']), ('\u1f81', &['\u1f01', '\u0345']),
        ('\u1f82', &['\u1f02', '\u0345']), ('\u1f83', &['\u1f03', '\u0345']), ('\u1f84', &['\u1f04',
        '\u0345']), ('\u1f85', &['\u1f05', '\u0345']), ('\u1f86', &['\u1f06', '\u0345']), ('\u1f87',
        &['\u1f07', '\u0345']), ('\u1f88', &['\u1f08', '\u0345']), ('\u1f89', &['\u1f09',
        '\u0345']), ('\u1f8a', &['\u1f0a', '\u0345']), ('\u1f8b', &['\u1f0b', '\u0345']), ('\u1f8c',
        &['\u1f0c', '\u0345']), ('\u1f8d', &['\u1f0d', '\u0345']), ('\u1f8e', &['\u1f0e',
        '\u0345']), ('\u1f8f', &['\u1f0f', '\u0345']), ('\u1f90', &['\u1f20', '\u0345']), ('\u1f91',
        &['\u1f21', '\u0345']), ('\u1f92', &['\u1f22', '\u0345']), ('\u1f93', &['\u1f23',
        '\u0345']), ('\u1f94', &['\u1f24', '\u0345']), ('\u1f95', &['\u1f25', '\u0345']), ('\u1f96',
        &['\u1f26', '\u0345']), ('\u1f97', &['\u1f27', '\u0345']), ('\u1f98', &['\u1f28',
        '\u0345']), ('\u1f99', &['\u1f29', '\u0345']), ('\u1f9a', &['\u1f2a', '\u0345']), ('\u1f9b',
        &['\u1f2b', '\u0345']), ('\u1f9c', &['\u1f2c', '\u0345']), ('\u1f9d', &['\u1f2d',
        '\u0345']), ('\u1f9e', &['\u1f2e', '\u0345']), ('\u1f9f', &['\u1f2f', '\u0345']), ('\u1fa0',
        &['\u1f60', '\u0345']), ('\u1fa1', &['\u1f61', '\u0345']), ('\u1fa2', &['\u1f62',
        '\u0345']), ('\u1fa3', &['\u1f63', '\u0345']), ('\u1fa4', &['\u1f64', '\u0345']), ('\u1fa5',
        &['\u1f65', '\u0345']), ('\u1fa6', &['\u1f66', '\u0345']), ('\u1fa7', &['\u1f67',
        '\u0345']), ('\u1fa8', &['\u1f68', '\u0345']), ('\u1fa9', &['\u1f69', '\u0345']), ('\u1faa',
        &['\u1f6a', '\u0345']), ('\u1fab', &['\u1f6b', '\u0345']), ('\u1fac', &['\u1f6c',
        '\u0345']), ('\u1fad', &['\u1f6d', '\u0345']), ('\u1fae', &['\u1f6e', '\u0345']), ('\u1faf',
        &['\u1f6f', '\u0345']), ('\u1fb0', &['\u03b1', '\u0306']), ('\u1fb1', &['\u03b1',
        '\u0304']), ('\u1fb2', &['\u1f70', '\u0345']), ('\u1fb3', &['\u03b1', '\u0345']), ('\u1fb4',
        &['\u03ac', '\u0345']), ('\u1fb6', &['\u03b1', '\u0342']), ('\u1fb7', &['\u1fb6',
        '\u0345']), ('\u1fb8', &['\u0391', '\u0306']), ('\u1fb9', &['\u0391', '\u0304']), ('\u1fba',
        &['\u0391', '\u0300']), ('\u1fbb', &['\u0386']), ('\u1fbc', &['\u0391', '\u0345']),
        ('\u1fbe', &['\u03b9']), ('\u1fc1', &['\xa8', '\u0342']), ('\u1fc2', &['\u1f74', '\u0345']),
        ('\u1fc3', &['\u03b7', '\u0345']), ('\u1fc4', &['\u03ae', '\u0345']), ('\u1fc6', &['\u03b7',
        '\u0342']), ('\u1fc7', &['\u1fc6', '\u0345']), ('\u1fc8', &['\u0395', '\u0300']), ('\u1fc9',
        &['\u0388']), ('\u1fca', &['\u0397', '\u0300']), ('\u1fcb', &['\u0389']), ('\u1fcc',
        &['\u0397', '\u0345']), ('\u1fcd', &['\u1fbf', '\u0300']), ('\u1fce', &['\u1fbf',
        '\u0301']), ('\u1fcf', &['\u1fbf', '\u0342']), ('\u1fd0', &['\u03b9', '\u0306']), ('\u1fd1',
        &['\u03b9', '\u0304']), ('\u1fd2', &['\u03ca', '\u0300']), ('\u1fd3', &['\u0390']),
        ('\u1fd6', &['\u03b9', '\u0342']), ('\u1fd7', &['\u03ca', '\u0342']), ('\u1fd8', &['\u0399',
        '\u0306']), ('\u1fd9', &['\u0399', '\u0304']), ('\u1fda', &['\u0399', '\u0300']), ('\u1fdb',
        &['\u038a']), ('\u1fdd', &['\u1ffe', '\u0300']), ('\u1fde', &['\u1ffe', '\u0301']),
        ('\u1fdf', &['\u1ffe', '\u0342']), ('\u1fe0', &['\u03c5', '\u0306']), ('\u1fe1', &['\u03c5',
        '\u0304']), ('\u1fe2', &['\u03cb', '\u0300']), ('\u1fe3', &['\u03b0']), ('\u1fe4',
        &['\u03c1', '\u0313']), ('\u1fe5', &['\u03c1', '\u0314']), ('\u1fe6', &['\u03c5',
        '\u0342']), ('\u1fe7', &['\u03cb', '\u0342']), ('\u1fe8', &['\u03a5', '\u0306']), ('\u1fe9',
        &['\u03a5', '\u0304']), ('\u1fea', &['\u03a5', '\u0300']), ('\u1feb', &['\u038e']),
        ('\u1fec', &['\u03a1', '\u0314']), ('\u1fed', &['\xa8', '\u0300']), ('\u1fee', &['\u0385']),
        ('\u1fef', &['\x60']), ('\u1ff2', &['\u1f7c', '\u0345']), ('\u1ff3', &['\u03c9', '\u0345']),
        ('\u1ff4', &['\u03ce', '\u0345']), ('\u1ff6', &['\u03c9', '\u0342']), ('\u1ff7', &['\u1ff6',
        '\u0345']), ('\u1ff8', &['\u039f', '\u0300']), ('\u1ff9', &['\u038c']), ('\u1ffa',
        &['\u03a9', '\u0300']), ('\u1ffb', &['\u038f']), ('\u1ffc', &['\u03a9', '\u0345']),
        ('\u1ffd', &['\xb4']), ('\u2000', &['\u2002']), ('\u2001', &['\u2003']), ('\u2126',
        &['\u03a9']), ('\u212a', &['\x4b']), ('\u212b', &['\xc5']), ('\u219a', &['\u2190',
        '\u0338']), ('\u219b', &['\u2192', '\u0338']), ('\u21ae', &['\u2194', '\u0338']), ('\u21cd',
        &['\u21d0', '\u0338']), ('\u21ce', &['\u21d4', '\u0338']), ('\u21cf', &['\u21d2',
        '\u0338']), ('\u2204', &['\u2203', '\u0338']), ('\u2209', &['\u2208', '\u0338']), ('\u220c',
        &['\u220b', '\u0338']), ('\u2224', &['\u2223', '\u0338']), ('\u2226', &['\u2225',
        '\u0338']), ('\u2241', &['\u223c', '\u0338']), ('\u2244', &['\u2243', '\u0338']), ('\u2247',
        &['\u2245', '\u0338']), ('\u2249', &['\u2248', '\u0338']), ('\u2260', &['\x3d', '\u0338']),
        ('\u2262', &['\u2261', '\u0338']), ('\u226d', &['\u224d', '\u0338']), ('\u226e', &['\x3c',
        '\u0338']), ('\u226f', &['\x3e', '\u0338']), ('\u2270', &['\u2264', '\u0338']), ('\u2271',
        &['\u2265', '\u0338']), ('\u2274', &['\u2272', '\u0338']), ('\u2275', &['\u2273',
        '\u0338']), ('\u2278', &['\u2276', '\u0338']), ('\u2279', &['\u2277', '\u0338']), ('\u2280',
        &['\u227a', '\u0338']), ('\u2281', &['\u227b', '\u0338']), ('\u2284', &['\u2282',
        '\u0338']), ('\u2285', &['\u2283', '\u0338']), ('\u2288', &['\u2286', '\u0338']), ('\u2289',
        &['\u2287', '\u0338']), ('\u22ac', &['\u22a2', '\u0338']), ('\u22ad', &['\u22a8',
        '\u0338']), ('\u22ae', &['\u22a9', '\u0338']), ('\u22af', &['\u22ab', '\u0338']), ('\u22e0',
        &['\u227c', '\u0338']), ('\u22e1', &['\u227d', '\u0338']), ('\u22e2', &['\u2291',
        '\u0338']), ('\u22e3', &['\u2292', '\u0338']), ('\u22ea', &['\u22b2', '\u0338']), ('\u22eb',
        &['\u22b3', '\u0338']), ('\u22ec', &['\u22b4', '\u0338']), ('\u22ed', &['\u22b5',
        '\u0338']), ('\u2329', &['\u3008']), ('\u232a', &['\u3009']), ('\u2adc', &['\u2add',
        '\u0338']), ('\u304c', &['\u304b', '\u3099']), ('\u304e', &['\u304d', '\u3099']), ('\u3050',
        &['\u304f', '\u3099']), ('\u3052', &['\u3051', '\u3099']), ('\u3054', &['\u3053',
        '\u3099']), ('\u3056', &['\u3055', '\u3099']), ('\u3058', &['\u3057', '\u3099']), ('\u305a',
        &['\u3059', '\u3099']), ('\u305c', &['\u305b', '\u3099']), ('\u305e', &['\u305d',
        '\u3099']), ('\u3060', &['\u305f', '\u3099']), ('\u3062', &['\u3061', '\u3099']), ('\u3065',
        &['\u3064', '\u3099']), ('\u3067', &['\u3066', '\u3099']), ('\u3069', &['\u3068',
        '\u3099']), ('\u3070', &['\u306f', '\u3099']), ('\u3071', &['\u306f', '\u309a']), ('\u3073',
        &['\u3072', '\u3099']), ('\u3074', &['\u3072', '\u309a']), ('\u3076', &['\u3075',
        '\u3099']), ('\u3077', &['\u3075', '\u309a']), ('\u3079', &['\u3078', '\u3099']), ('\u307a',
        &['\u3078', '\u309a']), ('\u307c', &['\u307b', '\u3099']), ('\u307d', &['\u307b',
        '\u309a']), ('\u3094', &['\u3046', '\u3099']), ('\u309e', &['\u309d', '\u3099']), ('\u30ac',
        &['\u30ab', '\u3099']), ('\u30ae', &['\u30ad', '\u3099']), ('\u30b0', &['\u30af',
        '\u3099']), ('\u30b2', &['\u30b1', '\u3099']), ('\u30b4', &['\u30b3', '\u3099']), ('\u30b6',
        &['\u30b5', '\u3099']), ('\u30b8', &['\u30b7', '\u3099']), ('\u30ba', &['\u30b9',
        '\u3099']), ('\u30bc', &['\u30bb', '\u3099']), ('\u30be', &['\u30bd', '\u3099']), ('\u30c0',
        &['\u30bf', '\u3099']), ('\u30c2', &['\u30c1', '\u3099']), ('\u30c5', &['\u30c4',
        '\u3099']), ('\u30c7', &['\u30c6', '\u3099']), ('\u30c9', &['\u30c8', '\u3099']), ('\u30d0',
        &['\u30cf', '\u3099']), ('\u30d1', &['\u30cf', '\u309a']), ('\u30d3', &['\u30d2',
        '\u3099']), ('\u30d4', &['\u30d2', '\u309a']), ('\u30d6', &['\u30d5', '\u3099']), ('\u30d7',
        &['\u30d5', '\u309a']), ('\u30d9', &['\u30d8', '\u3099']), ('\u30da', &['\u30d8',
        '\u309a']), ('\u30dc', &['\u30db', '\u3099']), ('\u30dd', &['\u30db', '\u309a']), ('\u30f4',
        &['\u30a6', '\u3099']), ('\u30f7', &['\u30ef', '\u3099']), ('\u30f8', &['\u30f0',
        '\u3099']), ('\u30f9', &['\u30f1', '\u3099']), ('\u30fa', &['\u30f2', '\u3099']), ('\u30fe',
        &['\u30fd', '\u3099']), ('\uf900', &['\u8c48']), ('\uf901', &['\u66f4']), ('\uf902',
        &['\u8eca']), ('\uf903', &['\u8cc8']), ('\uf904', &['\u6ed1']), ('\uf905', &['\u4e32']),
        ('\uf906', &['\u53e5']), ('\uf907', &['\u9f9c']), ('\uf908', &['\u9f9c']), ('\uf909',
        &['\u5951']), ('\uf90a', &['\u91d1']), ('\uf90b', &['\u5587']), ('\uf90c', &['\u5948']),
        ('\uf90d', &['\u61f6']), ('\uf90e', &['\u7669']), ('\uf90f', &['\u7f85']), ('\uf910',
        &['\u863f']), ('\uf911', &['\u87ba']), ('\uf912', &['\u88f8']), ('\uf913', &['\u908f']),
        ('\uf914', &['\u6a02']), ('\uf915', &['\u6d1b']), ('\uf916', &['\u70d9']), ('\uf917',
        &['\u73de']), ('\uf918', &['\u843d']), ('\uf919', &['\u916a']), ('\uf91a', &['\u99f1']),
        ('\uf91b', &['\u4e82']), ('\uf91c', &['\u5375']), ('\uf91d', &['\u6b04']), ('\uf91e',
        &['\u721b']), ('\uf91f', &['\u862d']), ('\uf920', &['\u9e1e']), ('\uf921', &['\u5d50']),
        ('\uf922', &['\u6feb']), ('\uf923', &['\u85cd']), ('\uf924', &['\u8964']), ('\uf925',
        &['\u62c9']), ('\uf926', &['\u81d8']), ('\uf927', &['\u881f']), ('\uf928', &['\u5eca']),
        ('\uf929', &['\u6717']), ('\uf92a', &['\u6d6a']), ('\uf92b', &['\u72fc']), ('\uf92c',
        &['\u90ce']), ('\uf92d', &['\u4f86']), ('\uf92e', &['\u51b7']), ('\uf92f', &['\u52de']),
        ('\uf930', &['\u64c4']), ('\uf931', &['\u6ad3']), ('\uf932', &['\u7210']), ('\uf933',
        &['\u76e7']), ('\uf934', &['\u8001']), ('\uf935', &['\u8606']), ('\uf936', &['\u865c']),
        ('\uf937', &['\u8def']), ('\uf938', &['\u9732']), ('\uf939', &['\u9b6f']), ('\uf93a',
        &['\u9dfa']), ('\uf93b', &['\u788c']), ('\uf93c', &['\u797f']), ('\uf93d', &['\u7da0']),
        ('\uf93e', &['\u83c9']), ('\uf93f', &['\u9304']), ('\uf940', &['\u9e7f']), ('\uf941',
        &['\u8ad6']), ('\uf942', &['\u58df']), ('\uf943', &['\u5f04']), ('\uf944', &['\u7c60']),
        ('\uf945', &['\u807e']), ('\uf946', &['\u7262']), ('\uf947', &['\u78ca']), ('\uf948',
        &['\u8cc2']), ('\uf949', &['\u96f7']), ('\uf94a', &['\u58d8']), ('\uf94b', &['\u5c62']),
        ('\uf94c', &['\u6a13']), ('\uf94d', &['\u6dda']), ('\uf94e', &['\u6f0f']), ('\uf94f',
        &['\u7d2f']), ('\uf950', &['\u7e37']), ('\uf951', &['\u964b']), ('\uf952', &['\u52d2']),
        ('\uf953', &['\u808b']), ('\uf954', &['\u51dc']), ('\uf955', &['\u51cc']), ('\uf956',
        &['\u7a1c']), ('\uf957', &['\u7dbe']), ('\uf958', &['\u83f1']), ('\uf959', &['\u9675']),
        ('\uf95a', &['\u8b80']), ('\uf95b', &['\u62cf']), ('\uf95c', &['\u6a02']), ('\uf95d',
        &['\u8afe']), ('\uf95e', &['\u4e39']), ('\uf95f', &['\u5be7']), ('\uf960', &['\u6012']),
        ('\uf961', &['\u7387']), ('\uf962', &['\u7570']), ('\uf963', &['\u5317']), ('\uf964',
        &['\u78fb']), ('\uf965', &['\u4fbf']), ('\uf966', &['\u5fa9']), ('\uf967', &['\u4e0d']),
        ('\uf968', &['\u6ccc']), ('\uf969', &['\u6578']), ('\uf96a', &['\u7d22']), ('\uf96b',
        &['\u53c3']), ('\uf96c', &['\u585e']), ('\uf96d', &['\u7701']), ('\uf96e', &['\u8449']),
        ('\uf96f', &['\u8aaa']), ('\uf970', &['\u6bba']), ('\uf971', &['\u8fb0']), ('\uf972',
        &['\u6c88']), ('\uf973', &['\u62fe']), ('\uf974', &['\u82e5']), ('\uf975', &['\u63a0']),
        ('\uf976', &['\u7565']), ('\uf977', &['\u4eae']), ('\uf978', &['\u5169']), ('\uf979',
        &['\u51c9']), ('\uf97a', &['\u6881']), ('\uf97b', &['\u7ce7']), ('\uf97c', &['\u826f']),
        ('\uf97d', &['\u8ad2']), ('\uf97e', &['\u91cf']), ('\uf97f', &['\u52f5']), ('\uf980',
        &['\u5442']), ('\uf981', &['\u5973']), ('\uf982', &['\u5eec']), ('\uf983', &['\u65c5']),
        ('\uf984', &['\u6ffe']), ('\uf985', &['\u792a']), ('\uf986', &['\u95ad']), ('\uf987',
        &['\u9a6a']), ('\uf988', &['\u9e97']), ('\uf989', &['\u9ece']), ('\uf98a', &['\u529b']),
        ('\uf98b', &['\u66c6']), ('\uf98c', &['\u6b77']), ('\uf98d', &['\u8f62']), ('\uf98e',
        &['\u5e74']), ('\uf98f', &['\u6190']), ('\uf990', &['\u6200']), ('\uf991', &['\u649a']),
        ('\uf992', &['\u6f23']), ('\uf993', &['\u7149']), ('\uf994', &['\u7489']), ('\uf995',
        &['\u79ca']), ('\uf996', &['\u7df4']), ('\uf997', &['\u806f']), ('\uf998', &['\u8f26']),
        ('\uf999', &['\u84ee']), ('\uf99a', &['\u9023']), ('\uf99b', &['\u934a']), ('\uf99c',
        &['\u5217']), ('\uf99d', &['\u52a3']), ('\uf99e', &['\u54bd']), ('\uf99f', &['\u70c8']),
        ('\uf9a0', &['\u88c2']), ('\uf9a1', &['\u8aaa']), ('\uf9a2', &['\u5ec9']), ('\uf9a3',
        &['\u5ff5']), ('\uf9a4', &['\u637b']), ('\uf9a5', &['\u6bae']), ('\uf9a6', &['\u7c3e']),
        ('\uf9a7', &['\u7375']), ('\uf9a8', &['\u4ee4']), ('\uf9a9', &['\u56f9']), ('\uf9aa',
        &['\u5be7']), ('\uf9ab', &['\u5dba']), ('\uf9ac', &['\u601c']), ('\uf9ad', &['\u73b2']),
        ('\uf9ae', &['\u7469']), ('\uf9af', &['\u7f9a']), ('\uf9b0', &['\u8046']), ('\uf9b1',
        &['\u9234']), ('\uf9b2', &['\u96f6']), ('\uf9b3', &['\u9748']), ('\uf9b4', &['\u9818']),
        ('\uf9b5', &['\u4f8b']), ('\uf9b6', &['\u79ae']), ('\uf9b7', &['\u91b4']), ('\uf9b8',
        &['\u96b8']), ('\uf9b9', &['\u60e1']), ('\uf9ba', &['\u4e86']), ('\uf9bb', &['\u50da']),
        ('\uf9bc', &['\u5bee']), ('\uf9bd', &['\u5c3f']), ('\uf9be', &['\u6599']), ('\uf9bf',
        &['\u6a02']), ('\uf9c0', &['\u71ce']), ('\uf9c1', &['\u7642']), ('\uf9c2', &['\u84fc']),
        ('\uf9c3', &['\u907c']), ('\uf9c4', &['\u9f8d']), ('\uf9c5', &['\u6688']), ('\uf9c6',
        &['\u962e']), ('\uf9c7', &['\u5289']), ('\uf9c8', &['\u677b']), ('\uf9c9', &['\u67f3']),
        ('\uf9ca', &['\u6d41']), ('\uf9cb', &['\u6e9c']), ('\uf9cc', &['\u7409']), ('\uf9cd',
        &['\u7559']), ('\uf9ce', &['\u786b']), ('\uf9cf', &['\u7d10']), ('\uf9d0', &['\u985e']),
        ('\uf9d1', &['\u516d']), ('\uf9d2', &['\u622e']), ('\uf9d3', &['\u9678']), ('\uf9d4',
        &['\u502b']), ('\uf9d5', &['\u5d19']), ('\uf9d6', &['\u6dea']), ('\uf9d7', &['\u8f2a']),
        ('\uf9d8', &['\u5f8b']), ('\uf9d9', &['\u6144']), ('\uf9da', &['\u6817']), ('\uf9db',
        &['\u7387']), ('\uf9dc', &['\u9686']), ('\uf9dd', &['\u5229']), ('\uf9de', &['\u540f']),
        ('\uf9df', &['\u5c65']), ('\uf9e0', &['\u6613']), ('\uf9e1', &['\u674e']), ('\uf9e2',
        &['\u68a8']), ('\uf9e3', &['\u6ce5']), ('\uf9e4', &['\u7406']), ('\uf9e5', &['\u75e2']),
        ('\uf9e6', &['\u7f79']), ('\uf9e7', &['\u88cf']), ('\uf9e8', &['\u88e1']), ('\uf9e9',
        &['\u91cc']), ('\uf9ea', &['\u96e2']), ('\uf9eb', &['\u533f']), ('\uf9ec', &['\u6eba']),
        ('\uf9ed', &['\u541d']), ('\uf9ee', &['\u71d0']), ('\uf9ef', &['\u7498']), ('\uf9f0',
        &['\u85fa']), ('\uf9f1', &['\u96a3']), ('\uf9f2', &['\u9c57']), ('\uf9f3', &['\u9e9f']),
        ('\uf9f4', &['\u6797']), ('\uf9f5', &['\u6dcb']), ('\uf9f6', &['\u81e8']), ('\uf9f7',
        &['\u7acb']), ('\uf9f8', &['\u7b20']), ('\uf9f9', &['\u7c92']), ('\uf9fa', &['\u72c0']),
        ('\uf9fb', &['\u7099']), ('\uf9fc', &['\u8b58']), ('\uf9fd', &['\u4ec0']), ('\uf9fe',
        &['\u8336']), ('\uf9ff', &['\u523a']), ('\ufa00', &['\u5207']), ('\ufa01', &['\u5ea6']),
        ('\ufa02', &['\u62d3']), ('\ufa03', &['\u7cd6']), ('\ufa04', &['\u5b85']), ('\ufa05',
        &['\u6d1e']), ('\ufa06', &['\u66b4']), ('\ufa07', &['\u8f3b']), ('\ufa08', &['\u884c']),
        ('\ufa09', &['\u964d']), ('\ufa0a', &['\u898b']), ('\ufa0b', &['\u5ed3']), ('\ufa0c',
        &['\u5140']), ('\ufa0d', &['\u55c0']), ('\ufa10', &['\u585a']), ('\ufa12', &['\u6674']),
        ('\ufa15', &['\u51de']), ('\ufa16', &['\u732a']), ('\ufa17', &['\u76ca']), ('\ufa18',
        &['\u793c']), ('\ufa19', &['\u795e']), ('\ufa1a', &['\u7965']), ('\ufa1b', &['\u798f']),
        ('\ufa1c', &['\u9756']), ('\ufa1d', &['\u7cbe']), ('\ufa1e', &['\u7fbd']), ('\ufa20',
        &['\u8612']), ('\ufa22', &['\u8af8']), ('\ufa25', &['\u9038']), ('\ufa26', &['\u90fd']),
        ('\ufa2a', &['\u98ef']), ('\ufa2b', &['\u98fc']), ('\ufa2c', &['\u9928']), ('\ufa2d',
        &['\u9db4']), ('\ufa2e', &['\u90de']), ('\ufa2f', &['\u96b7']), ('\ufa30', &['\u4fae']),
        ('\ufa31', &['\u50e7']), ('\ufa32', &['\u514d']), ('\ufa33', &['\u52c9']), ('\ufa34',
        &['\u52e4']), ('\ufa35', &['\u5351']), ('\ufa36', &['\u559d']), ('\ufa37', &['\u5606']),
        ('\ufa38', &['\u5668']), ('\ufa39', &['\u5840']), ('\ufa3a', &['\u58a8']), ('\ufa3b',
        &['\u5c64']), ('\ufa3c', &['\u5c6e']), ('\ufa3d', &['\u6094']), ('\ufa3e', &['\u6168']),
        ('\ufa3f', &['\u618e']), ('\ufa40', &['\u61f2']), ('\ufa41', &['\u654f']), ('\ufa42',
        &['\u65e2']), ('\ufa43', &['\u6691']), ('\ufa44', &['\u6885']), ('\ufa45', &['\u6d77']),
        ('\ufa46', &['\u6e1a']), ('\ufa47', &['\u6f22']), ('\ufa48', &['\u716e']), ('\ufa49',
        &['\u722b']), ('\ufa4a', &['\u7422']), ('\ufa4b', &['\u7891']), ('\ufa4c', &['\u793e']),
        ('\ufa4d', &['\u7949']), ('\ufa4e', &['\u7948']), ('\ufa4f', &['\u7950']), ('\ufa50',
        &['\u7956']), ('\ufa51', &['\u795d']), ('\ufa52', &['\u798d']), ('\ufa53', &['\u798e']),
        ('\ufa54', &['\u7a40']), ('\ufa55', &['\u7a81']), ('\ufa56', &['\u7bc0']), ('\ufa57',
        &['\u7df4']), ('\ufa58', &['\u7e09']), ('\ufa59', &['\u7e41']), ('\ufa5a', &['\u7f72']),
        ('\ufa5b', &['\u8005']), ('\ufa5c', &['\u81ed']), ('\ufa5d', &['\u8279']), ('\ufa5e',
        &['\u8279']), ('\ufa5f', &['\u8457']), ('\ufa60', &['\u8910']), ('\ufa61', &['\u8996']),
        ('\ufa62', &['\u8b01']), ('\ufa63', &['\u8b39']), ('\ufa64', &['\u8cd3']), ('\ufa65',
        &['\u8d08']), ('\ufa66', &['\u8fb6']), ('\ufa67', &['\u9038']), ('\ufa68', &['\u96e3']),
        ('\ufa69', &['\u97ff']), ('\ufa6a', &['\u983b']), ('\ufa6b', &['\u6075']), ('\ufa6c',
        &['\U000242ee']), ('\ufa6d', &['\u8218']), ('\ufa70', &['\u4e26']), ('\ufa71', &['\u51b5']),
        ('\ufa72', &['\u5168']), ('\ufa73', &['\u4f80']), ('\ufa74', &['\u5145']), ('\ufa75',
        &['\u5180']), ('\ufa76', &['\u52c7']), ('\ufa77', &['\u52fa']), ('\ufa78', &['\u559d']),
        ('\ufa79', &['\u5555']), ('\ufa7a', &['\u5599']), ('\ufa7b', &['\u55e2']), ('\ufa7c',
        &['\u585a']), ('\ufa7d', &['\u58b3']), ('\ufa7e', &['\u5944']), ('\ufa7f', &['\u5954']),
        ('\ufa80', &['\u5a62']), ('\ufa81', &['\u5b28']), ('\ufa82', &['\u5ed2']), ('\ufa83',
        &['\u5ed9']), ('\ufa84', &['\u5f69']), ('\ufa85', &['\u5fad']), ('\ufa86', &['\u60d8']),
        ('\ufa87', &['\u614e']), ('\ufa88', &['\u6108']), ('\ufa89', &['\u618e']), ('\ufa8a',
        &['\u6160']), ('\ufa8b', &['\u61f2']), ('\ufa8c', &['\u6234']), ('\ufa8d', &['\u63c4']),
        ('\ufa8e', &['\u641c']), ('\ufa8f', &['\u6452']), ('\ufa90', &['\u6556']), ('\ufa91',
        &['\u6674']), ('\ufa92', &['\u6717']), ('\ufa93', &['\u671b']), ('\ufa94', &['\u6756']),
        ('\ufa95', &['\u6b79']), ('\ufa96', &['\u6bba']), ('\ufa97', &['\u6d41']), ('\ufa98',
        &['\u6edb']), ('\ufa99', &['\u6ecb']), ('\ufa9a', &['\u6f22']), ('\ufa9b', &['\u701e']),
        ('\ufa9c', &['\u716e']), ('\ufa9d', &['\u77a7']), ('\ufa9e', &['\u7235']), ('\ufa9f',
        &['\u72af']), ('\ufaa0', &['\u732a']), ('\ufaa1', &['\u7471']), ('\ufaa2', &['\u7506']),
        ('\ufaa3', &['\u753b']), ('\ufaa4', &['\u761d']), ('\ufaa5', &['\u761f']), ('\ufaa6',
        &['\u76ca']), ('\ufaa7', &['\u76db']), ('\ufaa8', &['\u76f4']), ('\ufaa9', &['\u774a']),
        ('\ufaaa', &['\u7740']), ('\ufaab', &['\u78cc']), ('\ufaac', &['\u7ab1']), ('\ufaad',
        &['\u7bc0']), ('\ufaae', &['\u7c7b']), ('\ufaaf', &['\u7d5b']), ('\ufab0', &['\u7df4']),
        ('\ufab1', &['\u7f3e']), ('\ufab2', &['\u8005']), ('\ufab3', &['\u8352']), ('\ufab4',
        &['\u83ef']), ('\ufab5', &['\u8779']), ('\ufab6', &['\u8941']), ('\ufab7', &['\u8986']),
        ('\ufab8', &['\u8996']), ('\ufab9', &['\u8abf']), ('\ufaba', &['\u8af8']), ('\ufabb',
        &['\u8acb']), ('\ufabc', &['\u8b01']), ('\ufabd', &['\u8afe']), ('\ufabe', &['\u8aed']),
        ('\ufabf', &['\u8b39']), ('\ufac0', &['\u8b8a']), ('\ufac1', &['\u8d08']), ('\ufac2',
        &['\u8f38']), ('\ufac3', &['\u9072']), ('\ufac4', &['\u9199']), ('\ufac5', &['\u9276']),
        ('\ufac6', &['\u967c']), ('\ufac7', &['\u96e3']), ('\ufac8', &['\u9756']), ('\ufac9',
        &['\u97db']), ('\ufaca', &['\u97ff']), ('\ufacb', &['\u980b']), ('\ufacc', &['\u983b']),
        ('\ufacd', &['\u9b12']), ('\uface', &['\u9f9c']), ('\ufacf', &['\U0002284a']), ('\ufad0',
        &['\U00022844']), ('\ufad1', &['\U000233d5']), ('\ufad2', &['\u3b9d']), ('\ufad3',
        &['\u4018']), ('\ufad4', &['\u4039']), ('\ufad5', &['\U00025249']), ('\ufad6',
        &['\U00025cd0']), ('\ufad7', &['\U00027ed3']), ('\ufad8', &['\u9f43']), ('\ufad9',
        &['\u9f8e']), ('\ufb1d', &['\u05d9', '\u05b4']), ('\ufb1f', &['\u05f2', '\u05b7']),
        ('\ufb2a', &['\u05e9', '\u05c1']), ('\ufb2b', &['\u05e9', '\u05c2']), ('\ufb2c', &['\ufb49',
        '\u05c1']), ('\ufb2d', &['\ufb49', '\u05c2']), ('\ufb2e', &['\u05d0', '\u05b7']), ('\ufb2f',
        &['\u05d0', '\u05b8']), ('\ufb30', &['\u05d0', '\u05bc']), ('\ufb31', &['\u05d1',
        '\u05bc']), ('\ufb32', &['\u05d2', '\u05bc']), ('\ufb33', &['\u05d3', '\u05bc']), ('\ufb34',
        &['\u05d4', '\u05bc']), ('\ufb35', &['\u05d5', '\u05bc']), ('\ufb36', &['\u05d6',
        '\u05bc']), ('\ufb38', &['\u05d8', '\u05bc']), ('\ufb39', &['\u05d9', '\u05bc']), ('\ufb3a',
        &['\u05da', '\u05bc']), ('\ufb3b', &['\u05db', '\u05bc']), ('\ufb3c', &['\u05dc',
        '\u05bc']), ('\ufb3e', &['\u05de', '\u05bc']), ('\ufb40', &['\u05e0', '\u05bc']), ('\ufb41',
        &['\u05e1', '\u05bc']), ('\ufb43', &['\u05e3', '\u05bc']), ('\ufb44', &['\u05e4',
        '\u05bc']), ('\ufb46', &['\u05e6', '\u05bc']), ('\ufb47', &['\u05e7', '\u05bc']), ('\ufb48',
        &['\u05e8', '\u05bc']), ('\ufb49', &['\u05e9', '\u05bc']), ('\ufb4a', &['\u05ea',
        '\u05bc']), ('\ufb4b', &['\u05d5', '\u05b9']), ('\ufb4c', &['\u05d1', '\u05bf']), ('\ufb4d',
        &['\u05db', '\u05bf']), ('\ufb4e', &['\u05e4', '\u05bf']), ('\U0001109a', &['\U00011099',
        '\U000110ba']), ('\U0001109c', &['\U0001109b', '\U000110ba']), ('\U000110ab',
        &['\U000110a5', '\U000110ba']), ('\U0001112e', &['\U00011131', '\U00011127']),
        ('\U0001112f', &['\U00011132', '\U00011127']), ('\U0001d15e', &['\U0001d157',
        '\U0001d165']), ('\U0001d15f', &['\U0001d158', '\U0001d165']), ('\U0001d160',
        &['\U0001d15f', '\U0001d16e']), ('\U0001d161', &['\U0001d15f', '\U0001d16f']),
        ('\U0001d162', &['\U0001d15f', '\U0001d170']), ('\U0001d163', &['\U0001d15f',
        '\U0001d171']), ('\U0001d164', &['\U0001d15f', '\U0001d172']), ('\U0001d1bb',
        &['\U0001d1b9', '\U0001d165']), ('\U0001d1bc', &['\U0001d1ba', '\U0001d165']),
        ('\U0001d1bd', &['\U0001d1bb', '\U0001d16e']), ('\U0001d1be', &['\U0001d1bc',
        '\U0001d16e']), ('\U0001d1bf', &['\U0001d1bb', '\U0001d16f']), ('\U0001d1c0',
        &['\U0001d1bc', '\U0001d16f']), ('\U0002f800', &['\u4e3d']), ('\U0002f801', &['\u4e38']),
        ('\U0002f802', &['\u4e41']), ('\U0002f803', &['\U00020122']), ('\U0002f804', &['\u4f60']),
        ('\U0002f805', &['\u4fae']), ('\U0002f806', &['\u4fbb']), ('\U0002f807', &['\u5002']),
        ('\U0002f808', &['\u507a']), ('\U0002f809', &['\u5099']), ('\U0002f80a', &['\u50e7']),
        ('\U0002f80b', &['\u50cf']), ('\U0002f80c', &['\u349e']), ('\U0002f80d', &['\U0002063a']),
        ('\U0002f80e', &['\u514d']), ('\U0002f80f', &['\u5154']), ('\U0002f810', &['\u5164']),
        ('\U0002f811', &['\u5177']), ('\U0002f812', &['\U0002051c']), ('\U0002f813', &['\u34b9']),
        ('\U0002f814', &['\u5167']), ('\U0002f815', &['\u518d']), ('\U0002f816', &['\U0002054b']),
        ('\U0002f817', &['\u5197']), ('\U0002f818', &['\u51a4']), ('\U0002f819', &['\u4ecc']),
        ('\U0002f81a', &['\u51ac']), ('\U0002f81b', &['\u51b5']), ('\U0002f81c', &['\U000291df']),
        ('\U0002f81d', &['\u51f5']), ('\U0002f81e', &['\u5203']), ('\U0002f81f', &['\u34df']),
        ('\U0002f820', &['\u523b']), ('\U0002f821', &['\u5246']), ('\U0002f822', &['\u5272']),
        ('\U0002f823', &['\u5277']), ('\U0002f824', &['\u3515']), ('\U0002f825', &['\u52c7']),
        ('\U0002f826', &['\u52c9']), ('\U0002f827', &['\u52e4']), ('\U0002f828', &['\u52fa']),
        ('\U0002f829', &['\u5305']), ('\U0002f82a', &['\u5306']), ('\U0002f82b', &['\u5317']),
        ('\U0002f82c', &['\u5349']), ('\U0002f82d', &['\u5351']), ('\U0002f82e', &['\u535a']),
        ('\U0002f82f', &['\u5373']), ('\U0002f830', &['\u537d']), ('\U0002f831', &['\u537f']),
        ('\U0002f832', &['\u537f']), ('\U0002f833', &['\u537f']), ('\U0002f834', &['\U00020a2c']),
        ('\U0002f835', &['\u7070']), ('\U0002f836', &['\u53ca']), ('\U0002f837', &['\u53df']),
        ('\U0002f838', &['\U00020b63']), ('\U0002f839', &['\u53eb']), ('\U0002f83a', &['\u53f1']),
        ('\U0002f83b', &['\u5406']), ('\U0002f83c', &['\u549e']), ('\U0002f83d', &['\u5438']),
        ('\U0002f83e', &['\u5448']), ('\U0002f83f', &['\u5468']), ('\U0002f840', &['\u54a2']),
        ('\U0002f841', &['\u54f6']), ('\U0002f842', &['\u5510']), ('\U0002f843', &['\u5553']),
        ('\U0002f844', &['\u5563']), ('\U0002f845', &['\u5584']), ('\U0002f846', &['\u5584']),
        ('\U0002f847', &['\u5599']), ('\U0002f848', &['\u55ab']), ('\U0002f849', &['\u55b3']),
        ('\U0002f84a', &['\u55c2']), ('\U0002f84b', &['\u5716']), ('\U0002f84c', &['\u5606']),
        ('\U0002f84d', &['\u5717']), ('\U0002f84e', &['\u5651']), ('\U0002f84f', &['\u5674']),
        ('\U0002f850', &['\u5207']), ('\U0002f851', &['\u58ee']), ('\U0002f852', &['\u57ce']),
        ('\U0002f853', &['\u57f4']), ('\U0002f854', &['\u580d']), ('\U0002f855', &['\u578b']),
        ('\U0002f856', &['\u5832']), ('\U0002f857', &['\u5831']), ('\U0002f858', &['\u58ac']),
        ('\U0002f859', &['\U000214e4']), ('\U0002f85a', &['\u58f2']), ('\U0002f85b', &['\u58f7']),
        ('\U0002f85c', &['\u5906']), ('\U0002f85d', &['\u591a']), ('\U0002f85e', &['\u5922']),
        ('\U0002f85f', &['\u5962']), ('\U0002f860', &['\U000216a8']), ('\U0002f861',
        &['\U000216ea']), ('\U0002f862', &['\u59ec']), ('\U0002f863', &['\u5a1b']), ('\U0002f864',
        &['\u5a27']), ('\U0002f865', &['\u59d8']), ('\U0002f866', &['\u5a66']), ('\U0002f867',
        &['\u36ee']), ('\U0002f868', &['\u36fc']), ('\U0002f869', &['\u5b08']), ('\U0002f86a',
        &['\u5b3e']), ('\U0002f86b', &['\u5b3e']), ('\U0002f86c', &['\U000219c8']), ('\U0002f86d',
        &['\u5bc3']), ('\U0002f86e', &['\u5bd8']), ('\U0002f86f', &['\u5be7']), ('\U0002f870',
        &['\u5bf3']), ('\U0002f871', &['\U00021b18']), ('\U0002f872', &['\u5bff']), ('\U0002f873',
        &['\u5c06']), ('\U0002f874', &['\u5f53']), ('\U0002f875', &['\u5c22']), ('\U0002f876',
        &['\u3781']), ('\U0002f877', &['\u5c60']), ('\U0002f878', &['\u5c6e']), ('\U0002f879',
        &['\u5cc0']), ('\U0002f87a', &['\u5c8d']), ('\U0002f87b', &['\U00021de4']), ('\U0002f87c',
        &['\u5d43']), ('\U0002f87d', &['\U00021de6']), ('\U0002f87e', &['\u5d6e']), ('\U0002f87f',
        &['\u5d6b']), ('\U0002f880', &['\u5d7c']), ('\U0002f881', &['\u5de1']), ('\U0002f882',
        &['\u5de2']), ('\U0002f883', &['\u382f']), ('\U0002f884', &['\u5dfd']), ('\U0002f885',
        &['\u5e28']), ('\U0002f886', &['\u5e3d']), ('\U0002f887', &['\u5e69']), ('\U0002f888',
        &['\u3862']), ('\U0002f889', &['\U00022183']), ('\U0002f88a', &['\u387c']), ('\U0002f88b',
        &['\u5eb0']), ('\U0002f88c', &['\u5eb3']), ('\U0002f88d', &['\u5eb6']), ('\U0002f88e',
        &['\u5eca']), ('\U0002f88f', &['\U0002a392']), ('\U0002f890', &['\u5efe']), ('\U0002f891',
        &['\U00022331']), ('\U0002f892', &['\U00022331']), ('\U0002f893', &['\u8201']),
        ('\U0002f894', &['\u5f22']), ('\U0002f895', &['\u5f22']), ('\U0002f896', &['\u38c7']),
        ('\U0002f897', &['\U000232b8']), ('\U0002f898', &['\U000261da']), ('\U0002f899',
        &['\u5f62']), ('\U0002f89a', &['\u5f6b']), ('\U0002f89b', &['\u38e3']), ('\U0002f89c',
        &['\u5f9a']), ('\U0002f89d', &['\u5fcd']), ('\U0002f89e', &['\u5fd7']), ('\U0002f89f',
        &['\u5ff9']), ('\U0002f8a0', &['\u6081']), ('\U0002f8a1', &['\u393a']), ('\U0002f8a2',
        &['\u391c']), ('\U0002f8a3', &['\u6094']), ('\U0002f8a4', &['\U000226d4']), ('\U0002f8a5',
        &['\u60c7']), ('\U0002f8a6', &['\u6148']), ('\U0002f8a7', &['\u614c']), ('\U0002f8a8',
        &['\u614e']), ('\U0002f8a9', &['\u614c']), ('\U0002f8aa', &['\u617a']), ('\U0002f8ab',
        &['\u618e']), ('\U0002f8ac', &['\u61b2']), ('\U0002f8ad', &['\u61a4']), ('\U0002f8ae',
        &['\u61af']), ('\U0002f8af', &['\u61de']), ('\U0002f8b0', &['\u61f2']), ('\U0002f8b1',
        &['\u61f6']), ('\U0002f8b2', &['\u6210']), ('\U0002f8b3', &['\u621b']), ('\U0002f8b4',
        &['\u625d']), ('\U0002f8b5', &['\u62b1']), ('\U0002f8b6', &['\u62d4']), ('\U0002f8b7',
        &['\u6350']), ('\U0002f8b8', &['\U00022b0c']), ('\U0002f8b9', &['\u633d']), ('\U0002f8ba',
        &['\u62fc']), ('\U0002f8bb', &['\u6368']), ('\U0002f8bc', &['\u6383']), ('\U0002f8bd',
        &['\u63e4']), ('\U0002f8be', &['\U00022bf1']), ('\U0002f8bf', &['\u6422']), ('\U0002f8c0',
        &['\u63c5']), ('\U0002f8c1', &['\u63a9']), ('\U0002f8c2', &['\u3a2e']), ('\U0002f8c3',
        &['\u6469']), ('\U0002f8c4', &['\u647e']), ('\U0002f8c5', &['\u649d']), ('\U0002f8c6',
        &['\u6477']), ('\U0002f8c7', &['\u3a6c']), ('\U0002f8c8', &['\u654f']), ('\U0002f8c9',
        &['\u656c']), ('\U0002f8ca', &['\U0002300a']), ('\U0002f8cb', &['\u65e3']), ('\U0002f8cc',
        &['\u66f8']), ('\U0002f8cd', &['\u6649']), ('\U0002f8ce', &['\u3b19']), ('\U0002f8cf',
        &['\u6691']), ('\U0002f8d0', &['\u3b08']), ('\U0002f8d1', &['\u3ae4']), ('\U0002f8d2',
        &['\u5192']), ('\U0002f8d3', &['\u5195']), ('\U0002f8d4', &['\u6700']), ('\U0002f8d5',
        &['\u669c']), ('\U0002f8d6', &['\u80ad']), ('\U0002f8d7', &['\u43d9']), ('\U0002f8d8',
        &['\u6717']), ('\U0002f8d9', &['\u671b']), ('\U0002f8da', &['\u6721']), ('\U0002f8db',
        &['\u675e']), ('\U0002f8dc', &['\u6753']), ('\U0002f8dd', &['\U000233c3']), ('\U0002f8de',
        &['\u3b49']), ('\U0002f8df', &['\u67fa']), ('\U0002f8e0', &['\u6785']), ('\U0002f8e1',
        &['\u6852']), ('\U0002f8e2', &['\u6885']), ('\U0002f8e3', &['\U0002346d']), ('\U0002f8e4',
        &['\u688e']), ('\U0002f8e5', &['\u681f']), ('\U0002f8e6', &['\u6914']), ('\U0002f8e7',
        &['\u3b9d']), ('\U0002f8e8', &['\u6942']), ('\U0002f8e9', &['\u69a3']), ('\U0002f8ea',
        &['\u69ea']), ('\U0002f8eb', &['\u6aa8']), ('\U0002f8ec', &['\U000236a3']), ('\U0002f8ed',
        &['\u6adb']), ('\U0002f8ee', &['\u3c18']), ('\U0002f8ef', &['\u6b21']), ('\U0002f8f0',
        &['\U000238a7']), ('\U0002f8f1', &['\u6b54']), ('\U0002f8f2', &['\u3c4e']), ('\U0002f8f3',
        &['\u6b72']), ('\U0002f8f4', &['\u6b9f']), ('\U0002f8f5', &['\u6bba']), ('\U0002f8f6',
        &['\u6bbb']), ('\U0002f8f7', &['\U00023a8d']), ('\U0002f8f8', &['\U00021d0b']),
        ('\U0002f8f9', &['\U00023afa']), ('\U0002f8fa', &['\u6c4e']), ('\U0002f8fb',
        &['\U00023cbc']), ('\U0002f8fc', &['\u6cbf']), ('\U0002f8fd', &['\u6ccd']), ('\U0002f8fe',
        &['\u6c67']), ('\U0002f8ff', &['\u6d16']), ('\U0002f900', &['\u6d3e']), ('\U0002f901',
        &['\u6d77']), ('\U0002f902', &['\u6d41']), ('\U0002f903', &['\u6d69']), ('\U0002f904',
        &['\u6d78']), ('\U0002f905', &['\u6d85']), ('\U0002f906', &['\U00023d1e']), ('\U0002f907',
        &['\u6d34']), ('\U0002f908', &['\u6e2f']), ('\U0002f909', &['\u6e6e']), ('\U0002f90a',
        &['\u3d33']), ('\U0002f90b', &['\u6ecb']), ('\U0002f90c', &['\u6ec7']), ('\U0002f90d',
        &['\U00023ed1']), ('\U0002f90e', &['\u6df9']), ('\U0002f90f', &['\u6f6e']), ('\U0002f910',
        &['\U00023f5e']), ('\U0002f911', &['\U00023f8e']), ('\U0002f912', &['\u6fc6']),
        ('\U0002f913', &['\u7039']), ('\U0002f914', &['\u701e']), ('\U0002f915', &['\u701b']),
        ('\U0002f916', &['\u3d96']), ('\U0002f917', &['\u704a']), ('\U0002f918', &['\u707d']),
        ('\U0002f919', &['\u7077']), ('\U0002f91a', &['\u70ad']), ('\U0002f91b', &['\U00020525']),
        ('\U0002f91c', &['\u7145']), ('\U0002f91d', &['\U00024263']), ('\U0002f91e', &['\u719c']),
        ('\U0002f91f', &['\U000243ab']), ('\U0002f920', &['\u7228']), ('\U0002f921', &['\u7235']),
        ('\U0002f922', &['\u7250']), ('\U0002f923', &['\U00024608']), ('\U0002f924', &['\u7280']),
        ('\U0002f925', &['\u7295']), ('\U0002f926', &['\U00024735']), ('\U0002f927',
        &['\U00024814']), ('\U0002f928', &['\u737a']), ('\U0002f929', &['\u738b']), ('\U0002f92a',
        &['\u3eac']), ('\U0002f92b', &['\u73a5']), ('\U0002f92c', &['\u3eb8']), ('\U0002f92d',
        &['\u3eb8']), ('\U0002f92e', &['\u7447']), ('\U0002f92f', &['\u745c']), ('\U0002f930',
        &['\u7471']), ('\U0002f931', &['\u7485']), ('\U0002f932', &['\u74ca']), ('\U0002f933',
        &['\u3f1b']), ('\U0002f934', &['\u7524']), ('\U0002f935', &['\U00024c36']), ('\U0002f936',
        &['\u753e']), ('\U0002f937', &['\U00024c92']), ('\U0002f938', &['\u7570']), ('\U0002f939',
        &['\U0002219f']), ('\U0002f93a', &['\u7610']), ('\U0002f93b', &['\U00024fa1']),
        ('\U0002f93c', &['\U00024fb8']), ('\U0002f93d', &['\U00025044']), ('\U0002f93e',
        &['\u3ffc']), ('\U0002f93f', &['\u4008']), ('\U0002f940', &['\u76f4']), ('\U0002f941',
        &['\U000250f3']), ('\U0002f942', &['\U000250f2']), ('\U0002f943', &['\U00025119']),
        ('\U0002f944', &['\U00025133']), ('\U0002f945', &['\u771e']), ('\U0002f946', &['\u771f']),
        ('\U0002f947', &['\u771f']), ('\U0002f948', &['\u774a']), ('\U0002f949', &['\u4039']),
        ('\U0002f94a', &['\u778b']), ('\U0002f94b', &['\u4046']), ('\U0002f94c', &['\u4096']),
        ('\U0002f94d', &['\U0002541d']), ('\U0002f94e', &['\u784e']), ('\U0002f94f', &['\u788c']),
        ('\U0002f950', &['\u78cc']), ('\U0002f951', &['\u40e3']), ('\U0002f952', &['\U00025626']),
        ('\U0002f953', &['\u7956']), ('\U0002f954', &['\U0002569a']), ('\U0002f955',
        &['\U000256c5']), ('\U0002f956', &['\u798f']), ('\U0002f957', &['\u79eb']), ('\U0002f958',
        &['\u412f']), ('\U0002f959', &['\u7a40']), ('\U0002f95a', &['\u7a4a']), ('\U0002f95b',
        &['\u7a4f']), ('\U0002f95c', &['\U0002597c']), ('\U0002f95d', &['\U00025aa7']),
        ('\U0002f95e', &['\U00025aa7']), ('\U0002f95f', &['\u7aee']), ('\U0002f960', &['\u4202']),
        ('\U0002f961', &['\U00025bab']), ('\U0002f962', &['\u7bc6']), ('\U0002f963', &['\u7bc9']),
        ('\U0002f964', &['\u4227']), ('\U0002f965', &['\U00025c80']), ('\U0002f966', &['\u7cd2']),
        ('\U0002f967', &['\u42a0']), ('\U0002f968', &['\u7ce8']), ('\U0002f969', &['\u7ce3']),
        ('\U0002f96a', &['\u7d00']), ('\U0002f96b', &['\U00025f86']), ('\U0002f96c', &['\u7d63']),
        ('\U0002f96d', &['\u4301']), ('\U0002f96e', &['\u7dc7']), ('\U0002f96f', &['\u7e02']),
        ('\U0002f970', &['\u7e45']), ('\U0002f971', &['\u4334']), ('\U0002f972', &['\U00026228']),
        ('\U0002f973', &['\U00026247']), ('\U0002f974', &['\u4359']), ('\U0002f975',
        &['\U000262d9']), ('\U0002f976', &['\u7f7a']), ('\U0002f977', &['\U0002633e']),
        ('\U0002f978', &['\u7f95']), ('\U0002f979', &['\u7ffa']), ('\U0002f97a', &['\u8005']),
        ('\U0002f97b', &['\U000264da']), ('\U0002f97c', &['\U00026523']), ('\U0002f97d',
        &['\u8060']), ('\U0002f97e', &['\U000265a8']), ('\U0002f97f', &['\u8070']), ('\U0002f980',
        &['\U0002335f']), ('\U0002f981', &['\u43d5']), ('\U0002f982', &['\u80b2']), ('\U0002f983',
        &['\u8103']), ('\U0002f984', &['\u440b']), ('\U0002f985', &['\u813e']), ('\U0002f986',
        &['\u5ab5']), ('\U0002f987', &['\U000267a7']), ('\U0002f988', &['\U000267b5']),
        ('\U0002f989', &['\U00023393']), ('\U0002f98a', &['\U0002339c']), ('\U0002f98b',
        &['\u8201']), ('\U0002f98c', &['\u8204']), ('\U0002f98d', &['\u8f9e']), ('\U0002f98e',
        &['\u446b']), ('\U0002f98f', &['\u8291']), ('\U0002f990', &['\u828b']), ('\U0002f991',
        &['\u829d']), ('\U0002f992', &['\u52b3']), ('\U0002f993', &['\u82b1']), ('\U0002f994',
        &['\u82b3']), ('\U0002f995', &['\u82bd']), ('\U0002f996', &['\u82e6']), ('\U0002f997',
        &['\U00026b3c']), ('\U0002f998', &['\u82e5']), ('\U0002f999', &['\u831d']), ('\U0002f99a',
        &['\u8363']), ('\U0002f99b', &['\u83ad']), ('\U0002f99c', &['\u8323']), ('\U0002f99d',
        &['\u83bd']), ('\U0002f99e', &['\u83e7']), ('\U0002f99f', &['\u8457']), ('\U0002f9a0',
        &['\u8353']), ('\U0002f9a1', &['\u83ca']), ('\U0002f9a2', &['\u83cc']), ('\U0002f9a3',
        &['\u83dc']), ('\U0002f9a4', &['\U00026c36']), ('\U0002f9a5', &['\U00026d6b']),
        ('\U0002f9a6', &['\U00026cd5']), ('\U0002f9a7', &['\u452b']), ('\U0002f9a8', &['\u84f1']),
        ('\U0002f9a9', &['\u84f3']), ('\U0002f9aa', &['\u8516']), ('\U0002f9ab', &['\U000273ca']),
        ('\U0002f9ac', &['\u8564']), ('\U0002f9ad', &['\U00026f2c']), ('\U0002f9ae', &['\u455d']),
        ('\U0002f9af', &['\u4561']), ('\U0002f9b0', &['\U00026fb1']), ('\U0002f9b1',
        &['\U000270d2']), ('\U0002f9b2', &['\u456b']), ('\U0002f9b3', &['\u8650']), ('\U0002f9b4',
        &['\u865c']), ('\U0002f9b5', &['\u8667']), ('\U0002f9b6', &['\u8669']), ('\U0002f9b7',
        &['\u86a9']), ('\U0002f9b8', &['\u8688']), ('\U0002f9b9', &['\u870e']), ('\U0002f9ba',
        &['\u86e2']), ('\U0002f9bb', &['\u8779']), ('\U0002f9bc', &['\u8728']), ('\U0002f9bd',
        &['\u876b']), ('\U0002f9be', &['\u8786']), ('\U0002f9bf', &['\u45d7']), ('\U0002f9c0',
        &['\u87e1']), ('\U0002f9c1', &['\u8801']), ('\U0002f9c2', &['\u45f9']), ('\U0002f9c3',
        &['\u8860']), ('\U0002f9c4', &['\u8863']), ('\U0002f9c5', &['\U00027667']), ('\U0002f9c6',
        &['\u88d7']), ('\U0002f9c7', &['\u88de']), ('\U0002f9c8', &['\u4635']), ('\U0002f9c9',
        &['\u88fa']), ('\U0002f9ca', &['\u34bb']), ('\U0002f9cb', &['\U000278ae']), ('\U0002f9cc',
        &['\U00027966']), ('\U0002f9cd', &['\u46be']), ('\U0002f9ce', &['\u46c7']), ('\U0002f9cf',
        &['\u8aa0']), ('\U0002f9d0', &['\u8aed']), ('\U0002f9d1', &['\u8b8a']), ('\U0002f9d2',
        &['\u8c55']), ('\U0002f9d3', &['\U00027ca8']), ('\U0002f9d4', &['\u8cab']), ('\U0002f9d5',
        &['\u8cc1']), ('\U0002f9d6', &['\u8d1b']), ('\U0002f9d7', &['\u8d77']), ('\U0002f9d8',
        &['\U00027f2f']), ('\U0002f9d9', &['\U00020804']), ('\U0002f9da', &['\u8dcb']),
        ('\U0002f9db', &['\u8dbc']), ('\U0002f9dc', &['\u8df0']), ('\U0002f9dd', &['\U000208de']),
        ('\U0002f9de', &['\u8ed4']), ('\U0002f9df', &['\u8f38']), ('\U0002f9e0', &['\U000285d2']),
        ('\U0002f9e1', &['\U000285ed']), ('\U0002f9e2', &['\u9094']), ('\U0002f9e3', &['\u90f1']),
        ('\U0002f9e4', &['\u9111']), ('\U0002f9e5', &['\U0002872e']), ('\U0002f9e6', &['\u911b']),
        ('\U0002f9e7', &['\u9238']), ('\U0002f9e8', &['\u92d7']), ('\U0002f9e9', &['\u92d8']),
        ('\U0002f9ea', &['\u927c']), ('\U0002f9eb', &['\u93f9']), ('\U0002f9ec', &['\u9415']),
        ('\U0002f9ed', &['\U00028bfa']), ('\U0002f9ee', &['\u958b']), ('\U0002f9ef', &['\u4995']),
        ('\U0002f9f0', &['\u95b7']), ('\U0002f9f1', &['\U00028d77']), ('\U0002f9f2', &['\u49e6']),
        ('\U0002f9f3', &['\u96c3']), ('\U0002f9f4', &['\u5db2']), ('\U0002f9f5', &['\u9723']),
        ('\U0002f9f6', &['\U00029145']), ('\U0002f9f7', &['\U0002921a']), ('\U0002f9f8',
        &['\u4a6e']), ('\U0002f9f9', &['\u4a76']), ('\U0002f9fa', &['\u97e0']), ('\U0002f9fb',
        &['\U0002940a']), ('\U0002f9fc', &['\u4ab2']), ('\U0002f9fd', &['\U00029496']),
        ('\U0002f9fe', &['\u980b']), ('\U0002f9ff', &['\u980b']), ('\U0002fa00', &['\u9829']),
        ('\U0002fa01', &['\U000295b6']), ('\U0002fa02', &['\u98e2']), ('\U0002fa03', &['\u4b33']),
        ('\U0002fa04', &['\u9929']), ('\U0002fa05', &['\u99a7']), ('\U0002fa06', &['\u99c2']),
        ('\U0002fa07', &['\u99fe']), ('\U0002fa08', &['\u4bce']), ('\U0002fa09', &['\U00029b30']),
        ('\U0002fa0a', &['\u9b12']), ('\U0002fa0b', &['\u9c40']), ('\U0002fa0c', &['\u9cfd']),
        ('\U0002fa0d', &['\u4cce']), ('\U0002fa0e', &['\u4ced']), ('\U0002fa0f', &['\u9d67']),
        ('\U0002fa10', &['\U0002a0ce']), ('\U0002fa11', &['\u4cf8']), ('\U0002fa12',
        &['\U0002a105']), ('\U0002fa13', &['\U0002a20e']), ('\U0002fa14', &['\U0002a291']),
        ('\U0002fa15', &['\u9ebb']), ('\U0002fa16', &['\u4d56']), ('\U0002fa17', &['\u9ef9']),
        ('\U0002fa18', &['\u9efe']), ('\U0002fa19', &['\u9f05']), ('\U0002fa1a', &['\u9f0f']),
        ('\U0002fa1b', &['\u9f16']), ('\U0002fa1c', &['\u9f3b']), ('\U0002fa1d', &['\U0002a600'])
    ];

    // Compatibility decompositions
    static compatibility_table : &'static [(char, &'static [char])] = &[
        ('\xa0', &['\x20']), ('\xa8', &['\x20', '\u0308']), ('\xaa', &['\x61']), ('\xaf', &['\x20',
        '\u0304']), ('\xb2', &['\x32']), ('\xb3', &['\x33']), ('\xb4', &['\x20', '\u0301']),
        ('\xb5', &['\u03bc']), ('\xb8', &['\x20', '\u0327']), ('\xb9', &['\x31']), ('\xba',
        &['\x6f']), ('\xbc', &['\x31', '\u2044', '\x34']), ('\xbd', &['\x31', '\u2044', '\x32']),
        ('\xbe', &['\x33', '\u2044', '\x34']), ('\u0132', &['\x49', '\x4a']), ('\u0133', &['\x69',
        '\x6a']), ('\u013f', &['\x4c', '\xb7']), ('\u0140', &['\x6c', '\xb7']), ('\u0149',
        &['\u02bc', '\x6e']), ('\u017f', &['\x73']), ('\u01c4', &['\x44', '\u017d']), ('\u01c5',
        &['\x44', '\u017e']), ('\u01c6', &['\x64', '\u017e']), ('\u01c7', &['\x4c', '\x4a']),
        ('\u01c8', &['\x4c', '\x6a']), ('\u01c9', &['\x6c', '\x6a']), ('\u01ca', &['\x4e', '\x4a']),
        ('\u01cb', &['\x4e', '\x6a']), ('\u01cc', &['\x6e', '\x6a']), ('\u01f1', &['\x44', '\x5a']),
        ('\u01f2', &['\x44', '\x7a']), ('\u01f3', &['\x64', '\x7a']), ('\u02b0', &['\x68']),
        ('\u02b1', &['\u0266']), ('\u02b2', &['\x6a']), ('\u02b3', &['\x72']), ('\u02b4',
        &['\u0279']), ('\u02b5', &['\u027b']), ('\u02b6', &['\u0281']), ('\u02b7', &['\x77']),
        ('\u02b8', &['\x79']), ('\u02d8', &['\x20', '\u0306']), ('\u02d9', &['\x20', '\u0307']),
        ('\u02da', &['\x20', '\u030a']), ('\u02db', &['\x20', '\u0328']), ('\u02dc', &['\x20',
        '\u0303']), ('\u02dd', &['\x20', '\u030b']), ('\u02e0', &['\u0263']), ('\u02e1', &['\x6c']),
        ('\u02e2', &['\x73']), ('\u02e3', &['\x78']), ('\u02e4', &['\u0295']), ('\u037a', &['\x20',
        '\u0345']), ('\u0384', &['\x20', '\u0301']), ('\u03d0', &['\u03b2']), ('\u03d1',
        &['\u03b8']), ('\u03d2', &['\u03a5']), ('\u03d5', &['\u03c6']), ('\u03d6', &['\u03c0']),
        ('\u03f0', &['\u03ba']), ('\u03f1', &['\u03c1']), ('\u03f2', &['\u03c2']), ('\u03f4',
        &['\u0398']), ('\u03f5', &['\u03b5']), ('\u03f9', &['\u03a3']), ('\u0587', &['\u0565',
        '\u0582']), ('\u0675', &['\u0627', '\u0674']), ('\u0676', &['\u0648', '\u0674']), ('\u0677',
        &['\u06c7', '\u0674']), ('\u0678', &['\u064a', '\u0674']), ('\u0e33', &['\u0e4d',
        '\u0e32']), ('\u0eb3', &['\u0ecd', '\u0eb2']), ('\u0edc', &['\u0eab', '\u0e99']), ('\u0edd',
        &['\u0eab', '\u0ea1']), ('\u0f0c', &['\u0f0b']), ('\u0f77', &['\u0fb2', '\u0f81']),
        ('\u0f79', &['\u0fb3', '\u0f81']), ('\u10fc', &['\u10dc']), ('\u1d2c', &['\x41']),
        ('\u1d2d', &['\xc6']), ('\u1d2e', &['\x42']), ('\u1d30', &['\x44']), ('\u1d31', &['\x45']),
        ('\u1d32', &['\u018e']), ('\u1d33', &['\x47']), ('\u1d34', &['\x48']), ('\u1d35',
        &['\x49']), ('\u1d36', &['\x4a']), ('\u1d37', &['\x4b']), ('\u1d38', &['\x4c']), ('\u1d39',
        &['\x4d']), ('\u1d3a', &['\x4e']), ('\u1d3c', &['\x4f']), ('\u1d3d', &['\u0222']),
        ('\u1d3e', &['\x50']), ('\u1d3f', &['\x52']), ('\u1d40', &['\x54']), ('\u1d41', &['\x55']),
        ('\u1d42', &['\x57']), ('\u1d43', &['\x61']), ('\u1d44', &['\u0250']), ('\u1d45',
        &['\u0251']), ('\u1d46', &['\u1d02']), ('\u1d47', &['\x62']), ('\u1d48', &['\x64']),
        ('\u1d49', &['\x65']), ('\u1d4a', &['\u0259']), ('\u1d4b', &['\u025b']), ('\u1d4c',
        &['\u025c']), ('\u1d4d', &['\x67']), ('\u1d4f', &['\x6b']), ('\u1d50', &['\x6d']),
        ('\u1d51', &['\u014b']), ('\u1d52', &['\x6f']), ('\u1d53', &['\u0254']), ('\u1d54',
        &['\u1d16']), ('\u1d55', &['\u1d17']), ('\u1d56', &['\x70']), ('\u1d57', &['\x74']),
        ('\u1d58', &['\x75']), ('\u1d59', &['\u1d1d']), ('\u1d5a', &['\u026f']), ('\u1d5b',
        &['\x76']), ('\u1d5c', &['\u1d25']), ('\u1d5d', &['\u03b2']), ('\u1d5e', &['\u03b3']),
        ('\u1d5f', &['\u03b4']), ('\u1d60', &['\u03c6']), ('\u1d61', &['\u03c7']), ('\u1d62',
        &['\x69']), ('\u1d63', &['\x72']), ('\u1d64', &['\x75']), ('\u1d65', &['\x76']), ('\u1d66',
        &['\u03b2']), ('\u1d67', &['\u03b3']), ('\u1d68', &['\u03c1']), ('\u1d69', &['\u03c6']),
        ('\u1d6a', &['\u03c7']), ('\u1d78', &['\u043d']), ('\u1d9b', &['\u0252']), ('\u1d9c',
        &['\x63']), ('\u1d9d', &['\u0255']), ('\u1d9e', &['\xf0']), ('\u1d9f', &['\u025c']),
        ('\u1da0', &['\x66']), ('\u1da1', &['\u025f']), ('\u1da2', &['\u0261']), ('\u1da3',
        &['\u0265']), ('\u1da4', &['\u0268']), ('\u1da5', &['\u0269']), ('\u1da6', &['\u026a']),
        ('\u1da7', &['\u1d7b']), ('\u1da8', &['\u029d']), ('\u1da9', &['\u026d']), ('\u1daa',
        &['\u1d85']), ('\u1dab', &['\u029f']), ('\u1dac', &['\u0271']), ('\u1dad', &['\u0270']),
        ('\u1dae', &['\u0272']), ('\u1daf', &['\u0273']), ('\u1db0', &['\u0274']), ('\u1db1',
        &['\u0275']), ('\u1db2', &['\u0278']), ('\u1db3', &['\u0282']), ('\u1db4', &['\u0283']),
        ('\u1db5', &['\u01ab']), ('\u1db6', &['\u0289']), ('\u1db7', &['\u028a']), ('\u1db8',
        &['\u1d1c']), ('\u1db9', &['\u028b']), ('\u1dba', &['\u028c']), ('\u1dbb', &['\x7a']),
        ('\u1dbc', &['\u0290']), ('\u1dbd', &['\u0291']), ('\u1dbe', &['\u0292']), ('\u1dbf',
        &['\u03b8']), ('\u1e9a', &['\x61', '\u02be']), ('\u1fbd', &['\x20', '\u0313']), ('\u1fbf',
        &['\x20', '\u0313']), ('\u1fc0', &['\x20', '\u0342']), ('\u1ffe', &['\x20', '\u0314']),
        ('\u2002', &['\x20']), ('\u2003', &['\x20']), ('\u2004', &['\x20']), ('\u2005', &['\x20']),
        ('\u2006', &['\x20']), ('\u2007', &['\x20']), ('\u2008', &['\x20']), ('\u2009', &['\x20']),
        ('\u200a', &['\x20']), ('\u2011', &['\u2010']), ('\u2017', &['\x20', '\u0333']), ('\u2024',
        &['\x2e']), ('\u2025', &['\x2e', '\x2e']), ('\u2026', &['\x2e', '\x2e', '\x2e']), ('\u202f',
        &['\x20']), ('\u2033', &['\u2032', '\u2032']), ('\u2034', &['\u2032', '\u2032', '\u2032']),
        ('\u2036', &['\u2035', '\u2035']), ('\u2037', &['\u2035', '\u2035', '\u2035']), ('\u203c',
        &['\x21', '\x21']), ('\u203e', &['\x20', '\u0305']), ('\u2047', &['\x3f', '\x3f']),
        ('\u2048', &['\x3f', '\x21']), ('\u2049', &['\x21', '\x3f']), ('\u2057', &['\u2032',
        '\u2032', '\u2032', '\u2032']), ('\u205f', &['\x20']), ('\u2070', &['\x30']), ('\u2071',
        &['\x69']), ('\u2074', &['\x34']), ('\u2075', &['\x35']), ('\u2076', &['\x36']), ('\u2077',
        &['\x37']), ('\u2078', &['\x38']), ('\u2079', &['\x39']), ('\u207a', &['\x2b']), ('\u207b',
        &['\u2212']), ('\u207c', &['\x3d']), ('\u207d', &['\x28']), ('\u207e', &['\x29']),
        ('\u207f', &['\x6e']), ('\u2080', &['\x30']), ('\u2081', &['\x31']), ('\u2082', &['\x32']),
        ('\u2083', &['\x33']), ('\u2084', &['\x34']), ('\u2085', &['\x35']), ('\u2086', &['\x36']),
        ('\u2087', &['\x37']), ('\u2088', &['\x38']), ('\u2089', &['\x39']), ('\u208a', &['\x2b']),
        ('\u208b', &['\u2212']), ('\u208c', &['\x3d']), ('\u208d', &['\x28']), ('\u208e',
        &['\x29']), ('\u2090', &['\x61']), ('\u2091', &['\x65']), ('\u2092', &['\x6f']), ('\u2093',
        &['\x78']), ('\u2094', &['\u0259']), ('\u2095', &['\x68']), ('\u2096', &['\x6b']),
        ('\u2097', &['\x6c']), ('\u2098', &['\x6d']), ('\u2099', &['\x6e']), ('\u209a', &['\x70']),
        ('\u209b', &['\x73']), ('\u209c', &['\x74']), ('\u20a8', &['\x52', '\x73']), ('\u2100',
        &['\x61', '\x2f', '\x63']), ('\u2101', &['\x61', '\x2f', '\x73']), ('\u2102', &['\x43']),
        ('\u2103', &['\xb0', '\x43']), ('\u2105', &['\x63', '\x2f', '\x6f']), ('\u2106', &['\x63',
        '\x2f', '\x75']), ('\u2107', &['\u0190']), ('\u2109', &['\xb0', '\x46']), ('\u210a',
        &['\x67']), ('\u210b', &['\x48']), ('\u210c', &['\x48']), ('\u210d', &['\x48']), ('\u210e',
        &['\x68']), ('\u210f', &['\u0127']), ('\u2110', &['\x49']), ('\u2111', &['\x49']),
        ('\u2112', &['\x4c']), ('\u2113', &['\x6c']), ('\u2115', &['\x4e']), ('\u2116', &['\x4e',
        '\x6f']), ('\u2119', &['\x50']), ('\u211a', &['\x51']), ('\u211b', &['\x52']), ('\u211c',
        &['\x52']), ('\u211d', &['\x52']), ('\u2120', &['\x53', '\x4d']), ('\u2121', &['\x54',
        '\x45', '\x4c']), ('\u2122', &['\x54', '\x4d']), ('\u2124', &['\x5a']), ('\u2128',
        &['\x5a']), ('\u212c', &['\x42']), ('\u212d', &['\x43']), ('\u212f', &['\x65']), ('\u2130',
        &['\x45']), ('\u2131', &['\x46']), ('\u2133', &['\x4d']), ('\u2134', &['\x6f']), ('\u2135',
        &['\u05d0']), ('\u2136', &['\u05d1']), ('\u2137', &['\u05d2']), ('\u2138', &['\u05d3']),
        ('\u2139', &['\x69']), ('\u213b', &['\x46', '\x41', '\x58']), ('\u213c', &['\u03c0']),
        ('\u213d', &['\u03b3']), ('\u213e', &['\u0393']), ('\u213f', &['\u03a0']), ('\u2140',
        &['\u2211']), ('\u2145', &['\x44']), ('\u2146', &['\x64']), ('\u2147', &['\x65']),
        ('\u2148', &['\x69']), ('\u2149', &['\x6a']), ('\u2150', &['\x31', '\u2044', '\x37']),
        ('\u2151', &['\x31', '\u2044', '\x39']), ('\u2152', &['\x31', '\u2044', '\x31', '\x30']),
        ('\u2153', &['\x31', '\u2044', '\x33']), ('\u2154', &['\x32', '\u2044', '\x33']), ('\u2155',
        &['\x31', '\u2044', '\x35']), ('\u2156', &['\x32', '\u2044', '\x35']), ('\u2157', &['\x33',
        '\u2044', '\x35']), ('\u2158', &['\x34', '\u2044', '\x35']), ('\u2159', &['\x31', '\u2044',
        '\x36']), ('\u215a', &['\x35', '\u2044', '\x36']), ('\u215b', &['\x31', '\u2044', '\x38']),
        ('\u215c', &['\x33', '\u2044', '\x38']), ('\u215d', &['\x35', '\u2044', '\x38']), ('\u215e',
        &['\x37', '\u2044', '\x38']), ('\u215f', &['\x31', '\u2044']), ('\u2160', &['\x49']),
        ('\u2161', &['\x49', '\x49']), ('\u2162', &['\x49', '\x49', '\x49']), ('\u2163', &['\x49',
        '\x56']), ('\u2164', &['\x56']), ('\u2165', &['\x56', '\x49']), ('\u2166', &['\x56', '\x49',
        '\x49']), ('\u2167', &['\x56', '\x49', '\x49', '\x49']), ('\u2168', &['\x49', '\x58']),
        ('\u2169', &['\x58']), ('\u216a', &['\x58', '\x49']), ('\u216b', &['\x58', '\x49', '\x49']),
        ('\u216c', &['\x4c']), ('\u216d', &['\x43']), ('\u216e', &['\x44']), ('\u216f', &['\x4d']),
        ('\u2170', &['\x69']), ('\u2171', &['\x69', '\x69']), ('\u2172', &['\x69', '\x69', '\x69']),
        ('\u2173', &['\x69', '\x76']), ('\u2174', &['\x76']), ('\u2175', &['\x76', '\x69']),
        ('\u2176', &['\x76', '\x69', '\x69']), ('\u2177', &['\x76', '\x69', '\x69', '\x69']),
        ('\u2178', &['\x69', '\x78']), ('\u2179', &['\x78']), ('\u217a', &['\x78', '\x69']),
        ('\u217b', &['\x78', '\x69', '\x69']), ('\u217c', &['\x6c']), ('\u217d', &['\x63']),
        ('\u217e', &['\x64']), ('\u217f', &['\x6d']), ('\u2189', &['\x30', '\u2044', '\x33']),
        ('\u222c', &['\u222b', '\u222b']), ('\u222d', &['\u222b', '\u222b', '\u222b']), ('\u222f',
        &['\u222e', '\u222e']), ('\u2230', &['\u222e', '\u222e', '\u222e']), ('\u2460', &['\x31']),
        ('\u2461', &['\x32']), ('\u2462', &['\x33']), ('\u2463', &['\x34']), ('\u2464', &['\x35']),
        ('\u2465', &['\x36']), ('\u2466', &['\x37']), ('\u2467', &['\x38']), ('\u2468', &['\x39']),
        ('\u2469', &['\x31', '\x30']), ('\u246a', &['\x31', '\x31']), ('\u246b', &['\x31', '\x32']),
        ('\u246c', &['\x31', '\x33']), ('\u246d', &['\x31', '\x34']), ('\u246e', &['\x31', '\x35']),
        ('\u246f', &['\x31', '\x36']), ('\u2470', &['\x31', '\x37']), ('\u2471', &['\x31', '\x38']),
        ('\u2472', &['\x31', '\x39']), ('\u2473', &['\x32', '\x30']), ('\u2474', &['\x28', '\x31',
        '\x29']), ('\u2475', &['\x28', '\x32', '\x29']), ('\u2476', &['\x28', '\x33', '\x29']),
        ('\u2477', &['\x28', '\x34', '\x29']), ('\u2478', &['\x28', '\x35', '\x29']), ('\u2479',
        &['\x28', '\x36', '\x29']), ('\u247a', &['\x28', '\x37', '\x29']), ('\u247b', &['\x28',
        '\x38', '\x29']), ('\u247c', &['\x28', '\x39', '\x29']), ('\u247d', &['\x28', '\x31',
        '\x30', '\x29']), ('\u247e', &['\x28', '\x31', '\x31', '\x29']), ('\u247f', &['\x28',
        '\x31', '\x32', '\x29']), ('\u2480', &['\x28', '\x31', '\x33', '\x29']), ('\u2481',
        &['\x28', '\x31', '\x34', '\x29']), ('\u2482', &['\x28', '\x31', '\x35', '\x29']),
        ('\u2483', &['\x28', '\x31', '\x36', '\x29']), ('\u2484', &['\x28', '\x31', '\x37',
        '\x29']), ('\u2485', &['\x28', '\x31', '\x38', '\x29']), ('\u2486', &['\x28', '\x31',
        '\x39', '\x29']), ('\u2487', &['\x28', '\x32', '\x30', '\x29']), ('\u2488', &['\x31',
        '\x2e']), ('\u2489', &['\x32', '\x2e']), ('\u248a', &['\x33', '\x2e']), ('\u248b', &['\x34',
        '\x2e']), ('\u248c', &['\x35', '\x2e']), ('\u248d', &['\x36', '\x2e']), ('\u248e', &['\x37',
        '\x2e']), ('\u248f', &['\x38', '\x2e']), ('\u2490', &['\x39', '\x2e']), ('\u2491', &['\x31',
        '\x30', '\x2e']), ('\u2492', &['\x31', '\x31', '\x2e']), ('\u2493', &['\x31', '\x32',
        '\x2e']), ('\u2494', &['\x31', '\x33', '\x2e']), ('\u2495', &['\x31', '\x34', '\x2e']),
        ('\u2496', &['\x31', '\x35', '\x2e']), ('\u2497', &['\x31', '\x36', '\x2e']), ('\u2498',
        &['\x31', '\x37', '\x2e']), ('\u2499', &['\x31', '\x38', '\x2e']), ('\u249a', &['\x31',
        '\x39', '\x2e']), ('\u249b', &['\x32', '\x30', '\x2e']), ('\u249c', &['\x28', '\x61',
        '\x29']), ('\u249d', &['\x28', '\x62', '\x29']), ('\u249e', &['\x28', '\x63', '\x29']),
        ('\u249f', &['\x28', '\x64', '\x29']), ('\u24a0', &['\x28', '\x65', '\x29']), ('\u24a1',
        &['\x28', '\x66', '\x29']), ('\u24a2', &['\x28', '\x67', '\x29']), ('\u24a3', &['\x28',
        '\x68', '\x29']), ('\u24a4', &['\x28', '\x69', '\x29']), ('\u24a5', &['\x28', '\x6a',
        '\x29']), ('\u24a6', &['\x28', '\x6b', '\x29']), ('\u24a7', &['\x28', '\x6c', '\x29']),
        ('\u24a8', &['\x28', '\x6d', '\x29']), ('\u24a9', &['\x28', '\x6e', '\x29']), ('\u24aa',
        &['\x28', '\x6f', '\x29']), ('\u24ab', &['\x28', '\x70', '\x29']), ('\u24ac', &['\x28',
        '\x71', '\x29']), ('\u24ad', &['\x28', '\x72', '\x29']), ('\u24ae', &['\x28', '\x73',
        '\x29']), ('\u24af', &['\x28', '\x74', '\x29']), ('\u24b0', &['\x28', '\x75', '\x29']),
        ('\u24b1', &['\x28', '\x76', '\x29']), ('\u24b2', &['\x28', '\x77', '\x29']), ('\u24b3',
        &['\x28', '\x78', '\x29']), ('\u24b4', &['\x28', '\x79', '\x29']), ('\u24b5', &['\x28',
        '\x7a', '\x29']), ('\u24b6', &['\x41']), ('\u24b7', &['\x42']), ('\u24b8', &['\x43']),
        ('\u24b9', &['\x44']), ('\u24ba', &['\x45']), ('\u24bb', &['\x46']), ('\u24bc', &['\x47']),
        ('\u24bd', &['\x48']), ('\u24be', &['\x49']), ('\u24bf', &['\x4a']), ('\u24c0', &['\x4b']),
        ('\u24c1', &['\x4c']), ('\u24c2', &['\x4d']), ('\u24c3', &['\x4e']), ('\u24c4', &['\x4f']),
        ('\u24c5', &['\x50']), ('\u24c6', &['\x51']), ('\u24c7', &['\x52']), ('\u24c8', &['\x53']),
        ('\u24c9', &['\x54']), ('\u24ca', &['\x55']), ('\u24cb', &['\x56']), ('\u24cc', &['\x57']),
        ('\u24cd', &['\x58']), ('\u24ce', &['\x59']), ('\u24cf', &['\x5a']), ('\u24d0', &['\x61']),
        ('\u24d1', &['\x62']), ('\u24d2', &['\x63']), ('\u24d3', &['\x64']), ('\u24d4', &['\x65']),
        ('\u24d5', &['\x66']), ('\u24d6', &['\x67']), ('\u24d7', &['\x68']), ('\u24d8', &['\x69']),
        ('\u24d9', &['\x6a']), ('\u24da', &['\x6b']), ('\u24db', &['\x6c']), ('\u24dc', &['\x6d']),
        ('\u24dd', &['\x6e']), ('\u24de', &['\x6f']), ('\u24df', &['\x70']), ('\u24e0', &['\x71']),
        ('\u24e1', &['\x72']), ('\u24e2', &['\x73']), ('\u24e3', &['\x74']), ('\u24e4', &['\x75']),
        ('\u24e5', &['\x76']), ('\u24e6', &['\x77']), ('\u24e7', &['\x78']), ('\u24e8', &['\x79']),
        ('\u24e9', &['\x7a']), ('\u24ea', &['\x30']), ('\u2a0c', &['\u222b', '\u222b', '\u222b',
        '\u222b']), ('\u2a74', &['\x3a', '\x3a', '\x3d']), ('\u2a75', &['\x3d', '\x3d']), ('\u2a76',
        &['\x3d', '\x3d', '\x3d']), ('\u2c7c', &['\x6a']), ('\u2c7d', &['\x56']), ('\u2d6f',
        &['\u2d61']), ('\u2e9f', &['\u6bcd']), ('\u2ef3', &['\u9f9f']), ('\u2f00', &['\u4e00']),
        ('\u2f01', &['\u4e28']), ('\u2f02', &['\u4e36']), ('\u2f03', &['\u4e3f']), ('\u2f04',
        &['\u4e59']), ('\u2f05', &['\u4e85']), ('\u2f06', &['\u4e8c']), ('\u2f07', &['\u4ea0']),
        ('\u2f08', &['\u4eba']), ('\u2f09', &['\u513f']), ('\u2f0a', &['\u5165']), ('\u2f0b',
        &['\u516b']), ('\u2f0c', &['\u5182']), ('\u2f0d', &['\u5196']), ('\u2f0e', &['\u51ab']),
        ('\u2f0f', &['\u51e0']), ('\u2f10', &['\u51f5']), ('\u2f11', &['\u5200']), ('\u2f12',
        &['\u529b']), ('\u2f13', &['\u52f9']), ('\u2f14', &['\u5315']), ('\u2f15', &['\u531a']),
        ('\u2f16', &['\u5338']), ('\u2f17', &['\u5341']), ('\u2f18', &['\u535c']), ('\u2f19',
        &['\u5369']), ('\u2f1a', &['\u5382']), ('\u2f1b', &['\u53b6']), ('\u2f1c', &['\u53c8']),
        ('\u2f1d', &['\u53e3']), ('\u2f1e', &['\u56d7']), ('\u2f1f', &['\u571f']), ('\u2f20',
        &['\u58eb']), ('\u2f21', &['\u5902']), ('\u2f22', &['\u590a']), ('\u2f23', &['\u5915']),
        ('\u2f24', &['\u5927']), ('\u2f25', &['\u5973']), ('\u2f26', &['\u5b50']), ('\u2f27',
        &['\u5b80']), ('\u2f28', &['\u5bf8']), ('\u2f29', &['\u5c0f']), ('\u2f2a', &['\u5c22']),
        ('\u2f2b', &['\u5c38']), ('\u2f2c', &['\u5c6e']), ('\u2f2d', &['\u5c71']), ('\u2f2e',
        &['\u5ddb']), ('\u2f2f', &['\u5de5']), ('\u2f30', &['\u5df1']), ('\u2f31', &['\u5dfe']),
        ('\u2f32', &['\u5e72']), ('\u2f33', &['\u5e7a']), ('\u2f34', &['\u5e7f']), ('\u2f35',
        &['\u5ef4']), ('\u2f36', &['\u5efe']), ('\u2f37', &['\u5f0b']), ('\u2f38', &['\u5f13']),
        ('\u2f39', &['\u5f50']), ('\u2f3a', &['\u5f61']), ('\u2f3b', &['\u5f73']), ('\u2f3c',
        &['\u5fc3']), ('\u2f3d', &['\u6208']), ('\u2f3e', &['\u6236']), ('\u2f3f', &['\u624b']),
        ('\u2f40', &['\u652f']), ('\u2f41', &['\u6534']), ('\u2f42', &['\u6587']), ('\u2f43',
        &['\u6597']), ('\u2f44', &['\u65a4']), ('\u2f45', &['\u65b9']), ('\u2f46', &['\u65e0']),
        ('\u2f47', &['\u65e5']), ('\u2f48', &['\u66f0']), ('\u2f49', &['\u6708']), ('\u2f4a',
        &['\u6728']), ('\u2f4b', &['\u6b20']), ('\u2f4c', &['\u6b62']), ('\u2f4d', &['\u6b79']),
        ('\u2f4e', &['\u6bb3']), ('\u2f4f', &['\u6bcb']), ('\u2f50', &['\u6bd4']), ('\u2f51',
        &['\u6bdb']), ('\u2f52', &['\u6c0f']), ('\u2f53', &['\u6c14']), ('\u2f54', &['\u6c34']),
        ('\u2f55', &['\u706b']), ('\u2f56', &['\u722a']), ('\u2f57', &['\u7236']), ('\u2f58',
        &['\u723b']), ('\u2f59', &['\u723f']), ('\u2f5a', &['\u7247']), ('\u2f5b', &['\u7259']),
        ('\u2f5c', &['\u725b']), ('\u2f5d', &['\u72ac']), ('\u2f5e', &['\u7384']), ('\u2f5f',
        &['\u7389']), ('\u2f60', &['\u74dc']), ('\u2f61', &['\u74e6']), ('\u2f62', &['\u7518']),
        ('\u2f63', &['\u751f']), ('\u2f64', &['\u7528']), ('\u2f65', &['\u7530']), ('\u2f66',
        &['\u758b']), ('\u2f67', &['\u7592']), ('\u2f68', &['\u7676']), ('\u2f69', &['\u767d']),
        ('\u2f6a', &['\u76ae']), ('\u2f6b', &['\u76bf']), ('\u2f6c', &['\u76ee']), ('\u2f6d',
        &['\u77db']), ('\u2f6e', &['\u77e2']), ('\u2f6f', &['\u77f3']), ('\u2f70', &['\u793a']),
        ('\u2f71', &['\u79b8']), ('\u2f72', &['\u79be']), ('\u2f73', &['\u7a74']), ('\u2f74',
        &['\u7acb']), ('\u2f75', &['\u7af9']), ('\u2f76', &['\u7c73']), ('\u2f77', &['\u7cf8']),
        ('\u2f78', &['\u7f36']), ('\u2f79', &['\u7f51']), ('\u2f7a', &['\u7f8a']), ('\u2f7b',
        &['\u7fbd']), ('\u2f7c', &['\u8001']), ('\u2f7d', &['\u800c']), ('\u2f7e', &['\u8012']),
        ('\u2f7f', &['\u8033']), ('\u2f80', &['\u807f']), ('\u2f81', &['\u8089']), ('\u2f82',
        &['\u81e3']), ('\u2f83', &['\u81ea']), ('\u2f84', &['\u81f3']), ('\u2f85', &['\u81fc']),
        ('\u2f86', &['\u820c']), ('\u2f87', &['\u821b']), ('\u2f88', &['\u821f']), ('\u2f89',
        &['\u826e']), ('\u2f8a', &['\u8272']), ('\u2f8b', &['\u8278']), ('\u2f8c', &['\u864d']),
        ('\u2f8d', &['\u866b']), ('\u2f8e', &['\u8840']), ('\u2f8f', &['\u884c']), ('\u2f90',
        &['\u8863']), ('\u2f91', &['\u897e']), ('\u2f92', &['\u898b']), ('\u2f93', &['\u89d2']),
        ('\u2f94', &['\u8a00']), ('\u2f95', &['\u8c37']), ('\u2f96', &['\u8c46']), ('\u2f97',
        &['\u8c55']), ('\u2f98', &['\u8c78']), ('\u2f99', &['\u8c9d']), ('\u2f9a', &['\u8d64']),
        ('\u2f9b', &['\u8d70']), ('\u2f9c', &['\u8db3']), ('\u2f9d', &['\u8eab']), ('\u2f9e',
        &['\u8eca']), ('\u2f9f', &['\u8f9b']), ('\u2fa0', &['\u8fb0']), ('\u2fa1', &['\u8fb5']),
        ('\u2fa2', &['\u9091']), ('\u2fa3', &['\u9149']), ('\u2fa4', &['\u91c6']), ('\u2fa5',
        &['\u91cc']), ('\u2fa6', &['\u91d1']), ('\u2fa7', &['\u9577']), ('\u2fa8', &['\u9580']),
        ('\u2fa9', &['\u961c']), ('\u2faa', &['\u96b6']), ('\u2fab', &['\u96b9']), ('\u2fac',
        &['\u96e8']), ('\u2fad', &['\u9751']), ('\u2fae', &['\u975e']), ('\u2faf', &['\u9762']),
        ('\u2fb0', &['\u9769']), ('\u2fb1', &['\u97cb']), ('\u2fb2', &['\u97ed']), ('\u2fb3',
        &['\u97f3']), ('\u2fb4', &['\u9801']), ('\u2fb5', &['\u98a8']), ('\u2fb6', &['\u98db']),
        ('\u2fb7', &['\u98df']), ('\u2fb8', &['\u9996']), ('\u2fb9', &['\u9999']), ('\u2fba',
        &['\u99ac']), ('\u2fbb', &['\u9aa8']), ('\u2fbc', &['\u9ad8']), ('\u2fbd', &['\u9adf']),
        ('\u2fbe', &['\u9b25']), ('\u2fbf', &['\u9b2f']), ('\u2fc0', &['\u9b32']), ('\u2fc1',
        &['\u9b3c']), ('\u2fc2', &['\u9b5a']), ('\u2fc3', &['\u9ce5']), ('\u2fc4', &['\u9e75']),
        ('\u2fc5', &['\u9e7f']), ('\u2fc6', &['\u9ea5']), ('\u2fc7', &['\u9ebb']), ('\u2fc8',
        &['\u9ec3']), ('\u2fc9', &['\u9ecd']), ('\u2fca', &['\u9ed1']), ('\u2fcb', &['\u9ef9']),
        ('\u2fcc', &['\u9efd']), ('\u2fcd', &['\u9f0e']), ('\u2fce', &['\u9f13']), ('\u2fcf',
        &['\u9f20']), ('\u2fd0', &['\u9f3b']), ('\u2fd1', &['\u9f4a']), ('\u2fd2', &['\u9f52']),
        ('\u2fd3', &['\u9f8d']), ('\u2fd4', &['\u9f9c']), ('\u2fd5', &['\u9fa0']), ('\u3000',
        &['\x20']), ('\u3036', &['\u3012']), ('\u3038', &['\u5341']), ('\u3039', &['\u5344']),
        ('\u303a', &['\u5345']), ('\u309b', &['\x20', '\u3099']), ('\u309c', &['\x20', '\u309a']),
        ('\u309f', &['\u3088', '\u308a']), ('\u30ff', &['\u30b3', '\u30c8']), ('\u3131',
        &['\u1100']), ('\u3132', &['\u1101']), ('\u3133', &['\u11aa']), ('\u3134', &['\u1102']),
        ('\u3135', &['\u11ac']), ('\u3136', &['\u11ad']), ('\u3137', &['\u1103']), ('\u3138',
        &['\u1104']), ('\u3139', &['\u1105']), ('\u313a', &['\u11b0']), ('\u313b', &['\u11b1']),
        ('\u313c', &['\u11b2']), ('\u313d', &['\u11b3']), ('\u313e', &['\u11b4']), ('\u313f',
        &['\u11b5']), ('\u3140', &['\u111a']), ('\u3141', &['\u1106']), ('\u3142', &['\u1107']),
        ('\u3143', &['\u1108']), ('\u3144', &['\u1121']), ('\u3145', &['\u1109']), ('\u3146',
        &['\u110a']), ('\u3147', &['\u110b']), ('\u3148', &['\u110c']), ('\u3149', &['\u110d']),
        ('\u314a', &['\u110e']), ('\u314b', &['\u110f']), ('\u314c', &['\u1110']), ('\u314d',
        &['\u1111']), ('\u314e', &['\u1112']), ('\u314f', &['\u1161']), ('\u3150', &['\u1162']),
        ('\u3151', &['\u1163']), ('\u3152', &['\u1164']), ('\u3153', &['\u1165']), ('\u3154',
        &['\u1166']), ('\u3155', &['\u1167']), ('\u3156', &['\u1168']), ('\u3157', &['\u1169']),
        ('\u3158', &['\u116a']), ('\u3159', &['\u116b']), ('\u315a', &['\u116c']), ('\u315b',
        &['\u116d']), ('\u315c', &['\u116e']), ('\u315d', &['\u116f']), ('\u315e', &['\u1170']),
        ('\u315f', &['\u1171']), ('\u3160', &['\u1172']), ('\u3161', &['\u1173']), ('\u3162',
        &['\u1174']), ('\u3163', &['\u1175']), ('\u3164', &['\u1160']), ('\u3165', &['\u1114']),
        ('\u3166', &['\u1115']), ('\u3167', &['\u11c7']), ('\u3168', &['\u11c8']), ('\u3169',
        &['\u11cc']), ('\u316a', &['\u11ce']), ('\u316b', &['\u11d3']), ('\u316c', &['\u11d7']),
        ('\u316d', &['\u11d9']), ('\u316e', &['\u111c']), ('\u316f', &['\u11dd']), ('\u3170',
        &['\u11df']), ('\u3171', &['\u111d']), ('\u3172', &['\u111e']), ('\u3173', &['\u1120']),
        ('\u3174', &['\u1122']), ('\u3175', &['\u1123']), ('\u3176', &['\u1127']), ('\u3177',
        &['\u1129']), ('\u3178', &['\u112b']), ('\u3179', &['\u112c']), ('\u317a', &['\u112d']),
        ('\u317b', &['\u112e']), ('\u317c', &['\u112f']), ('\u317d', &['\u1132']), ('\u317e',
        &['\u1136']), ('\u317f', &['\u1140']), ('\u3180', &['\u1147']), ('\u3181', &['\u114c']),
        ('\u3182', &['\u11f1']), ('\u3183', &['\u11f2']), ('\u3184', &['\u1157']), ('\u3185',
        &['\u1158']), ('\u3186', &['\u1159']), ('\u3187', &['\u1184']), ('\u3188', &['\u1185']),
        ('\u3189', &['\u1188']), ('\u318a', &['\u1191']), ('\u318b', &['\u1192']), ('\u318c',
        &['\u1194']), ('\u318d', &['\u119e']), ('\u318e', &['\u11a1']), ('\u3192', &['\u4e00']),
        ('\u3193', &['\u4e8c']), ('\u3194', &['\u4e09']), ('\u3195', &['\u56db']), ('\u3196',
        &['\u4e0a']), ('\u3197', &['\u4e2d']), ('\u3198', &['\u4e0b']), ('\u3199', &['\u7532']),
        ('\u319a', &['\u4e59']), ('\u319b', &['\u4e19']), ('\u319c', &['\u4e01']), ('\u319d',
        &['\u5929']), ('\u319e', &['\u5730']), ('\u319f', &['\u4eba']), ('\u3200', &['\x28',
        '\u1100', '\x29']), ('\u3201', &['\x28', '\u1102', '\x29']), ('\u3202', &['\x28', '\u1103',
        '\x29']), ('\u3203', &['\x28', '\u1105', '\x29']), ('\u3204', &['\x28', '\u1106', '\x29']),
        ('\u3205', &['\x28', '\u1107', '\x29']), ('\u3206', &['\x28', '\u1109', '\x29']), ('\u3207',
        &['\x28', '\u110b', '\x29']), ('\u3208', &['\x28', '\u110c', '\x29']), ('\u3209', &['\x28',
        '\u110e', '\x29']), ('\u320a', &['\x28', '\u110f', '\x29']), ('\u320b', &['\x28', '\u1110',
        '\x29']), ('\u320c', &['\x28', '\u1111', '\x29']), ('\u320d', &['\x28', '\u1112', '\x29']),
        ('\u320e', &['\x28', '\u1100', '\u1161', '\x29']), ('\u320f', &['\x28', '\u1102', '\u1161',
        '\x29']), ('\u3210', &['\x28', '\u1103', '\u1161', '\x29']), ('\u3211', &['\x28', '\u1105',
        '\u1161', '\x29']), ('\u3212', &['\x28', '\u1106', '\u1161', '\x29']), ('\u3213', &['\x28',
        '\u1107', '\u1161', '\x29']), ('\u3214', &['\x28', '\u1109', '\u1161', '\x29']), ('\u3215',
        &['\x28', '\u110b', '\u1161', '\x29']), ('\u3216', &['\x28', '\u110c', '\u1161', '\x29']),
        ('\u3217', &['\x28', '\u110e', '\u1161', '\x29']), ('\u3218', &['\x28', '\u110f', '\u1161',
        '\x29']), ('\u3219', &['\x28', '\u1110', '\u1161', '\x29']), ('\u321a', &['\x28', '\u1111',
        '\u1161', '\x29']), ('\u321b', &['\x28', '\u1112', '\u1161', '\x29']), ('\u321c', &['\x28',
        '\u110c', '\u116e', '\x29']), ('\u321d', &['\x28', '\u110b', '\u1169', '\u110c', '\u1165',
        '\u11ab', '\x29']), ('\u321e', &['\x28', '\u110b', '\u1169', '\u1112', '\u116e', '\x29']),
        ('\u3220', &['\x28', '\u4e00', '\x29']), ('\u3221', &['\x28', '\u4e8c', '\x29']), ('\u3222',
        &['\x28', '\u4e09', '\x29']), ('\u3223', &['\x28', '\u56db', '\x29']), ('\u3224', &['\x28',
        '\u4e94', '\x29']), ('\u3225', &['\x28', '\u516d', '\x29']), ('\u3226', &['\x28', '\u4e03',
        '\x29']), ('\u3227', &['\x28', '\u516b', '\x29']), ('\u3228', &['\x28', '\u4e5d', '\x29']),
        ('\u3229', &['\x28', '\u5341', '\x29']), ('\u322a', &['\x28', '\u6708', '\x29']), ('\u322b',
        &['\x28', '\u706b', '\x29']), ('\u322c', &['\x28', '\u6c34', '\x29']), ('\u322d', &['\x28',
        '\u6728', '\x29']), ('\u322e', &['\x28', '\u91d1', '\x29']), ('\u322f', &['\x28', '\u571f',
        '\x29']), ('\u3230', &['\x28', '\u65e5', '\x29']), ('\u3231', &['\x28', '\u682a', '\x29']),
        ('\u3232', &['\x28', '\u6709', '\x29']), ('\u3233', &['\x28', '\u793e', '\x29']), ('\u3234',
        &['\x28', '\u540d', '\x29']), ('\u3235', &['\x28', '\u7279', '\x29']), ('\u3236', &['\x28',
        '\u8ca1', '\x29']), ('\u3237', &['\x28', '\u795d', '\x29']), ('\u3238', &['\x28', '\u52b4',
        '\x29']), ('\u3239', &['\x28', '\u4ee3', '\x29']), ('\u323a', &['\x28', '\u547c', '\x29']),
        ('\u323b', &['\x28', '\u5b66', '\x29']), ('\u323c', &['\x28', '\u76e3', '\x29']), ('\u323d',
        &['\x28', '\u4f01', '\x29']), ('\u323e', &['\x28', '\u8cc7', '\x29']), ('\u323f', &['\x28',
        '\u5354', '\x29']), ('\u3240', &['\x28', '\u796d', '\x29']), ('\u3241', &['\x28', '\u4f11',
        '\x29']), ('\u3242', &['\x28', '\u81ea', '\x29']), ('\u3243', &['\x28', '\u81f3', '\x29']),
        ('\u3244', &['\u554f']), ('\u3245', &['\u5e7c']), ('\u3246', &['\u6587']), ('\u3247',
        &['\u7b8f']), ('\u3250', &['\x50', '\x54', '\x45']), ('\u3251', &['\x32', '\x31']),
        ('\u3252', &['\x32', '\x32']), ('\u3253', &['\x32', '\x33']), ('\u3254', &['\x32', '\x34']),
        ('\u3255', &['\x32', '\x35']), ('\u3256', &['\x32', '\x36']), ('\u3257', &['\x32', '\x37']),
        ('\u3258', &['\x32', '\x38']), ('\u3259', &['\x32', '\x39']), ('\u325a', &['\x33', '\x30']),
        ('\u325b', &['\x33', '\x31']), ('\u325c', &['\x33', '\x32']), ('\u325d', &['\x33', '\x33']),
        ('\u325e', &['\x33', '\x34']), ('\u325f', &['\x33', '\x35']), ('\u3260', &['\u1100']),
        ('\u3261', &['\u1102']), ('\u3262', &['\u1103']), ('\u3263', &['\u1105']), ('\u3264',
        &['\u1106']), ('\u3265', &['\u1107']), ('\u3266', &['\u1109']), ('\u3267', &['\u110b']),
        ('\u3268', &['\u110c']), ('\u3269', &['\u110e']), ('\u326a', &['\u110f']), ('\u326b',
        &['\u1110']), ('\u326c', &['\u1111']), ('\u326d', &['\u1112']), ('\u326e', &['\u1100',
        '\u1161']), ('\u326f', &['\u1102', '\u1161']), ('\u3270', &['\u1103', '\u1161']), ('\u3271',
        &['\u1105', '\u1161']), ('\u3272', &['\u1106', '\u1161']), ('\u3273', &['\u1107',
        '\u1161']), ('\u3274', &['\u1109', '\u1161']), ('\u3275', &['\u110b', '\u1161']), ('\u3276',
        &['\u110c', '\u1161']), ('\u3277', &['\u110e', '\u1161']), ('\u3278', &['\u110f',
        '\u1161']), ('\u3279', &['\u1110', '\u1161']), ('\u327a', &['\u1111', '\u1161']), ('\u327b',
        &['\u1112', '\u1161']), ('\u327c', &['\u110e', '\u1161', '\u11b7', '\u1100', '\u1169']),
        ('\u327d', &['\u110c', '\u116e', '\u110b', '\u1174']), ('\u327e', &['\u110b', '\u116e']),
        ('\u3280', &['\u4e00']), ('\u3281', &['\u4e8c']), ('\u3282', &['\u4e09']), ('\u3283',
        &['\u56db']), ('\u3284', &['\u4e94']), ('\u3285', &['\u516d']), ('\u3286', &['\u4e03']),
        ('\u3287', &['\u516b']), ('\u3288', &['\u4e5d']), ('\u3289', &['\u5341']), ('\u328a',
        &['\u6708']), ('\u328b', &['\u706b']), ('\u328c', &['\u6c34']), ('\u328d', &['\u6728']),
        ('\u328e', &['\u91d1']), ('\u328f', &['\u571f']), ('\u3290', &['\u65e5']), ('\u3291',
        &['\u682a']), ('\u3292', &['\u6709']), ('\u3293', &['\u793e']), ('\u3294', &['\u540d']),
        ('\u3295', &['\u7279']), ('\u3296', &['\u8ca1']), ('\u3297', &['\u795d']), ('\u3298',
        &['\u52b4']), ('\u3299', &['\u79d8']), ('\u329a', &['\u7537']), ('\u329b', &['\u5973']),
        ('\u329c', &['\u9069']), ('\u329d', &['\u512a']), ('\u329e', &['\u5370']), ('\u329f',
        &['\u6ce8']), ('\u32a0', &['\u9805']), ('\u32a1', &['\u4f11']), ('\u32a2', &['\u5199']),
        ('\u32a3', &['\u6b63']), ('\u32a4', &['\u4e0a']), ('\u32a5', &['\u4e2d']), ('\u32a6',
        &['\u4e0b']), ('\u32a7', &['\u5de6']), ('\u32a8', &['\u53f3']), ('\u32a9', &['\u533b']),
        ('\u32aa', &['\u5b97']), ('\u32ab', &['\u5b66']), ('\u32ac', &['\u76e3']), ('\u32ad',
        &['\u4f01']), ('\u32ae', &['\u8cc7']), ('\u32af', &['\u5354']), ('\u32b0', &['\u591c']),
        ('\u32b1', &['\x33', '\x36']), ('\u32b2', &['\x33', '\x37']), ('\u32b3', &['\x33', '\x38']),
        ('\u32b4', &['\x33', '\x39']), ('\u32b5', &['\x34', '\x30']), ('\u32b6', &['\x34', '\x31']),
        ('\u32b7', &['\x34', '\x32']), ('\u32b8', &['\x34', '\x33']), ('\u32b9', &['\x34', '\x34']),
        ('\u32ba', &['\x34', '\x35']), ('\u32bb', &['\x34', '\x36']), ('\u32bc', &['\x34', '\x37']),
        ('\u32bd', &['\x34', '\x38']), ('\u32be', &['\x34', '\x39']), ('\u32bf', &['\x35', '\x30']),
        ('\u32c0', &['\x31', '\u6708']), ('\u32c1', &['\x32', '\u6708']), ('\u32c2', &['\x33',
        '\u6708']), ('\u32c3', &['\x34', '\u6708']), ('\u32c4', &['\x35', '\u6708']), ('\u32c5',
        &['\x36', '\u6708']), ('\u32c6', &['\x37', '\u6708']), ('\u32c7', &['\x38', '\u6708']),
        ('\u32c8', &['\x39', '\u6708']), ('\u32c9', &['\x31', '\x30', '\u6708']), ('\u32ca',
        &['\x31', '\x31', '\u6708']), ('\u32cb', &['\x31', '\x32', '\u6708']), ('\u32cc', &['\x48',
        '\x67']), ('\u32cd', &['\x65', '\x72', '\x67']), ('\u32ce', &['\x65', '\x56']), ('\u32cf',
        &['\x4c', '\x54', '\x44']), ('\u32d0', &['\u30a2']), ('\u32d1', &['\u30a4']), ('\u32d2',
        &['\u30a6']), ('\u32d3', &['\u30a8']), ('\u32d4', &['\u30aa']), ('\u32d5', &['\u30ab']),
        ('\u32d6', &['\u30ad']), ('\u32d7', &['\u30af']), ('\u32d8', &['\u30b1']), ('\u32d9',
        &['\u30b3']), ('\u32da', &['\u30b5']), ('\u32db', &['\u30b7']), ('\u32dc', &['\u30b9']),
        ('\u32dd', &['\u30bb']), ('\u32de', &['\u30bd']), ('\u32df', &['\u30bf']), ('\u32e0',
        &['\u30c1']), ('\u32e1', &['\u30c4']), ('\u32e2', &['\u30c6']), ('\u32e3', &['\u30c8']),
        ('\u32e4', &['\u30ca']), ('\u32e5', &['\u30cb']), ('\u32e6', &['\u30cc']), ('\u32e7',
        &['\u30cd']), ('\u32e8', &['\u30ce']), ('\u32e9', &['\u30cf']), ('\u32ea', &['\u30d2']),
        ('\u32eb', &['\u30d5']), ('\u32ec', &['\u30d8']), ('\u32ed', &['\u30db']), ('\u32ee',
        &['\u30de']), ('\u32ef', &['\u30df']), ('\u32f0', &['\u30e0']), ('\u32f1', &['\u30e1']),
        ('\u32f2', &['\u30e2']), ('\u32f3', &['\u30e4']), ('\u32f4', &['\u30e6']), ('\u32f5',
        &['\u30e8']), ('\u32f6', &['\u30e9']), ('\u32f7', &['\u30ea']), ('\u32f8', &['\u30eb']),
        ('\u32f9', &['\u30ec']), ('\u32fa', &['\u30ed']), ('\u32fb', &['\u30ef']), ('\u32fc',
        &['\u30f0']), ('\u32fd', &['\u30f1']), ('\u32fe', &['\u30f2']), ('\u3300', &['\u30a2',
        '\u30d1', '\u30fc', '\u30c8']), ('\u3301', &['\u30a2', '\u30eb', '\u30d5', '\u30a1']),
        ('\u3302', &['\u30a2', '\u30f3', '\u30da', '\u30a2']), ('\u3303', &['\u30a2', '\u30fc',
        '\u30eb']), ('\u3304', &['\u30a4', '\u30cb', '\u30f3', '\u30b0']), ('\u3305', &['\u30a4',
        '\u30f3', '\u30c1']), ('\u3306', &['\u30a6', '\u30a9', '\u30f3']), ('\u3307', &['\u30a8',
        '\u30b9', '\u30af', '\u30fc', '\u30c9']), ('\u3308', &['\u30a8', '\u30fc', '\u30ab',
        '\u30fc']), ('\u3309', &['\u30aa', '\u30f3', '\u30b9']), ('\u330a', &['\u30aa', '\u30fc',
        '\u30e0']), ('\u330b', &['\u30ab', '\u30a4', '\u30ea']), ('\u330c', &['\u30ab', '\u30e9',
        '\u30c3', '\u30c8']), ('\u330d', &['\u30ab', '\u30ed', '\u30ea', '\u30fc']), ('\u330e',
        &['\u30ac', '\u30ed', '\u30f3']), ('\u330f', &['\u30ac', '\u30f3', '\u30de']), ('\u3310',
        &['\u30ae', '\u30ac']), ('\u3311', &['\u30ae', '\u30cb', '\u30fc']), ('\u3312', &['\u30ad',
        '\u30e5', '\u30ea', '\u30fc']), ('\u3313', &['\u30ae', '\u30eb', '\u30c0', '\u30fc']),
        ('\u3314', &['\u30ad', '\u30ed']), ('\u3315', &['\u30ad', '\u30ed', '\u30b0', '\u30e9',
        '\u30e0']), ('\u3316', &['\u30ad', '\u30ed', '\u30e1', '\u30fc', '\u30c8', '\u30eb']),
        ('\u3317', &['\u30ad', '\u30ed', '\u30ef', '\u30c3', '\u30c8']), ('\u3318', &['\u30b0',
        '\u30e9', '\u30e0']), ('\u3319', &['\u30b0', '\u30e9', '\u30e0', '\u30c8', '\u30f3']),
        ('\u331a', &['\u30af', '\u30eb', '\u30bc', '\u30a4', '\u30ed']), ('\u331b', &['\u30af',
        '\u30ed', '\u30fc', '\u30cd']), ('\u331c', &['\u30b1', '\u30fc', '\u30b9']), ('\u331d',
        &['\u30b3', '\u30eb', '\u30ca']), ('\u331e', &['\u30b3', '\u30fc', '\u30dd']), ('\u331f',
        &['\u30b5', '\u30a4', '\u30af', '\u30eb']), ('\u3320', &['\u30b5', '\u30f3', '\u30c1',
        '\u30fc', '\u30e0']), ('\u3321', &['\u30b7', '\u30ea', '\u30f3', '\u30b0']), ('\u3322',
        &['\u30bb', '\u30f3', '\u30c1']), ('\u3323', &['\u30bb', '\u30f3', '\u30c8']), ('\u3324',
        &['\u30c0', '\u30fc', '\u30b9']), ('\u3325', &['\u30c7', '\u30b7']), ('\u3326', &['\u30c9',
        '\u30eb']), ('\u3327', &['\u30c8', '\u30f3']), ('\u3328', &['\u30ca', '\u30ce']), ('\u3329',
        &['\u30ce', '\u30c3', '\u30c8']), ('\u332a', &['\u30cf', '\u30a4', '\u30c4']), ('\u332b',
        &['\u30d1', '\u30fc', '\u30bb', '\u30f3', '\u30c8']), ('\u332c', &['\u30d1', '\u30fc',
        '\u30c4']), ('\u332d', &['\u30d0', '\u30fc', '\u30ec', '\u30eb']), ('\u332e', &['\u30d4',
        '\u30a2', '\u30b9', '\u30c8', '\u30eb']), ('\u332f', &['\u30d4', '\u30af', '\u30eb']),
        ('\u3330', &['\u30d4', '\u30b3']), ('\u3331', &['\u30d3', '\u30eb']), ('\u3332', &['\u30d5',
        '\u30a1', '\u30e9', '\u30c3', '\u30c9']), ('\u3333', &['\u30d5', '\u30a3', '\u30fc',
        '\u30c8']), ('\u3334', &['\u30d6', '\u30c3', '\u30b7', '\u30a7', '\u30eb']), ('\u3335',
        &['\u30d5', '\u30e9', '\u30f3']), ('\u3336', &['\u30d8', '\u30af', '\u30bf', '\u30fc',
        '\u30eb']), ('\u3337', &['\u30da', '\u30bd']), ('\u3338', &['\u30da', '\u30cb', '\u30d2']),
        ('\u3339', &['\u30d8', '\u30eb', '\u30c4']), ('\u333a', &['\u30da', '\u30f3', '\u30b9']),
        ('\u333b', &['\u30da', '\u30fc', '\u30b8']), ('\u333c', &['\u30d9', '\u30fc', '\u30bf']),
        ('\u333d', &['\u30dd', '\u30a4', '\u30f3', '\u30c8']), ('\u333e', &['\u30dc', '\u30eb',
        '\u30c8']), ('\u333f', &['\u30db', '\u30f3']), ('\u3340', &['\u30dd', '\u30f3', '\u30c9']),
        ('\u3341', &['\u30db', '\u30fc', '\u30eb']), ('\u3342', &['\u30db', '\u30fc', '\u30f3']),
        ('\u3343', &['\u30de', '\u30a4', '\u30af', '\u30ed']), ('\u3344', &['\u30de', '\u30a4',
        '\u30eb']), ('\u3345', &['\u30de', '\u30c3', '\u30cf']), ('\u3346', &['\u30de', '\u30eb',
        '\u30af']), ('\u3347', &['\u30de', '\u30f3', '\u30b7', '\u30e7', '\u30f3']), ('\u3348',
        &['\u30df', '\u30af', '\u30ed', '\u30f3']), ('\u3349', &['\u30df', '\u30ea']), ('\u334a',
        &['\u30df', '\u30ea', '\u30d0', '\u30fc', '\u30eb']), ('\u334b', &['\u30e1', '\u30ac']),
        ('\u334c', &['\u30e1', '\u30ac', '\u30c8', '\u30f3']), ('\u334d', &['\u30e1', '\u30fc',
        '\u30c8', '\u30eb']), ('\u334e', &['\u30e4', '\u30fc', '\u30c9']), ('\u334f', &['\u30e4',
        '\u30fc', '\u30eb']), ('\u3350', &['\u30e6', '\u30a2', '\u30f3']), ('\u3351', &['\u30ea',
        '\u30c3', '\u30c8', '\u30eb']), ('\u3352', &['\u30ea', '\u30e9']), ('\u3353', &['\u30eb',
        '\u30d4', '\u30fc']), ('\u3354', &['\u30eb', '\u30fc', '\u30d6', '\u30eb']), ('\u3355',
        &['\u30ec', '\u30e0']), ('\u3356', &['\u30ec', '\u30f3', '\u30c8', '\u30b2', '\u30f3']),
        ('\u3357', &['\u30ef', '\u30c3', '\u30c8']), ('\u3358', &['\x30', '\u70b9']), ('\u3359',
        &['\x31', '\u70b9']), ('\u335a', &['\x32', '\u70b9']), ('\u335b', &['\x33', '\u70b9']),
        ('\u335c', &['\x34', '\u70b9']), ('\u335d', &['\x35', '\u70b9']), ('\u335e', &['\x36',
        '\u70b9']), ('\u335f', &['\x37', '\u70b9']), ('\u3360', &['\x38', '\u70b9']), ('\u3361',
        &['\x39', '\u70b9']), ('\u3362', &['\x31', '\x30', '\u70b9']), ('\u3363', &['\x31', '\x31',
        '\u70b9']), ('\u3364', &['\x31', '\x32', '\u70b9']), ('\u3365', &['\x31', '\x33',
        '\u70b9']), ('\u3366', &['\x31', '\x34', '\u70b9']), ('\u3367', &['\x31', '\x35',
        '\u70b9']), ('\u3368', &['\x31', '\x36', '\u70b9']), ('\u3369', &['\x31', '\x37',
        '\u70b9']), ('\u336a', &['\x31', '\x38', '\u70b9']), ('\u336b', &['\x31', '\x39',
        '\u70b9']), ('\u336c', &['\x32', '\x30', '\u70b9']), ('\u336d', &['\x32', '\x31',
        '\u70b9']), ('\u336e', &['\x32', '\x32', '\u70b9']), ('\u336f', &['\x32', '\x33',
        '\u70b9']), ('\u3370', &['\x32', '\x34', '\u70b9']), ('\u3371', &['\x68', '\x50', '\x61']),
        ('\u3372', &['\x64', '\x61']), ('\u3373', &['\x41', '\x55']), ('\u3374', &['\x62', '\x61',
        '\x72']), ('\u3375', &['\x6f', '\x56']), ('\u3376', &['\x70', '\x63']), ('\u3377', &['\x64',
        '\x6d']), ('\u3378', &['\x64', '\x6d', '\xb2']), ('\u3379', &['\x64', '\x6d', '\xb3']),
        ('\u337a', &['\x49', '\x55']), ('\u337b', &['\u5e73', '\u6210']), ('\u337c', &['\u662d',
        '\u548c']), ('\u337d', &['\u5927', '\u6b63']), ('\u337e', &['\u660e', '\u6cbb']), ('\u337f',
        &['\u682a', '\u5f0f', '\u4f1a', '\u793e']), ('\u3380', &['\x70', '\x41']), ('\u3381',
        &['\x6e', '\x41']), ('\u3382', &['\u03bc', '\x41']), ('\u3383', &['\x6d', '\x41']),
        ('\u3384', &['\x6b', '\x41']), ('\u3385', &['\x4b', '\x42']), ('\u3386', &['\x4d', '\x42']),
        ('\u3387', &['\x47', '\x42']), ('\u3388', &['\x63', '\x61', '\x6c']), ('\u3389', &['\x6b',
        '\x63', '\x61', '\x6c']), ('\u338a', &['\x70', '\x46']), ('\u338b', &['\x6e', '\x46']),
        ('\u338c', &['\u03bc', '\x46']), ('\u338d', &['\u03bc', '\x67']), ('\u338e', &['\x6d',
        '\x67']), ('\u338f', &['\x6b', '\x67']), ('\u3390', &['\x48', '\x7a']), ('\u3391', &['\x6b',
        '\x48', '\x7a']), ('\u3392', &['\x4d', '\x48', '\x7a']), ('\u3393', &['\x47', '\x48',
        '\x7a']), ('\u3394', &['\x54', '\x48', '\x7a']), ('\u3395', &['\u03bc', '\u2113']),
        ('\u3396', &['\x6d', '\u2113']), ('\u3397', &['\x64', '\u2113']), ('\u3398', &['\x6b',
        '\u2113']), ('\u3399', &['\x66', '\x6d']), ('\u339a', &['\x6e', '\x6d']), ('\u339b',
        &['\u03bc', '\x6d']), ('\u339c', &['\x6d', '\x6d']), ('\u339d', &['\x63', '\x6d']),
        ('\u339e', &['\x6b', '\x6d']), ('\u339f', &['\x6d', '\x6d', '\xb2']), ('\u33a0', &['\x63',
        '\x6d', '\xb2']), ('\u33a1', &['\x6d', '\xb2']), ('\u33a2', &['\x6b', '\x6d', '\xb2']),
        ('\u33a3', &['\x6d', '\x6d', '\xb3']), ('\u33a4', &['\x63', '\x6d', '\xb3']), ('\u33a5',
        &['\x6d', '\xb3']), ('\u33a6', &['\x6b', '\x6d', '\xb3']), ('\u33a7', &['\x6d', '\u2215',
        '\x73']), ('\u33a8', &['\x6d', '\u2215', '\x73', '\xb2']), ('\u33a9', &['\x50', '\x61']),
        ('\u33aa', &['\x6b', '\x50', '\x61']), ('\u33ab', &['\x4d', '\x50', '\x61']), ('\u33ac',
        &['\x47', '\x50', '\x61']), ('\u33ad', &['\x72', '\x61', '\x64']), ('\u33ae', &['\x72',
        '\x61', '\x64', '\u2215', '\x73']), ('\u33af', &['\x72', '\x61', '\x64', '\u2215', '\x73',
        '\xb2']), ('\u33b0', &['\x70', '\x73']), ('\u33b1', &['\x6e', '\x73']), ('\u33b2',
        &['\u03bc', '\x73']), ('\u33b3', &['\x6d', '\x73']), ('\u33b4', &['\x70', '\x56']),
        ('\u33b5', &['\x6e', '\x56']), ('\u33b6', &['\u03bc', '\x56']), ('\u33b7', &['\x6d',
        '\x56']), ('\u33b8', &['\x6b', '\x56']), ('\u33b9', &['\x4d', '\x56']), ('\u33ba', &['\x70',
        '\x57']), ('\u33bb', &['\x6e', '\x57']), ('\u33bc', &['\u03bc', '\x57']), ('\u33bd',
        &['\x6d', '\x57']), ('\u33be', &['\x6b', '\x57']), ('\u33bf', &['\x4d', '\x57']), ('\u33c0',
        &['\x6b', '\u03a9']), ('\u33c1', &['\x4d', '\u03a9']), ('\u33c2', &['\x61', '\x2e', '\x6d',
        '\x2e']), ('\u33c3', &['\x42', '\x71']), ('\u33c4', &['\x63', '\x63']), ('\u33c5', &['\x63',
        '\x64']), ('\u33c6', &['\x43', '\u2215', '\x6b', '\x67']), ('\u33c7', &['\x43', '\x6f',
        '\x2e']), ('\u33c8', &['\x64', '\x42']), ('\u33c9', &['\x47', '\x79']), ('\u33ca', &['\x68',
        '\x61']), ('\u33cb', &['\x48', '\x50']), ('\u33cc', &['\x69', '\x6e']), ('\u33cd', &['\x4b',
        '\x4b']), ('\u33ce', &['\x4b', '\x4d']), ('\u33cf', &['\x6b', '\x74']), ('\u33d0', &['\x6c',
        '\x6d']), ('\u33d1', &['\x6c', '\x6e']), ('\u33d2', &['\x6c', '\x6f', '\x67']), ('\u33d3',
        &['\x6c', '\x78']), ('\u33d4', &['\x6d', '\x62']), ('\u33d5', &['\x6d', '\x69', '\x6c']),
        ('\u33d6', &['\x6d', '\x6f', '\x6c']), ('\u33d7', &['\x50', '\x48']), ('\u33d8', &['\x70',
        '\x2e', '\x6d', '\x2e']), ('\u33d9', &['\x50', '\x50', '\x4d']), ('\u33da', &['\x50',
        '\x52']), ('\u33db', &['\x73', '\x72']), ('\u33dc', &['\x53', '\x76']), ('\u33dd', &['\x57',
        '\x62']), ('\u33de', &['\x56', '\u2215', '\x6d']), ('\u33df', &['\x41', '\u2215', '\x6d']),
        ('\u33e0', &['\x31', '\u65e5']), ('\u33e1', &['\x32', '\u65e5']), ('\u33e2', &['\x33',
        '\u65e5']), ('\u33e3', &['\x34', '\u65e5']), ('\u33e4', &['\x35', '\u65e5']), ('\u33e5',
        &['\x36', '\u65e5']), ('\u33e6', &['\x37', '\u65e5']), ('\u33e7', &['\x38', '\u65e5']),
        ('\u33e8', &['\x39', '\u65e5']), ('\u33e9', &['\x31', '\x30', '\u65e5']), ('\u33ea',
        &['\x31', '\x31', '\u65e5']), ('\u33eb', &['\x31', '\x32', '\u65e5']), ('\u33ec', &['\x31',
        '\x33', '\u65e5']), ('\u33ed', &['\x31', '\x34', '\u65e5']), ('\u33ee', &['\x31', '\x35',
        '\u65e5']), ('\u33ef', &['\x31', '\x36', '\u65e5']), ('\u33f0', &['\x31', '\x37',
        '\u65e5']), ('\u33f1', &['\x31', '\x38', '\u65e5']), ('\u33f2', &['\x31', '\x39',
        '\u65e5']), ('\u33f3', &['\x32', '\x30', '\u65e5']), ('\u33f4', &['\x32', '\x31',
        '\u65e5']), ('\u33f5', &['\x32', '\x32', '\u65e5']), ('\u33f6', &['\x32', '\x33',
        '\u65e5']), ('\u33f7', &['\x32', '\x34', '\u65e5']), ('\u33f8', &['\x32', '\x35',
        '\u65e5']), ('\u33f9', &['\x32', '\x36', '\u65e5']), ('\u33fa', &['\x32', '\x37',
        '\u65e5']), ('\u33fb', &['\x32', '\x38', '\u65e5']), ('\u33fc', &['\x32', '\x39',
        '\u65e5']), ('\u33fd', &['\x33', '\x30', '\u65e5']), ('\u33fe', &['\x33', '\x31',
        '\u65e5']), ('\u33ff', &['\x67', '\x61', '\x6c']), ('\ua770', &['\ua76f']), ('\ua7f8',
        &['\u0126']), ('\ua7f9', &['\u0153']), ('\ufb00', &['\x66', '\x66']), ('\ufb01', &['\x66',
        '\x69']), ('\ufb02', &['\x66', '\x6c']), ('\ufb03', &['\x66', '\x66', '\x69']), ('\ufb04',
        &['\x66', '\x66', '\x6c']), ('\ufb05', &['\u017f', '\x74']), ('\ufb06', &['\x73', '\x74']),
        ('\ufb13', &['\u0574', '\u0576']), ('\ufb14', &['\u0574', '\u0565']), ('\ufb15', &['\u0574',
        '\u056b']), ('\ufb16', &['\u057e', '\u0576']), ('\ufb17', &['\u0574', '\u056d']), ('\ufb20',
        &['\u05e2']), ('\ufb21', &['\u05d0']), ('\ufb22', &['\u05d3']), ('\ufb23', &['\u05d4']),
        ('\ufb24', &['\u05db']), ('\ufb25', &['\u05dc']), ('\ufb26', &['\u05dd']), ('\ufb27',
        &['\u05e8']), ('\ufb28', &['\u05ea']), ('\ufb29', &['\x2b']), ('\ufb4f', &['\u05d0',
        '\u05dc']), ('\ufb50', &['\u0671']), ('\ufb51', &['\u0671']), ('\ufb52', &['\u067b']),
        ('\ufb53', &['\u067b']), ('\ufb54', &['\u067b']), ('\ufb55', &['\u067b']), ('\ufb56',
        &['\u067e']), ('\ufb57', &['\u067e']), ('\ufb58', &['\u067e']), ('\ufb59', &['\u067e']),
        ('\ufb5a', &['\u0680']), ('\ufb5b', &['\u0680']), ('\ufb5c', &['\u0680']), ('\ufb5d',
        &['\u0680']), ('\ufb5e', &['\u067a']), ('\ufb5f', &['\u067a']), ('\ufb60', &['\u067a']),
        ('\ufb61', &['\u067a']), ('\ufb62', &['\u067f']), ('\ufb63', &['\u067f']), ('\ufb64',
        &['\u067f']), ('\ufb65', &['\u067f']), ('\ufb66', &['\u0679']), ('\ufb67', &['\u0679']),
        ('\ufb68', &['\u0679']), ('\ufb69', &['\u0679']), ('\ufb6a', &['\u06a4']), ('\ufb6b',
        &['\u06a4']), ('\ufb6c', &['\u06a4']), ('\ufb6d', &['\u06a4']), ('\ufb6e', &['\u06a6']),
        ('\ufb6f', &['\u06a6']), ('\ufb70', &['\u06a6']), ('\ufb71', &['\u06a6']), ('\ufb72',
        &['\u0684']), ('\ufb73', &['\u0684']), ('\ufb74', &['\u0684']), ('\ufb75', &['\u0684']),
        ('\ufb76', &['\u0683']), ('\ufb77', &['\u0683']), ('\ufb78', &['\u0683']), ('\ufb79',
        &['\u0683']), ('\ufb7a', &['\u0686']), ('\ufb7b', &['\u0686']), ('\ufb7c', &['\u0686']),
        ('\ufb7d', &['\u0686']), ('\ufb7e', &['\u0687']), ('\ufb7f', &['\u0687']), ('\ufb80',
        &['\u0687']), ('\ufb81', &['\u0687']), ('\ufb82', &['\u068d']), ('\ufb83', &['\u068d']),
        ('\ufb84', &['\u068c']), ('\ufb85', &['\u068c']), ('\ufb86', &['\u068e']), ('\ufb87',
        &['\u068e']), ('\ufb88', &['\u0688']), ('\ufb89', &['\u0688']), ('\ufb8a', &['\u0698']),
        ('\ufb8b', &['\u0698']), ('\ufb8c', &['\u0691']), ('\ufb8d', &['\u0691']), ('\ufb8e',
        &['\u06a9']), ('\ufb8f', &['\u06a9']), ('\ufb90', &['\u06a9']), ('\ufb91', &['\u06a9']),
        ('\ufb92', &['\u06af']), ('\ufb93', &['\u06af']), ('\ufb94', &['\u06af']), ('\ufb95',
        &['\u06af']), ('\ufb96', &['\u06b3']), ('\ufb97', &['\u06b3']), ('\ufb98', &['\u06b3']),
        ('\ufb99', &['\u06b3']), ('\ufb9a', &['\u06b1']), ('\ufb9b', &['\u06b1']), ('\ufb9c',
        &['\u06b1']), ('\ufb9d', &['\u06b1']), ('\ufb9e', &['\u06ba']), ('\ufb9f', &['\u06ba']),
        ('\ufba0', &['\u06bb']), ('\ufba1', &['\u06bb']), ('\ufba2', &['\u06bb']), ('\ufba3',
        &['\u06bb']), ('\ufba4', &['\u06c0']), ('\ufba5', &['\u06c0']), ('\ufba6', &['\u06c1']),
        ('\ufba7', &['\u06c1']), ('\ufba8', &['\u06c1']), ('\ufba9', &['\u06c1']), ('\ufbaa',
        &['\u06be']), ('\ufbab', &['\u06be']), ('\ufbac', &['\u06be']), ('\ufbad', &['\u06be']),
        ('\ufbae', &['\u06d2']), ('\ufbaf', &['\u06d2']), ('\ufbb0', &['\u06d3']), ('\ufbb1',
        &['\u06d3']), ('\ufbd3', &['\u06ad']), ('\ufbd4', &['\u06ad']), ('\ufbd5', &['\u06ad']),
        ('\ufbd6', &['\u06ad']), ('\ufbd7', &['\u06c7']), ('\ufbd8', &['\u06c7']), ('\ufbd9',
        &['\u06c6']), ('\ufbda', &['\u06c6']), ('\ufbdb', &['\u06c8']), ('\ufbdc', &['\u06c8']),
        ('\ufbdd', &['\u0677']), ('\ufbde', &['\u06cb']), ('\ufbdf', &['\u06cb']), ('\ufbe0',
        &['\u06c5']), ('\ufbe1', &['\u06c5']), ('\ufbe2', &['\u06c9']), ('\ufbe3', &['\u06c9']),
        ('\ufbe4', &['\u06d0']), ('\ufbe5', &['\u06d0']), ('\ufbe6', &['\u06d0']), ('\ufbe7',
        &['\u06d0']), ('\ufbe8', &['\u0649']), ('\ufbe9', &['\u0649']), ('\ufbea', &['\u0626',
        '\u0627']), ('\ufbeb', &['\u0626', '\u0627']), ('\ufbec', &['\u0626', '\u06d5']), ('\ufbed',
        &['\u0626', '\u06d5']), ('\ufbee', &['\u0626', '\u0648']), ('\ufbef', &['\u0626',
        '\u0648']), ('\ufbf0', &['\u0626', '\u06c7']), ('\ufbf1', &['\u0626', '\u06c7']), ('\ufbf2',
        &['\u0626', '\u06c6']), ('\ufbf3', &['\u0626', '\u06c6']), ('\ufbf4', &['\u0626',
        '\u06c8']), ('\ufbf5', &['\u0626', '\u06c8']), ('\ufbf6', &['\u0626', '\u06d0']), ('\ufbf7',
        &['\u0626', '\u06d0']), ('\ufbf8', &['\u0626', '\u06d0']), ('\ufbf9', &['\u0626',
        '\u0649']), ('\ufbfa', &['\u0626', '\u0649']), ('\ufbfb', &['\u0626', '\u0649']), ('\ufbfc',
        &['\u06cc']), ('\ufbfd', &['\u06cc']), ('\ufbfe', &['\u06cc']), ('\ufbff', &['\u06cc']),
        ('\ufc00', &['\u0626', '\u062c']), ('\ufc01', &['\u0626', '\u062d']), ('\ufc02', &['\u0626',
        '\u0645']), ('\ufc03', &['\u0626', '\u0649']), ('\ufc04', &['\u0626', '\u064a']), ('\ufc05',
        &['\u0628', '\u062c']), ('\ufc06', &['\u0628', '\u062d']), ('\ufc07', &['\u0628',
        '\u062e']), ('\ufc08', &['\u0628', '\u0645']), ('\ufc09', &['\u0628', '\u0649']), ('\ufc0a',
        &['\u0628', '\u064a']), ('\ufc0b', &['\u062a', '\u062c']), ('\ufc0c', &['\u062a',
        '\u062d']), ('\ufc0d', &['\u062a', '\u062e']), ('\ufc0e', &['\u062a', '\u0645']), ('\ufc0f',
        &['\u062a', '\u0649']), ('\ufc10', &['\u062a', '\u064a']), ('\ufc11', &['\u062b',
        '\u062c']), ('\ufc12', &['\u062b', '\u0645']), ('\ufc13', &['\u062b', '\u0649']), ('\ufc14',
        &['\u062b', '\u064a']), ('\ufc15', &['\u062c', '\u062d']), ('\ufc16', &['\u062c',
        '\u0645']), ('\ufc17', &['\u062d', '\u062c']), ('\ufc18', &['\u062d', '\u0645']), ('\ufc19',
        &['\u062e', '\u062c']), ('\ufc1a', &['\u062e', '\u062d']), ('\ufc1b', &['\u062e',
        '\u0645']), ('\ufc1c', &['\u0633', '\u062c']), ('\ufc1d', &['\u0633', '\u062d']), ('\ufc1e',
        &['\u0633', '\u062e']), ('\ufc1f', &['\u0633', '\u0645']), ('\ufc20', &['\u0635',
        '\u062d']), ('\ufc21', &['\u0635', '\u0645']), ('\ufc22', &['\u0636', '\u062c']), ('\ufc23',
        &['\u0636', '\u062d']), ('\ufc24', &['\u0636', '\u062e']), ('\ufc25', &['\u0636',
        '\u0645']), ('\ufc26', &['\u0637', '\u062d']), ('\ufc27', &['\u0637', '\u0645']), ('\ufc28',
        &['\u0638', '\u0645']), ('\ufc29', &['\u0639', '\u062c']), ('\ufc2a', &['\u0639',
        '\u0645']), ('\ufc2b', &['\u063a', '\u062c']), ('\ufc2c', &['\u063a', '\u0645']), ('\ufc2d',
        &['\u0641', '\u062c']), ('\ufc2e', &['\u0641', '\u062d']), ('\ufc2f', &['\u0641',
        '\u062e']), ('\ufc30', &['\u0641', '\u0645']), ('\ufc31', &['\u0641', '\u0649']), ('\ufc32',
        &['\u0641', '\u064a']), ('\ufc33', &['\u0642', '\u062d']), ('\ufc34', &['\u0642',
        '\u0645']), ('\ufc35', &['\u0642', '\u0649']), ('\ufc36', &['\u0642', '\u064a']), ('\ufc37',
        &['\u0643', '\u0627']), ('\ufc38', &['\u0643', '\u062c']), ('\ufc39', &['\u0643',
        '\u062d']), ('\ufc3a', &['\u0643', '\u062e']), ('\ufc3b', &['\u0643', '\u0644']), ('\ufc3c',
        &['\u0643', '\u0645']), ('\ufc3d', &['\u0643', '\u0649']), ('\ufc3e', &['\u0643',
        '\u064a']), ('\ufc3f', &['\u0644', '\u062c']), ('\ufc40', &['\u0644', '\u062d']), ('\ufc41',
        &['\u0644', '\u062e']), ('\ufc42', &['\u0644', '\u0645']), ('\ufc43', &['\u0644',
        '\u0649']), ('\ufc44', &['\u0644', '\u064a']), ('\ufc45', &['\u0645', '\u062c']), ('\ufc46',
        &['\u0645', '\u062d']), ('\ufc47', &['\u0645', '\u062e']), ('\ufc48', &['\u0645',
        '\u0645']), ('\ufc49', &['\u0645', '\u0649']), ('\ufc4a', &['\u0645', '\u064a']), ('\ufc4b',
        &['\u0646', '\u062c']), ('\ufc4c', &['\u0646', '\u062d']), ('\ufc4d', &['\u0646',
        '\u062e']), ('\ufc4e', &['\u0646', '\u0645']), ('\ufc4f', &['\u0646', '\u0649']), ('\ufc50',
        &['\u0646', '\u064a']), ('\ufc51', &['\u0647', '\u062c']), ('\ufc52', &['\u0647',
        '\u0645']), ('\ufc53', &['\u0647', '\u0649']), ('\ufc54', &['\u0647', '\u064a']), ('\ufc55',
        &['\u064a', '\u062c']), ('\ufc56', &['\u064a', '\u062d']), ('\ufc57', &['\u064a',
        '\u062e']), ('\ufc58', &['\u064a', '\u0645']), ('\ufc59', &['\u064a', '\u0649']), ('\ufc5a',
        &['\u064a', '\u064a']), ('\ufc5b', &['\u0630', '\u0670']), ('\ufc5c', &['\u0631',
        '\u0670']), ('\ufc5d', &['\u0649', '\u0670']), ('\ufc5e', &['\x20', '\u064c', '\u0651']),
        ('\ufc5f', &['\x20', '\u064d', '\u0651']), ('\ufc60', &['\x20', '\u064e', '\u0651']),
        ('\ufc61', &['\x20', '\u064f', '\u0651']), ('\ufc62', &['\x20', '\u0650', '\u0651']),
        ('\ufc63', &['\x20', '\u0651', '\u0670']), ('\ufc64', &['\u0626', '\u0631']), ('\ufc65',
        &['\u0626', '\u0632']), ('\ufc66', &['\u0626', '\u0645']), ('\ufc67', &['\u0626',
        '\u0646']), ('\ufc68', &['\u0626', '\u0649']), ('\ufc69', &['\u0626', '\u064a']), ('\ufc6a',
        &['\u0628', '\u0631']), ('\ufc6b', &['\u0628', '\u0632']), ('\ufc6c', &['\u0628',
        '\u0645']), ('\ufc6d', &['\u0628', '\u0646']), ('\ufc6e', &['\u0628', '\u0649']), ('\ufc6f',
        &['\u0628', '\u064a']), ('\ufc70', &['\u062a', '\u0631']), ('\ufc71', &['\u062a',
        '\u0632']), ('\ufc72', &['\u062a', '\u0645']), ('\ufc73', &['\u062a', '\u0646']), ('\ufc74',
        &['\u062a', '\u0649']), ('\ufc75', &['\u062a', '\u064a']), ('\ufc76', &['\u062b',
        '\u0631']), ('\ufc77', &['\u062b', '\u0632']), ('\ufc78', &['\u062b', '\u0645']), ('\ufc79',
        &['\u062b', '\u0646']), ('\ufc7a', &['\u062b', '\u0649']), ('\ufc7b', &['\u062b',
        '\u064a']), ('\ufc7c', &['\u0641', '\u0649']), ('\ufc7d', &['\u0641', '\u064a']), ('\ufc7e',
        &['\u0642', '\u0649']), ('\ufc7f', &['\u0642', '\u064a']), ('\ufc80', &['\u0643',
        '\u0627']), ('\ufc81', &['\u0643', '\u0644']), ('\ufc82', &['\u0643', '\u0645']), ('\ufc83',
        &['\u0643', '\u0649']), ('\ufc84', &['\u0643', '\u064a']), ('\ufc85', &['\u0644',
        '\u0645']), ('\ufc86', &['\u0644', '\u0649']), ('\ufc87', &['\u0644', '\u064a']), ('\ufc88',
        &['\u0645', '\u0627']), ('\ufc89', &['\u0645', '\u0645']), ('\ufc8a', &['\u0646',
        '\u0631']), ('\ufc8b', &['\u0646', '\u0632']), ('\ufc8c', &['\u0646', '\u0645']), ('\ufc8d',
        &['\u0646', '\u0646']), ('\ufc8e', &['\u0646', '\u0649']), ('\ufc8f', &['\u0646',
        '\u064a']), ('\ufc90', &['\u0649', '\u0670']), ('\ufc91', &['\u064a', '\u0631']), ('\ufc92',
        &['\u064a', '\u0632']), ('\ufc93', &['\u064a', '\u0645']), ('\ufc94', &['\u064a',
        '\u0646']), ('\ufc95', &['\u064a', '\u0649']), ('\ufc96', &['\u064a', '\u064a']), ('\ufc97',
        &['\u0626', '\u062c']), ('\ufc98', &['\u0626', '\u062d']), ('\ufc99', &['\u0626',
        '\u062e']), ('\ufc9a', &['\u0626', '\u0645']), ('\ufc9b', &['\u0626', '\u0647']), ('\ufc9c',
        &['\u0628', '\u062c']), ('\ufc9d', &['\u0628', '\u062d']), ('\ufc9e', &['\u0628',
        '\u062e']), ('\ufc9f', &['\u0628', '\u0645']), ('\ufca0', &['\u0628', '\u0647']), ('\ufca1',
        &['\u062a', '\u062c']), ('\ufca2', &['\u062a', '\u062d']), ('\ufca3', &['\u062a',
        '\u062e']), ('\ufca4', &['\u062a', '\u0645']), ('\ufca5', &['\u062a', '\u0647']), ('\ufca6',
        &['\u062b', '\u0645']), ('\ufca7', &['\u062c', '\u062d']), ('\ufca8', &['\u062c',
        '\u0645']), ('\ufca9', &['\u062d', '\u062c']), ('\ufcaa', &['\u062d', '\u0645']), ('\ufcab',
        &['\u062e', '\u062c']), ('\ufcac', &['\u062e', '\u0645']), ('\ufcad', &['\u0633',
        '\u062c']), ('\ufcae', &['\u0633', '\u062d']), ('\ufcaf', &['\u0633', '\u062e']), ('\ufcb0',
        &['\u0633', '\u0645']), ('\ufcb1', &['\u0635', '\u062d']), ('\ufcb2', &['\u0635',
        '\u062e']), ('\ufcb3', &['\u0635', '\u0645']), ('\ufcb4', &['\u0636', '\u062c']), ('\ufcb5',
        &['\u0636', '\u062d']), ('\ufcb6', &['\u0636', '\u062e']), ('\ufcb7', &['\u0636',
        '\u0645']), ('\ufcb8', &['\u0637', '\u062d']), ('\ufcb9', &['\u0638', '\u0645']), ('\ufcba',
        &['\u0639', '\u062c']), ('\ufcbb', &['\u0639', '\u0645']), ('\ufcbc', &['\u063a',
        '\u062c']), ('\ufcbd', &['\u063a', '\u0645']), ('\ufcbe', &['\u0641', '\u062c']), ('\ufcbf',
        &['\u0641', '\u062d']), ('\ufcc0', &['\u0641', '\u062e']), ('\ufcc1', &['\u0641',
        '\u0645']), ('\ufcc2', &['\u0642', '\u062d']), ('\ufcc3', &['\u0642', '\u0645']), ('\ufcc4',
        &['\u0643', '\u062c']), ('\ufcc5', &['\u0643', '\u062d']), ('\ufcc6', &['\u0643',
        '\u062e']), ('\ufcc7', &['\u0643', '\u0644']), ('\ufcc8', &['\u0643', '\u0645']), ('\ufcc9',
        &['\u0644', '\u062c']), ('\ufcca', &['\u0644', '\u062d']), ('\ufccb', &['\u0644',
        '\u062e']), ('\ufccc', &['\u0644', '\u0645']), ('\ufccd', &['\u0644', '\u0647']), ('\ufcce',
        &['\u0645', '\u062c']), ('\ufccf', &['\u0645', '\u062d']), ('\ufcd0', &['\u0645',
        '\u062e']), ('\ufcd1', &['\u0645', '\u0645']), ('\ufcd2', &['\u0646', '\u062c']), ('\ufcd3',
        &['\u0646', '\u062d']), ('\ufcd4', &['\u0646', '\u062e']), ('\ufcd5', &['\u0646',
        '\u0645']), ('\ufcd6', &['\u0646', '\u0647']), ('\ufcd7', &['\u0647', '\u062c']), ('\ufcd8',
        &['\u0647', '\u0645']), ('\ufcd9', &['\u0647', '\u0670']), ('\ufcda', &['\u064a',
        '\u062c']), ('\ufcdb', &['\u064a', '\u062d']), ('\ufcdc', &['\u064a', '\u062e']), ('\ufcdd',
        &['\u064a', '\u0645']), ('\ufcde', &['\u064a', '\u0647']), ('\ufcdf', &['\u0626',
        '\u0645']), ('\ufce0', &['\u0626', '\u0647']), ('\ufce1', &['\u0628', '\u0645']), ('\ufce2',
        &['\u0628', '\u0647']), ('\ufce3', &['\u062a', '\u0645']), ('\ufce4', &['\u062a',
        '\u0647']), ('\ufce5', &['\u062b', '\u0645']), ('\ufce6', &['\u062b', '\u0647']), ('\ufce7',
        &['\u0633', '\u0645']), ('\ufce8', &['\u0633', '\u0647']), ('\ufce9', &['\u0634',
        '\u0645']), ('\ufcea', &['\u0634', '\u0647']), ('\ufceb', &['\u0643', '\u0644']), ('\ufcec',
        &['\u0643', '\u0645']), ('\ufced', &['\u0644', '\u0645']), ('\ufcee', &['\u0646',
        '\u0645']), ('\ufcef', &['\u0646', '\u0647']), ('\ufcf0', &['\u064a', '\u0645']), ('\ufcf1',
        &['\u064a', '\u0647']), ('\ufcf2', &['\u0640', '\u064e', '\u0651']), ('\ufcf3', &['\u0640',
        '\u064f', '\u0651']), ('\ufcf4', &['\u0640', '\u0650', '\u0651']), ('\ufcf5', &['\u0637',
        '\u0649']), ('\ufcf6', &['\u0637', '\u064a']), ('\ufcf7', &['\u0639', '\u0649']), ('\ufcf8',
        &['\u0639', '\u064a']), ('\ufcf9', &['\u063a', '\u0649']), ('\ufcfa', &['\u063a',
        '\u064a']), ('\ufcfb', &['\u0633', '\u0649']), ('\ufcfc', &['\u0633', '\u064a']), ('\ufcfd',
        &['\u0634', '\u0649']), ('\ufcfe', &['\u0634', '\u064a']), ('\ufcff', &['\u062d',
        '\u0649']), ('\ufd00', &['\u062d', '\u064a']), ('\ufd01', &['\u062c', '\u0649']), ('\ufd02',
        &['\u062c', '\u064a']), ('\ufd03', &['\u062e', '\u0649']), ('\ufd04', &['\u062e',
        '\u064a']), ('\ufd05', &['\u0635', '\u0649']), ('\ufd06', &['\u0635', '\u064a']), ('\ufd07',
        &['\u0636', '\u0649']), ('\ufd08', &['\u0636', '\u064a']), ('\ufd09', &['\u0634',
        '\u062c']), ('\ufd0a', &['\u0634', '\u062d']), ('\ufd0b', &['\u0634', '\u062e']), ('\ufd0c',
        &['\u0634', '\u0645']), ('\ufd0d', &['\u0634', '\u0631']), ('\ufd0e', &['\u0633',
        '\u0631']), ('\ufd0f', &['\u0635', '\u0631']), ('\ufd10', &['\u0636', '\u0631']), ('\ufd11',
        &['\u0637', '\u0649']), ('\ufd12', &['\u0637', '\u064a']), ('\ufd13', &['\u0639',
        '\u0649']), ('\ufd14', &['\u0639', '\u064a']), ('\ufd15', &['\u063a', '\u0649']), ('\ufd16',
        &['\u063a', '\u064a']), ('\ufd17', &['\u0633', '\u0649']), ('\ufd18', &['\u0633',
        '\u064a']), ('\ufd19', &['\u0634', '\u0649']), ('\ufd1a', &['\u0634', '\u064a']), ('\ufd1b',
        &['\u062d', '\u0649']), ('\ufd1c', &['\u062d', '\u064a']), ('\ufd1d', &['\u062c',
        '\u0649']), ('\ufd1e', &['\u062c', '\u064a']), ('\ufd1f', &['\u062e', '\u0649']), ('\ufd20',
        &['\u062e', '\u064a']), ('\ufd21', &['\u0635', '\u0649']), ('\ufd22', &['\u0635',
        '\u064a']), ('\ufd23', &['\u0636', '\u0649']), ('\ufd24', &['\u0636', '\u064a']), ('\ufd25',
        &['\u0634', '\u062c']), ('\ufd26', &['\u0634', '\u062d']), ('\ufd27', &['\u0634',
        '\u062e']), ('\ufd28', &['\u0634', '\u0645']), ('\ufd29', &['\u0634', '\u0631']), ('\ufd2a',
        &['\u0633', '\u0631']), ('\ufd2b', &['\u0635', '\u0631']), ('\ufd2c', &['\u0636',
        '\u0631']), ('\ufd2d', &['\u0634', '\u062c']), ('\ufd2e', &['\u0634', '\u062d']), ('\ufd2f',
        &['\u0634', '\u062e']), ('\ufd30', &['\u0634', '\u0645']), ('\ufd31', &['\u0633',
        '\u0647']), ('\ufd32', &['\u0634', '\u0647']), ('\ufd33', &['\u0637', '\u0645']), ('\ufd34',
        &['\u0633', '\u062c']), ('\ufd35', &['\u0633', '\u062d']), ('\ufd36', &['\u0633',
        '\u062e']), ('\ufd37', &['\u0634', '\u062c']), ('\ufd38', &['\u0634', '\u062d']), ('\ufd39',
        &['\u0634', '\u062e']), ('\ufd3a', &['\u0637', '\u0645']), ('\ufd3b', &['\u0638',
        '\u0645']), ('\ufd3c', &['\u0627', '\u064b']), ('\ufd3d', &['\u0627', '\u064b']), ('\ufd50',
        &['\u062a', '\u062c', '\u0645']), ('\ufd51', &['\u062a', '\u062d', '\u062c']), ('\ufd52',
        &['\u062a', '\u062d', '\u062c']), ('\ufd53', &['\u062a', '\u062d', '\u0645']), ('\ufd54',
        &['\u062a', '\u062e', '\u0645']), ('\ufd55', &['\u062a', '\u0645', '\u062c']), ('\ufd56',
        &['\u062a', '\u0645', '\u062d']), ('\ufd57', &['\u062a', '\u0645', '\u062e']), ('\ufd58',
        &['\u062c', '\u0645', '\u062d']), ('\ufd59', &['\u062c', '\u0645', '\u062d']), ('\ufd5a',
        &['\u062d', '\u0645', '\u064a']), ('\ufd5b', &['\u062d', '\u0645', '\u0649']), ('\ufd5c',
        &['\u0633', '\u062d', '\u062c']), ('\ufd5d', &['\u0633', '\u062c', '\u062d']), ('\ufd5e',
        &['\u0633', '\u062c', '\u0649']), ('\ufd5f', &['\u0633', '\u0645', '\u062d']), ('\ufd60',
        &['\u0633', '\u0645', '\u062d']), ('\ufd61', &['\u0633', '\u0645', '\u062c']), ('\ufd62',
        &['\u0633', '\u0645', '\u0645']), ('\ufd63', &['\u0633', '\u0645', '\u0645']), ('\ufd64',
        &['\u0635', '\u062d', '\u062d']), ('\ufd65', &['\u0635', '\u062d', '\u062d']), ('\ufd66',
        &['\u0635', '\u0645', '\u0645']), ('\ufd67', &['\u0634', '\u062d', '\u0645']), ('\ufd68',
        &['\u0634', '\u062d', '\u0645']), ('\ufd69', &['\u0634', '\u062c', '\u064a']), ('\ufd6a',
        &['\u0634', '\u0645', '\u062e']), ('\ufd6b', &['\u0634', '\u0645', '\u062e']), ('\ufd6c',
        &['\u0634', '\u0645', '\u0645']), ('\ufd6d', &['\u0634', '\u0645', '\u0645']), ('\ufd6e',
        &['\u0636', '\u062d', '\u0649']), ('\ufd6f', &['\u0636', '\u062e', '\u0645']), ('\ufd70',
        &['\u0636', '\u062e', '\u0645']), ('\ufd71', &['\u0637', '\u0645', '\u062d']), ('\ufd72',
        &['\u0637', '\u0645', '\u062d']), ('\ufd73', &['\u0637', '\u0645', '\u0645']), ('\ufd74',
        &['\u0637', '\u0645', '\u064a']), ('\ufd75', &['\u0639', '\u062c', '\u0645']), ('\ufd76',
        &['\u0639', '\u0645', '\u0645']), ('\ufd77', &['\u0639', '\u0645', '\u0645']), ('\ufd78',
        &['\u0639', '\u0645', '\u0649']), ('\ufd79', &['\u063a', '\u0645', '\u0645']), ('\ufd7a',
        &['\u063a', '\u0645', '\u064a']), ('\ufd7b', &['\u063a', '\u0645', '\u0649']), ('\ufd7c',
        &['\u0641', '\u062e', '\u0645']), ('\ufd7d', &['\u0641', '\u062e', '\u0645']), ('\ufd7e',
        &['\u0642', '\u0645', '\u062d']), ('\ufd7f', &['\u0642', '\u0645', '\u0645']), ('\ufd80',
        &['\u0644', '\u062d', '\u0645']), ('\ufd81', &['\u0644', '\u062d', '\u064a']), ('\ufd82',
        &['\u0644', '\u062d', '\u0649']), ('\ufd83', &['\u0644', '\u062c', '\u062c']), ('\ufd84',
        &['\u0644', '\u062c', '\u062c']), ('\ufd85', &['\u0644', '\u062e', '\u0645']), ('\ufd86',
        &['\u0644', '\u062e', '\u0645']), ('\ufd87', &['\u0644', '\u0645', '\u062d']), ('\ufd88',
        &['\u0644', '\u0645', '\u062d']), ('\ufd89', &['\u0645', '\u062d', '\u062c']), ('\ufd8a',
        &['\u0645', '\u062d', '\u0645']), ('\ufd8b', &['\u0645', '\u062d', '\u064a']), ('\ufd8c',
        &['\u0645', '\u062c', '\u062d']), ('\ufd8d', &['\u0645', '\u062c', '\u0645']), ('\ufd8e',
        &['\u0645', '\u062e', '\u062c']), ('\ufd8f', &['\u0645', '\u062e', '\u0645']), ('\ufd92',
        &['\u0645', '\u062c', '\u062e']), ('\ufd93', &['\u0647', '\u0645', '\u062c']), ('\ufd94',
        &['\u0647', '\u0645', '\u0645']), ('\ufd95', &['\u0646', '\u062d', '\u0645']), ('\ufd96',
        &['\u0646', '\u062d', '\u0649']), ('\ufd97', &['\u0646', '\u062c', '\u0645']), ('\ufd98',
        &['\u0646', '\u062c', '\u0645']), ('\ufd99', &['\u0646', '\u062c', '\u0649']), ('\ufd9a',
        &['\u0646', '\u0645', '\u064a']), ('\ufd9b', &['\u0646', '\u0645', '\u0649']), ('\ufd9c',
        &['\u064a', '\u0645', '\u0645']), ('\ufd9d', &['\u064a', '\u0645', '\u0645']), ('\ufd9e',
        &['\u0628', '\u062e', '\u064a']), ('\ufd9f', &['\u062a', '\u062c', '\u064a']), ('\ufda0',
        &['\u062a', '\u062c', '\u0649']), ('\ufda1', &['\u062a', '\u062e', '\u064a']), ('\ufda2',
        &['\u062a', '\u062e', '\u0649']), ('\ufda3', &['\u062a', '\u0645', '\u064a']), ('\ufda4',
        &['\u062a', '\u0645', '\u0649']), ('\ufda5', &['\u062c', '\u0645', '\u064a']), ('\ufda6',
        &['\u062c', '\u062d', '\u0649']), ('\ufda7', &['\u062c', '\u0645', '\u0649']), ('\ufda8',
        &['\u0633', '\u062e', '\u0649']), ('\ufda9', &['\u0635', '\u062d', '\u064a']), ('\ufdaa',
        &['\u0634', '\u062d', '\u064a']), ('\ufdab', &['\u0636', '\u062d', '\u064a']), ('\ufdac',
        &['\u0644', '\u062c', '\u064a']), ('\ufdad', &['\u0644', '\u0645', '\u064a']), ('\ufdae',
        &['\u064a', '\u062d', '\u064a']), ('\ufdaf', &['\u064a', '\u062c', '\u064a']), ('\ufdb0',
        &['\u064a', '\u0645', '\u064a']), ('\ufdb1', &['\u0645', '\u0645', '\u064a']), ('\ufdb2',
        &['\u0642', '\u0645', '\u064a']), ('\ufdb3', &['\u0646', '\u062d', '\u064a']), ('\ufdb4',
        &['\u0642', '\u0645', '\u062d']), ('\ufdb5', &['\u0644', '\u062d', '\u0645']), ('\ufdb6',
        &['\u0639', '\u0645', '\u064a']), ('\ufdb7', &['\u0643', '\u0645', '\u064a']), ('\ufdb8',
        &['\u0646', '\u062c', '\u062d']), ('\ufdb9', &['\u0645', '\u062e', '\u064a']), ('\ufdba',
        &['\u0644', '\u062c', '\u0645']), ('\ufdbb', &['\u0643', '\u0645', '\u0645']), ('\ufdbc',
        &['\u0644', '\u062c', '\u0645']), ('\ufdbd', &['\u0646', '\u062c', '\u062d']), ('\ufdbe',
        &['\u062c', '\u062d', '\u064a']), ('\ufdbf', &['\u062d', '\u062c', '\u064a']), ('\ufdc0',
        &['\u0645', '\u062c', '\u064a']), ('\ufdc1', &['\u0641', '\u0645', '\u064a']), ('\ufdc2',
        &['\u0628', '\u062d', '\u064a']), ('\ufdc3', &['\u0643', '\u0645', '\u0645']), ('\ufdc4',
        &['\u0639', '\u062c', '\u0645']), ('\ufdc5', &['\u0635', '\u0645', '\u0645']), ('\ufdc6',
        &['\u0633', '\u062e', '\u064a']), ('\ufdc7', &['\u0646', '\u062c', '\u064a']), ('\ufdf0',
        &['\u0635', '\u0644', '\u06d2']), ('\ufdf1', &['\u0642', '\u0644', '\u06d2']), ('\ufdf2',
        &['\u0627', '\u0644', '\u0644', '\u0647']), ('\ufdf3', &['\u0627', '\u0643', '\u0628',
        '\u0631']), ('\ufdf4', &['\u0645', '\u062d', '\u0645', '\u062f']), ('\ufdf5', &['\u0635',
        '\u0644', '\u0639', '\u0645']), ('\ufdf6', &['\u0631', '\u0633', '\u0648', '\u0644']),
        ('\ufdf7', &['\u0639', '\u0644', '\u064a', '\u0647']), ('\ufdf8', &['\u0648', '\u0633',
        '\u0644', '\u0645']), ('\ufdf9', &['\u0635', '\u0644', '\u0649']), ('\ufdfa', &['\u0635',
        '\u0644', '\u0649', '\x20', '\u0627', '\u0644', '\u0644', '\u0647', '\x20', '\u0639',
        '\u0644', '\u064a', '\u0647', '\x20', '\u0648', '\u0633', '\u0644', '\u0645']), ('\ufdfb',
        &['\u062c', '\u0644', '\x20', '\u062c', '\u0644', '\u0627', '\u0644', '\u0647']), ('\ufdfc',
        &['\u0631', '\u06cc', '\u0627', '\u0644']), ('\ufe10', &['\x2c']), ('\ufe11', &['\u3001']),
        ('\ufe12', &['\u3002']), ('\ufe13', &['\x3a']), ('\ufe14', &['\x3b']), ('\ufe15',
        &['\x21']), ('\ufe16', &['\x3f']), ('\ufe17', &['\u3016']), ('\ufe18', &['\u3017']),
        ('\ufe19', &['\u2026']), ('\ufe30', &['\u2025']), ('\ufe31', &['\u2014']), ('\ufe32',
        &['\u2013']), ('\ufe33', &['\x5f']), ('\ufe34', &['\x5f']), ('\ufe35', &['\x28']),
        ('\ufe36', &['\x29']), ('\ufe37', &['\x7b']), ('\ufe38', &['\x7d']), ('\ufe39',
        &['\u3014']), ('\ufe3a', &['\u3015']), ('\ufe3b', &['\u3010']), ('\ufe3c', &['\u3011']),
        ('\ufe3d', &['\u300a']), ('\ufe3e', &['\u300b']), ('\ufe3f', &['\u3008']), ('\ufe40',
        &['\u3009']), ('\ufe41', &['\u300c']), ('\ufe42', &['\u300d']), ('\ufe43', &['\u300e']),
        ('\ufe44', &['\u300f']), ('\ufe47', &['\x5b']), ('\ufe48', &['\x5d']), ('\ufe49',
        &['\u203e']), ('\ufe4a', &['\u203e']), ('\ufe4b', &['\u203e']), ('\ufe4c', &['\u203e']),
        ('\ufe4d', &['\x5f']), ('\ufe4e', &['\x5f']), ('\ufe4f', &['\x5f']), ('\ufe50', &['\x2c']),
        ('\ufe51', &['\u3001']), ('\ufe52', &['\x2e']), ('\ufe54', &['\x3b']), ('\ufe55',
        &['\x3a']), ('\ufe56', &['\x3f']), ('\ufe57', &['\x21']), ('\ufe58', &['\u2014']),
        ('\ufe59', &['\x28']), ('\ufe5a', &['\x29']), ('\ufe5b', &['\x7b']), ('\ufe5c', &['\x7d']),
        ('\ufe5d', &['\u3014']), ('\ufe5e', &['\u3015']), ('\ufe5f', &['\x23']), ('\ufe60',
        &['\x26']), ('\ufe61', &['\x2a']), ('\ufe62', &['\x2b']), ('\ufe63', &['\x2d']), ('\ufe64',
        &['\x3c']), ('\ufe65', &['\x3e']), ('\ufe66', &['\x3d']), ('\ufe68', &['\x5c']), ('\ufe69',
        &['\x24']), ('\ufe6a', &['\x25']), ('\ufe6b', &['\x40']), ('\ufe70', &['\x20', '\u064b']),
        ('\ufe71', &['\u0640', '\u064b']), ('\ufe72', &['\x20', '\u064c']), ('\ufe74', &['\x20',
        '\u064d']), ('\ufe76', &['\x20', '\u064e']), ('\ufe77', &['\u0640', '\u064e']), ('\ufe78',
        &['\x20', '\u064f']), ('\ufe79', &['\u0640', '\u064f']), ('\ufe7a', &['\x20', '\u0650']),
        ('\ufe7b', &['\u0640', '\u0650']), ('\ufe7c', &['\x20', '\u0651']), ('\ufe7d', &['\u0640',
        '\u0651']), ('\ufe7e', &['\x20', '\u0652']), ('\ufe7f', &['\u0640', '\u0652']), ('\ufe80',
        &['\u0621']), ('\ufe81', &['\u0622']), ('\ufe82', &['\u0622']), ('\ufe83', &['\u0623']),
        ('\ufe84', &['\u0623']), ('\ufe85', &['\u0624']), ('\ufe86', &['\u0624']), ('\ufe87',
        &['\u0625']), ('\ufe88', &['\u0625']), ('\ufe89', &['\u0626']), ('\ufe8a', &['\u0626']),
        ('\ufe8b', &['\u0626']), ('\ufe8c', &['\u0626']), ('\ufe8d', &['\u0627']), ('\ufe8e',
        &['\u0627']), ('\ufe8f', &['\u0628']), ('\ufe90', &['\u0628']), ('\ufe91', &['\u0628']),
        ('\ufe92', &['\u0628']), ('\ufe93', &['\u0629']), ('\ufe94', &['\u0629']), ('\ufe95',
        &['\u062a']), ('\ufe96', &['\u062a']), ('\ufe97', &['\u062a']), ('\ufe98', &['\u062a']),
        ('\ufe99', &['\u062b']), ('\ufe9a', &['\u062b']), ('\ufe9b', &['\u062b']), ('\ufe9c',
        &['\u062b']), ('\ufe9d', &['\u062c']), ('\ufe9e', &['\u062c']), ('\ufe9f', &['\u062c']),
        ('\ufea0', &['\u062c']), ('\ufea1', &['\u062d']), ('\ufea2', &['\u062d']), ('\ufea3',
        &['\u062d']), ('\ufea4', &['\u062d']), ('\ufea5', &['\u062e']), ('\ufea6', &['\u062e']),
        ('\ufea7', &['\u062e']), ('\ufea8', &['\u062e']), ('\ufea9', &['\u062f']), ('\ufeaa',
        &['\u062f']), ('\ufeab', &['\u0630']), ('\ufeac', &['\u0630']), ('\ufead', &['\u0631']),
        ('\ufeae', &['\u0631']), ('\ufeaf', &['\u0632']), ('\ufeb0', &['\u0632']), ('\ufeb1',
        &['\u0633']), ('\ufeb2', &['\u0633']), ('\ufeb3', &['\u0633']), ('\ufeb4', &['\u0633']),
        ('\ufeb5', &['\u0634']), ('\ufeb6', &['\u0634']), ('\ufeb7', &['\u0634']), ('\ufeb8',
        &['\u0634']), ('\ufeb9', &['\u0635']), ('\ufeba', &['\u0635']), ('\ufebb', &['\u0635']),
        ('\ufebc', &['\u0635']), ('\ufebd', &['\u0636']), ('\ufebe', &['\u0636']), ('\ufebf',
        &['\u0636']), ('\ufec0', &['\u0636']), ('\ufec1', &['\u0637']), ('\ufec2', &['\u0637']),
        ('\ufec3', &['\u0637']), ('\ufec4', &['\u0637']), ('\ufec5', &['\u0638']), ('\ufec6',
        &['\u0638']), ('\ufec7', &['\u0638']), ('\ufec8', &['\u0638']), ('\ufec9', &['\u0639']),
        ('\ufeca', &['\u0639']), ('\ufecb', &['\u0639']), ('\ufecc', &['\u0639']), ('\ufecd',
        &['\u063a']), ('\ufece', &['\u063a']), ('\ufecf', &['\u063a']), ('\ufed0', &['\u063a']),
        ('\ufed1', &['\u0641']), ('\ufed2', &['\u0641']), ('\ufed3', &['\u0641']), ('\ufed4',
        &['\u0641']), ('\ufed5', &['\u0642']), ('\ufed6', &['\u0642']), ('\ufed7', &['\u0642']),
        ('\ufed8', &['\u0642']), ('\ufed9', &['\u0643']), ('\ufeda', &['\u0643']), ('\ufedb',
        &['\u0643']), ('\ufedc', &['\u0643']), ('\ufedd', &['\u0644']), ('\ufede', &['\u0644']),
        ('\ufedf', &['\u0644']), ('\ufee0', &['\u0644']), ('\ufee1', &['\u0645']), ('\ufee2',
        &['\u0645']), ('\ufee3', &['\u0645']), ('\ufee4', &['\u0645']), ('\ufee5', &['\u0646']),
        ('\ufee6', &['\u0646']), ('\ufee7', &['\u0646']), ('\ufee8', &['\u0646']), ('\ufee9',
        &['\u0647']), ('\ufeea', &['\u0647']), ('\ufeeb', &['\u0647']), ('\ufeec', &['\u0647']),
        ('\ufeed', &['\u0648']), ('\ufeee', &['\u0648']), ('\ufeef', &['\u0649']), ('\ufef0',
        &['\u0649']), ('\ufef1', &['\u064a']), ('\ufef2', &['\u064a']), ('\ufef3', &['\u064a']),
        ('\ufef4', &['\u064a']), ('\ufef5', &['\u0644', '\u0622']), ('\ufef6', &['\u0644',
        '\u0622']), ('\ufef7', &['\u0644', '\u0623']), ('\ufef8', &['\u0644', '\u0623']), ('\ufef9',
        &['\u0644', '\u0625']), ('\ufefa', &['\u0644', '\u0625']), ('\ufefb', &['\u0644',
        '\u0627']), ('\ufefc', &['\u0644', '\u0627']), ('\uff01', &['\x21']), ('\uff02', &['\x22']),
        ('\uff03', &['\x23']), ('\uff04', &['\x24']), ('\uff05', &['\x25']), ('\uff06', &['\x26']),
        ('\uff07', &['\x27']), ('\uff08', &['\x28']), ('\uff09', &['\x29']), ('\uff0a', &['\x2a']),
        ('\uff0b', &['\x2b']), ('\uff0c', &['\x2c']), ('\uff0d', &['\x2d']), ('\uff0e', &['\x2e']),
        ('\uff0f', &['\x2f']), ('\uff10', &['\x30']), ('\uff11', &['\x31']), ('\uff12', &['\x32']),
        ('\uff13', &['\x33']), ('\uff14', &['\x34']), ('\uff15', &['\x35']), ('\uff16', &['\x36']),
        ('\uff17', &['\x37']), ('\uff18', &['\x38']), ('\uff19', &['\x39']), ('\uff1a', &['\x3a']),
        ('\uff1b', &['\x3b']), ('\uff1c', &['\x3c']), ('\uff1d', &['\x3d']), ('\uff1e', &['\x3e']),
        ('\uff1f', &['\x3f']), ('\uff20', &['\x40']), ('\uff21', &['\x41']), ('\uff22', &['\x42']),
        ('\uff23', &['\x43']), ('\uff24', &['\x44']), ('\uff25', &['\x45']), ('\uff26', &['\x46']),
        ('\uff27', &['\x47']), ('\uff28', &['\x48']), ('\uff29', &['\x49']), ('\uff2a', &['\x4a']),
        ('\uff2b', &['\x4b']), ('\uff2c', &['\x4c']), ('\uff2d', &['\x4d']), ('\uff2e', &['\x4e']),
        ('\uff2f', &['\x4f']), ('\uff30', &['\x50']), ('\uff31', &['\x51']), ('\uff32', &['\x52']),
        ('\uff33', &['\x53']), ('\uff34', &['\x54']), ('\uff35', &['\x55']), ('\uff36', &['\x56']),
        ('\uff37', &['\x57']), ('\uff38', &['\x58']), ('\uff39', &['\x59']), ('\uff3a', &['\x5a']),
        ('\uff3b', &['\x5b']), ('\uff3c', &['\x5c']), ('\uff3d', &['\x5d']), ('\uff3e', &['\x5e']),
        ('\uff3f', &['\x5f']), ('\uff40', &['\x60']), ('\uff41', &['\x61']), ('\uff42', &['\x62']),
        ('\uff43', &['\x63']), ('\uff44', &['\x64']), ('\uff45', &['\x65']), ('\uff46', &['\x66']),
        ('\uff47', &['\x67']), ('\uff48', &['\x68']), ('\uff49', &['\x69']), ('\uff4a', &['\x6a']),
        ('\uff4b', &['\x6b']), ('\uff4c', &['\x6c']), ('\uff4d', &['\x6d']), ('\uff4e', &['\x6e']),
        ('\uff4f', &['\x6f']), ('\uff50', &['\x70']), ('\uff51', &['\x71']), ('\uff52', &['\x72']),
        ('\uff53', &['\x73']), ('\uff54', &['\x74']), ('\uff55', &['\x75']), ('\uff56', &['\x76']),
        ('\uff57', &['\x77']), ('\uff58', &['\x78']), ('\uff59', &['\x79']), ('\uff5a', &['\x7a']),
        ('\uff5b', &['\x7b']), ('\uff5c', &['\x7c']), ('\uff5d', &['\x7d']), ('\uff5e', &['\x7e']),
        ('\uff5f', &['\u2985']), ('\uff60', &['\u2986']), ('\uff61', &['\u3002']), ('\uff62',
        &['\u300c']), ('\uff63', &['\u300d']), ('\uff64', &['\u3001']), ('\uff65', &['\u30fb']),
        ('\uff66', &['\u30f2']), ('\uff67', &['\u30a1']), ('\uff68', &['\u30a3']), ('\uff69',
        &['\u30a5']), ('\uff6a', &['\u30a7']), ('\uff6b', &['\u30a9']), ('\uff6c', &['\u30e3']),
        ('\uff6d', &['\u30e5']), ('\uff6e', &['\u30e7']), ('\uff6f', &['\u30c3']), ('\uff70',
        &['\u30fc']), ('\uff71', &['\u30a2']), ('\uff72', &['\u30a4']), ('\uff73', &['\u30a6']),
        ('\uff74', &['\u30a8']), ('\uff75', &['\u30aa']), ('\uff76', &['\u30ab']), ('\uff77',
        &['\u30ad']), ('\uff78', &['\u30af']), ('\uff79', &['\u30b1']), ('\uff7a', &['\u30b3']),
        ('\uff7b', &['\u30b5']), ('\uff7c', &['\u30b7']), ('\uff7d', &['\u30b9']), ('\uff7e',
        &['\u30bb']), ('\uff7f', &['\u30bd']), ('\uff80', &['\u30bf']), ('\uff81', &['\u30c1']),
        ('\uff82', &['\u30c4']), ('\uff83', &['\u30c6']), ('\uff84', &['\u30c8']), ('\uff85',
        &['\u30ca']), ('\uff86', &['\u30cb']), ('\uff87', &['\u30cc']), ('\uff88', &['\u30cd']),
        ('\uff89', &['\u30ce']), ('\uff8a', &['\u30cf']), ('\uff8b', &['\u30d2']), ('\uff8c',
        &['\u30d5']), ('\uff8d', &['\u30d8']), ('\uff8e', &['\u30db']), ('\uff8f', &['\u30de']),
        ('\uff90', &['\u30df']), ('\uff91', &['\u30e0']), ('\uff92', &['\u30e1']), ('\uff93',
        &['\u30e2']), ('\uff94', &['\u30e4']), ('\uff95', &['\u30e6']), ('\uff96', &['\u30e8']),
        ('\uff97', &['\u30e9']), ('\uff98', &['\u30ea']), ('\uff99', &['\u30eb']), ('\uff9a',
        &['\u30ec']), ('\uff9b', &['\u30ed']), ('\uff9c', &['\u30ef']), ('\uff9d', &['\u30f3']),
        ('\uff9e', &['\u3099']), ('\uff9f', &['\u309a']), ('\uffa0', &['\u3164']), ('\uffa1',
        &['\u3131']), ('\uffa2', &['\u3132']), ('\uffa3', &['\u3133']), ('\uffa4', &['\u3134']),
        ('\uffa5', &['\u3135']), ('\uffa6', &['\u3136']), ('\uffa7', &['\u3137']), ('\uffa8',
        &['\u3138']), ('\uffa9', &['\u3139']), ('\uffaa', &['\u313a']), ('\uffab', &['\u313b']),
        ('\uffac', &['\u313c']), ('\uffad', &['\u313d']), ('\uffae', &['\u313e']), ('\uffaf',
        &['\u313f']), ('\uffb0', &['\u3140']), ('\uffb1', &['\u3141']), ('\uffb2', &['\u3142']),
        ('\uffb3', &['\u3143']), ('\uffb4', &['\u3144']), ('\uffb5', &['\u3145']), ('\uffb6',
        &['\u3146']), ('\uffb7', &['\u3147']), ('\uffb8', &['\u3148']), ('\uffb9', &['\u3149']),
        ('\uffba', &['\u314a']), ('\uffbb', &['\u314b']), ('\uffbc', &['\u314c']), ('\uffbd',
        &['\u314d']), ('\uffbe', &['\u314e']), ('\uffc2', &['\u314f']), ('\uffc3', &['\u3150']),
        ('\uffc4', &['\u3151']), ('\uffc5', &['\u3152']), ('\uffc6', &['\u3153']), ('\uffc7',
        &['\u3154']), ('\uffca', &['\u3155']), ('\uffcb', &['\u3156']), ('\uffcc', &['\u3157']),
        ('\uffcd', &['\u3158']), ('\uffce', &['\u3159']), ('\uffcf', &['\u315a']), ('\uffd2',
        &['\u315b']), ('\uffd3', &['\u315c']), ('\uffd4', &['\u315d']), ('\uffd5', &['\u315e']),
        ('\uffd6', &['\u315f']), ('\uffd7', &['\u3160']), ('\uffda', &['\u3161']), ('\uffdb',
        &['\u3162']), ('\uffdc', &['\u3163']), ('\uffe0', &['\xa2']), ('\uffe1', &['\xa3']),
        ('\uffe2', &['\xac']), ('\uffe3', &['\xaf']), ('\uffe4', &['\xa6']), ('\uffe5', &['\xa5']),
        ('\uffe6', &['\u20a9']), ('\uffe8', &['\u2502']), ('\uffe9', &['\u2190']), ('\uffea',
        &['\u2191']), ('\uffeb', &['\u2192']), ('\uffec', &['\u2193']), ('\uffed', &['\u25a0']),
        ('\uffee', &['\u25cb']), ('\U0001d400', &['\x41']), ('\U0001d401', &['\x42']),
        ('\U0001d402', &['\x43']), ('\U0001d403', &['\x44']), ('\U0001d404', &['\x45']),
        ('\U0001d405', &['\x46']), ('\U0001d406', &['\x47']), ('\U0001d407', &['\x48']),
        ('\U0001d408', &['\x49']), ('\U0001d409', &['\x4a']), ('\U0001d40a', &['\x4b']),
        ('\U0001d40b', &['\x4c']), ('\U0001d40c', &['\x4d']), ('\U0001d40d', &['\x4e']),
        ('\U0001d40e', &['\x4f']), ('\U0001d40f', &['\x50']), ('\U0001d410', &['\x51']),
        ('\U0001d411', &['\x52']), ('\U0001d412', &['\x53']), ('\U0001d413', &['\x54']),
        ('\U0001d414', &['\x55']), ('\U0001d415', &['\x56']), ('\U0001d416', &['\x57']),
        ('\U0001d417', &['\x58']), ('\U0001d418', &['\x59']), ('\U0001d419', &['\x5a']),
        ('\U0001d41a', &['\x61']), ('\U0001d41b', &['\x62']), ('\U0001d41c', &['\x63']),
        ('\U0001d41d', &['\x64']), ('\U0001d41e', &['\x65']), ('\U0001d41f', &['\x66']),
        ('\U0001d420', &['\x67']), ('\U0001d421', &['\x68']), ('\U0001d422', &['\x69']),
        ('\U0001d423', &['\x6a']), ('\U0001d424', &['\x6b']), ('\U0001d425', &['\x6c']),
        ('\U0001d426', &['\x6d']), ('\U0001d427', &['\x6e']), ('\U0001d428', &['\x6f']),
        ('\U0001d429', &['\x70']), ('\U0001d42a', &['\x71']), ('\U0001d42b', &['\x72']),
        ('\U0001d42c', &['\x73']), ('\U0001d42d', &['\x74']), ('\U0001d42e', &['\x75']),
        ('\U0001d42f', &['\x76']), ('\U0001d430', &['\x77']), ('\U0001d431', &['\x78']),
        ('\U0001d432', &['\x79']), ('\U0001d433', &['\x7a']), ('\U0001d434', &['\x41']),
        ('\U0001d435', &['\x42']), ('\U0001d436', &['\x43']), ('\U0001d437', &['\x44']),
        ('\U0001d438', &['\x45']), ('\U0001d439', &['\x46']), ('\U0001d43a', &['\x47']),
        ('\U0001d43b', &['\x48']), ('\U0001d43c', &['\x49']), ('\U0001d43d', &['\x4a']),
        ('\U0001d43e', &['\x4b']), ('\U0001d43f', &['\x4c']), ('\U0001d440', &['\x4d']),
        ('\U0001d441', &['\x4e']), ('\U0001d442', &['\x4f']), ('\U0001d443', &['\x50']),
        ('\U0001d444', &['\x51']), ('\U0001d445', &['\x52']), ('\U0001d446', &['\x53']),
        ('\U0001d447', &['\x54']), ('\U0001d448', &['\x55']), ('\U0001d449', &['\x56']),
        ('\U0001d44a', &['\x57']), ('\U0001d44b', &['\x58']), ('\U0001d44c', &['\x59']),
        ('\U0001d44d', &['\x5a']), ('\U0001d44e', &['\x61']), ('\U0001d44f', &['\x62']),
        ('\U0001d450', &['\x63']), ('\U0001d451', &['\x64']), ('\U0001d452', &['\x65']),
        ('\U0001d453', &['\x66']), ('\U0001d454', &['\x67']), ('\U0001d456', &['\x69']),
        ('\U0001d457', &['\x6a']), ('\U0001d458', &['\x6b']), ('\U0001d459', &['\x6c']),
        ('\U0001d45a', &['\x6d']), ('\U0001d45b', &['\x6e']), ('\U0001d45c', &['\x6f']),
        ('\U0001d45d', &['\x70']), ('\U0001d45e', &['\x71']), ('\U0001d45f', &['\x72']),
        ('\U0001d460', &['\x73']), ('\U0001d461', &['\x74']), ('\U0001d462', &['\x75']),
        ('\U0001d463', &['\x76']), ('\U0001d464', &['\x77']), ('\U0001d465', &['\x78']),
        ('\U0001d466', &['\x79']), ('\U0001d467', &['\x7a']), ('\U0001d468', &['\x41']),
        ('\U0001d469', &['\x42']), ('\U0001d46a', &['\x43']), ('\U0001d46b', &['\x44']),
        ('\U0001d46c', &['\x45']), ('\U0001d46d', &['\x46']), ('\U0001d46e', &['\x47']),
        ('\U0001d46f', &['\x48']), ('\U0001d470', &['\x49']), ('\U0001d471', &['\x4a']),
        ('\U0001d472', &['\x4b']), ('\U0001d473', &['\x4c']), ('\U0001d474', &['\x4d']),
        ('\U0001d475', &['\x4e']), ('\U0001d476', &['\x4f']), ('\U0001d477', &['\x50']),
        ('\U0001d478', &['\x51']), ('\U0001d479', &['\x52']), ('\U0001d47a', &['\x53']),
        ('\U0001d47b', &['\x54']), ('\U0001d47c', &['\x55']), ('\U0001d47d', &['\x56']),
        ('\U0001d47e', &['\x57']), ('\U0001d47f', &['\x58']), ('\U0001d480', &['\x59']),
        ('\U0001d481', &['\x5a']), ('\U0001d482', &['\x61']), ('\U0001d483', &['\x62']),
        ('\U0001d484', &['\x63']), ('\U0001d485', &['\x64']), ('\U0001d486', &['\x65']),
        ('\U0001d487', &['\x66']), ('\U0001d488', &['\x67']), ('\U0001d489', &['\x68']),
        ('\U0001d48a', &['\x69']), ('\U0001d48b', &['\x6a']), ('\U0001d48c', &['\x6b']),
        ('\U0001d48d', &['\x6c']), ('\U0001d48e', &['\x6d']), ('\U0001d48f', &['\x6e']),
        ('\U0001d490', &['\x6f']), ('\U0001d491', &['\x70']), ('\U0001d492', &['\x71']),
        ('\U0001d493', &['\x72']), ('\U0001d494', &['\x73']), ('\U0001d495', &['\x74']),
        ('\U0001d496', &['\x75']), ('\U0001d497', &['\x76']), ('\U0001d498', &['\x77']),
        ('\U0001d499', &['\x78']), ('\U0001d49a', &['\x79']), ('\U0001d49b', &['\x7a']),
        ('\U0001d49c', &['\x41']), ('\U0001d49e', &['\x43']), ('\U0001d49f', &['\x44']),
        ('\U0001d4a2', &['\x47']), ('\U0001d4a5', &['\x4a']), ('\U0001d4a6', &['\x4b']),
        ('\U0001d4a9', &['\x4e']), ('\U0001d4aa', &['\x4f']), ('\U0001d4ab', &['\x50']),
        ('\U0001d4ac', &['\x51']), ('\U0001d4ae', &['\x53']), ('\U0001d4af', &['\x54']),
        ('\U0001d4b0', &['\x55']), ('\U0001d4b1', &['\x56']), ('\U0001d4b2', &['\x57']),
        ('\U0001d4b3', &['\x58']), ('\U0001d4b4', &['\x59']), ('\U0001d4b5', &['\x5a']),
        ('\U0001d4b6', &['\x61']), ('\U0001d4b7', &['\x62']), ('\U0001d4b8', &['\x63']),
        ('\U0001d4b9', &['\x64']), ('\U0001d4bb', &['\x66']), ('\U0001d4bd', &['\x68']),
        ('\U0001d4be', &['\x69']), ('\U0001d4bf', &['\x6a']), ('\U0001d4c0', &['\x6b']),
        ('\U0001d4c1', &['\x6c']), ('\U0001d4c2', &['\x6d']), ('\U0001d4c3', &['\x6e']),
        ('\U0001d4c5', &['\x70']), ('\U0001d4c6', &['\x71']), ('\U0001d4c7', &['\x72']),
        ('\U0001d4c8', &['\x73']), ('\U0001d4c9', &['\x74']), ('\U0001d4ca', &['\x75']),
        ('\U0001d4cb', &['\x76']), ('\U0001d4cc', &['\x77']), ('\U0001d4cd', &['\x78']),
        ('\U0001d4ce', &['\x79']), ('\U0001d4cf', &['\x7a']), ('\U0001d4d0', &['\x41']),
        ('\U0001d4d1', &['\x42']), ('\U0001d4d2', &['\x43']), ('\U0001d4d3', &['\x44']),
        ('\U0001d4d4', &['\x45']), ('\U0001d4d5', &['\x46']), ('\U0001d4d6', &['\x47']),
        ('\U0001d4d7', &['\x48']), ('\U0001d4d8', &['\x49']), ('\U0001d4d9', &['\x4a']),
        ('\U0001d4da', &['\x4b']), ('\U0001d4db', &['\x4c']), ('\U0001d4dc', &['\x4d']),
        ('\U0001d4dd', &['\x4e']), ('\U0001d4de', &['\x4f']), ('\U0001d4df', &['\x50']),
        ('\U0001d4e0', &['\x51']), ('\U0001d4e1', &['\x52']), ('\U0001d4e2', &['\x53']),
        ('\U0001d4e3', &['\x54']), ('\U0001d4e4', &['\x55']), ('\U0001d4e5', &['\x56']),
        ('\U0001d4e6', &['\x57']), ('\U0001d4e7', &['\x58']), ('\U0001d4e8', &['\x59']),
        ('\U0001d4e9', &['\x5a']), ('\U0001d4ea', &['\x61']), ('\U0001d4eb', &['\x62']),
        ('\U0001d4ec', &['\x63']), ('\U0001d4ed', &['\x64']), ('\U0001d4ee', &['\x65']),
        ('\U0001d4ef', &['\x66']), ('\U0001d4f0', &['\x67']), ('\U0001d4f1', &['\x68']),
        ('\U0001d4f2', &['\x69']), ('\U0001d4f3', &['\x6a']), ('\U0001d4f4', &['\x6b']),
        ('\U0001d4f5', &['\x6c']), ('\U0001d4f6', &['\x6d']), ('\U0001d4f7', &['\x6e']),
        ('\U0001d4f8', &['\x6f']), ('\U0001d4f9', &['\x70']), ('\U0001d4fa', &['\x71']),
        ('\U0001d4fb', &['\x72']), ('\U0001d4fc', &['\x73']), ('\U0001d4fd', &['\x74']),
        ('\U0001d4fe', &['\x75']), ('\U0001d4ff', &['\x76']), ('\U0001d500', &['\x77']),
        ('\U0001d501', &['\x78']), ('\U0001d502', &['\x79']), ('\U0001d503', &['\x7a']),
        ('\U0001d504', &['\x41']), ('\U0001d505', &['\x42']), ('\U0001d507', &['\x44']),
        ('\U0001d508', &['\x45']), ('\U0001d509', &['\x46']), ('\U0001d50a', &['\x47']),
        ('\U0001d50d', &['\x4a']), ('\U0001d50e', &['\x4b']), ('\U0001d50f', &['\x4c']),
        ('\U0001d510', &['\x4d']), ('\U0001d511', &['\x4e']), ('\U0001d512', &['\x4f']),
        ('\U0001d513', &['\x50']), ('\U0001d514', &['\x51']), ('\U0001d516', &['\x53']),
        ('\U0001d517', &['\x54']), ('\U0001d518', &['\x55']), ('\U0001d519', &['\x56']),
        ('\U0001d51a', &['\x57']), ('\U0001d51b', &['\x58']), ('\U0001d51c', &['\x59']),
        ('\U0001d51e', &['\x61']), ('\U0001d51f', &['\x62']), ('\U0001d520', &['\x63']),
        ('\U0001d521', &['\x64']), ('\U0001d522', &['\x65']), ('\U0001d523', &['\x66']),
        ('\U0001d524', &['\x67']), ('\U0001d525', &['\x68']), ('\U0001d526', &['\x69']),
        ('\U0001d527', &['\x6a']), ('\U0001d528', &['\x6b']), ('\U0001d529', &['\x6c']),
        ('\U0001d52a', &['\x6d']), ('\U0001d52b', &['\x6e']), ('\U0001d52c', &['\x6f']),
        ('\U0001d52d', &['\x70']), ('\U0001d52e', &['\x71']), ('\U0001d52f', &['\x72']),
        ('\U0001d530', &['\x73']), ('\U0001d531', &['\x74']), ('\U0001d532', &['\x75']),
        ('\U0001d533', &['\x76']), ('\U0001d534', &['\x77']), ('\U0001d535', &['\x78']),
        ('\U0001d536', &['\x79']), ('\U0001d537', &['\x7a']), ('\U0001d538', &['\x41']),
        ('\U0001d539', &['\x42']), ('\U0001d53b', &['\x44']), ('\U0001d53c', &['\x45']),
        ('\U0001d53d', &['\x46']), ('\U0001d53e', &['\x47']), ('\U0001d540', &['\x49']),
        ('\U0001d541', &['\x4a']), ('\U0001d542', &['\x4b']), ('\U0001d543', &['\x4c']),
        ('\U0001d544', &['\x4d']), ('\U0001d546', &['\x4f']), ('\U0001d54a', &['\x53']),
        ('\U0001d54b', &['\x54']), ('\U0001d54c', &['\x55']), ('\U0001d54d', &['\x56']),
        ('\U0001d54e', &['\x57']), ('\U0001d54f', &['\x58']), ('\U0001d550', &['\x59']),
        ('\U0001d552', &['\x61']), ('\U0001d553', &['\x62']), ('\U0001d554', &['\x63']),
        ('\U0001d555', &['\x64']), ('\U0001d556', &['\x65']), ('\U0001d557', &['\x66']),
        ('\U0001d558', &['\x67']), ('\U0001d559', &['\x68']), ('\U0001d55a', &['\x69']),
        ('\U0001d55b', &['\x6a']), ('\U0001d55c', &['\x6b']), ('\U0001d55d', &['\x6c']),
        ('\U0001d55e', &['\x6d']), ('\U0001d55f', &['\x6e']), ('\U0001d560', &['\x6f']),
        ('\U0001d561', &['\x70']), ('\U0001d562', &['\x71']), ('\U0001d563', &['\x72']),
        ('\U0001d564', &['\x73']), ('\U0001d565', &['\x74']), ('\U0001d566', &['\x75']),
        ('\U0001d567', &['\x76']), ('\U0001d568', &['\x77']), ('\U0001d569', &['\x78']),
        ('\U0001d56a', &['\x79']), ('\U0001d56b', &['\x7a']), ('\U0001d56c', &['\x41']),
        ('\U0001d56d', &['\x42']), ('\U0001d56e', &['\x43']), ('\U0001d56f', &['\x44']),
        ('\U0001d570', &['\x45']), ('\U0001d571', &['\x46']), ('\U0001d572', &['\x47']),
        ('\U0001d573', &['\x48']), ('\U0001d574', &['\x49']), ('\U0001d575', &['\x4a']),
        ('\U0001d576', &['\x4b']), ('\U0001d577', &['\x4c']), ('\U0001d578', &['\x4d']),
        ('\U0001d579', &['\x4e']), ('\U0001d57a', &['\x4f']), ('\U0001d57b', &['\x50']),
        ('\U0001d57c', &['\x51']), ('\U0001d57d', &['\x52']), ('\U0001d57e', &['\x53']),
        ('\U0001d57f', &['\x54']), ('\U0001d580', &['\x55']), ('\U0001d581', &['\x56']),
        ('\U0001d582', &['\x57']), ('\U0001d583', &['\x58']), ('\U0001d584', &['\x59']),
        ('\U0001d585', &['\x5a']), ('\U0001d586', &['\x61']), ('\U0001d587', &['\x62']),
        ('\U0001d588', &['\x63']), ('\U0001d589', &['\x64']), ('\U0001d58a', &['\x65']),
        ('\U0001d58b', &['\x66']), ('\U0001d58c', &['\x67']), ('\U0001d58d', &['\x68']),
        ('\U0001d58e', &['\x69']), ('\U0001d58f', &['\x6a']), ('\U0001d590', &['\x6b']),
        ('\U0001d591', &['\x6c']), ('\U0001d592', &['\x6d']), ('\U0001d593', &['\x6e']),
        ('\U0001d594', &['\x6f']), ('\U0001d595', &['\x70']), ('\U0001d596', &['\x71']),
        ('\U0001d597', &['\x72']), ('\U0001d598', &['\x73']), ('\U0001d599', &['\x74']),
        ('\U0001d59a', &['\x75']), ('\U0001d59b', &['\x76']), ('\U0001d59c', &['\x77']),
        ('\U0001d59d', &['\x78']), ('\U0001d59e', &['\x79']), ('\U0001d59f', &['\x7a']),
        ('\U0001d5a0', &['\x41']), ('\U0001d5a1', &['\x42']), ('\U0001d5a2', &['\x43']),
        ('\U0001d5a3', &['\x44']), ('\U0001d5a4', &['\x45']), ('\U0001d5a5', &['\x46']),
        ('\U0001d5a6', &['\x47']), ('\U0001d5a7', &['\x48']), ('\U0001d5a8', &['\x49']),
        ('\U0001d5a9', &['\x4a']), ('\U0001d5aa', &['\x4b']), ('\U0001d5ab', &['\x4c']),
        ('\U0001d5ac', &['\x4d']), ('\U0001d5ad', &['\x4e']), ('\U0001d5ae', &['\x4f']),
        ('\U0001d5af', &['\x50']), ('\U0001d5b0', &['\x51']), ('\U0001d5b1', &['\x52']),
        ('\U0001d5b2', &['\x53']), ('\U0001d5b3', &['\x54']), ('\U0001d5b4', &['\x55']),
        ('\U0001d5b5', &['\x56']), ('\U0001d5b6', &['\x57']), ('\U0001d5b7', &['\x58']),
        ('\U0001d5b8', &['\x59']), ('\U0001d5b9', &['\x5a']), ('\U0001d5ba', &['\x61']),
        ('\U0001d5bb', &['\x62']), ('\U0001d5bc', &['\x63']), ('\U0001d5bd', &['\x64']),
        ('\U0001d5be', &['\x65']), ('\U0001d5bf', &['\x66']), ('\U0001d5c0', &['\x67']),
        ('\U0001d5c1', &['\x68']), ('\U0001d5c2', &['\x69']), ('\U0001d5c3', &['\x6a']),
        ('\U0001d5c4', &['\x6b']), ('\U0001d5c5', &['\x6c']), ('\U0001d5c6', &['\x6d']),
        ('\U0001d5c7', &['\x6e']), ('\U0001d5c8', &['\x6f']), ('\U0001d5c9', &['\x70']),
        ('\U0001d5ca', &['\x71']), ('\U0001d5cb', &['\x72']), ('\U0001d5cc', &['\x73']),
        ('\U0001d5cd', &['\x74']), ('\U0001d5ce', &['\x75']), ('\U0001d5cf', &['\x76']),
        ('\U0001d5d0', &['\x77']), ('\U0001d5d1', &['\x78']), ('\U0001d5d2', &['\x79']),
        ('\U0001d5d3', &['\x7a']), ('\U0001d5d4', &['\x41']), ('\U0001d5d5', &['\x42']),
        ('\U0001d5d6', &['\x43']), ('\U0001d5d7', &['\x44']), ('\U0001d5d8', &['\x45']),
        ('\U0001d5d9', &['\x46']), ('\U0001d5da', &['\x47']), ('\U0001d5db', &['\x48']),
        ('\U0001d5dc', &['\x49']), ('\U0001d5dd', &['\x4a']), ('\U0001d5de', &['\x4b']),
        ('\U0001d5df', &['\x4c']), ('\U0001d5e0', &['\x4d']), ('\U0001d5e1', &['\x4e']),
        ('\U0001d5e2', &['\x4f']), ('\U0001d5e3', &['\x50']), ('\U0001d5e4', &['\x51']),
        ('\U0001d5e5', &['\x52']), ('\U0001d5e6', &['\x53']), ('\U0001d5e7', &['\x54']),
        ('\U0001d5e8', &['\x55']), ('\U0001d5e9', &['\x56']), ('\U0001d5ea', &['\x57']),
        ('\U0001d5eb', &['\x58']), ('\U0001d5ec', &['\x59']), ('\U0001d5ed', &['\x5a']),
        ('\U0001d5ee', &['\x61']), ('\U0001d5ef', &['\x62']), ('\U0001d5f0', &['\x63']),
        ('\U0001d5f1', &['\x64']), ('\U0001d5f2', &['\x65']), ('\U0001d5f3', &['\x66']),
        ('\U0001d5f4', &['\x67']), ('\U0001d5f5', &['\x68']), ('\U0001d5f6', &['\x69']),
        ('\U0001d5f7', &['\x6a']), ('\U0001d5f8', &['\x6b']), ('\U0001d5f9', &['\x6c']),
        ('\U0001d5fa', &['\x6d']), ('\U0001d5fb', &['\x6e']), ('\U0001d5fc', &['\x6f']),
        ('\U0001d5fd', &['\x70']), ('\U0001d5fe', &['\x71']), ('\U0001d5ff', &['\x72']),
        ('\U0001d600', &['\x73']), ('\U0001d601', &['\x74']), ('\U0001d602', &['\x75']),
        ('\U0001d603', &['\x76']), ('\U0001d604', &['\x77']), ('\U0001d605', &['\x78']),
        ('\U0001d606', &['\x79']), ('\U0001d607', &['\x7a']), ('\U0001d608', &['\x41']),
        ('\U0001d609', &['\x42']), ('\U0001d60a', &['\x43']), ('\U0001d60b', &['\x44']),
        ('\U0001d60c', &['\x45']), ('\U0001d60d', &['\x46']), ('\U0001d60e', &['\x47']),
        ('\U0001d60f', &['\x48']), ('\U0001d610', &['\x49']), ('\U0001d611', &['\x4a']),
        ('\U0001d612', &['\x4b']), ('\U0001d613', &['\x4c']), ('\U0001d614', &['\x4d']),
        ('\U0001d615', &['\x4e']), ('\U0001d616', &['\x4f']), ('\U0001d617', &['\x50']),
        ('\U0001d618', &['\x51']), ('\U0001d619', &['\x52']), ('\U0001d61a', &['\x53']),
        ('\U0001d61b', &['\x54']), ('\U0001d61c', &['\x55']), ('\U0001d61d', &['\x56']),
        ('\U0001d61e', &['\x57']), ('\U0001d61f', &['\x58']), ('\U0001d620', &['\x59']),
        ('\U0001d621', &['\x5a']), ('\U0001d622', &['\x61']), ('\U0001d623', &['\x62']),
        ('\U0001d624', &['\x63']), ('\U0001d625', &['\x64']), ('\U0001d626', &['\x65']),
        ('\U0001d627', &['\x66']), ('\U0001d628', &['\x67']), ('\U0001d629', &['\x68']),
        ('\U0001d62a', &['\x69']), ('\U0001d62b', &['\x6a']), ('\U0001d62c', &['\x6b']),
        ('\U0001d62d', &['\x6c']), ('\U0001d62e', &['\x6d']), ('\U0001d62f', &['\x6e']),
        ('\U0001d630', &['\x6f']), ('\U0001d631', &['\x70']), ('\U0001d632', &['\x71']),
        ('\U0001d633', &['\x72']), ('\U0001d634', &['\x73']), ('\U0001d635', &['\x74']),
        ('\U0001d636', &['\x75']), ('\U0001d637', &['\x76']), ('\U0001d638', &['\x77']),
        ('\U0001d639', &['\x78']), ('\U0001d63a', &['\x79']), ('\U0001d63b', &['\x7a']),
        ('\U0001d63c', &['\x41']), ('\U0001d63d', &['\x42']), ('\U0001d63e', &['\x43']),
        ('\U0001d63f', &['\x44']), ('\U0001d640', &['\x45']), ('\U0001d641', &['\x46']),
        ('\U0001d642', &['\x47']), ('\U0001d643', &['\x48']), ('\U0001d644', &['\x49']),
        ('\U0001d645', &['\x4a']), ('\U0001d646', &['\x4b']), ('\U0001d647', &['\x4c']),
        ('\U0001d648', &['\x4d']), ('\U0001d649', &['\x4e']), ('\U0001d64a', &['\x4f']),
        ('\U0001d64b', &['\x50']), ('\U0001d64c', &['\x51']), ('\U0001d64d', &['\x52']),
        ('\U0001d64e', &['\x53']), ('\U0001d64f', &['\x54']), ('\U0001d650', &['\x55']),
        ('\U0001d651', &['\x56']), ('\U0001d652', &['\x57']), ('\U0001d653', &['\x58']),
        ('\U0001d654', &['\x59']), ('\U0001d655', &['\x5a']), ('\U0001d656', &['\x61']),
        ('\U0001d657', &['\x62']), ('\U0001d658', &['\x63']), ('\U0001d659', &['\x64']),
        ('\U0001d65a', &['\x65']), ('\U0001d65b', &['\x66']), ('\U0001d65c', &['\x67']),
        ('\U0001d65d', &['\x68']), ('\U0001d65e', &['\x69']), ('\U0001d65f', &['\x6a']),
        ('\U0001d660', &['\x6b']), ('\U0001d661', &['\x6c']), ('\U0001d662', &['\x6d']),
        ('\U0001d663', &['\x6e']), ('\U0001d664', &['\x6f']), ('\U0001d665', &['\x70']),
        ('\U0001d666', &['\x71']), ('\U0001d667', &['\x72']), ('\U0001d668', &['\x73']),
        ('\U0001d669', &['\x74']), ('\U0001d66a', &['\x75']), ('\U0001d66b', &['\x76']),
        ('\U0001d66c', &['\x77']), ('\U0001d66d', &['\x78']), ('\U0001d66e', &['\x79']),
        ('\U0001d66f', &['\x7a']), ('\U0001d670', &['\x41']), ('\U0001d671', &['\x42']),
        ('\U0001d672', &['\x43']), ('\U0001d673', &['\x44']), ('\U0001d674', &['\x45']),
        ('\U0001d675', &['\x46']), ('\U0001d676', &['\x47']), ('\U0001d677', &['\x48']),
        ('\U0001d678', &['\x49']), ('\U0001d679', &['\x4a']), ('\U0001d67a', &['\x4b']),
        ('\U0001d67b', &['\x4c']), ('\U0001d67c', &['\x4d']), ('\U0001d67d', &['\x4e']),
        ('\U0001d67e', &['\x4f']), ('\U0001d67f', &['\x50']), ('\U0001d680', &['\x51']),
        ('\U0001d681', &['\x52']), ('\U0001d682', &['\x53']), ('\U0001d683', &['\x54']),
        ('\U0001d684', &['\x55']), ('\U0001d685', &['\x56']), ('\U0001d686', &['\x57']),
        ('\U0001d687', &['\x58']), ('\U0001d688', &['\x59']), ('\U0001d689', &['\x5a']),
        ('\U0001d68a', &['\x61']), ('\U0001d68b', &['\x62']), ('\U0001d68c', &['\x63']),
        ('\U0001d68d', &['\x64']), ('\U0001d68e', &['\x65']), ('\U0001d68f', &['\x66']),
        ('\U0001d690', &['\x67']), ('\U0001d691', &['\x68']), ('\U0001d692', &['\x69']),
        ('\U0001d693', &['\x6a']), ('\U0001d694', &['\x6b']), ('\U0001d695', &['\x6c']),
        ('\U0001d696', &['\x6d']), ('\U0001d697', &['\x6e']), ('\U0001d698', &['\x6f']),
        ('\U0001d699', &['\x70']), ('\U0001d69a', &['\x71']), ('\U0001d69b', &['\x72']),
        ('\U0001d69c', &['\x73']), ('\U0001d69d', &['\x74']), ('\U0001d69e', &['\x75']),
        ('\U0001d69f', &['\x76']), ('\U0001d6a0', &['\x77']), ('\U0001d6a1', &['\x78']),
        ('\U0001d6a2', &['\x79']), ('\U0001d6a3', &['\x7a']), ('\U0001d6a4', &['\u0131']),
        ('\U0001d6a5', &['\u0237']), ('\U0001d6a8', &['\u0391']), ('\U0001d6a9', &['\u0392']),
        ('\U0001d6aa', &['\u0393']), ('\U0001d6ab', &['\u0394']), ('\U0001d6ac', &['\u0395']),
        ('\U0001d6ad', &['\u0396']), ('\U0001d6ae', &['\u0397']), ('\U0001d6af', &['\u0398']),
        ('\U0001d6b0', &['\u0399']), ('\U0001d6b1', &['\u039a']), ('\U0001d6b2', &['\u039b']),
        ('\U0001d6b3', &['\u039c']), ('\U0001d6b4', &['\u039d']), ('\U0001d6b5', &['\u039e']),
        ('\U0001d6b6', &['\u039f']), ('\U0001d6b7', &['\u03a0']), ('\U0001d6b8', &['\u03a1']),
        ('\U0001d6b9', &['\u03f4']), ('\U0001d6ba', &['\u03a3']), ('\U0001d6bb', &['\u03a4']),
        ('\U0001d6bc', &['\u03a5']), ('\U0001d6bd', &['\u03a6']), ('\U0001d6be', &['\u03a7']),
        ('\U0001d6bf', &['\u03a8']), ('\U0001d6c0', &['\u03a9']), ('\U0001d6c1', &['\u2207']),
        ('\U0001d6c2', &['\u03b1']), ('\U0001d6c3', &['\u03b2']), ('\U0001d6c4', &['\u03b3']),
        ('\U0001d6c5', &['\u03b4']), ('\U0001d6c6', &['\u03b5']), ('\U0001d6c7', &['\u03b6']),
        ('\U0001d6c8', &['\u03b7']), ('\U0001d6c9', &['\u03b8']), ('\U0001d6ca', &['\u03b9']),
        ('\U0001d6cb', &['\u03ba']), ('\U0001d6cc', &['\u03bb']), ('\U0001d6cd', &['\u03bc']),
        ('\U0001d6ce', &['\u03bd']), ('\U0001d6cf', &['\u03be']), ('\U0001d6d0', &['\u03bf']),
        ('\U0001d6d1', &['\u03c0']), ('\U0001d6d2', &['\u03c1']), ('\U0001d6d3', &['\u03c2']),
        ('\U0001d6d4', &['\u03c3']), ('\U0001d6d5', &['\u03c4']), ('\U0001d6d6', &['\u03c5']),
        ('\U0001d6d7', &['\u03c6']), ('\U0001d6d8', &['\u03c7']), ('\U0001d6d9', &['\u03c8']),
        ('\U0001d6da', &['\u03c9']), ('\U0001d6db', &['\u2202']), ('\U0001d6dc', &['\u03f5']),
        ('\U0001d6dd', &['\u03d1']), ('\U0001d6de', &['\u03f0']), ('\U0001d6df', &['\u03d5']),
        ('\U0001d6e0', &['\u03f1']), ('\U0001d6e1', &['\u03d6']), ('\U0001d6e2', &['\u0391']),
        ('\U0001d6e3', &['\u0392']), ('\U0001d6e4', &['\u0393']), ('\U0001d6e5', &['\u0394']),
        ('\U0001d6e6', &['\u0395']), ('\U0001d6e7', &['\u0396']), ('\U0001d6e8', &['\u0397']),
        ('\U0001d6e9', &['\u0398']), ('\U0001d6ea', &['\u0399']), ('\U0001d6eb', &['\u039a']),
        ('\U0001d6ec', &['\u039b']), ('\U0001d6ed', &['\u039c']), ('\U0001d6ee', &['\u039d']),
        ('\U0001d6ef', &['\u039e']), ('\U0001d6f0', &['\u039f']), ('\U0001d6f1', &['\u03a0']),
        ('\U0001d6f2', &['\u03a1']), ('\U0001d6f3', &['\u03f4']), ('\U0001d6f4', &['\u03a3']),
        ('\U0001d6f5', &['\u03a4']), ('\U0001d6f6', &['\u03a5']), ('\U0001d6f7', &['\u03a6']),
        ('\U0001d6f8', &['\u03a7']), ('\U0001d6f9', &['\u03a8']), ('\U0001d6fa', &['\u03a9']),
        ('\U0001d6fb', &['\u2207']), ('\U0001d6fc', &['\u03b1']), ('\U0001d6fd', &['\u03b2']),
        ('\U0001d6fe', &['\u03b3']), ('\U0001d6ff', &['\u03b4']), ('\U0001d700', &['\u03b5']),
        ('\U0001d701', &['\u03b6']), ('\U0001d702', &['\u03b7']), ('\U0001d703', &['\u03b8']),
        ('\U0001d704', &['\u03b9']), ('\U0001d705', &['\u03ba']), ('\U0001d706', &['\u03bb']),
        ('\U0001d707', &['\u03bc']), ('\U0001d708', &['\u03bd']), ('\U0001d709', &['\u03be']),
        ('\U0001d70a', &['\u03bf']), ('\U0001d70b', &['\u03c0']), ('\U0001d70c', &['\u03c1']),
        ('\U0001d70d', &['\u03c2']), ('\U0001d70e', &['\u03c3']), ('\U0001d70f', &['\u03c4']),
        ('\U0001d710', &['\u03c5']), ('\U0001d711', &['\u03c6']), ('\U0001d712', &['\u03c7']),
        ('\U0001d713', &['\u03c8']), ('\U0001d714', &['\u03c9']), ('\U0001d715', &['\u2202']),
        ('\U0001d716', &['\u03f5']), ('\U0001d717', &['\u03d1']), ('\U0001d718', &['\u03f0']),
        ('\U0001d719', &['\u03d5']), ('\U0001d71a', &['\u03f1']), ('\U0001d71b', &['\u03d6']),
        ('\U0001d71c', &['\u0391']), ('\U0001d71d', &['\u0392']), ('\U0001d71e', &['\u0393']),
        ('\U0001d71f', &['\u0394']), ('\U0001d720', &['\u0395']), ('\U0001d721', &['\u0396']),
        ('\U0001d722', &['\u0397']), ('\U0001d723', &['\u0398']), ('\U0001d724', &['\u0399']),
        ('\U0001d725', &['\u039a']), ('\U0001d726', &['\u039b']), ('\U0001d727', &['\u039c']),
        ('\U0001d728', &['\u039d']), ('\U0001d729', &['\u039e']), ('\U0001d72a', &['\u039f']),
        ('\U0001d72b', &['\u03a0']), ('\U0001d72c', &['\u03a1']), ('\U0001d72d', &['\u03f4']),
        ('\U0001d72e', &['\u03a3']), ('\U0001d72f', &['\u03a4']), ('\U0001d730', &['\u03a5']),
        ('\U0001d731', &['\u03a6']), ('\U0001d732', &['\u03a7']), ('\U0001d733', &['\u03a8']),
        ('\U0001d734', &['\u03a9']), ('\U0001d735', &['\u2207']), ('\U0001d736', &['\u03b1']),
        ('\U0001d737', &['\u03b2']), ('\U0001d738', &['\u03b3']), ('\U0001d739', &['\u03b4']),
        ('\U0001d73a', &['\u03b5']), ('\U0001d73b', &['\u03b6']), ('\U0001d73c', &['\u03b7']),
        ('\U0001d73d', &['\u03b8']), ('\U0001d73e', &['\u03b9']), ('\U0001d73f', &['\u03ba']),
        ('\U0001d740', &['\u03bb']), ('\U0001d741', &['\u03bc']), ('\U0001d742', &['\u03bd']),
        ('\U0001d743', &['\u03be']), ('\U0001d744', &['\u03bf']), ('\U0001d745', &['\u03c0']),
        ('\U0001d746', &['\u03c1']), ('\U0001d747', &['\u03c2']), ('\U0001d748', &['\u03c3']),
        ('\U0001d749', &['\u03c4']), ('\U0001d74a', &['\u03c5']), ('\U0001d74b', &['\u03c6']),
        ('\U0001d74c', &['\u03c7']), ('\U0001d74d', &['\u03c8']), ('\U0001d74e', &['\u03c9']),
        ('\U0001d74f', &['\u2202']), ('\U0001d750', &['\u03f5']), ('\U0001d751', &['\u03d1']),
        ('\U0001d752', &['\u03f0']), ('\U0001d753', &['\u03d5']), ('\U0001d754', &['\u03f1']),
        ('\U0001d755', &['\u03d6']), ('\U0001d756', &['\u0391']), ('\U0001d757', &['\u0392']),
        ('\U0001d758', &['\u0393']), ('\U0001d759', &['\u0394']), ('\U0001d75a', &['\u0395']),
        ('\U0001d75b', &['\u0396']), ('\U0001d75c', &['\u0397']), ('\U0001d75d', &['\u0398']),
        ('\U0001d75e', &['\u0399']), ('\U0001d75f', &['\u039a']), ('\U0001d760', &['\u039b']),
        ('\U0001d761', &['\u039c']), ('\U0001d762', &['\u039d']), ('\U0001d763', &['\u039e']),
        ('\U0001d764', &['\u039f']), ('\U0001d765', &['\u03a0']), ('\U0001d766', &['\u03a1']),
        ('\U0001d767', &['\u03f4']), ('\U0001d768', &['\u03a3']), ('\U0001d769', &['\u03a4']),
        ('\U0001d76a', &['\u03a5']), ('\U0001d76b', &['\u03a6']), ('\U0001d76c', &['\u03a7']),
        ('\U0001d76d', &['\u03a8']), ('\U0001d76e', &['\u03a9']), ('\U0001d76f', &['\u2207']),
        ('\U0001d770', &['\u03b1']), ('\U0001d771', &['\u03b2']), ('\U0001d772', &['\u03b3']),
        ('\U0001d773', &['\u03b4']), ('\U0001d774', &['\u03b5']), ('\U0001d775', &['\u03b6']),
        ('\U0001d776', &['\u03b7']), ('\U0001d777', &['\u03b8']), ('\U0001d778', &['\u03b9']),
        ('\U0001d779', &['\u03ba']), ('\U0001d77a', &['\u03bb']), ('\U0001d77b', &['\u03bc']),
        ('\U0001d77c', &['\u03bd']), ('\U0001d77d', &['\u03be']), ('\U0001d77e', &['\u03bf']),
        ('\U0001d77f', &['\u03c0']), ('\U0001d780', &['\u03c1']), ('\U0001d781', &['\u03c2']),
        ('\U0001d782', &['\u03c3']), ('\U0001d783', &['\u03c4']), ('\U0001d784', &['\u03c5']),
        ('\U0001d785', &['\u03c6']), ('\U0001d786', &['\u03c7']), ('\U0001d787', &['\u03c8']),
        ('\U0001d788', &['\u03c9']), ('\U0001d789', &['\u2202']), ('\U0001d78a', &['\u03f5']),
        ('\U0001d78b', &['\u03d1']), ('\U0001d78c', &['\u03f0']), ('\U0001d78d', &['\u03d5']),
        ('\U0001d78e', &['\u03f1']), ('\U0001d78f', &['\u03d6']), ('\U0001d790', &['\u0391']),
        ('\U0001d791', &['\u0392']), ('\U0001d792', &['\u0393']), ('\U0001d793', &['\u0394']),
        ('\U0001d794', &['\u0395']), ('\U0001d795', &['\u0396']), ('\U0001d796', &['\u0397']),
        ('\U0001d797', &['\u0398']), ('\U0001d798', &['\u0399']), ('\U0001d799', &['\u039a']),
        ('\U0001d79a', &['\u039b']), ('\U0001d79b', &['\u039c']), ('\U0001d79c', &['\u039d']),
        ('\U0001d79d', &['\u039e']), ('\U0001d79e', &['\u039f']), ('\U0001d79f', &['\u03a0']),
        ('\U0001d7a0', &['\u03a1']), ('\U0001d7a1', &['\u03f4']), ('\U0001d7a2', &['\u03a3']),
        ('\U0001d7a3', &['\u03a4']), ('\U0001d7a4', &['\u03a5']), ('\U0001d7a5', &['\u03a6']),
        ('\U0001d7a6', &['\u03a7']), ('\U0001d7a7', &['\u03a8']), ('\U0001d7a8', &['\u03a9']),
        ('\U0001d7a9', &['\u2207']), ('\U0001d7aa', &['\u03b1']), ('\U0001d7ab', &['\u03b2']),
        ('\U0001d7ac', &['\u03b3']), ('\U0001d7ad', &['\u03b4']), ('\U0001d7ae', &['\u03b5']),
        ('\U0001d7af', &['\u03b6']), ('\U0001d7b0', &['\u03b7']), ('\U0001d7b1', &['\u03b8']),
        ('\U0001d7b2', &['\u03b9']), ('\U0001d7b3', &['\u03ba']), ('\U0001d7b4', &['\u03bb']),
        ('\U0001d7b5', &['\u03bc']), ('\U0001d7b6', &['\u03bd']), ('\U0001d7b7', &['\u03be']),
        ('\U0001d7b8', &['\u03bf']), ('\U0001d7b9', &['\u03c0']), ('\U0001d7ba', &['\u03c1']),
        ('\U0001d7bb', &['\u03c2']), ('\U0001d7bc', &['\u03c3']), ('\U0001d7bd', &['\u03c4']),
        ('\U0001d7be', &['\u03c5']), ('\U0001d7bf', &['\u03c6']), ('\U0001d7c0', &['\u03c7']),
        ('\U0001d7c1', &['\u03c8']), ('\U0001d7c2', &['\u03c9']), ('\U0001d7c3', &['\u2202']),
        ('\U0001d7c4', &['\u03f5']), ('\U0001d7c5', &['\u03d1']), ('\U0001d7c6', &['\u03f0']),
        ('\U0001d7c7', &['\u03d5']), ('\U0001d7c8', &['\u03f1']), ('\U0001d7c9', &['\u03d6']),
        ('\U0001d7ca', &['\u03dc']), ('\U0001d7cb', &['\u03dd']), ('\U0001d7ce', &['\x30']),
        ('\U0001d7cf', &['\x31']), ('\U0001d7d0', &['\x32']), ('\U0001d7d1', &['\x33']),
        ('\U0001d7d2', &['\x34']), ('\U0001d7d3', &['\x35']), ('\U0001d7d4', &['\x36']),
        ('\U0001d7d5', &['\x37']), ('\U0001d7d6', &['\x38']), ('\U0001d7d7', &['\x39']),
        ('\U0001d7d8', &['\x30']), ('\U0001d7d9', &['\x31']), ('\U0001d7da', &['\x32']),
        ('\U0001d7db', &['\x33']), ('\U0001d7dc', &['\x34']), ('\U0001d7dd', &['\x35']),
        ('\U0001d7de', &['\x36']), ('\U0001d7df', &['\x37']), ('\U0001d7e0', &['\x38']),
        ('\U0001d7e1', &['\x39']), ('\U0001d7e2', &['\x30']), ('\U0001d7e3', &['\x31']),
        ('\U0001d7e4', &['\x32']), ('\U0001d7e5', &['\x33']), ('\U0001d7e6', &['\x34']),
        ('\U0001d7e7', &['\x35']), ('\U0001d7e8', &['\x36']), ('\U0001d7e9', &['\x37']),
        ('\U0001d7ea', &['\x38']), ('\U0001d7eb', &['\x39']), ('\U0001d7ec', &['\x30']),
        ('\U0001d7ed', &['\x31']), ('\U0001d7ee', &['\x32']), ('\U0001d7ef', &['\x33']),
        ('\U0001d7f0', &['\x34']), ('\U0001d7f1', &['\x35']), ('\U0001d7f2', &['\x36']),
        ('\U0001d7f3', &['\x37']), ('\U0001d7f4', &['\x38']), ('\U0001d7f5', &['\x39']),
        ('\U0001d7f6', &['\x30']), ('\U0001d7f7', &['\x31']), ('\U0001d7f8', &['\x32']),
        ('\U0001d7f9', &['\x33']), ('\U0001d7fa', &['\x34']), ('\U0001d7fb', &['\x35']),
        ('\U0001d7fc', &['\x36']), ('\U0001d7fd', &['\x37']), ('\U0001d7fe', &['\x38']),
        ('\U0001d7ff', &['\x39']), ('\U0001ee00', &['\u0627']), ('\U0001ee01', &['\u0628']),
        ('\U0001ee02', &['\u062c']), ('\U0001ee03', &['\u062f']), ('\U0001ee05', &['\u0648']),
        ('\U0001ee06', &['\u0632']), ('\U0001ee07', &['\u062d']), ('\U0001ee08', &['\u0637']),
        ('\U0001ee09', &['\u064a']), ('\U0001ee0a', &['\u0643']), ('\U0001ee0b', &['\u0644']),
        ('\U0001ee0c', &['\u0645']), ('\U0001ee0d', &['\u0646']), ('\U0001ee0e', &['\u0633']),
        ('\U0001ee0f', &['\u0639']), ('\U0001ee10', &['\u0641']), ('\U0001ee11', &['\u0635']),
        ('\U0001ee12', &['\u0642']), ('\U0001ee13', &['\u0631']), ('\U0001ee14', &['\u0634']),
        ('\U0001ee15', &['\u062a']), ('\U0001ee16', &['\u062b']), ('\U0001ee17', &['\u062e']),
        ('\U0001ee18', &['\u0630']), ('\U0001ee19', &['\u0636']), ('\U0001ee1a', &['\u0638']),
        ('\U0001ee1b', &['\u063a']), ('\U0001ee1c', &['\u066e']), ('\U0001ee1d', &['\u06ba']),
        ('\U0001ee1e', &['\u06a1']), ('\U0001ee1f', &['\u066f']), ('\U0001ee21', &['\u0628']),
        ('\U0001ee22', &['\u062c']), ('\U0001ee24', &['\u0647']), ('\U0001ee27', &['\u062d']),
        ('\U0001ee29', &['\u064a']), ('\U0001ee2a', &['\u0643']), ('\U0001ee2b', &['\u0644']),
        ('\U0001ee2c', &['\u0645']), ('\U0001ee2d', &['\u0646']), ('\U0001ee2e', &['\u0633']),
        ('\U0001ee2f', &['\u0639']), ('\U0001ee30', &['\u0641']), ('\U0001ee31', &['\u0635']),
        ('\U0001ee32', &['\u0642']), ('\U0001ee34', &['\u0634']), ('\U0001ee35', &['\u062a']),
        ('\U0001ee36', &['\u062b']), ('\U0001ee37', &['\u062e']), ('\U0001ee39', &['\u0636']),
        ('\U0001ee3b', &['\u063a']), ('\U0001ee42', &['\u062c']), ('\U0001ee47', &['\u062d']),
        ('\U0001ee49', &['\u064a']), ('\U0001ee4b', &['\u0644']), ('\U0001ee4d', &['\u0646']),
        ('\U0001ee4e', &['\u0633']), ('\U0001ee4f', &['\u0639']), ('\U0001ee51', &['\u0635']),
        ('\U0001ee52', &['\u0642']), ('\U0001ee54', &['\u0634']), ('\U0001ee57', &['\u062e']),
        ('\U0001ee59', &['\u0636']), ('\U0001ee5b', &['\u063a']), ('\U0001ee5d', &['\u06ba']),
        ('\U0001ee5f', &['\u066f']), ('\U0001ee61', &['\u0628']), ('\U0001ee62', &['\u062c']),
        ('\U0001ee64', &['\u0647']), ('\U0001ee67', &['\u062d']), ('\U0001ee68', &['\u0637']),
        ('\U0001ee69', &['\u064a']), ('\U0001ee6a', &['\u0643']), ('\U0001ee6c', &['\u0645']),
        ('\U0001ee6d', &['\u0646']), ('\U0001ee6e', &['\u0633']), ('\U0001ee6f', &['\u0639']),
        ('\U0001ee70', &['\u0641']), ('\U0001ee71', &['\u0635']), ('\U0001ee72', &['\u0642']),
        ('\U0001ee74', &['\u0634']), ('\U0001ee75', &['\u062a']), ('\U0001ee76', &['\u062b']),
        ('\U0001ee77', &['\u062e']), ('\U0001ee79', &['\u0636']), ('\U0001ee7a', &['\u0638']),
        ('\U0001ee7b', &['\u063a']), ('\U0001ee7c', &['\u066e']), ('\U0001ee7e', &['\u06a1']),
        ('\U0001ee80', &['\u0627']), ('\U0001ee81', &['\u0628']), ('\U0001ee82', &['\u062c']),
        ('\U0001ee83', &['\u062f']), ('\U0001ee84', &['\u0647']), ('\U0001ee85', &['\u0648']),
        ('\U0001ee86', &['\u0632']), ('\U0001ee87', &['\u062d']), ('\U0001ee88', &['\u0637']),
        ('\U0001ee89', &['\u064a']), ('\U0001ee8b', &['\u0644']), ('\U0001ee8c', &['\u0645']),
        ('\U0001ee8d', &['\u0646']), ('\U0001ee8e', &['\u0633']), ('\U0001ee8f', &['\u0639']),
        ('\U0001ee90', &['\u0641']), ('\U0001ee91', &['\u0635']), ('\U0001ee92', &['\u0642']),
        ('\U0001ee93', &['\u0631']), ('\U0001ee94', &['\u0634']), ('\U0001ee95', &['\u062a']),
        ('\U0001ee96', &['\u062b']), ('\U0001ee97', &['\u062e']), ('\U0001ee98', &['\u0630']),
        ('\U0001ee99', &['\u0636']), ('\U0001ee9a', &['\u0638']), ('\U0001ee9b', &['\u063a']),
        ('\U0001eea1', &['\u0628']), ('\U0001eea2', &['\u062c']), ('\U0001eea3', &['\u062f']),
        ('\U0001eea5', &['\u0648']), ('\U0001eea6', &['\u0632']), ('\U0001eea7', &['\u062d']),
        ('\U0001eea8', &['\u0637']), ('\U0001eea9', &['\u064a']), ('\U0001eeab', &['\u0644']),
        ('\U0001eeac', &['\u0645']), ('\U0001eead', &['\u0646']), ('\U0001eeae', &['\u0633']),
        ('\U0001eeaf', &['\u0639']), ('\U0001eeb0', &['\u0641']), ('\U0001eeb1', &['\u0635']),
        ('\U0001eeb2', &['\u0642']), ('\U0001eeb3', &['\u0631']), ('\U0001eeb4', &['\u0634']),
        ('\U0001eeb5', &['\u062a']), ('\U0001eeb6', &['\u062b']), ('\U0001eeb7', &['\u062e']),
        ('\U0001eeb8', &['\u0630']), ('\U0001eeb9', &['\u0636']), ('\U0001eeba', &['\u0638']),
        ('\U0001eebb', &['\u063a']), ('\U0001f100', &['\x30', '\x2e']), ('\U0001f101', &['\x30',
        '\x2c']), ('\U0001f102', &['\x31', '\x2c']), ('\U0001f103', &['\x32', '\x2c']),
        ('\U0001f104', &['\x33', '\x2c']), ('\U0001f105', &['\x34', '\x2c']), ('\U0001f106',
        &['\x35', '\x2c']), ('\U0001f107', &['\x36', '\x2c']), ('\U0001f108', &['\x37', '\x2c']),
        ('\U0001f109', &['\x38', '\x2c']), ('\U0001f10a', &['\x39', '\x2c']), ('\U0001f110',
        &['\x28', '\x41', '\x29']), ('\U0001f111', &['\x28', '\x42', '\x29']), ('\U0001f112',
        &['\x28', '\x43', '\x29']), ('\U0001f113', &['\x28', '\x44', '\x29']), ('\U0001f114',
        &['\x28', '\x45', '\x29']), ('\U0001f115', &['\x28', '\x46', '\x29']), ('\U0001f116',
        &['\x28', '\x47', '\x29']), ('\U0001f117', &['\x28', '\x48', '\x29']), ('\U0001f118',
        &['\x28', '\x49', '\x29']), ('\U0001f119', &['\x28', '\x4a', '\x29']), ('\U0001f11a',
        &['\x28', '\x4b', '\x29']), ('\U0001f11b', &['\x28', '\x4c', '\x29']), ('\U0001f11c',
        &['\x28', '\x4d', '\x29']), ('\U0001f11d', &['\x28', '\x4e', '\x29']), ('\U0001f11e',
        &['\x28', '\x4f', '\x29']), ('\U0001f11f', &['\x28', '\x50', '\x29']), ('\U0001f120',
        &['\x28', '\x51', '\x29']), ('\U0001f121', &['\x28', '\x52', '\x29']), ('\U0001f122',
        &['\x28', '\x53', '\x29']), ('\U0001f123', &['\x28', '\x54', '\x29']), ('\U0001f124',
        &['\x28', '\x55', '\x29']), ('\U0001f125', &['\x28', '\x56', '\x29']), ('\U0001f126',
        &['\x28', '\x57', '\x29']), ('\U0001f127', &['\x28', '\x58', '\x29']), ('\U0001f128',
        &['\x28', '\x59', '\x29']), ('\U0001f129', &['\x28', '\x5a', '\x29']), ('\U0001f12a',
        &['\u3014', '\x53', '\u3015']), ('\U0001f12b', &['\x43']), ('\U0001f12c', &['\x52']),
        ('\U0001f12d', &['\x43', '\x44']), ('\U0001f12e', &['\x57', '\x5a']), ('\U0001f130',
        &['\x41']), ('\U0001f131', &['\x42']), ('\U0001f132', &['\x43']), ('\U0001f133', &['\x44']),
        ('\U0001f134', &['\x45']), ('\U0001f135', &['\x46']), ('\U0001f136', &['\x47']),
        ('\U0001f137', &['\x48']), ('\U0001f138', &['\x49']), ('\U0001f139', &['\x4a']),
        ('\U0001f13a', &['\x4b']), ('\U0001f13b', &['\x4c']), ('\U0001f13c', &['\x4d']),
        ('\U0001f13d', &['\x4e']), ('\U0001f13e', &['\x4f']), ('\U0001f13f', &['\x50']),
        ('\U0001f140', &['\x51']), ('\U0001f141', &['\x52']), ('\U0001f142', &['\x53']),
        ('\U0001f143', &['\x54']), ('\U0001f144', &['\x55']), ('\U0001f145', &['\x56']),
        ('\U0001f146', &['\x57']), ('\U0001f147', &['\x58']), ('\U0001f148', &['\x59']),
        ('\U0001f149', &['\x5a']), ('\U0001f14a', &['\x48', '\x56']), ('\U0001f14b', &['\x4d',
        '\x56']), ('\U0001f14c', &['\x53', '\x44']), ('\U0001f14d', &['\x53', '\x53']),
        ('\U0001f14e', &['\x50', '\x50', '\x56']), ('\U0001f14f', &['\x57', '\x43']), ('\U0001f16a',
        &['\x4d', '\x43']), ('\U0001f16b', &['\x4d', '\x44']), ('\U0001f190', &['\x44', '\x4a']),
        ('\U0001f200', &['\u307b', '\u304b']), ('\U0001f201', &['\u30b3', '\u30b3']), ('\U0001f202',
        &['\u30b5']), ('\U0001f210', &['\u624b']), ('\U0001f211', &['\u5b57']), ('\U0001f212',
        &['\u53cc']), ('\U0001f213', &['\u30c7']), ('\U0001f214', &['\u4e8c']), ('\U0001f215',
        &['\u591a']), ('\U0001f216', &['\u89e3']), ('\U0001f217', &['\u5929']), ('\U0001f218',
        &['\u4ea4']), ('\U0001f219', &['\u6620']), ('\U0001f21a', &['\u7121']), ('\U0001f21b',
        &['\u6599']), ('\U0001f21c', &['\u524d']), ('\U0001f21d', &['\u5f8c']), ('\U0001f21e',
        &['\u518d']), ('\U0001f21f', &['\u65b0']), ('\U0001f220', &['\u521d']), ('\U0001f221',
        &['\u7d42']), ('\U0001f222', &['\u751f']), ('\U0001f223', &['\u8ca9']), ('\U0001f224',
        &['\u58f0']), ('\U0001f225', &['\u5439']), ('\U0001f226', &['\u6f14']), ('\U0001f227',
        &['\u6295']), ('\U0001f228', &['\u6355']), ('\U0001f229', &['\u4e00']), ('\U0001f22a',
        &['\u4e09']), ('\U0001f22b', &['\u904a']), ('\U0001f22c', &['\u5de6']), ('\U0001f22d',
        &['\u4e2d']), ('\U0001f22e', &['\u53f3']), ('\U0001f22f', &['\u6307']), ('\U0001f230',
        &['\u8d70']), ('\U0001f231', &['\u6253']), ('\U0001f232', &['\u7981']), ('\U0001f233',
        &['\u7a7a']), ('\U0001f234', &['\u5408']), ('\U0001f235', &['\u6e80']), ('\U0001f236',
        &['\u6709']), ('\U0001f237', &['\u6708']), ('\U0001f238', &['\u7533']), ('\U0001f239',
        &['\u5272']), ('\U0001f23a', &['\u55b6']), ('\U0001f240', &['\u3014', '\u672c', '\u3015']),
        ('\U0001f241', &['\u3014', '\u4e09', '\u3015']), ('\U0001f242', &['\u3014', '\u4e8c',
        '\u3015']), ('\U0001f243', &['\u3014', '\u5b89', '\u3015']), ('\U0001f244', &['\u3014',
        '\u70b9', '\u3015']), ('\U0001f245', &['\u3014', '\u6253', '\u3015']), ('\U0001f246',
        &['\u3014', '\u76d7', '\u3015']), ('\U0001f247', &['\u3014', '\u52dd', '\u3015']),
        ('\U0001f248', &['\u3014', '\u6557', '\u3015']), ('\U0001f250', &['\u5f97']), ('\U0001f251',
        &['\u53ef'])
    ];

    static combining_class_table : &'static [(char, char, u8)] = &[
        ('\u0300', '\u0314', 230), ('\u0315', '\u0315', 232),
        ('\u0316', '\u0319', 220), ('\u031a', '\u031a', 232),
        ('\u031b', '\u031b', 216), ('\u031c', '\u0320', 220),
        ('\u0321', '\u0322', 202), ('\u0323', '\u0326', 220),
        ('\u0327', '\u0328', 202), ('\u0329', '\u0333', 220),
        ('\u0334', '\u0338', 1), ('\u0339', '\u033c', 220),
        ('\u033d', '\u0344', 230), ('\u0345', '\u0345', 240),
        ('\u0346', '\u0346', 230), ('\u0347', '\u0349', 220),
        ('\u034a', '\u034c', 230), ('\u034d', '\u034e', 220),
        ('\u0350', '\u0352', 230), ('\u0353', '\u0356', 220),
        ('\u0357', '\u0357', 230), ('\u0358', '\u0358', 232),
        ('\u0359', '\u035a', 220), ('\u035b', '\u035b', 230),
        ('\u035c', '\u035c', 233), ('\u035d', '\u035e', 234),
        ('\u035f', '\u035f', 233), ('\u0360', '\u0361', 234),
        ('\u0362', '\u0362', 233), ('\u0363', '\u036f', 230),
        ('\u0483', '\u0487', 230), ('\u0591', '\u0591', 220),
        ('\u0592', '\u0595', 230), ('\u0596', '\u0596', 220),
        ('\u0597', '\u0599', 230), ('\u059a', '\u059a', 222),
        ('\u059b', '\u059b', 220), ('\u059c', '\u05a1', 230),
        ('\u05a2', '\u05a7', 220), ('\u05a8', '\u05a9', 230),
        ('\u05aa', '\u05aa', 220), ('\u05ab', '\u05ac', 230),
        ('\u05ad', '\u05ad', 222), ('\u05ae', '\u05ae', 228),
        ('\u05af', '\u05af', 230), ('\u05b0', '\u05b0', 10),
        ('\u05b1', '\u05b1', 11), ('\u05b2', '\u05b2', 12),
        ('\u05b3', '\u05b3', 13), ('\u05b4', '\u05b4', 14),
        ('\u05b5', '\u05b5', 15), ('\u05b6', '\u05b6', 16),
        ('\u05b7', '\u05b7', 17), ('\u05b8', '\u05b8', 18),
        ('\u05b9', '\u05ba', 19), ('\u05bb', '\u05bb', 20),
        ('\u05bc', '\u05bc', 21), ('\u05bd', '\u05bd', 22),
        ('\u05bf', '\u05bf', 23), ('\u05c1', '\u05c1', 24),
        ('\u05c2', '\u05c2', 25), ('\u05c4', '\u05c4', 230),
        ('\u05c5', '\u05c5', 220), ('\u05c7', '\u05c7', 18),
        ('\u0610', '\u0617', 230), ('\u0618', '\u0618', 30),
        ('\u0619', '\u0619', 31), ('\u061a', '\u061a', 32),
        ('\u064b', '\u064b', 27), ('\u064c', '\u064c', 28),
        ('\u064d', '\u064d', 29), ('\u064e', '\u064e', 30),
        ('\u064f', '\u064f', 31), ('\u0650', '\u0650', 32),
        ('\u0651', '\u0651', 33), ('\u0652', '\u0652', 34),
        ('\u0653', '\u0654', 230), ('\u0655', '\u0656', 220),
        ('\u0657', '\u065b', 230), ('\u065c', '\u065c', 220),
        ('\u065d', '\u065e', 230), ('\u065f', '\u065f', 220),
        ('\u0670', '\u0670', 35), ('\u06d6', '\u06dc', 230),
        ('\u06df', '\u06e2', 230), ('\u06e3', '\u06e3', 220),
        ('\u06e4', '\u06e4', 230), ('\u06e7', '\u06e8', 230),
        ('\u06ea', '\u06ea', 220), ('\u06eb', '\u06ec', 230),
        ('\u06ed', '\u06ed', 220), ('\u0711', '\u0711', 36),
        ('\u0730', '\u0730', 230), ('\u0731', '\u0731', 220),
        ('\u0732', '\u0733', 230), ('\u0734', '\u0734', 220),
        ('\u0735', '\u0736', 230), ('\u0737', '\u0739', 220),
        ('\u073a', '\u073a', 230), ('\u073b', '\u073c', 220),
        ('\u073d', '\u073d', 230), ('\u073e', '\u073e', 220),
        ('\u073f', '\u0741', 230), ('\u0742', '\u0742', 220),
        ('\u0743', '\u0743', 230), ('\u0744', '\u0744', 220),
        ('\u0745', '\u0745', 230), ('\u0746', '\u0746', 220),
        ('\u0747', '\u0747', 230), ('\u0748', '\u0748', 220),
        ('\u0749', '\u074a', 230), ('\u07eb', '\u07f1', 230),
        ('\u07f2', '\u07f2', 220), ('\u07f3', '\u07f3', 230),
        ('\u0816', '\u0819', 230), ('\u081b', '\u0823', 230),
        ('\u0825', '\u0827', 230), ('\u0829', '\u082d', 230),
        ('\u0859', '\u085b', 220), ('\u08e4', '\u08e5', 230),
        ('\u08e6', '\u08e6', 220), ('\u08e7', '\u08e8', 230),
        ('\u08e9', '\u08e9', 220), ('\u08ea', '\u08ec', 230),
        ('\u08ed', '\u08ef', 220), ('\u08f0', '\u08f0', 27),
        ('\u08f1', '\u08f1', 28), ('\u08f2', '\u08f2', 29),
        ('\u08f3', '\u08f5', 230), ('\u08f6', '\u08f6', 220),
        ('\u08f7', '\u08f8', 230), ('\u08f9', '\u08fa', 220),
        ('\u08fb', '\u08fe', 230), ('\u093c', '\u093c', 7),
        ('\u094d', '\u094d', 9), ('\u0951', '\u0951', 230),
        ('\u0952', '\u0952', 220), ('\u0953', '\u0954', 230),
        ('\u09bc', '\u09bc', 7), ('\u09cd', '\u09cd', 9),
        ('\u0a3c', '\u0a3c', 7), ('\u0a4d', '\u0a4d', 9),
        ('\u0abc', '\u0abc', 7), ('\u0acd', '\u0acd', 9),
        ('\u0b3c', '\u0b3c', 7), ('\u0b4d', '\u0b4d', 9),
        ('\u0bcd', '\u0bcd', 9), ('\u0c4d', '\u0c4d', 9),
        ('\u0c55', '\u0c55', 84), ('\u0c56', '\u0c56', 91),
        ('\u0cbc', '\u0cbc', 7), ('\u0ccd', '\u0ccd', 9),
        ('\u0d4d', '\u0d4d', 9), ('\u0dca', '\u0dca', 9),
        ('\u0e38', '\u0e39', 103), ('\u0e3a', '\u0e3a', 9),
        ('\u0e48', '\u0e4b', 107), ('\u0eb8', '\u0eb9', 118),
        ('\u0ec8', '\u0ecb', 122), ('\u0f18', '\u0f19', 220),
        ('\u0f35', '\u0f35', 220), ('\u0f37', '\u0f37', 220),
        ('\u0f39', '\u0f39', 216), ('\u0f71', '\u0f71', 129),
        ('\u0f72', '\u0f72', 130), ('\u0f74', '\u0f74', 132),
        ('\u0f7a', '\u0f7d', 130), ('\u0f80', '\u0f80', 130),
        ('\u0f82', '\u0f83', 230), ('\u0f84', '\u0f84', 9),
        ('\u0f86', '\u0f87', 230), ('\u0fc6', '\u0fc6', 220),
        ('\u1037', '\u1037', 7), ('\u1039', '\u103a', 9),
        ('\u108d', '\u108d', 220), ('\u135d', '\u135f', 230),
        ('\u1714', '\u1714', 9), ('\u1734', '\u1734', 9),
        ('\u17d2', '\u17d2', 9), ('\u17dd', '\u17dd', 230),
        ('\u18a9', '\u18a9', 228), ('\u1939', '\u1939', 222),
        ('\u193a', '\u193a', 230), ('\u193b', '\u193b', 220),
        ('\u1a17', '\u1a17', 230), ('\u1a18', '\u1a18', 220),
        ('\u1a60', '\u1a60', 9), ('\u1a75', '\u1a7c', 230),
        ('\u1a7f', '\u1a7f', 220), ('\u1b34', '\u1b34', 7),
        ('\u1b44', '\u1b44', 9), ('\u1b6b', '\u1b6b', 230),
        ('\u1b6c', '\u1b6c', 220), ('\u1b6d', '\u1b73', 230),
        ('\u1baa', '\u1bab', 9), ('\u1be6', '\u1be6', 7),
        ('\u1bf2', '\u1bf3', 9), ('\u1c37', '\u1c37', 7),
        ('\u1cd0', '\u1cd2', 230), ('\u1cd4', '\u1cd4', 1),
        ('\u1cd5', '\u1cd9', 220), ('\u1cda', '\u1cdb', 230),
        ('\u1cdc', '\u1cdf', 220), ('\u1ce0', '\u1ce0', 230),
        ('\u1ce2', '\u1ce8', 1), ('\u1ced', '\u1ced', 220),
        ('\u1cf4', '\u1cf4', 230), ('\u1dc0', '\u1dc1', 230),
        ('\u1dc2', '\u1dc2', 220), ('\u1dc3', '\u1dc9', 230),
        ('\u1dca', '\u1dca', 220), ('\u1dcb', '\u1dcc', 230),
        ('\u1dcd', '\u1dcd', 234), ('\u1dce', '\u1dce', 214),
        ('\u1dcf', '\u1dcf', 220), ('\u1dd0', '\u1dd0', 202),
        ('\u1dd1', '\u1de6', 230), ('\u1dfc', '\u1dfc', 233),
        ('\u1dfd', '\u1dfd', 220), ('\u1dfe', '\u1dfe', 230),
        ('\u1dff', '\u1dff', 220), ('\u20d0', '\u20d1', 230),
        ('\u20d2', '\u20d3', 1), ('\u20d4', '\u20d7', 230),
        ('\u20d8', '\u20da', 1), ('\u20db', '\u20dc', 230),
        ('\u20e1', '\u20e1', 230), ('\u20e5', '\u20e6', 1),
        ('\u20e7', '\u20e7', 230), ('\u20e8', '\u20e8', 220),
        ('\u20e9', '\u20e9', 230), ('\u20ea', '\u20eb', 1),
        ('\u20ec', '\u20ef', 220), ('\u20f0', '\u20f0', 230),
        ('\u2cef', '\u2cf1', 230), ('\u2d7f', '\u2d7f', 9),
        ('\u2de0', '\u2dff', 230), ('\u302a', '\u302a', 218),
        ('\u302b', '\u302b', 228), ('\u302c', '\u302c', 232),
        ('\u302d', '\u302d', 222), ('\u302e', '\u302f', 224),
        ('\u3099', '\u309a', 8), ('\ua66f', '\ua66f', 230),
        ('\ua674', '\ua67d', 230), ('\ua69f', '\ua69f', 230),
        ('\ua6f0', '\ua6f1', 230), ('\ua806', '\ua806', 9),
        ('\ua8c4', '\ua8c4', 9), ('\ua8e0', '\ua8f1', 230),
        ('\ua92b', '\ua92d', 220), ('\ua953', '\ua953', 9),
        ('\ua9b3', '\ua9b3', 7), ('\ua9c0', '\ua9c0', 9),
        ('\uaab0', '\uaab0', 230), ('\uaab2', '\uaab3', 230),
        ('\uaab4', '\uaab4', 220), ('\uaab7', '\uaab8', 230),
        ('\uaabe', '\uaabf', 230), ('\uaac1', '\uaac1', 230),
        ('\uaaf6', '\uaaf6', 9), ('\uabed', '\uabed', 9),
        ('\ufb1e', '\ufb1e', 26), ('\ufe20', '\ufe26', 230),
        ('\U000101fd', '\U000101fd', 220), ('\U00010a0d', '\U00010a0d', 220),
        ('\U00010a0f', '\U00010a0f', 230), ('\U00010a38', '\U00010a38', 230),
        ('\U00010a39', '\U00010a39', 1), ('\U00010a3a', '\U00010a3a', 220),
        ('\U00010a3f', '\U00010a3f', 9), ('\U00011046', '\U00011046', 9),
        ('\U000110b9', '\U000110b9', 9), ('\U000110ba', '\U000110ba', 7),
        ('\U00011100', '\U00011102', 230), ('\U00011133', '\U00011134', 9),
        ('\U000111c0', '\U000111c0', 9), ('\U000116b6', '\U000116b6', 9),
        ('\U000116b7', '\U000116b7', 7), ('\U0001d165', '\U0001d166', 216),
        ('\U0001d167', '\U0001d169', 1), ('\U0001d16d', '\U0001d16d', 226),
        ('\U0001d16e', '\U0001d172', 216), ('\U0001d17b', '\U0001d182', 220),
        ('\U0001d185', '\U0001d189', 230), ('\U0001d18a', '\U0001d18b', 220),
        ('\U0001d1aa', '\U0001d1ad', 230), ('\U0001d242', '\U0001d244', 230)
    ];
    pub fn canonical(c: char, i: |char|) { d(c, i, false); }

    pub fn compatibility(c: char, i: |char|) { d(c, i, true); }

    pub fn canonical_combining_class(c: char) -> u8 {
        bsearch_range_value_table(c, combining_class_table)
    }

    fn d(c: char, i: |char|, k: bool) {
        use iter::Iterator;
        if c <= '\x7f' { i(c); return; }

        match bsearch_table(c, canonical_table) {
            Some(canon) => {
                for x in canon.iter() {
                    d(*x, |b| i(b), k);
                }
                return;
            }
            None => ()
        }

        if !k { i(c); return; }

        match bsearch_table(c, compatibility_table) {
            Some(compat) => {
                for x in compat.iter() {
                    d(*x, |b| i(b), k);
                }
                return;
            }
            None => ()
        }

        i(c);
    }
}

pub mod derived_property {
    static Alphabetic_table : &'static [(char,char)] = &[
        ('\x41', '\x5a'), ('\x61', '\x7a'),
        ('\xaa', '\xaa'), ('\xb5', '\xb5'),
        ('\xba', '\xba'), ('\xc0', '\xd6'),
        ('\xd8', '\xf6'), ('\xf8', '\u01ba'),
        ('\u01bb', '\u01bb'), ('\u01bc', '\u01bf'),
        ('\u01c0', '\u01c3'), ('\u01c4', '\u0293'),
        ('\u0294', '\u0294'), ('\u0295', '\u02af'),
        ('\u02b0', '\u02c1'), ('\u02c6', '\u02d1'),
        ('\u02e0', '\u02e4'), ('\u02ec', '\u02ec'),
        ('\u02ee', '\u02ee'), ('\u0345', '\u0345'),
        ('\u0370', '\u0373'), ('\u0374', '\u0374'),
        ('\u0376', '\u0377'), ('\u037a', '\u037a'),
        ('\u037b', '\u037d'), ('\u0386', '\u0386'),
        ('\u0388', '\u038a'), ('\u038c', '\u038c'),
        ('\u038e', '\u03a1'), ('\u03a3', '\u03f5'),
        ('\u03f7', '\u0481'), ('\u048a', '\u0527'),
        ('\u0531', '\u0556'), ('\u0559', '\u0559'),
        ('\u0561', '\u0587'), ('\u05b0', '\u05bd'),
        ('\u05bf', '\u05bf'), ('\u05c1', '\u05c2'),
        ('\u05c4', '\u05c5'), ('\u05c7', '\u05c7'),
        ('\u05d0', '\u05ea'), ('\u05f0', '\u05f2'),
        ('\u0610', '\u061a'), ('\u0620', '\u063f'),
        ('\u0640', '\u0640'), ('\u0641', '\u064a'),
        ('\u064b', '\u0657'), ('\u0659', '\u065f'),
        ('\u066e', '\u066f'), ('\u0670', '\u0670'),
        ('\u0671', '\u06d3'), ('\u06d5', '\u06d5'),
        ('\u06d6', '\u06dc'), ('\u06e1', '\u06e4'),
        ('\u06e5', '\u06e6'), ('\u06e7', '\u06e8'),
        ('\u06ed', '\u06ed'), ('\u06ee', '\u06ef'),
        ('\u06fa', '\u06fc'), ('\u06ff', '\u06ff'),
        ('\u0710', '\u0710'), ('\u0711', '\u0711'),
        ('\u0712', '\u072f'), ('\u0730', '\u073f'),
        ('\u074d', '\u07a5'), ('\u07a6', '\u07b0'),
        ('\u07b1', '\u07b1'), ('\u07ca', '\u07ea'),
        ('\u07f4', '\u07f5'), ('\u07fa', '\u07fa'),
        ('\u0800', '\u0815'), ('\u0816', '\u0817'),
        ('\u081a', '\u081a'), ('\u081b', '\u0823'),
        ('\u0824', '\u0824'), ('\u0825', '\u0827'),
        ('\u0828', '\u0828'), ('\u0829', '\u082c'),
        ('\u0840', '\u0858'), ('\u08a0', '\u08a0'),
        ('\u08a2', '\u08ac'), ('\u08e4', '\u08e9'),
        ('\u08f0', '\u08fe'), ('\u0900', '\u0902'),
        ('\u0903', '\u0903'), ('\u0904', '\u0939'),
        ('\u093a', '\u093a'), ('\u093b', '\u093b'),
        ('\u093d', '\u093d'), ('\u093e', '\u0940'),
        ('\u0941', '\u0948'), ('\u0949', '\u094c'),
        ('\u094e', '\u094f'), ('\u0950', '\u0950'),
        ('\u0955', '\u0957'), ('\u0958', '\u0961'),
        ('\u0962', '\u0963'), ('\u0971', '\u0971'),
        ('\u0972', '\u0977'), ('\u0979', '\u097f'),
        ('\u0981', '\u0981'), ('\u0982', '\u0983'),
        ('\u0985', '\u098c'), ('\u098f', '\u0990'),
        ('\u0993', '\u09a8'), ('\u09aa', '\u09b0'),
        ('\u09b2', '\u09b2'), ('\u09b6', '\u09b9'),
        ('\u09bd', '\u09bd'), ('\u09be', '\u09c0'),
        ('\u09c1', '\u09c4'), ('\u09c7', '\u09c8'),
        ('\u09cb', '\u09cc'), ('\u09ce', '\u09ce'),
        ('\u09d7', '\u09d7'), ('\u09dc', '\u09dd'),
        ('\u09df', '\u09e1'), ('\u09e2', '\u09e3'),
        ('\u09f0', '\u09f1'), ('\u0a01', '\u0a02'),
        ('\u0a03', '\u0a03'), ('\u0a05', '\u0a0a'),
        ('\u0a0f', '\u0a10'), ('\u0a13', '\u0a28'),
        ('\u0a2a', '\u0a30'), ('\u0a32', '\u0a33'),
        ('\u0a35', '\u0a36'), ('\u0a38', '\u0a39'),
        ('\u0a3e', '\u0a40'), ('\u0a41', '\u0a42'),
        ('\u0a47', '\u0a48'), ('\u0a4b', '\u0a4c'),
        ('\u0a51', '\u0a51'), ('\u0a59', '\u0a5c'),
        ('\u0a5e', '\u0a5e'), ('\u0a70', '\u0a71'),
        ('\u0a72', '\u0a74'), ('\u0a75', '\u0a75'),
        ('\u0a81', '\u0a82'), ('\u0a83', '\u0a83'),
        ('\u0a85', '\u0a8d'), ('\u0a8f', '\u0a91'),
        ('\u0a93', '\u0aa8'), ('\u0aaa', '\u0ab0'),
        ('\u0ab2', '\u0ab3'), ('\u0ab5', '\u0ab9'),
        ('\u0abd', '\u0abd'), ('\u0abe', '\u0ac0'),
        ('\u0ac1', '\u0ac5'), ('\u0ac7', '\u0ac8'),
        ('\u0ac9', '\u0ac9'), ('\u0acb', '\u0acc'),
        ('\u0ad0', '\u0ad0'), ('\u0ae0', '\u0ae1'),
        ('\u0ae2', '\u0ae3'), ('\u0b01', '\u0b01'),
        ('\u0b02', '\u0b03'), ('\u0b05', '\u0b0c'),
        ('\u0b0f', '\u0b10'), ('\u0b13', '\u0b28'),
        ('\u0b2a', '\u0b30'), ('\u0b32', '\u0b33'),
        ('\u0b35', '\u0b39'), ('\u0b3d', '\u0b3d'),
        ('\u0b3e', '\u0b3e'), ('\u0b3f', '\u0b3f'),
        ('\u0b40', '\u0b40'), ('\u0b41', '\u0b44'),
        ('\u0b47', '\u0b48'), ('\u0b4b', '\u0b4c'),
        ('\u0b56', '\u0b56'), ('\u0b57', '\u0b57'),
        ('\u0b5c', '\u0b5d'), ('\u0b5f', '\u0b61'),
        ('\u0b62', '\u0b63'), ('\u0b71', '\u0b71'),
        ('\u0b82', '\u0b82'), ('\u0b83', '\u0b83'),
        ('\u0b85', '\u0b8a'), ('\u0b8e', '\u0b90'),
        ('\u0b92', '\u0b95'), ('\u0b99', '\u0b9a'),
        ('\u0b9c', '\u0b9c'), ('\u0b9e', '\u0b9f'),
        ('\u0ba3', '\u0ba4'), ('\u0ba8', '\u0baa'),
        ('\u0bae', '\u0bb9'), ('\u0bbe', '\u0bbf'),
        ('\u0bc0', '\u0bc0'), ('\u0bc1', '\u0bc2'),
        ('\u0bc6', '\u0bc8'), ('\u0bca', '\u0bcc'),
        ('\u0bd0', '\u0bd0'), ('\u0bd7', '\u0bd7'),
        ('\u0c01', '\u0c03'), ('\u0c05', '\u0c0c'),
        ('\u0c0e', '\u0c10'), ('\u0c12', '\u0c28'),
        ('\u0c2a', '\u0c33'), ('\u0c35', '\u0c39'),
        ('\u0c3d', '\u0c3d'), ('\u0c3e', '\u0c40'),
        ('\u0c41', '\u0c44'), ('\u0c46', '\u0c48'),
        ('\u0c4a', '\u0c4c'), ('\u0c55', '\u0c56'),
        ('\u0c58', '\u0c59'), ('\u0c60', '\u0c61'),
        ('\u0c62', '\u0c63'), ('\u0c82', '\u0c83'),
        ('\u0c85', '\u0c8c'), ('\u0c8e', '\u0c90'),
        ('\u0c92', '\u0ca8'), ('\u0caa', '\u0cb3'),
        ('\u0cb5', '\u0cb9'), ('\u0cbd', '\u0cbd'),
        ('\u0cbe', '\u0cbe'), ('\u0cbf', '\u0cbf'),
        ('\u0cc0', '\u0cc4'), ('\u0cc6', '\u0cc6'),
        ('\u0cc7', '\u0cc8'), ('\u0cca', '\u0ccb'),
        ('\u0ccc', '\u0ccc'), ('\u0cd5', '\u0cd6'),
        ('\u0cde', '\u0cde'), ('\u0ce0', '\u0ce1'),
        ('\u0ce2', '\u0ce3'), ('\u0cf1', '\u0cf2'),
        ('\u0d02', '\u0d03'), ('\u0d05', '\u0d0c'),
        ('\u0d0e', '\u0d10'), ('\u0d12', '\u0d3a'),
        ('\u0d3d', '\u0d3d'), ('\u0d3e', '\u0d40'),
        ('\u0d41', '\u0d44'), ('\u0d46', '\u0d48'),
        ('\u0d4a', '\u0d4c'), ('\u0d4e', '\u0d4e'),
        ('\u0d57', '\u0d57'), ('\u0d60', '\u0d61'),
        ('\u0d62', '\u0d63'), ('\u0d7a', '\u0d7f'),
        ('\u0d82', '\u0d83'), ('\u0d85', '\u0d96'),
        ('\u0d9a', '\u0db1'), ('\u0db3', '\u0dbb'),
        ('\u0dbd', '\u0dbd'), ('\u0dc0', '\u0dc6'),
        ('\u0dcf', '\u0dd1'), ('\u0dd2', '\u0dd4'),
        ('\u0dd6', '\u0dd6'), ('\u0dd8', '\u0ddf'),
        ('\u0df2', '\u0df3'), ('\u0e01', '\u0e30'),
        ('\u0e31', '\u0e31'), ('\u0e32', '\u0e33'),
        ('\u0e34', '\u0e3a'), ('\u0e40', '\u0e45'),
        ('\u0e46', '\u0e46'), ('\u0e4d', '\u0e4d'),
        ('\u0e81', '\u0e82'), ('\u0e84', '\u0e84'),
        ('\u0e87', '\u0e88'), ('\u0e8a', '\u0e8a'),
        ('\u0e8d', '\u0e8d'), ('\u0e94', '\u0e97'),
        ('\u0e99', '\u0e9f'), ('\u0ea1', '\u0ea3'),
        ('\u0ea5', '\u0ea5'), ('\u0ea7', '\u0ea7'),
        ('\u0eaa', '\u0eab'), ('\u0ead', '\u0eb0'),
        ('\u0eb1', '\u0eb1'), ('\u0eb2', '\u0eb3'),
        ('\u0eb4', '\u0eb9'), ('\u0ebb', '\u0ebc'),
        ('\u0ebd', '\u0ebd'), ('\u0ec0', '\u0ec4'),
        ('\u0ec6', '\u0ec6'), ('\u0ecd', '\u0ecd'),
        ('\u0edc', '\u0edf'), ('\u0f00', '\u0f00'),
        ('\u0f40', '\u0f47'), ('\u0f49', '\u0f6c'),
        ('\u0f71', '\u0f7e'), ('\u0f7f', '\u0f7f'),
        ('\u0f80', '\u0f81'), ('\u0f88', '\u0f8c'),
        ('\u0f8d', '\u0f97'), ('\u0f99', '\u0fbc'),
        ('\u1000', '\u102a'), ('\u102b', '\u102c'),
        ('\u102d', '\u1030'), ('\u1031', '\u1031'),
        ('\u1032', '\u1036'), ('\u1038', '\u1038'),
        ('\u103b', '\u103c'), ('\u103d', '\u103e'),
        ('\u103f', '\u103f'), ('\u1050', '\u1055'),
        ('\u1056', '\u1057'), ('\u1058', '\u1059'),
        ('\u105a', '\u105d'), ('\u105e', '\u1060'),
        ('\u1061', '\u1061'), ('\u1062', '\u1062'),
        ('\u1065', '\u1066'), ('\u1067', '\u1068'),
        ('\u106e', '\u1070'), ('\u1071', '\u1074'),
        ('\u1075', '\u1081'), ('\u1082', '\u1082'),
        ('\u1083', '\u1084'), ('\u1085', '\u1086'),
        ('\u108e', '\u108e'), ('\u109c', '\u109c'),
        ('\u109d', '\u109d'), ('\u10a0', '\u10c5'),
        ('\u10c7', '\u10c7'), ('\u10cd', '\u10cd'),
        ('\u10d0', '\u10fa'), ('\u10fc', '\u10fc'),
        ('\u10fd', '\u1248'), ('\u124a', '\u124d'),
        ('\u1250', '\u1256'), ('\u1258', '\u1258'),
        ('\u125a', '\u125d'), ('\u1260', '\u1288'),
        ('\u128a', '\u128d'), ('\u1290', '\u12b0'),
        ('\u12b2', '\u12b5'), ('\u12b8', '\u12be'),
        ('\u12c0', '\u12c0'), ('\u12c2', '\u12c5'),
        ('\u12c8', '\u12d6'), ('\u12d8', '\u1310'),
        ('\u1312', '\u1315'), ('\u1318', '\u135a'),
        ('\u135f', '\u135f'), ('\u1380', '\u138f'),
        ('\u13a0', '\u13f4'), ('\u1401', '\u166c'),
        ('\u166f', '\u167f'), ('\u1681', '\u169a'),
        ('\u16a0', '\u16ea'), ('\u16ee', '\u16f0'),
        ('\u1700', '\u170c'), ('\u170e', '\u1711'),
        ('\u1712', '\u1713'), ('\u1720', '\u1731'),
        ('\u1732', '\u1733'), ('\u1740', '\u1751'),
        ('\u1752', '\u1753'), ('\u1760', '\u176c'),
        ('\u176e', '\u1770'), ('\u1772', '\u1773'),
        ('\u1780', '\u17b3'), ('\u17b6', '\u17b6'),
        ('\u17b7', '\u17bd'), ('\u17be', '\u17c5'),
        ('\u17c6', '\u17c6'), ('\u17c7', '\u17c8'),
        ('\u17d7', '\u17d7'), ('\u17dc', '\u17dc'),
        ('\u1820', '\u1842'), ('\u1843', '\u1843'),
        ('\u1844', '\u1877'), ('\u1880', '\u18a8'),
        ('\u18a9', '\u18a9'), ('\u18aa', '\u18aa'),
        ('\u18b0', '\u18f5'), ('\u1900', '\u191c'),
        ('\u1920', '\u1922'), ('\u1923', '\u1926'),
        ('\u1927', '\u1928'), ('\u1929', '\u192b'),
        ('\u1930', '\u1931'), ('\u1932', '\u1932'),
        ('\u1933', '\u1938'), ('\u1950', '\u196d'),
        ('\u1970', '\u1974'), ('\u1980', '\u19ab'),
        ('\u19b0', '\u19c0'), ('\u19c1', '\u19c7'),
        ('\u19c8', '\u19c9'), ('\u1a00', '\u1a16'),
        ('\u1a17', '\u1a18'), ('\u1a19', '\u1a1a'),
        ('\u1a1b', '\u1a1b'), ('\u1a20', '\u1a54'),
        ('\u1a55', '\u1a55'), ('\u1a56', '\u1a56'),
        ('\u1a57', '\u1a57'), ('\u1a58', '\u1a5e'),
        ('\u1a61', '\u1a61'), ('\u1a62', '\u1a62'),
        ('\u1a63', '\u1a64'), ('\u1a65', '\u1a6c'),
        ('\u1a6d', '\u1a72'), ('\u1a73', '\u1a74'),
        ('\u1aa7', '\u1aa7'), ('\u1b00', '\u1b03'),
        ('\u1b04', '\u1b04'), ('\u1b05', '\u1b33'),
        ('\u1b35', '\u1b35'), ('\u1b36', '\u1b3a'),
        ('\u1b3b', '\u1b3b'), ('\u1b3c', '\u1b3c'),
        ('\u1b3d', '\u1b41'), ('\u1b42', '\u1b42'),
        ('\u1b43', '\u1b43'), ('\u1b45', '\u1b4b'),
        ('\u1b80', '\u1b81'), ('\u1b82', '\u1b82'),
        ('\u1b83', '\u1ba0'), ('\u1ba1', '\u1ba1'),
        ('\u1ba2', '\u1ba5'), ('\u1ba6', '\u1ba7'),
        ('\u1ba8', '\u1ba9'), ('\u1bac', '\u1bad'),
        ('\u1bae', '\u1baf'), ('\u1bba', '\u1be5'),
        ('\u1be7', '\u1be7'), ('\u1be8', '\u1be9'),
        ('\u1bea', '\u1bec'), ('\u1bed', '\u1bed'),
        ('\u1bee', '\u1bee'), ('\u1bef', '\u1bf1'),
        ('\u1c00', '\u1c23'), ('\u1c24', '\u1c2b'),
        ('\u1c2c', '\u1c33'), ('\u1c34', '\u1c35'),
        ('\u1c4d', '\u1c4f'), ('\u1c5a', '\u1c77'),
        ('\u1c78', '\u1c7d'), ('\u1ce9', '\u1cec'),
        ('\u1cee', '\u1cf1'), ('\u1cf2', '\u1cf3'),
        ('\u1cf5', '\u1cf6'), ('\u1d00', '\u1d2b'),
        ('\u1d2c', '\u1d6a'), ('\u1d6b', '\u1d77'),
        ('\u1d78', '\u1d78'), ('\u1d79', '\u1d9a'),
        ('\u1d9b', '\u1dbf'), ('\u1e00', '\u1f15'),
        ('\u1f18', '\u1f1d'), ('\u1f20', '\u1f45'),
        ('\u1f48', '\u1f4d'), ('\u1f50', '\u1f57'),
        ('\u1f59', '\u1f59'), ('\u1f5b', '\u1f5b'),
        ('\u1f5d', '\u1f5d'), ('\u1f5f', '\u1f7d'),
        ('\u1f80', '\u1fb4'), ('\u1fb6', '\u1fbc'),
        ('\u1fbe', '\u1fbe'), ('\u1fc2', '\u1fc4'),
        ('\u1fc6', '\u1fcc'), ('\u1fd0', '\u1fd3'),
        ('\u1fd6', '\u1fdb'), ('\u1fe0', '\u1fec'),
        ('\u1ff2', '\u1ff4'), ('\u1ff6', '\u1ffc'),
        ('\u2071', '\u2071'), ('\u207f', '\u207f'),
        ('\u2090', '\u209c'), ('\u2102', '\u2102'),
        ('\u2107', '\u2107'), ('\u210a', '\u2113'),
        ('\u2115', '\u2115'), ('\u2119', '\u211d'),
        ('\u2124', '\u2124'), ('\u2126', '\u2126'),
        ('\u2128', '\u2128'), ('\u212a', '\u212d'),
        ('\u212f', '\u2134'), ('\u2135', '\u2138'),
        ('\u2139', '\u2139'), ('\u213c', '\u213f'),
        ('\u2145', '\u2149'), ('\u214e', '\u214e'),
        ('\u2160', '\u2182'), ('\u2183', '\u2184'),
        ('\u2185', '\u2188'), ('\u24b6', '\u24e9'),
        ('\u2c00', '\u2c2e'), ('\u2c30', '\u2c5e'),
        ('\u2c60', '\u2c7b'), ('\u2c7c', '\u2c7d'),
        ('\u2c7e', '\u2ce4'), ('\u2ceb', '\u2cee'),
        ('\u2cf2', '\u2cf3'), ('\u2d00', '\u2d25'),
        ('\u2d27', '\u2d27'), ('\u2d2d', '\u2d2d'),
        ('\u2d30', '\u2d67'), ('\u2d6f', '\u2d6f'),
        ('\u2d80', '\u2d96'), ('\u2da0', '\u2da6'),
        ('\u2da8', '\u2dae'), ('\u2db0', '\u2db6'),
        ('\u2db8', '\u2dbe'), ('\u2dc0', '\u2dc6'),
        ('\u2dc8', '\u2dce'), ('\u2dd0', '\u2dd6'),
        ('\u2dd8', '\u2dde'), ('\u2de0', '\u2dff'),
        ('\u2e2f', '\u2e2f'), ('\u3005', '\u3005'),
        ('\u3006', '\u3006'), ('\u3007', '\u3007'),
        ('\u3021', '\u3029'), ('\u3031', '\u3035'),
        ('\u3038', '\u303a'), ('\u303b', '\u303b'),
        ('\u303c', '\u303c'), ('\u3041', '\u3096'),
        ('\u309d', '\u309e'), ('\u309f', '\u309f'),
        ('\u30a1', '\u30fa'), ('\u30fc', '\u30fe'),
        ('\u30ff', '\u30ff'), ('\u3105', '\u312d'),
        ('\u3131', '\u318e'), ('\u31a0', '\u31ba'),
        ('\u31f0', '\u31ff'), ('\u3400', '\u4db5'),
        ('\u4e00', '\u9fcc'), ('\ua000', '\ua014'),
        ('\ua015', '\ua015'), ('\ua016', '\ua48c'),
        ('\ua4d0', '\ua4f7'), ('\ua4f8', '\ua4fd'),
        ('\ua500', '\ua60b'), ('\ua60c', '\ua60c'),
        ('\ua610', '\ua61f'), ('\ua62a', '\ua62b'),
        ('\ua640', '\ua66d'), ('\ua66e', '\ua66e'),
        ('\ua674', '\ua67b'), ('\ua67f', '\ua67f'),
        ('\ua680', '\ua697'), ('\ua69f', '\ua69f'),
        ('\ua6a0', '\ua6e5'), ('\ua6e6', '\ua6ef'),
        ('\ua717', '\ua71f'), ('\ua722', '\ua76f'),
        ('\ua770', '\ua770'), ('\ua771', '\ua787'),
        ('\ua788', '\ua788'), ('\ua78b', '\ua78e'),
        ('\ua790', '\ua793'), ('\ua7a0', '\ua7aa'),
        ('\ua7f8', '\ua7f9'), ('\ua7fa', '\ua7fa'),
        ('\ua7fb', '\ua801'), ('\ua803', '\ua805'),
        ('\ua807', '\ua80a'), ('\ua80c', '\ua822'),
        ('\ua823', '\ua824'), ('\ua825', '\ua826'),
        ('\ua827', '\ua827'), ('\ua840', '\ua873'),
        ('\ua880', '\ua881'), ('\ua882', '\ua8b3'),
        ('\ua8b4', '\ua8c3'), ('\ua8f2', '\ua8f7'),
        ('\ua8fb', '\ua8fb'), ('\ua90a', '\ua925'),
        ('\ua926', '\ua92a'), ('\ua930', '\ua946'),
        ('\ua947', '\ua951'), ('\ua952', '\ua952'),
        ('\ua960', '\ua97c'), ('\ua980', '\ua982'),
        ('\ua983', '\ua983'), ('\ua984', '\ua9b2'),
        ('\ua9b4', '\ua9b5'), ('\ua9b6', '\ua9b9'),
        ('\ua9ba', '\ua9bb'), ('\ua9bc', '\ua9bc'),
        ('\ua9bd', '\ua9bf'), ('\ua9cf', '\ua9cf'),
        ('\uaa00', '\uaa28'), ('\uaa29', '\uaa2e'),
        ('\uaa2f', '\uaa30'), ('\uaa31', '\uaa32'),
        ('\uaa33', '\uaa34'), ('\uaa35', '\uaa36'),
        ('\uaa40', '\uaa42'), ('\uaa43', '\uaa43'),
        ('\uaa44', '\uaa4b'), ('\uaa4c', '\uaa4c'),
        ('\uaa4d', '\uaa4d'), ('\uaa60', '\uaa6f'),
        ('\uaa70', '\uaa70'), ('\uaa71', '\uaa76'),
        ('\uaa7a', '\uaa7a'), ('\uaa80', '\uaaaf'),
        ('\uaab0', '\uaab0'), ('\uaab1', '\uaab1'),
        ('\uaab2', '\uaab4'), ('\uaab5', '\uaab6'),
        ('\uaab7', '\uaab8'), ('\uaab9', '\uaabd'),
        ('\uaabe', '\uaabe'), ('\uaac0', '\uaac0'),
        ('\uaac2', '\uaac2'), ('\uaadb', '\uaadc'),
        ('\uaadd', '\uaadd'), ('\uaae0', '\uaaea'),
        ('\uaaeb', '\uaaeb'), ('\uaaec', '\uaaed'),
        ('\uaaee', '\uaaef'), ('\uaaf2', '\uaaf2'),
        ('\uaaf3', '\uaaf4'), ('\uaaf5', '\uaaf5'),
        ('\uab01', '\uab06'), ('\uab09', '\uab0e'),
        ('\uab11', '\uab16'), ('\uab20', '\uab26'),
        ('\uab28', '\uab2e'), ('\uabc0', '\uabe2'),
        ('\uabe3', '\uabe4'), ('\uabe5', '\uabe5'),
        ('\uabe6', '\uabe7'), ('\uabe8', '\uabe8'),
        ('\uabe9', '\uabea'), ('\uac00', '\ud7a3'),
        ('\ud7b0', '\ud7c6'), ('\ud7cb', '\ud7fb'),
        ('\uf900', '\ufa6d'), ('\ufa70', '\ufad9'),
        ('\ufb00', '\ufb06'), ('\ufb13', '\ufb17'),
        ('\ufb1d', '\ufb1d'), ('\ufb1e', '\ufb1e'),
        ('\ufb1f', '\ufb28'), ('\ufb2a', '\ufb36'),
        ('\ufb38', '\ufb3c'), ('\ufb3e', '\ufb3e'),
        ('\ufb40', '\ufb41'), ('\ufb43', '\ufb44'),
        ('\ufb46', '\ufbb1'), ('\ufbd3', '\ufd3d'),
        ('\ufd50', '\ufd8f'), ('\ufd92', '\ufdc7'),
        ('\ufdf0', '\ufdfb'), ('\ufe70', '\ufe74'),
        ('\ufe76', '\ufefc'), ('\uff21', '\uff3a'),
        ('\uff41', '\uff5a'), ('\uff66', '\uff6f'),
        ('\uff70', '\uff70'), ('\uff71', '\uff9d'),
        ('\uff9e', '\uff9f'), ('\uffa0', '\uffbe'),
        ('\uffc2', '\uffc7'), ('\uffca', '\uffcf'),
        ('\uffd2', '\uffd7'), ('\uffda', '\uffdc'),
        ('\U00010000', '\U0001000b'), ('\U0001000d', '\U00010026'),
        ('\U00010028', '\U0001003a'), ('\U0001003c', '\U0001003d'),
        ('\U0001003f', '\U0001004d'), ('\U00010050', '\U0001005d'),
        ('\U00010080', '\U000100fa'), ('\U00010140', '\U00010174'),
        ('\U00010280', '\U0001029c'), ('\U000102a0', '\U000102d0'),
        ('\U00010300', '\U0001031e'), ('\U00010330', '\U00010340'),
        ('\U00010341', '\U00010341'), ('\U00010342', '\U00010349'),
        ('\U0001034a', '\U0001034a'), ('\U00010380', '\U0001039d'),
        ('\U000103a0', '\U000103c3'), ('\U000103c8', '\U000103cf'),
        ('\U000103d1', '\U000103d5'), ('\U00010400', '\U0001044f'),
        ('\U00010450', '\U0001049d'), ('\U00010800', '\U00010805'),
        ('\U00010808', '\U00010808'), ('\U0001080a', '\U00010835'),
        ('\U00010837', '\U00010838'), ('\U0001083c', '\U0001083c'),
        ('\U0001083f', '\U00010855'), ('\U00010900', '\U00010915'),
        ('\U00010920', '\U00010939'), ('\U00010980', '\U000109b7'),
        ('\U000109be', '\U000109bf'), ('\U00010a00', '\U00010a00'),
        ('\U00010a01', '\U00010a03'), ('\U00010a05', '\U00010a06'),
        ('\U00010a0c', '\U00010a0f'), ('\U00010a10', '\U00010a13'),
        ('\U00010a15', '\U00010a17'), ('\U00010a19', '\U00010a33'),
        ('\U00010a60', '\U00010a7c'), ('\U00010b00', '\U00010b35'),
        ('\U00010b40', '\U00010b55'), ('\U00010b60', '\U00010b72'),
        ('\U00010c00', '\U00010c48'), ('\U00011000', '\U00011000'),
        ('\U00011001', '\U00011001'), ('\U00011002', '\U00011002'),
        ('\U00011003', '\U00011037'), ('\U00011038', '\U00011045'),
        ('\U00011082', '\U00011082'), ('\U00011083', '\U000110af'),
        ('\U000110b0', '\U000110b2'), ('\U000110b3', '\U000110b6'),
        ('\U000110b7', '\U000110b8'), ('\U000110d0', '\U000110e8'),
        ('\U00011100', '\U00011102'), ('\U00011103', '\U00011126'),
        ('\U00011127', '\U0001112b'), ('\U0001112c', '\U0001112c'),
        ('\U0001112d', '\U00011132'), ('\U00011180', '\U00011181'),
        ('\U00011182', '\U00011182'), ('\U00011183', '\U000111b2'),
        ('\U000111b3', '\U000111b5'), ('\U000111b6', '\U000111be'),
        ('\U000111bf', '\U000111bf'), ('\U000111c1', '\U000111c4'),
        ('\U00011680', '\U000116aa'), ('\U000116ab', '\U000116ab'),
        ('\U000116ac', '\U000116ac'), ('\U000116ad', '\U000116ad'),
        ('\U000116ae', '\U000116af'), ('\U000116b0', '\U000116b5'),
        ('\U00012000', '\U0001236e'), ('\U00012400', '\U00012462'),
        ('\U00013000', '\U0001342e'), ('\U00016800', '\U00016a38'),
        ('\U00016f00', '\U00016f44'), ('\U00016f50', '\U00016f50'),
        ('\U00016f51', '\U00016f7e'), ('\U00016f93', '\U00016f9f'),
        ('\U0001b000', '\U0001b001'), ('\U0001d400', '\U0001d454'),
        ('\U0001d456', '\U0001d49c'), ('\U0001d49e', '\U0001d49f'),
        ('\U0001d4a2', '\U0001d4a2'), ('\U0001d4a5', '\U0001d4a6'),
        ('\U0001d4a9', '\U0001d4ac'), ('\U0001d4ae', '\U0001d4b9'),
        ('\U0001d4bb', '\U0001d4bb'), ('\U0001d4bd', '\U0001d4c3'),
        ('\U0001d4c5', '\U0001d505'), ('\U0001d507', '\U0001d50a'),
        ('\U0001d50d', '\U0001d514'), ('\U0001d516', '\U0001d51c'),
        ('\U0001d51e', '\U0001d539'), ('\U0001d53b', '\U0001d53e'),
        ('\U0001d540', '\U0001d544'), ('\U0001d546', '\U0001d546'),
        ('\U0001d54a', '\U0001d550'), ('\U0001d552', '\U0001d6a5'),
        ('\U0001d6a8', '\U0001d6c0'), ('\U0001d6c2', '\U0001d6da'),
        ('\U0001d6dc', '\U0001d6fa'), ('\U0001d6fc', '\U0001d714'),
        ('\U0001d716', '\U0001d734'), ('\U0001d736', '\U0001d74e'),
        ('\U0001d750', '\U0001d76e'), ('\U0001d770', '\U0001d788'),
        ('\U0001d78a', '\U0001d7a8'), ('\U0001d7aa', '\U0001d7c2'),
        ('\U0001d7c4', '\U0001d7cb'), ('\U0001ee00', '\U0001ee03'),
        ('\U0001ee05', '\U0001ee1f'), ('\U0001ee21', '\U0001ee22'),
        ('\U0001ee24', '\U0001ee24'), ('\U0001ee27', '\U0001ee27'),
        ('\U0001ee29', '\U0001ee32'), ('\U0001ee34', '\U0001ee37'),
        ('\U0001ee39', '\U0001ee39'), ('\U0001ee3b', '\U0001ee3b'),
        ('\U0001ee42', '\U0001ee42'), ('\U0001ee47', '\U0001ee47'),
        ('\U0001ee49', '\U0001ee49'), ('\U0001ee4b', '\U0001ee4b'),
        ('\U0001ee4d', '\U0001ee4f'), ('\U0001ee51', '\U0001ee52'),
        ('\U0001ee54', '\U0001ee54'), ('\U0001ee57', '\U0001ee57'),
        ('\U0001ee59', '\U0001ee59'), ('\U0001ee5b', '\U0001ee5b'),
        ('\U0001ee5d', '\U0001ee5d'), ('\U0001ee5f', '\U0001ee5f'),
        ('\U0001ee61', '\U0001ee62'), ('\U0001ee64', '\U0001ee64'),
        ('\U0001ee67', '\U0001ee6a'), ('\U0001ee6c', '\U0001ee72'),
        ('\U0001ee74', '\U0001ee77'), ('\U0001ee79', '\U0001ee7c'),
        ('\U0001ee7e', '\U0001ee7e'), ('\U0001ee80', '\U0001ee89'),
        ('\U0001ee8b', '\U0001ee9b'), ('\U0001eea1', '\U0001eea3'),
        ('\U0001eea5', '\U0001eea9'), ('\U0001eeab', '\U0001eebb'),
        ('\U00020000', '\U0002a6d6'), ('\U0002a700', '\U0002b734'),
        ('\U0002b740', '\U0002b81d'), ('\U0002f800', '\U0002fa1d')
    ];

    pub fn Alphabetic(c: char) -> bool {
        super::bsearch_range_table(c, Alphabetic_table)
    }

    static Lowercase_table : &'static [(char,char)] = &[
        ('\x61', '\x7a'), ('\xaa', '\xaa'),
        ('\xb5', '\xb5'), ('\xba', '\xba'),
        ('\xdf', '\xf6'), ('\xf8', '\xff'),
        ('\u0101', '\u0101'), ('\u0103', '\u0103'),
        ('\u0105', '\u0105'), ('\u0107', '\u0107'),
        ('\u0109', '\u0109'), ('\u010b', '\u010b'),
        ('\u010d', '\u010d'), ('\u010f', '\u010f'),
        ('\u0111', '\u0111'), ('\u0113', '\u0113'),
        ('\u0115', '\u0115'), ('\u0117', '\u0117'),
        ('\u0119', '\u0119'), ('\u011b', '\u011b'),
        ('\u011d', '\u011d'), ('\u011f', '\u011f'),
        ('\u0121', '\u0121'), ('\u0123', '\u0123'),
        ('\u0125', '\u0125'), ('\u0127', '\u0127'),
        ('\u0129', '\u0129'), ('\u012b', '\u012b'),
        ('\u012d', '\u012d'), ('\u012f', '\u012f'),
        ('\u0131', '\u0131'), ('\u0133', '\u0133'),
        ('\u0135', '\u0135'), ('\u0137', '\u0138'),
        ('\u013a', '\u013a'), ('\u013c', '\u013c'),
        ('\u013e', '\u013e'), ('\u0140', '\u0140'),
        ('\u0142', '\u0142'), ('\u0144', '\u0144'),
        ('\u0146', '\u0146'), ('\u0148', '\u0149'),
        ('\u014b', '\u014b'), ('\u014d', '\u014d'),
        ('\u014f', '\u014f'), ('\u0151', '\u0151'),
        ('\u0153', '\u0153'), ('\u0155', '\u0155'),
        ('\u0157', '\u0157'), ('\u0159', '\u0159'),
        ('\u015b', '\u015b'), ('\u015d', '\u015d'),
        ('\u015f', '\u015f'), ('\u0161', '\u0161'),
        ('\u0163', '\u0163'), ('\u0165', '\u0165'),
        ('\u0167', '\u0167'), ('\u0169', '\u0169'),
        ('\u016b', '\u016b'), ('\u016d', '\u016d'),
        ('\u016f', '\u016f'), ('\u0171', '\u0171'),
        ('\u0173', '\u0173'), ('\u0175', '\u0175'),
        ('\u0177', '\u0177'), ('\u017a', '\u017a'),
        ('\u017c', '\u017c'), ('\u017e', '\u0180'),
        ('\u0183', '\u0183'), ('\u0185', '\u0185'),
        ('\u0188', '\u0188'), ('\u018c', '\u018d'),
        ('\u0192', '\u0192'), ('\u0195', '\u0195'),
        ('\u0199', '\u019b'), ('\u019e', '\u019e'),
        ('\u01a1', '\u01a1'), ('\u01a3', '\u01a3'),
        ('\u01a5', '\u01a5'), ('\u01a8', '\u01a8'),
        ('\u01aa', '\u01ab'), ('\u01ad', '\u01ad'),
        ('\u01b0', '\u01b0'), ('\u01b4', '\u01b4'),
        ('\u01b6', '\u01b6'), ('\u01b9', '\u01ba'),
        ('\u01bd', '\u01bf'), ('\u01c6', '\u01c6'),
        ('\u01c9', '\u01c9'), ('\u01cc', '\u01cc'),
        ('\u01ce', '\u01ce'), ('\u01d0', '\u01d0'),
        ('\u01d2', '\u01d2'), ('\u01d4', '\u01d4'),
        ('\u01d6', '\u01d6'), ('\u01d8', '\u01d8'),
        ('\u01da', '\u01da'), ('\u01dc', '\u01dd'),
        ('\u01df', '\u01df'), ('\u01e1', '\u01e1'),
        ('\u01e3', '\u01e3'), ('\u01e5', '\u01e5'),
        ('\u01e7', '\u01e7'), ('\u01e9', '\u01e9'),
        ('\u01eb', '\u01eb'), ('\u01ed', '\u01ed'),
        ('\u01ef', '\u01f0'), ('\u01f3', '\u01f3'),
        ('\u01f5', '\u01f5'), ('\u01f9', '\u01f9'),
        ('\u01fb', '\u01fb'), ('\u01fd', '\u01fd'),
        ('\u01ff', '\u01ff'), ('\u0201', '\u0201'),
        ('\u0203', '\u0203'), ('\u0205', '\u0205'),
        ('\u0207', '\u0207'), ('\u0209', '\u0209'),
        ('\u020b', '\u020b'), ('\u020d', '\u020d'),
        ('\u020f', '\u020f'), ('\u0211', '\u0211'),
        ('\u0213', '\u0213'), ('\u0215', '\u0215'),
        ('\u0217', '\u0217'), ('\u0219', '\u0219'),
        ('\u021b', '\u021b'), ('\u021d', '\u021d'),
        ('\u021f', '\u021f'), ('\u0221', '\u0221'),
        ('\u0223', '\u0223'), ('\u0225', '\u0225'),
        ('\u0227', '\u0227'), ('\u0229', '\u0229'),
        ('\u022b', '\u022b'), ('\u022d', '\u022d'),
        ('\u022f', '\u022f'), ('\u0231', '\u0231'),
        ('\u0233', '\u0239'), ('\u023c', '\u023c'),
        ('\u023f', '\u0240'), ('\u0242', '\u0242'),
        ('\u0247', '\u0247'), ('\u0249', '\u0249'),
        ('\u024b', '\u024b'), ('\u024d', '\u024d'),
        ('\u024f', '\u0293'), ('\u0295', '\u02af'),
        ('\u02b0', '\u02b8'), ('\u02c0', '\u02c1'),
        ('\u02e0', '\u02e4'), ('\u0345', '\u0345'),
        ('\u0371', '\u0371'), ('\u0373', '\u0373'),
        ('\u0377', '\u0377'), ('\u037a', '\u037a'),
        ('\u037b', '\u037d'), ('\u0390', '\u0390'),
        ('\u03ac', '\u03ce'), ('\u03d0', '\u03d1'),
        ('\u03d5', '\u03d7'), ('\u03d9', '\u03d9'),
        ('\u03db', '\u03db'), ('\u03dd', '\u03dd'),
        ('\u03df', '\u03df'), ('\u03e1', '\u03e1'),
        ('\u03e3', '\u03e3'), ('\u03e5', '\u03e5'),
        ('\u03e7', '\u03e7'), ('\u03e9', '\u03e9'),
        ('\u03eb', '\u03eb'), ('\u03ed', '\u03ed'),
        ('\u03ef', '\u03f3'), ('\u03f5', '\u03f5'),
        ('\u03f8', '\u03f8'), ('\u03fb', '\u03fc'),
        ('\u0430', '\u045f'), ('\u0461', '\u0461'),
        ('\u0463', '\u0463'), ('\u0465', '\u0465'),
        ('\u0467', '\u0467'), ('\u0469', '\u0469'),
        ('\u046b', '\u046b'), ('\u046d', '\u046d'),
        ('\u046f', '\u046f'), ('\u0471', '\u0471'),
        ('\u0473', '\u0473'), ('\u0475', '\u0475'),
        ('\u0477', '\u0477'), ('\u0479', '\u0479'),
        ('\u047b', '\u047b'), ('\u047d', '\u047d'),
        ('\u047f', '\u047f'), ('\u0481', '\u0481'),
        ('\u048b', '\u048b'), ('\u048d', '\u048d'),
        ('\u048f', '\u048f'), ('\u0491', '\u0491'),
        ('\u0493', '\u0493'), ('\u0495', '\u0495'),
        ('\u0497', '\u0497'), ('\u0499', '\u0499'),
        ('\u049b', '\u049b'), ('\u049d', '\u049d'),
        ('\u049f', '\u049f'), ('\u04a1', '\u04a1'),
        ('\u04a3', '\u04a3'), ('\u04a5', '\u04a5'),
        ('\u04a7', '\u04a7'), ('\u04a9', '\u04a9'),
        ('\u04ab', '\u04ab'), ('\u04ad', '\u04ad'),
        ('\u04af', '\u04af'), ('\u04b1', '\u04b1'),
        ('\u04b3', '\u04b3'), ('\u04b5', '\u04b5'),
        ('\u04b7', '\u04b7'), ('\u04b9', '\u04b9'),
        ('\u04bb', '\u04bb'), ('\u04bd', '\u04bd'),
        ('\u04bf', '\u04bf'), ('\u04c2', '\u04c2'),
        ('\u04c4', '\u04c4'), ('\u04c6', '\u04c6'),
        ('\u04c8', '\u04c8'), ('\u04ca', '\u04ca'),
        ('\u04cc', '\u04cc'), ('\u04ce', '\u04cf'),
        ('\u04d1', '\u04d1'), ('\u04d3', '\u04d3'),
        ('\u04d5', '\u04d5'), ('\u04d7', '\u04d7'),
        ('\u04d9', '\u04d9'), ('\u04db', '\u04db'),
        ('\u04dd', '\u04dd'), ('\u04df', '\u04df'),
        ('\u04e1', '\u04e1'), ('\u04e3', '\u04e3'),
        ('\u04e5', '\u04e5'), ('\u04e7', '\u04e7'),
        ('\u04e9', '\u04e9'), ('\u04eb', '\u04eb'),
        ('\u04ed', '\u04ed'), ('\u04ef', '\u04ef'),
        ('\u04f1', '\u04f1'), ('\u04f3', '\u04f3'),
        ('\u04f5', '\u04f5'), ('\u04f7', '\u04f7'),
        ('\u04f9', '\u04f9'), ('\u04fb', '\u04fb'),
        ('\u04fd', '\u04fd'), ('\u04ff', '\u04ff'),
        ('\u0501', '\u0501'), ('\u0503', '\u0503'),
        ('\u0505', '\u0505'), ('\u0507', '\u0507'),
        ('\u0509', '\u0509'), ('\u050b', '\u050b'),
        ('\u050d', '\u050d'), ('\u050f', '\u050f'),
        ('\u0511', '\u0511'), ('\u0513', '\u0513'),
        ('\u0515', '\u0515'), ('\u0517', '\u0517'),
        ('\u0519', '\u0519'), ('\u051b', '\u051b'),
        ('\u051d', '\u051d'), ('\u051f', '\u051f'),
        ('\u0521', '\u0521'), ('\u0523', '\u0523'),
        ('\u0525', '\u0525'), ('\u0527', '\u0527'),
        ('\u0561', '\u0587'), ('\u1d00', '\u1d2b'),
        ('\u1d2c', '\u1d6a'), ('\u1d6b', '\u1d77'),
        ('\u1d78', '\u1d78'), ('\u1d79', '\u1d9a'),
        ('\u1d9b', '\u1dbf'), ('\u1e01', '\u1e01'),
        ('\u1e03', '\u1e03'), ('\u1e05', '\u1e05'),
        ('\u1e07', '\u1e07'), ('\u1e09', '\u1e09'),
        ('\u1e0b', '\u1e0b'), ('\u1e0d', '\u1e0d'),
        ('\u1e0f', '\u1e0f'), ('\u1e11', '\u1e11'),
        ('\u1e13', '\u1e13'), ('\u1e15', '\u1e15'),
        ('\u1e17', '\u1e17'), ('\u1e19', '\u1e19'),
        ('\u1e1b', '\u1e1b'), ('\u1e1d', '\u1e1d'),
        ('\u1e1f', '\u1e1f'), ('\u1e21', '\u1e21'),
        ('\u1e23', '\u1e23'), ('\u1e25', '\u1e25'),
        ('\u1e27', '\u1e27'), ('\u1e29', '\u1e29'),
        ('\u1e2b', '\u1e2b'), ('\u1e2d', '\u1e2d'),
        ('\u1e2f', '\u1e2f'), ('\u1e31', '\u1e31'),
        ('\u1e33', '\u1e33'), ('\u1e35', '\u1e35'),
        ('\u1e37', '\u1e37'), ('\u1e39', '\u1e39'),
        ('\u1e3b', '\u1e3b'), ('\u1e3d', '\u1e3d'),
        ('\u1e3f', '\u1e3f'), ('\u1e41', '\u1e41'),
        ('\u1e43', '\u1e43'), ('\u1e45', '\u1e45'),
        ('\u1e47', '\u1e47'), ('\u1e49', '\u1e49'),
        ('\u1e4b', '\u1e4b'), ('\u1e4d', '\u1e4d'),
        ('\u1e4f', '\u1e4f'), ('\u1e51', '\u1e51'),
        ('\u1e53', '\u1e53'), ('\u1e55', '\u1e55'),
        ('\u1e57', '\u1e57'), ('\u1e59', '\u1e59'),
        ('\u1e5b', '\u1e5b'), ('\u1e5d', '\u1e5d'),
        ('\u1e5f', '\u1e5f'), ('\u1e61', '\u1e61'),
        ('\u1e63', '\u1e63'), ('\u1e65', '\u1e65'),
        ('\u1e67', '\u1e67'), ('\u1e69', '\u1e69'),
        ('\u1e6b', '\u1e6b'), ('\u1e6d', '\u1e6d'),
        ('\u1e6f', '\u1e6f'), ('\u1e71', '\u1e71'),
        ('\u1e73', '\u1e73'), ('\u1e75', '\u1e75'),
        ('\u1e77', '\u1e77'), ('\u1e79', '\u1e79'),
        ('\u1e7b', '\u1e7b'), ('\u1e7d', '\u1e7d'),
        ('\u1e7f', '\u1e7f'), ('\u1e81', '\u1e81'),
        ('\u1e83', '\u1e83'), ('\u1e85', '\u1e85'),
        ('\u1e87', '\u1e87'), ('\u1e89', '\u1e89'),
        ('\u1e8b', '\u1e8b'), ('\u1e8d', '\u1e8d'),
        ('\u1e8f', '\u1e8f'), ('\u1e91', '\u1e91'),
        ('\u1e93', '\u1e93'), ('\u1e95', '\u1e9d'),
        ('\u1e9f', '\u1e9f'), ('\u1ea1', '\u1ea1'),
        ('\u1ea3', '\u1ea3'), ('\u1ea5', '\u1ea5'),
        ('\u1ea7', '\u1ea7'), ('\u1ea9', '\u1ea9'),
        ('\u1eab', '\u1eab'), ('\u1ead', '\u1ead'),
        ('\u1eaf', '\u1eaf'), ('\u1eb1', '\u1eb1'),
        ('\u1eb3', '\u1eb3'), ('\u1eb5', '\u1eb5'),
        ('\u1eb7', '\u1eb7'), ('\u1eb9', '\u1eb9'),
        ('\u1ebb', '\u1ebb'), ('\u1ebd', '\u1ebd'),
        ('\u1ebf', '\u1ebf'), ('\u1ec1', '\u1ec1'),
        ('\u1ec3', '\u1ec3'), ('\u1ec5', '\u1ec5'),
        ('\u1ec7', '\u1ec7'), ('\u1ec9', '\u1ec9'),
        ('\u1ecb', '\u1ecb'), ('\u1ecd', '\u1ecd'),
        ('\u1ecf', '\u1ecf'), ('\u1ed1', '\u1ed1'),
        ('\u1ed3', '\u1ed3'), ('\u1ed5', '\u1ed5'),
        ('\u1ed7', '\u1ed7'), ('\u1ed9', '\u1ed9'),
        ('\u1edb', '\u1edb'), ('\u1edd', '\u1edd'),
        ('\u1edf', '\u1edf'), ('\u1ee1', '\u1ee1'),
        ('\u1ee3', '\u1ee3'), ('\u1ee5', '\u1ee5'),
        ('\u1ee7', '\u1ee7'), ('\u1ee9', '\u1ee9'),
        ('\u1eeb', '\u1eeb'), ('\u1eed', '\u1eed'),
        ('\u1eef', '\u1eef'), ('\u1ef1', '\u1ef1'),
        ('\u1ef3', '\u1ef3'), ('\u1ef5', '\u1ef5'),
        ('\u1ef7', '\u1ef7'), ('\u1ef9', '\u1ef9'),
        ('\u1efb', '\u1efb'), ('\u1efd', '\u1efd'),
        ('\u1eff', '\u1f07'), ('\u1f10', '\u1f15'),
        ('\u1f20', '\u1f27'), ('\u1f30', '\u1f37'),
        ('\u1f40', '\u1f45'), ('\u1f50', '\u1f57'),
        ('\u1f60', '\u1f67'), ('\u1f70', '\u1f7d'),
        ('\u1f80', '\u1f87'), ('\u1f90', '\u1f97'),
        ('\u1fa0', '\u1fa7'), ('\u1fb0', '\u1fb4'),
        ('\u1fb6', '\u1fb7'), ('\u1fbe', '\u1fbe'),
        ('\u1fc2', '\u1fc4'), ('\u1fc6', '\u1fc7'),
        ('\u1fd0', '\u1fd3'), ('\u1fd6', '\u1fd7'),
        ('\u1fe0', '\u1fe7'), ('\u1ff2', '\u1ff4'),
        ('\u1ff6', '\u1ff7'), ('\u2071', '\u2071'),
        ('\u207f', '\u207f'), ('\u2090', '\u209c'),
        ('\u210a', '\u210a'), ('\u210e', '\u210f'),
        ('\u2113', '\u2113'), ('\u212f', '\u212f'),
        ('\u2134', '\u2134'), ('\u2139', '\u2139'),
        ('\u213c', '\u213d'), ('\u2146', '\u2149'),
        ('\u214e', '\u214e'), ('\u2170', '\u217f'),
        ('\u2184', '\u2184'), ('\u24d0', '\u24e9'),
        ('\u2c30', '\u2c5e'), ('\u2c61', '\u2c61'),
        ('\u2c65', '\u2c66'), ('\u2c68', '\u2c68'),
        ('\u2c6a', '\u2c6a'), ('\u2c6c', '\u2c6c'),
        ('\u2c71', '\u2c71'), ('\u2c73', '\u2c74'),
        ('\u2c76', '\u2c7b'), ('\u2c7c', '\u2c7d'),
        ('\u2c81', '\u2c81'), ('\u2c83', '\u2c83'),
        ('\u2c85', '\u2c85'), ('\u2c87', '\u2c87'),
        ('\u2c89', '\u2c89'), ('\u2c8b', '\u2c8b'),
        ('\u2c8d', '\u2c8d'), ('\u2c8f', '\u2c8f'),
        ('\u2c91', '\u2c91'), ('\u2c93', '\u2c93'),
        ('\u2c95', '\u2c95'), ('\u2c97', '\u2c97'),
        ('\u2c99', '\u2c99'), ('\u2c9b', '\u2c9b'),
        ('\u2c9d', '\u2c9d'), ('\u2c9f', '\u2c9f'),
        ('\u2ca1', '\u2ca1'), ('\u2ca3', '\u2ca3'),
        ('\u2ca5', '\u2ca5'), ('\u2ca7', '\u2ca7'),
        ('\u2ca9', '\u2ca9'), ('\u2cab', '\u2cab'),
        ('\u2cad', '\u2cad'), ('\u2caf', '\u2caf'),
        ('\u2cb1', '\u2cb1'), ('\u2cb3', '\u2cb3'),
        ('\u2cb5', '\u2cb5'), ('\u2cb7', '\u2cb7'),
        ('\u2cb9', '\u2cb9'), ('\u2cbb', '\u2cbb'),
        ('\u2cbd', '\u2cbd'), ('\u2cbf', '\u2cbf'),
        ('\u2cc1', '\u2cc1'), ('\u2cc3', '\u2cc3'),
        ('\u2cc5', '\u2cc5'), ('\u2cc7', '\u2cc7'),
        ('\u2cc9', '\u2cc9'), ('\u2ccb', '\u2ccb'),
        ('\u2ccd', '\u2ccd'), ('\u2ccf', '\u2ccf'),
        ('\u2cd1', '\u2cd1'), ('\u2cd3', '\u2cd3'),
        ('\u2cd5', '\u2cd5'), ('\u2cd7', '\u2cd7'),
        ('\u2cd9', '\u2cd9'), ('\u2cdb', '\u2cdb'),
        ('\u2cdd', '\u2cdd'), ('\u2cdf', '\u2cdf'),
        ('\u2ce1', '\u2ce1'), ('\u2ce3', '\u2ce4'),
        ('\u2cec', '\u2cec'), ('\u2cee', '\u2cee'),
        ('\u2cf3', '\u2cf3'), ('\u2d00', '\u2d25'),
        ('\u2d27', '\u2d27'), ('\u2d2d', '\u2d2d'),
        ('\ua641', '\ua641'), ('\ua643', '\ua643'),
        ('\ua645', '\ua645'), ('\ua647', '\ua647'),
        ('\ua649', '\ua649'), ('\ua64b', '\ua64b'),
        ('\ua64d', '\ua64d'), ('\ua64f', '\ua64f'),
        ('\ua651', '\ua651'), ('\ua653', '\ua653'),
        ('\ua655', '\ua655'), ('\ua657', '\ua657'),
        ('\ua659', '\ua659'), ('\ua65b', '\ua65b'),
        ('\ua65d', '\ua65d'), ('\ua65f', '\ua65f'),
        ('\ua661', '\ua661'), ('\ua663', '\ua663'),
        ('\ua665', '\ua665'), ('\ua667', '\ua667'),
        ('\ua669', '\ua669'), ('\ua66b', '\ua66b'),
        ('\ua66d', '\ua66d'), ('\ua681', '\ua681'),
        ('\ua683', '\ua683'), ('\ua685', '\ua685'),
        ('\ua687', '\ua687'), ('\ua689', '\ua689'),
        ('\ua68b', '\ua68b'), ('\ua68d', '\ua68d'),
        ('\ua68f', '\ua68f'), ('\ua691', '\ua691'),
        ('\ua693', '\ua693'), ('\ua695', '\ua695'),
        ('\ua697', '\ua697'), ('\ua723', '\ua723'),
        ('\ua725', '\ua725'), ('\ua727', '\ua727'),
        ('\ua729', '\ua729'), ('\ua72b', '\ua72b'),
        ('\ua72d', '\ua72d'), ('\ua72f', '\ua731'),
        ('\ua733', '\ua733'), ('\ua735', '\ua735'),
        ('\ua737', '\ua737'), ('\ua739', '\ua739'),
        ('\ua73b', '\ua73b'), ('\ua73d', '\ua73d'),
        ('\ua73f', '\ua73f'), ('\ua741', '\ua741'),
        ('\ua743', '\ua743'), ('\ua745', '\ua745'),
        ('\ua747', '\ua747'), ('\ua749', '\ua749'),
        ('\ua74b', '\ua74b'), ('\ua74d', '\ua74d'),
        ('\ua74f', '\ua74f'), ('\ua751', '\ua751'),
        ('\ua753', '\ua753'), ('\ua755', '\ua755'),
        ('\ua757', '\ua757'), ('\ua759', '\ua759'),
        ('\ua75b', '\ua75b'), ('\ua75d', '\ua75d'),
        ('\ua75f', '\ua75f'), ('\ua761', '\ua761'),
        ('\ua763', '\ua763'), ('\ua765', '\ua765'),
        ('\ua767', '\ua767'), ('\ua769', '\ua769'),
        ('\ua76b', '\ua76b'), ('\ua76d', '\ua76d'),
        ('\ua76f', '\ua76f'), ('\ua770', '\ua770'),
        ('\ua771', '\ua778'), ('\ua77a', '\ua77a'),
        ('\ua77c', '\ua77c'), ('\ua77f', '\ua77f'),
        ('\ua781', '\ua781'), ('\ua783', '\ua783'),
        ('\ua785', '\ua785'), ('\ua787', '\ua787'),
        ('\ua78c', '\ua78c'), ('\ua78e', '\ua78e'),
        ('\ua791', '\ua791'), ('\ua793', '\ua793'),
        ('\ua7a1', '\ua7a1'), ('\ua7a3', '\ua7a3'),
        ('\ua7a5', '\ua7a5'), ('\ua7a7', '\ua7a7'),
        ('\ua7a9', '\ua7a9'), ('\ua7f8', '\ua7f9'),
        ('\ua7fa', '\ua7fa'), ('\ufb00', '\ufb06'),
        ('\ufb13', '\ufb17'), ('\uff41', '\uff5a'),
        ('\U00010428', '\U0001044f'), ('\U0001d41a', '\U0001d433'),
        ('\U0001d44e', '\U0001d454'), ('\U0001d456', '\U0001d467'),
        ('\U0001d482', '\U0001d49b'), ('\U0001d4b6', '\U0001d4b9'),
        ('\U0001d4bb', '\U0001d4bb'), ('\U0001d4bd', '\U0001d4c3'),
        ('\U0001d4c5', '\U0001d4cf'), ('\U0001d4ea', '\U0001d503'),
        ('\U0001d51e', '\U0001d537'), ('\U0001d552', '\U0001d56b'),
        ('\U0001d586', '\U0001d59f'), ('\U0001d5ba', '\U0001d5d3'),
        ('\U0001d5ee', '\U0001d607'), ('\U0001d622', '\U0001d63b'),
        ('\U0001d656', '\U0001d66f'), ('\U0001d68a', '\U0001d6a5'),
        ('\U0001d6c2', '\U0001d6da'), ('\U0001d6dc', '\U0001d6e1'),
        ('\U0001d6fc', '\U0001d714'), ('\U0001d716', '\U0001d71b'),
        ('\U0001d736', '\U0001d74e'), ('\U0001d750', '\U0001d755'),
        ('\U0001d770', '\U0001d788'), ('\U0001d78a', '\U0001d78f'),
        ('\U0001d7aa', '\U0001d7c2'), ('\U0001d7c4', '\U0001d7c9'),
        ('\U0001d7cb', '\U0001d7cb')
    ];

    pub fn Lowercase(c: char) -> bool {
        super::bsearch_range_table(c, Lowercase_table)
    }

    static Uppercase_table : &'static [(char,char)] = &[
        ('\x41', '\x5a'), ('\xc0', '\xd6'),
        ('\xd8', '\xde'), ('\u0100', '\u0100'),
        ('\u0102', '\u0102'), ('\u0104', '\u0104'),
        ('\u0106', '\u0106'), ('\u0108', '\u0108'),
        ('\u010a', '\u010a'), ('\u010c', '\u010c'),
        ('\u010e', '\u010e'), ('\u0110', '\u0110'),
        ('\u0112', '\u0112'), ('\u0114', '\u0114'),
        ('\u0116', '\u0116'), ('\u0118', '\u0118'),
        ('\u011a', '\u011a'), ('\u011c', '\u011c'),
        ('\u011e', '\u011e'), ('\u0120', '\u0120'),
        ('\u0122', '\u0122'), ('\u0124', '\u0124'),
        ('\u0126', '\u0126'), ('\u0128', '\u0128'),
        ('\u012a', '\u012a'), ('\u012c', '\u012c'),
        ('\u012e', '\u012e'), ('\u0130', '\u0130'),
        ('\u0132', '\u0132'), ('\u0134', '\u0134'),
        ('\u0136', '\u0136'), ('\u0139', '\u0139'),
        ('\u013b', '\u013b'), ('\u013d', '\u013d'),
        ('\u013f', '\u013f'), ('\u0141', '\u0141'),
        ('\u0143', '\u0143'), ('\u0145', '\u0145'),
        ('\u0147', '\u0147'), ('\u014a', '\u014a'),
        ('\u014c', '\u014c'), ('\u014e', '\u014e'),
        ('\u0150', '\u0150'), ('\u0152', '\u0152'),
        ('\u0154', '\u0154'), ('\u0156', '\u0156'),
        ('\u0158', '\u0158'), ('\u015a', '\u015a'),
        ('\u015c', '\u015c'), ('\u015e', '\u015e'),
        ('\u0160', '\u0160'), ('\u0162', '\u0162'),
        ('\u0164', '\u0164'), ('\u0166', '\u0166'),
        ('\u0168', '\u0168'), ('\u016a', '\u016a'),
        ('\u016c', '\u016c'), ('\u016e', '\u016e'),
        ('\u0170', '\u0170'), ('\u0172', '\u0172'),
        ('\u0174', '\u0174'), ('\u0176', '\u0176'),
        ('\u0178', '\u0179'), ('\u017b', '\u017b'),
        ('\u017d', '\u017d'), ('\u0181', '\u0182'),
        ('\u0184', '\u0184'), ('\u0186', '\u0187'),
        ('\u0189', '\u018b'), ('\u018e', '\u0191'),
        ('\u0193', '\u0194'), ('\u0196', '\u0198'),
        ('\u019c', '\u019d'), ('\u019f', '\u01a0'),
        ('\u01a2', '\u01a2'), ('\u01a4', '\u01a4'),
        ('\u01a6', '\u01a7'), ('\u01a9', '\u01a9'),
        ('\u01ac', '\u01ac'), ('\u01ae', '\u01af'),
        ('\u01b1', '\u01b3'), ('\u01b5', '\u01b5'),
        ('\u01b7', '\u01b8'), ('\u01bc', '\u01bc'),
        ('\u01c4', '\u01c4'), ('\u01c7', '\u01c7'),
        ('\u01ca', '\u01ca'), ('\u01cd', '\u01cd'),
        ('\u01cf', '\u01cf'), ('\u01d1', '\u01d1'),
        ('\u01d3', '\u01d3'), ('\u01d5', '\u01d5'),
        ('\u01d7', '\u01d7'), ('\u01d9', '\u01d9'),
        ('\u01db', '\u01db'), ('\u01de', '\u01de'),
        ('\u01e0', '\u01e0'), ('\u01e2', '\u01e2'),
        ('\u01e4', '\u01e4'), ('\u01e6', '\u01e6'),
        ('\u01e8', '\u01e8'), ('\u01ea', '\u01ea'),
        ('\u01ec', '\u01ec'), ('\u01ee', '\u01ee'),
        ('\u01f1', '\u01f1'), ('\u01f4', '\u01f4'),
        ('\u01f6', '\u01f8'), ('\u01fa', '\u01fa'),
        ('\u01fc', '\u01fc'), ('\u01fe', '\u01fe'),
        ('\u0200', '\u0200'), ('\u0202', '\u0202'),
        ('\u0204', '\u0204'), ('\u0206', '\u0206'),
        ('\u0208', '\u0208'), ('\u020a', '\u020a'),
        ('\u020c', '\u020c'), ('\u020e', '\u020e'),
        ('\u0210', '\u0210'), ('\u0212', '\u0212'),
        ('\u0214', '\u0214'), ('\u0216', '\u0216'),
        ('\u0218', '\u0218'), ('\u021a', '\u021a'),
        ('\u021c', '\u021c'), ('\u021e', '\u021e'),
        ('\u0220', '\u0220'), ('\u0222', '\u0222'),
        ('\u0224', '\u0224'), ('\u0226', '\u0226'),
        ('\u0228', '\u0228'), ('\u022a', '\u022a'),
        ('\u022c', '\u022c'), ('\u022e', '\u022e'),
        ('\u0230', '\u0230'), ('\u0232', '\u0232'),
        ('\u023a', '\u023b'), ('\u023d', '\u023e'),
        ('\u0241', '\u0241'), ('\u0243', '\u0246'),
        ('\u0248', '\u0248'), ('\u024a', '\u024a'),
        ('\u024c', '\u024c'), ('\u024e', '\u024e'),
        ('\u0370', '\u0370'), ('\u0372', '\u0372'),
        ('\u0376', '\u0376'), ('\u0386', '\u0386'),
        ('\u0388', '\u038a'), ('\u038c', '\u038c'),
        ('\u038e', '\u038f'), ('\u0391', '\u03a1'),
        ('\u03a3', '\u03ab'), ('\u03cf', '\u03cf'),
        ('\u03d2', '\u03d4'), ('\u03d8', '\u03d8'),
        ('\u03da', '\u03da'), ('\u03dc', '\u03dc'),
        ('\u03de', '\u03de'), ('\u03e0', '\u03e0'),
        ('\u03e2', '\u03e2'), ('\u03e4', '\u03e4'),
        ('\u03e6', '\u03e6'), ('\u03e8', '\u03e8'),
        ('\u03ea', '\u03ea'), ('\u03ec', '\u03ec'),
        ('\u03ee', '\u03ee'), ('\u03f4', '\u03f4'),
        ('\u03f7', '\u03f7'), ('\u03f9', '\u03fa'),
        ('\u03fd', '\u042f'), ('\u0460', '\u0460'),
        ('\u0462', '\u0462'), ('\u0464', '\u0464'),
        ('\u0466', '\u0466'), ('\u0468', '\u0468'),
        ('\u046a', '\u046a'), ('\u046c', '\u046c'),
        ('\u046e', '\u046e'), ('\u0470', '\u0470'),
        ('\u0472', '\u0472'), ('\u0474', '\u0474'),
        ('\u0476', '\u0476'), ('\u0478', '\u0478'),
        ('\u047a', '\u047a'), ('\u047c', '\u047c'),
        ('\u047e', '\u047e'), ('\u0480', '\u0480'),
        ('\u048a', '\u048a'), ('\u048c', '\u048c'),
        ('\u048e', '\u048e'), ('\u0490', '\u0490'),
        ('\u0492', '\u0492'), ('\u0494', '\u0494'),
        ('\u0496', '\u0496'), ('\u0498', '\u0498'),
        ('\u049a', '\u049a'), ('\u049c', '\u049c'),
        ('\u049e', '\u049e'), ('\u04a0', '\u04a0'),
        ('\u04a2', '\u04a2'), ('\u04a4', '\u04a4'),
        ('\u04a6', '\u04a6'), ('\u04a8', '\u04a8'),
        ('\u04aa', '\u04aa'), ('\u04ac', '\u04ac'),
        ('\u04ae', '\u04ae'), ('\u04b0', '\u04b0'),
        ('\u04b2', '\u04b2'), ('\u04b4', '\u04b4'),
        ('\u04b6', '\u04b6'), ('\u04b8', '\u04b8'),
        ('\u04ba', '\u04ba'), ('\u04bc', '\u04bc'),
        ('\u04be', '\u04be'), ('\u04c0', '\u04c1'),
        ('\u04c3', '\u04c3'), ('\u04c5', '\u04c5'),
        ('\u04c7', '\u04c7'), ('\u04c9', '\u04c9'),
        ('\u04cb', '\u04cb'), ('\u04cd', '\u04cd'),
        ('\u04d0', '\u04d0'), ('\u04d2', '\u04d2'),
        ('\u04d4', '\u04d4'), ('\u04d6', '\u04d6'),
        ('\u04d8', '\u04d8'), ('\u04da', '\u04da'),
        ('\u04dc', '\u04dc'), ('\u04de', '\u04de'),
        ('\u04e0', '\u04e0'), ('\u04e2', '\u04e2'),
        ('\u04e4', '\u04e4'), ('\u04e6', '\u04e6'),
        ('\u04e8', '\u04e8'), ('\u04ea', '\u04ea'),
        ('\u04ec', '\u04ec'), ('\u04ee', '\u04ee'),
        ('\u04f0', '\u04f0'), ('\u04f2', '\u04f2'),
        ('\u04f4', '\u04f4'), ('\u04f6', '\u04f6'),
        ('\u04f8', '\u04f8'), ('\u04fa', '\u04fa'),
        ('\u04fc', '\u04fc'), ('\u04fe', '\u04fe'),
        ('\u0500', '\u0500'), ('\u0502', '\u0502'),
        ('\u0504', '\u0504'), ('\u0506', '\u0506'),
        ('\u0508', '\u0508'), ('\u050a', '\u050a'),
        ('\u050c', '\u050c'), ('\u050e', '\u050e'),
        ('\u0510', '\u0510'), ('\u0512', '\u0512'),
        ('\u0514', '\u0514'), ('\u0516', '\u0516'),
        ('\u0518', '\u0518'), ('\u051a', '\u051a'),
        ('\u051c', '\u051c'), ('\u051e', '\u051e'),
        ('\u0520', '\u0520'), ('\u0522', '\u0522'),
        ('\u0524', '\u0524'), ('\u0526', '\u0526'),
        ('\u0531', '\u0556'), ('\u10a0', '\u10c5'),
        ('\u10c7', '\u10c7'), ('\u10cd', '\u10cd'),
        ('\u1e00', '\u1e00'), ('\u1e02', '\u1e02'),
        ('\u1e04', '\u1e04'), ('\u1e06', '\u1e06'),
        ('\u1e08', '\u1e08'), ('\u1e0a', '\u1e0a'),
        ('\u1e0c', '\u1e0c'), ('\u1e0e', '\u1e0e'),
        ('\u1e10', '\u1e10'), ('\u1e12', '\u1e12'),
        ('\u1e14', '\u1e14'), ('\u1e16', '\u1e16'),
        ('\u1e18', '\u1e18'), ('\u1e1a', '\u1e1a'),
        ('\u1e1c', '\u1e1c'), ('\u1e1e', '\u1e1e'),
        ('\u1e20', '\u1e20'), ('\u1e22', '\u1e22'),
        ('\u1e24', '\u1e24'), ('\u1e26', '\u1e26'),
        ('\u1e28', '\u1e28'), ('\u1e2a', '\u1e2a'),
        ('\u1e2c', '\u1e2c'), ('\u1e2e', '\u1e2e'),
        ('\u1e30', '\u1e30'), ('\u1e32', '\u1e32'),
        ('\u1e34', '\u1e34'), ('\u1e36', '\u1e36'),
        ('\u1e38', '\u1e38'), ('\u1e3a', '\u1e3a'),
        ('\u1e3c', '\u1e3c'), ('\u1e3e', '\u1e3e'),
        ('\u1e40', '\u1e40'), ('\u1e42', '\u1e42'),
        ('\u1e44', '\u1e44'), ('\u1e46', '\u1e46'),
        ('\u1e48', '\u1e48'), ('\u1e4a', '\u1e4a'),
        ('\u1e4c', '\u1e4c'), ('\u1e4e', '\u1e4e'),
        ('\u1e50', '\u1e50'), ('\u1e52', '\u1e52'),
        ('\u1e54', '\u1e54'), ('\u1e56', '\u1e56'),
        ('\u1e58', '\u1e58'), ('\u1e5a', '\u1e5a'),
        ('\u1e5c', '\u1e5c'), ('\u1e5e', '\u1e5e'),
        ('\u1e60', '\u1e60'), ('\u1e62', '\u1e62'),
        ('\u1e64', '\u1e64'), ('\u1e66', '\u1e66'),
        ('\u1e68', '\u1e68'), ('\u1e6a', '\u1e6a'),
        ('\u1e6c', '\u1e6c'), ('\u1e6e', '\u1e6e'),
        ('\u1e70', '\u1e70'), ('\u1e72', '\u1e72'),
        ('\u1e74', '\u1e74'), ('\u1e76', '\u1e76'),
        ('\u1e78', '\u1e78'), ('\u1e7a', '\u1e7a'),
        ('\u1e7c', '\u1e7c'), ('\u1e7e', '\u1e7e'),
        ('\u1e80', '\u1e80'), ('\u1e82', '\u1e82'),
        ('\u1e84', '\u1e84'), ('\u1e86', '\u1e86'),
        ('\u1e88', '\u1e88'), ('\u1e8a', '\u1e8a'),
        ('\u1e8c', '\u1e8c'), ('\u1e8e', '\u1e8e'),
        ('\u1e90', '\u1e90'), ('\u1e92', '\u1e92'),
        ('\u1e94', '\u1e94'), ('\u1e9e', '\u1e9e'),
        ('\u1ea0', '\u1ea0'), ('\u1ea2', '\u1ea2'),
        ('\u1ea4', '\u1ea4'), ('\u1ea6', '\u1ea6'),
        ('\u1ea8', '\u1ea8'), ('\u1eaa', '\u1eaa'),
        ('\u1eac', '\u1eac'), ('\u1eae', '\u1eae'),
        ('\u1eb0', '\u1eb0'), ('\u1eb2', '\u1eb2'),
        ('\u1eb4', '\u1eb4'), ('\u1eb6', '\u1eb6'),
        ('\u1eb8', '\u1eb8'), ('\u1eba', '\u1eba'),
        ('\u1ebc', '\u1ebc'), ('\u1ebe', '\u1ebe'),
        ('\u1ec0', '\u1ec0'), ('\u1ec2', '\u1ec2'),
        ('\u1ec4', '\u1ec4'), ('\u1ec6', '\u1ec6'),
        ('\u1ec8', '\u1ec8'), ('\u1eca', '\u1eca'),
        ('\u1ecc', '\u1ecc'), ('\u1ece', '\u1ece'),
        ('\u1ed0', '\u1ed0'), ('\u1ed2', '\u1ed2'),
        ('\u1ed4', '\u1ed4'), ('\u1ed6', '\u1ed6'),
        ('\u1ed8', '\u1ed8'), ('\u1eda', '\u1eda'),
        ('\u1edc', '\u1edc'), ('\u1ede', '\u1ede'),
        ('\u1ee0', '\u1ee0'), ('\u1ee2', '\u1ee2'),
        ('\u1ee4', '\u1ee4'), ('\u1ee6', '\u1ee6'),
        ('\u1ee8', '\u1ee8'), ('\u1eea', '\u1eea'),
        ('\u1eec', '\u1eec'), ('\u1eee', '\u1eee'),
        ('\u1ef0', '\u1ef0'), ('\u1ef2', '\u1ef2'),
        ('\u1ef4', '\u1ef4'), ('\u1ef6', '\u1ef6'),
        ('\u1ef8', '\u1ef8'), ('\u1efa', '\u1efa'),
        ('\u1efc', '\u1efc'), ('\u1efe', '\u1efe'),
        ('\u1f08', '\u1f0f'), ('\u1f18', '\u1f1d'),
        ('\u1f28', '\u1f2f'), ('\u1f38', '\u1f3f'),
        ('\u1f48', '\u1f4d'), ('\u1f59', '\u1f59'),
        ('\u1f5b', '\u1f5b'), ('\u1f5d', '\u1f5d'),
        ('\u1f5f', '\u1f5f'), ('\u1f68', '\u1f6f'),
        ('\u1fb8', '\u1fbb'), ('\u1fc8', '\u1fcb'),
        ('\u1fd8', '\u1fdb'), ('\u1fe8', '\u1fec'),
        ('\u1ff8', '\u1ffb'), ('\u2102', '\u2102'),
        ('\u2107', '\u2107'), ('\u210b', '\u210d'),
        ('\u2110', '\u2112'), ('\u2115', '\u2115'),
        ('\u2119', '\u211d'), ('\u2124', '\u2124'),
        ('\u2126', '\u2126'), ('\u2128', '\u2128'),
        ('\u212a', '\u212d'), ('\u2130', '\u2133'),
        ('\u213e', '\u213f'), ('\u2145', '\u2145'),
        ('\u2160', '\u216f'), ('\u2183', '\u2183'),
        ('\u24b6', '\u24cf'), ('\u2c00', '\u2c2e'),
        ('\u2c60', '\u2c60'), ('\u2c62', '\u2c64'),
        ('\u2c67', '\u2c67'), ('\u2c69', '\u2c69'),
        ('\u2c6b', '\u2c6b'), ('\u2c6d', '\u2c70'),
        ('\u2c72', '\u2c72'), ('\u2c75', '\u2c75'),
        ('\u2c7e', '\u2c80'), ('\u2c82', '\u2c82'),
        ('\u2c84', '\u2c84'), ('\u2c86', '\u2c86'),
        ('\u2c88', '\u2c88'), ('\u2c8a', '\u2c8a'),
        ('\u2c8c', '\u2c8c'), ('\u2c8e', '\u2c8e'),
        ('\u2c90', '\u2c90'), ('\u2c92', '\u2c92'),
        ('\u2c94', '\u2c94'), ('\u2c96', '\u2c96'),
        ('\u2c98', '\u2c98'), ('\u2c9a', '\u2c9a'),
        ('\u2c9c', '\u2c9c'), ('\u2c9e', '\u2c9e'),
        ('\u2ca0', '\u2ca0'), ('\u2ca2', '\u2ca2'),
        ('\u2ca4', '\u2ca4'), ('\u2ca6', '\u2ca6'),
        ('\u2ca8', '\u2ca8'), ('\u2caa', '\u2caa'),
        ('\u2cac', '\u2cac'), ('\u2cae', '\u2cae'),
        ('\u2cb0', '\u2cb0'), ('\u2cb2', '\u2cb2'),
        ('\u2cb4', '\u2cb4'), ('\u2cb6', '\u2cb6'),
        ('\u2cb8', '\u2cb8'), ('\u2cba', '\u2cba'),
        ('\u2cbc', '\u2cbc'), ('\u2cbe', '\u2cbe'),
        ('\u2cc0', '\u2cc0'), ('\u2cc2', '\u2cc2'),
        ('\u2cc4', '\u2cc4'), ('\u2cc6', '\u2cc6'),
        ('\u2cc8', '\u2cc8'), ('\u2cca', '\u2cca'),
        ('\u2ccc', '\u2ccc'), ('\u2cce', '\u2cce'),
        ('\u2cd0', '\u2cd0'), ('\u2cd2', '\u2cd2'),
        ('\u2cd4', '\u2cd4'), ('\u2cd6', '\u2cd6'),
        ('\u2cd8', '\u2cd8'), ('\u2cda', '\u2cda'),
        ('\u2cdc', '\u2cdc'), ('\u2cde', '\u2cde'),
        ('\u2ce0', '\u2ce0'), ('\u2ce2', '\u2ce2'),
        ('\u2ceb', '\u2ceb'), ('\u2ced', '\u2ced'),
        ('\u2cf2', '\u2cf2'), ('\ua640', '\ua640'),
        ('\ua642', '\ua642'), ('\ua644', '\ua644'),
        ('\ua646', '\ua646'), ('\ua648', '\ua648'),
        ('\ua64a', '\ua64a'), ('\ua64c', '\ua64c'),
        ('\ua64e', '\ua64e'), ('\ua650', '\ua650'),
        ('\ua652', '\ua652'), ('\ua654', '\ua654'),
        ('\ua656', '\ua656'), ('\ua658', '\ua658'),
        ('\ua65a', '\ua65a'), ('\ua65c', '\ua65c'),
        ('\ua65e', '\ua65e'), ('\ua660', '\ua660'),
        ('\ua662', '\ua662'), ('\ua664', '\ua664'),
        ('\ua666', '\ua666'), ('\ua668', '\ua668'),
        ('\ua66a', '\ua66a'), ('\ua66c', '\ua66c'),
        ('\ua680', '\ua680'), ('\ua682', '\ua682'),
        ('\ua684', '\ua684'), ('\ua686', '\ua686'),
        ('\ua688', '\ua688'), ('\ua68a', '\ua68a'),
        ('\ua68c', '\ua68c'), ('\ua68e', '\ua68e'),
        ('\ua690', '\ua690'), ('\ua692', '\ua692'),
        ('\ua694', '\ua694'), ('\ua696', '\ua696'),
        ('\ua722', '\ua722'), ('\ua724', '\ua724'),
        ('\ua726', '\ua726'), ('\ua728', '\ua728'),
        ('\ua72a', '\ua72a'), ('\ua72c', '\ua72c'),
        ('\ua72e', '\ua72e'), ('\ua732', '\ua732'),
        ('\ua734', '\ua734'), ('\ua736', '\ua736'),
        ('\ua738', '\ua738'), ('\ua73a', '\ua73a'),
        ('\ua73c', '\ua73c'), ('\ua73e', '\ua73e'),
        ('\ua740', '\ua740'), ('\ua742', '\ua742'),
        ('\ua744', '\ua744'), ('\ua746', '\ua746'),
        ('\ua748', '\ua748'), ('\ua74a', '\ua74a'),
        ('\ua74c', '\ua74c'), ('\ua74e', '\ua74e'),
        ('\ua750', '\ua750'), ('\ua752', '\ua752'),
        ('\ua754', '\ua754'), ('\ua756', '\ua756'),
        ('\ua758', '\ua758'), ('\ua75a', '\ua75a'),
        ('\ua75c', '\ua75c'), ('\ua75e', '\ua75e'),
        ('\ua760', '\ua760'), ('\ua762', '\ua762'),
        ('\ua764', '\ua764'), ('\ua766', '\ua766'),
        ('\ua768', '\ua768'), ('\ua76a', '\ua76a'),
        ('\ua76c', '\ua76c'), ('\ua76e', '\ua76e'),
        ('\ua779', '\ua779'), ('\ua77b', '\ua77b'),
        ('\ua77d', '\ua77e'), ('\ua780', '\ua780'),
        ('\ua782', '\ua782'), ('\ua784', '\ua784'),
        ('\ua786', '\ua786'), ('\ua78b', '\ua78b'),
        ('\ua78d', '\ua78d'), ('\ua790', '\ua790'),
        ('\ua792', '\ua792'), ('\ua7a0', '\ua7a0'),
        ('\ua7a2', '\ua7a2'), ('\ua7a4', '\ua7a4'),
        ('\ua7a6', '\ua7a6'), ('\ua7a8', '\ua7a8'),
        ('\ua7aa', '\ua7aa'), ('\uff21', '\uff3a'),
        ('\U00010400', '\U00010427'), ('\U0001d400', '\U0001d419'),
        ('\U0001d434', '\U0001d44d'), ('\U0001d468', '\U0001d481'),
        ('\U0001d49c', '\U0001d49c'), ('\U0001d49e', '\U0001d49f'),
        ('\U0001d4a2', '\U0001d4a2'), ('\U0001d4a5', '\U0001d4a6'),
        ('\U0001d4a9', '\U0001d4ac'), ('\U0001d4ae', '\U0001d4b5'),
        ('\U0001d4d0', '\U0001d4e9'), ('\U0001d504', '\U0001d505'),
        ('\U0001d507', '\U0001d50a'), ('\U0001d50d', '\U0001d514'),
        ('\U0001d516', '\U0001d51c'), ('\U0001d538', '\U0001d539'),
        ('\U0001d53b', '\U0001d53e'), ('\U0001d540', '\U0001d544'),
        ('\U0001d546', '\U0001d546'), ('\U0001d54a', '\U0001d550'),
        ('\U0001d56c', '\U0001d585'), ('\U0001d5a0', '\U0001d5b9'),
        ('\U0001d5d4', '\U0001d5ed'), ('\U0001d608', '\U0001d621'),
        ('\U0001d63c', '\U0001d655'), ('\U0001d670', '\U0001d689'),
        ('\U0001d6a8', '\U0001d6c0'), ('\U0001d6e2', '\U0001d6fa'),
        ('\U0001d71c', '\U0001d734'), ('\U0001d756', '\U0001d76e'),
        ('\U0001d790', '\U0001d7a8'), ('\U0001d7ca', '\U0001d7ca')
    ];

    pub fn Uppercase(c: char) -> bool {
        super::bsearch_range_table(c, Uppercase_table)
    }

    static XID_Continue_table : &'static [(char,char)] = &[
        ('\x30', '\x39'), ('\x41', '\x5a'),
        ('\x5f', '\x5f'), ('\x61', '\x7a'),
        ('\xaa', '\xaa'), ('\xb5', '\xb5'),
        ('\xb7', '\xb7'), ('\xba', '\xba'),
        ('\xc0', '\xd6'), ('\xd8', '\xf6'),
        ('\xf8', '\u01ba'), ('\u01bb', '\u01bb'),
        ('\u01bc', '\u01bf'), ('\u01c0', '\u01c3'),
        ('\u01c4', '\u0293'), ('\u0294', '\u0294'),
        ('\u0295', '\u02af'), ('\u02b0', '\u02c1'),
        ('\u02c6', '\u02d1'), ('\u02e0', '\u02e4'),
        ('\u02ec', '\u02ec'), ('\u02ee', '\u02ee'),
        ('\u0300', '\u036f'), ('\u0370', '\u0373'),
        ('\u0374', '\u0374'), ('\u0376', '\u0377'),
        ('\u037b', '\u037d'), ('\u0386', '\u0386'),
        ('\u0387', '\u0387'), ('\u0388', '\u038a'),
        ('\u038c', '\u038c'), ('\u038e', '\u03a1'),
        ('\u03a3', '\u03f5'), ('\u03f7', '\u0481'),
        ('\u0483', '\u0487'), ('\u048a', '\u0527'),
        ('\u0531', '\u0556'), ('\u0559', '\u0559'),
        ('\u0561', '\u0587'), ('\u0591', '\u05bd'),
        ('\u05bf', '\u05bf'), ('\u05c1', '\u05c2'),
        ('\u05c4', '\u05c5'), ('\u05c7', '\u05c7'),
        ('\u05d0', '\u05ea'), ('\u05f0', '\u05f2'),
        ('\u0610', '\u061a'), ('\u0620', '\u063f'),
        ('\u0640', '\u0640'), ('\u0641', '\u064a'),
        ('\u064b', '\u065f'), ('\u0660', '\u0669'),
        ('\u066e', '\u066f'), ('\u0670', '\u0670'),
        ('\u0671', '\u06d3'), ('\u06d5', '\u06d5'),
        ('\u06d6', '\u06dc'), ('\u06df', '\u06e4'),
        ('\u06e5', '\u06e6'), ('\u06e7', '\u06e8'),
        ('\u06ea', '\u06ed'), ('\u06ee', '\u06ef'),
        ('\u06f0', '\u06f9'), ('\u06fa', '\u06fc'),
        ('\u06ff', '\u06ff'), ('\u0710', '\u0710'),
        ('\u0711', '\u0711'), ('\u0712', '\u072f'),
        ('\u0730', '\u074a'), ('\u074d', '\u07a5'),
        ('\u07a6', '\u07b0'), ('\u07b1', '\u07b1'),
        ('\u07c0', '\u07c9'), ('\u07ca', '\u07ea'),
        ('\u07eb', '\u07f3'), ('\u07f4', '\u07f5'),
        ('\u07fa', '\u07fa'), ('\u0800', '\u0815'),
        ('\u0816', '\u0819'), ('\u081a', '\u081a'),
        ('\u081b', '\u0823'), ('\u0824', '\u0824'),
        ('\u0825', '\u0827'), ('\u0828', '\u0828'),
        ('\u0829', '\u082d'), ('\u0840', '\u0858'),
        ('\u0859', '\u085b'), ('\u08a0', '\u08a0'),
        ('\u08a2', '\u08ac'), ('\u08e4', '\u08fe'),
        ('\u0900', '\u0902'), ('\u0903', '\u0903'),
        ('\u0904', '\u0939'), ('\u093a', '\u093a'),
        ('\u093b', '\u093b'), ('\u093c', '\u093c'),
        ('\u093d', '\u093d'), ('\u093e', '\u0940'),
        ('\u0941', '\u0948'), ('\u0949', '\u094c'),
        ('\u094d', '\u094d'), ('\u094e', '\u094f'),
        ('\u0950', '\u0950'), ('\u0951', '\u0957'),
        ('\u0958', '\u0961'), ('\u0962', '\u0963'),
        ('\u0966', '\u096f'), ('\u0971', '\u0971'),
        ('\u0972', '\u0977'), ('\u0979', '\u097f'),
        ('\u0981', '\u0981'), ('\u0982', '\u0983'),
        ('\u0985', '\u098c'), ('\u098f', '\u0990'),
        ('\u0993', '\u09a8'), ('\u09aa', '\u09b0'),
        ('\u09b2', '\u09b2'), ('\u09b6', '\u09b9'),
        ('\u09bc', '\u09bc'), ('\u09bd', '\u09bd'),
        ('\u09be', '\u09c0'), ('\u09c1', '\u09c4'),
        ('\u09c7', '\u09c8'), ('\u09cb', '\u09cc'),
        ('\u09cd', '\u09cd'), ('\u09ce', '\u09ce'),
        ('\u09d7', '\u09d7'), ('\u09dc', '\u09dd'),
        ('\u09df', '\u09e1'), ('\u09e2', '\u09e3'),
        ('\u09e6', '\u09ef'), ('\u09f0', '\u09f1'),
        ('\u0a01', '\u0a02'), ('\u0a03', '\u0a03'),
        ('\u0a05', '\u0a0a'), ('\u0a0f', '\u0a10'),
        ('\u0a13', '\u0a28'), ('\u0a2a', '\u0a30'),
        ('\u0a32', '\u0a33'), ('\u0a35', '\u0a36'),
        ('\u0a38', '\u0a39'), ('\u0a3c', '\u0a3c'),
        ('\u0a3e', '\u0a40'), ('\u0a41', '\u0a42'),
        ('\u0a47', '\u0a48'), ('\u0a4b', '\u0a4d'),
        ('\u0a51', '\u0a51'), ('\u0a59', '\u0a5c'),
        ('\u0a5e', '\u0a5e'), ('\u0a66', '\u0a6f'),
        ('\u0a70', '\u0a71'), ('\u0a72', '\u0a74'),
        ('\u0a75', '\u0a75'), ('\u0a81', '\u0a82'),
        ('\u0a83', '\u0a83'), ('\u0a85', '\u0a8d'),
        ('\u0a8f', '\u0a91'), ('\u0a93', '\u0aa8'),
        ('\u0aaa', '\u0ab0'), ('\u0ab2', '\u0ab3'),
        ('\u0ab5', '\u0ab9'), ('\u0abc', '\u0abc'),
        ('\u0abd', '\u0abd'), ('\u0abe', '\u0ac0'),
        ('\u0ac1', '\u0ac5'), ('\u0ac7', '\u0ac8'),
        ('\u0ac9', '\u0ac9'), ('\u0acb', '\u0acc'),
        ('\u0acd', '\u0acd'), ('\u0ad0', '\u0ad0'),
        ('\u0ae0', '\u0ae1'), ('\u0ae2', '\u0ae3'),
        ('\u0ae6', '\u0aef'), ('\u0b01', '\u0b01'),
        ('\u0b02', '\u0b03'), ('\u0b05', '\u0b0c'),
        ('\u0b0f', '\u0b10'), ('\u0b13', '\u0b28'),
        ('\u0b2a', '\u0b30'), ('\u0b32', '\u0b33'),
        ('\u0b35', '\u0b39'), ('\u0b3c', '\u0b3c'),
        ('\u0b3d', '\u0b3d'), ('\u0b3e', '\u0b3e'),
        ('\u0b3f', '\u0b3f'), ('\u0b40', '\u0b40'),
        ('\u0b41', '\u0b44'), ('\u0b47', '\u0b48'),
        ('\u0b4b', '\u0b4c'), ('\u0b4d', '\u0b4d'),
        ('\u0b56', '\u0b56'), ('\u0b57', '\u0b57'),
        ('\u0b5c', '\u0b5d'), ('\u0b5f', '\u0b61'),
        ('\u0b62', '\u0b63'), ('\u0b66', '\u0b6f'),
        ('\u0b71', '\u0b71'), ('\u0b82', '\u0b82'),
        ('\u0b83', '\u0b83'), ('\u0b85', '\u0b8a'),
        ('\u0b8e', '\u0b90'), ('\u0b92', '\u0b95'),
        ('\u0b99', '\u0b9a'), ('\u0b9c', '\u0b9c'),
        ('\u0b9e', '\u0b9f'), ('\u0ba3', '\u0ba4'),
        ('\u0ba8', '\u0baa'), ('\u0bae', '\u0bb9'),
        ('\u0bbe', '\u0bbf'), ('\u0bc0', '\u0bc0'),
        ('\u0bc1', '\u0bc2'), ('\u0bc6', '\u0bc8'),
        ('\u0bca', '\u0bcc'), ('\u0bcd', '\u0bcd'),
        ('\u0bd0', '\u0bd0'), ('\u0bd7', '\u0bd7'),
        ('\u0be6', '\u0bef'), ('\u0c01', '\u0c03'),
        ('\u0c05', '\u0c0c'), ('\u0c0e', '\u0c10'),
        ('\u0c12', '\u0c28'), ('\u0c2a', '\u0c33'),
        ('\u0c35', '\u0c39'), ('\u0c3d', '\u0c3d'),
        ('\u0c3e', '\u0c40'), ('\u0c41', '\u0c44'),
        ('\u0c46', '\u0c48'), ('\u0c4a', '\u0c4d'),
        ('\u0c55', '\u0c56'), ('\u0c58', '\u0c59'),
        ('\u0c60', '\u0c61'), ('\u0c62', '\u0c63'),
        ('\u0c66', '\u0c6f'), ('\u0c82', '\u0c83'),
        ('\u0c85', '\u0c8c'), ('\u0c8e', '\u0c90'),
        ('\u0c92', '\u0ca8'), ('\u0caa', '\u0cb3'),
        ('\u0cb5', '\u0cb9'), ('\u0cbc', '\u0cbc'),
        ('\u0cbd', '\u0cbd'), ('\u0cbe', '\u0cbe'),
        ('\u0cbf', '\u0cbf'), ('\u0cc0', '\u0cc4'),
        ('\u0cc6', '\u0cc6'), ('\u0cc7', '\u0cc8'),
        ('\u0cca', '\u0ccb'), ('\u0ccc', '\u0ccd'),
        ('\u0cd5', '\u0cd6'), ('\u0cde', '\u0cde'),
        ('\u0ce0', '\u0ce1'), ('\u0ce2', '\u0ce3'),
        ('\u0ce6', '\u0cef'), ('\u0cf1', '\u0cf2'),
        ('\u0d02', '\u0d03'), ('\u0d05', '\u0d0c'),
        ('\u0d0e', '\u0d10'), ('\u0d12', '\u0d3a'),
        ('\u0d3d', '\u0d3d'), ('\u0d3e', '\u0d40'),
        ('\u0d41', '\u0d44'), ('\u0d46', '\u0d48'),
        ('\u0d4a', '\u0d4c'), ('\u0d4d', '\u0d4d'),
        ('\u0d4e', '\u0d4e'), ('\u0d57', '\u0d57'),
        ('\u0d60', '\u0d61'), ('\u0d62', '\u0d63'),
        ('\u0d66', '\u0d6f'), ('\u0d7a', '\u0d7f'),
        ('\u0d82', '\u0d83'), ('\u0d85', '\u0d96'),
        ('\u0d9a', '\u0db1'), ('\u0db3', '\u0dbb'),
        ('\u0dbd', '\u0dbd'), ('\u0dc0', '\u0dc6'),
        ('\u0dca', '\u0dca'), ('\u0dcf', '\u0dd1'),
        ('\u0dd2', '\u0dd4'), ('\u0dd6', '\u0dd6'),
        ('\u0dd8', '\u0ddf'), ('\u0df2', '\u0df3'),
        ('\u0e01', '\u0e30'), ('\u0e31', '\u0e31'),
        ('\u0e32', '\u0e33'), ('\u0e34', '\u0e3a'),
        ('\u0e40', '\u0e45'), ('\u0e46', '\u0e46'),
        ('\u0e47', '\u0e4e'), ('\u0e50', '\u0e59'),
        ('\u0e81', '\u0e82'), ('\u0e84', '\u0e84'),
        ('\u0e87', '\u0e88'), ('\u0e8a', '\u0e8a'),
        ('\u0e8d', '\u0e8d'), ('\u0e94', '\u0e97'),
        ('\u0e99', '\u0e9f'), ('\u0ea1', '\u0ea3'),
        ('\u0ea5', '\u0ea5'), ('\u0ea7', '\u0ea7'),
        ('\u0eaa', '\u0eab'), ('\u0ead', '\u0eb0'),
        ('\u0eb1', '\u0eb1'), ('\u0eb2', '\u0eb3'),
        ('\u0eb4', '\u0eb9'), ('\u0ebb', '\u0ebc'),
        ('\u0ebd', '\u0ebd'), ('\u0ec0', '\u0ec4'),
        ('\u0ec6', '\u0ec6'), ('\u0ec8', '\u0ecd'),
        ('\u0ed0', '\u0ed9'), ('\u0edc', '\u0edf'),
        ('\u0f00', '\u0f00'), ('\u0f18', '\u0f19'),
        ('\u0f20', '\u0f29'), ('\u0f35', '\u0f35'),
        ('\u0f37', '\u0f37'), ('\u0f39', '\u0f39'),
        ('\u0f3e', '\u0f3f'), ('\u0f40', '\u0f47'),
        ('\u0f49', '\u0f6c'), ('\u0f71', '\u0f7e'),
        ('\u0f7f', '\u0f7f'), ('\u0f80', '\u0f84'),
        ('\u0f86', '\u0f87'), ('\u0f88', '\u0f8c'),
        ('\u0f8d', '\u0f97'), ('\u0f99', '\u0fbc'),
        ('\u0fc6', '\u0fc6'), ('\u1000', '\u102a'),
        ('\u102b', '\u102c'), ('\u102d', '\u1030'),
        ('\u1031', '\u1031'), ('\u1032', '\u1037'),
        ('\u1038', '\u1038'), ('\u1039', '\u103a'),
        ('\u103b', '\u103c'), ('\u103d', '\u103e'),
        ('\u103f', '\u103f'), ('\u1040', '\u1049'),
        ('\u1050', '\u1055'), ('\u1056', '\u1057'),
        ('\u1058', '\u1059'), ('\u105a', '\u105d'),
        ('\u105e', '\u1060'), ('\u1061', '\u1061'),
        ('\u1062', '\u1064'), ('\u1065', '\u1066'),
        ('\u1067', '\u106d'), ('\u106e', '\u1070'),
        ('\u1071', '\u1074'), ('\u1075', '\u1081'),
        ('\u1082', '\u1082'), ('\u1083', '\u1084'),
        ('\u1085', '\u1086'), ('\u1087', '\u108c'),
        ('\u108d', '\u108d'), ('\u108e', '\u108e'),
        ('\u108f', '\u108f'), ('\u1090', '\u1099'),
        ('\u109a', '\u109c'), ('\u109d', '\u109d'),
        ('\u10a0', '\u10c5'), ('\u10c7', '\u10c7'),
        ('\u10cd', '\u10cd'), ('\u10d0', '\u10fa'),
        ('\u10fc', '\u10fc'), ('\u10fd', '\u1248'),
        ('\u124a', '\u124d'), ('\u1250', '\u1256'),
        ('\u1258', '\u1258'), ('\u125a', '\u125d'),
        ('\u1260', '\u1288'), ('\u128a', '\u128d'),
        ('\u1290', '\u12b0'), ('\u12b2', '\u12b5'),
        ('\u12b8', '\u12be'), ('\u12c0', '\u12c0'),
        ('\u12c2', '\u12c5'), ('\u12c8', '\u12d6'),
        ('\u12d8', '\u1310'), ('\u1312', '\u1315'),
        ('\u1318', '\u135a'), ('\u135d', '\u135f'),
        ('\u1369', '\u1371'), ('\u1380', '\u138f'),
        ('\u13a0', '\u13f4'), ('\u1401', '\u166c'),
        ('\u166f', '\u167f'), ('\u1681', '\u169a'),
        ('\u16a0', '\u16ea'), ('\u16ee', '\u16f0'),
        ('\u1700', '\u170c'), ('\u170e', '\u1711'),
        ('\u1712', '\u1714'), ('\u1720', '\u1731'),
        ('\u1732', '\u1734'), ('\u1740', '\u1751'),
        ('\u1752', '\u1753'), ('\u1760', '\u176c'),
        ('\u176e', '\u1770'), ('\u1772', '\u1773'),
        ('\u1780', '\u17b3'), ('\u17b4', '\u17b5'),
        ('\u17b6', '\u17b6'), ('\u17b7', '\u17bd'),
        ('\u17be', '\u17c5'), ('\u17c6', '\u17c6'),
        ('\u17c7', '\u17c8'), ('\u17c9', '\u17d3'),
        ('\u17d7', '\u17d7'), ('\u17dc', '\u17dc'),
        ('\u17dd', '\u17dd'), ('\u17e0', '\u17e9'),
        ('\u180b', '\u180d'), ('\u1810', '\u1819'),
        ('\u1820', '\u1842'), ('\u1843', '\u1843'),
        ('\u1844', '\u1877'), ('\u1880', '\u18a8'),
        ('\u18a9', '\u18a9'), ('\u18aa', '\u18aa'),
        ('\u18b0', '\u18f5'), ('\u1900', '\u191c'),
        ('\u1920', '\u1922'), ('\u1923', '\u1926'),
        ('\u1927', '\u1928'), ('\u1929', '\u192b'),
        ('\u1930', '\u1931'), ('\u1932', '\u1932'),
        ('\u1933', '\u1938'), ('\u1939', '\u193b'),
        ('\u1946', '\u194f'), ('\u1950', '\u196d'),
        ('\u1970', '\u1974'), ('\u1980', '\u19ab'),
        ('\u19b0', '\u19c0'), ('\u19c1', '\u19c7'),
        ('\u19c8', '\u19c9'), ('\u19d0', '\u19d9'),
        ('\u19da', '\u19da'), ('\u1a00', '\u1a16'),
        ('\u1a17', '\u1a18'), ('\u1a19', '\u1a1a'),
        ('\u1a1b', '\u1a1b'), ('\u1a20', '\u1a54'),
        ('\u1a55', '\u1a55'), ('\u1a56', '\u1a56'),
        ('\u1a57', '\u1a57'), ('\u1a58', '\u1a5e'),
        ('\u1a60', '\u1a60'), ('\u1a61', '\u1a61'),
        ('\u1a62', '\u1a62'), ('\u1a63', '\u1a64'),
        ('\u1a65', '\u1a6c'), ('\u1a6d', '\u1a72'),
        ('\u1a73', '\u1a7c'), ('\u1a7f', '\u1a7f'),
        ('\u1a80', '\u1a89'), ('\u1a90', '\u1a99'),
        ('\u1aa7', '\u1aa7'), ('\u1b00', '\u1b03'),
        ('\u1b04', '\u1b04'), ('\u1b05', '\u1b33'),
        ('\u1b34', '\u1b34'), ('\u1b35', '\u1b35'),
        ('\u1b36', '\u1b3a'), ('\u1b3b', '\u1b3b'),
        ('\u1b3c', '\u1b3c'), ('\u1b3d', '\u1b41'),
        ('\u1b42', '\u1b42'), ('\u1b43', '\u1b44'),
        ('\u1b45', '\u1b4b'), ('\u1b50', '\u1b59'),
        ('\u1b6b', '\u1b73'), ('\u1b80', '\u1b81'),
        ('\u1b82', '\u1b82'), ('\u1b83', '\u1ba0'),
        ('\u1ba1', '\u1ba1'), ('\u1ba2', '\u1ba5'),
        ('\u1ba6', '\u1ba7'), ('\u1ba8', '\u1ba9'),
        ('\u1baa', '\u1baa'), ('\u1bab', '\u1bab'),
        ('\u1bac', '\u1bad'), ('\u1bae', '\u1baf'),
        ('\u1bb0', '\u1bb9'), ('\u1bba', '\u1be5'),
        ('\u1be6', '\u1be6'), ('\u1be7', '\u1be7'),
        ('\u1be8', '\u1be9'), ('\u1bea', '\u1bec'),
        ('\u1bed', '\u1bed'), ('\u1bee', '\u1bee'),
        ('\u1bef', '\u1bf1'), ('\u1bf2', '\u1bf3'),
        ('\u1c00', '\u1c23'), ('\u1c24', '\u1c2b'),
        ('\u1c2c', '\u1c33'), ('\u1c34', '\u1c35'),
        ('\u1c36', '\u1c37'), ('\u1c40', '\u1c49'),
        ('\u1c4d', '\u1c4f'), ('\u1c50', '\u1c59'),
        ('\u1c5a', '\u1c77'), ('\u1c78', '\u1c7d'),
        ('\u1cd0', '\u1cd2'), ('\u1cd4', '\u1ce0'),
        ('\u1ce1', '\u1ce1'), ('\u1ce2', '\u1ce8'),
        ('\u1ce9', '\u1cec'), ('\u1ced', '\u1ced'),
        ('\u1cee', '\u1cf1'), ('\u1cf2', '\u1cf3'),
        ('\u1cf4', '\u1cf4'), ('\u1cf5', '\u1cf6'),
        ('\u1d00', '\u1d2b'), ('\u1d2c', '\u1d6a'),
        ('\u1d6b', '\u1d77'), ('\u1d78', '\u1d78'),
        ('\u1d79', '\u1d9a'), ('\u1d9b', '\u1dbf'),
        ('\u1dc0', '\u1de6'), ('\u1dfc', '\u1dff'),
        ('\u1e00', '\u1f15'), ('\u1f18', '\u1f1d'),
        ('\u1f20', '\u1f45'), ('\u1f48', '\u1f4d'),
        ('\u1f50', '\u1f57'), ('\u1f59', '\u1f59'),
        ('\u1f5b', '\u1f5b'), ('\u1f5d', '\u1f5d'),
        ('\u1f5f', '\u1f7d'), ('\u1f80', '\u1fb4'),
        ('\u1fb6', '\u1fbc'), ('\u1fbe', '\u1fbe'),
        ('\u1fc2', '\u1fc4'), ('\u1fc6', '\u1fcc'),
        ('\u1fd0', '\u1fd3'), ('\u1fd6', '\u1fdb'),
        ('\u1fe0', '\u1fec'), ('\u1ff2', '\u1ff4'),
        ('\u1ff6', '\u1ffc'), ('\u203f', '\u2040'),
        ('\u2054', '\u2054'), ('\u2071', '\u2071'),
        ('\u207f', '\u207f'), ('\u2090', '\u209c'),
        ('\u20d0', '\u20dc'), ('\u20e1', '\u20e1'),
        ('\u20e5', '\u20f0'), ('\u2102', '\u2102'),
        ('\u2107', '\u2107'), ('\u210a', '\u2113'),
        ('\u2115', '\u2115'), ('\u2118', '\u2118'),
        ('\u2119', '\u211d'), ('\u2124', '\u2124'),
        ('\u2126', '\u2126'), ('\u2128', '\u2128'),
        ('\u212a', '\u212d'), ('\u212e', '\u212e'),
        ('\u212f', '\u2134'), ('\u2135', '\u2138'),
        ('\u2139', '\u2139'), ('\u213c', '\u213f'),
        ('\u2145', '\u2149'), ('\u214e', '\u214e'),
        ('\u2160', '\u2182'), ('\u2183', '\u2184'),
        ('\u2185', '\u2188'), ('\u2c00', '\u2c2e'),
        ('\u2c30', '\u2c5e'), ('\u2c60', '\u2c7b'),
        ('\u2c7c', '\u2c7d'), ('\u2c7e', '\u2ce4'),
        ('\u2ceb', '\u2cee'), ('\u2cef', '\u2cf1'),
        ('\u2cf2', '\u2cf3'), ('\u2d00', '\u2d25'),
        ('\u2d27', '\u2d27'), ('\u2d2d', '\u2d2d'),
        ('\u2d30', '\u2d67'), ('\u2d6f', '\u2d6f'),
        ('\u2d7f', '\u2d7f'), ('\u2d80', '\u2d96'),
        ('\u2da0', '\u2da6'), ('\u2da8', '\u2dae'),
        ('\u2db0', '\u2db6'), ('\u2db8', '\u2dbe'),
        ('\u2dc0', '\u2dc6'), ('\u2dc8', '\u2dce'),
        ('\u2dd0', '\u2dd6'), ('\u2dd8', '\u2dde'),
        ('\u2de0', '\u2dff'), ('\u3005', '\u3005'),
        ('\u3006', '\u3006'), ('\u3007', '\u3007'),
        ('\u3021', '\u3029'), ('\u302a', '\u302d'),
        ('\u302e', '\u302f'), ('\u3031', '\u3035'),
        ('\u3038', '\u303a'), ('\u303b', '\u303b'),
        ('\u303c', '\u303c'), ('\u3041', '\u3096'),
        ('\u3099', '\u309a'), ('\u309d', '\u309e'),
        ('\u309f', '\u309f'), ('\u30a1', '\u30fa'),
        ('\u30fc', '\u30fe'), ('\u30ff', '\u30ff'),
        ('\u3105', '\u312d'), ('\u3131', '\u318e'),
        ('\u31a0', '\u31ba'), ('\u31f0', '\u31ff'),
        ('\u3400', '\u4db5'), ('\u4e00', '\u9fcc'),
        ('\ua000', '\ua014'), ('\ua015', '\ua015'),
        ('\ua016', '\ua48c'), ('\ua4d0', '\ua4f7'),
        ('\ua4f8', '\ua4fd'), ('\ua500', '\ua60b'),
        ('\ua60c', '\ua60c'), ('\ua610', '\ua61f'),
        ('\ua620', '\ua629'), ('\ua62a', '\ua62b'),
        ('\ua640', '\ua66d'), ('\ua66e', '\ua66e'),
        ('\ua66f', '\ua66f'), ('\ua674', '\ua67d'),
        ('\ua67f', '\ua67f'), ('\ua680', '\ua697'),
        ('\ua69f', '\ua69f'), ('\ua6a0', '\ua6e5'),
        ('\ua6e6', '\ua6ef'), ('\ua6f0', '\ua6f1'),
        ('\ua717', '\ua71f'), ('\ua722', '\ua76f'),
        ('\ua770', '\ua770'), ('\ua771', '\ua787'),
        ('\ua788', '\ua788'), ('\ua78b', '\ua78e'),
        ('\ua790', '\ua793'), ('\ua7a0', '\ua7aa'),
        ('\ua7f8', '\ua7f9'), ('\ua7fa', '\ua7fa'),
        ('\ua7fb', '\ua801'), ('\ua802', '\ua802'),
        ('\ua803', '\ua805'), ('\ua806', '\ua806'),
        ('\ua807', '\ua80a'), ('\ua80b', '\ua80b'),
        ('\ua80c', '\ua822'), ('\ua823', '\ua824'),
        ('\ua825', '\ua826'), ('\ua827', '\ua827'),
        ('\ua840', '\ua873'), ('\ua880', '\ua881'),
        ('\ua882', '\ua8b3'), ('\ua8b4', '\ua8c3'),
        ('\ua8c4', '\ua8c4'), ('\ua8d0', '\ua8d9'),
        ('\ua8e0', '\ua8f1'), ('\ua8f2', '\ua8f7'),
        ('\ua8fb', '\ua8fb'), ('\ua900', '\ua909'),
        ('\ua90a', '\ua925'), ('\ua926', '\ua92d'),
        ('\ua930', '\ua946'), ('\ua947', '\ua951'),
        ('\ua952', '\ua953'), ('\ua960', '\ua97c'),
        ('\ua980', '\ua982'), ('\ua983', '\ua983'),
        ('\ua984', '\ua9b2'), ('\ua9b3', '\ua9b3'),
        ('\ua9b4', '\ua9b5'), ('\ua9b6', '\ua9b9'),
        ('\ua9ba', '\ua9bb'), ('\ua9bc', '\ua9bc'),
        ('\ua9bd', '\ua9c0'), ('\ua9cf', '\ua9cf'),
        ('\ua9d0', '\ua9d9'), ('\uaa00', '\uaa28'),
        ('\uaa29', '\uaa2e'), ('\uaa2f', '\uaa30'),
        ('\uaa31', '\uaa32'), ('\uaa33', '\uaa34'),
        ('\uaa35', '\uaa36'), ('\uaa40', '\uaa42'),
        ('\uaa43', '\uaa43'), ('\uaa44', '\uaa4b'),
        ('\uaa4c', '\uaa4c'), ('\uaa4d', '\uaa4d'),
        ('\uaa50', '\uaa59'), ('\uaa60', '\uaa6f'),
        ('\uaa70', '\uaa70'), ('\uaa71', '\uaa76'),
        ('\uaa7a', '\uaa7a'), ('\uaa7b', '\uaa7b'),
        ('\uaa80', '\uaaaf'), ('\uaab0', '\uaab0'),
        ('\uaab1', '\uaab1'), ('\uaab2', '\uaab4'),
        ('\uaab5', '\uaab6'), ('\uaab7', '\uaab8'),
        ('\uaab9', '\uaabd'), ('\uaabe', '\uaabf'),
        ('\uaac0', '\uaac0'), ('\uaac1', '\uaac1'),
        ('\uaac2', '\uaac2'), ('\uaadb', '\uaadc'),
        ('\uaadd', '\uaadd'), ('\uaae0', '\uaaea'),
        ('\uaaeb', '\uaaeb'), ('\uaaec', '\uaaed'),
        ('\uaaee', '\uaaef'), ('\uaaf2', '\uaaf2'),
        ('\uaaf3', '\uaaf4'), ('\uaaf5', '\uaaf5'),
        ('\uaaf6', '\uaaf6'), ('\uab01', '\uab06'),
        ('\uab09', '\uab0e'), ('\uab11', '\uab16'),
        ('\uab20', '\uab26'), ('\uab28', '\uab2e'),
        ('\uabc0', '\uabe2'), ('\uabe3', '\uabe4'),
        ('\uabe5', '\uabe5'), ('\uabe6', '\uabe7'),
        ('\uabe8', '\uabe8'), ('\uabe9', '\uabea'),
        ('\uabec', '\uabec'), ('\uabed', '\uabed'),
        ('\uabf0', '\uabf9'), ('\uac00', '\ud7a3'),
        ('\ud7b0', '\ud7c6'), ('\ud7cb', '\ud7fb'),
        ('\uf900', '\ufa6d'), ('\ufa70', '\ufad9'),
        ('\ufb00', '\ufb06'), ('\ufb13', '\ufb17'),
        ('\ufb1d', '\ufb1d'), ('\ufb1e', '\ufb1e'),
        ('\ufb1f', '\ufb28'), ('\ufb2a', '\ufb36'),
        ('\ufb38', '\ufb3c'), ('\ufb3e', '\ufb3e'),
        ('\ufb40', '\ufb41'), ('\ufb43', '\ufb44'),
        ('\ufb46', '\ufbb1'), ('\ufbd3', '\ufc5d'),
        ('\ufc64', '\ufd3d'), ('\ufd50', '\ufd8f'),
        ('\ufd92', '\ufdc7'), ('\ufdf0', '\ufdf9'),
        ('\ufe00', '\ufe0f'), ('\ufe20', '\ufe26'),
        ('\ufe33', '\ufe34'), ('\ufe4d', '\ufe4f'),
        ('\ufe71', '\ufe71'), ('\ufe73', '\ufe73'),
        ('\ufe77', '\ufe77'), ('\ufe79', '\ufe79'),
        ('\ufe7b', '\ufe7b'), ('\ufe7d', '\ufe7d'),
        ('\ufe7f', '\ufefc'), ('\uff10', '\uff19'),
        ('\uff21', '\uff3a'), ('\uff3f', '\uff3f'),
        ('\uff41', '\uff5a'), ('\uff66', '\uff6f'),
        ('\uff70', '\uff70'), ('\uff71', '\uff9d'),
        ('\uff9e', '\uff9f'), ('\uffa0', '\uffbe'),
        ('\uffc2', '\uffc7'), ('\uffca', '\uffcf'),
        ('\uffd2', '\uffd7'), ('\uffda', '\uffdc'),
        ('\U00010000', '\U0001000b'), ('\U0001000d', '\U00010026'),
        ('\U00010028', '\U0001003a'), ('\U0001003c', '\U0001003d'),
        ('\U0001003f', '\U0001004d'), ('\U00010050', '\U0001005d'),
        ('\U00010080', '\U000100fa'), ('\U00010140', '\U00010174'),
        ('\U000101fd', '\U000101fd'), ('\U00010280', '\U0001029c'),
        ('\U000102a0', '\U000102d0'), ('\U00010300', '\U0001031e'),
        ('\U00010330', '\U00010340'), ('\U00010341', '\U00010341'),
        ('\U00010342', '\U00010349'), ('\U0001034a', '\U0001034a'),
        ('\U00010380', '\U0001039d'), ('\U000103a0', '\U000103c3'),
        ('\U000103c8', '\U000103cf'), ('\U000103d1', '\U000103d5'),
        ('\U00010400', '\U0001044f'), ('\U00010450', '\U0001049d'),
        ('\U000104a0', '\U000104a9'), ('\U00010800', '\U00010805'),
        ('\U00010808', '\U00010808'), ('\U0001080a', '\U00010835'),
        ('\U00010837', '\U00010838'), ('\U0001083c', '\U0001083c'),
        ('\U0001083f', '\U00010855'), ('\U00010900', '\U00010915'),
        ('\U00010920', '\U00010939'), ('\U00010980', '\U000109b7'),
        ('\U000109be', '\U000109bf'), ('\U00010a00', '\U00010a00'),
        ('\U00010a01', '\U00010a03'), ('\U00010a05', '\U00010a06'),
        ('\U00010a0c', '\U00010a0f'), ('\U00010a10', '\U00010a13'),
        ('\U00010a15', '\U00010a17'), ('\U00010a19', '\U00010a33'),
        ('\U00010a38', '\U00010a3a'), ('\U00010a3f', '\U00010a3f'),
        ('\U00010a60', '\U00010a7c'), ('\U00010b00', '\U00010b35'),
        ('\U00010b40', '\U00010b55'), ('\U00010b60', '\U00010b72'),
        ('\U00010c00', '\U00010c48'), ('\U00011000', '\U00011000'),
        ('\U00011001', '\U00011001'), ('\U00011002', '\U00011002'),
        ('\U00011003', '\U00011037'), ('\U00011038', '\U00011046'),
        ('\U00011066', '\U0001106f'), ('\U00011080', '\U00011081'),
        ('\U00011082', '\U00011082'), ('\U00011083', '\U000110af'),
        ('\U000110b0', '\U000110b2'), ('\U000110b3', '\U000110b6'),
        ('\U000110b7', '\U000110b8'), ('\U000110b9', '\U000110ba'),
        ('\U000110d0', '\U000110e8'), ('\U000110f0', '\U000110f9'),
        ('\U00011100', '\U00011102'), ('\U00011103', '\U00011126'),
        ('\U00011127', '\U0001112b'), ('\U0001112c', '\U0001112c'),
        ('\U0001112d', '\U00011134'), ('\U00011136', '\U0001113f'),
        ('\U00011180', '\U00011181'), ('\U00011182', '\U00011182'),
        ('\U00011183', '\U000111b2'), ('\U000111b3', '\U000111b5'),
        ('\U000111b6', '\U000111be'), ('\U000111bf', '\U000111c0'),
        ('\U000111c1', '\U000111c4'), ('\U000111d0', '\U000111d9'),
        ('\U00011680', '\U000116aa'), ('\U000116ab', '\U000116ab'),
        ('\U000116ac', '\U000116ac'), ('\U000116ad', '\U000116ad'),
        ('\U000116ae', '\U000116af'), ('\U000116b0', '\U000116b5'),
        ('\U000116b6', '\U000116b6'), ('\U000116b7', '\U000116b7'),
        ('\U000116c0', '\U000116c9'), ('\U00012000', '\U0001236e'),
        ('\U00012400', '\U00012462'), ('\U00013000', '\U0001342e'),
        ('\U00016800', '\U00016a38'), ('\U00016f00', '\U00016f44'),
        ('\U00016f50', '\U00016f50'), ('\U00016f51', '\U00016f7e'),
        ('\U00016f8f', '\U00016f92'), ('\U00016f93', '\U00016f9f'),
        ('\U0001b000', '\U0001b001'), ('\U0001d165', '\U0001d166'),
        ('\U0001d167', '\U0001d169'), ('\U0001d16d', '\U0001d172'),
        ('\U0001d17b', '\U0001d182'), ('\U0001d185', '\U0001d18b'),
        ('\U0001d1aa', '\U0001d1ad'), ('\U0001d242', '\U0001d244'),
        ('\U0001d400', '\U0001d454'), ('\U0001d456', '\U0001d49c'),
        ('\U0001d49e', '\U0001d49f'), ('\U0001d4a2', '\U0001d4a2'),
        ('\U0001d4a5', '\U0001d4a6'), ('\U0001d4a9', '\U0001d4ac'),
        ('\U0001d4ae', '\U0001d4b9'), ('\U0001d4bb', '\U0001d4bb'),
        ('\U0001d4bd', '\U0001d4c3'), ('\U0001d4c5', '\U0001d505'),
        ('\U0001d507', '\U0001d50a'), ('\U0001d50d', '\U0001d514'),
        ('\U0001d516', '\U0001d51c'), ('\U0001d51e', '\U0001d539'),
        ('\U0001d53b', '\U0001d53e'), ('\U0001d540', '\U0001d544'),
        ('\U0001d546', '\U0001d546'), ('\U0001d54a', '\U0001d550'),
        ('\U0001d552', '\U0001d6a5'), ('\U0001d6a8', '\U0001d6c0'),
        ('\U0001d6c2', '\U0001d6da'), ('\U0001d6dc', '\U0001d6fa'),
        ('\U0001d6fc', '\U0001d714'), ('\U0001d716', '\U0001d734'),
        ('\U0001d736', '\U0001d74e'), ('\U0001d750', '\U0001d76e'),
        ('\U0001d770', '\U0001d788'), ('\U0001d78a', '\U0001d7a8'),
        ('\U0001d7aa', '\U0001d7c2'), ('\U0001d7c4', '\U0001d7cb'),
        ('\U0001d7ce', '\U0001d7ff'), ('\U0001ee00', '\U0001ee03'),
        ('\U0001ee05', '\U0001ee1f'), ('\U0001ee21', '\U0001ee22'),
        ('\U0001ee24', '\U0001ee24'), ('\U0001ee27', '\U0001ee27'),
        ('\U0001ee29', '\U0001ee32'), ('\U0001ee34', '\U0001ee37'),
        ('\U0001ee39', '\U0001ee39'), ('\U0001ee3b', '\U0001ee3b'),
        ('\U0001ee42', '\U0001ee42'), ('\U0001ee47', '\U0001ee47'),
        ('\U0001ee49', '\U0001ee49'), ('\U0001ee4b', '\U0001ee4b'),
        ('\U0001ee4d', '\U0001ee4f'), ('\U0001ee51', '\U0001ee52'),
        ('\U0001ee54', '\U0001ee54'), ('\U0001ee57', '\U0001ee57'),
        ('\U0001ee59', '\U0001ee59'), ('\U0001ee5b', '\U0001ee5b'),
        ('\U0001ee5d', '\U0001ee5d'), ('\U0001ee5f', '\U0001ee5f'),
        ('\U0001ee61', '\U0001ee62'), ('\U0001ee64', '\U0001ee64'),
        ('\U0001ee67', '\U0001ee6a'), ('\U0001ee6c', '\U0001ee72'),
        ('\U0001ee74', '\U0001ee77'), ('\U0001ee79', '\U0001ee7c'),
        ('\U0001ee7e', '\U0001ee7e'), ('\U0001ee80', '\U0001ee89'),
        ('\U0001ee8b', '\U0001ee9b'), ('\U0001eea1', '\U0001eea3'),
        ('\U0001eea5', '\U0001eea9'), ('\U0001eeab', '\U0001eebb'),
        ('\U00020000', '\U0002a6d6'), ('\U0002a700', '\U0002b734'),
        ('\U0002b740', '\U0002b81d'), ('\U0002f800', '\U0002fa1d'),
        ('\U000e0100', '\U000e01ef')
    ];

    pub fn XID_Continue(c: char) -> bool {
        super::bsearch_range_table(c, XID_Continue_table)
    }

    static XID_Start_table : &'static [(char,char)] = &[
        ('\x41', '\x5a'), ('\x61', '\x7a'),
        ('\xaa', '\xaa'), ('\xb5', '\xb5'),
        ('\xba', '\xba'), ('\xc0', '\xd6'),
        ('\xd8', '\xf6'), ('\xf8', '\u01ba'),
        ('\u01bb', '\u01bb'), ('\u01bc', '\u01bf'),
        ('\u01c0', '\u01c3'), ('\u01c4', '\u0293'),
        ('\u0294', '\u0294'), ('\u0295', '\u02af'),
        ('\u02b0', '\u02c1'), ('\u02c6', '\u02d1'),
        ('\u02e0', '\u02e4'), ('\u02ec', '\u02ec'),
        ('\u02ee', '\u02ee'), ('\u0370', '\u0373'),
        ('\u0374', '\u0374'), ('\u0376', '\u0377'),
        ('\u037b', '\u037d'), ('\u0386', '\u0386'),
        ('\u0388', '\u038a'), ('\u038c', '\u038c'),
        ('\u038e', '\u03a1'), ('\u03a3', '\u03f5'),
        ('\u03f7', '\u0481'), ('\u048a', '\u0527'),
        ('\u0531', '\u0556'), ('\u0559', '\u0559'),
        ('\u0561', '\u0587'), ('\u05d0', '\u05ea'),
        ('\u05f0', '\u05f2'), ('\u0620', '\u063f'),
        ('\u0640', '\u0640'), ('\u0641', '\u064a'),
        ('\u066e', '\u066f'), ('\u0671', '\u06d3'),
        ('\u06d5', '\u06d5'), ('\u06e5', '\u06e6'),
        ('\u06ee', '\u06ef'), ('\u06fa', '\u06fc'),
        ('\u06ff', '\u06ff'), ('\u0710', '\u0710'),
        ('\u0712', '\u072f'), ('\u074d', '\u07a5'),
        ('\u07b1', '\u07b1'), ('\u07ca', '\u07ea'),
        ('\u07f4', '\u07f5'), ('\u07fa', '\u07fa'),
        ('\u0800', '\u0815'), ('\u081a', '\u081a'),
        ('\u0824', '\u0824'), ('\u0828', '\u0828'),
        ('\u0840', '\u0858'), ('\u08a0', '\u08a0'),
        ('\u08a2', '\u08ac'), ('\u0904', '\u0939'),
        ('\u093d', '\u093d'), ('\u0950', '\u0950'),
        ('\u0958', '\u0961'), ('\u0971', '\u0971'),
        ('\u0972', '\u0977'), ('\u0979', '\u097f'),
        ('\u0985', '\u098c'), ('\u098f', '\u0990'),
        ('\u0993', '\u09a8'), ('\u09aa', '\u09b0'),
        ('\u09b2', '\u09b2'), ('\u09b6', '\u09b9'),
        ('\u09bd', '\u09bd'), ('\u09ce', '\u09ce'),
        ('\u09dc', '\u09dd'), ('\u09df', '\u09e1'),
        ('\u09f0', '\u09f1'), ('\u0a05', '\u0a0a'),
        ('\u0a0f', '\u0a10'), ('\u0a13', '\u0a28'),
        ('\u0a2a', '\u0a30'), ('\u0a32', '\u0a33'),
        ('\u0a35', '\u0a36'), ('\u0a38', '\u0a39'),
        ('\u0a59', '\u0a5c'), ('\u0a5e', '\u0a5e'),
        ('\u0a72', '\u0a74'), ('\u0a85', '\u0a8d'),
        ('\u0a8f', '\u0a91'), ('\u0a93', '\u0aa8'),
        ('\u0aaa', '\u0ab0'), ('\u0ab2', '\u0ab3'),
        ('\u0ab5', '\u0ab9'), ('\u0abd', '\u0abd'),
        ('\u0ad0', '\u0ad0'), ('\u0ae0', '\u0ae1'),
        ('\u0b05', '\u0b0c'), ('\u0b0f', '\u0b10'),
        ('\u0b13', '\u0b28'), ('\u0b2a', '\u0b30'),
        ('\u0b32', '\u0b33'), ('\u0b35', '\u0b39'),
        ('\u0b3d', '\u0b3d'), ('\u0b5c', '\u0b5d'),
        ('\u0b5f', '\u0b61'), ('\u0b71', '\u0b71'),
        ('\u0b83', '\u0b83'), ('\u0b85', '\u0b8a'),
        ('\u0b8e', '\u0b90'), ('\u0b92', '\u0b95'),
        ('\u0b99', '\u0b9a'), ('\u0b9c', '\u0b9c'),
        ('\u0b9e', '\u0b9f'), ('\u0ba3', '\u0ba4'),
        ('\u0ba8', '\u0baa'), ('\u0bae', '\u0bb9'),
        ('\u0bd0', '\u0bd0'), ('\u0c05', '\u0c0c'),
        ('\u0c0e', '\u0c10'), ('\u0c12', '\u0c28'),
        ('\u0c2a', '\u0c33'), ('\u0c35', '\u0c39'),
        ('\u0c3d', '\u0c3d'), ('\u0c58', '\u0c59'),
        ('\u0c60', '\u0c61'), ('\u0c85', '\u0c8c'),
        ('\u0c8e', '\u0c90'), ('\u0c92', '\u0ca8'),
        ('\u0caa', '\u0cb3'), ('\u0cb5', '\u0cb9'),
        ('\u0cbd', '\u0cbd'), ('\u0cde', '\u0cde'),
        ('\u0ce0', '\u0ce1'), ('\u0cf1', '\u0cf2'),
        ('\u0d05', '\u0d0c'), ('\u0d0e', '\u0d10'),
        ('\u0d12', '\u0d3a'), ('\u0d3d', '\u0d3d'),
        ('\u0d4e', '\u0d4e'), ('\u0d60', '\u0d61'),
        ('\u0d7a', '\u0d7f'), ('\u0d85', '\u0d96'),
        ('\u0d9a', '\u0db1'), ('\u0db3', '\u0dbb'),
        ('\u0dbd', '\u0dbd'), ('\u0dc0', '\u0dc6'),
        ('\u0e01', '\u0e30'), ('\u0e32', '\u0e32'),
        ('\u0e40', '\u0e45'), ('\u0e46', '\u0e46'),
        ('\u0e81', '\u0e82'), ('\u0e84', '\u0e84'),
        ('\u0e87', '\u0e88'), ('\u0e8a', '\u0e8a'),
        ('\u0e8d', '\u0e8d'), ('\u0e94', '\u0e97'),
        ('\u0e99', '\u0e9f'), ('\u0ea1', '\u0ea3'),
        ('\u0ea5', '\u0ea5'), ('\u0ea7', '\u0ea7'),
        ('\u0eaa', '\u0eab'), ('\u0ead', '\u0eb0'),
        ('\u0eb2', '\u0eb2'), ('\u0ebd', '\u0ebd'),
        ('\u0ec0', '\u0ec4'), ('\u0ec6', '\u0ec6'),
        ('\u0edc', '\u0edf'), ('\u0f00', '\u0f00'),
        ('\u0f40', '\u0f47'), ('\u0f49', '\u0f6c'),
        ('\u0f88', '\u0f8c'), ('\u1000', '\u102a'),
        ('\u103f', '\u103f'), ('\u1050', '\u1055'),
        ('\u105a', '\u105d'), ('\u1061', '\u1061'),
        ('\u1065', '\u1066'), ('\u106e', '\u1070'),
        ('\u1075', '\u1081'), ('\u108e', '\u108e'),
        ('\u10a0', '\u10c5'), ('\u10c7', '\u10c7'),
        ('\u10cd', '\u10cd'), ('\u10d0', '\u10fa'),
        ('\u10fc', '\u10fc'), ('\u10fd', '\u1248'),
        ('\u124a', '\u124d'), ('\u1250', '\u1256'),
        ('\u1258', '\u1258'), ('\u125a', '\u125d'),
        ('\u1260', '\u1288'), ('\u128a', '\u128d'),
        ('\u1290', '\u12b0'), ('\u12b2', '\u12b5'),
        ('\u12b8', '\u12be'), ('\u12c0', '\u12c0'),
        ('\u12c2', '\u12c5'), ('\u12c8', '\u12d6'),
        ('\u12d8', '\u1310'), ('\u1312', '\u1315'),
        ('\u1318', '\u135a'), ('\u1380', '\u138f'),
        ('\u13a0', '\u13f4'), ('\u1401', '\u166c'),
        ('\u166f', '\u167f'), ('\u1681', '\u169a'),
        ('\u16a0', '\u16ea'), ('\u16ee', '\u16f0'),
        ('\u1700', '\u170c'), ('\u170e', '\u1711'),
        ('\u1720', '\u1731'), ('\u1740', '\u1751'),
        ('\u1760', '\u176c'), ('\u176e', '\u1770'),
        ('\u1780', '\u17b3'), ('\u17d7', '\u17d7'),
        ('\u17dc', '\u17dc'), ('\u1820', '\u1842'),
        ('\u1843', '\u1843'), ('\u1844', '\u1877'),
        ('\u1880', '\u18a8'), ('\u18aa', '\u18aa'),
        ('\u18b0', '\u18f5'), ('\u1900', '\u191c'),
        ('\u1950', '\u196d'), ('\u1970', '\u1974'),
        ('\u1980', '\u19ab'), ('\u19c1', '\u19c7'),
        ('\u1a00', '\u1a16'), ('\u1a20', '\u1a54'),
        ('\u1aa7', '\u1aa7'), ('\u1b05', '\u1b33'),
        ('\u1b45', '\u1b4b'), ('\u1b83', '\u1ba0'),
        ('\u1bae', '\u1baf'), ('\u1bba', '\u1be5'),
        ('\u1c00', '\u1c23'), ('\u1c4d', '\u1c4f'),
        ('\u1c5a', '\u1c77'), ('\u1c78', '\u1c7d'),
        ('\u1ce9', '\u1cec'), ('\u1cee', '\u1cf1'),
        ('\u1cf5', '\u1cf6'), ('\u1d00', '\u1d2b'),
        ('\u1d2c', '\u1d6a'), ('\u1d6b', '\u1d77'),
        ('\u1d78', '\u1d78'), ('\u1d79', '\u1d9a'),
        ('\u1d9b', '\u1dbf'), ('\u1e00', '\u1f15'),
        ('\u1f18', '\u1f1d'), ('\u1f20', '\u1f45'),
        ('\u1f48', '\u1f4d'), ('\u1f50', '\u1f57'),
        ('\u1f59', '\u1f59'), ('\u1f5b', '\u1f5b'),
        ('\u1f5d', '\u1f5d'), ('\u1f5f', '\u1f7d'),
        ('\u1f80', '\u1fb4'), ('\u1fb6', '\u1fbc'),
        ('\u1fbe', '\u1fbe'), ('\u1fc2', '\u1fc4'),
        ('\u1fc6', '\u1fcc'), ('\u1fd0', '\u1fd3'),
        ('\u1fd6', '\u1fdb'), ('\u1fe0', '\u1fec'),
        ('\u1ff2', '\u1ff4'), ('\u1ff6', '\u1ffc'),
        ('\u2071', '\u2071'), ('\u207f', '\u207f'),
        ('\u2090', '\u209c'), ('\u2102', '\u2102'),
        ('\u2107', '\u2107'), ('\u210a', '\u2113'),
        ('\u2115', '\u2115'), ('\u2118', '\u2118'),
        ('\u2119', '\u211d'), ('\u2124', '\u2124'),
        ('\u2126', '\u2126'), ('\u2128', '\u2128'),
        ('\u212a', '\u212d'), ('\u212e', '\u212e'),
        ('\u212f', '\u2134'), ('\u2135', '\u2138'),
        ('\u2139', '\u2139'), ('\u213c', '\u213f'),
        ('\u2145', '\u2149'), ('\u214e', '\u214e'),
        ('\u2160', '\u2182'), ('\u2183', '\u2184'),
        ('\u2185', '\u2188'), ('\u2c00', '\u2c2e'),
        ('\u2c30', '\u2c5e'), ('\u2c60', '\u2c7b'),
        ('\u2c7c', '\u2c7d'), ('\u2c7e', '\u2ce4'),
        ('\u2ceb', '\u2cee'), ('\u2cf2', '\u2cf3'),
        ('\u2d00', '\u2d25'), ('\u2d27', '\u2d27'),
        ('\u2d2d', '\u2d2d'), ('\u2d30', '\u2d67'),
        ('\u2d6f', '\u2d6f'), ('\u2d80', '\u2d96'),
        ('\u2da0', '\u2da6'), ('\u2da8', '\u2dae'),
        ('\u2db0', '\u2db6'), ('\u2db8', '\u2dbe'),
        ('\u2dc0', '\u2dc6'), ('\u2dc8', '\u2dce'),
        ('\u2dd0', '\u2dd6'), ('\u2dd8', '\u2dde'),
        ('\u3005', '\u3005'), ('\u3006', '\u3006'),
        ('\u3007', '\u3007'), ('\u3021', '\u3029'),
        ('\u3031', '\u3035'), ('\u3038', '\u303a'),
        ('\u303b', '\u303b'), ('\u303c', '\u303c'),
        ('\u3041', '\u3096'), ('\u309d', '\u309e'),
        ('\u309f', '\u309f'), ('\u30a1', '\u30fa'),
        ('\u30fc', '\u30fe'), ('\u30ff', '\u30ff'),
        ('\u3105', '\u312d'), ('\u3131', '\u318e'),
        ('\u31a0', '\u31ba'), ('\u31f0', '\u31ff'),
        ('\u3400', '\u4db5'), ('\u4e00', '\u9fcc'),
        ('\ua000', '\ua014'), ('\ua015', '\ua015'),
        ('\ua016', '\ua48c'), ('\ua4d0', '\ua4f7'),
        ('\ua4f8', '\ua4fd'), ('\ua500', '\ua60b'),
        ('\ua60c', '\ua60c'), ('\ua610', '\ua61f'),
        ('\ua62a', '\ua62b'), ('\ua640', '\ua66d'),
        ('\ua66e', '\ua66e'), ('\ua67f', '\ua67f'),
        ('\ua680', '\ua697'), ('\ua6a0', '\ua6e5'),
        ('\ua6e6', '\ua6ef'), ('\ua717', '\ua71f'),
        ('\ua722', '\ua76f'), ('\ua770', '\ua770'),
        ('\ua771', '\ua787'), ('\ua788', '\ua788'),
        ('\ua78b', '\ua78e'), ('\ua790', '\ua793'),
        ('\ua7a0', '\ua7aa'), ('\ua7f8', '\ua7f9'),
        ('\ua7fa', '\ua7fa'), ('\ua7fb', '\ua801'),
        ('\ua803', '\ua805'), ('\ua807', '\ua80a'),
        ('\ua80c', '\ua822'), ('\ua840', '\ua873'),
        ('\ua882', '\ua8b3'), ('\ua8f2', '\ua8f7'),
        ('\ua8fb', '\ua8fb'), ('\ua90a', '\ua925'),
        ('\ua930', '\ua946'), ('\ua960', '\ua97c'),
        ('\ua984', '\ua9b2'), ('\ua9cf', '\ua9cf'),
        ('\uaa00', '\uaa28'), ('\uaa40', '\uaa42'),
        ('\uaa44', '\uaa4b'), ('\uaa60', '\uaa6f'),
        ('\uaa70', '\uaa70'), ('\uaa71', '\uaa76'),
        ('\uaa7a', '\uaa7a'), ('\uaa80', '\uaaaf'),
        ('\uaab1', '\uaab1'), ('\uaab5', '\uaab6'),
        ('\uaab9', '\uaabd'), ('\uaac0', '\uaac0'),
        ('\uaac2', '\uaac2'), ('\uaadb', '\uaadc'),
        ('\uaadd', '\uaadd'), ('\uaae0', '\uaaea'),
        ('\uaaf2', '\uaaf2'), ('\uaaf3', '\uaaf4'),
        ('\uab01', '\uab06'), ('\uab09', '\uab0e'),
        ('\uab11', '\uab16'), ('\uab20', '\uab26'),
        ('\uab28', '\uab2e'), ('\uabc0', '\uabe2'),
        ('\uac00', '\ud7a3'), ('\ud7b0', '\ud7c6'),
        ('\ud7cb', '\ud7fb'), ('\uf900', '\ufa6d'),
        ('\ufa70', '\ufad9'), ('\ufb00', '\ufb06'),
        ('\ufb13', '\ufb17'), ('\ufb1d', '\ufb1d'),
        ('\ufb1f', '\ufb28'), ('\ufb2a', '\ufb36'),
        ('\ufb38', '\ufb3c'), ('\ufb3e', '\ufb3e'),
        ('\ufb40', '\ufb41'), ('\ufb43', '\ufb44'),
        ('\ufb46', '\ufbb1'), ('\ufbd3', '\ufc5d'),
        ('\ufc64', '\ufd3d'), ('\ufd50', '\ufd8f'),
        ('\ufd92', '\ufdc7'), ('\ufdf0', '\ufdf9'),
        ('\ufe71', '\ufe71'), ('\ufe73', '\ufe73'),
        ('\ufe77', '\ufe77'), ('\ufe79', '\ufe79'),
        ('\ufe7b', '\ufe7b'), ('\ufe7d', '\ufe7d'),
        ('\ufe7f', '\ufefc'), ('\uff21', '\uff3a'),
        ('\uff41', '\uff5a'), ('\uff66', '\uff6f'),
        ('\uff70', '\uff70'), ('\uff71', '\uff9d'),
        ('\uffa0', '\uffbe'), ('\uffc2', '\uffc7'),
        ('\uffca', '\uffcf'), ('\uffd2', '\uffd7'),
        ('\uffda', '\uffdc'), ('\U00010000', '\U0001000b'),
        ('\U0001000d', '\U00010026'), ('\U00010028', '\U0001003a'),
        ('\U0001003c', '\U0001003d'), ('\U0001003f', '\U0001004d'),
        ('\U00010050', '\U0001005d'), ('\U00010080', '\U000100fa'),
        ('\U00010140', '\U00010174'), ('\U00010280', '\U0001029c'),
        ('\U000102a0', '\U000102d0'), ('\U00010300', '\U0001031e'),
        ('\U00010330', '\U00010340'), ('\U00010341', '\U00010341'),
        ('\U00010342', '\U00010349'), ('\U0001034a', '\U0001034a'),
        ('\U00010380', '\U0001039d'), ('\U000103a0', '\U000103c3'),
        ('\U000103c8', '\U000103cf'), ('\U000103d1', '\U000103d5'),
        ('\U00010400', '\U0001044f'), ('\U00010450', '\U0001049d'),
        ('\U00010800', '\U00010805'), ('\U00010808', '\U00010808'),
        ('\U0001080a', '\U00010835'), ('\U00010837', '\U00010838'),
        ('\U0001083c', '\U0001083c'), ('\U0001083f', '\U00010855'),
        ('\U00010900', '\U00010915'), ('\U00010920', '\U00010939'),
        ('\U00010980', '\U000109b7'), ('\U000109be', '\U000109bf'),
        ('\U00010a00', '\U00010a00'), ('\U00010a10', '\U00010a13'),
        ('\U00010a15', '\U00010a17'), ('\U00010a19', '\U00010a33'),
        ('\U00010a60', '\U00010a7c'), ('\U00010b00', '\U00010b35'),
        ('\U00010b40', '\U00010b55'), ('\U00010b60', '\U00010b72'),
        ('\U00010c00', '\U00010c48'), ('\U00011003', '\U00011037'),
        ('\U00011083', '\U000110af'), ('\U000110d0', '\U000110e8'),
        ('\U00011103', '\U00011126'), ('\U00011183', '\U000111b2'),
        ('\U000111c1', '\U000111c4'), ('\U00011680', '\U000116aa'),
        ('\U00012000', '\U0001236e'), ('\U00012400', '\U00012462'),
        ('\U00013000', '\U0001342e'), ('\U00016800', '\U00016a38'),
        ('\U00016f00', '\U00016f44'), ('\U00016f50', '\U00016f50'),
        ('\U00016f93', '\U00016f9f'), ('\U0001b000', '\U0001b001'),
        ('\U0001d400', '\U0001d454'), ('\U0001d456', '\U0001d49c'),
        ('\U0001d49e', '\U0001d49f'), ('\U0001d4a2', '\U0001d4a2'),
        ('\U0001d4a5', '\U0001d4a6'), ('\U0001d4a9', '\U0001d4ac'),
        ('\U0001d4ae', '\U0001d4b9'), ('\U0001d4bb', '\U0001d4bb'),
        ('\U0001d4bd', '\U0001d4c3'), ('\U0001d4c5', '\U0001d505'),
        ('\U0001d507', '\U0001d50a'), ('\U0001d50d', '\U0001d514'),
        ('\U0001d516', '\U0001d51c'), ('\U0001d51e', '\U0001d539'),
        ('\U0001d53b', '\U0001d53e'), ('\U0001d540', '\U0001d544'),
        ('\U0001d546', '\U0001d546'), ('\U0001d54a', '\U0001d550'),
        ('\U0001d552', '\U0001d6a5'), ('\U0001d6a8', '\U0001d6c0'),
        ('\U0001d6c2', '\U0001d6da'), ('\U0001d6dc', '\U0001d6fa'),
        ('\U0001d6fc', '\U0001d714'), ('\U0001d716', '\U0001d734'),
        ('\U0001d736', '\U0001d74e'), ('\U0001d750', '\U0001d76e'),
        ('\U0001d770', '\U0001d788'), ('\U0001d78a', '\U0001d7a8'),
        ('\U0001d7aa', '\U0001d7c2'), ('\U0001d7c4', '\U0001d7cb'),
        ('\U0001ee00', '\U0001ee03'), ('\U0001ee05', '\U0001ee1f'),
        ('\U0001ee21', '\U0001ee22'), ('\U0001ee24', '\U0001ee24'),
        ('\U0001ee27', '\U0001ee27'), ('\U0001ee29', '\U0001ee32'),
        ('\U0001ee34', '\U0001ee37'), ('\U0001ee39', '\U0001ee39'),
        ('\U0001ee3b', '\U0001ee3b'), ('\U0001ee42', '\U0001ee42'),
        ('\U0001ee47', '\U0001ee47'), ('\U0001ee49', '\U0001ee49'),
        ('\U0001ee4b', '\U0001ee4b'), ('\U0001ee4d', '\U0001ee4f'),
        ('\U0001ee51', '\U0001ee52'), ('\U0001ee54', '\U0001ee54'),
        ('\U0001ee57', '\U0001ee57'), ('\U0001ee59', '\U0001ee59'),
        ('\U0001ee5b', '\U0001ee5b'), ('\U0001ee5d', '\U0001ee5d'),
        ('\U0001ee5f', '\U0001ee5f'), ('\U0001ee61', '\U0001ee62'),
        ('\U0001ee64', '\U0001ee64'), ('\U0001ee67', '\U0001ee6a'),
        ('\U0001ee6c', '\U0001ee72'), ('\U0001ee74', '\U0001ee77'),
        ('\U0001ee79', '\U0001ee7c'), ('\U0001ee7e', '\U0001ee7e'),
        ('\U0001ee80', '\U0001ee89'), ('\U0001ee8b', '\U0001ee9b'),
        ('\U0001eea1', '\U0001eea3'), ('\U0001eea5', '\U0001eea9'),
        ('\U0001eeab', '\U0001eebb'), ('\U00020000', '\U0002a6d6'),
        ('\U0002a700', '\U0002b734'), ('\U0002b740', '\U0002b81d'),
        ('\U0002f800', '\U0002fa1d')
    ];

    pub fn XID_Start(c: char) -> bool {
        super::bsearch_range_table(c, XID_Start_table)
    }
}

pub mod property {
    static White_Space_table : &'static [(char,char)] = &[
        ('\x09', '\x0d'), ('\x20', '\x20'),
        ('\x85', '\x85'), ('\xa0', '\xa0'),
        ('\u1680', '\u1680'), ('\u2000', '\u200a'),
        ('\u2028', '\u2028'), ('\u2029', '\u2029'),
        ('\u202f', '\u202f'), ('\u205f', '\u205f'),
        ('\u3000', '\u3000')
    ];

    pub fn White_Space(c: char) -> bool {
        super::bsearch_range_table(c, White_Space_table)
    }
}

pub mod conversions {
    use cmp::{Equal, Less, Greater};
    use slice::ImmutableVector;
    use tuple::Tuple2;
    use option::{Option, Some, None};

    pub fn to_lower(c: char) -> char {
        match bsearch_case_table(c, LuLl_table) {
          None        => c,
          Some(index) => LuLl_table[index].val1()
        }
    }

    pub fn to_upper(c: char) -> char {
        match bsearch_case_table(c, LlLu_table) {
            None        => c,
            Some(index) => LlLu_table[index].val1()
        }
    }

    fn bsearch_case_table(c: char, table: &'static [(char, char)]) -> Option<uint> {
        table.bsearch(|&(key, _)| {
            if c == key { Equal }
            else if key < c { Less }
            else { Greater }
        })
    }

    static LuLl_table : &'static [(char, char)] = &[
        ('\x41', '\x61'), ('\x42', '\x62'),
        ('\x43', '\x63'), ('\x44', '\x64'),
        ('\x45', '\x65'), ('\x46', '\x66'),
        ('\x47', '\x67'), ('\x48', '\x68'),
        ('\x49', '\x69'), ('\x4a', '\x6a'),
        ('\x4b', '\x6b'), ('\x4c', '\x6c'),
        ('\x4d', '\x6d'), ('\x4e', '\x6e'),
        ('\x4f', '\x6f'), ('\x50', '\x70'),
        ('\x51', '\x71'), ('\x52', '\x72'),
        ('\x53', '\x73'), ('\x54', '\x74'),
        ('\x55', '\x75'), ('\x56', '\x76'),
        ('\x57', '\x77'), ('\x58', '\x78'),
        ('\x59', '\x79'), ('\x5a', '\x7a'),
        ('\xc0', '\xe0'), ('\xc1', '\xe1'),
        ('\xc2', '\xe2'), ('\xc3', '\xe3'),
        ('\xc4', '\xe4'), ('\xc5', '\xe5'),
        ('\xc6', '\xe6'), ('\xc7', '\xe7'),
        ('\xc8', '\xe8'), ('\xc9', '\xe9'),
        ('\xca', '\xea'), ('\xcb', '\xeb'),
        ('\xcc', '\xec'), ('\xcd', '\xed'),
        ('\xce', '\xee'), ('\xcf', '\xef'),
        ('\xd0', '\xf0'), ('\xd1', '\xf1'),
        ('\xd2', '\xf2'), ('\xd3', '\xf3'),
        ('\xd4', '\xf4'), ('\xd5', '\xf5'),
        ('\xd6', '\xf6'), ('\xd8', '\xf8'),
        ('\xd9', '\xf9'), ('\xda', '\xfa'),
        ('\xdb', '\xfb'), ('\xdc', '\xfc'),
        ('\xdd', '\xfd'), ('\xde', '\xfe'),
        ('\u0100', '\u0101'), ('\u0102', '\u0103'),
        ('\u0104', '\u0105'), ('\u0106', '\u0107'),
        ('\u0108', '\u0109'), ('\u010a', '\u010b'),
        ('\u010c', '\u010d'), ('\u010e', '\u010f'),
        ('\u0110', '\u0111'), ('\u0112', '\u0113'),
        ('\u0114', '\u0115'), ('\u0116', '\u0117'),
        ('\u0118', '\u0119'), ('\u011a', '\u011b'),
        ('\u011c', '\u011d'), ('\u011e', '\u011f'),
        ('\u0120', '\u0121'), ('\u0122', '\u0123'),
        ('\u0124', '\u0125'), ('\u0126', '\u0127'),
        ('\u0128', '\u0129'), ('\u012a', '\u012b'),
        ('\u012c', '\u012d'), ('\u012e', '\u012f'),
        ('\u0130', '\x69'), ('\u0132', '\u0133'),
        ('\u0134', '\u0135'), ('\u0136', '\u0137'),
        ('\u0139', '\u013a'), ('\u013b', '\u013c'),
        ('\u013d', '\u013e'), ('\u013f', '\u0140'),
        ('\u0141', '\u0142'), ('\u0143', '\u0144'),
        ('\u0145', '\u0146'), ('\u0147', '\u0148'),
        ('\u014a', '\u014b'), ('\u014c', '\u014d'),
        ('\u014e', '\u014f'), ('\u0150', '\u0151'),
        ('\u0152', '\u0153'), ('\u0154', '\u0155'),
        ('\u0156', '\u0157'), ('\u0158', '\u0159'),
        ('\u015a', '\u015b'), ('\u015c', '\u015d'),
        ('\u015e', '\u015f'), ('\u0160', '\u0161'),
        ('\u0162', '\u0163'), ('\u0164', '\u0165'),
        ('\u0166', '\u0167'), ('\u0168', '\u0169'),
        ('\u016a', '\u016b'), ('\u016c', '\u016d'),
        ('\u016e', '\u016f'), ('\u0170', '\u0171'),
        ('\u0172', '\u0173'), ('\u0174', '\u0175'),
        ('\u0176', '\u0177'), ('\u0178', '\xff'),
        ('\u0179', '\u017a'), ('\u017b', '\u017c'),
        ('\u017d', '\u017e'), ('\u0181', '\u0253'),
        ('\u0182', '\u0183'), ('\u0184', '\u0185'),
        ('\u0186', '\u0254'), ('\u0187', '\u0188'),
        ('\u0189', '\u0256'), ('\u018a', '\u0257'),
        ('\u018b', '\u018c'), ('\u018e', '\u01dd'),
        ('\u018f', '\u0259'), ('\u0190', '\u025b'),
        ('\u0191', '\u0192'), ('\u0193', '\u0260'),
        ('\u0194', '\u0263'), ('\u0196', '\u0269'),
        ('\u0197', '\u0268'), ('\u0198', '\u0199'),
        ('\u019c', '\u026f'), ('\u019d', '\u0272'),
        ('\u019f', '\u0275'), ('\u01a0', '\u01a1'),
        ('\u01a2', '\u01a3'), ('\u01a4', '\u01a5'),
        ('\u01a6', '\u0280'), ('\u01a7', '\u01a8'),
        ('\u01a9', '\u0283'), ('\u01ac', '\u01ad'),
        ('\u01ae', '\u0288'), ('\u01af', '\u01b0'),
        ('\u01b1', '\u028a'), ('\u01b2', '\u028b'),
        ('\u01b3', '\u01b4'), ('\u01b5', '\u01b6'),
        ('\u01b7', '\u0292'), ('\u01b8', '\u01b9'),
        ('\u01bc', '\u01bd'), ('\u01c4', '\u01c6'),
        ('\u01c7', '\u01c9'), ('\u01ca', '\u01cc'),
        ('\u01cd', '\u01ce'), ('\u01cf', '\u01d0'),
        ('\u01d1', '\u01d2'), ('\u01d3', '\u01d4'),
        ('\u01d5', '\u01d6'), ('\u01d7', '\u01d8'),
        ('\u01d9', '\u01da'), ('\u01db', '\u01dc'),
        ('\u01de', '\u01df'), ('\u01e0', '\u01e1'),
        ('\u01e2', '\u01e3'), ('\u01e4', '\u01e5'),
        ('\u01e6', '\u01e7'), ('\u01e8', '\u01e9'),
        ('\u01ea', '\u01eb'), ('\u01ec', '\u01ed'),
        ('\u01ee', '\u01ef'), ('\u01f1', '\u01f3'),
        ('\u01f4', '\u01f5'), ('\u01f6', '\u0195'),
        ('\u01f7', '\u01bf'), ('\u01f8', '\u01f9'),
        ('\u01fa', '\u01fb'), ('\u01fc', '\u01fd'),
        ('\u01fe', '\u01ff'), ('\u0200', '\u0201'),
        ('\u0202', '\u0203'), ('\u0204', '\u0205'),
        ('\u0206', '\u0207'), ('\u0208', '\u0209'),
        ('\u020a', '\u020b'), ('\u020c', '\u020d'),
        ('\u020e', '\u020f'), ('\u0210', '\u0211'),
        ('\u0212', '\u0213'), ('\u0214', '\u0215'),
        ('\u0216', '\u0217'), ('\u0218', '\u0219'),
        ('\u021a', '\u021b'), ('\u021c', '\u021d'),
        ('\u021e', '\u021f'), ('\u0220', '\u019e'),
        ('\u0222', '\u0223'), ('\u0224', '\u0225'),
        ('\u0226', '\u0227'), ('\u0228', '\u0229'),
        ('\u022a', '\u022b'), ('\u022c', '\u022d'),
        ('\u022e', '\u022f'), ('\u0230', '\u0231'),
        ('\u0232', '\u0233'), ('\u023a', '\u2c65'),
        ('\u023b', '\u023c'), ('\u023d', '\u019a'),
        ('\u023e', '\u2c66'), ('\u0241', '\u0242'),
        ('\u0243', '\u0180'), ('\u0244', '\u0289'),
        ('\u0245', '\u028c'), ('\u0246', '\u0247'),
        ('\u0248', '\u0249'), ('\u024a', '\u024b'),
        ('\u024c', '\u024d'), ('\u024e', '\u024f'),
        ('\u0370', '\u0371'), ('\u0372', '\u0373'),
        ('\u0376', '\u0377'), ('\u0386', '\u03ac'),
        ('\u0388', '\u03ad'), ('\u0389', '\u03ae'),
        ('\u038a', '\u03af'), ('\u038c', '\u03cc'),
        ('\u038e', '\u03cd'), ('\u038f', '\u03ce'),
        ('\u0391', '\u03b1'), ('\u0392', '\u03b2'),
        ('\u0393', '\u03b3'), ('\u0394', '\u03b4'),
        ('\u0395', '\u03b5'), ('\u0396', '\u03b6'),
        ('\u0397', '\u03b7'), ('\u0398', '\u03b8'),
        ('\u0399', '\u03b9'), ('\u039a', '\u03ba'),
        ('\u039b', '\u03bb'), ('\u039c', '\u03bc'),
        ('\u039d', '\u03bd'), ('\u039e', '\u03be'),
        ('\u039f', '\u03bf'), ('\u03a0', '\u03c0'),
        ('\u03a1', '\u03c1'), ('\u03a3', '\u03c3'),
        ('\u03a4', '\u03c4'), ('\u03a5', '\u03c5'),
        ('\u03a6', '\u03c6'), ('\u03a7', '\u03c7'),
        ('\u03a8', '\u03c8'), ('\u03a9', '\u03c9'),
        ('\u03aa', '\u03ca'), ('\u03ab', '\u03cb'),
        ('\u03cf', '\u03d7'), ('\u03d8', '\u03d9'),
        ('\u03da', '\u03db'), ('\u03dc', '\u03dd'),
        ('\u03de', '\u03df'), ('\u03e0', '\u03e1'),
        ('\u03e2', '\u03e3'), ('\u03e4', '\u03e5'),
        ('\u03e6', '\u03e7'), ('\u03e8', '\u03e9'),
        ('\u03ea', '\u03eb'), ('\u03ec', '\u03ed'),
        ('\u03ee', '\u03ef'), ('\u03f4', '\u03b8'),
        ('\u03f7', '\u03f8'), ('\u03f9', '\u03f2'),
        ('\u03fa', '\u03fb'), ('\u03fd', '\u037b'),
        ('\u03fe', '\u037c'), ('\u03ff', '\u037d'),
        ('\u0400', '\u0450'), ('\u0401', '\u0451'),
        ('\u0402', '\u0452'), ('\u0403', '\u0453'),
        ('\u0404', '\u0454'), ('\u0405', '\u0455'),
        ('\u0406', '\u0456'), ('\u0407', '\u0457'),
        ('\u0408', '\u0458'), ('\u0409', '\u0459'),
        ('\u040a', '\u045a'), ('\u040b', '\u045b'),
        ('\u040c', '\u045c'), ('\u040d', '\u045d'),
        ('\u040e', '\u045e'), ('\u040f', '\u045f'),
        ('\u0410', '\u0430'), ('\u0411', '\u0431'),
        ('\u0412', '\u0432'), ('\u0413', '\u0433'),
        ('\u0414', '\u0434'), ('\u0415', '\u0435'),
        ('\u0416', '\u0436'), ('\u0417', '\u0437'),
        ('\u0418', '\u0438'), ('\u0419', '\u0439'),
        ('\u041a', '\u043a'), ('\u041b', '\u043b'),
        ('\u041c', '\u043c'), ('\u041d', '\u043d'),
        ('\u041e', '\u043e'), ('\u041f', '\u043f'),
        ('\u0420', '\u0440'), ('\u0421', '\u0441'),
        ('\u0422', '\u0442'), ('\u0423', '\u0443'),
        ('\u0424', '\u0444'), ('\u0425', '\u0445'),
        ('\u0426', '\u0446'), ('\u0427', '\u0447'),
        ('\u0428', '\u0448'), ('\u0429', '\u0449'),
        ('\u042a', '\u044a'), ('\u042b', '\u044b'),
        ('\u042c', '\u044c'), ('\u042d', '\u044d'),
        ('\u042e', '\u044e'), ('\u042f', '\u044f'),
        ('\u0460', '\u0461'), ('\u0462', '\u0463'),
        ('\u0464', '\u0465'), ('\u0466', '\u0467'),
        ('\u0468', '\u0469'), ('\u046a', '\u046b'),
        ('\u046c', '\u046d'), ('\u046e', '\u046f'),
        ('\u0470', '\u0471'), ('\u0472', '\u0473'),
        ('\u0474', '\u0475'), ('\u0476', '\u0477'),
        ('\u0478', '\u0479'), ('\u047a', '\u047b'),
        ('\u047c', '\u047d'), ('\u047e', '\u047f'),
        ('\u0480', '\u0481'), ('\u048a', '\u048b'),
        ('\u048c', '\u048d'), ('\u048e', '\u048f'),
        ('\u0490', '\u0491'), ('\u0492', '\u0493'),
        ('\u0494', '\u0495'), ('\u0496', '\u0497'),
        ('\u0498', '\u0499'), ('\u049a', '\u049b'),
        ('\u049c', '\u049d'), ('\u049e', '\u049f'),
        ('\u04a0', '\u04a1'), ('\u04a2', '\u04a3'),
        ('\u04a4', '\u04a5'), ('\u04a6', '\u04a7'),
        ('\u04a8', '\u04a9'), ('\u04aa', '\u04ab'),
        ('\u04ac', '\u04ad'), ('\u04ae', '\u04af'),
        ('\u04b0', '\u04b1'), ('\u04b2', '\u04b3'),
        ('\u04b4', '\u04b5'), ('\u04b6', '\u04b7'),
        ('\u04b8', '\u04b9'), ('\u04ba', '\u04bb'),
        ('\u04bc', '\u04bd'), ('\u04be', '\u04bf'),
        ('\u04c0', '\u04cf'), ('\u04c1', '\u04c2'),
        ('\u04c3', '\u04c4'), ('\u04c5', '\u04c6'),
        ('\u04c7', '\u04c8'), ('\u04c9', '\u04ca'),
        ('\u04cb', '\u04cc'), ('\u04cd', '\u04ce'),
        ('\u04d0', '\u04d1'), ('\u04d2', '\u04d3'),
        ('\u04d4', '\u04d5'), ('\u04d6', '\u04d7'),
        ('\u04d8', '\u04d9'), ('\u04da', '\u04db'),
        ('\u04dc', '\u04dd'), ('\u04de', '\u04df'),
        ('\u04e0', '\u04e1'), ('\u04e2', '\u04e3'),
        ('\u04e4', '\u04e5'), ('\u04e6', '\u04e7'),
        ('\u04e8', '\u04e9'), ('\u04ea', '\u04eb'),
        ('\u04ec', '\u04ed'), ('\u04ee', '\u04ef'),
        ('\u04f0', '\u04f1'), ('\u04f2', '\u04f3'),
        ('\u04f4', '\u04f5'), ('\u04f6', '\u04f7'),
        ('\u04f8', '\u04f9'), ('\u04fa', '\u04fb'),
        ('\u04fc', '\u04fd'), ('\u04fe', '\u04ff'),
        ('\u0500', '\u0501'), ('\u0502', '\u0503'),
        ('\u0504', '\u0505'), ('\u0506', '\u0507'),
        ('\u0508', '\u0509'), ('\u050a', '\u050b'),
        ('\u050c', '\u050d'), ('\u050e', '\u050f'),
        ('\u0510', '\u0511'), ('\u0512', '\u0513'),
        ('\u0514', '\u0515'), ('\u0516', '\u0517'),
        ('\u0518', '\u0519'), ('\u051a', '\u051b'),
        ('\u051c', '\u051d'), ('\u051e', '\u051f'),
        ('\u0520', '\u0521'), ('\u0522', '\u0523'),
        ('\u0524', '\u0525'), ('\u0526', '\u0527'),
        ('\u0531', '\u0561'), ('\u0532', '\u0562'),
        ('\u0533', '\u0563'), ('\u0534', '\u0564'),
        ('\u0535', '\u0565'), ('\u0536', '\u0566'),
        ('\u0537', '\u0567'), ('\u0538', '\u0568'),
        ('\u0539', '\u0569'), ('\u053a', '\u056a'),
        ('\u053b', '\u056b'), ('\u053c', '\u056c'),
        ('\u053d', '\u056d'), ('\u053e', '\u056e'),
        ('\u053f', '\u056f'), ('\u0540', '\u0570'),
        ('\u0541', '\u0571'), ('\u0542', '\u0572'),
        ('\u0543', '\u0573'), ('\u0544', '\u0574'),
        ('\u0545', '\u0575'), ('\u0546', '\u0576'),
        ('\u0547', '\u0577'), ('\u0548', '\u0578'),
        ('\u0549', '\u0579'), ('\u054a', '\u057a'),
        ('\u054b', '\u057b'), ('\u054c', '\u057c'),
        ('\u054d', '\u057d'), ('\u054e', '\u057e'),
        ('\u054f', '\u057f'), ('\u0550', '\u0580'),
        ('\u0551', '\u0581'), ('\u0552', '\u0582'),
        ('\u0553', '\u0583'), ('\u0554', '\u0584'),
        ('\u0555', '\u0585'), ('\u0556', '\u0586'),
        ('\u10a0', '\u2d00'), ('\u10a1', '\u2d01'),
        ('\u10a2', '\u2d02'), ('\u10a3', '\u2d03'),
        ('\u10a4', '\u2d04'), ('\u10a5', '\u2d05'),
        ('\u10a6', '\u2d06'), ('\u10a7', '\u2d07'),
        ('\u10a8', '\u2d08'), ('\u10a9', '\u2d09'),
        ('\u10aa', '\u2d0a'), ('\u10ab', '\u2d0b'),
        ('\u10ac', '\u2d0c'), ('\u10ad', '\u2d0d'),
        ('\u10ae', '\u2d0e'), ('\u10af', '\u2d0f'),
        ('\u10b0', '\u2d10'), ('\u10b1', '\u2d11'),
        ('\u10b2', '\u2d12'), ('\u10b3', '\u2d13'),
        ('\u10b4', '\u2d14'), ('\u10b5', '\u2d15'),
        ('\u10b6', '\u2d16'), ('\u10b7', '\u2d17'),
        ('\u10b8', '\u2d18'), ('\u10b9', '\u2d19'),
        ('\u10ba', '\u2d1a'), ('\u10bb', '\u2d1b'),
        ('\u10bc', '\u2d1c'), ('\u10bd', '\u2d1d'),
        ('\u10be', '\u2d1e'), ('\u10bf', '\u2d1f'),
        ('\u10c0', '\u2d20'), ('\u10c1', '\u2d21'),
        ('\u10c2', '\u2d22'), ('\u10c3', '\u2d23'),
        ('\u10c4', '\u2d24'), ('\u10c5', '\u2d25'),
        ('\u10c7', '\u2d27'), ('\u10cd', '\u2d2d'),
        ('\u1e00', '\u1e01'), ('\u1e02', '\u1e03'),
        ('\u1e04', '\u1e05'), ('\u1e06', '\u1e07'),
        ('\u1e08', '\u1e09'), ('\u1e0a', '\u1e0b'),
        ('\u1e0c', '\u1e0d'), ('\u1e0e', '\u1e0f'),
        ('\u1e10', '\u1e11'), ('\u1e12', '\u1e13'),
        ('\u1e14', '\u1e15'), ('\u1e16', '\u1e17'),
        ('\u1e18', '\u1e19'), ('\u1e1a', '\u1e1b'),
        ('\u1e1c', '\u1e1d'), ('\u1e1e', '\u1e1f'),
        ('\u1e20', '\u1e21'), ('\u1e22', '\u1e23'),
        ('\u1e24', '\u1e25'), ('\u1e26', '\u1e27'),
        ('\u1e28', '\u1e29'), ('\u1e2a', '\u1e2b'),
        ('\u1e2c', '\u1e2d'), ('\u1e2e', '\u1e2f'),
        ('\u1e30', '\u1e31'), ('\u1e32', '\u1e33'),
        ('\u1e34', '\u1e35'), ('\u1e36', '\u1e37'),
        ('\u1e38', '\u1e39'), ('\u1e3a', '\u1e3b'),
        ('\u1e3c', '\u1e3d'), ('\u1e3e', '\u1e3f'),
        ('\u1e40', '\u1e41'), ('\u1e42', '\u1e43'),
        ('\u1e44', '\u1e45'), ('\u1e46', '\u1e47'),
        ('\u1e48', '\u1e49'), ('\u1e4a', '\u1e4b'),
        ('\u1e4c', '\u1e4d'), ('\u1e4e', '\u1e4f'),
        ('\u1e50', '\u1e51'), ('\u1e52', '\u1e53'),
        ('\u1e54', '\u1e55'), ('\u1e56', '\u1e57'),
        ('\u1e58', '\u1e59'), ('\u1e5a', '\u1e5b'),
        ('\u1e5c', '\u1e5d'), ('\u1e5e', '\u1e5f'),
        ('\u1e60', '\u1e61'), ('\u1e62', '\u1e63'),
        ('\u1e64', '\u1e65'), ('\u1e66', '\u1e67'),
        ('\u1e68', '\u1e69'), ('\u1e6a', '\u1e6b'),
        ('\u1e6c', '\u1e6d'), ('\u1e6e', '\u1e6f'),
        ('\u1e70', '\u1e71'), ('\u1e72', '\u1e73'),
        ('\u1e74', '\u1e75'), ('\u1e76', '\u1e77'),
        ('\u1e78', '\u1e79'), ('\u1e7a', '\u1e7b'),
        ('\u1e7c', '\u1e7d'), ('\u1e7e', '\u1e7f'),
        ('\u1e80', '\u1e81'), ('\u1e82', '\u1e83'),
        ('\u1e84', '\u1e85'), ('\u1e86', '\u1e87'),
        ('\u1e88', '\u1e89'), ('\u1e8a', '\u1e8b'),
        ('\u1e8c', '\u1e8d'), ('\u1e8e', '\u1e8f'),
        ('\u1e90', '\u1e91'), ('\u1e92', '\u1e93'),
        ('\u1e94', '\u1e95'), ('\u1e9e', '\xdf'),
        ('\u1ea0', '\u1ea1'), ('\u1ea2', '\u1ea3'),
        ('\u1ea4', '\u1ea5'), ('\u1ea6', '\u1ea7'),
        ('\u1ea8', '\u1ea9'), ('\u1eaa', '\u1eab'),
        ('\u1eac', '\u1ead'), ('\u1eae', '\u1eaf'),
        ('\u1eb0', '\u1eb1'), ('\u1eb2', '\u1eb3'),
        ('\u1eb4', '\u1eb5'), ('\u1eb6', '\u1eb7'),
        ('\u1eb8', '\u1eb9'), ('\u1eba', '\u1ebb'),
        ('\u1ebc', '\u1ebd'), ('\u1ebe', '\u1ebf'),
        ('\u1ec0', '\u1ec1'), ('\u1ec2', '\u1ec3'),
        ('\u1ec4', '\u1ec5'), ('\u1ec6', '\u1ec7'),
        ('\u1ec8', '\u1ec9'), ('\u1eca', '\u1ecb'),
        ('\u1ecc', '\u1ecd'), ('\u1ece', '\u1ecf'),
        ('\u1ed0', '\u1ed1'), ('\u1ed2', '\u1ed3'),
        ('\u1ed4', '\u1ed5'), ('\u1ed6', '\u1ed7'),
        ('\u1ed8', '\u1ed9'), ('\u1eda', '\u1edb'),
        ('\u1edc', '\u1edd'), ('\u1ede', '\u1edf'),
        ('\u1ee0', '\u1ee1'), ('\u1ee2', '\u1ee3'),
        ('\u1ee4', '\u1ee5'), ('\u1ee6', '\u1ee7'),
        ('\u1ee8', '\u1ee9'), ('\u1eea', '\u1eeb'),
        ('\u1eec', '\u1eed'), ('\u1eee', '\u1eef'),
        ('\u1ef0', '\u1ef1'), ('\u1ef2', '\u1ef3'),
        ('\u1ef4', '\u1ef5'), ('\u1ef6', '\u1ef7'),
        ('\u1ef8', '\u1ef9'), ('\u1efa', '\u1efb'),
        ('\u1efc', '\u1efd'), ('\u1efe', '\u1eff'),
        ('\u1f08', '\u1f00'), ('\u1f09', '\u1f01'),
        ('\u1f0a', '\u1f02'), ('\u1f0b', '\u1f03'),
        ('\u1f0c', '\u1f04'), ('\u1f0d', '\u1f05'),
        ('\u1f0e', '\u1f06'), ('\u1f0f', '\u1f07'),
        ('\u1f18', '\u1f10'), ('\u1f19', '\u1f11'),
        ('\u1f1a', '\u1f12'), ('\u1f1b', '\u1f13'),
        ('\u1f1c', '\u1f14'), ('\u1f1d', '\u1f15'),
        ('\u1f28', '\u1f20'), ('\u1f29', '\u1f21'),
        ('\u1f2a', '\u1f22'), ('\u1f2b', '\u1f23'),
        ('\u1f2c', '\u1f24'), ('\u1f2d', '\u1f25'),
        ('\u1f2e', '\u1f26'), ('\u1f2f', '\u1f27'),
        ('\u1f38', '\u1f30'), ('\u1f39', '\u1f31'),
        ('\u1f3a', '\u1f32'), ('\u1f3b', '\u1f33'),
        ('\u1f3c', '\u1f34'), ('\u1f3d', '\u1f35'),
        ('\u1f3e', '\u1f36'), ('\u1f3f', '\u1f37'),
        ('\u1f48', '\u1f40'), ('\u1f49', '\u1f41'),
        ('\u1f4a', '\u1f42'), ('\u1f4b', '\u1f43'),
        ('\u1f4c', '\u1f44'), ('\u1f4d', '\u1f45'),
        ('\u1f59', '\u1f51'), ('\u1f5b', '\u1f53'),
        ('\u1f5d', '\u1f55'), ('\u1f5f', '\u1f57'),
        ('\u1f68', '\u1f60'), ('\u1f69', '\u1f61'),
        ('\u1f6a', '\u1f62'), ('\u1f6b', '\u1f63'),
        ('\u1f6c', '\u1f64'), ('\u1f6d', '\u1f65'),
        ('\u1f6e', '\u1f66'), ('\u1f6f', '\u1f67'),
        ('\u1fb8', '\u1fb0'), ('\u1fb9', '\u1fb1'),
        ('\u1fba', '\u1f70'), ('\u1fbb', '\u1f71'),
        ('\u1fc8', '\u1f72'), ('\u1fc9', '\u1f73'),
        ('\u1fca', '\u1f74'), ('\u1fcb', '\u1f75'),
        ('\u1fd8', '\u1fd0'), ('\u1fd9', '\u1fd1'),
        ('\u1fda', '\u1f76'), ('\u1fdb', '\u1f77'),
        ('\u1fe8', '\u1fe0'), ('\u1fe9', '\u1fe1'),
        ('\u1fea', '\u1f7a'), ('\u1feb', '\u1f7b'),
        ('\u1fec', '\u1fe5'), ('\u1ff8', '\u1f78'),
        ('\u1ff9', '\u1f79'), ('\u1ffa', '\u1f7c'),
        ('\u1ffb', '\u1f7d'), ('\u2126', '\u03c9'),
        ('\u212a', '\x6b'), ('\u212b', '\xe5'),
        ('\u2132', '\u214e'), ('\u2183', '\u2184'),
        ('\u2c00', '\u2c30'), ('\u2c01', '\u2c31'),
        ('\u2c02', '\u2c32'), ('\u2c03', '\u2c33'),
        ('\u2c04', '\u2c34'), ('\u2c05', '\u2c35'),
        ('\u2c06', '\u2c36'), ('\u2c07', '\u2c37'),
        ('\u2c08', '\u2c38'), ('\u2c09', '\u2c39'),
        ('\u2c0a', '\u2c3a'), ('\u2c0b', '\u2c3b'),
        ('\u2c0c', '\u2c3c'), ('\u2c0d', '\u2c3d'),
        ('\u2c0e', '\u2c3e'), ('\u2c0f', '\u2c3f'),
        ('\u2c10', '\u2c40'), ('\u2c11', '\u2c41'),
        ('\u2c12', '\u2c42'), ('\u2c13', '\u2c43'),
        ('\u2c14', '\u2c44'), ('\u2c15', '\u2c45'),
        ('\u2c16', '\u2c46'), ('\u2c17', '\u2c47'),
        ('\u2c18', '\u2c48'), ('\u2c19', '\u2c49'),
        ('\u2c1a', '\u2c4a'), ('\u2c1b', '\u2c4b'),
        ('\u2c1c', '\u2c4c'), ('\u2c1d', '\u2c4d'),
        ('\u2c1e', '\u2c4e'), ('\u2c1f', '\u2c4f'),
        ('\u2c20', '\u2c50'), ('\u2c21', '\u2c51'),
        ('\u2c22', '\u2c52'), ('\u2c23', '\u2c53'),
        ('\u2c24', '\u2c54'), ('\u2c25', '\u2c55'),
        ('\u2c26', '\u2c56'), ('\u2c27', '\u2c57'),
        ('\u2c28', '\u2c58'), ('\u2c29', '\u2c59'),
        ('\u2c2a', '\u2c5a'), ('\u2c2b', '\u2c5b'),
        ('\u2c2c', '\u2c5c'), ('\u2c2d', '\u2c5d'),
        ('\u2c2e', '\u2c5e'), ('\u2c60', '\u2c61'),
        ('\u2c62', '\u026b'), ('\u2c63', '\u1d7d'),
        ('\u2c64', '\u027d'), ('\u2c67', '\u2c68'),
        ('\u2c69', '\u2c6a'), ('\u2c6b', '\u2c6c'),
        ('\u2c6d', '\u0251'), ('\u2c6e', '\u0271'),
        ('\u2c6f', '\u0250'), ('\u2c70', '\u0252'),
        ('\u2c72', '\u2c73'), ('\u2c75', '\u2c76'),
        ('\u2c7e', '\u023f'), ('\u2c7f', '\u0240'),
        ('\u2c80', '\u2c81'), ('\u2c82', '\u2c83'),
        ('\u2c84', '\u2c85'), ('\u2c86', '\u2c87'),
        ('\u2c88', '\u2c89'), ('\u2c8a', '\u2c8b'),
        ('\u2c8c', '\u2c8d'), ('\u2c8e', '\u2c8f'),
        ('\u2c90', '\u2c91'), ('\u2c92', '\u2c93'),
        ('\u2c94', '\u2c95'), ('\u2c96', '\u2c97'),
        ('\u2c98', '\u2c99'), ('\u2c9a', '\u2c9b'),
        ('\u2c9c', '\u2c9d'), ('\u2c9e', '\u2c9f'),
        ('\u2ca0', '\u2ca1'), ('\u2ca2', '\u2ca3'),
        ('\u2ca4', '\u2ca5'), ('\u2ca6', '\u2ca7'),
        ('\u2ca8', '\u2ca9'), ('\u2caa', '\u2cab'),
        ('\u2cac', '\u2cad'), ('\u2cae', '\u2caf'),
        ('\u2cb0', '\u2cb1'), ('\u2cb2', '\u2cb3'),
        ('\u2cb4', '\u2cb5'), ('\u2cb6', '\u2cb7'),
        ('\u2cb8', '\u2cb9'), ('\u2cba', '\u2cbb'),
        ('\u2cbc', '\u2cbd'), ('\u2cbe', '\u2cbf'),
        ('\u2cc0', '\u2cc1'), ('\u2cc2', '\u2cc3'),
        ('\u2cc4', '\u2cc5'), ('\u2cc6', '\u2cc7'),
        ('\u2cc8', '\u2cc9'), ('\u2cca', '\u2ccb'),
        ('\u2ccc', '\u2ccd'), ('\u2cce', '\u2ccf'),
        ('\u2cd0', '\u2cd1'), ('\u2cd2', '\u2cd3'),
        ('\u2cd4', '\u2cd5'), ('\u2cd6', '\u2cd7'),
        ('\u2cd8', '\u2cd9'), ('\u2cda', '\u2cdb'),
        ('\u2cdc', '\u2cdd'), ('\u2cde', '\u2cdf'),
        ('\u2ce0', '\u2ce1'), ('\u2ce2', '\u2ce3'),
        ('\u2ceb', '\u2cec'), ('\u2ced', '\u2cee'),
        ('\u2cf2', '\u2cf3'), ('\ua640', '\ua641'),
        ('\ua642', '\ua643'), ('\ua644', '\ua645'),
        ('\ua646', '\ua647'), ('\ua648', '\ua649'),
        ('\ua64a', '\ua64b'), ('\ua64c', '\ua64d'),
        ('\ua64e', '\ua64f'), ('\ua650', '\ua651'),
        ('\ua652', '\ua653'), ('\ua654', '\ua655'),
        ('\ua656', '\ua657'), ('\ua658', '\ua659'),
        ('\ua65a', '\ua65b'), ('\ua65c', '\ua65d'),
        ('\ua65e', '\ua65f'), ('\ua660', '\ua661'),
        ('\ua662', '\ua663'), ('\ua664', '\ua665'),
        ('\ua666', '\ua667'), ('\ua668', '\ua669'),
        ('\ua66a', '\ua66b'), ('\ua66c', '\ua66d'),
        ('\ua680', '\ua681'), ('\ua682', '\ua683'),
        ('\ua684', '\ua685'), ('\ua686', '\ua687'),
        ('\ua688', '\ua689'), ('\ua68a', '\ua68b'),
        ('\ua68c', '\ua68d'), ('\ua68e', '\ua68f'),
        ('\ua690', '\ua691'), ('\ua692', '\ua693'),
        ('\ua694', '\ua695'), ('\ua696', '\ua697'),
        ('\ua722', '\ua723'), ('\ua724', '\ua725'),
        ('\ua726', '\ua727'), ('\ua728', '\ua729'),
        ('\ua72a', '\ua72b'), ('\ua72c', '\ua72d'),
        ('\ua72e', '\ua72f'), ('\ua732', '\ua733'),
        ('\ua734', '\ua735'), ('\ua736', '\ua737'),
        ('\ua738', '\ua739'), ('\ua73a', '\ua73b'),
        ('\ua73c', '\ua73d'), ('\ua73e', '\ua73f'),
        ('\ua740', '\ua741'), ('\ua742', '\ua743'),
        ('\ua744', '\ua745'), ('\ua746', '\ua747'),
        ('\ua748', '\ua749'), ('\ua74a', '\ua74b'),
        ('\ua74c', '\ua74d'), ('\ua74e', '\ua74f'),
        ('\ua750', '\ua751'), ('\ua752', '\ua753'),
        ('\ua754', '\ua755'), ('\ua756', '\ua757'),
        ('\ua758', '\ua759'), ('\ua75a', '\ua75b'),
        ('\ua75c', '\ua75d'), ('\ua75e', '\ua75f'),
        ('\ua760', '\ua761'), ('\ua762', '\ua763'),
        ('\ua764', '\ua765'), ('\ua766', '\ua767'),
        ('\ua768', '\ua769'), ('\ua76a', '\ua76b'),
        ('\ua76c', '\ua76d'), ('\ua76e', '\ua76f'),
        ('\ua779', '\ua77a'), ('\ua77b', '\ua77c'),
        ('\ua77d', '\u1d79'), ('\ua77e', '\ua77f'),
        ('\ua780', '\ua781'), ('\ua782', '\ua783'),
        ('\ua784', '\ua785'), ('\ua786', '\ua787'),
        ('\ua78b', '\ua78c'), ('\ua78d', '\u0265'),
        ('\ua790', '\ua791'), ('\ua792', '\ua793'),
        ('\ua7a0', '\ua7a1'), ('\ua7a2', '\ua7a3'),
        ('\ua7a4', '\ua7a5'), ('\ua7a6', '\ua7a7'),
        ('\ua7a8', '\ua7a9'), ('\ua7aa', '\u0266'),
        ('\uff21', '\uff41'), ('\uff22', '\uff42'),
        ('\uff23', '\uff43'), ('\uff24', '\uff44'),
        ('\uff25', '\uff45'), ('\uff26', '\uff46'),
        ('\uff27', '\uff47'), ('\uff28', '\uff48'),
        ('\uff29', '\uff49'), ('\uff2a', '\uff4a'),
        ('\uff2b', '\uff4b'), ('\uff2c', '\uff4c'),
        ('\uff2d', '\uff4d'), ('\uff2e', '\uff4e'),
        ('\uff2f', '\uff4f'), ('\uff30', '\uff50'),
        ('\uff31', '\uff51'), ('\uff32', '\uff52'),
        ('\uff33', '\uff53'), ('\uff34', '\uff54'),
        ('\uff35', '\uff55'), ('\uff36', '\uff56'),
        ('\uff37', '\uff57'), ('\uff38', '\uff58'),
        ('\uff39', '\uff59'), ('\uff3a', '\uff5a'),
        ('\U00010400', '\U00010428'), ('\U00010401', '\U00010429'),
        ('\U00010402', '\U0001042a'), ('\U00010403', '\U0001042b'),
        ('\U00010404', '\U0001042c'), ('\U00010405', '\U0001042d'),
        ('\U00010406', '\U0001042e'), ('\U00010407', '\U0001042f'),
        ('\U00010408', '\U00010430'), ('\U00010409', '\U00010431'),
        ('\U0001040a', '\U00010432'), ('\U0001040b', '\U00010433'),
        ('\U0001040c', '\U00010434'), ('\U0001040d', '\U00010435'),
        ('\U0001040e', '\U00010436'), ('\U0001040f', '\U00010437'),
        ('\U00010410', '\U00010438'), ('\U00010411', '\U00010439'),
        ('\U00010412', '\U0001043a'), ('\U00010413', '\U0001043b'),
        ('\U00010414', '\U0001043c'), ('\U00010415', '\U0001043d'),
        ('\U00010416', '\U0001043e'), ('\U00010417', '\U0001043f'),
        ('\U00010418', '\U00010440'), ('\U00010419', '\U00010441'),
        ('\U0001041a', '\U00010442'), ('\U0001041b', '\U00010443'),
        ('\U0001041c', '\U00010444'), ('\U0001041d', '\U00010445'),
        ('\U0001041e', '\U00010446'), ('\U0001041f', '\U00010447'),
        ('\U00010420', '\U00010448'), ('\U00010421', '\U00010449'),
        ('\U00010422', '\U0001044a'), ('\U00010423', '\U0001044b'),
        ('\U00010424', '\U0001044c'), ('\U00010425', '\U0001044d'),
        ('\U00010426', '\U0001044e'), ('\U00010427', '\U0001044f')
    ];

   static LlLu_table : &'static [(char, char)] = &[
        ('\x61', '\x41'), ('\x62', '\x42'),
        ('\x63', '\x43'), ('\x64', '\x44'),
        ('\x65', '\x45'), ('\x66', '\x46'),
        ('\x67', '\x47'), ('\x68', '\x48'),
        ('\x69', '\x49'), ('\x6a', '\x4a'),
        ('\x6b', '\x4b'), ('\x6c', '\x4c'),
        ('\x6d', '\x4d'), ('\x6e', '\x4e'),
        ('\x6f', '\x4f'), ('\x70', '\x50'),
        ('\x71', '\x51'), ('\x72', '\x52'),
        ('\x73', '\x53'), ('\x74', '\x54'),
        ('\x75', '\x55'), ('\x76', '\x56'),
        ('\x77', '\x57'), ('\x78', '\x58'),
        ('\x79', '\x59'), ('\x7a', '\x5a'),
        ('\xb5', '\u039c'), ('\xe0', '\xc0'),
        ('\xe1', '\xc1'), ('\xe2', '\xc2'),
        ('\xe3', '\xc3'), ('\xe4', '\xc4'),
        ('\xe5', '\xc5'), ('\xe6', '\xc6'),
        ('\xe7', '\xc7'), ('\xe8', '\xc8'),
        ('\xe9', '\xc9'), ('\xea', '\xca'),
        ('\xeb', '\xcb'), ('\xec', '\xcc'),
        ('\xed', '\xcd'), ('\xee', '\xce'),
        ('\xef', '\xcf'), ('\xf0', '\xd0'),
        ('\xf1', '\xd1'), ('\xf2', '\xd2'),
        ('\xf3', '\xd3'), ('\xf4', '\xd4'),
        ('\xf5', '\xd5'), ('\xf6', '\xd6'),
        ('\xf8', '\xd8'), ('\xf9', '\xd9'),
        ('\xfa', '\xda'), ('\xfb', '\xdb'),
        ('\xfc', '\xdc'), ('\xfd', '\xdd'),
        ('\xfe', '\xde'), ('\xff', '\u0178'),
        ('\u0101', '\u0100'), ('\u0103', '\u0102'),
        ('\u0105', '\u0104'), ('\u0107', '\u0106'),
        ('\u0109', '\u0108'), ('\u010b', '\u010a'),
        ('\u010d', '\u010c'), ('\u010f', '\u010e'),
        ('\u0111', '\u0110'), ('\u0113', '\u0112'),
        ('\u0115', '\u0114'), ('\u0117', '\u0116'),
        ('\u0119', '\u0118'), ('\u011b', '\u011a'),
        ('\u011d', '\u011c'), ('\u011f', '\u011e'),
        ('\u0121', '\u0120'), ('\u0123', '\u0122'),
        ('\u0125', '\u0124'), ('\u0127', '\u0126'),
        ('\u0129', '\u0128'), ('\u012b', '\u012a'),
        ('\u012d', '\u012c'), ('\u012f', '\u012e'),
        ('\u0131', '\x49'), ('\u0133', '\u0132'),
        ('\u0135', '\u0134'), ('\u0137', '\u0136'),
        ('\u013a', '\u0139'), ('\u013c', '\u013b'),
        ('\u013e', '\u013d'), ('\u0140', '\u013f'),
        ('\u0142', '\u0141'), ('\u0144', '\u0143'),
        ('\u0146', '\u0145'), ('\u0148', '\u0147'),
        ('\u014b', '\u014a'), ('\u014d', '\u014c'),
        ('\u014f', '\u014e'), ('\u0151', '\u0150'),
        ('\u0153', '\u0152'), ('\u0155', '\u0154'),
        ('\u0157', '\u0156'), ('\u0159', '\u0158'),
        ('\u015b', '\u015a'), ('\u015d', '\u015c'),
        ('\u015f', '\u015e'), ('\u0161', '\u0160'),
        ('\u0163', '\u0162'), ('\u0165', '\u0164'),
        ('\u0167', '\u0166'), ('\u0169', '\u0168'),
        ('\u016b', '\u016a'), ('\u016d', '\u016c'),
        ('\u016f', '\u016e'), ('\u0171', '\u0170'),
        ('\u0173', '\u0172'), ('\u0175', '\u0174'),
        ('\u0177', '\u0176'), ('\u017a', '\u0179'),
        ('\u017c', '\u017b'), ('\u017e', '\u017d'),
        ('\u017f', '\x53'), ('\u0180', '\u0243'),
        ('\u0183', '\u0182'), ('\u0185', '\u0184'),
        ('\u0188', '\u0187'), ('\u018c', '\u018b'),
        ('\u0192', '\u0191'), ('\u0195', '\u01f6'),
        ('\u0199', '\u0198'), ('\u019a', '\u023d'),
        ('\u019e', '\u0220'), ('\u01a1', '\u01a0'),
        ('\u01a3', '\u01a2'), ('\u01a5', '\u01a4'),
        ('\u01a8', '\u01a7'), ('\u01ad', '\u01ac'),
        ('\u01b0', '\u01af'), ('\u01b4', '\u01b3'),
        ('\u01b6', '\u01b5'), ('\u01b9', '\u01b8'),
        ('\u01bd', '\u01bc'), ('\u01bf', '\u01f7'),
        ('\u01c6', '\u01c4'), ('\u01c9', '\u01c7'),
        ('\u01cc', '\u01ca'), ('\u01ce', '\u01cd'),
        ('\u01d0', '\u01cf'), ('\u01d2', '\u01d1'),
        ('\u01d4', '\u01d3'), ('\u01d6', '\u01d5'),
        ('\u01d8', '\u01d7'), ('\u01da', '\u01d9'),
        ('\u01dc', '\u01db'), ('\u01dd', '\u018e'),
        ('\u01df', '\u01de'), ('\u01e1', '\u01e0'),
        ('\u01e3', '\u01e2'), ('\u01e5', '\u01e4'),
        ('\u01e7', '\u01e6'), ('\u01e9', '\u01e8'),
        ('\u01eb', '\u01ea'), ('\u01ed', '\u01ec'),
        ('\u01ef', '\u01ee'), ('\u01f3', '\u01f1'),
        ('\u01f5', '\u01f4'), ('\u01f9', '\u01f8'),
        ('\u01fb', '\u01fa'), ('\u01fd', '\u01fc'),
        ('\u01ff', '\u01fe'), ('\u0201', '\u0200'),
        ('\u0203', '\u0202'), ('\u0205', '\u0204'),
        ('\u0207', '\u0206'), ('\u0209', '\u0208'),
        ('\u020b', '\u020a'), ('\u020d', '\u020c'),
        ('\u020f', '\u020e'), ('\u0211', '\u0210'),
        ('\u0213', '\u0212'), ('\u0215', '\u0214'),
        ('\u0217', '\u0216'), ('\u0219', '\u0218'),
        ('\u021b', '\u021a'), ('\u021d', '\u021c'),
        ('\u021f', '\u021e'), ('\u0223', '\u0222'),
        ('\u0225', '\u0224'), ('\u0227', '\u0226'),
        ('\u0229', '\u0228'), ('\u022b', '\u022a'),
        ('\u022d', '\u022c'), ('\u022f', '\u022e'),
        ('\u0231', '\u0230'), ('\u0233', '\u0232'),
        ('\u023c', '\u023b'), ('\u023f', '\u2c7e'),
        ('\u0240', '\u2c7f'), ('\u0242', '\u0241'),
        ('\u0247', '\u0246'), ('\u0249', '\u0248'),
        ('\u024b', '\u024a'), ('\u024d', '\u024c'),
        ('\u024f', '\u024e'), ('\u0250', '\u2c6f'),
        ('\u0251', '\u2c6d'), ('\u0252', '\u2c70'),
        ('\u0253', '\u0181'), ('\u0254', '\u0186'),
        ('\u0256', '\u0189'), ('\u0257', '\u018a'),
        ('\u0259', '\u018f'), ('\u025b', '\u0190'),
        ('\u0260', '\u0193'), ('\u0263', '\u0194'),
        ('\u0265', '\ua78d'), ('\u0266', '\ua7aa'),
        ('\u0268', '\u0197'), ('\u0269', '\u0196'),
        ('\u026b', '\u2c62'), ('\u026f', '\u019c'),
        ('\u0271', '\u2c6e'), ('\u0272', '\u019d'),
        ('\u0275', '\u019f'), ('\u027d', '\u2c64'),
        ('\u0280', '\u01a6'), ('\u0283', '\u01a9'),
        ('\u0288', '\u01ae'), ('\u0289', '\u0244'),
        ('\u028a', '\u01b1'), ('\u028b', '\u01b2'),
        ('\u028c', '\u0245'), ('\u0292', '\u01b7'),
        ('\u0371', '\u0370'), ('\u0373', '\u0372'),
        ('\u0377', '\u0376'), ('\u037b', '\u03fd'),
        ('\u037c', '\u03fe'), ('\u037d', '\u03ff'),
        ('\u03ac', '\u0386'), ('\u03ad', '\u0388'),
        ('\u03ae', '\u0389'), ('\u03af', '\u038a'),
        ('\u03b1', '\u0391'), ('\u03b2', '\u0392'),
        ('\u03b3', '\u0393'), ('\u03b4', '\u0394'),
        ('\u03b5', '\u0395'), ('\u03b6', '\u0396'),
        ('\u03b7', '\u0397'), ('\u03b8', '\u0398'),
        ('\u03b9', '\u0399'), ('\u03ba', '\u039a'),
        ('\u03bb', '\u039b'), ('\u03bc', '\u039c'),
        ('\u03bd', '\u039d'), ('\u03be', '\u039e'),
        ('\u03bf', '\u039f'), ('\u03c0', '\u03a0'),
        ('\u03c1', '\u03a1'), ('\u03c2', '\u03a3'),
        ('\u03c3', '\u03a3'), ('\u03c4', '\u03a4'),
        ('\u03c5', '\u03a5'), ('\u03c6', '\u03a6'),
        ('\u03c7', '\u03a7'), ('\u03c8', '\u03a8'),
        ('\u03c9', '\u03a9'), ('\u03ca', '\u03aa'),
        ('\u03cb', '\u03ab'), ('\u03cc', '\u038c'),
        ('\u03cd', '\u038e'), ('\u03ce', '\u038f'),
        ('\u03d0', '\u0392'), ('\u03d1', '\u0398'),
        ('\u03d5', '\u03a6'), ('\u03d6', '\u03a0'),
        ('\u03d7', '\u03cf'), ('\u03d9', '\u03d8'),
        ('\u03db', '\u03da'), ('\u03dd', '\u03dc'),
        ('\u03df', '\u03de'), ('\u03e1', '\u03e0'),
        ('\u03e3', '\u03e2'), ('\u03e5', '\u03e4'),
        ('\u03e7', '\u03e6'), ('\u03e9', '\u03e8'),
        ('\u03eb', '\u03ea'), ('\u03ed', '\u03ec'),
        ('\u03ef', '\u03ee'), ('\u03f0', '\u039a'),
        ('\u03f1', '\u03a1'), ('\u03f2', '\u03f9'),
        ('\u03f5', '\u0395'), ('\u03f8', '\u03f7'),
        ('\u03fb', '\u03fa'), ('\u0430', '\u0410'),
        ('\u0431', '\u0411'), ('\u0432', '\u0412'),
        ('\u0433', '\u0413'), ('\u0434', '\u0414'),
        ('\u0435', '\u0415'), ('\u0436', '\u0416'),
        ('\u0437', '\u0417'), ('\u0438', '\u0418'),
        ('\u0439', '\u0419'), ('\u043a', '\u041a'),
        ('\u043b', '\u041b'), ('\u043c', '\u041c'),
        ('\u043d', '\u041d'), ('\u043e', '\u041e'),
        ('\u043f', '\u041f'), ('\u0440', '\u0420'),
        ('\u0441', '\u0421'), ('\u0442', '\u0422'),
        ('\u0443', '\u0423'), ('\u0444', '\u0424'),
        ('\u0445', '\u0425'), ('\u0446', '\u0426'),
        ('\u0447', '\u0427'), ('\u0448', '\u0428'),
        ('\u0449', '\u0429'), ('\u044a', '\u042a'),
        ('\u044b', '\u042b'), ('\u044c', '\u042c'),
        ('\u044d', '\u042d'), ('\u044e', '\u042e'),
        ('\u044f', '\u042f'), ('\u0450', '\u0400'),
        ('\u0451', '\u0401'), ('\u0452', '\u0402'),
        ('\u0453', '\u0403'), ('\u0454', '\u0404'),
        ('\u0455', '\u0405'), ('\u0456', '\u0406'),
        ('\u0457', '\u0407'), ('\u0458', '\u0408'),
        ('\u0459', '\u0409'), ('\u045a', '\u040a'),
        ('\u045b', '\u040b'), ('\u045c', '\u040c'),
        ('\u045d', '\u040d'), ('\u045e', '\u040e'),
        ('\u045f', '\u040f'), ('\u0461', '\u0460'),
        ('\u0463', '\u0462'), ('\u0465', '\u0464'),
        ('\u0467', '\u0466'), ('\u0469', '\u0468'),
        ('\u046b', '\u046a'), ('\u046d', '\u046c'),
        ('\u046f', '\u046e'), ('\u0471', '\u0470'),
        ('\u0473', '\u0472'), ('\u0475', '\u0474'),
        ('\u0477', '\u0476'), ('\u0479', '\u0478'),
        ('\u047b', '\u047a'), ('\u047d', '\u047c'),
        ('\u047f', '\u047e'), ('\u0481', '\u0480'),
        ('\u048b', '\u048a'), ('\u048d', '\u048c'),
        ('\u048f', '\u048e'), ('\u0491', '\u0490'),
        ('\u0493', '\u0492'), ('\u0495', '\u0494'),
        ('\u0497', '\u0496'), ('\u0499', '\u0498'),
        ('\u049b', '\u049a'), ('\u049d', '\u049c'),
        ('\u049f', '\u049e'), ('\u04a1', '\u04a0'),
        ('\u04a3', '\u04a2'), ('\u04a5', '\u04a4'),
        ('\u04a7', '\u04a6'), ('\u04a9', '\u04a8'),
        ('\u04ab', '\u04aa'), ('\u04ad', '\u04ac'),
        ('\u04af', '\u04ae'), ('\u04b1', '\u04b0'),
        ('\u04b3', '\u04b2'), ('\u04b5', '\u04b4'),
        ('\u04b7', '\u04b6'), ('\u04b9', '\u04b8'),
        ('\u04bb', '\u04ba'), ('\u04bd', '\u04bc'),
        ('\u04bf', '\u04be'), ('\u04c2', '\u04c1'),
        ('\u04c4', '\u04c3'), ('\u04c6', '\u04c5'),
        ('\u04c8', '\u04c7'), ('\u04ca', '\u04c9'),
        ('\u04cc', '\u04cb'), ('\u04ce', '\u04cd'),
        ('\u04cf', '\u04c0'), ('\u04d1', '\u04d0'),
        ('\u04d3', '\u04d2'), ('\u04d5', '\u04d4'),
        ('\u04d7', '\u04d6'), ('\u04d9', '\u04d8'),
        ('\u04db', '\u04da'), ('\u04dd', '\u04dc'),
        ('\u04df', '\u04de'), ('\u04e1', '\u04e0'),
        ('\u04e3', '\u04e2'), ('\u04e5', '\u04e4'),
        ('\u04e7', '\u04e6'), ('\u04e9', '\u04e8'),
        ('\u04eb', '\u04ea'), ('\u04ed', '\u04ec'),
        ('\u04ef', '\u04ee'), ('\u04f1', '\u04f0'),
        ('\u04f3', '\u04f2'), ('\u04f5', '\u04f4'),
        ('\u04f7', '\u04f6'), ('\u04f9', '\u04f8'),
        ('\u04fb', '\u04fa'), ('\u04fd', '\u04fc'),
        ('\u04ff', '\u04fe'), ('\u0501', '\u0500'),
        ('\u0503', '\u0502'), ('\u0505', '\u0504'),
        ('\u0507', '\u0506'), ('\u0509', '\u0508'),
        ('\u050b', '\u050a'), ('\u050d', '\u050c'),
        ('\u050f', '\u050e'), ('\u0511', '\u0510'),
        ('\u0513', '\u0512'), ('\u0515', '\u0514'),
        ('\u0517', '\u0516'), ('\u0519', '\u0518'),
        ('\u051b', '\u051a'), ('\u051d', '\u051c'),
        ('\u051f', '\u051e'), ('\u0521', '\u0520'),
        ('\u0523', '\u0522'), ('\u0525', '\u0524'),
        ('\u0527', '\u0526'), ('\u0561', '\u0531'),
        ('\u0562', '\u0532'), ('\u0563', '\u0533'),
        ('\u0564', '\u0534'), ('\u0565', '\u0535'),
        ('\u0566', '\u0536'), ('\u0567', '\u0537'),
        ('\u0568', '\u0538'), ('\u0569', '\u0539'),
        ('\u056a', '\u053a'), ('\u056b', '\u053b'),
        ('\u056c', '\u053c'), ('\u056d', '\u053d'),
        ('\u056e', '\u053e'), ('\u056f', '\u053f'),
        ('\u0570', '\u0540'), ('\u0571', '\u0541'),
        ('\u0572', '\u0542'), ('\u0573', '\u0543'),
        ('\u0574', '\u0544'), ('\u0575', '\u0545'),
        ('\u0576', '\u0546'), ('\u0577', '\u0547'),
        ('\u0578', '\u0548'), ('\u0579', '\u0549'),
        ('\u057a', '\u054a'), ('\u057b', '\u054b'),
        ('\u057c', '\u054c'), ('\u057d', '\u054d'),
        ('\u057e', '\u054e'), ('\u057f', '\u054f'),
        ('\u0580', '\u0550'), ('\u0581', '\u0551'),
        ('\u0582', '\u0552'), ('\u0583', '\u0553'),
        ('\u0584', '\u0554'), ('\u0585', '\u0555'),
        ('\u0586', '\u0556'), ('\u1d79', '\ua77d'),
        ('\u1d7d', '\u2c63'), ('\u1e01', '\u1e00'),
        ('\u1e03', '\u1e02'), ('\u1e05', '\u1e04'),
        ('\u1e07', '\u1e06'), ('\u1e09', '\u1e08'),
        ('\u1e0b', '\u1e0a'), ('\u1e0d', '\u1e0c'),
        ('\u1e0f', '\u1e0e'), ('\u1e11', '\u1e10'),
        ('\u1e13', '\u1e12'), ('\u1e15', '\u1e14'),
        ('\u1e17', '\u1e16'), ('\u1e19', '\u1e18'),
        ('\u1e1b', '\u1e1a'), ('\u1e1d', '\u1e1c'),
        ('\u1e1f', '\u1e1e'), ('\u1e21', '\u1e20'),
        ('\u1e23', '\u1e22'), ('\u1e25', '\u1e24'),
        ('\u1e27', '\u1e26'), ('\u1e29', '\u1e28'),
        ('\u1e2b', '\u1e2a'), ('\u1e2d', '\u1e2c'),
        ('\u1e2f', '\u1e2e'), ('\u1e31', '\u1e30'),
        ('\u1e33', '\u1e32'), ('\u1e35', '\u1e34'),
        ('\u1e37', '\u1e36'), ('\u1e39', '\u1e38'),
        ('\u1e3b', '\u1e3a'), ('\u1e3d', '\u1e3c'),
        ('\u1e3f', '\u1e3e'), ('\u1e41', '\u1e40'),
        ('\u1e43', '\u1e42'), ('\u1e45', '\u1e44'),
        ('\u1e47', '\u1e46'), ('\u1e49', '\u1e48'),
        ('\u1e4b', '\u1e4a'), ('\u1e4d', '\u1e4c'),
        ('\u1e4f', '\u1e4e'), ('\u1e51', '\u1e50'),
        ('\u1e53', '\u1e52'), ('\u1e55', '\u1e54'),
        ('\u1e57', '\u1e56'), ('\u1e59', '\u1e58'),
        ('\u1e5b', '\u1e5a'), ('\u1e5d', '\u1e5c'),
        ('\u1e5f', '\u1e5e'), ('\u1e61', '\u1e60'),
        ('\u1e63', '\u1e62'), ('\u1e65', '\u1e64'),
        ('\u1e67', '\u1e66'), ('\u1e69', '\u1e68'),
        ('\u1e6b', '\u1e6a'), ('\u1e6d', '\u1e6c'),
        ('\u1e6f', '\u1e6e'), ('\u1e71', '\u1e70'),
        ('\u1e73', '\u1e72'), ('\u1e75', '\u1e74'),
        ('\u1e77', '\u1e76'), ('\u1e79', '\u1e78'),
        ('\u1e7b', '\u1e7a'), ('\u1e7d', '\u1e7c'),
        ('\u1e7f', '\u1e7e'), ('\u1e81', '\u1e80'),
        ('\u1e83', '\u1e82'), ('\u1e85', '\u1e84'),
        ('\u1e87', '\u1e86'), ('\u1e89', '\u1e88'),
        ('\u1e8b', '\u1e8a'), ('\u1e8d', '\u1e8c'),
        ('\u1e8f', '\u1e8e'), ('\u1e91', '\u1e90'),
        ('\u1e93', '\u1e92'), ('\u1e95', '\u1e94'),
        ('\u1e9b', '\u1e60'), ('\u1ea1', '\u1ea0'),
        ('\u1ea3', '\u1ea2'), ('\u1ea5', '\u1ea4'),
        ('\u1ea7', '\u1ea6'), ('\u1ea9', '\u1ea8'),
        ('\u1eab', '\u1eaa'), ('\u1ead', '\u1eac'),
        ('\u1eaf', '\u1eae'), ('\u1eb1', '\u1eb0'),
        ('\u1eb3', '\u1eb2'), ('\u1eb5', '\u1eb4'),
        ('\u1eb7', '\u1eb6'), ('\u1eb9', '\u1eb8'),
        ('\u1ebb', '\u1eba'), ('\u1ebd', '\u1ebc'),
        ('\u1ebf', '\u1ebe'), ('\u1ec1', '\u1ec0'),
        ('\u1ec3', '\u1ec2'), ('\u1ec5', '\u1ec4'),
        ('\u1ec7', '\u1ec6'), ('\u1ec9', '\u1ec8'),
        ('\u1ecb', '\u1eca'), ('\u1ecd', '\u1ecc'),
        ('\u1ecf', '\u1ece'), ('\u1ed1', '\u1ed0'),
        ('\u1ed3', '\u1ed2'), ('\u1ed5', '\u1ed4'),
        ('\u1ed7', '\u1ed6'), ('\u1ed9', '\u1ed8'),
        ('\u1edb', '\u1eda'), ('\u1edd', '\u1edc'),
        ('\u1edf', '\u1ede'), ('\u1ee1', '\u1ee0'),
        ('\u1ee3', '\u1ee2'), ('\u1ee5', '\u1ee4'),
        ('\u1ee7', '\u1ee6'), ('\u1ee9', '\u1ee8'),
        ('\u1eeb', '\u1eea'), ('\u1eed', '\u1eec'),
        ('\u1eef', '\u1eee'), ('\u1ef1', '\u1ef0'),
        ('\u1ef3', '\u1ef2'), ('\u1ef5', '\u1ef4'),
        ('\u1ef7', '\u1ef6'), ('\u1ef9', '\u1ef8'),
        ('\u1efb', '\u1efa'), ('\u1efd', '\u1efc'),
        ('\u1eff', '\u1efe'), ('\u1f00', '\u1f08'),
        ('\u1f01', '\u1f09'), ('\u1f02', '\u1f0a'),
        ('\u1f03', '\u1f0b'), ('\u1f04', '\u1f0c'),
        ('\u1f05', '\u1f0d'), ('\u1f06', '\u1f0e'),
        ('\u1f07', '\u1f0f'), ('\u1f10', '\u1f18'),
        ('\u1f11', '\u1f19'), ('\u1f12', '\u1f1a'),
        ('\u1f13', '\u1f1b'), ('\u1f14', '\u1f1c'),
        ('\u1f15', '\u1f1d'), ('\u1f20', '\u1f28'),
        ('\u1f21', '\u1f29'), ('\u1f22', '\u1f2a'),
        ('\u1f23', '\u1f2b'), ('\u1f24', '\u1f2c'),
        ('\u1f25', '\u1f2d'), ('\u1f26', '\u1f2e'),
        ('\u1f27', '\u1f2f'), ('\u1f30', '\u1f38'),
        ('\u1f31', '\u1f39'), ('\u1f32', '\u1f3a'),
        ('\u1f33', '\u1f3b'), ('\u1f34', '\u1f3c'),
        ('\u1f35', '\u1f3d'), ('\u1f36', '\u1f3e'),
        ('\u1f37', '\u1f3f'), ('\u1f40', '\u1f48'),
        ('\u1f41', '\u1f49'), ('\u1f42', '\u1f4a'),
        ('\u1f43', '\u1f4b'), ('\u1f44', '\u1f4c'),
        ('\u1f45', '\u1f4d'), ('\u1f51', '\u1f59'),
        ('\u1f53', '\u1f5b'), ('\u1f55', '\u1f5d'),
        ('\u1f57', '\u1f5f'), ('\u1f60', '\u1f68'),
        ('\u1f61', '\u1f69'), ('\u1f62', '\u1f6a'),
        ('\u1f63', '\u1f6b'), ('\u1f64', '\u1f6c'),
        ('\u1f65', '\u1f6d'), ('\u1f66', '\u1f6e'),
        ('\u1f67', '\u1f6f'), ('\u1f70', '\u1fba'),
        ('\u1f71', '\u1fbb'), ('\u1f72', '\u1fc8'),
        ('\u1f73', '\u1fc9'), ('\u1f74', '\u1fca'),
        ('\u1f75', '\u1fcb'), ('\u1f76', '\u1fda'),
        ('\u1f77', '\u1fdb'), ('\u1f78', '\u1ff8'),
        ('\u1f79', '\u1ff9'), ('\u1f7a', '\u1fea'),
        ('\u1f7b', '\u1feb'), ('\u1f7c', '\u1ffa'),
        ('\u1f7d', '\u1ffb'), ('\u1f80', '\u1f88'),
        ('\u1f81', '\u1f89'), ('\u1f82', '\u1f8a'),
        ('\u1f83', '\u1f8b'), ('\u1f84', '\u1f8c'),
        ('\u1f85', '\u1f8d'), ('\u1f86', '\u1f8e'),
        ('\u1f87', '\u1f8f'), ('\u1f90', '\u1f98'),
        ('\u1f91', '\u1f99'), ('\u1f92', '\u1f9a'),
        ('\u1f93', '\u1f9b'), ('\u1f94', '\u1f9c'),
        ('\u1f95', '\u1f9d'), ('\u1f96', '\u1f9e'),
        ('\u1f97', '\u1f9f'), ('\u1fa0', '\u1fa8'),
        ('\u1fa1', '\u1fa9'), ('\u1fa2', '\u1faa'),
        ('\u1fa3', '\u1fab'), ('\u1fa4', '\u1fac'),
        ('\u1fa5', '\u1fad'), ('\u1fa6', '\u1fae'),
        ('\u1fa7', '\u1faf'), ('\u1fb0', '\u1fb8'),
        ('\u1fb1', '\u1fb9'), ('\u1fb3', '\u1fbc'),
        ('\u1fbe', '\u0399'), ('\u1fc3', '\u1fcc'),
        ('\u1fd0', '\u1fd8'), ('\u1fd1', '\u1fd9'),
        ('\u1fe0', '\u1fe8'), ('\u1fe1', '\u1fe9'),
        ('\u1fe5', '\u1fec'), ('\u1ff3', '\u1ffc'),
        ('\u214e', '\u2132'), ('\u2184', '\u2183'),
        ('\u2c30', '\u2c00'), ('\u2c31', '\u2c01'),
        ('\u2c32', '\u2c02'), ('\u2c33', '\u2c03'),
        ('\u2c34', '\u2c04'), ('\u2c35', '\u2c05'),
        ('\u2c36', '\u2c06'), ('\u2c37', '\u2c07'),
        ('\u2c38', '\u2c08'), ('\u2c39', '\u2c09'),
        ('\u2c3a', '\u2c0a'), ('\u2c3b', '\u2c0b'),
        ('\u2c3c', '\u2c0c'), ('\u2c3d', '\u2c0d'),
        ('\u2c3e', '\u2c0e'), ('\u2c3f', '\u2c0f'),
        ('\u2c40', '\u2c10'), ('\u2c41', '\u2c11'),
        ('\u2c42', '\u2c12'), ('\u2c43', '\u2c13'),
        ('\u2c44', '\u2c14'), ('\u2c45', '\u2c15'),
        ('\u2c46', '\u2c16'), ('\u2c47', '\u2c17'),
        ('\u2c48', '\u2c18'), ('\u2c49', '\u2c19'),
        ('\u2c4a', '\u2c1a'), ('\u2c4b', '\u2c1b'),
        ('\u2c4c', '\u2c1c'), ('\u2c4d', '\u2c1d'),
        ('\u2c4e', '\u2c1e'), ('\u2c4f', '\u2c1f'),
        ('\u2c50', '\u2c20'), ('\u2c51', '\u2c21'),
        ('\u2c52', '\u2c22'), ('\u2c53', '\u2c23'),
        ('\u2c54', '\u2c24'), ('\u2c55', '\u2c25'),
        ('\u2c56', '\u2c26'), ('\u2c57', '\u2c27'),
        ('\u2c58', '\u2c28'), ('\u2c59', '\u2c29'),
        ('\u2c5a', '\u2c2a'), ('\u2c5b', '\u2c2b'),
        ('\u2c5c', '\u2c2c'), ('\u2c5d', '\u2c2d'),
        ('\u2c5e', '\u2c2e'), ('\u2c61', '\u2c60'),
        ('\u2c65', '\u023a'), ('\u2c66', '\u023e'),
        ('\u2c68', '\u2c67'), ('\u2c6a', '\u2c69'),
        ('\u2c6c', '\u2c6b'), ('\u2c73', '\u2c72'),
        ('\u2c76', '\u2c75'), ('\u2c81', '\u2c80'),
        ('\u2c83', '\u2c82'), ('\u2c85', '\u2c84'),
        ('\u2c87', '\u2c86'), ('\u2c89', '\u2c88'),
        ('\u2c8b', '\u2c8a'), ('\u2c8d', '\u2c8c'),
        ('\u2c8f', '\u2c8e'), ('\u2c91', '\u2c90'),
        ('\u2c93', '\u2c92'), ('\u2c95', '\u2c94'),
        ('\u2c97', '\u2c96'), ('\u2c99', '\u2c98'),
        ('\u2c9b', '\u2c9a'), ('\u2c9d', '\u2c9c'),
        ('\u2c9f', '\u2c9e'), ('\u2ca1', '\u2ca0'),
        ('\u2ca3', '\u2ca2'), ('\u2ca5', '\u2ca4'),
        ('\u2ca7', '\u2ca6'), ('\u2ca9', '\u2ca8'),
        ('\u2cab', '\u2caa'), ('\u2cad', '\u2cac'),
        ('\u2caf', '\u2cae'), ('\u2cb1', '\u2cb0'),
        ('\u2cb3', '\u2cb2'), ('\u2cb5', '\u2cb4'),
        ('\u2cb7', '\u2cb6'), ('\u2cb9', '\u2cb8'),
        ('\u2cbb', '\u2cba'), ('\u2cbd', '\u2cbc'),
        ('\u2cbf', '\u2cbe'), ('\u2cc1', '\u2cc0'),
        ('\u2cc3', '\u2cc2'), ('\u2cc5', '\u2cc4'),
        ('\u2cc7', '\u2cc6'), ('\u2cc9', '\u2cc8'),
        ('\u2ccb', '\u2cca'), ('\u2ccd', '\u2ccc'),
        ('\u2ccf', '\u2cce'), ('\u2cd1', '\u2cd0'),
        ('\u2cd3', '\u2cd2'), ('\u2cd5', '\u2cd4'),
        ('\u2cd7', '\u2cd6'), ('\u2cd9', '\u2cd8'),
        ('\u2cdb', '\u2cda'), ('\u2cdd', '\u2cdc'),
        ('\u2cdf', '\u2cde'), ('\u2ce1', '\u2ce0'),
        ('\u2ce3', '\u2ce2'), ('\u2cec', '\u2ceb'),
        ('\u2cee', '\u2ced'), ('\u2cf3', '\u2cf2'),
        ('\u2d00', '\u10a0'), ('\u2d01', '\u10a1'),
        ('\u2d02', '\u10a2'), ('\u2d03', '\u10a3'),
        ('\u2d04', '\u10a4'), ('\u2d05', '\u10a5'),
        ('\u2d06', '\u10a6'), ('\u2d07', '\u10a7'),
        ('\u2d08', '\u10a8'), ('\u2d09', '\u10a9'),
        ('\u2d0a', '\u10aa'), ('\u2d0b', '\u10ab'),
        ('\u2d0c', '\u10ac'), ('\u2d0d', '\u10ad'),
        ('\u2d0e', '\u10ae'), ('\u2d0f', '\u10af'),
        ('\u2d10', '\u10b0'), ('\u2d11', '\u10b1'),
        ('\u2d12', '\u10b2'), ('\u2d13', '\u10b3'),
        ('\u2d14', '\u10b4'), ('\u2d15', '\u10b5'),
        ('\u2d16', '\u10b6'), ('\u2d17', '\u10b7'),
        ('\u2d18', '\u10b8'), ('\u2d19', '\u10b9'),
        ('\u2d1a', '\u10ba'), ('\u2d1b', '\u10bb'),
        ('\u2d1c', '\u10bc'), ('\u2d1d', '\u10bd'),
        ('\u2d1e', '\u10be'), ('\u2d1f', '\u10bf'),
        ('\u2d20', '\u10c0'), ('\u2d21', '\u10c1'),
        ('\u2d22', '\u10c2'), ('\u2d23', '\u10c3'),
        ('\u2d24', '\u10c4'), ('\u2d25', '\u10c5'),
        ('\u2d27', '\u10c7'), ('\u2d2d', '\u10cd'),
        ('\ua641', '\ua640'), ('\ua643', '\ua642'),
        ('\ua645', '\ua644'), ('\ua647', '\ua646'),
        ('\ua649', '\ua648'), ('\ua64b', '\ua64a'),
        ('\ua64d', '\ua64c'), ('\ua64f', '\ua64e'),
        ('\ua651', '\ua650'), ('\ua653', '\ua652'),
        ('\ua655', '\ua654'), ('\ua657', '\ua656'),
        ('\ua659', '\ua658'), ('\ua65b', '\ua65a'),
        ('\ua65d', '\ua65c'), ('\ua65f', '\ua65e'),
        ('\ua661', '\ua660'), ('\ua663', '\ua662'),
        ('\ua665', '\ua664'), ('\ua667', '\ua666'),
        ('\ua669', '\ua668'), ('\ua66b', '\ua66a'),
        ('\ua66d', '\ua66c'), ('\ua681', '\ua680'),
        ('\ua683', '\ua682'), ('\ua685', '\ua684'),
        ('\ua687', '\ua686'), ('\ua689', '\ua688'),
        ('\ua68b', '\ua68a'), ('\ua68d', '\ua68c'),
        ('\ua68f', '\ua68e'), ('\ua691', '\ua690'),
        ('\ua693', '\ua692'), ('\ua695', '\ua694'),
        ('\ua697', '\ua696'), ('\ua723', '\ua722'),
        ('\ua725', '\ua724'), ('\ua727', '\ua726'),
        ('\ua729', '\ua728'), ('\ua72b', '\ua72a'),
        ('\ua72d', '\ua72c'), ('\ua72f', '\ua72e'),
        ('\ua733', '\ua732'), ('\ua735', '\ua734'),
        ('\ua737', '\ua736'), ('\ua739', '\ua738'),
        ('\ua73b', '\ua73a'), ('\ua73d', '\ua73c'),
        ('\ua73f', '\ua73e'), ('\ua741', '\ua740'),
        ('\ua743', '\ua742'), ('\ua745', '\ua744'),
        ('\ua747', '\ua746'), ('\ua749', '\ua748'),
        ('\ua74b', '\ua74a'), ('\ua74d', '\ua74c'),
        ('\ua74f', '\ua74e'), ('\ua751', '\ua750'),
        ('\ua753', '\ua752'), ('\ua755', '\ua754'),
        ('\ua757', '\ua756'), ('\ua759', '\ua758'),
        ('\ua75b', '\ua75a'), ('\ua75d', '\ua75c'),
        ('\ua75f', '\ua75e'), ('\ua761', '\ua760'),
        ('\ua763', '\ua762'), ('\ua765', '\ua764'),
        ('\ua767', '\ua766'), ('\ua769', '\ua768'),
        ('\ua76b', '\ua76a'), ('\ua76d', '\ua76c'),
        ('\ua76f', '\ua76e'), ('\ua77a', '\ua779'),
        ('\ua77c', '\ua77b'), ('\ua77f', '\ua77e'),
        ('\ua781', '\ua780'), ('\ua783', '\ua782'),
        ('\ua785', '\ua784'), ('\ua787', '\ua786'),
        ('\ua78c', '\ua78b'), ('\ua791', '\ua790'),
        ('\ua793', '\ua792'), ('\ua7a1', '\ua7a0'),
        ('\ua7a3', '\ua7a2'), ('\ua7a5', '\ua7a4'),
        ('\ua7a7', '\ua7a6'), ('\ua7a9', '\ua7a8'),
        ('\uff41', '\uff21'), ('\uff42', '\uff22'),
        ('\uff43', '\uff23'), ('\uff44', '\uff24'),
        ('\uff45', '\uff25'), ('\uff46', '\uff26'),
        ('\uff47', '\uff27'), ('\uff48', '\uff28'),
        ('\uff49', '\uff29'), ('\uff4a', '\uff2a'),
        ('\uff4b', '\uff2b'), ('\uff4c', '\uff2c'),
        ('\uff4d', '\uff2d'), ('\uff4e', '\uff2e'),
        ('\uff4f', '\uff2f'), ('\uff50', '\uff30'),
        ('\uff51', '\uff31'), ('\uff52', '\uff32'),
        ('\uff53', '\uff33'), ('\uff54', '\uff34'),
        ('\uff55', '\uff35'), ('\uff56', '\uff36'),
        ('\uff57', '\uff37'), ('\uff58', '\uff38'),
        ('\uff59', '\uff39'), ('\uff5a', '\uff3a'),
        ('\U00010428', '\U00010400'), ('\U00010429', '\U00010401'),
        ('\U0001042a', '\U00010402'), ('\U0001042b', '\U00010403'),
        ('\U0001042c', '\U00010404'), ('\U0001042d', '\U00010405'),
        ('\U0001042e', '\U00010406'), ('\U0001042f', '\U00010407'),
        ('\U00010430', '\U00010408'), ('\U00010431', '\U00010409'),
        ('\U00010432', '\U0001040a'), ('\U00010433', '\U0001040b'),
        ('\U00010434', '\U0001040c'), ('\U00010435', '\U0001040d'),
        ('\U00010436', '\U0001040e'), ('\U00010437', '\U0001040f'),
        ('\U00010438', '\U00010410'), ('\U00010439', '\U00010411'),
        ('\U0001043a', '\U00010412'), ('\U0001043b', '\U00010413'),
        ('\U0001043c', '\U00010414'), ('\U0001043d', '\U00010415'),
        ('\U0001043e', '\U00010416'), ('\U0001043f', '\U00010417'),
        ('\U00010440', '\U00010418'), ('\U00010441', '\U00010419'),
        ('\U00010442', '\U0001041a'), ('\U00010443', '\U0001041b'),
        ('\U00010444', '\U0001041c'), ('\U00010445', '\U0001041d'),
        ('\U00010446', '\U0001041e'), ('\U00010447', '\U0001041f'),
        ('\U00010448', '\U00010420'), ('\U00010449', '\U00010421'),
        ('\U0001044a', '\U00010422'), ('\U0001044b', '\U00010423'),
        ('\U0001044c', '\U00010424'), ('\U0001044d', '\U00010425'),
        ('\U0001044e', '\U00010426'), ('\U0001044f', '\U00010427')
    ];

}
