const bun = @import("root").bun;
const string = bun.string;
const Output = bun.Output;
const Global = bun.Global;
const Environment = bun.Environment;
const strings = bun.strings;
const MutableString = bun.MutableString;
const stringZ = bun.stringZ;
const default_allocator = bun.default_allocator;
const C = bun.C;
const std = @import("std");
const defines = @import("./defines.zig");

// If something is in this list, then a direct identifier expression or property
// access chain matching this will be assumed to have no side effects and will
// be removed.
//
// This also means code is allowed to be reordered past things in this list. For
// example, if "console.log" is in this list, permitting reordering allows for
// "if (a) console.log(b); else console.log(c)" to be reordered and transformed
// into "console.log(a ? b : c)". Notice that "a" and "console.log" are in a
// different order, which can only happen if evaluating the "console.log"
// property access can be assumed to not change the value of "a".
//
// Note that membership in this list says nothing about whether calling any of
// these functions has any side effects. It only says something about
// referencing these function without calling them.
pub const GlobalDefinesKey = [_][]const string{
    // Object: Static methods
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object#Static_methods
    &[_]string{ "Object", "assign" },
    &[_]string{ "Object", "create" },
    &[_]string{ "Object", "defineProperties" },
    &[_]string{ "Object", "defineProperty" },
    &[_]string{ "Object", "entries" },
    &[_]string{ "Object", "freeze" },
    &[_]string{ "Object", "fromEntries" },
    &[_]string{ "Object", "getOwnPropertyDescriptor" },
    &[_]string{ "Object", "getOwnPropertyDescriptors" },
    &[_]string{ "Object", "getOwnPropertyNames" },
    &[_]string{ "Object", "getOwnPropertySymbols" },
    &[_]string{ "Object", "getPrototypeOf" },
    &[_]string{ "Object", "is" },
    &[_]string{ "Object", "isExtensible" },
    &[_]string{ "Object", "isFrozen" },
    &[_]string{ "Object", "isSealed" },
    &[_]string{ "Object", "keys" },
    &[_]string{ "Object", "preventExtensions" },
    &[_]string{ "Object", "seal" },
    &[_]string{ "Object", "setPrototypeOf" },
    &[_]string{ "Object", "values" },

    // Object: Instance methods
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object#Instance_methods
    &[_]string{ "Object", "prototype", "__defineGetter__" },
    &[_]string{ "Object", "prototype", "__defineSetter__" },
    &[_]string{ "Object", "prototype", "__lookupGetter__" },
    &[_]string{ "Object", "prototype", "__lookupSetter__" },
    &[_]string{ "Object", "prototype", "hasOwnProperty" },
    &[_]string{ "Object", "prototype", "isPrototypeOf" },
    &[_]string{ "Object", "prototype", "propertyIsEnumerable" },
    &[_]string{ "Object", "prototype", "toLocaleString" },
    &[_]string{ "Object", "prototype", "toString" },
    &[_]string{ "Object", "prototype", "unwatch" },
    &[_]string{ "Object", "prototype", "valueOf" },
    &[_]string{ "Object", "prototype", "watch" },

    // Math: Static properties
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Math#Static_properties
    &[_]string{ "Math", "E" },
    &[_]string{ "Math", "LN10" },
    &[_]string{ "Math", "LN2" },
    &[_]string{ "Math", "LOG10E" },
    &[_]string{ "Math", "LOG2E" },
    &[_]string{ "Math", "PI" },
    &[_]string{ "Math", "SQRT1_2" },
    &[_]string{ "Math", "SQRT2" },

    // Math: Static methods
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Math#Static_methods
    &[_]string{ "Math", "abs" },
    &[_]string{ "Math", "acos" },
    &[_]string{ "Math", "acosh" },
    &[_]string{ "Math", "asin" },
    &[_]string{ "Math", "asinh" },
    &[_]string{ "Math", "atan" },
    &[_]string{ "Math", "atan2" },
    &[_]string{ "Math", "atanh" },
    &[_]string{ "Math", "cbrt" },
    &[_]string{ "Math", "ceil" },
    &[_]string{ "Math", "clz32" },
    &[_]string{ "Math", "cos" },
    &[_]string{ "Math", "cosh" },
    &[_]string{ "Math", "exp" },
    &[_]string{ "Math", "expm1" },
    &[_]string{ "Math", "floor" },
    &[_]string{ "Math", "fround" },
    &[_]string{ "Math", "hypot" },
    &[_]string{ "Math", "imul" },
    &[_]string{ "Math", "log" },
    &[_]string{ "Math", "log10" },
    &[_]string{ "Math", "log1p" },
    &[_]string{ "Math", "log2" },
    &[_]string{ "Math", "max" },
    &[_]string{ "Math", "min" },
    &[_]string{ "Math", "pow" },
    &[_]string{ "Math", "random" },
    &[_]string{ "Math", "round" },
    &[_]string{ "Math", "sign" },
    &[_]string{ "Math", "sin" },
    &[_]string{ "Math", "sinh" },
    &[_]string{ "Math", "sqrt" },
    &[_]string{ "Math", "tan" },
    &[_]string{ "Math", "tanh" },
    &[_]string{ "Math", "trunc" },

    &[_]string{ "Symbol", "for" },
    &[_]string{ "Symbol", "keyFor" },

    // Reflect: Static methods
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Reflect#static_methods
    &[_]string{ "Reflect", "apply" },
    &[_]string{ "Reflect", "construct" },
    &[_]string{ "Reflect", "defineProperty" },
    &[_]string{ "Reflect", "deleteProperty" },
    &[_]string{ "Reflect", "get" },
    &[_]string{ "Reflect", "getOwnPropertyDescriptor" },
    &[_]string{ "Reflect", "getPrototypeOf" },
    &[_]string{ "Reflect", "has" },
    &[_]string{ "Reflect", "isExtensible" },
    &[_]string{ "Reflect", "ownKeys" },
    &[_]string{ "Reflect", "preventExtensions" },
    &[_]string{ "Reflect", "set" },
    &[_]string{ "Reflect", "setPrototypeOf" },

    // Console method references are assumed to have no side effects
    // https://developer.mozilla.org/en-US/docs/Web/API/console
    &[_]string{ "console", "assert" },
    &[_]string{ "console", "clear" },
    &[_]string{ "console", "count" },
    &[_]string{ "console", "countReset" },
    &[_]string{ "console", "debug" },
    &[_]string{ "console", "dir" },
    &[_]string{ "console", "dirxml" },
    &[_]string{ "console", "error" },
    &[_]string{ "console", "group" },
    &[_]string{ "console", "groupCollapsed" },
    &[_]string{ "console", "groupEnd" },
    &[_]string{ "console", "info" },
    &[_]string{ "console", "log" },
    &[_]string{ "console", "table" },
    &[_]string{ "console", "time" },
    &[_]string{ "console", "timeEnd" },
    &[_]string{ "console", "timeLog" },
    &[_]string{ "console", "trace" },
    &[_]string{ "console", "warn" },
};

const pure_global_identifier_define = defines.IdentifierDefine{
    .valueless = true,
    .can_be_removed_if_unused = true,
    .value = .{ .e_undefined = .{} },
};
const js_ast = bun.JSAst;

const identifiers = struct {
    const nan_val = js_ast.E.Number{ .value = std.math.nan(f64) };

    const inf_val = js_ast.E.Number{ .value = std.math.inf(f64) };

    // Step 2. Swap in certain literal values because those can be constant folded
    pub const @"undefined" = defines.IdentifierDefine{
        .value = js_ast.Expr.Data{ .e_undefined = .{} },
        .valueless = false,
        .can_be_removed_if_unused = true,
    };
    pub const NaN = defines.IdentifierDefine{
        .value = js_ast.Expr.Data{ .e_number = nan_val },
    };
    pub const Infinity = defines.IdentifierDefine{
        .value = js_ast.Expr.Data{ .e_number = inf_val },
    };
};

pub const pure_global_identifiers = .{
    .{ "NaN", identifiers.NaN },
    .{ "Infinity", identifiers.Infinity },
    .{ "undefined", identifiers.undefined },

    // These global identifiers should exist in all JavaScript environments. This
    // deliberately omits "NaN", "Infinity", and "undefined" because these are
    // treated as automatically-inlined constants instead of identifiers.
    .{ "Array", pure_global_identifier_define },
    .{ "Boolean", pure_global_identifier_define },
    .{ "Function", pure_global_identifier_define },
    .{ "Math", pure_global_identifier_define },
    .{ "Number", pure_global_identifier_define },
    .{ "Object", pure_global_identifier_define },
    .{ "RegExp", pure_global_identifier_define },
    .{ "String", pure_global_identifier_define },

    // Other globals present in both the browser and node (except "eval" because
    // it has special behavior)
    .{ "AbortController", pure_global_identifier_define },
    .{ "AbortSignal", pure_global_identifier_define },
    .{ "AggregateError", pure_global_identifier_define },
    .{ "ArrayBuffer", pure_global_identifier_define },
    .{ "BigInt", pure_global_identifier_define },
    .{ "DataView", pure_global_identifier_define },
    .{ "Date", pure_global_identifier_define },
    .{ "Error", pure_global_identifier_define },
    .{ "EvalError", pure_global_identifier_define },
    .{ "Event", pure_global_identifier_define },
    .{ "EventTarget", pure_global_identifier_define },
    .{ "Float32Array", pure_global_identifier_define },
    .{ "Float64Array", pure_global_identifier_define },
    .{ "Int16Array", pure_global_identifier_define },
    .{ "Int32Array", pure_global_identifier_define },
    .{ "Int8Array", pure_global_identifier_define },
    .{ "Intl", pure_global_identifier_define },
    .{ "JSON", pure_global_identifier_define },
    .{ "Map", pure_global_identifier_define },
    .{ "MessageChannel", pure_global_identifier_define },
    .{ "MessageEvent", pure_global_identifier_define },
    .{ "MessagePort", pure_global_identifier_define },
    .{ "Promise", pure_global_identifier_define },
    .{ "Proxy", pure_global_identifier_define },
    .{ "RangeError", pure_global_identifier_define },
    .{ "ReferenceError", pure_global_identifier_define },
    .{ "Reflect", pure_global_identifier_define },
    .{ "Set", pure_global_identifier_define },
    .{ "Symbol", pure_global_identifier_define },
    .{ "SyntaxError", pure_global_identifier_define },
    .{ "TextDecoder", pure_global_identifier_define },
    .{ "TextEncoder", pure_global_identifier_define },
    .{ "TypeError", pure_global_identifier_define },
    .{ "URIError", pure_global_identifier_define },
    .{ "URL", pure_global_identifier_define },
    .{ "URLSearchParams", pure_global_identifier_define },
    .{ "Uint16Array", pure_global_identifier_define },
    .{ "Uint32Array", pure_global_identifier_define },
    .{ "Uint8Array", pure_global_identifier_define },
    .{ "Uint8ClampedArray", pure_global_identifier_define },
    .{ "WeakMap", pure_global_identifier_define },
    .{ "WeakSet", pure_global_identifier_define },
    .{ "WebAssembly", pure_global_identifier_define },
    .{ "clearInterval", pure_global_identifier_define },
    .{ "clearTimeout", pure_global_identifier_define },
    .{ "console", pure_global_identifier_define },
    .{ "decodeURI", pure_global_identifier_define },
    .{ "decodeURIComponent", pure_global_identifier_define },
    .{ "encodeURI", pure_global_identifier_define },
    .{ "encodeURIComponent", pure_global_identifier_define },
    .{ "escape", pure_global_identifier_define },
    .{ "globalThis", pure_global_identifier_define },
    .{ "isFinite", pure_global_identifier_define },
    .{ "isNaN", pure_global_identifier_define },
    .{ "parseFloat", pure_global_identifier_define },
    .{ "parseInt", pure_global_identifier_define },
    .{ "queueMicrotask", pure_global_identifier_define },
    .{ "setInterval", pure_global_identifier_define },
    .{ "setTimeout", pure_global_identifier_define },
    .{ "unescape", pure_global_identifier_define },

    // CSSOM APIs
    .{ "CSSAnimation", pure_global_identifier_define },
    .{ "CSSFontFaceRule", pure_global_identifier_define },
    .{ "CSSImportRule", pure_global_identifier_define },
    .{ "CSSKeyframeRule", pure_global_identifier_define },
    .{ "CSSKeyframesRule", pure_global_identifier_define },
    .{ "CSSMediaRule", pure_global_identifier_define },
    .{ "CSSNamespaceRule", pure_global_identifier_define },
    .{ "CSSPageRule", pure_global_identifier_define },
    .{ "CSSRule", pure_global_identifier_define },
    .{ "CSSRuleList", pure_global_identifier_define },
    .{ "CSSStyleDeclaration", pure_global_identifier_define },
    .{ "CSSStyleRule", pure_global_identifier_define },
    .{ "CSSStyleSheet", pure_global_identifier_define },
    .{ "CSSSupportsRule", pure_global_identifier_define },
    .{ "CSSTransition", pure_global_identifier_define },

    // SVG DOM
    .{ "SVGAElement", pure_global_identifier_define },
    .{ "SVGAngle", pure_global_identifier_define },
    .{ "SVGAnimateElement", pure_global_identifier_define },
    .{ "SVGAnimateMotionElement", pure_global_identifier_define },
    .{ "SVGAnimateTransformElement", pure_global_identifier_define },
    .{ "SVGAnimatedAngle", pure_global_identifier_define },
    .{ "SVGAnimatedBoolean", pure_global_identifier_define },
    .{ "SVGAnimatedEnumeration", pure_global_identifier_define },
    .{ "SVGAnimatedInteger", pure_global_identifier_define },
    .{ "SVGAnimatedLength", pure_global_identifier_define },
    .{ "SVGAnimatedLengthList", pure_global_identifier_define },
    .{ "SVGAnimatedNumber", pure_global_identifier_define },
    .{ "SVGAnimatedNumberList", pure_global_identifier_define },
    .{ "SVGAnimatedPreserveAspectRatio", pure_global_identifier_define },
    .{ "SVGAnimatedRect", pure_global_identifier_define },
    .{ "SVGAnimatedString", pure_global_identifier_define },
    .{ "SVGAnimatedTransformList", pure_global_identifier_define },
    .{ "SVGAnimationElement", pure_global_identifier_define },
    .{ "SVGCircleElement", pure_global_identifier_define },
    .{ "SVGClipPathElement", pure_global_identifier_define },
    .{ "SVGComponentTransferFunctionElement", pure_global_identifier_define },
    .{ "SVGDefsElement", pure_global_identifier_define },
    .{ "SVGDescElement", pure_global_identifier_define },
    .{ "SVGElement", pure_global_identifier_define },
    .{ "SVGEllipseElement", pure_global_identifier_define },
    .{ "SVGFEBlendElement", pure_global_identifier_define },
    .{ "SVGFEColorMatrixElement", pure_global_identifier_define },
    .{ "SVGFEComponentTransferElement", pure_global_identifier_define },
    .{ "SVGFECompositeElement", pure_global_identifier_define },
    .{ "SVGFEConvolveMatrixElement", pure_global_identifier_define },
    .{ "SVGFEDiffuseLightingElement", pure_global_identifier_define },
    .{ "SVGFEDisplacementMapElement", pure_global_identifier_define },
    .{ "SVGFEDistantLightElement", pure_global_identifier_define },
    .{ "SVGFEDropShadowElement", pure_global_identifier_define },
    .{ "SVGFEFloodElement", pure_global_identifier_define },
    .{ "SVGFEFuncAElement", pure_global_identifier_define },
    .{ "SVGFEFuncBElement", pure_global_identifier_define },
    .{ "SVGFEFuncGElement", pure_global_identifier_define },
    .{ "SVGFEFuncRElement", pure_global_identifier_define },
    .{ "SVGFEGaussianBlurElement", pure_global_identifier_define },
    .{ "SVGFEImageElement", pure_global_identifier_define },
    .{ "SVGFEMergeElement", pure_global_identifier_define },
    .{ "SVGFEMergeNodeElement", pure_global_identifier_define },
    .{ "SVGFEMorphologyElement", pure_global_identifier_define },
    .{ "SVGFEOffsetElement", pure_global_identifier_define },
    .{ "SVGFEPointLightElement", pure_global_identifier_define },
    .{ "SVGFESpecularLightingElement", pure_global_identifier_define },
    .{ "SVGFESpotLightElement", pure_global_identifier_define },
    .{ "SVGFETileElement", pure_global_identifier_define },
    .{ "SVGFETurbulenceElement", pure_global_identifier_define },
    .{ "SVGFilterElement", pure_global_identifier_define },
    .{ "SVGForeignObjectElement", pure_global_identifier_define },
    .{ "SVGGElement", pure_global_identifier_define },
    .{ "SVGGeometryElement", pure_global_identifier_define },
    .{ "SVGGradientElement", pure_global_identifier_define },
    .{ "SVGGraphicsElement", pure_global_identifier_define },
    .{ "SVGImageElement", pure_global_identifier_define },
    .{ "SVGLength", pure_global_identifier_define },
    .{ "SVGLengthList", pure_global_identifier_define },
    .{ "SVGLineElement", pure_global_identifier_define },
    .{ "SVGLinearGradientElement", pure_global_identifier_define },
    .{ "SVGMPathElement", pure_global_identifier_define },
    .{ "SVGMarkerElement", pure_global_identifier_define },
    .{ "SVGMaskElement", pure_global_identifier_define },
    .{ "SVGMatrix", pure_global_identifier_define },
    .{ "SVGMetadataElement", pure_global_identifier_define },
    .{ "SVGNumber", pure_global_identifier_define },
    .{ "SVGNumberList", pure_global_identifier_define },
    .{ "SVGPathElement", pure_global_identifier_define },
    .{ "SVGPatternElement", pure_global_identifier_define },
    .{ "SVGPoint", pure_global_identifier_define },
    .{ "SVGPointList", pure_global_identifier_define },
    .{ "SVGPolygonElement", pure_global_identifier_define },
    .{ "SVGPolylineElement", pure_global_identifier_define },
    .{ "SVGPreserveAspectRatio", pure_global_identifier_define },
    .{ "SVGRadialGradientElement", pure_global_identifier_define },
    .{ "SVGRect", pure_global_identifier_define },
    .{ "SVGRectElement", pure_global_identifier_define },
    .{ "SVGSVGElement", pure_global_identifier_define },
    .{ "SVGScriptElement", pure_global_identifier_define },
    .{ "SVGSetElement", pure_global_identifier_define },
    .{ "SVGStopElement", pure_global_identifier_define },
    .{ "SVGStringList", pure_global_identifier_define },
    .{ "SVGStyleElement", pure_global_identifier_define },
    .{ "SVGSwitchElement", pure_global_identifier_define },
    .{ "SVGSymbolElement", pure_global_identifier_define },
    .{ "SVGTSpanElement", pure_global_identifier_define },
    .{ "SVGTextContentElement", pure_global_identifier_define },
    .{ "SVGTextElement", pure_global_identifier_define },
    .{ "SVGTextPathElement", pure_global_identifier_define },
    .{ "SVGTextPositioningElement", pure_global_identifier_define },
    .{ "SVGTitleElement", pure_global_identifier_define },
    .{ "SVGTransform", pure_global_identifier_define },
    .{ "SVGTransformList", pure_global_identifier_define },
    .{ "SVGUnitTypes", pure_global_identifier_define },
    .{ "SVGUseElement", pure_global_identifier_define },
    .{ "SVGViewElement", pure_global_identifier_define },

    // Other browser APIs
    //
    // This list contains all globals present in modern versions of Chrome, Safari,
    // and Firefox except for the following properties, since they have a side effect
    // of triggering layout (https://gist.github.com/paulirish/5d52fb081b3570c81e3a):
    //
    //   - scrollX
    //   - scrollY
    //   - innerWidth
    //   - innerHeight
    //   - pageXOffset
    //   - pageYOffset
    //
    // The following globals have also been removed since they sometimes throw an
    // exception when accessed, which is a side effect (for more information see
    // https://stackoverflow.com/a/33047477):
    //
    //   - localStorage
    //   - sessionStorage
    //
    .{ "AnalyserNode", pure_global_identifier_define },
    .{ "Animation", pure_global_identifier_define },
    .{ "AnimationEffect", pure_global_identifier_define },
    .{ "AnimationEvent", pure_global_identifier_define },
    .{ "AnimationPlaybackEvent", pure_global_identifier_define },
    .{ "AnimationTimeline", pure_global_identifier_define },
    .{ "Attr", pure_global_identifier_define },
    .{ "Audio", pure_global_identifier_define },
    .{ "AudioBuffer", pure_global_identifier_define },
    .{ "AudioBufferSourceNode", pure_global_identifier_define },
    .{ "AudioDestinationNode", pure_global_identifier_define },
    .{ "AudioListener", pure_global_identifier_define },
    .{ "AudioNode", pure_global_identifier_define },
    .{ "AudioParam", pure_global_identifier_define },
    .{ "AudioProcessingEvent", pure_global_identifier_define },
    .{ "AudioScheduledSourceNode", pure_global_identifier_define },
    .{ "BarProp", pure_global_identifier_define },
    .{ "BeforeUnloadEvent", pure_global_identifier_define },
    .{ "BiquadFilterNode", pure_global_identifier_define },
    .{ "Blob", pure_global_identifier_define },
    .{ "BlobEvent", pure_global_identifier_define },
    .{ "ByteLengthQueuingStrategy", pure_global_identifier_define },
    .{ "CDATASection", pure_global_identifier_define },
    .{ "CSS", pure_global_identifier_define },
    .{ "CanvasGradient", pure_global_identifier_define },
    .{ "CanvasPattern", pure_global_identifier_define },
    .{ "CanvasRenderingContext2D", pure_global_identifier_define },
    .{ "ChannelMergerNode", pure_global_identifier_define },
    .{ "ChannelSplitterNode", pure_global_identifier_define },
    .{ "CharacterData", pure_global_identifier_define },
    .{ "ClipboardEvent", pure_global_identifier_define },
    .{ "CloseEvent", pure_global_identifier_define },
    .{ "Comment", pure_global_identifier_define },
    .{ "CompositionEvent", pure_global_identifier_define },
    .{ "ConvolverNode", pure_global_identifier_define },
    .{ "CountQueuingStrategy", pure_global_identifier_define },
    .{ "Crypto", pure_global_identifier_define },
    .{ "CustomElementRegistry", pure_global_identifier_define },
    .{ "CustomEvent", pure_global_identifier_define },
    .{ "DOMException", pure_global_identifier_define },
    .{ "DOMImplementation", pure_global_identifier_define },
    .{ "DOMMatrix", pure_global_identifier_define },
    .{ "DOMMatrixReadOnly", pure_global_identifier_define },
    .{ "DOMParser", pure_global_identifier_define },
    .{ "DOMPoint", pure_global_identifier_define },
    .{ "DOMPointReadOnly", pure_global_identifier_define },
    .{ "DOMQuad", pure_global_identifier_define },
    .{ "DOMRect", pure_global_identifier_define },
    .{ "DOMRectList", pure_global_identifier_define },
    .{ "DOMRectReadOnly", pure_global_identifier_define },
    .{ "DOMStringList", pure_global_identifier_define },
    .{ "DOMStringMap", pure_global_identifier_define },
    .{ "DOMTokenList", pure_global_identifier_define },
    .{ "DataTransfer", pure_global_identifier_define },
    .{ "DataTransferItem", pure_global_identifier_define },
    .{ "DataTransferItemList", pure_global_identifier_define },
    .{ "DelayNode", pure_global_identifier_define },
    .{ "Document", pure_global_identifier_define },
    .{ "DocumentFragment", pure_global_identifier_define },
    .{ "DocumentTimeline", pure_global_identifier_define },
    .{ "DocumentType", pure_global_identifier_define },
    .{ "DragEvent", pure_global_identifier_define },
    .{ "DynamicsCompressorNode", pure_global_identifier_define },
    .{ "Element", pure_global_identifier_define },
    .{ "ErrorEvent", pure_global_identifier_define },
    .{ "EventSource", pure_global_identifier_define },
    .{ "File", pure_global_identifier_define },
    .{ "FileList", pure_global_identifier_define },
    .{ "FileReader", pure_global_identifier_define },
    .{ "FocusEvent", pure_global_identifier_define },
    .{ "FontFace", pure_global_identifier_define },
    .{ "FormData", pure_global_identifier_define },
    .{ "GainNode", pure_global_identifier_define },
    .{ "Gamepad", pure_global_identifier_define },
    .{ "GamepadButton", pure_global_identifier_define },
    .{ "GamepadEvent", pure_global_identifier_define },
    .{ "Geolocation", pure_global_identifier_define },
    .{ "GeolocationPositionError", pure_global_identifier_define },
    .{ "HTMLAllCollection", pure_global_identifier_define },
    .{ "HTMLAnchorElement", pure_global_identifier_define },
    .{ "HTMLAreaElement", pure_global_identifier_define },
    .{ "HTMLAudioElement", pure_global_identifier_define },
    .{ "HTMLBRElement", pure_global_identifier_define },
    .{ "HTMLBaseElement", pure_global_identifier_define },
    .{ "HTMLBodyElement", pure_global_identifier_define },
    .{ "HTMLButtonElement", pure_global_identifier_define },
    .{ "HTMLCanvasElement", pure_global_identifier_define },
    .{ "HTMLCollection", pure_global_identifier_define },
    .{ "HTMLDListElement", pure_global_identifier_define },
    .{ "HTMLDataElement", pure_global_identifier_define },
    .{ "HTMLDataListElement", pure_global_identifier_define },
    .{ "HTMLDetailsElement", pure_global_identifier_define },
    .{ "HTMLDirectoryElement", pure_global_identifier_define },
    .{ "HTMLDivElement", pure_global_identifier_define },
    .{ "HTMLDocument", pure_global_identifier_define },
    .{ "HTMLElement", pure_global_identifier_define },
    .{ "HTMLEmbedElement", pure_global_identifier_define },
    .{ "HTMLFieldSetElement", pure_global_identifier_define },
    .{ "HTMLFontElement", pure_global_identifier_define },
    .{ "HTMLFormControlsCollection", pure_global_identifier_define },
    .{ "HTMLFormElement", pure_global_identifier_define },
    .{ "HTMLFrameElement", pure_global_identifier_define },
    .{ "HTMLFrameSetElement", pure_global_identifier_define },
    .{ "HTMLHRElement", pure_global_identifier_define },
    .{ "HTMLHeadElement", pure_global_identifier_define },
    .{ "HTMLHeadingElement", pure_global_identifier_define },
    .{ "HTMLHtmlElement", pure_global_identifier_define },
    .{ "HTMLIFrameElement", pure_global_identifier_define },
    .{ "HTMLImageElement", pure_global_identifier_define },
    .{ "HTMLInputElement", pure_global_identifier_define },
    .{ "HTMLLIElement", pure_global_identifier_define },
    .{ "HTMLLabelElement", pure_global_identifier_define },
    .{ "HTMLLegendElement", pure_global_identifier_define },
    .{ "HTMLLinkElement", pure_global_identifier_define },
    .{ "HTMLMapElement", pure_global_identifier_define },
    .{ "HTMLMarqueeElement", pure_global_identifier_define },
    .{ "HTMLMediaElement", pure_global_identifier_define },
    .{ "HTMLMenuElement", pure_global_identifier_define },
    .{ "HTMLMetaElement", pure_global_identifier_define },
    .{ "HTMLMeterElement", pure_global_identifier_define },
    .{ "HTMLModElement", pure_global_identifier_define },
    .{ "HTMLOListElement", pure_global_identifier_define },
    .{ "HTMLObjectElement", pure_global_identifier_define },
    .{ "HTMLOptGroupElement", pure_global_identifier_define },
    .{ "HTMLOptionElement", pure_global_identifier_define },
    .{ "HTMLOptionsCollection", pure_global_identifier_define },
    .{ "HTMLOutputElement", pure_global_identifier_define },
    .{ "HTMLParagraphElement", pure_global_identifier_define },
    .{ "HTMLParamElement", pure_global_identifier_define },
    .{ "HTMLPictureElement", pure_global_identifier_define },
    .{ "HTMLPreElement", pure_global_identifier_define },
    .{ "HTMLProgressElement", pure_global_identifier_define },
    .{ "HTMLQuoteElement", pure_global_identifier_define },
    .{ "HTMLScriptElement", pure_global_identifier_define },
    .{ "HTMLSelectElement", pure_global_identifier_define },
    .{ "HTMLSlotElement", pure_global_identifier_define },
    .{ "HTMLSourceElement", pure_global_identifier_define },
    .{ "HTMLSpanElement", pure_global_identifier_define },
    .{ "HTMLStyleElement", pure_global_identifier_define },
    .{ "HTMLTableCaptionElement", pure_global_identifier_define },
    .{ "HTMLTableCellElement", pure_global_identifier_define },
    .{ "HTMLTableColElement", pure_global_identifier_define },
    .{ "HTMLTableElement", pure_global_identifier_define },
    .{ "HTMLTableRowElement", pure_global_identifier_define },
    .{ "HTMLTableSectionElement", pure_global_identifier_define },
    .{ "HTMLTemplateElement", pure_global_identifier_define },
    .{ "HTMLTextAreaElement", pure_global_identifier_define },
    .{ "HTMLTimeElement", pure_global_identifier_define },
    .{ "HTMLTitleElement", pure_global_identifier_define },
    .{ "HTMLTrackElement", pure_global_identifier_define },
    .{ "HTMLUListElement", pure_global_identifier_define },
    .{ "HTMLUnknownElement", pure_global_identifier_define },
    .{ "HTMLVideoElement", pure_global_identifier_define },
    .{ "HashChangeEvent", pure_global_identifier_define },
    .{ "Headers", pure_global_identifier_define },
    .{ "History", pure_global_identifier_define },
    .{ "IDBCursor", pure_global_identifier_define },
    .{ "IDBCursorWithValue", pure_global_identifier_define },
    .{ "IDBDatabase", pure_global_identifier_define },
    .{ "IDBFactory", pure_global_identifier_define },
    .{ "IDBIndex", pure_global_identifier_define },
    .{ "IDBKeyRange", pure_global_identifier_define },
    .{ "IDBObjectStore", pure_global_identifier_define },
    .{ "IDBOpenDBRequest", pure_global_identifier_define },
    .{ "IDBRequest", pure_global_identifier_define },
    .{ "IDBTransaction", pure_global_identifier_define },
    .{ "IDBVersionChangeEvent", pure_global_identifier_define },
    .{ "Image", pure_global_identifier_define },
    .{ "ImageData", pure_global_identifier_define },
    .{ "InputEvent", pure_global_identifier_define },
    .{ "IntersectionObserver", pure_global_identifier_define },
    .{ "IntersectionObserverEntry", pure_global_identifier_define },
    .{ "KeyboardEvent", pure_global_identifier_define },
    .{ "KeyframeEffect", pure_global_identifier_define },
    .{ "Location", pure_global_identifier_define },
    .{ "MediaCapabilities", pure_global_identifier_define },
    .{ "MediaElementAudioSourceNode", pure_global_identifier_define },
    .{ "MediaEncryptedEvent", pure_global_identifier_define },
    .{ "MediaError", pure_global_identifier_define },
    .{ "MediaList", pure_global_identifier_define },
    .{ "MediaQueryList", pure_global_identifier_define },
    .{ "MediaQueryListEvent", pure_global_identifier_define },
    .{ "MediaRecorder", pure_global_identifier_define },
    .{ "MediaSource", pure_global_identifier_define },
    .{ "MediaStream", pure_global_identifier_define },
    .{ "MediaStreamAudioDestinationNode", pure_global_identifier_define },
    .{ "MediaStreamAudioSourceNode", pure_global_identifier_define },
    .{ "MediaStreamTrack", pure_global_identifier_define },
    .{ "MediaStreamTrackEvent", pure_global_identifier_define },
    .{ "MimeType", pure_global_identifier_define },
    .{ "MimeTypeArray", pure_global_identifier_define },
    .{ "MouseEvent", pure_global_identifier_define },
    .{ "MutationEvent", pure_global_identifier_define },
    .{ "MutationObserver", pure_global_identifier_define },
    .{ "MutationRecord", pure_global_identifier_define },
    .{ "NamedNodeMap", pure_global_identifier_define },
    .{ "Navigator", pure_global_identifier_define },
    .{ "Node", pure_global_identifier_define },
    .{ "NodeFilter", pure_global_identifier_define },
    .{ "NodeIterator", pure_global_identifier_define },
    .{ "NodeList", pure_global_identifier_define },
    .{ "Notification", pure_global_identifier_define },
    .{ "OfflineAudioCompletionEvent", pure_global_identifier_define },
    .{ "Option", pure_global_identifier_define },
    .{ "OscillatorNode", pure_global_identifier_define },
    .{ "PageTransitionEvent", pure_global_identifier_define },
    .{ "Path2D", pure_global_identifier_define },
    .{ "Performance", pure_global_identifier_define },
    .{ "PerformanceEntry", pure_global_identifier_define },
    .{ "PerformanceMark", pure_global_identifier_define },
    .{ "PerformanceMeasure", pure_global_identifier_define },
    .{ "PerformanceNavigation", pure_global_identifier_define },
    .{ "PerformanceObserver", pure_global_identifier_define },
    .{ "PerformanceObserverEntryList", pure_global_identifier_define },
    .{ "PerformanceResourceTiming", pure_global_identifier_define },
    .{ "PerformanceTiming", pure_global_identifier_define },
    .{ "PeriodicWave", pure_global_identifier_define },
    .{ "Plugin", pure_global_identifier_define },
    .{ "PluginArray", pure_global_identifier_define },
    .{ "PointerEvent", pure_global_identifier_define },
    .{ "PopStateEvent", pure_global_identifier_define },
    .{ "ProcessingInstruction", pure_global_identifier_define },
    .{ "ProgressEvent", pure_global_identifier_define },
    .{ "PromiseRejectionEvent", pure_global_identifier_define },
    .{ "RTCCertificate", pure_global_identifier_define },
    .{ "RTCDTMFSender", pure_global_identifier_define },
    .{ "RTCDTMFToneChangeEvent", pure_global_identifier_define },
    .{ "RTCDataChannel", pure_global_identifier_define },
    .{ "RTCDataChannelEvent", pure_global_identifier_define },
    .{ "RTCIceCandidate", pure_global_identifier_define },
    .{ "RTCPeerConnection", pure_global_identifier_define },
    .{ "RTCPeerConnectionIceEvent", pure_global_identifier_define },
    .{ "RTCRtpReceiver", pure_global_identifier_define },
    .{ "RTCRtpSender", pure_global_identifier_define },
    .{ "RTCRtpTransceiver", pure_global_identifier_define },
    .{ "RTCSessionDescription", pure_global_identifier_define },
    .{ "RTCStatsReport", pure_global_identifier_define },
    .{ "RTCTrackEvent", pure_global_identifier_define },
    .{ "RadioNodeList", pure_global_identifier_define },
    .{ "Range", pure_global_identifier_define },
    .{ "ReadableStream", pure_global_identifier_define },
    .{ "Request", pure_global_identifier_define },
    .{ "ResizeObserver", pure_global_identifier_define },
    .{ "ResizeObserverEntry", pure_global_identifier_define },
    .{ "Response", pure_global_identifier_define },
    .{ "Screen", pure_global_identifier_define },
    .{ "ScriptProcessorNode", pure_global_identifier_define },
    .{ "SecurityPolicyViolationEvent", pure_global_identifier_define },
    .{ "Selection", pure_global_identifier_define },
    .{ "ShadowRoot", pure_global_identifier_define },
    .{ "SourceBuffer", pure_global_identifier_define },
    .{ "SourceBufferList", pure_global_identifier_define },
    .{ "SpeechSynthesisEvent", pure_global_identifier_define },
    .{ "SpeechSynthesisUtterance", pure_global_identifier_define },
    .{ "StaticRange", pure_global_identifier_define },
    .{ "Storage", pure_global_identifier_define },
    .{ "StorageEvent", pure_global_identifier_define },
    .{ "StyleSheet", pure_global_identifier_define },
    .{ "StyleSheetList", pure_global_identifier_define },
    .{ "Text", pure_global_identifier_define },
    .{ "TextMetrics", pure_global_identifier_define },
    .{ "TextTrack", pure_global_identifier_define },
    .{ "TextTrackCue", pure_global_identifier_define },
    .{ "TextTrackCueList", pure_global_identifier_define },
    .{ "TextTrackList", pure_global_identifier_define },
    .{ "TimeRanges", pure_global_identifier_define },
    .{ "TrackEvent", pure_global_identifier_define },
    .{ "TransitionEvent", pure_global_identifier_define },
    .{ "TreeWalker", pure_global_identifier_define },
    .{ "UIEvent", pure_global_identifier_define },
    .{ "VTTCue", pure_global_identifier_define },
    .{ "ValidityState", pure_global_identifier_define },
    .{ "VisualViewport", pure_global_identifier_define },
    .{ "WaveShaperNode", pure_global_identifier_define },
    .{ "WebGLActiveInfo", pure_global_identifier_define },
    .{ "WebGLBuffer", pure_global_identifier_define },
    .{ "WebGLContextEvent", pure_global_identifier_define },
    .{ "WebGLFramebuffer", pure_global_identifier_define },
    .{ "WebGLProgram", pure_global_identifier_define },
    .{ "WebGLQuery", pure_global_identifier_define },
    .{ "WebGLRenderbuffer", pure_global_identifier_define },
    .{ "WebGLRenderingContext", pure_global_identifier_define },
    .{ "WebGLSampler", pure_global_identifier_define },
    .{ "WebGLShader", pure_global_identifier_define },
    .{ "WebGLShaderPrecisionFormat", pure_global_identifier_define },
    .{ "WebGLSync", pure_global_identifier_define },
    .{ "WebGLTexture", pure_global_identifier_define },
    .{ "WebGLUniformLocation", pure_global_identifier_define },
    .{ "WebKitCSSMatrix", pure_global_identifier_define },
    .{ "WebSocket", pure_global_identifier_define },
    .{ "WheelEvent", pure_global_identifier_define },
    .{ "Window", pure_global_identifier_define },
    .{ "Worker", pure_global_identifier_define },
    .{ "XMLDocument", pure_global_identifier_define },
    .{ "XMLHttpRequest", pure_global_identifier_define },
    .{ "XMLHttpRequestEventTarget", pure_global_identifier_define },
    .{ "XMLHttpRequestUpload", pure_global_identifier_define },
    .{ "XMLSerializer", pure_global_identifier_define },
    .{ "XPathEvaluator", pure_global_identifier_define },
    .{ "XPathExpression", pure_global_identifier_define },
    .{ "XPathResult", pure_global_identifier_define },
    .{ "XSLTProcessor", pure_global_identifier_define },
    .{ "alert", pure_global_identifier_define },
    .{ "atob", pure_global_identifier_define },
    .{ "blur", pure_global_identifier_define },
    .{ "btoa", pure_global_identifier_define },
    .{ "cancelAnimationFrame", pure_global_identifier_define },
    .{ "captureEvents", pure_global_identifier_define },
    .{ "close", pure_global_identifier_define },
    .{ "closed", pure_global_identifier_define },
    .{ "confirm", pure_global_identifier_define },
    .{ "customElements", pure_global_identifier_define },
    .{ "devicePixelRatio", pure_global_identifier_define },
    .{ "document", pure_global_identifier_define },
    .{ "event", pure_global_identifier_define },
    .{ "fetch", pure_global_identifier_define },
    .{ "find", pure_global_identifier_define },
    .{ "focus", pure_global_identifier_define },
    .{ "frameElement", pure_global_identifier_define },
    .{ "frames", pure_global_identifier_define },
    .{ "getComputedStyle", pure_global_identifier_define },
    .{ "getSelection", pure_global_identifier_define },
    .{ "history", pure_global_identifier_define },
    .{ "indexedDB", pure_global_identifier_define },
    .{ "isSecureContext", pure_global_identifier_define },
    .{ "length", pure_global_identifier_define },
    .{ "location", pure_global_identifier_define },
    .{ "locationbar", pure_global_identifier_define },
    .{ "matchMedia", pure_global_identifier_define },
    .{ "menubar", pure_global_identifier_define },
    .{ "moveBy", pure_global_identifier_define },
    .{ "moveTo", pure_global_identifier_define },
    .{ "name", pure_global_identifier_define },
    .{ "navigator", pure_global_identifier_define },
    .{ "onabort", pure_global_identifier_define },
    .{ "onafterprint", pure_global_identifier_define },
    .{ "onanimationend", pure_global_identifier_define },
    .{ "onanimationiteration", pure_global_identifier_define },
    .{ "onanimationstart", pure_global_identifier_define },
    .{ "onbeforeprint", pure_global_identifier_define },
    .{ "onbeforeunload", pure_global_identifier_define },
    .{ "onblur", pure_global_identifier_define },
    .{ "oncanplay", pure_global_identifier_define },
    .{ "oncanplaythrough", pure_global_identifier_define },
    .{ "onchange", pure_global_identifier_define },
    .{ "onclick", pure_global_identifier_define },
    .{ "oncontextmenu", pure_global_identifier_define },
    .{ "oncuechange", pure_global_identifier_define },
    .{ "ondblclick", pure_global_identifier_define },
    .{ "ondrag", pure_global_identifier_define },
    .{ "ondragend", pure_global_identifier_define },
    .{ "ondragenter", pure_global_identifier_define },
    .{ "ondragleave", pure_global_identifier_define },
    .{ "ondragover", pure_global_identifier_define },
    .{ "ondragstart", pure_global_identifier_define },
    .{ "ondrop", pure_global_identifier_define },
    .{ "ondurationchange", pure_global_identifier_define },
    .{ "onemptied", pure_global_identifier_define },
    .{ "onended", pure_global_identifier_define },
    .{ "onerror", pure_global_identifier_define },
    .{ "onfocus", pure_global_identifier_define },
    .{ "ongotpointercapture", pure_global_identifier_define },
    .{ "onhashchange", pure_global_identifier_define },
    .{ "oninput", pure_global_identifier_define },
    .{ "oninvalid", pure_global_identifier_define },
    .{ "onkeydown", pure_global_identifier_define },
    .{ "onkeypress", pure_global_identifier_define },
    .{ "onkeyup", pure_global_identifier_define },
    .{ "onlanguagechange", pure_global_identifier_define },
    .{ "onload", pure_global_identifier_define },
    .{ "onloadeddata", pure_global_identifier_define },
    .{ "onloadedmetadata", pure_global_identifier_define },
    .{ "onloadstart", pure_global_identifier_define },
    .{ "onlostpointercapture", pure_global_identifier_define },
    .{ "onmessage", pure_global_identifier_define },
    .{ "onmousedown", pure_global_identifier_define },
    .{ "onmouseenter", pure_global_identifier_define },
    .{ "onmouseleave", pure_global_identifier_define },
    .{ "onmousemove", pure_global_identifier_define },
    .{ "onmouseout", pure_global_identifier_define },
    .{ "onmouseover", pure_global_identifier_define },
    .{ "onmouseup", pure_global_identifier_define },
    .{ "onoffline", pure_global_identifier_define },
    .{ "ononline", pure_global_identifier_define },
    .{ "onpagehide", pure_global_identifier_define },
    .{ "onpageshow", pure_global_identifier_define },
    .{ "onpause", pure_global_identifier_define },
    .{ "onplay", pure_global_identifier_define },
    .{ "onplaying", pure_global_identifier_define },
    .{ "onpointercancel", pure_global_identifier_define },
    .{ "onpointerdown", pure_global_identifier_define },
    .{ "onpointerenter", pure_global_identifier_define },
    .{ "onpointerleave", pure_global_identifier_define },
    .{ "onpointermove", pure_global_identifier_define },
    .{ "onpointerout", pure_global_identifier_define },
    .{ "onpointerover", pure_global_identifier_define },
    .{ "onpointerup", pure_global_identifier_define },
    .{ "onpopstate", pure_global_identifier_define },
    .{ "onprogress", pure_global_identifier_define },
    .{ "onratechange", pure_global_identifier_define },
    .{ "onrejectionhandled", pure_global_identifier_define },
    .{ "onreset", pure_global_identifier_define },
    .{ "onresize", pure_global_identifier_define },
    .{ "onscroll", pure_global_identifier_define },
    .{ "onseeked", pure_global_identifier_define },
    .{ "onseeking", pure_global_identifier_define },
    .{ "onselect", pure_global_identifier_define },
    .{ "onstalled", pure_global_identifier_define },
    .{ "onstorage", pure_global_identifier_define },
    .{ "onsubmit", pure_global_identifier_define },
    .{ "onsuspend", pure_global_identifier_define },
    .{ "ontimeupdate", pure_global_identifier_define },
    .{ "ontoggle", pure_global_identifier_define },
    .{ "ontransitioncancel", pure_global_identifier_define },
    .{ "ontransitionend", pure_global_identifier_define },
    .{ "ontransitionrun", pure_global_identifier_define },
    .{ "ontransitionstart", pure_global_identifier_define },
    .{ "onunhandledrejection", pure_global_identifier_define },
    .{ "onunload", pure_global_identifier_define },
    .{ "onvolumechange", pure_global_identifier_define },
    .{ "onwaiting", pure_global_identifier_define },
    .{ "onwebkitanimationend", pure_global_identifier_define },
    .{ "onwebkitanimationiteration", pure_global_identifier_define },
    .{ "onwebkitanimationstart", pure_global_identifier_define },
    .{ "onwebkittransitionend", pure_global_identifier_define },
    .{ "onwheel", pure_global_identifier_define },
    .{ "open", pure_global_identifier_define },
    .{ "opener", pure_global_identifier_define },
    .{ "origin", pure_global_identifier_define },
    .{ "outerHeight", pure_global_identifier_define },
    .{ "outerWidth", pure_global_identifier_define },
    .{ "parent", pure_global_identifier_define },
    .{ "performance", pure_global_identifier_define },
    .{ "personalbar", pure_global_identifier_define },
    .{ "postMessage", pure_global_identifier_define },
    .{ "print", pure_global_identifier_define },
    .{ "prompt", pure_global_identifier_define },
    .{ "releaseEvents", pure_global_identifier_define },
    .{ "requestAnimationFrame", pure_global_identifier_define },
    .{ "resizeBy", pure_global_identifier_define },
    .{ "resizeTo", pure_global_identifier_define },
    .{ "screen", pure_global_identifier_define },
    .{ "screenLeft", pure_global_identifier_define },
    .{ "screenTop", pure_global_identifier_define },
    .{ "screenX", pure_global_identifier_define },
    .{ "screenY", pure_global_identifier_define },
    .{ "scroll", pure_global_identifier_define },
    .{ "scrollBy", pure_global_identifier_define },
    .{ "scrollTo", pure_global_identifier_define },
    .{ "scrollbars", pure_global_identifier_define },
    .{ "self", pure_global_identifier_define },
    .{ "speechSynthesis", pure_global_identifier_define },
    .{ "status", pure_global_identifier_define },
    .{ "statusbar", pure_global_identifier_define },
    .{ "stop", pure_global_identifier_define },
    .{ "toolbar", pure_global_identifier_define },
    .{ "top", pure_global_identifier_define },
    .{ "webkitURL", pure_global_identifier_define },
    .{ "window", pure_global_identifier_define },
};

pub const pure_global_identifier_map = bun.ComptimeStringMap(defines.IdentifierDefine, pure_global_identifiers);
