/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.cluster.impl;

import com.hazelcast.cluster.impl.ClusterClockImpl;
import com.hazelcast.cluster.impl.ClusterServiceImpl;
import com.hazelcast.cluster.impl.operations.HeartbeatOperation;
import com.hazelcast.cluster.impl.operations.MasterConfirmationOperation;
import com.hazelcast.cluster.impl.operations.MemberInfoUpdateOperation;
import com.hazelcast.instance.GroupProperties;
import com.hazelcast.instance.GroupProperty;
import com.hazelcast.instance.MemberImpl;
import com.hazelcast.instance.Node;
import com.hazelcast.instance.NodeState;
import com.hazelcast.internal.metrics.Probe;
import com.hazelcast.logging.ILogger;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.Connection;
import com.hazelcast.spi.impl.NodeEngineImpl;
import com.hazelcast.spi.impl.executionservice.InternalExecutionService;
import com.hazelcast.util.Clock;
import com.hazelcast.util.EmptyStatement;
import java.net.ConnectException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

public class ClusterHeartbeatManager {
    private static final long CLOCK_JUMP_THRESHOLD = 10000L;
    private static final int HEART_BEAT_INTERVAL_FACTOR = 10;
    private static final int MAX_PING_RETRY_COUNT = 5;
    private final ILogger logger;
    private final Node node;
    private final NodeEngineImpl nodeEngine;
    private final ClusterServiceImpl clusterService;
    private final ClusterClockImpl clusterClock;
    private final ConcurrentMap<MemberImpl, Long> heartbeatTimes = new ConcurrentHashMap<MemberImpl, Long>();
    private final ConcurrentMap<MemberImpl, Long> masterConfirmationTimes = new ConcurrentHashMap<MemberImpl, Long>();
    private final long maxNoHeartbeatMillis;
    private final long maxNoMasterConfirmationMillis;
    private final long heartbeatIntervalMillis;
    private final long pingIntervalMillis;
    private final boolean icmpEnabled;
    private final int icmpTtl;
    private final int icmpTimeoutMillis;
    @Probe(name="lastHeartBeat")
    private volatile long lastHeartBeat;
    private volatile long lastClusterTimeDiff;

    public ClusterHeartbeatManager(Node node, ClusterServiceImpl clusterService) {
        this.node = node;
        this.clusterService = clusterService;
        this.nodeEngine = node.getNodeEngine();
        this.clusterClock = clusterService.getClusterClock();
        this.logger = node.getLogger(this.getClass());
        this.maxNoHeartbeatMillis = node.groupProperties.getMillis(GroupProperty.MAX_NO_HEARTBEAT_SECONDS);
        this.maxNoMasterConfirmationMillis = node.groupProperties.getMillis(GroupProperty.MAX_NO_MASTER_CONFIRMATION_SECONDS);
        this.heartbeatIntervalMillis = ClusterHeartbeatManager.getHeartBeatInterval(node.groupProperties);
        this.pingIntervalMillis = this.heartbeatIntervalMillis * 10L;
        this.icmpEnabled = node.groupProperties.getBoolean(GroupProperty.ICMP_ENABLED);
        this.icmpTtl = node.groupProperties.getInteger(GroupProperty.ICMP_TTL);
        this.icmpTimeoutMillis = (int)node.groupProperties.getMillis(GroupProperty.ICMP_TIMEOUT);
    }

    private static long getHeartBeatInterval(GroupProperties groupProperties) {
        long heartbeatInterval = groupProperties.getMillis(GroupProperty.HEARTBEAT_INTERVAL_SECONDS);
        return heartbeatInterval > 0L ? heartbeatInterval : TimeUnit.SECONDS.toMillis(1L);
    }

    void init() {
        InternalExecutionService executionService = this.nodeEngine.getExecutionService();
        executionService.scheduleWithFixedDelay("hz:cluster", new Runnable(){

            @Override
            public void run() {
                ClusterHeartbeatManager.this.heartBeat();
            }
        }, this.heartbeatIntervalMillis, this.heartbeatIntervalMillis, TimeUnit.MILLISECONDS);
        long masterConfirmationInterval = this.node.groupProperties.getSeconds(GroupProperty.MASTER_CONFIRMATION_INTERVAL_SECONDS);
        masterConfirmationInterval = masterConfirmationInterval > 0L ? masterConfirmationInterval : 1L;
        executionService.scheduleWithFixedDelay("hz:cluster", new Runnable(){

            @Override
            public void run() {
                ClusterHeartbeatManager.this.sendMasterConfirmation();
            }
        }, masterConfirmationInterval, masterConfirmationInterval, TimeUnit.SECONDS);
        long memberListPublishInterval = this.node.groupProperties.getSeconds(GroupProperty.MEMBER_LIST_PUBLISH_INTERVAL_SECONDS);
        memberListPublishInterval = memberListPublishInterval > 0L ? memberListPublishInterval : 1L;
        executionService.scheduleWithFixedDelay("hz:cluster", new Runnable(){

            @Override
            public void run() {
                ClusterHeartbeatManager.this.sendMemberListToOthers();
            }
        }, memberListPublishInterval, memberListPublishInterval, TimeUnit.SECONDS);
    }

    public void onHeartbeat(MemberImpl member, long timestamp) {
        if (member != null) {
            long clusterTime = this.clusterClock.getClusterTime();
            if (this.logger.isFineEnabled()) {
                this.logger.fine(String.format("Received heartbeat from %s (now: %s, timestamp: %s)", member, new Date(clusterTime), new Date(timestamp)));
            }
            if (clusterTime - timestamp > this.maxNoHeartbeatMillis / 2L) {
                this.logger.warning(String.format("Ignoring heartbeat from %s since it is expired (now: %s, timestamp: %s)", member, new Date(clusterTime), new Date(timestamp)));
                return;
            }
            if (this.isMaster(member)) {
                this.clusterClock.setMasterTime(timestamp);
            }
            this.heartbeatTimes.put(member, this.clusterClock.getClusterTime());
        }
    }

    public void acceptMasterConfirmation(MemberImpl member, long timestamp) {
        if (member != null) {
            long clusterTime;
            if (this.logger.isFinestEnabled()) {
                this.logger.finest("MasterConfirmation has been received from " + member);
            }
            if ((clusterTime = this.clusterClock.getClusterTime()) - timestamp > this.maxNoMasterConfirmationMillis / 2L) {
                this.logger.warning(String.format("Ignoring master confirmation from %s, since it is expired (now: %s, timestamp: %s)", member, new Date(clusterTime), new Date(timestamp)));
                return;
            }
            this.masterConfirmationTimes.put(member, clusterTime);
        }
    }

    void heartBeat() {
        if (!this.node.joined()) {
            return;
        }
        this.checkClockDrift(this.heartbeatIntervalMillis);
        long clusterTime = this.clusterClock.getClusterTime();
        if (this.node.isMaster()) {
            this.heartBeatWhenMaster(clusterTime);
        } else {
            this.heartBeatWhenSlave(clusterTime);
        }
    }

    private void checkClockDrift(long intervalMillis) {
        long now = Clock.currentTimeMillis();
        if (this.lastHeartBeat != 0L) {
            long clockJump = now - this.lastHeartBeat - intervalMillis;
            long absoluteClockJump = Math.abs(clockJump);
            if (absoluteClockJump > 10000L) {
                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
                this.logger.info(String.format("System clock apparently jumped from %s to %s since last heartbeat (%+d ms)", sdf.format(new Date(this.lastHeartBeat)), sdf.format(new Date(now)), clockJump));
                long currentClusterTimeDiff = this.clusterClock.getClusterTimeDiff();
                if (Math.abs(this.lastClusterTimeDiff - currentClusterTimeDiff) < 10000L) {
                    this.clusterClock.setClusterTimeDiff(currentClusterTimeDiff - clockJump);
                }
            }
            if (absoluteClockJump >= this.maxNoMasterConfirmationMillis / 2L) {
                this.logger.warning(String.format("Resetting master confirmation timestamps because of huge system clock jump! Clock-Jump: %d ms, Master-Confirmation-Timeout: %d ms", clockJump, this.maxNoMasterConfirmationMillis));
                this.resetMemberMasterConfirmations();
            }
            if (absoluteClockJump >= this.maxNoHeartbeatMillis / 2L) {
                this.logger.warning(String.format("Resetting heartbeat timestamps because of huge system clock jump! Clock-Jump: %d ms, Heartbeat-Timeout: %d ms", clockJump, this.maxNoHeartbeatMillis));
                this.resetHeartbeats();
            }
        }
        this.lastClusterTimeDiff = this.clusterClock.getClusterTimeDiff();
        this.lastHeartBeat = now;
    }

    private void heartBeatWhenMaster(long now) {
        Collection<MemberImpl> members = this.clusterService.getMemberImpls();
        for (MemberImpl member : members) {
            if (member.localMember()) continue;
            try {
                this.logIfConnectionToEndpointIsMissing(now, member);
                if (this.removeMemberIfNotHeartBeating(now, member) || this.removeMemberIfMasterConfirmationExpired(now, member)) continue;
                this.pingMemberIfRequired(now, member);
                this.sendHeartbeat(member.getAddress());
            }
            catch (Throwable e) {
                this.logger.severe(e);
            }
        }
    }

    private boolean removeMemberIfNotHeartBeating(long now, MemberImpl member) {
        long heartbeatTime = this.getHeartbeatTime(member);
        if (now - heartbeatTime > this.maxNoHeartbeatMillis) {
            this.logger.warning(String.format("Removing %s because it has not sent any heartbeats for %d ms. Now: %s, last heartbeat time was %s", member, this.maxNoHeartbeatMillis, new Date(now), new Date(heartbeatTime)));
            this.clusterService.removeAddress(member.getAddress());
            return true;
        }
        if (this.logger.isFinestEnabled() && now - heartbeatTime > this.heartbeatIntervalMillis * 10L) {
            this.logger.finest(String.format("Not receiving any heartbeats from %s since %s", member, new Date(heartbeatTime)));
        }
        return false;
    }

    private boolean removeMemberIfMasterConfirmationExpired(long now, MemberImpl member) {
        Long lastConfirmation = (Long)this.masterConfirmationTimes.get(member);
        if (lastConfirmation == null) {
            lastConfirmation = 0L;
        }
        if (now - lastConfirmation > this.maxNoMasterConfirmationMillis) {
            this.logger.warning(String.format("Removing %s because it has not sent any master confirmation for %d ms.  Last confirmation time was %s", member, this.maxNoMasterConfirmationMillis, new Date(lastConfirmation)));
            this.clusterService.removeAddress(member.getAddress());
            return true;
        }
        return false;
    }

    private void heartBeatWhenSlave(long now) {
        Collection<MemberImpl> members = this.clusterService.getMemberImpls();
        for (MemberImpl member : members) {
            if (member.localMember()) continue;
            try {
                this.logIfConnectionToEndpointIsMissing(now, member);
                if (this.isMaster(member) && this.removeMemberIfNotHeartBeating(now, member)) continue;
                this.pingMemberIfRequired(now, member);
                this.sendHeartbeat(member.getAddress());
            }
            catch (Throwable e) {
                this.logger.severe(e);
            }
        }
    }

    private boolean isMaster(MemberImpl member) {
        return member.getAddress().equals(this.node.getMasterAddress());
    }

    private void pingMemberIfRequired(long now, MemberImpl member) {
        if (!this.icmpEnabled) {
            return;
        }
        if (now - this.getHeartbeatTime(member) >= this.pingIntervalMillis) {
            this.ping(member);
        }
    }

    private void ping(final MemberImpl memberImpl) {
        this.nodeEngine.getExecutionService().execute("hz:system", new Runnable(){

            @Override
            public void run() {
                try {
                    Address address = memberImpl.getAddress();
                    ClusterHeartbeatManager.this.logger.warning(String.format("%s will ping %s", ClusterHeartbeatManager.this.node.getThisAddress(), address));
                    for (int i = 0; i < 5; ++i) {
                        try {
                            if (!address.getInetAddress().isReachable(null, ClusterHeartbeatManager.this.icmpTtl, ClusterHeartbeatManager.this.icmpTimeoutMillis)) continue;
                            ClusterHeartbeatManager.this.logger.info(String.format("%s pinged %s successfully", ClusterHeartbeatManager.this.node.getThisAddress(), address));
                            return;
                        }
                        catch (ConnectException ignored) {
                            EmptyStatement.ignore(ignored);
                        }
                    }
                    ClusterHeartbeatManager.this.logger.warning(String.format("%s could not ping %s", ClusterHeartbeatManager.this.node.getThisAddress(), address));
                    ClusterHeartbeatManager.this.clusterService.removeAddress(address);
                }
                catch (Throwable ignored) {
                    EmptyStatement.ignore(ignored);
                }
            }
        });
    }

    private void sendHeartbeat(Address target) {
        block3: {
            if (target == null) {
                return;
            }
            try {
                this.node.nodeEngine.getOperationService().send(new HeartbeatOperation(this.clusterClock.getClusterTime()), target);
            }
            catch (Exception e) {
                if (!this.logger.isFinestEnabled()) break block3;
                this.logger.finest(String.format("Error while sending heartbeat -> %s[%s]", e.getClass().getName(), e.getMessage()));
            }
        }
    }

    private void logIfConnectionToEndpointIsMissing(long now, MemberImpl member) {
        Connection conn;
        long heartbeatTime = this.getHeartbeatTime(member);
        if (!(now - heartbeatTime < this.pingIntervalMillis || (conn = this.node.connectionManager.getOrConnect(member.getAddress())) != null && conn.isAlive())) {
            this.logger.warning("This node does not have a connection to " + member);
        }
    }

    private long getHeartbeatTime(MemberImpl member) {
        Long heartbeatTime = (Long)this.heartbeatTimes.get(member);
        return heartbeatTime != null ? heartbeatTime : 0L;
    }

    public void sendMasterConfirmation() {
        if (!this.node.joined() || this.node.getState() == NodeState.SHUT_DOWN || this.node.isMaster()) {
            return;
        }
        Address masterAddress = this.node.getMasterAddress();
        if (masterAddress == null) {
            this.logger.finest("Could not send MasterConfirmation, masterAddress is null!");
            return;
        }
        MemberImpl masterMember = this.clusterService.getMember(masterAddress);
        if (masterMember == null) {
            this.logger.finest("Could not send MasterConfirmation, masterMember is null!");
            return;
        }
        if (this.logger.isFinestEnabled()) {
            this.logger.finest("Sending MasterConfirmation to " + masterMember);
        }
        this.nodeEngine.getOperationService().send(new MasterConfirmationOperation(this.clusterClock.getClusterTime()), masterAddress);
    }

    private void sendMemberListToOthers() {
        if (!this.node.isMaster()) {
            return;
        }
        Collection<MemberImpl> members = this.clusterService.getMemberImpls();
        MemberInfoUpdateOperation op = new MemberInfoUpdateOperation(ClusterServiceImpl.createMemberInfoList(members), this.clusterClock.getClusterTime(), false);
        for (MemberImpl member : members) {
            if (member.localMember()) continue;
            this.nodeEngine.getOperationService().send(op, member.getAddress());
        }
    }

    void resetMemberMasterConfirmations() {
        long now = this.clusterClock.getClusterTime();
        for (MemberImpl member : this.clusterService.getMemberImpls()) {
            this.masterConfirmationTimes.put(member, now);
        }
    }

    private void resetHeartbeats() {
        long now = this.clusterClock.getClusterTime();
        for (MemberImpl member : this.clusterService.getMemberImpls()) {
            this.heartbeatTimes.put(member, now);
        }
    }

    void removeMember(MemberImpl member) {
        this.masterConfirmationTimes.remove(member);
        this.heartbeatTimes.remove(member);
    }

    void reset() {
        this.masterConfirmationTimes.clear();
        this.heartbeatTimes.clear();
    }
}

