// Copyright (c) 2007-2010 Satoshi Nakamoto
// Copyright (c) 2009-2015 The Bitcoin Core developers
// Copyright (c) 2011-2017 The Litecoin Core developers
// Copyright (c) 2013-2025 The Goldcoin Core developers
// Distributed under the MIT software license, see the accompanying
// file COPYING or http://www.opensource.org/licenses/mit-license.php.


#ifndef BITCOIN_VALIDATIONINTERFACE_H
#define BITCOIN_VALIDATIONINTERFACE_H

#include <memory>
#include <functional>
#include <vector>
#include <algorithm>

// C++23 Signal/Slot replacement for boost::signals2
template<typename Signature>
class Signal;

// Specialization for non-void return types
template<typename R, typename... Args>
class Signal<R(Args...)> {
public:
    using SlotType = std::function<R(Args...)>;
    using FunctionPtr = R(*)(Args...);
    
    void connect(SlotType slot) {
        slots_.push_back(std::move(slot));
    }
    
    void connect(FunctionPtr fn) {
        function_ptrs_.push_back(fn);
    }
    
    void disconnect(FunctionPtr fn) {
        function_ptrs_.erase(
            std::remove(function_ptrs_.begin(), function_ptrs_.end(), fn),
            function_ptrs_.end()
        );
    }
    
    void disconnect_all_slots() {
        slots_.clear();
        function_ptrs_.clear();
    }
    
    R operator()(Args... args) {
        R result{};
        for (auto& slot : slots_) {
            if (slot) result = slot(args...);
        }
        for (auto fn : function_ptrs_) {
            if (fn) result = fn(args...);
        }
        return result;
    }
    
private:
    std::vector<SlotType> slots_;
    std::vector<FunctionPtr> function_ptrs_;
};

// Specialization for void return type
template<typename... Args>
class Signal<void(Args...)> {
public:
    using SlotType = std::function<void(Args...)>;
    using FunctionPtr = void(*)(Args...);
    
    void connect(SlotType slot) {
        slots_.push_back(std::move(slot));
    }
    
    void connect(FunctionPtr fn) {
        function_ptrs_.push_back(fn);
    }
    
    void disconnect(FunctionPtr fn) {
        function_ptrs_.erase(
            std::remove(function_ptrs_.begin(), function_ptrs_.end(), fn),
            function_ptrs_.end()
        );
    }
    
    void disconnect_all_slots() {
        slots_.clear();
        function_ptrs_.clear();
    }
    
    void operator()(Args... args) {
        for (auto& slot : slots_) {
            if (slot) slot(args...);
        }
        for (auto fn : function_ptrs_) {
            if (fn) fn(args...);
        }
    }
    
private:
    std::vector<SlotType> slots_;
    std::vector<FunctionPtr> function_ptrs_;
};

class CBlock;
class CBlockIndex;
struct CBlockLocator;
class CBlockIndex;
class CConnman;
class CReserveScript;
class CTransaction;
class CValidationInterface;
class CValidationState;
class uint256;

// These functions dispatch to one or all registered wallets

/** Register a wallet to receive updates from core */
void RegisterValidationInterface(CValidationInterface* pwalletIn);
/** Unregister a wallet from core */
void UnregisterValidationInterface(CValidationInterface* pwalletIn);
/** Unregister all wallets from core */
void UnregisterAllValidationInterfaces();

class CValidationInterface {
protected:
    virtual void UpdatedBlockTip(const CBlockIndex *pindexNew, const CBlockIndex *pindexFork, bool fInitialDownload) {}
    virtual void SyncTransaction(const CTransaction &tx, const CBlockIndex *pindex, int posInBlock) {}
    virtual void SetBestChain(const CBlockLocator &locator) {}
    virtual void UpdatedTransaction(const uint256 &hash) {}
    virtual void Inventory(const uint256 &hash) {}
    virtual void ResendWalletTransactions(int64_t nBestBlockTime, CConnman* connman) {}
    virtual void BlockChecked(const CBlock&, const CValidationState&) {}
    virtual void GetScriptForMining(std::shared_ptr<CReserveScript>&) {};
    virtual void ResetRequestCount(const uint256 &hash) {};
    virtual void NewPoWValidBlock(const CBlockIndex *pindex, const std::shared_ptr<const CBlock>& block) {};
    friend void ::RegisterValidationInterface(CValidationInterface*);
    friend void ::UnregisterValidationInterface(CValidationInterface*);
    friend void ::UnregisterAllValidationInterfaces();
};

struct CMainSignals {
    /** Notifies listeners of updated block chain tip */
    Signal<void (const CBlockIndex *, const CBlockIndex *, bool)> UpdatedBlockTip;
    /** A posInBlock value for SyncTransaction calls for tranactions not
     * included in connected blocks such as transactions removed from mempool,
     * accepted to mempool or appearing in disconnected blocks.*/
    static const int SYNC_TRANSACTION_NOT_IN_BLOCK = -1;
    /** Notifies listeners of updated transaction data (transaction, and
     * optionally the block it is found in). Called with block data when
     * transaction is included in a connected block, and without block data when
     * transaction was accepted to mempool, removed from mempool (only when
     * removal was due to conflict from connected block), or appeared in a
     * disconnected block.*/
    Signal<void (const CTransaction &, const CBlockIndex *, int)> SyncTransaction;
    /** Notifies listeners of an updated transaction without new data (for now: a coinbase potentially becoming visible). */
    Signal<void (const uint256 &)> UpdatedTransaction;
    /** Notifies listeners of a new active block chain. */
    Signal<void (const CBlockLocator &)> SetBestChain;
    /** Notifies listeners about an inventory item being seen on the network. */
    Signal<void (const uint256 &)> Inventory;
    /** Tells listeners to broadcast their data. */
    Signal<void (int64_t, CConnman*)> Broadcast;
    /** Notifies listeners of a block validation result */
    Signal<void (const CBlock&, const CValidationState&)> BlockChecked;
    /** Notifies listeners that a key for mining is required (coinbase) */
    Signal<void (std::shared_ptr<CReserveScript>&)> ScriptForMining;
    /** Notifies listeners that a block has been successfully mined */
    Signal<void (const uint256 &)> BlockFound;
    /**
     * Notifies listeners that a block which builds directly on our current tip
     * has been received and connected to the headers tree, though not validated yet */
    Signal<void (const CBlockIndex *, const std::shared_ptr<const CBlock>&)> NewPoWValidBlock;
};

CMainSignals& GetMainSignals();

#endif // BITCOIN_VALIDATIONINTERFACE_H
