# About Rexl

* [Heritage](#heritage)
* [Concepts](#concepts)

**Rexl**, also written **REXL** or **RExL**, stands for **_Research EXpression Language_**.
Rexl is designed to be embedded in host applications. This document covers the language
Rexl and how to use it. To learn about features of a particular host, refer to its
documentation.

Rexl is still evolving, so features or functionality that are not included in this document
may be added in the future via new types, operators, syntax, and functions.

## Heritage

Rexl shares heritage with [**_Power Fx_**](https://learn.microsoft.com/en-us/power-platform/power-fx/overview),
the language of Microsoft Power Apps. These languages are both derived from a language
named **_TEXL_**, the **_Tangram Expression Language_**, developed around 2010 as part of
an incubation project at Microsoft. These languages draw some degree of inspiration from the
Excel language,  which is perhaps the most widely used (computational) language on the planet.

Similarities between Rexl and the Excel language include:
* The language is used to write formulas, also known as expressions.
* A formula is evaluated to produce a value.
* A value, whether imported or produced by evaluating formulas, is often associated with a name.
  The name associated with the value is not defined in the language itself. In Excel, the name is
  the address of a cell, such as C7, that contains the value. In a Rexl host, there may be a
  variety of named entities, some with values imported from external data sources and others with
  values determined by evaluating some combination of formulas. For example, the value of a named
  node in a flow graph may come from a Parquet file or SQL table, or may be computed from
  a Rexl formula that the user typed or that was generated by a Copilot.
* A named value (whether generated by formulas or not) can be referenced in formulas via its name.
* Each value has a type, for example, number, text, or logical.
* Formulas do not perform actions, such as changing the contents of a file or sending an email message. 
  The formula produces a value and has no side effects.
* The value produced by a formula may be formatted in several ways. The formatting of values is not
  part of the language but depends on user actions in the containing application. For example, in Excel
  you can display a value as a percentage by selecting the cell and pressing the appropriate button on
  the ribbon. Formatting the value does not change the value, but only changes how the value is
  displayed for the user.

Differences between Rexl and the Excel language include:
* Rexl supports many types of values that Excel does not. Excel has one numeric type while Rexl
  supports many numeric types. More significantly, Rexl supports structured types, such as record,
  tuple, tensor, and sequence types.
* The Excel language knows that value names (cell names) are spatially related, so supports cell ranges
  such as `C7:D12`. In Rexl, there is no concept of an underlying grid. A Rexl name is either an **_identifier_**
  or a **_dotted name_** consisting of multiple identifiers separated by the `.` (dot) token.
* Rexl supports the concept of **_local name_**, which is introduced in one part of a formula and then used
  in other parts of the formula. For example, in `With(x: 3 * 2, x + x^2)`, the name `x` is a local name 
  introduced in the first argument of the `With` function and used in the second argument.

## Concepts

* [Formulas](#formulas)
* [Names](#names)
* [Evaluation and Errors](#evaluation-and-errors)

### Formulas

The Rexl language is used to write **_formulas_**, also known as **_expressions_**.

Each of the following lines is a formula:
```
3 + 5

SinD(45) * 2

Employees->{ Name: LastName & ", " & FirstName, Age }

Employees->TakeIf(Age >= 55)
```
The third and fourth formulas reference a **_global named value_** (**_global_** for short), named `Employees`.

The second and fourth formulas use **_functions_**. The second uses the function `SinD`, which computes the sine of
an angle expressed in degrees. The fourth uses the function `TakeIf` to filter the **_sequence_** of `Employees` to
keep only the items that satisfy the **_predicate_** expression `Age >= 55`. Note that this does not change the value
of `Employees`, which is impossible to do in Rexl. Instead, it produces a value that is computed from `Employees`.

All four formulas use **_operators_**. The first uses the addition operator, `+`. The second uses the multiplication
operator, `*`. The third uses the text concatenation operator, `&`, as well as the record projection operator, `->`.
The fourth uses the comparison operator, `>=`, and the function projection operator `->`.

A formula has an associated **_type_** that is automatically inferred from the specific form of the formula as well as 
from the types of any **_globals_** that the formula references.

A formula may contain **_errors_**, meaning that it is invalid for one reason or another. For example, each of the
following formulas has an error:
```
3 + "hello"

SinD("45") * 2

Employees->{ Name: LastName ++ ", " & FirstName, Age }

Employees->Takeif(Age >= 55)
```

In the first formula, the addition operator can't be applied to a text value such as `"hello"`.

In the second formula, the `SinD` function does not accept a text value such as `"45"`.

In the third formula, the sequence concatenation operator, `++`, is incorrectly applied to text values.

In the fourth formula, there is no function named `Takeif`, though there is a function named `TakeIf`.

When a formula contains no errors, the formula is **_evaluated_** to produce a **_value_** of the **_type_**
of the formula. It is important to keep in mind that the term **_value_** can mean a compound piece of information,
such as an entire **_table_**, represented as a **_sequence of records_**. See [Types and Values](02-TypesAndValues.md)
for a complete description of types and values.

### Names

In Rexl, a **_simple name_** or **_identifier_** is a sequence of Unicode characters that contains at least one
non-whitespace character and contains no line terminator characters.

A simple name may be **_quoted_**, where it is placed between single quotation characters. Generally, if the simple
name starts with a letter or underscore character `_` and contains only letters, digits, and underscore 
characters, then the name does not need to be quoted, but it still can be. Otherwise, the name must be 
quoted. For example, the following specify the same simple name:
```
MyValue_3_Temp

'MyValue_3_Temp'
```

The following identifiers must be quoted:
```
' MyValue_3_Temp'

'My Value'

'Beth''sValue'

'123X'
```

The corresponding names (without quotes) are the character sequences:
```
 MyValue_3_Temp

My Value

Beth'sValue

123X
```

Note that the surrounding quotes used in the Rexl formula are not part of the name itself.

The reasons these names require quoting vary.
* The first two lines each contain a space, which is not a letter, digit, or underscore.
* The third line contains a single quotation character, which is also not a letter, digit, or underscore.
  Note that this quotation character needs to be doubled to distinguish it from a terminating quotation 
  character. The resulting name contains only one quotation character.
* The last does not start with a letter or underscore.

A **_field name_** in a **_record_** is a simple name. A **_local name_** used in a function invocation is also a simple name.

A **_dotted name_** consists of one or more simple names, separated by the dot token. By this definition, a simple 
name is also a dotted name.

Both **_functions_** and **_global named values_** are identified by dotted names. For example, `Text.Len` is a function
that produces the number of characters in a text value. The portion of a dotted name consisting of all but the last name
is called the namespace of the dotted name. When a dotted name consists of a single simple name, it is said to be in the
global namespace.

In certain Rexl language constructs a simple name may be **_implicit_**. Consider the formula:
```
ForEach(Item:MyTable, { Item:Item, Name:Name, Age:Item.Age, Addr:Item.HomeAddr })
```

This uses an existing sequence, `MyTable`, to produce a new sequence of records. The resulting records have 
fields named `Item`, `Name`, `Age`, and `Addr`.

In the field specifications `Item:Item` and `Name:Name`, the value expression consists of a simple name matching 
the corresponding field name. In this situation, the field name and colon may be omitted, shortening the 
above to:
```
ForEach(Item:MyTable, { Item, Name, Age:Item.Age, Addr:Item.HomeAddr })
```

Similarly, in the field specification `Age:Item.Age`, the value expression is a dotted expression with the final
simple name matching the field name. In this situation, the field name and colon may also be omitted, shortening
this further to:
```
ForEach(Item:MyTable, { Item, Name, Item.Age, Addr:Item.HomeAddr })
```

In this case, the field names `Item`, `Name`, and `Age` are implicit.

### Evaluation and Errors

**_Evaluation_** of an expression consists of two phases, namely **_compilation_** and **_execution_**.

Before an expression can be executed, it is **_compiled_**. Compilation can generate errors. For example, each of
the following formulas contains a compilation error:
```
"abc

a + * b

"hello" + 1
```

The second phase of evaluation is **_execution_**. An expression that has compilation errors is not executed, so 
cannot be fully evaluated to produce a value.

Errors generated by execution are known as **_runtime errors_** or **_execution errors_**. A runtime error is when an
expression that successfully compiles fails to (fully) execute and does not produce a (complete) value. Note
that items in a sequence value may be evaluated lazily as needed, so when there is a runtime error, it may 
come long after the initial execution of an expression.

Generally, Rexl is designed with the ideal that there should be no runtime errors. That is, when an expression
compiles successfully, it should execute without generating runtime errors. Colloquially, "if it compiles, it
runs." This is an ideal and not a rule since there can be extreme circumstances where something goes wrong,
for example, running out of memory.

Throughout this document, the term **_error_** generally means a compilation error, not a **_runtime error_**.

[_Back to Contents_](../RexlUserGuide.md)

[_Next section - Types and Values_](02-TypesAndValues.md)
