﻿// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.

using System;
using System.Collections.Generic;
using System.Linq;

using Microsoft.Rexl.Lex;
using Microsoft.Rexl.Parse;
using Microsoft.Rexl.Private;
using Microsoft.Rexl.Utility;

namespace Microsoft.Rexl.Bind;

using ArgTuple = Immutable.Array<BoundNode>;
using Conditional = System.Diagnostics.ConditionalAttribute;
using Integer = System.Numerics.BigInteger;
using NameTuple = Immutable.Array<DName>;
using NodeTuple = Immutable.Array<ExprNode>;
using ScopeTuple = Immutable.Array<ArgScope>;

/// <summary>
/// Options for binding. The defaults are represented by zero/default.
/// </summary>
[Flags]
public enum BindOptions
{
    Default = 0x00,

    /// <summary>
    /// When this is off, volatile functions are still found, but uses of them
    /// are an error. Note that a function isn't "volatile", but particular invocations
    /// are so a function can have both volatile and non-volatile invocations.
    /// </summary>
    AllowVolatile = 0x01,

    /// <summary>
    /// Whether a top level procedure invocation is allowed (not an error). It is also
    /// an error if a procedure invocation is not top level.
    /// REVIEW: Should we support <c>With(..., Proc(...))</c>?
    /// </summary>
    AllowProc = 0x02,

    /// <summary>
    /// When reduce is turned off, optimize is as well, even though that fact
    /// is not represented in the flag values.
    /// </summary>
    DontReduce = 0x04,

    /// <summary>
    /// Whether optimization should be inhibited (after reduction).
    /// </summary>
    DontOptimize = 0x08,

    /// <summary>
    /// Whether to prohibit all module usage.
    /// </summary>
    ProhibitModule = 0x10,

    /// <summary>
    /// Whether the general type should be freely allowed (not discouraged).
    /// </summary>
    AllowGeneral = 0x20,

    /// <summary>
    /// This one allows both volatile and proc.
    /// </summary>
    AllowImpure = AllowVolatile | AllowProc,
}

partial class BoundFormula
{
    /// <summary>
    /// The Binder does the following:
    /// * Records semantic errors and warnings.
    /// * Translates from parse nodes to semantic "bound" nodes.
    /// * Fully type annotates the bound nodes.
    /// * Makes sequence operations and operator lifting explicit.
    /// * Builds a mapping from parse node to type and scope information.
    /// * Records all usage of globals.
    /// * Optionally invokes the reducer and records any warnings generated by it.
    ///
    /// The binding is done in the context of:
    /// * A bind host that provides global type info, namespace info, function info, etc.
    /// * A current namespace.
    /// * The type of "this" or invalid if there isn't a "this".
    /// * Optional outer scopes.
    ///
    /// General notes about opt-ness:
    /// * Items in a BndTupleNode and Args in a BndCallNode should have exactly the types expected,
    ///   including opt-ness.
    /// * Args in BndBinaryOpNode, BndVariadicOpNode, and BndCompareNode may have some args being non-opt
    ///   while others are opt (if the operator directly supports opt-ness). This makes it easier for
    ///   code-gen to be efficient.
    /// </summary>
    private sealed partial class Binder : RexlTreeVisitor, IReducerHost
    {
        /// <summary>
        /// Persistent stack of box value and use count.
        /// </summary>
        private sealed class BoxValueStack
        {
            public BoundNode Value { get; }
            public int Uses { get; }
            public BoxValueStack Rest { get; }
            public int Depth { get; }

            public static readonly BoxValueStack Empty = new BoxValueStack();

            /// <summary>
            /// This ctor creates the last item. This avoids having to support null links.
            /// </summary>
            private BoxValueStack()
            {
            }

            /// <summary>
            /// This ctor adds a new item. This is for "push".
            /// </summary>
            private BoxValueStack(BoundNode value, int uses, BoxValueStack rest)
            {
                Validation.AssertValue(value);
                Validation.Assert(uses >= 0);
                Validation.AssertValue(rest);

                Value = value;
                Uses = uses;
                Rest = rest;
                Depth = rest != null ? rest.Depth + 1 : 1;
            }

            public BoxValueStack Push(BoundNode value)
            {
                return new BoxValueStack(value, 0, this);
            }

            public BoxValueStack Use()
            {
                Validation.Assert(Depth > 0);
                return new BoxValueStack(Value, Uses + 1, Rest);
            }
        }

        /// <summary>
        /// Persistent map from bound node to parse node.
        /// </summary>
        private sealed class BndToParse : BoundNodeMapping<BndToParse, BoundNode, RexlNode>
        {
            public static readonly BndToParse Empty = new BndToParse(null);

            private BndToParse(Node root)
                : base(root)
            {
            }

            protected override bool ValIsValid(RexlNode val)
            {
                return val != null;
            }

            protected override bool ValEquals(RexlNode val0, RexlNode val1)
            {
                return val0 == val1;
            }

            protected override int ValHash(RexlNode val)
            {
                return val.GetHashCode();
            }

            protected override BndToParse Wrap(Node root)
            {
                return root == _root ? this : root == null ? Empty : new BndToParse(root);
            }
        }

        /// <summary>
        /// This represents a "scope". When binding a module, the <see cref="ModuleBuilder"/> will be wrapped
        /// rather than an <see cref="ArgScope"/>.
        /// </summary>
        private sealed class ScopeWrapper
        {
            // Except at the root/top level (where both are null), exactly one of these will be non-null.
            private readonly ArgScope _scope;
            private readonly ModuleBuilder _module;

            // The ScopeInfo chain.
            public readonly ScopeInfo Info;

            /// <summary>
            /// This is null for the root scope and module scopes. Non-null for normal arg scopes.
            /// </summary>
            public ArgScope Scope => _scope;

            /// <summary>
            /// Non-null for module scopes.
            /// </summary>
            public ModuleBuilder Module => _module;

            // The associated scope for indexing.
            public readonly ArgScope IndexScope;

            // The next outer scope wrapper.
            public readonly ScopeWrapper Outer;

            // This is the nesting level, useful for debugging and assertions.
            public readonly int Nest;

            // REVIEW: Switch this to be of type DName.
            public readonly string Variable;
            public readonly ExprNode SrcImplicitName;

            public bool VarIsImplicit => SrcImplicitName != null;

            /// <summary>
            /// A scope can selectively be "disabled" for some slots.
            /// </summary>
            public bool Enabled;

            /// <summary>
            /// Constructs the root scope.
            /// </summary>
            public static ScopeWrapper CreateRoot()
            {
                return new ScopeWrapper();
            }

            private ScopeWrapper()
            {
            }

            public static ScopeWrapper Create(
                ScopeWrapper outer, ScopeKind kind, DType type, ArgScope index,
                string variable = null, ExprNode srcImplicitName = null)
            {
                Validation.Assert(ScopeKind.None < kind && kind < ScopeKind.SeqIndex);
                Validation.Assert(type.IsValid);
                Validation.Assert(index == null || kind == ScopeKind.SeqItem && index.Kind == ScopeKind.SeqIndex && index.Type == DType.I8Req);

                var scope = ArgScope.Create(kind, type);
                if (kind == ScopeKind.Range)
                    index = scope;
                return new ScopeWrapper(outer, scope, index, variable, srcImplicitName);
            }

            public static ScopeWrapper Create(ScopeWrapper outer, ModuleBuilder module)
            {
                Validation.AssertValue(outer);
                Validation.AssertValue(module);

                return new ScopeWrapper(outer, module);
            }

            public static ScopeWrapper Create(
                ScopeWrapper outer, ArgScope scope,
                string variable = null, ExprNode srcImplicitName = null)
            {
                return new ScopeWrapper(outer, scope, null, variable, srcImplicitName);
            }

            private ScopeWrapper(
                ScopeWrapper outer, ArgScope scope, ArgScope index,
                string variable = null, ExprNode srcImplicitName = null)
            {
                Validation.AssertValue(outer);
                Validation.AssertValue(scope);
#if DEBUG
                switch (scope.Kind)
                {
                case ScopeKind.SeqItem:
                    Validation.Assert(index == null || index.Kind == ScopeKind.SeqIndex);
                    break;
                case ScopeKind.Range:
                    Validation.Assert(index == scope);
                    break;
                default:
                    Validation.Assert(index == null);
                    break;
                }
#endif
                Validation.AssertValueOrNull(variable);
                Validation.Assert(srcImplicitName == null || srcImplicitName is FirstNameNode || srcImplicitName is DottedNameNode);
                Validation.Assert(srcImplicitName == null || variable != null);

                _scope = scope;
                _module = null;
                IndexScope = index;
                Outer = outer;
                Nest = Outer.Nest + 1;
                Variable = variable;
                SrcImplicitName = srcImplicitName;

                Info = Variable != null ?
                    new ScopeInfo(scope.Type, Outer.Info, new DName(Variable), VarIsImplicit) :
                    new ScopeInfo(scope.Type, Outer.Info);
            }

            private ScopeWrapper(ScopeWrapper outer, ModuleBuilder module)
            {
                Validation.AssertValue(outer);
                Validation.AssertValue(module);

                _scope = null;
                _module = module;
                Outer = outer;
                Nest = Outer.Nest + 1;

                // REVIEW: Need to figure out how to update this Info once the module is finished. It will matter for
                // flow graph analysis and intellisense. Note that when this is fixed, both module symbol references
                // and globals used in module construction will need to be considered. The former are represented
                // as slot accesses in the module's items tuple and the latter as slot accesses in the module's
                // "externals" tuple.
                Info = new ScopeInfo(DType.EmptyModuleReq, Outer.Info);
            }
        }

        /// <summary>
        /// Contains state of the Visitor, to support <see cref="Rewind(in CheckPoint)"/>.
        /// </summary>
        private struct CheckPoint
        {
            public NodeToInfo NodeToInfo;
            public CallToOper CallToOper;
            public int DiagCount;
            public bool HasErrors;
            public BoxValueStack BoxValueStack;
            public BndToParse BndToParse;
            public int StackDepth;
        }

        #region Inputs

        private readonly ExprNode _nodeTop;
        private readonly BindHost _host;
        private readonly BindOptions _options;

        // REVIEW: Perhaps remove _typeThis and always ask the bind host?
        private readonly DType _typeThis;

        #endregion Inputs

        #region Outputs

        /// <summary>
        /// Map from streaming global to the first node that referenced it. This is allocated lazily
        /// so not readonly.
        /// </summary>
        private Dictionary<NPath, RexlNode> _streamToNode;

        /// <summary>
        /// This maps from parse node to inferred info about it, including its type and scope-chain information.
        /// This is useful for intellisense features like suggestion generation. Note that some parse nodes, like ExprListNode
        /// and the record node in an augmenting projection node, won't have a valid type, but will still have
        /// an entry in this dictionary, specifying scope information.
        /// </summary>
        private NodeToInfo _nodeToInfo;

        /// <summary>
        /// This maps from call node to ArgTraits, which includes the Func and arity. Useful for
        /// intellisense features like signature help generation.
        /// </summary>
        private CallToOper _callToOper;

        // Generated errors and warnings.
        private List<BaseDiagnostic> _diagnostics;
        // Tracks whether _diagnostics contains any errors. May be useful for debugging, etc.
        private bool _hasErrors;

        #endregion Outputs

        #region Working Values

        // The root (bottom) of the scope stack.
        private readonly ScopeWrapper _scopeRoot;

        // The base of the scope stack - at and below here are only pre-defined scopes.
        private readonly ScopeWrapper _scopeBase;

        // The top of the scope stack.
        private ScopeWrapper _scopeCur;

        private readonly List<BoundNode> _stack;
        private BoxValueStack _boxValueStack;

        /// <summary>
        /// Maps from bound node to parse node. Used primarily for diagnostics. Note that the associated parse
        /// node can change as binding progresses. When a diagnostic is generated, the current mapping is used.
        /// REVIEW: Seems like this could be useful in the <see cref="BoundFormula"/>.
        /// </summary>
        private BndToParse _bndToParse;

        #endregion Working Values

        private Binder(ExprNode node, BindHost host, BindOptions options, DType typeThis,
            Immutable.Array<(ArgScope scope, string name)> scopes)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(host);

            // Inputs.
            _nodeTop = node;
            _host = host;
            _options = options;
            _typeThis = typeThis;

            // Outputs.
            _nodeToInfo = NodeToInfo.Empty;
            _callToOper = CallToOper.Empty;
            _diagnostics = null;

            // Working values.
            _scopeRoot = ScopeWrapper.CreateRoot();
            _scopeCur = _scopeRoot;
            _stack = new List<BoundNode>();
            _boxValueStack = BoxValueStack.Empty;
            _bndToParse = BndToParse.Empty;

            foreach (var (scope, name) in scopes)
            {
                if (scope != null)
                    PushScope(scope, name);
            }

            _scopeBase = _scopeCur;
        }

        /// <summary>
        /// Capture the current state of this visitor, to support rewind.
        /// </summary>
        private void Capture(out CheckPoint cp)
        {
            cp.NodeToInfo = _nodeToInfo;
            cp.CallToOper = _callToOper;
            cp.DiagCount = Util.Size(_diagnostics);
            cp.HasErrors = _hasErrors;
            cp.BoxValueStack = _boxValueStack;
            cp.BndToParse = _bndToParse;
            cp.StackDepth = _stack.Count;
        }

        /// <summary>
        /// Rewind to a previously captured state. Note that this does not support re-jumping forward.
        /// If we need that, we'll need to make the diagnostics be a persistent data structure.
        /// </summary>
        private void Rewind(in CheckPoint cp)
        {
            Validation.AssertValue(cp.NodeToInfo);
            Validation.AssertValue(cp.CallToOper);
            Validation.AssertIndexInclusive(cp.DiagCount, Util.Size(_diagnostics));
            Validation.Assert(!cp.HasErrors || _hasErrors);
            Validation.AssertValue(cp.BoxValueStack);
            Validation.AssertValue(cp.BndToParse);

            // Caller should ensure that the stack is the same depth.
            Validation.Assert(cp.StackDepth == _stack.Count);

            _nodeToInfo = cp.NodeToInfo;
            _callToOper = cp.CallToOper;
            if (cp.DiagCount == 0)
                _diagnostics = null;
            else if (cp.DiagCount < _diagnostics.VerifyValue().Count)
                _diagnostics.RemoveRange(cp.DiagCount, _diagnostics.Count - cp.DiagCount);
            _hasErrors = cp.HasErrors;
            _boxValueStack = cp.BoxValueStack;
            _bndToParse = cp.BndToParse;
        }

        [Conditional("DEBUG")]
        private void AssertValid()
        {
#if DEBUG
            Validation.AssertValue(_scopeRoot);
            Validation.AssertValue(_scopeBase);
            Validation.AssertValue(_scopeCur);

            ScopeWrapper scope = _scopeCur;
            while (scope != null && scope != _scopeBase)
                scope = scope.Outer;
            Validation.Assert(scope == _scopeBase, "_scopeBase should be in the parent chain of _scopeCur!");
            while (scope != null && scope != _scopeRoot)
                scope = scope.Outer;
            Validation.Assert(scope == _scopeRoot, "_scopeRoot should be in the parent chain of _scopeCur!");
#endif
        }

        /// <summary>
        /// Binds a Rexl parse tree (node).
        /// </summary>
        public static BoundNode Run(
            ExprNode node, BindHost host, BindOptions options, DType typeThis,
            Immutable.Array<(ArgScope scope, string name)> scopes,
            out BoundNode boundTreeRaw,
            out Immutable.Array<BaseDiagnostic> diagnostics,
            out NodeToInfo nodeToInfo,
            out CallToOper callNodeToOper)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(host);

            var vis = new Binder(node, host, options, typeThis, scopes);

            Validation.Assert(vis._nodeTop == node);
            Validation.Assert(vis._scopeCur == vis._scopeBase);
            Validation.Assert(vis._nodeToInfo.Count == 0);
            Validation.Assert(vis._diagnostics == null);
            Validation.Assert(vis._boxValueStack.Depth == 0);
            Validation.Assert(vis._stack.Count == 0);

            node.Accept(vis);

            Validation.Assert(vis._scopeCur == vis._scopeBase);
            Validation.Assert(vis._boxValueStack.Depth == 0);
            Validation.Assert(vis._stack.Count == 1);

#if DEBUG
            NodeInfoVerifier.Run(node, vis._nodeToInfo);
#endif

            // Get the bound node and label the scopes.
            var bnd = vis.Pop();
            boundTreeRaw = bnd;
            Validation.Assert(vis._stack.Count == 0);

            // If there are any streaming globals, and we're in a proc, ensure that they are used properly (non-eagerly).
            if (vis._streamToNode != null && bnd.IsProcCall && !vis._hasErrors)
            {
                foreach (var kvp in vis._streamToNode)
                {
                    Validation.Assert(kvp.Value != null);
                    var bad = StreamAnalysis.FindEagerStreamUse(bnd, kvp.Key);
                    if (bad != null)
                    {
                        if (!vis._bndToParse.TryGetValue(bad, out var nodeBad))
                        {
                            // Just use the first occurrence.
                            nodeBad = kvp.Value;
                        }
                        vis.Error(nodeBad, ErrorStrings.ErrBadStreamUse);
                    }
                }
            }

            // REVIEW: If there were any errors should we still reduce? If there are errors, code gen
            // shouldn't happen, so isn't needed from that end. However, reduction may generate some
            // warnings that could be useful to the user. On the other hand, if there are errors, just
            // showing those initially might be sufficiently useful.
            // Always reducing for tests is useful since it encourages the reduction code to be robust and
            // not assume more than is guaranteed by the IR.
            if ((options & BindOptions.DontReduce) == 0)
            {
                // Run the reducer. This may record more warnings.
                bnd = Reducer.Run(vis, bnd);
                if ((options & BindOptions.DontOptimize) == 0)
                {
                    var bnd0 = bnd;
                    bnd = Optimizer.Run(vis, bnd);
#if DEBUG
                    var bnds = new List<BoundNode>() { bnd0 };
#endif
                    const int limit = 10;
                    int i = 0;
                    while (bnd != bnd0 && i++ < limit)
                    {
#if DEBUG
                        Validation.Assert(!bnds.Any(b => b.Equivalent(bnd)));
                        bnds.Add(bnd);
#endif
                        bnd = Reducer.Run(vis, bnd0 = bnd);
                        if (bnd == bnd0)
                            break;
#if DEBUG
                        Validation.Assert(!bnds.Any(b => b.Equivalent(bnd)));
                        bnds.Add(bnd);
#endif
                        bnd = Optimizer.Run(vis, bnd0 = bnd);
                    }
                }
                Validation.Assert(vis._stack.Count == 0);
            }

            // Fill the outputs.
            diagnostics = vis._diagnostics == null ? Immutable.Array<BaseDiagnostic>.Empty : vis._diagnostics.ToImmutableArray();
            nodeToInfo = vis._nodeToInfo;
            callNodeToOper = vis._callToOper;

            Validation.Assert(bnd != null);
            Validation.Assert(!diagnostics.IsDefault);

            return bnd;
        }

        BoundNode IReducerHost.OnMapped(BoundNode bndOld, BoundNode bndNew)
        {
            Validation.BugCheckValue(bndOld, nameof(bndOld));
            Validation.BugCheckValue(bndNew, nameof(bndNew));

            if (bndOld == bndNew)
                return bndNew;

            Validation.BugCheckParam(bndOld.Type == bndNew.Type, nameof(bndNew));

            // REVIEW: What should we do if there is no corresponding parse node?
            if (!_bndToParse.TryGetValue(bndOld, out var node))
                return bndNew;

            Validation.Assert(node != null);
            return Associate(node, bndNew);
        }

        BoundNode IReducerHost.Associate(BoundNode bndOld, BoundNode bndNew)
        {
            Validation.BugCheckValueOrNull(bndOld);
            Validation.BugCheckValue(bndNew, nameof(bndNew));

            if (bndOld == null || bndOld == bndNew)
                return bndNew;

            // REVIEW: What should we do if there is no corresponding parse node?
            if (!_bndToParse.TryGetValue(bndOld, out var node))
                return bndNew;

            Validation.Assert(node != null);
            return Associate(node, bndNew);
        }

        void IReducerHost.Warn(BoundNode bnd, StringId msg)
        {
            if (!_bndToParse.TryGetValue(bnd, out var node))
            {
                // REVIEW: What should we do if there is no corresponding parse node?
                return;
            }

            Validation.Assert(node != null);
            Warning(node, msg);
        }

        /// <summary>
        /// This maintains the mapping from bound node to parse node, for diagnostics. In particular, this
        /// assigns <paramref name="node"/> to <paramref name="bnd"/>.
        /// </summary>
        private TNode Associate<TNode>(RexlNode node, TNode bnd)
            where TNode : BoundNode
        {
            Validation.AssertValue(node);
            Validation.AssertValue(bnd);
            _bndToParse = _bndToParse.SetItem(bnd, node);
            return bnd;
        }

        /// <summary>
        /// This maintains the mapping from bound node to parse node, for diagnostics. In particular, this
        /// looks up the parse node currently assigned to <paramref name="src"/> and assigns it to
        /// <paramref name="bnd"/>. If there is no node currently assigned to <paramref name="src"/>,
        /// this uses <paramref name="node"/> instead.
        /// </summary>
        private TNode Associate<TNode>(BoundNode src, RexlNode node, TNode bnd)
            where TNode : BoundNode
        {
            Validation.AssertValue(src);
            Validation.AssertValue(node);
            Validation.AssertValue(bnd);
            if (!_bndToParse.TryGetValue(src, out var nodeSrc))
                nodeSrc = node;
            _bndToParse = _bndToParse.SetItem(bnd, nodeSrc);
            return bnd;
        }

        private TDiag AddDiagnostic<TDiag>(TDiag diag)
            where TDiag : BaseDiagnostic
        {
            Validation.AssertValue(diag);
            Util.Add(ref _diagnostics, diag);
            _hasErrors |= diag.IsError;
            return diag;
        }

        private RexlDiagnostic Error(RexlNode node, StringId msg)
        {
            Validation.AssertValue(node);
            return AddDiagnostic(RexlDiagnostic.Error(node, msg));
        }

        private RexlDiagnostic Error(Token tok, RexlNode node, StringId msg)
        {
            Validation.AssertValue(tok);
            Validation.AssertValue(node);
            return AddDiagnostic(RexlDiagnostic.Error(tok, node, msg));
        }

        private RexlDiagnostic Error(RexlNode node, StringId msg, params object[] args)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(args);
            return AddDiagnostic(RexlDiagnostic.Error(node, msg, args));
        }

        private RexlDiagnostic ErrorGuess(RexlNode node, StringId msg, string guess, SourceRange rngGuess, params object[] args)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(args);
            Validation.AssertValue(guess);
            return AddDiagnostic(RexlDiagnostic.ErrorGuess(node, msg, guess, rngGuess, args));
        }

        private RexlDiagnostic Error(Token tok, RexlNode node, StringId msg, params object[] args)
        {
            Validation.AssertValue(tok);
            Validation.AssertValue(node);
            Validation.AssertValue(args);
            return AddDiagnostic(RexlDiagnostic.Error(tok, node, msg, args));
        }

        private RexlDiagnostic ErrorGuess(Token tok, RexlNode node, StringId msg, string guess, SourceRange rngGuess, params object[] args)
        {
            Validation.AssertValue(tok);
            Validation.AssertValue(node);
            Validation.AssertValue(args);
            Validation.AssertValue(guess);
            return AddDiagnostic(RexlDiagnostic.ErrorGuess(tok, node, msg, guess, rngGuess, args));
        }

        private RexlDiagnostic Warning(RexlNode node, StringId msg)
        {
            Validation.AssertValue(node);
            return AddDiagnostic(RexlDiagnostic.Warning(node, msg));
        }

        private RexlDiagnostic WarningGuess(RexlNode node, StringId msg, string guess, SourceRange rngGuess)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(guess);
            return AddDiagnostic(RexlDiagnostic.WarningGuess(node, msg, guess, rngGuess));
        }

        private RexlDiagnostic Warning(RexlNode node, StringId msg, params object[] args)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(args);
            return AddDiagnostic(RexlDiagnostic.Warning(node, msg, args));
        }

        private RexlDiagnostic WarningGuess(RexlNode node, StringId msg, string guess, SourceRange rngGuess, params object[] args)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(args);
            Validation.AssertValue(guess);
            return AddDiagnostic(RexlDiagnostic.WarningGuess(node, msg, guess, rngGuess, args));
        }

        private RexlDiagnostic WarningGuess(Token tok, RexlNode node, StringId msg, string guess, SourceRange rngGuess, params object[] args)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(args);
            Validation.AssertValue(guess);
            return AddDiagnostic(RexlDiagnostic.WarningGuess(tok, node, msg, guess, rngGuess, args));
        }

        private RexlDiagnostic Warning(Token tok, RexlNode node, StringId msg, params object[] args)
        {
            Validation.AssertValue(tok);
            Validation.AssertValue(node);
            Validation.AssertValue(args);
            return AddDiagnostic(RexlDiagnostic.Warning(tok, node, msg, args));
        }

        /// <summary>
        /// Push the bound node on the stack and associate it with the given parse node.
        /// </summary>
        private void Push(RexlNode node, BoundNode bnd)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(bnd);
            _stack.Add(Associate(node, bnd));
        }

        /// <summary>
        /// Push the bound node on the stack. It should already be associated with a parse node.
        /// </summary>
        private void Push(BoundNode val)
        {
            Validation.AssertValue(val);
            Validation.Assert(_bndToParse.ContainsKey(val));
            _stack.Add(val);
        }

        /// <summary>
        /// Sets a parse node type and scope chain information as inferred by the binder.
        /// Asserts that the information has not previously been set.
        /// </summary>
        private void SetNodeInfoCore(RexlNode node, DType type, BoundNode bnd, bool isUsedImplicitName = false)
        {
            Validation.AssertValue(node);

            // Shouldn't assign a type to a non-ExprNode. Note that we can't assert that the type
            // is valid iff the node is an ExprNode, since the record node in an augmenting projection
            // node is not given a valid type (intentionally).
            Validation.Assert((node is ExprNode) || !type.IsValid);
            Validation.AssertValueOrNull(bnd);

            Validation.Assert(!_nodeToInfo.ContainsKey(node));
            _nodeToInfo = _nodeToInfo.SetItem(node, new NodeInfo(type, _scopeCur.Info, isUsedImplicitName, bnd));
        }

        /// <summary>
        /// Sets the type and scope information for the node.
        /// </summary>
        private void SetNodeTypeAndScope(RexlNode node, DType type)
        {
            Validation.AssertValue(node);
            Validation.Assert(type.IsValid);
            SetNodeInfoCore(node, type, null);
        }

        /// <summary>
        /// Sets the type and scope information for the node. For the type, it peeks the bound node from
        /// the top of the stack (without removing it).
        /// </summary>
        private void SetNodeInfo(RexlNode node)
        {
            Validation.AssertValue(node);
            var bnd = Peek();
            Validation.AssertValue(bnd);
            SetNodeInfoCore(node, bnd.Type, bnd);
        }

        /// <summary>
        /// Sets the scope information, but no type, for the node.
        /// </summary>
        private void SetNodeScope(RexlNode node)
        {
            Validation.AssertValue(node);
            SetNodeInfoCore(node, default(DType), null);
        }

        /// <summary>
        /// Peeks (without removing) the top bound node from the stack.
        /// </summary>
        private BoundNode Peek()
        {
            Validation.Assert(_stack.Count > 0);
            int index = _stack.Count - 1;
            return _stack[index];
        }

        /// <summary>
        /// Pop the top bound node from the stack.
        /// </summary>
        private BoundNode Pop()
        {
            Validation.Assert(_stack.Count > 0);
            int index = _stack.Count - 1;
            var res = _stack[index];
            _stack.RemoveAt(index);
            return res;
        }

        /// <summary>
        /// Pop the top count bound nodes from the stack.
        /// </summary>
        private ArgTuple PopMulti(int count)
        {
            return PopToBuilder(count).ToImmutable();
        }

        /// <summary>
        /// Pop the top count bound nodes from the stack.
        /// </summary>
        private ArgTuple.Builder PopToBuilder(int count, int extra = 0)
        {
            Validation.Assert(0 <= count & count <= _stack.Count);
            Validation.Assert(extra >= 0);

            int index = _stack.Count - count;
            var bldr = ArgTuple.CreateBuilder(count + extra);
            for (int i = 0; i < count; i++)
                bldr.Add(_stack[index + i]);
            _stack.RemoveRange(index, count);
            return bldr;
        }

        /// <summary>
        /// Reverse the top <paramref name="count"/> items on the stack.
        /// </summary>
        private void Reverse(int count)
        {
            Validation.AssertIndexInclusive(count, _stack.Count);
            for (int min = _stack.Count - count, max = _stack.Count - 1; min < max; min++, max--)
            {
                var tmp = _stack[min];
                _stack[min] = _stack[max];
                _stack[max] = tmp;
            }
        }

        private BndErrorNode CreateBndError(RexlNode node, DType? typeGuess, StringId msg)
        {
            return Associate(node, BndErrorNode.Create(Error(node, msg), typeGuess));
        }

        private void PushError(RexlNode node, DType? typeGuess, StringId msg)
        {
            Push(CreateBndError(node, typeGuess, msg));
        }

        private BndErrorNode CreateBndError(Token tok, RexlNode node, DType? typeGuess, StringId msg)
        {
            return Associate(node, BndErrorNode.Create(Error(tok, node, msg), typeGuess));
        }

        private void PushError(Token tok, RexlNode node, DType? typeGuess, StringId msg)
        {
            Push(CreateBndError(tok, node, typeGuess, msg));
        }

        private BndErrorNode CreateBndError(RexlNode node, DType? typeGuess, StringId msg, params object[] args)
        {
            return Associate(node, BndErrorNode.Create(Error(node, msg, args), typeGuess));
        }

        private void PushError(RexlNode node, DType? typeGuess, StringId msg, params object[] args)
        {
            Push(CreateBndError(node, typeGuess, msg, args));
        }

        private BndErrorNode CreateBndError(Token tok, RexlNode node, DType? typeGuess, StringId msg, params object[] args)
        {
            return Associate(node, BndErrorNode.Create(Error(tok, node, msg, args), typeGuess));
        }

        private void PushError(Token tok, RexlNode node, DType? typeGuess, StringId msg, params object[] args)
        {
            Push(CreateBndError(tok, node, typeGuess, msg, args));
        }

        /// <summary>
        /// Push a pre-created scope.
        /// </summary>
        private ScopeWrapper PushScope(ArgScope scope, string variable = null)
        {
            Validation.AssertValue(_scopeCur);
            Validation.AssertValue(scope);
            Validation.AssertValueOrNull(variable);

            _scopeCur = ScopeWrapper.Create(_scopeCur, scope, variable);
            _scopeCur.Enabled = true;
            return _scopeCur;
        }

        /// <summary>
        /// Push a new scope.
        /// </summary>
        private ScopeWrapper PushScope(ScopeKind kind, DType type,
            ArgScope index = null, string variable = null, ExprNode srcImplicitNode = null)
        {
            Validation.AssertValue(_scopeCur);
            Validation.Assert(type.IsValid);
            Validation.Assert(index == null || index.IsIndex);
            Validation.AssertValueOrNull(variable);

            _scopeCur = ScopeWrapper.Create(_scopeCur, kind, type, index, variable, srcImplicitNode);
            _scopeCur.Enabled = true;
            return _scopeCur;
        }

        /// <summary>
        /// Pop the top scope. This asserts that the top-most scope, which represents the root, is never popped.
        /// </summary>
        private ScopeWrapper PopScope()
        {
            Validation.AssertValue(_scopeCur);
            Validation.AssertValue(_scopeCur.Outer);
            var res = _scopeCur;
            _scopeCur = _scopeCur.Outer;
            res.Enabled = false;
            return res;
        }

        /// <summary>
        /// Cast the given <paramref name="bnd"/> to the given <paramref name="typeDst"/>.
        /// This asserts that the destination type accepts the source type. It does not generate
        /// any diagnostics.
        ///
        /// Note that the <paramref name="node"/> is used so we can use the "Lift" functions.
        /// Other than establishing some associations, it really shouldn't be necessary, since
        /// this should never generate errors. On the flip side, we may want to generate warnings.
        /// </summary>
        private BoundNode CastGeneral(RexlNode node, BoundNode bnd, DType typeDst, bool union)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(bnd);
            Validation.Assert(typeDst.Accepts(bnd.Type, union));

            if (!_bndToParse.ContainsKey(bnd))
                _bndToParse = _bndToParse.SetItem(bnd, node);
            return Conversion.CastBnd(this, bnd, typeDst, union);
        }

        /// <summary>
        /// Checks that the bound node root type can be converted to the indicated <paramref name="typeWant"/>
        /// (issues an error if not). On output <paramref name="bnd"/> will have type <paramref name="typeWant"/>.
        /// </summary>
        private bool CheckGeneralType(RexlNode node, ref BoundNode bnd, DType typeWant, bool union = DType.UseUnionOper)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(bnd);

            if (typeWant.Accepts(bnd.Type, union))
            {
                bnd = CastGeneral(node, bnd, typeWant, union);
                Validation.Assert(bnd.Type == typeWant);
                return true;
            }

            bnd = CreateBndError(node, typeWant, ErrorStrings.ErrBadType_Src_Dst, bnd.Type, typeWant);
            Validation.Assert(bnd.Type == typeWant);
            return false;
        }

        /// <summary>
        /// Checks that the bound node root type can be converted to the indicated <paramref name="typeWant"/>
        /// (issues an error if not). On output <paramref name="bnd"/> will have type <paramref name="typeWant"/>.
        /// </summary>
        private bool CheckGeneralType(RexlNode node, ref BoundNode bnd, DType typeWant, bool union, StringId errMsg, params object[] args)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(bnd);

            if (typeWant.Accepts(bnd.Type, union))
            {
                bnd = CastGeneral(node, bnd, typeWant, union);
                return true;
            }

            bnd = CreateBndError(node, typeWant, errMsg, args);
            return false;
        }

        /// <summary>
        /// Return the common super type of the given types. If this results in "new" general types, report an error.
        /// </summary>
        private DType GetSuperType(RexlNode node, DType type0, DType type1, bool union = DType.UseUnionOper)
        {
            bool toGen = false;
            var type = DType.GetSuperType(type0, type1, union, ref toGen);
            if (toGen)
            {
                Validation.Assert(type.HasGeneral);
                if ((_options & BindOptions.AllowGeneral) == 0)
                    Error(node, ErrorStrings.ErrIncompatibleTypes_Type_Type, type0, type1);
            }
            return type;
        }

        /// <summary>
        /// Return the common super type of the given types. Doesn't report an error when "new" general types
        /// are in the result type. This is for situations where the presence of general is known to cause other
        /// errors to be reported. For example, when the result type must be equatable, etc.
        /// </summary>
        private DType GetSuperTypeRaw(DType type0, DType type1, bool union = DType.UseUnionOper)
        {
            return DType.GetSuperType(type0, type1, union);
        }

        [Flags]
        private enum LiftKinds : byte
        {
            None = 0x00,

            /// <summary>
            /// Lift over sequence using <see cref="ForEachFunc"/>.
            /// </summary>
            Seq = 0x01,

            /// <summary>
            /// Lift over tensor, both req and opt. When a tensor is opt, the lifting is in two phases,
            /// an outer <see cref="WithFunc.Guard"/> and an inner <see cref="TensorForEachFunc"/>.
            /// </summary>
            Ten = 0x02,

            /// <summary>
            /// Lift over "opt". More precisely, when an arg type has a required form, wrap with an
            /// invocation of <see cref="WithFunc.Guard"/>.
            /// </summary>
            Opt = 0x04,

            // Combinations.
            SeqOpt = Seq | Opt,
            SeqTen = Seq | Ten,
            SeqTenOpt = Seq | Ten | Opt,
        }

        /// <summary>
        /// Lifts a unary operation over the indicated lifting kinds.
        /// </summary>
        private void LiftUnary<TNode>(LiftKinds kinds, TNode node, BoundNode arg, Action<TNode, BoundNode> action)
            where TNode : RexlNode
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg);
            Validation.AssertValue(action);

            Action<TNode, ArgTuple, BitSet, Action<TNode, ArgTuple>> lifter;
            if ((kinds & LiftKinds.Seq) != 0 && arg.Type.IsSequence)
                lifter = LiftOverSeq;
            else if ((kinds & LiftKinds.Ten) != 0 && arg.Type.IsTensorXxx)
            {
                if (arg.Type.IsOpt)
                    lifter = LiftOverOpt;
                else
                    lifter = LiftOverTen;
            }
            else if ((kinds & LiftKinds.Opt) != 0 && arg.Type.HasReq)
                lifter = LiftOverOpt;
            else
            {
                action(node, arg);
                return;
            }

            lifter(
                node, Immutable.Array.Create(arg), 0x01,
                (n, args) => { Validation.Assert(args.Length == 1); LiftUnary(kinds, n, args[0], action); });
        }

        /// <summary>
        /// Lifts a binary operation over the indicated lifting kinds.
        /// </summary>
        private void LiftBinary<TNode>(LiftKinds kinds, TNode node, BoundNode arg0, BoundNode arg1, Action<TNode, BoundNode, BoundNode> action)
            where TNode : RexlNode
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.AssertValue(action);

            Action<TNode, ArgTuple, BitSet, Action<TNode, ArgTuple>> lifter;
            bool f0, f1;
            if ((kinds & LiftKinds.Seq) != 0 && ((f0 = arg0.Type.IsSequence) | (f1 = arg1.Type.IsSequence)))
                lifter = LiftOverSeq;
            else if ((kinds & LiftKinds.Ten) != 0 && ((f0 = arg0.Type.IsTensorXxx) | (f1 = arg1.Type.IsTensorXxx)))
            {
                if (f0 && arg0.Type.IsOpt || f1 && arg1.Type.IsOpt)
                {
                    f0 &= arg0.Type.IsOpt;
                    f1 &= arg1.Type.IsOpt;
                    lifter = LiftOverOpt;
                }
                else
                    lifter = LiftOverTen;
            }
            else if ((kinds & LiftKinds.Opt) != 0 && ((f0 = arg0.Type.HasReq) | (f1 = arg1.Type.HasReq)))
                lifter = LiftOverOpt;
            else
            {
                action(node, arg0, arg1);
                return;
            }

            Validation.Assert(f0 | f1);
            lifter(
                node, Immutable.Array.Create(arg0, arg1), (f0 ? 0x01u : 0) | (f1 ? 0x02u : 0),
                (n, args) => { Validation.Assert(args.Length == 2); LiftBinary(kinds, n, args[0], args[1], action); });
        }

        private static void GetLiftSlots(ArgTuple args,
            out BitSet slotsSeq, out BitSet slotsTen, out BitSet slotsOpt, BitSet maskLiftSeqNeedsSeq = default)
        {
            slotsSeq = default;
            slotsTen = default;
            slotsOpt = default;
            for (int i = 0; i < args.Length; i++)
            {
                var type = args[i].Type;
                if (type.SeqCount > maskLiftSeqNeedsSeq.TestBit(i).ToNum())
                    slotsSeq = slotsSeq.SetBit(i);
                else
                {
                    if (type.IsTensorXxx)
                        slotsTen = slotsTen.SetBit(i);
                    if (type.HasReq)
                        slotsOpt = slotsOpt.SetBit(i);
                }
            }
        }

        /// <summary>
        /// Lifts an operation over the indicated lifting kinds.
        /// </summary>
        private void Lift<TNode>(LiftKinds kinds, TNode node, ArgTuple args, Action<TNode, ArgTuple> action)
            where TNode : RexlNode
        {
            Validation.AssertValue(node);
            Validation.Assert(!args.IsDefaultOrEmpty);
            Validation.AssertValue(action);

            GetLiftSlots(args, out var slotsSeq, out var slotsTen, out var slotsOpt);

            BitSet slots;
            Action<TNode, ArgTuple, BitSet, Action<TNode, ArgTuple>> lifter;
            if ((kinds & LiftKinds.Seq) != 0 && !(slots = slotsSeq).IsEmpty)
                lifter = LiftOverSeq;
            else if ((kinds & LiftKinds.Ten) != 0 && !(slots = slotsTen).IsEmpty)
            {
                if (!(slotsOpt &= slotsTen).IsEmpty)
                {
                    slots = slotsOpt;
                    lifter = LiftOverOpt;
                }
                else
                    lifter = LiftOverTen;
            }
            else if ((kinds & LiftKinds.Opt) != 0 && !(slots = slotsOpt).IsEmpty)
                lifter = LiftOverOpt;
            else
            {
                action(node, args);
                return;
            }

            lifter(
                node, args, slots,
                (n, a) =>
                {
                    Validation.Assert(a.Length == args.Length);
                    Lift(kinds, n, a, action);
                });
        }

        /// <summary>
        /// Lifts an operation over the indicated lifting kinds, with the given prohibitions.
        /// </summary>
        private void Lift<TNode>(LiftKinds kinds, TNode node, ArgTuple args, Action<TNode, ArgTuple> action,
                BitSet slotsNotSeq = default, BitSet slotsNotTen = default, BitSet slotsNotOpt = default)
            where TNode : RexlNode
        {
            Validation.AssertValue(node);
            Validation.Assert(!args.IsDefaultOrEmpty);
            Validation.AssertValue(action);

            GetLiftSlots(args, out var slotsSeq, out var slotsTen, out var slotsOpt);
            if (!slotsNotSeq.IsEmpty)
                slotsSeq -= slotsNotSeq;
            if (!slotsNotTen.IsEmpty)
                slotsTen -= slotsNotTen;
            if (!slotsNotOpt.IsEmpty)
                slotsOpt -= slotsNotOpt;

            BitSet slots;
            Action<TNode, ArgTuple, BitSet, Action<TNode, ArgTuple>> lifter;
            if ((kinds & LiftKinds.Seq) != 0 && !(slots = slotsSeq).IsEmpty)
                lifter = LiftOverSeq;
            else if ((kinds & LiftKinds.Ten) != 0 && !(slots = slotsTen).IsEmpty)
            {
                if (!(slotsOpt &= slotsTen).IsEmpty)
                {
                    slots = slotsOpt;
                    lifter = LiftOverOpt;
                }
                else
                    lifter = LiftOverTen;
            }
            else if ((kinds & LiftKinds.Opt) != 0 && !(slots = slotsOpt).IsEmpty)
                lifter = LiftOverOpt;
            else
            {
                action(node, args);
                return;
            }

            lifter(
                node, args, slots,
                (n, a) =>
                {
                    Validation.Assert(a.Length == args.Length);
                    Lift(kinds, n, a, action, slotsNotSeq, slotsNotTen, slotsNotOpt);
                });
        }

        /// <summary>
        /// Kinds that should not be "repeated" when lifting over sequence. These should always
        /// be pulled into a "with".
        /// </summary>
        private const BndNodeKindMask k_maskDontRepeat =
            BndNodeKindMask.CallVolatile |
            BndNodeKindMask.CallProcedure;

        /// <summary>
        /// Kinds that may be "repeated" when lifting over sequence. These don't need to be pulled
        /// into a "with".
        /// </summary>
        private const BndNodeKindMask k_maskCanRepeat =
            BndNodeKindMask.Error |
            BndNodeKindMask.MissingValue |
            BndNodeKindMask.Namespace |
            BndNodeKindMask.Null |
            BndNodeKindMask.Default |
            BndNodeKindMask.Int |
            BndNodeKindMask.Flt |
            BndNodeKindMask.Str |
            BndNodeKindMask.CmpConst |
            BndNodeKindMask.This |
            BndNodeKindMask.Global |
            BndNodeKindMask.FreeVar |
            BndNodeKindMask.ArgScopeRef |
            BndNodeKindMask.IndScopeRef;

        /// <summary>
        /// These are cheap enough that they can be repeated if the child can.
        /// </summary>
        private const BndNodeKindMask k_maskCast =
            BndNodeKindMask.CastNum |
            BndNodeKindMask.CastRef |
            // Boxing allocates memory, so we don't want to repeat it.
            // BndNodeKindMask.CastBox |
            BndNodeKindMask.CastOpt |
            BndNodeKindMask.CastVac;

        /// <summary>
        /// Returns true if it is ok to repeatedly "compute" the given node. This is true for things like
        /// constants, globals, scope references, etc. It must be false for anything impure, like volatile
        /// calls or procedure calls. For casts (except boxing), this keeps the cast with the child.
        /// For other things, the hoisting code should be able to extract it from a repeated context but
        /// we might as well say that it shouldn't be repeated.
        /// </summary>
        private bool CanRepeat(BoundNode bnd)
        {
            Validation.AssertValue(bnd);

            if ((bnd.AllKinds & k_maskDontRepeat) != 0)
                return false;
            if (bnd.IsCheap)
                return true;
            while (true)
            {
                var mask = bnd.ThisKindMask;
                if ((k_maskCanRepeat & mask) != 0)
                    return true;
                if ((k_maskCast & mask) == 0)
                    break;

                // REVIEW: Currently it seems that this can't get hit since the casts are (typically)
                // introduced only after lifting has been completed. Perhaps this should be a property on
                // BoundNode so it can be used in more contexts as needed?
                var cast = (BndCastNode)bnd;
                if (cast.Kind == BndNodeKind.CastNum)
                {
                    if (cast.Type == DType.IAReq)
                        return false;
                    if (cast.Child.Type == DType.IAReq)
                        return false;
                }
                bnd = cast.Child;
            }

            // These could really go either way. See the doc comment above.
            return false;
        }

        /// <summary>
        /// Lifts an operation over sequence via (repeated) Map/Zip on the slots indicated by <paramref name="slotsLift"/>.
        /// </summary>
        private void LiftOverSeq<TNode>(TNode node, ArgTuple args, BitSet slotsLift, Action<TNode, ArgTuple> action)
            where TNode : RexlNode
        {
            LiftOverSeq(node, args, slotsLift, action, slotsNeedSeq: default);
        }

        /// <summary>
        /// Lifts an operation over sequence via (repeated) Map/Zip on the slots indicated by
        /// <paramref name="slotsLift"/>. The slots that should end with a sequence count of one are indicated
        /// by <paramref name="slotsNeedSeq"/>. Note that any slots in <paramref name="slotsNeedSeq"/> that
        /// are not in <paramref name="slotsLift"/> are ignored.
        /// </summary>
        private void LiftOverSeq<TNode>(
                TNode node, ArgTuple args, BitSet slotsLift, Action<TNode, ArgTuple> action, BitSet slotsNeedSeq)
            where TNode : RexlNode
        {
            Validation.AssertValue(node);
            Validation.Assert(!args.IsDefault);
            Validation.Assert(args.Length > 0);
            Validation.Assert(!slotsLift.TestAtOrAbove(args.Length));
            Validation.Assert(!slotsNeedSeq.TestAtOrAbove(args.Length));

            if (slotsLift.IsEmpty)
            {
                // No need for lifting.
                action(node, args);
                return;
            }

            int depthBase = _stack.Count;
            var scopeBase = _scopeCur;

            // These are the scopes.
            var scopeIndex = ArgScope.CreateIndex();
            var infos = new List<(BoundNode arg, ArgScope scope)>();
            BitSet slotsLiftNew = slotsLift;
            ScopeTuple.Builder scopesWith = null;
            ArgTuple.Builder argsWith = null;
            var bldrInner = args.ToBuilder();
            for (int slot = 0; slot < args.Length; slot++)
            {
                var arg = args[slot];

                ArgScope scope;
                if (!slotsLift.TestBit(slot))
                {
                    if (CanRepeat(arg))
                        continue;
                    scope = ArgScope.Create(ScopeKind.With, arg.Type);
                    (scopesWith ??= ScopeTuple.CreateBuilder()).Add(scope);
                    (argsWith ??= ArgTuple.CreateBuilder()).Add(arg);
                }
                else
                {
                    Validation.Assert(arg.Type.IsSequence);
                    Validation.Assert(!slotsNeedSeq.TestBit(slot) | arg.Type.SeqCount > 1);

                    PushScope(ScopeKind.SeqItem, arg.Type.ItemTypeOrThis, scopeIndex);
                    scope = _scopeCur.Scope;
                    infos.Add((arg, scope));
                    if (scope.Type.SeqCount == 0 || scope.Type.SeqCount == 1 && slotsNeedSeq.TestBit(slot))
                        slotsLiftNew = slotsLiftNew.ClearBit(slot);
                }

                bldrInner[slot] = Associate(arg, node, BndScopeRefNode.Create(scope));
            }
            Validation.Assert(infos.Count > 0);
            Validation.Assert(_stack.Count == depthBase);

            var scopeInner = _scopeCur;
            Validation.Assert(scopeInner.Nest == scopeBase.Nest + infos.Count);

            // Recurse down.
            LiftOverSeq(node, bldrInner.ToImmutable(), slotsLiftNew, action, slotsNeedSeq);

            Validation.Assert(_scopeCur == scopeInner);
            _scopeCur = scopeBase;

            // The recursive call should have left one additional stack element.
            Validation.Assert(_stack.Count == depthBase + 1);
            var value = Pop();

            var bldrOuter = ArgTuple.CreateBuilder(infos.Count + 1, init: true);
            for (int i = 0; i < infos.Count; i++)
                bldrOuter[i] = infos[i].arg;
            bldrOuter[infos.Count] = value;
            var bnds = bldrOuter.ToImmutable();

            var scopes = infos.Select(x => x.scope).ToImmutableArray();
            Validation.Assert(bnds.Length == scopes.Length + 1);

            var res = BndCallNode.Create(ForEachFunc.ForEach,
                value.Type.ToSequence(), bnds, scopes, ScopeTuple.Create(scopeIndex));
            if (argsWith != null)
            {
                Validation.Assert(argsWith.Count == scopesWith.Count);
                argsWith.Add(Associate(node, res));
                res = BndCallNode.Create(WithFunc.With, res.Type, argsWith.ToImmutable(),
                    scopesWith.ToImmutable());
            }
            Push(node, res);
        }

        /// <summary>
        /// Lifts an operation over tensor via Tensor.ForEach on the slots indicated by <paramref name="slotsLift"/>.
        /// </summary>
        private void LiftOverTen<TNode>(TNode node, ArgTuple args, BitSet slotsLift, Action<TNode, ArgTuple> action)
            where TNode : RexlNode
        {
            Validation.AssertValue(node);
            Validation.Assert(!args.IsDefaultOrEmpty);
            Validation.Assert(!slotsLift.IsEmpty);
            Validation.Assert(!slotsLift.TestAtOrAbove(args.Length));

            int depthBase = _stack.Count;
            var scopeBase = _scopeCur;

            var bldrInner = args.ToBuilder();
            int numScopes = slotsLift.Count;
            var bldrScopes = ScopeTuple.CreateBuilder(numScopes, init: true);
            var bldrOuter = ArgTuple.CreateBuilder(numScopes + 1, init: true);
            int iscope = 0;
            for (int slot = 0; slot < args.Length; slot++)
            {
                if (!slotsLift.TestBit(slot))
                    continue;
                Validation.AssertIndex(iscope, numScopes);
                var arg = args[slot];
                Validation.Assert(arg.Type.IsTensorReq);
                PushScope(ScopeKind.TenItem, arg.Type.GetTensorItemType());
                var scope = _scopeCur.Scope;
                bldrInner[slot] = Associate(arg, node, BndScopeRefNode.Create(scope));
                bldrScopes[iscope] = scope;
                bldrOuter[iscope] = arg;
                iscope++;
            }
            Validation.Assert(_stack.Count == depthBase);
            Validation.Assert(iscope == numScopes);

            var scopeInner = _scopeCur;
            Validation.Assert(scopeInner.Nest == scopeBase.Nest + numScopes);

            action(node, bldrInner.ToImmutable());

            Validation.Assert(_scopeCur == scopeInner);
            _scopeCur = scopeBase;

            // The action call should have left one additional stack element, namely the selector.
            Validation.Assert(_stack.Count == depthBase + 1);
            var value = Pop();

            var typeSel = value.Type;
            int rank = 0;
            for (int i = 0; i < numScopes; i++)
            {
                var arg = bldrOuter[i];
                int rankCur = arg.Type.TensorRank;
                if (rank < rankCur)
                    rank = rankCur;
            }

            bldrOuter[numScopes] = value;
            BoundNode res = BndCallNode.Create(TensorForEachFunc.Eager, typeSel.ToTensor(false, rank), bldrOuter.ToImmutable(), bldrScopes.ToImmutable());
            Push(node, res);
        }

        /// <summary>
        /// Lifts an operation over opt via Guard on the slots indicated by <paramref name="slotsLift"/>.
        /// Compare to <see cref="LiftOverSeq"/>.
        /// </summary>
        private void LiftOverOpt<TNode>(TNode node, ArgTuple args, BitSet slotsLift, Action<TNode, ArgTuple> action)
            where TNode : RexlNode
        {
            Validation.AssertValue(node);
            Validation.Assert(!args.IsDefaultOrEmpty);
            Validation.Assert(!slotsLift.IsEmpty);
            Validation.Assert(!slotsLift.TestAtOrAbove(args.Length));

            int depthBase = _stack.Count;
            var scopeBase = _scopeCur;

            var bldrInner = args.ToBuilder();
            int numScopes = slotsLift.Count;
            var bldrScopes = ScopeTuple.CreateBuilder(numScopes, init: true);
            var bldrOuter = ArgTuple.CreateBuilder(numScopes + 1, init: true);
            int iscope = 0;
            for (int slot = 0; slot < args.Length; slot++)
            {
                if (!slotsLift.TestBit(slot))
                    continue;
                Validation.AssertIndex(iscope, numScopes);
                var arg = args[slot];
                Validation.Assert(arg.Type.HasReq);
                PushScope(ScopeKind.Guard, arg.Type.ToReq());
                var scope = _scopeCur.Scope;
                bldrInner[slot] = Associate(arg, node, BndScopeRefNode.Create(scope));
                bldrScopes[iscope] = scope;
                bldrOuter[iscope] = arg;
                iscope++;
            }
            Validation.Assert(_stack.Count == depthBase);
            Validation.Assert(iscope == numScopes);

            var scopeInner = _scopeCur;
            Validation.Assert(scopeInner.Nest == scopeBase.Nest + numScopes);

            action(node, bldrInner.ToImmutable());

            Validation.Assert(_scopeCur == scopeInner);
            _scopeCur = scopeBase;

            // The action call should have left one additional stack element, namely the "selector" for the guard invocation.
            Validation.Assert(_stack.Count == depthBase + 1);
            var value = Pop();

            bldrOuter[numScopes] = value;
            BoundNode res = BndCallNode.Create(WithFunc.Guard, value.Type.ToOpt(), bldrOuter.ToImmutable(), bldrScopes.ToImmutable());
            Push(node, res);
        }

        /// <summary>
        /// Bind numeric or time negation on a non-sequence.
        /// </summary>
        private void BindNegateCore(UnaryOpNode node, BoundNode arg)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg);
            Validation.Assert(!arg.Type.IsSequence);
            Validation.Assert(!arg.Type.HasReq);

            // Apply negation to a constant. We do this here rather than in the reducer (for integer types),
            // since it affects the type. For example, we want -3i1 to have type i1 rather than i4.
            if (arg.TryGetIntegral(out var bi))
            {
                DKind kindRes = DType.SuperNum(DKind.I1, arg.Type.Kind);
                Validation.Assert(kindRes.IsSignedIntegral());
                var bnd = BndIntNode.CreateCast(DType.GetNumericType(kindRes), -bi, out bool overflow);
                if (overflow)
                    Warning(node, ErrorStrings.WrnIntOverflow);
                Push(node, bnd);
                return;
            }
            // Since we handled integer constants, we might as well do floating point too.
            if (arg.TryGetFractional(out var fr))
            {
                Push(node, BndFltNode.Create(arg.Type, -fr));
                return;
            }

            DType typeRes;
            BoundNode res;
            if (arg.Type.RootKind.IsChrono())
            {
                typeRes = DType.TimeReq;
                CheckGeneralType(node.Arg, ref arg, typeRes);
                res = BndBinaryOpNode.Create(typeRes, BinaryOp.ChronoMul, arg, BndIntNode.CreateI8(-1));
            }
            else
            {
                typeRes = DType.GetNumericBinaryType(arg.Type.RootKind, DKind.I1);
                Validation.Assert(typeRes.IsNumericReq);
                CheckGeneralType(node.Arg, ref arg, typeRes);
                res = BndVariadicOpNode.Create(typeRes, BinaryOp.Add, ArgTuple.Create(arg), invs: 0x1);
            }

            Push(node, res);
        }

        /// <summary>
        /// Bind bitwise negation on a non-sequence.
        /// </summary>
        private void BindNegateBitWiseCore(UnaryOpNode node, BoundNode arg)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg);
            Validation.Assert(!arg.Type.IsSequence);
            Validation.Assert(!arg.Type.HasReq);
            Validation.Assert(!arg.Type.IsTensorXxx);

            DType typeRes = DType.GetIntegerBinaryType(arg.Type.RootKind, arg.Type.RootKind, forceSize: false);
            Validation.Assert(typeRes.IsIntegralReq);
            CheckGeneralType(node.Arg, ref arg, typeRes);

            // Bitwise-xor with all ones. This helps with reduction, eg, in something like "~(~X ^ ~Y)".
            var arg1 = Associate(node, BndIntNode.CreateAllOnes(typeRes));
            Push(node, BndVariadicOpNode.Create(typeRes, BinaryOp.BitXor, arg, arg1, inv: false));
        }

        /// <summary>
        /// Bind logical negation on a non-sequence.
        /// </summary>
        private void BindNegateLogicalCore(UnaryOpNode node, BoundNode arg)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg);
            Validation.Assert(!arg.Type.IsSequence);
            Validation.Assert(!arg.Type.HasReq);
            Validation.Assert(!arg.Type.IsTensorXxx);

            DType typeRes = DType.BitReq;
            CheckGeneralType(node.Arg, ref arg, typeRes);

            // Xor with true. This helps with reduction, eg, in something like "not (not X) xor not Y".
            var arg1 = Associate(node, BndIntNode.True);
            Push(node, BndVariadicOpNode.Create(typeRes, BinaryOp.Xor, arg, arg1, inv: false));
        }

        /// <summary>
        /// Bind percent on a non-sequence.
        /// </summary>
        private void BindPercentCore(UnaryOpNode node, BoundNode arg)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg);
            Validation.Assert(!arg.Type.IsSequence);
            Validation.Assert(!arg.Type.HasReq);
            Validation.Assert(!arg.Type.IsTensorXxx);

            DType typeRes = DType.GetNumericBinaryType(arg.Type.RootKind, arg.Type.RootKind, forceFrac: true);
            Validation.Assert(typeRes.IsFractionalReq);
            CheckGeneralType(node.Arg, ref arg, typeRes);

            // Divide by 100.
            var arg1 = Associate(node, BndFltNode.CreateCast(typeRes, 100));
            Push(node, BndVariadicOpNode.Create(typeRes, BinaryOp.Div, arg, arg1, inv: false));
        }

        /// <summary>
        /// Binds a unary operation. The operand is on the stack.
        /// </summary>
        private void BindUop(UnaryOpNode node)
        {
            Validation.AssertValue(node);

            var arg = Pop();

            switch (node.Op)
            {
            default:
                break;

            case UnaryOp.Posate:
                if (!arg.Type.CoreKind.IsNumeric())
                {
                    DType typeRes = (arg.Type.IsRootOpt ? DType.I8Opt : DType.I8Req).ToSequence(arg.Type.SeqCount);
                    CheckGeneralType(node.Arg, ref arg, typeRes);
                }
                Push(node, arg);
                return;

            case UnaryOp.Negate:
                LiftUnary(LiftKinds.SeqTenOpt, node, arg, BindNegateCore);
                return;

            case UnaryOp.BitNot:
                LiftUnary(LiftKinds.SeqTenOpt, node, arg, BindNegateBitWiseCore);
                return;

            case UnaryOp.Not:
                // Note: we use Guard to handle b? values. This produces slightly worse IL, but facilitates
                // composing Guards.
                LiftUnary(LiftKinds.SeqTenOpt, node, arg, BindNegateLogicalCore);
                return;

            case UnaryOp.Percent:
                LiftUnary(LiftKinds.SeqTenOpt, node, arg, BindPercentCore);
                return;
            }

            Validation.Assert(false);
            PushError(node, null, ErrorStrings.ErrInternalError);
        }

        private void BindAddSubChronoCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.Assert(node.Op == BinaryOp.Add | node.Op == BinaryOp.Sub);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert(!arg0.Type.IsSequence);
            Validation.Assert(!arg1.Type.IsSequence);
            Validation.Assert(!arg0.Type.HasReq);
            Validation.Assert(!arg1.Type.HasReq);

            // Signatures:
            // * D + T => D
            // * D - T => D
            // * D - D => T
            // * T + D => D
            // * T + T => T
            // * T - T => T

            bool sub = node.Op == BinaryOp.Sub;
            // D + D not supported
            if (arg0.Type.RootKind == DKind.Date && arg1.Type.RootKind == DKind.Date && !sub)
            {
                PushError(node, null, ErrorStrings.ErrBadOperatorForType_Op_Left_Right, node.Token.GetTextString(), arg0.Type, arg1.Type);
            }
            else
            {
                DType typeRes;
                if (arg0.Type.RootKind == DKind.Date)
                {
                    if (arg1.Type.RootKind == DKind.Date && sub)
                    {
                        // * D - D => T
                        typeRes = DType.TimeReq;
                    }
                    else
                    {
                        // * D + T => D
                        // * D - T => D
                        typeRes = DType.DateReq;
                        CheckGeneralType(node.Arg1, ref arg1, DType.TimeReq);
                    }
                }
                else
                {
                    CheckGeneralType(node.Arg0, ref arg0, DType.TimeReq);
                    if (arg1.Type.RootKind == DKind.Date && !sub)
                    {
                        // * T + D => D
                        // Normalize to D + T => D.
                        typeRes = DType.DateReq;
                        Util.Swap(ref arg0, ref arg1);
                    }
                    else
                    {
                        // * T + T => T
                        // * T - T => T
                        typeRes = DType.TimeReq;
                        CheckGeneralType(node.Arg1, ref arg1, DType.TimeReq);
                    }
                }
                Push(node, BndBinaryOpNode.Create(typeRes, sub ? BinaryOp.ChronoSub : BinaryOp.ChronoAdd, arg0, arg1));
            }
        }

        private void BindMulDivModChronoCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            // Allowed scenarios
            //   t * r8 => t
            //   t * i8 => t
            //   r8 * t => t
            //   i8 * t => t
            //
            //   t / r8 => t
            //   t / t => r8
            //   t div i8 => t
            //   t div t => i8
            //   t mod t => t

            // Validate caller assumptions
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert((arg0.Type.SeqCount | arg1.Type.SeqCount) == 0);
            Validation.Assert(!arg0.Type.HasReq);
            Validation.Assert(!arg1.Type.HasReq);
            Validation.Assert(arg0.Type.RootKind.IsChrono() || arg1.Type.RootKind.IsChrono());

            bool mul = node.Op == BinaryOp.Mul;

            var parg0 = node.Arg0;
            var parg1 = node.Arg1;
            DType type1;
            DType typeRes;
            BinaryOp op;
            switch (node.Op)
            {
            case BinaryOp.Mul:
                // Normalize to have time in arg0.
                if (!arg0.Type.RootKind.IsChrono() || arg1.Type.RootKind == DKind.Time && arg0.Type.RootKind != DKind.Time)
                {
                    Util.Swap(ref arg0, ref arg1);
                    Util.Swap(ref parg0, ref parg1);
                }
                // Either i8 or r8.
                type1 = DType.I8Req.Accepts(arg1.Type, DType.UseUnionDefault) ? DType.I8Req : DType.R8Req;
                typeRes = DType.TimeReq;
                op = BinaryOp.ChronoMul;
                break;

            case BinaryOp.Div:
                // Either time or r8.
                type1 = DType.R8Req;
                typeRes = DType.TimeReq;
                if (arg1.Type.RootKind.IsChrono())
                    Util.Swap(ref type1, ref typeRes);
                op = BinaryOp.ChronoDiv;
                break;

            case BinaryOp.IntDiv:
                // Either time or i8.
                type1 = DType.I8Req;
                typeRes = DType.TimeReq;
                if (arg1.Type.RootKind.IsChrono())
                    Util.Swap(ref type1, ref typeRes);
                op = BinaryOp.ChronoDiv;
                break;

            case BinaryOp.IntMod:
            default:
                Validation.Assert(node.Op == BinaryOp.IntMod);

                // i8.
                type1 = DType.TimeReq;
                typeRes = DType.TimeReq;
                op = BinaryOp.ChronoMod;
                break;
            }

            CheckGeneralType(parg0, ref arg0, DType.TimeReq);
            CheckGeneralType(parg1, ref arg1, type1);

            Push(node, BndBinaryOpNode.Create(typeRes, op, arg0, arg1));
        }

        /// <summary>
        /// For binding variadic ops with operand types matching the result type.
        /// </summary>
        private void BindVariadicOpCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert((arg0.Type.SeqCount | arg1.Type.SeqCount) == 0);
            Validation.Assert(!arg0.Type.HasReq);
            Validation.Assert(!arg1.Type.HasReq);

            // The result type depends on the operator (and args).
            DType typeRes;
            var op = node.Op;
            switch (node.Op)
            {
            case BinaryOp.Div:
                typeRes = DType.GetNumericBinaryType(arg0.Type.RootKind, arg1.Type.RootKind, forceFrac: true);
                Validation.Assert(typeRes.IsFractionalReq);
                break;
            case BinaryOp.BitAnd:
            case BinaryOp.BitOr:
            case BinaryOp.BitXor:
                typeRes = DType.GetIntegerBinaryType(arg0.Type.RootKind, arg1.Type.RootKind, forceSize: false);
                break;
            case BinaryOp.Xor:
                typeRes = DType.BitReq;
                break;
            default:
                typeRes = DType.GetNumericBinaryType(arg0.Type.RootKind, arg1.Type.RootKind);
                Validation.Assert(typeRes.IsNumericReq);
                break;
            }

            // Operand types must match result type.
            CheckGeneralType(node.Arg0, ref arg0, typeRes);
            CheckGeneralType(node.Arg1, ref arg1, typeRes);

            Push(node, BndVariadicOpNode.Create(typeRes, op, arg0, arg1, inv: false));
        }

        /// <summary>
        /// For binding binary (non-variadic) ops with operand types matching the result type.
        /// </summary>
        private void BindBinaryOpCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert((arg0.Type.SeqCount | arg1.Type.SeqCount) == 0);

            // The result type depends on the operator (and args).
            DType typeRes;
            switch (node.Op)
            {
            case BinaryOp.IntDiv:
            case BinaryOp.IntMod:
                typeRes = DType.GetIntegerBinaryType(arg0.Type.RootKind, arg1.Type.RootKind, forceSize: true);
                Validation.Assert(typeRes.IsIntegralReq);
                break;
            case BinaryOp.Min:
            case BinaryOp.Max:
                typeRes = GetSuperTypeRaw(arg0.Type, arg1.Type);
                if (typeRes.IsComparable)
                {
                    Validation.Assert(!typeRes.HasGeneral);
                    break;
                }
                if (!arg0.Type.IsComparable)
                    PushError(node.Arg0, arg1.Type, ErrorStrings.ErrIncomparableType_Type, arg0.Type);
                else if (!arg1.Type.IsComparable)
                    PushError(node.Arg1, arg0.Type, ErrorStrings.ErrIncomparableType_Type, arg1.Type);
                else
                {
                    // Arbitrarily use the first arg type.
                    PushError(node, arg0.Type, ErrorStrings.ErrIncomparableTypes_Type_Type, arg0.Type, arg1.Type);
                }
                return;

            default:
                // Assume any numeric type will work.
                typeRes = DType.GetNumericBinaryType(arg0.Type.RootKind, arg1.Type.RootKind);
                Validation.Assert(typeRes.IsNumericReq);
                break;
            }
            Validation.Assert(!typeRes.HasReq);

            // Operand types must match result type.
            CheckGeneralType(node.Arg0, ref arg0, typeRes);
            CheckGeneralType(node.Arg1, ref arg1, typeRes);

            Push(node, BndBinaryOpNode.Create(typeRes, node.Op, arg0, arg1));
        }

        /// <summary>
        /// For binding power.
        /// </summary>
        private void BindPowerCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.Assert(node.Op == BinaryOp.Power);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);

            var type0 = arg0.Type;
            var type1 = arg1.Type;
            Validation.Assert((type0.SeqCount | type1.SeqCount) == 0);
            Validation.Assert(!type0.HasReq);
            Validation.Assert(!type1.HasReq);
            var kind0 = type0.RootKind;
            var kind1 = type1.RootKind;

            DType typeRes;
            DType typePow;
            if ((kind0.IsIxOrUx() || kind0 == DKind.Vac) && (kind1.IsIxOrUx() || kind1 == DKind.Vac))
            {
                typeRes = kind0 == DKind.U8 ? DType.U8Req : DType.I8Req;
                typePow = kind1.IsSignedIntegral() ? DType.I8Req : DType.U8Req;
            }
            else
                typePow = typeRes = DType.R8Req;

            CheckGeneralType(node.Arg0, ref arg0, typeRes);
            CheckGeneralType(node.Arg1, ref arg1, typePow);

            Push(node, BndBinaryOpNode.Create(typeRes, BinaryOp.Power, arg0, arg1));
        }

        private void BindBitShiftCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert((arg0.Type.SeqCount | arg1.Type.SeqCount) == 0);
            Validation.Assert(!arg0.Type.HasReq);
            Validation.Assert(!arg1.Type.HasReq);

            var bop = node.Op;
            Validation.Assert(bop == BinaryOp.Shl | bop == BinaryOp.Shr | bop == BinaryOp.Shri | bop == BinaryOp.Shru);

            DType typeRes = arg0.Type;
            if (!typeRes.IsIntegralReq)
            {
                typeRes = DType.I8Req;
                CheckGeneralType(node.Arg0, ref arg0, typeRes);
            }
            Validation.Assert(arg0.Type == typeRes);
            var kind = typeRes.RootKind;

            // REVIEW: Should we support any integral type for shift amount?
            // That should be safe unless the value type is Integer and bop is <<.
            // REVIEW: Should we limit the left shift amount for Integer type?
            CheckGeneralType(node.Arg1, ref arg1, DType.I8Req);

            // Warn for shift right unsigned with big integer.
            if (bop == BinaryOp.Shru && kind == DKind.IA)
            {
                WarningGuess(node, ErrorStrings.WrnShruOn_Type, "shr", node.Token.Range, typeRes);
                bop = BinaryOp.Shri;
            }

            // Replace unspecified shift right with specific one.
            if (bop == BinaryOp.Shr)
                bop = kind.IsUx() ? BinaryOp.Shru : BinaryOp.Shri;

            Push(node, BndBinaryOpNode.Create(typeRes, bop, arg0, arg1));
        }

        private void BindOrAndCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.Assert(node.Op == BinaryOp.Or | node.Op == BinaryOp.And);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert((arg0.Type.SeqCount | arg1.Type.SeqCount) == 0);

            CheckGeneralType(node.Arg0, ref arg0, arg0.Type.IsOpt ? DType.BitOpt : DType.BitReq);
            CheckGeneralType(node.Arg1, ref arg1, arg1.Type.IsOpt ? DType.BitOpt : DType.BitReq);
            DType typeRes = arg0.Type.IsOpt || arg1.Type.IsOpt ? DType.BitOpt : DType.BitReq;

            Push(node, BndVariadicOpNode.Create(typeRes, node.Op, arg0, arg1, inv: false));
        }

        private void BindStrConcatCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert((arg0.Type.SeqCount | arg1.Type.SeqCount) == 0);

            DType typeRes = DType.Text;
            CheckGeneralType(node.Arg0, ref arg0, typeRes);
            CheckGeneralType(node.Arg1, ref arg1, typeRes);

            Push(node, BndVariadicOpNode.Create(typeRes, BinaryOp.StrConcat, arg0, arg1, inv: false));
        }

        private void BindTupleConcatCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert((arg0.Type.SeqCount | arg1.Type.SeqCount) == 0);
            Validation.Assert(!arg0.Type.HasReq);
            Validation.Assert(!arg1.Type.HasReq);

            var type0 = arg0.Type;
            var type1 = arg1.Type;

            if (type0.RootKind != DKind.Tuple)
            {
                Error(node.Arg0, ErrorStrings.ErrNotTuple);
                if (type0.RootKind == DKind.Vac)
                    arg0 = Associate(node.Arg0, BndTupleNode.Create(ArgTuple.Empty));
                else
                    arg0 = Associate(node.Arg0, BndTupleNode.Create(ArgTuple.Create(arg0)));
            }
            if (type1.RootKind != DKind.Tuple)
            {
                Error(node.Arg1, ErrorStrings.ErrNotTuple);
                if (type1.RootKind == DKind.Vac)
                    arg1 = Associate(node.Arg1, BndTupleNode.Create(ArgTuple.Empty));
                else
                    arg1 = Associate(node.Arg1, BndTupleNode.Create(ArgTuple.Create(arg1)));
            }

            Push(node, BndVariadicOpNode.CreateTupleConcat(arg0, arg1));
        }

        private void BindRecordConcatCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert((arg0.Type.SeqCount | arg1.Type.SeqCount) == 0);
            Validation.Assert(!arg0.Type.HasReq);
            Validation.Assert(!arg1.Type.HasReq);

            var type0 = arg0.Type;
            var type1 = arg1.Type;

            if (type0.RootKind != DKind.Record)
                arg0 = CreateBndError(node.Arg0, DType.EmptyRecordReq, ErrorStrings.ErrNotRecord);
            if (type1.RootKind != DKind.Record)
                arg1 = CreateBndError(node.Arg1, DType.EmptyRecordReq, ErrorStrings.ErrNotRecord);

            Push(node, BndVariadicOpNode.CreateRecordConcat(arg0, arg1));
        }

        private void BindSeqConcat(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);

            // Use operator acceptance.
            bool union = DType.UseUnionOper;

            // Do the type checking and resolution.
            DType type0 = arg0.Type;
            if (!type0.IsSequence && type0 != DType.Null)
                type0 = type0.ToSequence();
            DType type1 = arg1.Type;
            if (!type1.IsSequence && type1 != DType.Null)
                type1 = type1.ToSequence();
            DType typeRes = GetSuperType(node, type0, type1, union);
            if (!typeRes.IsSequence)
            {
                Validation.Assert(typeRes.IsNull);
                typeRes = DType.Vac.ToSequence();
            }

            // Delay the type conversions until we have the individual sequences in the variadic,
            // unless the conversions generate an error.
            if (!typeRes.Accepts(arg0.Type, union) || !typeRes.Accepts(arg1.Type, union))
            {
                CheckGeneralType(node.Arg0, ref arg0, typeRes, union);
                CheckGeneralType(node.Arg1, ref arg1, typeRes, union);
                Push(node, BndVariadicOpNode.Create(typeRes, BinaryOp.SeqConcat, arg0, arg1, inv: false));
                return;
            }

            Validation.Assert(typeRes.Accepts(arg0.Type, union));
            Validation.Assert(typeRes.Accepts(arg1.Type, union));

            var (bldr, invs) = BndVariadicOpNode.GetVarArgs(BinaryOp.SeqConcat, false, arg0, arg1, out int index);
            Validation.Assert(invs.IsEmpty);
            Validation.Assert(0 < index & index < bldr.Count);

            for (int i = 0; i < bldr.Count; i++)
            {
                var seq = bldr[i];
                Validation.Assert(i < index & seq.Type == arg0.Type | index <= i & seq.Type == arg1.Type);
                if (seq.Type != typeRes)
                    bldr[i] = CastGeneral(node, seq, typeRes, union);
            }

            Push(node, BndVariadicOpNode.Create(typeRes, BinaryOp.SeqConcat, bldr.ToImmutable(), default));
        }

        /// <summary>
        /// Bind an in operation.
        /// </summary>
        private void BindIn(InHasNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);

            var type0 = arg0.Type;
            var type1 = arg1.Type;

            LiftOverSeq(
                node,
                Immutable.Array.Create(arg0, arg1),
                new BitSet(type0.SeqCount > 0, type1.SeqCount > 1),
                (n, args) =>
                {
                    Validation.Assert(args.Length == 2);
                    BindInCore(n, args[0], args[1]);
                },
                // The 2nd arg needs a sequence (of non-sequence).
                0x02);
        }

        private void BindInCore(InHasNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert(node.Op == BinaryOp.In | node.Op == BinaryOp.InNot | node.Op == BinaryOp.InCi | node.Op == BinaryOp.InCiNot);
            Validation.Assert(arg0.Type.SeqCount == 0);
            Validation.Assert(arg1.Type.SeqCount <= 1);

            var root0 = arg0.Type.RootType;
            var root1 = arg1.Type.RootType;

            bool union = DType.UseUnionOper;
            DType typeCmp = GetSuperTypeRaw(root0, root1, union);

            if (typeCmp.IsVacXxx)
                typeCmp = DType.Text;
            if (!typeCmp.IsEquatable)
            {
                if (!arg1.Type.IsSequence)
                    Error(node.Arg1, ErrorStrings.ErrNeedSequenceForIn);
                if (root0 == typeCmp)
                    PushError(node.Arg0, DType.BitReq, ErrorStrings.ErrInequatableType_Type, typeCmp);
                else if (root1 == typeCmp)
                    PushError(node.Arg1, DType.BitReq, ErrorStrings.ErrInequatableType_Type, typeCmp);
                else
                    PushError(node, DType.BitReq, ErrorStrings.ErrIncompatibleTypes_Type_Type, root0, root1);
                return;
            }
            Validation.Assert(!typeCmp.HasGeneral);

            // Cast to the common comparison type.
            arg0 = CastGeneral(node.Arg0, arg0, typeCmp, union);
            CheckGeneralType(node.Arg1, ref arg1, typeCmp.ToSequence(1), union, ErrorStrings.ErrNeedSequenceForIn);

            var op = node.Op;
            if (!typeCmp.HasText && (op == BinaryOp.InCi || op == BinaryOp.InCiNot))
            {
                Validation.Assert(node.Tld != null);
                WarningGuess(node.Tld, node, ErrorStrings.WrnCmpCi_Type, "", node.Tld.Range, typeCmp);
                op = op.GetInHasCs();
            }

            Push(node, BndBinaryOpNode.Create(DType.BitReq, op, arg0, arg1));
        }

        private void BindHasCore(InHasNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert(node.Op == BinaryOp.Has | node.Op == BinaryOp.HasNot | node.Op == BinaryOp.HasCi | node.Op == BinaryOp.HasCiNot);
            Validation.Assert((arg0.Type.SeqCount | arg1.Type.SeqCount) == 0);

            CheckGeneralType(node.Arg0, ref arg0, DType.Text);
            CheckGeneralType(node.Arg1, ref arg1, DType.Text);

            Push(node, BndBinaryOpNode.Create(DType.BitReq, node.Op, arg0, arg1));
        }

        private void BindCoalesceCore(BinaryOpNode node, BoundNode arg0, BoundNode arg1)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg0);
            Validation.AssertValue(arg1);
            Validation.Assert((arg0.Type.SeqCount | arg1.Type.SeqCount) == 0);

            if (!arg0.Type.IsRootOpt)
                Warning(node.Arg0, ErrorStrings.WrnCoalesceLeftNotOpt_Type, arg0.Type.RootType);
            bool isOpt = arg1.Type.IsRootOpt;

            bool union = DType.UseUnionOper;
            DType typeResOpt = GetSuperType(node, arg0.Type, arg1.Type, union);
            Validation.Assert(typeResOpt.IsRootOpt || !arg0.Type.IsOpt);

            // Remove opt-ness if the second arg isn't opt.
            DType typeRes = isOpt ? typeResOpt : typeResOpt.ToReq();

            // Do conversion, which can't fail, since we got the super type.
            arg0 = CastGeneral(node, arg0, typeResOpt, union);
            arg1 = CastGeneral(node, arg1, typeRes, union);

            if (!typeResOpt.IsOpt)
            {
                // We warned above. Don't coalesce.
                Push(node, arg0);
                return;
            }

            Push(node, BndBinaryOpNode.Create(typeRes, BinaryOp.Coalesce, arg0, arg1));
        }

        /// <summary>
        /// Binds a binary operation. The operands are on the stack.
        /// </summary>
        private void BindBop(BinaryOpNode node)
        {
            Validation.AssertValue(node);

            var arg1 = Pop();
            var arg0 = Pop();

            // Core binding function.
            Action<BinaryOpNode, BoundNode, BoundNode> core;

            DKind kind0, kind1;
            LiftKinds lifts;
            switch (node.Op)
            {
            default:
                // If this assert goes off, it's likely that someone introduced a new binary operator
                // without extending this switch.
                Validation.Assert(false);
                PushError(node, null, ErrorStrings.ErrInternalError);
                return;

            // Special cases.
            case BinaryOp.Error:
                PushError(node, null, ErrorStrings.ErrOperatorExpected);
                return;
            case BinaryOp.SeqConcat:
                BindSeqConcat(node, arg0, arg1);
                return;
            case BinaryOp.Or:
            case BinaryOp.And:
                lifts = LiftKinds.SeqTen;
                core = BindOrAndCore;
                break;
            case BinaryOp.Coalesce:
                lifts = LiftKinds.Seq;
                core = BindCoalesceCore;
                break;
            case BinaryOp.GenConcat:
                // Give left arg priority in determining the kind of concatenation.
                lifts = LiftKinds.SeqTenOpt;
                switch (arg0.Type.CoreKind)
                {
                case DKind.Tuple:
                    core = BindTupleConcatCore;
                    break;
                case DKind.Record:
                    core = BindRecordConcatCore;
                    break;
                case DKind.Text:
                    // Note that lifting over tensor includes over opt-ness of tensors.
                    lifts = LiftKinds.SeqTen;
                    core = BindStrConcatCore;
                    break;

                default:
                    switch (arg1.Type.CoreKind)
                    {
                    case DKind.Tuple:
                        core = BindTupleConcatCore;
                        break;
                    case DKind.Record:
                        core = BindRecordConcatCore;
                        break;
                    case DKind.Text:
                        lifts = LiftKinds.SeqTen;
                        core = BindStrConcatCore;
                        break;

                    default:
                        // Assume text.
                        lifts = LiftKinds.SeqTen;
                        core = BindStrConcatCore;
                        break;
                    }
                    break;
                }
                break;

            // Lift over seq and opt (unless string).
            case BinaryOp.Sub:
                lifts = LiftKinds.SeqTenOpt;
                if (arg0.Type.CoreKind.IsChrono() || arg1.Type.CoreKind.IsChrono())
                    core = BindAddSubChronoCore;
                else
                    core = BindVariadicOpCore;
                break;
            case BinaryOp.Add:
                lifts = LiftKinds.SeqTenOpt;
                if ((kind0 = arg0.Type.CoreKind).IsChrono() || (kind1 = arg1.Type.CoreKind).IsChrono())
                    core = BindAddSubChronoCore;
                else if ((kind0 == DKind.Text || kind1 == DKind.Text) && !(kind0.IsNumeric() || kind1.IsNumeric()))
                {
                    WarningGuess(node, ErrorStrings.WrnDeprecatedStrCat, "&", node.Token.Range);
                    lifts = LiftKinds.SeqTen;
                    core = BindStrConcatCore;
                }
                else
                    core = BindVariadicOpCore;
                break;
            case BinaryOp.Mul:
                lifts = LiftKinds.SeqTenOpt;
                if (arg0.Type.CoreKind.IsChrono() || arg1.Type.CoreKind.IsChrono())
                    core = BindMulDivModChronoCore;
                else
                    core = BindVariadicOpCore;
                break;
            case BinaryOp.Div:
                lifts = LiftKinds.SeqTenOpt;
                if (arg0.Type.CoreKind.IsChrono() || arg1.Type.CoreKind.IsChrono())
                    core = BindMulDivModChronoCore;
                else
                    core = BindVariadicOpCore;
                break;
            case BinaryOp.BitOr:
            case BinaryOp.BitXor:
            case BinaryOp.BitAnd:
                lifts = LiftKinds.SeqTenOpt;
                core = BindVariadicOpCore;
                break;

            case BinaryOp.Xor:
                // This bool operator lifts over opt and is fully associative and commutative.
                lifts = LiftKinds.SeqTenOpt;
                core = BindVariadicOpCore;
                break;

            case BinaryOp.Shl:
            case BinaryOp.Shr:
            case BinaryOp.Shri:
            case BinaryOp.Shru:
                lifts = LiftKinds.SeqTenOpt;
                core = BindBitShiftCore;
                break;

            case BinaryOp.Power:
                lifts = LiftKinds.SeqTenOpt;
                core = BindPowerCore;
                break;

            case BinaryOp.IntDiv:
            case BinaryOp.IntMod:
                lifts = LiftKinds.SeqTenOpt;
                if (arg0.Type.CoreKind.IsChrono() || arg1.Type.CoreKind.IsChrono())
                    core = BindMulDivModChronoCore;
                else
                    core = BindBinaryOpCore;
                break;

            case BinaryOp.Min:
            case BinaryOp.Max:
                lifts = LiftKinds.SeqTenOpt;
                core = BindBinaryOpCore;
                if (arg0.Type.CoreKind == DKind.Text || arg1.Type.CoreKind == DKind.Text)
                    lifts = LiftKinds.SeqTen;
                break;
            }
            Validation.Assert(core != null);

            LiftBinary(lifts, node, arg0, arg1, core);
        }

        /// <summary>
        /// Binds a binary operation. The operands are on the stack.
        /// </summary>
        private void BindInHas(InHasNode node)
        {
            Validation.AssertValue(node);

            var arg1 = Pop();
            var arg0 = Pop();

            switch (node.Op)
            {
            default:
                // If this assert goes off, it's likely that someone introduced a new binary operator
                // without extending this switch.
                Validation.Assert(false);
                PushError(node, null, ErrorStrings.ErrInternalError);
                return;
            case BinaryOp.In:
            case BinaryOp.InNot:
            case BinaryOp.InCi:
            case BinaryOp.InCiNot:
                BindIn(node, arg0, arg1);
                return;

            // Lift over seq, but not opt.
            case BinaryOp.Has:
            case BinaryOp.HasNot:
            case BinaryOp.HasCi:
            case BinaryOp.HasCiNot:
                LiftBinary(LiftKinds.SeqTen, node, arg0, arg1, BindHasCore);
                return;
            }
        }

        private void BindCompare(CompareNode node)
        {
            Validation.AssertValue(node);
            Validation.Assert(node.Count >= 2);

            var args = PopMulti(node.Count);
            Lift(LiftKinds.SeqTen, node, args, BindCompareCore);
        }

        private void BindCompareCore(CompareNode node, ArgTuple args)
        {
            Validation.AssertValue(node);
            Validation.Assert(!args.IsDefault);
            Validation.Assert(args.Length == node.Count);
            Validation.Assert(args.All(x => x.Type.SeqCount == 0));

            var ops = node.Operators;
            Validation.Assert(ops.Length == node.Count - 1);

            CompareModifiers mods = default;
            bool hasOrder = false;
            for (int i = 0; i < ops.Length; i++)
            {
                var op = ops[i];
                mods |= op.Mods;
                hasOrder |= op.IsOrdered;
            }

            // First determine the comparison type. For this, ignore types that aren't compatible with the
            // operator(s), and types that push the super type to be not compatible. For example, when
            // comparing a numeric type and text, keep the first encountered and let CheckGeneralType
            // (invoked later) do the error reporting.
            DType typeCmp = default;
            ExprNode nodeBad = null;
            DType typeBad = default;
            for (int i = 0; i < args.Length; i++)
            {
                var type = args[i].Type;
                if (!hasOrder && !type.IsEquatable || hasOrder && !type.IsComparable)
                {
                    // Ignore types that aren't compatible with the operator(s). We'll report errors later.
                    if (nodeBad == null)
                    {
                        nodeBad = node.Children[i];
                        typeBad = type;
                    }
                    continue;
                }

                if (!typeCmp.IsValid)
                    typeCmp = type;
                else if (!typeCmp.Accepts(type, DType.UseUnionOper))
                {
                    var typeTmp = GetSuperTypeRaw(typeCmp, type);
                    if (typeTmp.IsEquatable)
                        typeCmp = typeTmp;
                    else
                        Validation.Assert(typeTmp.HasGeneral);
                }
            }

            if (!typeCmp.IsValid)
            {
                if (nodeBad != null)
                {
                    Error(nodeBad,
                        hasOrder ?
                            ErrorStrings.ErrIncomparableType_Type :
                            ErrorStrings.ErrInequatableType_Type,
                        typeBad);
                }
                typeCmp = mods.IsCi() ? DType.Text : DType.I8Req;
            }
            else
                typeCmp = typeCmp.ToReq();

            Validation.Assert(typeCmp.IsEquatable);
            Validation.Assert(!hasOrder || typeCmp.IsComparable);

            // The type to use for opt args.
            var typeCmpOpt = typeCmp.ToOpt();

            ArgTuple.Builder bldrArgs = null;
            Immutable.Array<CompareOp>.Builder bldrOps = null;

            var arg = args[0];
            var typePrev = arg.Type.IsOpt ? typeCmpOpt : typeCmp;
            bool hasErrors = !CheckGeneralType(node.Children[0], ref arg, typePrev);
            if (arg != args[0])
                (bldrArgs ??= args.ToBuilder())[0] = arg;

            for (int i = 1; i < args.Length; i++)
            {
                arg = args[i];
                var typeArg = arg.Type.IsOpt ? typeCmpOpt : typeCmp;
                hasErrors |= !CheckGeneralType(node.Children[i], ref arg, typeArg);
                if (arg != args[i])
                    (bldrArgs ??= args.ToBuilder())[i] = arg;

                var op = ops[i - 1];
                Validation.Assert(op != CompareOp.None);

                var opNew = op.SimplifyForType(typePrev, typeArg);
                if (opNew != op)
                {
                    if (op.IsCi && !opNew.IsCi)
                    {
                        var tokTld = node.Tokens[i - 1].Tld;
                        Validation.Assert(tokTld != null);
                        WarningGuess(tokTld, node, ErrorStrings.WrnCmpCi_Type, "", tokTld.Range, typeCmp);
                    }
                    (bldrOps ??= ops.ToBuilder())[i - 1] = opNew;
                }
                typePrev = typeArg;
            }

            if (bldrArgs is not null)
                args = bldrArgs.ToImmutable();
            if (bldrOps is not null)
                ops = bldrOps.ToImmutable();

            Push(node, BndCompareNode.Create(ops, args));
        }

        private bool TryBindName(FirstNameNode node)
        {
            Validation.Assert(node == PeekSrc());

            var name = node.Ident.Name;
            var isRooted = node.Ident.IsGlobal;

            if (!_host.TryFindName(node, name, isRooted, out NPath path, out DType type, out bool isStream))
                return false;

            Validation.Assert(!path.IsRoot);
            if (type.IsValid)
            {
                if (type.HasModule && (_options & BindOptions.ProhibitModule) != 0)
                    Error(node, ErrorStrings.ErrModuleNotSupported);
                if (isStream)
                    Util.GetOrAdd(ref _streamToNode, path, node);
                // When building a module, global references go through an "externals" tuple, so the module
                // builder needs to provide the reference.
                if (_moduleCur is not null)
                    Push(node, _moduleCur.GetGlobalRef(path, type));
                else
                    Push(node, BndGlobalNode.Create(path, type));
                return true;
            }
            Validation.Assert(!isStream);

            // If there is no parent or the parent is not a DottedName, the namespace is
            // being used as a value, which is illegal.
            if (PeekSrc(1)?.Kind != NodeKind.DottedName)
                Error(node, ErrorStrings.ErrNamespaceNotValue);
            Push(node, BndNamespaceNode.Create(path));

            return true;
        }

        private bool TryBindNameFuzzy(FirstNameNode node)
        {
            var name = node.Ident.Name;
            var isRooted = node.Ident.IsGlobal;
            if (!_host.TryFindNameFuzzy(node, name, isRooted, out DName nameGuess, out NPath path, out DType type, out bool isStream))
                return false;

            Validation.Assert(name != nameGuess);
            string strGuess = nameGuess.Escape();
            Validation.Coverage(strGuess.Contains('\'') ? 1 : 0);
            // Note: use node.GetRange() so we don't delete an @ prefix.
            ErrorGuess(node, ErrorStrings.ErrNameDoesNotExist_Guess, strGuess, node.GetRange(), nameGuess);

            if (type.IsValid)
            {
                // Note: use the original name as specified, NOT nameGuess. The guess is used for the error
                // message and the type, but not put in the bound tree!
                Push(node, BndGlobalNode.Create(NPath.Root.Append(name), type));
            }
            else
                Push(node, BndNamespaceNode.Create(path));

            return true;
        }

        private bool TryBindNamespaceItem(DottedNameNode node, NPath ns)
        {
            Validation.Assert(!ns.IsRoot);

            var name = node.Right.Name;
            if (!_host.TryFindNamespaceItem(node, ns, name, out NPath path, out DType type, out bool isStream))
                return false;

            Validation.Assert(!path.IsRoot);

            if (type.IsValid)
            {
                if (isStream)
                    Util.GetOrAdd(ref _streamToNode, path, node);
                Push(node, BndGlobalNode.Create(path, type));
                return true;
            }
            Validation.Assert(!isStream);

            // If there is no parent or the parent is not a DottedName, the namespace is
            // being used as a value, which is illegal.
            if (PeekSrc(1)?.Kind != NodeKind.DottedName)
                Error(node, ErrorStrings.ErrNamespaceNotValue);
            Push(node, BndNamespaceNode.Create(path));

            return true;
        }

        private bool TryBindNamespaceItemFuzzy(DottedNameNode node, NPath ns)
        {
            Validation.Assert(!ns.IsRoot);

            var name = node.Right.Name;
            if (!_host.TryFindNamespaceItemFuzzy(node, ns, name, out var nameGuess, out var type, out var isStream))
                return false;

            string strGuess = nameGuess.Escape();
            Validation.Coverage(strGuess.Contains('\'') ? 1 : 0);
            ErrorGuess(node, ErrorStrings.ErrBadNamespaceChild_Ns_Child_Guess, strGuess, node.Right.Range, ns, name, nameGuess);

            if (type.IsValid)
            {
                // Note: use the original name as specified, NOT nameGuess. The guess is used for the error
                // message and the type, but not put in the bound tree!
                Push(node, BndGlobalNode.Create(ns.Append(name), type));
            }
            else
                Push(node, BndNamespaceNode.Create(ns.Append(nameGuess)));

            return true;
        }

        private void BindDotted(DottedNameNode node)
        {
            Validation.AssertValue(node);

            var lhs = Pop();
            if (lhs is BndNamespaceNode bns)
            {
                if (!TryBindNamespaceItem(node, bns.Path) && !TryBindNamespaceItemFuzzy(node, bns.Path))
                    PushError(node, null, ErrorStrings.ErrBadNamespaceChild_Ns_Child, bns.Path, node.Right.Name);
                return;
            }

            // Don't lift dot over tensor since we want tensor properties to work, eg, Rank, Shape, etc.
            LiftUnary(LiftKinds.SeqOpt, node, lhs, BindDottedCore);
        }

        private void BindDottedCore(DottedNameNode node, BoundNode arg)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg);
            Validation.Assert(arg.Type.SeqCount == 0);
            Validation.Assert(!arg.Type.HasReq);

            BindFieldAccessCore(node, arg, node.Right);
        }

        private void BindImplicitFieldAccessCore(FirstNameNode node, BoundNode arg)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg);
            Validation.Assert(arg.Type.SeqCount == 0);
            Validation.Assert(!arg.Type.HasReq);

            BindFieldAccessCore(node, arg, node.Ident);
        }

        /// <summary>
        /// Determine whether this scope has a field/symbol/property with the given <paramref name="name"/>.
        /// For a record type, we look at fields. For a module, we look at symbols. For other types, we look
        /// for a matching "property function". For example, if the scope type is 'd?*', we'll look in the
        /// Date namespace for a function with the given <paramref name="name"/> that takes one arg of type 'd'.
        /// </summary>
        private bool HasField(ScopeWrapper scope, DName name, bool guess, out DName corrected)
        {
            Validation.AssertValue(scope);
            Validation.Assert(name.IsValid);

            corrected = default;

            DType type;
            if (scope.Scope is not null)
                type = scope.Scope.Type;
            else if (scope.Module is not null)
                type = scope.Module.TypeCur;
            else
                return false;

            switch (type.RootKind)
            {
            case DKind.Record:
                break;
            case DKind.Module:
                // This handles both the case of building a module and of a module being the value of a scope.
                break;

            default:
                {
                    var func = PropInvokeInfo.GetPropFunc(this, name, type, guess);
                    if (func != null)
                    {
                        if (guess)
                            corrected = func.Path.Leaf;
                        // The type is unknown in this case.
                        return true;
                    }
                    return false;
                }
            }

            // This handles both records and modules.
            if (type.Contains(name))
                return true;
            if (!guess)
                return false;

            foreach (var tn in type.GetNames())
            {
                if (_host.IsFuzzyMatch(name, tn.Name))
                {
                    corrected = tn.Name;
                    return true;
                }
            }

            return false;
        }

        private void BindFieldAccessCore(ExprNode node, BoundNode arg, Identifier ident)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(arg);
            Validation.AssertValue(ident);
            Validation.Assert(arg.Type.SeqCount == 0);
            Validation.Assert(!arg.Type.HasReq);

            DType type = arg.Type;
            switch (type.RootKind)
            {
            default:
                if (!PropInvokeInfo.TryRun(this, node, ident, type, arg))
                    PushError(node, null, ErrorStrings.ErrInvalidDot);
                return;

            case DKind.Record:
                break;
            case DKind.Module:
                if ((_options & BindOptions.ProhibitModule) != 0)
                    Error(node, ErrorStrings.ErrModuleNotSupported);
                arg = Associate(node, BndModToRecNode.Create(arg));
                type = arg.Type;
                break;
            }
            Validation.Assert(type.IsRecordReq);

            var name = ident.Name;
            if (type.Contains(name))
            {
                Push(node, BndGetFieldNode.Create(name, arg));
                return;
            }

            foreach (var tn in type.GetNames())
            {
                if (_host.IsFuzzyMatch(name, tn.Name))
                {
                    string strGuess = tn.Name.Escape();
                    Validation.Coverage(strGuess.Contains('\'') ? 1 : 0);
                    ErrorGuess(ident.Token, node, ErrorStrings.ErrFieldDoesNotExist_Type_Guess, strGuess, ident.Range, type, tn.Name);
                    Push(node, BndGetFieldNode.Create(tn.Name, arg));
                    return;
                }
            }

            PushError(ident.Token, node, null, ErrorStrings.ErrFieldDoesNotExist_Type, type);
        }

        private bool BindPipePre(BinaryOpNode node)
        {
            Validation.AssertValue(node);
            Validation.Assert(node.Op == BinaryOp.Pipe);

            // Visit the left.
            node.Arg0.Accept(this);
            var arg0 = Pop();

            // Push the result on the box value stack.
            int depth = _boxValueStack.Depth;
            _boxValueStack = _boxValueStack.Push(arg0);

            node.Arg1.Accept(this);
            var arg1 = Pop();

            Validation.Assert(_boxValueStack.Depth == depth + 1);
            Validation.Assert(_boxValueStack.Value == arg0);
            if (_boxValueStack.Uses == 0)
                Warning(node, ErrorStrings.WrnUnusedPipeValue);
            _boxValueStack = _boxValueStack.Rest;

            Push(node, arg1);
            return false;
        }

        private void BindIndexingCore(IndexingNode node, ArgTuple args)
        {
            Validation.AssertValue(node);

            var type = args[0].Type;

            // Lifting should have already been done.
            Validation.Assert(type.SeqCount == 0);
            Validation.Assert(!type.HasReq);

            if (type.RootKind == DKind.Text)
                BindTextIndexing(node, args);
            else if (type.IsTensorReq)
                BindTensorIndexing(node, args);
            else if (type.IsTupleReq)
                BindTupleIndexing(node, args);
            else
                PushError(node, type, ErrorStrings.ErrNotIndexable);
        }

        /// <summary>
        /// Used while building a tensor slicing operation to add a tuple defined slice. If the tuple
        /// can be unpacked, adds a range slice rather than a tuple slice. Generates an error if there
        /// are any index modifiers in the parse node. If <paramref name="values"/> is <c>null</c>,
        /// the values aren't actually added.
        /// </summary>
        private SliceItemFlags AddTupleSlice(ArgTuple.Builder values, SliceItemNode item, BoundNode tuple,
            out DType type)
        {
            Validation.AssertValueOrNull(values);
            Validation.AssertValue(item);
            Validation.AssertValue(tuple);
            Validation.Assert(tuple.Type.IsTupleReq);

            var tok = item.StartBack ?? item.StartEdge;
            if (tok != null)
                Error(tok, item, ErrorStrings.ErrBadIndexModifierWithTuple_Tok, tok.GetStdString());

            // Determine the correct tuple slot types.
            var slotsArg = tuple.Type.GetTupleSlotTypes();
            var type0 = DType.I8Opt;
            var type1 = DType.I8Opt;
            var type2 = DType.I8Opt;
            bool withBacks = slotsArg.Length > 3;
            if (slotsArg.Length > 0)
            {
                int islot = 0;
                if (!slotsArg[islot++].IsOpt)
                    type0 = DType.I8Req;
                if (withBacks)
                    islot++;
                if (slotsArg.Length > islot)
                {
                    if (!slotsArg[islot++].IsOpt)
                        type1 = DType.I8Req;
                    if (withBacks)
                        islot++;
                    if (slotsArg.Length > islot)
                    {
                        if (!slotsArg[islot++].IsOpt)
                            type2 = DType.I8Req;
                    }
                }
            }

            // Construct the correct tuple type, cast to it, and try to unpack the tuple to a range.
            if (withBacks)
            {
                var typeTup = type = DType.CreateTuple(false, type0, DType.BitReq, type1, DType.BitReq, type2);

                CheckGeneralType(item, ref tuple, typeTup);

                if (values != null && tuple is BndTupleNode btn &&
                    btn.Items[1].TryGetBool(out bool backStart) &&
                    btn.Items[3].TryGetBool(out bool backStop))
                {
                    // Unpack to a range.
                    return SliceUtils.Add(values,
                        btn.Items[0], backStart, btn.Items[2], backStop, star2: false, btn.Items[4]);
                }
            }
            else
            {
                var typeTup = type = DType.CreateTuple(false, type0, type1, type2);

                CheckGeneralType(item, ref tuple, typeTup);

                if (values != null && tuple is BndTupleNode btn)
                {
                    // Unpack to a range.
                    return SliceUtils.Add(values, btn.Items[0], btn.Items[1], btn.Items[2]);
                }
            }

            values?.Add(tuple);
            return SliceItemFlags.Tuple;
        }

        /// <summary>
        /// Process the given slice item, <paramref name="item"/>, adding values to <paramref name="bldrVals"/>
        /// if it is provided. Returns the flags defining the item. Updates <paramref name="iarg"/> as it digests
        /// values (which have already been bound).
        /// </summary>
        private SliceItemFlags AddSliceItem(ArgTuple.Builder bldrVals, SliceItemNode item,
            ArgTuple args, ref int iarg)
        {
            Validation.AssertValueOrNull(bldrVals);
            Validation.AssertValue(item);
            Validation.Assert(!args.IsDefault);

            int carg = args.Length;

            if (item.IsSimple)
            {
                Validation.Assert(iarg < carg);
                var arg = args[iarg++];

                // Lifting should have ensured these.
                Validation.Assert(!arg.Type.IsSequence);
                Validation.Assert(!arg.Type.HasReq);

                if (arg.Type.RootKind == DKind.Tuple)
                {
                    // This is a tuple encoded slice. We unpack the tuple if possible.
                    return AddTupleSlice(bldrVals, item, arg, out var typeTup);
                }

                CheckGeneralType(item, ref arg, DType.I8Req);
                bldrVals?.Add(arg);

                return GetIndexFlags(item).ToSliceFlags();
            }

            // Compound.
            BoundNode start = null;
            BoundNode stop = null;
            BoundNode step = null;
            if (item.Start != null)
            {
                Validation.Assert(iarg < carg);
                start = args[iarg++];
                CheckGeneralType(item.Start, ref start, start.Type.IsOpt ? DType.I8Opt : DType.I8Req);
            }
            if (item.Stop != null)
            {
                Validation.Assert(iarg < carg);
                stop = args[iarg++];
                CheckGeneralType(item.Stop, ref stop, stop.Type.IsOpt ? DType.I8Opt : DType.I8Req);
            }
            if (item.Step != null)
            {
                Validation.Assert(iarg < carg);
                step = args[iarg++];
                CheckGeneralType(item.Step, ref step, step.Type.IsOpt ? DType.I8Opt : DType.I8Req);
            }

            return SliceUtils.Add(bldrVals,
                start, item.StartBack != null, stop, item.StopBack != null, item.StopStar != null, step);
        }

        /// <summary>
        /// Bind indexing and slicing of text.
        /// </summary>
        private void BindTextIndexing(IndexingNode node, ArgTuple args)
        {
            Validation.Assert(args.Length > 0);
            Validation.Assert(args[0].Type == DType.Text);

            int carg = args.Length;
            var items = node.Items.Children;
            int citem = items.Length;

            if (citem == 0)
            {
                Error(node.TokOpen, node.Items, ErrorStrings.ErrExpectedIndexOrSlice);
                Push(node, args[0]);
                return;
            }

            if (citem > 1)
                Error(items[1], ErrorStrings.ErrUnexpectedIndex);

            BoundNode bnd;
            int iarg = 1;
            var item = items[0];
            if (item.IsSimple && args[1].Type.RootKind != DKind.Tuple)
            {
                // Lifting should have ensured these.
                var arg = args[iarg++];
                Validation.Assert(!arg.Type.IsSequence);
                Validation.Assert(!arg.Type.HasReq);
                CheckGeneralType(item, ref arg, DType.I8Req);
                var flags = GetIndexFlags(item);
                bnd = BndIdxTextNode.Create(args[0], arg, flags);
            }
            else
            {
                var bldrVals = ArgTuple.CreateBuilder(carg - 1);
                var flags = AddSliceItem(bldrVals, item, args, ref iarg);
                Validation.Assert(!flags.IsIndex());
                bnd = BndTextSliceNode.Create(args[0], flags, bldrVals.ToImmutable());
            }

            // Processing remaining ones just for diagnostics.
            for (int iitem = 1; iitem < citem; iitem++)
                AddSliceItem(null, items[iitem], args, ref iarg);
            Validation.Assert(iarg == carg);

            Push(node, bnd);
        }

        /// <summary>
        /// Bind an indexing operation on a tensor. This can produce either an indexing or slicing
        /// bound node, depending on the arguments. Note that the args generally do NOT have a 1:1
        /// correspondence with the indexing slots, since range items can have from zero to three
        /// values in args.
        /// </summary>
        private void BindTensorIndexing(IndexingNode node, ArgTuple args)
        {
            Validation.Assert(args.Length > 0);

            var tensor = args[0];
            var type = tensor.Type;
            Validation.Assert(type.IsTensorReq);

            int carg = args.Length;
            var items = node.Items.Children;
            int citem = items.Length;

            int rank = type.TensorRank;
            if (citem > rank)
                Error(items[rank], ErrorStrings.ErrWrongNumberOfTensorIndices_Rank, rank);

            // Check for all indices (rather than slices) but only up to rank. We discard the rest. If the
            // number of indices is less than rank, then it's also slicing since the remainder is padded with
            // full slice ranges.
            if (citem >= rank)
            {
                for (int iitem = 0; iitem < rank; iitem++)
                {
                    var item = items[iitem];
                    if (!item.IsSimple)
                        goto LSlicing;
                    Validation.Assert(iitem + 1 < args.Length);
                    if (args[iitem + 1].Type.RootKind == DKind.Tuple)
                        goto LSlicing;
                }

                // The pure indexing case.
                var bldrInds = ArgTuple.CreateBuilder(rank, init: true);
                var bldrFlags = Immutable.Array<IndexFlags>.CreateBuilder(rank, init: true);
                for (int i = 0; i < rank; i++)
                {
                    var item = items[i];
                    Validation.Assert(item.IsSimple);
                    Validation.Assert(i + 1 < args.Length);
                    var arg = args[i + 1];
                    CheckGeneralType(item, ref arg, DType.I8Req);
                    bldrInds[i] = arg;
                    bldrFlags[i] = GetIndexFlags(item);
                }

                Push(node, BndIdxTensorNode.Create(tensor, bldrInds.ToImmutable(), bldrFlags.ToImmutable()));
                return;
            }

        LSlicing:
            // The slicing case.
            var bldrItem = Immutable.Array<SliceItemFlags>.CreateBuilder(rank, init: true);
            var bldrVals = ArgTuple.CreateBuilder(carg - 1);
            int iarg = 1;
            for (int iitem = 0; iitem < citem; iitem++)
            {
                var itemNew = AddSliceItem(iitem < rank ? bldrVals : null, items[iitem], args, ref iarg);
                if (iitem < rank)
                    bldrItem[iitem] = itemNew;
            }
            Validation.Assert(iarg == carg);

            // Pad with "full range" items. These need no values.
            for (int i = citem; i < rank; i++)
                bldrItem[i] = SliceItemFlags.Range;

            var bnd = BndTensorSliceNode.Create(tensor, bldrItem.ToImmutable(), bldrVals.ToImmutable());
            Push(node, bnd);
        }

        /// <summary>
        /// Bind indexing of a tuple. This also handles explicit slicing. Note that we don't support
        /// tuple-based slicing, where a slice is encoded as a tuple. This is partly because slicing
        /// start/stop/step values need to be fully resolved at bind time so there is really no need
        /// for tuple-based slicing of tuples.
        /// </summary>
        private void BindTupleIndexing(IndexingNode node, ArgTuple args)
        {
            Validation.Assert(args.Length > 0);

            var tup = args[0];
            var typeTup = tup.Type;
            int arity = typeTup.TupleArity;

            int carg = args.Length;
            var items = node.Items.Children;

            RexlNode pind = node.Items;
            BoundNode ind;
            RexlDiagnostic errMissingIdx = null;

            if (arity == 0)
            {
                PushError(node, null, ErrorStrings.ErrEmptyTupleNoIndexing);
                return;
            }

            SliceItemNode item;
            if (items.Length == 0)
            {
                errMissingIdx = Error(node.TokOpen, pind, ErrorStrings.ErrExpectedIndexOrSlice);
                ind = BndMissingValueNode.Create(DType.I8Req);
                item = null;
            }
            else
            {
                if (items.Length > 1)
                    Error(items[1], ErrorStrings.ErrUnexpectedIndex);

                item = items[0];
                int iarg = 1;
                if (!item.IsSimple)
                {
                    // Range.
                    long? startOpt = null;
                    long? stopOpt = null;
                    long? stepOpt = null;
                    SliceItemFlags sifs = SliceItemFlags.Range | SliceItemFlags.Step;
                    if (item.Start != null)
                    {
                        Validation.Assert(iarg < carg);
                        var arg = args[iarg++];
                        CheckGeneralType(item.Start, ref arg, arg.Type.IsOpt ? DType.I8Opt : DType.I8Req);
                        if (!arg.TryGetI8Opt(out startOpt))
                        {
                            PushError(item.Start, null, ErrorStrings.ErrBadTupleSliceRange);
                            return;
                        }
                        if (startOpt.HasValue)
                        {
                            sifs |= SliceItemFlags.Start;
                            if (item.StartBack != null)
                                sifs |= SliceItemFlags.StartBack;
                        }
                    }
                    if (item.Stop != null)
                    {
                        Validation.Assert(iarg < carg);
                        var arg = args[iarg++];
                        CheckGeneralType(item.Stop, ref arg, arg.Type.IsOpt ? DType.I8Opt : DType.I8Req);
                        if (!arg.TryGetI8Opt(out stopOpt))
                        {
                            PushError(item.Stop, null, ErrorStrings.ErrBadTupleSliceRange);
                            return;
                        }
                        if (stopOpt.HasValue)
                        {
                            sifs |= SliceItemFlags.Stop;
                            if (item.StopBack != null)
                                sifs |= SliceItemFlags.StopBack;
                            if (item.StopStar != null)
                                sifs |= SliceItemFlags.StopStar;
                        }
                    }
                    if (item.Step != null)
                    {
                        Validation.Assert(iarg < carg);
                        var arg = args[iarg++];
                        CheckGeneralType(item.Step, ref arg, arg.Type.IsOpt ? DType.I8Opt : DType.I8Req);
                        if (!arg.TryGetI8Opt(out stepOpt))
                        {
                            PushError(item.Step, null, ErrorStrings.ErrBadTupleSliceRange);
                            return;
                        }
                    }

                    var (start, step, count) = SliceUtil.GetRange(arity, sifs,
                        startOpt.GetValueOrDefault(), stopOpt.GetValueOrDefault(), stepOpt.GetValueOrDefault());
                    Validation.AssertIndexInclusive(count, arity);

                    if (count == 0)
                        Push(node, BndTupleNode.Create(ArgTuple.Empty));
                    else if (count == arity && (step > 0 || arity == 1))
                    {
                        Validation.Assert(step == 1 || arity == 1);
                        Validation.Assert(start == 0);
                        Push(node, tup);
                    }
                    else
                        Push(node, BndTupleSliceNode.Create(tup, (int)start, (int)step, (int)count));
                    return;
                }

                // Not a slice (just an index).
                Validation.Assert(iarg < carg);
                ind = args[iarg++];

                CheckGeneralType(item, ref ind, DType.I8Req);
                pind = item;
            }

            DType typeItem;
            var flags = GetIndexFlags(item);
            if (ind.TryGetI8(out long slot))
            {
                slot = flags.AdjustIndex(slot, arity);
                if (Validation.IsValidIndex(slot, arity))
                {
                    Push(node, BndGetSlotNode.Create((int)slot, tup));
                    return;
                }

                if (typeTup.IsHomTuple(out typeItem))
                {
                    Warning(pind, ErrorStrings.WrnHomTupleIndexOutOfRange);
                    Push(node, BndDefaultNode.Create(typeItem));
                    return;
                }

                if (flags == 0 && slot < 0 && -arity < slot)
                    PushError(pind, null, ErrorStrings.ErrTupleNegativeIndex_Index, -slot);
                else if ((flags & IndexFlags.Back) != 0)
                    PushError(pind, null, ErrorStrings.ErrHetTupleOffsetOutOfRange_Max, arity);
                else
                    PushError(pind, null, ErrorStrings.ErrHetTupleIndexOutOfRange_Arity, arity);
                return;
            }

            if (typeTup.IsHomTuple(out typeItem))
            {
                var bnd = BndIdxHomTupNode.Create(tup, ind, flags, out bool oor);
                Validation.Assert(!oor);
                Push(node, bnd);
                return;
            }

            if (errMissingIdx != null)
            {
                Push(node, BndErrorNode.Create(errMissingIdx));
                return;
            }

            PushError(pind, null, ErrorStrings.ErrBadHetTupleIndex);
        }

        /// <summary>
        /// The binder handles only expressions, not statements.
        /// </summary>
        private void ThrowOnStmt()
        {
            throw Validation.BugExcept("Binder handles only expressions");
        }

        protected override void VisitImpl(ErrorNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            Validation.Assert(node.Error.IsError);

            // Log the error.
            Push(node, BndErrorNode.Create(AddDiagnostic(node.Error)));
            SetNodeInfo(node);
        }

        protected override void VisitImpl(MissingValueNode node)
        {
            AssertValid();
            Validation.AssertValue(node);

            // Log an error and push a missing.
            Error(node, ErrorStrings.ErrOperandExpected);
            Push(node, BndMissingValueNode.Create(DType.Vac));
            SetNodeInfo(node);
        }

        protected override void VisitImpl(NullLitNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            Push(node, BndNullNode.Null);
            SetNodeInfo(node);
        }

        protected override void VisitImpl(BoolLitNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            Push(node, BndIntNode.CreateBit(node.Value));
            SetNodeInfo(node);
        }

        protected override void VisitImpl(NumLitNode node)
        {
            AssertValid();
            Validation.AssertValue(node);

            NumLitToken nlt = node.Value;
            Validation.Assert(nlt.Kind == TokKind.IntLit || nlt.Kind == TokKind.FltLit);

            if (nlt.Kind == TokKind.FltLit)
            {
                var flt = nlt.Cast<FltLitToken>();
                switch (flt.Size)
                {
                case NumLitSize.FourBytes:
                    var fltValue = (float)flt.Value;
                    // REVIEW: These overflow/underflow errors should probably be warnings.
                    // REVIEW: Removed these since there is no error / warning when the lexer produces 0 or infinity from
                    // something that is clearly not 0 or infinite, eg, 1e-400 or 1e400. It would be inconsistent to provide a
                    // diagnostic here when we don't in those cases.
                    //
                    // if (!NumUtil.IsFinite(fltValue) && NumUtil.IsFinite(flt.Value))
                    //     Error(node, ErrorStrings.ErrFloatOverflow);
                    // else if (fltValue == 0f && flt.Value != 0.0)
                    //     Error(node, ErrorStrings.ErrFloatUnderflow);
                    Push(node, BndFltNode.CreateR4(fltValue));
                    break;
                case NumLitSize.Unspecified:
                case NumLitSize.UnlimitedSize:
                case NumLitSize.EightBytes:
                    Push(node, BndFltNode.CreateR8(flt.Value));
                    break;
                default:
                    Error(node, ErrorStrings.ErrUnsupportedRatSuffix);
                    Push(node, BndFltNode.CreateR8(flt.Value));
                    break;
                }
            }
            else
            {
                var ilt = nlt.Cast<IntLitToken>();
                Validation.Assert(ilt.Value >= 0);

                var isUnsigned = (ilt.Flags & IntLitFlags.Unsigned) != 0;
                var isHexOrBin = (ilt.Flags & (IntLitFlags.Hex | IntLitFlags.Bin)) != 0;

                Integer value = ilt.Value;
                bool validSize;
                BoundNode res;
                switch (ilt.Size)
                {
                default:
                    Validation.Assert(ilt.Size == NumLitSize.Unspecified);
                    if (isUnsigned)
                    {
                        if (value <= ulong.MaxValue)
                            res = BndIntNode.CreateU8((ulong)value);
                        else
                        {
                            Warning(node, ErrorStrings.WrnUnsignedIntLiteralOverflow);
                            res = BndIntNode.CreateI(value);
                        }
                    }
                    else
                    {
                        if (value <= long.MaxValue)
                            res = BndIntNode.CreateI8((long)value);
                        else
                            res = BndIntNode.CreateI(value);
                    }
                    validSize = true;
                    break;

                case NumLitSize.OneByte:
                    validSize = isUnsigned || isHexOrBin ? value <= byte.MaxValue : value <= sbyte.MaxValue;
                    res = isUnsigned ? BndIntNode.CreateU1((byte)value.CastUlong()) : BndIntNode.CreateI1((sbyte)value.CastLong());
                    break;
                case NumLitSize.TwoBytes:
                    validSize = isUnsigned || isHexOrBin ? value <= ushort.MaxValue : value <= short.MaxValue;
                    res = isUnsigned ? BndIntNode.CreateU2((ushort)value.CastUlong()) : BndIntNode.CreateI2((short)value.CastLong());
                    break;
                case NumLitSize.FourBytes:
                    validSize = isUnsigned || isHexOrBin ? value <= uint.MaxValue : value <= int.MaxValue;
                    res = isUnsigned ? BndIntNode.CreateU4((uint)value.CastUlong()) : BndIntNode.CreateI4((int)value.CastLong());
                    break;
                case NumLitSize.EightBytes:
                    validSize = isUnsigned || isHexOrBin ? value <= ulong.MaxValue : value <= long.MaxValue;
                    res = isUnsigned ? BndIntNode.CreateU8(value.CastUlong()) : BndIntNode.CreateI8(value.CastLong());
                    break;
                case NumLitSize.UnlimitedSize:
                    Validation.Assert(!isUnsigned);
                    validSize = true;
                    res = BndIntNode.CreateI(value);
                    break;
                }
                if (!validSize)
                    Warning(node, ErrorStrings.WrnIntLiteralOutOfRange);
                Push(node, res);
            }

            SetNodeInfo(node);
        }

        protected override void VisitImpl(TextLitNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            Push(node, BndStrNode.Create(node.Value));
            SetNodeInfo(node);
        }

        protected override void VisitImpl(BoxNode node)
        {
            AssertValid();
            Validation.AssertValue(node);

            if (_boxValueStack.Depth <= 0)
            {
                PushError(node, null, ErrorStrings.ErrUnboundBox);
                SetNodeInfo(node);
                return;
            }

            var value = _boxValueStack.Value;
            int uses = _boxValueStack.Uses;
            Validation.AssertValue(value);
            Validation.Assert(uses >= 0);
            _boxValueStack = _boxValueStack.Use();

            if (uses > 0)
            {
                // REVIEW: Relax this requirement when we support value sharing.
                PushError(node, value.Type, ErrorStrings.ErrMultipleUseOfBox);
                SetNodeInfo(node);
                return;
            }

            Push(node, value);
            SetNodeInfo(node);
        }

        /// <summary>
        /// Returns the first module builder that contains the given scope wrapper. Returns
        /// <c>null</c> if there isn't none.
        /// </summary>
        private ModuleBuilder GetOuterModule(ScopeWrapper scope)
        {
            if (_moduleCur is null)
                return null;

            var moduleOuter = _moduleCur;
            for (var scopeCur = _scopeCur; ; scopeCur = scopeCur.Outer)
            {
                Validation.Assert(scopeCur is not null);
                if (scopeCur == scope)
                    return moduleOuter;
                if (scopeCur.Module is not null)
                {
                    Validation.Assert(moduleOuter == scopeCur.Module);
                    moduleOuter = moduleOuter.Outer;
                    if (moduleOuter is null)
                        return null;
                }
            }
        }

        /// <summary>
        /// Create a reference to the given scope wrapper. This should not be called with a module
        /// builder scope wrapper.
        /// </summary>
        private BoundNode MakeScopeRef(ScopeWrapper scope, bool index = false)
        {
            Validation.Assert(scope.Scope is not null);
            return MakeScopeRef(scope, GetOuterModule(scope), index);
        }

        /// <summary>
        /// Create a reference to the given scope wrapper. The <paramref name="moduleOuter"/> must be
        /// the closest containing module builder (may be null).
        /// </summary>
        private BoundNode MakeScopeRef(ScopeWrapper scope, ModuleBuilder moduleOuter, bool index = false)
        {
            Validation.Assert(scope.Module is null);
            Validation.Assert(moduleOuter == GetOuterModule(scope));

            var sc = index ? scope.IndexScope : scope.Scope;
            Validation.Assert(sc is not null);
            if (_moduleCur == moduleOuter)
                return BndScopeRefNode.Create(sc);
            // The scope is external to _moduleCur so must be referenced via _moduleCur's externals.
            return _moduleCur.GetScopeRefForExt(sc, moduleOuter);
        }

        /// <summary>
        /// Bind it[$slot].
        /// </summary>
        protected override void VisitImpl(ItNameNode node)
        {
            // This walks the scope chain. It assigns a "slot" to each active scope, regardless of
            // whether it is explicitly named. This is so introducing a name for a scope doesn't change
            // the meaning of an ItNameNode.
            AssertValid();
            Validation.AssertValue(node);
            Validation.Assert(node.UpCount >= 0);

            if (TryFindIt(node.UpCount, out var scope, out int up))
            {
                if (scope.Module is not null)
                    PushError(node, scope.Module.TypeCur, ErrorStrings.ErrBadModuleIt);
                else
                    Push(node, MakeScopeRef(scope));
            }
            else if (up > 0)
            {
                Validation.Assert(scope is not null);
                if (scope.Module is not null)
                    PushError(node, null, ErrorStrings.ErrBadItSlot);
                else
                {
                    Error(node, ErrorStrings.ErrBadItSlot);
                    Push(node, MakeScopeRef(scope));
                }
            }
            else
                PushError(node, null, ErrorStrings.ErrBadIt);

            SetNodeInfo(node);
        }

        private bool TryFindIt(int upCount, out ScopeWrapper scopeLast, out int up)
        {
            // This walks the scope chain. It assigns a "slot" to each active scope, regardless of
            // whether it is explicitly named. This is so introducing a name for a scope doesn't change
            // the meaning of an ItNameNode.
            AssertValid();
            Validation.Assert(upCount >= 0);

            scopeLast = null;
            up = 0;
            for (ScopeWrapper scope = _scopeCur; scope != _scopeRoot; scope = scope.Outer)
            {
                Validation.AssertValue(scope);
                Validation.AssertValue(scope.Outer);

                if (!scope.Enabled)
                    continue;

                scopeLast = scope;
                if (up == upCount)
                    return true;
                up++;
            }
            Validation.AssertIndexInclusive(up, upCount);
            return false;
        }

        private bool TryFindScopeFromName(DName name, bool guess, out ScopeWrapper scopeLast, out DName corrected)
        {
            AssertValid();
            Validation.Assert(name.IsValid);

            scopeLast = null;
            corrected = default;
            for (ScopeWrapper scope = _scopeCur; scope != _scopeRoot; scope = scope.Outer)
            {
                Validation.AssertValue(scope);
                Validation.AssertValue(scope.Outer);

                if (!scope.Enabled)
                    continue;

                scopeLast = scope;

                // Check for scope variable.
                var nameScope = scope.Variable;
                if (nameScope != null)
                {
                    if (nameScope == name)
                        return true;
                    // Fuzzy test.
                    if (guess && _host.IsFuzzyMatch(nameScope, name))
                    {
                        corrected = new DName(nameScope);
                        return true;
                    }
                }
                if (HasField(scope, name, guess, out corrected))
                    return true;
            }
            return false;
        }

        private bool TryFindIndexScope(int slotCount, out ScopeWrapper scopeLast, out int slot)
        {
            AssertValid();
            Validation.Assert(slotCount >= 0);

            scopeLast = null;
            slot = 0;

            ArgScope indexCur = null;
            for (ScopeWrapper scope = _scopeCur; scope != _scopeRoot; scope = scope.Outer)
            {
                Validation.AssertValue(scope);
                Validation.AssertValue(scope.Outer);

                if (!scope.Enabled || scope.IndexScope == indexCur)
                    continue;
                indexCur = scope.IndexScope;
                if (indexCur == null)
                    continue;

                scopeLast = scope;
                if (slot == slotCount)
                    return true;
                slot++;
            }
            Validation.Assert(0 <= slot & slot <= slotCount);
            return false;
        }

        protected override void VisitImpl(ThisNameNode node)
        {
            VisitThisCore(node);
        }

        private void VisitThisCore(RexlNode node)
        {
            Validation.AssertValue(node);
            Validation.Assert(node is ThisNameNode | node is FirstNameNode);
            Validation.Assert(node is FirstNameNode == (node.Token.AltFuzzy && node.Token.TokenAlt.Kind == TokKind.KwdThis));

            var type = _typeThis;
            if (!type.IsValid)
            {
                if (_host.TryGetThisType(out type))
                    Validation.Assert(type.IsValid);
                else
                {
                    Error(node, ErrorStrings.ErrInvalidThis);
                    type = DType.Vac;
                }
            }
            Push(node, BndThisNode.Create(type));
            SetNodeInfo(node);
        }

        /// <summary>
        /// Bind an initial (unqualified) identifier.
        /// </summary>
        protected override void VisitImpl(FirstNameNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            Validation.Assert(PeekSrc() == node);

            var ident = node.Ident;
            DName name = ident.Name;

            // REVIEW: Error reporting should be finer grained.
            for (int i = 0; i < 2; i++)
            {
                // Second time, use guesses (fuzzy matching).
                bool guess = i > 0;

                if (!ident.IsGlobal)
                {
                    ModuleBuilder moduleOuter = _moduleCur;
                    for (ScopeWrapper scope = _scopeCur; scope != _scopeRoot; scope = scope.Outer)
                    {
                        Validation.AssertValue(scope);
                        Validation.AssertValue(scope.Outer);

                        if (!scope.Enabled)
                        {
                            Validation.Assert(scope.Module is null);
                            continue;
                        }

                        if (scope.Module is not null)
                        {
                            Validation.Assert(moduleOuter == scope.Module);
                            moduleOuter = scope.Module.Outer;
                        }

                        // Check for scope variable.
                        bool varMatch = scope.Variable != null &&
                            (!guess ? scope.Variable == name : _host.IsFuzzyMatch(scope.Variable, name));

                        // If the scope variable isn't a match, or if the scope variable is implicit,
                        // look for a "field" of the scope.
                        if ((!varMatch || scope.VarIsImplicit) && HasField(scope, name, guess, out var corrected))
                        {
                            if (scope.Scope is not null)
                            {
                                // This handles field of opt and/or seq as well as the "property" case.
                                // See if the scope is external to the current module.
                                BoundNode src = Associate(node, MakeScopeRef(scope, moduleOuter));
                                LiftUnary(LiftKinds.SeqOpt, node, src, BindImplicitFieldAccessCore);
                            }
                            else
                            {
                                Validation.Assert(scope.Module is not null);

                                if (corrected.IsValid)
                                {
                                    Validation.Assert(guess);
                                    string strGuess = corrected.Escape();
                                    Validation.Coverage(strGuess.Contains('\'') ? 1 : 0);
                                    ErrorGuess(node, ErrorStrings.ErrNameDoesNotExist_Guess, strGuess, node.GetRange(), corrected);
                                    name = corrected;
                                }

                                scope.Module.TryFindSym(name, out var symFnd).Verify();
                                Validation.Assert(name == symFnd.Name);

                                // If this is a variable sym, report errors if we're currently building a constant value.
                                // The constant values include the domain and default for free variables. If we've reported
                                // a fuzzy error, don't bother with this.
                                if (symFnd.IsVariableSym && !corrected.IsValid)
                                {
                                    // None of the modules being bound should be binding exprs for constants
                                    // or free variables.
                                    for (var mod = _moduleCur; mod is not null; mod = mod.Outer)
                                    {
                                        switch (mod.SymCur.SymKind)
                                        {
                                        case ModSymKind.Constant:
                                            Error(node, ErrorStrings.ErrModuleConstCantReferenceVar);
                                            break;
                                        case ModSymKind.Parameter:
                                            Error(node, ErrorStrings.ErrModuleParamDefCantReferenceVar);
                                            break;
                                        case ModSymKind.FreeVariable:
                                            Error(node, ErrorStrings.ErrModuleVarDomCantReferenceVar);
                                            break;
                                        }
                                    }
                                }

                                var sr = _moduleCur.GetSymbolRef(scope.Module, symFnd);
                                Validation.Assert(sr.Type == symFnd.Type);
                                Push(node, sr);
                            }
                            SetNodeInfo(node);
                            return;
                        }

                        if (varMatch)
                        {
                            Validation.Assert(scope.Module is null);

                            // Matched the scope variable.
                            if (guess)
                            {
                                string strGuess = LexUtils.EscapeName(scope.Variable);
                                Validation.Coverage(strGuess.Contains('\'') ? 1 : 0);
                                ErrorGuess(node, ErrorStrings.ErrNameDoesNotExist_Guess, strGuess, node.GetRange(), scope.Variable);
                            }
                            else if (scope.VarIsImplicit)
                            {
                                _nodeToInfo.TryGetValue(scope.SrcImplicitName, out var info).Verify();
                                Validation.AssertValue(info);
                                if (!info.IsUsedImplicitName)
                                    _nodeToInfo = _nodeToInfo.SetItem(scope.SrcImplicitName, info.WithUsedImplicitName());
                            }

                            // This handles field of opt and/or seq as well as the "property" case.
                            // See if the scope is external to the current module.
                            BoundNode res = Associate(node, MakeScopeRef(scope, moduleOuter));

                            Push(node, res);
                            SetNodeInfo(node);
                            return;
                        }
                    }
                }

                if (!guess ? TryBindName(node) : TryBindNameFuzzy(node))
                {
                    SetNodeInfo(node);
                    return;
                }
            }

            // Guess if the identifier is a keyword.
            var tok = ident.Token;
            if (tok.AltFuzzy)
            {
                var tid = tok.TokenAlt.Kind;
                switch (tid)
                {
                case TokKind.KwdTrue:
                case TokKind.KwdFalse:
                    {
                        string rep = RexlLexer.Instance.GetFixedText(tid);
                        ErrorGuess(node, ErrorStrings.ErrNameDoesNotExist_Guess, rep, node.GetRange(), rep);
                        Push(node, BndIntNode.CreateBit(tid == TokKind.KwdTrue));
                        SetNodeInfo(node);
                        return;
                    }
                case TokKind.KwdThis:
                    if (_typeThis.IsValid || _host.TryGetThisType(out _))
                    {
                        string rep = RexlLexer.Instance.GetFixedText(tid);
                        ErrorGuess(node, ErrorStrings.ErrNameDoesNotExist_Guess, rep, node.GetRange(), rep);
                        VisitThisCore(node);
                        return;
                    }
                    break;
                case TokKind.KwdNull:
                    {
                        string rep = RexlLexer.Instance.GetFixedText(tid);
                        ErrorGuess(node, ErrorStrings.ErrNameDoesNotExist_Guess, rep, node.GetRange(), rep);
                        Push(node, BndNullNode.Null);
                        SetNodeInfo(node);
                        return;
                    }
                default:
                    break;
                }
            }

            PushError(node, null, ErrorStrings.ErrNameDoesNotExist);
            SetNodeInfo(node);
        }

        protected override void VisitImpl(MetaPropNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            Validation.Assert(PeekSrc() == node);

            var path = node.Left.FullName;
            var name = node.Right.Name;
            if (!_host.TryGetMetaProp(path, name, out var bnd))
            {
                if (!_host.IsNamespace(path))
                    PushError(node.Left.Last.Token, node, null, ErrorStrings.ErrMetaContainerUnknown);
                else
                    PushError(node.Right.Token, node, null, ErrorStrings.ErrMetaPropUnknown_Name, name.Value);
            }
            else
            {
                Validation.Assert(bnd != null);
                Validation.Assert(!bnd.IsProcCall);
                Push(node, bnd);
            }
            SetNodeInfo(node);
        }

        protected override void VisitImpl(GotoStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void VisitImpl(LabelStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(StmtListNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(ExprListNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            // Just set the scope information, since the children have already been pushed.
            SetNodeScope(node);
        }

        protected override void PostVisitImpl(SymListNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            // Just set the scope information, since the children have already been pushed.
            SetNodeScope(node);
        }

        protected override void PostVisitImpl(SliceListNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            // Just set the scope information, since the children have already been pushed.
            SetNodeScope(node);
        }

        protected override void PostVisitImpl(BlockStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(NamespaceStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(WithStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(IfStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(WhileStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(ExecuteStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(ImportStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(ExprStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(DefinitionStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(FuncStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(ParenNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            // This records the mapping between node and the bound node.
            Push(node, Pop());
            SetNodeInfo(node);
        }

        protected override void PostVisitImpl(DottedNameNode node)
        {
            BindDotted(node);
            SetNodeInfo(node);
        }

        protected override bool PreVisitImpl(GetIndexNode node)
        {
            Validation.AssertValue(node);

            if (node.ItChild != null)
            {
                Validation.Assert(node.NameChild == null);
                Validation.Assert(node.Slot < 0);
                var child = node.ItChild;

                if (TryFindIt(child.UpCount, out var scope, out int up))
                    PushIndexScope(node, child, scope);
                else
                {
                    SetNodeTypeAndScope(child, DType.Vac);
                    if (up > 0)
                    {
                        Validation.Assert(scope != null);
                        PushError(child.Token, node, DType.I8Req, ErrorStrings.ErrBadItSlot);
                    }
                    else
                        PushError(node, DType.I8Req, ErrorStrings.ErrBadIt);
                }
            }
            else if (node.NameChild != null)
            {
                Validation.Assert(node.Slot < 0);
                var child = node.NameChild;
                if (TryFindScopeFromName(child.Ident.Name, guess: false, out var scope, out _))
                    PushIndexScope(node, child, scope);
                else if (TryFindScopeFromName(child.Ident.Name, guess: true, out scope, out var corrected))
                {
                    Validation.Assert(corrected.IsValid);
                    string strGuess = corrected.Escape();
                    Validation.Coverage(strGuess.Contains('\'') ? 1 : 0);
                    ErrorGuess(node, ErrorStrings.ErrNameDoesNotExist_Guess, strGuess, node.GetRange(), corrected);
                    PushIndexScope(node, child, scope);
                }
                else
                {
                    SetNodeTypeAndScope(child, DType.Vac);
                    PushError(node, DType.I8Req, ErrorStrings.ErrNotActiveScope);
                }
            }
            else
            {
                Validation.Assert(node.Slot >= 0);
                if (TryFindIndexScope(node.Slot, out var scope, out int slot))
                    Push(node, MakeScopeRef(scope, index: true));
                else if (slot > 0)
                    PushError(node, DType.I8Req, ErrorStrings.ErrBadItIndSlot);
                else
                    PushError(node, DType.I8Req, ErrorStrings.ErrBadItInd);
            }
            SetNodeInfo(node);
            return false;
        }

        private void PushIndexScope(GetIndexNode node, RexlNode child, ScopeWrapper scope)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(child);
            Validation.Assert(child == node.ItChild ^ child == node.NameChild);
            Validation.AssertValue(scope);

            SetNodeTypeAndScope(child, DType.I8Req);
            if (scope.IndexScope != null)
                Push(node, MakeScopeRef(scope, index: true));
            else
                PushError(node, DType.I8Req, ErrorStrings.ErrNotIndexedScope);
        }

        protected override void PostVisitImpl(GetIndexNode node)
        {
            Validation.Assert(false);
        }

        protected override bool PreVisitImpl(UnaryOpNode node)
        {
            Validation.AssertValue(node);

            // Special case the negation of the min value of a fixed-sized signed
            // integer type, since normal processing will generate warnings.
            if (node.Op != UnaryOp.Negate)
                return true;
            if (!(node.Arg is NumLitNode nlt))
                return true;
            if (!(nlt.Value is IntLitToken ilt))
                return true;
            if ((ilt.Flags & (IntLitFlags.Unsigned | IntLitFlags.Hex)) != 0)
                return true;
            if (!ilt.Value.IsPowerOfTwo)
                return true;

            Validation.Assert(ilt.Value > 0);
            var val = -ilt.Value;
            DType type;
            switch (ilt.Size)
            {
            default:
                Validation.Assert(ilt.Size == NumLitSize.UnlimitedSize);
                return true;

            case NumLitSize.OneByte:
                if (val != sbyte.MinValue)
                    return true;
                type = DType.I1Req;
                break;
            case NumLitSize.TwoBytes:
                if (val != short.MinValue)
                    return true;
                type = DType.I2Req;
                break;
            case NumLitSize.FourBytes:
                if (val != int.MinValue)
                    return true;
                type = DType.I4Req;
                break;
            case NumLitSize.EightBytes:
            case NumLitSize.Unspecified:
                if (val != long.MinValue)
                    return true;
                type = DType.I8Req;
                break;
            }

            Push(node, BndIntNode.Create(type, val));
            SetNodeInfo(node.Arg);
            SetNodeInfo(node);
            return false;
        }

        protected override void PostVisitImpl(UnaryOpNode node)
        {
            Validation.AssertValue(node);
            BindUop(node);
            SetNodeInfo(node);
        }

        protected override bool PreVisitImpl(BinaryOpNode node)
        {
            Validation.AssertValue(node);
            if (node.Op == BinaryOp.Pipe && !BindPipePre(node))
            {
                // If BindPipePre(node) returns false, PostVisit on the node
                // won't be called. So we need to call SetNodeType here.
                SetNodeInfo(node);
                return false;
            }

            return true;
        }

        protected override void PostVisitImpl(BinaryOpNode node)
        {
            Validation.AssertValue(node);
            BindBop(node);
            SetNodeInfo(node);
        }

        protected override void PostVisitImpl(InHasNode node)
        {
            Validation.AssertValue(node);
            BindInHas(node);
            SetNodeInfo(node);
        }

        /// <summary>
        /// For comparison operators, we mimic C#. More precisely, for numeric x:
        /// * x < null, x <= null, x >= null, and x > null are all false.
        /// * x == null, x != null do null testing.
        /// </summary>
        protected override void PostVisitImpl(CompareNode node)
        {
            BindCompare(node);
            SetNodeInfo(node);
        }

        protected override bool PreVisitImpl(CallNode node)
        {
            AssertValid();
            Validation.AssertValue(node);

            SetNodeScope(node.Args);

            var info = CallInfo.Create(this, node);

            // Set the oper and traits for the CallNode.
            SetNodeRexlOper(node, info.Oper, info.Info, info.Traits);

            if (info.Oper is GroupByFunc gbf)
                BindGroupBy(node, gbf, info.Traits, info.First, info.ErrRoot);
            else if (info.Oper is SetFieldsFunc sff)
                BindSetFields(node, sff, info.Traits, info.First, info.ErrRoot);
            else
                info.Visit();

            SetNodeInfo(node);
            return false;
        }

        protected override void PostVisitImpl(CallNode node)
        {
            // Shouldn't get here - PreVisit should always return false.
            Validation.Assert(false);
            throw new InvalidOperationException("Internal binding error");
        }

        private void SetNodeRexlOper(CallNode node, RexlOper oper, OperInfo info, ArgTraits traits)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(oper);
            Validation.AssertValue(traits);
            Validation.Assert(traits.Oper == oper);
            Validation.Assert(!_callToOper.ContainsKey(node));
            _callToOper = _callToOper.SetItem(node, (info, traits));
        }

        private void BindGroupBy(CallNode call, GroupByFunc gbf, ArgTraits traits, BoundNode first, RexlDiagnostic err)
        {
            Validation.AssertValue(call);
            Validation.AssertValue(gbf);
            Validation.AssertValue(traits);
            Validation.Assert(traits.SlotCount >= 2);
            Validation.Assert(call.Args.Count == traits.SlotCount || err != null);
            Validation.AssertValueOrNull(first);
            Validation.Assert(err == null || err.IsError);

            var pargs = call.Args.Children;
            int cargRaw = pargs.Length;
            int carg = traits.SlotCount;
            Validation.Assert(cargRaw <= carg);

            // Process the first arg.
            BoundNode src;
            RexlNode nodeSrc;
            string variable = null;
            if (cargRaw > 0)
            {
                nodeSrc = call.Args.Children[0];
                if (nodeSrc is VariableDeclNode vdn)
                {
                    variable = vdn.Variable?.Name.Value;
                    nodeSrc = vdn.Value;
                }
                if (first != null)
                    src = first;
                else
                {
                    nodeSrc.Accept(this);
                    src = Pop();
                }
            }
            else
            {
                Validation.Assert(err != null);
                src = Associate(call.Args, BndMissingValueNode.Create(DType.Vac.ToSequence()));
                nodeSrc = call.Args;
            }

            DType typeSrc = src.Type;
            if (typeSrc.SeqCount == 0)
            {
                err = Error(nodeSrc, ErrorStrings.ErrNeedSequence_Slot_Func, 1, gbf.Name);
                typeSrc = typeSrc.ToSequence();
            }
            Validation.Assert(typeSrc.SeqCount > 0);

            DType typeItemSrc = typeSrc.ItemTypeOrThis;

            // Get the arg indices for the key, agg, and map items.
            var iargsKey = new List<int>();
            List<int> iargsAgg = null;
            List<(int index, Directive kind)> iargsMap = null;
            for (int iarg = 1; iarg < carg; iarg++)
            {
                // Determine the kind of argument: key, agg, map, auto, none.
                Directive kind = Directive.None;

                var parg = iarg < cargRaw ? pargs[iarg] : null;
                if (parg is DirectiveNode dn)
                {
                    switch (dn.Directive)
                    {
                    case Directive.Ci:
                    case Directive.Eq:
                    case Directive.EqCi:
                    case Directive.Key:
                        kind = Directive.Key;
                        break;
                    case Directive.Agg:
                    case Directive.Map:
                        kind = dn.Directive;
                        break;

                    case Directive.Auto:
                        kind = Directive.Auto;
                        if (!IsSimpleName(dn.Value, out _))
                        {
                            var errTmp = Error(dn.DirToken, dn, ErrorStrings.ErrBadAutoDirective);
                            err ??= errTmp;
                            kind = Directive.Map;
                        }
                        break;

                    default:
                        {
                            var errTmp = Error(dn.DirToken, dn, ErrorStrings.ErrBadDirective);
                            err ??= errTmp;
                        }
                        break;
                    }
                }

                if (kind == Directive.None && (iarg < carg - 1 || carg == 2))
                    kind = Directive.Key;

                switch (kind)
                {
                case Directive.Key:
                    iargsKey.Add(iarg);
                    break;
                case Directive.Agg:
                    Util.Add(ref iargsAgg, iarg);
                    break;
                default:
                    Validation.Assert(kind == Directive.Map | kind == Directive.Auto | kind == Directive.None);
                    Util.Add(ref iargsMap, (iarg, kind));
                    break;
                }
            }

            if (iargsKey.Count == 0)
            {
                var errTmp = Error(call, ErrorStrings.ErrGroupByNeedsKey);
                err ??= errTmp;
            }

            // The fields to add to the top level record.
            DType typeAdd = DType.EmptyRecordReq;
            // The fields to remove from the nested level record.
            DType typeDel = DType.EmptyRecordReq;

            // Create the scope.
            var scopeBase = _scopeCur;
            var swKey = PushScope(ScopeKind.SeqItem, typeItemSrc, ArgScope.CreateIndex(), variable);

            // Process the keys.
            var keysPure = ArgTuple.CreateBuilder(iargsKey.Count);
            var keysKeep = NamedItems.Empty;
            BitSet keysCi = default;
            HashSet<DName> namedKeysCi = null;
            foreach (int iarg in iargsKey)
            {
                BoundNode arg;
                RexlNode node;
                if (iarg < cargRaw)
                {
                    node = pargs[iarg];
                    node.Accept(this);
                    arg = Pop();
                }
                else
                {
                    Validation.Assert(err != null);
                    node = call.Args;
                    arg = Associate(node, BndMissingValueNode.Create(DType.I8Req));
                }

                // REVIEW: Handle ci keys.
                bool ci = false;
                var dn = node as DirectiveNode;
                if (dn is not null)
                {
                    node = dn.Value;
                    ci = dn.Directive.IsCi();
                }

                if (arg.HasVolatile)
                    Error(node, ErrorStrings.ErrBadVolatileKey_Func, gbf.Name);

                // Accept should leave the scope as it found it.
                Validation.Assert(_scopeCur == swKey);

                // Get the "value" to use as a key, the (optional) field name to assign it to, and
                // the (optional) field name to drop from the inner record type (if typeItemSrc is a record).
                Identifier fldAdd = null;
                Identifier fldSrc = null;
                BoundNode key = arg;
                if (node is VariableDeclNode vdn && vdn.Variable != null)
                {
                    fldAdd = vdn.Variable;
                    Validation.Assert(fldAdd.Token != null);
                    Validation.Assert(fldAdd.Name.IsValid);
                    fldSrc = GetSimpleName(vdn.Value);
                }
                else
                    fldSrc = GetSimpleName(node);

                if (!key.Type.IsEquatable)
                {
                    var errTmp = Error(node, ErrorStrings.ErrNotGroupableType_Type, key.Type);
                    if (err == null)
                        err = errTmp;
                }
                else if (ci && !key.Type.HasText)
                {
                    Validation.Assert(dn is not null);
                    WarningGuess(dn, ErrorStrings.WrnCmpCi_Type, "[key]", dn.DirToken.Range, key.Type);
                    ci = false;
                }

                // See if the value is a field to be dropped from the inner table.
                Token tokAdd = fldAdd?.Token;
                DName nameAdd = fldAdd != null ? fldAdd.Name : default;
                DName nameDel = default;
                if (fldSrc != null && typeItemSrc.IsRecordXxx &&
                    BndGetFieldNode.IsScopeField(key, swKey.Scope, out DName nameFldDel))
                {
                    Validation.Assert(typeItemSrc.Contains(nameFldDel));
                    Validation.Assert(fldSrc.Name == nameFldDel ||
                        _host.IsFuzzyMatch(fldSrc.Name, nameFldDel) && _hasErrors);
                    if (fldAdd == null)
                    {
                        tokAdd = fldSrc.Token;
                        nameAdd = nameFldDel;
                    }
                    nameDel = nameFldDel;
                }

                if (!nameAdd.IsValid)
                {
                    Validation.Assert(!nameDel.IsValid);
                    if (ci)
                        keysCi = keysCi.SetBit(keysPure.Count);
                    keysPure.Add(key);
                }
                else
                {
                    Validation.Assert(tokAdd != null);

                    if (typeAdd.Contains(nameAdd))
                    {
                        var errTmp = Error(tokAdd, node, ErrorStrings.ErrDuplicateFieldName_Name, nameAdd);
                        err ??= errTmp;
                    }
                    else
                    {
                        typeAdd = typeAdd.AddNameType(nameAdd, key.Type);
                        keysKeep = keysKeep.SetItem(nameAdd, key);
                        if (ci)
                        {
                            // Case insensitive matching. Note that we don't drop the name in the auto result,
                            // to avoid losing information.
                            Util.Add(ref namedKeysCi, nameAdd);
                        }
                        else if (nameDel.IsValid)
                        {
                            bool tmp = typeDel.TryGetNameType(nameDel, out DType typeTmp);
                            Validation.Assert(!tmp | typeTmp == key.Type);
                            if (!tmp)
                                typeDel = typeDel.AddNameType(nameDel, key.Type);
                        }
                    }
                }
            }
            Validation.Assert(typeItemSrc.IsRecordXxx || typeDel.FieldCount == 0);
            Validation.Assert(typeAdd.FieldCount > 0 || typeDel.FieldCount == 0);

            if (namedKeysCi is not null)
            {
                foreach (var name in namedKeysCi)
                {
                    keysKeep.TryGetValue(name, out var b, out int index).Verify();
                    keysCi = keysCi.SetBit(index + keysPure.Count);
                }
            }

            Validation.Assert(_scopeCur == swKey);
            PopScope();
            ArgScope scopeKey = swKey.Scope;
            ArgScope indexKey = swKey.IndexScope;
            Validation.Assert(scopeKey.Kind == ScopeKind.SeqItem);
            Validation.Assert(indexKey.Kind == ScopeKind.SeqIndex);
            Validation.Assert(_scopeCur == scopeBase);

            DType typeAuto = typeItemSrc;
            if (typeDel.FieldCount > 0)
            {
                // The type is the original item type with some fields dropped. Note that opt-ness
                // is maintained.
                Validation.Assert(typeAuto.IsRecordXxx);
                foreach (var tn in typeDel.GetNames())
                    typeAuto = typeAuto.DropName(tn.Name);
            }

            int cagg = Util.Size(iargsAgg);
            int cmap = Util.Size(iargsMap);

            // Process the auto and map items. A null value in maps indicates auto.
            ArgScope scopeMap = null;
            ArgScope indexMap = null;
            var maps = NamedItems.EmptyOptBoth;
            if (cmap > 0)
            {
                indexMap = ArgScope.CreateIndex();
                var swMap = PushScope(ScopeKind.SeqItem, typeItemSrc, indexMap, variable ?? "item");

                foreach (var (iarg, kind) in iargsMap)
                {
                    // If there aren't enough args provided, then the args can't be map or agg.
                    Validation.Assert(iarg < cargRaw);

                    var nodeFull = pargs[iarg];
                    var node = nodeFull;
                    if (node is DirectiveNode dn)
                        node = dn.Value;

                    DName name;
                    BoundNode value;
                    DType typeItem;

                    if ((kind == Directive.Auto || kind != Directive.Map && (typeAdd.FieldCount > 0 || typeItemSrc.IsRecordXxx)) && IsSimpleName(node, out var fnn))
                    {
                        // An auto slot. Note that if typeItemSrc isn't a record type, then typeAuto is the same as typeItemSrc. Otherwise, it
                        // consists of a subset of the fields of typeItemSrc.
                        Validation.Assert(typeAuto.IsRecordXxx == typeItemSrc.IsRecordXxx);
                        Validation.Assert(typeAuto == typeItemSrc | typeAuto.IsRecordXxx);
                        name = fnn.Ident.Name;
                        value = null;
                        typeItem = typeAuto;
                        // We don't call Accept here, so we need to record the scope & type information explicitly. Note that recording the type
                        // information is a bit dubious, but is none the less useful.
                        SetNodeTypeAndScope(node, typeItem);
                        if (node != nodeFull)
                            SetNodeTypeAndScope(nodeFull, typeItem);
                    }
                    else
                    {
                        // If the kind was auto, the call to IsSimpleName should have succeeded.
                        Validation.Assert(kind == Directive.Map | kind == Directive.None);

                        // The nested item value is explicitly specified. In this case, typeDel is irrelevant.
                        nodeFull.Accept(this);
                        var arg = Pop();

                        // Accept should leave the scope as it found it.
                        Validation.Assert(_scopeCur == swMap);

                        // Get the "value" to use as the nested item, and the (optional) field name to use.
                        value = arg;
                        if (node is VariableDeclNode vdn && vdn.Variable != null)
                            name = vdn.Variable.Name;
                        else if (typeAdd.FieldCount > 0 || cmap + cagg > 1)
                        {
                            if (IsImplicitName(node, true, out var ident))
                                name = ident.Name;
                            else
                            {
                                var errTmp = Error(node, ErrorStrings.ErrNeedFieldName_Slot_Func, iarg + 1, gbf.Name);
                                err ??= errTmp;
                                name = default(DName);
                            }
                        }
                        else
                            name = default(DName);
                        typeItem = value.Type;
                    }

                    if (name.IsValid)
                    {
                        if (typeAdd.Contains(name))
                        {
                            var errTmp = Error(node, ErrorStrings.ErrDuplicateFieldName_Name, name);
                            err ??= errTmp;
                        }

                        // In the error case, we force the field type, hence Set instead of Add.
                        typeAdd = typeAdd.SetNameType(name, typeItem.ToSequence());
                    }

                    maps = maps.SetItem(name, value);
                }

                scopeMap = PopScope().Scope;
                Validation.Assert(scopeMap.Kind == ScopeKind.SeqItem);
            }
            // Note that duplicate names (including multiple nameless) will cause maps.Count to
            // be less than cmap.
            Validation.Assert(maps.Count <= cmap);
            Validation.Assert(_scopeCur == scopeBase);

            // Process the agg items.
            ArgScope scopeAgg = null;
            var aggs = NamedItems.EmptyOptName;
            if (cagg > 0)
            {
                var swAgg = PushScope(ScopeKind.With, typeSrc, null, "group");

                foreach (int iarg in iargsAgg)
                {
                    // If there aren't enough args provided, then the args can't be map or agg.
                    Validation.Assert(iarg < cargRaw);

                    var node = pargs[iarg];
                    node.Accept(this);
                    var arg = Pop();
                    if (node is DirectiveNode dn)
                        node = dn.Value;

                    // Accept should leave the scope as it found it.
                    Validation.Assert(_scopeCur == swAgg);

                    // Get the value, and the (optional) field name.
                    DName name;
                    BoundNode value = arg;
                    Token tokName = null;
                    if (node is VariableDeclNode vdn && vdn.Variable != null)
                    {
                        name = vdn.Variable.Name;
                        tokName = vdn.Variable.Token;
                    }
                    else if (typeAdd.FieldCount > 0 || cmap + cagg > 1)
                    {
                        if (IsImplicitName(node, true, out var ident))
                            name = ident.Name;
                        else
                        {
                            var errTmp = Error(node, ErrorStrings.ErrNeedFieldName_Slot_Func, iarg + 1, gbf.Name);
                            err ??= errTmp;
                            name = default(DName);
                        }
                    }
                    else
                        name = default(DName);

                    if (name.IsValid)
                    {
                        if (typeAdd.Contains(name))
                        {
                            var errTmp = tokName != null ?
                                Error(tokName, node, ErrorStrings.ErrDuplicateFieldName_Name, name) :
                                Error(node, ErrorStrings.ErrDuplicateFieldName_Name, name);
                            err ??= errTmp;
                        }

                        // In the error case, we force the field type, hence Set instead of Add.
                        typeAdd = typeAdd.SetNameType(name, value.Type);
                    }

                    aggs = aggs.SetItem(name, value);
                }

                scopeAgg = PopScope().Scope;
                Validation.Assert(scopeAgg.Kind == ScopeKind.With);
            }
            Validation.Assert(_scopeCur == scopeBase);

            // Determine the final type.
            DType typeOuter;
            if (typeAdd.FieldCount > 0)
                typeOuter = typeAdd.ToSequence();
            else
            {
                // No field names specified. Either there should be an error or at most one agg/map.
                Validation.Assert(err != null || maps.Count + aggs.Count <= 1);
                if (maps.Count + aggs.Count == 0)
                {
                    // Just group the input items.
                    typeOuter = typeSrc.ToSequence();
                }
                else if (maps.Count > 0)
                {
                    Validation.Assert(maps.FirstVal != null);
                    typeOuter = maps.FirstVal.Type.ToSequence(2);
                }
                else
                    typeOuter = aggs.FirstVal.Type.ToSequence();
            }

            BoundNode res;
            if (err != null)
            {
                // There was at least one error, just produce an error node.
                res = BndErrorNode.Create(err, typeOuter);
            }
            else
            {
                res = BndGroupByNode.Create(
                    typeOuter, src,
                    scopeKey, indexKey, keysPure.ToImmutable(), keysKeep, keysCi,
                    scopeMap, indexMap, maps,
                    scopeAgg, aggs);
            }

            Push(call, res);
        }

        private void BindSetFields(CallNode call, SetFieldsFunc sff, ArgTraits traits, BoundNode first, RexlDiagnostic err)
        {
            Validation.AssertValue(call);
            Validation.AssertValue(sff);
            Validation.AssertValue(traits);
            Validation.Assert(traits.SlotCount >= 2);
            Validation.Assert(traits.ScopeCount == 1);
            Validation.Assert(traits.GetScopeKind(0) == ScopeKind.With);
            Validation.Assert(call.Args.Count == traits.SlotCount || err != null);
            Validation.Assert(call.Args.Count <= traits.SlotCount);
            Validation.AssertValueOrNull(first);
            Validation.Assert(err == null || err.IsError);

            // Get the first arg.
            if (first == null)
            {
                if (call.Args.Count > 0)
                {
                    call.Args.Children[0].Accept(this);
                    first = Pop();
                }
                else
                {
                    Validation.Assert(err != null);
                    first = Associate(call.Args, BndMissingValueNode.Create(DType.EmptyRecordReq));
                }
            }

            // Deal with lifting.
            DType typeSrc = first.Type;
            if (typeSrc.IsSequence || typeSrc.IsTensorXxx || typeSrc.HasReq)
                LiftUnary(LiftKinds.SeqTenOpt, call, first, (n, s) => BindSetFieldsCore(n, sff, traits, s, err));
            else
                BindSetFieldsCore(call, sff, traits, first, err);
        }

        private void BindSetFieldsCore(CallNode call, SetFieldsFunc sff, ArgTraits traits, BoundNode src, RexlDiagnostic err)
        {
            Validation.AssertValue(call);
            Validation.AssertValue(sff);
            Validation.AssertValue(traits);
            Validation.Assert(traits.SlotCount >= 2);
            Validation.Assert(traits.ScopeCount == 1);
            Validation.Assert(traits.GetScopeKind(0) == ScopeKind.With);
            Validation.Assert(call.Args.Count == traits.SlotCount || err != null);
            Validation.Assert(call.Args.Count <= traits.SlotCount);
            Validation.AssertValue(src);
            Validation.Assert(err == null || err.IsError);

            var pargs = call.Args.Children;
            int cargRaw = pargs.Length;
            Validation.Assert(cargRaw <= traits.SlotCount);

            var nodeSrc = cargRaw > 0 ? pargs[0] : null;
            if (nodeSrc is DirectiveNode dn)
            {
                Error(dn.DirToken, dn, ErrorStrings.ErrBadDirective);
                nodeSrc = dn.Value;
            }

            string variable = null;
            ExprNode impName = null;
            if (nodeSrc is VariableDeclNode vdnSrc)
            {
                variable = vdnSrc.Variable?.Name.Value;
                nodeSrc = vdnSrc.Value;
            }
            else if (call.TokPipe == null && IsImplicitName(nodeSrc, dotted: false, out var ident))
            {
                variable = ident.Name.Value;
                impName = nodeSrc;
            }

            var typeSrc = src.Type;
            if (!typeSrc.IsRecordReq)
            {
                CheckGeneralType(nodeSrc ?? call, ref src, DType.EmptyRecordReq);
                typeSrc = src.Type;
            }
            Validation.Assert(typeSrc.IsRecordReq);

            // Create the scope.
            var scopeBase = _scopeCur;

            var sw = PushScope(ScopeKind.With, typeSrc, null, variable, impName);

            var res = BindSetFieldsCore2(sff, src, call.Args, 1, sw);

            ArgScope scope = PopScope().Scope;
            Validation.Assert(scope.Kind == ScopeKind.With);
            Validation.Assert(_scopeCur == scopeBase);

            Push(call, res);
        }

        private BoundNode BindSetFieldsCore2(SetFieldsFunc sff, BoundNode src, ExprListNode nodes, int slotMin, ScopeWrapper sw)
        {
            var pargs = nodes.Children;
            var typeSrc = src.Type;
            var typeRet = typeSrc;
            var typeDel = DType.EmptyRecordReq;
            var flds = NamedItems.Empty;
            int slotLim = pargs.Length;
            var nameHints = NameTuple.Empty;
            if (slotMin < slotLim)
            {
                var bldrNames = NameTuple.CreateBuilder(slotLim - slotMin);
                for (int slot = slotMin; slot < slotLim; slot++)
                {
                    var node = pargs[slot];
                    node.Accept(this);
                    var arg = Pop();

                    // Accept should leave the scope as it found it.
                    Validation.Assert(_scopeCur == sw);

                    var nodeCur = node;
                    if (node is DirectiveNode dn)
                    {
                        Error(dn.DirToken, dn, ErrorStrings.ErrBadDirective);
                        node = dn.Value;
                    }

                    Identifier identAdd = null;
                    if (node is VariableDeclNode vdn)
                    {
                        identAdd = vdn.Variable;
                        if (identAdd != null)
                            node = vdn.Value;
                    }
                    else if (IsImplicitName(node, dotted: true, out var ident))
                        identAdd = ident;

                    if (identAdd == null)
                    {
                        if (!arg.HasErrors)
                            Error(node, ErrorStrings.ErrNeedName_Slot_Func, slot + 1, sff.Name);
                        continue;
                    }

                    DName nameFld = identAdd.Name;
                    if (flds.ContainsKey(nameFld))
                    {
                        Error(identAdd.Token, nodeCur, ErrorStrings.ErrDuplicateFieldName_Name, nameFld);
                        continue;
                    }

                    DName nameDrop;
                    if (node.Kind == NodeKind.NullLit)
                    {
                        // Note: null means drop the field.
                        if (!typeSrc.Contains(nameFld))
                        {
                            // REVIEW: Make this a warning?
                            Error(identAdd.Token, nodeCur, ErrorStrings.ErrUnknownFieldNameForDrop);
                            continue;
                        }
                        nameDrop = nameFld;
                    }
                    else
                    {
                        // Capture the value and adjust the types.
                        flds = flds.SetItem(nameFld, arg);
                        typeRet = typeRet.SetNameType(nameFld, arg.Type);
                        bldrNames.Add(nameFld);

                        // If this is a rename, process the dropped field.
                        if (!sff.IsRename || !BndGetFieldNode.IsScopeField(arg, sw.Scope, out nameDrop))
                            continue;
                        Validation.Assert(typeSrc.Contains(nameDrop));

                        // Make sure that we aren't already setting the field.
                        if (flds.ContainsKey(nameDrop))
                            continue;

                        // The field should be dropped when the user code says "B : A", but not when they say
                        // "B : 1 * A" or "B : --A", etc. So we have to look back at the parse tree to see the
                        // precise form.
                        if (!(node is FirstNameNode fnn) || fnn.Ident.IsGlobal)
                            continue;
                    }

                    // Drop it, if it hasn't been already.
                    if (!typeDel.Contains(nameDrop))
                    {
                        typeDel = typeDel.AddNameType(nameDrop, DType.Null);
                        typeRet = typeRet.DropName(nameDrop);
                    }
                }
                nameHints = bldrNames.ToImmutable();
            }

            return BndSetFieldsNode.Create(typeRet, src, sw.Scope, flds, nameHints);
        }

        /// <summary>
        /// Returns true iff the given parse node is a simple name.
        /// </summary>
        private bool IsImplicitName(ExprNode node, bool dotted, out Identifier ident)
        {
            Validation.AssertValueOrNull(node);
            if (node is FirstNameNode pfn && !pfn.Ident.IsGlobal)
            {
                ident = pfn.Ident;
                return true;
            }

            if (dotted && node is DottedNameNode dnn)
            {
                ident = dnn.Right;
                return true;
            }

            ident = null;
            return false;
        }

        /// <summary>
        /// Returns true iff the given parse node is a simple name.
        /// </summary>
        private bool IsSimpleName(ExprNode node, out FirstNameNode fnn)
        {
            Validation.AssertValueOrNull(node);
            if (node is FirstNameNode pfn && !pfn.Ident.IsGlobal)
            {
                fnn = pfn;
                return true;
            }

            fnn = null;
            return false;
        }

        /// <summary>
        /// Returns the identifier if the given parse node is a simple name, otherwise returns null.
        /// </summary>
        private Identifier GetSimpleName(RexlNode node)
        {
            Validation.AssertValueOrNull(node);
            if (node is FirstNameNode fnn && !fnn.Ident.IsGlobal)
                return fnn.Ident;
            return null;
        }

        protected override void PostVisitImpl(SliceItemNode node)
        {
            // Shouldn't get here - PreVisit(IndexingNode) should handle these directly.
            Validation.Assert(false);
            throw new InvalidOperationException("Internal binding error");
        }

        /// <summary>
        /// Get the <see cref="IndexFlags"/> value from the associated parse tokens.
        /// </summary>
        private static IndexFlags GetIndexFlags(SliceItemNode item)
        {
            Validation.AssertValueOrNull(item);

            if (item == null)
                return 0;

            IndexFlags flags = 0;
            if (item.StartBack != null)
                flags |= IndexFlags.Back;
            if (item.StartEdge != null)
            {
                if (item.StartEdge.Kind == TokKind.Per)
                    flags |= IndexFlags.Wrap;
                else if (item.StartEdge.Kind == TokKind.Amp)
                    flags |= IndexFlags.Clip;
                else
                    Validation.Assert(false, "Unexpected edge mode token kind");
            }
            return flags;
        }

        protected override bool PreVisitImpl(IndexingNode node)
        {
            Validation.AssertValue(node);

            int cchd = 0;
            node.Child.Accept(this);
            cchd++;

            // This deconstructs the SliceItemNodes, adopting their children into the
            // "args" associated with this node.
            BitSet slotsNotOpt = default;
            var items = node.Items.Children;
            for (int i = 0; i < items.Length; i++)
            {
                var item = items[i];
                if (item.IsSimple)
                {
                    Validation.Assert(item.Start != null);
                    Validation.Assert(item.Stop == null);
                    Validation.Assert(item.Step == null);
                    item.Start.Accept(this);
                    cchd++;
                }
                else
                {
                    Validation.Assert(item.StartEdge == null);

                    // Get the start/stop/step components. For these, we don't lift over opt, since
                    // null is meaningful for them.
                    if (item.Start != null)
                    {
                        item.Start.Accept(this);
                        slotsNotOpt = slotsNotOpt.SetBit(cchd);
                        cchd++;
                    }
                    if (item.Stop != null)
                    {
                        item.Stop.Accept(this);
                        slotsNotOpt = slotsNotOpt.SetBit(cchd);
                        cchd++;
                    }
                    if (item.Step != null)
                    {
                        item.Step.Accept(this);
                        slotsNotOpt = slotsNotOpt.SetBit(cchd);
                        cchd++;
                    }
                }

                // The item isn't an ExprNode, so just set its scope.
                SetNodeScope(item);
            }

            // node.Items isn't an ExprNode, so just set its scope.
            SetNodeScope(node.Items);

            var args = PopMulti(cchd);
            Lift(LiftKinds.SeqOpt, node, args, BindIndexingCore, slotsNotOpt: slotsNotOpt);

            SetNodeInfo(node);
            return false;
        }

        protected override void PostVisitImpl(IndexingNode node)
        {
            // Shouldn't get here - PreVisit should always return false.
            Validation.Assert(false);
            throw new InvalidOperationException("Internal binding error");
        }

        protected override void PostVisitImpl(VariableDeclNode node)
        {
            Validation.AssertValue(node);
            SetNodeInfo(node);
        }

        protected override void PostVisitImpl(DirectiveNode node)
        {
            Validation.AssertValue(node);
            SetNodeInfo(node);
        }

        protected override void PostVisitImpl(IfNode node)
        {
            Validation.AssertValue(node);

            var bndFalse = Pop();
            var bndCond = Pop();
            var bndTrue = Pop();

            // REVIEW: Should the condition type allow opt? If not, the message should guide the user to
            // consider using ?? to disambiguate.
            CheckGeneralType(node.Condition, ref bndCond, DType.BitReq, DType.UseUnionOper,
                ErrorStrings.ErrIfNeedsBoolCondition_Type_Type, DType.BitReq, bndCond.Type);

            DType typeRes = GetSuperType(node, bndTrue.Type, bndFalse.Type);
            CheckGeneralType(node.TrueValue, ref bndTrue, typeRes);
            CheckGeneralType(node.FalseValue, ref bndFalse, typeRes);

            Push(node, BndIfNode.Create(bndCond, bndTrue, bndFalse));
            SetNodeInfo(node);
        }

        protected override void PostVisitImpl(RecordNode node)
        {
            Validation.AssertValue(node);

            int count = node.Items.Children.Length;
            Reverse(count);
            var type = DType.EmptyRecordReq;
            var items = NamedItems.Empty;
            var bldrNames = NameTuple.CreateBuilder(count);
            for (int i = 0; i < count; i++)
            {
                var nodeCur = node.Items.Children[i];
                var val = Pop();
                Identifier idCur;
                if (nodeCur is VariableDeclNode vdn && vdn.Variable != null)
                    idCur = vdn.Variable;
                else
                {
                    // The parser guarantees this assertion.
                    Validation.Assert(nodeCur.Kind == NodeKind.FirstName || nodeCur.Kind == NodeKind.DottedName);
                    idCur = nodeCur.Kind == NodeKind.FirstName ? nodeCur.Cast<FirstNameNode>().Ident : nodeCur.Cast<DottedNameNode>().Right;
                    _nodeToInfo.TryGetValue(nodeCur, out var info).Verify();
                    if (!info.IsUsedImplicitName)
                        _nodeToInfo = _nodeToInfo.SetItem(nodeCur, info.WithUsedImplicitName());
                }

                if (type.TryGetNameType(idCur.Name, out DType typeFld))
                    Error(idCur.Token, node, ErrorStrings.ErrDuplicateFieldName_Name, idCur.Name);
                else
                {
                    type = type.AddNameType(idCur.Name, val.Type);
                    items = items.SetItem(idCur.Name, val);
                    bldrNames.Add(idCur.Name);
                }
            }

            Validation.Assert(type.FieldCount == items.Count);
            Push(node, BndRecordNode.Create(type, items, bldrNames.ToImmutable()));
            SetNodeInfo(node);
        }

        protected override void PostVisitImpl(SequenceNode node)
        {
            Validation.AssertValue(node);

            bool union = DType.UseUnionOper;

            var bldr = PopToBuilder(node.Items.Count);
            DType type;
            if (bldr.Count == 0)
            {
                // When the count is zero, the item type is Vac.
                type = DType.Vac;
            }
            else
            {
                type = bldr[0].Type;
                for (int i = 1; i < bldr.Count; i++)
                    type = GetSuperType(node.Items.Children[i], type, bldr[i].Type, union);
            }

            for (int i = 0; i < bldr.Count; i++)
            {
                Validation.Assert(type.Accepts(bldr[i].Type, union));
                bldr[i] = CastGeneral(node, bldr[i], type, union);
            }

            Push(node, BndSequenceNode.Create(type.ToSequence(), bldr.ToImmutable()));
            SetNodeInfo(node);
        }

        protected override void PostVisitImpl(TupleNode node)
        {
            Validation.AssertValue(node);
            var bldr = PopToBuilder(node.Items.Count);
            Push(node, BndTupleNode.Create(bldr.ToImmutable()));
            SetNodeInfo(node);
        }

        private void Project(ExprNode node, ExprNode source, ExprNode value)
        {
            Validation.AssertValue(node);
            Validation.AssertValue(source);
            Validation.AssertValue(value);

            // Lift the Guard function.
            source.Accept(this);
            var first = Pop();
            LiftUnary(LiftKinds.Seq, node, first,
                (n, src) => CallInfo.Create(this, n, NodeTuple.Create(source, value), WithFunc.Guard, src).Visit());
        }

        protected override bool PreVisitImpl(RecordProjectionNode node)
        {
            Validation.AssertValue(node);

            if (!node.IsConcat)
            {
                Project(node, node.Source, node.Record);
                SetNodeInfo(node);
                return false;
            }

            bool visitedRecord = false;
            node.Source.Accept(this);
            LiftUnary(LiftKinds.SeqOpt, node, Pop(), (n, src) =>
            {
                var typeSrc = src.Type;
                Validation.Assert(typeSrc.SeqCount == 0);
                Validation.Assert(!typeSrc.HasReq);

                var pargs = n.Record.Items.Children;
                int count = pargs.Length;

                BoundNode res;
                ScopeWrapper sw = PushScope(ScopeKind.With, src.Type);
                var scopes = ScopeTuple.Create(sw.Scope);
                if (typeSrc.RootKind != DKind.Record)
                {
                    // The src isn't a record, so it's an error. Don't concat.
                    Error(n.Source, ErrorStrings.ErrNotRecord);
                    visitedRecord = true;
                    n.Record.Accept(this);
                    res = Pop();
                    res = BndCallNode.Create(WithFunc.With, res.Type, ArgTuple.Create(src, res), scopes);
                }
                else
                    res = BindSetFieldsCore2(SetFieldsFunc.SetFields, src, n.Record.Items, 0, sw);

                Validation.Assert(_scopeCur == sw);
                PopScope();

                Push(node, res);
            });

            if (!visitedRecord)
            {
                SetNodeScope(node.Record.Items);
                SetNodeScope(node.Record);
            }

            SetNodeInfo(node);
            return false;
        }

        protected override void PostVisitImpl(RecordProjectionNode node)
        {
            // Shouldn't get here - PreVisit should always return false.
            Validation.Assert(false);
            throw new InvalidOperationException("Internal binding error");
        }

        protected override bool PreVisitImpl(TupleProjectionNode node)
        {
            Validation.AssertValue(node);

            if (!node.IsConcat)
            {
                Project(node, node.Source, node.Tuple);
                SetNodeInfo(node);
                return false;
            }

            node.Source.Accept(this);
            LiftUnary(LiftKinds.SeqOpt, node, Pop(), (n, src) =>
            {
                var typeSrc = src.Type;
                Validation.Assert(typeSrc.SeqCount == 0);
                Validation.Assert(!typeSrc.HasReq);

                // If the src isn't a tuple, it's an error. In that case, don't concat.
                bool isTup = typeSrc.RootKind == DKind.Tuple;
                if (!isTup)
                    Error(node.Source, ErrorStrings.ErrNotTuple);

                bool isTopScope = src is BndScopeRefNode bsr && _scopeCur.Scope == bsr.Scope;
                var sw = !isTopScope ? PushScope(ScopeKind.With, src.Type) : null;
                node.Tuple.Accept(this);
                var tup = Pop();

                BoundNode res = tup;
                if (sw != null)
                {
                    PopScope();
                    if (isTup)
                        res = Associate(node, BndVariadicOpNode.CreateTupleConcat(BndScopeRefNode.Create(sw.Scope), res));
                    res = BndCallNode.Create(WithFunc.With, res.Type, ArgTuple.Create(src, res), ScopeTuple.Create(sw.Scope));
                }
                else if (isTup)
                    res = BndVariadicOpNode.CreateTupleConcat(src, res);

                Push(node, res);
            });

            SetNodeInfo(node);
            return false;
        }

        protected override void PostVisitImpl(TupleProjectionNode node)
        {
            // Shouldn't get here - PreVisit should always return false.
            Validation.Assert(false);
            throw new InvalidOperationException("Internal binding error");
        }

        protected override bool PreVisitImpl(ValueProjectionNode node)
        {
            Validation.AssertValue(node);

            Project(node, node.Source, node.Value);
            SetNodeInfo(node);
            return false;
        }

        protected override void PostVisitImpl(ValueProjectionNode node)
        {
            // Shouldn't get here - PreVisit should always return false.
            Validation.Assert(false);
            throw new InvalidOperationException("Internal binding error");
        }

        protected override void PostVisitImpl(ValueSymDeclNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            // REVIEW: Although this isn't a statement node, it should only be contained
            // in a statement node (at this point).
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(FreeVarDeclNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            // REVIEW: Although this isn't a statement node, it should only be contained
            // in a statement node (at this point).
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(TaskCmdStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(TaskProcStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(TaskBlockStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }

        protected override void PostVisitImpl(UserProcStmtNode node)
        {
            AssertValid();
            Validation.AssertValue(node);
            ThrowOnStmt();
        }
    }

#if DEBUG
    /// <summary>
    /// This validates the _nodeToInfo dictionary generated during binding. It ensures that all nodes
    /// in the parse tree have an entry in the dictionary and that ExprNodes have types, except in the
    /// rare exceptions.
    /// </summary>
    private sealed class NodeInfoVerifier : NoopTreeVisitor
    {
        private readonly NodeToInfo _nodeToInfo;
        private int _count;

        private NodeInfoVerifier(NodeToInfo nodeToInfo) : base()
        {
            _nodeToInfo = nodeToInfo;
            _count = 0;
        }

        public static void Run(RexlNode tree, NodeToInfo nodeToInfo)
        {
            Validation.AssertValue(tree);
            var verifier = new NodeInfoVerifier(nodeToInfo);
            tree.Accept(verifier);

            // We should visit the same number of nodes as are in the dictionary.
            Validation.Assert(verifier._count == nodeToInfo.Count);
        }

        protected override void VisitCore(RexlNode node)
        {
            Validation.Assert(node == PeekSrc());

            if (!_nodeToInfo.TryGetValue(node, out var info))
            {
                // If this assert goes off, the binder didn't properly set type/scope information
                // for this node.
                Validation.Assert(false);
                return;
            }

            _count++;

            // Non-expr nodes should not have a type.
            if (!(node is ExprNode))
            {
                Validation.Assert(!info.Type.IsValid);
                Validation.Assert(info.BoundNode == null);
                return;
            }

            bool expectType = true;
            if (!info.Type.IsValid)
            {
                // An ExprNode should have a type unless it is one of these cases:
                switch (node.Kind)
                {
                case NodeKind.Record:
                    // * value+>{ <fields> }, the RecordNode won't have a type.
                    switch (PeekSrc(1))
                    {
                    case RecordProjectionNode rpn:
                        if (rpn.Record == node && rpn.IsConcat)
                            expectType = false;
                        break;
                    case ModuleProjectionNode mpn:
                        expectType = false;
                        break;
                    }
                    break;
                }
                Validation.Assert(!expectType);
            }

            // If an ExprNode does not have a bound node, it should be one of these cases:
            // * The node has no type.
            // * The node's parent is a GetIndexNode.
            // * The node is a FirstNameNode and has a GroupBy CallNode parent or [auto] DirectiveNode parent that has a GroupBy CallNode parent.
            // * The node is an [auto] DirectiveNode with a GroupBy CallNode parent and a FirstNameNode child.

            if (info.BoundNode != null)
                return;
            if (!expectType)
                return;
            if (PeekSrc(1) is GetIndexNode)
                return;

            Validation.Assert(expectType);
            int isrc = 0;
            switch (node)
            {
            default:
                Validation.Assert(false);
                return;
            case FirstNameNode fnn:
                Validation.Assert(!fnn.Ident.IsGlobal);
                isrc++;
                break;
            case DirectiveNode dn:
                Validation.Assert(dn.Value is FirstNameNode);
                break;
            case SymbolDeclNode sdn:
                return;
            }

            if (PeekSrc(isrc) is DirectiveNode d)
            {
                if (!(PeekSrc(isrc + 1) is ExprListNode eln))
                {
                    Validation.Assert(false);
                    return;
                }

                var kindDir = d.Directive;
                if (kindDir != Directive.Auto)
                {
                    Validation.Assert(kindDir != Directive.Key);
                    Validation.Assert(kindDir != Directive.Agg);
                    Validation.Assert(kindDir != Directive.Map);
                    Validation.Assert(eln.Count > 2 && eln.Children[eln.Count - 1] == d);
                }
                isrc += 2;
            }
            else
            {
                if (!(PeekSrc(isrc) is ExprListNode))
                {
                    Validation.Assert(false);
                    return;
                }
                isrc++;
            }

            Validation.Assert(PeekSrc(isrc) is CallNode cn && cn.IdentPath.FullName == NPath.Root.Append(new DName("GroupBy")));
        }

        protected override void PostVisitCore(RexlNode node)
        {
            VisitCore(node);
        }
    }
#endif
}
