﻿// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.

using Microsoft.Rexl.Bind;
using Microsoft.Rexl.Private;

namespace Microsoft.Rexl;

/// <summary>
/// Produces the type string. This is fully resolved at Bind time.
/// </summary>
public sealed partial class GetTypeFunc : RexlOper
{
    public static readonly GetTypeFunc Instance = new GetTypeFunc();

    private GetTypeFunc()
        : base(isFunc: true, new DName("GetType"), 1, 1)
    {
    }

    protected override ArgTraits GetArgTraitsCore(int carg)
    {
        Validation.Assert(SupportsArity(carg));
        return ArgTraitsSimple.Create(this, eager: false, carg);
    }

    protected override (DType, Immutable.Array<DType>) SpecializeTypesCore(InvocationInfo info)
    {
        Validation.AssertValue(info);
        Validation.Assert(info.Arity == 1);

        return (DType.Text, Immutable.Array.Create(info.Args[0].Type));
    }

    protected override bool CertifyCore(BndCallNode call, ref bool full)
    {
        if (call.Type != DType.Text)
            return false;
        // Fully reduced.
        full = false;
        return true;
    }

    protected override BoundNode ReduceCore(IReducer reducer, BndCallNode call)
    {
        Validation.AssertValue(reducer);
        Validation.Assert(IsValidCall(call));

        return BndStrNode.Create(call.Args[0].Type.Serialize());
    }
}

/// <summary>
/// Produces a string with information about the bound node. This is fully resolved at Bind time.
/// </summary>
public sealed partial class GetBindInfoFunc : RexlOper
{
    public static readonly GetBindInfoFunc Instance = new GetBindInfoFunc();

    private GetBindInfoFunc()
        : base(isFunc: true, new DName("GetBindInfo"), 1, 1)
    {
    }

    protected override ArgTraits GetArgTraitsCore(int carg)
    {
        Validation.Assert(SupportsArity(carg));
        return ArgTraitsSimple.Create(this, eager: false, carg);
    }

    protected override (DType, Immutable.Array<DType>) SpecializeTypesCore(InvocationInfo info)
    {
        Validation.AssertValue(info);
        Validation.Assert(info.Arity == 1);

        return (DType.Text, Immutable.Array.Create(info.Args[0].Type));
    }

    protected override bool CertifyCore(BndCallNode call, ref bool full)
    {
        if (call.Type != DType.Text)
            return false;
        // Fully reduced.
        full = false;
        return true;
    }

    protected override BoundNode ReduceCore(IReducer reducer, BndCallNode call)
    {
        Validation.AssertValue(reducer);
        Validation.Assert(IsValidCall(call));

        // This seizes the argument bound node, labels the scopes, converts to a string, and tosses
        // the bound node, so it isn't part of the resulting bound tree.
        // REVIEW: Ideally, it would also toss or seize any errors generated by binding the argument.
        var bnd = call.Args[0];

        string res = string.Format("BoundNode: {0}\n     Type: {1}", bnd.ToString(), bnd.Type.Serialize());

        return BndStrNode.Create(res);
    }
}
