﻿// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.

using System;
using System.Reflection;

using Microsoft.Rexl.Private;

namespace Microsoft.Rexl.Code;

/// <summary>
/// Base class for synthesized record and tuple classes.
/// </summary>
public abstract class AggBase
{
    // Max possible arity is limited by the max numbers of type parameters and members in a class.
    // These seem to be 2^16 - 1. The precise limits don't seem to be documented.
    // However, generating the code for the `IEquatable<>` methods takes a very long time when the arity
    // is large, and there doesn't seem to be a compelling reason to support extremely large arities.
    //
    // REVIEW: The time for code gen seems to be O(n^2) rather than the expected O(n). Why is that?
    // See the durations of the `RecordSizes` and `TupleSizes` tests. 1000 seems to take about 100 ms, with
    // every doubling multiplying that time by 4.
    public const int ArityMax = 1_000;

    /// <summary>
    /// This is a cache of the hash value. It is zero if the hash hasn't been computed yet.
    /// </summary>
    private int _hash;

    /// <summary>
    /// This is a cache of the case insensitive hash value. It is zero if the hash hasn't been computed yet.
    /// This is set by the case insensitive equality comparer for the specific agg type. The code that sets
    /// it is generated by the type manager, which is why this is public.
    /// REVIEW: Is there a better way that doesn't force this to be public?
    /// </summary>
    public int _hashCi;

    private protected AggBase()
    {
    }

    public sealed override int GetHashCode()
    {
        int hash = _hash;
        if (hash != 0)
            return hash;

        hash = GetHashCodeCore();

        // The core method should ensure that the result isn't zero.
        Validation.Assert(hash != 0);
        return _hash = hash;
    }

    protected abstract int GetHashCodeCore();

    public sealed override bool Equals(object obj) => EqualsCore(obj);

    protected abstract bool EqualsCore(object obj);

    /// <summary>
    /// This is used to clone a "default instance" of an agg to be used as the result of building a new
    /// instance. If you are tempted to use it for any other purpose, you shouldn't.
    /// </summary>
    public AggBase Clone()
    {
        var res = (AggBase)MemberwiseClone();
        // Since the clone will be modified, clear the hash caches.
        res._hash = 0;
        res._hashCi = 0;
        return res;
    }
}

/// <summary>
/// This is the base class for all tuple sytem types generated by the type manager.
/// </summary>
public abstract class TupleBase : AggBase
{
    protected TupleBase()
        : base()
    {
    }
}

/// <summary>
/// This is the base class for all record sytem types generated by the type manager.
/// </summary>
public abstract class RecordBase : AggBase
{
    public static readonly FieldInfo FinRrti =
        typeof(RecordBase).GetField("_rrti", BindingFlags.Public | BindingFlags.Instance).VerifyValue();

    public RecordRuntimeTypeInfo _rrti;

    public RecordRuntimeTypeInfo Rrti => _rrti;

    protected RecordBase()
        : base()
    {
    }

    /// <summary>
    /// Get the <see cref="DType"/> for this instance. Note that the bind time type may have been looser,
    /// allowing more null fields than this result.
    /// </summary>
    public DType GetDType()
    {
        Validation.BugCheck(Rrti is not null);
        Validation.Assert(Rrti.TypeReq.IsRecordReq);

        var nulls = GetNullBits();
        return Rrti.TypeReq.SetFieldOpts(nulls);
    }

    /// <summary>
    /// Gets a bit set indicating which fields have a <c>null</c> value where the field type also has
    /// a required form. To emphasize, the bits for always-opt types like text, uri, sequence, etc,
    /// will be clear (not set).
    /// </summary>
    public BitSet GetNullBits()
    {
        Validation.AssertValue(_rrti);

        int cb = _rrti.ReqFieldBits.Length;
        Validation.Assert(cb <= ((ArityMax + 7) >> 3));

        Span<byte> flags = stackalloc byte[cb];
        FillFlags(flags);

        var req = _rrti.ReqFieldBits;
        for (int i = 0; i < cb; i++)
            flags[i] = (byte)(~flags[i] & req[i]);

        int len = flags.Length;
        while (len > 0 && flags[len - 1] == 0)
            len--;

        return new BitSet(flags.Slice(0, len));
    }

    /// <summary>
    /// Fill a span with the flags byte values.
    /// </summary>
    protected abstract void FillFlags(Span<byte> span);
}

/// <summary>
/// This is the base class for record runtime type information.
/// </summary>
public abstract class RecordRuntimeTypeInfo
{
    public readonly DType TypeReq;
    public readonly Immutable.Array<byte> ReqFieldBits;

    /// <summary>
    /// The record system type.
    /// </summary>
    public abstract Type RecSysType { get; }

    protected RecordRuntimeTypeInfo(DType type)
    {
        Validation.Assert(type.IsRecordXxx);
        TypeReq = type.GetReqFieldType();

        // REVIEW: Optimize this.
        int arity = TypeReq.FieldCount;
        int cb = (arity + 7) >> 3;
        var bldr = Immutable.Array<byte>.CreateBuilder(cb, init: true);
        foreach (var (name, typeFld, idx) in TypeReq.GetFields())
        {
            Validation.Assert(!typeFld.HasReq);
            Validation.AssertIndex(idx, arity);
            if (!typeFld.IsOpt)
                bldr[idx >> 3] |= (byte)(1 << (idx & 7));
        }
        ReqFieldBits = bldr.ToImmutable();
    }
}
