// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

package org.bondlib;

import org.bondlib.test.*;
import org.junit.Test;

import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.util.Map;
import java.util.AbstractMap;

import org.junit.Assert;

public class DeserializerControlsTest {

    @Test(expected = IllegalArgumentException.class)
    public void testInvalidSetMaxDepthArguments() {
        DeserializerControls.setMaxDepth(-1);
    }

    @Test
    public void testSetMaxDepthArgumentsRountrip() {
        Assert.assertEquals(64, DeserializerControls.getMaxDepth());

        DeserializerControls.setMaxDepth(1);
        Assert.assertEquals(1, DeserializerControls.getMaxDepth());

        DeserializerControls.setMaxDepth(64);
        Assert.assertEquals(64, DeserializerControls.getMaxDepth());
    }

    @Test(expected = InvalidBondDataException.class)
    public void test_BadPayload_RecursionViaNesting_Throws() throws IOException
    {
        // Manually-constructed payload with a nesting depth of n
        int n = 100;
        byte[] bytes = new byte[8 * n];
        for (int i = 0; i < n; i++) bytes[8 * i + 4] = 1;

        SimpleBinaryReader reader = new SimpleBinaryReader(new ByteArrayInputStream(bytes), 1);
        new Deserializer<StructWithRecursiveReference>(StructWithRecursiveReference.BOND_TYPE).deserialize(reader);
    }

    @Test
    public void test_BadPayload_RecursionViaSkip_FastBinary_Throws() throws IOException
    {
        byte[][] payloads = new byte[][] {
            // Depth 130, BT_STRUCT
            new byte[] { (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x0A, (byte)0x32, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00 },
            // Depth 130, BT_LIST
            new byte[] { (byte)0x0B, (byte)0x32, (byte)0x00, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x64, (byte)0x00 },
            // Depth 130, BT_SET
            new byte[] { (byte)0x0C, (byte)0x32, (byte)0x00, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x64, (byte)0x00 },
            // Depth 130, BT_MAP
            new byte[] { (byte)0x0D, (byte)0x32, (byte)0x00, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x32, (byte)0x64, (byte)0x00 },
        };

        for (byte[] payload : payloads) {
            try
            {
                new Deserializer<NestedWithBase1>(NestedWithBase1.BOND_TYPE).deserialize(new FastBinaryReader(new ByteArrayInputStream(payload), 1));
                Assert.assertTrue(false);
            }
            catch (InvalidBondDataException e) { }
        }
    }
    
    @Test
    public void test_BadPayload_RecursionViaSkip_CompactBinary_Throws() throws IOException
    {
        Object[] payloads = new Object[] {
            // Depth 130, BT_STRUCT, Protocol v1
            new AbstractMap.SimpleEntry<Integer,byte[]>(1, new byte[] { (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0xCA, (byte)0x32, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00 }),
            // Depth 130, BT_LIST, Protocol v1
            new AbstractMap.SimpleEntry<Integer,byte[]>(1, new byte[] { (byte)0xCB, (byte)0x32, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x0B, (byte)0x01, (byte)0x64, (byte)0x00 }),
            // Depth 130, BT_LIST, Protocol v2
            new AbstractMap.SimpleEntry<Integer,byte[]>(2, new byte[] { (byte)0x86, (byte)0x01, (byte)0xCB, (byte)0x32, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x4B, (byte)0x64, (byte)0x00 }),
            // Depth 130, BT_SET, Protocol v1
            new AbstractMap.SimpleEntry<Integer,byte[]>(1, new byte[] { (byte)0xCC, (byte)0x32, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x0C, (byte)0x01, (byte)0x64, (byte)0x00 }),
            // Depth 130, BT_SET, Protocol v2
            new AbstractMap.SimpleEntry<Integer,byte[]>(2, new byte[] { (byte)0x86, (byte)0x01, (byte)0xCC, (byte)0x32, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x4C, (byte)0x64, (byte)0x00 }),
            // Depth 130, BT_MAP, Protocol v1
            new AbstractMap.SimpleEntry<Integer,byte[]>(1, new byte[] { (byte)0xCD, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x32, (byte)0x64, (byte)0x00 }),
            // Depth 130, BT_MAP, Protocol v2
            new AbstractMap.SimpleEntry<Integer,byte[]>(2, new byte[] { (byte)0x8D, (byte)0x04, (byte)0xCD, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x0E, (byte)0x0D, (byte)0x01, (byte)0x32, (byte)0x32, (byte)0x64, (byte)0x00 }), 
        };

        for (Object obj : payloads) {
            AbstractMap.SimpleEntry<Integer,byte[]> payload = (AbstractMap.SimpleEntry<Integer,byte[]>)obj;
            try
            {
                new Deserializer<NestedWithBase1>(NestedWithBase1.BOND_TYPE).deserialize(new CompactBinaryReader(new ByteArrayInputStream(payload.getValue()), payload.getKey()));
                Assert.assertTrue(false);
            }
            catch (InvalidBondDataException e) { }
        }
    }

    @Test
    public void test_BadPayload_LargeContainerInvalidTypes_FastBinary_Throws() throws IOException
    {
        byte[][] payloads = new byte[][] {
            new byte[] { (byte)0x0b, (byte)0x04, (byte)0x00, (byte)0x1e, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff },             // BT_LIST<0x1e>
            new byte[] { (byte)0x0b, (byte)0x04, (byte)0x00, (byte)0x8d, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff },             // BT_LIST<0x8d>
            new byte[] { (byte)0x0c, (byte)0x04, (byte)0x00, (byte)0x1e, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff },             // BT_SET<0x1e>
            new byte[] { (byte)0x0c, (byte)0x04, (byte)0x00, (byte)0x8d, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff },             // BT_SET<0x8d>
            new byte[] { (byte)0x0d, (byte)0x04, (byte)0x00, (byte)0x1e, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff }, // BT_MAP<0x1e, BT_FLOAT>
            new byte[] { (byte)0x0d, (byte)0x04, (byte)0x00, (byte)0x8d, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff }, // BT_MAP<0x8d, BT_FLOAT>
            new byte[] { (byte)0x0d, (byte)0x04, (byte)0x00, (byte)0x07, (byte)0x1e, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff }, // BT_MAP<BT_FLOAT, 0x1e>
            new byte[] { (byte)0x0d, (byte)0x04, (byte)0x00, (byte)0x07, (byte)0x8d, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff }, // BT_MAP<BT_FLOAT, 0x8d>
        };

        for (byte[] payload : payloads) {
            try
            {
                new Deserializer<NestedStruct>(NestedStruct.BOND_TYPE).deserialize(new FastBinaryReader(new ByteArrayInputStream(payload), 1));
                Assert.assertTrue(false);
            }
            catch (InvalidBondDataException e) { }
            catch (EOFException e) { }
        }
    }

    @Test
    public void test_BadPayload_LargeContainerInvalidTypes_CompactBinary_Throws() throws IOException
    {
        byte[][] payloads = new byte[][] {
            new byte[] { (byte)0x8b, (byte)0x1e, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff },             // BT_LIST<0x1e>
            new byte[] { (byte)0x8b, (byte)0x8d, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff },             // BT_LIST<0x8d>
            new byte[] { (byte)0x8c, (byte)0x1e, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff },             // BT_SET<0x1e>
            new byte[] { (byte)0x8c, (byte)0x8d, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff },             // BT_SET<0x8d>
            new byte[] { (byte)0x8d, (byte)0x1e, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff }, // BT_MAP<0x1e, BT_FLOAT>
            new byte[] { (byte)0x8d, (byte)0x8d, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff }, // BT_MAP<0x8d, BT_FLOAT>
            new byte[] { (byte)0x8d, (byte)0x07, (byte)0x1e, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff }, // BT_MAP<BT_FLOAT, 0x1e>
            new byte[] { (byte)0x8d, (byte)0x07, (byte)0x8d, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0x07, (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xff }, // BT_MAP<BT_FLOAT, 0x8d>
        };

        for (int protocol = 1; protocol <= 2; protocol++)
        {
            for (byte[] payload : payloads) {
                try
                {
                    new Deserializer<NestedStruct>(NestedStruct.BOND_TYPE).deserialize(new CompactBinaryReader(new ByteArrayInputStream(payload), protocol));
                    Assert.assertTrue(false);
                }
                catch (InvalidBondDataException e) { }
                catch (EOFException e) { }
            }
        }
    }
}