// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

package org.bondlib;

import java.util.HashSet;

/**
 * Contains settings for generation of structs using the {@link StructGenerator} class.
 */
public final class StructGeneratorSettings {

    /**
     * The maximum allowed depth of recursive structs, where a struct contains (either directly or indirectly)
     * an instance of the same struct. Only structs of the same type with the same generic type parameters are
     * considered against this threshold.
     */
    public int maxRecursiveStructDepth = 4;

    /**
     * The maximum length of a randomly generated container. The lengths of generated containers are uniformly
     * distributed between 0 and this value. Once a length is determined, the container is generated by adding
     * a sequence of generated values.
     */
    public int maxGeneratedContainerLength = 15;

    /**
     * The maximum length of a randomly generated string in Unicode code points. The lengths of generated strings
     * (in terms of Unicode code points) are uniformly distributed between 0 and this value. Once a length is
     * determined, the string is generated as a sequence of pseudo-random valid Unicode code points.
     */
    public int maxGeneratedStringLength = 10;

    /**
     * The probability of an optional field being set to a value, as opposed to being left with the default.
     * This covers both explicit defaults (including nothing) and implicit defaults for all optional fields.
     */
    public double probabilityOfSettingOptionalField = 0.75;

    /**
     * The probability of a nullable value being assigned null. This setting is used for all nullable values,
     * whether struct fields or collection elements. For optional nullable fields, the probability of the field
     * set to null is conditional on the probability of that field set to any value at all, which is determined
     * from the {@link #probabilityOfSettingOptionalField} setting.
     */
    public double probabilityOfValueAssignedToNull = 0.25;

    /**
     * The probability that a generated floating point value is coming from a pool of special-case floating point
     * values, as opposed to being randomly generated. The special pool includes values that cannot be generated
     * randomly, such as infinities and NaNs values.
     */
    public double probabilityOfSpecialFloatingPointValue = 0.50;

    /**
     * A collection of struct type descriptors that are considered when generating values of bonded containers.
     * For each generated bonded value, a struct type is selected randomly from the union of two sets:
     * (1) the set of all types present in this collection that can be assigned to the type within the bonded
     * and (2) the exact type within the bonded.
     */
    public final HashSet<StructBondType> polymorphicBondTypes = new HashSet<StructBondType>();
}
