# Copyright 2013-2020 Lawrence Livermore National Security, LLC and other
# Spack Project Developers. See the top-level COPYRIGHT file for details.
#
# SPDX-License-Identifier: (Apache-2.0 OR MIT)

"""Utilities for managing paths in Spack.

TODO: this is really part of spack.config. Consolidate it.
"""
import os
import re
import getpass
import subprocess
import tempfile

import llnl.util.tty as tty
from llnl.util.lang import memoized

import spack.paths


__all__ = [
    'substitute_config_variables',
    'substitute_path_variables',
    'canonicalize_path']

# Substitutions to perform
replacements = {
    'spack': spack.paths.prefix,
    'user': getpass.getuser(),
    'tempdir': tempfile.gettempdir(),
}

# This is intended to be longer than the part of the install path
# spack generates from the root path we give it.  Included in the
# estimate:
#
#   os-arch      ->   30
#   compiler     ->   30
#   package name ->   50   (longest is currently 47 characters)
#   version      ->   20
#   hash         ->   32
#   buffer       ->  138
#  ---------------------
#   total        ->  300
SPACK_MAX_INSTALL_PATH_LENGTH = 300
SPACK_PATH_PADDING_CHARS = 'spack_path_placeholder'


@memoized
def get_system_path_max():
    # Choose a conservative default
    sys_max_path_length = 256
    try:
        path_max_proc  = subprocess.Popen(['getconf', 'PATH_MAX', '/'],
                                          stdout=subprocess.PIPE,
                                          stderr=subprocess.STDOUT)
        proc_output = str(path_max_proc.communicate()[0].decode())
        sys_max_path_length = int(proc_output)
    except (ValueError, subprocess.CalledProcessError, OSError):
        tty.msg('Unable to find system max path length, using: {0}'.format(
            sys_max_path_length))

    return sys_max_path_length


def substitute_config_variables(path):
    """Substitute placeholders into paths.

    Spack allows paths in configs to have some placeholders, as follows:

    - $spack     The Spack instance's prefix
    - $user      The current user's username
    - $tempdir   Default temporary directory returned by tempfile.gettempdir()

    These are substituted case-insensitively into the path, and users can
    use either ``$var`` or ``${var}`` syntax for the variables.

    """
    # Look up replacements for re.sub in the replacements dict.
    def repl(match):
        m = match.group(0).strip('${}')
        return replacements.get(m.lower(), match.group(0))

    # Replace $var or ${var}.
    return re.sub(r'(\$\w+\b|\$\{\w+\})', repl, path)


def substitute_path_variables(path):
    """Substitute config vars, expand environment vars, expand user home."""
    path = substitute_config_variables(path)
    path = os.path.expandvars(path)
    path = os.path.expanduser(path)
    return path


def _get_padding_string(length):
    spack_path_padding_size = len(SPACK_PATH_PADDING_CHARS)
    num_reps = int(length / (spack_path_padding_size + 1))
    extra_chars = length % (spack_path_padding_size + 1)
    reps_list = [SPACK_PATH_PADDING_CHARS for i in range(num_reps)]
    reps_list.append(SPACK_PATH_PADDING_CHARS[:extra_chars])
    return os.path.sep.join(reps_list)


def _add_computed_padding(path):
    """Subtitute in padding of os-specific length.  The intent is to leave
    SPACK_MAX_INSTALL_PATH_LENGTH characters available for parts of the
    path generated by spack.  This is to allow for not-completely-known
    lengths of things like os/arch, compiler, package name, hash length,
    etc.
    """
    padding_regex = re.compile(r'(\$[\w\d\:]+\b|\$\{[\w\d\:]+\})')
    m = padding_regex.search(path)
    if m and m.group(0).strip('${}').startswith('padding'):
        padding_part = m.group(0)
        len_pad_part = len(m.group(0))
        p_match = re.search(r'\:(\d+)', padding_part)
        if p_match:
            computed_padding = _get_padding_string(int(p_match.group(1)))
        else:
            # Take whatever has been computed/substituted so far and add some
            # room
            path_len = len(path) - len_pad_part + SPACK_MAX_INSTALL_PATH_LENGTH
            system_max_path = get_system_path_max()
            needed_pad_len = system_max_path - path_len
            computed_padding = _get_padding_string(needed_pad_len)
        return padding_regex.sub(computed_padding, path)
    return path


def canonicalize_path(path):
    """Same as substitute_path_variables, but also take absolute path."""
    path = substitute_path_variables(path)
    path = os.path.abspath(path)
    path = _add_computed_padding(path)

    return path
