---
title: Installation
description: 'Learn how to install and configure Nuxt UI in your Vue application, compatible with both plain Vite and Inertia.'
framework: vue
links:
  - label: Template
    to: https://codesandbox.io/p/devbox/nuxt-ui-vue-f2qp34
    icon: i-lucide-codesandbox
navigation.icon: i-lucide-square-play
---

::callout{to="/docs/getting-started/installation/nuxt" icon="i-logos-nuxt-icon" class="hidden"}
Looking for the **Nuxt** version?
::

## Setup

### Add to a Vue project

::steps{level="4"}

#### Install the Nuxt UI package

::code-group{sync="pm"}

```bash [pnpm]
pnpm add @nuxt/ui
```

```bash [yarn]
yarn add @nuxt/ui
```

```bash [npm]
npm install @nuxt/ui
```

```bash [bun]
bun add @nuxt/ui
```

::

::warning
If you're using **pnpm**, ensure that you either set [`shamefully-hoist=true`](https://pnpm.io/npmrc#shamefully-hoist) in your `.npmrc` file or install `tailwindcss`, `vue-router` and `@unhead/vue` in your project's root directory.
::

#### Add the Nuxt UI Vite plugin in your `vite.config.ts`{lang="ts-type"}

::code-group{sync="vite"}

```ts [vite.config.ts (Vite)]{3,8}
import { defineConfig } from 'vite'
import vue from '@vitejs/plugin-vue'
import ui from '@nuxt/ui/vite'

export default defineConfig({
  plugins: [
    vue(),
    ui()
  ]
})
```

```ts [vite.config.ts (Laravel Inertia)]{3,20-22}
import { defineConfig } from 'vite'
import vue from '@vitejs/plugin-vue'
import ui from '@nuxt/ui/vite'
import laravel from 'laravel-vite-plugin'

export default defineConfig({
  plugins: [
    laravel({
      input: ['resources/js/app.ts'],
      refresh: true
    }),
    vue({
      template: {
        transformAssetUrls: {
          base: null,
          includeAbsolute: false
        }
      }
    }),
    ui({
      inertia: true
    })
  ]
})
```

```ts [vite.config.ts (AdonisJS Inertia)]{3,15-17}
import { defineConfig } from 'vite'
import vue from '@vitejs/plugin-vue'
import ui from '@nuxt/ui/vite'
import adonisjs from '@adonisjs/vite/client'
import inertia from '@adonisjs/inertia/client'

export default defineConfig({
  plugins: [
    adonisjs({
      entrypoints: ['inertia/app/app.ts'],
      reload: ['resources/views/**/*.edge']
    }),
    inertia(),
    vue(),
    ui({
      inertia: true
    })
  ]
})
```

::

::tip
Nuxt UI registers `unplugin-auto-import` and `unplugin-vue-components`, which will generate `auto-imports.d.ts` and `components.d.ts` type declaration files. You will likely want to gitignore these, and add them to your `tsconfig`.

```json [tsconfig.app.json]
{
  "include": ["src/**/*.ts", "src/**/*.tsx", "src/**/*.vue", "auto-imports.d.ts", "components.d.ts"]
}
```

```bash [.gitignore]
# Auto-generated type declarations
auto-imports.d.ts
components.d.ts
```

::

::tip
Internally, Nuxt UI relies on custom alias to resolve the theme types. If you're using TypeScript, you should add an alias to your `tsconfig` to enable auto-completion in your `vite.config.ts`.

```json [tsconfig.node.json]
{
  "compilerOptions": {
    "paths": {
      "#build/ui": [
        "./node_modules/@nuxt/ui/.nuxt/ui"
      ]
    }
  }
}
```
::

#### Use the Nuxt UI Vue plugin

::code-group{sync="vite"}

```ts [src/main.ts (Vite)]{3,14}
import { createApp } from 'vue'
import { createRouter, createWebHistory } from 'vue-router'
import ui from '@nuxt/ui/vue-plugin'
import App from './App.vue'

const app = createApp(App)

const router = createRouter({
  routes: [],
  history: createWebHistory()
})

app.use(router)
app.use(ui)

app.mount('#app')
```

```ts [resources/js/app.ts (Laravel Inertia)]{3,19}
import type { DefineComponent } from 'vue'
import { createInertiaApp } from '@inertiajs/vue3'
import ui from '@nuxt/ui/vue-plugin'
import { resolvePageComponent } from 'laravel-vite-plugin/inertia-helpers'
import { createApp, h } from 'vue'

const appName = import.meta.env.VITE_APP_NAME || 'Laravel x Nuxt UI'

createInertiaApp({
  title: title => (title ? `${title} - ${appName}` : appName),
  resolve: name =>
    resolvePageComponent(
      `./pages/${name}.vue`,
      import.meta.glob<DefineComponent>('./pages/**/*.vue')
    ),
  setup({ el, App, props, plugin }) {
    createApp({ render: () => h(App, props) })
      .use(plugin)
      .use(ui)
      .mount(el)
  }
})
```

```ts [inertia/app/app.ts (AdonisJS Inertia)]{3,19}
import type { DefineComponent } from 'vue'
import { createInertiaApp } from '@inertiajs/vue3'
import ui from '@nuxt/ui/vue-plugin'
import { resolvePageComponent } from '@adonisjs/inertia/helpers'
import { createApp, h } from 'vue'

const appName = import.meta.env.VITE_APP_NAME || 'AdonisJS x Nuxt UI'

createInertiaApp({
  title: title => (title ? `${title} - ${appName}` : appName),
  resolve: name =>
    resolvePageComponent(
      `../pages/${name}.vue`,
      import.meta.glob<DefineComponent>('../pages/**/*.vue')
    ),
  setup({ el, App, props, plugin }) {
    createApp({ render: () => h(App, props) })
      .use(plugin)
      .use(ui)
      .mount(el)
  }
})
```

::

#### Import Tailwind CSS and Nuxt UI in your CSS

::code-group{sync="vite"}

```css [assets/main.css (Vite)]
@import "tailwindcss";
@import "@nuxt/ui";
```

```css [resources/css/app.css (Laravel Inertia)]
@import "tailwindcss";
@import "@nuxt/ui";
```

```css [inertia/css/app.css (AdonisJS Inertia)]
@import "tailwindcss";
@import "@nuxt/ui";
```

::

::tip
Import the CSS file in your entrypoint.

  ::code-group{sync="vite"}

  ```ts [src/main.ts]{1}
  import './assets/main.css'

  import { createApp } from 'vue'
  import { createRouter, createWebHistory } from 'vue-router'
  import ui from '@nuxt/ui/vue-plugin'
  import App from './App.vue'

  const app = createApp(App)

  const router = createRouter({
    routes: [],
    history: createWebHistory()
  })

  app.use(router)
  app.use(ui)

  app.mount('#app')
  ```

  ```ts [resources/js/app.ts (Laravel Inertia)]{1}
  import '../css/app.css'
  import type { DefineComponent } from 'vue'
  import { createInertiaApp } from '@inertiajs/vue3'
  import ui from '@nuxt/ui/vue-plugin'
  import { resolvePageComponent } from 'laravel-vite-plugin/inertia-helpers'
  import { createApp, h } from 'vue'

  const appName = import.meta.env.VITE_APP_NAME || 'Laravel x Nuxt UI'

  createInertiaApp({
    title: title => (title ? `${title} - ${appName}` : appName),
    resolve: name =>
      resolvePageComponent(
        `./pages/${name}.vue`,
        import.meta.glob<DefineComponent>('./pages/**/*.vue')
      ),
    setup({ el, App, props, plugin }) {
      createApp({ render: () => h(App, props) })
        .use(plugin)
        .use(ui)
        .mount(el)
    }
  })
  ```

  ```ts [inertia/app/app.ts (AdonisJS Inertia)]{1}
  import '../css/app.css'
  import type { DefineComponent } from 'vue'
  import { createInertiaApp } from '@inertiajs/vue3'
  import ui from '@nuxt/ui/vue-plugin'
  import { resolvePageComponent } from '@adonisjs/inertia/helpers'
  import { createApp, h } from 'vue'

  const appName = import.meta.env.VITE_APP_NAME || 'AdonisJS x Nuxt UI'

  createInertiaApp({
    title: title => (title ? `${title} - ${appName}` : appName),
    resolve: name =>
      resolvePageComponent(
        `../pages/${name}.vue`,
        import.meta.glob<DefineComponent>('../pages/**/*.vue')
      ),
    setup({ el, App, props, plugin }) {
      createApp({ render: () => h(App, props) })
        .use(plugin)
        .use(ui)
        .mount(el)
    }
  })
  ```

  ::

::

::callout{icon="i-simple-icons-visualstudiocode"}
It's recommended to install the [Tailwind CSS IntelliSense](https://marketplace.visualstudio.com/items?itemName=bradlc.vscode-tailwindcss) extension for VSCode and add the following settings:

```json [.vscode/settings.json]
{
  "files.associations": {
    "*.css": "tailwindcss"
  },
  "editor.quickSuggestions": {
    "strings": "on"
  },
  "tailwindCSS.classAttributes": ["class", "ui"],
  "tailwindCSS.experimental.classRegex": [
    ["ui:\\s*{([^)]*)\\s*}", "(?:'|\"|`)([^']*)(?:'|\"|`)"]
  ]
}
```

::

#### Wrap your app with App component

::code-group{sync="vite"}

```vue [src/App.vue (Vite)]
<template>
  <UApp>
    <RouterView />
  </UApp>
</template>
```

```vue [resources/js/pages/index.vue (Laravel Inertia)]
<template>
  <UApp>
    <!-- Your content goes here -->
  </UApp>
</template>
```

```vue [inertia/pages/index.vue (AdonisJS Inertia)]
<template>
  <UApp>
    <!-- Your content goes here -->
  </UApp>
</template>
```

::

::note{to="/docs/components/app"}
The `App` component sets up global config and is required for **Toast**, **Tooltip** and **programmatic overlays**.
::

#### Add the `isolate` class to your root container

::code-group{sync="vite"}

```html [index.html (Vite)]{9}
<!DOCTYPE html>
<html lang="en">
  <head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>Nuxt UI</title>
  </head>
  <body>
    <div id="app" class="isolate"></div>
    <script type="module" src="/src/main.ts"></script>
  </body>
</html>
```

```blade [resources/views/app.blade.php (Laravel Inertia)]{10}
<!DOCTYPE html>
<html>
  <head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1">
    @inertiaHead
    @vite('resources/js/app.ts')
  </head>
  <body>
    <div class="isolate">
        @inertia
    </div>
  </body>
</html>
```

```edge [resources/views/inertia_layout.edge (AdonisJS Inertia)]{10}
<!DOCTYPE html>
<html>
  <head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1">
    @inertiaHead()
    @vite(['inertia/app/app.ts', `inertia/pages/${page.component}.vue`])
  </head>
  <body>
    @inertia({ class: 'isolate' })
  </body>
</html>
```

::

::note
This ensures styles are scoped to your app and prevents issues with overlays and stacking contexts.
::

::

### Use a Vue template

To quickly get started with Nuxt UI, you can use the [starter template](https://github.com/nuxt-ui-templates/starter-vue) by running:

```bash [Terminal]
npm create nuxt@latest -- --no-modules -t ui-vue
```

You can also get started with one of our [official templates](/templates):

::card-group
  ::card
  ---
  title: Starter
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/starter-vue
  target: _blank
  color: neutral
  ---
  A minimal template to get started with Nuxt UI.
  ::

  ::card
  ---
  title: Dashboard
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/dashboard-vue
  target: _blank
  color: neutral
  variant: subtle
  ---
  A dashboard template with multi-column layout for building sophisticated admin interfaces.
  ::

  ::card
  ---
  title: Starter Adonis
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/starter-adonis
  target: _blank
  color: neutral
  ---
  A minimal Nuxt UI template for AdonisJS using Inertia.js.
  ::

  ::card
  ---
  title: Starter Laravel
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/starter-laravel
  target: _blank
  color: neutral
  ---
  A minimal Nuxt UI template for Laravel using Inertia.js.
  ::
::

You can use the `Use this template` button on GitHub to create a new repository or use the CLI:

::code-group

```bash [Starter]
npm create nuxt@latest -- --no-modules -t ui-vue
```

```bash [Dashboard]
npm create nuxt@latest -- --no-modules -t ui-vue/dashboard
```

::

## Options

You can customize Nuxt UI by providing options in your `vite.config.ts`.

### `prefix`

Use the `prefix` option to change the prefix of the components.

- Default: `U`{lang="ts-type"}

```ts [vite.config.ts]
import { defineConfig } from 'vite'
import vue from '@vitejs/plugin-vue'
import ui from '@nuxt/ui/vite'

export default defineConfig({
  plugins: [
    vue(),
    ui({
      prefix: 'Nuxt'
    })
  ]
})
```

### `ui`

Use the `ui` option to provide configuration for Nuxt UI.

```ts [vite.config.ts]
import { defineConfig } from 'vite'
import vue from '@vitejs/plugin-vue'
import ui from '@nuxt/ui/vite'

export default defineConfig({
  plugins: [
    vue(),
    ui({
      ui: {
        colors: {
          primary: 'green',
          neutral: 'slate'
        }
      }
    })
  ]
})
```

### `colorMode`

Use the `colorMode` option to enable or disable the color mode integration from `@vueuse/core`.

- Default: `true`{lang="ts-type"}

```ts [vite.config.ts]
import { defineConfig } from 'vite'
import vue from '@vitejs/plugin-vue'
import ui from '@nuxt/ui/vite'

export default defineConfig({
  plugins: [
    vue(),
    ui({
      colorMode: false
    })
  ]
})
```

### `theme.colors`

Use the `theme.colors` option to define the dynamic color aliases used to generate components theme.

- Default: `['primary', 'secondary', 'success', 'info', 'warning', 'error']`{lang="ts-type" class="inline"}

```ts [vite.config.ts]
import { defineConfig } from 'vite'
import vue from '@vitejs/plugin-vue'
import ui from '@nuxt/ui/vite'

export default defineConfig({
  plugins: [
    vue(),
    ui({
      theme: {
        colors: ['primary', 'error']
      }
    })
  ]
})
```

::tip{to="/docs/getting-started/theme/design-system#colors"}
Learn more about color customization and theming in the Theme section.
::

### `theme.transitions`

Use the `theme.transitions` option to enable or disable transitions on components.

- Default: `true`{lang="ts-type"}

```ts [vite.config.ts]
import { defineConfig } from 'vite'
import vue from '@vitejs/plugin-vue'
import ui from '@nuxt/ui/vite'

export default defineConfig({
  plugins: [
    vue(),
    ui({
      theme: {
        transitions: false
      }
    })
  ]
})
```

::note
This option adds the `transition-colors` class on components with hover or active states.
::

### `theme.defaultVariants`

Use the `theme.defaultVariants` option to override the default `color` and `size` variants for components.

- Default: `{ color: 'primary', size: 'md' }`{lang="ts-type"}

```ts [vite.config.ts]
import { defineConfig } from 'vite'
import vue from '@vitejs/plugin-vue'
import ui from '@nuxt/ui/vite'

export default defineConfig({
  plugins: [
    vue(),
    ui({
      theme: {
        defaultVariants: {
          color: 'neutral',
          size: 'sm'
        }
      }
    })
  ]
})
```

### `inertia`

Use the `inertia` option to enable compatibility with [Inertia.js](https://inertiajs.com/).

```ts [vite.config.ts]
import { defineConfig } from 'vite'
import vue from '@vitejs/plugin-vue'
import ui from '@nuxt/ui/vite'

export default defineConfig({
  plugins: [
    vue(),
    ui({
      inertia: true
    })
  ]
})
```

::note
When using this option, `vue-router` is not required as Inertia.js provides its own routing system. The components that would normally use `RouterLink` will automatically use Inertia's `InertiaLink` component instead.
::

## Continuous releases

Nuxt UI uses [pkg.pr.new](https://github.com/stackblitz-labs/pkg.pr.new) for continuous preview releases, providing developers with instant access to the latest features and bug fixes without waiting for official releases.

Automatic preview releases are created for all commits and PRs to the `v4` branch. Use them by replacing your package version with the specific commit hash or PR number.

```diff [package.json]
{
  "dependencies": {
-   "@nuxt/ui": "^4.0.0",
+   "@nuxt/ui": "https://pkg.pr.new/@nuxt/ui@4c96909",
  }
}
```

::note
**pkg.pr.new** will automatically comment on PRs with the installation URL, making it easy to test changes.
::
