# coding: utf-8

# Copyright 2014 The Oppia Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS-IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Controllers for the editor view."""

__author__ = 'sll@google.com (Sean Lip)'

import imghdr
import logging

from core.controllers import base
from core.controllers import reader
from core.domain import config_domain
from core.domain import dependency_registry
from core.domain import event_services
from core.domain import exp_domain
from core.domain import exp_services
from core.domain import fs_domain
from core.domain import param_domain
from core.domain import rights_manager
from core.domain import obj_services
from core.domain import skins_services
from core.domain import stats_services
from core.domain import user_services
from core.domain import value_generators_domain
from core.domain import widget_registry
from core.platform import models
current_user_services = models.Registry.import_current_user_services()
import feconf
import utils

import jinja2

# The frontend template for a new state. It is sent to the frontend when the
# exploration editor page is first loaded, so that new states can be
# added in a way that is completely client-side.
# IMPORTANT: Before adding this state to an existing exploration, the
# state name and the destination of the default rule should first be
# changed to the desired new state name.
NEW_STATE_TEMPLATE = {
    'content': [{
        'type': 'text',
        'value': ''
    }],
    'param_changes': [],
    'widget': {
        'handlers': [{
            'name': 'submit',
            'rule_specs': [{
                'dest': feconf.DEFAULT_INIT_STATE_NAME,
                'definition': {
                    'rule_type': 'default'
                },
                'feedback': [],
                'param_changes': [],
                'description': 'Default',
            }],
        }],
        'widget_id': 'TextInput',
        'customization_args': {
            'rows': {'value': 1},
            'placeholder': {'value': 'Type your answer here.'}
        },
        'sticky': False
    },
    'unresolved_answers': {},
}


def get_value_generators_js():
    """Return a string that concatenates the JS for all value generators."""
    all_value_generators = (
        value_generators_domain.Registry.get_all_generator_classes())
    value_generators_js = ''
    for _, generator_cls in all_value_generators.iteritems():
        value_generators_js += generator_cls.get_js_template()
    return value_generators_js

VALUE_GENERATORS_JS = config_domain.ComputedProperty(
    'value_generators_js', 'UnicodeString',
    'JavaScript code for the value generators', get_value_generators_js)

OBJECT_EDITORS_JS = config_domain.ComputedProperty(
    'object_editors_js', 'UnicodeString',
    'JavaScript code for the object editors',
    obj_services.get_all_object_editor_js_templates)

EDITOR_PAGE_ANNOUNCEMENT = config_domain.ConfigProperty(
    'editor_page_announcement', 'Html',
    'A persistent announcement to display on top of all editor pages.',
    default_value='')
MODERATOR_REQUEST_FORUM_URL = config_domain.ConfigProperty(
    'moderator_request_forum_url', 'UnicodeString',
    'A link to the forum for nominating explorations for release.',
    default_value='https://moderator/request/forum/url')


def _require_valid_version(version_from_payload, exploration_version):
    """Check that the payload version matches the given exploration version."""
    if version_from_payload is None:
        raise base.BaseHandler.InvalidInputException(
            'Invalid POST request: a version must be specified.')

    if version_from_payload != exploration_version:
        raise base.BaseHandler.InvalidInputException(
            'Trying to update version %s of exploration from version %s, '
            'which is too old. Please reload the page and try again.'
            % (exploration_version, version_from_payload))


def require_editor(handler):
    """Decorator that checks if the user can edit the given entity."""
    def test_editor(self, exploration_id, escaped_state_name=None, **kwargs):
        """Gets the user and exploration id if the user can edit it.

        Args:
            self: the handler instance
            exploration_id: the exploration id
            escaped_state_name: the URL-escaped state name, if it exists
            **kwargs: any other arguments passed to the handler

        Returns:
            The relevant handler, if the user is authorized to edit this
            exploration.

        Raises:
            self.PageNotFoundException: if no such exploration or state exists.
            self.UnauthorizedUserException: if the user exists but does not
                have the right credentials.
        """
        if not self.user_id:
            self.redirect(current_user_services.create_login_url(
                self.request.uri))
            return

        if self.username in config_domain.BANNED_USERNAMES.value:
            raise self.UnauthorizedUserException(
                'You do not have the credentials to access this page.')

        redirect_url = feconf.EDITOR_PREREQUISITES_URL

        if not user_services.has_user_registered_as_editor(self.user_id):
            redirect_url = utils.set_url_query_parameter(
                redirect_url, 'return_url', self.request.uri)
            self.redirect(redirect_url)
            return

        try:
            exploration = exp_services.get_exploration_by_id(exploration_id)
        except:
            raise self.PageNotFoundException

        if not rights_manager.Actor(self.user_id).can_edit(exploration_id):
            raise self.UnauthorizedUserException(
                'You do not have the credentials to edit this exploration.',
                self.user_id)

        if not escaped_state_name:
            return handler(self, exploration_id, **kwargs)

        state_name = self.unescape_state_name(escaped_state_name)
        if state_name not in exploration.states:
            logging.error('Could not find state: %s' % state_name)
            logging.error('Available states: %s' % exploration.states.keys())
            raise self.PageNotFoundException

        return handler(self, exploration_id, state_name, **kwargs)

    return test_editor


class EditorHandler(base.BaseHandler):
    """Base class for all handlers for the editor page."""

    # The page name to use as a key for generating CSRF tokens.
    PAGE_NAME_FOR_CSRF = 'editor'


class ExplorationPage(EditorHandler):
    """The editor page for a single exploration."""

    EDITOR_PAGE_DEPENDENCY_IDS = ['codemirror']

    def get(self, exploration_id):
        """Handles GET requests."""
        try:
            exp_services.get_exploration_by_id(exploration_id)
        except:
            raise self.PageNotFoundException

        if not rights_manager.Actor(self.user_id).can_view(exploration_id):
            raise self.PageNotFoundException

        can_edit = (
            bool(self.user_id) and
            self.username not in config_domain.BANNED_USERNAMES.value and
            rights_manager.Actor(self.user_id).can_edit(exploration_id))

        if (can_edit and not
                user_services.has_user_registered_as_editor(self.user_id)):
            redirect_url = utils.set_url_query_parameter(
                feconf.EDITOR_PREREQUISITES_URL, 'return_url',
                self.request.uri)
            self.redirect(redirect_url)
            return

        # TODO(sll): Consider including the obj_generator html in a ng-template
        # to remove the need for an additional RPC?
        object_editors_js = OBJECT_EDITORS_JS.value
        value_generators_js = VALUE_GENERATORS_JS.value

        all_interactive_widget_ids = (
            widget_registry.Registry.get_widget_ids_of_type(
                feconf.INTERACTIVE_PREFIX))

        widget_dependency_ids = (
            widget_registry.Registry.get_deduplicated_dependency_ids(
                all_interactive_widget_ids))
        dependencies_html, additional_angular_modules = (
            dependency_registry.Registry.get_deps_html_and_angular_modules(
                widget_dependency_ids + self.EDITOR_PAGE_DEPENDENCY_IDS))

        widget_templates = (
            widget_registry.Registry.get_noninteractive_widget_html() +
            widget_registry.Registry.get_interactive_widget_html(
                all_interactive_widget_ids))

        skin_templates = skins_services.Registry.get_skin_templates(
            skins_services.Registry.get_all_skin_ids())

        self.values.update({
            'additional_angular_modules': additional_angular_modules,
            'announcement': jinja2.utils.Markup(
                EDITOR_PAGE_ANNOUNCEMENT.value),
            'can_delete': rights_manager.Actor(
                self.user_id).can_delete(exploration_id),
            'can_edit': can_edit,
            'can_modify_roles': rights_manager.Actor(
                self.user_id).can_modify_roles(exploration_id),
            'can_publicize': rights_manager.Actor(
                self.user_id).can_publicize(exploration_id),
            'can_publish': rights_manager.Actor(self.user_id).can_publish(
                exploration_id),
            'can_release_ownership': rights_manager.Actor(
                self.user_id).can_release_ownership(exploration_id),
            'can_unpublicize': rights_manager.Actor(
                self.user_id).can_unpublicize(exploration_id),
            'can_unpublish': rights_manager.Actor(self.user_id).can_unpublish(
                exploration_id),
            'dependencies_html': jinja2.utils.Markup(dependencies_html),
            'moderator_request_forum_url': MODERATOR_REQUEST_FORUM_URL.value,
            'nav_mode': feconf.NAV_MODE_CREATE,
            'object_editors_js': jinja2.utils.Markup(object_editors_js),
            'value_generators_js': jinja2.utils.Markup(value_generators_js),
            'widget_templates': jinja2.utils.Markup(widget_templates),
            'skin_js_urls': [
                skins_services.Registry.get_skin_js_url(skin_id)
                for skin_id in skins_services.Registry.get_all_skin_ids()],
            'skin_templates': jinja2.utils.Markup(skin_templates),
            'ALL_LANGUAGE_CODES': feconf.ALL_LANGUAGE_CODES,
            'NEW_STATE_TEMPLATE': NEW_STATE_TEMPLATE,
            'SHOW_SKIN_CHOOSER': feconf.SHOW_SKIN_CHOOSER,
        })

        self.render_template('editor/exploration_editor.html')


class ExplorationHandler(EditorHandler):
    """Page with editor data for a single exploration."""

    PAGE_NAME_FOR_CSRF = 'editor'

    def _get_exploration_data(self, exploration_id):
        """Returns a description of the given exploration."""
        try:
            exploration = exp_services.get_exploration_by_id(exploration_id)
        except:
            raise self.PageNotFoundException

        states = {}
        for state_name in exploration.states:
            state_frontend_dict = exploration.export_state_to_frontend_dict(
                state_name)
            state_frontend_dict['unresolved_answers'] = (
                stats_services.get_top_unresolved_answers_for_default_rule(
                    exploration_id, state_name))
            states[state_name] = state_frontend_dict

        editor_dict = {
            'exploration_id': exploration_id,
            'init_state_name': exploration.init_state_name,
            'category': exploration.category,
            'objective': exploration.objective,
            'language_code': exploration.language_code,
            'title': exploration.title,
            'states': states,
            'param_changes': exploration.param_change_dicts,
            'param_specs': exploration.param_specs_dict,
            'version': exploration.version,
            'rights': rights_manager.get_exploration_rights(
                exploration_id).to_dict(),
            'ALL_INTERACTIVE_WIDGETS': {
                widget.id: widget.to_dict()
                for widget in widget_registry.Registry.get_widgets_of_type(
                    feconf.INTERACTIVE_PREFIX)
            },
        }

        if feconf.SHOW_SKIN_CHOOSER:
            editor_dict['all_skin_ids'] = (
                skins_services.Registry.get_all_skin_ids())
            editor_dict['default_skin_id'] = exploration.default_skin

        return editor_dict

    def get(self, exploration_id):
        """Gets the data for the exploration overview page."""
        if not rights_manager.Actor(self.user_id).can_view(exploration_id):
            raise self.PageNotFoundException

        self.values.update(self._get_exploration_data(exploration_id))
        self.render_json(self.values)

    @require_editor
    def put(self, exploration_id):
        """Updates properties of the given exploration."""
        exploration = exp_services.get_exploration_by_id(exploration_id)
        version = self.payload.get('version')
        _require_valid_version(version, exploration.version)

        commit_message = self.payload.get('commit_message')
        change_list = self.payload.get('change_list')

        try:
            exp_services.update_exploration(
                self.user_id, exploration_id, change_list, commit_message)
        except utils.ValidationError as e:
            raise self.InvalidInputException(e)

        self.values.update(self._get_exploration_data(exploration_id))
        self.render_json(self.values)

    @require_editor
    def delete(self, exploration_id):
        """Deletes the given exploration."""
        role = self.request.get('role')
        if not role:
            role = None

        if role == rights_manager.ROLE_ADMIN:
            if not self.is_admin:
                logging.error(
                    '%s tried to delete an exploration, but is not an admin.'
                    % self.user_id)
                raise self.UnauthorizedUserException(
                    'User %s does not have permissions to delete exploration '
                    '%s' % (self.user_id, exploration_id))
        elif role == rights_manager.ROLE_MODERATOR:
            if not self.is_moderator:
                logging.error(
                    '%s tried to delete an exploration, but is not a '
                    'moderator.' % self.user_id)
                raise self.UnauthorizedUserException(
                    'User %s does not have permissions to delete exploration '
                    '%s' % (self.user_id, exploration_id))
        elif role is not None:
            raise self.InvalidInputException('Invalid role: %s' % role)

        logging.info(
            '%s %s tried to delete exploration %s' %
            (role, self.user_id, exploration_id))

        exploration = exp_services.get_exploration_by_id(exploration_id)
        can_delete = rights_manager.Actor(self.user_id).can_delete(
            exploration.id)
        if not can_delete:
            raise self.UnauthorizedUserException(
                'User %s does not have permissions to delete exploration %s' %
                (self.user_id, exploration_id))

        is_exploration_cloned = rights_manager.is_exploration_cloned(
            exploration_id)
        exp_services.delete_exploration(
            self.user_id, exploration_id, force_deletion=is_exploration_cloned)

        logging.info(
            '%s %s deleted exploration %s' %
            (role, self.user_id, exploration_id))


class ExplorationRightsHandler(EditorHandler):
    """Handles management of exploration editing rights."""

    PAGE_NAME_FOR_CSRF = 'editor'

    @require_editor
    def put(self, exploration_id):
        """Updates the editing rights for the given exploration."""
        exploration = exp_services.get_exploration_by_id(exploration_id)
        version = self.payload.get('version')
        _require_valid_version(version, exploration.version)

        is_public = self.payload.get('is_public')
        is_publicized = self.payload.get('is_publicized')
        is_community_owned = self.payload.get('is_community_owned')
        new_member_username = self.payload.get('new_member_username')
        new_member_role = self.payload.get('new_member_role')
        viewable_if_private = self.payload.get('viewable_if_private')

        if new_member_username:
            if not rights_manager.Actor(self.user_id).can_modify_roles(
                    exploration_id):
                raise self.UnauthorizedUserException(
                    'Only an owner of this exploration can add or change '
                    'roles.')

            new_member_id = user_services.get_user_id_from_username(
                new_member_username)
            if new_member_id is None:
                raise Exception(
                    'Sorry, we could not find the specified user.')

            rights_manager.assign_role(
                self.user_id, exploration_id, new_member_id, new_member_role)

        elif is_public is not None:
            exploration = exp_services.get_exploration_by_id(exploration_id)
            if is_public:
                try:
                    exploration.validate(strict=True)
                except utils.ValidationError as e:
                    raise self.InvalidInputException(e)

                rights_manager.publish_exploration(
                    self.user_id, exploration_id)
            else:
                rights_manager.unpublish_exploration(
                    self.user_id, exploration_id)

        elif is_publicized is not None:
            exploration = exp_services.get_exploration_by_id(exploration_id)
            if is_publicized:
                try:
                    exploration.validate(strict=True)
                except utils.ValidationError as e:
                    raise self.InvalidInputException(e)

                rights_manager.publicize_exploration(
                    self.user_id, exploration_id)
            else:
                rights_manager.unpublicize_exploration(
                    self.user_id, exploration_id)

        elif is_community_owned:
            exploration = exp_services.get_exploration_by_id(exploration_id)
            try:
                exploration.validate(strict=True)
            except utils.ValidationError as e:
                raise self.InvalidInputException(e)

            rights_manager.release_ownership(self.user_id, exploration_id)

        elif viewable_if_private is not None:
            rights_manager.set_private_viewability(
                self.user_id, exploration_id, viewable_if_private)

        else:
            raise self.InvalidInputException(
                'No change was made to this exploration.')

        self.render_json({
            'rights': rights_manager.get_exploration_rights(
                exploration_id).to_dict()
        })


class ResolvedAnswersHandler(EditorHandler):
    """Allows readers' answers for a state to be marked as resolved."""

    PAGE_NAME_FOR_CSRF = 'editor'

    @require_editor
    def put(self, exploration_id, state_name):
        """Marks readers' answers as resolved."""
        resolved_answers = self.payload.get('resolved_answers')

        if not isinstance(resolved_answers, list):
            raise self.InvalidInputException(
                'Expected a list of resolved answers; received %s.' %
                resolved_answers)

        if 'resolved_answers' in self.payload:
            event_services.DefaultRuleAnswerResolutionEventHandler.record(
                exploration_id, state_name, 'submit', resolved_answers)

        self.render_json({})


class ExplorationDownloadHandler(EditorHandler):
    """Downloads an exploration as a zip file or JSON."""

    def get(self, exploration_id):
        """Handles GET requests."""
        try:
            exploration = exp_services.get_exploration_by_id(exploration_id)
        except:
            raise self.PageNotFoundException

        version = self.request.get('v', default_value=exploration.version)
        output_format = self.request.get('output_format', default_value='zip')

        # If the title of the exploration has changed, we use the new title
        filename = 'oppia-%s-v%s' % (
            utils.to_ascii(exploration.title.replace(' ', '')), version)

        if output_format == feconf.OUTPUT_FORMAT_ZIP:
            self.response.headers['Content-Type'] = 'text/plain'
            self.response.headers['Content-Disposition'] = (
                'attachment; filename=%s.zip' % str(filename))
            self.response.write(
                exp_services.export_to_zip_file(exploration_id, version))
        elif output_format == feconf.OUTPUT_FORMAT_JSON:
            self.render_json(
                exp_services.export_to_dict(exploration_id, version))
        else:
            raise self.InvalidInputException(
                'Unrecognized output format %s' % output_format)


class ExplorationResourcesHandler(EditorHandler):
    """Manages assets associated with an exploration."""

    @require_editor
    def get(self, exploration_id):
        """Handles GET requests."""
        fs = fs_domain.AbstractFileSystem(
            fs_domain.ExplorationFileSystem(exploration_id))
        dir_list = fs.listdir('')

        self.render_json({'filepaths': dir_list})


class ExplorationSnapshotsHandler(EditorHandler):
    """Returns the exploration snapshot history."""

    def get(self, exploration_id):
        """Handles GET requests."""

        try:
            snapshots = exp_services.get_exploration_snapshots_metadata(exploration_id)
        except:
            raise self.PageNotFoundException

        # Patch `snapshots` to use the editor's display name.
        for snapshot in snapshots:
            if snapshot['committer_id'] != feconf.ADMIN_COMMITTER_ID:
                snapshot['committer_id'] = user_services.get_username(
                    snapshot['committer_id'])

        self.render_json({
            'snapshots': snapshots,
        })


class ExplorationRevertHandler(EditorHandler):
    """Reverts an exploration to an older version."""

    @require_editor
    def post(self, exploration_id):
        """Handles POST requests."""
        current_version = self.payload.get('current_version')
        revert_to_version = self.payload.get('revert_to_version')

        if not isinstance(revert_to_version, int):
            raise self.InvalidInputException(
                'Expected an integer version to revert to; received %s.' %
                revert_to_version)
        if not isinstance(current_version, int):
            raise self.InvalidInputException(
                'Expected an integer current version; received %s.' %
                current_version)

        if revert_to_version < 1 or revert_to_version >= current_version:
            raise self.InvalidInputException(
                'Cannot revert to version %s from version %s.' %
                (revert_to_version, current_version))

        exp_services.revert_exploration(
            self.user_id, exploration_id, current_version, revert_to_version)
        self.render_json({})


class ExplorationStatisticsHandler(EditorHandler):
    """Returns statistics for an exploration."""

    def get(self, exploration_id):
        """Handles GET requests."""
        try:
            exp_services.get_exploration_by_id(exploration_id)
        except:
            raise self.PageNotFoundException

        self.render_json({
            'num_starts': stats_services.get_exploration_start_count(
                exploration_id),
            'num_completions': stats_services.get_exploration_completed_count(
                exploration_id),
            'state_stats': stats_services.get_state_stats_for_exploration(
                exploration_id),
            'improvements': stats_services.get_state_improvements(
                exploration_id),
        })


class StateRulesStatsHandler(EditorHandler):
    """Returns detailed reader answer statistics for a state."""

    def get(self, exploration_id, escaped_state_name):
        """Handles GET requests."""
        try:
            exploration = exp_services.get_exploration_by_id(exploration_id)
        except:
            raise self.PageNotFoundException

        state_name = self.unescape_state_name(escaped_state_name)
        if state_name not in exploration.states:
            logging.error('Could not find state: %s' % state_name)
            logging.error('Available states: %s' % exploration.states.keys())
            raise self.PageNotFoundException

        self.render_json({
            'rules_stats': stats_services.get_state_rules_stats(
                exploration_id, state_name)
        })


class ImageUploadHandler(EditorHandler):
    """Handles image uploads."""

    @require_editor
    def post(self, exploration_id):
        """Saves an image uploaded by a content creator."""

        raw = self.request.get('image')
        filename = self.payload.get('filename')
        if not raw:
            raise self.InvalidInputException('No image supplied')

        file_format = imghdr.what(None, h=raw)
        if file_format not in feconf.ACCEPTED_IMAGE_FORMATS_AND_EXTENSIONS:
            allowed_formats = ', '.join(
                feconf.ACCEPTED_IMAGE_FORMATS_AND_EXTENSIONS.keys())
            raise Exception('Image file not recognized: it should be in '
                            'one of the following formats: %s.' %
                            allowed_formats)

        if not filename:
            raise self.InvalidInputException('No filename supplied')
        if '/' in filename or '..' in filename:
            raise self.InvalidInputException(
                'Filenames should not include slashes (/) or consecutive dot '
                'characters.')
        if '.' in filename:
            dot_index = filename.rfind('.')
            primary_name = filename[:dot_index]
            extension = filename[dot_index + 1:].lower()
            if (extension not in
                    feconf.ACCEPTED_IMAGE_FORMATS_AND_EXTENSIONS[file_format]):
                raise self.InvalidInputException(
                    'Expected a filename ending in .%s; received %s' %
                    (file_format, filename))
        else:
            primary_name = filename

        filepath = '%s.%s' % (primary_name, file_format)

        fs = fs_domain.AbstractFileSystem(
            fs_domain.ExplorationFileSystem(exploration_id))
        if fs.isfile(filepath):
            raise self.InvalidInputException(
                'A file with the name %s already exists. Please choose a '
                'different name.' % filepath)
        fs.commit(self.user_id, filepath, raw)

        self.render_json({'filepath': filepath})


class ChangeListSummaryHandler(EditorHandler):
    """Returns a summary of a changelist applied to a given exploration."""

    @require_editor
    def post(self, exploration_id):
        """Handles POST requests."""
        change_list = self.payload.get('change_list')
        version = self.payload.get('version')
        current_exploration = exp_services.get_exploration_by_id(
            exploration_id)

        if version != current_exploration.version:
            # TODO(sll): Improve this.
            self.render_json({
                'error': (
                    'Sorry! Someone else has edited and committed changes to '
                    'this exploration while you were editing it. We suggest '
                    'opening another browser tab -- which will load the new '
                    'version of the exploration -- then transferring your '
                    'changes there. We will try to make this easier in the '
                    'future -- we have not done it yet because figuring out '
                    'how to merge different people\'s changes is hard. '
                    '(Trying to edit version %s, but the current version is '
                    '%s.).' % (version, current_exploration.version)
                )
            })
        else:
            utils.recursively_remove_key(change_list, '$$hashKey')

            summary = exp_services.get_summary_of_change_list(
                current_exploration, change_list)
            updated_exploration = exp_services.apply_change_list(
                exploration_id, change_list)
            warning_message = ''
            try:
                updated_exploration.validate(strict=True)
            except utils.ValidationError as e:
                warning_message = unicode(e)

            self.render_json({
                'summary': summary,
                'warning_message': warning_message
            })


class InitExplorationHandler(EditorHandler):
    """Performs a get_init_html_and_params() operation server-side and
    returns the result. This is done while maintaining no state.
    """

    @require_editor
    def post(self, exploration_id):
        """Handles POST requests."""
        exp_param_specs_dict = self.payload.get('exp_param_specs', {})
        exp_param_specs = {
            ps_name: param_domain.ParamSpec.from_dict(ps_val)
            for (ps_name, ps_val) in exp_param_specs_dict.iteritems()
        }
        # A domain object representing the old state.
        init_state = exp_domain.State.from_dict(self.payload.get('init_state'))
        # A domain object representing the exploration-level parameter changes.
        exp_param_changes = [
            param_domain.ParamChange.from_dict(param_change_dict)
            for param_change_dict in self.payload.get('exp_param_changes')]

        init_html, init_params = reader.get_init_html_and_params(
            exp_param_changes, init_state, exp_param_specs)

        self.render_json({
            'init_html': init_html,
            'params': init_params,
        })


class ClassifyHandler(EditorHandler):
    """Performs a classify() operation server-side and returns the result.
    This is done while maintaining no state.
    """

    @require_editor
    def post(self, exploration_id):
        """Handles POST requests."""
        exp_param_specs_dict = self.payload.get('exp_param_specs', {})
        exp_param_specs = {
            ps_name: param_domain.ParamSpec.from_dict(ps_val)
            for (ps_name, ps_val) in exp_param_specs_dict.iteritems()
        }
        # A domain object representing the old state.
        old_state = exp_domain.State.from_dict(self.payload.get('old_state'))
        # The name of the rule handler triggered.
        handler_name = self.payload.get('handler')
        # The learner's raw answer.
        answer = self.payload.get('answer')
        # The learner's parameter values.
        params = self.payload.get('params')

        rule_spec, input_type = reader.classify(
            exploration_id, exp_param_specs, old_state, handler_name,
            answer, params)

        self.render_json({
            'rule_spec': rule_spec.to_dict(),
            'input_type': input_type,
        })


class NextStateHandler(EditorHandler):
    """Performs a get_new_state_dict() operation server-side and returns the
    result. This is done while maintaining no state.
    """

    @require_editor
    def post(self, exploration_id):
        """Handles POST requests."""
        exp_param_specs_dict = self.payload.get('exp_param_specs', {})
        exp_param_specs = {
            ps_name: param_domain.ParamSpec.from_dict(ps_val)
            for (ps_name, ps_val) in exp_param_specs_dict.iteritems()
        }
        # The old state name.
        old_state_name = self.payload.get('old_state_name')
        # The learner's parameter values.
        params = self.payload.get('params')
        # The input type of the answer.
        input_type = self.payload.get('input_type')
        # The rule spec matching the learner's answer.
        rule_spec = exp_domain.RuleSpec.from_dict_and_obj_type(
            self.payload.get('rule_spec'), input_type)
        # A domain object representing the new state.
        new_state_dict = self.payload.get('new_state')
        new_state = (
            exp_domain.State.from_dict(new_state_dict) if new_state_dict
            else None)

        self.render_json(reader.get_next_state_dict(
            exp_param_specs, old_state_name, params, rule_spec, new_state))
