// Copyright 2014 The Oppia Authors. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS-IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview Utilities for exploration creation, publication ect. when
 * carrrying out end-to-end testing with protractor.
 *
 * @author Jacob Davis (jacobdavis11@gmail.com)
 */

forms = require('./forms.js');
editor = require('./editor.js');
general = require('./general.js');

// Creates an exploration and opens its editor.
var createExploration = function(name, category) {
  browser.get('/gallery');
  element(by.css('.protractor-test-create-exploration')).click();
  element(by.model('newExplorationTitle')).sendKeys(name);
  forms.editAutocompleteDropdown(element(by.tagName('select2-dropdown'))).
    setText(category);
  element(by.buttonText('Create New Exploration')).click();

  // We now want to wait for the editor to fully load.
  protractor.getInstance().waitForAngular();
};

// This will only work if all changes have been saved and there are no 
// outstanding warnings; run from the editor.
var publishExploration = function() {
  element(by.css('.protractor-test-publish-exploration')).click();
  protractor.getInstance().waitForAngular();
  general.waitForSystem();
  element(by.css('.protractor-test-confirm-publish')).click();
};

// Creates and publishes a minimal exploration
var createAndPublishExploration = function(name, category, objective, language) {
  createExploration(name, category);
  editor.editRule('default').setDestination('END');
  editor.setObjective(objective);
  if (language) {
    editor.setLanguage(language);
  }
  editor.selectContinueWidget();
  editor.saveChanges();
  publishExploration();
};

// Run from the editor, requires user to be a moderator / admin.
var releaseExploration = function() {
  editor.runFromSettingsTab(function() {
    element(by.css('.protractor-test-release-exploration')).click();
  });
};

// Role management (state editor settings tab)

// roleName here is the user-visible form of the role name (e.g. 'Manager')
var _addExplorationRole = function(roleName, username) {
  editor.runFromSettingsTab(function() {
    element(by.css('.protractor-test-edit-roles')).click();
    element(by.css('.protractor-test-role-username')).sendKeys(username);
    element(by.css('.protractor-test-role-select')).
      element(by.cssContainingText('option', roleName)).click();
    element(by.css('.protractor-test-save-role')).click();
  });
};

var addExplorationManager = function(username) {
  _addExplorationRole('Manager', username);
};

var addExplorationCollaborator = function(username) {
  _addExplorationRole('Collaborator', username);
};

var addExplorationPlaytester = function(username) {
  _addExplorationRole('Playtester', username);
};

// roleName here is the server-side form of the name (e.g. 'owner')
var _getExplorationRoles = function(roleName) {
  var result = editor.runFromSettingsTab(function() {
    var itemName = roleName + 'Name';
    var listName = roleName + 'Names';
    return element.all(by.repeater(
          itemName + ' in explorationRightsService.' + listName + ' track by $index'
        )).map(function(elem) {
      return elem.getText();
    }); 
  });
  return result;
};

var getExplorationManagers = function() {
  return _getExplorationRoles('owner');
};

var getExplorationCollaborators = function() {
  return _getExplorationRoles('editor');
};

var getExplorationPlaytesters = function() {
  return _getExplorationRoles('viewer');
};

exports.createExploration = createExploration;
exports.publishExploration = publishExploration;
exports.createAndPublishExploration = createAndPublishExploration;
exports.releaseExploration = releaseExploration;

exports.addExplorationManager = addExplorationManager;
exports.addExplorationCollaborator = addExplorationCollaborator;
exports.addExplorationPlaytester = addExplorationPlaytester;
exports.getExplorationManagers = getExplorationManagers;
exports.getExplorationCollaborators = getExplorationCollaborators;
exports.getExplorationPlaytesters = getExplorationPlaytesters;