// Copyright (C) Krzysztof Jakubowski <nadult@fastmail.fm>
// This file is part of libfwk. See license.txt for details.

#pragma once

#include "fwk/enum.h"
#include "fwk/math_base.h"
#include "fwk/str.h"
#include "fwk/sys/expected.h"

namespace fwk {

class TextParser;
struct NotParsable;

namespace detail {

	template <class T> struct IsParsable {
		template <class U> static auto test(U &) -> decltype(DECLVAL(TextParser &) >> DECLVAL(U &));
		static char test(...);
		static constexpr bool value = is_same<decltype(test(DECLVAL(T &))), TextParser &>;
	};

	template <class T> struct VariableParseElements {
		static constexpr bool value = false;
	};
	template <class T> struct VariableParseElements<vector<T>> {
		static constexpr bool value = true;
	};
}

template <class T> constexpr bool is_parsable = detail::IsParsable<T>::value;
template <class T> concept c_parsable = is_parsable<T>;

// Parsing white-space separated elements
// Output generated by TextFormatter in plain mode can be parsed
// Errors are reported with exceptions
// TODO: strings with whitespace in them
// TODO: drop requirement of 0-termination
class TextParser {
  public:
	TextParser(ZStr str) : m_current(str) {}

	// These will work even when parser is empty
	TextParser &operator>>(Str &);
	TextParser &operator>>(string &);

	TextParser &operator>>(bool &) EXCEPT;
	TextParser &operator>>(double &) EXCEPT;
	TextParser &operator>>(float &) EXCEPT;
	TextParser &operator>>(short &) EXCEPT;
	TextParser &operator>>(unsigned short &) EXCEPT;
	TextParser &operator>>(int &) EXCEPT;
	TextParser &operator>>(unsigned int &) EXCEPT;
	TextParser &operator>>(long &) EXCEPT;
	TextParser &operator>>(unsigned long &) EXCEPT;
	TextParser &operator>>(long long &) EXCEPT;
	TextParser &operator>>(unsigned long long &) EXCEPT;

	template <c_span TSpan, class T = SpanBase<TSpan>>
		requires is_parsable<T>
	void parseSpan(TSpan &span) EXCEPT {
		for(auto &elem : span)
			*this >> elem;
	}

	void parseNotEmpty(Span<Str>) EXCEPT;
	void parseNotEmpty(Span<string>) EXCEPT;
	void parseInts(Span<int>) EXCEPT;
	void parseFloats(Span<float>) EXCEPT;
	void parseDoubles(Span<double>) EXCEPT;

	Str current() const { return m_current; }
	void advance(int offset) { m_current = m_current.advance(offset); }

	bool empty() const { return m_current.empty(); }

	// Also skips whitespace on both sides
	Str parseElement();
	void advanceWhitespace();
	int countElements() const;

	void parseUints(Span<uint> out) EXCEPT;
	void parseStrings(Span<string> out) EXCEPT;

	template <c_parsable T> T parse() EXCEPT {
		T value;
		*this >> value;
		return value;
	}

	void errorTrailingData() EXCEPT;

  private:
	template <class Func> auto parseSingle(Func func, const char *) EXCEPT;
	template <class T, class Func> T parseSingleRanged(Func, const char *) EXCEPT;

	ZStr m_current;
};

// To make new type parsable: simply overload operator>>:
// TextParser &operator>>(TextParser&, MyNewType &rhs);

TextParser &operator>>(TextParser &, short2 &) EXCEPT;
TextParser &operator>>(TextParser &, short3 &) EXCEPT;
TextParser &operator>>(TextParser &, short4 &) EXCEPT;
TextParser &operator>>(TextParser &, int2 &) EXCEPT;
TextParser &operator>>(TextParser &, int3 &) EXCEPT;
TextParser &operator>>(TextParser &, int4 &) EXCEPT;
TextParser &operator>>(TextParser &, double2 &) EXCEPT;
TextParser &operator>>(TextParser &, double3 &) EXCEPT;
TextParser &operator>>(TextParser &, double4 &) EXCEPT;
TextParser &operator>>(TextParser &, float2 &) EXCEPT;
TextParser &operator>>(TextParser &, float3 &) EXCEPT;
TextParser &operator>>(TextParser &, float4 &) EXCEPT;
TextParser &operator>>(TextParser &, DRect &) EXCEPT;
TextParser &operator>>(TextParser &, FRect &) EXCEPT;
TextParser &operator>>(TextParser &, IRect &) EXCEPT;
TextParser &operator>>(TextParser &, FBox &) EXCEPT;
TextParser &operator>>(TextParser &, IBox &) EXCEPT;
TextParser &operator>>(TextParser &, DBox &) EXCEPT;
TextParser &operator>>(TextParser &, Matrix4 &) EXCEPT;
TextParser &operator>>(TextParser &, Quat &) EXCEPT;

TextParser &operator>>(TextParser &, vector<string> &) EXCEPT;
TextParser &operator>>(TextParser &, vector<int> &) EXCEPT;
TextParser &operator>>(TextParser &, vector<float> &) EXCEPT;

template <c_parsable T>
	requires(!detail::VariableParseElements<T>::value)
TextParser &operator>>(TextParser &parser, vector<T> &vec) EXCEPT {
	parser.advanceWhitespace();
	vec.clear();
	while(!parser.empty()) {
		vec.emplace_back();
		parser >> vec.back();
	}
	return parser;
}

template <class T>
	requires(is_one_of<T, Str, ZStr, string>)
T fromString(ZStr v) {
	return v;
}

template <c_parsable T>
	requires(!is_enum<T> && !is_one_of<T, Str, ZStr, string>)
T fromString(ZStr str) EXCEPT {
	TextParser parser(str);
	T out;
	parser >> out;
	if(!parser.empty()) {
		out = T();
		parser.errorTrailingData();
	}
	return out;
}

template <c_parsable T>
	requires(!is_enum<T>)
T tryFromString(ZStr str, const T &on_error = {}) NOEXCEPT {
	QuietExceptionBlock quiet;
	TextParser parser(str);
	T out;
	parser >> out;
	if(exceptionRaised()) {
		clearExceptions();
		return on_error;
	}
	if(!parser.empty())
		return on_error;
	return out;
}

template <c_parsable T>
	requires(!is_enum<T>)
Maybe<T> maybeFromString(ZStr str) NOEXCEPT {
	QuietExceptionBlock quiet;
	TextParser parser(str);
	T out;
	parser >> out;
	if(exceptionRaised()) {
		clearExceptions();
		return none;
	}
	if(!parser.empty())
		return none;
	return out;
}
}
