'use strict';
/**
 * Merges an array of buffers into a new buffer.
 *
 * @param {Buffer[]} list The array of buffers to concat
 * @param {Number} totalLength The total length of buffers in the list
 * @return {Buffer} The resulting buffer
 * @public
 */
function concat(list, totalLength) {
    var target = Buffer.allocUnsafe(totalLength);
    var offset = 0;
    for (var i = 0; i < list.length; i++) {
        var buf = list[i];
        buf.copy(target, offset);
        offset += buf.length;
    }
    return target;
}
/**
 * Masks a buffer using the given mask.
 *
 * @param {Buffer} source The buffer to mask
 * @param {Buffer} mask The mask to use
 * @param {Buffer} output The buffer where to store the result
 * @param {Number} offset The offset at which to start writing
 * @param {Number} length The number of bytes to mask.
 * @public
 */
function _mask(source, mask, output, offset, length) {
    for (var i = 0; i < length; i++) {
        output[offset + i] = source[i] ^ mask[i & 3];
    }
}
/**
 * Unmasks a buffer using the given mask.
 *
 * @param {Buffer} buffer The buffer to unmask
 * @param {Buffer} mask The mask to use
 * @public
 */
function _unmask(buffer, mask) {
    // Required until https://github.com/nodejs/node/issues/9006 is resolved.
    var length = buffer.length;
    for (var i = 0; i < length; i++) {
        buffer[i] ^= mask[i & 3];
    }
}
try {
    var bufferUtil = require('bufferutil');
    var bu_1 = bufferUtil.BufferUtil || bufferUtil;
    module.exports = {
        mask: function (source, mask, output, offset, length) {
            if (length < 48)
                _mask(source, mask, output, offset, length);
            else
                bu_1.mask(source, mask, output, offset, length);
        },
        unmask: function (buffer, mask) {
            if (buffer.length < 32)
                _unmask(buffer, mask);
            else
                bu_1.unmask(buffer, mask);
        },
        concat: concat
    };
}
catch (e) /* istanbul ignore next */ {
    module.exports = { concat: concat, mask: _mask, unmask: _unmask };
}
