'use strict';
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
        return extendStatics(d, b);
    }
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
/**
 * Class representing an event.
 *
 * @private
 */
var Event = /** @class */ (function () {
    /**
     * Create a new `Event`.
     *
     * @param {String} type The name of the event
     * @param {Object} target A reference to the target to which the event was dispatched
     */
    function Event(type, target) {
        this.target = target;
        this.type = type;
    }
    return Event;
}());
/**
 * Class representing a message event.
 *
 * @extends Event
 * @private
 */
var MessageEvent = /** @class */ (function (_super) {
    __extends(MessageEvent, _super);
    /**
     * Create a new `MessageEvent`.
     *
     * @param {(String|Buffer|ArrayBuffer|Buffer[])} data The received data
     * @param {WebSocket} target A reference to the target to which the event was dispatched
     */
    function MessageEvent(data, target) {
        var _this = _super.call(this, 'message', target) || this;
        _this.data = data;
        return _this;
    }
    return MessageEvent;
}(Event));
/**
 * Class representing a close event.
 *
 * @extends Event
 * @private
 */
var CloseEvent = /** @class */ (function (_super) {
    __extends(CloseEvent, _super);
    /**
     * Create a new `CloseEvent`.
     *
     * @param {Number} code The status code explaining why the connection is being closed
     * @param {String} reason A human-readable string explaining why the connection is closing
     * @param {WebSocket} target A reference to the target to which the event was dispatched
     */
    function CloseEvent(code, reason, target) {
        var _this = _super.call(this, 'close', target) || this;
        _this.wasClean = target._closeFrameReceived && target._closeFrameSent;
        _this.reason = reason;
        _this.code = code;
        return _this;
    }
    return CloseEvent;
}(Event));
/**
 * Class representing an open event.
 *
 * @extends Event
 * @private
 */
var OpenEvent = /** @class */ (function (_super) {
    __extends(OpenEvent, _super);
    /**
     * Create a new `OpenEvent`.
     *
     * @param {WebSocket} target A reference to the target to which the event was dispatched
     */
    function OpenEvent(target) {
        return _super.call(this, 'open', target) || this;
    }
    return OpenEvent;
}(Event));
/**
 * Class representing an error event.
 *
 * @extends Event
 * @private
 */
var ErrorEvent = /** @class */ (function (_super) {
    __extends(ErrorEvent, _super);
    /**
     * Create a new `ErrorEvent`.
     *
     * @param {Object} error The error that generated this event
     * @param {WebSocket} target A reference to the target to which the event was dispatched
     */
    function ErrorEvent(error, target) {
        var _this = _super.call(this, 'error', target) || this;
        _this.message = error.message;
        _this.error = error;
        return _this;
    }
    return ErrorEvent;
}(Event));
/**
 * This provides methods for emulating the `EventTarget` interface. It's not
 * meant to be used directly.
 *
 * @mixin
 */
var EventTarget = {
    /**
     * Register an event listener.
     *
     * @param {String} method A string representing the event type to listen for
     * @param {Function} listener The listener to add
     * @public
     */
    addEventListener: function (method, listener) {
        if (typeof listener !== 'function')
            return;
        function onMessage(data) {
            listener.call(this, new MessageEvent(data, this));
        }
        function onClose(code, message) {
            listener.call(this, new CloseEvent(code, message, this));
        }
        function onError(error) {
            listener.call(this, new ErrorEvent(error, this));
        }
        function onOpen() {
            listener.call(this, new OpenEvent(this));
        }
        if (method === 'message') {
            onMessage._listener = listener;
            this.on(method, onMessage);
        }
        else if (method === 'close') {
            onClose._listener = listener;
            this.on(method, onClose);
        }
        else if (method === 'error') {
            onError._listener = listener;
            this.on(method, onError);
        }
        else if (method === 'open') {
            onOpen._listener = listener;
            this.on(method, onOpen);
        }
        else {
            this.on(method, listener);
        }
    },
    /**
     * Remove an event listener.
     *
     * @param {String} method A string representing the event type to remove
     * @param {Function} listener The listener to remove
     * @public
     */
    removeEventListener: function (method, listener) {
        var listeners = this.listeners(method);
        for (var i = 0; i < listeners.length; i++) {
            if (listeners[i] === listener || listeners[i]._listener === listener) {
                this.removeListener(method, listeners[i]);
            }
        }
    }
};
module.exports = EventTarget;
