'use strict';
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
        return extendStatics(d, b);
    }
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var EventEmitter = require('events');
var crypto = require('crypto');
var http = require('http');
var url = require('url');
var PerMessageDeflate = require('./permessage-deflate');
var extension = require('./extension');
var constants = require('./constants');
var WebSocket = require('./websocket');
/**
 * Class representing a WebSocket server.
 *
 * @extends EventEmitter
 */
var WebSocketServer = /** @class */ (function (_super) {
    __extends(WebSocketServer, _super);
    /**
     * Create a `WebSocketServer` instance.
     *
     * @param {Object} options Configuration options
     * @param {String} options.host The hostname where to bind the server
     * @param {Number} options.port The port where to bind the server
     * @param {http.Server} options.server A pre-created HTTP/S server to use
     * @param {Function} options.verifyClient An hook to reject connections
     * @param {Function} options.handleProtocols An hook to handle protocols
     * @param {String} options.path Accept only connections matching this path
     * @param {Boolean} options.noServer Enable no server mode
     * @param {Boolean} options.clientTracking Specifies whether or not to track clients
     * @param {(Boolean|Object)} options.perMessageDeflate Enable/disable permessage-deflate
     * @param {Number} options.maxPayload The maximum allowed message size
     * @param {Function} callback A listener for the `listening` event
     */
    function WebSocketServer(options, callback) {
        var _this = _super.call(this) || this;
        options = Object.assign({
            maxPayload: 100 * 1024 * 1024,
            perMessageDeflate: false,
            handleProtocols: null,
            clientTracking: true,
            verifyClient: null,
            noServer: false,
            backlog: null,
            server: null,
            host: null,
            path: null,
            port: null
        }, options);
        if (options.port == null && !options.server && !options.noServer) {
            throw new TypeError('One of the "port", "server", or "noServer" options must be specified');
        }
        if (options.port != null) {
            _this._server = http.createServer(function (req, res) {
                var body = http.STATUS_CODES[426];
                res.writeHead(426, {
                    'Content-Length': body.length,
                    'Content-Type': 'text/plain'
                });
                res.end(body);
            });
            _this._server.listen(options.port, options.host, options.backlog, callback);
        }
        else if (options.server) {
            _this._server = options.server;
        }
        if (_this._server) {
            _this._removeListeners = addListeners(_this._server, {
                listening: _this.emit.bind(_this, 'listening'),
                error: _this.emit.bind(_this, 'error'),
                upgrade: function (req, socket, head) {
                    _this.handleUpgrade(req, socket, head, function (ws) {
                        _this.emit('connection', ws, req);
                    });
                }
            });
        }
        if (options.perMessageDeflate === true)
            options.perMessageDeflate = {};
        if (options.clientTracking)
            _this.clients = new Set();
        _this.options = options;
        return _this;
    }
    /**
     * Returns the bound address, the address family name, and port of the server
     * as reported by the operating system if listening on an IP socket.
     * If the server is listening on a pipe or UNIX domain socket, the name is
     * returned as a string.
     *
     * @return {(Object|String|null)} The address of the server
     * @public
     */
    WebSocketServer.prototype.address = function () {
        if (this.options.noServer) {
            throw new Error('The server is operating in "noServer" mode');
        }
        if (!this._server)
            return null;
        return this._server.address();
    };
    /**
     * Close the server.
     *
     * @param {Function} cb Callback
     * @public
     */
    WebSocketServer.prototype.close = function (cb) {
        var _this = this;
        if (cb)
            this.once('close', cb);
        //
        // Terminate all associated clients.
        //
        if (this.clients) {
            for (var _i = 0, _a = this.clients; _i < _a.length; _i++) {
                var client = _a[_i];
                client.terminate();
            }
        }
        var server = this._server;
        if (server) {
            this._removeListeners();
            this._removeListeners = this._server = null;
            //
            // Close the http server if it was internally created.
            //
            if (this.options.port != null) {
                server.close(function () { return _this.emit('close'); });
                return;
            }
        }
        process.nextTick(emitClose, this);
    };
    /**
     * See if a given request should be handled by this server instance.
     *
     * @param {http.IncomingMessage} req Request object to inspect
     * @return {Boolean} `true` if the request is valid, else `false`
     * @public
     */
    WebSocketServer.prototype.shouldHandle = function (req) {
        if (this.options.path && url.parse(req.url).pathname !== this.options.path) {
            return false;
        }
        return true;
    };
    /**
     * Handle a HTTP Upgrade request.
     *
     * @param {http.IncomingMessage} req The request object
     * @param {net.Socket} socket The network socket between the server and client
     * @param {Buffer} head The first packet of the upgraded stream
     * @param {Function} cb Callback
     * @public
     */
    WebSocketServer.prototype.handleUpgrade = function (req, socket, head, cb) {
        var _this = this;
        socket.on('error', socketOnError);
        var version = +req.headers['sec-websocket-version'];
        var extensions = {};
        if (req.method !== 'GET' || req.headers.upgrade.toLowerCase() !== 'websocket' ||
            !req.headers['sec-websocket-key'] || (version !== 8 && version !== 13) ||
            !this.shouldHandle(req)) {
            return abortHandshake(socket, 400);
        }
        if (this.options.perMessageDeflate) {
            var perMessageDeflate = new PerMessageDeflate(this.options.perMessageDeflate, true, this.options.maxPayload);
            try {
                var offers = extension.parse(req.headers['sec-websocket-extensions']);
                if (offers[PerMessageDeflate.extensionName]) {
                    perMessageDeflate.accept(offers[PerMessageDeflate.extensionName]);
                    extensions[PerMessageDeflate.extensionName] = perMessageDeflate;
                }
            }
            catch (err) {
                return abortHandshake(socket, 400);
            }
        }
        //
        // Optionally call external client verification handler.
        //
        if (this.options.verifyClient) {
            var info = {
                origin: req.headers["" + (version === 8 ? 'sec-websocket-origin' : 'origin')],
                secure: !!(req.connection.authorized || req.connection.encrypted),
                req: req
            };
            if (this.options.verifyClient.length === 2) {
                this.options.verifyClient(info, function (verified, code, message, headers) {
                    if (!verified) {
                        return abortHandshake(socket, code || 401, message, headers);
                    }
                    _this.completeUpgrade(extensions, req, socket, head, cb);
                });
                return;
            }
            if (!this.options.verifyClient(info))
                return abortHandshake(socket, 401);
        }
        this.completeUpgrade(extensions, req, socket, head, cb);
    };
    /**
     * Upgrade the connection to WebSocket.
     *
     * @param {Object} extensions The accepted extensions
     * @param {http.IncomingMessage} req The request object
     * @param {net.Socket} socket The network socket between the server and client
     * @param {Buffer} head The first packet of the upgraded stream
     * @param {Function} cb Callback
     * @private
     */
    WebSocketServer.prototype.completeUpgrade = function (extensions, req, socket, head, cb) {
        var _this = this;
        var _a;
        //
        // Destroy the socket if the client has already sent a FIN packet.
        //
        if (!socket.readable || !socket.writable)
            return socket.destroy();
        var key = crypto.createHash('sha1')
            .update(req.headers['sec-websocket-key'] + constants.GUID, 'binary')
            .digest('base64');
        var headers = [
            'HTTP/1.1 101 Switching Protocols',
            'Upgrade: websocket',
            'Connection: Upgrade',
            "Sec-WebSocket-Accept: " + key
        ];
        var ws = new WebSocket(null);
        var protocol = req.headers['sec-websocket-protocol'];
        if (protocol) {
            protocol = protocol.trim().split(/ *, */);
            //
            // Optionally call external protocol selection handler.
            //
            if (this.options.handleProtocols) {
                protocol = this.options.handleProtocols(protocol, req);
            }
            else {
                protocol = protocol[0];
            }
            if (protocol) {
                headers.push("Sec-WebSocket-Protocol: " + protocol);
                ws.protocol = protocol;
            }
        }
        if (extensions[PerMessageDeflate.extensionName]) {
            var params = extensions[PerMessageDeflate.extensionName].params;
            var value = extension.format((_a = {},
                _a[PerMessageDeflate.extensionName] = [params],
                _a));
            headers.push("Sec-WebSocket-Extensions: " + value);
            ws._extensions = extensions;
        }
        //
        // Allow external modification/inspection of handshake headers.
        //
        this.emit('headers', headers, req);
        socket.write(headers.concat('\r\n').join('\r\n'));
        socket.removeListener('error', socketOnError);
        ws.setSocket(socket, head, this.options.maxPayload);
        if (this.clients) {
            this.clients.add(ws);
            ws.on('close', function () { return _this.clients.delete(ws); });
        }
        cb(ws);
    };
    return WebSocketServer;
}(EventEmitter));
module.exports = WebSocketServer;
/**
 * Add event listeners on an `EventEmitter` using a map of <event, listener>
 * pairs.
 *
 * @param {EventEmitter} server The event emitter
 * @param {Object.<String, Function>} map The listeners to add
 * @return {Function} A function that will remove the added listeners when called
 * @private
 */
function addListeners(server, map) {
    for (var _i = 0, _a = Object.keys(map); _i < _a.length; _i++) {
        var event = _a[_i];
        server.on(event, map[event]);
    }
    return function removeListeners() {
        for (var _i = 0, _a = Object.keys(map); _i < _a.length; _i++) {
            var event = _a[_i];
            server.removeListener(event, map[event]);
        }
    };
}
/**
 * Emit a `'close'` event on an `EventEmitter`.
 *
 * @param {EventEmitter} server The event emitter
 * @private
 */
function emitClose(server) {
    server.emit('close');
}
/**
 * Handle premature socket errors.
 *
 * @private
 */
function socketOnError() {
    this.destroy();
}
/**
 * Close the connection when preconditions are not fulfilled.
 *
 * @param {net.Socket} socket The socket of the upgrade request
 * @param {Number} code The HTTP response status code
 * @param {String} [message] The HTTP response body
 * @param {Object} [headers] Additional HTTP response headers
 * @private
 */
function abortHandshake(socket, code, message, headers) {
    if (socket.writable) {
        message = message || http.STATUS_CODES[code];
        headers = Object.assign({
            'Connection': 'close',
            'Content-type': 'text/html',
            'Content-Length': Buffer.byteLength(message)
        }, headers);
        socket.write("HTTP/1.1 " + code + " " + http.STATUS_CODES[code] + "\r\n" +
            Object.keys(headers).map(function (h) { return h + ": " + headers[h]; }).join('\r\n') +
            '\r\n\r\n' +
            message);
    }
    socket.removeListener('error', socketOnError);
    socket.destroy();
}
