package se.jbee.inject.bind;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static se.jbee.inject.Dependency.dependency;

import java.util.List;

import org.junit.Test;

import se.jbee.inject.Dependency;
import se.jbee.inject.Injector;
import se.jbee.inject.bind.BasicBinder.TargetedBasicBinder;
import se.jbee.inject.bind.Binder.TargetedBinder;

/**
 * A test that demonstrates how to overlay general binds in specified packages and/or sub-packages
 * using {@link TargetedBasicBinder#in(se.jbee.inject.Packages)} or any of the utility methods on
 * top of it: {@link TargetedBinder#inPackageOf(Class)},
 * {@link TargetedBinder#inSubPackagesOf(Class)},
 * {@link TargetedBinder#inPackageAndSubPackagesOf(Class)}.
 * 
 * @author Jan Bernitt (jan.bernitt@gmx.de)
 */
public class TestPackageLocalisedBinds {

	private static class PackageLocalisedBindsModule
			extends BinderModule {

		@Override
		protected void declare() {
			bind( String.class ).to( "default" );
			inPackageOf( TestPackageLocalisedBinds.class ).bind( String.class ).to( "test" );
			inSubPackagesOf( Object.class ).bind( String.class ).to( "java-lang.*" );
			inPackageAndSubPackagesOf( List.class ).bind( String.class ).to( "java-util.*" );
		}

	}

	private static final Dependency<String> stringGlobal = dependency( String.class );

	private final Injector injector = Bootstrap.injector( PackageLocalisedBindsModule.class );

	@Test
	public void thatDepedencyWithoutTargetResolvedToGlobalBind() {
		assertThat( injector.resolve( stringGlobal ), is( "default" ) );
	}

	@Test
	public void thatDependencyWithTargetResolvedToSpecificBindInThatPackage() {
		Dependency<String> stringInBind = stringGlobal.injectingInto( TestPackageLocalisedBinds.class );
		assertThat( injector.resolve( stringInBind ), is( "test" ) );
	}

	@Test
	public void thatDependencyWithTargetSomewhereElseResolvedToGlobalBind() {
		Dependency<String> stringSomewhereElse = stringGlobal.injectingInto( java.awt.font.TextAttribute.class );
		assertThat( injector.resolve( stringSomewhereElse ), is( "default" ) );
	}

	@Test
	public void thatDependencyWithTargetResolvedToRelevantSubPackagesBind() {
		Dependency<String> stringInAnnotation = stringGlobal.injectingInto( java.lang.annotation.Target.class );
		assertThat( injector.resolve( stringInAnnotation ), is( "java-lang.*" ) );
	}

	@Test
	public void thatDependencyWithTargetResolvedToRelevantPackageOfPackageAndSubPackagesBind() {
		Dependency<String> stringInUtil = stringGlobal.injectingInto( java.util.List.class );
		assertThat( injector.resolve( stringInUtil ), is( "java-util.*" ) );
	}

	@Test
	public void thatDependencyWithTargetResolvedToRelevantSubPackageOfPackageAndSubPackagesBind() {
		Dependency<String> stringInUtil = stringGlobal.injectingInto( java.util.concurrent.Callable.class );
		assertThat( injector.resolve( stringInUtil ), is( "java-util.*" ) );
	}

}
