from geosnap import analyze, DataStore
from geosnap.analyze.incs import lincs_from_gdf
from geosnap.io import get_census
from geosnap.harmonize import harmonize

from numpy.testing import assert_array_almost_equal
import numpy as np

linc = analyze.incs.linc


def test_linc():
    labels_0 = [1, 1, 1, 1, 2, 2, 3, 3, 3, 4]
    labels_1 = [1, 1, 1, 1, 1, 2, 3, 3, 3, 4]
    res = linc([labels_0, labels_1])
    assert res[4] == 1.0
    assert res[7] == 0.0 == res[-1]

    labels_2 = [1, 1, 1, 1, 1, 2, 3, 3, 3, 4]
    res = linc([labels_1, labels_2])
    assert res[0] == 0.0

    res = linc([labels_0, labels_1, labels_2])
    assert res[0] == 0.25


def test_linc_from_gdf():
    columns = [
        "median_household_income",
        "p_poverty_rate",
        "p_unemployment_rate",
    ]
    reno = get_census(DataStore(), msa_fips="39900")
    rdf = harmonize(reno, target_year=1990, intensive_variables=columns, unit_index='geoid')

    rdf = analyze.cluster(reno, columns=columns, method="ward", unit_index='geoid')
    l = lincs_from_gdf(
        rdf.reset_index(), unit_index="geoid", temporal_index="year", cluster_col="ward"
    )
    assert_array_almost_equal(
        l.linc.values,
        np.array(
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.85714286,
                0.5,
                1.0,
                0.8,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.5,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.8,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.85714286,
                0.5,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.5,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.5,
                1.0,
                1.0,
                1.0,
                0.0,
                0.5,
                0.5,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        ),
        decimal=3,
    )


def test_linc_from_gdf_subset():
    columns = [
        "median_household_income",
        "p_poverty_rate",
        "p_unemployment_rate",
        "n_total_pop",
    ]
    reno = get_census(DataStore(), msa_fips="39900")
    rdf = harmonize(reno, target_year=1990, intensive_variables=columns, unit_index='geoid')

    rdf = analyze.cluster(
        rdf.reset_index(),
        columns=columns,
        method="ward",
    )
    l = lincs_from_gdf(
        rdf,
        unit_index="geoid",
        temporal_index="year",
        cluster_col="ward",
        periods=[2000, 2010],
    )

    assert_array_almost_equal(
        l.linc.values,
        np.array(
            [
                0.96969697,
                0.78571429,
                0.8,
                0.75,
                0.66666667,
                0.8125,
                0.78571429,
                0.80952381,
                1.0,
                0.8,
                0.75,
                0.74074074,
                0.80952381,
                0.80952381,
                0.92307692,
                1.0,
                0.8,
                0.78571429,
                0.78571429,
                0.75,
                0.8125,
                0.75,
                0.74074074,
                0.74074074,
                0.8,
                0.75,
                0.66666667,
                0.90909091,
                0.66666667,
                0.92307692,
                1.0,
                1.0,
                0.74074074,
                0.80952381,
                1.0,
                1.0,
                1.0,
                0.74074074,
                0.96969697,
                1.0,
                0.8125,
                0.74074074,
                0.74074074,
                1.0,
                0.80952381,
                0.8125,
                0.96153846,
                0.90909091,
                0.74074074,
                0.66666667,
                0.66666667,
                0.66666667,
                0.66666667,
                0.66666667,
                0.66666667,
                0.96153846,
                0.66666667,
                0.66666667,
            ]
        ),
        decimal=3,
    )


def test_linc_method():
    columns = [
        "median_household_income",
        "p_poverty_rate",
        "p_unemployment_rate",
        "n_total_pop",
    ]
    reno = get_census(DataStore(), msa_fips="39900")
    rdf = harmonize(reno, target_year=2010, intensive_variables=columns, unit_index='geoid')

    _, model = analyze.cluster(rdf.reset_index(), columns=columns, method="ward", return_model=True)

    l = model.lincs.linc.values

    assert_array_almost_equal(
        l,
        np.array(
            [
                0.9047619,
                0.94594595,
                0.82608696,
                0.875,
                0.97142857,
                0.9047619,
                1.0,
                0.96428571,
                0.97560976,
                1.0,
                0.82608696,
                1.0,
                0.92682927,
                0.94285714,
                1.0,
                0.94285714,
                0.92682927,
                1.0,
                0.90909091,
                0.94285714,
                1.0,
                1.0,
                1.0,
                0.975,
                0.9047619,
                0.97560976,
                1.0,
                0.82608696,
                0.82608696,
                0.94594595,
                0.875,
                0.875,
                0.96428571,
                0.875,
                0.90625,
                1.0,
                0.9137931,
                0.98360656,
                1.0,
                0.875,
                1.0,
                0.98181818,
                0.97619048,
                0.90909091,
                0.98181818,
                0.90909091,
                0.94594595,
                0.82608696,
                0.97619048,
                0.90909091,
                0.90625,
                0.9137931,
                0.93333333,
                0.93333333,
                1.0,
                1.0,
                0.93333333,
                0.93333333,
                0.975,
                0.90625,
                0.96666667,
                0.96666667,
                0.98507463,
                0.9137931,
                0.94339623,
                0.93939394,
                0.93939394,
                0.94339623,
                0.94339623,
                0.9137931,
                0.97142857,
                0.875,
                0.93939394,
                0.93939394,
                0.93939394,
                0.98507463,
                1.0,
                1.0,
                0.9047619,
                0.96666667,
                0.9047619,
                0.90909091,
                0.94339623,
                0.90625,
                0.90625,
                0.9137931,
                0.9137931,
                0.98214286,
                0.984375,
                0.95918367,
                0.95918367,
                0.95918367,
                0.92682927,
                0.92682927,
                0.98360656,
                0.96551724,
                0.98214286,
                0.96551724,
                0.984375,
                1.0,
                1.0,
                0.98214286,
                0.96551724,
                0.90625,
                0.90625,
                0.98214286,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        ),
        decimal=3,
    )
